<?php

defined('ABSPATH') || exit;

/**
 * FontAwesome Icons for Themify Icon Picker
 *
 * Version 6.4.2
 * Updated 02-10-23
 *
 * @link http://fontawesome.io/
 */
class Themify_Icon_FontAwesome extends Themify_Icon_Font {

    public function __construct() {
        parent::__construct();
    }

    public function get_id() {
        return 'fa';
    }

    public function get_label() {
        return __('FontAwesome', 'themify');
    }

    /**
     * Check if the icon name belongs to the Themify icon font
     * Assumes yes, if the icon name has "fa-" prefix
     *
     * @return bool
     */
    public function is_valid_icon($name) {
        return
            strpos($name, 'fa-') === 0 // FA v4
            || strpos($name, 'fa ') === 0 || in_array(substr($name, 0, 4), ['fas ', 'far ', 'fab '], true); // FA v5
    }

    private static function findIcon($ic, $type = 'fab') {
        $font = $type === 'fab' ? self::fab() : ($type === 'far' ? self::far() : self::fas());
        if (!isset($font[$ic]) && !isset($font[$ic . '-alt']) && !isset($font[$ic . '-o'])) {
            $tmp = explode('-', $ic);
            $s = '';
            foreach ($tmp as $t) {
                $s .= $t;
                if (isset($font[$s])) {
                    break;
                }
                $s .= '-';
            }
            if (isset($font[$s])) {
                $f = $font[$s];
                $font = is_string($f) && isset($font[$f]) ? $font[$f] : $f;
            } else {
                $font = $s = $tmp = null;
                if ($type === 'fab') {
                    return self::findIcon($ic, 'far');
                } elseif ($type === 'far') {
                    return self::findIcon($ic, 'fas');
                } else {
                    return false;
                }
            }
        } else {
            $f = isset($font[$ic]) ? $font[$ic] : (isset($font[$ic . '-alt']) ? $font[$ic . '-alt'] : $font[$ic . '-o']);

            $font = is_string($f) && isset($font[$f]) ? $font[$f] : $f;
        }
        return [$type => $font];
    }

    function get_classname($ic, $lazy = null, $data_only = false, $attrs = []) {
        $ic = trim($ic);
        $postfix = '';
        $find = ['fab ', 'far ', 'fas '];
        foreach ($find as $f) {
            if (strpos($ic, $f) !== false) {
                $postfix = trim($f);
                break;
            }
        }
        $find[] = 'fa ';
        $find[] = 'fa-';
        $ic = str_replace($find, '', $ic);
        $find = null;
        if (!$postfix) {
            $postfix = 'fab';
        }

        $font = self::findIcon($ic, $postfix);
        if ($font === false) {
            $postfix = 'fas';
        } else {
            $postfix = key($font);
            $font = $font[$postfix];
        }
        $k = $postfix . '-' . $ic;
        if (isset($font) && !isset(self::$usedIcons[$k])) {
            self::$usedIcons[$k] = is_string($font) ? ['p' => $font] : $font;
        }
        if ($data_only === true) {
            return self::$usedIcons[$k];
        }
        return self::get_svg($k, $attrs);
    }

    function get_categories() {
        return [
            'fas' => __('Solid', 'themify'),
            'far' => __('Regular', 'themify'),
            'fab' => __('Brand', 'themify')
        ];
    }

    public function get_icons_by_category($cat = '') {
        if ($cat === '') {
            $cats = $this->get_categories();
            $res = [];
            foreach ($cats as $k => $v) {
                $res[$k] = [];
                $icons = $this->get_icons_by_category($k);
                $icons = $icons[$k];
                foreach ($icons as $ic) {
                    $res[$k][$ic] = $this->get_classname($k . ' ' . $ic, false, true);
                }
            }
            return $res;
        }
        if ($cat === 'fas' || $cat === 'far' || $cat === 'fab') {
            return [$cat => array_keys(call_user_func("Themify_Icon_FontAwesome::$cat"))];
        }
        return [];
    }

    public static function fab() {
        return [
    '42-group' => [
        'p' => 'M320 96V416C341.011 416 361.818 411.861 381.23 403.821C400.641 395.78 418.28 383.995 433.138 369.138C447.995 354.28 459.78 336.641 467.821 317.23C475.861 297.818 480 277.011 480 256C480 234.989 475.861 214.182 467.821 194.771C459.78 175.359 447.995 157.72 433.138 142.863C418.28 128.005 400.641 116.22 381.23 108.179C361.818 100.139 341.011 96 320 96ZM0 256L160.002 416L320.003 256L160.002 96L0 256ZM480 256C480 277.011 484.138 297.818 492.179 317.23C500.219 336.643 512.005 354.28 526.862 369.138C541.72 383.995 559.357 395.781 578.77 403.821C598.182 411.862 618.989 416 640 416V96C597.565 96 556.869 112.858 526.862 142.863C496.857 172.869 480 213.565 480 256Z',
        'vw' => 640,
        'vh' => 512
    ],
    '500px' => [
        'p' => 'M6.46 21.52c-.4-.89-.44-1.14.46-1.45 1.6-.5.5.58 2.7 3.08h.02v-5.87a6.03 6.03 0 0 1 6.1-5.76 6.09 6.09 0 0 1 6.11 6.05c0 3.96-3.8 7.07-8.03 5.83-.66-.26-.13-1.98.53-1.79 3.31 0 5.59-.63 5.59-4.02 0-3.82-4.82-5.6-7.3-2.79-1.48 1.65-1.1 2.63-1.1 9.85a8.26 8.26 0 0 0 10.01-1.26A8.2 8.2 0 0 0 15.7 9.34c-2.21 0-4.3.87-5.84 2.41-.02.02-1 1.03-1.32 1.5l-.03.03c-.2.3-.4.57-1.26.38-.43-.1-.9-.36-.9-.73V1.25c0-.31.25-.66.66-.66H22.1c.52 0 .52.73.52.95 0 .24 0 .94-.52.94H8.14v8.3h.02c6.52-6.85 17.68-2.24 17.68 6.81 0 11.14-15.3 13.77-19.38 3.93Zm3.95-16.3c-.03.26.29 1.53.91 1.29 7.8-2.97 12.68 2.52 13.1 2.52.3 0 1.42-.96.88-1.42-5.82-5.57-14.65-3.57-14.89-2.4Zm14.15 20.7A12.18 12.18 0 0 1 3.81 19.4c0-.77-1.9-.46-1.8.2 1.5 10.84 15.37 16.06 23.85 7.59.43-.5-.79-1.78-1.3-1.28Zm-11.2-6.76c0 .25.26.46.34.54.18.18.37.27.53.27.23 0 .16.01 1.39-1.22 1.22 1.2 1.2 1.22 1.4 1.22.33 0 1.15-.65.66-1.14l-1.08-1.08 1.14-1.14c.4-.42-.63-1.36-1.01-.98l-1.12 1.12c-1.16-1.18-1.15-1.22-1.35-1.22-.31 0-1.12.73-.77 1.08l1.13 1.14c-1.13 1.12-1.27 1.2-1.27 1.41Zm0 0',
        'vw' => 28
    ],
    'accessible-icon' => 'm26.5 16-.81 9.81q-.13 1.2-1.1 1.6t-1.87-.16-.85-1.75l.63-7.62-2.56.12q1 2 1 4.25 0 3.88-2.63 6.69l-2.44-2.44q1.25-1.44 1.5-3.31t-.44-3.44-2.21-2.66T11.25 16Q8.8 16 7 17.63l-2.44-2.44q2.19-2 5.06-2.5l4.75-5.38-2.69-1.56L8.5 8.62q-.94.82-1.94.41T5.34 7.66t.6-1.85L10.17 2q1-.88 2.2-.19l8.74 5.06q.75.5.9 1.5t-.52 1.75l-3.63 4.2 6.63-.38q.87-.07 1.47.56t.53 1.5zm-4.06-9.63q-1.32 0-2.25-.93t-.94-2.25.94-2.25T22.44 0q.87 0 1.6.44t1.15 1.15.44 1.6q0 1.31-.94 2.25t-2.25.93zM11.25 28.5q1.88 0 3.5-1.06l2.44 2.5q-1.82 1.43-4 1.87t-4.22-.12-3.72-1.78-2.72-3.25-1.03-4.41q0-3.31 2.06-5.94l2.5 2.44Q5.06 20.25 5 22t.69 3.19 2.22 2.37 3.34.94z',
    'accusoft' => [
        'p' => 'm20.13 15.75-1.44 1.31-2.82 2.6-1.37 1.28-5.81 2.12q-.44.2-.44.57l.03.8.03.95q-7.13 2.31-7.19 2.31-1.12.38-1.12.19 0-.06.12-.31l3.75-3.7q3.7-3.62 7.47-7.37t4.03-3.97.47-.4.4-.28.35-.16.28-.12h.06l3.2 4.12v.06zm10 7.5q-.2-.19-7.47-9.53t-7.54-9.6q-.3-.43-.34-.65t.22-.31.5-.1h.88Q20 3 23.63 3q1.18 0 1.56.12t.81.7q1.06 1.3 13 15.5.94 1.18.44 1.43l-3.66 1.53q-3.53 1.47-3.72 1.53-.43.31-.9.22t-1.03-.78zm9.56.13q.62.12.12.5-.12.06-3.94 2.43t-3.93 2.44q-.32.13-.6.19t-.75.03-.65-.06-.75-.16-.63-.12q-.19-.07-4.97-1.13t-9.5-2.12-4.78-1.07q-.25-.12-.25-.25-.06-.18.69-.43l4.6-1.7 4.78-1.74q.3-.13.87-.2.44 0 4.13.63l3.56 4.63q1.19 1.37 2.25.87l7.37-3.25.9.2 1 .21q.42.1.48.1z',
        'w' => 1.25,
        'vw' => 40
    ],
    'acquisitions-incorporated' => [
        'p' => 'M21.13 29.25q.06.25.21 1.22t.22 1.53H.5q-.38 0-.31-.38.06-.62.31-2.43 0-.38.44-.38 7.12.07 7.62-.19.94-.37.94-1.43-.06-6.7-.06-8.07 0-.31-.25-.37-.44-.13-3-.75-.13.37-.47 1.12t-.47 1.13q.06.19 2.19.19v1.75H.25l.25-1.57q0-.18.31-.18.63-.07 1.63 0 .31 0 .44-.32.43-1 1.37-3.28t2.88-6.78q.25-.69.5-1.31t.43-1.03.35-.85.25-.62.22-.47.12-.35.06-.25.07-.21.03-.2-.03-.21q0-.75-.78-1.06t-1.6-.32H.44Q.13 1.56.1 1.03T0 .3q-.06-.32.32-.32H21.5q-.12.7-.12 1-.06.2-.16.85t-.1 1.03q-.06.25-.3.25h-3.88q-2.75.06-3.5.12-.44.07-.75.28t-.44.6 0 .81q.44 1.31 2.32 5.56.8 1.94 2 4.75 1.56 3.75 1.96 4.44t1.29.75h1.25l.25 1.7h-7.25v-1.7H16l.2-.25-.35-.78-.47-1.15q-.19-.32-.57-.2-.87.26-2.3.57-.38.13-.38.5 0 .56.03 1.22t.03 1.25v5.6q.06 1.24 1.31 1.43.82.19 7.07.19.37 0 .46.03t.1.4zM10.24 16.37q.56.25 1.13 0 .37-.06.78-.18t.9-.28.75-.22q.07 0 .16-.07l.1-.06q-.95-2.56-1.2-3.19t-.56-1.62-.6-2-.52-1.57h-.63l-3.19 8.25q.32.2.38.2.44.12 1.31.4t1.19.34z',
        'w' => 0.673,
        'vw' => 22
    ],
    'adn' => 'm15.5 10.5 4.06 6.13h-8.12zM31 16q0 6.44-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5t10.97 4.53T31 16zm-6.25 5.19L15.5 7.25 6.25 21.19h1.88l2.12-3.25h10.5l2.13 3.25h1.87z',
    'adobe' => [
        'p' => 'M18.13 4H28.8v24zm-7.44 0L0 28V4h10.69zm3.68 8.88L21.13 28h-4.57l-1.94-4.75H9.75z',
        'w' => 0.898,
        'vw' => 29
    ],
    'adversal' => 'M30.13 2q1.12 0 1.5.38t.37 1.5v24.25q0 1.18-.38 1.53t-1.5.34H1.82q-1.13 0-1.48-.34T0 28.19V3.8q0-1.06.37-1.44T1.81 2h28.32zm-19 11.75q.87.63.74 1.75-.06.5-1.28.72t-3.06.47-2.78.69q-1.31.68-1.97 1.9t-.53 2.6q.19 1.93 1.72 2.97t3.62.93 3.85-1.22q.5-.37.75-.06.06.06.12.19t.13.19q.31.43 1.4.6t1.53-.17q.32-.24.13-.62-.38-.88-.47-1.94t-.06-1.87-.03-1.75v-2.94l-.03-1.22-.1-1.12-.22-.88-.4-.81-.6-.57-.84-.53q-4-1.62-7.81.32-.57.3-1.25 1.12t-.94 1.66.62 1.22q.44.25 1.44.37.38.13.63.03t.43-.53q.57-1.5 2.28-1.97t2.97.47zm17.93 13.88q0-.32-.12-.47t-.22-.16H3.25l-.34-.03q-.28-.03-.38 0t-.22.19-.12.47q-.07.3.06.47t.25.18.4 0 .35-.03h24.69l.34.03.4.03q.13 0 .26-.15t.12-.53zm.25-21.25V6.3q0-.94-.87-.87h-.57q-1.25 0-1.43.22t-.2 1.4v4.44q-1.43-.81-2.74-1.03t-2.22.03-1.69.78-1.19 1.06-.65 1.1q-1.56 3.12-1 6.25.37 2.06 1.37 3.5t2.28 2.03 2.7.53 2.65-.75q.06-.06.19-.13.12-.12.28-.12t.34.31q.25.38.6.44t.9 0 .75-.06q.5 0 .5-.88V6.38zm-3.25 14.5q-.56 1.43-1.72 1.93t-2.25.03-1.65-1.65Q19 18 20.5 14.87q.75-1.62 2.4-1.68t2.66 1.37q.88 1.19.9 3.44t-.4 2.88zM7.7 23.05l-.19-.03-.28-.03-.31-.06-.32-.13-.28-.15-.31-.22-.22-.28-.16-.35-.06-.43q-.06-1.32 1.25-1.63l4.56-.94q.32-.06.41.07t.1.37v.19q.06 1.56-1.13 2.62t-3.06 1z',
    'affiliatetheme' => 'M9.97 14.84q-2.4 3.35-5.06 4.97t-4.04.6-.62-4.1 3.16-6.37T8.47 5t4.03-.63.63 4.07-3.16 6.4zm21.97-3.59q.06.81.06 1.56 0 4.7-2.28 8.63t-6.19 6.25T15 30q-3.81 0-7.19-1.62t-5.75-4.44q3.32-.88 6-3.06t4.25-5.25q.82 3.18 3.07 4.62 3.43 2.06 8.28-.56t8.28-8.44z',
    'airbnb' => [
        'p' => 'M14 23.32c-1.58-1.98-2.5-3.71-2.81-5.2-1.41-5.5 7.04-5.5 5.63 0-.34 1.52-1.27 3.25-2.82 5.2Zm8.63 4.58c-2.63 1.14-5.22-.68-7.45-3.16 6.5-8.13 2.88-12.5-1.18-12.5-3.43 0-5.32 2.9-4.58 6.28.43 1.83 1.58 3.9 3.4 6.22-2.03 2.26-3.78 3.3-5.32 3.44-3.13.46-5.57-2.57-4.46-5.7.95-2.44 6.99-14.45 7.25-15.1.98-1.88 1.6-3.58 3.7-3.58 2.03 0 2.72 1.62 3.78 3.74 2.25 4.41 5.59 11.1 7.18 14.94a4.18 4.18 0 0 1-2.32 5.42Zm2.94-8.5C17.52 2.24 17.07 2 14 2c-2.84 0-4.05 1.98-5.3 4.55C2.08 19.82 1.44 21.7 1.39 21.86A5.96 5.96 0 0 0 6.97 30c1.35 0 3.79-.38 7.02-3.9 3.67 3.99 6.33 3.9 7.02 3.9 3.93 0 7.18-3.8 5.6-8.14 0-.24-1.05-2.43-1.05-2.47Zm0 0',
        'vw' => 28
    ],
    'algolia' => [
        'p' => 'M14.31 11.44q2.32 0 3.94 1.62T19.88 17t-1.63 3.94-3.94 1.62-3.94-1.62T8.75 17t1.63-3.94 3.93-1.62zm3.94 3.5q.13-.06.06-.19-1.25-2.25-3.81-2.38-.19 0-.19.2v4.18q0 .25.25.13zM24.31 2q1 0 1.85.5t1.34 1.34.5 1.85V26.3q0 1.5-1.1 2.6T24.32 30H3.7q-1.5 0-2.6-1.1T0 26.3V5.7q0-1.5 1.1-2.6T3.68 2H24.3zM11.63 7.31h.06v.94q0 .25.19.19 1.18-.31 2.43-.31t2.44.3q.19.07.19-.18v-.94q0-.5-.38-.87t-.87-.38h-2.81q-.5 0-.88.38t-.37.87zM6.37 9.62q-.37.32-.37.82t.38.87l.5.5q.12.13.25-.06.43-.56.93-1.06.44-.5 1.07-.94.18-.13 0-.25L8.69 9q-.38-.31-.88-.31t-.87.3zm7.97 15.26q3.29 0 5.6-2.32t2.31-5.6-2.31-5.59-5.6-2.3-5.59 2.3-2.31 5.6 2.31 5.6 5.6 2.3z',
        'w' => 0.873,
        'vw' => 28
    ],
    'alipay' => 'M23.63 2q1.8 0 3.09 1.28T28 6.38v14.68q-1.5-.5-4.63-1.62l-4.68-1.69q.62-1.25 1.25-2.66t.87-2.21l.25-.82h-5.5v-1.94h6.82V8.94h-6.82v-3.2h-3.19v3.2H5.5v1.18h6.87v1.94H6.62v1.07h11.13q-.63 1.87-1.69 3.75Q11 15.3 8.12 15.3q-3.62 0-5.03 1.47t-1.15 3.66q.18 1.81 1.62 3.28T8 25.19q2.81 0 5.22-1.4t4.03-3.67q4.69 2.13 10.75 5.5 0 1.2-.6 2.2t-1.59 1.59-2.19.6H4.37q-1.81 0-3.1-1.29T0 25.62V6.39q0-1.82 1.28-3.1T4.38 2h19.24zM2.93 20.19q-.06-1.13 1.04-2.16T7.87 17q2.13 0 6.82 2-3.44 4.56-7.38 4.56-4.19 0-4.37-3.37z',
    'amazon' => [
        'p' => 'M16.06 10.19V8.16l-.03-.6-.1-.47-.18-.43-.28-.35-.4-.25-.54-.12-.65-.07h-.29l-.62.2-.88.46-.87.9-.72 1.5-4.62-.43Q5.88 6 8.16 4t6.15-2q1.94 0 3.38.56T19.88 4t1.09 1.66.34 1.46v9.2q0 .8.5 1.62t1 1.25l.5.44-3.56 3.5q-2.37-2.25-2.81-2.94-.81 1.31-2.25 2.1t-3 .84-2.97-.44-2.31-1.85-.91-3.34q0-1.81.81-3.19t1.97-2.12 2.75-1.22 2.75-.63 2.28-.15zm0 5.44v-2.57q-5.25.19-5.25 3.63 0 1.81 1.75 2.37 1.82.63 2.88-1 .62-.94.62-2.44zm8.5 10.18q-.43.57-1.56 1.4t-3.75 1.82-5.56.97q-3 0-5.94-1.28t-4.53-2.56-2.6-2.47q-.18-.25-.15-.4t.19-.16.28.06l2.19 1.28q.69.41 2.4 1.22t2.9 1.12 3.2.66 3.78.13 4.09-.82 4.75-1.72q.19-.06.38-.03t.18.25-.25.53zm2.5.13q-.56 1.31-1.31 1.94-.25.25-.4.15t-.04-.4q.38-.7.72-1.88t.1-1.56q-.38-.44-3-.19-.82.06-.88-.03t.1-.22.4-.25.56-.25.66-.22.66-.15q2.37-.25 2.87.37.13.25 0 1.1t-.44 1.59z',
        'w' => 0.873,
        'vw' => 28
    ],
    'amazon-pay' => [
        'p' => 'M0 20.31q.19-.37.63-.12 1.75.94 1.93 1 3.75 1.87 7.94 2.94 2.06.43 4 .75 2.94.37 5.88.3 1.56-.05 3.3-.24 5.32-.63 10.2-2.69.31-.12.56-.06.31.06.4.34t-.15.53q-.13.13-.56.44-2.7 1.88-6.25 3-2.32.75-4.7 1.13-1.8.25-3.3.25l-.2.06h-1.3l-.13-.06h-.69q-1.5-.07-3.19-.32-2.56-.43-5-1.31-5.06-1.81-9.12-5.44 0-.06-.25-.37v-.13zM9.88 4.06h.5l.62.13q1.94.37 2.75 2.31.56 1.25.5 3.06-.06 1.57-.63 2.75-.93 1.88-2.74 2.2-2.13.43-3.82-.95l-.12-.12v4.69q-.07.31-.32.31-.68.06-1.3 0-.32 0-.32-.31V4.88q0-.32.1-.4t.4-.1h.69q.43 0 .5.37 0 .19.06.56l.06.07q0-.07.1-.13l.18-.12.16-.07q1.06-.8 2.37-.93l.1-.03.15-.04zM6.94 6.7v5.25q0 .12.06.19 1.2.86 2.38.86 2.06.13 2.62-1.75.25-.81.25-1.94 0-1.19-.31-2.06-.5-1.5-2.07-1.63-1.5-.06-2.87.88-.06.06-.06.19zm12.81-2.63h.75q.13 0 .25.03t.25.03h.19q.81.13 1.37.44 1 .63 1.2 1.94.05.31.05.81v6.75q-.06.32-.3.32h-1q-.32 0-.38-.32 0-.12-.13-.68l-.06.06q-1.19.94-2.57 1.15T17 14.25q-1.13-.62-1.31-2-.13-1.06.25-1.87.56-1.13 1.81-1.57.88-.31 2.13-.25.93.07 2 .32h.06V7.25q0-.62-.32-1.03t-1.06-.53q-1.06-.19-2.31.06l-1.5.38q-.44.12-.44-.32V5.2q0-.38.38-.5.87-.31 1.06-.31.87-.25 1.81-.32h.19zm2.19 8v-1.81q0-.13-.13-.13-.18-.06-.5-.09t-.56-.06-.5-.1q-.88-.06-1.5.13-.81.31-1.06 1-.2.62 0 1.22t.81.84q.44.19 1 .13 1.25-.13 2.38-.94.06-.06.06-.19zm16.25 8.06v1.07l-.03.31q-.03.25-.1.38-.43 2.06-1.56 3.5-.19.25-.75.8-.13.13-.31.2-.13 0-.22-.07t-.03-.18q0-.07.06-.2.75-1.87 1-2.8t.19-1.5q0-.5-.44-.63-.56-.13-.94-.19-.87-.06-2.12.07-.25 0-.75.06t-.7.12h-.18q-.19 0-.22-.12t.03-.25.25-.25q1-.69 2.5-.94 1.2-.25 2.44-.13.81.07 1.31.25.5.13.57.5zm-8.25-7.68.06.12q0-.06.5-1.37l2.13-6.25.09-.13q.03-.06.03-.12.13-.32.5-.32h1.25q.25 0 .19.25 0 .2-.06.38-3.57 9.06-4.44 11.19-.19.56-.56 1.19-.82 1.37-2.5 1.3-.38 0-1-.12-.5-.06-.5-.56v-.62l.3-.25h.5q.5.06.95-.07.68-.12 1-.81.06-.25.25-.72t.3-.65q.13-.38 0-.76-1.55-3.8-3.68-9.12-.12-.13-.12-.38-.07-.25.18-.25h1.44q.25 0 .38.32.37.94.37 1.06.38.94.81 2.13t.88 2.4.75 2.16z',
        'w' => 1.193,
        'vw' => 38
    ],
    'amilia' => 'M15 2q6.06 0 9.03 2.28T27 10.5v18.56q0 .44-.31.57h-4.2q-3.3 0-3.68-.13-.44-.06-.44-.56v-1.57q-1.93 1.07-4.68 1.85T8.8 30q-1.94 0-3.44-.56T3.1 28.06t-1.28-1.78-.66-1.65T1 23.5v-1.44l.03-1 .1-1.1.21-1.06.35-1.03.5-.8.75-.57q1.37-.56 6.15-1.5t9.03-1.5v-2.38q0-1.31-.06-1.93t-.4-1.35-1.1-1-2.06-.28q-2.5 0-5.2.9T4.94 9.5q-.25.25-.65.15t-.6-.53q-.25-.5-.62-2.44-.19-.87.44-1.25Q8.25 2 15 2zm3.13 21.75v-6.69q-1.94.19-4.1.82t-3.4 1.3q-.2.07-.35.23t-.25.28-.19.34-.15.38-.1.37-.06.38-.03.37v.97q0 .31.12.72t.47.9 1.16.82 1.94.31q2.5 0 4.93-1.5z',
    'android' => 'M5.63 12.75V20q0 .75-.5 1.25t-1.25.5-1.32-.5T2 20v-7.25q0-.69.56-1.19t1.32-.5q.68 0 1.21.5t.54 1.19zm.62 9.88V11.38h15.5v11.24q0 .75-.56 1.29t-1.32.53h-1.25v3.81q0 1.12-.87 1.56t-1.72 0-.84-1.56v-3.81H12.8v3.81q0 1.06-.84 1.53t-1.72.03-.88-1.56v-3.81H8.12q-.75 0-1.31-.54t-.56-1.28zm15.5-11.88H6.19q0-.75.19-1.53t.53-1.44.84-1.28 1.13-1.1 1.3-.84L9 2.37q-.19-.25.13-.37l.24.12 1.26 2.25q3.37-1.5 6.74 0l1.25-2.25q.13-.25.32-.06.12.13.06.31l-1.19 2.2q1.82.93 2.88 2.59t1.06 3.59zM11.12 7.31q0-.25-.18-.47t-.47-.22-.47.22-.19.47.19.44.44.19q.31 0 .5-.19t.19-.44zm7.07 0q0-.25-.19-.47t-.44-.22-.47.22-.22.47q0 .2.1.35t.25.22.34.06q.25 0 .44-.19t.19-.44zm5.93 3.75q.5 0 .91.22t.69.63.28.84V20q0 .75-.56 1.25t-1.32.5q-.68 0-1.22-.5T22.37 20v-7.25q0-.69.53-1.19t1.22-.5z',
    'angellist' => 'M21.69 13.44q2.06.5 2.75 2t.69 3.87q0 5.5-3.29 9.1T13.2 32q-4.38 0-7.35-2.88t-2.96-6.44 3.18-4.44q-.5-1.37-.5-1.87 0-1.13.97-2.22t2.1-1.1q.5 0 1.25.32Q6.94 5.06 6.94 3.25q0-.69.19-1.28T7.66.94t.87-.69T9.81 0q2.32 0 5.75 10.38.13.3.2.37.06-.13.62-1.72t1.06-2.75 1.22-2.62 1.53-2.28 1.63-.82q1.25 0 1.96.88t.72 2.19q0 1.93-2.81 9.8zM19.5 6.75l-2.06 5.88 2.12.37q2.94-8.13 2.94-9.25 0-1-.69-1-1 0-2.31 4zM8.87 3q0 1.81 2.88 9.31.38-.25 2.19-.06L11.8 5.94q-1.43-4.2-2.44-3.88-.5.13-.5.94zm-.12 12.25q-.25 0-.53.22t-.5.53-.22.56q0 1.07 1.9 3.6t2.85 2.53q.31 0 .5-.25t.19-.5q0-.94-1.5-3.82t-2.69-2.87zm11.5 11.63q1.63-1.75 2.28-4.25T23 18.3t-.81-2.3q-1-.88-3.78-1.38t-4.85-.5q-.5 0-.75.16t-.28.28-.03.44q0 2.5 7.13 2.44H20q.94 0 1.06 1.31-.5.5-1.68 1-1.2.44-1.7.81-1.18.82-1.96 2.03t-.78 2.47q0 .5.34 1.53t.35 1.6l-.13.44q-2.5-.13-2.63-3.82h-.74q.24.88-.47 1.75t-1.91.88q-1.88 0-3.4-2.03T6.11 22q.38.55.88 1.21t.72.94.53.6.53.4.53.1q.19 0 .47-.16t.28-.41q0-.62-1.44-2.6t-2-1.96q-.93 0-1.43 1.12t.18 3.19Q6.44 27 8.53 28.47t4.9 1.47q4.07 0 6.82-3.06zM14.12 19q.44 1.25.57 1.56.69-.75 1.25-1.18z',
    'angrycreative' => [
        'p' => 'm40 14.88-.19 1.74-2.19.2-.12 1.12 2.19-.19-.2 1.81-2.18.13-.12 1.25 2.12-.13-.19 1.63-4 .25.75-7.07-2.94 7.2-2 .12-1.5-7.38 1.94-.12.82 5 2-5.19zm-13.31 8.37.81-7.5 1.75-.13-.81 7.5zm-16.57 1L8.95 22l-.25 2.38-1.75.06.18-1.82q-1.3 2.13-3.56 2.25-1.62.07-2.65-.93T0 21.24q.13-1.5 1.28-2.74t2.97-1.38q.5 0 .9.07t.72.18.54.29.4.37.28.38.22.34l.13.19.18-1.94 2.44-.13q2.32-.18 2.63 1.5l.19-1.68 4.06-.25-.19 1.75-2.25.12-.12 1.13 2.25-.13L16.5 21l-2.31.19-.13 1.25 2.32-.19-.13 1.31 3.38-7.31 1.5-.13 2 6.94.56-5.37-1.38.12.2-1.75 4.68-.31L27 17.5l-1.5.13-.62 5.75-3.63.18-.25-.93-2.5.12-.5 1zm-2.74-4.38-1.7.13q-.56-1.06-1.62-1-.44 0-.81.19t-.66.5-.47.65-.25.72q-.06.82.41 1.4t1.4.54q1-.06 1.82-1.19l1.69-.06zm3.5-.5q.06-.56-.32-.68t-1.31-.07l-.19 1.75q.44 0 .66-.03t.53-.12.44-.31.18-.54zm1.3 4.44.38-3.75q-.43 1.13-1.75 1.7zM20.64 21l-.5-2.38-.94 2.5zM10 16.31l-.25-1.06-2.5.12-.5 1.2-2 .12 3.56-7.63 1.5-.12 1.94 6.37.63-6.5 1.68-.12 2.38 3.94.37-4.13 1.82-.12-.25 2.56q.37-.75.65-1.16t1.22-1.06T22.31 8q.44-.06.82 0t.65.16.53.22.44.28.35.31.25.31.15.28.13.22l.06.1.19-2 2.44-.2q1.62-.12 2.31.63l-.31-.75 2-.12.81 2.69 1.44-2.88 1.94-.12L33.8 12l-.3 2.82-1.82.12.31-2.81-1-2.7q.13 2.26-2 3.13L30.63 15l-2.31.13-1.2-2.32-.24 2.44-1.75.13.19-2q-.07.06-.22.31t-.28.4-.38.41-.5.44-.6.38-.77.3-1 .13q-2.57.2-3.44-2.43l-.25 2.5-1.56.12L13.94 12l-.37 4.13zm17.31-5.12q.94-.07 1.32-.25.43-.25.5-.75t-.2-.69q-.37-.19-1.43-.13zm-6.81 2.19q.5.56 1.31.5.44-.07.82-.26.5-.3.93-.87l-2.12.13.19-1.5 3.93-.26.07-.74-1.94.12q-.44-.38-.69-.5-.44-.19-.81-.19-.69.06-1.38.56-.68.63-.8 1.57-.07.87.5 1.44zm-12.63.5 1.5-.07-.5-2.37z',
        'w' => 1.25,
        'vw' => 40
    ],
    'angular' => [
        'p' => 'M10.63 16.75 13 11l2.38 5.75h-4.76zM13 2l13 4.63-2 17.25L13 30 2 23.87 0 6.63zm8.13 21.38L13 5.13 4.87 23.38h3l1.7-4.13h6.87l1.62 4.13h3.07z',
        'w' => 0.81,
        'vw' => 26
    ],
    'app-store' => 'm16 7.56.56-1q.25-.43.78-.6t.97.13.6.78-.16 1l-5.44 9.44h3.94q.94 0 1.4.88t.04 1.69H7.12q-.56 0-.93-.38t-.38-.9.38-.91.93-.38h3.25l4.13-7.18-1.25-2.25q-.31-.5-.16-1t.6-.78.97-.13.78.6zM11.06 21.2l-1.19 2.1q-.3.44-.8.6t-.98-.13-.6-.78.13-1l.94-1.56q1.5-.44 2.5.75zm10.56-3.88h3.32q.56 0 .94.38t.37.9-.37.91-.94.37h-1.82L24.38 22q.25.5.1 1t-.6.78-.96.13-.79-.6q-1.06-1.94-2.62-4.62t-2.06-3.57q-.7-1.43-.47-2.59t.9-1.72l3.75 6.5zM16 .5q6.44 0 10.97 4.53T31.5 16t-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5zM29.5 16q0-5.63-3.97-9.56T16 2.5q-5.63 0-9.56 3.97T2.5 16q0 5.62 3.97 9.56T16 29.5q5.62 0 9.56-3.97T29.5 16z',
    'app-store-ios' => 'M25 2q1.25 0 2.13.88T28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22zM7.94 24l1.19-2.06q-.94-1.2-2.44-.7l-.94 1.5q-.25.5-.1 1t.6.76.94.12.75-.62zm8.68-3.31q.44-.82-.03-1.66t-1.4-.84H11.3l5.38-9.32q.25-.43.12-.93t-.6-.75-.96-.13-.75.56l-.56.94-.57-.94q-.25-.43-.75-.56t-.93.13-.57.75.13.93l1.25 2.2-4.07 7.12H5.25q-.5 0-.88.34t-.37.88.37.9.88.38h11.37zm6.13 0q.5 0 .88-.38t.37-.9-.38-.88-.87-.34H19.5q-1-1.75-3.69-6.38-.69.5-.9 1.63t.46 2.56l2.16 3.75L20 24q.25.5.75.63t.94-.13.6-.75-.1-1l-1.25-2.06h1.81z',
    'apper' => [
        'p' => 'M2.63 14.94q1.06 0 1.46.19t.66.75v-1.44q0-1-1.13-1-.25 0-.4.03t-.28.12-.16.25-.1.41H.32q0-1.12.38-1.5.56-.75 2.87-.75 2.75 0 3.38 1.19.25.56.25 1.5v4.75H4.75l.13-1.19H4.8q-.19.5-.44.75t-.68.38-1.13.12q-1.44 0-2-.53T0 17.19q0-1.19.62-1.72t2-.53zm1.06 3q.62 0 .84-.16t.22-.65q0-.44-.25-.6t-1-.15q-1.06 0-1.06.75 0 .5.25.65t1 .16zM8.56 12h2.38l-.06 1.31h.06q.37-.81.84-1.06t1.47-.25q1.31 0 2 .56.75.7.75 3 0 2.5-.75 3.32-.56.62-2.06.62-.94 0-1.4-.28t-.73-1.03H11v4.37H8.56V12zm5.07 3.81q0-1.31-.26-1.68t-1.06-.38-1.06.34-.25 1.47q0 1.32.25 1.72t1.12.4q.75 0 1-.37t.25-1.5zM17.25 12h2.38l-.07 1.31q.44-.81.9-1.06t1.48-.25q1.31 0 1.93.56.82.7.82 3 0 2.5-.75 3.32-.57.62-2.07.62-1 0-1.46-.28t-.66-1.03h-.06v4.37h-2.44V12zm5.06 3.81q0-1.31-.25-1.68T21 13.74t-1.06.34-.25 1.47q0 .94.12 1.35t.38.6.81.18q.81 0 1.06-.38t.25-1.5zm3.31-.25q0-2.19.82-2.87.75-.69 2.81-.69 2 0 2.75.75t.75 2.75v.69H28v.31q0 .5.13.81t.37.44.75.13q.69 0 .9-.2t.23-.68h2.37v.13q0 1-.44 1.56-.62.81-2.81.81-2.38 0-3.13-.75-.75-.69-.75-3.19zm4.82-.68v-.26q0-.56-.25-.74t-1-.2q-.69 0-.88.2t-.25.74v.26h2.38zM33.87 12h2.44l-.12 1.25h.06Q36.81 12 38.06 12q1.07 0 1.5.6t.44 2.09v.62h-2.19v-.25q0-.75-.12-1t-.5-.25q-.63 0-.81.57-.07.25-.07.8v4.26h-2.44V12z',
        'w' => 1.248,
        'vw' => 40
    ],
    'apple' => [
        'p' => 'M19.69 16.81q0 3.94 3.81 5.75-.75 2.38-2.56 4.88T17.12 30q-.87 0-2.3-.56t-2.5-.57-2.54.57-2.16.56q-1.37.06-2.9-1.5t-2.44-3.25-1.4-3.06Q0 19.56 0 17.12 0 13.2 2.16 11t5.03-2.25q.93 0 2.68.63T12 10q.19 0 2.19-.69t3.31-.62q3.56.31 5.31 2.81-3.18 1.88-3.12 5.31zM16.12 6.5q-2 2.38-4.3 2.19-.26-2.5 1.56-4.5.75-.88 1.93-1.5T17.63 2q.18 2.5-1.5 4.5z',
        'w' => 0.734,
        'vw' => 24
    ],
    'apple-pay' => [
        'p' => 'M7.31 9.88q-.87 1.06-2 .93-.12-1.06.75-2 .75-.93 1.94-1 .13 1.13-.69 2.07zM8 11q1.63.13 2.44 1.31h-.06q-1.44.94-1.38 2.5 0 .32.06.63t.16.53.22.44.28.37l.28.28.28.2.25.12.16.09.06.03q-.19.88-.87 1.88Q8.75 21 7.8 21q-.44 0-1.06-.25t-1.13-.25-1.12.25-1 .25q-.94.06-2.13-1.69-.93-1.37-1.25-3.15t.44-3.47q.95-1.63 2.7-1.69.38 0 .81.16t.82.3.56.17 1.06-.35T8 11zm6.25-2.25h4.56q1.2 0 2.1.5t1.43 1.4.54 2.1q0 1.75-1.16 2.88t-2.97 1.12h-2.62v4.13h-1.88V8.74zm1.88 1.56v4.82h2.18q1.25 0 1.94-.63t.69-1.78-.69-1.78-1.94-.63h-2.19zM26.3 21q-1.31 0-2.15-.72t-.85-1.9q0-2.38 3.38-2.63l2.37-.13v-.68q0-1.5-1.81-1.5-1.56 0-1.81 1.25h-1.7q0-.82.48-1.44t1.28-.97 1.81-.34q1.63 0 2.6.78t.97 2.1v6.05h-1.75v-1.43h-.07Q28.25 21 26.31 21zm.5-1.44q.94 0 1.6-.56t.65-1.38v-.68l-2.12.12q-1.82.07-1.82 1.25 0 .38.22.66t.6.44.87.15zm6.38 4.63h-.69v-1.5q.31.06.56.06 1.2 0 1.57-1.25l.18-.56-3.18-8.88h1.93l2.25 7.2h.07l2.19-7.2H40l-3.31 9.32q-.56 1.62-1.31 2.21t-2.2.6z',
        'w' => 1.25,
        'vw' => 40
    ],
    'artstation' => 'M0 23.56h19.69L23.38 30H5.56q-2 0-2.87-1.75zm31.25-1.68q1.06 1.68-.13 3.68L28.57 30 12.5 2H18q1.94 0 2.87 1.69zM17.06 19H2.63L9.88 6.5z',
    'asymmetrik' => [
        'p' => 'M32.38 19.31q-1.32 1.38-3.07 2.5l4.57 6.88H27.5l-2.88-4.44q-1.56.63-3.37 1.13l2.06 3.3h-8.5l1.44-2.3q-2.25.25-4.31.18l-1.38 2.13H4.25l2.13-3.19q-4.32-1.75-4.25-5.19 0-2.56 2.62-5.19 3.69-3.8 10.06-5.8-1.81.62-3.87 1.93Q7.88 12.8 6 14.87q-2.69 2.88-2.31 5.47t3.5 3.97L19.06 6.44l9.44 14.18q1.19-.93 2.19-1.93 4.56-4.88 2.75-9.13T24.5 4.25q-6.69-1.06-13.81.87Q5.44 6-.01 8.75q5.63-3.13 12.54-4.56t12.9-.57q3.38.5 5.79 1.72t3.6 2.94T36 12.06q-.06 3.56-3.63 7.25zm-11.76 5.07q1.7-.5 3.38-1.25l-4.94-7.63-6.37 9.94q2.06.06 4.31-.31l2.06-3.32z',
        'w' => 1.125,
        'vw' => 36
    ],
    'atlassian' => 'M9.5 14.75q3.13 3.38 4.06 8.22t-.87 8.53q-.25.5-.82.5H.94q-.5 0-.78-.44t-.04-.94l7.82-15.68q.25-.5.72-.53t.84.34zM15.25.5q.38-.5.88-.5t.75.5l15 30.13q.25.5-.04.93t-.78.44H20.2q-.57 0-.88-.5-.56-1.12-1.78-3.44t-1.87-3.62-1.63-3.57-1.34-3.84-.66-3.81 0-4.1 1.07-4.15T15.25.5z',
    'audible' => [
        'p' => 'M40 12.5v3.38l-20 12.5-20-12.5V12.5L20 25zM27.87 17q-3.37-4.44-8.65-5.1t-9.53 2.85q-.31.25-.44.38h-.06q1.81-2.63 4.65-4.2T20 9.38q2.19 0 4.25.72t3.72 2 2.84 3.03zm-13.8 1.19q2.18-3.07 5.93-3.07t5.94 3.07L23.06 20q-2.12-3-5.37-3-1.94 0-3.63 1.19zM6.5 10.06q-.88.7-2.19 2.07 2.56-3.94 6.72-6.22T20 3.63t8.97 2.28 6.72 6.22L32.63 14l-.07-.06q-3.06-4.25-7.62-6.28t-9.5-1.4-8.94 3.8z',
        'w' => 1.25,
        'vw' => 40
    ],
    'autoprefixer' => [
        'p' => 'm19.88 1 10.25 30H25.3l-1.56-5.06h-7.5L14.69 31H9.8zm-2.5 21.38h5.25l-2.57-8.13h-.12zM40 25.3l-9.88.57-1.25-3.5 10.5 1zm-28.88-2.94-1.24 3.5L0 25.32l.63-1.94z',
        'w' => 1.25,
        'vw' => 40
    ],
    'avianex' => 'M28.31 2q1.82 0 2.88 1.28t.75 3.1l-3.57 19.25q-.3 1.8-1.84 3.09T23.2 30H3.69Q1.87 30 .8 28.72t-.75-3.1L3.63 6.38q.3-1.8 1.84-3.09T8.8 2h19.5zM24.7 23.69l.37-.5-5.93-7.44-.25-.44.84-.62.66-.5.46-.35.22-.15q.38-.32.32-.88t-.5-1.06-1-.66-.88.16q-.94.87-2.12 2l-.32-.25-6.43-7.06-.57.3-1.19 1.82 4.32 6.25 1.3 1.81q-1.31 1.38-2.06 2.2L8.3 16.36l-1 1 3.25 3 2.56 3.82 1.13-.82-1.38-4 2.63-1.68 7.19 6.87z',
    'aviato' => [
        'p' => 'M6.69 17.75 5.5 15.12H2.25l-1.19 2.63H0L3.87 9.5l3.88 8.25H6.69zm-2.82-6.19-1.18 2.7h2.43zM10.95 18 7.06 9.75h1.07l2.8 6 2.82-6h1.07zm5.06-.25v-8h.94v8H16zm13.06-7.25v7.25h-1.12V10.5h-2.56v-.75h6.25v.75h-2.57zM40 13.69q0 .87-.31 1.62t-.88 1.32-1.31.87-1.63.31-1.62-.3-1.31-.92-.88-1.3-.31-1.6.31-1.63.88-1.31q.3-.31 1.3-.75.76-.31 1.63-.31t1.63.31q1 .44 1.31.75.56.56.88 1.31T40 13.7zm-1 0q0-.69-.22-1.28t-.66-1.03-1-.7q-.62-.24-1.28-.24t-1.22.25-1 .69-.68 1.03-.25 1.28.25 1.25.68 1 1 .69 1.22.25 1.28-.25q.57-.25 1-.7t.66-1T39 13.7zm-5.88 6.06q.63.06.63.75 0 .31-.22.53t-.53.22-.53-.22-.22-.53q0-.69.63-.75v-.38l-5 .57v.31q.62.13.62.75t-.78.63-.78-.63.62-.75v-.31l-1.81.19v1.5h.38v.87h-.38v-.44h-.31v.44h-.32v-.87h.32v-1.44l-2.38.25q-.25.31-.53.53t-.47.28l-.19.06v.44h.32v.75h-.25v-.31h-.32v.31h-.25v-.75h.32v-.44q-.7-.12-1.2-.87l-2.37-.25v1.44h.38v.87h-.38v-.44h-.3v.44h-.38v-.87h.37v-1.5l-1.81-.2v.32q.62.13.62.75t-.75.63-.75-.63.57-.75v-.31l-5-.57v.38q.68.06.68.75 0 .31-.25.53t-.56.22-.53-.22-.22-.53q0-.69.63-.75v-.44l-5.57-.62 15.2.87q-.07-.25-.07-.44-.06-.25-.06-.43l-4.07-.5 4.07.12q0-.06.06-.06.06-.5.44-.88t.75-.5l.37-.12.06-1.63h-1.5L19 17.7h-1.13l3.94-8.2 3.88 8.2h-1.07l-1.18-2.57h-1.5l.06 1.63q1.44.37 1.62 1.5v.06l4-.12-4 .5q0 .18-.06.37 0 .25-.06.5l15.19-.87-5.57.62v.44zm-12.56-5.5h1.13l.06-2.5.13 2.5H23l-1.25-2.69z',
        'w' => 1.25,
        'vw' => 40
    ],
    'aws' => [
        'p' => 'M11.25 12.69q0 .62.16 1.12t.34.85.19.47q0 .25-.25.37l-.81.56q-.13.13-.32.13h-.19l-.4-.4-.72-1.16q-.38.5-1.03.9t-1.4.66-1.38.25h-.07q-1.18.06-2.47-.75t-1.15-2.75Q1.62 11.19 3 10.13t3.12-.94q1.32 0 2.94.37v-1q.19-1.25-.6-2.06t-2.15-.63q-1.31 0-2.87.63-.38.12-.7.12-.3-.06-.3-.65t.12-.85q.31-.3 1.56-.71T6.7 4h.43q1.7 0 3.07 1.06Q11.3 6.31 11.3 8q0 .19-.06.38v4.3zm-5.38 2q1.38 0 2.13-.6t.94-1.28q.12-.5.12-1.68-1.25-.32-2.44-.32-.93-.06-1.8.4T4 12.82q-.06.81.47 1.38t1.4.5zm10.7 1.44q-.63.06-.82-.63L12.62 5.19q-.06-.25-.06-.5-.06 0-.06-.07 0-.25.25-.25h1.44q.62-.06.75.63l2.25 8.81L19.25 5q.12-.69.81-.63h1.07q.68-.06.8.63L24 13.94 26.31 5q.13-.69.82-.63h1.31l.16.13.1.37q-.07.13-3.32 10.63-.13.75-.82.69h-1.12q-.69.06-.81-.7l-2.07-8.55-2.06 8.56q-.12.75-.81.62h-1.13zm17.06.37q-2.07 0-3.57-.75-.5-.25-.5-.75v-.69q0-.56.57-.37 1.18.5 1.8.62 2.2.44 3.57-.31.56-.31.66-1.03t-.35-1.16q-.12-.12-.25-.19l-.31-.15-.28-.16-.4-.15-.5-.16-.7-.22-.87-.28q-.44-.06-.9-.31t-1.16-1.07-.69-1.87q-.06-1.38 1.16-2.44T34.06 4q.5 0 1.16.1t1.37.3.97.6q.13.19.13.44v.62q0 .44-.32.44-.12-.06-.56-.22t-1.1-.31-1.4-.16q-.25 0-.6.03t-.8.16-.76.47-.28.9q0 1.2 1.88 1.7.25.05.87.24t.78.22.57.19.53.22.4.22.44.34.35.38q1.25 1.56.3 3.5-1.12 2.18-4.37 2.12zm2.56 6.56q-3.19 2.38-7.6 3.66T20 28h-.19q-4.69 0-10.44-2.22T.2 20.44q-.32-.25-.13-.5t.56-.07q3.82 2.2 9.6 3.72t10.15 1.53h.07q3.25 0 7.68-.9t7.38-2.16q.63-.25.9.19t-.21.81zm1.81-2.1q-.56-.71-5.13-.15-.3.07-.34-.15t.22-.47q1.19-.75 2.81-1.03t2.82-.1 1.5.5q.43.57-.25 2.97t-2.25 3.72q-.25.19-.47.1t-.1-.35q1.75-4.31 1.2-5.03z',
        'w' => 1.25,
        'vw' => 40
    ],
    'bandcamp' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm3 20.38 5.31-9.75H12.5l-5.31 9.74H18.5z',
    'battle-net' => 'M28.04 14.1c1.68.01 2.22-.46 2.43-.77.78-1.02-.44-2.98-3.3-4.46 1.1-2.1 1.88-3.98 2.27-5.33.2-.74.07-1.19.03-1.27-.11.66-1 3.03-3.02 6.26a18.85 18.85 0 0 0-5.86-1.5c-.55-1.05-2.18-3.98-3.78-5.55C15.76.45 14.91.07 14.26.02c-.87-.1-1.42.36-1.71.68-1.08 1.16-1.53 3.05-1.57 5.26-.45-.78-1.07-1.54-1.78-1.62-1.3-.22-2.4 1.82-2.26 5.07-2.4.09-4.44.36-5.81.7-.62.16-1.02.46-1.11.62.06-.03 1.4-.58 6.97-.58.33 3.31 1.86 6.36 1.63 5.82-.61.97-2.4 3.9-2.96 6.1-.37 1.44-.27 2.36 0 2.96.36.8 1.04 1.04 1.46 1.14 1.56.35 3.46-.23 5.42-1.33-.47.8-.87 1.79-.57 2.46.46 1.23 2.78 1.17 5.53-.59a29.39 29.39 0 0 0 3.48 4.64c.16.17.34.32.55.44.32.2.53.21.53.21-.51-.42-2.12-2.38-3.9-5.73 1.38-1 2.85-2.43 4.21-4.34 7.68.3 8.96-1.54 9.25-1.97.92-1.25.22-3.6-3.58-5.86Zm-4.87 6.64c1.49-2.36 1.9-4.24 1.84-5.77 1.74 1.1 2.95 2.35 3.07 3.67.07.81-.5 1.82-4.9 2.1Zm-9.62 3.49c.62-.39 1.23-.82 1.83-1.28.42.83.85 1.63 1.29 2.39-2.54 1.36-4.3.8-3.12-1.11ZM27 13.52c-.64-.33-1.32-.64-2.02-.94.5-.75.97-1.52 1.42-2.3 2.44 1.5 2.87 3.32.6 3.24Zm-9.54 11.36c-.34-.72-.69-1.47-1.03-2.28 2.7.08 3.9-1.17 3.95-1.28 0 0-1.56.98-3.9.77a44.99 44.99 0 0 0 6.87-7.2c-.02-.02-.1-.18-1.03-.75A35.16 35.16 0 0 1 14 21.4a8.72 8.72 0 0 1-3.48-3.84 18.78 18.78 0 0 1-1.98-7.26c.77.06 1.58.14 2.43.24-1.39 2.3-.9 3.94-.84 4.02 0 0-.06-1.83 1.26-3.73a43.5 43.5 0 0 0 2.8 9.55c.05-.02.1.06 1.16-.51a35.06 35.06 0 0 1-2.13-10.84 8.7 8.7 0 0 1 5.1-1.11c2.8.16 5.4.95 7.28 1.92-.45.65-.93 1.33-1.46 2.03-1.3-2.36-2.95-2.75-3.06-2.73.01 0 1.62.87 2.6 2.95a44.75 44.75 0 0 0-9.67-2.37c0 .04-.11.12-.14 1.28 4.87.9 8.54 2.49 10.46 3.57a8.68 8.68 0 0 1-1.6 4.98 18.76 18.76 0 0 1-5.3 5.34Zm1.71-17.94c-2.79-.1-4.6.46-5.92 1.29.13-3.27 1.33-4.77 2.4-4.7 1.05-.27 3.42 3.26 3.52 3.4Zm-8.16.2c.03.75.1 1.53.2 2.32a44.2 44.2 0 0 0-2.65-.08c0 .2-.01-3.18 1.54-3.1.36.07.67.43.9.85ZM9.24 17.25c1.3 2.48 2.7 3.78 4.08 4.52-2.93 1.54-4.85 1.25-5.3.28-.02-.02-.7-.96 1.22-4.8Zm13.13 4.67',
    'behance' => [
        'p' => 'M14.5 14.81q4.06 1.13 4.06 5.63 0 3.19-2.25 4.81t-5.56 1.63H0V4.68h10.44q7.06 0 7.06 5.5 0 3.2-3 4.63zM4.87 8.5v5.19h4.94q2.94 0 2.94-2.69 0-2.5-3.31-2.5H4.88zm5.2 14.63q3.62 0 3.62-2.94 0-3.2-3.5-3.2H4.88v6.13h5.18zM32.5 8.05h-9V5.87h9v2.2zm3.5 11q0 .32-.06.82H24.37q0 2 1.03 3.06T28.31 24q.69 0 1.4-.25t1.32-.72.78-1.15h3.88q-1.75 5.37-7.5 5.37-3.75 0-5.97-2.31T20 18.88q0-3.7 2.28-6.1t5.9-2.4 5.72 2.5 2.1 6.18zm-11.63-1.93h7.2q-.32-3.44-3.5-3.44-1.57 0-2.57.94t-1.13 2.5z',
        'w' => 1.125,
        'vw' => 36
    ],
    'behance-square' => 'M11.63 18.31q0 1.57-1.94 1.57H6.87v-3.25h2.88q1.87 0 1.87 1.68zm-.44-5.12q0 1.44-1.63 1.44H6.87V11.8h2.5q1.82 0 1.82 1.38zm8.25 1.43q1.75 0 1.94 1.88h-3.94q.06-.88.6-1.38t1.4-.5zM28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22q1.25 0 2.13.88T28 5zm-11 6.56h4.88v-1.19H17v1.2zm-2.69 6.88q0-2.57-2.19-3.19 1.63-.81 1.63-2.31 0-3-3.88-3H4.25v12h5.81q1.82 0 3.03-.88t1.22-2.62zm9.44-.94q0-2-1.13-3.34t-3.09-1.35-3.22 1.32-1.25 3.3q0 1 .31 1.85t.88 1.44 1.4.94 1.85.34q3.13 0 4.13-2.94H21.5q-.19.5-.75.82t-1.13.31q-2.12 0-2.12-2.19h6.25v-.5z',
    'bilibili' => [
        'p' => 'M488.6 104.1C505.3 122.2 513 143.8 511.9 169.8V372.2C511.5 398.6 502.7 420.3 485.4 437.3C468.2 454.3 446.3 463.2 419.9 464H92.02C65.57 463.2 43.81 454.2 26.74 436.8C9.682 419.4 .7667 396.5 0 368.2V169.8C.7667 143.8 9.682 122.2 26.74 104.1C43.81 87.75 65.57 78.77 92.02 78H121.4L96.05 52.19C90.3 46.46 87.42 39.19 87.42 30.4C87.42 21.6 90.3 14.34 96.05 8.603C101.8 2.868 109.1 0 117.9 0C126.7 0 134 2.868 139.8 8.603L213.1 78H301.1L375.6 8.603C381.7 2.868 389.2 0 398 0C406.8 0 414.1 2.868 419.9 8.603C425.6 14.34 428.5 21.6 428.5 30.4C428.5 39.19 425.6 46.46 419.9 52.19L394.6 78L423.9 78C450.3 78.77 471.9 87.75 488.6 104.1H488.6zM449.8 173.8C449.4 164.2 446.1 156.4 439.1 150.3C433.9 144.2 425.1 140.9 416.4 140.5H96.05C86.46 140.9 78.6 144.2 72.47 150.3C66.33 156.4 63.07 164.2 62.69 173.8V368.2C62.69 377.4 65.95 385.2 72.47 391.7C78.99 398.2 86.85 401.5 96.05 401.5H416.4C425.6 401.5 433.4 398.2 439.7 391.7C446 385.2 449.4 377.4 449.8 368.2L449.8 173.8zM185.5 216.5C191.8 222.8 195.2 230.6 195.6 239.7V273C195.2 282.2 191.9 289.9 185.8 296.2C179.6 302.5 171.8 305.7 162.2 305.7C152.6 305.7 144.7 302.5 138.6 296.2C132.5 289.9 129.2 282.2 128.8 273V239.7C129.2 230.6 132.6 222.8 138.9 216.5C145.2 210.2 152.1 206.9 162.2 206.5C171.4 206.9 179.2 210.2 185.5 216.5H185.5zM377 216.5C383.3 222.8 386.7 230.6 387.1 239.7V273C386.7 282.2 383.4 289.9 377.3 296.2C371.2 302.5 363.3 305.7 353.7 305.7C344.1 305.7 336.3 302.5 330.1 296.2C323.1 289.9 320.7 282.2 320.4 273V239.7C320.7 230.6 324.1 222.8 330.4 216.5C336.7 210.2 344.5 206.9 353.7 206.5C362.9 206.9 370.7 210.2 377 216.5H377z',
        'vw' => 512,
        'vh' => 512
    ],
    'bimobject' => 'M26 2q.81 0 1.4.6T28 4v24q0 .81-.6 1.4T26 30H2q-.81 0-1.4-.6T0 28V4q0-.81.6-1.4T2 2h24zm-4 16.06v-2.19q0-2.62-1.25-3.87t-5.25-1.25h-1.06q-4 0-4.38 2.44H10V8.5H6V23h4v-2.19h.06q.81 2.44 4.38 2.44h1.06q4 0 5.25-1.25T22 18.06zm-4-1.81v1.44q0 1.19-.44 1.65t-1.94.47h-2.87q-.94 0-1.5-.15t-.9-.66-.35-1.37v-1.25q0-1.25.66-1.72t2.09-.47h2.87q1.5 0 1.94.47t.44 1.6z',
    'bitbucket' => 'M1 2h29.38q.18.06.34.16t.28.25.19.34 0 .38l-4.25 26.06q-.13.87-1 .81H5.56q-.25 0-.47-.06t-.37-.22-.28-.38-.19-.47L0 3.12v-.18q0-.38.31-.66t.7-.28zm17.88 18.56 1.56-9.19h-9.82l1.75 9.2h6.5z',
    'bitcoin' => 'M31.5 16q0 6.44-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5t10.97 4.53T31.5 16zm-8.88-2.19q.44-2.62-3.37-3.94l.69-2.8-1.69-.38-.69 2.68q-.5-.12-1.37-.3l.68-2.76-1.68-.44-.7 2.82q-.74-.13-1.12-.25l-2.3-.57-.45 1.82h.07l.21.03.32.06.31.1.22.06h.06q.88.25.82 1l-.82 3.19q.13.06.2.06h-.07l-.13-.06-1.12 4.5q-.19.5-.75.37l-1.25-.31-.82 1.94 2.2.56q.12.06.3.1t.32.06.28.09.28.06l-.69 2.82 1.7.43L13 22q.63.13 1.31.32l-.68 2.81 1.68.37.75-2.8q2.13.55 3.72.05t2.28-2.3q.94-2.82-1.56-4.13 1.81-.44 2.13-2.5zm-3.87 5.44q-.06.31-.25.56t-.47.38-.6.22-.71.1-.72-.04-.75-.12-.66-.16-.62-.16-.47-.1l.94-3.74q.06 0 .62.12t.88.22.87.28.88.41.65.5.44.69-.03.84zm.56-5.5q-.12.31-.25.5t-.31.34-.44.22-.5.1-.56 0-.53-.03-.53-.1-.5-.12-.47-.1-.35-.12l.88-3.38q.06 0 .44.1t.6.15.62.2.62.24.6.34.46.44.25.57-.03.65z',
    'bity' => 'M4.88 4.19Q9.3 0 15.55 0t10.82 4.44q.37.44.22.97t-.66.65-1-.31Q21 1.94 15.56 1.94t-9.3 3.62q-.45.44-.98.25t-.65-.65.25-.97zm6.18 26.19q-6-1.82-9.06-7.22T.5 11.75q.13-.63.66-.75t.93.25.29.94Q1 17.44 3.66 22.16t7.9 6.34q.63.19.75.72t-.28.94-.97.22zm9.25-.13q-.56.19-.93-.19t-.29-.94.72-.75q4.94-1.62 7.5-6.18t1.44-9.63q-.19-.62.25-1t.97-.25.66.75q1.3 5.81-1.63 11.06t-8.69 7.13zm-3.94-4.94q0 .38-.25.66t-.62.28-.66-.28-.28-.66v-4.5q0-.37.28-.65t.66-.28.66.28.28.65zm4.5-16.81q0-.63.5-.88t1 0 .5.88v4.44q0 3-2.15 4.34t-5.22 1.35-5.22-1.35-2.16-4.34V8.5q0-.63.5-.88t1 0 .5.88q-.06 2.06 0 3.6t.5 2.56 1.6 1.56 3.28.53q2.06 0 3.22-.5t1.62-1.5.53-2.56 0-3.69z',
    'black-tie' => 'M0 2h28v28H0V2zm19.75 20.31-4-11.5 4-5.37H8.25l4 5.37-4 11.5L14 27.88z',
    'blackberry' => 'M10.38 7.31q0 1.25-1.07 2.16t-3.44.9H1.44l1.3-5.56H7q1.81 0 2.6.78t.77 1.72zm7.87-2.5q1.81 0 2.6.78t.77 1.72q0 1.25-1.03 2.16t-3.34.9h-4.38L14 4.82h4.25zM5.56 13q1.75 0 2.57.78t.8 1.72q0 3.06-4.55 3.06H0L1.31 13h4.25zm11.25 0q1.75 0 2.57.78t.8 1.72q0 3.06-4.43 3.06h-4.37L12.56 13h4.25zm11.81-3.38q1.82 0 2.6.79t.78 1.71q0 3.07-4.38 3.07h-4.37l1.13-5.56h4.25zm-1.75 8.63q1.25 0 2.04.44t1.06.97.28 1.1q0 3.05-4.38 3.05H21.5l1.13-5.56h4.25zm-11.8 3.38q1.74 0 2.56.75t.75 1.75q0 3.06-4.38 3.06H9.63l1.18-5.57h4.25z',
    'blogger' => [
        'p' => 'M10.13 12.25q.12-.13.21-.19t.38-.1.6-.02h1.12q1.75 0 2 .12.5.25.5.88 0 .5-.44.81-.19.13-1.97.13t-1.97-.07q-.5-.12-.62-.65t.18-.91zM14 18.13q2.56 0 3.03.06t.72.25q.38.31.31.81t-.5.75l-.56.06-3 .07q-3.25 0-3.5-.13-.5-.19-.6-.72t.35-.84q.25-.31 3.75-.31zm13.94 8.18q-.13.88-.53 1.6t-1.1 1.25-1.56.72q-.5.12-10.81.12t-10.75-.12q-.75-.2-1.38-.63-.69-.44-1.06-1-.44-.62-.63-1.37Q0 26.44 0 16 0 5.63.12 5.2q.32-1.19 1.16-2t2.03-1.13Q3.62 2 13.91 2t10.65.06q2.06.44 3 2.32.31.68.38 1.68t.06 9.57q0 10.12-.06 10.69zM22.63 14.7q-.13-.5-.5-.69-.13-.06-.97-.1t-1.1-.21q-.5-.25-.5-1.13 0-1.94-1.56-3.56-1.19-1.13-2.56-1.56-.32-.07-2.44-.13-3.13 0-4.19.44-2.69 1-3.37 3.87-.07.38-.13 4 0 4.44.25 5.32.44 1.37 1.6 2.37t2.46 1.25q.44.07 4.13.1t4.25-.03q1.81-.25 3.12-1.63 1-1 1.38-2.44.12-.44.15-3.03t-.03-2.84z',
        'w' => 0.873,
        'vw' => 28
    ],
    'blogger-b' => 'M27.94 13.94q.06.31.03 4.5t-.22 4.94q-.63 2.3-2.19 3.87-.75.81-1.53 1.35t-1.69.84-1.9.44q-.88.12-6.85.12t-6.65-.12q-2.13-.44-3.97-2.03T.44 23.94Q.12 23 .06 21.6T0 15.44q.06-5.87.19-6.5.56-2.19 1.97-3.84t3.4-2.47q1.13-.44 2.35-.53t4.46-.1q3.5.07 3.94.19 2.32.69 4.13 2.56 2.5 2.57 2.56 5.7 0 1.37.75 1.8.44.32 1.81.35t1.57.15q.62.32.81 1.2zm-20.19-4q-.56.56-.31 1.44t1 1.12q.37.06 3.19.06t3.12-.18q.75-.5.75-1.38 0-.94-.81-1.37-.38-.2-3.25-.2-2.25 0-2.78.07t-.91.44zm12 12.5q.69-.38.78-1.16T20.06 20q-.37-.37-1.12-.44T14 19.5q-4.13 0-4.94.06t-1.19.32q-.62.62-.5 1.43.13.88.94 1.2.44.18 5.69.12l4.81-.07z',
    'bluetooth' => 'm18.31 10.69-2.69 2.69L15.56 8zm-2.75 13.75.07-5.38 2.62 2.7zM26 16.19q0 3.19-.34 5.65t-1.1 4.2-1.69 2.9-2.3 1.84-2.82.94-3.31.28q-2.88 0-4.97-.69t-3.88-2.37-2.69-4.88-.9-7.84.9-7.94 2.63-5.03T9.37.75 14.31 0q2.75 0 4.81.75t3.66 2.53 2.4 5 .82 7.9zm-9.88 0 4.94-5.5-7.81-8.44v11.06L8.62 8.75l-1.68 1.63 5.81 5.87L6.94 22l1.68 1.69 4.63-4.57.13 10.63 7.93-8z',
    'bluetooth-b' => [
        'p' => 'm12.25 16.25 6.13 6.44L9.13 32l-.2-12.37L3.57 25l-1.94-2 6.75-6.75L1.62 9.5l1.94-2 5.38 5.38V0l9.12 9.81zm2.56-6.44L11.7 6.7v6.25zm-3.12 9.75v6.25l3.12-3.12z',
        'w' => 0.625,
        'vw' => 20
    ],
    'bootstrap' => [
        'p' => 'M18.27 19.5c0 2.64-2.48 2.59-2.75 2.59h-5.05v-5.11h5.05c2.66 0 2.75 2 2.75 2.52Zm-3.14-4.57c.05 0 2.4.06 2.4-2.27 0-.97-.21-2.25-2.4-2.25h-4.66v4.52ZM28 6.67v18.66c0 2.58-2.1 4.66-4.67 4.67H4.67A4.68 4.68 0 0 1 0 25.33V6.67C0 4.09 2.1 2 4.67 2h18.66C25.91 2 28 4.1 28 6.67Zm-6.87 13.2c0-1.35-.42-3.65-3.07-4.21v-.05a3.57 3.57 0 0 0 2.33-3.48c0-.43.13-4.04-4.84-4.04H7.6v16.33c8.02 0 8.74.1 10.22-.36.9-.27 3.3-1.12 3.3-4.2Zm0 0',
        'vw' => 28
    ],
    'bots' => [
        'p' => 'M86.344,197.834a51.767,51.767,0,0,0-41.57,20.058V156.018a8.19,8.19,0,0,0-8.19-8.19H8.19A8.19,8.19,0,0,0,0,156.018V333.551a8.189,8.189,0,0,0,8.19,8.189H36.584a8.189,8.189,0,0,0,8.19-8.189v-8.088c11.628,13.373,25.874,19.769,41.573,19.769,34.6,0,61.922-26.164,61.922-73.843C148.266,225.452,121.229,197.834,86.344,197.834ZM71.516,305.691c-9.593,0-21.221-4.942-26.745-12.5V250.164c5.528-7.558,17.152-12.791,26.745-12.791,17.734,0,31.107,13.082,31.107,34.013C102.623,292.609,89.25,305.691,71.516,305.691Zm156.372-59.032a17.4,17.4,0,1,0,17.4,17.4A17.4,17.4,0,0,0,227.888,246.659ZM273.956,156.7V112.039a13.308,13.308,0,1,0-10.237,0V156.7a107.49,107.49,0,1,0,10.237,0Zm85.993,107.367c0,30.531-40.792,55.281-91.112,55.281s-91.111-24.75-91.111-55.281,40.792-55.281,91.111-55.281S359.949,233.532,359.949,264.062Zm-50.163,17.4a17.4,17.4,0,1,0-17.4-17.4h0A17.4,17.4,0,0,0,309.786,281.466ZM580.7,250.455c-14.828-2.617-22.387-3.78-22.387-9.885,0-5.523,7.268-9.884,17.735-9.884a65.56,65.56,0,0,1,34.484,10.1,8.171,8.171,0,0,0,11.288-2.468c.07-.11.138-.221.2-.333l8.611-14.886a8.2,8.2,0,0,0-2.867-11.123,99.863,99.863,0,0,0-52.014-14.138c-38.956,0-60.179,21.514-60.179,46.225,0,36.342,33.725,41.864,57.563,45.642,13.373,2.326,24.13,4.361,24.13,11.048,0,6.4-5.523,10.757-18.9,10.757-13.552,0-30.994-6.222-42.623-13.579a8.206,8.206,0,0,0-11.335,2.491c-.035.054-.069.108-.1.164l-10.2,16.891a8.222,8.222,0,0,0,2.491,11.066c15.224,10.3,37.663,16.692,59.441,16.692,40.409,0,63.957-19.769,63.957-46.515C640,260.63,604.537,254.816,580.7,250.455Zm-95.928,60.787a8.211,8.211,0,0,0-9.521-5.938,23.168,23.168,0,0,1-4.155.387c-7.849,0-12.5-6.106-12.5-14.245V240.28h20.349a8.143,8.143,0,0,0,8.141-8.143V209.466a8.143,8.143,0,0,0-8.141-8.143H458.594V171.091a8.143,8.143,0,0,0-8.143-8.143H422.257a8.143,8.143,0,0,0-8.143,8.143h0v30.232H399a8.143,8.143,0,0,0-8.143,8.143h0v22.671A8.143,8.143,0,0,0,399,240.28h15.115v63.667c0,27.037,15.408,41.282,43.9,41.282,12.183,0,21.383-2.2,27.6-5.446a8.161,8.161,0,0,0,4.145-9.278Z',
        'vw' => 640,
        'vh' => 512
    ],
    'btc' => [
        'p' => 'M19.38 15.19q4.8 1.12 4.37 6.25-.38 3.31-2.6 4.69t-6.34 1.62V32h-3.06v-4.19H9.37v4.2H6.31v-4.26H.25l.62-3.62h.63q1.06.06 1.47.03t.78-.28.44-.75V9.56q-.2-1-.91-1.22T.75 8.2h-.5V4.94q5.25.06 6.06 0V0h3.06v4.88q1.57-.07 2.38-.07V0h3.06v4.94q1.2.12 2.2.34t1.9.6 1.53.93 1.03 1.4.47 1.85q.37 3.63-2.57 5.13zM9.44 8.44v6l.65.03q.6.03.91 0t.97-.03 1.1-.1 1.02-.21.97-.38.72-.56.53-.75.2-1q0-.75-.29-1.28t-.84-.88-1.1-.53-1.37-.25-1.32-.1-1.28 0-.87.04zm0 15.69.78.03q.72.03 1.1 0t1.15-.07 1.28-.12 1.19-.22 1.12-.4.85-.6.62-.81.22-1.13q0-.81-.34-1.4t-1-.94-1.25-.6-1.63-.31-1.53-.1-1.5 0-1.06.04v6.62z',
        'w' => 0.748,
        'vw' => 24
    ],
    'buffer' => [
        'p' => 'm26.74 23.8-12.28 6.1c-.3.13-.62.13-.92 0L1.26 23.8c-.25-.13-.25-.34 0-.46l2.94-1.46c.3-.13.63-.13.92 0l8.42 4.18c.3.13.63.13.92 0l8.42-4.18c.3-.13.62-.13.92 0l2.94 1.46c.25.12.25.33 0 .45Zm0-8.54-2.94-1.47c-.3-.12-.62-.12-.92 0L14.46 18c-.3.12-.62.12-.92 0l-8.42-4.2c-.3-.12-.62-.12-.92 0l-2.94 1.47c-.25.12-.25.33 0 .45l12.28 6.12c.3.12.63.12.92 0l12.28-6.12c.25-.12.25-.33 0-.45ZM1.26 8.16l12.28 5.63c.3.12.62.12.92 0l12.28-5.64c.25-.11.25-.3 0-.42L14.46 2.09c-.3-.12-.62-.12-.92 0L1.26 7.73c-.25.11-.25.3 0 .42Zm0 0',
        'vw' => 28
    ],
    'buromobelexperte' => 'M0 2h8v8H0V2zm7.5 7.5v-7h-7v7h7zM10 2h8v8h-8V2zm7.5 7.5v-7h-7v7h7zM20 2h8v8h-8V2zm7.5 7.5v-7h-7v7h7zM0 12h8v8H0v-8zm7.5 7.5v-7h-7v7h7zM10 12h8v8h-8v-8zm7.5 7.5v-7h-7v7h7zM20 12h8v8h-8v-8zm7.5 7.5v-7h-7v7h7zM0 22h8v8H0v-8zm7.5 7.5v-7h-7v7h7zM10 22h8v8h-8v-8zm7.5 7.5v-7h-7v7h7zM20 22h8v8h-8v-8z',
    'buy-n-large' => [
        'p' => 'M18 2C8.33 2 .49 8.27.49 16S8.33 30 18 30c9.67 0 17.51-6.27 17.51-14S27.67 2 18 2Zm-5.34 22.32-8.65.09L8.84 6.24h8.34c3.94 0 5.3 1.79 4.87 4.55a4 4 0 0 1-.1.43 4.76 4.76 0 0 0-5.87 4.6c0 2.33 1.7 4.26 3.94 4.66-1.14 2.32-3.5 3.8-7.36 3.84Zm9.71-11.37h2L23 18.59h-2.21l-.7-2.23-.49 2.23h-2.37l1.67-5.64h1.96l.94 2.3Zm9.12 11.37H19.18l.98-3.82a4.76 4.76 0 0 0 5.51-4.68 4.73 4.73 0 0 0-3.14-4.44l1.33-5.15h5.78l-3.32 12.84h6.5Zm-18.26-7.48h-1.54l-.87 3.53h1.55c1 0 2-.2 2.37-1.67.35-1.4-.5-1.86-1.51-1.86Zm1.33-6.21h-1.33l-.73 2.98h1.34a2.7 2.7 0 0 0 2.45-1.89c.29-1.18-.6-1.1-1.73-1.1Zm0 0',
        'vw' => 36
    ],
    'buysellads' => 'm14 9.44 2.69 10H11.3zM28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22q1.25 0 2.13.88T28 5zm-4.06 20.31L18 6.7h-8L4.06 25.3h5.69l7-5.68 1.5 5.68h5.69z',
    'canadian-maple-leaf' => [
        'p' => 'M22.56 22q-.37.31-.4.44t.1.94l.62 2.18q-5.94-1.25-6.57-1.37-1.18-.13-1.18.75L15.5 32h-1.87l.03-.37q.03-.25.06-.9t.06-1.29.07-1.44.06-1.4.06-1.07.03-.6q0-.87-1.12-.74l-6.63 1.37q.32-1.25.63-2.18.12-.82.1-.94T6.55 22L0 16.19l1.13-.44q.62-.31.25-1.13L.13 10.44q3 .62 3.62.8.5.13.81-.5l.94-2q3.25 3.76 3.44 3.88.44.32.81.1T10 12L8.32 3.87q1.87 1.13 2.5 1.44.68.44 1.06-.31l2.69-5q2.37 4.5 2.69 5 .37.75 1.06.31l2.5-1.44q-1.69 7.5-1.69 8.13-.12.5.25.72t.81-.1l3.44-3.87q.81 1.69.94 2 .31.62.81.5l3.63-.81-1.25 4.18q-.38.82.25 1.13l1.12.44Q22.7 21.8 22.57 22z',
        'vw' => 29
    ],
    'cc-amazon-pay' => [
        'p' => 'M7.81 12.63V8.18q0-.13.07-.2Q9 7.24 10.25 7.3q1.31.13 1.75 1.38.25.68.25 1.75 0 .93-.19 1.56-.5 1.62-2.25 1.5-.94 0-1.94-.69-.06-.12-.06-.19zm9.69.87q-.5-.19-.66-.69t-.03-1q.2-.62.88-.81.56-.19 1.31-.13.25 0 .69.1t.56.1.13.12v1.5l-.07.12q-.93.69-1.93.81-.5.07-.88-.12zM36 5v22q0 1.25-.88 2.13T33 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h30q1.25 0 2.13.88T36 5zM23 6.5q.06.19.13.31.25.57 1.37 3.35t1.75 4.28q.13.31 0 .62-.06.2-.22.57t-.28.62q-.25.5-.81.69-.38.06-.82.06l-.15-.03-.22-.03q-.25 0-.25.25v.5q0 .44.37.5.57.06.88.06 1.37.07 2.06-1.12.32-.5.5-.94 1.25-3.13 3.69-9.44l.13-.25q0-.25-.25-.25h-1q-.32 0-.44.25 0 .06-.03.13t-.03.12L27.56 12l-.44 1.13V13q-.37-.94-1.03-2.81t-1.03-2.81q-.12-.38-.31-.88-.13-.25-.38-.25H23.2q-.2 0-.2.25zm-7.88 6.38q.2 1.12 1.13 1.68.88.44 2.03.25t2.1-.93l.06-.07q.12.44.12.56.07.26.25.26h.88q.25 0 .25-.26V8.7q0-.32-.07-.7-.12-1.05-1-1.55-.43-.25-1.12-.38l-.25-.03Q19.31 6 19.19 6h-.75q-.88.06-1.57.25-.12.06-.87.31-.31.07-.31.38v.56q0 .31.37.25.88-.25 1.19-.31 1.12-.2 2-.07.56.07.87.44.25.32.25.82V10h-.06q-.87-.19-1.69-.25-1-.06-1.75.19-1.06.44-1.5 1.31-.37.69-.25 1.63zM6.2 6.67V17.8q0 .25.25.25h1.12q.25 0 .25-.25v-3.94l.13.07q1.37 1.19 3.19.81 1.5-.31 2.25-1.81.56-1 .56-2.32Q14 9.12 13.5 8q-.69-1.63-2.25-1.94-.12 0-.37-.06h-.63l-.19.06q-1.06.13-2 .75L7.87 7q-.12.06-.18.13v-.07q0-.12-.03-.25t-.03-.19q-.07-.37-.44-.37h-.57q-.43 0-.43.44zm24.68 14.5q-.18-.06-.5.07-4.06 1.69-8.56 2.25-1.37.13-2.69.19-2.5.06-5-.25-1.68-.25-3.3-.63-3.57-.87-6.7-2.43L2.5 19.5q-.32-.19-.5.13v.12l.25.31q3.37 3 7.62 4.57 2 .68 4.2 1.06 1.3.19 2.62.31h1.93q1.38-.06 2.82-.25 2-.31 3.93-.94 2.94-.93 5.2-2.5.37-.25.5-.37.18-.19.12-.44t-.32-.31zM34 19.45q-.06-.25-.5-.44-.44-.13-1.06-.19-1.07-.06-2.07.13-1.25.18-2.12.81-.13.06-.19.19v.19l.19.12h.12l1.2-.19q1.12-.06 1.8 0 .32 0 .75.13.38.06.41.53t-.16 1.22q-.24.94-.87 2.37l-.06.13q0 .12.06.19t.19.06l.25-.13q.5-.5.62-.68.94-1.2 1.32-2.94.06-.19.12-.57v-.93z',
        'w' => 1.125,
        'vw' => 36
    ],
    'cc-amex' => [
        'p' => 'M20.31 10.5q0 .88-.75 1.13.7.24.7 1.18v.82h-1v-.82l-.07-.28-.16-.25-.28-.15-.44-.07H17.2v1.57h-1V9.3h2.44q1.68 0 1.68 1.2zm-1.81.69q.69 0 .69-.5 0-.44-.69-.44h-1.31v.94h1.31zm-1.06 5.62q1.75 0 1.75 1.38 0 .37-.1.65t-.3.44-.45.25-.56.13-.53.03h-1.13v1.44h-2.06l-1.25-1.44-1.31 1.44H7.37V16.8h4.2l1.24 1.38 1.31-1.38h3.32zm-6.5 3.38 1.19-1.25L11 17.69H8.44v.75h2.25v.94H8.44v.8h2.5zm4.12.56v-3.5l-1.56 1.69zm2.38-2q.62 0 .62-.56 0-.5-.62-.5h-1.32v1.06h1.32zm-2-8.5H13V11h2.38v.88H13v.87h2.44v.88H12V9.3h3.44v.94zm8.37 7.69q0 .87-.68 1.19.62.25.62 1.12v.88h-1v-.75q0-.32-.1-.5t-.27-.25-.57-.07h-1.06l-.06 1.57h-1V16.8h2.43q1.7 0 1.7 1.13zm-1.81.68q.69 0 .69-.5 0-.43-.69-.43h-1.25v.93H22zm-10.81-5h-1v-3.37l-1.5 3.38h-.94l-1.5-3.38v3.38H4.12l-.37-.94H1.56l-.37.93H.06l1.88-4.3h1.5l1.75 4.12V9.3h1.68l1.38 2.94L9.5 9.31h1.69v4.31zm-9.25-1.8h1.44l-.75-1.76zm29.87 9.93h2.2l.77-.03.72-.13.5-.21V27q0 1.25-.87 2.13T33 30H3q-1.25 0-2.12-.88T0 27V14.25h1.7q.12-.38.37-.94h1.2q.24.56.43.94H7v-.69l.32.69h1.8l.13-.22q.06-.09.1-.22t.1-.25v.69h8.5v-1.56h.56q0 .69.06 1.56h4.12v-.56q.7.56 1.88.56h1.69q.12-.37.37-.94h1.19q.19.44.19.5.06 0 .1.13t.15.31h3.25v-1.37q.43.75.6.96t.27.41h2.56V8.5h-2.5v1.19q-.3-.56-.68-1.19h-2.7v1.31q-.18-.5-.55-1.31h-3.88q-1.19 0-1.94.56V8.5h-2.5v.31l-.22-.1-.25-.05q-.15-.02-.25-.07t-.3-.06-.38-.03H9q-.32.75-.75 1.87-.13-.18-.47-.93t-.41-.94H4.5v1.3l-.4-.84q-.1-.22-.16-.47h-2.5q-.12.25-.62 1.44T0 11.87V5q0-1.25.87-2.13T3.01 2h30q1.25 0 2.12.87T36.01 5v10.94h-.63q-2.19 0-2.75.31v-.31h-2.19l-.62.03-.53.06-.38.1-.34.18v-.37h-4.94v.31l-.25-.1-.28-.09q-.16-.06-.38-.06t-.4-.03-.6-.03h-2.78v.37q-.62-.37-1.56-.37h-3.63q-.06.12-.18.28t-.28.28-.29.31-.18.25q-.25-.31-.44-.5l-.25-.25-.13-.15-.18-.22h-5.2v5.81h5.13l1-1.12.63.71.37.41h3q0-.5.03-.75t.03-.69q1.44 0 2.13-.44v1.88h2.44v-1.63H22v1.63q9.19 0 9.56-.19.07 0 .25-.12v.31zm2.25-4.06q-.56 0-.56.44 0 .3.47.34t1.1.06.87.4v1.63q-.38.57-1.44.57h-2l.06-.94h2q.5 0 .5-.38 0-.3-.4-.37t-.9-.06-.95-.32-.43-1q0-1.25 1.5-1.25h2.06v.88h-1.88zm-3.56 2.5q.56 0 .56-.38 0-.31-.43-.4t-.94-.07-.9-.28-.41-1q0-1.31 1.5-1.31h2.06v.94H30q-.56 0-.56.44 0 .25.43.3t.94.04.9.28.41 1q0 1.38-1.62 1.38h-2v-.94h2zm-2.62-2.5h-2.44v.75h2.37v.94h-2.37v.8l2.44.07v.88h-3.5V16.8h3.5v.88zm-3.57-6.75q.07-.07.5-.07h.2l-.5 1.07q-.13 0-.2-.07t-.06-.46.06-.47zm.32-1.63h1v.94h-1.16l-.31.06-.25.16-.22.22-.1.34-.03.47q0 1.25 1 1.25H25l1.5-3.44H28l1.75 4.13V9.3h1.63l1.8 3.07V9.3h1.07v4.32h-1.5l-1.94-3.25v3.25H28.7l-.44-.94h-2.12l-.38.94h-1.19q-2.06 0-2.06-2.13 0-2.19 2.12-2.19zm2.62.75-.75 1.75h1.44zm-6.31-.75H22v4.32h-1.06V9.3z',
        'w' => 1.125,
        'vw' => 36
    ],
    'cc-apple-pay' => [
        'p' => 'M18.88 13.66q0 1.72-1.82 1.72h-1.5v-3.44h1.5q1.82 0 1.82 1.72zm3 3.9q0-.81 1.25-.87l1.43-.13v.5q0 .38-.18.7t-.57.5-.81.18q-1.13 0-1.13-.88zM36 4.94v22q0 1.25-.88 2.12t-2.12.88H3q-1.25 0-2.13-.88T0 26.94v-22Q0 3.69.88 2.8T3 1.94h30q1.25 0 2.13.87T36 4.94zM8 12.3q.75.07 1.38-.62.56-.69.5-1.5-.82.06-1.38.69-.63.68-.5 1.43zm3.75 4.7q-1.19-.56-1.19-1.88-.06-1.06 1-1.74-.62-.88-1.69-.94-.37-.07-.72.06t-.65.25-.44.13-.66-.22-.84-.22q-1.31.06-1.94 1.19-.5 1.18-.28 2.43t.9 2.2q.82 1.18 1.45 1.18.25 0 .68-.19t.82-.19.81.2.75.18q.62 0 1.37-1.13.38-.56.63-1.3zm8.5-3.38q0-1.24-.78-2t-2.03-.74h-3.19v8.5h1.31v-2.94h1.82q.8 0 1.47-.35t1.03-1 .37-1.47zm5.63 1.5q0-.62-.32-1.06t-.87-.69-1.32-.25q-.68 0-1.25.22t-.9.66-.35 1h1.2q.18-.88 1.24-.88 1.25 0 1.25 1.07v.5l-1.62.06q-2.38.19-2.38 1.88 0 .5.28.93t.75.66 1.03.22q1.44 0 2-1.07v1h1.25v-4.25zm6.37-1.93h-1.38l-1.5 5.06h-.06l-1.56-5.06h-1.38l2.25 6.19-.12.37q-.25.88-1.13.88H27v1q.06.06.5.06 1 0 1.53-.44t.9-1.56z',
        'w' => 1.125,
        'vw' => 36
    ],
    'cc-diners-club' => [
        'p' => 'M15 5q4.56 0 7.78 3.22T26 15.97t-3.22 7.78T15 27t-7.78-3.25T4 15.97t3.22-7.75T15 5zm-2.5 17.5v-13q-2 .75-3.25 2.53T8 16t1.25 3.94 3.25 2.56zm5 0q1.94-.81 3.19-2.56T21.94 16q0-.88-.22-1.72t-.6-1.56-.93-1.35-1.25-1.12-1.44-.75v13zM33 2q1.25 0 2.13.88T36 5v22q0 1.25-.88 2.13T33 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h30zM20.62 28q4.94 0 8.72-3.53t3.79-8.34q0-5.25-3.72-8.7T20.62 4h-5.68q-3.32 0-6.07 1.5T4.5 9.8t-1.63 6.31q0 4.88 3.57 8.38t8.5 3.5h5.68z',
        'w' => 1.125,
        'vw' => 36
    ],
    'cc-discover' => [
        'p' => 'M32.5 12.25q0 .81-.94.81h-.31V11.5h.31q.94 0 .94.75zM33 2q1.25 0 2.13.88T36 5v22q0 1.25-.88 2.13T33 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h30zm-2.75 8.69v5.12h1v-2.06h.13l1.37 2.06H34l-1.63-2.19q1.2-.18 1.2-1.43 0-.38-.1-.63t-.34-.44-.44-.28-.6-.12-.56-.03h-1.28zm-3.5 0v5.12h2.81v-.87h-1.81v-1.38h1.75v-.87h-1.75v-1.13h1.81v-.87h-2.81zm-4.31 0h-1.07l2.2 5.25h.5l2.24-5.25H25.2l-1.38 3.44zm-3.47-.19q-1.16 0-1.97.81t-.81 1.97.81 1.97 1.97.81 1.97-.8.81-1.98-.81-1.97-1.97-.81zm-3.1.38q-1-.44-1.87-.26t-1.47.94-.6 1.7q0 .74.35 1.37t.9.93 1.29.38 1.4-.31v-1.2q-.87.88-1.9.45t-1.03-1.63q0-1.12 1-1.6t1.93.41v-1.18zM9.81 15q-.75 0-1.12-.63l-.63.63q.82 1.19 2.2.9t1.37-1.65q0-.63-.32-.94t-1.18-.62q-.76-.32-.76-.7 0-.43.54-.52t1.03.4l.5-.68q-.88-.82-2-.5T8.3 12.06q0 1 1.32 1.44.68.25.87.62t-.06.63-.63.25zm-6.37.81q1.37 0 2.1-.78t.71-1.78q0-.75-.37-1.34t-1-.9-1.44-.32H2v5.12h1.44zm4.25 0V10.7h-1v5.12h1zM34 27.06V18q-3.19 2-10.78 5.28T8 28h25.06q.38 0 .66-.28t.28-.66zM4.62 12q.57.44.57 1.25t-.57 1.31q-.43.38-1.37.38H3v-3.38h.25q.94 0 1.38.44z',
        'w' => 1.125,
        'vw' => 36
    ],
    'cc-jcb' => [
        'p' => 'M27 15.25v-2h2.38q.8.19.8 1 0 .88-.8 1H27zm2.63 1.25q.87.19.87 1.1t-.88 1.02q-.12.07-2.62.07V16.5h2.63zM36 5v22q0 1.25-.88 2.13T33 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h30q1.25 0 2.13.88T36 5zm-24.63 7H7.81q0 .69.03 1.69t.03 1.53-.03 1.25-.15 1.06-.38.69-.69.5-1.06.16q-1.69 0-3.56-.94v1.75q.5.19 1.15.28t1.16.15.97.07H6l.25.06q1.12 0 1.97-.19t1.4-.37.94-.69.56-.88.25-1.21.07-1.38-.03-1.69-.03-1.84zm11.13.31q-1.56-.43-3.28-.5t-3.31.22-2.63 1.32T12.25 16q0 1.69 1.06 2.72T16.06 20t3.35.16 3.1-.47v-1.75q-1.5.81-3.04.94t-2.6-.6T15.82 16t1.07-2.28 2.59-.6 3.03.94v-1.75zM34 17.87q0-.25-.1-.5t-.21-.46-.35-.41-.5-.31-.6-.2l-.62-.12q.88-.12 1.38-.65t.5-1.22q0-.63-.28-1.06t-.82-.66-1.2-.28h-7.5v8h7.68q1.13 0 1.88-.56t.75-1.57z',
        'w' => 1.125,
        'vw' => 36
    ],
    'cc-mastercard' => [
        'p' => 'M30.19 25.66q0 .71-.72.71t-.72-.71.72-.72.72.72zm-19.44-.72q.69 0 .69.72t-.69.72-.69-.72.69-.72zm7.34-.07q.54 0 .6.57H17.5q.06-.57.6-.57zm6.72.07q.75 0 .75.72t-.75.72q-.68 0-.68-.72t.68-.72zm6.63 1.62q.06 0 .06.07l-.06.06v.06h-.25l-.06-.06v-.13q0-.06.06-.06v-.06h.19l.06.06v.06zm-.13.07h.07v-.13h-.2v.19h.07v-.07l.06.07h.07zM36 5.05v22q0 1.25-.88 2.13t-2.12.87H3q-1.25 0-2.13-.87T0 27.06v-22q0-1.25.88-2.12T3 2.06h30q1.25 0 2.13.88T36 5.06zM4 13.78q0 3.6 2.53 6.13t6.1 2.53q2.62 0 4.8-1.44-3.43-2.81-3.4-7.22t3.4-7.16q-2.18-1.5-4.8-1.5-3.57 0-6.1 2.53T4 13.79zm14 6.78q3.31-2.56 3.31-6.78T18 7q-3.31 2.56-3.31 6.78T18 20.56zm-8.88 4.82q0-.94-.93-.94-.57 0-.82.44-.25-.44-.75-.44-.43 0-.68.31v-.25h-.5v2.31h.5v-1.25l.06-.28.12-.19.16-.15.22-.06q.37 0 .47.3t.1 1.07v.56h.43v-.9l.03-.35.06-.28.1-.18.15-.16.28-.06q.32 0 .41.3t.1 1.07v.57h.5v-1.44zm2.82-.88h-.5v.25q-.31-.31-.75-.31-.5 0-.81.34t-.32.88.31.87.82.35.75-.38v.31h.5V24.5zm2.5 1.63q0-.57-.72-.7t-.72-.3q0-.26.44-.26t.75.2l.19-.38q-.44-.38-1.16-.25t-.72.75q0 .5.72.65t.72.28q0 .32-.5.32t-.81-.25l-.25.37q.56.38 1.3.25t.76-.69zm2.19.56-.13-.44q-.75.44-.75-.25v-1.06h.81v-.44h-.81v-.69h-.5v.69h-.5v.44h.5V26q0 .75.6.84t.77-.15zm.87-.81h1.69q0-1.44-1.07-1.44-.5 0-.8.34t-.32.85q0 .93.78 1.18t1.34-.31l-.25-.37q-.37.37-.84.28t-.53-.53zm3.69-1.38q-.63-.25-.94.25v-.25h-.5v2.31h.5V25.5q0-.5.31-.6t.44.1zm.68 1.16q0-.53.41-.7t.84.16l.25-.43q-.56-.44-1.3-.16t-.76 1.1q0 .87.78 1.15t1.28-.22l-.25-.37q-.43.3-.84.15t-.4-.69zM26 24.5h-.5v.25q-.38-.5-1.13-.22t-.75 1.1q0 .87.75 1.15t1.13-.28v.31h.5V24.5zm2.13 0q-.63-.31-.94.25v-.25h-.5v2.31h.5V25.5q0-.44.31-.56t.5.06zm2.5-.94h-.5v1.2q-.38-.5-1.13-.23t-.75 1.13.78 1.12 1.1-.28v.31h.5v-3.25zm.5-4.68v.3h.06v-.3h.06v-.07H31v.07h.13zm.37 7.75v-.13l-.13-.13h-.18l-.13.13v.19q0 .06.07.06.06.06.09.06h.16l.12-.12v-.06zm.25-7.82h-.13l-.06.25-.12-.25h-.07v.07l.13.25h.06l.07-.25v.25h.12v-.32zm.25-5q0-3.62-2.53-6.15t-6.1-2.53q-2.62 0-4.8 1.43 3.37 2.82 3.4 7.22T18.57 21q2.18 1.44 4.8 1.44 3.57 0 6.1-2.53T32 13.8z',
        'w' => 1.125,
        'vw' => 36
    ],
    'cc-paypal' => [
        'p' => 'M11.63 16.13q0 .56-.38.96t-1 .41q-1 0-1-.94 0-.56.4-.97t.98-.4q1 0 1 .94zm-6.63-3q.69 0 .97.15t.16.78q-.13.88-1.38.88h-.5l.31-1.7q0-.12.2-.12H5zm17.75 0q1.13 0 1.19.68 0 1.13-1.38 1.13h-.5l.25-1.69q0-.13.2-.13h.24zM36 5v22q0 1.25-.88 2.13T33 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h30q1.25 0 2.13.88T36 5zM8 13.44q0-1.75-2.13-1.75h-2.5q-.3 0-.37.31l-1 6.38q-.06.25.19.25h1.19q.3 0 .37-.38L4 16.63q0-.13.1-.22t.25-.1h.78q1.37 0 2.12-.75T8 13.44zm5.25.56h-1.13q-.24 0-.3.5-.45-.63-1.45-.63-1.12 0-1.93.85t-.82 2.03q0 .88.57 1.44t1.43.56q1 0 1.7-.75-.07.19-.07.38 0 .25.19.25h1.06q.25 0 .38-.38l.62-4q0-.25-.25-.25zm2.56 6.13 4-5.75v-.13q0-.25-.18-.25h-1.2q-.18 0-.3.19l-1.63 2.44-.69-2.38q-.12-.25-.37-.25H14.3q-.25 0-.25.25l.31.9.66 1.88.4 1.1q-1.3 1.8-1.3 2t.18.18h1.19q.19 0 .31-.18zm9.94-6.7q0-1.74-2.13-1.74h-2.5q-.3 0-.37.31l-1 6.38q0 .25.19.25h1.31q.19 0 .25-.2l.25-1.8q.06-.13.16-.22t.22-.1h.75q1.37 0 2.12-.75t.75-2.12zM31 14h-1.13q-.25 0-.3.5-.38-.63-1.45-.63-1.18 0-1.96.85t-.79 2.03q0 .88.54 1.44t1.46.56q1 0 1.63-.75v.38q0 .25.19.25h1.06q.25 0 .38-.38l.62-4q0-.25-.25-.25zm3-2.06q0-.25-.19-.25h-1.19q-.18 0-.18.19l-1 6.5v.06q0 .19.18.19h1.07q.25 0 .31-.38zm-5.63 3.25q1 0 1 .94 0 .56-.37.97t-1 .4q-1 0-1-.94 0-.62.4-1t.98-.37z',
        'w' => 1.125,
        'vw' => 36
    ],
    'cc-stripe' => [
        'p' => 'M30.75 13.81q.5 0 .81.35t.32 1.1h-2.25q0-.5.15-.85t.44-.47.53-.13zm-7.31.13q.37 0 .69.28t.5.75.18 1.03q0 .94-.37 1.53t-1 .6-1.06-.44v-3.32q.37-.43 1.06-.43zM33 2q1.25 0 2.13.88T36 5v22q0 1.25-.88 2.13T33 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h30zM7.62 17.56q0-1.12-.84-1.72T5.1 15t-.84-.63q0-.43.75-.43 1.12 0 2.19.56v-2.13Q6.2 12 5 12q-.87 0-1.55.28t-1.07.88T2 14.5q0 1.06.84 1.63t1.69.84.84.66q0 .56-.87.56-1.06 0-2.38-.75v2.12q1.13.5 2.38.5 1.44 0 2.28-.65t.84-1.85zM11.95 14v-1.88h-1.69V10.2l-2.19.5v7.12q0 .94.66 1.6t1.65.65q1.07 0 1.57-.31V18q-.2.19-.94.1t-.75-.91v-3.2h1.69zm4.62.19v-2.06q-1.25-.44-1.87.68l-.13-.68h-1.93v7.8h2.25v-5.3q.62-.82 1.68-.44zm2.75 5.75v-7.81h-2.25v7.8h2.25zm0-8.94V9.19l-2.25.43v1.82zm4.63 9.06q1.31 0 2.19-1.03T27 16q0-1.87-.9-2.94T23.94 12q-.94 0-1.69.69l-.12-.56h-2V22.5l2.25-.5v-2.5q.8.56 1.56.56zM34 16.88v-.82q0-1.81-.84-2.94T30.8 12q-.81 0-1.44.31t-1.06.85-.69 1.28-.25 1.62q0 1.32.5 2.22t1.32 1.35 1.87.43q1.44 0 2.5-.56v-1.81q-1 .5-2.19.5-1.62 0-1.75-1.32H34z',
        'w' => 1.125,
        'vw' => 36
    ],
    'cc-visa' => [
        'p' => 'M29.38 14.44q.5 2.37.56 2.81h-2.07l1-2.69.07-.15.12-.4.13-.38zM36 5v22q0 1.25-.88 2.13T33 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h30q1.25 0 2.13.88T36 5zM9.5 20.69l4-9.69h-2.69l-2.44 6.62-.25-1.37-.87-4.44q-.19-.75-1.19-.81h-4l-.06.19q1.44.37 2.62 1.06l2.25 8.44H9.5zm5.94 0L17 11h-2.5l-1.56 9.69h2.5zm8.75-3.13q0-1.62-2.13-2.69-1.43-.68-1.43-1.18.06-.82 1.5-.82.93-.06 1.8.38l.26.06.31-2.06q-1-.44-2.25-.44-1.19 0-2.16.4t-1.5 1.16-.53 1.63q-.06 1.56 2.2 2.69 1.3.62 1.3 1.18-.06.94-1.5.94-1.18 0-2.37-.5l-.31-.19-.38 2.2q1.06.5 2.81.55 1.94 0 3.16-.9t1.22-2.4zM33 20.7 31 11h-2q-.94 0-1.25.81L24 20.7h2.63q.43-1.19.5-1.44h3.25l.3 1.44H33z',
        'w' => 1.125,
        'vw' => 36
    ],
    'centercode' => 'M20.56 16.78q-.18 1.66-1.47 2.69t-2.93.87-2.7-1.47-.84-2.96 1.47-2.7 2.94-.84 2.69 1.47.84 2.94zM15.2 31.5q-4.44-.25-8.03-2.75T1.8 22.31.62 14q.5-4 2.85-7.16T9.3 2Q12.25.62 15.44.5t5.97 1 5.03 3.06 3.6 4.85 1.46 6.15q.19 6.82-4.69 11.53T15.2 31.5zm11.25-11.25q.5-2.13-.44-4.22t-2.6-3.66-4-3-4.18-2.28-3.6-1.47Q8.94 8.2 8.41 14T9.5 27.13q3.19.75 7.65-1.1t9.29-5.78z',
    'centos' => [
        'p' => 'm18.13 6.13 1.93 2-4.75 4.74V6.13h2.82zm-10.2 2 2-2h2.82v6.75zm2.63-2.63L14 2l3.5 3.5h-2.8v8l-.69.69-.69-.69v-8h-2.75zM12.2 16l-.69.69h-8v2.81L0 16l3.5-3.5v2.81h8zm4.94-1.31 4.75-4.75 2 1.94v2.8h-6.75zM28 16l-3.5 3.5v-2.81h-8l-.63-.7.63-.68h8V12.5zM4.06 11V6.12h5l-2 2 5.7 5.63v.94h-1L6.12 9.06zm19.82-4.88V11l-2-1.94-5.63 5.63h-.94v-.94l5.63-5.63-1.94-2h4.87zM12.74 25.88h-2.8l-2-1.93 4.81-4.82v6.75zM4.06 14.7v-2.81l2.07-1.94 4.75 4.75H4.06zM23.88 21v4.88H19l1.94-2-5.63-5.63v-.94h.94l5.63 5.63zm0-3.69v2.82l-2 1.93-4.75-4.75h6.75zm-3.82 6.57-1.93 2H15.3v-6.75zM6.13 22.05l-2.07-1.93V17.3h6.81zM17.5 26.5 14 30l-3.44-3.5h2.75v-8l.69-.69.69.69v8h2.81zM7.06 23.87l2 2h-5V21l2.07 1.94 5.62-5.63h1v.94z',
        'w' => 0.873,
        'vw' => 28
    ],
    'chrome' => 'M8.25 13.63 3.44 6.25Q5.63 3.5 8.8 2T15.44.5q4.18 0 7.81 2.06 4 2.32 6.06 6.44L16.5 8.31q-1.38-.06-2.66.28t-2.37 1-1.94 1.7-1.28 2.34zm2 2.37q0-1.44.72-2.63t1.9-1.9 2.63-.72q2.19 0 3.72 1.53T20.75 16t-1.53 3.72-3.72 1.53-3.72-1.53T10.25 16zm19.69-5.56q2.06 5.19.19 10.65t-6.88 8.35q-4 2.31-8.62 2.06l7-10.81q1.56-2.25 1.46-5t-1.84-4.81zM8.37 19q1.26 2.44 3.66 3.72t5.03.78l-3.94 7.81q-5.62-.87-9.37-5.22T0 16q0-4.63 2.5-8.5z',
    'chromecast' => 'M27.99 4H4a2.67 2.67 0 0 0-2.67 2.67v4H4v-4h23.99v18.66h-9.33V28H28c1.48 0 2.67-1.2 2.67-2.67V6.67c0-1.47-1.2-2.67-2.68-2.67ZM1.33 23.98v3.99h4c0-2.2-1.8-4-4-4Zm0-5.32v2.65A6.7 6.7 0 0 1 8 28h2.67a9.34 9.34 0 0 0-9.33-9.34Zm12 9.34h2.68A14.71 14.71 0 0 0 1.33 13.34v2.65a12 12 0 0 1 12 12.01Zm0 0',
    'cloudflare' => [
        'p' => 'm25.5 20-14.43-.19a.29.29 0 0 1-.23-.12.29.29 0 0 1-.03-.26.38.38 0 0 1 .33-.25l14.56-.19a5.2 5.2 0 0 0 4.26-3.17l.83-2.16a.5.5 0 0 0 .02-.29 9.49 9.49 0 0 0-18.23-.97 4.25 4.25 0 0 0-6.7 4.45 6.06 6.06 0 0 0-5.82 6.9c.03.14.14.25.28.25h26.64c.15 0 .29-.1.33-.25l.2-.7c.25-.84.16-1.62-.25-2.19-.38-.52-1-.83-1.77-.86Zm6.62-6.18h-.4a.26.26 0 0 0-.21.18l-.57 1.95a2.5 2.5 0 0 0 .26 2.18c.37.52 1 .83 1.76.86l3.08.19a.28.28 0 0 1 .25.37.39.39 0 0 1-.34.26l-3.2.19a5.2 5.2 0 0 0-4.25 3.17l-.23.6c-.02.05-.01.1.02.15.03.05.08.08.13.08h11.01a.3.3 0 0 0 .28-.21 7.86 7.86 0 0 0-7.59-9.97Zm0 0',
        'vw' => 40
    ],
    'cloudscale' => 'M19.88 9.63q.18-.2.4-.13t.28.28-.06.47l-4.75 5.5q.12.88-.5 1.5-.5.5-1.22.5t-1.25-.53-.53-1.25.5-1.22q.62-.62 1.44-.56l1.62-1.25q-1.69-1.57-3.94-1.57-2.43 0-4.15 1.75T6.06 17.5Q6 16.94 6 16.25q0-3.38 2.37-5.81T14.07 8q2.8 0 5.24 2.12zm-5.25 6.93q.62-.62 0-1.25t-1.26 0 0 1.25 1.26 0zM14 2q5.81 0 9.9 4.1T28 16t-4.1 9.9T14 30t-9.9-4.1T0 16t4.1-9.9T14 2zm0 24q4.13 0 7.06-2.94T24 16t-2.94-7.06T14 6 6.94 8.94 4 16t2.94 7.06T14 26z',
    'cloudsmith' => [
        'p' => 'M20.75 26.25q0 1.44-.69 2.63t-1.9 1.9-2.66.72q-2.12 0-3.69-1.53t-1.56-3.72 1.56-3.72T15.53 21t3.69 1.53 1.53 3.72zM15.5 11q-2.13 0-3.56 1.4T10.5 16q0 2.13-1.5 3.56T5.28 21t-3.75-1.44T0 16q0-2.19 1.56-3.84t3.69-1.66q2.19 0 3.47-1.28T10 5.75q0-2.13 1.66-3.69T15.5.5t3.6 1.53 1.4 3.72-1.4 3.72T15.5 11z',
        'w' => 0.65,
        'vw' => 21
    ],
    'cloudversify' => [
        'p' => 'M9.31 19H6.25q-.31 0-.53-.22t-.22-.53v-2.5q0-.31.22-.53t.47-.22h2.94v.44q0 1 1.18 1h1.25q.38 0 .66-.16t.44-.44.15-.59v-1.31q0-1.2-.93-1.2 1.12-.93 2.8-1.5.7-2.74 2.13-4.4t4.07-2.47q1.25-.5 2.97-.37t3.37.6 3.13 1.8 2.15 3.3q3.38 1.06 4.94 3.84t.81 5.85q-.62 2.68-2.84 4.46t-5.34 1.79q-2.25 2.3-5.79 2.37t-5.84-2.06q-3.69.19-6.22-1.75T9.32 19zm14.19.5q.38 2.31 2.16 3.19t3.59.5 3.19-1.66 1.81-3.1q.5-2.37-.62-4.5L33 12.7q.06 1.25-.72 2.69T30 17.5q1.94-2.19 1.69-4.19t-1.97-3.47-3.97-1.72q-3.87-.56-7.12 2.7 5.12.5 6.37 4.68-.87-1.44-2.56-2.25t-3.19-.75-2.63.44q-1.68.87-2.68 2.47t-.88 3.4q.06.5.16.94t.28.81l.34.7.47.62.47.5.5.44.5.37.5.34.4.29q-.3-1.38.07-3.16T18.5 18q-.81 1.38-.56 2.97t1.43 2.72 3.13 1.6 4.44-.35q-4.25-1.87-3.44-5.44zM8 13H5.5q-.38 0-.69-.31t-.31-.7V9.5q0-.37.31-.68t.69-.31H8q.44 0 .72.31T9 9.5V12q0 .37-.28.69T8 13zM.62 10.5Q0 10.5 0 9.87V8.13q0-.63.63-.63h1.75q.25 0 .43.19t.2.44v1.75q0 .62-.63.62H.63zm9.88-1.56V7.56q0-.56.56-.56h1.38q.56 0 .56.56v1.38q0 .56-.56.56h-1.38q-.56 0-.56-.56zm-7 5.81v1.5q0 .31-.22.53t-.47.22H1.2q-.25 0-.47-.22t-.22-.53v-1.5q0-.31.22-.53T1.2 14h1.56q.75 0 .75.75z',
        'w' => 1.201,
        'vw' => 38
    ],
    'cmplid' => [
        'p' => 'M226.119,388.165a3.816,3.816,0,0,0-2.294-3.5,3.946,3.946,0,0,0-1.629-.385L72.6,384.3a19.243,19.243,0,0,1-17.924-26.025L81.585,255.692a35.72,35.72,0,0,1,32.373-26H262.525a7.07,7.07,0,0,0,6.392-5.194l10.769-41.131a3.849,3.849,0,0,0-2.237-4.937,3.755,3.755,0,0,0-1.377-.261c-.063,0-.126,0-.189.005H127.38a106.8,106.8,0,0,0-96.99,77.1L3.483,358.824A57.469,57.469,0,0,0,57.314,436q1.43,0,2.86-.072H208.742a7.131,7.131,0,0,0,6.391-5.193L225.839,389.6A3.82,3.82,0,0,0,226.119,388.165ZM306.658,81.2a3.861,3.861,0,0,0,.251-1.367A3.813,3.813,0,0,0,303.079,76c-.064,0-.128,0-.192,0h-41A7.034,7.034,0,0,0,255.5,81.2l-21.347,80.915h51.131ZM180.364,368.249H231.5L263.452,245.69H212.321ZM511.853,79.723a3.809,3.809,0,0,0-3.8-3.661c-.058,0-.137,0-.23.007h-41a7.1,7.1,0,0,0-6.584,5.129L368.91,430.634a3.54,3.54,0,0,0-.262,1.335,3.873,3.873,0,0,0,3.864,3.863c.056,0,.112,0,.169,0h41a7.068,7.068,0,0,0,6.392-5.193L511.533,81.2A3.624,3.624,0,0,0,511.853,79.723ZM324.649,384.47h-41a7.2,7.2,0,0,0-6.392,5.194L266.52,430.8a3.662,3.662,0,0,0-.268,1.374A3.783,3.783,0,0,0,270.023,436c.06,0,.166,0,.3-.012h40.905a7.036,7.036,0,0,0,6.391-5.193l10.769-41.131a3.75,3.75,0,0,0-3.445-5.208c-.108,0-.217,0-.326.014Zm311.324-308.4h-41a7.066,7.066,0,0,0-6.392,5.129l-91.46,349.436a4.073,4.073,0,0,0-.229,1.347,3.872,3.872,0,0,0,3.863,3.851c.056,0,.112,0,.169,0h40.968a7.1,7.1,0,0,0,6.392-5.193L639.68,81.2a3.624,3.624,0,0,0,.32-1.475,3.841,3.841,0,0,0-3.821-3.564c-.068,0-.137,0-.206.006ZM371.562,225.236l10.8-41.1a4.369,4.369,0,0,0,.227-1.388,3.869,3.869,0,0,0-3.861-3.842c-.057,0-.113,0-.169,0h-41.1a7.292,7.292,0,0,0-6.391,5.226l-10.834,41.1a4.417,4.417,0,0,0-.26,1.493c0,.069,0,.138,0,.206a3.776,3.776,0,0,0,3.757,3.507c.076,0,.18,0,.3-.012h41.129A7.034,7.034,0,0,0,371.562,225.236Z',
        'vw' => 640,
        'vh' => 512
    ],
    'codepen' => 'M31.38 10q.62.38.62 1.13v9.74q0 .75-.63 1.13l-14.62 9.75q-.75.5-1.5 0L.62 22Q0 21.62 0 20.87v-9.75q0-.74.63-1.12L15.24.25q.75-.5 1.5 0zm-14-6.06v6.44l5.93 3.93 4.82-3.18zm-2.75 0L3.87 11.13 8.7 14.3l5.94-3.93V3.94zM2.74 13.69v4.62L6.19 16zm11.88 14.37v-6.43l-5.94-3.94-4.82 3.19zM16 19.25 20.88 16 16 12.75 11.12 16zm1.38 8.81 10.75-7.18-4.82-3.2-5.94 3.95v6.43zm11.87-9.75V13.7L25.81 16z',
    'codiepie' => [
        'p' => 'M26.38 12.69q.93 0 1.34.81t.03 1.66-1.38.84h-.68v2.75h-1.63V12.7h2.32zm3.12 9.37q-1.31 2.88-3.34 4.97t-4.79 3.28-5.87 1.2q-6.44 0-10.97-4.54T0 16 4.53 5.03 15.5.5q4.44 0 7.94 2.28t5.69 6.35l-11.63 6.8zm-2.38.75-3.8-1.93q-2.7 4.5-7.63 4.5-3.88 0-6.6-2.72t-2.72-6.6 2.72-6.6 6.6-2.71q4.5 0 7.31 3.94l3.69-2.2q-1.94-2.93-4.85-4.58T15.5 2.25q-5.69 0-9.72 4.03T1.75 16t4.03 9.72 9.72 4.03q3.63 0 6.69-1.85t4.94-5.09z',
        'vw' => 30
    ],
    'confluence' => 'M.13 25.75q.18-.25.5-.75.87-1.38 1.4-2.19t1.78-2.25 2.38-2.22 3-1.53 3.84-.78 4.69.6 5.72 2.12l6.62 3.13q.38.18.53.56t-.03.75l-3.12 7.12v.07q-.19.37-.6.5t-.78-.07q-.81-.37-2.25-1.06-2.44-1.19-3.5-1.69t-2.69-1.12-2.21-.75-1.63-.16-1.47.28-1.19.94-1.28 1.47-1.34 2.16q-.5.87-1.38.37L.5 27.12q-.88-.5-.38-1.37zm31.75-19.5q-.2.25-.5.75-.82 1.38-1.38 2.19t-1.81 2.25-2.38 2.22-3 1.56-3.84.78-4.69-.62T8.5 13.3q-3.06-1.5-6.63-3.12-.37-.19-.5-.6t0-.78l3.2-7.12q.18-.38.59-.53t.78.03q.81.37 2.31 1.12 3.19 1.5 4.47 2.13t3.1 1.19 2.46.43T20 5.41t1.72-1.5 1.72-2.78l.06-.07q.57-.8 1.38-.25l6.62 4.07q.88.5.38 1.37z',
    'connectdevelop' => [
        'p' => 'M34.38 15.06q.74.13.74.94 0 .75-.74.94l-3.44 5.94q.06.18.06.31 0 .75-.75.94L27 29.75q.06.13.06.25 0 .25-.12.47t-.35.35-.47.12q-.43 0-.68-.25h-6.7q-.3.31-.74.31t-.75-.3h-6.63q-.3.3-.75.3-.37 0-.65-.28t-.28-.65q0-.13.06-.38l-3.25-5.56Q5 23.94 5 23.19q0-.12.06-.31l-3.43-5.94Q.88 16.75.88 16q0-.81.74-.94L5 9.31q0-.06-.03-.09l-.03-.03q0-.63.56-.88L9 2.25q-.06-.19-.06-.31 0-.25.15-.47t.38-.34.47-.13q.44 0 .69.38h6.62Q17.56 1 18 1t.75.38h6.63q.3-.38.75-.38.25 0 .47.13t.34.34.12.47q0 .12-.06.31l3.5 6q.38 0 .63.28t.25.66q0 .19-.13.44zM9.61 28.2v-4.75H6.88zm0-5.25v-3.19l-2.8 3q.05.13.05.19h2.75zm0-3.94v-5.75L6.45 10q-.2.13-.44.13l-3.25 5.62V16q0 .13-.06.25l3.5 6 .25.13zm0-6.38v-4.5l-2.74 1v.07q0 .25-.13.5zm0-5V3.88l-2.74 4.7zM30 10.06V10l-6.13 6.38 4 4.18 2.2-10.44zM18.25 22.2l-.75.75h1.44zm-.06-.69 5-5.13-5.19-5.5-5 5.26zm.37.38 1.07 1.06h7.75l.37-1.75-4.25-4.44zm7.07-19.13L18.3 10.5l5.2 5.5 6.13-6.31q-.13-.25-.13-.5v-.07l-3.63-6.25q-.06-.06-.25-.12zm-.57-.13h-.25L14.44 6.38 18 10.2zm-7.68 0h-6.5L14.05 6l9.25-3.38h-4.68q-.25.32-.63.32t-.62-.31zM10.13 3v4.44l3.37-1.25-3.19-3.38q-.06 0-.12.07zm0 4.94v5.19l2.56 2.68 4.94-5.3-3.75-3.95zm0 5.81v4.75l2.24-2.38zm0 5.5v3.69h6.68l1.07-1.13-5.13-5.37zm.43 10.13h.07l5.68-5.94h-6.18v5.62l.06.07q.19.06.37.25zm8.06 0h6.57l-5.75-5.94h-2.38l-5.75 5.93h6.07q.25-.3.62-.3t.63.3zm7.5-.75 1.13-5.2h-7.19l5.63 5.76q.12-.07.18-.13zm.82-1.38 2.19-3.81h-1.38zm2.18-4.31q0-.13.07-.19l-1.07-1.13-.25 1.32h1.25zm4.13-6.75V16l.06-.31-2.87-4.94L28.25 21l1.31 1.37q.07-.06.25-.12z',
        'w' => 1.125,
        'vw' => 36
    ],
    'contao' => 'm2.81 19.06 1 4.25q.3 1.31 1.2 3.35T7.12 30h-5q-.88 0-1.5-.62T0 27.88V4.13q0-.88.63-1.5T2.13 2h3.62Q4.44 3.2 3.44 4.75 2.63 6 2.19 7.5t-.53 2.6.12 3.03.4 2.78.63 3.15zM29.88 2q.87 0 1.5.63t.62 1.5v23.75q0 .87-.63 1.5t-1.5.62h-3.25q3.88-3.81 3.57-10.25l-8.57 1.88q-.18 3.87-3.62 4.56-2.19.5-3.63-.69-.87-.75-1.53-2.34t-1.53-5.91q-.62-2.94-.84-4.78t-.13-2.56.41-1.41q.81-1.69 3.06-2.13 3.38-.75 5.13 2.7l8.56-1.82Q26.25 4.06 24.25 2h5.63z',
    'cotton-bureau' => 'M29.64 20.65c-1.48 5.74-5.89 9.04-12.61 9.27v-3.07c0-3 1.65-4.65 4.65-4.65 3.87 0 6.2-2.32 6.2-6.2 0-3.84-2.29-6.14-6.1-6.2-2.05-4.32-9.15-4.03-11.07 0-3.8.06-6.09 2.36-6.09 6.2 0 3.88 2.33 6.21 6.22 6.21 3 0 4.66 1.64 4.66 4.65v3.08C7.1 29.62 2.3 24.62 2.3 16c0-8.86 5.1-13.95 13.95-13.95 7.18 0 11.87 3.33 13.42 9.3h1.58C29.62 4.47 24.27.5 16.24.5 6.56.5.75 6.32.75 15.99c0 9.7 5.82 15.53 15.49 15.51 8.01 0 13.37-3.99 14.98-10.85Zm-7.33-9.26c2.59.22 4.01 1.82 4.01 4.61 0 3-1.65 4.65-4.64 4.65-1.8 0-3.09-.6-3.86-1.7 5.2-1.04 4.73-6.24 4.5-7.56Zm-5.1 6.09c-.16-.64-1.02-5.44 3.51-6.06.14.63 1.03 5.44-3.51 6.06Zm-.96-9.23c1.79 0 3.06.6 3.84 1.73a5.29 5.29 0 0 0-3.85 2.71 5.3 5.3 0 0 0-3.85-2.71c.78-1.1 2.08-1.71 3.85-1.71Zm-4.47 3.17c4.57.66 3.68 5.42 3.53 6.06-4.52-.61-3.69-5.43-3.51-6.06Zm-.96 9.23c-3 0-4.64-1.65-4.64-4.65 0-2.77 1.43-4.38 4-4.6-.41 2.32-.08 6.65 4.49 7.54-.76 1.1-2.06 1.7-3.83 1.7Zm3.33.78a5 5 0 0 0 2.1-2.1c.48.9 1.2 1.6 2.11 2.1a5 5 0 0 0-2.1 2.1 5.06 5.06 0 0 0-2.1-2.1Zm0 0',
    'cpanel' => [
        'p' => 'M13.13 13.75q.18.69.03 1.4t-.6 1.32-1.12 1-1.5.4h-2l.37-1.5q.19-.68.88-.68h.69q.56 0 .9-.44t.22-1q-.19-.88-1.12-.88H8.63L7 19.31q-.19.69-.88.69H4.56l2.19-8.13q.19-.68.88-.68h2.3q.57 0 1.1.18t.94.54.72.8.44 1.04zM3.37 15.5q-.75 0-1.06.6t0 1.18 1.07.6h.68q.13 0 .25.06t.16.19.03.25L4 20h-.75q-1.56 0-2.56-1.25t-.57-2.81q.32-1.25 1.2-1.9t2-.66h2.5l-.38 1.43q-.2.7-.94.7H3.37zm15.44-2.19q.94 0 1.47.72t.35 1.6l-.88 3.25Q19.44 20 18.25 20H14.7q-.75 0-1.25-.44t-.6-1 .04-1.15.62-1 1.25-.4h3l-.25.8q-.19.7-.87.7H15.3q-.25 0-.25.24t.25.25h2.19q.25 0 .31-.37l.57-1.94q.06-.38-.13-.38h-3.62q-.44 0-.72-.34t-.16-.78l.19-.88h4.87zm20.75-2.12q.2 0 .32.15t.12.35l-1.69 6.19q-.25.93-1.03 1.53t-1.78.6l2.25-8.32q.13-.5.63-.5h1.18zm-14.75 2.19q1.07 0 1.88.56t1.19 1.53.06 1.97l-.56 2.06q-.13.5-.63.5h-1.19q-.18 0-.31-.15t-.12-.35l.68-2.5q.2-.56-.18-1.03t-.94-.47h-1.25l-1.07 4q-.12.5-.62.5h-1.19q-.25 0-.37-.15t-.07-.35l1.63-6.12h3.06zM30.94 18h4.56l-.31 1.31q-.25.7-.94.7h-4.13q-.93 0-1.46-.73t-.29-1.6l.63-2.5q.19-.5.53-.93t.84-.66 1.07-.22h2.94q.93 0 1.46.7t.29 1.56l-.2.75q-.24 1.12-1.43 1.12h-3.31l.18-.81q.2-.69.88-.69h1.37q.2 0 .25-.19l.07-.12q.06-.32-.25-.32h-2.07q-.43 0-.5.38l-.5 1.87q-.12.38.32.38z',
        'w' => 1.248,
        'vw' => 40
    ],
    'creative-commons' => 'M15.38 13.44 13.3 14.5q-.62-1.25-1.75-1.25-2.06 0-2.06 2.75t2.06 2.75q1.38 0 1.94-1.31l1.88.94q-1.32 2.43-4.07 2.43-.62 0-1.31-.18T8.5 20t-1.31-1.5-.5-2.5q0-2.5 1.4-3.65t3.16-1.16q2.87 0 4.12 2.25zm8.93 0-2.06 1.06q-.56-1.25-1.75-1.25-2.06 0-2.06 2.75t2.06 2.75q1.38 0 1.94-1.31l1.94.94Q23 20.8 20.3 20.8q-.68 0-1.37-.18T17.47 20t-1.28-1.47-.5-2.53q0-2.5 1.4-3.65t3.1-1.16q2.87 0 4.12 2.25zM15.5.5Q22.06.5 26.53 5T31 16q0 3.38-1.28 6.31t-3.44 4.9-4.97 3.13-5.81 1.16q-6.38 0-10.94-4.53T0 16q0-4.13 1.97-7.69t5.56-5.69T15.5.5zm0 28.19q5.06 0 8.88-3.6t3.8-9.09q0-5.31-3.68-9t-9-3.69q-5.38 0-9.03 3.81T2.8 16q0 5.19 3.72 8.94t8.97 3.75z',
    'creative-commons-by' => 'M19.69 12.13v6.37h-1.75V26h-4.82v-7.5h-1.8v-6.38q0-.37.3-.68t.7-.32h6.37q.37 0 .69.32t.3.68zM13.3 8.19Q13.31 6 15.47 6t2.15 2.19q0 2.12-2.15 2.12T13.3 8.2zM15.5.5Q22.06.5 26.53 5T31 16q0 6.69-4.63 11.1T15.5 31.5q-6.38 0-10.94-4.53T0 16Q0 9.69 4.44 5.1T15.5.5zm0 2.81q-5.38 0-9.03 3.82T2.8 16q0 5.19 3.72 8.94t8.97 3.75q3.25 0 6.16-1.56t4.72-4.5 1.8-6.63q0-5.31-3.68-9t-9-3.69z',
    'creative-commons-nc' => 'M15.5.5q6.56 0 11.03 4.47T31 16q0 6.69-4.63 11.1T15.5 31.5q-6.38 0-10.94-4.56T0 16Q0 9.69 4.44 5.1T15.5.5zm-12 11.31q-.69 1.94-.69 4.2 0 5.18 3.75 8.9t8.94 3.72q1.56 0 3.03-.35t2.53-.87 2.03-1.28 1.57-1.35 1.12-1.31.66-.9.25-.44l-5.82-2.57q-.37 1.57-1.59 2.44t-2.72 1v2.38h-1.81V23q-2.57 0-4.69-1.88L12.19 19q1.56 1.38 3.46 1.4t1.97-1.52q0-.63-.43-1-1.75-.75-13.7-6.07zm12-8.56q-6.63 0-10.63 5.81l5.94 2.7q.44-1.38 1.57-2.04T14.75 9V6.56h1.81V9q2.13.06 3.88 1.38l-2 2.12q-2.19-1.56-3.94-.62-1 .5-.44 1.3l5.82 2.57 7.93 3.57q.38-1.63.38-3.32 0-5.37-3.69-9-3.69-3.75-9-3.75z',
    'creative-commons-nc-eu' => 'M15.5.5q6.56 0 11.03 4.47T31 16q0 2.69-.81 5.13t-2.32 4.3-3.43 3.26-4.25 2.1-4.7.71q-6.37 0-10.93-4.56T0 16Q0 9.69 4.43 5.1T15.5.5zm0 28.19q2.06 0 3.94-.66t3.03-1.44 2.22-1.87 1.37-1.56.63-.97v-.07l-8-3.5H13q.75 3.13 4.25 3.13 1.69 0 2.94-.69l.62 2.88q-1.75 1-4.06 1-2.81 0-5.03-1.66T9 18.62H7.31v-1.87h1.44v-1.19H7.31V13.7h.25L3.5 11.87q-.7 1.94-.7 4.13 0 5.19 3.75 8.94t8.94 3.75zm-1-11.94zm4.81-1.19 8.5 3.75q.38-1.56.38-3.31 0-5.31-3.69-9t-9-3.69q-6.75 0-10.63 5.82l5.07 2.25q.44-.7.87-1.25 2.38-2.63 6-2.63 2.13 0 3.88.75l-.69 3q-1.5-.62-2.88-.62-2.06 0-3.25 1.37-.24.31-.5.88l1.76.8h4.37v1.88h-.19z',
    'creative-commons-nc-jp' => 'M15.5.5Q22.06.5 26.53 5T31 16q0 6.69-4.63 11.1T15.5 31.5q-6.38 0-10.94-4.56T0 16Q0 9.69 4.44 5.1T15.5.5zm0 28.19q6.75 0 11-6.19l-5.13-2.25v2.25h-4V26h-3.8v-3.5h-4v-2.44h4v-1.12l-.38-.75H9.56v-2.44h1.75l-7.94-3.5Q2.81 14 2.81 16q0 5.19 3.75 8.94t8.94 3.75zm5.5-8.63-3.44-1.5-.19.38v1.12H21zm.38-3.19 6.3 2.82q.5-1.75.5-3.69 0-5.31-3.68-9t-9-3.69q-2 0-3.81.53t-3.13 1.5-2.25 1.94T4.7 9.44l5.06 2.25-1.69-3.2h4.13l2.5 5.38 1.43.63 2.7-6h4.12L19 15.75h2.38v1.12z',
    'creative-commons-nd' => 'M15.5.5Q22.06.5 26.53 5T31 16q0 6.69-4.63 11.1T15.5 31.5q-6.38 0-10.94-4.53T0 16Q0 9.69 4.44 5.1T15.5.5zm0 2.81q-5.38 0-9.03 3.82T2.8 16q0 5.19 3.72 8.94t8.97 3.75q3.25 0 6.16-1.56t4.72-4.5 1.8-6.63q0-5.31-3.68-9t-9-3.69zm5.88 9v2.7H10.13v-2.7h11.24zm0 5v2.63H10.13V17.3h11.24z',
    'creative-commons-pd' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm0 28.13q2.31 0 4.47-.85t3.81-2.31 2.78-3.4Q26.5 22 12.94 16q0 1.06.34 2.06t1.13 1.72 1.9.72q1.32 0 2.25-1.06l2.25 2.75Q18.75 24 15.88 24q-1.26 0-2.44-.38t-2.4-1.21-1.98-2.47-.75-3.82q0-1.06.25-2.06l-4.94-2.19q-1.18 3-.59 6t2.19 5.38 4.34 3.87 5.94 1.5zm12.13-9.38h.06q.44-1.63.44-3.25 0-5.19-3.72-8.9t-8.9-3.73q-3.26 0-6.04 1.57T5 9.13l4.63 2.06q2.06-3.25 6.3-3.25 2.88 0 4.88 1.75l-2.5 2.56q-.87-1-2.37-.87T13.63 13z',
    'creative-commons-pd-alt' => 'M15.5.5Q22.06.5 26.53 5T31 16q0 6.69-4.63 11.1T15.5 31.5q-6.38 0-10.94-4.53T0 16Q0 9.69 4.44 5.1T15.5.5zm0 28.19q3.25 0 6.16-1.57t4.71-4.5T28.2 16q0-5.31-3.69-9t-9-3.69q-5.38 0-9.03 3.81T2.8 16q0 5.19 3.72 8.94t8.97 3.75zm4.31-17.07q4.38 0 4.38 4.29t-4.38 4.28H16.5v-8.57h3.31zm.07 6.76q2.12 0 2.12-2.44 0-.94-.28-1.5t-.75-.75-.66-.22-.5-.03h-1.18v4.93h1.25zm-7.13-6.75q3.13 0 3.13 2.87 0 .38-.1.75t-.4.94-1.1.9-1.97.35h-1.69v2.75H8.5v-8.57h4.25zm-.31 4.18q.81 0 1.19-.65t0-1.28-1.25-.63h-1.75v2.56h1.8z',
    'creative-commons-remix' => 'M15.5.5Q22.06.5 26.53 5T31 16q0 6.69-4.63 11.1T15.5 31.5q-6.38 0-10.94-4.53T0 16Q0 9.69 4.44 5.1T15.5.5zm0 2.81q-5.38 0-9.03 3.82T2.8 16q0 5.19 3.72 8.94t8.97 3.75q3.25 0 6.16-1.56t4.72-4.5 1.8-6.63q0-5.31-3.68-9t-9-3.69zm10.13 12.94.3.19v4.37l-1.18.5-1.94.85q-.94.42-1.06.4-.19 0-2-.75T15.8 20.2t-2.69-1.13l-4 1.7L5.2 19v-4l3.68-1.5V9.06l4.2-1.81 9.5 3.88V15zm-4.38 5.13v-2.75h-.06v-.13l-7.07-2.88v2.82l7.07 2.94h.06zm.44-3.63 2.5-1-2.32-1-2.43 1.06zM25 20.13v-2.7l-2.81 1.2v2.68z',
    'creative-commons-sa' => 'M15.5.5Q22.06.5 26.53 5T31 16q0 6.69-4.63 11.1T15.5 31.5q-6.38 0-10.94-4.53T0 16Q0 9.69 4.44 5.1T15.5.5zm0 2.81q-5.38 0-9.03 3.82T2.8 16q0 5.19 3.72 8.94t8.97 3.75q3.25 0 6.16-1.56t4.72-4.5 1.8-6.63q0-5.31-3.68-9t-9-3.69zm-6.88 10.5q.26-1.81 1.1-3.12t1.97-1.88 2-.78 1.75-.22q2.19 0 3.84.85t2.5 2.19T23 13.53t.38 2.72q0 2.07-.85 3.9t-2.72 3.1-4.5 1.25q-1.37 0-2.72-.47t-2.56-1.9-1.53-3.7h3.88q.18 2.88 3.43 2.88 1.38 0 2.5-1.25t1.13-3.93q0-1.44-.35-2.5t-.75-1.53-1.03-.72-.87-.28-.56-.04q-3.13 0-3.5 2.75h1.12l-3.06 3.07-3.07-3.07h1.25z',
    'creative-commons-sampling' => 'M15.5.5Q22.06.5 26.53 5T31 16q0 6.69-4.63 11.1T15.5 31.5q-6.38 0-10.94-4.53T0 16Q0 9.69 4.44 5.1T15.5.5zm0 2.81q-5.38 0-9.03 3.82T2.8 16q0 5.19 3.72 8.94t8.97 3.75q3.25 0 6.16-1.56t4.72-4.5 1.8-6.63q0-5.31-3.68-9t-9-3.69zm.25 3.31q.25 0 .5.16t.25.53l.38 6.75.3-3.75q0-.62.66-.62t.66.62l.37 5.13.38-4q.06-.63.62-.63.63 0 .63.56l.75 6.07.31-.82q.2-.43.7-.43h3.62v1.37h-3.2l-1.18 2.75q-.19.5-.69.44t-.56-.63l-.25-2-.5 5.75q-.06.63-.63.63-.62 0-.68-.63l-.38-4.8-.31 4.5q-.13.56-.69.56t-.62-.57l-.38-5.5-.38 5.94q0 .75-.65.75t-.66-.75l-.37-5.44-.32 5.32q-.12.62-.68.62t-.63-.63l-.03-.03-.03-.09-.38-5.56-.3 3.56q-.07.56-.7.56-.5 0-.62-.56L9.37 17q-.25.5-.69.5H5.43v-1.31h2.82l.87-1.75q.19-.44.66-.38t.6.57l.18 1.25.5-6.13q0-.25.19-.44t.5-.19q.62 0 .62.75l.32 4.32.43-5.75q0-.32.16-.47t.28-.19.19-.03.19.03.3.19.2.47l.31 5 .38-6.13q.06-.62.62-.69z',
    'creative-commons-sampling-plus' => 'M15.5.5Q22.06.5 26.53 5T31 16q0 6.69-4.63 11.1T15.5 31.5q-6.38 0-10.94-4.53T0 16Q0 9.69 4.44 5.1T15.5.5zm0 2.81q-5.38 0-9.03 3.82T2.8 16q0 5.19 3.72 8.94t8.97 3.75q3.25 0 6.16-1.56t4.72-4.5 1.8-6.63q0-5.31-3.68-9t-9-3.69zm6.69 12.81h3.69v1.44h-3.2l-1.12 2.75q-.18.5-.75.4t-.62-.59L20 18.7l-.38 5.18q-.06.7-.75.7t-.75-.7l-.25-4-.3 3.63q-.07.69-.76.69t-.69-.69l-.3-4.5-.32 5.06q0 .7-.69.7t-.75-.7l-.31-4.56-.25 4.38q-.07.68-.75.68t-.75-.68L11.68 19l-.25 2.69q-.06.69-.72.69t-.71-.63l-.7-4.56q-.18.37-.62.37H5.5v-1.43h2.68l.88-1.75q.19-.44.72-.38t.65.63l.07.74.5-5.62q.06-.63.72-.63t.71.63l.25 3.56.38-5q0-.68.69-.68t.75.68l.25 4.2.31-5.26q.06-.69.75-.69t.69.7l.37 5.87.25-2.82q.07-.62.72-.65t.72.65l.31 4.32.32-3.2q0-.68.65-.68t.78.63l.7 5.8.24-.56q.13-.5.63-.5zm-4.88.38q.5 0 .5-.5t-.5-.5H16v-1.31q0-.5-.5-.5t-.5.5v1.31h-1.31q-.5 0-.5.5t.5.5H15v1.31q0 .5.5.5t.5-.5V16.5h1.31z',
    'creative-commons-share' => 'M15.5.5Q22.06.5 26.53 5T31 16q0 6.69-4.63 11.1T15.5 31.5q-6.38 0-10.94-4.53T0 16Q0 9.69 4.44 5.1T15.5.5zm0 2.81q-5.38 0-9.03 3.82T2.8 16q0 5.19 3.72 8.94t8.97 3.75q3.25 0 6.16-1.56t4.72-4.5 1.8-6.63q0-5.31-3.68-9t-9-3.69zm6.31 8.25q.38 0 .63.25t.25.63V23.8q0 .38-.25.63t-.63.25h-8.44q-.3 0-.56-.25t-.25-.63v-3.37H9.2q-.2 0-.35-.06t-.28-.16-.19-.28-.06-.31V8.19q0-.7.75-.88h8.56q.2 0 .41.13t.31.31.1.44v3.37h3.37zM10 18.75h2.56v-6.31q0-.75.75-.88h3.44v-2.5H10v9.69zm11-5.5h-6.75V23H21v-9.75z',
    'creative-commons-zero' => 'M15.5.5Q22.06.5 26.53 5T31 16q0 6.69-4.63 11.1T15.5 31.5q-6.38 0-10.94-4.53T0 16Q0 9.69 4.44 5.1T15.5.5zm0 2.81q-5.38 0-9.03 3.82T2.8 16q0 5.19 3.72 8.94t8.97 3.75q3.25 0 6.16-1.56t4.72-4.5 1.8-6.63q0-5.31-3.68-9t-9-3.69zm0 3.75q1.81 0 3.13.82t2 2.15.96 2.82.29 3.15-.29 3.16-.96 2.81-2 2.16-3.13.8-3.13-.8-2-2.16-.96-2.81T9.13 16t.28-3.15.97-2.82 2-2.15 3.12-.82zm0 3.38q-2.75 0-2.75 5.56 0 1.25.13 2.13l3.37-6.25q.5-.82-.19-1.38-.31-.06-.56-.06zm2.56 2.87-3.81 6.57q-.56.8-.06 1.25t1.31.43q2.75 0 2.75-5.56 0-1.56-.19-2.69z',
    'critical-role' => [
        'p' => 'M14.06 0q13.25 7.69 13.57 7.81.25.07.25.38v15.69q0 .25-.25.3-.57.38-3.78 2.2t-6.5 3.68T13.94 32h-.19Q.38 24.32.13 24.13 0 24.07 0 23.94V8.13q0-.2.19-.32.62-.37 7-4t6.75-3.8h.12zm-.68 1.25L1.24 8.19q.25.12 1.94.75-.5.37-.69.31-1.31-.5-1.69-.62V23q.07 0 .22-.31t.44-.78l.66-1.1.75-1.28.71-1.28.63-1.1.65-1.09q.07-.12.57-.06.06 0 .06.06-3 5.13-4.25 7.32-.06.06.06.06H13q.05 0 .05-.06.07-.32 0-.57h.07l-.38-.06h-.81q-.75 0-1.31-.81-.5-.63-1.38-1.82Q9.13 20 8.2 20q-.13 0-.13.06.07 1.69.07 2.07.06.25.62.3.19 0 .19.13t-.13.2H7.75l-1.06-.04-.13-.16.13-.12h.19l.18-.06.07-.2q.06-.8.06-5 0-.74-.31-.8-.07-.07-.16-.07h-.15q-.2-.12.12-.31 3 0 3.5.31.75.44.78 1.28T10 19.62q-.06.07 0 .13t.22.28.28.34q1.32 1.57 1.53 1.75t1.04.32q.06 0 .06-.07v-.68l.12-.07q.63.07 1.44-.06.13 0 .13.13l.03.8.03.88.06.07h11.69l-5.06-8.82.43-.06q.13 0 .2.06.18.38 1.4 2.47t2.31 4l1.1 1.9V8.63q-1 .32-6.63 2.5-.06 0-.06-.06-.07-.19-.2-.38 0-.06.07-.06l.35-.12.87-.35 1.19-.47 1.31-.5 1.25-.43 1-.35.47-.22q-1.37-.81-12.25-7.06l2.56 4.44 2.88 5q-.63-.07-.75-.25-2.88-5.07-5.13-8.88-.25.32-1.56 2.66T9.85 8.5t-1.28 2.07H8l2.84-4.85q1.84-3.16 2.53-4.47zm3.24 8.13q.07 0 .38-.07.13 0 .13.13v1.31q-.13.13-.2 0t-.12-.31q0-.25-.31-.32t-2-.12q-.06 0-.06.06l.25 7.63q0 .06.06.06.5.25.56.25.63.44.75 1.25.07.75-.37 1.5t-1.63.81q-1.75.07-2.12-1.5-.25-.81.15-1.44t1.1-.8q.06-.07.06-.2.13-4.5.25-7.56 0-.06-.06-.06h-.57q-1.18 0-1.5.1t-.43.4q-.2.31-.25.06.06-.56.18-1.12.07-.2.2-.13.3.06 2.37.13.06 0 .06-.13.06-1.37.06-1.87.07-.13-.06-.2-.56-.24-.56-.8 0-.7.62-.97t1.07.22q.3.3.28.78t-.41.72q-.12.12-.12.3l.06 1.88q0 .07.06.07 1.19 0 2.13-.07zM4.2 8.88q.44-.13.9-.13h.72l.85.1.84.09q.25.06.19.25 0 1.19-.06 1.5 0 .06-.07.12-.25 0-.25-.37l-.06-.07q-.06-.75-1.22-1.06t-2.28.31-1.12 2.32q0 1.25.87 2.28t2.5.9q.75-.06 1.06-.37t.35-.63.15-.3q.2 0 .2.24L7.5 15.2v.06q-.06.38-1.4.5t-2.79-.6-1.62-2.52q-.13-1.32.53-2.38t1.97-1.38zm5 5.43H8l-.19-.06q.07-.13.22-.16t.16-.28v-2.5q0-.25-.2-.31-.24 0-.24-.06l.19-.13q1.37 0 1.87.06.44.07.66.5t-.16.88q0 .06-.37.44.56.75 1.06 1.25.19.19.63.19.12 0 .12-.13v-2.56q0-.38-.19-.44-.19 0-.19-.06t.2-.13h1.18q.31 0 .06.19h-.06q-.25.06-.25.31 0 2.38.06 2.7 0 .12.32.12.12 0 .12.06.06.06 0 .1t-.12.02h-2.2l-.24-.06-.19-.12-.19-.22-.19-.22-.21-.32-.22-.3q-.07-.13-.2-.13H9l-.13.06q.07.19.07.5 0 .63.18.63t.25.06q-.06.12-.18.12zm17.19-.62q-.2.62-.32.69l-3.12-.07h-.25q-.13-.06-.07-.12.07-.13 0-.25-.06-.25-.25-.63 0-.12-.12-.12h-.94l-.06.06-.31.69q0 .12.12.16t.31 0 .38.03q.25 0 .12.18h-1.75q-.12 0-.06-.18.25 0 .38-.2.06-.18.25-.55 1.12-2.57 1.18-2.63.07-.13.16-.03t.69 1.56.78 1.72q.06.13.44.13.18 0 .25-.25v-2.63q0-.19-.38-.25-.19 0 0-.19h1.38q.12 0 .15.07t-.1.12h-.12q-.19.06-.19.19v2.56q.07.19.2.19.93 0 1-.25.18-.25.24 0zm-6.5 4.06q-.2 0-.25-.13.12-.06.18-.12h2.32q.06-.06.09-.03t.03.1q0 .24-.06.68 0 .06-.1.06t-.1-.11q0-.19-.13-.22t-.5-.03q-.25-.06-.37-.06-.06 0-.06.12v1q0 .13.06.06l1-.06q.19-.06.19.13 0 .56-.07.68-.06.2-.18 0-.32-.37-.2-.3-.74-.07-.77-.04t-.03.4q0 1 .25 1 .69.07.78 0t.22-.3q.06-.13.12-.13t.07.12q0 .32-.07.63 0 .12-.31.12h-2.2q-.12 0-.12-.1t.12-.15q.25 0 .25-.12.06-3.19-.19-3.19zm-.07-6.88q.2.07.2.32v.56q-.07.13-.17.13t-.09-.13q0-.44-.44-.56-1.12-.31-1.6.37t-.15 1.5q.25.63.82.82t1 .06q.3-.06.43-.38v-.12q.13-.13.2.12.12.32-.13.63l-.25.12q-1.32.2-2-.18-.94-.5-.97-1.47t.53-1.35q.81-.87 2.62-.43zm-1.75 6.63q.13 0 .16.1t-.16.15H18q-.25 0-.25.31v2.57q.06.25.25.25.63.06.84.03t.25-.2.1-.21q.12-.25.12.06.07.38-.03.57t-.12.18h-2.53q-.07 0-.1-.09t.1-.1q.3-.12.3-.18.07-1.82 0-3 0-.19-.37-.25-.06 0-.06-.1t.13-.09h1.43zm-2-3.63q0 .26.38.26.12 0 .12.09t-.12.1h-1.32q-.37 0-.18-.13.06-.07.18-.07t.16-.15.03-1.16v-1.56q0-.19-.19-.25H15q-.13-.06-.1-.13t.16-.06q1.19-.06 1.31 0 .13.13-.06.19h-.06q-.19.06-.19.19v2.69zm-2.31 17.2q.13.06.25 0 .5-.32 6.16-3.54T26.3 24H14.94q-.07 0-.07.06v.57l.03.56.04.66v1.4q0 .19-.07.44L14 29.75q-.07.13-.13 0l-.75-1.94q-.18-.37-.12-.68 0-.63.06-3 0-.13-.12-.13H1.5q.06 0 1.22.69t2.87 1.66T9 28.3t3.1 1.79 1.66.96zm-5.7-11.83q0 .19.2.19.5.12.96.06t.66-.5q.3-.69.09-1.47t-.9-.97q-.2-.06-.54-.06t-.47.12v2.63zm6.63-.94q0 .94.06 2.44 0 .31-.37.38T13.6 21t-.41-.44l.06-2.44q-.69.32-.6 1.5t1.04 1.57q.87.37 1.37-.44.25-.38.22-1.22t-.6-1.22zm-.75-12.37q-.5 0-.5.47t.5.47q.25 0 .37-.25t.03-.47-.4-.22zm-5 6.47q0 .15.12.15.38.06.57-.12t.18-.4v-.29q-.06-.31-.28-.47t-.4-.12-.2.09v1.16zm13.25.4q-.13-.31-.38-1-.12.38-.37 1h.75z',
        'w' => 0.869,
        'vw' => 28
    ],
    'css3' => [
        'p' => 'm30 2-4 23-13.94 5L0 25l1.25-5.94h5.12l-.5 2.57 7.25 2.75 8.38-2.75 1.19-6.07H1.87l1-5.12H23.7l.68-3.32H3.5L4.56 2H30z',
        'vw' => 30
    ],
    'css3-alt' => [
        'p' => 'M0 2h24l-2.19 24.75L12 30l-9.81-3.25zm19.56 5H4.44l.37 2.94h7.7l-.45.19-7 2.93.25 2.88L15.7 16l-.25 3.82-3.38.93-3.31-.81-.19-2.38H5.5l.37 4.63 6.2 1.81 6.12-1.81.81-9.13h-6.94l7.19-3.12z',
        'w' => 0.75,
        'vw' => 24
    ],
    'cuttlefish' => [
        'p' => 'M21.5 19.13q1.38 5.12 5.44 7.3-2.19 2.38-5.16 3.73T15.5 31.5q-6.44 0-10.97-4.53T0 16 4.53 5.03 15.5.5q3.31 0 6.28 1.35t5.16 3.71q-4.07 2.2-5.44 7.32-.81-1.44-2.5-2.4t-3.5-.98q-2.63 0-4.56 1.94T9 16t1.94 4.57 4.56 1.93q1.19 0 2.38-.43T20 20.85t1.5-1.72z',
        'w' => 0.841,
        'vw' => 27
    ],
    'd-and-d' => [
        'p' => 'M5.13 6.19q-.07-1.88.8-3Q6 4 6.2 4.5 6.75 1.94 8.88.44q-.25.69-.2 1.06Q10 .38 11.45.13q1.31-.25 2.62.06 3.5.81 4.38 4.5.44 1.87-.5 4.19-.56 1.18-1.31 2.18l-.38-.37Q14.5 9 13.94 8.38q-.13-.13 0-.25.75-1.32.62-2.75-.12-1-.75-1.63-.69-.75-1.65-.72t-1.66.72q-1 1.13-.69 2.75v.25q-.37 1.38-.19 2.63v.12q-2-1-2.68-3.25Q6.12 8 6.5 9.63 5.25 8.38 5.12 6.19zm29.37 7.68.13.13q-1.13-.25-2.82.06-.94.25-1.62.57-.13.06-.5.3l-1.13-.43q-2.31-.5-3.87 1.25-1.7 1.88-2.94 4.69l-.06.12v.07q-.13-.07-.94-.75T19.37 19q.07 0 .07-.03v-.03q2.12-2.88 4.68-4.44 1.63-.94 3.25-1.12.2 0 .32-.07h-.82q-1.75.2-3.68 1.25-1.38.82-2.25 1.88l-.07.12-.06-.06q1-2.81 3.56-4.62l.07.06q-.2.25-.2.31l-.3.63q-.13.31.12.31t.63-.25q.18-.19.59-.53t.62-.5.54-.34.62-.32.69-.19H28q-.06 0-.13.07-.81.56-.93.75t-.04.34.29.1l.5-.13q1-.37 1.8-.44 1.26-.06 2.5.5 1.45.57 2.5 1.63zm-14.69 11q.13.07.07.25-2.75 5.38-7.25 6.2-2.7.43-4.88-.57-1.87-.81-3-2.25 0-.06-.06-.13t-.16-.18-.1-.2q-.05.63.07 1.38l-.62-.62q-1.07-1.31-1.5-3.25-.75-3.06.5-6.38-.7.38-1.5 1.07.93-2.7 3.5-5.32.25.2.18.5-.37 1.94-.3 2.75 1.68-1.75 4-2.75-.63 1-.7 2.38 1.44-1.38 2.88-.88-2 2.07-2.44 4-.5 2.25.56 3.94t3.07 1.69q1 0 2.12-.69 1.63-1.12 2.75-3l.13-.19 2.68 2.25zm10.75-2.93q2 .69 2.7 2.44.43 1.06.15 2.21t-1.22 1.91q-1.5 1.25-3.56 1.13h-.13q1.5 1.06 3.19.62 0 .06-.16.22l-.22.22q-1.06.94-2.69 1.25-1.87.31-3.37-.63t-1.94-2.93l-.06-.44q-.56.44-.63 1.4t.63 1.91q-.06 0-.13-.06H23q-1-.44-1.56-1.28t-.5-1.97q.12-1 0-1.94t-.82-1.56q-.37-.25-.81-.63t-.75-.62-.78-.66-.78-.66q-.88-.68-1.06-1.56-.07-.25-.13-.87 0-.57-.5-.94-.12-.06-.25-.19t-.31-.25-.25-.25q-1-.81-.5-1.87.25 1.12 1.44 1.3h.25l.3.04.26.03q.75.13.87.88.13.56.2.62.12.5.62.56h.34l.53.07q.69.06 1.25.5.56.5 1.16 1t1.34 1.09 1.19.97q.56.5 1.19.5.81.06 1.28-.56t.28-1.44v-.13q.44.32.44.38.62.62.81 1.65t-.25 1.85q-.06 0-.5.62.06.07.13.07 1.43.12 2.43-.57.75-.5.85-1.34t-.47-1.53q-.38-.38-1.19-.75.75-.13 1.81.19zM6.2 11.19q-.4-.69-1.4-1.31 1.44-.07 2.5.24Q6.7 9 7 7.7q.44 1 1.5 1.68.38.26 1.44.88t1.44 1l.62.69q0-.06-.06-.19t-.06-.19q-.44-.94-1.44-1.56-.2-.13-.2-.31-.3-1.07-.05-2.32.06-.31.25-1 .06-.25.25-.44.18 1.75 1.25 2.88.5.5 2.12 2.06 1.19 1.13 1.56 1.63l.25.31q-.37-1-1.5-2.12L12.57 9q-.26-.24-.32-.68-.13-1 .25-1.88.06-.06.12-.18l.07.68q.18.94.75 1.63t2 2l.37.37q1.13 1.07 1.56 1.94.5.94.38 1.88 0 .06.06.12.63.25.63.88.56-.19 1-.88.12.44 0 1t-.35 1.1-.6.93-.74.6q.12-1.5-1.06-1.69-.13-.06-1.44-.19.69-1.25.06-1.37-1.25-.13-2.25 0t-1.37.31q1 0 1.5.19-.38 1.12-.38 1.19-.12.31.06.56t.5 0q.07 0 .13-.06t.12-.07q-.18 1-.18 1.13 0 .44.34.5t.6-.06q-.07.56-1.07 1.12t-1.75.5q.75-.5 1-1.12-.62.06-1.19.18l-.06-.06q.44-.31.69-.81.37-1-.1-1.72t-1.47-.78q-.93-.06-1.62.31.56-1.37 2-2.12-.75-.25-1.5-.44 1.19-.75 2.87-.88-.3-.06-.96-.1t-1.72.35-1.7 1.07q.26.06 1.45.25-2 .56-3.88 2.25 0-.13.03-.25t.07-.22.06-.22.03-.25q.06-.44 0-.81-.13-.94-1.19-1.07-.56-.06-1.5.13h-.19q.44-.44 1.03-.78t1.25-.7T6 12.13q.5-.37.18-.94zm8.06 1.69q.06.5.28.84t.53.53.69.13q-.5-1.07-1.5-1.5z',
        'w' => 1.123,
        'vw' => 36
    ],
    'd-and-d-beyond' => [
        'p' => 'M19.63 15.13q-.32 0-.54-.13t-.34-.22-.13-.15q-.18-.2-.18-.07t.06.19q-.69-.63-.31-1.63l-.07-.12-.12.13q0-.26.15-.47t.29-.35l.12-.06q.06.06.06.1t-.03.15-.03.31q.19-.31.69-.5-.13.19-.13.44.25-.31.5-.19-.5.57-.4 1.13t.6.56.87-.69q.06.07.25.22t.25.22q-.57 1.13-1.57 1.13zm3.25-1.35q0 .22-.32.28.13-.31 0-.56.32.06.32.28zm.62.35q0 .3-.25.5t-.63.12q.32.25.57.13-.19.37-.82.28t-.62-.66q-.31.31 0 .56-.38-.12-.44-.43 0-.2.03-.41t-.1-.31-.55-.41l-.2-.22q-.06-.09-.05-.19v-.15l-.13-.16-.16-.16-.03-.25q0 .2.2.25.05 0 .15.04t.15.03q.07.06.07.18 0 .07.06.13l.15.03.29.1.18.15.25.22q.13.12.2.12.24.2.4.07t.1-.32q.18.13.21.38t-.16.37q.57.13.63-.3 0-.32-.31-.38.31-.07.56.15t.25.53zm-2.44-1.2q.7-.93 1.5-1-.68-.05-1.25.57.13-.5.63-.81l-.07.25q.07 0 .16-.06t.25-.2.28-.12l-.19.25.13-.06.37-.06q.5 0 .82.43-.5-.18-.82.13-.8-.5-1.37 1-.31-.25-.44-.31zm15.19 3.26q1.69 0 2.72.84T40 19.25q0 3.06-3.69 3.06-1.5 0-3.25-.12l.5-.5V16.8l-.5-.5q1.69-.12 3.19-.12zm-1 4.75q2.94.37 2.94-1.69 0-2-2.94-1.69v3.38zM18.87 11.3l-.25-.43q0-.2.04-.44T19 9.9t.88-.22q1.06 0 1.06 1 0 .5-.32.87l-.5-.5q.2-.25.1-.56t-.4-.31q-.2 0-.32.18t-.06.44q-.13.25-.07.5-.3-.19-.43-.56-.2.25-.07.56zm-.06.38q.07-.19-.31-.31h.44q-.13-.13-.07-.38.13.19.47.38t.41.37q0-.25-.25-.37-.19-.44.06-.7 0 .32.44.73t.44.53q-.06-.25-.44-.57l-.19-.18q-.06-.25.07-.44 0 .25.46.72t.41.78q.13.06.13.12.12 0 .18-.12.07.44-.3.62.05-.18-.07-.28t-.32-.03v-.25q-.06-.19-.68.06.18-.06.31 0-.13.25.06.25v.32h.2q-.2.31-.57.31.19-.06.19-.25-.07.06-.2.06.32-.18.07-.5-.19-.25-.5-.06.06-.25.38-.38-.13 0-.32-.06.25-.19.57-.12-.5-.13-.82.18.13 0 .25.07-.37.06-.68.37l.06-.31q-.07-.19-.5-.13.06-.06.22-.15t.28-.16.12-.12zm1.63.44q-.07-.2-.25-.25.06.25.25.25zm2.43.25.07-.07.12-.06v1.12q-.12-.06-.31-.06l-.31.06q-.2-.18-.32-.18.32.3-.06.43-.06.07-.12 0v-1l.03-.06.1-.06q.18-.25.43-.25.19 0 .31.12h.07zM5 19.12q.63.2.9.57t.29.5v.5q0 .75-.63 1.12-.12.07-.28.13t-.31.1-.44.06-.5.06-.66.03H.2l.5-.56v-2l-.7-.94h.7v-1.82l-.5-.56h3.5q.87 0 1.59.35T6 17.88q0 1-1 1.25zm-2.56-1.75v1.32l.31.03q.25.03.44.03t.4-.03.35-.1.22-.18.1-.32V18q0-.25-.07-.38t-.31-.15-.47-.06-.72-.04h-.25zm0 3.75H3.5l.28-.03.22-.06.22-.06.16-.1.09-.15.03-.22h-.06v-.25q0-.13-.06-.25t-.2-.19-.27-.1-.38-.02h-1.1v1.43zm6.31-1.25v1h3.75l-.88 1.32H6.57l.44-.57v-4.8l-.44-.5h4.57l.87 1.3H8.75v.94h2.31l-.87 1.32H8.75zm13.38-3.75q1 0 1.75.35t1.15.9.6 1.22 0 1.32-.57 1.22-1.15.9-1.78.35q-1.38 0-2.29-.66t-1.09-1.56.03-1.82 1.13-1.56 2.22-.65zm0 4.88q1.18 0 1.62-.88t-.03-1.75-1.56-.87-1.57.88 0 1.75 1.53.87zm-5.7-.56v1.19l.45.56h-2.63l.44-.56v-1.2l-2.75-4.12h2.81l-.37.44 1.31 2.38 1.25-2.38-.38-.44h2.63zM14.64 11q-.5-.38-.57-.94.44.25 2.5.25 1.56 0 2 1.44l-.28.19-.22.15-.12.16q.18-.06.56-.06-.63.37-.63 1.25l.2-.2q-.13.38-.07.76-.69.5-1.63.5H14.2q.25-.06.5-.25.12-.13.12-.38v-2.43q0-.32-.19-.44zm1.37.38v2.12h.69l.28-.1.22-.18.19-.32.06-.46-.06-.47-.2-.32-.21-.18-.28-.1-.28-.03h-.25zm14.31 4.93h2.44l-.44.57v5.3h-1.37q-.13-.43-1.5-1.8t-1.5-1.7v3l.44.5h-2.44l.44-.56v-4.75l-.44-.56h1.94q.12.25 1.4 1.47t1.47 1.72v-2.62zm-6.68-6q.93.07 1.46.57t.6.87.06.69-.12.72-.7.9-1.5.5q.45-.56-.12-1.06 1.2 0 1.2-1.06 0-.25-.04-.47t-.16-.31l-.25-.2-.28-.09h-.72v.25q-.25-.06-.43 0 .18-.12.37-.12-.56-.19-.94.19.07-.13.25-.32-.18.07-.37.25 0-.5-.19-.62-.5-.38-.62-.94.43.25 2.5.25z',
        'w' => 1.25,
        'vw' => 40
    ],
    'dailymotion' => [
        'p' => 'M18.68 16.69a3 3 0 0 0-1.52-.39 2.87 2.87 0 0 0-2.94 2.92c0 .88.28 1.6.83 2.16a2.8 2.8 0 0 0 2.1.84 2.93 2.93 0 0 0 2.98-3 2.93 2.93 0 0 0-1.45-2.53ZM0 2v28h28V2Zm23.42 23.33H20.1v-1.5h-.04c-.66 1.1-1.82 1.64-3.49 1.64a5.55 5.55 0 0 1-5.12-3.05c-.47-.95-.72-2-.72-3.18 0-1.15.25-2.2.74-3.14a5.64 5.64 0 0 1 5.04-3.06 4.5 4.5 0 0 1 3.42 1.4V9.55l3.49-.75Zm0 0',
        'vw' => 28
    ],
    'dashcube' => 'M20.44 6.5 26.8 0v26.69q0 2.31-1.65 3.81T21.13 32H6.88q-2.38 0-4.03-1.5t-1.66-3.8V12.37q0-2.43 1.66-4.15T6.88 6.5h13.56zM9.62 26v.06h14l-3.18-3.31v-7.69q0-.56-.28-1.03t-.75-.75-.97-.28H9.62q-.8 0-1.4.63t-.6 1.43V24q0 .81.6 1.4t1.4.6z',
    'debian' => [
        'p' => 'M248.2 .9c-4 .2-8.1 .4-11.4 1.6l-3.4-.5c5.4-.7 10.9-1.1 16.4-1.5c2.4-.2 4.8-.4 7.2-.6c-2.7 .6-5.7 .8-8.8 .9zm132 244.7c3-7.6 5.5-14 5.2-24.4l-4.3 9c4.4-13.2 4-27.1 3.6-40.4c-.2-6-.3-11.8 0-17.4l-1.8-.5c-1.5-45.2-40.6-93.1-75.3-109.4c-30-13.8-76.1-16.2-97.3-5.8c1.3-1.1 4.2-2 6.8-2.7c3.4-1 6.3-1.8 4.3-3c-19.2 1.9-24.9 5.5-31.1 9.4c-4.6 2.9-9.5 6-20.3 8.7c-3.5 3.4 1.7 2 5.8 .9c4.1-1.1 7.2-1.9-.1 2.4c-3.6 1-6.7 1.3-9.7 1.6c-8.3 .8-15.8 1.6-30.7 17c.8 1.3 3.5-.3 5.4-1.4c2.3-1.4 3.4-2-1.7 4.4c-19.1-2.4-60.3 43.7-69.1 59l4.6 .8c-3.2 8-6.8 14.8-10 20.8c-4.3 8.1-7.9 14.9-8.7 21.3c-.4 5.9-1.2 12.9-2 20.3c-3 27.4-6.8 61.3 3.8 73.6l-1.3 13c.6 1.2 1.1 2.3 1.6 3.5c1.2 2.5 2.3 5 3.9 7.4l-3 .2c7 22.2 10.8 22.5 15.1 22.9c4.4 .4 9.3 .9 18.7 24.2c-2.7-.9-5.5-1.9-9.4-7.2c-.5 4.1 5.8 16.3 13.1 25.8l-3.1 3.6c3.2 5.8 6.1 8.1 8.6 10l0 0c.8 .6 1.5 1.1 2.1 1.7c-11.9-6.5 3.2 13.7 11.9 25.2c2.5 3.4 4.5 6 5.1 7l2.4-4.2c-.3 6.1 4.3 13.9 13.1 24.7l7.3-.3c3 6 14 16.7 20.7 17.2l-4.4 5.8c8.1 2.6 10.3 4.3 12.7 6.2c2.6 2.1 5.4 4.3 16.1 8.1l-4.2-7.4c3.5 3 6.2 5.9 8.8 8.7c5.2 5.6 9.9 10.7 19.8 15.4c11.2 3.9 17.1 4.8 23.6 5.9c5.4 .8 11.2 1.8 20.8 4.5c-2.2-.1-4.4-.1-6.7-.2l0 0h0c-16.3-.4-34-.8-46.4-5.2C107.8 480.5 19.5 367.2 26 250.6c-.6-9.9-.3-20.9 0-30.7c.4-13.5 .7-24.8-1.6-28.3l1-3.1c5.3-17.4 11.7-38.2 23.8-62.8l-.1-.2 0-.1 0 0 0 0c.4 .4 3.4 3.4 8.8-5.8c1.3-2.9 2.5-5.8 3.8-8.8l0 0c2.5-6.1 5.1-12.3 8.4-17.9l2.6-.6c1.7-10.1 17-23.8 29.8-35.2l0 0c6.1-5.5 11.6-10.4 14.7-14.1l.7 4.4c17.1-16 45-27.7 66.1-36.6c4.8-2 9.3-3.9 13.3-5.7c-3.4 3.8 2.2 2.7 10 1c4.8-1 10.4-2.1 15.3-2.4c-1.3 .7-2.6 1.4-3.9 2.1l0 0 0 0c-2.7 1.4-5.4 2.8-8 4.6c8.3-2 11.9-1.4 16-.8c3.5 .6 7.3 1.2 14.6 .2c-5.6 .8-12.3 3-11.2 3.8c7.9 .9 12.8-.1 17.2-1c5.6-1.1 10.4-2.1 19.5 .9l-1-4.8c7.5 2.7 13.1 4.4 18 5.9c10 3 17.6 5.3 34.2 14.1c3.2 .2 5.3-.5 7.4-1.2c3.6-1.1 7-2.2 15.3 1.2c.5 .8 .7 1.5 1 2.1c1 2.6 1.7 4.6 14.6 12.2c1.8-.7-3.1-5.1-7-8.7l-.2-.1c32.3 17.3 67.5 54.1 78.2 93.6c-6-11.1-5.2-5.5-4.3 .5c.6 4 1.2 8.1-.2 7.5c4.5 12.1 8.1 24.5 10.4 37.4l-.8-2.9c-3.3-11.8-9.6-34.5-20-49.6c-.4 4.4-2.9 3.9-5.3 3.5c-3.3-.6-6.3-1.2-1.9 12.6c2.6 3.8 3.1 2.4 3.5 1.1c.5-1.5 .9-2.8 4.7 5.2c.1 4.3 1.1 8.5 2.2 13.3l0 0 0 0 0 0 0 0 0 0 0 0c.7 3 1.5 6.2 2.1 9.8c-1.1-.2-2.3-2.2-3.4-4.2l0 0 0 0c-1.4-2.4-2.8-4.7-3.7-3.2c2.4 11.5 6.5 17.4 8 18.3c-.3 .6-.6 .7-1.1 .7c-.8 0-1.8 .1-1.9 5.3c.7 13.7 3.3 12.5 5.3 11.6c.6-.3 1.2-.6 1.7-.4c-.6 2.5-1.6 5.1-2.7 7.9l0 0c-2.8 7.1-6 15.4-3.4 26.1c-.8-3.1-2.1-6.3-3.3-9.3l0 0 0 0c-.5-1.3-1.1-2.6-1.5-3.8c-.6 4.8-.3 8.2-.1 11.3c.4 5.3 .7 10-3 19.9c4.3-14.2 3.8-26.9-.2-20.8c1 11-3.8 20.5-8.1 29.1c-3.6 7.1-6.8 13.5-5.9 19.3l-5.2-7.1c-7.6 11-7 13.3-6.5 15.6c.5 1.9 1 3.8-3.4 10.8c1.7-2.9 1.3-3.6 1-4.2c-.4-.8-.7-1.5 1.7-5.1c-1.6 .1-5.5 3.9-10.1 8.5l0 0c-3.9 3.9-8.5 8.4-12.8 11.8c-37.5 30.1-82.3 34-125.6 17.8l0 0c.2-1-.2-2.1-3.1-4.1c-36.8-28.2-58.5-52.1-50.9-107.5c2.2-1.7 3.7-6.2 5.6-11.6c2.9-8.4 6.5-18.9 14.3-23.9c7.8-17.3 31.3-33.3 56.4-33.7c25.6-1.4 47.2 13.7 58.1 27.9c-19.8-18.4-52.1-24-79.7-10.4c-28.2 12.7-45 43.8-42.5 74.7c.3-.5 .7-.7 1-.9c.6-.5 1.2-.9 1.5-3.4c-.9 60.2 64.8 104.3 112.1 82l.6 1.3c12.7-3.5 15.9-6.5 20.3-10.7c2.2-2.1 4.7-4.5 9-7.4c-.3 .7-1.3 1.7-2.4 2.7c-2.2 2.1-4.6 4.5-1.6 4.6c5.3-1.4 20.1-14.8 30.2-23.8l0 0 0 0c1.6-1.4 3-2.7 4.3-3.9c2-4.3 1.6-5.7 1.3-7.1c-.4-1.6-.8-3.3 2.4-9.6l7.3-3.7c1-2.8 2-5.4 2.9-7.8l0 0zM233.1 321.9a.9 .9 0 1 0 -1.7 0 .9 .9 0 1 0 1.7 0zm-.2 .5l-.2 .4c-.3 .7-.6 1.4-.3 2.4c-12.2-5.7-23.4-14.3-32.6-24.9c4.9 7.1 10.1 14.1 17 19.5c-6.9-2.3-15.2-11.9-21.7-19.4l0 0 0 0c-4.3-5-7.9-9.1-9.7-9.5c19.8 35.5 80.5 62.3 112.3 49c-14.7 .5-33.4 .3-49.9-5.8c-6.3-3.2-14.6-9.6-14.9-11.8zM237.5 7c3.8 .6 7.3 1.2 6.7 2.1c5-1.1 6.1-2.1-9-2.5c.8 .1 1.6 .3 2.4 .4zm92.2 208.4c-1 3.9-1.8 1.4-2.7-1.2c-.5-1.5-1.1-3.1-1.7-3.4c1.4-5.8 5.4-10.7 4.4 4.6zm-6.8 21.2c-1.3 7.9-5 15.5-10.1 22.5c.2-2-1.2-2.4-2.6-2.8c-2.9-.8-5.9-1.6 5.6-16.1c-.5 2-2.3 4.9-4 7.7l0 0c-3.6 5.9-6.7 11 4 4.3l1-1.8c2.6-4.5 5-8.8 6-13.8zM280 276.6c-11.1-1.7-21.2-6-12.7-6.1c7.1 .6 14.1 .6 21-1.1c-2.5 2.4-5.2 4.8-8.3 7.2zM244.2 9.1l-.2 .4-3 .3 3.2-.7zm-69.5 273c3.7 7.2 6.5 11.7 9.1 15.9c2.3 3.7 4.4 7.1 6.8 11.7c-5.2-4.3-8.9-9.8-12.8-15.5c-1.4-2.1-2.8-4.2-4.4-6.2l1.2-5.9zm7.3-10c1.7 3.4 3.3 6.7 5.9 9.5l2.6 7.7-1.3-2.1c-3.2-5.3-6.3-10.6-8-16.7l.8 1.6zm239.1-41.2c-2.3 17.4-7.7 34.6-16 50.3c7.6-14.9 12.5-30.9 14.8-47.2l1.2-3.1zM35.4 109.6c0 .3 0 .5 .1 .7c0-.2 0-.5-.1-.7zm.1 .7c.3 1.2 1.4 .9 2.4 .6c1.9-.5 3.6-.9-.1 7.6c-2.4 1.7-3.8 2.8-4.6 3.4c-.6 .4-.8 .6-.8 .6c0 0 .1-.2 .3-.5c.8-1.4 3.4-5.5 2.9-11.7zm-10.2 42c-.7 3.7-1.5 7.9-3.4 13.9c.2-1.9 0-3.5-.2-4.9c-.4-3.4-.8-6.3 4.3-12.9c-.3 1.2-.5 2.5-.7 3.8z',
        'vw' => 448,
        'vh' => 512
    ],
    'deezer' => [
        'p' => 'M28.21 15.3H36v-4.55h-7.79Zm0-10.87v4.54H36V4.43Zm0 17.19H36v-4.55h-7.79ZM0 27.94h7.79V23.4H0Zm9.4 0h7.79V23.4H9.4Zm9.41 0h7.78V23.4h-7.78Zm9.4 0H36V23.4h-7.79Zm-9.4-6.32h7.78v-4.55h-7.78Zm-9.4 0h7.78v-4.55H9.4Zm0-6.33h7.78v-4.54H9.4Zm0 0',
        'vw' => 36
    ],
    'delicious' => 'M27.88 4.25q.12.31.12.75v22q0 1.25-.88 2.13T25 30H3q-.38 0-.75-.06-.69-.2-1.25-.7L.75 29q-.19-.25-.38-.56t-.28-.7T0 27V5q0-1.25.88-2.13T3 2h22q.38 0 .75.12.19 0 .34.07t.35.18l.25.16q.13.09.19.16.25.18.43.43.2.13.32.38t.18.5q.07.19.07.25zM26 27V16H14V4H3q-.44 0-.72.28T2 5v11h12v12h11q.44 0 .72-.28T26 27z',
    'deploydog' => 'M23.88 8.5h3.25v15h-3.25v-1.31q-.88 1.06-2.29 1.25t-2.34-.32q-1.25-.56-2-1.75t-.75-2.75q0-1.43.72-2.65t1.97-1.78q.87-.5 2.31-.35T23.88 15V8.5zm-4 10.13q0 .87.56 1.46t1.47.6 1.5-.63.6-1.43-.6-1.44-1.47-.63q-.63 0-1.1.32t-.72.78-.25.97zM11.74 8.5H15v15h-3.25v-1.31q-.81 1.06-2.25 1.25t-2.31-.32q-.82-.37-1.44-1.03t-1-1.56-.38-1.9q0-1.44.72-2.66t2.03-1.78q.88-.5 2.29-.35T11.75 15V8.5zm-4 10.13q0 .87.6 1.46t1.46.6 1.47-.63.6-1.43q0-.5-.25-.97t-.75-.78-1.07-.32q-.87 0-1.47.63t-.59 1.44zM28 6H4q-.81 0-1.4.6T2 8v16q0 .81.6 1.4T4 26h24q.81 0 1.4-.6T30 24V8q0-.81-.6-1.4T28 6zm0-2q1.63 0 2.81 1.19T32.01 8v16q0 1.63-1.2 2.81T28 28.01H4q-1.63 0-2.81-1.2t-1.2-2.8V8q0-1.63 1.2-2.82T4 4h24z',
    'deskpro' => [
        'p' => 'm12.88 32 1.93-2.38q1.44-.06 2.32-.43.43-.25.84-.6t.66-.68.43-.75.25-.72.13-.6.06-.44v-.21q0-.26-.87-1.26-.2-.18-.28-.3t-.29-.35-.37-.4h4.94q0 3.18-.63 4.74-.56 1.31-2.19 2.66t-3.74 1.65q-.7.07-3.2.07zm12.5-10.5h-7.44L19 19.56h11q-1.69 1.94-4.63 1.94zm-2.7-14v10.56h-4.62L16 21.5h-.38q-.93 0-1.62-.37t-1.06-.85-.6-1.1-.28-.9-.06-.47q-.06-.62-.69-.56t-.56.69q.19 2.19 1.62 3.56H6.7q-.76 0-1.79-.37T2.68 20 .73 17.78 0 14.38V1.06Q0 .06.93 0h5.2l.59.03q.4.03 1.03.25t1.06.85.44 1.62q0 .63.62.63t.63-.63q0-1.69-1-2.75h5.25q2.37 0 4.06.78t2.47 2.06 1.1 2.41.3 2.25zm-8.3 3.5 4 1.88q.43-1.38-.32-2.35t-1.87-.87T14.37 11z',
        'vw' => 30
    ],
    'dev' => 'M7.5 13.03q.38.28.38.78v4.38q0 .56-.38.81t-.75.25H5.69v-6.5h1.06q.38 0 .75.28zM25.25 2q1.13 0 1.94.81T28 4.75v22.5q0 1.13-.81 1.94t-1.94.81H2.75q-1.13 0-1.94-.81T0 27.25V4.75q0-1.12.8-1.94T2.75 2h22.5zM9.62 18.19v-4.44q0-1.06-.71-2t-2.22-.94h-3v10.31h2.94q1.56 0 2.28-.9t.72-2.03zm6.32-5.5V10.8H12q-.56 0-.9.38t-.35.94v7.8q0 .5.38.88t.93.32h3.88V19.3h-3.31v-2.37h2v-1.88h-2V12.7h3.3zm6.5 7.18 2.37-9.06h-2.06l-1.81 7.13-1.88-7.13h-2l2.38 9.06q.5 1.2 1.4 1.29t1.6-1.29z',
    'deviantart' => [
        'p' => 'm20 5.81-6.13 11.2.44.62h5.7v8H9.93l-.81.56-2.76 5.25-.56.56H0v-5.8l5.8-11.26-.44-.56H0v-8h9.75l.87-.57 2.7-5.25.55-.56H20v5.81z',
        'w' => 0.625,
        'vw' => 20
    ],
    'dhl' => [
        'p' => 'm14.88 18.81 1.37-1.87h3.69l-1.38 1.87h-3.69zM0 17.7h5.38l-.25.37H0v-.37zm10.81-.75h4.25Q13.7 18.8 11.5 18.8H5.25l2.56-3.44h3.2q.43 0 .18.32-.06.06-.69.94-.25.3.31.3zm19.82-.44h-3.88l2.5-3.31h3.88zm-24.7.44-.24.37H0v-.37h5.94zm6.94-1.69q.32-.38-.12-.38h-7l1.31-1.68h7.32q1.3 0 1.8.78t.07 1.4l-.87 1.13h-2.82q-.56 0-.3-.31.05-.13.18-.29t.25-.34.19-.31zM0 18.81v-.37h4.88l-.32.37H0zm20.19 0 1.37-1.87h3.7l-1.38 1.87h-3.7zm13.87 0 .32-.37H40v.37h-5.94zm1.38-1.87H40v.37h-4.8zm-.81 1.12.24-.37H40v.37h-5.38zM24.37 13.2H28l-2.44 3.31h-9l2.5-3.31h3.63l-1.38 1.87h1.63zm2.07 3.75h8.31l-1.37 1.87H27.5q-1.06 0-1.47-.43t.03-.88z',
        'w' => 1.25,
        'vw' => 40
    ],
    'diaspora' => [
        'p' => 'M14.19 22.19q-.06 0-2.81 3.75l-2.76 3.75q-.06 0-2.74-1.9t-2.7-1.95 2.73-3.96 2.71-4-2.03-.75-4.25-1.41T0 14.94q-.06 0 .94-3.19t1.03-3.22 2.28.72 4.53 1.5l2.28.75q.06 0 .06-.13l.13-9.3Q11.31 2 14.53 2t3.25.03.06 1.03.1 2.44.1 2.87.06 2.44.09 1 2.22-.75 4.31-1.47l2.16-.72q.06.07 1.03 3.25l.97 3.2-4.38 1.5-4.44 1.56q0 .06 2.6 3.84t2.6 3.81-2.7 2L19.88 30q-.07 0-2.85-3.9t-2.84-3.91z',
        'vw' => 29
    ],
    'digg' => 'M5.13 10.75V6H8.3v15.69H0V10.75h5.13zm0 8.38V13.3H3.19v5.82h1.93zm18.56-8.38H32V26h-8.3v-2.56h5.12v-1.75H23.7V10.75zm5.12 8.38V13.3h-1.93v5.82h1.93zm-14.75 2.56V10.75h8.32V26h-8.32v-2.56h5.13v-1.75h-5.13zm3.19-8.38v5.82h1.94V13.3h-1.94zM9.56 6h3.25v3.19H9.56V6zm0 4.75h3.25v10.94H9.56V10.75z',
    'digital-ocean' => 'M5.44 30.13V25.5h4.62v4.63H5.44zm-3.88-8.44h3.88v3.81H1.56v-3.8zm29.19-10.63q1.31 4 .53 7.7t-2.84 6.52-5.38 4.53-7.06 1.7v-6q3.19 0 5.69-1.88t3.37-4.79-.19-6.34q-.68-1.87-2.09-3.28t-3.28-2.1q-3.44-1.06-6.35-.18t-4.78 3.4T6.5 16h-6q0-3.75 1.69-7.06t4.53-5.38T13.25.72t7.69.53q3.5 1.12 6.1 3.72t3.71 6.1zM16 25.5h-5.94v-5.94H16v5.94z',
    'discord' => 'M18.56 15.19q0 .75-.47 1.28t-1.15.53q-.44 0-.82-.25t-.59-.66-.22-.9q0-.38.13-.69t.34-.56.53-.38.63-.12q.68 0 1.15.5t.47 1.25zm-7.43-1.75q.68 0 1.15.5t.47 1.25-.47 1.28-1.15.53-1.16-.53-.47-1.28.47-1.25 1.15-.5zM28 3.3V32q-3.63-3.2-7.44-6.76l.88 3H3.24q-1.3 0-2.27-.97T0 24.94V3.3Q0 1.94.97.97T3.25 0h21.5q1.31 0 2.28.97T28 3.3zm-4.56 15.13q0-2.25-.57-4.6t-1.18-3.53l-.57-1.18q-.62-.5-1.34-.85t-1.25-.5-.97-.25-.69-.1h-.25l-.18.26q1.18.37 2.18.9t1.38.85l.37.31q-2.75-1.56-6-1.63T8.44 9.25l-.94.5q1.3-1.25 4.19-2.13l-.13-.18q-2.19 0-4.5 1.68-.25.5-.62 1.32T5.4 13.72t-.65 4.72q.12.31.47.69t1.6 1.06 2.8.69q.63-.7 1.07-1.32-.88-.25-1.6-.71T8.12 18l-.25-.31.29.15.28.16.12.06q2.13 1.2 4.72 1.4t5.28-.84q.94-.3 1.81-.93-.8 1.31-2.87 1.93l1.06 1.25q1 0 1.85-.28t1.37-.62.94-.69.53-.6z',
    'discourse' => 'M14.13 2q5.74 0 9.8 4.13T28 15.96t-4.06 9.87T14.12 30H0V15.75q0-5.69 4.19-9.72T14.13 2zM14 24q3.31 0 5.66-2.34T22 16t-2.34-5.66T14 8t-5.66 2.34T6 16q0 2.06.94 3.81l-1.44 4.7 5.13-1.26Q12.19 24 14 24z',
    'dochub' => [
        'p' => 'M24.88 10H16V1.25zM19 12h6v7.88Q25 32 11.81 32H0V0h11.8q1.13 0 2.19.19v5.25q-.94-.13-2.06-.13H6v21.07h5.94q7.06 0 7.06-6.25V12z',
        'w' => 0.781,
        'vw' => 25
    ],
    'docker' => [
        'p' => 'M21.88 14.75h-4.13v-3.69h4.13v3.69zm0-12.75v3.81h-4.13V2h4.13zm4.87 9.06v3.7h-4.13v-3.7h4.13zM17 6.56v3.75h-4.13V6.56H17zm4.88 0v3.75h-4.13V6.56h4.13zm17.3 6.25.82.57q-.5.93-.5 1.06-1.56 2.87-5.75 2.81-2.56 6.19-7.84 9.47T13.25 30q-6.87 0-10.37-4.12Q1.18 23.8.47 21t-.34-5.25H27.3q1.75 0 3.07-.69-.94-1.5-.82-3.3t1-3.2l.57-.87.87.62q2.31 1.7 2.63 3.94 1.18-.19 2.53-.06t2.03.62zM7.26 11.06v3.7H3.12v-3.7h4.13zm4.88 0v3.7H8v-3.7h4.13zm4.87 0v3.7h-4.13v-3.7H17zm-4.88-4.5v3.75H8V6.56h4.13z',
        'w' => 1.25,
        'vw' => 40
    ],
    'draft2digital' => [
        'p' => 'M30 24.88 21 30v-4H0q5.56-5.81 10.06-10.94.2-.19.57-.62 1.3-1.5 2.03-2.38T14 9.72t.44-2.6q-.25-1.43-1.75-2.15T9.25 5q-.44.19-1.03.69t-.66.87q.7.06 1.25.4t.94.82.6 1.06.22 1.28q0 1.44-1.04 2.5T7 13.7q-1.69 0-2.75-1.31T3.44 9.5t1.13-3q2.06-3.25 6.56-4.19 1.69-.25 2.97-.31t2.53.31 2 .66 1.47.78.97.72.43.4q.94 1 1.5 2.44t.44 3.38-1.19 3.75q-2.12 3.87-6.93 9.37H21v-4.06zm-6.88-1.7v3.38l2.94-1.68zM8.38 10.1q0-.59-.4-1T7 8.7t-1 .4-.44 1 .44 1 1 .41.97-.4.4-1zM5.13 23.75h7.18q5.63-6.13 8-10.38 1-2.06.94-3.65t-.69-2.53L20 6.25Q17.81 4 14.81 3.87q.94.75 1.56 2.25 1.25 3.25-1.8 7.07-3.38 4.12-9.45 10.56z',
        'vw' => 30
    ],
    'dribbble' => 'M16 .5q6.44 0 10.97 4.53T31.5 16t-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5zm10.25 7.13q-.38.5-.97 1.12t-2.4 1.94-4 2.19q.56 1.25 1.12 2.62 4.56-.62 9.25.38-.06-4.63-3-8.25zm-1.5-1.5q-2.44-2.2-5.56-2.97t-6.32-.04q2.7 3.57 4.94 7.75 2.13-.74 3.85-1.93t2.25-1.78.84-1.04zM10.37 4.05Q8.5 4.94 7 6.31T4.47 9.47t-1.4 3.84q5.87.07 12.18-1.62Q13 7.62 10.38 4.06zM2.75 16q0 5.13 3.44 8.88Q7.25 23 8.94 21.25t3.94-3.19T17.44 16q-.44-1.13-1.06-2.25-6.75 2-13.63 1.87V16zm5.13 10.44q2.87 2.31 6.5 2.75t6.8-1q-.8-4.81-2.8-10-3.82 1.25-6.6 3.6t-3.9 4.65zm15.5.56q4.8-3.25 5.68-8.87-3.94-1.25-8.25-.57 1.82 5 2.57 9.44z',
    'dribbble-square' => 'M5.63 14.25q.43-1.94 1.68-3.53t3.07-2.47q1.62 2.25 3.18 5-4.18 1.06-7.93 1zm14.06-4.63q-1.38 1.94-4.5 3.07-1.44-2.69-3.19-5 2.06-.44 4.06.06t3.63 1.88zM8.75 22.76q.5-.94 1.38-1.94t2.3-1.94 3.07-1.5q1.31 3.44 1.81 6.5-2 .88-4.34.6t-4.22-1.72zM14.94 16q-1.5.44-2.94 1.34t-2.53 2.04-1.85 2.37Q5.44 19.25 5.44 16v-.25q4.25.13 8.81-1.19.44.88.69 1.44zm2.19 1q2.8-.44 5.3.38-.56 3.62-3.68 5.75-.44-2.88-1.63-6.13zm-1.25-3q3.24-1.38 4.74-3.38 1.88 2.32 1.94 5.32-2.94-.69-5.93-.32-.38-.87-.75-1.62zM28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22q1.25 0 2.13.88T28 5zm-4 11q0-4.13-2.94-7.06T14 6 6.94 8.94 4 16t2.94 7.06T14 26t7.06-2.94T24 16z',
    'dropbox' => 'm16.5 7.25-8.25 5.31 8.25 5.25-8.25 5.25L0 17.75l8.25-5.25L0 7.25 8.25 2zm-8.25 17.5 8.25-5.31 8.25 5.31L16.5 30zm8.25-7 8.25-5.25-8.25-5.25L24.75 2 33 7.25l-8.25 5.31L33 17.81l-8.25 5.25z',
    'drupal' => 'M20 7.19q.56.31 1.31 1t1.94 2.06 1.97 3.53.78 4.6q0 5.12-3.5 8.37T14.13 30t-8.5-3.4T2 18.13q0-2.32.81-4.44t2-3.44 2.07-2.03T8.3 7.19q2.5-1.38 3.32-2.06 1.19-.88 1.37-3.07 2 2.38 2.94 3 .69.44 2.03 1.07T20 7.19zm.13 20.31q.3-.31.06-.56-.13-.07-.32.06-1.18.88-3.43.88-.94 0-1.75-.22t-1.13-.41l-.31-.25q-.31 0-.47.28t.03.47q1.31 1.13 3.81.94t3.5-1.19zm-3.38-3.25q-.5.19-1.1.69t-.34.87q.19.2.75-.31.88-.69 1.69-.69.56 0 .88.19t.37.4.16.35.21.06q.38-.31.16-.81t-.53-.69q-.31-.25-1.1-.25t-1.15.19zm5.75 1.06q1.44.13 2.25-2.19.94-2.62-.06-3.93-.44-.5-1.2-.5-.62 0-2.52 1.31t-2.6 1.31q-.62 0-1.68-.75t-2.41-1.53-2.66-.78q-1.5 0-2.59.81T7.5 20.94t-.22 2.19 1.1 1.75q1.18.87 2.78.56T14 24.41t2.56-1.47 1.88-.69q.62 0 2.12 1.53t1.94 1.53z',
    'dyalog' => [
        'p' => 'M0 2h10.69q7.12 0 11.44 4.38Q26 10.3 26 16q0 2.63-1 5.13t-2.87 4.5-4.85 3.18-6.6 1.2H0v-4h10.69q5.37 0 8.34-3.04T22 16q0-4-3.12-7t-8.2-3H4v3.44H0V2z',
        'w' => 0.812,
        'vw' => 26
    ],
    'earlybirds' => [
        'p' => 'M19.56 3q.07-.63.82-.75t1.5.19q.06 0 .25.1l.5.24.43.32v.28Q22.5 3.25 22 3.3t-.94.16-.78.16-.53-.1-.19-.53zM7 3.38q-.31-.2.31-.57t1-.44q.82-.3 1.63-.18t.87.75q.07.5-.15.65t-.6.1-.87-.2-1.03-.18T7 3.37zM19.97 10q.4 0 .69.28t.28.72-.28.72-.72.28q-.25 0-.47-.13t-.38-.37-.15-.5q0-.44.31-.72t.72-.28zm-10 0q.4 0 .72.28T11 11t-.31.72-.72.28-.69-.28T9 11t.28-.72.69-.28zm19.9 10.19q-.68 1.69-4-.06-1.25 6.5-7.87 7.75 0 .06.13.06h1.37q2.06.12 6.06-.5-.5-.38-.68-.97t-.1-.97.22-.44l.25.03.56.22.5.5q.25-.37.7-.6t.77-.27.44 0 .12.4-.06.85-.53.93q.5-.12.63-.12.3-.13.56.38.19.37-.38.43-.12 0-2.93.7-2.82.62-7.32.74l.07.75q.12.88-.72 1.06t-1.03-.68q0 .56-.41.8t-.88.04-.46-.9l-.07.3q0 .88-.87.88t-.88-.87q-.37.62-1.06.4t-.63-.97l.13-.75q-1.06 0-1.97.07t-1.4.1l-.5.02q-.5 0-.76-.31t-.25-.69.25-.69.75-.31l.54.03 1.46.1q1 .06 1.94.06-6.12-1.57-7.19-7.38-3.56 1.82-4.25.07-.37-.94.38-1.75t2.87-2.2q.63-.43.94-.68l.13-1.19q-.32-.44-.47-.94t-.22-.84-.13-.97-.18-1.06Q2 11 2 9.3q0-.75.12-.94.07-.06.22-.1t.35-.06.25-.03q.12-.68.62-1.12t1.47-.66 1.63-.28 1.96-.06Q10.88 3.56 13.7 3q-.44-.5-.47-1.13t.1-.97.18-.4.4.06.73.5.5.87q.18-.25.46-.4t.5-.22.29-.03.12.28-.03.65-.4.66q3.37.31 6 3.19 2.12.06 3.37.5t1.44 1.62l.28.03q.22.03.37.07t.16.1q.12.18.12.93-.06 1.63-1.43 1.44 0 .12-.16 1.28t-.4 1.72q.24 1.06.3 2.25.32.25.76.53t.72.47.65.47.6.47.46.4.38.44.22.47.1.47-.13.47zM12.95 10q.06-1-.35-1.6t-1.09-.74-1.6-.22-1.84.06q-2.44.06-2.87.88-.5.93.06 3.4t1.38 2.9q.43.26 2.65.23t2.78-.66q.25-.31.47-1.13t.35-1.68.06-1.44zm3.25 3.63q0-.13-.19-.76t-.34-.9-.41-.35q-.06-.06-.31-.06t-.38.06q-.19 0-.34.29t-.25.56-.22.72-.25.75q.56 0 1 .28t.5.28.22-.16l.4-.4.57-.31zm8.5-5.26q-.38-.8-2.88-.87-1.31-.06-2.4-.03t-1.78.62-.7 1.91q0 .94.26 2.38t.56 1.87q.56.63 2.81.69t2.63-.25q.75-.38 1.4-2.85t.1-3.46z',
        'vw' => 30
    ],
    'ebay' => [
        'p' => 'M37.88 11.88H40L33.81 24h-2.25l1.75-3.38-3.81-7.25q.31.63.31 1.5V19q0 .94.06 1.63H28q-.06-.57-.06-1.25Q26.62 21 24.06 21q-1.81 0-2.81-.81t-1-2q0-.32.06-.57-.37 1.5-1.62 2.44t-2.94.94q-2.44 0-3.69-1.56 0 .56-.06 1.18h-2q.12-1.37.12-2.06V8h2v5q1.25-1.5 3.63-1.5 1.87 0 3.31 1.25t1.44 3.5q0 .56-.13 1.06.63-2.06 4.82-2.19h2.62v-.25q0-2.06-2.56-2.06-2.06 0-2.19 1.38h-2.12q.25-2.7 4.43-2.7 3 0 4 1.63l-.68-1.25h2.3l3.45 6.82zm-22.66 7.74q1.4 0 2.28-.93t.88-2.44-.88-2.44-2.28-.94-2.25.94-.85 2.44.85 2.44 2.25.93zm12.6-3v-.18q-1.5 0-2.5.06-2.94.06-2.94 1.56 0 .44.21.78t.72.6 1.25.25q1.57 0 2.4-.82t.85-2.25zm-25.76.13q.07 1.69 1.35 2.44t2.62.37 1.72-1.43h2.06q-.18 1-.96 1.68t-1.7.94T5.2 21q-2.63 0-3.9-1.28T0 16.38q0-2.25 1.31-3.57T5.2 11.5q2.25 0 3.6 1.19t1.34 3.56v.5H2.06zM8 15.44q-.06-1.63-1.47-2.31t-2.81-.04-1.66 2.35H8z',
        'w' => 1.25,
        'vw' => 40
    ],
    'edge' => 'M1.63 14.25zm28.75 1v3.25H10.75q.06 2.81 2 4.5t4.66 1.97 5.59-.31 4.75-1.97v6.56q-1.56.94-4 1.6t-5.5.65-5.5-.94q-3.25-1.25-5.38-4.31T5.25 19.8q-.13-8.19 7.5-11.75-1.63 2.06-2.13 4.94h11q.2-1.88-.4-3.22T19.69 7.8t-1.9-.97-1.66-.4h-.75q-4.44.12-7.97 2.34t-5.78 5.47q.75-5.81 4.56-9.78t10-3.97q4 0 7.25 1.84t5.12 5.22q1.82 3.25 1.82 7.69z',
    'edge-legacy' => 'm1.6 14.26.03-.03v.03Zm28.8.97c0-2.75-.5-5.28-1.8-7.65C26.02 2.99 21.5.5 16.17.5 7.44.48 2.54 7.07 1.63 14.23 4.28 10.4 8.95 6.64 15.4 6.42c0 0 6.86 0 6.22 6.56h-11a10 10 0 0 1 2.15-4.93c-4.69 2.18-7.61 6-7.55 11.77a12.12 12.12 0 0 0 7.55 10.75c5.21 1.99 12.05.45 15.01-1.34v-6.52c-5.05 3.53-16.93 3.8-17.02-4.23H30.4Zm0 0',
    'elementor' => 'M26.63 2q.56 0 .96.4t.41.98v25.25q0 .56-.4.96t-.98.41H1.38q-.56 0-.96-.4T0 28.61V3.38q0-.56.4-.96T1.38 2h25.25zM10.25 22.25V9.75h-2.5v12.5h2.5zm10 0v-2.5h-7.5v2.5h7.5zm0-5v-2.5h-7.5v2.5h7.5zm0-5v-2.5h-7.5v2.5h7.5z',
    'ello' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm9 17.81q.13-.31-.06-.65t-.57-.4q-.37-.13-.68.05t-.38.57q-.69 2.44-2.72 4.03t-4.6 1.6-4.59-1.6-2.72-4.03q-.06-.38-.37-.53t-.69-.1q-.31.06-.5.38t-.12.68q.81 3.07 3.31 5.04t5.69 1.96 5.69-1.96 3.3-5.04z',
    'ember' => [
        'p' => 'M40 15.94q0 .25-.06.5t-.2.31l-.05.06-.5.32q-.3.19-1.1.43t-1.34.13q-.38-.13-.66-.44t-.4-.69-.2-.75-.12-.62v-.25h-.12L34 18.56q-.07.2-.19.47t-.47.7-.69.24-.53-.44-.19-.53v-.19q-.12.13-.4.35t-1.16.56-1.87.22q-.88-.06-1.5-.47t-.88-.78l-.19-.37q-1.3.87-4.93 1.62-.75.06-1.44-.16t-1.07-.4l-.3-.25-1.13.62q-1.38.63-1.94-.31-.25-.44-.28-1.44t.03-1.81l.06-.75h-.06q-.81 1.62-1.25 3.12-.19.63-.4.97t-.41.35h-.13q-.94.06-1.19-.7-.06-.5 0-1.33t.25-1.6l.13-.81h-.13l-.75 1.5q-.62 1.5-1 2.15t-.62.79-.69.06q-.31-.06-.5-.35T8 19.13l-.06-.25-1 .56q-.06 0-.25.1t-.66.21-.9.16-.97-.07-.85-.34q-1-.75-1.25-1.87Q1 17.56.3 17.25q-.3-.12-.3-.37t.18-.5l.12-.25q.25-.32.63 0 .19.12.47.18t.53 0h.19q.12-1.18.5-2 .87-2 2.56-2.06 1.31 0 2 .94.5 1.31-.28 2.37t-1.78 1.5l-1.06.5v.07l.06.18.12.22.32.16.56.06q.5 0 1.25-.31t1.19-.63l.5-.3q.06-.7.4-2.38T9 12.56q.07-.18.29-.28t.4-.1h.19q.56-.05.62.5.07.26-.03 1t-.22 1.38l-.12.57.06.12 1.32-2.25q.06-.19.21-.37t.63-.53.9-.29q.32 0 .41.35t.03.84-.15.97-.22.84l-.13.32.13.12.06-.12 1.75-2.82.16-.18q.1-.12.4-.28t.57-.1q.56.19.62.88l-.44 3.31v1.37q.13.32.32.38.3 0 .62-.16t.5-.34l.19-.13q.44-3.62.75-6.56.06-.62.19-1.22t.28-1.1.34-.93.35-.75.28-.53.18-.34L20.5 6q.44-.62 1.5-1.06 1.37-.5 1.81 1.44.69 2.8-2.31 6.62.87-.87 1.69-.37 1.12.75 1.25 2t-.32 2.18l-.44.88q.32-.06.82-.35t.87-.53l.38-.25v-.43q.44-2.32 2.31-3.75.25-.2.9-.35t1.38.22.78 1.63q.07 1.18-.78 2.12t-1.65 1.25l-.88.31.13.25.46.29.97.03q.38 0 .78-.2t.72-.37.6-.37l.47-.32.12-.12q.06-.75.44-2.28t.5-1.97q.13-.25.63-.28t.56.15l.06 2.32h.06q.13-.31.29-.63l.28-.56.28-.44.22-.25.06-.12q.12-.07.22-.16t.4-.22.54-.06.3.53.04.9l-.06.44q-.25 1.94.62 2 .38.07.85-.09t.71-.34l.25-.2q.63-.24.7.45zm-36.13.31q.38.13 1.25-.5.63-.44.79-1.03t-.22-.9q-.25-.13-.66 0t-.78.8-.38 1.63zM20.95 8.5q-1 3.44-1.07 4.44l.32-.5q.19-.31.65-1.19t.82-1.6.6-1.53.12-1.18Q22.13 5.56 21 8.3zm1.4 8.66q.22-.6.28-1.1t-.03-.81-.18-.53-.22-.35l-.07-.06q-.06-.06-.15-.1t-.38-.06-.53.07-.56.43-.47.91-.28 1.1-.13.8v.91q.07.07.22.1t.53.06.7-.06.68-.4.6-.91zm5.28-.91q1 .06 1.63-.88.31-.43.34-.84t-.09-.6l-.13-.18q-.56-.69-1.25.56-.3.57-.43 1.07t-.07.68v.2z',
        'w' => 1.25,
        'vw' => 40
    ],
    'empire' => 'M18 3.38q-1-.2-2.13-.2V2q3.63.12 6.66 1.9t4.9 4.79l-1 .56q-.62-1-1.24-1.69l-1.13.94Q21.5 5.62 17.7 4.87zM3.31 20.13q.44 1.06.88 2l-1 .56Q1.5 19.56 1.5 16t1.69-6.63l1 .57q-.5.94-.88 1.94l1.44.43q-.62 1.75-.62 3.7 0 1.87.62 3.68zm10 7-.31 1.5q1.19.18 2.13.18v1.2q-3.63-.13-6.7-1.92t-4.87-4.78l1-.56q.5.81 1.25 1.69l1.13-1q2.56 2.94 6.37 3.69zM5.81 7.55q-.62.7-1.25 1.7l-1-.57q1.88-3 4.9-4.78T15.14 2v1.18q-1.13 0-2.13.19l.31 1.5Q9.5 5.63 6.94 8.5zM25.2 24.44q.75-.88 1.25-1.69l1 .56q-1.81 3-4.88 4.78T15.88 30v-1.19q.93 0 2.12-.19l-.31-1.5q3.81-.75 6.37-3.68zM31 16q0 6.44-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5t10.97 4.53T31 16zm-.75 0q0-6.13-4.31-10.44T15.5 1.25 5.06 5.56.75 16t4.31 10.44 10.44 4.31 10.44-4.31T30.25 16zm-2.44-6.69q1.7 3.13 1.7 6.7t-1.7 6.62l-1-.57q.5-.87.88-1.93l-1.44-.44q.63-1.82.63-3.7 0-1.93-.63-3.68l1.44-.44q-.38-1-.88-2zM16.38 20.44l.8 4.19q-.87.18-1.68.18t-1.69-.18l.81-4.2q-1.56-.3-2.5-1.43l-3.25 2.75q-1.12-1.31-1.68-2.88l4-1.37q-.2-.75-.2-1.5t.2-1.5l-4-1.38q.56-1.56 1.68-2.87L12.12 13q1-1.13 2.5-1.44l-.8-4.19q.74-.18 1.68-.18.87 0 1.69.18l-.82 4.2q1.5.3 2.5 1.43l3.25-2.75q1.13 1.31 1.7 2.88l-4 1.37q.18.75.18 1.5t-.19 1.5l4 1.38q-.56 1.56-1.68 2.87L18.88 19q-1 1.13-2.5 1.44z',
    'envira' => 'M0 2q5.38 0 9.53.88t6.66 2.25 4.15 3.3 2.32 3.85.87 4.07.03 3.68-.4 3.03-.22 1.82L28 30h-1.63L22 25.56q-.38 0-1.22.16t-1.56.25-1.75.19-2.03-.03-2.16-.4-2.34-.92-2.38-1.56q-1.62-1.31-2.72-2.75T4 17.03t-1.2-3.81-1.24-5.16T0 2zm5 2.88q-1.44-.63-1.5-.5t.56.62l.57.44q1.37.94 2.5 2.31T8.72 10t1.6 2.94q4.43 8.56 9 10.5 2.37 1 2.37.94.06-.07-.75-.63-.81-.5-1.25-.87-2.31-1.7-5.69-8.13-.87-1.62-1.25-2.28t-1.28-2.03-1.69-2.16-2.03-1.75T5 4.88z',
    'erlang' => [
        'p' => 'M5.44 3.38q-4.07 4.93-4.07 12 0 8.06 4.88 13.24H0V3.38h5.44zm14.87.56q2.2 0 3.6 1.44t1.47 3.75H14.75q.19-2.25 1.78-3.72t3.78-1.47zm14.44-.57H40V28.7h-5.06q2-2.13 3.62-5.07l-6-3q-4.06 6.57-9.5 6.63-4.44-.06-6.56-3.38T14.37 15H37.7v-1.5q.18-5.94-2.94-10.13z',
        'w' => 1.25,
        'vw' => 40
    ],
    'ethereum' => [
        'p' => 'M19.5 16.31 10 22.13.5 16.3 10 0zM10 23.94l9.5-5.75L10 32 .5 18.2z',
        'w' => 0.625,
        'vw' => 20
    ],
    'etsy' => [
        'p' => 'M24 21.75q-.63 3.75-1 8.25-10.69-.38-23 0v-1.63l.84-.15q.53-.1.79-.13t.62-.12.53-.16.38-.18.3-.22.2-.28.12-.41.03-.53q.32-13 0-20.13 0-.25-.03-.47t-.1-.37-.15-.28l-.19-.25-.3-.19-.38-.12-.5-.13-.6-.12-.75-.16L0 3.8V2.25q16.19.5 22.69-.25l-.5 7.88h-1.44L20.53 9l-.25-.9-.25-.85-.28-.84-.31-.7-.38-.62-.47-.44-.6-.3-.68-.1H8.75q-.44 0-.53.12t-.1.5v10.25l5.5-.18q.32 0 .5-.04l.38-.06.34-.16.25-.18.19-.32.16-.37.15-.53.16-.66.19-.81.25-.94h1.56q-.19 4.69-.19 5.06t.13 4.94h-1.63q-.06-.31-.22-.97t-.22-.9-.19-.7-.25-.55-.37-.35-.53-.28-.72-.12q-1.31-.13-5.5-.13v8.69q0 2.44 2.81 2.44h5.57q.74 0 1.34-.13t1-.19.84-.47.63-.56.56-.9.53-1.04.63-1.4.65-1.56H24z',
        'w' => 0.75,
        'vw' => 24
    ],
    'evernote' => [
        'p' => 'M7.55 8.26c.1 1.4-1.1 1.35-1.35 1.35-4.3 0-4.6-.06-5.22.21-.04.02-.05 0-.03-.02l6.79-6.9c.02-.02.03 0 .02.03-.27.62-.2.94-.2 5.33Zm4.94 19.25a3.5 3.5 0 0 1 3.28-4.79c1.1 0 1.41 1.46.5 1.97-.39.2-1.56.1-1.57 1.2 0 1.07 1.23 1.56 1.95 1.56a2.85 2.85 0 0 0 2.85-2.86c0-.73-.49-2.96-2.97-3.46-.48-.1-4.07-.4-4.27-3.16-.24 1.06-1.1 3.97-2.7 4.32-.54.12-4.35.48-7.06-2.3 0 0-1.16-.95-1.76-3.62-.2-.99-.58-2.48-.7-3.88 0-1.12.7-1.9 1.57-2.01 5.06 0 5.62.14 6.31-.49.62-.58.49-.97.49-6.42.06-.52.49-1.93 3.34-1.51.38.05 2 .26 2.34 1.92l4.02.7c1.28.22 4.43.43 5.03 3.61a51.9 51.9 0 0 1 .5 14.9c-1 7.15-6.95 6.81-6.95 6.81-1.18-.02-3.39-.59-4.2-2.49Zm5.06-12.8c-.07.12-.14.38.05.44.88.3 2.49.42 2.87.34.2-.01.19-.27.16-.41-.23-1.37-2.56-1.66-3.08-.37Zm0 0',
        'vw' => 24
    ],
    'expeditedssl' => 'M15.5 2.69q5.5 0 9.4 3.9T28.82 16t-3.9 9.4-9.41 3.91-9.4-3.9T2.18 16t3.9-9.4 9.41-3.91zM9.44 11h-.06v1.69q0 .56.56.56h1.12q.57 0 .57-.56V11q0-1.44.78-2.44t1.94-1.25 2.28 0 1.93 1.25.82 2.44v1.69q0 .56.56.56H21q.57 0 .57-.56V11q0-2.5-1.79-4.28T15.5 4.94q-1.62 0-3.03.81t-2.22 2.22-.8 3.03zm14.94 12.75v-8.88q0-.43-.35-.74t-.78-.32H7.75q-.44 0-.78.31t-.34.76v8.87q0 .44.34.78t.78.34h15.5q.44 0 .78-.34t.34-.78zm-15.5-8.56v8.31q0 .25-.32.25H8q-.25 0-.25-.25v-8.3q0-.31.25-.31h.56q.31 0 .31.3zm8.8 3q0 1.31-1.05 1.94v1.93q0 .57-.57.57h-1.12q-.57 0-.57-.57v-1.93q-1.12-.63-1.12-1.94 0-.88.66-1.53T15.5 16t1.56.66.63 1.53zM15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm0 29.88q5.94 0 10.16-4.22T29.87 16 25.66 5.84 15.5 1.63 5.34 5.83 1.13 16t4.21 10.16 10.16 4.21z',
    'facebook' => [
        'p' => 'M13.5 5.31q-1.13 0-1.78.38t-.85.94-.18 1.43V12H16l-.75 5.69h-4.56V32H4.8V17.7H0V12h4.8V7.5q0-3.56 2-5.53T12.13 0q2.68 0 4.37.25v5.06h-3z',
        'vw' => 17
    ],
    'facebook-f' => 'facebook',
    'facebook-messenger' => [
        'p' => 'M14 2q5.75 0 9.81 3.78t4.07 9.16q0 4.25-2.6 7.6t-6.75 4.68-8.53.16L5.31 30v-4.94q-2.12-1.62-3.4-3.72T.3 17.1t.07-4.34 1.56-4.1 2.87-3.4 4.1-2.38T14 2zm1.44 17.38 7.56-8-6.87 3.8-3.57-3.74L5 19.5l6.88-3.87z',
        'vw' => 28
    ],
    'facebook-square' => 'M28 5v22q0 1.25-.88 2.13T25 30h-5.31V18.94h3.75l.56-4.25h-4.31v-2.7q0-.68.12-1.08t.63-.7 1.31-.27H24V6.19q-1.31-.2-3.25-.2-2.5 0-3.97 1.48t-1.47 4.1v3.12H11.5v4.25h3.81V30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.87-2.13T3 2h22q1.25 0 2.12.87T28 5z',
    'fantasy-flight-games' => 'M16 2.06 29.94 16 16 29.94 2.06 16zM5.5 16q1.38 1.31 2.56 2.56l3.88-3.81 1.15 1.16 1.16 1.22q.94-.94 2.25-2.2l-2.25-2.24q-.13-.2 0-.32.75-.75 1.75-.84t1.81.53q.5.38 3.07 2.75 1.56-1.62 2.06-2.19.06 0 .03-.1l-.03-.08q-2.13-2.38-3.88-3.7-1.56-1.24-3.59-1t-3.9 2.13L5.5 16zm14.69 6.31L26.5 16q-.25-.25-.53-.5t-.53-.5l-1.5-1.5-1.19 1.16-1.4 1.37-1.22 1.22L18.9 16q-.78-.81-1.16-1.13l-.12.07-2 2q-.13.12.06.31l2 2q.19.25 0 .44-.81.75-1.84.78t-1.91-.66q-.38-.3-2.75-2.62l-2.25 2.25q3.31 3.37 4.56 4.19 1.32.87 3.22.6t3.47-1.92zM16 0l16 16-16 16L0 16zM1 16l15 15 15-15L16 1z',
    'fedex' => [
        'p' => 'M36.63 17.75 40 21.56h-4L34.56 20l-1.37 1.56H19.94v-.75h-.07q-.75 1.2-2.12 1.2-1.5 0-2.56-1.07t-1.2-2.5H8.63q0 1.12 1.1 1.5t1.78-.63h2.62q-.81 2.07-2.81 2.57t-3.66-.66T6 17.88q0-.63.12-1.13H3.06v4.81H0v-11.5h6.8v2.57H3.06v1.62H6.5v1.5q.75-1.31 2.34-1.81t3.16.22 2.12 2.47q.38-1.25 1.32-2.03t2.25-.79q1.37 0 2.18.94h.07V10h9.43v3.07h-3.5v1h7.44l1.38 1.5 1.37-1.5h3.88zm-27.94-1h2.94q-.25-1.19-1.44-1.22t-1.5 1.22zm9.6 3.06q1.02 0 1.43-1t-.03-1.97-1.38-.97q-1 0-1.43.97t-.04 1.97 1.44 1zM28.74 21v-1.88h-3.5v-2.75h3.5v-1.75H25.3V12.5h3.5v-1.94h-5.94V21h5.88zm-2.81-2.5h3.5v2.88l3.12-3.57-3.12-3.56V17h-3.5v1.5zm8.62.63 1.7 1.87h2.5l-2.88-3.25 2.8-3.13H36.3l-1.62 1.82-1.63-1.82h-2.5l2.88 3.2L30.56 21h2.38z',
        'w' => 1.25,
        'vw' => 40
    ],
    'fedora' => 'M14.06 2q5.82 0 9.9 4.13t4.07 9.9-4.12 9.88T14 30H3.37q-1.44 0-2.41-1T0 26.62V15.95Q.06 10.2 4.16 6.1t9.9-4.1zm10.63 9.81q-.06-.75-.19-1.37l-3.44-3.5V7q0 .44-.25.94zm-4-5.93q.19.3.25.62l3.44 3.38q-.94-2.82-3.7-4zM7.37 15.44q-.43.06-1 .19l.5.56q.25-.44.5-.75zm-1.3.25q-.45.12-.88.37l1.69 1.7q-.13-.45-.13-.82l.06-.5zm-1.13.5q-.5.25-.69.44l2.19 2.25q.37-.2.87-.32zM4 16.8q-.5.32-.63.57l2.13 2.18q.31-.3.69-.56zm-.88.75-.5.63 2.25 2.25q.2-.38.44-.69zm-.68.88-.38.75 2.44 2.5q.06-.5.25-.94zm-.56 1.06q-.13.38-.25.88l3.25 3.37q-.38-.88-.38-1.69zm-.44 2.63q0 .68.19 1.37L5.06 27v-.06q0-.44.2-.88zm.31 1.93q.94 2.82 3.63 4-.13-.25-.25-.56zm-.31-2.31 4 4q.31-.31.62-.5l-4.5-4.56q-.12.62-.12 1.06zm15.81-3.25q.94 0 1.34-.75t0-1.53-1.34-.78h-2.13q-.5 0-.5-.56v-2.94q0-1.82 1.35-2.78t2.78-.66q.44.13.88-.06t.71-.57.29-.87q0-.56-.35-1t-.9-.5q-.63-.13-1.25-.13-2.75 0-4.7 1.91t-1.93 4.66v3.5H8.87q-1 0-1.4.75t0 1.53 1.4.78h2.07q.19 0 .37.19t.19.37V22q0 1.44-1.03 2.47T8 25.5q-.31 0-.56-.06T7 25.37q-.63 0-1.06.47t-.44 1.1.44 1.06.8.44q2.88.62 5.35-1.32T14.56 22v-3.5h2.69zm1.44 0q.5-.06 1.06-.19l-.56-.5q-.19.38-.5.69zm1.31-.25q.5-.13.88-.31l-1.7-1.75q.2.43.2.81l-.07.5zm1.13-.5q.37-.19.75-.44l-2.25-2.25q-.32.19-.88.32zm1-.56q.37-.32.62-.57l-2.19-2.18q-.31.31-.62.5zm.8-.82q.38-.37.57-.62l-2.25-2.25q-.25.44-.5.69zm.7-.87q.25-.38.43-.75l-2.5-2.5q0 .5-.18 1zm.87-1.94-3.38-3.31q.44.75.5 1.63l2.57 2.62q.19-.5.31-.94zm.06-.31q.07-.44.13-1.06l-4-4.07q-.32.38-.63.63z',
    'figma' => [
        'p' => 'M16 10.69h-5.31v16q0 2.19-1.57 3.75T5.34 32t-3.78-1.56T0 26.66t1.56-3.78 3.75-1.57q-2.19 0-3.75-1.56T0 16t1.56-3.75 3.75-1.56q-2.19 0-3.75-1.57T0 5.34t1.56-3.78T5.31 0H16q1.06 0 2.06.44t1.72 1.12 1.13 1.69.4 2.06q0 2.25-1.56 3.82T16 10.69zm0 0q2.19 0 3.75 1.56T21.31 16t-1.56 3.75T16 21.31t-3.75-1.56T10.69 16t1.56-3.75T16 10.7z',
        'w' => 0.666,
        'vw' => 21
    ],
    'firefox' => [
        'p' => 'M29.88 14.69q.12.69.12 1.37v.2q-.13 3.24-1.88 6l-.06.08-.06.16q.25.69-.06 1.38-.44 1.87-2.07 3-1 .87-2.18 1.37l-.13.06-.06.07q-.13 0-.32.12-.18.44-.62.81-.06.13-.25.32t-.78.53-1.35.62-2 .5-2.68.22q-2.25 0-4.38-.62l.13.06-.5-.19Q9 30.2 7.43 29.13q-3.93-2.38-5.81-6.5-1.37-3-1.19-6.32-.25.88-.44 1.63 0-2.81 1-5.38-.5.75-.87 1.57Q.68 12 1.8 10.06q.37-.62.75-1.06 0-.25.06-.56.13-1.5.75-2.94h.06v.13q0-.2.1-.38t.28-.37v.25q.06-.25.12-.32.2-.56.7-.87 0 .81.24 1.62l.06-.06H5q.06.19.12.31.19.25.25.38h.07q.18.37.56.69 1.75-.32 3.44.3.18-.24.43-.43v.13q.38-.44.94-.7v.13q.69-.37 1.44-.56-.06 0-.31.19.37-.13.75-.19.56 0 .81.19-.25.06-.5.25h.13q-1.38.87-1.94 2.44.25 1.12 1.44 1.3 2.3.2 2.37.57v.25q-.12.69-.62 1.13l-.32.18q0 .07-.12.1t-.32.12-.37.16q-.75.31-1.38.69-.12.12-.25.19l-.06.12q.06.13.06.19l-.12.06q.12.25.06.5.07.44-.12.88-.25-.2-.44-.25.25.12.31.37v.19q-.12-.19-.31-.31-.38-.2-.75-.32l.12.25q-.12-.12-.3-.18-.38.18-.57.56-.38.75.06 1.5.38.56.94.94h-.13l-.12-.07q.5.32.94.57h-.13l-.06-.07q.06.13.25.2.5.24 1 .3t.84-.03.85-.25.81-.22q1.69-.3 2.75 1.07.31.37.1.69t-.66.18q-.06-.06-.19-.06t-.22.03-.25.06-.25.1-.28.15l-.31.16-.38.22-.37.22q-1.7.81-3.5.44l-.88-.25-.12-.07v.07q.87 1.12 2.06 1.87.81.38 1.69.56.75.2 1.56.2 2.88 0 4.88-2.04t2-4.84q.06-1.57-.57-2.94 2 1.19 3.07 3.19-.25-.69-.57-1.28t-.84-1.16-.78-.78-1-.81-.88-.72q-1.87-1.44-2.68-3.63-.44-1.31-.41-2.5t.44-2.25T19 2.3t.81-1.3l.38-.44q-.25.94 1.69 2.75Q23.69 4.93 23.69 5q.19-.88.5-1.69.19 1.37.94 2.56.31.5.97 1.28t.97 1.22q.18.25.3.5.5.7.88 1.44.57 1.13.88 2.31.31 1 .37 2l.32-.37q0 .19.06.44zm-18.7-6.06q-.05.06-.05.18l.03-.1z',
        'vw' => 30
    ],
    'firefox-browser' => 'M11.84 9.55ZM8.16 7.72c.02 0 0 0 0 0Zm21.97 2.83c-.67-1.6-2-3.31-3.06-3.85a15.92 15.92 0 0 1 1.55 4.64c-1.73-4.3-4.65-6.02-7.04-9.8l-.36-.58A4.13 4.13 0 0 1 20.8 0h-.03l-.01.01a11.15 11.15 0 0 0-5.16 7.67 8.2 8.2 0 0 0-3.02.77.39.39 0 0 0-.19.48c.07.2.29.3.49.24l.03-.01c.83-.4 1.72-.63 2.63-.7h.1l.37-.01c.74 0 1.48.1 2.19.3l.12.04a6.18 6.18 0 0 1 .6.21l.21.09a11.15 11.15 0 0 1 .75.38l.19.11a7.81 7.81 0 0 1 2.66 2.76 5.24 5.24 0 0 0-3.67-.9c5.48 2.75 4 12.19-3.59 11.83a6.74 6.74 0 0 1-2.69-.69 6.42 6.42 0 0 1-3.6-4.98s.7-2.62 5.04-2.62c.47 0 1.81-1.3 1.84-1.7 0-.12-2.66-1.17-3.7-2.19-.55-.54-.81-.8-1.04-1-.13-.11-.26-.2-.4-.3a7.06 7.06 0 0 1-.04-3.73A11.33 11.33 0 0 0 7.2 8.9c-.61-.77-.57-3.3-.53-3.82-.01-.03-.46.23-.51.27A13.25 13.25 0 0 0 .9 14.72c-.04.23-.06.48-.1.87v.05l-.02.5v.08a15.24 15.24 0 0 0 30.27 2.58l.07-.59c.3-2.61-.04-5.36-1-7.66Zm0 0',
    'first-order' => 'M14 6.06q.75 0 1.38.07l-.26 4.25 1.38-4q1.31.3 2.5 1.06l-1.81 3.81L20 8.06q1.06.82 1.88 1.94l-3.13 2.82 3.81-1.88q.63 1.13 1 2.56l-4 1.38 4.25-.25q.07.56.07 1.37t-.07 1.38l-4.18-.25 4 1.37q-.38 1.32-1.07 2.5l-3.81-1.81L21.87 22q-.75 1.07-1.87 1.94l-2.81-3.19 1.8 3.82q-1.05.68-2.5 1.06l-1.37-4.06.25 4.3q-.68.07-1.37.07t-1.38-.06l.25-4.25-1.37 4q-1.31-.31-2.5-1.06l1.87-3.82-2.8 3.2Q7 23.18 6.11 22l3.2-2.81-3.82 1.88q-.75-1.25-1.06-2.57l4-1.37-4.25.25q-.07-.82-.07-1.38 0-.69.07-1.37l4.25.25-4-1.44q.3-1.25 1.06-2.5L9.3 12.8 6.11 10q.82-1.07 1.94-1.95l2.81 3.2L9 7.37q1.19-.7 2.5-1l1.37 4-.18-4.25q.62-.07 1.3-.07zM27.69 8v16L14 32 .31 24V8L14 0zm-1.07.63L14 1.3 1.37 8.62v14.7L14 30.68l12.63-7.38V8.63zM14 2.3l11.75 6.82V22.8L14 29.7 2.25 22.8V9.2zm0 3.19q-4.31 0-7.38 3.06T3.56 16q0 4.31 3.06 7.4T14 26.5t7.38-3.1 3.06-7.4q0-4.37-3.07-7.44T14 5.5z',
    'first-order-alt' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm0 30.5q6.19 0 10.6-4.4T30.5 16 26.1 5.4 15.5 1 4.9 5.4.5 16t4.4 10.6T15.5 31zm0-28.75q5.69 0 9.72 4.03T29.25 16t-4.03 9.72-9.72 4.03-9.72-4.03T1.75 16t4.03-9.72 9.72-4.03zm0 27q5.5 0 9.38-3.88T28.75 16t-3.88-9.38-9.37-3.87-9.38 3.88T2.25 16t3.88 9.38 9.37 3.87zm11.63-10.19q-.38 1.57-1.25 3l-4.63-2.25 3.81 3.44q-1 1.31-2.31 2.31l-3.44-3.8 2.25 4.62q-1.44.87-3 1.25l-1.75-4.88.32 5.19q-.82.12-1.63.12t-1.63-.12l.26-5.19-1.7 4.88q-1.62-.38-3.06-1.25l2.25-4.63-3.37 3.88q-1.38-1.07-2.38-2.38l3.88-3.44-4.63 2.25q-.87-1.43-1.25-3l4.82-1.75-5.13.32q-.12-.82-.12-1.63t.12-1.62l5.19.25-4.88-1.7q.38-1.62 1.25-3.05l4.63 2.24-3.81-3.37q1-1.38 2.31-2.38l3.44 3.88-2.25-4.63q1.44-.87 3-1.25l1.69 4.82-.25-5.13q.8-.12 1.62-.12t1.63.12L16.8 9.2l1.75-4.82q1.57.38 3 1.25l-2.25 4.63 3.44-3.88q1.31 1 2.31 2.38l-3.8 3.38 4.62-2.2q.8 1.38 1.25 3l-4.88 1.7 5.19-.26q.06.82.06 1.63t-.06 1.63l-5.13-.32z',
    'firstdraft' => [
        'p' => 'M24 12h-4v8h-8v8H0v-1.63h10.38v-8h8v-8H24V12zm-1.63 2.38H24V24h-8v8H4v-1.63h10.38v-8h8v-8zm1.63 12V28h-4v4h-1.63v-5.63H24zM0 0h24v8h-8v8H8v8H0V0z',
        'w' => 0.75,
        'vw' => 24
    ],
    'flickr' => 'M25 2q1.25 0 2.13.88T28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22zM9.03 19.94q1.66 0 2.81-1.16T13 15.97t-1.16-2.81T9.04 12t-2.82 1.16-1.16 2.8 1.16 2.82 2.81 1.16zm9.94 0q1.66 0 2.81-1.16t1.16-2.81-1.16-2.81T18.97 12t-2.81 1.16-1.16 2.8 1.16 2.82 2.8 1.16z',
    'flipboard' => 'M0 2h28v28H0V2zm22.38 11.19V7.63H5.63v16.75h5.56V18.8h5.62v-5.6h5.57z',
    'fly' => [
        'p' => 'M12.38 26.75q2.12 1.88 2.06 3.13v.18q-.06.88-.75 1.4T12 32q-1.13 0-1.81-.66t-.63-1.59q.06-1.19 2.06-2.94.2-.19.32-.25.25 0 .44.19zm8.37-22.56q1.94 3.31 1.31 7.06-.62 2.94-3.6 6.69t-6.15 6.81l-.25.13h-.19l-.37-.38q-1.75-1.69-3.1-3.12t-2.93-3.5-2.57-4.1-1.15-3.84q-.13-6.06 5.19-8.88Q9.56 0 12 0q2.5 0 4.81.94 2.7 1.25 3.94 3.25zM11.69 22V1.62q-1.69.2-2.75 1.75T7.56 6.72t-.18 3.47q.3 4.5 4.3 11.81zm8.87-10.75q.82-2.88-.78-5.81t-5.47-3.75q1.63 1.12 2.35 3.44t.65 4.75q-.12 4.43-4.18 12 6.18-6.25 7.43-10.63z',
        'w' => 0.748,
        'vw' => 24
    ],
    'font-awesome' => 'M24.88 2q1.25 0 2.18.94T28 5.13v21.75q0 1.25-.94 2.18t-2.19.94H3.13q-1.25 0-2.18-.94T0 26.88V5.13q0-1.25.94-2.2T3.13 2h21.75zM22 19.75V11q0-.5-.5-.5-.06 0-1.31.5t-2.07.5q-.5 0-.8-.13-2.32-.87-3.82-.87-1.75 0-3.88.88-.12.12-.31.18v-1.18q.81-.57.81-1.63 0-.88-.62-1.47T8 6.68t-1.47.6-.6 1.47q0 1 .82 1.63v13.68q0 .5.37.88t.88.37q.94 0 1.19-.87v-.06q.06-.07.06-.32v-2.81q.06-.06.12-.06t.13-.07q2.37-1.06 4.19-1.06 1.06 0 2.37.53t2.13.53q1.37 0 3.37-.87.44-.2.44-.5z',
    'font-awesome-alt' => 'M21.19 10.69q.25 0 .4.12t.1.32v8.43q0 .25-.44.44-1.88.88-3.25.88-.81 0-2.06-.5t-2.32-.5q-1.62 0-4.06 1.06-.06.06-.19.06v2.75q0 .2-.06.32v.06q-.25.87-1.13.87-.5 0-.87-.37t-.38-.88V10.5q-.8-.62-.8-1.62 0-.81.59-1.4t1.43-.6 1.44.6.6 1.4q0 1.06-.75 1.62v1.13q.06 0 .25-.13 2.18-.8 3.75-.8t3.68.8q.38.2.82.2.81 0 2-.5t1.25-.5zM24.87 2q1.25 0 2.2.94T28 5.13v21.75q0 1.25-.94 2.18t-2.19.94H3.13q-1.25 0-2.18-.94T0 26.88V5.13q0-1.25.94-2.2T3.13 2h21.75zm.88 24.88V5.13q0-.38-.25-.63t-.63-.25H3.13q-.37 0-.62.25t-.25.63v21.75q0 .3.25.59t.63.28h21.75q.37 0 .62-.25t.25-.63z',
    'font-awesome-flag' => [
        'p' => 'M27.75 22.44q0 .5-.81.87-3.44 1.44-5.75 1.44-1.38 0-3.63-.87T13.5 23q-3.13 0-7.25 1.81-.06.07-.19.1t-.19.03v4.87q0 1-.62 1.6t-1.5.6-1.5-.63-.63-1.57V6.38Q.18 5.3.18 3.56q0-1.43 1.07-2.5T3.78 0 6.3 1.06t1.06 2.5q0 1.82-1.44 2.82v2q.07 0 .41-.16t.81-.34 1.2-.41 1.43-.4 1.63-.35 1.71-.16q2.32 0 4.82.88t3.12.87q1.44 0 3.53-.87t2.22-.88q.38 0 .66.25t.28.63v15z',
        'w' => 0.873,
        'vw' => 28
    ],
    'fonticons' => 'M0 2h28v28H0V2zm11.69 8.81q.37 0 .6.13t.3.44.1.87l3.19-.37q0-1.94-1.1-2.63t-2.97-.69q-1.62 0-2.65.44t-1.6 1.4T7 12.95v1.31H5.25v2.31h1.38q.37 0 .37.2v6.93q0 .44-.44.5l-1.31.13v1.62h8.19v-1.56l-2.75-.25q-.25-.07-.19-.25v-7.32H14l.69-2.3h-4.25q0-.13.06-.26v-1.5q0-.62.06-.94t.32-.53.81-.22zm4.62 15.13h6.44v-1.56l-1-.2q-.19 0-.19-.06v-9.87h-5l-.43 1.87 1.56.38q.37.12.37.5v6.75q0 .37-.43.44l-1.32.19v1.56zm5.07-16 1.37-1.5-.25-.63h-1.69l-1-1.75h-.56l-1 1.75h-1.69l-.25.63 1.38 1.5-.57 2.06.44.5 2-1.06 1.94 1.06.44-.5z',
    'fonticons-fi' => [
        'p' => 'M7.13 14h5.8L12 17.19H7.19v10.19q-.07.25.31.3l3.69.38v2.13H0V28l1.81-.19q.57-.06.57-.68v-9.75q0-.2-.5-.2H0V14h2.38v-1.81q0-3.13 1.65-4.57T9 6.2q2.63 0 4.1.96t1.46 3.6l-4.3.5q0-1.19-.32-1.6t-1.13-.4q-.75 0-1.12.31t-.44.75-.06 1.31v2q0 .07-.03.13l-.07.12-.03.1.07.03zM24 30.19h-8.81v-2.13l1.75-.25q.68-.06.68-.62v-9.25q0-.5-.62-.69l-2.06-.56.56-2.5h6.87v13.37q-.06.13 0 .16t.25.1l1.38.24v2.13zM22.12 8.5l.75 2.81-.62.63-2.63-1.38-2.68 1.38-.63-.63.75-2.81-1.87-2.31.31-.63h2.38l1.3-2.37H20l1.31 2.37h2.38l.31.88z',
        'w' => 0.75,
        'vw' => 24
    ],
    'fort-awesome' => 'M30.56 18q.32 0 .32.25V32H19.44v-6q0-1.63-1.07-2.53t-2.37-.9-2.38.9T12.56 26v6H1.12V18.25q0-.25.32-.25h1.69q.3 0 .3.25v2H5.7V9.12q0-.24.31-.24H7.7q.31 0 .31.24v2h2.25v-2q0-.24.32-.24h1.68q.32 0 .32.24v2h2.3v-2q0-.24.7-.24v-7Q15 1.63 15 1t.5-.88 1 0T17 1t-.62.88v.3q.81-.18 1.5-.18.37 0 1.06.13t1 .12.85-.13.65-.12q.25 0 .25.25V6q0 .5-1.69.5-.37 0-1-.13t-1-.12q-.68 0-1.62.25v2.38q.75 0 .75.24v2h2.31v-2q0-.24.25-.24h1.75q.25 0 .25.24v2H24v-2q0-.24.32-.24H26q.3 0 .3.24v11.13h2.26v-2q0-.25.31-.25h1.69zm-18-.31v-4q0-.25-.31-.25h-1.69q-.31 0-.31.25v4q0 .31.31.31h1.69q.31 0 .31-.31zm9.19 0v-4q0-.25-.31-.25h-1.7q-.3 0-.3.25v4q0 .31.3.31h1.7q.31 0 .31-.31z',
    'fort-awesome-alt' => 'M13 14.81q.25 0 .25.25v3.25q0 .2-.25.2h-1.38q-.24 0-.24-.2v-3.25q0-.25.24-.25H13zm7.38 0q.25 0 .25.25v3.25q0 .2-.25.2H19q-.25 0-.25-.2v-3.25q0-.25.25-.25h1.38zM28.63 7q2.87 4.06 2.87 9v.88l-.06.06v.53l-.03.1-.03.12v.12q-.07.2-.07.25v.13l-.06.37v.07q0 .12-.06.25v.18q-.07 0-.07.1v.1l-.03.05-.03.1v.1l-.03.09-.03.1v.09l-.03.09-.03.06q0 .07-.07.19v.19q-.06 0-.06.06v.06q-.06.13-.06.2l-.13.3q0 .13-.06.2l-.03.06-.03.09-.03.1-.03.06-.07.18-.06.07v.06q-.06.13-.06.19l-.07.19q-.06 0-.12.12v.1l-.03.06-.06.1-.04.12-.06.06q0 .06-.03.13t-.1.12q-.24.5-.43.75v.06l-.1.13q-.04.06-.03.12l-.06.07q-.12.18-.12.22t-.04.06l-.09.1-.06.12-.07.06q0 .06-.06.13t-.06.12h-.06q-3.7 5.38-10 6.44h-.25l-.1.03-.1.03h-.24q-.13.06-.2.06h-.55l-.13.03-.1.04h-1.8l-.1-.04-.1-.03h-.52q-.13-.06-.2-.06h-.37q-.06-.06-.19-.06h-.18q-6.25-1.07-10-6.25V25l-.2-.18q0-.07-.02-.1t-.07-.1-.09-.05v-.04l-.06-.09-.1-.16q-.03-.04-.03-.09l-.1-.12-.02-.07q0-.06-.03-.06t-.03-.06q-.2-.25-.44-.75 0-.07-.07-.13t-.06-.12q-.06-.07-.06-.1t-.03-.1-.03-.09-.03-.06-.04-.06-.03-.1-.03-.12l-.06-.13q-.06 0-.1-.06t-.03-.1v-.06l-.06-.09-.06-.19-.06-.12q-.07-.13-.07-.19 0-.13-.06-.19 0-.06-.03-.15t-.1-.16v-.19q-.06 0-.06-.06v-.06q-.06-.13-.06-.2t-.03-.09-.03-.06v-.1l-.03-.09-.04-.1v-.05q-.06-.07-.06-.2v-.12l-.03-.1-.03-.09v-.09l-.03-.06-.03-.1v-.09l-.07-.16v-.06l-.06-.37v-.13Q.61 18.06.61 18v-.5q-.06-.12-.06-.25v-.62l-.07-.13V16q0-4.94 2.88-9l.44-.56q.5-.63 1-1.13.25-.25.5-.56.37-.31.56-.5.44-.31.87-.69l.32-.19Q9.67 1.5 12.85.81 14.36.5 15.99.5q1.57 0 3.13.31 3.19.63 5.87 2.56l.25.2q.5.37.88.68 1.12 1 2.06 2.19.19.19.44.56zM28 23.25v-4.5q0-.25-.25-.25h-1.38q-.25 0-.25.25v1.63h-1.8v-9q0-.26-.26-.26H22.7q-.25 0-.25.26V13h-1.82v-1.63q0-.24-.25-.24H19q-.25 0-.25.24V13h-1.81v-1.63q0-.24-.63-.24V9.25q.75-.19 1.31-.19.32 0 .82.1t.81.09q1.38 0 1.38-.38V5.81q0-.19-.25-.19-.13 0-.54.1t-.65.1-.81-.1-.88-.1q-.5 0-1.19.13V5.5q.5-.19.5-.69Q16.81 4 16 4t-.81.81q0 .5.5.69v5.62q-.63 0-.63.25V13h-1.81v-1.63q0-.25-.25-.25h-1.38q-.25 0-.25.25V13H9.56v-1.63q0-.25-.25-.25H7.93q-.25 0-.25.25v9h-1.8v-1.62q0-.25-.26-.25H4.25q-.25 0-.25.25v4.5q.87 1.44 2.12 2.62 1.94 1.94 4.44 3 1.31.57 2.69.88V25q0-1.75 1.37-2.47t2.75 0T18.75 25v4.75q1.37-.31 2.69-.88 2.5-1.06 4.43-3 1.25-1.18 2.13-2.62z',
    'forumbee' => 'M.38 19.38Q0 17.75 0 16.13q0-5.75 4.1-9.85t9.84-4.1q1.5 0 3.06.32-5.75 2.25-10.12 6.66T.38 19.37zM24.93 7.5q-7.07 2.06-12.28 7.34T5.38 27.25q-1.5-1.19-2.82-3.06 2.07-7 7.22-12.22t12.16-7.28q1.69 1.19 3 2.81zm.93 15.81q1.13 3 2.13 6.88-3.44-.88-6.69-2.13-2.5 1.57-5.31 2 4.19-7.81 11.75-11.93-.38 2.8-1.88 5.18zm1.88-9.06q-5.31 2.19-9.38 6.31t-6.25 9.5q-1.93-.25-3.56-.93 2.13-6.5 6.94-11.41t11.25-7.03q.75 1.69 1 3.56z',
    'foursquare' => [
        'p' => 'M20.19.19q1.81 0 2.44 1t.18 3.06l-3 15.19q-.12.56-.25.81t-.4.75-.82.72-1.34.22h-5.19q-.25 0-.44.19-.12.19-8.18 9.5-.82.94-2.07.44Q0 31.63 0 30.19V3.07Q0 2 .78 1.1T3.12.2H20.2zm-1 4.62q.06-.43-.19-.78t-.69-.34H4.62q-.43 0-.72.34t-.28.72v21.19q0 .12.13 0l5.56-6.75q.44-.5.72-.66t.97-.15h4.56q.32 0 .53-.16t.35-.34.19-.32l.68-3.68q.13-.44-.18-.78t-.75-.35h-5.63q-.56 0-.9-.37t-.35-.94v-.81q0-.25.1-.5t.24-.41.41-.25.5-.1h6.63q.18 0 .37-.09t.34-.19.25-.25.1-.34z',
        'w' => 0.718,
        'vw' => 23
    ],
    'free-code-camp' => [
        'p' => 'M4.31 9q-1.87 3.31-1.81 7.22t1.88 7.03q.5.75 1.09 1.5t1 1.13.72.78.31.71-.25.57-.56.25q-.5 0-1.47-.88t-2.06-2.4-1.88-3.97T.5 15.8q0-2.44.78-4.72t1.82-3.7 2-2.32 1.47-.87q.2 0 .4.1t.38.24.16.35q0 .37-.47.87t-1.28 1.4T4.31 9zm12.25 13.38q-1.81-.63-2.94-2.1T12.5 17q0-1.5 1.81-4.28t1.82-4.22q0-.75-.41-1.44t-.84-.94q-.32-.18-.32-.3.2-.45.94-.32t1.38.44q1.43.68 2 1.72t.87 2.9q.38 2.06 1 2.06.75 0 .75-.75 0-.18-.5-1.5-.06-.37.06-.43.44 0 1.44 1.06 2 2.12 2 5.12 0 1.32-.47 2.44t-1.19 1.9-1.65 1.29q-.75.5-.75-.13 0-.12.44-.59t.9-1.25.47-1.66q0-1.12-.5-1.93-.5-.63-.75-.63v.06q0 .2.1.54t.09.53q0 .62-.88.75-1-.07-1.06-1.2v-1.3q0-.38-.12-.7-.2-.37-.63-.8t-.8-.44q-.19 0-.19.12 0 .06.25.4t.25.85q0 1-1.37 2.19t-1.38 2.19q0 1.43.63 2.37.43.69 1.18 1 .25.06.25.19 0 .37-.75.06zm12.82-18.2q.56 0 1.53.91t2.03 2.44 1.81 3.97.75 5.06q0 2.32-.75 4.57T33 24.88t-2.03 2.4-1.66.9q-.25 0-.53-.24t-.28-.47.44-.75 1.03-1.16.72-.8q1.94-2.45 2.56-5.63t-.28-6.6-3.34-5.97Q28.5 5.38 28.5 5q0-.31.31-.56t.57-.25zm-2.63 20.2q.75 0 .75 1t-.75 1H10.44q-.38 0-.66-.29t-.28-.71q0-.25.1-.47t.3-.38.54-.16h16.31z',
        'w' => 1.123,
        'vw' => 36
    ],
    'freebsd' => 'M19 6q.94-.94 4.34-2.6t4.35-.71-.72 4.34-2.6 4.35q-.56.56-1.71.12T20.4 9.97t-1.54-2.25T19 6zM6.87 4.25q-3 1.88-4.75 4.94Q-.94 3.88.38 2.56q1.25-1.25 6.5 1.7zm18.57 6.63.69-1.2q1.68 3.07 1.68 6.63 0 5.56-3.93 9.53t-9.54 3.97-9.53-3.97-3.93-9.53q0-3.69 1.8-6.78t4.91-4.9 6.79-1.82q3.25 0 6.18 1.56-.56.32-1.18.7-1.25-.2-1.54.68t.6 2.38q1.06 2.06 3.47 3.68t3.34.7q.5-.5.19-1.63z',
    'fulcrum' => [
        'p' => 'M4.44 10.25 2.19 13 0 10.25l2.19-2.69zM7.44 0v13.19L4.87 16l2.57 2.81v13.2L6.19 19.62 3 16l3.19-3.62zm5 10.25 2.19-2.69 2.25 2.69L14.63 13zm-3.07 2.94V0l1.32 12.38L13.88 16l-3.2 3.63L9.39 32V18.81L12 16z',
        'w' => 0.525,
        'vw' => 17
    ],
    'galactic-republic' => 'M15.5 31.5q-6.44 0-10.97-4.53T0 16 4.53 5.03 15.5.5t10.97 4.53T31 16t-4.53 10.97T15.5 31.5zm0-29.94q-6 0-10.22 4.22T1.06 16t4.22 10.22 10.22 4.22 10.22-4.22T29.94 16 25.72 5.78 15.5 1.56zm1.75 1.31q3.5.5 6.31 2.63l-1.12 1.12q-.94-.75-2.47-1.37t-2.72-.81V2.87zm-3.5 0v1.57q-2.88.44-5.19 2.19L7.44 5.5q2.81-2.12 6.31-2.62zm.75 3.2h2v5.12q.88.19 1.69.75l3.69-3.69 1.37 1.37-3.69 3.7q.57.8.75 1.68h5.13v1.94H20.3q-.18.94-.75 1.75l3.7 3.69-1.38 1.37-3.7-3.69q-.8.57-1.68.75v5.13h-2V20.8q-.88-.25-1.69-.75l-3.69 3.7-1.37-1.38 3.69-3.7q-.57-.8-.75-1.68H5.56v-2h5.13q.19-.87.75-1.69l-3.7-3.69 1.38-1.37 3.7 3.69q.74-.56 1.68-.75V6.06zM26 7.93q2.13 2.81 2.63 6.31h-1.57q-.12-.75-.44-1.75t-.78-1.9-.97-1.54zm-21 0 1.13 1.12q-1.75 2.32-2.2 5.2H2.38q.5-3.5 2.63-6.32zm-2.63 9.81h1.57q.44 2.88 2.19 5.19L5 24.06q-2.12-2.81-2.62-6.31zm24.7 0h1.56q-.5 3.5-2.63 6.31l-1.12-1.12q1.75-2.32 2.18-5.19zm-18.5 7.63q2.3 1.75 5.18 2.18v1.57q-3.5-.5-6.31-2.63zm13.87 0 1.12 1.12q-2.81 2.13-6.31 2.63v-1.57q2.88-.44 5.19-2.19z',
    'galactic-senate' => 'M15.63 2.06h.74v1.7q.63.93 1.07 4.68t.5 8.75q-1.19-1.63-1.19-3v-1.06q0-.5-.19-1-.06-.25-.12-.38-.2-.19-.44-.19t-.44.2q-.06.12-.12.37-.2.5-.13 1v1.06q0 1.37-1.25 3 .06-3.31.25-6.16t.53-4.75.78-2.53V2.06zM14 11.31q-.19 2.07-.25 4.5-2.38 1.2-2.88 3.75-.8.25-2.56.25-.75-.37-1.62-.37.18-1.38.75-2.63.68.38 1.34.38t.88-.35-.13-.9-1.03-1q.81-1.13 1.87-1.94.5.75 1.07 1.1t.94.12.34-.9-.47-1.45q.81-.37 1.75-.56zm4 0q.94.2 1.75.57-.44.75-.47 1.43t.34.91.94-.12 1.07-1.1q1.06.81 1.87 1.94-.69.44-1.03 1t-.13.9.88.35 1.34-.38q.57 1.25.75 2.63-.87 0-1.62.37-1.75 0-2.56-.25-.5-2.56-2.88-3.75-.06-2.37-.25-4.5zm-2 .57q.38 0 .38 1.24v1.07q0 1.94 1.9 3.94t3.97 2q2.5 0 3.69.03t3.15.47 2.91 1.3v.13q-.88 1.63-2.06 3-.63-1.06-2.44-1.8t-4.31-.76q-2.07.06-3.6.78t-2.1 1.78q-.74.2-1.5.2t-1.5-.2q-.37-.68-1.15-1.25t-1.97-.93-2.56-.38q-2.5 0-4.31.75t-2.44 1.81Q.87 23.7 0 22.06v-.12q.93-.88 2.9-1.32t3.16-.46 3.69-.04q2.06 0 3.97-2t1.9-3.93v-1.07q0-1.25.38-1.25zm0 6q-1.06 0-1.81.75t-.75 1.8.75 1.82T16 23t1.81-.75.75-1.81-.75-1.82-1.81-.75zm0 .5q.88 0 1.47.59t.6 1.47-.6 1.47-1.47.6-1.47-.6-.6-1.47.6-1.47 1.47-.6zm0 .37q-.69 0-1.19.5t-.5 1.19.5 1.19 1.19.5 1.19-.5.5-1.2-.5-1.18-1.19-.5zm-7.19 4.19q1.7 0 3.07.62 1.43 1.82 3.06 2.07H15q.5.06 1 .06t1-.06h.07q1.62-.25 3.06-2.07 1.37-.62 3.06-.62 1.25 0 1.88.12-.13 1.2-.44 1.82-.88-.25-1.69-.25-1.94.06-3.25 1.4T18 29.31v.32q-1 .18-2 .18t-2-.18v-.32q-.37-1.94-1.69-3.28t-3.25-1.4q-.8 0-1.68.25-.32-.63-.44-1.82.62-.12 1.87-.12z',
    'get-pocket' => 'M25.5 4q1.06 0 1.78.75T28 6.56V15q0 5.81-4.1 9.9T14 29t-9.9-4.1T0 15V6.56Q0 5.5.75 4.75T2.56 4H25.5zM15.37 20.75q1.5-1.44 3.63-3.44 2.56-2.44 3.03-3t.47-1.25q0-.75-.56-1.31t-1.32-.56q-.62 0-1.12.37T16.87 14L14 16.75q-1.19-1.19-2.94-2.88-2-1.93-2.5-2.3t-1.12-.38q-.82 0-1.38.56t-.56 1.31q0 .69.47 1.22t3.1 3.03q2.12 2 3.62 3.44.56.56 1.3.56t1.38-.56z',
    'gg' => 'm11.19 14.38 6.44 6.43-6.44 6.38L0 16 11.19 4.81 14 7.63l-1.62 1.56L11.18 8l-8 8 8 8 3.26-3.25L9.63 16zM20.8 4.8 32 16 20.8 27.18l-2.8-2.8 1.63-1.57 1.18 1.2 8-8-8-8-3.25 3.24L22.38 16l-1.57 1.63-6.43-6.44z',
    'gg-circle' => 'M16.06.5q6.44 0 10.97 4.53T31.56 16t-4.53 10.97-10.97 4.53T5.1 26.97.56 16 5.1 5.03 16.06.5zM13 23.94l4.63-4.69-4.82-4.88-1.5 1.57 3.32 3.31-1.7 1.63-4.8-4.82 4.8-4.81.7.69 1.56-1.5L13 8.19l-7.88 7.87zm6.13-.13L27 15.94l-7.88-7.88-4.62 4.7 4.81 4.87 1.5-1.57-3.31-3.31 1.63-1.63L24 15.95l-4.88 4.81-.68-.69-1.5 1.5z',
    'git' => 'M1.19 13.88q0-2.07 1.31-3.47t3.06-1.57 3.57.66q1.5 0 3.37-.69v3.13q-.56.18-1.25.37.25.69.25 1.31 0 1.94-1.12 3.35T7.3 18.75q-1.25.25-1.25 1.31 0 .81 1 1.22t2.2.6 2.15 1.03.97 2.34q0 4.75-5.63 4.75-2.37 0-4.06-.97T1 25.87q0-2.56 2.81-3.5v-.06q-1.12-.75-1.06-2.22t1-1.9v-.07q-1.12-.37-1.84-1.68t-.72-2.57zm5.44 10.24q-2.44 0-2.44 1.63 0 1.56 2.68 1.56 2.38 0 2.38-1.68 0-1.5-2.63-1.5zM6.42 16q1.88 0 1.88-2.13 0-.87-.4-1.46t-.97-.72-1.1 0-.93.72-.41 1.47Q4.5 16 6.44 16zM27 19.13v3q-1.13.62-2.88.62-1.18 0-1.96-.4t-1.13-1.26-.47-1.68-.12-1.97v-5.57q-1.07 0-1.5.07v-3h1.5q0-2-.07-2.57h3.5q-.06.7-.06 2.57H27v3q-1.19-.07-3.19-.07v5.7q0 2.06 1.38 2.06 1.06 0 1.8-.5zM17.87 4.28q0 .9-.62 1.6t-1.5.68q-.56 0-1.06-.31t-.78-.84-.29-1.1q0-.94.63-1.62t1.5-.7q.94 0 1.53.7t.6 1.6zm-.37 4.66q-.13.69-.1 6.72t.1 6.9H14q.13-.87.13-6.87T14 8.94h3.5z',
    'git-alt' => [
        'p' => 'M27.47 14.75 15.25 2.53a1.8 1.8 0 0 0-2.55 0l-2.54 2.54 3.22 3.22a2.14 2.14 0 0 1 2.71 2.73l3.1 3.1c2.14-.74 3.83 1.94 2.22 3.54-1.66 1.66-4.39-.18-3.5-2.33l-2.9-2.9v7.62c1.58.79 1.4 2.62.57 3.44a2.14 2.14 0 1 1-2.33-3.5V12.3a2.14 2.14 0 0 1-1.16-2.8L8.9 6.3.53 14.7c-.7.7-.7 1.84 0 2.55l12.22 12.22c.7.7 1.85.7 2.55 0L27.47 17.3c.7-.7.7-1.84 0-2.55Zm0 0',
        'vw' => 28
    ],
    'git-square' => 'M8.75 21.75q1.88 0 1.88 1.13 0 1.18-1.7 1.18Q7 24.06 7 22.94q0-1.19 1.75-1.19zM7.19 14.5q0-1.06.69-1.47t1.37 0 .69 1.47q0 1.5-1.31 1.5-1.44 0-1.44-1.5zM28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22q1.25 0 2.13.88T28 5zm-15.06 5.88q-1.38.5-2.44.5-1.81-1.07-3.75-.1T4.81 14.5q0 .88.53 1.81t1.35 1.2v.05q-.7.32-.72 1.38t.78 1.56v.06q-2.06.7-2.06 2.5 0 1.57 1.21 2.25t2.91.7q4.06 0 4.06-3.38 0-1.07-.72-1.66t-1.56-.75-1.53-.44-.69-.84q0-.81.88-.94.87-.19 1.56-.72T11.87 16t.38-1.69q0-.43-.19-.93.5-.13.88-.25v-2.25zm3.56.12H14q.06.5.06 4.78t-.06 4.9h2.5q-.06-.62-.06-4.93T16.5 11zm.25-3.38q0-.62-.44-1.12T15.25 6t-1.06.5-.44 1.16.44 1.12 1.06.47 1.06-.5.44-1.13zM23.31 11H21q0-1.44.06-1.88h-2.5q.07.38.07 1.88h-1.07v2.06h1.07V17q0 1.81.5 2.81t2.12 1q1.25 0 2.06-.43v-2.2q-.56.38-1.3.38-1 0-1-1.43v-4.07h2.3V11z',
    'github' => 'M10.38 24.81q0 .25-.32.25-.37.07-.37-.25 0-.18.31-.18.38-.07.38.18zm-1.97-.25q.09-.25.4-.12.32.06.25.31t-.37.13q-.38-.07-.28-.32zm2.78-.12q.31 0 .34.19t-.28.3q-.31.07-.34-.18t.28-.31zM15.3.5q4.25 0 7.88 2T28.9 8t2.1 7.75q0 5.19-2.92 9.28T20.5 30.7q-.5.12-.78-.13t-.28-.62V24.7q0-2-1.06-2.88 1.37-.19 2.12-.31t1.88-.56 1.72-1.13 1.03-1.94.43-3q0-1.18-.37-1.93t-1.25-1.7q.75-1.93-.13-4.24-.62-.2-1.68.22t-1.88.9l-.75.57q-1.94-.57-3.97-.57t-3.9.57q-.32-.25-.88-.6t-1.72-.81T7.31 7q-.93 2.31-.18 4.25-1.44 1.56-1.44 3.62 0 1.75.44 3t1 1.94 1.62 1.13 1.84.59 2.1.28q-.81.75-1 2.13-.63.24-1.28.37t-1.66-.25-1.62-1.44q-.38-.69-.97-1.1t-1.03-.46L4.75 21q-1.37 0-.12.93.43.25.87.82t.63 1l.25.5q.25.75.87 1.25t1.38.62 1.43.13 1.13-.07l.44-.06.03 2.25.03 1.56q0 .38-.28.63t-.78.12q-4.75-1.56-7.7-5.65T0 15.75q0-6.5 4.4-10.88T15.32.5zM6.06 22.06q.13-.12.32.07t.06.3-.31-.05-.07-.32zm-.68-.5q.12-.12.3-.06.2.13.13.25-.06.19-.25.06-.25-.12-.18-.25zm2.06 2.19q.12-.19.37.06t.13.44q-.19.19-.44-.06t-.06-.44zm-.75-.88q.19-.18.37.13.2.25 0 .38t-.34-.16-.03-.34z',
    'github-alt' => [
        'p' => 'M11.63 20.56q0 1.2-.6 2.32T9.31 24q-.75 0-1.28-.62T7.28 22t-.22-1.44q0-1.18.6-2.31t1.68-1.13 1.7 1.13.59 2.31zM30 17.38q0 3.68-1.13 5.93-.56 1.2-1.4 2.03t-2.13 1.35-2.28.78-2.72.37-2.46.13-2.5.03h-2.22L11 27.9l-2.44-.21-2.15-.47-2.16-.85-1.63-1.28-1.37-1.78Q0 20.81 0 17.37q0-4.06 2.63-7.06-.5-1.5-.5-3.06Q2.13 5.8 3 4q2.19 0 3.72.5t3.1 1.75q2.62-.63 5.56-.63 2.62 0 5 .57 1.56-1.25 3.09-1.72T27.13 4q.93 1.81.93 3.25 0 1.56-.5 3Q30 13.12 30 17.37zm-4 3.18q0-2.19-1.25-3.69t-3.38-1.5q-.75 0-3.5.38-1.3.19-2.8.19t-2.82-.19q-2.69-.38-3.5-.38-2.13 0-3.38 1.5t-1.25 3.7q0 1.56.6 2.74t1.47 1.85 2.18 1.06 2.5.53 2.7.13h3q1.87 0 3.37-.22t2.97-.85 2.28-1.97.81-3.28zm-5.16-3.44q1.1 0 1.7 1.13t.59 2.31-.6 2.32T20.84 24t-1.71-1.12-.63-2.32.63-2.31 1.71-1.13z',
        'vw' => 30
    ],
    'github-square' => 'M25 2q1.25 0 2.13.88T28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22zm-7.69 24q3.13-1.06 5.03-3.78t1.91-6.16q0-4.25-3.03-7.15T13.87 6 6.66 8.9t-2.91 7.15q0 3.44 1.94 6.16T10.75 26q.75.13.75-.5l-.03-1.06-.03-1.5q-.13.06-.31.1t-.72.02-.97-.1-.88-.4-.62-.87q-.44-1.2-1.13-1.5l-.44-.44q0-.19.5-.19 1 .06 1.57 1.06 1.06 1.82 3.06.88.13-.94.63-1.38-1.13-.12-1.75-.28t-1.41-.6-1.13-1.37-.34-2.3 1-2.45q-.56-1.25.06-2.81.38-.13 1.1.16t1.28.59l.5.31q1.25-.31 2.6-.31t2.59.31q.25-.12.6-.34t1.12-.53 1.15-.19q.63 1.5.13 2.81 1.06 1.13 1.06 2.44 0 1.38-.38 2.31t-1.18 1.38-1.44.6-1.75.27q.69.57.69 1.88v3.5q0 .62.68.5zm-6.1-3.84q-.21.03-.24-.13t.15-.16q.25-.06.29.1t-.2.19zm-.59-.04q0-.18-.25-.18t-.25.18q0 .13.25.13t.25-.13zm-.87-.09q-.06.16-.25.1-.25-.07-.19-.22t.25-.1q.25.06.19.22zM9 21.7q.06-.07-.1-.25t-.27-.07q-.07.13.06.32.19.12.31 0zm-.56-.57q.12-.06 0-.25t-.25-.06q-.13.07 0 .22t.25.1zm-.4-.59q.09-.1-.04-.22t-.22-.03.03.22.22.03zm-.48-.47q.07-.06-.06-.18-.12 0-.19.06t.13.12q.12.13.12 0z',
    'gitkraken' => [
        'p' => 'M35.4 7.4q1.1 3 1.1 6.3 0 7.3-5 12.5t-12.3 5.5v-9.1h-1.4v9q-7.2-.2-12.3-5.4t-5-12.4q0-3.4 1.1-6.4.3-.7 1-.4.6.3.4.9-1 2.8-1 5.8 0 6.3 4 11T16.4 30v-7.7L15 22v6.5q-3.3-.8-6-3t-4.1-5.2-1.6-6.6q0-4.2 2.2-7.8T11.1.4q.7-.4 1 .3.3.6-.3 1-3.2 1.7-5.2 5t-2 7q0 4.4 2.5 7.8t6.3 5v-5.7l-.7-.5-.6-.7q-.3-.4-.2-1 0-1.2 1.1-1.8-.2-1.7-.7-2.7t-.9-1.2-.8-.6-.4-.7v-1q0-1.9 3.1-6t5-4.3h.4q1.9.1 5 4.3t3.1 6v1l-.2.6q-.2.3-.6.5t-.8.5-.7 1.3-.5 2.3q1.1.6 1.1 1.9 0 .7-.4 1.3t-1.1.8v5.8q3.8-1.6 6.3-5.1t2.4-7.8q0-3.8-2-7t-5.1-5q-.6-.4-.3-1 .3-.7 1-.3 2.3 1.3 4.1 3.3t2.7 4.6 1 5.4q0 5.4-3.2 9.5T22 28.5V22l-1.4.4V30q4-.5 7.3-2.8t5.2-5.9 1.9-7.7q0-3-1.1-5.8-.3-.7.4-.9.7-.3 1 .4zm-12.5 9.8q-.6 0-1 .5t-.5 1l.2.8q.2.4.5.5t.8.2q.6 0 1-.5t.5-1-.5-1-1-.5zm-8.8 3 .6-.1q.3-.1.5-.4t.3-.4.1-.6q0-.6-.4-1t-1-.5-1.1.4-.5 1 .5 1.1 1 .5z',
        'w' => 1.156,
        'vw' => 37
    ],
    'gitlab' => 'm6.56 1.56 3.57 10.94H1.88l3.5-10.94q.18-.43.62-.43t.56.43zM.06 18l1.82-5.5L16 30.88.5 19.38Q-.19 18.8.06 18zm10.06-5.5h11.76L16 30.88zM31.95 18q.25.81-.44 1.38L16 30.88 30.13 12.5zM26.62 1.56l3.5 10.94h-8.25l3.57-10.94q.12-.44.56-.44t.63.44z',
    'gitter' => [
        'p' => 'M4.13 20.13H1V0h3.13v20.13zm6.3-15.38V32H7.26V4.75h3.19zm6.32 0V32h-3.19V4.75h3.19zm6.25 0v15.44h-3.13V4.75H23z',
        'w' => 0.75,
        'vw' => 24
    ],
    'glide' => 'M15.81 9.31q0 .63-.25 1.63l-1.06 5.25q-.06.31-.88.31-1 0-1.43-.87t-.44-2q0-1.57.6-3.07t1.77-2.12q.5-.25.88-.25.81 0 .81 1.12zM28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22q1.25 0 2.13.88T28 5zm-4 11.69q0-.25-.69-1.31t-.94-1.13q-.3-.19-.56-.19-.31 0-.6.03t-.56.13l-.46.15-.54.25-.46.22-.54.28-.46.32-.07-.07q.07-.43.5-2.4t.44-3.03Q19.06 6 14.81 6q-1.56 0-2.9.75t-2.16 1.9-1.25 2.47-.44 2.6.37 2.28 1.07 1.65 1.69 1.04 2.3.37h.2q.25 0 .25.06l-.07.07q-.43 3.75-2.25 4.43-1 .38-1.18-.5l-.07-.25q0-.43.2-.84t.43-.66.5-.43.44-.32l.19-.12q0-.31-.91-.69t-1.34-.37q-.5 0-.91.3t-.66.82-.37 1.03-.13.97q0 1 .4 1.78t1.17 1.22 1.75.44q2.12 0 3.71-1.75t2.16-4q.13-.38.22-.75t.12-.63.1-.43.1-.32.02-.22.03-.12.1-.06.12-.1q2.07-1.06 3.63-1.06.31 0 2.31.31.25.07.25-.19z',
    'glide-g' => [
        'p' => 'M25.44 13.19q.37.12 1.47 1.84T28 17.2q0 .31-.45.31-3.12-.5-3.68-.5-2.5 0-5.82 1.69l-.19.1-.09.12-.06.15-.1.25-.1.38-.08.5-.16.69-.22.9-.25 1.16q-.94 3.5-3.5 6.28T7.37 32q-2.5 0-3.94-1.53T2 26.5q0-1.56.9-3.25t2.35-1.69q.75 0 2.22.6t1.46 1.1q0 .05-.21.18t-.54.34-.68.57-.7.71-.5.91-.18 1.1q0 1.24 1.19 1.37 1.75.06 3.12-2.56 1-1.88 1.38-4.7 0-.24-.06-.24h-.63q-4.12 0-6.4-2.22t-2.29-6.34q0-2.82 1.16-5.57T7.3 2.03 13.25 0Q15.5 0 17 .63t2.28 2.03.78 3.65q0 1.75-.69 4.94t-.81 3.81l.06.07q.19-.13.72-.4t.72-.41.65-.38.72-.34.7-.25.74-.22.75-.13.88-.06q.37 0 .93.25zM14.5 7.94q.31-1.63.31-2.69 0-.38-.03-.66t-.12-.56-.38-.4-.72-.13q-.68 0-1.37.38-1.94 1.06-2.9 3.46t-.98 4.91q0 1.13.25 2.06t.97 1.72 1.85.78h.56l.28-.03.25-.06.16-.13.12-.28q.81-3.93 1.75-8.37z',
        'vw' => 30
    ],
    'gofore' => [
        'p' => 'M20.25 20q1.19 0 2.06.5t1.32 1.31.62 1.53.25 1.53Q20.25 30 13.81 30 8 30 4 25.87T0 16q0-5.81 4.03-9.9T13.81 2q5.69 0 9.75 4.13l-3.94 3.93q-2.37-2.44-5.8-2.44t-5.88 2.44T5.5 16t2.44 5.94 5.87 2.44q3.25 0 5.63-2.25V20h.8zm-.75-7.06q2.31 0 3.9 1.65t1.6 4.66v1.88q-1.5-2.63-4.75-2.63h-7.38v-5.56h6.63z',
        'w' => 0.781,
        'vw' => 25
    ],
    'golang' => [
        'p' => 'M400.1 194.8C389.2 197.6 380.2 199.1 371 202.4C363.7 204.3 356.3 206.3 347.8 208.5L347.2 208.6C343 209.8 342.6 209.9 338.7 205.4C334 200.1 330.6 196.7 324.1 193.5C304.4 183.9 285.4 186.7 267.7 198.2C246.5 211.9 235.6 232.2 235.9 257.4C236.2 282.4 253.3 302.9 277.1 306.3C299.1 309.1 316.9 301.7 330.9 285.8C333 283.2 334.9 280.5 337 277.5V277.5L337 277.5C337.8 276.5 338.5 275.4 339.3 274.2H279.2C272.7 274.2 271.1 270.2 273.3 264.9C277.3 255.2 284.8 239 289.2 230.9C290.1 229.1 292.3 225.1 296.1 225.1H397.2C401.7 211.7 409 198.2 418.8 185.4C441.5 155.5 468.1 139.9 506 133.4C537.8 127.8 567.7 130.9 594.9 149.3C619.5 166.1 634.7 188.9 638.8 218.8C644.1 260.9 631.9 295.1 602.1 324.4C582.4 345.3 557.2 358.4 528.2 364.3C522.6 365.3 517.1 365.8 511.7 366.3C508.8 366.5 506 366.8 503.2 367.1C474.9 366.5 449 358.4 427.2 339.7C411.9 326.4 401.3 310.1 396.1 291.2C392.4 298.5 388.1 305.6 382.1 312.3C360.5 341.9 331.2 360.3 294.2 365.2C263.6 369.3 235.3 363.4 210.3 344.7C187.3 327.2 174.2 304.2 170.8 275.5C166.7 241.5 176.7 210.1 197.2 184.2C219.4 155.2 248.7 136.8 284.5 130.3C313.8 124.1 341.8 128.4 367.1 145.6C383.6 156.5 395.4 171.4 403.2 189.5C405.1 192.3 403.8 193.9 400.1 194.8zM48.3 200.4C47.05 200.4 46.74 199.8 47.36 198.8L53.91 190.4C54.53 189.5 56.09 188.9 57.34 188.9H168.6C169.8 188.9 170.1 189.8 169.5 190.7L164.2 198.8C163.6 199.8 162 200.7 161.1 200.7L48.3 200.4zM1.246 229.1C0 229.1-.3116 228.4 .3116 227.5L6.855 219.1C7.479 218.2 9.037 217.5 10.28 217.5H152.4C153.6 217.5 154.2 218.5 153.9 219.4L151.4 226.9C151.1 228.1 149.9 228.8 148.6 228.8L1.246 229.1zM75.72 255.9C75.1 256.8 75.41 257.7 76.65 257.7L144.6 258C145.5 258 146.8 257.1 146.8 255.9L147.4 248.4C147.4 247.1 146.8 246.2 145.5 246.2H83.2C81.95 246.2 80.71 247.1 80.08 248.1L75.72 255.9zM577.2 237.9C577 235.3 576.9 233.1 576.5 230.9C570.9 200.1 542.5 182.6 512.9 189.5C483.9 196 465.2 214.4 458.4 243.7C452.8 268 464.6 292.6 487 302.6C504.2 310.1 521.3 309.2 537.8 300.7C562.4 287.1 575.8 268 577.4 241.2C577.3 240 577.3 238.9 577.2 237.9z',
        'vw' => 640,
        'vh' => 512
    ],
    'goodreads' => 'M18.75 11.94q.19 1.25 0 2.4t-.75 2.2-1.5 1.71q-1 .69-2.31.75t-2.13-.37q-1.62-.82-2.34-2.47t-.6-3.66q.2-2.62 1.57-4.03t3.19-1.4q1.43 0 2.46.62t1.63 1.69.78 2.56zM28 5.5v21q0 1.44-1.03 2.47T24.5 30h-21q-1.44 0-2.47-1.03T0 26.5v-21q0-1.44 1.03-2.47T3.5 2h21q1.44 0 2.47 1.03T28 5.5zm-7.38 14.06V6h-1.8v2.5q-.07 0-.13-.06-.5-1.13-1.75-2t-3-.88q-2.38 0-4 1.31t-2.25 3.57q-.38 1.31-.38 2.81.07 3.63 1.94 5.5t5.13 1.75q2.87-.12 4.3-2.87l.07-.13h.06v2.13q0 1.56-.43 2.75-.75 2-2.82 2.43-1.68.38-3.31-.06-2.19-.62-2.56-2.62 0-.07-.13-.07H7.87q.07 1.07.5 1.82 1.07 1.8 3.53 2.4t4.54-.03q4.18-1.06 4.18-6.69z',
    'goodreads-g' => [
        'p' => 'M2.69 25.19h2.56q.19 0 .19.19.62 3 3.94 3.93 2.44.7 5.06.13 3.13-.69 4.25-3.75.69-1.75.69-4.19.06-.44.06-1.75t-.06-1.5l-.06-.06q0 .06-.07.15t-.06.16q-1.06 2-2.78 3.1t-3.85 1.21q-3.18.13-5.56-1.1t-3.72-3.74-1.47-6.22q0-2.31.57-4.31Q3.38 4 5.85 2T12 0q1.75 0 3.31.72t2.5 1.65 1.44 2q.07.2.13.13V.69h2.75l.06 20.75q-.06 8.56-6.44 10.12-2.31.44-4.6.4t-4.43-1.09T3.5 28q-.69-1.19-.81-2.81zM11.8 2.25q-2.68 0-4.8 2.16t-2.38 6.21q-.18 3.07.88 5.6t3.56 3.72q1.25.69 3.28.6t3.54-1.16q2.12-1.44 2.96-4.1t.41-5.53q-.44-3.5-2.31-5.5t-5.13-2z',
        'w' => 0.748,
        'vw' => 24
    ],
    'google' => 'M30.5 16.38q0 6.62-4.16 10.87T15.5 31.5q-6.44 0-10.97-4.53T0 16 4.53 5.03 15.5.5q6.06 0 10.38 4.06l-4.2 4.07q-2-1.94-4.77-2.29t-5.2.7-4.12 3.43T5.87 16q0 4.06 2.82 6.94t6.81 2.87q2.19 0 3.9-.71t2.7-1.82 1.5-2.15.71-2H15.5v-5.38h14.75q.25 1.38.25 2.63z',
    'google-drive' => 'M21.19 19.69 10.94 2h10.12l10.2 17.69H21.18zm-8.57 1.43H32L26.94 30H7.56zm-3-16.93 5.2 8.87L5.05 30 0 21.13z',
    'google-pay' => [
        'p' => 'M6.6 13.44v2.58h3.58a3.1 3.1 0 0 1-1.32 2.03 3.96 3.96 0 0 1-5.96-2.12 4.1 4.1 0 0 1 0-2.56 3.95 3.95 0 0 1 3.7-2.77c.95-.02 1.86.34 2.54 1l1.9-1.91A6.32 6.32 0 0 0 6.6 7.95a6.6 6.6 0 0 0-5.9 3.7 6.75 6.75 0 0 0 0 6v.01a6.6 6.6 0 0 0 5.9 3.7c1.78 0 3.28-.6 4.37-1.63a6.57 6.57 0 0 0 1.97-4.93c0-.46-.04-.91-.11-1.36Zm24.35-.25a3.66 3.66 0 0 0-2.6-.89c-1.4 0-2.45.52-3.15 1.56l1.3.83a2.24 2.24 0 0 1 3.38-.52c.39.34.6.82.6 1.33v.34a4.4 4.4 0 0 0-2.16-.48c-1.03 0-1.85.24-2.46.73a2.4 2.4 0 0 0-.93 1.98c-.02.75.3 1.47.87 1.95.57.52 1.3.78 2.17.78a2.8 2.8 0 0 0 2.44-1.36h.06v1.1h1.42v-4.91c0-1.04-.31-1.86-.94-2.44Zm-1.2 5.58c-.45.44-1.04.7-1.67.7-.41 0-.82-.13-1.14-.4a1.2 1.2 0 0 1-.49-.97c0-.44.2-.8.6-1.09.4-.29.9-.44 1.5-.44.82 0 1.47.18 1.93.55a2.2 2.2 0 0 1-.74 1.65Zm-5.86-8.88a3.48 3.48 0 0 0-2.53-1.02h-3.92v11.68h1.48v-4.73h2.44c1 0 1.84-.34 2.53-1l.16-.16a3.4 3.4 0 0 0-.16-4.77Zm-1.03 3.9c-.38.4-.91.62-1.47.6h-2.47V10.3h2.47a2 2 0 0 1 1.42.58c.8.8.82 2.07.05 2.9Zm15.53-1.23-2.28 5.73h-.03l-2.34-5.73h-1.6l3.23 7.48-1.83 4.01h1.52L40 12.56Zm0 0',
        'vw' => 40
    ],
    'google-play' => 'M20.31 14.63 6.56.8l17.5 10.06zM2.94 0 19 16 2.94 32q-1.38-.69-1.38-2.19V2.2Q1.56.7 2.94 0zM29.5 14.13q.88.62.9 1.87t-.84 1.88L25.81 20l-4.06-4 4.06-4zM6.56 31.18l13.75-13.82 3.75 3.75z',
    'google-plus' => [
        'p' => 'M24.13 14.25q.18 1.06.18 2 0 5.19-3.19 8.47T12.82 28q-5 0-8.5-3.5T.81 16t3.5-8.5 8.5-3.5q4.68 0 8 3.13l-3.26 3.12q-1.87-1.81-4.75-1.81-2 0-3.72 1t-2.72 2.75-1 3.81q0 3.13 2.2 5.34t5.24 2.22q3.13 0 4.82-1.68t2-3.44H12.8v-4.2h11.32zm11.56.44h3.5v3.5h-3.5v3.5h-3.5v-3.5h-3.44v-3.5h3.44v-3.5h3.5v3.5z',
        'w' => 1.25,
        'vw' => 40
    ],
    'google-plus-g' => 'google-plus',
    'google-plus-square' => 'M25 2q1.25 0 2.13.88T28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22zM10.25 22.25q2.69 0 4.34-1.69t1.66-4.44q0-.5-.13-1h-5.87v2.13h3.56q-.18.94-1.06 1.81t-2.5.88q-1.06 0-1.94-.53t-1.4-1.44T6.37 16q0-1.63 1.13-2.78t2.75-1.16q1.5 0 2.5.94l1.69-1.62q-1.75-1.63-4.19-1.63-2.56 0-4.4 1.84T4 16t1.84 4.4 4.41 1.85zM24 17.12v-1.8h-1.81V13.5h-1.82v1.81h-1.8v1.81h1.8v1.82h1.82v-1.82H24z',
    'google-wallet' => 'M9.81 7.94q3.32 5.31 5.25 10.12-.75 3.13-1.93 6.2-1.57-6.2-3.5-9.32.12-.82.18-1.75t.03-1.69-.03-1.81V7.94zm-3 4.56q.5 0 .82.38Q11.06 17.62 12.8 25H6.3Q4.38 18.12.63 13.31q-.2-.25-.07-.53T1 12.5h5.81zm3-5.5h6.82q7.87 10.88 8.93 23h-7.12Q17.8 19.62 9.8 7zm15.75-4.25Q27.5 9.63 27.5 16q0 6.63-1.56 12.69-1.06-8.63-5.69-16.94-.37-4.62-1.62-9.12-.07-.25.06-.44t.37-.2h5.57q.75 0 .93.76z',
    'gratipay' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm7.19 14.13q.19-.32.34-.82t.19-1.15-.34-1.35-1.07-1.19q-1.43-.68-2.47-.4t-1.78 1.15q-.75.82-1.93.85t-1.94-.85-1.82-1.15-2.5.4q-.87.63-1.18 1.63t-.13 1.72.5 1.15l7.06 9.57z',
    'grav' => 'M18.81 13.25q.5.5 0 1l-.62.63q-.5.56-1 0l-.69-.63q-.44-.56 0-1.06l.63-.63q.5-.5 1.06 0zM16.94 12q-.32.31-.69 0-.31-.31 0-.63.38-.37.69 0 .31.32 0 .63zm-1.63.38q.32-.38.63 0 .37.3 0 .62-.32.31-.63 0-.37-.31 0-.63zm4.5-.88q-1-.75-1.31-1.88t.63-2.37q1.62-1.88 3.68.06.07 0 .25.2l.25.24.22.25.22.25.19.28.15.32.07.3v.38l-.1.38-.18.4q-.57 1.13-1.85 1.5t-2.22-.3zm3-2.75q-.25-.31-.6-.16t-.46.5.12.66q.38.5 1 .13.5-.44-.06-1.13zM31.5 16q0 6.44-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5t10.97 4.53T31.5 16zm-4.13 2.69q.07-.63-.71-1.07t-.85-.56q-.69-1.25-1.47-1.72t-1.97-.22q.32-.18.7-.28t.55-.1h.25q0-.74-.62-1.5l.1-.46q.03-.22.02-.37t-.03-.32-.03-.22l-.06-.06q1.62-.87 1.87-2.69.13-1.44-.78-2.6T22 5.19q-.69-.06-1.34.1t-1.2.56-.84.97q-.56.88-.62 1.75t.25 1.5.75 1.1 1 .71q-.81-.06-1.53-.62t-.97-1.5q-.5-1.88.56-3.88l-.5-.5q-.87 0-1.56.32.31-.44.72-.66t.65-.22l.2-.06q0-.82-.2-1.32-.25-.56-.93-.56t-1.07.75v-.06q-.25.44.03 2.28T16.43 9q-.43.25-.8.62-.38.19-.88.66t-.94.97-.85 1-.59.8l-.25.38Q10.56 14 9 15.6t-1.25 2.78q0 .31.12.5l-.5.5q-.62.69-.25 1.35t1.38.21q1.43-.68 2.25-1.93l-.1-.07-.34-.12-.57-.06q2.44-.57 2.88-.57.25.07.37-.47t.13-1.1V16q0-1.75-.69-2.56 1.81 1.75 1.75 3.81 0 .32-.12.57t-.25.4-.25.25-.25.16h-.07l-.3.56-.48.82q-.23.38-.47.9t-.44 1.06-.3 1.1-.1 1.12.15 1.07q-.56-.88-.62-1.38-2.94 3.31-1.5 4.06 1.19.63 4.1-2.8t3.77-6.07q2.88-1.75 4-4.06 1.13 2.18 3.63 3.75 2.44 1.43 2.69-.07z',
    'gripfire' => [
        'p' => 'M7 18.81q0 .32.06.66t.13.56.25.56.25.5.31.57.31.47.41.53.34.53q.13.12.32.37l1.25 1.75q.44.62.81 1.57t.38 1.68q0 1.57-1.07 2.94 0-.12.16-.6t.15-.77q0-1.94-4.93-5.44-1.32-1.13-2.07-1.9t-1.4-2.1T2 17.94q0-2 .94-3.94t2.31-3.47T7.97 7.5t2.28-3.15.94-3.22q0-.32-.06-.63.25.13.81.85t1.13 2.09.56 2.69q0 1.69-1.03 3.47t-2.28 3.06-2.28 2.97T7 18.82zm12.88-2.44q.12.82.12 1.57 0 4.19-4.19 8.56l-.19.13q-.25 0-1.25-1.41t-1-1.9q.32-.38.75-1.04T15.28 20t.72-2.75q0-1.06-.56-2.06-.07.87-.94 2.25t-1.75 2.28-.94.9-.16-.09-.22-.28-.25-.44-.25-.53-.18-.62-.07-.66q0-.5.2-1l.59-.84 1.21-1.85L13.84 12l.53-2.25q0-.63-.19-1.06l-.25-.44q1.2.44 3.32 2.94t2.62 5.18z',
        'w' => 0.687,
        'vw' => 22
    ],
    'grunt' => [
        'p' => 'M3.81 11.81q.07-.56.44-.75-.44-.12-.81-.25 2.87-.68 5.19 2 .43.57 1 .7t.87 0l.38-.13q0 .5-.41.84t-.97.34q.06.32-.03.6t-.22.53-.25.47-.25.34l-.12.13q.06-.07.06-.13t.06-.25.1-.37.03-.44T8.8 15q-.06-.25-.18-.5-.57.56-1.32.5t-1.28-.31-.84-.56q-.75-.82-.81-2.13-.2.38-.25 1 0-.06-.07-.16t-.15-.4-.1-.63zm1.82-.06q-.2.38-.1 1.13T6 13.94q1.13.81 2.38.06-.76-1.12-2.76-2.25zm8.87 2.81q-.56 0-.97-.34t-.4-.84q1.18.62 2.24-.57 2.32-2.69 5.2-2-.38.13-.82.32.38.12.44.68 0 .32-.07.6t-.18.47l-.07.12q-.06-.56-.25-1-.12 1.31-.8 2.13-.2.18-.57.4t-.88.38-1.03.06-.97-.47l-.18.5q-.07.38-.04.72t.1.66l.12.25q-.12-.13-.28-.35t-.4-.75-.2-.97zm3.88-2.81q-2 1.13-2.75 2.25 1.18.75 2.3-.06.45-.32.54-1.07t-.1-1.12zM10 26.13q.38 0 .53.18t.1.44q-.26.63-.82 1.25-.31.25-.72.25T8.37 28q-.62-.56-.87-1.12-.13-.38.06-.57t.63-.18H10zm11.81-6.32.5.5.7 1.47.12 2.47q-.44 2.5-2.94 3.44-1.25 2.81-4.19 3.06Q14.5 32 12 32t-3.94-1.25q-2.94-.25-4.25-3.06-1.19-.44-1.94-1.28t-1-2.16.1-2.44.78-1.56l.44-.44q.19 1.63 1.56 2.75.19-1.68.19-3.87 0-.63-.28-1.06t-.47-.57-.5-.31q-.81-.44-1.22-.69t-.88-.81T.13 14q0-1 1-1.69 0-.06.06-.1l.06-.02q.25-.2.25-.32.19-.37.22-1.28t-.1-1.03Q.57 8.31.89 7.12q.18-.75.87-1.06.5-.25 1.25-.31 0-.13.06-.38.07-.31.07-.5-.07-.56-.2-.69-.18-.37-.65-.65t-.78-.38-.37-.1l-.75-.12.5-.56Q2 1.25 3.75 1.25q.75 0 1.5.25 1.69.5 3.13 2.43.25-.12.37-.12-.06-1.13.6-2.03T10.5.8v.31l.03.53.13.5.28.35Q11.19.75 13.38 0q-.38 1.06.12 2.3.57-.5 1.1-.74t.78-.25l.25.06q-.13.13-.25.4t-.28.91-.03 1.07q.3.06.56.19Q17.07 2 18.75 1.5q.75-.25 1.5-.25.82 0 1.53.28t1.03.5.32.28l.5.62-.75.13q-.07 0-.38.1t-.75.37-.69.65q-.12.13-.12.7-.06.18.06.5v.37q.75 0 1.25.25.7.37.88 1.12.12.5-.07 1.13t-.37.9-.31.4-.1 1.04.22 1.28q.06.13.25.31.06 0 .13.07 1 .75 1 1.75 0 .62-.47 1.22t-.88.84-1.22.69q-.43.25-.5.25-.18.19-.47.62t-.28 1.07q0 2.18.25 3.87 1.32-1.12 1.5-2.75zM19.13 2.38q-1.32.37-2.38 2l.31.12q.25.13.44.25.38-.75 1.25-1.44.81-.56 1.81-.56.32 0 .57.06.31-.19.62-.31-.75-.31-1.56-.31-.56 0-1.06.18zm1.3 1.12q-.68 0-1.24.44t-.82 1.37q.38.38.63.75.31-.12.75-.18.19-.32.25-1.1t.25-1.1zM2.89 2.81q.25 0 .56-.06 1 0 1.75.56.94.7 1.31 1.44.31-.19.44-.25t.31-.12q-1.06-1.63-2.37-2-.5-.2-1.07-.2-.8 0-1.56.32.31.13.63.31zm2.75 2.5q-.25-.93-.82-1.37-.5-.44-1.25-.44l.19.19q.19.37.25 1.12t.25 1.07q.31.06.75.18.19-.37.63-.75zm-2 10.94q1.12.63 1.18 2.13.13 1.56-.18 4.75.93.37 2.43.43.25-.43.38-.81 1.06-3 1.62-3.88-.37 2.7-1.43 5.57.37-.07.75-.07h7.24q.44 0 .76.07-1.07-2.88-1.44-5.57.68 1.07 1.62 3.88.13.38.38.81 1.5-.06 2.43-.43-.3-3.2-.18-4.75.06-1.5 1.18-2.13.44-.25.5-.31.63-.32.94-.5t.66-.6.34-.78q0-.69-.56-1.12l-.06-.07-.07-.03-.06-.1q.13.82.19.95-.75-.94-1-2.75-.19-1.57.31-2.7v.88L21.7 9q.75-.94.56-1.63-.06-.37-.44-.5-.31-.18-.94-.18-.87 0-2.06.37-1.62.5-4 2.38l-.37.25.25-.38q1.37-2.06 3.3-2.94-.3-.37-1.37-.94-2-.93-4.62-.93t-4.63.93Q6.31 6 6 6.38q2 .88 3.31 2.94l.25.38-.37-.25q-2.38-1.88-4-2.38-1.2-.37-2.07-.37-.56 0-.93.18-.38.13-.44.5-.19.7.56 1.63l.13.12v-.87q.5 1.12.3 2.69-.12.81-.34 1.5t-.4 1l-.25.25q.12-.13.19-.94l-.2.19q-.5.43-.5 1.12 0 .56.48.97t1.4.9q.38.26.5.32zM2.2 25.31q1.12 1.82 3.62 1.82-.43-1.25.25-2l.07-.07q.3-.31.5-.62-3.82-.19-4.7-2.81-.3.12-.27 1.46t.53 2.22zM5 28q.81 1.13 1.94 1.63-.38-.57-.75-1.5-.57 0-1.19-.13zm7 3.13q3.88 0 5.31-4 .32-1-.06-1.5-.38-.44-1.56-.44H8.3q-1.19 0-1.56.47t-.06 1.47q1.43 4 5.3 4zm5.06-1.5Q18.2 29.13 19 28q-.62.13-1.19.13-.37.93-.75 1.5zm4.75-4.32q.5-.87.53-2.22t-.28-1.47q-.87 2.63-4.62 2.82.19.31.44.62l.06.07q.69.75.25 2 2.56 0 3.62-1.82zM12 11.75q-.38 0-.78-.28t-.66-.6l-.25-.24q.82.5 1.7.5t1.68-.5q-.88 1.12-1.69 1.12zm3.88 14.38q.37.06.56.21t.06.47q-.25.63-.87 1.13-.32.31-.72.31t-.72-.31q-.57-.63-.82-1.19-.06-.25.1-.44t.53-.19h1.87z',
        'w' => 0.75,
        'vw' => 24
    ],
    'guilded' => [
        'p' => 'M27.71 4H.3c0 6.45 1.38 11.25 2.71 13.9C7 25.88 14 28 14.08 28a19.53 19.53 0 0 0 8.78-6.47 24.32 24.32 0 0 0 4.12-9.1H10.74a8.37 8.37 0 0 0 2.82 5.43h5.53a12.03 12.03 0 0 1-5.02 4.34c-1.96-.83-4.32-2.91-6.04-6.15a16.83 16.83 0 0 1-1.69-6.62h21c.26-1.8.38-3.61.37-5.43Zm0 0',
        'vw' => 28
    ],
    'gulp' => [
        'p' => 'M13.13 24.44 12.24 26l-.31 5q0 .44-1.16.72T8 32q-1.06 0-1.97-.12T4.6 31.5t-.53-.5l-.37-5-.94-1.56q1.94.81 5.22.81t5.15-.81zM.87 8.62q1.44.82 7.13.79t7.13-.78l-.94 8.5q-.75.68-1 .87-.44.31-.5.31-.2 0-.2-.12.07 0 .38-.38.82-1 .82-1.43 0-.25-.2-.41t-.55-.1-.82.44l-.31.32q.06-.2.06-.38 0-.25-.25-.25-.06 0-.19.06-.12.2-.3.75 0 .07-.13.32l-.07.06q-1 1.06-1.43 1.06-.13 0-.07-.37.13-.25.57-1.38 0-.12.3-.84t.5-1.16q.13-.31.26-.56 0-.06.06-.13.06-.25-.06-.37-.13-.06-.19-.06-.31 0-.44.31-1.56 4-1.62 4.25-.25.19-.44.25-.44.19-.44-.06t.19-.75q.06-.2.22-.53t.22-.47q.12-.32 0-.44t-.25-.13q-.19 0-.32.32l-.12.31q-.06.31-.31.81-.32.57-.7.82-.24.12-.43.12-.25 0-.12-.56.12-.32.62-1.25.13-.38-.06-.5H6.8q-.3-.13-.5.25l-.43 1q-.44 1.19-.32 1.5t.57.31q.87 0 1.37-.94-.19.88.25.97t1-.47q0 .07.03.1l.03.03q.13.56.88.25.44-.19.94-.69l-.07.19q-.93 2.06-.97 2.12t-.03.2.13.18q.12.12.37.06.13-.12.25-.31 0-.06.03-.1t.04-.06.06-.25.19-.47q.18-.5.56-1.37.25-.57.56-.94.25-.44.5-.63.63-.43.75-.18 0 .18-.31.5-.88 1.18-.88 1.37 0 .38.41.44t.66-.06q.43-.2 1.25-.94l-.7 5.94q-.05.12-.74.37t-2.03.47-2.75.22-2.72-.22-2-.47-.75-.37l-.5-5q.19.12.5.12.56 0 1.31-.75.19-.12.44-.37-.44 1.31-.5 1.56 0 .31.06.37.07.13.2.13.24 0 .37-.5l.5-1.63q.12-.5.3-.96t.26-.7.06-.46q0-.19-.12-.28t-.28-.03-.22.31l-.32.94q-.25.44-.8 1-.63.62-1 .62-.2 0-.32-.19-.19-.43.06-1.25.25-.87.75-1.78t1.06-1.28q.7-.5 1.07.2.25.43.43.43.38 0 .25-.56-.06-.44-.62-.75T4.18 13q-.81.38-1.5 1.38t-1 2.06zm14.38-.56q0 .63-3.63.88t-7.24 0-3.63-.88q0-.37 2.13-.68T8 7.06q.94 0 1.63.07l.8-3.07 3.82-4q.19-.19.63.19.5.44.3.56v.07l-3.55 3.75L11 7.19q4.25.25 4.25.87zm-4.38.32q0-.07-.12-.13l-.06.06q0 .13-.7.13t-.68-.13v-.06h.06q-.25.06-.25.13 0 .24.88.24t.87-.24z',
        'w' => 0.498,
        'vw' => 16
    ],
    'hacker-news' => 'M0 2h28v28H0V2zm14.94 15.69L20 8h-2.19q-1.5 2.81-2.22 4.28t-1.06 2.34-.47 1.25l-.4-1-.5-1.21q-.22-.51-.94-2.04T10.32 8H8l5 9.56V24h1.94v-6.3z',
    'hacker-news-square' => 'M25 2q1.25 0 2.13.88T28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22zM14.94 17.69 20 8h-2.19q-1.5 2.81-2.22 4.28t-1.06 2.34-.47 1.25l-.4-1-.5-1.21q-.22-.51-.94-2.04T10.32 8H8l5 9.56V24h1.94v-6.3z',
    'hackerrank' => [
        'p' => 'M28.34 8q.66 1.19.66 8t-.66 8-6.56 4.6T14.5 32t-7.25-3.4T.69 24t-.7-8 .7-8 6.56-4.6T14.5 0t7.28 3.4T28.34 8zM18.25 25.88q.19 0 1.34-1.13t1.04-1.25q-.07-.13-1.13-.13 0-.5-.03-1.25t-.03-1.59V17l.12-6.81q0-.32-.31-.32h-2.06q-.25 0-.25.32.06 2.31.06 5.25h-5l.13-6.63v-.19q1 0 1.06-.12.12-.13-1.03-1.25T10.8 6.12q-.06 0-.22.13t-.4.31-.5.44l-.5.5-.44.47-.28.37-.1.16q.13.12 1.13.12v.2q.13 7.3 0 13.3 0 .38.31.38h1.94q.38 0 .38-.37-.13-1.7-.13-4.5h5q0 .06.03.15t.03.16q-.06 3-.06 5.44-1.06 0-1.13.09t.47.69 1.16 1.15.75.57z',
        'vw' => 29
    ],
    'hashnode' => [
        'p' => 'M35.19 171.1C-11.72 217.1-11.72 294 35.19 340.9L171.1 476.8C217.1 523.7 294 523.7 340.9 476.8L476.8 340.9C523.7 294 523.7 217.1 476.8 171.1L340.9 35.19C294-11.72 217.1-11.72 171.1 35.19L35.19 171.1zM315.5 315.5C282.6 348.3 229.4 348.3 196.6 315.5C163.7 282.6 163.7 229.4 196.6 196.6C229.4 163.7 282.6 163.7 315.5 196.6C348.3 229.4 348.3 282.6 315.5 315.5z',
        'vw' => 512,
        'vh' => 512
    ],
    'hips' => [
        'p' => 'M15.75 9.88v12.56q0 .19-.19.19H13q-.19 0-.19-.2V9.89q0-.2.19-.2h2.56q.19 0 .19.2zm-6 .62q1.75 1.25 1.75 3.44v8.5q0 .19-.13.19H9.63q-.38 0-.7-.28t-.3-.72v-7.7q.06-1.12-1.44-1.5l-1.44-.18q-.82 0-1.44.19t-.88.37-.37.53-.16.41-.03.19v8.5q0 .19-.12.19H.19q-.2 0-.2-.2V5q0-.13.2-.13h2.56q.12 0 .12.13v5.06Q4.31 9.37 6 9.37q2.25 0 3.75 1.13zm29.94 6.88q.5 1.18.19 2.43-.57 2.2-2.82 2.88H37q-.88.25-2.25.25-1.56 0-3-.5-.94-.32-.94-1.32V19q0-.13.1-.16t.22.04q1.56 1.18 3.62 1.18 1.94 0 2.31-.87.32-.82-.81-1.32l-3.31-1.5q-2-.87-2.32-2.68-.37-1.94 1.38-3.38 1.31-.94 3.69-.94 1.75 0 3.44.63.12 0 .12.12v2.82q0 .31-.25.12-2.06-.81-3.4-.81t-1.85.44q-.31.18-.28.5t.4.5l3.63 1.56q1.56.69 2.19 2.13zm-16.19-8q2.88 0 4.84 1.96t1.97 4.82-2 4.81-4.81 1.97q-2.06 0-3.62-1.19v5.56q0 .2-.13.2h-2.56q-.2 0-.2-.2v-11.5q0-1.8.73-3.15t1.81-2.03 2.1-.97 1.87-.28zm0 10.68q1.63 0 2.75-1.15t1.13-2.78q0-1.07-.5-1.94t-1.41-1.4-1.97-.54q-1.62 0-2.75 1.16t-1.12 2.75 1.12 2.75 2.75 1.15zM14.25 4.5q.75 0 1.28.53t.53 1.28-.53 1.28-1.28.53-1.28-.53-.53-1.28.53-1.28 1.28-.53z',
        'w' => 1.25,
        'vw' => 40
    ],
    'hire-a-helper' => 'M27.69 0q.12 1.13.72 2.1t1.53 1.59T32 4.5v23.13q-1.69.18-2.9 1.43T27.74 32H4.5q-.31-1.69-1.53-2.87T0 27.69V4.5q1.75-.19 3.03-1.47T4.5 0h23.2zm-2.32 25.31q.38 0 .35-.34t-.34-.35q-.88-.3-1.38-.87-.31-.56-.4-2.1t-.1-5.9V9.44q0-.63.28-1.07t.63-.62.9-.31q.38-.13.28-.44t-.34-.32q-1.13 0-1.78.07t-1.78.06q-1 0-1.53-.06t-1.54-.07q-.18 0-.28.29t.22.4l.2.03.27.1.31.18.35.29.28.4.22.63.06.81q0 3.62-.06 4.31l-.03.25q-.03.18-.25.38t-.47.19h-6.88q-.25 0-.47-.2t-.24-.37-.04-.25q-.06-.69-.06-4.31 0-.44.06-.81t.22-.63.28-.4.35-.29.31-.18.28-.1l.19-.03q.31-.12.22-.4t-.28-.29q-1 0-1.53.07t-1.53.06q-1.13 0-1.79-.06t-1.78-.07q-.25 0-.34.32t.28.44q.88.18 1.34.62t.47 1.38v6.3q0 4.38-.09 5.91t-.4 2.1q-.5.56-1.38.87-.31 0-.35.35t.35.34q1.12 0 1.72-.06t1.72-.06 1.78.06 1.72.06q.25 0 .25-.28t-.25-.34q-1.25-.44-1.57-.82-.18-.3-.28-1.28t-.1-1.84.04-2.22.03-1.72q0-.25.25-.53t.56-.28h7q.32 0 .57.28t.25.53l.03 1.72q.03 1.34.03 2.22t-.1 1.84-.28 1.28q-.25.38-1.56.82-.25.06-.25.34t.25.28q1.06 0 1.72-.06t1.78-.06 1.72.06 1.72.06z',
    'hive' => 'M16.27 15.93 8.22 2.07A.14.14 0 0 0 8.1 2c-.05 0-.1.03-.12.07L.02 15.93a.13.13 0 0 0 0 .14l8.05 13.86c.02.04.07.07.12.07.05 0 .1-.03.12-.07l7.96-13.86a.15.15 0 0 0 0-.14Zm2.44-1.6c.03.04.08.06.12.06H23c.05 0 .1-.03.12-.07a.14.14 0 0 0 0-.14L16.2 2.07a.14.14 0 0 0-.13-.07h-4.15c-.05 0-.1.03-.12.07a.14.14 0 0 0 0 .14Zm13.27 1.6L24.05 2.07a.14.14 0 0 0-.11-.07h-4.17a.14.14 0 0 0-.11.07.12.12 0 0 0 0 .14L27.54 16l-7.88 13.8a.12.12 0 0 0 0 .13c.02.04.07.07.11.07h4.17c.04 0 .1-.03.11-.07l7.93-13.86a.13.13 0 0 0 0-.14Zm-9.1 1.88h-4.16c-.05 0-.1.02-.12.07l-6.8 11.91a.14.14 0 0 0 0 .14c.02.04.07.07.12.07h4.15c.05 0 .1-.03.13-.07L23 18.02a.14.14 0 0 0 0-.14.14.14 0 0 0-.13-.07Zm0 0',
    'hooli' => [
        'p' => 'M9 22v-1q1.19.63 2.44 1.06zm3.63-.31q2.37.81 4.74.81-2.56.69-5.93-.44l1.19-.06v-.31zm25.56-7.19q-.75 0-1.28-.53t-.53-1.28.53-1.16 1.31-.4 1.28.4.5 1.16-.53 1.28-1.28.53zM36.38 22v-6.88H40V22h-3.63zm-4.63 0V11.5l3.63-1.69V22h-3.63zm-1.94-7.44q.44.38.66.75t.31.9.1.7v1.93l-.1.72-.34.82-.63.75q-.31.31-.75.53t-.84.31-.75.16-.6.06h-.18q-2 .06-3.13-1.06-.44-.44-.65-.82t-.32-.9-.1-.69v-1.16q0-.68.04-1.03t.28-.9.75-1.07 1.28-.78 1.28-.28h.57q2-.12 3.12 1.06zM27.38 19v-2.5q0-.69-.7-.69t-.68.69V19q0 .81.69.81t.68-.81zm-6.7-4.44q.45.38.66.75t.32.9.1.7v1.9l-.04.35-.06.31-.1.31-.12.32-.19.34-.25.31-.31.32q-.32.3-.75.56t-.85.34-.75.16-.6.06l-.18-.06q-2 .12-3.12-1.06-.44-.38-.66-.75t-.31-.91-.1-.69v-.9q0-1.88.57-2.75-2.2-1.63-6.25-3.63v3.38q.8-1.38 2.43-1.38 1.2 0 1.85.69t.65 1.69v5.87q-1.87-.75-3.56-1.87v-3.07q0-.93-.69-.93T7.7 17v1.82l-.26-.2q-2.06-1.68-3.37-3.18v1.94q1.13 1 2.44 2 .56.44 1.19.81V22H4.06v-4.62q-1.06-1-1.87-1.9t-1.25-1.6-.66-1.25-.25-.97 0-.69.1-.4l.06-.07q.44-.93 2-.97t4.12.91l1.38-.62v1.3q4 1.95 6.37 3.82l.38-.37q.31-.32.75-.57t.84-.34.75-.16.6-.06l.18.06q2-.12 3.13 1.07zM4.07 11.5l1.82-.88q-2.88-1.06-4-.12-.2.19-.25.6t.5 1.62 1.93 2.72V11.5zM18.25 19v-2.5q0-.69-.69-.69t-.69.69V19q0 .81.7.81t.68-.81z',
        'w' => 1.25,
        'vw' => 40
    ],
    'hornbill' => 'M4.69 23.13q.25 1.75-1.32 2.5-.87.37-1.78.06T.25 24.5t-.13-1.81 1.2-1.38Q2 21 2.68 21.12 1.3 19.2 1.16 17.25t.6-3.34 1.65-2.5 1.72-1.66l.8-.56-.5 5.75q-1 .56-1.43 1.15t-.31 1.03.5.82 1.1.68 1.34.5 1.3.32q.95 2.56 3.2 4.15t5.06 1.6h.31v1.18q-7.3.13-11.8-3.25zm6.75-20.44q1.94-1.32 3.84-1.44t3.28.6 2.47 1.68 1.63 1.72.53.81L17.5 5.5q-.69-1-1.25-1.44t-1.03-.28-.85.6-.68 1.18-.5 1.38-.32 1.37q-2.43 1-3.97 3.2t-1.53 4.93v.19H6Q5.94 9.19 9.3 4.75 7.7 4.88 7 3.38q-.44-.88-.13-1.78T8.07.25t1.8-.12 1.38 1.19q.31.62.19 1.37zm19.06 8.37q-.56.25-1.25.2 1.5 2 1.72 3.96t-.5 3.4-1.66 2.57-1.75 1.69l-.87.56.56-5.69q1.13-.75 1.53-1.4t.13-1.13-1-.9-1.47-.7-1.63-.43q-1-2.44-3.22-4t-4.9-1.56l-.57.06V6.25q7-.06 11.5 3-.12-.75.22-1.44t1.1-1.06q.87-.37 1.78-.06t1.34 1.19.13 1.8-1.19 1.38zm-7.19 16.19q1.5 0 2.2 1.38.37.87.05 1.78t-1.18 1.34-1.82.13-1.37-1.2q-.19-.43-.19-.87-1.13.75-2.28 1.16t-2.1.44-1.87-.22-1.69-.7-1.4-.93-1.13-1.03-.84-.97-.57-.69L9 26.62l5.69.5q.75 1.2 1.4 1.57t1.13.09.9-1.03.66-1.5.47-1.63q2.5-.93 4.1-3.18t1.59-5v-.38h1.18q.13 6.81-2.8 11.19z',
    'hotjar' => 'M25.94 10.13q1.44 2.5 1.87 5.46t-.25 5.79q-.75 2.87-3.22 5.43t-6.09 3.7q1.69-2.38 2.1-5.38t-.6-5.32T16.56 17q2 3.44.07 6.25-1.57 2.19-3.94 1.81-1.75-.25-3.07-2.19l-.4 1.6q-.22.9-.16 2.94t.88 3.6Q3 30 .87 25.18q-1.56-3.75-.25-6.97t5.2-5.84Q8.05 10.8 9.08 9t.85-3.34-.7-2.85T8.2.75L7.56 0q.56.06 1.5.25t3.56.94 4.88 1.78 4.69 2.9 3.75 4.25z',
    'houzz' => [
        'p' => 'M16.19 20.69H9.62V30H0V2h6.88v6.5l19.06 5.38V30h-9.75v-9.3z',
        'w' => 0.808,
        'vw' => 26
    ],
    'html5' => [
        'p' => 'M0 2h24l-2.19 24.75L12 30l-9.81-3.25zm19.25 8 .31-3H4.5l.75 9.13h10.44L15.3 20l-3.31.88L8.62 20l-.24-2.38H5.44l.37 4.75 6.19 1.7h.06l6.07-1.7.87-9.3H8L7.75 10h11.5z',
        'w' => 0.75,
        'vw' => 24
    ],
    'hubspot' => 'M16.69 13.25 6.5 6.06q-.81.44-1.69.44-1.31 0-2.28-.94t-.97-2.28.97-2.31T4.84 0t2.32.97.97 2.28q0 .44-.13.87 1.81 1.38 5.88 4.35t4.68 3.47q1.25-.7 2.63-.88V7.25q-1.75-.75-1.75-2.69 0-1.25.84-2.12t2.07-.88 2.09.88.88 2.12q0 1.94-1.75 2.69v3.87q2.93.44 4.9 2.72t1.97 5.28q0 3.38-2.38 5.75t-5.74 2.38q-2.75 0-4.94-1.63l-3.13 3.13q.13.31.13.75 0 1.06-.75 1.75-.75.75-1.82.75t-1.75-.75q-.75-.69-.75-1.75t.72-1.78 1.78-.72q.44 0 .88.12L15.8 24q-1.62-2.13-1.62-4.88 0-3.5 2.5-5.87zm5.6 10.19q1.77 0 3.05-1.25t1.28-3.06q0-1.2-.59-2.16t-1.56-1.56-2.16-.6q-1.81 0-3.06 1.28T18 19.16t1.25 3.03 3.03 1.25z',
    'ideal' => [
        'p' => 'M7.85 10.34a3.07 3.07 0 1 0 0 6.14 3.07 3.07 0 0 0 0-6.14ZM5.38 26.61h4.94v-8.78H5.38Zm9.47-13.22c0-1.25-.62-1.41-1.17-1.41h-.88v2.86h.88c.6 0 1.17-.17 1.17-1.45Zm12.6 2.88v-5.71h1.43v4.28h2.1c-.57-7.77-6.7-9.45-12.25-9.45H12.8v5.17h.88c1.6 0 2.59 1.09 2.59 2.83 0 1.8-.97 2.88-2.6 2.88h-.87v10.35h5.93c9.04 0 12.18-4.2 12.3-10.35Zm-6.85 0h-3.52v-5.7h3.4v1.41H18.5v.66h1.88v1.42H18.5v.78h2.1Zm4.66 0-.32-1.11h-1.86l-.32 1.1h-1.48l1.72-5.7h2.02l1.72 5.7ZM18.73 2H2v28h16.73c10.11 0 15.68-4.98 15.68-14.03C34.41 10.75 32.38 2 18.73 2Zm0 26.68H3.32V3.32h15.4c8.89 0 14.38 4.04 14.38 12.65 0 8.38-5.06 12.71-14.37 12.71Zm5.24-16.55-.47 1.61h1.02l-.47-1.61Zm0 0',
        'vw' => 36
    ],
    'imdb' => 'M25 2q1.25 0 2.13.88T28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22zM6.06 20v-8H4v8h2.06zm7.06 0v-8h-2.68L10 15.75 9.44 12h-2.7v8h1.82v-5.31L9.31 20h1.31l.75-5.44V20h1.75zm.76 0h.93q2.88 0 3.35-.44t.4-3.06v-1.7q0-1.12-.12-1.5-.13-.62-.66-.94t-.97-.34-1.18-.03h-1.76v8zm10-1.81v-2.25q0-.69-.04-1.03t-.37-.63-.97-.28q-.75 0-1.31.63V12h-1.94v8h1.87l.07-.5q.56.63 1.31.63 1.38 0 1.38-1.94zm-7.32-3.82v3.79l-.15.3-.47.16v-5.25q.31 0 .44.13t.15.25.03.44v.19zM21.88 16v2.06q0 .88-.38.88-.13 0-.22-.16t-.1-2.78q0-.81.32-.81.37 0 .37.81z',
    'innosoft' => [
        'p' => 'M26.41 9.98a1.7 1.7 0 0 0-.87-1.46 1.64 1.64 0 0 0-1.63.01L4.62 19.66v-8.64c0-.27.15-.52.38-.65l8.63-4.98a.76.76 0 0 1 .74 0l2.04 1.18-9.37 5.41v3.5l15.44-8.91-7.41-4.28a2.1 2.1 0 0 0-2.11-.01L2.67 8.2a2.12 2.12 0 0 0-1.08 1.83V22a1.68 1.68 0 0 0 2.51 1.47l19.28-11.13v8.64c0 .27-.14.51-.37.64l-8.64 4.99a.76.76 0 0 1-.74 0l-2.04-1.18 9.37-5.41v-3.5L5.52 25.43l7.43 4.29c.66.38 1.48.37 2.14-.02l10.28-5.94a2.1 2.1 0 0 0 1.04-1.84Zm0 0',
        'vw' => 28
    ],
    'instagram' => 'M14 8.81q3 0 5.1 2.1t2.09 5.1-2.1 5.08T14 23.2t-5.1-2.1T6.82 16t2.1-5.1T14 8.82zm0 11.88q1.94 0 3.31-1.38T18.7 16t-1.38-3.31T14 11.3t-3.31 1.38T9.3 16t1.38 3.31T14 20.7zM23.13 8.5q0 .69-.47 1.19t-1.16.5-1.19-.5-.5-1.19.5-1.16 1.19-.46 1.16.46.46 1.16zm4.8 1.75Q28 11.94 28 16t-.06 5.75q-.19 3.81-2.28 5.9t-5.9 2.23Q18.05 30 14 30t-5.75-.13q-3.81-.18-5.87-2.25-.82-.75-1.32-1.75t-.72-1.93-.21-2.2Q0 20.07 0 16t.13-5.81Q.3 6.44 2.38 4.37q2.06-2.12 5.87-2.31Q9.94 2 14 2t5.75.06q3.81.19 5.9 2.28t2.29 5.9zm-3 14q.2-.5.32-1.28t.19-1.88.06-1.78V12.7l-.06-1.78-.19-1.88-.31-1.28q-.75-1.94-2.69-2.69-.5-.18-1.28-.31t-1.88-.19-1.75-.06H10.7l-1.78.06-1.88.19-1.28.31q-1.94.75-2.69 2.7-.18.5-.31 1.27t-.19 1.88-.06 1.75v6l.03 1.53.1 1.6.18 1.34.25 1.12q.82 1.94 2.69 2.69.5.19 1.28.31t1.88.19 1.75.06h6.65l1.78-.06 1.88-.19 1.28-.31q1.94-.81 2.69-2.69z',
    'instagram-square' => [
        'p' => 'M14 12.67a3.33 3.33 0 1 0 0 6.66 3.33 3.33 0 0 0 0-6.66Zm7.8-2.56a3.37 3.37 0 0 0-1.9-1.9c-1.32-.53-4.44-.4-5.9-.4s-4.58-.13-5.9.4a3.37 3.37 0 0 0-1.9 1.9c-.51 1.3-.4 4.43-.4 5.89 0 1.45-.11 4.58.4 5.9a3.37 3.37 0 0 0 1.9 1.9c1.32.51 4.45.4 5.9.4 1.46 0 4.58.12 5.9-.4a3.37 3.37 0 0 0 1.9-1.9c.52-1.32.4-4.45.4-5.9 0-1.45.12-4.58-.4-5.9ZM14 21.13a5.12 5.12 0 1 1 0-10.25 5.12 5.12 0 0 1 0 10.24Zm5.34-9.27ZM25 2H3a3 3 0 0 0-3 3v22a3 3 0 0 0 3 3h22a3 3 0 0 0 3-3V5a3 3 0 0 0-3-3Zm-1.07 18.13c-.08 1.6-.45 3.01-1.62 4.18-1.17 1.17-2.58 1.54-4.18 1.62-1.65.09-6.6.09-8.25 0-1.6-.08-3.02-.45-4.2-1.62-1.16-1.17-1.53-2.59-1.6-4.18-.1-1.66-.1-6.6 0-8.25.08-1.6.44-3.02 1.6-4.2 1.18-1.16 2.6-1.53 4.2-1.6 1.65-.1 6.6-.1 8.24 0 1.61.08 3.03.44 4.2 1.61 1.16 1.17 1.54 2.6 1.61 4.2.1 1.64.1 6.58 0 8.23Zm0 0',
        'vw' => 28
    ],
    'instalod' => 'M9.59 30h14.6l7.22-12.77-18.64 3.6Zm21.96-15L24.2 2H9.73l12.79 14.74ZM7.77 3.05.45 16 7.7 28.82l6.4-18.47Zm0 0',
    'intercom' => 'M24.5 2q1.44 0 2.47 1.03T28 5.5v21q0 1.44-1.03 2.47T24.5 30h-21q-1.44 0-2.47-1.03T0 26.5v-21q0-1.44 1.03-2.47T3.5 2h21zm-6.75 5.13v12.5q0 .87.94.87t.94-.88V7.13q0-.93-.94-.93t-.94.93zm-4.69-.44v13.5q0 .94.94.94t.94-.94V6.69q0-.94-.94-.94t-.94.94zm-4.69.43v12.5q0 .88.94.88t.94-.88V7.13q0-.93-.94-.93t-.93.93zM3.75 9v8.38q0 .93.94.93t.94-.93V9q0-.94-.94-.94T3.75 9zm20.19 14.25q.25-.25.31-.56t-.06-.53-.35-.41-.53-.19-.56.25q-3 2.57-8.69 2.57T5.25 21.8q-.25-.25-.56-.25t-.54.19-.34.4-.06.54.31.56q3.63 3 10.03 2.97t9.85-2.97zm.31-5.88V9q0-.94-.94-.94t-.94.94v8.38q0 .93.94.93t.94-.93z',
    'internet-explorer' => 'M30.19 10q2.37 4.25 1.69 8.81H11.3q0 2.25 1.47 3.85t3.44 2.1 4.1-.26 3.3-2.75h7.57q-1.5 4.25-5.22 6.69t-8.34 2.44q-4.38 0-6.38-.94Q9.13 31 7.2 31.54t-3.6.43T.98 30.7 0 27.31q0-2 .81-4.53t1.94-4.53q3.57-6.43 8.5-10.81-3.31 1.38-7.62 6.31Q4.82 8.44 9.1 5.1t9.34-3.03Q24.5-.63 28 .16t3.5 4.34q0 2.44-1.31 5.5zM4.06 21.69Q2.94 24 2.53 25.79t-.12 2.8 1 1.63 1.81.66 2.37-.32 2.72-1.18q-4.56-2.63-6.25-7.7zm7.25-6.82h12.94q0-1.68-.94-3.06t-2.4-2.12-3.1-.75q-2.62 0-4.53 1.68t-1.97 4.26zm11.5-11.74q4.2 1.62 7 6.25.94-2.38.94-4.26 0-1-.44-1.75T29.1 2.2t-1.72-.44q-1.68 0-4.56 1.37z',
    'invision' => 'M25.44 2q1.06 0 1.81.75T28 4.56v22.88q0 1.06-.75 1.81t-1.81.75H2.56q-1.06 0-1.81-.75T0 27.44V4.56Q0 3.5.75 2.75T2.56 2h22.88zM11 9.13q0-1.07-.84-1.47t-1.7 0-.84 1.46q0 .7.5 1.2t1.25.5q.2 0 .53-.1t.72-.53.38-1.06zm9.81 14.43q2.38 0 3.32-3.25L23 19.88q-.69 1.87-1.44 1.87-.56-.06-.56-.81.06-.38.94-3.7.44-1.55.22-2.74t-1.03-1.75-1.75-.6-2 .7-1.82 2.02l.57-2.5h-4.25l-.63 2.25h2L12 19.56q-.81 1.88-2.44 2.13-1.12.19-1.12-.7 0-.5 2.06-8.62H5.7l-.57 2.25h2q-1.43 5.82-1.5 6.5 0 1.44 1.07 2.13t2.5.19 2.37-2.07l-.5 2h2.82l1.62-6.43q.56-2.25 2.13-2.5 1.3-.2 1.68 1.06.2.62-.06 1.37l-.31 1q-.25.8-.35 1.25t-.25 1.22-.06 1.25.31.97.78.75 1.44.25z',
    'ioxhost' => [
        'p' => 'M38.5 10q.63 0 1.06.44T40 11.5t-.44 1.06-1.06.44h-3.31q.31 1.44.31 3 0 6.44-4.53 10.97T20 31.5q-4.75 0-8.63-2.62T5.7 22H1.5q-.62 0-1.05-.44T0 20.5t.44-1.06T1.5 19h3.3q-.31-1.44-.31-3 0-6.44 4.53-10.97T20 .5q4.75 0 8.62 2.63t5.7 6.87h4.18zm-6 6q0-1.5-.38-3H13.5q-.63 0-1.06-.44T12 11.5t.44-1.06T13.5 10H31q-1.63-2.94-4.56-4.72T20 3.5q-5.19 0-8.84 3.65T7.5 16q0 1.5.37 3H26.5q.62 0 1.06.44T28 20.5t-.44 1.06-1.06.44H9q1.62 2.94 4.56 4.72T20 28.5q5.19 0 8.84-3.66T32.5 16zm-19-1.5h13q.63 0 1.06.44T28 16t-.44 1.06-1.06.44h-13q-.63 0-1.06-.44T12 16t.44-1.06 1.06-.44z',
        'w' => 1.25,
        'vw' => 40
    ],
    'itch-io' => 'M4.5 2.17C3.14 2.98.46 6.05.44 6.86v1.33c0 1.7 1.58 3.18 3.01 3.18a3.18 3.18 0 0 0 3.16-3.12 3.13 3.13 0 0 0 3.11 3.12 3.1 3.1 0 0 0 3.07-3.12 3.2 3.2 0 0 0 3.19 3.12h.03c1.72 0 3.2-1.43 3.2-3.12a3.1 3.1 0 0 0 3.06 3.12 3.13 3.13 0 0 0 3.11-3.12 3.18 3.18 0 0 0 3.16 3.12c1.44 0 3.02-1.49 3.02-3.18V6.86c-.03-.8-2.7-3.88-4.06-4.69C23.29 2.02 20.36 2 16 2s-10.3.07-11.5.17Zm8.27 8.4a3.6 3.6 0 0 1-6.25.02c-.82 1.45-2.7 2-3.5 1.73-.24 2.5-.85 14.82 1.11 16.82 5 1.16 18.88 1.13 23.74 0 1.97-2.02 1.33-14.5 1.1-16.82-.8.27-2.67-.29-3.5-1.73a3.6 3.6 0 0 1-6.24-.02A3.58 3.58 0 0 1 16 12.38a3.6 3.6 0 0 1-3.23-1.8Zm-2.6 3.36c1.03 0 1.94 0 3.07 1.24a26.7 26.7 0 0 1 5.51 0c1.14-1.23 2.05-1.23 3.08-1.23 3.27 0 4.08 4.84 5.24 9.02 1.08 3.9-.34 3.99-2.12 3.99-2.63-.1-4.1-2.01-4.1-3.93-2.44.4-6.36.55-9.71 0 0 1.92-1.46 3.83-4.1 3.93-1.77 0-3.2-.1-2.12-3.99 1.17-4.18 1.97-9.02 5.24-9.02Zm5.83 3s-2.77 2.54-3.27 3.45l1.8-.08v1.58c0 .1 1.35.01 1.47.01.73.04 1.46.07 1.46 0V20.3l1.81.08c-.5-.9-3.27-3.45-3.27-3.45Zm0 0',
    'itunes' => 'M13.97 5q4.4 0 7.56 3.16t3.16 7.56-3.13 7.56T14 26.44t-7.6-3.16-3.15-7.56 3.16-7.56T13.97 5zm4.97 15q.06-.25.06-10.44 0-.62-.56-.56-.07 0-7.19 1.31-.19 0-.44.19-.19.13-.19.38-.06.12-.03 4.21t-.15 4.16q-.13.19-.5.25-1.88.31-2.38.81-.69.63-.37 1.57T8.5 23q1.06.19 1.94-.37.75-.38.94-1.2.06-.37.06-7.43 0-.44.44-.5 5.5-1 5.75-1.06.5-.13.5.37v2.75q.06 2.13-.1 2.32t-.53.25q-1.87.3-2.37.75-.5.5-.47 1.18t.6 1.13q.68.56 2 .31 1.37-.25 1.68-1.5zm2.62-18q2.7 0 4.57 1.88T28 8.43v15.12q0 2.7-1.87 4.57T21.56 30H6.44q-2.69 0-4.57-1.87T0 23.56V8.44q0-2.69 1.88-4.56T6.43 2h15.12zM14 27.75q3.25 0 6.03-1.63t4.38-4.37T26 15.72t-1.6-6.03-4.37-4.38T14 3.7q-5 0-8.53 3.53t-3.53 8.5 3.53 8.5T14 27.75z',
    'itunes-note' => [
        'p' => 'M23.88 24.25q-.32 1.25-1.22 2t-2.22 1q-2.63.5-4.07-.63-1.06-.93-1.12-2.3t1-2.32q.5-.5 1.31-.81t1.35-.44 2.09-.38q.69-.12 1-.5.25-.25.28-.81t.03-3.81q-.06-3.2-.06-5.57 0-.93-1.06-.75l-11.57 2.2q-.5.05-.68.27t-.2.72v8.07q.07 6.18-.12 6.87v.06q-.37 1.7-1.87 2.44-.82.44-2 .63t-2 .06q-2-.38-2.6-2.25t.78-3.19q.5-.5 1.35-.81t1.37-.4 2.1-.41q.68-.13 1-.44.31-.44.28-8.4t.03-8.48q.06-.56.44-.81.37-.37.81-.37 14.37-2.7 14.56-2.7Q24 1.94 24 3.13v11.94q0 8.63-.13 9.13v.06z',
        'w' => 0.75,
        'vw' => 24
    ],
    'java' => [
        'p' => 'M17.19 19.56q-4.94 1.32-11.32.75-3.5-.31-3.3-1.06.05-.19.55-.44 1.25-.62 2.5-.9t1.94-.28h.63l-2.5.75q-1.88.8.44 1.12 2.56.31 7.68-.06l4.82-.7q-.82.38-1.44.82zm-5.32-2q-1.06-.93-1.74-1.62t-1.35-1.5-.9-1.5-.25-1.4.43-1.35q.7-1 2.2-2.25T12.9 5.8t1.84-2.44T15 0l.16.19q.1.13.31.6t.31.93 0 1.25-.47 1.56-1.25 1.82-2.18 2.09q-1.5 1.19-2.16 2.25t-.47 2 .56 1.66T10.88 16t1 1.56zm7.13-11Q17.06 7.7 15.87 8.6t-1.71 1.53-.47 1.35.28 1.12.84 1.22l.19.32q.5.75.47 1.43t-.44 1.25-.87.97-.91.66l-.38.25.2-.19.37-.5.37-.75.03-.9-.53-1.04q-3.87-4.43 2.44-7.56Q17.38 7 19 6.56zm-.38 16.88q4.2-2.19 3.57-4.38-.38-1.25-1.75-1.37-.25 0-.5.06l-.69.19q.13-.31.5-.44 2.13-.75 3.31.75 1.32 1.44-.31 3.19-1.38 1.37-4.25 2.19v-.07q.13-.06.13-.12zm2.94 3.87q1.7.75-.75 1.75-4.43 1.75-12.87 1.38-3.19-.19-6.32-.69-2.62-.44-1.06-1.5 1.56-1.13 3.69-1.19.63 0 .88.19-.38 0-1.13.13-1.88.43-2 1 0 .18.38.3 2.5 1 8.9.95t9.28-.94q.88-.32 1.13-.63t.06-.5zM7.63 24.75l-.38.38q-.13.43 1.25.56 3.19.37 6.63-.25.68.44 1.74.81-4.25 1.88-8.68.81-2.25-.5-2.13-1.3.06-.57 1.56-1zm11.18 6.06q1.94-.37 3.16-.84t1.4-.72l.25-.25q-.18 1.81-4.9 2.53t-9.97.34q-3.25-.18-4.5-.8 2.5.5 6.78.46t7.78-.72zm-2.68-8.75q.5.5 1.3.82-5.55 1.62-10.24.68-2.25-.43-2.06-1.25.12-.56 1.68-1.06-1.31 1 .75 1.25 4.2.44 8.57-.44z',
        'w' => 0.736,
        'vw' => 24
    ],
    'jedi-order' => 'M24.88 23.38q.18-.13.5-.41t1.15-1.53T28 18.5q0 .25-.06.66t-.44 1.59-.9 2.28-1.63 2.44-2.47 2.28-3.56 1.62-4.82.75h-.25q-2.62-.06-4.78-.71t-3.56-1.7-2.47-2.24-1.62-2.44-.91-2.25-.4-1.66L0 18.5q.69 1.69 1.44 2.9T2.69 23l.43.37q-1.25-1.56-1.96-3.22t-.88-3.03-.1-2.68.38-2.32.63-1.78.56-1.15L2 8.75q-.75 1.5-.78 3.15t.5 2.94 1.19 2.38 1.15 1.65l.56.57q-1.8-3-2.12-5.85t.6-4.78 2.09-3.44 2.19-2.19l1-.68q-1.7 1.25-2.25 2.93T6 8.63t1.1 2.84 1.28 2.15l.62.75q-1.06 2.07-1.4 3.94t-.07 3.28.94 2.63 1.5 2.03 1.56 1.4 1.28.85l.5.25q.07-1.07.13-5.32l-2.13 2.25 1.5-3-3.93-.56 3.93-.56-1.25-3.44L13.5 21 14 1.8l.5 19.2 1.94-2.88-1.25 3.44 3.94.56-3.94.56 1.5 3-2.13-2.25.13 5.32 1-.53q.63-.34 2-1.66t2.16-2.84.71-4.03T19 14.37l.66-.81q.4-.5 1.25-2.03t1.15-2.94-.22-3.12-2.21-2.97l.56.37 1.31 1.1 1.69 1.8 1.44 2.5.9 3.13-.28 3.72-1.87 4.32q.25-.2.6-.6t1.12-1.6 1.22-2.4.43-2.87T26 8.75q.2.31.47.81t.82 2.13.62 3.25-.62 3.93-2.41 4.5z',
    'jenkins' => 'M30.44 26.56q.06.44-.57 2.5-.5 1.63-.87 2.94H2.62q-.06-.06-.12-.25v-.12q-.13-.25-.28-.7t-.28-.74-.22-.72-.16-.72q-.06-.56.16-.9t.84-.63.82-.4q.62-.45 3.5-2.13 1.68-1 1.68-1.13.13-.25.03-.56t-.34-.69-.31-.5q-.44-.94-.5-1.87-1.56-.25-2.44-1.57-.69-1.06-.94-2.46t.19-2.79l.16-.15.15-.22.13-.25q0-.19-.03-.44t-.1-.5-.06-.44Q4.37 7.2 6.37 6.25q.2-.56.47-1.1t.6-.87.87-.75.9-.6l1.04-.56 1-.62q.69-.38 2.5-1.07 2.31-.87 5.53-.62t5.1 1.69q.43.31.96.78t1 1 .9 1.1.7 1q1.18 2.5 1.25 6.37t-.88 6.87q-.37 1.31-.69 2l-.3.56q-.25.43-.41.75t-.13.5.4.57.91.81.6.5q.12.13.5.44t.59.53.44.5.22.53zM12.87 2.13q-1.74.5-3.37 1.56T7.31 6.06q.25-.06.82-.25t.93-.18q.07-.07.4 0t.42 0q.25-.07.53-.41t.53-.81.44-.66.4-.37.47-.35.38-.34q.06 0 .21-.07t.25-.15.1-.22q-.19-.19-.31-.13zm-6.56 5.8q-.87 1-.62 4.26 1.25-.75 2.56-.13t1.31 2q.2 0 .32-.12t.1-.32-.07-.34-.13-.44-.1-.4q-.18-.94.17-3.13t-.1-3q-2.12.19-3.44 1.63zM8.63 20.5q.56 2.25 1.62 3.81.94 1.25 4.31 1.5.38-.87 2.25-.43-1-.38-2.5-2-1.62-1.82-1.69-3.07 1.07 1.5 1.6 2.13t1.62 1.5 2.41 1.31q2.81.94 6.19-1.5.44-.25 1.56-1.88 2.75-3.93 2.44-9.8-.13-3.13-.56-3.95-.44-.87-1.44-1.25t-1.82.25q-.12-.5.16-.87t.78-.47 1.13.03q-.13-.12-.38-.47t-.4-.53-.41-.47-.44-.5-.44-.44-.47-.4-.53-.32q-1.75-1.06-3.93-1.4t-4.13.4q-4.19 2.07-5.19 4.38.25.62.28 1.37t-.09 1.32-.28 1.37-.22 1.38q0 .56.25 1.53t.31 1.47q-.37.56-1.68.5-.44-2.07-1.57-2.19-1-.13-1.72.56t-.78 1.7q-.06 1.18.78 2.46t1.97 1.16q.32-.07.75-.38t.75-.31q.19.25.1.47t-.25.37-.32.35-.15.3q-.07.26.12 1zm13.87 7.75q-.06.13-.03.44t-.03.56q1.37.38 3.19.44.3-.44.25-1.44 0-.69-.16-1.4t-.47-.91q-.37-.19-.97.06t-1.25.69l-.84.56q.19.57.19.63.62-.19 1.18.12-.93.13-1.06.25zM21.44 27q.19-.19.62-.56t.75-.57q-.37.07-1.22.29t-1.28.25-1.18-.22q0 .06-.07.06t-.06.06q1 .75 2.44.69zm-.82 2.06q1 .32 1.13-.56.19-.81-.5-.94-.31-.06-.75.25-.06.13.13 1.25zm-6.5-1.62q0-.19.1-.57t.1-.5q-.32-.06-.91-.09t-1-.1-.66-.24q-.44.37.63.9t1.75.6zm-10.5 3.62h11.25l-.12-.43q-.44-1.38-.5-2.25-2.63-1.25-3.56-1.88-.07 0-.44-.47t-.75-.87-.44-.35q-2.56 1-6.37 3.94.5 1.06.93 2.31zm15 .38zm.5-.38h.57q-.07-.06-.19-.25-.06 0-.38.25zm.94-1.56v-.63q-.68-.37-1.43-.37.18-.13.65-.22t.72-.22v-.31q-.38 0-1.38-.56-1.3-.75-2.75-1-.43-.07-.65.18t-.28.66-.1.9.03.88q0 .44.7 2.31.18.57.3.63.7.12 2.13-.63.44-.25 1.13-.84t.93-.78zm3.31 1.56.07-.81q-.38.06-.88-.28t-.68-.34q-.2-.07-.63.15t-.75.1l-.13.12-.12.16-.13.15q.25.31.57.75h1.06q0-.37.38-.37t.37.37h.88zm.57 0h2.25q-.88-1.31-2.19-.68-.06.37-.06.68zm5.18-2.31q.32-1.38.25-1.56 0-.25-.21-.5t-.63-.57-.47-.37q-.31-.25-.97-.94T26 23.75q-.18.25-.43.47t-.5.4-.37.38q1.37-.69 1.75 1.69.37 2-.25 3.18.06.25.34.54t.34.46q-.06.07-.12.2h1.75q.44-1.32.62-2.32zM13.88 8.12q.32-.74 1.2-1.4t1.62-.6q.44.13.72.44t.03.7q-2.19-.5-3.38 2.05-.56-.12-.18-1.18zm11.5 4q.25-.12.94-.15t.94-.22q-.62-1.69-.56-2.31h.06q1.06 2.19 1.31 2.56-.12.44-.8.75t-1.29.19-.6-.81zm-6 .13q-1.25-1.94-.93-2.81.06.06.4.9t.72 1.16q0 .06.13.16t.18.15.13.19.1.22.02.25-.06.28q-.06.19-.31.37t-.44.25q-.81.25-1.78-.03t-.47-1.03q.13 0 .31.03t.35.07.34.06l.38.06h.34l.31-.1.29-.18zm1.7 4.38q.06.12.12.46t.19.57.43.28q.44.06.97-.07t.94-.28.94-.43.78-.35q.19-.25.22-.47t-.03-.43-.16-.32-.31-.34-.28-.32l-1.16-1.43q-.35-.43-.78-1.4t-.44-1.85q.38-.25.5.43v.2q.7 1.5 2.94 3.93.06.13.25.28t.28.25l.16.16q.06.31-.13.84t-.18.66q-.88.56-1.44.87t-1.6.57-2.03.12q-.37-.31-.44-.87t.25-1.07zM8.5 16.05q-.19-.06-.34-.4t-.32-.63-.4-.28q-.13 0-.25.06t-.2.22-.08.28-.04.41v1.22q-.37-.31-.43-1.16t.18-1.34q-.12-.13-.25-.06t-.18.12-.2.16-.18.15q.19-1.36 1.34-1.1t1.35 1.56q.06.38 0 .81zm18.69 1.82q-1.32 2.5-3.38 2.56-.06-.94 0-1.19.5-.06 1-.22t.82-.34.78-.44.78-.38zm-8.88.75q1.44.75 4.94.62.13.31.13 1.25-1.75.06-3.22-.38t-1.85-1.5zm5.75 2.93q-.18.57-.56.7-.69.24-1.84.3t-2.1-.3q-.31-.2-.56-.45t-.53-.68-.35-.5l-.18-.2q-.7-.62.18-.62.38.94 1.38 1.38t2.15.4 2.41-.03z',
    'jira' => 'M30.63 15.13q.37.37.37.87t-.38.88l-4.43 4.5.06-.13L15.5 32Q13.75 30.19.37 16.87 0 16.5 0 16t.38-.88L15.5 0l7.69 7.69zM15.5 20.75 20.25 16l-4.75-4.75L10.75 16z',
    'joget' => 'M23.63 2.81q5.5 3.38 6.96 9.63t-1.9 11.72-9.63 6.93-11.72-1.9-6.93-9.63T2.3 7.88q2.19-3.5 5.72-5.44T15.5.5q4.38 0 8.13 2.31zM26.88 23q.5-.75.87-1.63l-2.34.04q-2.21.04-3.82.09t-3.69.31-3.21.63q-.7.25-1.16.56t-.75.56-.4.6-.16.56.03.6.12.5.22.46l.16.35q.44 1.12 1.5 2.69.69.06 1.25.06 3.44 0 6.47-1.69t4.9-4.69zM14.3 17.62l14.38.32q.56-3.88-1.1-7.53T22.5 4.63q-4.69-2.88-10.06-1.6T4.12 9q-2.68 4.31-1.75 9.31.32-1.25.94-2.43t1.82-2.66 3.37-2.47 5-1.19q2.56-.19 2.38.82-.13.56-.88 1-.13.12-.72.43t-.9.53-.85.57-1 .78-.84.94q-1.07 1.25-.03 2.1t3.65.9zm11.5-3.43q.63.19.94.44T27 15v.19l-2.87 1.44-10.63.12q2.13-.69 4.06-2 .32-.19.94-.6t.97-.62.81-.5.85-.37.69-.1.62.13q.31.19.44.37t-.03.38-.32.28-.5.28-.4.25l-3.19 2.13.38.12q.56-.06 1.56-.62 3.94-2.07 5.44-1.7z',
    'joomla' => 'M.06 5.75q0-1.56 1.1-2.66T3.76 2q1.43 0 2.46.9T7.5 5.14q3.38-.75 6 1.87l-2.75 2.75q-1.06-.88-1.94-.88t-1.53.7-.65 1.62.62 1.62l6.25 6.25-2.75 2.75q-3.19-3.12-4.22-4.18T4.5 15.56q-1.25-1.19-1.66-2.87t.1-3.32Q1.69 9.12.88 8.1T.06 5.75zM8.12 13l6.26-6.19q1.18-1.25 2.87-1.65t3.25.09q.19-1.37 1.25-2.31t2.5-.94q1.5 0 2.6 1.1t1.09 2.65q0 1.44-.94 2.47T24.69 9.5q1 3.5-1.82 6.31L20.12 13q.94-1 .94-1.88t-.69-1.56-1.62-.69-1.63.7l-6.25 6.24zm16.63 9.56q1.38.2 2.31 1.22t.94 2.47q0 1.56-1.1 2.66T24.26 30q-.87 0-1.66-.41t-1.3-1.1-.73-1.5q-1.56.5-3.34.06t-3.03-1.8l2.75-2.76q1 .88 1.87.88t1.57-.7.68-1.62-.68-1.62l-6.25-6.25 2.8-2.75q5.7 5.69 6.2 6.25 2.5 2.5 1.62 5.87zm-5.25-3.31-6.19 6.19q-2.5 2.5-5.94 1.62-.18.88-.68 1.53T5.4 29.62 3.75 30q-1.56 0-2.66-1.1T0 26.26q0-1.31.82-2.34t2.07-1.29Q2 19.2 4.75 16.45l2.75 2.81q-.88 1-.88 1.88t.63 1.56q.69.69 1.66.69t1.59-.7q5.56-5.5 6.25-6.24z',
    'js' => 'M0 2h28v28H0V2zm15.25 21.81v-8.94h-2.63v8.94q0 1.63-1.43 1.63-.94 0-1.7-1.32l-2.12 1.25q1.13 2.44 3.94 2.44 1.81 0 2.88-1t1.06-3zm6.19 4q1.94 0 3.15-.97t1.22-2.72q0-1.56-.84-2.43t-2.66-1.7l-.62-.24q-.94-.44-1.28-.75t-.35-.88q0-.3.16-.56t.44-.4.65-.16q1.07 0 1.7 1.12l2.05-1.3q-1.18-2.13-3.75-2.13-1.75 0-2.8 1t-1.07 2.5.81 2.4 2.38 1.53l.62.32.84.37.6.35.4.46.1.57q0 .56-.5.93t-1.31.38q-1.57 0-2.57-1.63l-2.12 1.25q1.31 2.7 4.75 2.7z',
    'js-square' => 'M25 2q1.25 0 2.13.88T28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22zm-9.75 21.81v-8.94h-2.63v8.94q0 1.63-1.43 1.63-.94 0-1.7-1.32l-2.12 1.25q1.13 2.44 3.94 2.44 1.81 0 2.88-1t1.06-3zm6.19 4q1.94 0 3.15-.97t1.22-2.72q0-1.56-.84-2.43t-2.66-1.7l-.62-.24q-.94-.44-1.28-.75t-.35-.88q0-.3.16-.56t.44-.4.65-.16q1.07 0 1.7 1.12l2.05-1.3q-1.18-2.13-3.75-2.13-1.75 0-2.8 1t-1.07 2.5.81 2.4 2.38 1.53l.62.32.84.37.6.35.4.46.1.57q0 .56-.5.93t-1.31.38q-1.57 0-2.57-1.63l-2.12 1.25q1.31 2.7 4.75 2.7z',
    'jsfiddle' => [
        'p' => 'M31.94 14.81q2.12 1.2 3.1 3.32t.4 4.37q-.25 1.19-.88 2.19t-1.5 1.72-2 1.12-2.37.4Q26.5 28 12.94 28q-4.63 0-5.69-.06-2.25-.06-4.1-1.5T.7 22.88q-.7-2.2.03-4.28t2.53-3.54q.37-.25.31-.56-.37-3.06 2.07-4.81 2.37-1.75 5.3-.56l.5.24q1.57-2.56 4.07-4 2.13-1.18 4.53-1.34t4.53.63 3.85 2.46 2.53 4q.25.57.56 3.07.06.5.44.62zm1.31 9.81q1.81-2.25 1.16-5t-3.35-4q-.62-.3-.62-1-.07-1.87-.72-3.56T28 8.22t-2.53-1.94-3.03-1.06-3.25 0-3.13 1.1q-2.5 1.37-3.81 3.87-.13.25-.25.37t-.28.13h-.22l-.31-.16-.32-.22q-2.37-1.25-4.5.19-2.19 1.5-1.69 3.94.2.94-.62 1.44-1.81 1.18-2.4 3.25t.27 4q1.7 3.75 5.94 3.75 10.25.12 10.25 0h10.63q2.69-.07 4.5-2.25zm-5.5-5q-.25 1.44-1.47 2.44t-2.72 1q-1.56 0-2.75-1.12-.5-.44-2.78-2.82t-3.22-3q-1.87-1.43-3.75.25-.87.75-1.06 1.79t.28 1.87 1.22 1.44 1.78.56 1.97-.84l.37-.4q.25-.28.44-.41.38-.32.75-.1t.06.78q-.8 1.2-2.18 1.72t-2.75.1-2.22-1.63-.85-2.75q.2-2.37 2.25-3.5 2.32-1.31 4.2.13 1.62 1.25 3.68 3.5t3 2.8q1.19.76 2.4.45t1.91-1.5q.63-1.13.38-2.28t-1.13-1.82-2-.78-2.19.94l-.3.34q-.25.28-.38.35-.5.31-.79 0t-.03-.63q2.13-2.94 5.2-1.75 1.5.5 2.24 1.88t.44 3z',
        'w' => 1.123,
        'vw' => 36
    ],
    'kaggle' => [
        'p' => 'm18.13 31.38.12.43q-.06.2-.38.2H13.7q-.38 0-.7-.32l-6.87-8.82-1.93 1.88v6.75q0 .5-.5.5H.49q-.5 0-.5-.5V.5Q0 0 .5 0h3.2q.5 0 .5.5v19.13l8.25-8.38q.3-.38.68-.38h4.32q.31 0 .4.29t-.1.4L9 20z',
        'w' => 0.568,
        'vw' => 18
    ],
    'keybase' => [
        'p' => 'M11.06 26.94q0 .44-.31.78t-.78.34-.78-.34-.32-.78.32-.78.75-.35q.25 0 .47.1t.34.25.22.34.1.44zm5.82-1.13q.43 0 .78.32t.34.78-.34.81-.79.34-.78-.34-.34-.81.34-.78.79-.32zm8.87-2.25q0 3.7-1.38 6.44H22.7q1.06-1.75 1.43-4.19t-.06-4.18q-.75 1.3-2.28 1.78t-3.1-.04q-2.12-.62-4.09-.68t-3.28.15-2.78.9-2.16 1.13-1.8 1.32q-.38.25-.5.37l1.18-3.69-2.5 2.63Q3.2 27.94 4.7 30H2.87q-.93-1.5-1.37-3.19L0 28.37v-2.84l.03-1.31.13-1.4.21-1.29.32-1.4.47-1.29.65-1.37.9-1.22 1.17-1.31Q5.75 13 8.18 12q-.68-1.25-.5-2.88l-1.24-.06Q5.62 9 5.06 8.37t-.56-1.5l.12-1.62q.07-.82.66-1.35t1.34-.53l1.82.07q.87.06 1.44.75.3-.5 1.5-2.2l1.3.76q-.93 1.94-.55 2.31.93 0 2 .31 1.93.7 2.8 2.5.88 2 .07 3.94 2.94.94 5.19 3.19 3.56 3.75 3.56 8.56zM8 7.63q.25-.88.81-1.82.07-.81-.5-.87L6.7 4.87q-.5 0-.56.5L6 7q0 .5.5.56zm1.63 3.87q.5.81 1.37 1.25 0-1.13 1.1-1.63t2.21.57l.5.62q.94-.81 1.07-1.97t-.32-1.84q-.5-1.19-1.6-1.6t-1.77-.4q-1.06.06-1.5-.94Q8.3 9.31 9.63 11.5zm5.18 4.25-1.25 1.06q-.19.2-.06.38l.56.69q.2.18.44 0l1.19-1 .37.43q.25.32.57 0 .18-.18 0-.43-1.94-2.32-3.5-4.25-.25-.32-.5 0-.25.18-.07.43l.66.78.6.72-.57.5q-.31.25.06.57.2.18.38.06l.62-.5zm7.38 1.63q-1.82-2.5-4.57-3.75-.68-.32-1.43-.57-.25.25-.44.38l2 2.5q.5.56.44 1.34t-.7 1.22q-.8.69-1.8.31-.75.63-1.38.63-.87 0-1.37-.63l-.57-.68q-.62-.75-.31-1.7-.75-.87-.31-1.87-1.57-.25-2.63-1.31-1.62.56-3 1.66T4 16.96t-1.25 1.97q-1.13 2.25-1.25 5.62 2.87-3 7.12-7.56l-1.87 5.87q5.5-2.93 12.37-.93 2.38.75 3.5-.82t-.44-3.75zM6.94 6.68 7 5.76l.94.06-.06.94z',
        'w' => 0.804,
        'vw' => 26
    ],
    'keycdn' => 'm4 25.56 3.75-3.69q3.06 4.13 7.94 4.25 2.94.07 5.37-1.43l.47.47.53.56.32.34q-3.07 2-6.75 1.94-4.2-.12-7.38-2.81.06 0-1.06 1.1t-1.32 1.27q.44 1.44-.46 2.66T3 31.44q-1.31 0-2.16-.88T0 28.44q0-1.63 1.28-2.5T4 25.56zM26.19 9.81q1.87 3.07 1.75 6.5-.07 4.25-3 7.57l1.5 1.56q.94-.13 1.62.56.5.56.5 1.35t-.56 1.3-1.34.54-1.35-.56q-.62-.7-.44-1.63l-2.8-3q3.93-3.12 4.05-7.75.07-2.62-1.3-5.12zM6.5 3.31q.69.7.5 1.63L9.88 8q-.94.69-1.44 1.31Q6 12.06 5.87 15.75q-.06 2.75 1.25 5.13l-1.37 1.37Q3.94 19.2 4 15.7q.12-4.25 3-7.56l-1.63-1.7q-.93.13-1.62-.55-.5-.57-.47-1.32t.56-1.31 1.32-.53 1.34.6zM29.06.5q1.2 0 2.07.9T32 3.54t-.87 2.1-2.07.87q-.68 0-1-.13l-3.94 3.75q-3.06-4.06-7.93-4.18-2.94-.07-5.38 1.37L9.5 5.94Q12.56 4 16.25 4.06q4.19.13 7.37 2.82l2.57-2.44q-.13-.44-.13-.94 0-.81.4-1.5t1.1-1.1 1.5-.4zm-13 6.56q.44 0 1 .07 3.7.5 5.94 3.4t1.81 6.57-3.37 5.93-6.6 1.82-5.9-3.38-1.81-6.6q.43-3.37 2.96-5.56t5.97-2.25zm-3.69 7.5q.07 2.25 2 3.25l-.68 3.25h4.62l-.69-3.25q2-1.06 2-3.25 0-1.56-1.06-2.6T16 10.88q-1.5.07-2.57 1.1t-1.06 2.6z',
    'kickstarter' => 'M25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22q1.25 0 2.13.88T28 5v22q0 1.25-.88 2.13T25 30zM12.5 11.12q0-.68-.22-1.21t-.53-.88-.78-.53-.97-.19q-1.19 0-1.84.81t-.66 2v9.7q0 .56.16 1.09t.47.9.8.6 1.07.22q1.13 0 1.81-.72t.7-2.1V18.2l3.74 4.75q.63.69 1.66.72t1.84-.72q.69-.69.72-1.63T20 19.75l-3.06-3.94 2.5-3.62q.5-.63.44-1.53t-.75-1.53q-.75-.82-1.94-.75t-1.94 1.18l-2.75 4v-2.43z',
    'kickstarter-k' => [
        'p' => 'M9.19 7.13v4.43l5.19-7.37q1.37-2 3.56-2.13t3.56 1.32q1.25 1.18 1.38 2.84T22.13 9l-4.7 6.69 5.7 7.19q.93 1.12.84 2.81t-1.28 2.94q-1.5 1.44-3.44 1.37t-3.06-1.37l-7-8.63v4.75q0 2.63-1.25 3.94T4.69 30q-2.25 0-3.47-1.47T0 24.75V7.07Q0 5.63.53 4.5t1.6-1.81T4.63 2q2 0 3.28 1.31t1.28 3.82z',
        'w' => 0.75,
        'vw' => 24
    ],
    'korvue' => [
        'p' => 'M24.13 2.13q1.56 0 2.65 1.09t1.1 2.66v20.37q0 1.56-1.1 2.66T24.18 30H3.76Q2.2 30 1.1 28.9T0 26.32V5.87q0-1.56 1.1-2.65t2.66-1.1h20.37zM5.43 7.55v8.25H17.2l5.06-8.25h-6.94l-3.87 7.25V7.56h-6zm10.13 17h7.38l-5.5-8.19h-12v8.2h6v-7.13z',
        'w' => 0.871,
        'vw' => 28
    ],
    'laravel' => [
        'p' => 'M39.88 15.13q.37.43-.32.62l-5.44 1.5 4.94 6.63q.19.43.06.59t-.56.34-6.44 2.32-6.56 2.3q-.75.32-1.22.26t-.9-.69q-.63-.81-4.57-7.94l-8.37 2.2q-1.44.3-2.07-1L6.4 17.83l-4-8.65Q.37 4.82.2 4.3q-.57-1.25.5-1.37 6-.5 8.06-.63h.5l.34.16.22.22.25.37 9.63 16.07 12.12-2.88q-.37-.56-2.12-3.03t-1.88-2.66q-.44-.62.56-.75.13-.06 1-.21t1.82-.29 1.78-.28.97-.15q.75-.13 1.31.56.38.38 2.4 2.94t2.23 2.75zm-22.07 4.5q.32-.07.07-.38l-9-15.63q-.13-.18-.32-.18l-3.31.28L1.94 4q-.2 0 0 .37.12.2 4.15 8.5t4.03 8.38q.07.19.44.06.19 0 3.69-.84t3.56-.85zm19.13 3.56q0-.07-1.94-2.69t-2.06-2.81q-.2-.25-.57-.13l-11.8 3.07q3.62 6.25 3.93 6.62.25.5.62.31.13-.06 2.97-1.03t5.72-1.94 3-1.03q.31-.06.13-.37zm.75-8.38q.5-.12.37-.31-3.06-3.81-3.44-4.31-.18-.32-.5-.25-3 .62-4.5.81-.25.06-.06.31l3.57 4.88 4.56-1.13z',
        'w' => 1.25,
        'vw' => 40
    ],
    'lastfm' => 'm14.13 22.94-.26.22q-.18.16-.74.47t-1.2.56-1.71.47-2.22.22q-3.88 0-5.94-2.2T0 16.32q0-4.37 2.12-6.78t6.13-2.4q1.44 0 2.6.22t1.96.56 1.53 1.06 1.16 1.25.94 1.69.71 1.84.72 2.2q.32.87.44 1.37.5 1.44 1.16 2.4t1.94 1.6 3.03.62q3.81 0 3.81-1.94 0-.37-.1-.71t-.4-.6-.56-.4-.82-.35-.87-.28-1.13-.25-1.12-.28q-4.25-1-4.25-4.75 0-2.81 1.87-4.03t4.7-1.22q5.87 0 6.3 4.62l-3.68.44q-.2-2.2-2.89-2.2-.81 0-1.37.22t-.88.66-.31 1.06q0 .88.44 1.38t1.62.75q.13.06 1 .25t1.03.22.94.25 1 .34.84.4.82.54.6.65.55.85.29 1.03.12 1.22q0 5.06-7.62 5.06-4.38 0-6.54-1.69T14.7 18.5l-.6-1.8-.53-1.7q-.13-.45-.5-1.4t-.62-1.28-.69-.91-.9-.78-1.1-.38-1.37-.18q-1.94 0-3.22 1.47t-1.28 4.65q0 1.69.53 3t1.53 2 2.25.69 2.44-.53 1.75-1.03l.56-.57z',
    'lastfm-square' => 'M25 2q1.25 0 2.13.88T28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22zm-5.75 19.56q4.75 0 4.75-3.19 0-.37-.1-.75t-.15-.65-.34-.53-.41-.4-.5-.35l-.5-.25-.63-.19-.59-.16-.69-.15-.6-.13q-1.3-.31-1.3-1.31 0-1.25 1.62-1.25 1.69 0 1.81 1.38l2.25-.25q-.25-2.94-3.87-2.94-1.81 0-2.97.78t-1.16 2.53q0 2.31 2.7 3 .18 0 .5.06t.5.13.43.12.4.13.35.12.31.16l.25.19.2.21.12.29.06.34q0 1.19-2.38 1.19-2.87 0-3.87-2.88l-.35-1.06-.34-1.06q-.05-.18-.34-.9t-.4-.88-.45-.7-.56-.65-.69-.44-.87-.4-1-.19-1.25-.1q-2.5 0-3.85 1.5T4 16.2q0 1.75.6 2.93t1.68 1.82 2.72.62q1.13 0 2.1-.31t1.34-.63l.37-.25-.69-2q-1.25 1.32-3 1.32-1.18 0-1.93-.97t-.75-2.6q0-2 .81-2.93t2-.94q.44 0 .84.12t.7.25.55.5.44.6.38.81.31.84.34 1.07.38 1.12q.19.69.5 1.25t.78 1.1 1.13.87 1.56.56 2.1.22z',
    'leanpub' => [
        'p' => 'm24.19 6.94.94 15.56h-.7q-3.5-.06-6.43 1.75-2.81-1.75-6.44-1.75-3.5 0-7.69 1.75L5.81 8.13q2.75-1.2 5.75-1.2 3.82 0 6.44 2.7 2.56-2.57 6.19-2.7zM35.56 28q-2.25 0-4.37-1-3.44-1.56-6.75-1.56-3.88 0-6.44 2.56-2.56-2.56-6.44-2.56-3.31 0-6.75 1.56-2.19 1-4.31 1H.44L3.06 6.19q3.88-2.2 8.31-2.2 4 0 6.63 1.82Q20.62 4 24.62 4q4.44 0 8.32 2.19zm-2.69-2.81-2.12-17.5q-2.94-1.32-6.31-1.32-3.88 0-6.44 2.38-2.56-2.38-6.44-2.38-3.37 0-6.31 1.32l-2.13 17.5q1.82-.75 2.85-1.13t2.56-.65 3.03-.28q3.5 0 6.44 1.8 2.94-1.8 6.44-1.8 2.19 0 3.97.5t4.47 1.56z',
        'w' => 1.125,
        'vw' => 36
    ],
    'less' => [
        'p' => 'M38.31 13.69q0 1.37 1.7 1.44v1.75q-1.7.06-1.7 1.43 0 .63.1 1.66t.1 1.78q0 1.56-.6 2.19t-1.97.62h-1.25V23h.37v-.06q.44 0 .66-.13t.31-.4.1-.85-.03-1.6-.04-1.65q0-2 1.44-2.31v-.13q-1.44-.3-1.44-2.3 0-.63.04-1.7t.03-1.5q0-.87-.2-1.15t-.87-.28h-.37v-1.5h1.25q1.37 0 1.97.62t.6 2.19q0 .75-.1 1.78t-.1 1.66zm-6.62 2q2.69.87 2.69 3 0 .87-.44 1.6t-1.38 1.15-2.19.44q-2.12 0-3.74-1.38l1.3-1.94q1.32 1.07 2.5 1.07 1.32 0 1.32-.88 0-.56-1.81-1.25-.13-.06-.19-.12-2.62-1-2.62-2.88 0-.69.28-1.25t.75-.97 1.18-.62 1.54-.22q1.75 0 3.43 1.25L33 14.44q-1.13-.81-2.13-.81-1.06 0-1.06.74 0 .2.03.29t.16.22.22.18.31.16.34.16.41.15.4.16zM9.25 19.38h.38L10 21.5q-.63.25-1.5.25-2.69 0-2.69-3.38V9h-.87q-.63 0-.85.28t-.22 1.1l.03 1.56.04 1.62q0 2-1.44 2.32V16q1.44.38 1.44 2.31 0 .63-.04 1.7t-.03 1.55q0 .82.22 1.13t.85.31h.37v1.5H4.06q-1.37 0-1.97-.62t-.6-2.25q0-.69.1-1.72t.1-1.6q0-1.43-1.7-1.43v-1.75q1.7-.07 1.7-1.44 0-.56-.1-1.63t-.1-1.8q0-1.57.6-2.2t1.97-.62h4.69v11.12q0 .82.5.82zm14.5-3.7q2.69.88 2.69 3 0 .7-.28 1.26t-.78 1-1.25.69-1.75.25q-2.07 0-3.75-1.38l1.3-1.94q1.32 1.07 2.5 1.07 1.32 0 1.32-.88 0-.56-1.81-1.25-.13-.06-.19-.12-2.62-1-2.62-2.88 0-.69.28-1.25t.78-.97 1.19-.62 1.5-.22q1.8 0 3.43 1.25l-1.3 1.75q-1.13-.81-2.07-.81-1.13 0-1.13.75 0 .18.03.28t.16.22.25.18l.31.16.35.16.43.15.41.16zm-9.69-4.3q2 0 3.1 1.34t1.1 3.4q0 .44-.13 1.2h-5.94q.37 2.18 2.56 2.18 1.06 0 2.13-.62l1 1.68q-1.63 1.13-3.5 1.13-1.44 0-2.53-.6t-1.75-1.78-.66-2.75q-.06-2.3 1.34-3.75t3.28-1.43zM12.2 15.5h3.56q0-1.94-1.69-1.94-1.56 0-1.87 1.94z',
        'w' => 1.25,
        'vw' => 40
    ],
    'line' => 'M17 12.75v4.44q0 .19-.19.19h-.69q-.12 0-.18-.07l-2.07-2.75v2.63q0 .25-.18.25h-.7q-.24 0-.24-.25v-4.44q0-.19.25-.19h.68q.13 0 .2.06l2 2.76v-2.63q0-.19.18-.19h.75q.19 0 .19.19zm-5.13-.19q.2 0 .2.2v4.43q0 .19-.2.19h-.68q-.2 0-.2-.2v-4.43q0-.19.2-.19h.68zm-1.68 3.75q.19 0 .19.2v.68q0 .19-.25.19H7.3q-.06 0-.12-.03t-.07-.16v-4.44q0-.19.2-.19H8q.25 0 .25.2v3.56h1.94zm10.56-3.75q.19 0 .19.2v.74q0 .19-.19.19h-1.94v.75h1.94q.19 0 .19.19v.75q0 .19-.19.19h-1.94v.75h1.94q.19 0 .19.18v.7q0 .18-.19.18h-2.88q-.18 0-.18-.19v-4.44q0-.18.18-.18h2.88zM28 7.12v17.82q0 2.12-1.5 3.6T22.87 30H5.07q-2.13 0-3.6-1.5T0 24.88V7.06q0-2.12 1.5-3.6T5.12 2h17.82q2.12 0 3.6 1.5T28 7.13zm-3.88 7.63q0-3.38-2.96-5.81t-7.2-2.44-7.21 2.44-3 5.81q0 3.06 2.44 5.38t6.12 2.8q.81.2.9.66t-.15 1.66v.25l.03.28.25.16.6-.2q.62-.24 2.06-1.12t3.22-2.25 2.78-2.5q2.13-2.43 2.13-5.12z',
    'linkedin' => 'M26 2q.81 0 1.4.6T28 4v24q0 .81-.6 1.4T26 30H2q-.81 0-1.4-.6T0 28V4q0-.81.6-1.4T2 2h24zM8.44 26h.06V12.62H4.31V26h4.13zM6.38 10.81q1 0 1.71-.72t.72-1.68-.72-1.7T6.37 6t-1.68.72-.7 1.69.7 1.68 1.68.72zM24 26v-7.31q0-1.5-.19-2.57t-.69-1.96-1.53-1.38-2.53-.47q-1.44 0-2.47.63t-1.47 1.5h-.06v-1.82h-4V26h4.19v-6.63q0-1.56.5-2.5t1.94-.93q.75 0 1.25.3t.65.95.22 1.09.07 1.22V26H24z',
    'linkedin-in' => 'M6.25 28H.44V9.31h5.81V28zM3.37 6.75Q2 6.75 1 5.75T0 3.37 1 1t2.38-1 2.37 1 1 2.38-1 2.37-2.38 1zM28 28h-5.81v-9.13q0-.8-.03-1.34t-.2-1.28-.43-1.19-.87-.78-1.47-.34q-2.07 0-2.78 1.31t-.72 3.5V28H9.87V9.31h5.63v2.56h.06q.63-1.25 2.06-2.15t3.44-.9q2.13 0 3.53.65t2.13 1.94 1 2.78.28 3.56V28z',
    'linode' => [
        'p' => 'M27.31 14.13q.07.06-.56 4.56 0 .12-.06.12l-3.44 2.75q-.13.07-.25 0l-1.25-.87-.13 2.06q0 .13-.06.13l-4.19 3.37q-.12.13-.25 0l-1.75-1.5.13 2.88q0 .12-.06.12l-5.25 4.19q-.07.06-.13.06l-.12-.06-4.07-4.31q-.12-.07-1.06-4.94-.06-.13.06-.19L6 21.81q-1.7-1.56-1.7-1.62L3 13.94q-.07-.13.12-.25l1.69-.82q-2.38-1.8-2.44-1.93L.68 2.62q-.06-.12.13-.18L8.5 0h.18l5.63 2.75q.12.06.12.18l.38 8.25q0 .13-.13.2l-2.12 1.06L14.8 14q.12 0 .12.12l.07 2.2 2.2-1.32q.06-.06.18 0l1.5 1 .07-2q0-.06.06-.13l3.69-2.25q.12-.06.18 0L27.25 14zm-12.75.3L8.25 18l.88 6.06 5.68-3.8zm-.5-11.24L6.5 6.13l1.25 8.37 6.63-3.44zM2.75 10.8l4.56 3.57-1.18-8.32-5-3.06zm1.94 9.25 4 3.82L7.87 18 3.5 14.38zm1.5 7.38 3.62 3.81L9.2 27l-3.9-3.75zm4 4 4.94-3.88-.25-4.37L9.56 27zm5.12-7.19 1.7 1.44v-4.25L15.12 20q0 .44-.07.5l-1.37.94 1.5 1.25q.12.06.12 1.56zm5.94-1.56.25-4.19-4.13 2.94.07 4.25zm1.69-1.63.37-4.18-4-2.5-.06 1.87 2.56 1.69q.13.06.07.19l-.13 2.12zm3.43-2.5.5-3.93-3.18 2.25-.32 4.06z',
        'w' => 0.873,
        'vw' => 28
    ],
    'linux' => 'M13.81 7.69h-.12l-.13-.07q0-.06.07-.12.12-.06.37.06.19.07.19.16t-.19.1q-.06 0-.12-.07zm-1.37.12q-.19 0-.19-.06 0-.13.19-.19.25-.12.37 0 .07 0 .03.03t-.15.1h-.06l-.07.06-.12.06zm13.81 17.44q1 1.13 1 1.88-.06.62-.88 1.12l-1.03.47q-.73.34-1.25.72t-.9.81Q21.8 31.88 20.13 32t-2.5-1.44q-.07-.18-.13-.43-.75.06-1.9-.13t-1.6-.13q-.44 0-.9.1l-.79.16-.69.12-.62.06q-.44 1-1.63 1.25-1.56.38-3.5-.68-.5-.25-1.15-.38t-1.4-.22-1.13-.16q-.94-.18-1.25-.75-.32-.68.12-2 .13-.3-.06-1.3-.06-.5-.06-.76 0-.44.18-.75.32-.62 1.13-.94.87-.3 1.12-.5.07-.12.41-.59t.6-.66q-.26-1.5.37-3.3.94-2.88 3.62-6.07 1.26-1.75 1.44-4.06 0-.32-.06-1.32t-.1-1.78.23-1.84.71-1.78T12.1.5t2.6-.5q1.44 0 2.44.6t1.4 1.5.66 2 .22 2.15.03 1.94q-.07.87.31 1.75t.69 1.37 1.06 1.38q2.06 2.5 2.88 4.68 1.06 2.63.25 4.94l.25.13.25.18q.43.32.68 1.38.25 1 .44 1.25zM14 5.44q-.31.69-.19 1.37.5.13.81.32-.12-.57.13-1.04t.63-.46q.37.06.56.62t0 1.13q-.06 0-.25.25.56.18.75.3.94-1.05.31-2.5-.5-1-1.4-1t-1.35 1zm-2.56-.69q-.88 0-1.06 1.28T10.94 8l.37-.31-.06-.07q-.31-.3-.28-1t.4-.75q.2 0 .35.2t.22.46.06.6q.25-.2.63-.32.06-.87-.32-1.47t-.87-.6zm-.88 3.75q-.37.31-.25.81.2.5.94.94.19.13.6.47t.65.47.63.12q.62.07 1.43-.4t.94-.53q1.13-.32 1.38-1.13.31-.75-.63-1.12-.25-.13-.72-.28t-.72-.29Q13.94 7 13.2 7t-1.69.88q-.06 0-.94.62zm.25 20.88q.07-.7-1.18-2.35t-1.38-1.9L7 22.88q-.63-.88-1.38-1-.62-.13-1.12.43-.13.13-.4.53t-.47.6q-.38.31-1.2.62-.68.25-.93.7-.19.37-.06 1.24.18 1.13.06 1.5-.44 1.19-.19 1.69.19.37.97.5t1.9.28 1.88.53q1.5.94 3.06.69t1.7-1.81zm-.12-1.57q1.94 1.07 3.72.88t3.47-1.38q.37-1.31.37-1.81.13-1.75.31-2.5.32-1.31 1.32-1.69.18-1.75 2.37-.75 1.75.75 1.44 1.63H24q.38-1.25-1.94-2.2.25-1.05-.06-2.24-.5-2.19-2.19-3.69-.25 0 .2.44.8.75 1.4 2.28t.03 3.03q-.38-.12-.69-.12-.31-1.94-1.44-4-1.56-2.88-2.37-6-.44.62-1.38.93-.12.07-.28.13t-.37.22-.35.22q-1.5.87-2.62-.07l-.35-.3q-.28-.25-.47-.32-.12-.13-.37-.32-.06.7-.22 1.47t-.5 1.6-.6 1.44-.62 1.46-.5 1.1q-.81 1.94-.87 3.87-1.5-2 .19-5.18.75-1.38.5-1.32-1.38 2.25-1.7 3.75-.3 1.2 0 2.2.38 1.12 1.57 1.87 1.56.87 2.38 2 .68.93.5 1.56-.26.56-1.07.56.63.75.88 1.25zm16.12-.75q0-.56-.87-1.56-.25-.25-.4-.81t-.29-1.07-.37-.68q-.44-.38-1.44-.32-.38.44-1.1.78t-1.28.22q-1.06-.25-1.18-1.81h-.07q-.62.31-.87 1.31-.13.7-.25 2.44 0 .19-.63 2.56-.37 1.44-.06 2.25.69 1.32 2.12 1.2 1.5-.13 2.75-1.57.57-.69 1.57-1.22t1.65-.9.72-.82zm-16-17.75q-.25-.25-.06-.44.06 0 .16.16l.09.16q.31.31.5.43.75.63 1.5.63t1.88-.63q.12-.12.87-.62.13-.06.22-.22t.15-.1q.25 0-.18.38-.44.44-.94.69-1.25.63-1.94.63-.87 0-1.75-.63-.19-.19-.5-.44z',
    'lyft' => 'M0 5.06h4.88v13.07q0 2.68 1.68 3.8-.75.63-2.12.73t-2.88-.9Q0 20.5 0 18.05v-13zm30.38 10.88q0 .62.46 1.1t1.16.46v4.81q-1.75 0-3.22-.87t-2.34-2.35-.88-3.21v-4.82q0-1-.81-1.44t-1.63 0-.8 1.44v1.44h2.43v4.81H22.3v.44q0 2.44-1.56 3.69-1.31 1.12-3.19.87V11.13q0-2.7 1.88-4.57t4.5-1.87q2.3 0 4.06 1.44t2.25 3.62h1.62v4.82h-1.5v1.37zM12 9.75v-.06h4.81V20.8q0 2.5-1.31 4.19T12 27.13q-2.38.43-4.88-.32v-4.37q1.5.62 2.82.5 2-.2 2.19-1.57l-.07.04-.25.15-.37.22-.53.22-.7.16h-.77L8.5 22q-2.81-.75-2.81-4V9.75h4.81v7.06q0 .69.75.69t.75-.69V9.75z',
    'magento' => 'M27.88 8v16l-4 2.25V10.31L14 4.56l-9.94 5.75.06 16-4-2.31V8L14 0zM16 26.25V10.31l3.94 2.25v16L14 32l-5.94-3.44v-16L12 10.31v16l2 1.13z',
    'mailchimp' => [
        'p' => 'M13.94 4.56q-.31.2-.44.25l-.19-.75zM6.3 19.63q0 .06.25.18t.44.28.25.47q.06.5-.25.88t-.75.37q-.19 0-.34-.06t-.22-.13-.03-.09.15-.03q.82.13.94-.56v-.32q0-.18-.31-.4t-.38-.28q-.37-.38-.06-.75.25-.2.44-.07 0 .07-.13.13t-.12.25zm.25-.63q-.37-.63-1.25-.44l-.12.07h-.07q-.06.06-.12.06-.38.31-.44.31-.31.44-.31 1.13 0 .3.12.62.07.13 0 .19-.12.06-.25-.13-.25-.25-.3-.62-.32-1.56 1.05-2.06.07-.07.2-.07.3-.12.74-.06.57.06.88.5.31.31.31.63-.06.25-.19.25 0 .06-.06 0l-.03-.13-.16-.25zm6.07-14.69-.38-.12-.13-.44.2-.56zm.62.69q-.06.06-.44.31l-.8-.68zM19 16.63q-.38-.25-.25-.57.19-.31.56-.06t.22.53-.53.1zm-2.38-.13q-.62-.06-1.12.19-.25.06-.31.06-.07-.06.12-.31.19-.13.44-.19 0-.06.06-.06t.2-.07q.24 0 .5.07t.33.15.07.13-.28.03zm10.07 3.75q.12.38.03.78t-.22.72l-.19.25-.15.6q-.1.34-.5 1.18t-.88 1.56-1.31 1.63-1.85 1.53-2.43 1.06-3.07.44q-1.93 0-3.72-.62t-2.93-1.53-2-1.79-1.22-1.5l-.44-.62q-.69 0-1.44-.25-1.5-.56-2.19-2.06t-.12-3l-1.13-.88q-1.25-.94-.81-3.19T2.15 10t3.66-4.47 4.12-2.97 3.38.07l1.06.87Q16.81 2 18.75 2q1.43 0 2.18.81.82.94.57 2.53t-1.38 3.41q.44.5.75 1.19.75.56 1 1.75l.13.94q.07.57.09 1.06t.1.69q.43.19.68.37 1.07.5 1.69 1.13.81.19 1.1.9t-.22 1.35l-.2.19q.38.93.38 1.12.75.19 1.07.81zm-25.44-3.5q.06.13.38.38t.56.43.31.2q.56-.82 1.31-1.26h-.06q.5-1.56 1.47-3.25t2.16-3.12 2.56-2.57l1.94-1.44-1.5-1.3-.13-.76 2.07 1.69q-.07 0-.16.1l-.1.09q-.3.25-.62.56l-.87.88q-.88.87-1.82 2-1.31 1.68-2.06 3.06 0 .06.9-1.03t2-2.35 1.47-1.62q.7-.69 1.75-1.53t1.72-1.32.72-.46q1.63-1 2.81-1.25-1.56.12-3.68 1.37l-1.5-1.31-.57.25Q10.75 2.8 7.88 5T3 10.44q-2.81 4.5-1.75 6.31zM5.56 23q1.32 0 2.16-1.06t.65-2.44q-.18.44-.37.62.06-.93-.25-1.93-.13.37-.25.8l-.13-.5-.12-.43-.13-.31-.15-.22-.22-.22h-.06q-.5-.25-1.13-.25-1.19 0-2 .88t-.81 2.09.81 2.1 2 .87zm3.5-10.5q1.32-1.69 3.2-2.94 2.18-1.44 4.33-1.78t2.85.34l-.06-.06q-.38-.81-1.44-1.12.19.18.31.43-.69-.5-1.56-.68l.06.06q.31.25.38.44-1-.38-1.88-.25L15.2 7h.12q.44.13.57.31-1.57-.25-2.63.38h.13q.62 0 .8.12-1.37.32-2 .82h.76q-1.19.68-2.16 1.65t-1.34 1.6zm16.44 9.13q.44-.5.34-.91t-.65-.4l-.38.06q0-.94-.43-1.7-.82.95-2.5 1.76-2.13 1-4.63 1.06h-.4l-.2.03-.15.1-.16.18-.09.25q-.06.07-.06.2.56.18 1.34.18t1.5-.13 1.31-.28 1.07-.3l.47-.17h.06q.06 0-.13.13t-.59.31-.94.38-1.31.31-1.53.13q-.69 0-1.19-.13.19.69 1.06.88l.82.12q1.68 0 3.47-.75t2.56-1.37 1.03-.88l-.06.06q-.5 1.13-2.63 2.32t-4.37 1.18q-1.5-.06-2.13-.8-.56-.7-.31-1.57t1.12-.94h.44q4.69-.12 7.44-3.44.19-.25 0-.5t-.56-.25h-.07q-.56-.68-1.75-1.25-.18-.06-.43-.18-.5-.26-.63-.82-.06-.31-.12-1.37l-.07-.44q-.19-1.69-.87-2.06-.2-.2-.5-.2-.07 0-.32.07-.37.06-.8.5-.38.31-.7.5t-.68.25-.63.1-.78 0-.78-.04q-1 0-1.78.63t-.9 1.69q-.2 1.12.05 1.87-.37-.25-.5-.5 0 .31.1.56t.22.47.28.41.28.31.25.22.19.16l.06.06h-.5q.69.56 1.25.69-.06.12-.28.34t-.53.66-.5.94q-.13.43-.19.97t-.06.84l.06.31.31-.44q-.18 1.07.2 2.13l.43-.88q-.06 1.13.5 2.07l.12-.75q.32 1 1.07 1.69.69.62 1.97.84t2.87-.19 2.9-1.4q.5-.38.88-.79t.63-.75.47-.68.34-.66.19-.53.1-.34zm-9.31.8zm.06.26zm3.94-6.88q0-.19.1-.28t.21-.1q.31.07.25.45 0 .18-.12.28t-.25.1q-.25-.07-.2-.45zm-3.63-.06q-.93-.38-2 0-.37.12-.43.12-.25 0 .25-.43.87-.7 1.87-.32.5.2.69.5.12.2.06.22t-.44-.1zm2.88-1.88q0-.5.25-.59t.53.4.28 1.13q-.44-.25-.94-.19-.06-.3-.12-.75z',
        'w' => 0.835,
        'vw' => 27
    ],
    'mandalorian' => [
        'p' => 'M12.69 32q-.13-.44-.06-1.56.06-1.44.06-1.75.12-1.25.25-1.38v-.94l-.06-1.12v-1.31q-.07-1.2-.2-1.5-.06-.2-.06-.75v-.82q-.25-1.43-.25-2.06.07-.44.13-.62.31-.5.56-1.47t.31-1.28q.07-.44.26-.94.06-.2.09-.44t0-.4-.1-.16l-.12.19-.5.43q-.75.57-.81.82-.07.12-.07.28t.03.19.16.34q.13.44.06.94-.06.19-.12.5 0 .19-.06.5-.07.25-.13.31t-.12-.03-.1-.31-.15-.72q-.32-.94-.07-1.32.13-.25.13-.3 0-.2-.5-.63t-.63-.7q-.18-.3-.31-.12-.13.25-.19 1.38V16l.13.19q.56.62.87 1.18.25.57.25 1.5 0 1.32.13 1.57.06.06.03.1t.06.27.13.57.15.62q.13.75.13 1.25 0 .38-.07 1.06l-.12 3.88v1.31q-.06 1.63-.19 1.75-.06.13-.75-.56-.25-.25-.19-.63t.2-2.06q.05-.56.05-.75.07-.94.2-2.19.06-1 .09-1.12t.09-1.2V21.7l-.13-.25q-.12-.32-.8-1l-.82-.82-.13-.18v-.7q.07-.74.07-1.96t-.07-1.72v-.56l-.25-.25q-.25-.38-.3-.66t-.07-1.47q0-.69-.07-.75-.06-.12-.5-.19-.62 0-1.5-.87l-.84-.85-.03-.59.06-.31-.12-.2q-.07-.18-.13-.24-.31-.31-.37-.81-.07-.2.06-.25.19-.25.25-.7.31-1.43.69-1.8.12-.2.31-.1t.25.4q.13.38 0 1.13-.06.44-.06.72t.19.47q.25.38.93.63.2.12.25.18 1.07 1 1.44 1.07.5.06.63.8.19.76.37.76.07 0 .2-.1t.12-.28q0-.56-.13-1.12V8.8q-.06-.13-.06-.38-.25-.94-.32-1-.18-.25-.3-1-.13-.81-.76-1.06-.44-.19-.87-.5l-.38-.31-.28-.22-.16-.16q-.56-.31-.75-.75l-.12-.31V2.3l-.06-.8.43-.38q.44-.44.7-.57t.68-.12q.63 0 .75.12.06.07.06.25t-.09.28-.1.16-.3.56q-.13.13-.13.38.31.25.75.44.31.18.34.3t-.15.82q-.07.19-.07.25l.7.19.55-1.62q0-.32-.43-.38t-.57-.19q-.12-.31.38-1.19.25-.37.56-.25.13 0 .47-.09t.47-.1.37-.09.41-.03.37.1.38.12.31.13.4-.1l.32-.19h.63q.68 0 1-.18l.25-.13.18.06q.13.13.29.4t.09.48 0 .4.12.35.07.25q.06.31-.13.44-.44.3-.44.56 0 .06-.06.19-.37.81-.94 1.06-.68.38-1 .38-.25 0-.28.06t-.25.22-.28.31-.15.19-.16.1-.44.34-.47.37-.15.1-.16.09-.06.28.1.38q.12.37 0 .68-.13.25.05.57.2.25.07.56-.07.19-.07.56t.1.53.22.16q.12-.06.31-.75.06-.38.13-.53t.21-.28.5-.28.63-.35.6-.31q.56-.25.93-.75.56-.69.81-2.06.07-.32.2-.5t.18-.44q.06-.5.31-.69.32-.31.32.19v.5q-.07.37-.13.87v.44l.25.25.31.25v.56q0 .5.07.7V7l-.38.37q-.75.75-1.06 1.38-.19.44-.25.5-.25.37-.56 1.25l-.13.5-.44.37q-.43.44-.56.5-.31.25-.31.63 0 .25.12.47t.16.3.1.17.06.59-.07.84q-.06.75-.25.75-.25 0-.31 1.32-.06.25-.1.56t-.09 1q-.06 1.06.13 1.44.31.56-.7 1.44l-.24.18v.38l.12.56q.25 1 .25 2.31-.06 2.38.07 2.75 0 .2.03.41t.1.47.05 1.44v1.18l-.12.2q-.13.06-.34.27t-.44.35zm1.87-2.81q-.06-.07-.06-.94 0-1.44-.13-2.13-.06-.25-.06-1.06 0-1.44-.12-1.69-.2-.3-.2-1.3 0-.32.32-.57t.31-.06q.07.18.2.62.24 1 .24 1.38t-.12 1.12q-.06.5-.06 1.75.06 2-.13 2.56-.06.25-.19.32zm-4.47-2.35q-.03-.03-.09-.68t-.1-.7-.06-.4-.1-.56q-.18-1-.18-1.31 0-.13-.06-.5-.06-.25-.06-1t.06-.75q.12-.13.69.43l.3.38v.37q-.05.44-.05.94t-.03.72-.1 1.03q-.06 1.94-.12 2t-.1.03zM4.7 10.2l.69.47.43.28.25.15q.14.1.38.16 1.19.38 1.44 1.06-4.25 3.57-4.75 4.44-.88 1.69-.82 3.06t1.35 2.1 3.34.28q-.62 1.37-2.9 1.31T.88 21.88Q0 20.56-.06 18.75t1.12-4.19T4.7 10.2zm13.1 11.17q2.7.32 3.63-1.43t-.69-4.2l-4-4.24q.31-1.69 1.88-3.31 1.56 1.06 2.75 2.43T23.09 13t.88 2.16.37 1.53.1.8q.06 1.57-.72 2.73t-1.88 1.6-2.25.34-1.78-.78z',
        'w' => 0.761,
        'vw' => 24
    ],
    'markdown' => [
        'p' => 'M37.13 3.69q1.18 0 2.03.84T40 6.56v18.88q0 1.19-.84 2.03t-2.03.84H2.87q-1.18 0-2.03-.84T0 25.44V6.56q0-1.19.84-2.03t2.04-.84h34.24zm-16 18.87V9.44H17.3l-3.87 4.81-3.82-4.81H5.75v13.12h3.87v-7.5l3.88 4.82 3.81-4.82v7.5h3.81zm8.5.19L35.38 16H31.5V9.44h-3.81V16h-3.82z',
        'w' => 1.25,
        'vw' => 40
    ],
    'mastodon' => [
        'p' => 'M26.13 11.19q.06 4.5-.57 7.62-.12.57-.53 1.16t-1.1 1.16-1.5 1-1.8.78-2 .46q-6.38.75-11.2-.43 0 .43.07.87.12 1.07.62 1.78t1.41 1.03 1.63.41 1.9.16q1.19 0 2.38-.16t1.8-.34l.63-.13.13 2.31q-1.75.94-4.94 1.13-3.12.12-6.44-.75-1.25-.31-2.22-.9T2.75 26.8 1.56 25t-.78-2.28-.47-2.38-.25-2.69T0 15.04V11.2q0-1.7.4-3.13t1-2.28 1.2-1.44.96-.78L4 3.31Q6.87 2 13.03 2t9.1 1.31q.18.13.43.28t.94.78 1.22 1.44.97 2.25.44 3.13zM21.43 19v-6.94l-.09-1.4-.22-.97-.34-.9-.53-.73Q19.12 6.7 17 6.72t-3.25 1.72l-.69 1.25-.75-1.25q-1.06-1.63-3.19-1.7T5.87 8.07q-.37.38-.56.72t-.34.9-.22.98-.07 1.4V19h2.94v-7.13q0-1.93 1.5-2.24 1.5-.38 2.2 1.18.3.7.3 1.5v3.94h2.88v-3.94q0-2.25 1.62-2.69 1.38-.37 2.13 1 .25.5.25 1.26V19h2.94z',
        'w' => 0.814,
        'vw' => 26
    ],
    'maxcdn' => 'M28.81 27.69h-6.06L26 12.5q.19-1-.31-1.6t-1.5-.59h-3.07L17.37 27.7h-6.06L15 10.3H9.8L6.06 27.7H0l3.75-17.4-2.81-6h23.25q3.87 0 6.12 2.82 2.31 2.8 1.5 6.62z',
    'mdb' => [
        'p' => 'M1.09 10.03.44 22h2.74l.35-4.99L5.28 22h2.79l1.6-4.84.3 4.84h2.84l-.8-11.97H9.17l-2.54 7.33-2.65-7.33Zm17.56 0h-3V22h3s5.94.05 5.89-5.99c-.05-5.89-5.9-5.98-5.9-5.98Zm-.08 9.15V12.8s2.88.27 2.93 3.16c.05 2.9-2.93 3.22-2.93 3.22Zm14.9-4.64a3.6 3.6 0 0 0 .5-2.4C33.63 9.91 30.52 10 30.52 10h-3.24v11.97h2.84s5.44.3 5.44-3.99c0-2.7-2.1-3.44-2.1-3.44Zm-3.25-2s.85-.1 1 .6c.1.42-.25.75-.25.75h-.75Zm0 6.84V16.7s2.6-.3 2.58 1.4c-.02 1.6-2.58 1.3-2.58 1.3Zm0 0',
        'vw' => 36
    ],
    'medapps' => [
        'p' => 'M7.38 14.88q.06-.13.15-.47t.16-.6l.12-.5.19-.37.25-.13q.19.07.34.44t.38.94.4.87q.13-.5.22-1.93t.25-2.44.41-1q.19 0 .38.4t.34 1.1.28 1.28.25 1.28.13.75q.3.12 1.06.22t1.37.15l.63.07q.06.12-.16.25t-.69.15-.78.03-.87.07l-.19.78q-.12.53-.25.72t-.25.25q-.13 0-.19-.2t-.15-.52-.22-.72-.25-.84-.32-.85l-.56 4.06q-.19.57-.69-.62-.19-.38-.5-1.25l-.19-.56h-.75l-1.12-.07-1.13-.15v-.28zM14.25 28h-8.5q-1 0-1-1t1-1h8.5q1 0 1 1t-1 1zm-1.5 4h-5.5q-1 0-1-1t1-1h5.5q1 0 1 1t-1 1zm2.13-8.88q-.07.7-.63.85t-1.03-.19-.35-.97q.44-2.75.94-3.62.5-1 1.44-2.53t1.4-2.38.88-2.1.4-2.5q0-3.18-2.34-5.43T10 2 4.4 4.25 2.07 9.69q0 1.25.4 2.5t.88 2.1 1.4 2.37 1.45 2.53q.5.93.93 3.68.13.57-.34.91t-1.03.19-.63-.78q-.43-2.44-.75-3.07-.56-1.06-1.5-2.56-1-1.69-1.46-2.56t-.94-2.35T0 9.7q0-2.63 1.34-4.85T5 1.31 10 0q4.12 0 7.06 2.84T20 9.7q0 1.5-.47 2.97T18.6 15t-1.47 2.56q-1 1.63-1.5 2.56-.3.63-.75 3z',
        'w' => 0.625,
        'vw' => 20
    ],
    'medium' => 'M0 2h28v28H0V2zm23.25 6.63V8.3h-5.19l-3.69 9.25-4.25-9.25H4.7v.31l1.75 2.13q.3.25.25.62v8.32q.06.44-.25.81l-2 2.38v.3h5.62v-.3l-2-2.38q-.31-.37-.25-.81V12.5l4.94 10.69h.56L17.5 12.5v8.56q0 .25-.13.38l-1.5 1.5v.31h7.38v-.31l-1.5-1.44q-.19-.19-.13-.44V10.5q-.06-.25.13-.44z',
    'medium-m' => 'M4.5 8.88q0-.57-.44-1L1.25 4.5V4H10l6.81 14.88L22.75 4h8.38v.5l-2.44 2.31q-.32.25-.25.7v17q-.07.43.25.68l2.37 2.31v.5H19.2v-.5l2.44-2.37q.25-.25.25-.7V10.7l-6.75 17.25h-.94L6.25 10.69v11.56q-.06.75.44 1.32l3.18 3.87v.5h-9v-.5l3.2-3.88q.56-.56.43-1.3V8.87z',
    'medrt' => [
        'p' => 'M7.13 16q0 5.69 3.43 9.94t8.63 5.12q-1.75.44-3.63.44-6.43 0-11-4.53T0 16 4.56 5.03t11-4.53q1.88 0 3.63.44-5.19.87-8.63 5.12T7.12 16zm18.56 6.25q3.62-2.56 4.81-6.75t-.62-7.88q1.06 1 1.93 2.2 2.88 4.12 2.03 9.09t-4.93 7.9-9.03 2.07-7.88-5q-.81-1.2-1.31-2.5 2.87 2.93 7.12 3.21t7.88-2.34zM12 7q3.44-2.56 7.47-2.19t6.4 3.44q2.13 2.81 1.7 6.38t-3.07 6.24q.88-2.37.5-5.03T22.94 11q-2-2.63-5.07-3.63t-6.12-.18l.12-.07zm6.5 7H21q.5 0 .5.5v3q0 .5-.5.5h-2.5q-.5 0-.5.5V21q0 .5-.5.5h-3q-.5 0-.5-.5v-2.5q0-.5-.5-.5H11q-.5 0-.5-.5v-3q0-.5.5-.5h2.5q.5 0 .5-.5V11q0-.5.5-.5h3q.5 0 .5.5v2.5q0 .5.5.5z',
        'vw' => 34
    ],
    'meetup' => 'M6.19 25.9q.12.66-.5.79t-.75-.53.5-.79.75.54zm8.93 4.48q.76-.57 1.26.18t-.25 1.25q-.7.57-1.25-.18-.44-.75.24-1.25zM9.79 1.46q-.72.16-.9-.6Q8.74.2 9.44 0q.31-.06.56.1t.31.46q.2.75-.53.9zm-7.63 12.7q.47.34.56.9T2.5 16.1t-.88.6-1.03-.22-.56-.9.22-1.07q.31-.44.88-.56t1.03.21zm17-10.7q-.47.29-1.04.17t-.8-.63q-.32-.5-.2-1.06t.63-.85 1.03-.15.84.62.16 1.06-.63.85zM7.22 7.07q-.34.25-.78.2t-.69-.45T5.59 6 6 5.31q.38-.19.81-.12t.66.4.15.78-.4.7zm23.28 9.75q.31-.06.63.16t.37.53q.06.37-.13.69t-.56.37q-.31.07-.62-.12t-.38-.57q-.06-.25 0-.46t.25-.38.44-.22zm2.44-3.5q.19.57-.44.75-.69.13-.81-.5t.5-.75.75.5zm-5-4.65q.37-.29.84-.2t.75.5.19.91-.47.75-.87.2-.75-.48-.16-.9.47-.78zm-.38 8.71q1.25 2.07.75 4.41t-2.43 3.66q-1.57 1.12-3.5 1.06-.57 2.25-2.75 2.9t-3.88-.84l-.1.06-.09.13q-1.87 1.25-4.03.84t-3.47-2.22q-.87-1.3-.94-2.93Q5.2 24.12 4 22.56t-1.06-3.53 1.62-3.34q-1.31-2.2-.22-4.5t3.53-2.7q1.25-3.12 4.44-4.24t5.81.87q3.2-1.06 5.82.97t2.37 5.35q1.94.62 2.35 2.62t-1.1 3.31zm-4.37 4.7q-.13-.63-1.38-.82t-1.31-.88q-.12-1.06 1.22-4.12T23 12.37q-.25-1.8-2.13-1.87-.8 0-1.3.31-.45.25-.76-.06-.75-.75-1.19-.75-.8-.19-1.68.56t-1.25.57l-1.5-.7q-.5-.18-1.1-.09t-1.12.47-.85.88q-.18.37-.5 1.25t-.65 1.94-.63 1.93-.53 1.72-.25.97q-.44 1.25.31 2.13t2 .81q1-.06 1.57-1.06.12-.25 1.4-3.6t1.4-3.47q.32-.62.88-.3.5.24.38.93 0 .31-.88 2.5t-.93 2.37q-.07.7.37 1t1.06.13 1-.88l.54-1.09 1-2.03 1.53-3.06q.37-.5.75-.5.56 0 .43.68 0 .32-1.15 2.66t-1.28 2.84q-.13.63.03 1.25t.56 1.13.97.81q.31.13.94.25t1.28.16 1.1-.25.37-.85zm.31 5.74q.5-.37 1.1-.25t.96.63q.32.5.22 1.1t-.6.93-1.09.22-.93-.6-.22-1.09.56-.94z',
    'megaport' => 'M13.38 13.13 15.5 11l2.06 2.13v4.12l-2.06 2.06-2.13-2.06v-4.13zM15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm9.06 25.88v-5.44l-3.68-3.7v-5.5L17.13 8V3.3L15.5 2.12l-1.63 1.2V8l-3.74 3.75v5.5l-3.7 3.69v5.43l1.63 1.25 1.63-1.25v-4.06l2.06-2.06 2.13 2.06v4.06l1.62 1.25 1.63-1.25v-4.06l2.12-2.06 2.06 2.06v4.06l1.63 1.25z',
    'mendeley' => [
        'p' => 'M39.06 20.31q1 1.13.94 2.97t-.94 2.6q-.87.87-1.97 1.12t-2.06-.06-1.72-1.19-.81-2.12q-.13-1.07.31-2.07 1.44-3-1.62-4.37-2.7-1.2-5.5-.2-3.07 1.13-2.07 3.88 0 .13.07.2v.06q1.25 1.93 0 3.96t-3.7 1.97q-2.43.07-3.68-1.93t.06-4v-.07q1.13-2.81-1.93-4-2.7-1.06-5.44.07-3.07 1.3-1.88 4.25.07.12.07.18 1 2.13-.29 3.97t-3.53 1.53Q1.62 26.88.75 25.5t-.78-2.78.78-2.1q1.56-1.56 3.31-1.37Q6 19.19 6.5 17.62t-.13-4.56q-1.12-2.81.54-5.5t4.96-2.69q1.57.07 2.85.85t1.9 1.75 1.54 1.75T20 10q.88 0 1.56-.5t1.28-1.25 1.25-1.5 1.72-1.28 2.38-.6q3.31 0 4.97 2.7t.53 5.5q-.69 3-.16 4.56T36 19.25q1.81-.19 3.06 1.06zm-19-1.18q1.82 0 2.88-1.25t1.06-2.7q0-1.62-1.15-2.77t-2.79-1.16H20q-1.62 0-2.78 1.16t-1.16 2.78q0 .87.41 1.75t1.34 1.53 2.2.65h.06z',
        'w' => 1.25,
        'vw' => 40
    ],
    'meta' => [
        'p' => 'M640 317.9C640 409.2 600.6 466.4 529.7 466.4C467.1 466.4 433.9 431.8 372.8 329.8L341.4 277.2C333.1 264.7 326.9 253 320.2 242.2C300.1 276 273.1 325.2 273.1 325.2C206.1 441.8 168.5 466.4 116.2 466.4C43.42 466.4 0 409.1 0 320.5C0 177.5 79.78 42.4 183.9 42.4C234.1 42.4 277.7 67.08 328.7 131.9C365.8 81.8 406.8 42.4 459.3 42.4C558.4 42.4 640 168.1 640 317.9H640zM287.4 192.2C244.5 130.1 216.5 111.7 183 111.7C121.1 111.7 69.22 217.8 69.22 321.7C69.22 370.2 87.7 397.4 118.8 397.4C149 397.4 167.8 378.4 222 293.6C222 293.6 246.7 254.5 287.4 192.2V192.2zM531.2 397.4C563.4 397.4 578.1 369.9 578.1 322.5C578.1 198.3 523.8 97.08 454.9 97.08C421.7 97.08 393.8 123 360 175.1C369.4 188.9 379.1 204.1 389.3 220.5L426.8 282.9C485.5 377 500.3 397.4 531.2 397.4L531.2 397.4z',
        'vw' => 640,
        'vh' => 512
    ],
    'microblog' => [
        'p' => 'M24.96 22.64a12.09 12.09 0 0 0 2.94-7.86C27.9 7.72 21.68 2 14 2 6.32 2 .1 7.72.1 14.78S6.32 27.55 14 27.55c1.7 0 3.37-.28 4.96-.84.2-.06.4 0 .51.16a8.5 8.5 0 0 0 5 3.13c.14.02.3-.07.32-.22a.27.27 0 0 0-.05-.23 5.44 5.44 0 0 1 .23-6.91Zm-4.37-9.37L17.02 16l1.3 4.3a.4.4 0 0 1-.63.46L14 18.18l-3.7 2.57c-.13.1-.32.1-.46 0a.41.41 0 0 1-.15-.45l1.3-4.3-3.58-2.72a.4.4 0 0 1-.14-.45.4.4 0 0 1 .37-.28l4.5-.1 1.47-4.24a.4.4 0 0 1 .77 0l1.48 4.24 4.5.1a.4.4 0 0 1 .24.73Zm0 0',
        'vw' => 28
    ],
    'microsoft' => 'M0 2h13.44v13.44H0V2zm14.56 0H28v13.44H14.56V2zM0 16.56h13.44V30H0V16.56zm14.56 0H28V30H14.56V16.56z',
    'mix' => 'M0 4h28v12.75q0 1.69-1.38 2.4t-2.75.04-1.37-2.44v-1.5q0-1.69-1.31-2.4t-2.66-.1T17 15v7.25q0 1.31-.94 2.03T14 25t-2.06-.72-.94-2.03V11.3q0-1.25-.78-2t-1.78-.8-1.88.56-1.06 1.81v14.94q0 1.69-1.38 2.38t-2.75-.03T0 25.8V4z',
    'mixcloud' => [
        'p' => 'M26.5 13.75q1.31.25 2.38 1.1t1.68 2.06.63 2.6q0 2.43-1.75 4.18t-4.19 1.75H7.12q-2.93 0-5-2.1T.06 18.38q0-2.7 1.72-4.66t4.35-2.34q1.06-3.2 3.8-5.16t6.13-1.97q4.07 0 7.07 2.75t3.37 6.75zm-1.25 9q.69 0 1.28-.25t1.03-.69.72-1.03.28-1.28q0-1-.6-1.84t-1.52-1.22q-.13.81-.38 1.62-.25.75-1 .88t-1.28-.38-.22-1.31q.38-1.25.38-2.5 0-3.25-2.32-5.56t-5.56-2.32q-2.31 0-4.28 1.28t-2.9 3.35q1.87.5 3.24 1.87.63.57.38 1.28t-.9.91-1.35-.31q-1.31-1.32-3.13-1.32T4 15.22t-1.31 3.12 1.3 3.13 3.13 1.28h18.13zm14.69-3.28q0 4.22-2.31 7.72-.44.56-1.13.56-.75 0-1.13-.69t.07-1.37q1.87-2.81 1.87-6.19t-1.87-6.19q-.5-.69-.13-1.34t1.07-.72 1.25.62q2.3 3.38 2.3 7.6zm-4.44 0q0 3.03-1.69 5.53-.37.56-1.06.56-.81 0-1.19-.68t.06-1.38q1.25-1.75 1.25-4t-1.25-4.06q-.43-.63-.06-1.32t1.03-.71 1.22.59q1.69 2.44 1.69 5.47z',
        'w' => 1.25,
        'vw' => 40
    ],
    'mixer' => 'M7.16 4.75a2.85 2.85 0 0 0-4.22-.4 2.96 2.96 0 0 0-.3 3.93l5.74 7.68-5.77 7.76c-.9 1.2-.82 2.92.3 3.93a2.86 2.86 0 0 0 4.21-.4l8.06-10.83c.2-.27.2-.63 0-.89Zm22.23 18.97-5.77-7.76 5.74-7.68c.89-1.2.8-2.91-.3-3.92a2.86 2.86 0 0 0-4.22.4l-8 10.75c-.2.27-.2.63 0 .9l8.04 10.84a2.85 2.85 0 0 0 4.21.4 2.94 2.94 0 0 0 .3-3.93Zm0 0',
    'mizuni' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm-5 22v-14q0-1.06-.72-1.78T8 6t-1.78.72T5.5 8.5v17q2.06-2 5-3zm7.5-.63V8.5q0-1.06-.72-1.78T15.5 6t-1.78.72T13 8.5v13.38q1.25-.2 2.5-.2t2.5.2zm7.5 3.63v-17q0-1.06-.72-1.78T23 6t-1.78.72-.72 1.78v14q2.94 1 5 3z',
    'modx' => 'm22.25 15.13 2.31 1.5V30l-8.31-5.25zM27.5 4.69l-5.19 8.37-9.62-6 1.44-2.37H27.5zm-5.56 8.93L6.3 17.25l-2.87-1.88V2zM6.06 18.38l15.63-3.62-7.82 12.56H.5z',
    'monero' => 'M22 24h6.75q-1.31 2.25-3.34 3.94t-4.6 2.62-5.31.94q-4.13 0-7.66-2.03T2.24 24H9v-8l6.5 6.56L22 16v8zM5.5 21H.81Q0 18.56 0 16 0 9.56 4.53 5.03T15.5.5t10.97 4.53T31 16q0 2.56-.81 5h-4.7V8l-10.05 9.94L5.5 8v13z',
    'napster' => 'M18.63 23.38q-1.38 1.3-3.13 1.87-1.81-.63-3.13-1.88h6.26zm2.75-12.5Q18.75 9 15.5 9t-5.94 1.88Q7.62 9.25 5.25 8.62V18.7q0 2.81 1.12 5t2.94 3.34 4 1.69 4.38 0 4-1.66 2.93-3.34 1.13-5.03V8.63q-2.44.62-4.38 2.25zm-13-3.63q.62.31 1.24.69Q12.38 6.5 15.5 6.5t5.88 1.44l1.25-.75q-3-2.82-7.13-2.82T8.37 7.25zM2.75 5.81q1.56.07 3.13.5Q9.75 2 15.5 2t9.63 4.31q1.5-.43 3.12-.5v4.32Q31 11.88 31 15.09t-2.75 4.97v.07q-.44 3.62-2.38 6.25t-4.62 3.81-5.78 1.22-5.75-1.16-4.63-3.84-2.34-6.34Q0 18.32 0 15.1t2.75-4.97V5.8zM16.19 22q.56-2.5 1.65-3t4.72-.5q.44.06.7.06v.22l-.04.31-.06.35-.1.44-.18.4-.29.44-.4.4-.56.38q-.63.38-1.63.38zm-8.44-3.44h.69q1.62-.06 2.31-.03t1.56.19 1.22.53.7 1.03.59 1.72q-2.25 0-3.88-.06-.94-.07-1.56-.44-.32-.19-.57-.38t-.4-.4-.28-.44-.2-.4-.09-.44-.06-.35-.03-.31v-.22zm10.56 7.69q.13.25.2.56-1.38.82-3 .82t-3-.82q.05-.37.24-.62l.38-.38q2.37.7 4.75 0 .25.13.43.44z',
    'neos' => [
        'p' => 'M24.19 32h-5.94l-6.75-9.69v5.75L6.12 32H0V1.87L2.5 0h6.81l7.7 11V3.94L22.43 0h6.06v28.87zM.69 2.19V31l4.5-3.31V12.12l13.44 19.26h5.3l3.26-2.44H22.3L2.56.8zM5.8 31.37l5-3.68v-6.32l-4.94-7.12v13.81L1.3 31.38h4.5zM3.31.7 22.7 28.3h5.19V.7h-5v19.8L8.94.7H3.3zM17.7 12l4.5 6.38V1l-4.5 3.31v7.7z',
        'w' => 0.89,
        'vw' => 29
    ],
    'nfc-directional' => [
        'p' => 'M211.8 488.6C213.4 491.1 213.9 494.2 213.2 497.1C212.6 500 210.8 502.6 208.3 504.2C205.7 505.8 202.7 506.3 199.7 505.7C138.3 491.8 84.1 455.8 47.53 404.5C10.97 353.2-5.395 290.3 1.57 227.7C8.536 165 38.34 107.2 85.29 65.21C132.2 23.2 193-.0131 256 0C257.5 0 258.1 .2931 260.3 .8627C261.7 1.432 262.1 2.267 264 3.319C265.1 4.371 265.9 5.619 266.5 6.993C267 8.367 267.3 9.839 267.3 11.32V112.3L291.8 86.39C292.8 85.31 294 84.44 295.4 83.84C296.7 83.23 298.2 82.9 299.7 82.86C301.2 82.81 302.6 83.06 304 83.59C305.4 84.12 306.7 84.92 307.8 85.94C308.8 86.96 309.7 88.18 310.3 89.54C310.9 90.89 311.3 92.35 311.3 93.84C311.3 95.32 311.1 96.8 310.6 98.18C310 99.57 309.2 100.8 308.2 101.9L264.2 148.5C263.1 149.6 261.9 150.5 260.5 151.1C259 151.7 257.5 152 255.1 152C254.5 152 252.9 151.7 251.5 151.1C250.1 150.5 248.8 149.6 247.8 148.5L203.7 101.9C201.7 99.74 200.6 96.83 200.7 93.84C200.7 90.84 202 87.1 204.2 85.94C206.4 83.88 209.3 82.77 212.3 82.86C215.3 82.94 218.1 84.21 220.2 86.39L244.7 112.4V22.89C188.3 25.64 134.9 48.73 94.23 87.87C53.58 127 28.49 179.6 23.61 235.8C18.73 292 34.38 348.1 67.68 393.7C100.1 439.2 149.7 471.2 204.7 483.6C207.6 484.3 210.2 486.1 211.8 488.6L211.8 488.6zM171.4 126.1C170.6 127.4 169.5 128.5 168.3 129.3C147.8 143.2 131.1 161.9 119.5 183.8C107.9 205.7 101.8 230.1 101.8 254.9C101.8 279.7 107.9 304.1 119.5 325.1C131.1 347.9 147.8 366.6 168.3 380.5C170.8 382.2 172.5 384.8 173 387.8C173.6 390.7 172.1 393.8 171.3 396.2C169.6 398.7 166.1 400.4 164 400.1C161.1 401.5 158 400.9 155.6 399.2C132 383.2 112.8 361.7 99.46 336.5C86.15 311.4 79.19 283.4 79.19 254.9C79.19 226.5 86.15 198.4 99.46 173.3C112.8 148.1 132 126.6 155.6 110.6C156.8 109.8 158.2 109.2 159.6 108.8C161.1 108.5 162.6 108.5 164.1 108.8C165.5 109 166.9 109.6 168.2 110.4C169.5 111.2 170.5 112.3 171.4 113.5C172.2 114.7 172.8 116.1 173.1 117.6C173.4 119.1 173.4 120.6 173.1 122C172.8 123.5 172.3 124.9 171.4 126.1H171.4zM340.9 383.5C341.7 382.3 342.8 381.2 343.1 380.4V380.3C364.4 366.3 381.1 347.6 392.7 325.7C404.2 303.9 410.2 279.5 410.2 254.8C410.2 230.1 404.2 205.7 392.7 183.8C381.1 161.1 364.4 143.3 343.1 129.3C342.8 128.5 341.7 127.4 340.9 126.2C340.1 124.9 339.5 123.5 339.3 122.1C338.1 120.6 339 119.1 339.3 117.7C339.6 116.2 340.2 114.8 341 113.6C341.9 112.4 342.1 111.3 344.2 110.5C345.4 109.7 346.8 109.2 348.3 108.9C349.8 108.6 351.2 108.6 352.7 108.9C354.2 109.2 355.5 109.8 356.8 110.7C380.2 126.7 399.5 148.2 412.7 173.3C426 198.4 432.1 226.4 432.1 254.8C432.1 283.3 426 311.3 412.7 336.4C399.5 361.5 380.2 383 356.8 399C355.5 399.9 354.2 400.5 352.7 400.8C351.2 401.1 349.8 401.1 348.3 400.8C346.8 400.5 345.4 399.1 344.2 399.2C342.1 398.4 341.9 397.3 341 396.1C340.2 394.9 339.6 393.5 339.3 392C339 390.6 338.1 389.1 339.3 387.6C339.5 386.2 340.1 384.8 340.9 383.5V383.5zM312.3 6.307C368.5 19.04 418.7 50.28 455 95.01C485.4 132.6 504.6 178 510.3 226C515.9 274 507.9 322.7 487.1 366.3C466.2 409.9 433.5 446.8 392.6 472.6C351.7 498.3 304.4 512 256 512C254.5 512 253.1 511.7 251.7 511.1C250.3 510.6 249.1 509.7 248 508.7C246.1 507.6 246.1 506.4 245.6 505C245 503.6 244.7 502.2 244.7 500.7V401.5L220.2 427.5C218.1 429.7 215.3 430.1 212.3 431.1C209.3 431.2 206.4 430 204.2 427.1C202 425.9 200.7 423.1 200.7 420.1C200.6 417.1 201.7 414.2 203.7 412L247.8 365.4C249.1 363.2 252.9 362 255.1 362C259.1 362 262 363.2 264.2 365.4L308.2 412C310.3 414.2 311.4 417.1 311.3 420.1C311.2 423.1 309.9 425.9 307.8 427.1C305.6 430 302.7 431.2 299.7 431.1C296.7 430.1 293.8 429.7 291.8 427.5L267.3 401.6V489.1C323.7 486.3 377.1 463.3 417.8 424.1C458.5 384.1 483.6 332.4 488.5 276.2C493.3 219.1 477.7 163.9 444.4 118.3C411.1 72.75 362.4 40.79 307.4 28.36C305.9 28.03 304.6 27.42 303.3 26.57C302.1 25.71 301.1 24.63 300.3 23.37C299.5 22.12 298.1 20.72 298.7 19.26C298.5 17.8 298.5 16.3 298.8 14.85C299.2 13.41 299.8 12.04 300.6 10.82C301.5 9.61 302.6 8.577 303.8 7.784C305.1 6.99 306.5 6.451 307.9 6.198C309.4 5.945 310.9 5.982 312.3 6.307L312.3 6.307zM353.1 256.1C353.1 287.5 335.6 317.2 303.8 339.6C301.7 341.1 299 341.9 296.4 341.6C293.7 341.4 291.2 340.3 289.4 338.4L219.3 268.6C217.1 266.5 215.1 263.6 215.9 260.6C215.9 257.6 217.1 254.7 219.2 252.6C221.4 250.5 224.2 249.3 227.2 249.3C230.2 249.3 233.1 250.5 235.2 252.6L298.3 315.4C319.1 298.3 330.5 277.5 330.5 256.1C330.5 232.2 316.4 209.1 290.8 191C288.3 189.3 286.7 186.7 286.2 183.7C285.7 180.8 286.3 177.7 288.1 175.3C289.8 172.8 292.4 171.2 295.4 170.7C298.3 170.2 301.4 170.8 303.8 172.6C335.6 195 353.1 224.7 353.1 256.1V256.1zM216.7 341.5C213.7 342 210.7 341.3 208.2 339.6C176.5 317.2 158.1 287.5 158.1 256.1C158.1 224.7 176.5 195 208.2 172.6C210.4 171 213.1 170.3 215.7 170.5C218.4 170.8 220.8 171.9 222.7 173.8L292.8 243.6C294.9 245.7 296.1 248.6 296.1 251.6C296.1 254.6 294.1 257.4 292.8 259.6C290.7 261.7 287.8 262.9 284.9 262.9C281.9 262.9 278.1 261.7 276.9 259.6L213.8 196.7C192.9 214 181.6 234.7 181.6 256.1C181.6 279.1 195.7 303.1 221.3 321.1C223.7 322.9 225.4 325.5 225.9 328.5C226.4 331.4 225.7 334.4 224 336.9C222.3 339.3 219.6 341 216.7 341.5L216.7 341.5z',
        'vw' => 512,
        'vh' => 512
    ],
    'nfc-symbol' => [
        'p' => 'M392.9 32.43C400.6 31.1 408.6 32.89 414.1 37.41C498.2 96.14 544 173.7 544 255.1C544 338.2 498.2 415.9 414.1 474.6C409.3 478.6 402.4 480.5 395.5 479.9C388.5 479.3 382 476.3 377.1 471.4L193.7 288.7C188.1 283.2 185 275.7 184.1 267.8C184.1 260 188.1 252.5 193.6 246.9C199.2 241.4 206.7 238.2 214.5 238.2C222.4 238.2 229.9 241.3 235.4 246.8L400.5 411.2C455.1 366.5 484.8 312 484.8 255.1C484.8 193.5 447.9 132.9 380.9 85.76C374.5 81.24 370.1 74.35 368.8 66.62C367.4 58.89 369.2 50.94 373.8 44.53C378.3 38.12 385.2 33.77 392.9 32.43V32.43zM186.9 479.6C179.2 480.9 171.3 479.1 164.8 474.6C81.67 415.9 35.84 338.2 35.84 255.1C35.84 173.7 81.67 96.14 164.8 37.41C170.5 33.4 177.4 31.53 184.4 32.12C191.3 32.71 197.8 35.72 202.7 40.63L386.1 223.3C391.7 228.8 394.8 236.3 394.8 244.2C394.9 251.1 391.8 259.5 386.2 265.1C380.7 270.6 373.2 273.8 365.3 273.8C357.5 273.8 349.1 270.7 344.4 265.2L179.3 100.7C124.7 145.9 95.03 199.9 95.03 255.1C95.03 318.5 131.9 379.1 198.1 426.2C205.4 430.8 209.7 437.6 211.1 445.4C212.4 453.1 210.6 461.1 206.1 467.5C201.6 473.9 194.7 478.2 186.9 479.6V479.6z',
        'vw' => 576,
        'vh' => 512
    ],
    'nimblr' => [
        'p' => 'M14.53 18.69q.72 0 1.22.5t.5 1.19q0 .37-.13.68t-.37.53-.56.35-.63.12q-.75 0-1.25-.47t-.5-1.18.5-1.22 1.22-.5zM6.2 20.37q0-.43.22-.84t.62-.62.88-.22.84.22.6.62.21.84q0 .75-.47 1.22t-1.18.47-1.22-.47-.5-1.21zm4.94-10.43q4.56 0 7.84 3.22t3.28 7.81-3.28 7.81T11.09 32t-7.84-3.22T0 20.94V0l2.81 14.2q1.32-2.06 3.82-3.16t4.5-1.1zm0 19.25q3.43 0 5.84-2.4t2.4-5.82-2.4-5.81-5.84-2.41-5.88 2.4-2.44 5.82 2.44 5.81 5.88 2.4z',
        'w' => 0.693,
        'vw' => 22
    ],
    'nintendo-switch' => 'M6 2.13Q6.44 2 9.94 2t3.56.13.06 13.87v11.72l-.06 2.22q-.12.06-3.69.06-3.56-.06-4-.12-1.5-.38-2.65-1.16t-1.9-1.94-1.13-2.6Q0 23.64 0 16.5q0-8.06.25-9.13.56-2.06 2.1-3.46T6 2.13zM11.25 16V4.25H9q-2.75 0-3.63.38-2.43.93-3 3.56-.06.44-.06 7.81t.06 7.75q.2.94.7 1.72t1.27 1.31 1.66.78q.5.13 2.88.2h2.37V16zM6.5 7.81q1.06-.25 2 .4T9.63 10q.12.94-.38 1.75t-1.44 1.13q-.37.12-.94.12t-1-.25q-.8-.38-1.25-1.25-.18-.38-.18-.94-.13-1.31.8-2.12.57-.5 1.26-.63zm9.88-5.75Q16.43 2 18.96 2t3.03.13q1.5.25 2.75 1.03t2.06 2.03 1.13 2.75Q28 8.37 28 16v.38q0 5.75-.03 6.84T27.69 25q0 .06-.06.13-.2.5-.44 1-.81 1.56-2.28 2.59t-3.28 1.22q-.38.06-2.72.06h-2.47q-.06-.06-.06-14V2.06zm6.18 12.63q-1.06-.32-2.06.25t-1.31 1.62q-.25.82 0 1.63.31 1.06 1.31 1.62t2.06.32q1.38-.38 1.94-1.82.38-.87 0-1.81-.5-1.44-1.94-1.81z',
    'node' => [
        'p' => 'M19.75 28.25q-.19 0-.38-.13l-1.18-.68q-.2-.13-.07-.2.32-.05.57-.24.06 0 .12.06l.88.5q.06.07.12 0l3.63-2.06q.06-.06.06-.13v-4.12q0-.06-.06-.13l-3.63-2.06q-.06-.06-.12 0l-3.57 2.06q-.06.07-.06.13v4.12q0 .07.06.13l.94.56q.44.19.66.03t.22-.4v-4.13q0-.06.12-.06h.44q.13 0 .13.06v4.13q0 1.12-1.07 1.12-.37 0-.81-.19l-1-.56q-.38-.25-.38-.69v-4.12q0-.44.38-.69l3.62-2.06q.38-.25.75 0l3.63 2.06q.37.25.37.69v4.12q0 .5-.37.7l-3.63 2.05q-.12.13-.37.13zm2.94-4.13q0 1.25-1.82 1.25-1.93 0-1.93-1.3 0-.13.12-.13h.5q.07 0 .07.06.12.75 1.25.75t1.12-.56q0-.44-1.31-.57-1.57-.12-1.57-1.06 0-1.19 1.63-1.19 1.69 0 1.81 1.25 0 .07-.03.1t-.1.03H22q-.13 0-.13-.06-.12-.7-1.12-.7-.94 0-.94.5 0 .2.22.29t1.03.22q.88.12 1.25.34t.38.78zM26.12 21q0 .19-.09.34t-.28.25-.38.1q-.25 0-.46-.19t-.22-.5.22-.5.5-.19.5.19.21.5zM26 21q0-.56-.6-.56t-.59.6.6.55T26 21zm-.31.38h-.13q-.06-.2-.06-.25t-.06-.07h-.13v.32h-.19v-.75h.25q.32 0 .32.18 0 .13-.2.2.13 0 .2.18v.19zm-.19-.54q0-.09-.06-.09h-.13v.19h.07q.12 0 .12-.1zm-16.94-8.9v5.94q0 .12-.12.18t-.19 0l-2.31-1.3q-.2-.13-.2-.38v-2.75q0-.32-.24-.44l-1-.56q-.06-.07-.19-.07t-.25.06l-1 .57q-.19.12-.19.44v2.75q0 .25-.25.37L.37 18.06q-.12.07-.25 0T0 17.88v-5.94q0-.25.25-.38l3.81-2.25q.13-.06.19-.06h.06q.13 0 .19.06l3.88 2.25q.18.13.18.38zm20.94-6.5V16.5q0 .25-.19.38l-3.81 2.25q-.25.12-.5 0l-3.81-2.25q-.25-.13-.25-.38v-4.44q0-.25.25-.43l3.8-2.2q.26-.12.5 0l.95.57q.25.13.25-.12V4q0-.13.1-.22t.21-.03l2.31 1.31q.2.13.2.38zm-2.88 8.06v-.06l-1.3-.82h-.13l-1.32.82q-.06 0-.06.06V15q0 .13.06.13l1.32.74q.06.07.06 0l1.38-.74V13.5zm13.13 0-2.25 1.31q-.13.07-.25 0t-.13-.18V13.5q0-.13-.12-.19l-1.19-.69q-.12-.06-.25 0l-1.19.7q-.12.05-.12.18v1.38q0 .12.12.24l3.88 2.2q.12.06.12.18t-.12.19L35.94 19q-.25.13-.44 0l-3.82-2.19q-.25-.12-.25-.37V12q0-.25.25-.37l3.82-2.25q.25-.13.44 0l3.8 2.25q.26.12.26.37v1.06q0 .25-.25.44zm-4.81.19.75-.44h.06l.75.44q.06 0 .06.06v.88l-.06.06-.75.44h-.06l-.75-.44q-.06 0-.06-.06v-.88zm-15.88 2.69v.12q-.06.19-.25.31L15 19.01q-.25.12-.44 0l-3.87-2.2q-.25-.12-.25-.43V12q0-.25.25-.38l3.81-2.24q.25-.13.5 0l3.81 2.24q.25.13.25.38v4.38zm-4.68-7.75zm4.8 8.12-.05-.06v.06h.06z',
        'w' => 1.25,
        'vw' => 40
    ],
    'node-js' => 'M14 31.75q-.63 0-1.19-.31l-3.87-2.32q-.63-.3-.13-.5.94-.3 1.75-.75.13-.12.31 0l2.94 1.75q.2.13.38 0L25.75 23q.13-.13.13-.38V9.32q0-.2-.2-.32l-11.5-6.63q-.18-.12-.37 0L2.31 9q-.18.12-.18.31v13.31q0 .25.18.32l3.13 1.87q1.31.5 2.06.1t.75-1.28V10.5q0-.38.31-.38H10q.38 0 .38.38v13.13q0 1.68-.9 2.65t-2.54.97q-1.19 0-2.69-.75l-3-1.75Q.06 24.06.06 22.62V9.32q0-1.38 1.2-2.07L12.8.56q1.2-.69 2.38 0l11.56 6.69Q28 7.94 28 9.3v13.31q0 1.44-1.25 2.13l-11.55 6.7q-.56.31-1.19.31zm9.31-13.13q0 1.88-1.5 2.91t-4.25 1.03q-6.12 0-6.12-4.25 0-.31.37-.31h1.44q.31 0 .38.25.18 1.25.97 1.78t2.96.53q3.44 0 3.44-1.81 0-.32-.03-.47t-.28-.35-.66-.34-1.18-.28-1.91-.25q-1.69-.19-2.75-.5T12.5 15.5t-.62-2q0-1.75 1.37-2.72t3.88-.97q5.37 0 5.75 4 0 .19-.1.28t-.28.1H21q-.25 0-.31-.26-.25-1.18-1.03-1.65t-2.53-.47q-3 0-3 1.62 0 .63.69.91t3.25.66q2.87.37 4.06 1.12t1.19 2.5z',
    'npm' => [
        'p' => 'M18 18h-2v-4h2v4zm18-8v12H18v2h-8v-2H0V10h36zm-26 2H2v8h4v-6h2v6h2v-8zm10 0h-8v10h4v-2h4v-8zm14 0H22v8h4v-6h2v6h2v-6h2v6h2v-8z',
        'w' => 1.125,
        'vw' => 36
    ],
    'ns8' => [
        'p' => 'M11.69 10h2.81l-3 10H7.94l-3.19-6.63L2.81 20H0l3.06-10h3.07l3.43 7.13zM40 9.94q.19 1.19-1.1 2t-3.02 1.19q.87 1.43.96 3.12t-.78 3.44q-1.18 2.37-3.65 3.6t-4.78.65q-.94-.25-1.66-.75t-1.22-1.25-.69-1.69q-.06-.31-.06-.69-.44.25-1.13.32-1.37.12-10.06.12l.69-2.5h7.94q.69 0 .87-.44t.25-.93q.07-.2-.03-.29t-.4-.09h-5.5q-1.5 0-1.91-.4t-.1-1.41q.63-2.25 1.07-2.94.56-.81 1.94-.94.69-.06 10-.06L27 12h-8.3q-.69 0-.88.44-.06.18-.18.75-.07.12 0 .25t.25.15.43.03h5.5q1.82 0 2.13 1.07 2.06-2.25 5.87-3v-.07q-.56-1.87 1.35-2.97T37.38 8q2.37.37 2.62 1.94zm-6.81 6.56q.25-1.44-.44-3-3.06.38-4.31 1.69-1.13 1.19-.85 2.5t1.72 1.56q1.25.19 2.4-.62t1.48-2.13zm1.31-5.19q1.5-.25 2.19-.62.62-.38.53-.88t-.84-.62q-1.32-.32-2 .44-.57.62.12 1.68z',
        'w' => 1.248,
        'vw' => 40
    ],
    'nutritionix' => [
        'p' => 'M5.5.5h.31l.75.03 1.1.1L8.9.9l1.31.5 1.22.8 1 1.17.56 1.56.07 2.06q0-.06.06-.22t.28-.62.53-.82.84-.72 1.16-.56 1.56-.1 2 .54q0 .06-.06.22t-.25.65-.47.85-.72.84-1 .72-1.31.38T14 8q0-.06.06-.19t.22-.5.35-.69.53-.71.75-.7 1-.5 1.22-.27q-.13 0-.28-.03t-.75.06-1.16.34-1.37.97T13 7.5l-.15-.66q-.1-.4-.57-1.43t-1.06-1.85-1.69-1.5-2.4-.81l.53.19q.35.13 1.15.62t1.44 1.19 1.16 1.94T12 8q0 .19-1.67-.06t-3.28-2.1T5.5.5zM25 20.25q0 4.94-2 8.25-4.13 3-10.5 3T2 28.5q-2-3.31-2-8.25Q0 16.12 2.5 12q4.13-3.06 10-3.06t10 3.06q2.5 4.13 2.5 8.25zm-17.5 6.5q0-.75-.5-1.25T5.75 25t-1.25.5-.5 1.25.5 1.25 1.25.5T7 28t.5-1.25zm0-4.16q0-.71-.5-1.21t-1.25-.5-1.25.5-.5 1.21.5 1.25 1.25.54T7 23.84t.5-1.25zm0-4.12q0-.72-.5-1.22t-1.25-.5-1.25.5-.5 1.22.5 1.25 1.25.53T7 19.72t.5-1.25zm4.5 8.28q0-.75-.5-1.25t-1.25-.5-1.25.5-.5 1.25T9 28t1.25.5 1.25-.5.5-1.25zm0-4.16q0-.71-.5-1.21t-1.25-.5-1.25.5-.5 1.21.5 1.25 1.25.54 1.25-.54.5-1.25zm0-4.12q0-.72-.5-1.22t-1.25-.5-1.25.5-.5 1.22.5 1.25 1.25.53 1.25-.53.5-1.25zm4.5 8.28q0-.75-.5-1.25t-1.25-.5-1.25.5-.5 1.25.5 1.25 1.25.5T16 28t.5-1.25zm0-4.16q0-.71-.5-1.21t-1.25-.5-1.25.5-.5 1.21.5 1.25 1.25.54 1.25-.54.5-1.25zm0-4.12q0-.72-.5-1.22t-1.25-.5-1.25.5-.5 1.22.5 1.25 1.25.53 1.25-.53.5-1.25zm4.5 8.28q0-.75-.5-1.25t-1.25-.5-1.25.5-.5 1.25T18 28t1.25.5 1.25-.5.5-1.25zm0-4.16q0-.71-.5-1.21t-1.25-.5-1.25.5-.5 1.21.5 1.25 1.25.54 1.25-.54.5-1.25zm0-4.12q0-.72-.5-1.22t-1.25-.5-1.25.5-.5 1.22.5 1.25 1.25.53 1.25-.53.5-1.25zM22.5 16q-.5-2.44-1-3.5-3.88-2-9-2t-9 2q-.44 1-1 3.5 4.94-.94 10-.94t10 .94z',
        'w' => 0.781,
        'vw' => 25
    ],
    'octopus-deploy' => 'M28.48 21.82c-2.87-2.44-2.3-4.86-1.01-8C29.7 8.38 26 2.13 20.62.52c-5.8-1.75-12.23 1-14.39 6.8a11.25 11.25 0 0 0-.69 4.6c.11 1.85.92 3.31 1.5 5.02 1.08 3.13-1.75 5.8-4.16 7.35-2.92 1.88-2.27 2.49-.52 2.61 1.46.1 2.78-.28 4.07-.94.58-.29 2.55-1.18 2.82-1.78-.76 1.66-2.3 4.54-1.34 6.37 1.2 2.27 4.2-1.98 4.8-2.86.54-.78 2.68-5.08 3.97-2.93 1.18 1.96.54 4.78 2.24 6.54 2.05 2.14 3.2-1.14 3.21-2.76.01-1.03-.38-6 1.87-3.75 1.34 1.34 3.3 4.45 5.54 4.19 2.42-.28-1.39-4.25-1.77-4.92.34.27 3.36 2.13 3.36.6.01-1.18-1.88-2.17-2.65-2.84Zm0 0',
    'odnoklassniki' => [
        'p' => 'M17.19 20.88q-2.07 1.3-5.63 1.68l6.07 6.07q.8.8.43 1.9t-1.34 1.38-1.9-.4L10 26.68l-4.75 4.75q-.81.81-1.9.44t-1.38-1.35.47-1.9l4.75-4.82 1.31-1.25q-3.69-.37-5.69-1.68-1.68-1.13-2.18-1.85t.06-1.84q.37-.82 1.22-1.03t2.15.65q.25.19.75.5t2.04.81 3.15.5 3.13-.43 2.12-.94l.69-.44q1.31-.87 2.16-.65t1.28 1.03q.18.68.25 1.06t-.32.87-.81.85-1.31.9zM1.87 8.13q0-3.38 2.38-5.76T10 0q2.19 0 4.06 1.1t2.97 2.93 1.1 4.1q0 3.3-2.38 5.68T10 16.2t-5.75-2.4-2.37-5.67zm4.13 0q0 1.62 1.19 2.78T10 12.06t2.81-1.15 1.2-2.78q0-1.7-1.2-2.85t-2.8-1.15q-1.07 0-2 .53T6.52 6.09 6 8.13z',
        'w' => 0.625,
        'vw' => 20
    ],
    'odnoklassniki-square' => 'M11.5 11.1q0-1.04.75-1.79T14 8.56t1.75.75.75 1.78-.75 1.75-1.75.72-1.75-.72-.75-1.75zM28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22q1.25 0 2.13.88T28 5zM8.94 11.06q0 2.13 1.47 3.6t3.6 1.47 3.59-1.47 1.46-3.57T17.6 7.5 14 6t-3.6 1.47-1.46 3.6zm10.87 5.69q-.25-.5-.75-.63t-1.37.38Q16.3 17.63 14 17.63q-1 0-1.94-.29t-1.37-.53l-.38-.31q-.87-.5-1.37-.37t-.75.62q-.38.44-.2.88t.48.68 1.03.75q1.25.82 3.56 1.07l-.81.75q0 .06-3 3-.81.8.06 1.72t1.69.09l3-3 3 3q.81.81 1.69-.06t.06-1.75l-2.94-3-.81-.75q2.19-.25 3.5-1.07 1.06-.68 1.37-1.15t-.06-1.16z',
    'odysee' => [
        'p' => 'M406.7 463c-42.3 30.8-94.4 49-150.7 49C144.9 512 50.3 441.2 14.9 342.2c2.4 1.7 5.9 3.6 7.9 4.4c16.3 7.4 40.1-5.4 62.9-28.7c6.9-6.9 14.4-12.4 22.8-17.3c18.3-11.9 37.6-20.8 58.4-27.2c0 0 22.3 34.2 43.1 74.8s-22.3 54-27.2 54c-.3 0-.8 0-1.5-.1c-11-.5-70-3-56 51.1c14.9 57.4 97.5 36.6 139.6 8.9s31.7-118.3 31.7-118.3c41.1-6.4 54 37.1 57.9 59.4c.8 4.6 1.1 9.9 1.4 15.5c1.1 21.2 2.3 45.6 35.3 46.4c5.3 0 10.6-.8 15.5-2zm-95.3-23.7c-2-.5-3.5-2.5-3-5c1-2.5 3-3.5 5-3s3.5 3 3 5s-2.5 3.5-5 3zm-207-95.6c1.5-.5 3.5 1 4 3c0 2-1 4-3 4c-1.5 .5-3.5-1-4-3c-.5-1.5 1-3.5 3-4zM451.8 421C489.3 376.4 512 318.8 512 256c0-67.5-26.1-128.9-68.8-174.7c-.1 23.5-6.1 48.2-16.8 69.2c-11.9 20.3-49 58.9-69.8 78.7c-.7 .3-1.1 .9-1.5 1.4c-.2 .2-.3 .4-.5 .6c-5 6.9-4 16.8 3 21.8c21.3 15.8 56.4 45.6 59.4 72.8c3.5 34.9 27.9 75.6 34.2 86.2l0 0c.8 1.3 1.3 2.1 1.4 2.4c0 2.2-.4 4.3-.8 6.5zM390.7 251c-.5 3 1 5.9 4 6.4s5.9-1 6.4-4s-1-5.9-4-6.4c-3-1-5.9 1-6.4 4zm61.4-60.9l-11.4 5.4-3 12.9-5.4-11.4-12.9-3 11.4-5.4 3-12.9 5.4 11.4 12.9 3zM395.5 41.3c-16.2 8.2-22.1 32.8-29 61.4l0 0c-.3 1.4-.7 2.8-1 4.2c-9.5 38.5-30.6 37.6-41.7 37.2c-1.1 0-2-.1-2.9-.1c-5.1 0-6-4-8.9-17.1c-2.6-12.1-6.9-32-17.9-63.6C271.4-2.5 211.4 13.9 165.9 41.1C110.6 74.2 131.5 143 146.1 190.5c.7 2.2 1.4 4.4 2 6.6c-4 4-13.8 7.5-26 11.9c-12.1 4.3-26.6 9.5-40.3 16.9C47.9 243.9 11.5 274.9 2 288.5C.7 277.8 0 267 0 256C0 114.6 114.6 0 256 0c51.4 0 99.4 15.2 139.5 41.3zM58.9 189.6c-1.5-2-4.5-3-6.4-1.5s-3 4.5-1.5 6.4s4.5 3 6.4 1.5c2.5-1.5 3-4.5 1.5-6.4zM327.3 64.9c2-1.5 5-.5 6.4 1.5c1.5 2.5 1 5.4-1.5 6.4c-2 1.5-5 .5-6.4-1.5s-.5-5 1.5-6.4zM95.1 105c-.5 1.5 .5 3 2 3c1.5 .5 3-.5 3-2c.5-1.5-.5-3-2-3s-3 .5-3 2zm84.7-.5c-3.5-43.1 37.1-54 37.1-54c44.1-15.4 56 5.9 66.4 37.6s3 42.6-38.6 58.9s-61.9-4.5-64.9-42.6zm89.6 14.9h1c2.5 0 5-2 5-5c2-6.9 1-14.4-2-20.8c-1.5-2-4-3.5-6.4-2.5c-3 1-4.5 4-3.5 6.9c2 4.5 3 9.9 1.5 14.9c-.5 3 1.5 5.9 4.5 6.4zm-9.9-41.6c-2 0-4-1-5-3s-2-3.5-3-5c-2-2-2-5.4 0-7.4s5.4-2 7.4 0c2 2.5 3.5 5 5 7.4s.5 5.9-2.5 7.4c-.6 0-1 .2-1.3 .3c-.2 .1-.4 .2-.6 .2z',
        'vw' => 512,
        'vh' => 512
    ],
    'old-republic' => 'M14.75.63h1.38q.3 0 .68.06l.94.06q.06.06.16.06t.18.03.16.04l1.5.3q1.13.32 2.25.88 2.5 1.13 4.44 3.07T29.5 9.5l.1.19.15.25.44 1.25q.44 1.19.62 2.38 0 .3.07.56.06.37.06.75.06.56.06 1.56 0 .5-.03.9t-.1.66v.16l-.02.19-.03.15q-.13.75-.2 1-.37 1.75-1.12 3.32-1.5 3.12-4.18 5.3-1.63 1.38-3.57 2.2-.69.31-1.19.5-.56.12-.68.18-.7.2-1.25.32l-.88.12q-.75.13-1 .13-.5.06-1.25.06t-1.25-.06q-.37 0-.62-.07h-.16l-.19-.03-.15-.03q-.7-.12-.82-.12l-.75-.19q-.68-.19-1.62-.5-1.06-.44-1.88-.88l-.43-.25q-1-.62-2-1.37Q2.3 25.38.88 21.32q-.32-.82-.5-1.7l-.2-1.12-.02-.28L.13 18l-.04-.18q-.03-.12-.03-.29v-.21Q0 16.62 0 15.5q0-.37.06-.62v-.22l.03-.22.04-.19v-.15l.03-.2.03-.15L.3 13q.32-1.62.94-3l.16-.31.1-.25Q2.24 7.8 3.37 6.5q1.87-2.37 4.56-3.94.81-.44 1.44-.69 1.43-.62 3.06-.93.06-.07.69-.13l.25-.03.25-.03q.12-.06.37-.06h.25q.13 0 .5-.07zm.5 1.56q-.75 0-1.69.12-.62.07-1.37.25-1.32.32-2.5.94-1.07.44-1.94 1.06L7 5.13q-.5.37-.94.75 0 .06-.44.43-2.69 2.82-3.56 6.57-.19.81-.31 1.62-.07.75-.07 1.7 0 .74.07 1.3.06.5.19 1.32.18.75.25.87.56 2.19 1.87 4.13.75 1.18 1.88 2.25 1.5 1.43 3.43 2.43.82.38 1.38.57.69.25 1 .31 1.06.31 2.12.5h.57q1.3.13 2.56 0l.37-.03q.31-.03.5-.1.57-.06.63-.12.62-.12.81-.19 2.5-.69 4.57-2.25 1.5-1.12 2.62-2.56l.63-.81q.3-.44.56-1 .25-.44.44-.82.56-1.12.87-2.3.25-1.07.38-1.95.12-1.12.06-2.56l-.03-.4-.03-.29q-.2-1.87-.88-3.5-.56-1.44-1.37-2.69-1.25-1.8-2.88-3.12-2.44-1.94-5.37-2.63-.7-.18-1.5-.3h-.44l-.25-.07q-.69-.06-1.44 0zM8.69 5.25q1.19-.81 2.75-1.31-.06.06-.28.19t-.35.18q-1.06.63-1.87 1.5-.69.75-1.06 1.57-.32.68-.44 1.37-.13.44-.07.88 0 1.25.7 2.43.5.94 1.37 1.75.12.07.47.35t.47.4l.43.44q.25.38.38.69v2q.06.13.06.25v2.19h-.19q-1 0-2-.44-.62-.31-1.06-.63-.56-.43-1-.87-.56-.69-1-1.5-.19-.38-.25-.38 0-.12-.06-.18t-.1-.16-.03-.15q-.37-.88-.5-1.94-.06-.63-.12-.82-.07-1.5.31-2.93.25-.82.5-1.5.63-1.32 1.69-2.38.56-.56 1.25-1zm11-1.31q.12.06.5.19.69.25 1.12.5 1.32.62 2.28 1.56t1.54 2.12q.25.5.25.57.62 1.37.68 2.94.07.3.07.87 0 1.06-.25 2.13-.25.8-.5 1.37-.38.69-.7 1.19-.68 1-1.62 1.69-1 .75-2.19.93-.3.07-.68.13-.25 0-.38-.06v-.38q0-.12.03-.34t.03-.35v-3.3q.07-.19.25-.44.25-.5 1.07-1.12.56-.44.93-.88.75-.81 1.2-1.75.24-.62.37-1.12.06-.38.06-.88 0-.69-.25-1.44-.12-.43-.44-1-.37-.62-.87-1.25-.94-.93-2.19-1.68l-.13-.07-.18-.12zm-6.31 3.68q.12-.12.18-.25-.06.13-.12.2-.31.37-.44.8t-.12.63q0 .31.06.5.06.69.5 1.31.25.32.44.44.06.07.37.19.44.25.56.38.25.18.25.56v14.06l.13.25q.06.19.16.28t.21.1q.13-.07.2-.2.12-.12.18-.3v-2.6l.03-.34.03-.38v-4.5q.07.38.07.69l.03.19q.03.13.03.25v.19l.03.28.03.31.03.25.03.28.06.25q.2 1.19.32 1.69t.25.87q.25.88.62 1.44.13.19.38.44h.06q.06.06.12 0 0-.06.07-.19.31-.62.81-1.44.06-.18.13-.25.12-.12.37-.25.31-.12.56-.12.38-.06 1.25.12.44.07.7.07h.68q.81-.07 1.56-.38.57-.25.94-.5 1.06-.62 1.88-1.69l.12-.12.13-.19q-.07.25-.25.63l-.32.5q-.5 1-1.06 1.75-.38.43-.88.87-.68.69-1.37.94-.81.37-1.63.25l-.68-.19-.5-.25q-.25-.12-.57-.37l.5.5q.5.43.88.68t.81.32q.06.06.25.06-.06.06-.25.19-.81.56-1.87 1-2 .87-4.13 1h-.87q-.7 0-.94-.07-1-.06-2.06-.3-1.07-.32-1.91-.7t-1.53-.8l-.38-.32q.63-.07 1.2-.38.55-.25 1-.69.18-.18.37-.43-.38.25-.7.37-.37.19-.8.31-.5.13-1.13.07-.75-.13-1.44-.57l-.75-.62q-.75-.81-1.56-2.13-.25-.37-.69-1.18-.06-.07-.12-.2T3.75 21q.13.06.22.16l.34.34q.94 1 1.82 1.56t1.56.7l.28.02.28.03h1.5q.13 0 .63-.06.19-.06.44-.06.37-.13.68 0 .44.19.75.62.2.25.38.57 0 .06.06.21t.13.29q.12.18.18.37l.2.19q.06-.06.12-.19.12-.19.31-.81.19-.69.31-1.25.13-.44.32-1.75 0-.13.03-.25t.03-.28v-.22q.12-1.57.19-2v-2.57q0-.5-.13-1.25-.06-.68-.19-1.12-.12-.63-.56-1.31-.37-.57-.75-.88-.37-.44-.81-.56-.07 0-.32-.06 0-.13.07-.5.19-.75.44-1.44l.3-.75q.38-.63.82-1.13zm1.93.13.07-.06v3.06h-.07q0-.13-.06-.38t-.12-.3V9.9l-.07-.16H13.5v-.06q.06-.07.16-.1l.09-.03q.12 0 .34-.06t.35-.13q.19-.06.25-.06v-.1q-.07 0-.25-.07-.07-.06-.2-.1t-.21-.05-.22-.04q0-.06-.12-.09t-.2-.03v-.06h.95q.12-.07.34-.03t.28-.04h.03l.03-.06q0-.12.03-.28t.1-.31.06-.28zm.38-.06q.06 0 .06.06.06.06.1.31t.09.38q0 .06.03.16l.03.09q.06.06.25.03t.25-.03q.38.06.94.06h.12v.07q-.06 0-.25.06-.06 0-.06.06-.19 0-.75.19l-.06.06-.06.13q.12 0 .18.06.13 0 .44.1t.5.15q.06 0 .06.03v.1H16q0 .06-.06.18v.13q-.06.12-.1.25t-.03.25-.06.19h-.06V7.69zM4.06 10.5l.2-.38q-.13.63-.2 1.44Q4 12.2 4 13q.13 2 .88 3.63.37.62.68 1.06.7.87 1.25 1.31.44.44.94.69l.63.31 1 .32q.06 0 .3.06t.38.06q.5.19.82.56.43.44.62 1.13.13.19.19.56v.2q-.69.24-1.25.37-.31.06-.81.06-.88 0-1.88-.31l-.69-.32q-.75-.37-1.43-.87l-.5-.5q-.82-.88-1.38-2.13-.25-.62-.37-1.06l-.2-.75q-.05-.13-.09-.31t-.03-.25L3 16.44l-.13-1.06q.07-.75.13-1.13v-.4l.06-.16q.07-.62.2-1.06.3-1 .8-2.13zm22.75-.38q.07.07.2.32.24.62.37.87.75 1.82.8 3.7v.3q0 .75-.18 1.75l-.19.7q-.43 1.5-1.12 2.62-.81 1.25-2 1.93-.56.38-1.13.57-.62.25-1.12.37-.63.07-1.25.07-.75-.07-1.44-.32-.06 0-.12-.06l-.25-.06q0-.07.06-.32.12-.56.37-1 .2-.5.57-.75.18-.18.5-.3.12-.07.75-.2.68-.18 1.3-.43.95-.44 1.7-1.2.25-.3.31-.3l.62-.76q.63-.87 1.07-2.06.19-.69.25-1 0-.12.06-.56.13-.69.13-1.13 0-.81-.07-1.25-.06-.81-.19-1.5z',
    'opencart' => [
        'p' => 'M26.44 27.56q0 1.2-.82 2t-2 .82-2.03-.82-.84-2 .84-2.03 2.04-.84 2 .84.8 2.03zM10.56 24.7q1.2 0 2.03.84t.85 2.03-.85 2-2.03.82-2-.82-.81-2 .81-2.03 2-.84zM28.81 7.8q2.44 0 4.28.07t3.1.3l2.19.45 1.18.72.44.87-.37 1.16-.94 1.31-1.6 1.6-1.96 1.74-2.44 2.1q-3.19 2.75-4.94 4.31.75-1.37 2.34-3.12t2.79-2.91 1.8-2.38-.09-1.93-3.46-1.13-7.82-.34q-3.62.06-6.65-.19t-5.1-.63-3.84-1.09-2.75-1.28-1.9-1.5-1.32-1.47-.97-1.53T0 1.63q2.62 2.75 5.78 4.03t8.1 1.72 14.93.43z',
        'w' => 1.248,
        'vw' => 40
    ],
    'openid' => 'm17 27-4.25 2q-5.44-.5-9.1-3.06T0 19.87q0-3.3 3.4-5.84t8.6-3.16v2.7q-3.44.56-5.6 2.3t-2.15 4q0 2.38 2.4 4.2t6.04 2.24V5.06L16.94 3v24H17zm11-8.81-8.19-1.81 2.31-1.25q-1.87-1.13-4.37-1.57v-2.69q4.31.5 7.5 2.44l2.19-1.19z',
    'opera' => 'M19.63 2.06q-3.82 0-6.82 2.32T8.5 10.03t-.9 7.28 2.77 6.94q3.82 4.69 9.25 4.69 3.32 0 6.2-1.88Q21.43 31 15.5 31h-.75q-6.19-.31-10.47-4.78T0 15.5Q0 9.06 4.53 4.53T15.5 0h.06q5.88 0 10.25 3.94Q23 2.06 19.62 2.06zM26 26.88q-1.94 1.18-4.16 1.09t-4.09-1.4q2.63-1 4.34-4.07t1.72-7-1.68-7-4.38-4.06q1.94-1.32 4.16-1.4t4.15 1.09Q31 8.75 31 15.53t-5 11.34z',
    'optin-monster' => [
        'p' => 'M35.81 26.31q.13.44.07.7.06 1.43-1.53 2.71t-3.04 1.34q-1.87.13-2.68-1.43-11.57.37-21.25 0-.88 1.5-2.7 1.43-1.43-.06-3.02-1.34T.06 27l.13-.69q-.25-.43-.19-.65t.56-.03q.25-.44.7-.88-.26-1.06.68-.56.44-.31 1-.56 0-.7.4-.72t.66.34q1-.19 1.69.38v-1.25q-.44 0-1.03-.25t-1.16-.79-.69-1.21q-.5-2.13 1.57-3 .06-.25.34-.57t.6-.44q.62-.18.87.63 1.06 0 1.31.5.2 0 .53-.03t.54-.03q-.25-.2-.5-.32-.44-.68-1.57-.75V16q-2.5-.06-3.44-1.13-1-1.12-.75-3.34t1.57-3.16q.37-.25.84-.06t.53.69l.03.4q.03.29.07.32t.06.28.06.22.13.15.19.13.28.1.37.02.44 0q.81-.75 1.69-1.25-.7-.06-2.75.25 1.75-1.75 3.56-2.68 2.44-1.2 4.75-1.44-.81-.31-3.75-1 10.06-2.13 15.31 2.3 1.63 1.44 2.63 3.75.5.06.84.06h.6l.4-.09.25-.12.16-.29.1-.3.06-.38.03-.44q.06-.44.53-.66t.9.1q1.2.87 1.5 3 .32 2.25-.75 3.43-.93 1.07-3.37 1.13v.19q-1.19 0-1.63.69-.25.12-.5.3.2 0 .53.07t.53.06q.2-.56 1.32-.56.25-.75.87-.62.32.06.6.37t.34.63q1.31.56 1.56 1.62.13.69 0 1.38-.25 1.06-1.18 1.65t-1.7.6v1.25q.7-.5 1.7-.38.25-.37.65-.34t.4.72q.57.25 1 .56.5-.25.66-.06t.03.62q.5.5.7.88.5-.2.56.03t-.2.65zM31.2 10.44q1.37.56 1.87 2.25.07-.63-.15-1.66T32.2 9.5q-.19-.19-.4-.25t-.35.03-.13.28q0 .5-.12.88zm-.31.43q-.25.25-.88.38 1.31 1.5.81 3.94 1.13-.25 1.63-.75.19-.25.31-.5-.12-2.5-1.87-3.07zm-1.5.44h-.75q.75 1.94.93 4 .32 0 .7-.06.3-1.13.02-2.25t-.9-1.69zM3 12.7q.5-1.69 1.88-2.25-.2-.44-.2-.88 0-.31-.24-.34t-.63.28q-.44.5-.69 1.53T3 12.7zm.56 1.75q.5.5 1.63.75-.44-2.44.81-3.94-.62-.13-.87-.38-1.7.57-1.88 3.07l.31.5zm3 .87q.25-2.12 1.25-3.94-.31.07-.93.32l-.75.31v-.06q-.82 1.56-.38 3.31.5.06.81.06zm11.5-10.19q-1.31 0-2.4.66t-1.72 1.75-.63 2.34q0 2 1.4 3.41t3.38 1.4 3.35-1.4 1.37-3.4q0-1.94-1.37-3.35t-3.38-1.4zm7.25 10.94q-.37-.06-2.5-.06.25 1.13.38 2.63.12 1.06.37 1.06.13 0 .32-.06.68-.38 1.03-1.47t.4-2.1zm-3.37-.12h-3.57q-.06.44 0 1.44t.25 1.12q.5.25 1.6.25t1.6-.19q.18-.31.18-1.22t-.06-1.4zm-4.57 3.37q.2-.37.2-1.56t-.07-1.81h-1.16l-1.34.03-1.13.03q-.12.62-.15 1.75T14 19.2q.56.25 1.69.28t1.69-.16zm-6.62-3.18q-.31 1 .06 2t1.32 1.43q.5.25.62-.31.06-.25.13-1.5t.18-1.75l-2.3.13zM8.5 18.06q.31.5-.5.82 4.69 4.93 9.75 6.62 2.94-.81 5.28-2.44T28 18.81q-.63-.25-.5-.68-.25-.07-.53-.1t-.6-.06-.5-.03Q25.25 20 24 20.37q-.69.25-1.06-.5-.13-.18-.32-1-.18.44-1.5.57-1.8.12-2.75-.19-.12.38-.37.56t-.47.22-.84.1q-3.07.3-3.44-.7-.07 1.5-1.57.7-.87-.44-1.25-1.13-.3-.5-.43-1.12zM7.19 28.88q.06-.25.06-.5-.19-.82-.5-1.5T6 25.66 5.13 25q-1.13-.38-2.7.81T1 28.38q.31.43.75.87t.94.72 1.03.47.97.25q.87 0 1.6-.5t.9-1.31zm-.57-4.13Q7.56 26.44 7.7 28q3.31-1.94 6.62-2.94-2.69-1.44-6.12-4.81-.07.06-.44.19-.06.12.19.34t.12.47q-.19.38-.94.56-.18.2-.5.38-.06 1.44 0 2.56zm21.13 4-1.44-.78-1.37-.72q-.74-.38-1.44-.69t-1.47-.6-1.53-.52q-1.25.56-2.63 1-.12.06-.25 0-1.06-.32-2.18-.82-3.57 1-7.07 3.13 8.94.37 19.38 0zm.63-.75q.12-1.69 1.06-3.25h-.07q.07-1.31.07-2.56-.25-.13-.5-.44-.69-.13-.94-.5-.13-.38.38-.69-.07-.06-.07-.12-.25-.07-.44-.2-3.43 3.2-6.5 4.76 3.44 1.06 7 3zm6.62.38q.13-1.38-1.44-2.57T30.87 25q-.62.19-1.22 1.19t-.9 2.19q0 1.06.78 1.68t1.78.63q.88-.07 2-.72t1.69-1.6zM23.31 4.68q-1.68-1.8-3.75-2.21t-4.12.78q.75-1.75 2.62-2.22t3.5.53 1.75 3.13zM4.7 25.45q.69-.07 1.06.65t.06 1.28q-.12.32-.5.66t-.87.66-.81.37q-.82.25-1.44-.06t-.63-.88q0-.75 1.16-1.68t1.97-1zm13.5-11q-1.5 0-2.5-1.1T15 10.7q.63 1 1.78.69T17.94 10t-1.07-1.37q1.57-.63 3.07.25t1.5 2.5q0 1.3-.97 2.18t-2.28.88zM4.25 29.63q0-.25.34-.6t.79-.62.68-.35q.38-.06.57.38t.06.75q-.38.75-1.06 1.06-.44.12-.91-.1t-.47-.53zm27.06-4.2q.82.07 1.97 1t1.16 1.7q0 .56-.63.87t-1.44.06q-.56-.12-1.28-.68t-.97-1q-.25-.57.13-1.29t1.06-.65zm-1.37 2.63q.44.07 1.15.63t.66.94q0 .3-.47.53t-.84.1q-.82-.32-1.13-1.07-.12-.32.06-.75t.57-.38z',
        'w' => 1.125,
        'vw' => 36
    ],
    'orcid' => 'M18.42 11.76h-2.87v9.62h2.97c4.23 0 5.2-3.21 5.2-4.81 0-2.6-1.67-4.8-5.3-4.8ZM16 .5a15.5 15.5 0 1 0 0 31 15.5 15.5 0 0 0 0-31Zm-5.05 22.55H9.09V10.08h1.86ZM10.02 8.6a1.22 1.22 0 1 1 0-2.44 1.22 1.22 0 0 1 0 2.44Zm8.73 14.46h-5.06V10.08h5.04c4.79 0 6.9 3.42 6.9 6.49 0 3.33-2.6 6.5-6.88 6.5Zm0 0',
    'osi' => 'M0 16.19Q.13 10 3.94 5.72T13.38.69q4.18-.57 7.96.97t6.25 4.62 3.16 6.97q1 5.5-1.69 10.28t-7.81 6.85q-.81.37-1.13-.5-.43-1.2-1.06-2.75T17.84 24t-1.03-2.75q-.31-.75.44-1.06 2.31-1.06 2.69-3.5.31-1.94-.9-3.47t-3.1-1.66q-1.81-.18-3.22.9t-1.66 2.79q-.25 1.56.47 2.9t2.16 1.98q.81.37.5 1.12-2.25 5.88-3.38 8.75-.25.69-.93.44-5.5-2.13-8.2-7.44-.3-.56-.52-1.12t-.38-1.04-.28-1-.19-.87-.12-.84-.1-.72-.06-.7T0 16.2zm1.31-.13v.44l.03.5.03.44Q1.7 21 3.79 24.06t5.78 4.82q.25.12.38-.13 1.37-3.62 2.81-7.31.06-.25-.19-.44-3-1.87-2.81-5.37.06-2 1.47-3.54t3.4-1.8 3.82.71q1.5.88 2.28 2.4t.53 3.29q-.31 2.81-2.81 4.31-.25.19-.2.38l2.88 7.37q.07.25.32.13 3.43-1.57 5.5-4.5 3.3-4.5 2.68-10-.68-5.75-5.25-9.47T13.62 2q-3.3.37-6.09 2.24T3.06 9.22t-1.75 6.84z',
    'padlet' => [
        'p' => 'M297.9 0L298 .001C305.6 .1078 312.4 4.72 315.5 11.78L447.5 320.3L447.8 320.2L448 320.6L445.2 330.6L402.3 488.6C398.6 504.8 382.6 514.9 366.5 511.2L298.1 495.6L229.6 511.2C213.5 514.9 197.5 504.8 193.8 488.6L150.9 330.6L148.2 320.6L148.3 320.2L280.4 11.78C283.4 4.797 290.3 .1837 297.9 .0006L297.9 0zM160.1 322.1L291.1 361.2L298 483.7L305.9 362.2L436.5 322.9L436.7 322.8L305.7 347.9L297.1 27.72L291.9 347.9L160.1 322.1zM426 222.6L520.4 181.6H594.2L437.2 429.2L468.8 320.2L426 222.6zM597.5 181.4L638.9 257.6C642.9 265.1 635 273.5 627.3 269.8L579.7 247.1L597.5 181.4zM127.3 318.5L158.7 430L1.61 154.5C-4.292 144.1 7.128 132.5 17.55 138.3L169.4 222.5L127.3 318.5z',
        'vw' => 640,
        'vh' => 512
    ],
    'page4' => 'M15.5 31.5q-6.44 0-10.97-4.53T0 16 4.53 5.03 15.5.5q1.94 0 3.81.5L2.63 24.5H15.5v7zm0-9H6.19L15.5 9.19V22.5zm6 2h6.94q-2.57 3.94-6.94 5.81V24.5zm3.56-8.63v1l-1.31-.5zm-1.25 3.38q.5 0 .5.69v.62h-1v-.5q0-.81.5-.81zM31 16q0 3.44-1.44 6.5H21.5V1.69q2.06.87 3.81 2.34t3.03 3.28 1.97 4.07T31 16zm-8.5-7h4.31V6h-.87v2.06h-.88V6.2h-.87v1.87h-.82V6h-.87v3zm4.25 11.56h-1.63v-.68q0-1.57-1.3-1.57t-1.32 1.63v1.56h4.31v-.94h-.06zm0-6.43L22.5 16v.75l4.25 1.88v-1.07l-.88-.37v-1.63l.88-.37v-1.07zM26.5 9.8h-2.25v1.7h.88v-.76h.68q.44.75.1 1.47t-1.28.72q-1 0-1.28-.88t.34-1.5l-.75-.62q-.5.56-.56 1.5t.53 1.72 1.72.78q1.56 0 2.09-1.38T26.5 9.8z',
    'pagelines' => [
        'p' => 'M24 19.56q-.94 2.38-2.5 3.6t-3.1 1.28-2.93-.25-2.28-.82l-.88-.43q-1.94 3.87-4.97 6.12t-6.6 2.25q-.74 0-.74-.75t.75-.75q3 0 5.53-1.84t4.34-5.03q-1.18.43-2.34.56t-2.63-.16-2.84-1.68T.5 17.8q1.94-.81 3.65-1t2.88.13 2.12.97 1.5 1.3.97 1.42q.94-2.25 1.38-5-.13.06-.32.09t-.78.03-1.18-.03-1.41-.16-1.53-.37-1.44-.75-1.28-1.13-.94-1.6-.5-2.21q2.25-.81 3.88-.63t2.72 1.07 1.75 1.84.9 1.78l.31.75q.2-2.12.2-3.31l-.7-.53q-.44-.34-1.34-1.47T9.97 6.75t-.32-2.81T11 .62q2.19.82 3.37 2.13t1.35 2.66 0 2.59-.54 2l-.3.81v2.07q0-.13.12-.32t.53-.72.87-1 1.22-.97 1.56-.75 1.91-.25 2.28.44q-.06 2.32-1 3.88t-2.25 2.1-2.6.71-2.15.06l-.87-.12q-.44 2.56-1.25 4.87.18-.25.6-.62t1.58-1.1 2.38-1.06 2.9-.03T24 19.56z',
        'w' => 0.75,
        'vw' => 24
    ],
    'palfed' => [
        'p' => 'M24.06 12.13q0 2.37-1.68 4.18t-3.5 1.82q-.88 0-1-.38.06-2.94.25-7.5 1.37-.5 2.65-.6t2.28.5 1 1.98zm-12.19 4.5v-.25q0-1.13 1.2-2.5t2.43-2.26q-.37 11.82-.06 15.13.06.94.53 1.4t.94.41.78-.44q.37-.37.37-1.18-.18-2.5-.18-6.25 2.87.68 5.8-2.16t2.95-6.47q0-3.25-2.47-4.4t-5.97-.1q-.13-.69-.72-.9t-1.16.21-.62 1.75q-2.75 1.44-4.53 3.7T9.38 16.5q0 .81.3 1.4t.85.98 1.16.37q1.5 0 1.87-1.31.38-1.2-.68-1.44-.13-.06-.25-.06-.25 0-.75.18zM.5 11.3q0-.68.25-1.15t.6-.7.68-.34.6-.12h2.06q.25-.94.75-1.66t1.19-1.18T8.1 5.3t1.5-.53 1.3-.28.98-.12l.37-.07H14V3.12q0-.5.35-.78T15 2h5.7q.5 0 .83.28t.41.6l.06.24v1.2h1.75q6.13 0 7.44 4.68h1.94q1.06 0 1.66.56t.65 1.2l.06.55q0 1.07-.59 1.66t-1.22.66h-2.44l-1.43 13.62q0 1.19-.7 1.88t-1.37.8L27 30H9q-.81 0-1.37-.28t-.85-.69-.43-.81-.16-.72v-.25L4.75 13.63H2.88q-1.07 0-1.66-.57t-.66-1.19z',
        'w' => 1.123,
        'vw' => 36
    ],
    'patreon' => 'M32 12.19q0 4.75-3.38 8.12T20.5 23.7t-8.13-3.38T9 12.2t3.38-8.16T20.5.63t8.13 3.4T32 12.2zM0 31.37V.63h5.63v30.75H0z',
    'paypal' => [
        'p' => 'M6.94 18.5q0 .25-.13.9t-.25 1.44-.25 1.7-.25 1.71-.25 1.53-.18 1.1q-.07.12-.2.12H.76q-.31 0-.56-.25t-.2-.62l3.7-23.2q.06-.5.4-.77t.85-.28q1.75 0 4-.04t3.03-.03 1.9.03 1.6.13 1.03.22 1.19.37q2.62 1.07 3.25 3.53t-.5 5.22q-1.94 5.57-8.75 5.7-.5 0-1.47-.04t-1.32 0-.87.16-.75.47-.35.9zm15.37-9q2.25 1.69 1.5 5.81-1.37 7-8.94 7h-.8l-.26.03-.19.1-.18.15-.16.22-.16.32-.06.43q-.87 5.44-.94 5.7-.19.93-1.06.93h-4q-.31 0-.5-.25t-.13-.56q.25-1.88 1.7-10.63.06-.56.68-.56h2.4l1.35-.06 1.5-.2 1.44-.34 1.5-.56 1.34-.81 1.28-1.13 1.03-1.5.91-1.9q.38-1.13.56-2.13.07-.19.2-.06z',
        'w' => 0.75,
        'vw' => 24
    ],
    'penny-arcade' => [
        'p' => 'm26.38 10.25 1.56 3.44-2.5.69q.87-3.88.94-4.13zM13.5 13.44q1.31-.31 1.69.75.37 1.25-.94 1.94-.38.18-.81.3-.63.2-2.13.57L11 14.13q1.44-.38 2.5-.7zm20.38-1.63Q40 14.88 40 18.38q.06 2.43-3.4 4.18t-9.07 2.1T15 23.63q0 .62-.34 1.12t-.91.69q-7.44 1.94-7.69 1.94-.69 0-1.19-.44t-.56-1.13l-.25-2.75v-.18q0-.5.31-.97t.82-.66l-.13-1.12Q0 17.3 0 14.19q0-2.25 2.69-3.84t7.1-2.2T19.3 8q.32-.81 1.19-1.06 4.94-1.32 8.5-2.25.63-.19 1.19.1t.87.84zm-17.94 6.82q.81-.2 1.75-.79t1.44-1.28q.5-.81.62-1.93t-.12-2.16-1.1-1.75-2.1-.72h-.06q-.68 0-6.18 1.5-3.13.81-5.38 1.44l.25 2.75 1.32-.31.68 7.18-1.25.32.25 2.75 7.2-1.88-.26-2.81-1.31.37-.13-1.5 3-.81zm12.43 1.06 6.82-1.82-.25-2.8-1.31.37-4.2-9.07q-3.8 1-8.43 2.25l.25 2.75 1.06-.25L20.25 19l-1.31.37.25 2.75 7.06-1.87L26 17.8l-1.19.31.25-1.12 3.63-.94.37.88-.94.25z',
        'w' => 1.248,
        'vw' => 40
    ],
    'perbyte' => [
        'p' => 'M19.08 17.79h-3.67v6.17h3.67c1.02 0 1.81-.3 2.39-.87.57-.57.86-1.32.86-2.25 0-.91-.3-1.65-.88-2.21a3.28 3.28 0 0 0-2.37-.84ZM9.34 8.04H5.67v6.17h3.67c1.02 0 1.81-.28 2.39-.86.57-.57.86-1.32.86-2.25 0-.91-.3-1.65-.88-2.21a3.28 3.28 0 0 0-2.37-.85ZM22.91 2H5.1A5.1 5.1 0 0 0 0 7.09V24.9A5.1 5.1 0 0 0 5.09 30H22.9a5.1 5.1 0 0 0 5.1-5.09V7.1A5.1 5.1 0 0 0 22.91 2Zm3.98 22.91c0 2.2-1.78 3.98-3.98 3.98H5.1a3.99 3.99 0 0 1-3.98-3.98V7.1c0-2.2 1.78-3.98 3.98-3.98h17.8c2.2 0 3.98 1.78 3.98 3.98ZM19.1 8.04h-3.7v6.17h3.67c1.02 0 1.81-.28 2.39-.86.57-.57.86-1.32.86-2.25 0-.91-.3-1.65-.88-2.21a3.28 3.28 0 0 0-2.37-.85Zm0 0',
        'vw' => 28
    ],
    'periscope' => 'M23.13 4q3.75 3.94 3.75 9.38 0 5.18-5.13 12.06-.44.56-1.1 1.34t-1.87 2.13-2.47 2.22-2.12.87q-1.5 0-3.7-1.97t-4.18-4.6q-5.19-6.8-5.19-12.05 0-5.5 3.85-9.44T14.19 0q5.18 0 8.93 4zm-8.94 26.88q.87 0 2.78-1.79t3.84-4.3 3.4-5.73 1.48-5.69q0-5-3.4-8.62t-8.1-3.63q-5 0-8.44 3.63t-3.44 8.62q0 2.5 1.47 5.7t3.44 5.71 3.97 4.31 3 1.79zm6.94-17.94q0 2.75-2.07 4.78t-4.87 2.03q-2.25 0-4.03-1.13t-2.6-2.78-.65-3.68 1.47-3.72q0 1.12.84 1.97t2 .84 1.97-.84.81-1.97q0-1.82-1.69-2.57 2.32-.56 4.38.28t3.25 2.7 1.19 4.09z',
    'phabricator' => 'm20.19 16.38 1.31 1.3-.5 1.26h-1.87l-.57.56v2l-1.25.5-1.37-1.5h-.82L13.87 22l-1.25-.5-.12-2.06-.56-.57q-1.82.07-1.88-.06l-.56-1.25 1.31-1.31v-.81L9.5 14.12l.5-1.25h1.87l.57-.56V10.5l1.25-.5L15 11.3h.88l1.23-1.3 1.25.56.13 1.81.56.57q1.82 0 1.82.06l.62 1.25-1.31 1.31v.82zm-2.82-.57q0-.75-.62-1.31t-1.44-.53-1.34.62-.5 1.41.62 1.34 1.4.54q.57-.07 1-.35t.67-.72.21-1zm12.57-2.31Q31 14.5 31 16t-1.06 2.5l-5.94 6Q20.5 28 15.56 28t-8.44-3.44v.07Q4.56 22 1.06 18.5 0 17.5 0 16t1.06-2.5Q5 9.56 7 7.5 10.5 4 15.44 4t8.44 3.44v-.06zM8.78 22.72q2.78 2.78 6.72 2.81t6.75-2.78T25.06 16t-2.81-6.72T15.5 6.5 8.78 9.28 6 16t2.78 6.72z',
    'phoenix-framework' => [
        'p' => 'M13.31 21.5q-2.12.06-4.65-1.66T5.3 14.88Q5 13.62 4.97 12T4.8 9.5q-.62-4.25-3.5-4.81-.56-.07-1.12 0H0l.13-.13Q1.8 3.31 4.16 2.75t4.46-.38q2.2.2 3.94 1.25t3.03 2.54 2.5 2.93 2.85 2.63 3.5 1.47q.25.06.65.1t.54.02h.12q-.13.13-.9.47t-1.35.4q-1.81.38-3.25-.62-.19-.12-1.03-.84t-1.4-.85q-.76-.25-1.44.04t-.79 1.18.79 2.28 2.43 2.38q1.82 1.06 4.38 1.31t4.06-.37l.06-.07h.07q-1.13 1.38-3.1 2.22t-3.65.66q-1.38-.19-2.63-1.38t-2.44-2.68-2.06-2.07-2-.25-1.25 1.44q-.06.75.38 1.69.3.62.71 1.12t.79.82.78.53.75.34.6.19.43.1l.19.02q-.32.13-1.57.13zM4.7 4.06q.62.63 1.4.47t.85-.9q-1.25.06-2.31.3.06.07.06.13zM17 25.94q-1.94.06-4.06-1.03T9.3 22.13q-.87-.94-1.19-1.44v-.06q2.13 1.3 3.44 1.68 2.32.7 4.5-.18.5-.2 1-.25t.85.03.56.18.28.22q-1.25.25-1.5.5-.63.57.44 1.07.5.25 2.69.4t3.37 1.16q.44.37.63.69-1.7-.5-2.5-.57-.88-.06-2.47.16t-2.4.22zm10.94-15.75q2.62-.82 4.19-.19 1.37.56 2 2.19.3.81.3 1.75-.05 0-.05.06v-.12q-.32-1.2-1.47-1.82T30.13 12q-1.7.56-3.6.56t-4-.9T19.06 9q-.12-.12-.06-.19 0-.06.19.07 1.62 1.25 3.75 1.68 2.44.44 5-.37zm6.44 7.37q-.7 0-1.29.5T32 19.22t-1.56 1.34-2.57 1q-1.62.38-2.8.32v-.07l.3-.1.72-.3.85-.66q.44-.37 1.44-1.37.8-.88 1.34-1.32t1.56-.84 2.22-.28q1.19.06 2.25.75v.06q-.56-.19-1.37-.19zm-.82-1.5q-1.25-.25-2.31-.25t-1.94.5q-.06.07-.5.32t-.56.34-.53.28-.69.25-.72.19-.87.19q-2.5.31-4.13-.5l-.06-.07q1.06.2 1.84.13t1.6-.5q.25-.13.8-.47t.85-.5.78-.4 1-.38 1-.19q2.57-.37 4.44 1.06zm-16.75-10-.18-.12q2.06 0 3.37.5.44.19.88.47t.71.5.75.56.79.53q.93.63 2.25.5h.28l.3-.03.23-.03h.12L26.2 9q-.75.38-1.63.44-2.25.25-4.56-.94-1.75-.94-3.19-2.44zM40 21q-.25.19-.94.34t-1.75-.03q-.31-.12-.62-.34t-.7-.6-.62-.56q-.62-.44-1.75-.56l.07-.06q1.43-.88 2.94-.38.18.06.37.16t.31.19.28.25l.91.9q.25.19.53.35t.44.18.34.07l.2.03V21zm-13.25 8.56q.06.07.06.13-1.81-.19-2.93-1.44-1.2-1.38-3-1.81l.06-.07q1.5.07 2.44.25l.59.25q.29.12.53.29t.4.3.35.41.28.35.25.34.16.32q.31.37.81.68z',
        'w' => 1.25,
        'vw' => 40
    ],
    'phoenix-squadron' => 'M6 3.94Q10.63.38 16.31.5q2.7 0 5.44.94-2.5-.44-4.56-.13-4.13.5-7 3.7.19 0 1-.38 1.81-.75 3.56-.88 1.44-.19 3.75.13-2.25.37-2.75.56-3.37 1.12-5.37 3.94-1.44 2.06-1.57 4.18 0 1.94 1.25 2.76 1.13.68 2.38.43 1.25-.18 2.69-1 1.3-.75 2.06-1.5.12-.18.12-.81-.3-.06-.37 0-1.06.56-2.25.94-2.25.69-3.88.06 1.32.19 2.32-.31.5-.25.97-.56t.8-.57.88-.69.78-.62q2.7-2.06 5.57-3.5 3.06-1.38 5.12-1.5L24.31 8.3q-1 1-2.5 2.82-.06.12-.87.62-2.13 1.5-3.25 3.75-.94 2.13-.75 4 .25 1.57 1.31 2.19 1.07.69 2.75.31 3.07-.68 5.07-3.5 2-2.62 1.87-6-.06-.06-.1-.6t-.02-.83q.87 2.25 1.06 3.8.12 1.13-.13 2.7-.06.37-.25 1.06t-.25 1q1.94-2.19 2.5-5.13.5-3.37-.93-6.5 1.93 2.88 2.25 6.7.06 1.24 0 2.43-.25 3.25-1.88 6.13-2.5 4.56-7.31 6.74-3.69 1.7-7.9 1.47t-7.66-2.4q-3.75-2.31-5.75-6.19Q.19 20.13 0 17.07v-1.94q.2-3.25 1.78-6.22T6 3.94zm13.94 5q.37-.06 1-.25.56-.19.62-.69-1.18.63-1.62.94z',
    'php' => [
        'p' => 'M20 6.5q-7.88 0-13.4 2.84T1.05 16t5.53 6.66T20 25.5t13.4-2.84T38.95 16 33.4 9.34 20 6.5zm0-1q5.44 0 10.03 1.4t7.28 3.82T40 16t-2.69 5.28-7.28 3.81T20 26.5 9.97 25.1t-7.28-3.82T0 16q0-1.44.72-2.81t2-2.5 3.12-2.13 4.07-1.65 4.78-1.04T20 5.5zm-6.38 9.63q.07-.32.07-.6v-.5l-.16-.37-.25-.28-.34-.2-.4-.09-.54-.06-.56-.03h-1.32l-.87 4.44h1.4l.6-.03.6-.1.46-.15.47-.29.35-.4.3-.56.2-.78zm-7.56 6.74 2.07-10.5h4.43q2 0 2.88 1t.62 2.7-1.5 2.8q-.3.32-.62.54t-.72.34-.66.19-.84.1-.78.06-1 .03h-1l-.56 2.75H6.06zM17.7 8.57H20l-.56 2.8.81-.02.84-.04q.28-.02.72.03t.72.07.6.12.56.2.44.27q.12.19.25.32t.15.34.07.4 0 .57-.07.65-.15.88-.22 1.06-.25 1.32-.35 1.53h-2.3q.18-.94.5-2.38.43-2.12.46-2.6t-.15-.77q-.07-.07-.13-.1t-.16-.06-.21-.06-.29-.03h-2.15l-1.19 6h-2.31zm13.87 6.56q.07-.32.07-.6t-.04-.5-.15-.37-.22-.28-.34-.2-.41-.09-.53-.06-.56-.03h-1.32l-.87 4.44h1.53l.62-.07.63-.12.5-.25.5-.4.34-.6.25-.88zM24 21.87l2.06-10.5h4.44q2 0 2.88 1t.59 2.7-1.47 2.8q-.25.26-.56.44t-.53.35-.63.22-.6.12-.74.07-.75.03-.9.03h-.91l-.57 2.75H24z',
        'w' => 1.25,
        'vw' => 40
    ],
    'pied-piper' => 'm2 26.19-2 3.75.06-20.5q0-3.07 2.2-5.25T7.5 2h20.44q-4.31 1.38-8.56 4.19t-7.32 6.37Q6.5 18.94 2 26.2zM28 2v20.56q0 3.07-2.19 5.25t-5.25 2.2H0q.81-1.07 2.4-2.26t2.91-1.87q.69-.32 2.16-.94t2.56-1.22 1.9-1.35q1.13-1.12 1.88-2.75t1.4-3.87 1.16-3.31q.5-1.25 1.47-2.53t2.03-2.29 2.22-1.9 2.19-1.56T26.16 3t1.34-.75z',
    'pied-piper-alt' => [
        'p' => 'm15.25 15.38-1.75.37-.06-.31q.18 0 .56-.13t.63-.12q.3 0 .62.18zm8.5 12.37q.19 0 .4.28t.5.78.48.88.4.9.28.6q-3.19 1.18-6.62.62L18.75 29q.69-.5 2.37-.88t2.63-.37zm-2.44-16.69q-.18-.25-2.47-.15t-3.72.78q.5-.94 1.63-1.38.63-.19 1.19-.62t.81-.69.88-.97.8-.9q.26.18.54.59t.6.66.8.18v1.38h.44q4.44-1.69 6.63-3.38.5-.43.9-.75t.79-.65.6-.57.52-.62.44-.53.44-.66.4-.69.47-.84.57-.94l.12-.19q-.06 1-.81 2.5-2.75 5.75-9.19 7.32-.19 0-.44.06t-.5.1l-.5.06-.47.12-.4.22q.06.31.43.78t.38.6q0 .25-.44.56-1.12-1.25-1.44-1.38zM34.75 0l-.06.06zM31 23.88q.63 1 1.44 1.96t1.34 1.44 1.78 1.6q-.87.56-1.97 1.06t-2.47 1.03-2 .84q-1.12-.93-1.78-3.43t-.93-4.85-1.35-4.22-2.94-1.87q-.18.06-.3.12.12.07.24.25.13.13.4.35t.41.31.32.28l.28.28.22.28.18.32.16.34.12.44.07.47.03.62v.56q.06.82-.63 4-.43-.06-2.4.22t-2.47-.03v-1.22l.06-1.4q0-.19 1.03-1.22t1.03-1.47q-.43-.06-.8.44-.63.94-2 1.78t-2.45.84q-.37 0-1.03-.75t-1.28-1.62-1.34-1.82-1.1-1.18l-.5-.57q-8.56 2-9.25 2-.25 0-.47-.18t-.22-.5q0-.63.57-.75l8.75-1.88q-.13-.31-.19-.5t.19-.34.4-.22.5-.13.41-.06.28-.13.28-.12q.56 0 .88.87 3-.62 3.12-.62.69 0 .75.63t-.56.75l-3.31.75v.25q0 .5 1.46 2.12t1.85 1.63q.87 0 2.15-1.44t1.29-2.32q0-.62-.41-.93t-.9-.35-.91-.19-.4-.59q0-.31.12-.62l-1.2-.38q1-1 .7-3.12 1.43-.32 2.56-.32t1.69.07.97.37.5.53.4 1.06.57 1.41l.87-.44q-.06.88-.62 1.19t-1.07.38-.5.37v.38q1.38-.07 2.6.34t2.12 1.03 1.81 1.72 1.5 1.94T31 23.87zm-12.25 2.87h.56l.2 1.5-.76.19v-1.69zm-4.31-9.38.87-.25 1.94 3.2-1.25.87zm5.31 9.7 1.13-.2q.69-.12 1.25-.18t1.3-.07l.07.32q0 .12-.31.25t-.82.22-.96.18-.97.16-.57.06z',
        'w' => 1.125,
        'vw' => 36
    ],
    'pied-piper-hat' => [
        'p' => 'M40 1.56q-1.44.94-2.5 1.85t-1.69 1.68-.97 1.32-.6 1.06-.24.6q-.19.3-.63 2.37t-.87 2.88q-.31.62-1.1 1.24t-1.9 1.25l-1.13.63q-.8.56-2.62 4-1.25-.06-2.44.06t-2.06.22-2.1.56-1.8.66-1.98.94-1.75.97-2.03 1.22-2.03 1.25l2.69-1.07q.87-.3 2.69-1.06t2.59-1.03 2.37-.72 2.82-.53 3.1-.06 4.05.34l.13.06q.44.25.19.63l-3.32 6q-.3.5-.87.37-2.56-.56-5.63-.43t-5.12.47-4.72.71-4.4.44q-2.07 0-3.6-.53T.5 28.88t-.5-.63.13-.12q2.37 0 6.43-.94 3.32-6.37 6.5-9.12t6.63-2.75q.19 0 .5.03t1.15.25 1.53.56 1.44 1.16 1.07 1.87l1.75-3.06.53-1.03 1.1-2.1 1.62-2.68 2.18-3q1.44-1.88 3-3.13t2.38-1.69T40 1.57z',
        'w' => 1.25,
        'vw' => 40
    ],
    'pied-piper-pp' => 'M12.81 10.9q0 .98-.56 1.7t-1.38.71q-.68 0-1.12-.25V8.81q.44-.25 1.13-.25.8 0 1.37.7t.56 1.65zm3.31 4.23q.82 0 1.41.68t.6 1.66-.6 1.69-1.4.72q-.7 0-1.07-.25v-4.25q.44-.25 1.06-.25zM28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22q1.25 0 2.13.88T28 5zM11.56 15.94q1.94 0 3.28-1.47T16.2 11q0-1.38-.63-2.53t-1.69-1.81-2.3-.66q-1.13 0-2.2.56h-2.8v12.88l3.18-.63v-3.19q.87.32 1.81.32zm9.88 1.56q0-2.06-1.35-3.5t-3.21-1.44q-.38 0-.63.06-.38 1.2-1.19 2.07-1.37 1.44-3.25 1.56V26l3.25-.62v-3.2q.75.32 1.75.32 1.94 0 3.28-1.47t1.35-3.53z',
    'pied-piper-square' => [
        'p' => 'm2 26.19-2 3.76.05-20.5A7.44 7.44 0 0 1 7.5 2h20.45a36.47 36.47 0 0 0-15.87 10.54A96.87 96.87 0 0 0 2 26.19M28 2s-8.93 4.18-11.65 10.45c-1.45 3.34-1.8 7.37-4.44 9.9-1.8 1.74-4.36 2.4-6.58 3.52-1.45.75-4.15 2.54-5.3 4.13h20.52a7.45 7.45 0 0 0 7.46-7.45L28 2Zm0 0',
        'vw' => 28
    ],
    'pinterest' => 'M31 16q0 6.44-4.53 10.97T15.5 31.5q-2.38 0-4.56-.69 1.5-2.5 1.87-4.06l1-3.69q.38.75 1.38 1.25t2.19.5q3.43 0 5.71-2.78t2.29-6.84q0-3.82-2.75-6.38T15.8 6.25q-4.75 0-7.47 2.75t-2.72 6.62q0 1.82.88 3.6t2.25 2.4q.44.2.5-.18.06-.13.22-.85t.22-.9q.06-.25-.13-.44-1.12-1.44-1.12-3.57 0-2.68 1.9-4.71t5.1-2.04q2.87 0 4.69 1.79t1.8 4.53q0 2.06-.65 3.68t-1.78 2.54-2.44.9q-1.18 0-1.87-.84t-.44-1.97q.13-.63.5-1.75t.56-1.97.2-1.47q0-.56-.2-1.06t-.65-.78-1.1-.29q-1.18 0-2 1.07t-.8 2.69q0 .25.03.53t.06.5.1.43.1.35.09.25.06.19l.06.06L9.88 26q-.32 1.56 0 4.44-4.44-1.7-7.16-5.63T0 16Q0 9.56 4.53 5.03T15.5.5t10.97 4.53T31 16z',
    'pinterest-p' => [
        'p' => 'M12.75.38q4.56 0 7.9 2.87T24 10.69q0 2.25-.6 4.37T21.7 19t-3.03 2.88-4.29 1.06q-1.24 0-2.5-.6t-1.8-1.65l-.54 2.12-.44 1.79-.37 1.34-.5 1.34-.63 1.16-.87 1.44-1.16 1.62q-.19.13-.22.13t-.22-.2q-.25-3-.25-3.55 0-1.7.41-3.85t1.25-5.4.97-3.82q-.63-1.18-.63-3.19 0-3 2.32-4 1.87-.8 3 .57.56.75.56 1.87 0 1.2-.81 3.53t-.82 3.54.85 1.96 2.03.79q1.5 0 2.69-1t1.78-2.57.87-3.1.28-2.9q0-3.25-2.03-5.06t-5.34-1.81Q8.5 3.44 6 5.87t-2.5 6.2q0 1.43.72 2.8t.72 1.44q0 .5-.28 1.35T4 18.5q-1.31 0-2.66-1.94T0 11.63q0-5 3.84-8.13T12.75.37z',
        'w' => 0.75,
        'vw' => 24
    ],
    'pinterest-square' => 'M28 5v22q0 1.25-.88 2.13T25 30H9.62q1.38-2.25 1.76-3.69l.93-3.69q.38.75 1.38 1.25t2.12.5q2.32 0 4.13-1.25t2.84-3.43 1.03-4.82q0-3.81-2.72-6.34T14.31 6q-4.68 0-7.4 2.72t-2.72 6.6q0 1.74.87 3.5t2.25 2.43q.44.19.5-.25 0-.06.2-.75t.24-.94-.12-.44Q7 17.44 7 15.37q0-2.68 1.9-4.68t5.04-2q2.81 0 4.62 1.75t1.82 4.5q0 2-.66 3.62t-1.75 2.5-2.47.88q-1.12 0-1.81-.82t-.44-1.94q.13-.68.47-1.78t.56-1.93.22-1.41q0-1.88-1.56-2.13-1.69-.25-2.63 1.5-.5.94-.5 2.2 0 .37.03.74t.1.66.12.47.13.31l.06.13-.72 3.03q-.4 1.73-.56 2.34t-.38 1.63-.25 1.37-.06 1.06 0 1.22.1 1.35l.18.06H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22q1.25 0 2.13.87T28 5z',
    'pix' => [
        'p' => 'M242.4 292.5C247.8 287.1 257.1 287.1 262.5 292.5L339.5 369.5C353.7 383.7 372.6 391.5 392.6 391.5H407.7L310.6 488.6C280.3 518.1 231.1 518.1 200.8 488.6L103.3 391.2H112.6C132.6 391.2 151.5 383.4 165.7 369.2L242.4 292.5zM262.5 218.9C256.1 224.4 247.9 224.5 242.4 218.9L165.7 142.2C151.5 127.1 132.6 120.2 112.6 120.2H103.3L200.7 22.76C231.1-7.586 280.3-7.586 310.6 22.76L407.8 119.9H392.6C372.6 119.9 353.7 127.7 339.5 141.9L262.5 218.9zM112.6 142.7C126.4 142.7 139.1 148.3 149.7 158.1L226.4 234.8C233.6 241.1 243 245.6 252.5 245.6C261.9 245.6 271.3 241.1 278.5 234.8L355.5 157.8C365.3 148.1 378.8 142.5 392.6 142.5H430.3L488.6 200.8C518.9 231.1 518.9 280.3 488.6 310.6L430.3 368.9H392.6C378.8 368.9 365.3 363.3 355.5 353.5L278.5 276.5C264.6 262.6 240.3 262.6 226.4 276.6L149.7 353.2C139.1 363 126.4 368.6 112.6 368.6H80.78L22.76 310.6C-7.586 280.3-7.586 231.1 22.76 200.8L80.78 142.7H112.6z',
        'vw' => 512,
        'vh' => 512
    ],
    'playstation' => [
        'p' => 'M35.69 23.25q-.19.25-.53.5t-.66.44-.6.31-.46.19l-.19.12-12.81 4.57V26l9.43-3.37q.82-.25.91-.63t-.53-.56q-1.56-.56-3.5.19l-6.31 2.18v-3.5q2.31-.8 4.75-1.06 4.31-.5 8.12.94 1.31.44 2 1t.69 1.1-.31.96zM21.63 17.5V8.81q0-1.94-1.13-2.25-.5-.19-.84.25t-.35 1.25v21.69l-5.87-1.81V2q3.37.62 8.12 2.19 3.69 1.31 5.16 3.18t1.47 5.44q0 3.44-1.9 4.75t-4.66-.06zM2.68 25.63q-2-.57-2.53-1.66t.53-1.78q.5-.38 1.31-.75t1.38-.56l.56-.25 8.37-2.94v3.37l-6.06 2.19q-.81.25-.9.63t.53.56q1.62.56 3.56-.19l2.87-1.06v3.06q-5 .94-9.62-.63z',
        'w' => 1.125,
        'vw' => 36
    ],
    'product-hunt' => 'M20.38 13.66q0 .97-.7 1.65t-1.62.69H13.7v-4.63h4.37q.94 0 1.63.66t.68 1.63zM31.5 16q0 6.44-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5t10.97 4.53T31.5 16zm-8-2.31q0-2.25-1.6-3.85t-3.84-1.59h-7.5v15.5h3.13v-4.63h4.37q2.25 0 3.85-1.59t1.59-3.84z',
    'pushed' => [
        'p' => 'M25.44 7q1.06.06 1.44 1.03t-.38 1.78L11.81 24.25l-2.19 5.06q-.5.82-1.59.7t-1.4-1.13L.06 7.13q-.19-.88.22-1.53t1.1-.79l17-2.8q.87-.13 1.56.59t.19 1.78l-.88 2.06zm-6.82-3.63L6.82 5.31l11.06 1 1.06-2.5q.12-.25 0-.37t-.32-.07zM1.45 6.75 11.3 22l6.07-14.38L1.8 6.2q-.25-.07-.4.12t.03.44zM8.5 28.5l.19-.4.47-1.04.6-1.34.52-1.25q.27-.67.4-.9L2.76 11.24l5.19 17.19q.06.19.25.22t.31-.16zM25.56 8.87q.38-.37-.18-.56l-6.63-.56-5.63 13.31z',
        'w' => 0.843,
        'vw' => 27
    ],
    'px' => [
        'p' => 'M6.44 21.5q-.32-.63-.25-.9t.75-.54q.69-.18.84 0t.56 1.13 1.29 1.94v-5.88q.06-2.31 1.84-4.03t4.28-1.72 4.31 1.78 1.82 4.28q0 3.07-2.47 4.9t-5.6.92q-.5-.2-.22-1.07t.79-.69q1.8 0 2.93-.28t1.88-1.22.75-2.56q0-1.81-1.31-3t-3.07-1.15-2.94 1.4l-.5.66q-.2.28-.34.75t-.22.84-.06 1.22v6.34q2.44 1.5 5.25 1.13t4.81-2.38q2.38-2.37 2.38-5.78t-2.38-5.84q-2.44-2.38-5.87-2.38t-5.82 2.38q-.93.94-1.3 1.5H8.5q-.38.62-1.25.44-.88-.25-.88-.75V1.24q0-.62.63-.62h15.12q.5 0 .5.94t-.5.94h-14v8.31h.07q2.3-2.5 5.46-3.1t5.82.48 4.5 3.6 1.84 5.84q0 2.62-1.15 4.78t-2.94 3.37-4.03 1.63-4.38-.03-4-1.97-2.87-3.9zm4-16.25q0-.31.56-.56t1.47-.47 2.06-.31 2.53.06 2.79.53 2.84 1.15 2.62 1.97q.25.25.07.57t-.5.56-.44.25q-.06 0-.75-.53t-1.85-1.19T19.2 6.1t-3.6-.47-4.28.88q-.37.19-.65-.38t-.22-.87zm14.12 20.69q.38-.38 1 .25t.32 1q-3.07 3.06-7.1 3.87t-7.5-.37-6.1-4.16T2 19.63q-.06-.32.4-.53t.94-.13.47.4q.63 3 2.28 5.32t3.94 3.47 4.85 1.4 5.12-.68 4.56-2.94zm-11.19-6.75q0-.19 1.25-1.44l-1.12-1.13q-.19-.18-.06-.43t.4-.47.4-.22 1.38 1.25l1.13-1.13q.25-.3.75.2t.25.8l-1.13 1.13 1.07 1.06q.31.38-.03.78T17 20q-.13 0-1.38-1.25l-.62.63q-.56.62-.75.62-.25 0-.56-.31t-.32-.5z',
        'w' => 0.873,
        'vw' => 28
    ],
    'python' => 'M27.5 12.5q.94 3.94 0 6.81-.56 1.75-1.28 2.57t-2.1.8h-10v.88h6.7v2.57q0 2.37-3.32 3.37-3.38.94-6.69 0-3.37-1-3.37-3.37v-6.38q0-1.37 1-2.37t2.37-1h6.69q1.63 0 2.88-1.28t1.25-2.97v-3h2.5q2.5 0 3.37 3.37zm-9.6 12.75q-.52 0-.9.38t-.38.9.38.9.88.38q.37 0 .65-.19t.44-.47.15-.65q0-.5-.34-.88t-.87-.37zm-7.4-9.75q-1.13 0-2.1.56T6.89 17.6t-.57 2.03v3.07H4q-1.38 0-2.25-.9T.5 19.3q-.88-3.37 0-6.81.37-1.56 1.5-2.47t2.62-.9h9.2V8.3h-6.7V5.75q0-.63.1-1.06t.31-.85.56-.69.94-.5 1.47-.4q3.19-.57 6.69 0 1.37.25 2.34 1.22t.97 2.28v6.37q0 1.38-.97 2.38t-2.34 1H10.5zm-.44-8.88q.38 0 .66-.18t.44-.47.15-.63-.15-.65-.44-.47-.66-.16q-.5 0-.87.38t-.38.87q0 .25.1.5t.28.4.4.29.47.12z',
    'qq' => 'M27.13 26.28q-.2.03-.57-.31t-.75-.81-.72-.97-.6-.88l-.18-.31q0 1.56-.81 3.28t-2.38 3.03l.97.38q.47.18 1.03.46t.82.66.06.69q-.94.5-4.38.5T14 31.75q-2.2.25-5.6.25T4 31.5q-.38-.63.62-1.19t2.25-1Q5.31 28 4.5 26.28T3.69 23q-.07.06-.13.19t-.28.46-.47.66-.53.69-.53.65-.47.47-.4.13q-.2 0-.32-.6t.1-2.09.84-3.5q.44-1.56 2.5-6.62-.19-5.88 2.34-9.66T14 0q5.06 0 7.62 3.75T24 13.44q2.06 5.06 2.5 6.62.62 2 .84 3.5t.1 2.1-.32.62z',
    'quinscape' => 'M18.88 29.69q.62.62.8.75-2.06.62-4.43.62-6.31 0-10.78-4.47T0 15.8 4.47 4.96 15.28.5 26.1 4.97t4.46 10.84q0 3.19-1.25 6.06l-.68-.93q.5-4.38-2.44-7.72T18.8 9.87q-4.12 0-7.03 2.91t-2.9 7 2.9 7 7.03 2.9h.07zm0-.07zm4.87-10.5q2.56 0 4.38 1.82t1.8 4.37-1.8 4.38-4.38 1.81q-1.25 0-2.4-.5t-2-1.31-1.32-1.97-.47-2.4q0-2.57 1.82-4.38t4.37-1.82z',
    'quora' => [
        'p' => 'M27.5 24.19q.06.56-.1 1.4t-.65 1.91T25 29.28 22 30q-3.69 0-6-3.88-3.63 1-7.25-.25T2.78 21.5.44 14.31Q.44 9 4.19 5.5T12.8 2q4.94 0 8.66 3.47t3.72 8.84q0 5.94-4.63 9.56 1.44 2.25 3.07 2.25.5 0 .9-.18t.63-.5.37-.63.16-.62h1.81zm-8.94-3.63q1.07-2.31 1.07-6.25 0-5-1.66-7.44t-5.16-2.43-5.09 2.43-1.6 7.44q0 4.94 1.6 7.38t5.1 2.44q1.18 0 2.06-.32-.44-.81-.82-1.37t-.9-1.19-1.19-.94T10.5 20q-.69 0-1.19.25l-.81-1.5q1.69-1.44 4.37-1.44 2.07 0 3.35.85t2.34 2.4z',
        'w' => 0.873,
        'vw' => 28
    ],
    'r-project' => [
        'p' => 'M36.31 14.19q0 5.12-5.5 8.69l4.2 7.12h-7l-2.5-4.69q-2.07.57-4.38.82V30h-6.2v-3.87Q8.57 25.38 4.29 22T0 14.2q0-5.06 5.31-8.63T18.13 2q4.93 0 9.12 1.63t6.63 4.43 2.43 6.13zm-29.18.87q0 2.44 2.15 4.47t5.66 3.03v-13h12.78l.81.13 1.13.31 1.18.6 1.13.96.78 1.47.31 2.03q0 1.5-.56 2.66t-1.34 1.72-1.6.9-1.37.41l-.57.06q1.13.38 1.75.7.2.05.44.24 4.38-2.19 4.38-6.69 0-4.12-3.63-6.22t-9.62-2.09q-5.7 0-9.75 2.44t-4.07 5.87zm14.06 2.38q.5 0 1.28.03t1.19.03 1-.03.87-.13.6-.28.43-.53.13-.84q0-.38-.06-.63t-.22-.43-.32-.32-.5-.19-.62-.12-.75-.06h-.84l-1.04.03-1.15.03v3.44zM21.12 22v1.38q1.75 0 3.2-.2Q23.5 22 22.81 22h-1.7z',
        'w' => 1.134,
        'vw' => 36
    ],
    'raspberry-pi' => [
        'p' => 'M23.25 14.5q.94.63 1.6 1.78T25.4 19t-1.28 2.88q-.07.5-.7 2.25-.24 1.8-1.4 2.93T20 28.38q-1.69 1.3-3.25 1.87Q15 32 12.75 32h-.06q-2.25 0-4-1.75-1.56-.56-3.31-1.87-.82-.2-1.97-1.32T2 24.13q-.63-1.75-.69-2.25Q.12 20.55.03 19t.56-2.72 1.6-1.78l.25-.38q0-1.37.44-2.37t1-1.53 1.43-1.1q-.93-.24-1.06-.8-.94-.26-1.06-1-.94-.63-.88-1.2-.68-.5-.5-1.12-.62-.69-.37-1.25-.5-.63-.06-1.31-.32-.5-.04-.94t1.04-.38q.37-.5 1.56-.5.25-.37 1-.25.5-.5 1.44-.18.25-.2.5-.2.12 0 .28.04t.34.12.25.1q.44-.07.56-.07t.35.13T9 .5q.69-.13 1.38.5h.37q1.38.81 2 2.75Q13.31 1.8 14.7 1h.44l.06-.07q.55-.62 1.3-.43 0-.07.29-.2t.37-.12.53.07q.06 0 .31-.1t.38-.12.31 0 .44.15q.87-.31 1.37.2.7-.13 1 .24 1.2 0 1.5.5.82-.06 1.1.38t-.03.93q.44.7-.07 1.32.25.56-.37 1.25.19.62-.5 1.12.06.56-.88 1.19-.12.75-1.06 1-.12.56-1.06.81 1.37.82 2.12 1.9t.75 3.1zm-1.38-.5q.07-1.44-.65-2.5T19.34 10t-2.59-.19q-.75.13 1.94 2.44 2.94 2.44 3.19 1.75zm-2.71 5.84q1.15-.71 1.34-2.18t-.72-2.82-2.34-1.75-2.57.35-1.3 2.22.68 2.78 2.31 1.72 2.6-.32zM14.94 1.81q-.63.63-1.06 1.4t-.38 1.73 1.06 1.37Q17.13 4.25 20.94 3q-4.25 2.19-5.88 3.88.2.68.72 1.15t1.22.63 1.19.22 1 0q-.38-.13-.44-.32.13-.06.75-.12t.94-.25q-.31-.07-.44-.25 1.25-.38 1.5-.63-.37 0-.62-.12 1.25-.63 1.5-.88-.57-.06-.63-.12.69-.44 1.19-.94-.5.06-.63-.06.94-.94 1-1.07-.5.2-.68 0 0-.12.37-.4t.5-.66q-.56.06-.62 0 .12-.5.56-1.12h-.56q-.94 0-.94-.07l.37-.37q-.56-.13-1.62.19-.19-.2.25-.57-.75.07-1.38.32-.06-.13-.06-.16t.19-.22l.19-.19q-.82.13-1.44.57-.19-.2.12-.63-.56.19-1.18.75-.25-.31-.07-.56-.62.31-1 .81-.12-.12-.12-.56-.2.12-1.07 1.06-.12-.06-.18-.5zM12.75 9q-1.19-.06-2.25.66T9.44 11q0 .69.93 1.25t2.44.63q1.5 0 2.4-.5t.92-1.2q.06-.8-1.07-1.5T12.75 9zm-7.63-.81q.32.19.94.25t.75.12q-.06.2-.43.32.68.06 1.4 0t1.6-.57 1.06-1.43Q8.88 5.13 4.63 3 8.43 4.25 11 6.31q.94-.43 1.03-1.37t-.37-1.72-1.03-1.4q-.13.43-.2.5-.87-.95-1.06-1.07 0 .44-.12.56-.38-.5-1-.81.19.25-.06.56Q7.49 1 6.94.81q.37.44.18.63Q6.44 1 5.7.88l.18.18.16.22-.03.16q-.63-.19-1.38-.32.44.38.25.57-1.06-.32-1.68-.19l.37.38q.06.06-1.5.06.5.62.63 1.12-.07.07-.7 0 .2.38.54.66t.4.4q-.25.2-.68 0 .06.13 1 1.07-.2.12-.63.06.5.5 1.19.94-.06.06-.63.12.25.25 1.44.88-.19.12-.62.12.31.25 1.56.63-.12.19-.44.25zm3.75 1.5q-2.18-.38-3.71.81t-1.41 3.44q.06.87 2.94-1.63 2.69-2.3 2.19-2.62zm-6.5 11.69q.57.25 1-1 .94-2.32.57-4.32-.25-.87-.75-.81-1.32.69-1.75 1.97t-.13 2.4 1.07 1.75zm5.7 6.12q.62-.5.4-1.81t-1.03-2.32q-1-1.06-2.03-1.53t-1.78-.03q-.57.32-.57 1.53t.7 2.4q1 1.45 2.4 1.85t1.9-.1zM6.4 20.03q1.15.72 2.59.31t2.34-1.75.72-2.78-1.34-2.19-2.6-.34-2.3 1.75-.73 2.81 1.32 2.2zM12.8 31q1.25 0 2.44-.81t1.13-1.38q0-.56-1.16-.9t-2.34-.28q-1.2-.07-2.44.37t-1.25 1q0 .25.5.75t1.37.9 1.75.35zm3.69-7.81v-.07q0-1.37-1.1-2.34t-2.62-.97-2.6 1-1.06 2.38v.06q0 1.37 1.1 2.34t2.62.97 2.6-1 1.06-2.37zm5.13 2.68q.8-1.18.75-2.59t-.63-1.9q-1.31-1-3.69 1.62-.87 1-1.22 2.44t.28 2.06q.5.25 1.2.31t1.62-.43 1.68-1.5zm1.43-4.56q.75-.62 1.07-1.75t-.13-2.4-1.75-2.03q-.5-.13-.69.68-.5 1.94.44 4.38.56 1.25 1.06 1.12z',
        'w' => 0.792,
        'vw' => 25
    ],
    'ravelry' => 'M25.44 3.88q3.37 1.75 5.12 5.15t1.38 7.28q-.25 2.06-.63 3.25-1.06 3.25-2.78 5.38t-4.9 3.87q-.75.5-1.25.69-1.13.37-2.5.25h-1.13q-4.37 0-7.78-2.66t-4.53-6.78h-.19q-.12 1.2.28 2.82t.9 2.68l.57 1 .81 1.25q-1.75-.93-2.75-2.97t-1.18-3.65l-.13-1.56Q.5 18.25 0 17.5q2.56 1.38 4.75 1.69l.06-.5q.13-2.31 1.44-4.44.63-3.12 2.69-5.62.44-1.82 1.25-3.32.44-.81 1.25-1.19 7.69-3.56 14-.25zm-16.75 6.3q-.69 1-1.19 2.32.62-.69 1.12-1.06zm-1.94 7.63q.06.94.25 1.75l2.56.44q-.37-1.19-.62-2.6t-.31-2.21l-.07-.82q-1.25 1.25-1.81 3.44zm.5 2.69q1 3.25 3.6 5.56t6.02 2.82q-.12 0-.18-.07l-.57-.34q-.37-.22-1.34-1T13 25.84t-1.72-2.12-1.34-2.66l-2.7-.56zm16.88 7.25q3-1.5 4.84-4.44t1.84-6.44q0-5-3.53-8.56t-8.53-3.56q-4.62 0-8 3.06-.5.88-.75 2.44 1.5-1.19 3.78-1.72t3.85-.47l1.5.13q1.37.06 1.3 1.75-3.68-.32-6 .53t-4.55 2.78q.56 4.87 1.68 6.87 3 .13 5.97-.68T22 17.69l1.44-.88q1.19-.87 1.31.25.25 1.5-.56 1.88Q21 20.44 17 21q-2.06.31-4.87.25.62 1.5 1.69 2.72t2.09 1.94 2 1.21 1.6.7l.56.18q2.25.38 4.06-.25z',
    'react' => 'M26.13 11.06q2.62.94 4.25 2.22T32 16t-1.72 2.81-4.53 2.25l-.56.2q.12.43.18.87.38 1.81.32 3.31t-.53 2.62-1.28 1.57q-1.25.69-3.1.06t-3.9-2.31l-.82-.82q-.31.32-.68.63-2.07 1.81-4.04 2.47t-3.21-.03T6.5 27t.13-4.56l.25-1.13q-.32-.06-.88-.24-2.69-.94-4.34-2.32T0 16t1.56-2.69 4.2-2.18l1.12-.38q-.13-.5-.2-1.06-.55-2.63-.18-4.6t1.63-2.72q1.24-.68 3.24 0T15.5 5l.5.5q.44-.44.81-.75 2.07-1.81 3.9-2.44t3.1.06q1.32.7 1.7 2.79t-.26 4.9q-.06.44-.12.69.5.13 1 .31zm-8.44-5.31q-.38.31-.75.69Q18.38 8 19.63 9.75q2.06.19 4.18.69.07-.25.13-.63.5-2.43.28-4.12T23.19 3.5q-.81-.44-2.31.19t-3.2 2.06zm-7.25 13.5-.88-1.69q-.62 1.5-1 2.82 1.38.3 2.88.43-.44-.62-1-1.56zm-1.88-7.56q.44 1.37 1 2.75.38-.75.88-1.63l.94-1.62q-1.5.19-2.82.5zM10.25 16q.69 1.38 1.38 2.56.68 1.25 1.5 2.44 1.43.06 2.87.06 1.5 0 2.88-.06.87-1.31 1.5-2.5.75-1.25 1.37-2.5-.69-1.37-1.38-2.56t-1.5-2.38q-1.43-.12-2.87-.12t-2.88.12q-.87 1.25-1.5 2.44-.74 1.31-1.37 2.5zm11.31 3.19q-.56 1-1 1.62 1.5-.18 2.94-.5-.44-1.43-1-2.75-.5.94-.94 1.63zm.88-4.75q.62-1.38 1-2.75-1.31-.32-2.81-.5.43.81.93 1.62.38.63.88 1.63zM16 7.38q-1 1.06-1.88 2.25 1.88-.07 3.76 0Q17 8.43 16 7.37zM8.75 3.56Q8 4 7.78 5.6T8 9.44l.19.94q2.12-.44 4.19-.63Q13.63 8 15.06 6.44L14.63 6q-1.82-1.63-3.44-2.28t-2.44-.16zm-1.5 16.5q.63-2.12 1.5-4.06-.88-1.94-1.5-4-.63.19-1.06.38-2.2.74-3.5 1.74T1.37 16q.13 1.19 1.13 1.88Q4 19 6.44 19.8l.81.25zm7.25 6.13q.31-.25.63-.57-1.5-1.62-2.7-3.37-2.12-.13-4.24-.63-.13.5-.25 1.07-.38 2.25-.16 3.78t1.03 1.97q1.06.44 2.19-.07 1.94-.75 3.5-2.18zm1.56-1.57q1-1.12 1.82-2.25-.88.07-1.88.07-.94 0-1.87-.07.93 1.25 1.93 2.25zm8.19 1.88q.25-2.13-.19-4.13-.06-.37-.19-.75-2.06.44-4.25.63Q18.44 24 17 25.63q.5.5.81.68 1.07 1 2.16 1.6t1.87.72 1.35-.13q.93-.75 1.06-2zm1.13-6.69q1.5-.5 2.71-1.18t1.88-1.38.66-1.31q0-.82-1.38-1.85t-3.56-1.78q-.5-.12-.94-.25-.63 2.06-1.5 3.94.94 2 1.56 3.94.19 0 .57-.13zM16 18.88q-1.19 0-2.03-.85T13.12 16t.85-2.03 2.03-.85 2.03.85.84 2.03-.84 2.03-2.03.84z',
    'reacteurope' => [
        'p' => 'm15.5 13.25.13-.44-.38-.25h.44l.12-.44.2.44h.43l-.38.25.13.44-.38-.25zm3.94 0 .12-.44-.31-.25h.44l.12-.44.13.44h.44l-.32.25.13.44-.38-.25zm-5.69 3.13q.19 0 .22.12t.03.75q0 .88-1.06.88t-1.07-.88v-2.5q0-.88 1.07-.88t1.06.88q0 .75-.25.75h-.19q-.25 0-.25-.63 0-.37-.37-.37t-.38.38v2.24q0 .38.38.38t.37-.38v-.5l.06-.15.2-.1h.18zM8.69 17.5q.25 0 .25.25v.13q0 .18-.25.18H7.3q-.18 0-.18-.18v-3.75q0-.2.18-.2h1.4q.25 0 .25.2v.18q0 .2-.25.2h-.9v1.18h.7q.24 0 .24.19v.19q0 .18-.25.18H7.8v1.25h.88zm-2.63-1.19.5 1.5q.13.32-.18.32h-.2q-.18 0-.24-.25l-.5-1.44h-.32v1.44q0 .25-.25.25h-.25q-.18 0-.18-.25v-3.7q0-.24.18-.24h.88q1.06 0 1.06.87v.75q0 .57-.5.75zm-.18-.87v-.57q0-.37-.44-.37h-.32v1.31h.32q.44 0 .44-.37zm4.93-1.32.7 3.7q.05.24-.2.24h-.25q-.18 0-.25-.18l-.12-.7h-.75l-.13.7q0 .18-.25.18h-.18q-.26 0-.2-.25l.7-3.69q.06-.18.24-.18h.44q.2 0 .25.18zm-.75 2.44h.57l-.32-1.81zm5.57 1.32q0 .18-.26.18h-.18q-.25 0-.25-.18V14.5h-.44q-.25 0-.25-.19v-.19q0-.18.25-.18h1.62q.2 0 .2.18v.2q0 .18-.2.18h-.5v3.38zm15.5-.07q.12 0 .12.13V18q0 .13-.13.13h-1.37q-.13 0-.13-.13v-3.94q0-.19.13-.12h1.38q.12 0 .12.12v.07q0 .12-.13.12h-1.18v1.56h1q.12 0 .12.13V16q0 .13-.12.13h-1v1.68h1.18zm-5.82-3.94q.94 0 .94.88v2.5q0 .88-.94.88h-.06q-.94 0-.94-.88v-2.5q0-.88.94-.88h.06zm.63 3.25v-2.25q0-.68-.66-.68t-.65.68v2.26q0 .68.65.68t.66-.68zm-2.88-.87.63 1.69q.06.12-.07.12h-.12q-.06 0-.13-.06l-.62-1.69H22V18q0 .13-.13.13h-.06q-.12 0-.12-.13v-3.94q0-.12.12-.12h.82q1 0 1 .87v.56q0 .7-.57.88zm-.43-.25q.68 0 .68-.69v-.44q0-.68-.68-.68H22v1.8h.63zM28 13.94q.94 0 .94.87v.63q0 .94-.94.94h-.63V18q0 .13-.12.13h-.13q-.12 0-.12-.13v-3.94q0-.18.13-.12H28zm.63 1.44v-.5q0-.7-.63-.63h-.63v1.81H28q.63 0 .63-.68zM16 19.25h.5l-.38.25.13.44-.38-.25-.37.25.13-.44-.32-.25h.44l.13-.44zm14.25-8.5q2.81 1.06 4.22 2.56t1.19 3-2.13 2.97-5.4 2.47q1.56 5.69 0 8.38-.94 1.75-2.94 1.75-2.13 0-5.19-2.22T13.81 23q-2.81-.25-5.31-.8-.88 3.94-.53 5.94T9.5 30.8l.28.16q.16.09.88.13t1.5-.25 2.09-1.13T17 27.5q.31.31.69.63-1.44 1.37-2.78 2.28t-2.41 1.25-1.94.31-1.56-.4q-1.5-.88-1.9-3.32t.52-6.31q-.62-.13-1.25-.35t-1.87-.75-2.16-1.12-1.62-1.56T0 16.19q0-1.88 2.03-3.44t5.47-2.5Q6 4.56 7.5 1.87q1-1.75 3-1.75 2.12 0 5.16 2.22T21.87 9q2.88.25 5.32.81.81-3.87.5-5.9t-1.5-2.72l-.4-.16q-.2-.09-.91-.1t-1.47.26-2.07 1.1-2.72 2.21L18 3.87Q20.5 1.44 22.78.5t3.88-.06 1.93 3.4-.6 6.22q1.2.32 2.26.69zM8.31 2.31q-.25.38-.37.9t-.19 1.07-.06 1.13.06 1.12l.13 1.06.15.94.16.72.12.53.07.22q2.56-.62 5.25-.94 1.56-2.18 3.18-4l.63.63Q16.13 7.19 14.8 9q2.88-.25 5.88-.06-2.88-3.88-5.63-5.88t-4.56-2q-1.5 0-2.19 1.25zM27.38 29.7q1.3-2.44-.13-7.69-2.25.63-5.25.94-1.5 2.19-3.19 4l-.62-.63q1.37-1.5 2.62-3.31-2.81.25-5.81.06 2.81 3.88 5.56 5.9T25.2 31q1.43 0 2.19-1.31zm.56-8.63q5.25-1.5 6.56-3.93.19-.25.25-.44t.1-.88-.35-1.31-1.5-1.47-3-1.53q-1.06-.44-2.19-.69-.31 1.19-.75 2.44-.37.06-.56.37-.19-.18-.44-.3.5-1.38.88-2.76-2-.44-4.5-.69 1.12 1.63 2.19 3.44-.44.13-.7.44-.37-.44-1-1.47t-1.62-2.53q-3.69-.25-7.06.06-1.13 1.63-2.19 3.57-.37.18-.62.5-.07-.32-.38-.5.94-1.75 2.06-3.44-2.5.25-4.5.81.38 1.31.94 2.69-.06.06-.19.12-.25-.31-.87-.31-.44-1.06-.75-2.25-3.25.94-5.06 2.28T.75 16.03t1.65 2.81 5.41 2.41q.38-1.25.81-2.5.25 0 .44-.06.13.06.5.06-.5 1.38-.87 2.69 2.37.56 4.5.75-1.07-1.63-2.07-3.44.5 0 .75-.19.13.07.2.13 1.05 1.87 2.3 3.56 3.38.25 7-.06 1.13-1.63 2.13-3.44.44 0 .69-.25.18.12.37.19-.94 1.81-2 3.37 2.69-.25 4.5-.81-.37-1.31-.94-2.63.2-.06.38-.18v-.07q.19.38.62.38.44 1.12.82 2.31zm-7.19-7.12q.13 0 .19.06v3.25q0 .88-.94.88h-.06q-.94 0-.94-.88V14q0-.06.12-.06h.07q.12 0 .12.06v3.13q0 .68.63.68t.62-.68V14q0-.06.13-.06h.06zm-.81 5.31h.44l-.38.25.13.44-.38-.25-.31.25.12-.44-.37-.25h.44l.12-.44zm-1.94.5h.44l-.38.25.13.44-.38-.25-.31.25.13-.44-.38-.25h.44l.12-.44zm.31-1.94q.13 0 .13.13V18q0 .13-.13.13h-1.43q-.13 0-.13-.13v-3.94q0-.12.13-.12h1.37q.13 0 .13.12t-.13.13h-1.19v1.62h1q.13 0 .13.13V16q0 .13-.13.13h-1v1.68h1.25zm-.5-6.25.2.44h.43l-.38.25.13.44-.38-.25-.3.25.12-.44-.38-.25h.44z',
        'w' => 1.113,
        'vw' => 36
    ],
    'readme' => [
        'p' => 'M33 2.88q1.25.06 2.13.93T36 5.94v15.31q0 1.25-.88 2.13t-2.12.87h-5.63q-4.56 0-6.5 1.1t-2.68 3.59q-.07.12-.2.12t-.18-.12q-.5-1.69-1.37-2.66t-2.78-1.5-5.04-.53H3q-1.25 0-2.13-.87T0 21.25V5.88q0-1.2.87-2.1T3 2.88h8.75q2.25 0 4 1.43T18 7.94q.5-2.19 2.25-3.63t4-1.43H33zM15.12 19.5v-1.44q0-.19-.18-.19H4.87q-.18 0-.18.2v1.43q0 .19.18.19h10q.25 0 .25-.19zm0-3.81v-1.44q0-.19-.18-.19H4.87q-.18 0-.18.19v1.44q0 .19.18.19h10q.25 0 .25-.2zm0-3.81v-1.44q0-.2-.18-.2H4.87q-.18 0-.18.2v1.43q0 .26.18.26h10q.25 0 .25-.26zm16.2 7.62v-1.44q0-.25-.2-.25h-10q-.25 0-.25.25v1.44q0 .19.25.19h10q.2 0 .2-.19zm0-3.81v-1.44q0-.25-.2-.25h-10q-.25 0-.25.25v1.44q0 .19.25.19h10q.2 0 .2-.2zm0-3.81v-1.44q0-.2-.2-.2h-10q-.25 0-.25.2v1.43q0 .26.25.26h10q.2 0 .2-.26z',
        'w' => 1.125,
        'vw' => 36
    ],
    'rebel' => 'M16 31.5q-6.44 0-10.9-4.69T.8 15.62q.13-3.75 2-7.03t5.32-5.47h.06q-.25.25-.84 1.03T5.96 6.53 4.84 9.81t.2 3.75 2.27 3.75q.63.63 1.29.97t1.18.4 1.03-.02.88-.29.66-.4.4-.35l.19-.18q.5-.63.75-1.41t.28-1.5-.06-1.47-.25-1.37-.35-1.1-.31-.72l-.13-.31q-.24-.69-.96-1.31t-1.28-.88L10 7.06l1.81-2q.44.19.88.5t.75.66.56.62.38.47l.12.19q.06-.82-.28-1.78t-.72-1.53l-.37-.5 2.8-3.2 2.76 3.13q-.56.81-.9 1.78t-.41 1.53v.63q.12-.31.37-.6t.5-.53.53-.47.5-.37.4-.28.26-.19l.12-.06 1.82 2q-.82.25-1.5.87T19.3 9.06l-.3.5q-.5 1-.82 2.34t-.13 2.94 1 2.66q.38.62.9.9t1.04.31 1.03-.09.97-.34.78-.4.53-.38l.19-.13q1.81-1.62 2.4-3.66t.16-3.8-1.12-3.32-1.38-2.4l-.69-.88q.07-.13.07-.19 1.56 1.12 2.5 1.97t2.18 2.37 1.88 3.66.69 4.81q.06 6.82-4.19 11.2T16 31.5z',
    'red-river' => 'M22.06 2q2.44 0 4.2 1.75T28 7.94v16.12q0 2.44-1.75 4.2T22.06 30H5.94q-2.44 0-4.19-1.75T0 24.06V7.94Q0 5.5 1.75 3.75T5.94 2h16.12zm-13 10.56Q9 12 9.5 11.5t1.06-.44h3.5q1.25 0 2.16-.9t.9-2.16H7.5q-.62 0-1.06.44T6 9.5v9.63q1.25 0 2.16-.91t.9-2.16v-3.5zm11 4.5q1.25 0 2.16-.9t.9-2.16H13.5q-.63 0-1.06.44T12 15.5v9.63q1.25 0 2.15-.9t.91-2.16v-3.5Q15 18 15.5 17.5t1.06-.44h3.5z',
    'reddit' => 'M12.6 19.13q-.66 0-1.1-.47t-.44-1.1.44-1.1 1.1-.46 1.09.47.44 1.1-.44 1.09-1.1.46zM31.5 16q0 6.44-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5t10.97 4.53T31.5 16zm-8.25-2.56q-.88 0-1.5.62-2.25-1.56-5.38-1.68l1.07-4.88 3.5.75q0 .69.44 1.13t1.09.43 1.1-.47.43-1.09-.44-1.1-1.12-.46q-.94 0-1.38.87l-3.8-.87q-.32-.07-.45.31l-1.18 5.38q-3.13.12-5.38 1.68-.56-.62-1.5-.62-1 0-1.6.75t-.4 1.72 1.13 1.47q-.13.43-.13.93 0 2.5 2.44 4.22T16 24.25q1.69 0 3.22-.47t2.66-1.25 1.75-1.9.62-2.32q0-.5-.06-1 .94-.43 1.1-1.4t-.45-1.72-1.59-.75zm-4.31 7.25q.25-.25.5 0 .31.31 0 .56-1.07 1.06-3.44 1.06t-3.44-1.06q-.31-.25 0-.56.25-.25.5 0 .81.81 2.9.84t2.98-.84zm.43-4.69q.7 0 1.13.47t.44 1.1-.44 1.09-1.1.47-1.09-.47-.44-1.1q0-.43.2-.78t.55-.56.75-.22z',
    'reddit-alien' => 'M27.5 12.75q1.69 0 2.63 1.19t.65 2.72T29 18.94q.19.69.19 1.56 0 3.94-3.85 6.72T16 30q-3.56 0-6.6-1.28t-4.78-3.44-1.75-4.78q0-.81.2-1.5-1.57-.81-1.85-2.31t.65-2.72 2.63-1.22q1.38 0 2.38 1 3.56-2.5 8.5-2.69l1.93-8.62q.13-.57.7-.44l6.05 1.37Q24.81 2 26.26 2q1.05 0 1.77.72t.72 1.75-.72 1.75-1.78.72q-1 0-1.72-.72t-.72-1.72l-5.5-1.25-1.75 7.81q5 .19 8.57 2.63 1-.94 2.37-.94zM8.06 19.31q0 1 .75 1.72t1.75.72q.5 0 .97-.19t.78-.53.5-.78.2-.94q0-1.06-.73-1.78t-1.72-.72q-.68 0-1.25.35t-.9.9-.35 1.25zm13.44 5.82q.44-.38 0-.82t-.88 0q-1.3 1.32-4.65 1.32T11.3 24.3q-.37-.44-.81 0t0 .82q1.69 1.75 5.47 1.75t5.53-1.75zm-.06-3.38q.69 0 1.25-.31t.9-.88.35-1.25q0-1.06-.75-1.78t-1.75-.72-1.72.75-.72 1.75.72 1.72 1.72.72z',
    'reddit-square' => 'M17.69 21.63q.31.25 0 .56Q16.56 23.3 14 23.3t-3.69-1.12q-.31-.32 0-.57.25-.3.56 0 .88.88 3.13.88t3.12-.88q.32-.3.57 0zM12 18.25q0 .69-.5 1.16t-1.19.46-1.15-.46-.47-1.16.47-1.19 1.15-.5 1.19.5.5 1.19zm5.69-1.69q.69 0 1.15.5t.47 1.2-.47 1.15-1.15.47-1.19-.47-.5-1.16.5-1.19 1.19-.5zM28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22q1.25 0 2.13.88T28 5zm-6.25 8.81q-.88 0-1.56.63-2.44-1.69-5.82-1.75l1.2-5.31 3.68.87q0 .44.25.81t.63.6.8.22q.7 0 1.2-.47t.5-1.16-.5-1.19-1.2-.5q-1 0-1.5.94l-4.12-.94q-.31-.06-.43.32l-1.25 5.8q-3.38.13-5.75 1.82-.7-.69-1.63-.69-1.12 0-1.78.82t-.44 1.84T5.25 18q-.12.5-.12 1.06 0 2.63 2.62 4.5T14 25.44q3.69 0 6.28-1.88t2.6-4.5q0-.62-.13-1.06 1-.5 1.22-1.53t-.44-1.84-1.78-.82z',
    'redhat' => 'M19.5 25.06q.31.25.06.7-.12.24-.37.5-.57.43-1.25 0-.44-.32-.82-.26-.25.06-.34-.12t.03-.38q.25-.69 1.19-.19.69.44 1 .2.19-.13.44-.45h.06zM16 1.88q6.63 0 11.31 4.68t4.7 11.32q0 4.68-2.57 8.68-2.69-.62-4.81.25-.2.07-.2.25 0 .07.04.13t.03.19q-.06.31-.81.5-.5.06-.91.4t-.81.82-.47.53q-.13.12-.38.25t-.5.18-.47.1-.21-.03q-.07-.07 0-.07.93-1.3 1.06-1.56.19-.31.53-.72t.47-.65q.62-.82.75-1.25.06-.25.1-.75t.02-.57.22-.65.22-.85q-.06-.12-.28-.03t-.62.22-.6.13q-.25 0-.72-.16t-.65-.15q-.25 0-.7.18t-.68.2q-.12-.07-.44-.2t-.56-.12-1.25.15-1.88.16q-1-.06-1.9-.88t-1.28-1.37q-.32-.56-.82-.13-.25.2-.46.13t-.35-.25q-.5-.94-1-1.31-.69-.5-1.37-.32-.44.13-.7.57t-.18.87q0 .5.56 1.13.13.12.28.37t.32.44.4.31q.44.2.32.57-.07.25-.6.28t-.97-.22q-.18-.13-.72-.78t-1.21-.66q-.32 0-.63.13-1.25.62-2.69.62H.81q-.82-2.44-.82-5.06 0-6.63 4.7-11.31t11.3-4.7zM29 17.5q.25-1.38-.81-2.4t-3.2-1.54q-.3 1.25-1.37 1.75-.12.07-.15 0t.03-.19l.56-1q.32-.75-.06-2-.31-1.12-.78-2.3t-1-2.23-.97-1.47q-1.44-1.43-3.94-.8-.06 0-.37.08t-.57.16-.65.13-.88.03-1.03-.16q-3.37-.75-4 1.63l-.69 2.8q-.18.32-.03.76t.47.84 1.44.88 2.94.78q1.75.37 2.25.87t.6 1.16.65 1.03q.44.25-.25.44h-.13q-.81.12-3.12-.6T10 14.5q-2.25-1.38-1.44-3-1.44-.13-2.53 0t-1.81.53-.9 1.1Q3 14.18 4.5 15.71t4.25 3 5.87 2.4q5.07 1.57 9.44.53T29 17.5zM14.37 8.44q1.07 0 1.32.5.19.31-.38.44l-.56.12q-1.88.13-2.38.69-.24.19-.5.03t-.43-.4l-.2-.32q-.3-.06-.55-.47t-.07-.78q.2-.44.47-.44t.4.2q.26.3.54.43t.5.12.78-.06 1.06-.06zm3.82-2.5q.75-.06 1.34.06t.85.35.25.46q0 .38-.57.6t-1.31.22q-.5 0-.94-.13v-.12q1.13-.25 1.07-.63-.2-.62-2.2-.31-.06 0-.06-.06t.07-.07q.56-.31 1.5-.37z',
    'renren' => 'M13.38 10.56q0 5.13-2.57 9.28T4.12 26Q.5 21.69.5 16.06q0-3.81 1.69-7.1T6.8 3.54 13.38.75v9.81zm2.56 20.94q-3.94 0-7.44-1.88 2.81-1.75 4.78-4.28t2.66-5.4q.68 2.87 2.65 5.4t4.78 4.29q-3.43 1.87-7.43 1.87zM27.88 26q-2.07-.94-3.75-2.53t-2.91-3.56-1.9-4.38-.7-4.97V.75q5.5.94 9.2 5.25t3.68 10.06q0 5.63-3.62 9.94z',
    'replyd' => 'M20 30H8q-3.31 0-5.66-2.34T0 22V10q0-3.31 2.34-5.66T8 2h12q3.31 0 5.66 2.34T28 10v12q0 3.31-2.34 5.66T20 30zm-7.94-12.94q-.56-.18-1-.18-.69 0-1.19.37-.5.31-.8 1H9l-.25-1.19H7v8.69h2.25v-5.62q.12-.5.53-.82t1.03-.31q.44 0 1 .19zm7.25 6.5q-.31.2-.87.44-.57.19-1.32.19-1.12 0-1.62-.56T14.94 22h5.18q.13-.44.13-.87.06-.38.06-.75 0-1-.31-1.7t-.75-1.12q-.5-.37-1.13-.56t-1.25-.19q-2 0-3.09 1.16t-1.1 3.47q0 2.12.98 3.34T16.75 26q1 0 1.88-.31.93-.25 1.43-.7zm-1.4-4.43q.34.5.28 1.43h-3.2q.07-.87.5-1.4t1.32-.53q.75 0 1.1.5zm5.03.18q.37.38 1 .38t1.03-.38.4-1-.4-.97-1.03-.34-1 .34-.38.97.38 1zm0 6.25q.37.38 1 .38t1.03-.38.4-.97-.4-.93-1.03-.35-1 .35-.38.93.38.97z',
    'researchgate' => 'M0 2h28v28H0V2zm16.38 20.88v-.44q-.82 0-1.7-.94-1.43-1.56-2.62-3.63 1.06-.25 1.75-1.12t.69-1.88q0-1.37-.97-2.12T11 12l-2 .06q-1.87.06-2.44 0v.5q.13 0 .35.07t.28.06.25.06.22.13.12.15.1.22.03.35.03.46v6.82q0 .31-.03.5t-.04.31-.09.25-.16.19-.21.1-.22.06-.28.06-.35.03v.5h4.31v-.5l-.4-.03q-.28-.03-.35-.07t-.24-.06-.26-.1l-.15-.15-.13-.25-.03-.34V18q.38.07 1.32.07 1.43 2.62 2.62 3.93.69.82 1.81 1t1.32-.12zm1.43-8.44q.5 0 .88-.06t.6-.32.37-.4.22-.6.06-.53v-1.16h-1.88V12h1.13q0 1.69-1.38 1.69-1.12 0-1.12-1.44v-1.69q0-.56.4-.9t.85-.35q.37 0 .69.19t.43.44l.13.19.62-.5q0-.13-.09-.28t-.6-.47-1.18-.32q-1.06 0-1.6.6t-.53 1.28v2q0 .87.5 1.44t1.5.56zm-7.25 2.81q-.62 0-1.25-.06v-4.32q.38-.06 1.38-.06Q13 12.81 13 15q0 1-.66 1.62t-1.78.63z',
    'resolving' => 'M17.56 17.38 5.2 20.93 3.44 14.8l11.75-3.37q2.5-.69 3.31-.47t1.13 1.28l.68 2.44q.32 1-.18 1.56t-2.57 1.13zM15.5.5q4.25 0 7.81 2.06t5.63 5.66T31 16q0 6.38-4.56 10.94T15.5 31.5q-4.31 0-7.94-2.19l-1.44-5 17.94 2.32 3-.82-.62-2.12L12.69 22l5.81-1.69q2.25-.62 3.81-2.06 2.2-2.06 1.44-4.63l-.62-2.3q-.88-3-3.88-3.5-2.12-.26-4.94.55l-13.87 4Q1.69 7.2 5.88 3.84T15.5.5zM2.37 24.31Q0 20.44 0 16.06z',
    'rev' => [
        'p' => 'M16.94 17.19q0 1.69-1.22 2.87t-2.9 1.2-2.88-1.2-1.19-2.87 1.19-2.9 2.87-1.23 2.9 1.22 1.23 2.9zm8.68-.32V30H12.5q-5.19-.13-8.84-3.88T0 17.2q0-5.07 3.47-8.75t8.47-4.07V2l6.81 3.94-6.81 3.94v-2.5q-3.75.37-6.35 3.18T3 17.2q0 4.06 2.88 6.94T12.8 27t6.94-2.87 2.88-6.94q0-2.57-1.22-4.75t-3.28-3.5L21 7.3q4.5 3.75 4.63 9.57z',
        'w' => 0.8,
        'vw' => 26
    ],
    'rocketchat' => [
        'p' => 'M30.75 6.63Q34.44 9 35.78 12.5t0 7.03-5.03 5.84q-3.5 2.32-8.16 3.07t-8.53-.07q-2.12 2-4.5 2.91T5.2 32t-3.25-.53T0 30.75l.5-.47q.31-.28 1.06-1.16t1.28-1.71.88-1.97.15-2.07q-1.18-1.18-2-2.78T.73 17.03t.44-4.19 2.72-4.22q.3-1.5-.66-3.34t-2.1-2.9L0 1.24Q.75.87 2 .53T5.22 0t4.34.72 4.5 2.9q3.88-.81 8.54-.06t8.15 3.06zM18.62 25.38q6 0 10.25-2.75T33.13 16q0-2.56-1.96-4.72t-5.29-3.4-7.25-1.26q-6.06 0-10.28 2.75T4.13 16q0 3.38 3.25 5.94.87 2.75-1 6.25l-.07.12-.12.2q1.75-.13 3.25-.95 1.31-.68 2.31-1.68L13 24.63q2.75.75 5.63.75zm-7-7.13q-.56 0-1.06-.28t-.81-.78-.31-1.06q0-1.38 1.06-1.94t2.15-.03 1.1 1.84v.06q0 .88-.63 1.53t-1.5.66zm4.66-2.16q-.03-1.34 1.06-1.9t2.2-.03T20.62 16v.06q0 1.38-1.07 1.94t-2.15 0-1.13-1.9zm9.06 2.16q-.59 0-1.09-.28t-.78-.78-.28-1.06q-.06-1.38 1.03-1.94t2.19-.03T27.5 16v.06q0 .57-.29 1.07t-.78.8-1.1.32z',
        'w' => 1.136,
        'vw' => 36
    ],
    'rockrms' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm9.81 26.25L19 19.31l4.63-.06q1.37.06 2-1.19t-.25-2.31L15.5 4.19q-.63-.82-1.69-.82t-1.69.82l-9.25 10.8H8.5l5.31-6.18L19.06 15h-4.69q-1.37 0-1.93 1.28t.25 2.22l7 8.25h5.62z',
    'rust' => 'm31.78 15.6-1.36-.83-.04-.37 1.17-1.1a.46.46 0 0 0-.15-.76l-1.5-.56-.1-.36.93-1.3c.1-.13.12-.3.06-.44a.45.45 0 0 0-.35-.28l-1.6-.26-.16-.32.66-1.47a.46.46 0 0 0-.03-.44.45.45 0 0 0-.4-.2l-1.61.05-.23-.27.37-1.58a.47.47 0 0 0-.12-.43.47.47 0 0 0-.43-.12l-1.58.37-.27-.22.05-1.62a.46.46 0 0 0-.2-.4.46.46 0 0 0-.44-.03l-1.47.67-.32-.17-.26-1.6a.45.45 0 0 0-.28-.34.45.45 0 0 0-.44.05l-1.3.93-.36-.1-.56-1.5a.46.46 0 0 0-.34-.3.47.47 0 0 0-.42.15l-1.1 1.17-.37-.04L16.4.22A.45.45 0 0 0 16 0c-.16 0-.3.08-.4.22l-.83 1.36-.37.04L13.3.45a.46.46 0 0 0-.76.15l-.56 1.5-.36.1-1.3-.93a.45.45 0 0 0-.45-.06.47.47 0 0 0-.27.36l-.26 1.58-.32.17-1.47-.66a.45.45 0 0 0-.44.04.46.46 0 0 0-.2.4l.05 1.61-.27.23-1.58-.38a.47.47 0 0 0-.43.12.47.47 0 0 0-.12.43l.37 1.58-.22.27-1.62-.05a.5.5 0 0 0-.4.2c-.08.13-.1.3-.03.44l.67 1.47-.17.32-1.6.26a.45.45 0 0 0-.34.28.45.45 0 0 0 .05.44l.94 1.3-.1.36-1.5.56a.46.46 0 0 0-.16.76l1.17 1.1-.04.37-1.36.84A.45.45 0 0 0 0 16c0 .16.08.3.22.4l1.36.83.04.37-1.17 1.1a.46.46 0 0 0 .15.76l1.5.56.1.36-.93 1.3a.45.45 0 0 0 .3.72l1.58.26.17.32-.66 1.47a.45.45 0 0 0 .04.44c.08.13.23.21.4.2l1.6-.05.23.27-.37 1.58a.5.5 0 0 0 .12.43c.12.1.28.15.43.12l1.58-.37.27.22-.05 1.62c-.01.16.07.3.2.4.13.08.3.1.44.03l1.47-.67.32.17.26 1.6a.46.46 0 0 0 .72.29l1.3-.94.36.1.56 1.5a.46.46 0 0 0 .76.16l1.1-1.17c.12.01.24.03.37.03l.84 1.37a.46.46 0 0 0 .78 0l.84-1.37c.13 0 .25-.02.37-.03l1.1 1.17a.46.46 0 0 0 .76-.15l.56-1.5.36-.1 1.3.93c.13.1.3.11.45.05a.45.45 0 0 0 .27-.35l.26-1.59.32-.17 1.47.67c.14.07.3.05.44-.04a.45.45 0 0 0 .2-.4l-.05-1.6.27-.23 1.58.37a.5.5 0 0 0 .43-.12c.1-.12.15-.28.12-.43l-.37-1.58.22-.27 1.62.05c.16.01.3-.07.4-.2.08-.13.1-.3.03-.44l-.67-1.47.17-.32 1.6-.26a.45.45 0 0 0 .34-.28.45.45 0 0 0-.05-.44l-.94-1.3.1-.36 1.5-.56a.46.46 0 0 0 .16-.76l-1.17-1.1c.01-.12.03-.24.03-.37l1.37-.84A.46.46 0 0 0 32 16c0-.16-.09-.3-.22-.4Zm-9.44 8.08a.87.87 0 0 0-1.03.66l-.47 2.23a11.72 11.72 0 0 1-9.78-.04l-.47-2.23a.87.87 0 0 0-1.04-.67l-1.96.42c-.37-.37-.7-.77-1.02-1.2h9.57c.11 0 .18-.01.18-.12v-3.38c0-.1-.07-.12-.18-.12h-2.8v-2.15h3.04c.27 0 1.48.08 1.86 1.62.12.47.38 2 .56 2.5.18.55.91 1.65 1.7 1.65h4.94c-.33.44-.7.86-1.08 1.26Zm1.62 2.15a.95.95 0 1 1-.96-.95h.03c.51.01.92.44.92.95Zm-14.1-.04a.95.95 0 1 1-.96-.95h.03c.52.02.92.44.92.95ZM4.35 14.63l2.05-.9c.2-.1.37-.27.46-.49a.87.87 0 0 0-.02-.66l-.42-.96h1.66v7.49H4.73a11.78 11.78 0 0 1-.38-4.48Zm-.7-2.25c0-.52.42-.95.95-.95h.03a.96.96 0 0 1-.04 1.9.95.95 0 0 1-.95-.95Zm9.7 1.53v-2.2h3.95c.2 0 1.44.23 1.44 1.16 0 .77-.95 1.04-1.73 1.04Zm11.59 5.26c-.62.07-1.3-.26-1.38-.63-.36-2.03-.96-2.46-1.9-3.21 1.17-.75 2.4-1.86 2.4-3.33 0-1.6-1.1-2.6-1.84-3.1a5.26 5.26 0 0 0-2.52-.82H7.27a11.7 11.7 0 0 1 6.55-3.7l1.47 1.54a.9.9 0 0 0 1.23.03l1.64-1.57c3.38.64 6.32 2.73 8.02 5.71l-1.12 2.54c-.2.44 0 .95.44 1.15l2.16.95c.07.68.08 1.36.03 2.04h-1.21c-.12 0-.17.08-.17.2v.55c0 1.3-.73 1.58-1.37 1.65ZM15 3.77c0-.53.43-.96.95-.96h.03c.52.02.93.45.93.97a.96.96 0 0 1-.97.94.96.96 0 0 1-.94-.96Zm12.3 9.6a.95.95 0 0 1 0-1.9h.03a.96.96 0 0 1-.03 1.9Zm0 0',
    'safari' => 'M14.81 16.06q0-.25.07-.43t.18-.35.35-.25.4-.1q.44 0 .79.29t.34.72q0 .31-.13.56t-.37.4-.56.16q-.38 0-.72-.28t-.35-.72zM31.5 16q0 6.44-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5t10.97 4.53T31.5 16zm-1.69 0q0-5.75-4.03-9.78T16 2.19 6.22 6.22 2.19 16t4.03 9.78T16 29.81t9.78-4.03T29.81 16zm-4.5 6.06q0 .13 1 .75-2.69 4.07-7.37 5.2l-.25-1.13q0-.2-.25-.2t-.19.26l.25 1.19q-1.25.25-2.5.25-3.44 0-6.44-1.82l.75-1.25q0-.25-.19-.25-.12 0-.3.25t-.42.6-.21.4q-4.13-2.69-5.25-7.5l1.18-.25q.2-.06.2-.25t-.26-.18l-1.19.25q-.25-1.2-.25-2.38 0-3.56 1.88-6.56 1.12.69 1.19.69.18 0 .18-.2 0-.12-.21-.3t-.54-.38-.37-.19Q8.56 5 13.25 3.94l.25 1.18q.06.13.28.13t.16-.25l-.25-1.13q1.25-.25 2.31-.25 3.56 0 6.56 1.88-.68 1-.68 1.12 0 .25.25.25.12 0 .8-1.12 4 2.69 5.07 7.37l-.94.25q-.18 0-.18.25 0 .2.25.2l1-.26q.25 1.25.25 2.44 0 3.5-1.82 6.44-.06-.07-.28-.22t-.44-.28-.28-.13q-.25 0-.25.25zM23.25 6.88 14.62 15l-6 10q8.5-7.94 8.57-8.06z',
    'salesforce' => [
        'p' => 'M15.55 15.35h-1.64c.04-.32.2-.88.85-.88.42 0 .75.24.8.88Zm8.55-.87c-.03 0-.89-.1-.89 1.25 0 1.36.86 1.25.89 1.25.81 0 .88-.84.88-1.25 0-1.36-.86-1.25-.88-1.25ZM8.9 15.97a.54.54 0 0 0-.2.46c0 .3.12.37.2.44.29.23.94.13 1.3.06v-1.06c-.33-.07-1.04-.12-1.3.1ZM40 14.5c0 5.47-5 9.65-10.34 8.53a6.33 6.33 0 0 1-8.26 2.6 7.2 7.2 0 0 1-13.36-.33c-7.48 1.5-11.18-8.64-4.7-12.45a7.78 7.78 0 0 1 13.3-7.8 7.04 7.04 0 0 1 5.07-2.18 7.1 7.1 0 0 1 6.18 3.66A8.65 8.65 0 0 1 40 14.5ZM7.53 16.49c0-.74-.73-.95-1.12-1.08-.33-.13-.84-.21-.84-.55 0-.6 1.07-.42 1.57-.14 0 0 .08.05.11-.02l.16-.46c.01-.04 0-.08-.04-.1-.77-.47-2.55-.52-2.55.8 0 .78.72.97 1.12 1.08.3.1.83.18.83.54 0 .25-.23.44-.58.44-.43 0-.84-.14-1.19-.4-.03-.01-.09-.04-.1.05l-.15.47c-.03.06.02.07.02.08.1.1.64.42 1.42.42.83 0 1.34-.44 1.34-1.14Zm2-2.66c-.64 0-1.17.2-1.34.32a.07.07 0 0 0-.03.04c0 .02 0 .03.02.05l.16.44c0 .03.04.05.07.04.04 0 .43-.25 1.06-.25.25 0 .44.05.57.15.23.18.2.52.2.66-.3-.02-1.2-.21-1.85.24-.3.2-.46.54-.45.9 0 .37.1.65.4.9.77.51 2.28.13 2.39.09.1-.02.22-.04.22-.12v-2.11c0-.3.02-1.36-1.42-1.36Zm2.9-1.31a.1.1 0 0 0-.01-.06.1.1 0 0 0-.06-.02h-.61a.1.1 0 0 0-.05.02.1.1 0 0 0-.02.06v4.93c0 .02 0 .04.02.05.01.02.03.03.05.03h.62a.1.1 0 0 0 .05-.02.1.1 0 0 0 .02-.06Zm3.5 1.8c-.14-.14-.43-.47-1.11-.47-.22 0-.89.02-1.3.56-.4.48-.4 1.13-.4 1.34 0 .2 0 .9.43 1.32.17.18.57.52 1.43.52.68 0 1.03-.15 1.16-.24.03-.01.05-.04.02-.12l-.15-.42a.08.08 0 0 0-.09-.04c-.16.05-.4.17-.96.17-1.08 0-1.05-.92-1.05-1.05h2.32c.04 0 .07-.02.07-.05-.01 0 .13-.92-.38-1.52Zm2.28 3.3c.83 0 1.34-.44 1.34-1.14 0-.73-.73-.94-1.11-1.07-.26-.1-.84-.21-.84-.56 0-.23.2-.4.52-.4.37.01.73.1 1.05.27 0 0 .08.05.1-.03.02-.04.15-.4.16-.46a.07.07 0 0 0-.04-.08c-.5-.31-1.05-.31-1.27-.31-.75 0-1.27.45-1.27 1.1 0 .78.72.97 1.11 1.08.39.12.83.2.83.54 0 .25-.22.44-.58.44a2 2 0 0 1-1.18-.4.06.06 0 0 0-.07 0 .06.06 0 0 0-.03.04l-.15.48c-.03.06.01.07.01.08.1.1.65.42 1.42.42Zm4.1-3.62c0-.04 0-.07-.06-.07h-.74c0-.01.06-.56.28-.78.26-.26.73-.1.75-.1.07.02.09 0 .1-.03l.18-.49c.04-.06 0-.07-.02-.09-.32-.12-1.08-.18-1.53.27-.34.34-.43.87-.5 1.22h-.52a.08.08 0 0 0-.08.07l-.09.48c0 .05.02.08.08.08h.51c-.53 3-.55 3.14-.65 3.47-.06.22-.2.43-.36.48 0 0-.25.1-.6-.01 0 0-.06-.03-.1.04 0 .05-.16.43-.17.47-.02.05 0 .1.03.1.32.12.81.1 1.12 0 .39-.15.6-.5.72-.82.17-.48.17-.61.73-3.73h.77c.04 0 .07-.04.07-.08Zm3.35 1c-.04-.1-.32-1.13-1.58-1.13-.95 0-1.44.62-1.57 1.13-.06.19-.2.88 0 1.47 0 .02.28 1.13 1.57 1.13.94 0 1.43-.6 1.58-1.13.2-.6.06-1.28 0-1.47Zm2.83-1.04c-.31-.1-1.04-.12-1.38.33v-.27c0-.02 0-.04-.02-.06a.08.08 0 0 0-.05-.02h-.6a.1.1 0 0 0-.05.02.1.1 0 0 0-.02.06v3.45l.02.05c.02.02.04.02.06.02h.6c.02 0 .04 0 .06-.02a.07.07 0 0 0 .02-.05v-1.74c0-.18 0-.7.28-.94.3-.3.75-.2.83-.19a.1.1 0 0 0 .1-.06c.06-.16.13-.33.18-.5.02-.03 0-.07-.03-.08Zm2.93 3.38-.13-.46c-.03-.07-.1-.04-.1-.04a2 2 0 0 1-.7.12c-.29 0-1.07-.07-1.07-1.24 0-.39.12-1.24 1.03-1.24.24 0 .49.04.72.11 0 0 .06.03.07-.05l.16-.47c.02-.05-.02-.07-.04-.07a2.53 2.53 0 0 0-1.73 0c-.1.05-1.02.4-1.02 1.72 0 .18-.04 1.88 1.8 1.88.34 0 .67-.06.98-.17.03-.02.04-.06.03-.1Zm3.37-2.47a1.37 1.37 0 0 0-1.4-1.02c-1 0-1.47.63-1.6 1.17-.08.23-.11.48-.1.73 0 1.62 1.17 1.84 1.86 1.84.68 0 1.03-.15 1.16-.24.03-.01.04-.04.02-.12l-.15-.42a.07.07 0 0 0-.09-.04c-.16.05-.4.17-.95.17-1.1 0-1.06-.92-1.06-1.04h2.32c.04 0 .06-.03.08-.06-.02 0 .05-.45-.1-.97Zm-1.46-.4c-.65 0-.81.56-.85.88h1.65c-.06-.74-.48-.88-.8-.88Zm0 0',
        'vw' => 40
    ],
    'sass' => [
        'p' => 'M34.44 18.25q2.56 0 4.1 1.38T40 22.68q-.06 1.06-.75 1.69t-1.12.75q-.32.06-.38-.13 0-.12.31-.31.94-.63 1.1-1.6T38.38 21t-3.25-1.38q-2.2-.25-3.88.44.32.82.38 1.44 0 1.38-1.5 2.5-.63.44-1.38.69-.69.31-1.06.25-1.19-.25-.56-2.06.5-1.44 2.12-2.75-.62-1.25-.75-1.63l-.37-1.62q-.94 2-2.2 3.8 0 .07-.18.32.38 1 .44 1.75.06 1.56-1.19 2.5-.62.38-1.31.63-.57.31-1.44.37-.69 0-1-.19-.38-.3-.19-.87.13-.19 1.88-2.31.37-.38.62-.82.19-.18.38-.5-.75-1.68-.82-1.87l-.37-1.63q-1 2.5-2 4.5-.94 1.88-1.56 2.88 0 .06-.07.12l-.06.07q-.75 1.18-1.44 1.18t-.94-.8-.18-1.63l.12-.88q-.5 1.25-1 2.25-.25.56-.44.84t-.53.5-.72.22q-.62 0-.94-.3-.56-.5-.93-1.67t-.32-2.15q.2-1.63.57-2.69-.88.5-1.5.81-.5.32-.75.5v.07q.56.81.47 2.75t-1.1 3.44q-1.06 1.68-2.94 2.46t-3.75.29q-.81-.25-1.25-1.41t.25-3.03q1-2.25 5.82-4.63.06-.06.15-.1t.16-.08q-.56-.44-1.97-1.44t-2.5-1.78-2.1-2.07-1.24-2.53q-.25-1.06.5-2.78t2.8-3.6q3.57-3.3 9.45-5.43 4.12-1.5 7.84-.69t4.4 3.44q.44 1.69-.27 3.53t-2.25 3.31-3.47 2.22q-2.32.94-4.57 1.03t-2.87-.28q-.94-.31-1.53-.84t-.78-.97q-.2-.5 0-.63.18-.06.5.32 1.25 1.31 4.06 1 4.5-.5 7.22-3.32t1.72-4.93q-.7-1.63-3.94-1.72t-7.94 2.03Q3.24 9.3 3 13.38q-.06 1.43.9 2.65t2.82 2.57 2.47 2.09q.06 0 .1.03l.03.03q.18-.06.4-.19t.47-.28.44-.22q1-.56 2.62-1.43.7-.94 1.72-1.7t1.85-.74 1.12.65.19 1.35l-.12.69q.06-.13.12-.13l.06-.06.16-.13.25-.06.34.06q.38.19.44.69l-.15.66-.41 1.65-.38 1.63q-.06.31-.09.62t0 .47l.03.16q.13 0 .31-.38.07-.19.2-.44v.07l.02-.03.03-.1.63-1.31q2.44-5.38 2.62-6 .13-.25.22-.66t.16-.53q.12-.37 1.56-.37.88 0 .88.44l-.1.28-.22.68-.12.72q-.13.57.06 1.57.07.43.57 1.37 2.18-3.62 2.43-4.69.13-.62.32-1.18.12-.38 1.56-.38.87 0 .94.44l-.13.28-.22.69-.15.72q-.07.56.06 1.56.06.37.81 1.87 1.7-.8 3.63-.8zM7.63 27.31q1.93-2.12 1.18-4.93l-.06.06-.1.06-.09.06-.22.13-.18.1-.2.09q-.11.06-.15.12-1.69 1-2.69 1.97t-1.28 2.16.28 1.5 1.63-.07 1.87-1.25zM16 21.56q1.31-3.19 1-3.69-.06-.25-.53-.06t-.9.69q-.88 1-1.38 2.5-.31.94-.38 1.84t.1 1.47.34.63q.13 0 .31-.22t.41-.66.38-.81.37-.94.28-.75zm6.94 3.31q1.56-.93 1.56-2.18v-.07q-.31.44-.69.88l-1.18 1.31-.07.19q.07.06.38-.13zm5.37-1.18q1.63-.63 1.63-2.19 0-.25-.13-.63-.81.7-1.18 1.5-.57 1.13-.32 1.32z',
        'w' => 1.25,
        'vw' => 40
    ],
    'schlix' => 'M21.88 9.88 18.5 7l4.63-2.44L28 7.31zM12 7.62l2.88-1.75L17 8l-3.44 1.88zm-4.06.44 2-1.37L11 8.44 8.7 9.88zm-1.44 5.5-.56-2.19 1.81-1.12.88 2.19zM5.12 8.31l1.5-1.12.57 1.5L5.5 9.8zM3.7 12.88l-.25-1.76 1.44-.93.37 1.75zm-1.94 1.06 1.31-.81.2 2.12-1.38.75zm-1.7 2.81 1.25-.63v2.38L0 19.06zm3.69 3.69L2 21l-.06-2.94 1.56-.68zm2.44-4-1.94.81-.31-2.56 1.69-.88zm1.5 7.12-2.56.38-.5-4 2.18-.63zm1.81-8.75 1.31 3.57-2.87.87-.88-3.37zm5.31 14.44L10.37 29l-1.5-6 3.44-.5zM9.5 11.75l2.69-1.38 1.75 2.82-3.19 1.37zm2.63 5.94 3.8-1.19 3.32 5.44-5 .69zm3.18-5.38 4.2-1.81 4.12 4.12-5.57 1.63z',
    'screenpal' => [
        'p' => 'M233.5 22.49C233.5 10.07 243.6 0 256 0C268.4 0 278.5 10.07 278.5 22.49C278.5 34.91 268.4 44.98 256 44.98C243.6 44.98 233.5 34.91 233.5 22.49zM313.4 259C313.4 290.7 287.7 316.4 256 316.4C224.3 316.4 198.6 290.7 198.6 259C198.6 227.3 224.3 201.6 256 201.6C287.7 201.6 313.4 227.3 313.4 259zM337.2 350C359.5 330.1 373.7 302.7 377.1 273H496.6C493.1 334.4 466.2 392.2 421.4 434.4C376.7 476.6 317.5 500.2 256 500.2C194.5 500.2 135.3 476.6 90.56 434.4C45.83 392.2 18.94 334.4 15.39 273H135.1C138.5 302.7 152.7 330.1 175 350C197.3 369.9 226.2 380.9 256.1 380.9C285.1 380.9 314.8 369.9 337.2 350zM73.14 140.3C73.54 152.7 63.81 163.1 51.39 163.5C38.97 163.9 28.59 154.2 28.18 141.8C27.78 129.3 37.52 118.9 49.94 118.5C62.35 118.1 72.74 127.9 73.14 140.3zM438.9 141C438.9 128.6 448.9 118.5 461.4 118.5C473.8 118.5 483.8 128.6 483.8 141C483.8 153.5 473.8 163.5 461.4 163.5C448.9 163.5 438.9 153.5 438.9 141zM317.9 95.27C300.6 109.1 278.7 118.1 256 118.1C233.3 118.1 211.4 109.1 194.1 95.27C176.8 80.55 165.3 60.18 161.7 37.78C176.8 31.37 192.5 26.52 208.6 23.31C208.6 35.88 213.6 47.93 222.5 56.82C231.4 65.7 243.4 70.7 256 70.7C268.6 70.7 280.6 65.7 289.5 56.82C298.4 47.93 303.4 35.88 303.4 23.31C319.5 26.52 335.2 31.37 350.3 37.78C346.7 60.18 335.2 80.55 317.9 95.27H317.9zM82.78 231C61.42 238.6 38.06 238.4 16.86 230.4C18.82 214.1 22.46 198.1 27.71 182.5C33.1 185.6 39.05 187.6 45.22 188.5C51.39 189.3 57.67 188.9 63.68 187.3C69.69 185.6 75.33 182.9 80.27 179.1C85.21 175.3 89.36 170.6 92.47 165.2C95.58 159.8 97.61 153.8 98.42 147.7C99.23 141.5 98.83 135.2 97.22 129.2C95.61 123.2 92.83 117.6 89.04 112.6C85.25 107.7 80.53 103.5 75.14 100.4C85.96 88.11 98.01 76.94 111.1 67.07C128.7 81.42 140.6 101.6 144.7 123.9C148.8 146.2 144.8 169.3 133.5 188.9C122.1 208.5 104.1 223.4 82.78 231V231zM429.2 231.1C407.9 223.5 389.9 208.5 378.5 188.9C367.2 169.3 363.3 146.2 367.4 123.9C371.5 101.7 383.4 81.54 400.9 67.19C414 77.04 426.1 88.21 436.9 100.5C426.2 106.9 418.5 117.2 415.4 129.3C412.2 141.3 413.1 154.1 420.2 164.9C426.4 175.7 436.6 183.6 448.6 186.9C460.6 190.2 473.5 188.6 484.3 182.6C489.6 198.1 493.2 214.2 495.2 230.4C473.1 238.5 450.6 238.7 429.2 231.1L429.2 231.1z',
        'vw' => 512,
        'vh' => 512
    ],
    'scribd' => [
        'p' => 'M2.63 15.81q-1.5-1.81-1.5-5 0-4.56 2.93-7.06t7.5-2.5q2.13-.06 4 .34t3.53 1.38 2.2 1.34 1.77 1.32l-3.19 4.5-2.3-1.5 1.68-2.44q-3.5-2.31-7.69-2.25-2.56 0-4.78 1.31t-2.22 3.5q0 1.5 1.82.94 2.06-.44 3.5.87 1.18 1.2 1.15 2.88T9.81 16.3q-1.37 1.44-3.6 1.29T2.63 15.8zm19.87.38q.63.75 1.03 1.87T24 20.5t-.16 2.66-.9 2.62T21.25 28q-3.25 2.75-8.37 2.75-6.82.06-11.94-4.5L0 25.38l3.81-4.5L6 22.63l-2.06 2.43q4 3 9 3 5.93 0 7.56-4.06.13-.37.13-.75 0-1.5-1.82-1-2.12.5-3.5-.87-1.25-1.13-1.19-2.82 0-1.81 1.32-3.06t3.12-1.12q2.38-.07 3.94 1.8z',
        'w' => 0.75,
        'vw' => 24
    ],
    'searchengin' => [
        'p' => 'M13.81 8.13 9.56 21.24v-8.19L6.2 14.62 9.56 2.7v7.18zm-5.25-6-.06.25-.94 3.3q-1.94.76-3.12 2.48T3.24 12q0 2.44 1.48 4.28t3.72 2.28v3.38q-3.63-.5-6.03-3.35T0 12t2.47-6.56 6.1-3.32zm19.5 27.93q-.93.94-1.81.63l-3.75-2.75q-.5-.38-.87-.72t-.63-.6-.44-.56-.28-.5-.25-.6-.22-.59Q18.75 22 16.2 21.12l-.57-.93q-2.37 1.62-5.12 1.8l.12-.43 1-3.06q2.25-.56 3.66-2.38t1.4-4.12q0-2.63-1.65-4.56t-4.34-2.2V2q1.93.12 3.65.97t2.97 2.15 1.97 3.1T20 12q0 3-1.81 5.62l.93.63q.88 2.5 3.25 3.56.2.06.72.28t.7.34.55.38.79.66.8 1.03l2.76 3.75q.31.87-.63 1.81zm-.62-1.12q0-.5-.5-.5t-.5.5.5.5.5-.5z',
        'w' => 0.896,
        'vw' => 29
    ],
    'sellcast' => 'M22.06 2q1.63 0 3 .78t2.16 2.16.78 3v16.12q0 2.44-1.75 4.19T22.06 30H5.94q-2.44 0-4.2-1.75T0 24.06V7.94q0-2.44 1.75-4.2T5.94 2h16.12zm-3.12 19.78q1.31-2.28.62-4.81t-2.94-3.85q-.25-.18-.4-.25l-.16-.06q0-.06-.03-.06t-.15-.07-.38-.25q-1.19-.68-1.84.5t.47 1.88q.56.31 1.18.62 1.25.75 1.66 2.22t-.34 2.75-2.22 1.7-2.72-.35q-.31-.19-.44-.25t-.16-.07h-.03v-.03l-.15-.09-.4-.25q-.5-.25-1.04-.13t-.78.63l-1 1.69Q7.13 24.3 8 25q1.13.94 2.19-.69.06-.13.13-.19 2.24 1.25 4.78.6t3.84-2.94zM20.3 8.75q.44-.75 0-1.4t-1.12-.66-1.19.69l-.31.5q-2.32-1.25-4.85-.57t-3.81 2.94-.63 4.78 2.97 3.85q.25.18.38.25t.19.06v.06l.15.06.4.25q1.2.7 1.85-.5t-.47-1.87l-1.18-.63q-1.32-.75-1.7-2.21t.38-2.75 2.22-1.7 2.72.35q.31.19.44.25l.12.06q.07 0 .07.04t.12.09.44.25q.94.44 1.62-.25.13-.13 1.2-1.94z',
    'sellsy' => [
        'p' => 'M33.75 14.81q2.56.63 4.22 2.72t1.66 4.78q0 3.2-2.29 5.44T31.94 30H8.06q-3.12 0-5.4-2.25T.38 22.31q0-2.25 1.21-4.12t3.22-2.82q-.18-.68-.18-1.43 0-1.44.68-2.63t1.91-1.9 2.66-.72q1.93 0 3.43 1.31.88-3.5 3.72-5.75T23.5 2q4.31 0 7.4 3.1t3.1 7.4q0 1.25-.25 2.31zM12.5 25.12V18.2q0-.44-.28-.72t-.66-.28H9.62q-.43 0-.71.28t-.28.72v6.93q0 .38.28.66t.72.28h1.93q.38 0 .66-.28t.28-.66zm5.56 0v-8.25q0-.37-.28-.65t-.65-.28h-1.94q-.38 0-.66.28t-.28.66v8.25q0 .37.28.65t.66.28h1.94q.37 0 .65-.28t.28-.66zm5.63 0V14.95q0-.25-.13-.47t-.34-.35-.47-.12h-1.94q-.44 0-.72.28t-.28.66v10.18q0 .38.28.66t.72.28h1.94q.37 0 .66-.28t.28-.66zm5.43 0V11.06q0-.37-.28-.69t-.65-.3h-1.81q-.38 0-.66.3t-.28.7v14.05q0 .38.28.66t.66.28h1.8q.38 0 .66-.28t.29-.66z',
        'w' => 1.25,
        'vw' => 40
    ],
    'servicestack' => 'M5.5 13.5q2.81.38 6.75 2.16t7.66 5.18T24.5 28H0q3.13-.25 5.53-2.66t2.6-5.56T5.5 13.5zm2-9.5q2.56.38 6.06 2.31t7.1 4.94 6.53 7.5T31 28h-6q-.56-2.81-2.22-5.44t-3.81-4.6-4.31-3.37-4.29-2.21Q10.5 7.3 7.5 4z',
    'shirtsinbulk' => 'M6.25 25.63 6.5 25l1.94.88-.25.62zm2.44 1.12.31-.63 1.88.82-.25.62zm10.81-.88 1.88-.87.25.63-1.88.87zm-8.31 1.94.31-.62 1.88.87-.32.63zm-7.44-3.25.31-.62 1.88.87-.25.57zM17 26.94l1.88-.82.3.63-1.93.81zm-2.5 1.12 1.94-.87.31.62-1.94.88zm7.44-3.25 1.94-.87.25.62-1.94.82zM10.62 2.94v.62H8.5v-.62h2.13zm-2.93 0v.62H5.56v-.62H7.7zm-2.94 0v.62H2.62v-.62h2.13zm8.81 0v.62H11.5v-.62h2.06zm5.94 0v.62h-2.13v-.62h2.13zm2.94 0v.62H20.3v-.62h2.13zm-5.94 0v.62h-2.06v-.62h2.06zm8.88 0v.62h-2.13v-.62h2.13zm-22.07 19v2.12h-.69v-2.12h.7zM7.7 8.44v.69H5.56v-.7H7.7zm-4.38.68v1.44h-.69V8.44h2.13v.69H3.31zm10.56 14.25q-2.37 0-4.06-1.65t-1.69-4.03 1.7-4.03T13.86 12t4.04 1.66 1.65 4.03-1.65 4.03-4.04 1.66zm-3-6.93q0 .94.75 1.31t1.66.31 1.66.1.75.65q0 .94-2.07.94-1.75 0-2.25-.94l-.56 1.13q1.19.75 2.94.75 1.19 0 2.19-.44t1-1.5q0-1-.78-1.44t-1.7-.37-1.68-.03-.78-.53q0-.94 1.87-.94 1.7 0 2.25.81h.07l.5-1q-1.82-.75-2.75-.75-.75 0-1.38.13t-1.15.59-.54 1.22zm-7.56-5.32v2.13h-.69v-2.13h.7zM24.7 24.06v-2.12h.69v2.12h-.7zM3.3 19.25v2.06h-.69v-2.06h.7zM0 .25h28v25.38l-14.13 6.12L0 25.62V.25zm26.19 24.19V7.56H1.8v16.88l12.07 5.31zm0-18.69V2.06H1.8v3.69h24.4zM3.3 13.88v2.06h-.69v-2.06h.7zm0 2.68v2.07h-.69v-2.07h.7zm7.31-8.12v.69H8.5v-.7h2.13zM24.7 18.62v-2.06h.69v2.07h-.7zm0-2.68v-2.06h.69v2.06h-.7zm-11.13-7.5v.69H11.5v-.7h2.06zm11.13 4.81v-2.13h.69v2.13h-.7zm0 8.13v-2.13h.69v2.13h-.7zM16.5 8.44v.69h-2.06v-.7h2.06zm3 0v.69h-2.13v-.7h2.13zm5.19 2.12V9.12h-1.44v-.68h2.13v2.12h-.7zm-2.25-2.12v.69H20.3v-.7h2.13z',
    'shopify' => [
        'p' => 'M24.27 6.5a.28.28 0 0 0-.27-.24l-2.33-.05-1.85-1.8v27.04l7.85-1.95-3.4-23Zm-6.23-2.1a6.93 6.93 0 0 0-.45-1.1c-.65-1.25-1.63-1.92-2.78-1.92a.9.9 0 0 0-.25.02c-.02-.05-.07-.08-.1-.13-.5-.54-1.15-.8-1.92-.77-1.5.05-3 1.13-4.2 3.05a11.96 11.96 0 0 0-1.68 4.38c-1.73.53-2.93.9-2.96.93-.87.27-.9.3-1 1.12-.07.63-2.37 18.24-2.37 18.24l18.91 3.28V4.1c-.1 0-.18.02-.28.03l-.92.28Zm-3.45 1.08-3.18.98c.3-1.18.9-2.35 1.6-3.13.28-.28.65-.6 1.08-.8a7 7 0 0 1 .5 2.95Zm-2.05-3.95c.31-.01.62.06.9.22a5 5 0 0 0-1.18.9A8.93 8.93 0 0 0 10.3 6.8c-.9.28-1.8.56-2.63.8.55-2.4 2.57-6.02 4.88-6.07Zm-2.9 13.76c.1 1.6 4.32 1.95 4.57 5.73.18 2.98-1.57 5-4.1 5.15a6.14 6.14 0 0 1-4.73-1.6l.65-2.75S7.71 23.1 9.06 23a1.2 1.2 0 0 0 1.17-1.28c-.12-2.1-3.57-1.98-3.8-5.43-.2-2.9 1.7-5.83 5.9-6.1a5.3 5.3 0 0 1 2.46.3l-.95 3.6s-1.08-.5-2.36-.4c-1.85.12-1.87 1.3-1.85 1.6Zm5.95-10.11c0-.75-.1-1.82-.45-2.73 1.15.23 1.7 1.5 1.95 2.28-.45.13-.95.27-1.5.45Zm0 0',
        'vw' => 28
    ],
    'shopware' => 'M24.75 28.44q-4.13 3.06-9.25 3.06-6.44 0-10.97-4.53T0 16 4.53 5.03 15.5.5q5.94 0 10.38 3.94.12.12.03.28t-.22.16q-1.7-.25-3.57-.25-6.25 0-10.06 2.53T8.25 14.3q0 2 .69 3.63t1.59 2.65 2.62 2.07 2.88 1.53 3.28 1.31q4.06 1.56 5.44 2.63.06.06.06.15t-.06.16zm6.19-13.88q.31 3.7-1 7.07-.07.25-.32.12-1.8-1-5.8-2.5-1.07-.37-1.6-.6t-1.4-.59-1.35-.65-1.06-.7-.88-.77-.5-.88-.22-1.06v-.06q-.06-2.07 2.31-2.94 1.13-.38 2.7-.38 4.5 0 9.05 3.82.07 0 .07.12z',
    'simplybuilt' => 'M30.06 4q.7 0 1.2.47t.5 1.16v20.75q0 .68-.5 1.15t-1.2.47H1.94q-.69 0-1.19-.47t-.5-1.16V5.63q0-.68.5-1.15T1.94 4h6.62q.44 0 .82.22t.59.6.22.8v2.5H21.8v-2.5q0-.68.47-1.15T23.44 4h6.62zM9.34 22.25q1.72 0 2.94-1.22t1.22-2.94-1.22-2.94-2.94-1.21-2.93 1.22-1.22 2.93 1.22 2.94 2.93 1.22zm13.32 0q1.71 0 2.93-1.22t1.22-2.94-1.22-2.94-2.93-1.21-2.94 1.22-1.22 2.93 1.22 2.94 2.94 1.22z',
    'sistrix' => 'M28 28.06 26.12 30 17 20.75q-2.88 2.38-6.63 2.38-4.24 0-7.3-3.1T0 12.56 3.06 5.1 10.41 2t7.34 3.1 3.06 7.46q0 3.44-2 6.2zM10.44 20.7q3.31 0 5.65-2.38t2.35-5.75-2.38-5.75-5.65-2.37T4.75 6.8t-2.37 5.75 2.37 5.75 5.69 2.38z',
    'sith' => 'm0 2 7.44 4.38-.75-3.7 5.69 4.38q.8-.19 1.62-.19t1.63.2l5.68-4.38-.75 3.68L28 2l-4.37 7.44 3.69-.75-4.38 5.69q.31 1.62 0 3.25l4.37 5.68-3.68-.75L28 30l-7.44-4.37.75 3.68-5.68-4.37q-1.63.31-3.25 0l-5.7 4.37.76-3.68L0 30l4.38-7.44-3.7.75 4.38-5.68q-.19-.82-.19-1.63t.2-1.62L.68 8.68l3.69.76zm13.97 6.25q-3.22 0-5.47 2.25t-2.25 5.47 2.28 5.5T14 23.75t5.47-2.28T21.75 16t-2.28-5.47-5.5-2.28zM14 10.5q2.25 0 3.88 1.63T19.5 16t-1.63 3.88T14 21.5t-3.88-1.63T8.5 16t1.63-3.88T14 10.5z',
    'sitrox' => [
        'p' => 'M212.439 0.00846128V0H448V128H64C64 57.6008 141.755 0.475338 212.439 0.00846128ZM237.256 192V192.007C307.135 192.475 384 249.6 384 320H210.809V319.995C140.915 319.563 64 262.424 64 192H237.256ZM235.565 511.993C306.251 511.521 384 454.399 384 384H0V512H235.565V511.993Z',
        'vw' => 448,
        'vh' => 512
    ],
    'sketch' => 'M1.13 10.13 6.05 3.5l-.43 8.19H0zm23.06-7.26.44 8.32L16.19 2zM6.44 13.63l7.69 15L.06 12.25h5.7zm.12-1.93 8.88-9.63 3.81 4.13 5.06 5.5H6.56zm18.57.56h5.68l-14 16.38zm.25-7.94 5.5 7.38h-5.63l-.38-7.07-.06-1.12zM6.5 5.88l.19-3L14.75 2l-8.5 9.19zm18 6.37L15.44 30 9.06 17.44l-2.69-5.2H24.5z',
    'skyatlas' => [
        'p' => 'M40 20.56q0 3.07-2.13 5.13t-5.24 2.06q-1.82 0-3.5-.69t-2.94-1.78-2.53-2.5-2.32-2.87-2.31-2.85-2.5-2.5-2.87-1.78-3.47-.65q-2.63 0-4.41 1.37t-2.19 3.31.03 3.88T5.94 24t4.56 1.38q4.19 0 7.87-2.63l.29-.28.4-.37.38-.29.3-.12q.32 0 .5.22t.2.47q0 .5-1.13 1.5-2 1.75-4.6 2.81t-5.09 1.06q-4 0-6.8-2.56T0 18.7q0-2.62 1.28-4.78t3.47-3.34 4.81-1.2q2.06 0 3.94.66t3.25 1.72 2.75 2.44 2.44 2.85 2.28 2.84 2.37 2.44 2.6 1.75 3 .68q1.75 0 2.94-.9t1.46-2.19 0-2.56-1.43-2.19-2.78-.9q-.57 0-1.6.34t-1.34.34q-.38 0-.7-.28t-.3-.65.18-1.13.2-1.12q0-3.13-2.1-5.16T21.5 6.32q-1.5 0-2.75.56t-1.97 1.16-.84.6q-.32 0-.5-.23t-.2-.53.5-.87q2.38-2.75 6.38-2.75 1.88 0 3.47.65t2.75 1.82 1.82 2.75.65 3.4q0 .75-.06 1.31 1.12-.3 2.25-.3 2.87 0 4.94 1.93T40 20.57z',
        'w' => 1.25,
        'vw' => 40
    ],
    'skype' => [
        'p' => 'M26.56 18.75Q28 20.63 28 23q0 2.88-2.06 4.94T21 30q-2.38 0-4.25-1.44-1.38.25-2.75.25-5.31 0-9.06-3.75T1.19 16q0-1.31.25-2.75Q0 11.37 0 9q0-2.88 2.06-4.94T7 2q2.38 0 4.25 1.44 1.38-.25 2.75-.25 5.31 0 9.06 3.75T26.81 16q0 1.31-.25 2.75zm-12.19 5.69q2.82 0 4.91-1.4t2.1-3.98q0-3.56-5.7-4.87-.3-.07-1-.19-1.06-.25-1.5-.38t-.96-.34-.69-.47-.16-.69q0-1.37 2.63-1.37 1.25 0 2.03.5t1.38 1.03 1.28.53q.43 0 .8-.16t.6-.43.38-.63.15-.78q0-1.5-2.12-2.47t-4.85-.97T8.9 8.62t-2.03 3.82q0 .68.1 1.25t.37 1 .53.75.82.62.84.5 1.06.4 1.1.32 1.28.31 1.28.28q1.06.32 1.44.41t.78.44.4.9q0 .75-.75 1.22t-1.93.47q-1.13 0-2-.44t-1.28-.97-1-.96-1.22-.44q-.88 0-1.35.53t-.47 1.34q0 1.7 2.22 2.88t5.28 1.19z',
        'w' => 0.873,
        'vw' => 28
    ],
    'slack' => 'M5.88 19.69q0 1.25-.88 2.1t-2.06.84-2.07-.85T0 19.72t.87-2.1 2.07-.87h2.94v2.94zm1.5 0q0-1.19.84-2.07t2.06-.87 2.1.88.87 2.06v7.37q0 1.2-.87 2.07t-2.1.87-2.06-.87-.85-2.07V19.7zM10.3 7.87q-1.25 0-2.1-.87t-.83-2.06.84-2.07T10.28 2t2.1.87.87 2.07v2.93h-2.94zm0 1.5q1.2 0 2.07.85t.87 2.06-.87 2.1-2.07.87H2.94q-1.19 0-2.06-.87T0 12.3q0-.81.4-1.5t1.07-1.06 1.47-.38h7.37zm11.81 2.94q0-1.25.88-2.1t2.06-.84 2.07.85.87 2.06-.87 2.1-2.07.87h-2.94v-2.94zm-1.5 0q0 1.2-.84 2.07t-2.06.87-2.1-.87-.87-2.07V4.94q0-1.19.87-2.07T17.7 2q.81 0 1.5.4t1.06 1.07.38 1.47v7.37zM17.7 24.12q1.25 0 2.1.88t.84 2.06-.85 2.07-2.06.87-2.1-.87-.87-2.07v-2.94h2.94zm0-1.5q-1.19 0-2.07-.84t-.87-2.06.87-2.1 2.07-.87h7.37q1.2 0 2.07.87t.87 2.1-.87 2.06-2.07.84H17.7z',
    'slack-hash' => 'M27.88 16.88q.3.93-.13 1.78t-1.31 1.09l-2.82.94 1 2.94q.25.93-.15 1.75t-1.28 1.12q-1 .31-1.82-.16t-1.06-1.28l-1-2.93-5.87 2 1 2.87q.18.63.06 1.22t-.53 1.03-.97.63q-.94.3-1.75-.13t-1.13-1.25l-1-3q-3.25 1.13-3.56 1.06-.69 0-1.28-.43t-.84-1.07q-.13-.43-.1-.9t.22-.85.53-.68.78-.44l2.82-.94-1.88-5.63q-3.31 1.07-3.62 1-.7 0-1.28-.4t-.79-1.1q-.18-.62-.06-1.18t.5-1.03 1-.66l2.81-.94-1-2.94q-.12-.43-.09-.9t.22-.85.53-.68.78-.44q.94-.32 1.78.12t1.1 1.32l1 2.93 5.8-1.93L13.57 5q-.31-.94.13-1.75T15 2.12q.45-.19.88-.12t.85.25.72.53.44.78l1 2.94 2.8-.94q.88-.31 1.73.1t1.12 1.3-.12 1.76-1.28 1.15l-2.88.94 1.94 5.63 2.8-.94q.63-.2 1.19-.1t1.03.5.66.97zm-15.88 3 5.88-2-1.94-5.63-5.82 1.94z',
    'slideshare' => 'M11.72 9.63q1.6 0 2.72 1.03t1.12 2.53-1.12 2.56-2.72 1.06T9 15.75t-1.13-2.56T9 10.66t2.72-1.03zm9 0q1.6 0 2.72 1.03t1.12 2.53-1.12 2.56-2.72 1.06-2.75-1.06-1.16-2.56 1.16-2.53 2.75-1.03zm9.78 5.62q.69-.5 1.06-.03t-.06 1.15q-2.19 2.63-6.63 4.5 1.88 6.38-1.5 9.63-2.12 2.06-4.62 1.38-2.38-.63-2.31-3.25v-5.88q-.2 0-.47-.1t-.4-.09v6.06q.06 2.57-2.32 3.25-2.5.7-4.63-1.37-3.37-3.25-1.43-9.69-4.5-1.81-6.7-4.5-.43-.62-.05-1.1t1.06.04q.06 0 .22.1t.22.15V3.12q0-1.3.84-2.21T4.81 0h22.44q.75 0 1.4.4t1.07 1.13.4 1.6V15.5zm-1.94 1.19V4.5q0-1-.25-1.63t-.78-.9T26 1.69H6.2q-1.5 0-2.03.6T3.63 4.5v12.06q1.93 1 4 1.44t3.34.38 2.53-.07q1.19-.06 1.69.44.62.63 1.25 1.13.12-1.7 2.12-1.57 1.2 0 2.5.07t3.44-.4 4.06-1.54z',
    'snapchat' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm10.63 21.19q.18-.57-.32-.69-.81-.13-1.56-.56t-1.19-.85-.84-.97-.5-.78-.16-.4h-.06q-.19-.5-.06-.82.12-.37 1.18-.68.25-.13.32-.13Q24 15.37 24 14.75q0-.5-.63-.75-.5-.13-.87 0-.5.25-.94.31-.19 0-.31-.12.19-3.75-.25-4.82-.5-1.12-1.35-1.87t-1.84-1.13T15.68 6h-.37q-.5 0-1.03.1t-1.38.4-1.62 1.06-1.22 1.81q-.5 1.07-.25 4.82-.19.12-.44.12-.44 0-1-.25-.25-.12-.53-.06t-.53.25-.25.44q-.19.69 1.06 1.12.07.07.35.16l.47.15.37.22.31.28q.13.32-.12.82-.06.18-.19.4t-.5.78-.84 1-1.2.85-1.52.53q-.38.06-.38.5.07.87 2.82 1.31.06.06.09.16t.06.25l.07.31.09.28q.06.38.5.38.12 0 .66-.1t.96-.1q1.13 0 2.32.88 1.75 1.2 3.06 1.13 1.38.06 3.06-1.13 1.2-.87 2.32-.87.5 0 1 .1t.62.09q.44 0 .5-.38.06-.12.13-.37t.1-.4.08-.23q2.44-.37 2.82-1.12z',
    'snapchat-ghost' => 'M31.94 24.56q-.5 1.13-4.25 1.7 0 .05-.07.3t-.15.6-.16.6q-.18.55-.75.55-.25 0-1-.15T24.13 28q-1.2 0-1.82.28t-1.75 1.03q-2.5 1.82-4.62 1.7-1.88.18-4.5-1.7-1.13-.75-1.75-1.03T7.87 28q-.62 0-1.43.16t-1 .15q-.57 0-.75-.56-.07-.19-.16-.56l-.16-.63-.06-.3q-1-.13-1.75-.32t-1.62-.66T0 24.25v-.06q0-.56.56-.63.88-.18 1.72-.53t1.44-.81 1.15-1.03.91-1.03.63-.97.37-.69l.16-.31v-.07q.37-.68.18-1.12-.12-.31-.43-.53t-.6-.31-.72-.22-.5-.2Q3.06 15 3.25 14q.12-.56.84-.88t1.22-.06q.88.38 1.5.38.38 0 .63-.13-.2-1.5-.2-3.78t.5-3.47q.76-1.62 1.91-2.75t2.41-1.6 2.06-.62T15.7.94h.62q1.69 0 3.16.56t2.75 1.72 2.03 2.84q.5 1.19.5 3.47t-.19 3.78q.25.13.56.13.63 0 1.38-.38.62-.31 1.31 0 .94.31.94 1.13 0 .87-1.63 1.56-.12.06-.43.12-1.57.5-1.82 1.13-.18.44.2 1.12v.07q.05.25.24.6t.78 1.15 1.25 1.47 1.78 1.3 2.32.85q.3.07.46.35t.04.65z',
    'snapchat-square' => 'M25 2q1.25 0 2.13.88T28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22zm-.38 19.69q.2-.57-.3-.69-.63-.06-1.2-.31t-.93-.57-.75-.68-.6-.7-.44-.65-.28-.44-.06-.21H20q-.2-.5-.07-.82.13-.37 1.2-.69.24-.12.3-.12 1.07-.44 1.07-1.06 0-.5-.63-.75-.5-.13-.87 0-.5.25-.94.3-.19 0-.31-.12.19-3.75-.25-4.8-.75-1.63-2.2-2.5T14.19 6h-.37q-.5 0-1.03.09t-1.38.4-1.62 1.07-1.22 1.81q-.5 1.07-.25 4.82-.19.12-.44.12-.44 0-1-.25-.31-.19-.78.03t-.53.6q-.19.68 1.06 1.12l.44.16.69.3.37.35q.13.32-.12.82-.06.18-.19.4t-.5.78-.84 1-1.2.85-1.52.53q-.38.06-.38.5.07.87 2.82 1.31.06.06.09.22t.1.4.12.38q.06.38.5.38.12 0 .65-.1t.97-.1q1.13 0 2.32.88Q12.69 26.07 14 26q1.37.06 3.06-1.13 1.19-.87 2.32-.87.5 0 1 .1t.62.09q.44 0 .5-.38.06-.12.13-.37t.09-.41.1-.22q2.43-.37 2.8-1.12z',
    'soundcloud' => [
        'p' => 'm6.94 16 .37 4.06-.37 4.32q0 .25-.25.25t-.25-.25l-.38-4.32.38-4.06q0-.25.25-.25t.25.25zm1.37-2.81q.32 0 .32.31l.3 6.56-.3 4.32q0 .31-.32.31T8 24.38l-.31-4.32L8 13.5q0-.31.31-.31zm1.56-1.5q.38 0 .38.31l.31 8.13-.31 4.25q0 .3-.38.3-.3 0-.3-.3l-.32-4.25.31-8.13q0-.31.31-.31zm-9.43 6q.12 0 .19.19l.3 2.18-.3 2.2q-.07.18-.2.18t-.12-.19L0 20.06l.31-2.18q0-.2.13-.2zm1.5-1.38q.12 0 .19.2l.43 3.55-.43 3.57q-.07.12-.2.12-.18 0-.18-.19l-.38-3.5.38-3.56q0-.19.19-.19zm1.56-.68q.19 0 .19.18l.44 4.25-.44 4.13q0 .19-.19.19t-.19-.2l-.37-4.12.37-4.25q0-.18.19-.18zm1.6-.13q.21 0 .21.19l.44 4.37-.44 4.25q0 .25-.22.25t-.22-.25l-.37-4.25.37-4.37q0-.19.22-.19zm15.09-6.94q.25.13.25.44l.25 11.13-.25 4q0 .5-.5.5-.57 0-.57-.5l-.25-4 .25-11.07V9q0-.25.2-.44.12-.12.37-.12.12 0 .25.12zm-1.69.94q.25.13.25.44l.19 10.19-.19 3.62v.44q0 .19-.19.31-.12.19-.31.19-.25 0-.44-.19-.06-.12-.06-.31l-.19-4.13.13-10.06.06-.06q0-.32.19-.44.18-.06.3-.06t.26.06zm-6.97 1.44q.34 0 .4.37l.32 8.75-.31 4.2q-.07.37-.38.37-.37 0-.37-.38l-.32-4.19.32-8.75q0-.37.34-.37zm23.53 3.87q2.07 0 3.5 1.47T40 19.75q0 2.06-1.44 3.5t-3.5 1.44H21.44q-.57-.06-.57-.63V8.5q0-.44.5-.62 1.5-.57 3.13-.57 3.44 0 5.9 2.28t2.79 5.66q.93-.44 1.87-.44zm-18.5-3.75q.44 0 .44.5l.25 8.57-.25 4.06q0 .5-.44.5-.5 0-.5-.5l-.18-4.06.18-8.57q0-.5.5-.5zm-3.37-.5q.37 0 .44.44l.25 9.06-.25 4.2q-.07.43-.44.43t-.38-.44l-.25-4.19.25-9.06q0-.44.38-.44zm1.69.25q.43 0 .43.44l.25 8.81-.25 4.2q0 .43-.43.43t-.44-.44l-.25-4.19.25-8.8q0-.45.44-.45z',
        'w' => 1.248,
        'vw' => 40
    ],
    'sourcetree' => [
        'p' => 'M25.38 12.69q0 4.12-2.44 7.47t-6.32 4.6v6.37q0 .37-.28.62t-.65.25h-6q-.38 0-.66-.25t-.28-.62v-6.38q-3.87-1.25-6.31-4.6T0 12.7q0-5.25 3.72-8.97T12.69 0t8.97 3.72 3.71 8.97zm-16.94 0q0 1.62.87 2.72t2.13 1.34 2.53 0 2.15-1.34.88-2.72q0-2-1.34-3.16t-2.94-1.15-2.94 1.15-1.34 3.16z',
        'w' => 0.792,
        'vw' => 25
    ],
    'space-awesome' => [
        'p' => 'M96 256H128V512H0V352H32V320H64V288H96V256zM512 352V512H384V256H416V288H448V320H480V352H512zM320 64H352V448H320V416H192V448H160V64H192V32H224V0H288V32H320V64zM288 128H224V192H288V128z',
        'vw' => 512,
        'vh' => 512
    ],
    'speakap' => [
        'p' => 'M3.44 24.5Q-.31 20.37 0 14.78t4.47-9.31 9.72-3.44 9.28 4.4 3.44 9.7-4.47 9.3q-.63.57 1.44 1.82 1.37.88 1.43 1.19 0 .19-.25.25-2.75 1.19-6.56 1.37t-8.1-1.18-6.96-4.38zm13.31-13.25 2.44-2.56q-1.82-1.63-4.75-1.7-2.57-.05-4.28 1.29T8.38 12q-.07 2.25 1.8 3.44.82.43 2.29.9t2.31.97.78 1.38q0 .87-.78 1.25t-1.6.31q-1.93 0-3-1.5l-2.5 2.63q2.13 2.06 5.26 2.06.37.06.87.03t1.5-.31 1.75-.75 1.35-1.54.6-2.5q.06-2.37-1.76-3.5-.81-.5-2.25-.93t-2.25-.9-.81-1.29q0-.63.56-1.03t1-.47.82-.06q1.62 0 2.43 1.06z',
        'w' => 0.839,
        'vw' => 27
    ],
    'speaker-deck' => 'M13.37 18.5H6.25a6.25 6.25 0 1 1 0-12.5h8.3a2.5 2.5 0 1 1 0 5H6.12c-1.65 0-1.65 2.5 0 2.5h7.12a6.25 6.25 0 1 1 0 12.5H2.5a2.5 2.5 0 0 1 0-5h10.87c1.65 0 1.65-2.5 0-2.5Zm5.26 7.5a7.51 7.51 0 0 0 3.19-5h4.04c.68 0 1.23-.57 1.22-1.25v-7.5A1.23 1.23 0 0 0 25.86 11h-7.33a3.8 3.8 0 0 0 0-5h8.56A4.96 4.96 0 0 1 32 11v10c0 2.76-2.2 5-4.91 5Zm0 0',
    'spotify' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm6.31 22.81q.38 0 .66-.28t.28-.65q0-.7-.63-1.07-6-3.56-14.8-1.62-.82.19-.82 1.06 0 .38.25.66t.69.28q.12 0 .75-.13 7.75-1.62 12.93 1.5.38.25.7.25zm1.69-4.12q.5 0 .84-.35t.35-.84q0-.81-.69-1.25-4.81-2.81-11.06-2.81-3.13 0-6.13.81-1 .31-1 1.31 0 .5.38.85t.87.34q.13 0 .75-.12 3.94-1 7.9-.47t6.98 2.28q.5.25.81.25zm1.94-4.75q.56 0 1-.4t.44-1.04q0-.94-.82-1.44-5.12-3-12.81-3-4.13 0-7.38 1-1.06.32-1.06 1.44 0 .63.4 1.06t1.04.44q.19 0 .81-.12 3.75-1.13 9-.63t8.56 2.44q.38.25.82.25z',
    'square-behance' => [
        'p' => 'M186.5 293c0 19.3-14 25.4-31.2 25.4h-45.1v-52.9h46c18.6.1 30.3 7.8 30.3 27.5zm-7.7-82.3c0-17.7-13.7-21.9-28.9-21.9h-39.6v44.8H153c15.1 0 25.8-6.6 25.8-22.9zm132.3 23.2c-18.3 0-30.5 11.4-31.7 29.7h62.2c-1.7-18.5-11.3-29.7-30.5-29.7zM448 80v352c0 26.5-21.5 48-48 48H48c-26.5 0-48-21.5-48-48V80c0-26.5 21.5-48 48-48h352c26.5 0 48 21.5 48 48zM271.7 185h77.8v-18.9h-77.8V185zm-43 110.3c0-24.1-11.4-44.9-35-51.6 17.2-8.2 26.2-17.7 26.2-37 0-38.2-28.5-47.5-61.4-47.5H68v192h93.1c34.9-.2 67.6-16.9 67.6-55.9zM380 280.5c0-41.1-24.1-75.4-67.6-75.4-42.4 0-71.1 31.8-71.1 73.6 0 43.3 27.3 73 71.1 73 33.2 0 54.7-14.9 65.1-46.8h-33.7c-3.7 11.9-18.6 18.1-30.2 18.1-22.4 0-34.1-13.1-34.1-35.3h100.2c.1-2.3.3-4.8.3-7.2z',
        'vw' => 448,
        'vh' => 512
    ],
    'square-dribbble' => [
        'p' => 'M90.2 228.2c8.9-42.4 37.4-77.7 75.7-95.7 3.6 4.9 28 38.8 50.7 79-64 17-120.3 16.8-126.4 16.7zM314.6 154c-33.6-29.8-79.3-41.1-122.6-30.6 3.8 5.1 28.6 38.9 51 80 48.6-18.3 69.1-45.9 71.6-49.4zM140.1 364c40.5 31.6 93.3 36.7 137.3 18-2-12-10-53.8-29.2-103.6-55.1 18.8-93.8 56.4-108.1 85.6zm98.8-108.2c-3.4-7.8-7.2-15.5-11.1-23.2C159.6 253 93.4 252.2 87.4 252c0 1.4-.1 2.8-.1 4.2 0 35.1 13.3 67.1 35.1 91.4 22.2-37.9 67.1-77.9 116.5-91.8zm34.9 16.3c17.9 49.1 25.1 89.1 26.5 97.4 30.7-20.7 52.5-53.6 58.6-91.6-4.6-1.5-42.3-12.7-85.1-5.8zm-20.3-48.4c4.8 9.8 8.3 17.8 12 26.8 45.5-5.7 90.7 3.4 95.2 4.4-.3-32.3-11.8-61.9-30.9-85.1-2.9 3.9-25.8 33.2-76.3 53.9zM448 80v352c0 26.5-21.5 48-48 48H48c-26.5 0-48-21.5-48-48V80c0-26.5 21.5-48 48-48h352c26.5 0 48 21.5 48 48zm-64 176c0-88.2-71.8-160-160-160S64 167.8 64 256s71.8 160 160 160 160-71.8 160-160z',
        'vw' => 448,
        'vh' => 512
    ],
    'square-facebook' => [
        'p' => 'M400 32H48A48 48 0 0 0 0 80v352a48 48 0 0 0 48 48h137.25V327.69h-63V256h63v-54.64c0-62.15 37-96.48 93.67-96.48 27.14 0 55.52 4.84 55.52 4.84v61h-31.27c-30.81 0-40.42 19.12-40.42 38.73V256h68.78l-11 71.69h-57.78V480H400a48 48 0 0 0 48-48V80a48 48 0 0 0-48-48z',
        'vw' => 448,
        'vh' => 512
    ],
    'square-font-awesome' => [
        'p' => 'M384.5,32.5h-320c-35.3,0-64,28.7-64,64v320c0,35.3,28.7,64,64,64h320c35.3,0,64-28.7,64-64v-320 C448.5,61.2,419.8,32.5,384.5,32.5z M336.5,312.5c-31.6,11.2-41.2,16-59.8,16c-31.4,0-43.2-16-74.6-16c-10.2,0-18.2,1.6-25.6,4v-32 c7.4-2.2,15.4-4,25.6-4c31.2,0,43.2,16,74.6,16c10.2,0,17.8-1.4,27.8-4.6v-96c-10,3.2-17.6,4.6-27.8,4.6c-31.4,0-43.2-16-74.6-16 c-25.4,0-37.4,10.4-57.6,14.4v153.6c0,8.8-7.2,16-16,16c-8.8,0-16-7.2-16-16v-192c0-8.8,7.2-16,16-16c8.8,0,16,7.2,16,16v6.4 c20.2-4,32.2-14.4,57.6-14.4c31.2,0,43.2,16,74.6,16c18.6,0,28.2-4.8,59.8-16V312.5z',
        'vw' => 448,
        'vh' => 512
    ],
    'square-font-awesome-stroke' => [
        'p' => 'M201.6,152c-25.4,0-37.4,10.4-57.6,14.4V160c0-8.8-7.2-16-16-16s-16,7.2-16,16v192c0,0.8,0.1,1.6,0.2,2.4 c0.1,0.4,0.1,0.8,0.2,1.2c1.6,7.1,8,12.4,15.6,12.4s14-5.3,15.6-12.4c0.1-0.4,0.2-0.8,0.2-1.2c0.1-0.8,0.2-1.6,0.2-2.4V198.4 c4-0.8,7.7-1.8,11.2-3c14.3-4.7,26-11.4,46.4-11.4c31.4,0,43.2,16,74.6,16c8.9,0,15.9-1.1,24.2-3.5c1.2-0.3,2.4-0.7,3.6-1.1v96 c-10,3.2-17.6,4.6-27.8,4.6c-31.4,0-43.4-16-74.6-16c-10.2,0-18.2,1.8-25.6,4v32c7.4-2.4,15.4-4,25.6-4c31.4,0,43.2,16,74.6,16 c18.6,0,28.2-4.8,59.8-16V152c-31.6,11.2-41.2,16-59.8,16C244.8,168,232.8,152,201.6,152z M384,32H64C28.7,32,0,60.7,0,96v320 c0,35.3,28.7,64,64,64h320c35.3,0,64-28.7,64-64V96C448,60.7,419.3,32,384,32z M416,416c0,17.6-14.4,32-32,32H64 c-17.6,0-32-14.4-32-32V96c0-17.6,14.4-32,32-32h320c17.6,0,32,14.4,32,32V416z',
        'vw' => 448,
        'vh' => 512
    ],
    'square-git' => [
        'p' => 'M100.59 334.24c48.57 3.31 58.95 2.11 58.95 11.94 0 20-65.55 20.06-65.55 1.52.01-5.09 3.29-9.4 6.6-13.46zm27.95-116.64c-32.29 0-33.75 44.47-.75 44.47 32.51 0 31.71-44.47.75-44.47zM448 80v352a48 48 0 0 1-48 48H48a48 48 0 0 1-48-48V80a48 48 0 0 1 48-48h352a48 48 0 0 1 48 48zm-227 69.31c0 14.49 8.38 22.88 22.86 22.88 14.74 0 23.13-8.39 23.13-22.88S258.62 127 243.88 127c-14.48 0-22.88 7.84-22.88 22.31zM199.18 195h-49.55c-25-6.55-81.56-4.85-81.56 46.75 0 18.8 9.4 32 21.85 38.11C74.23 294.23 66.8 301 66.8 310.6c0 6.87 2.79 13.22 11.18 16.76-8.9 8.4-14 14.48-14 25.92C64 373.35 81.53 385 127.52 385c44.22 0 69.87-16.51 69.87-45.73 0-36.67-28.23-35.32-94.77-39.38l8.38-13.43c17 4.74 74.19 6.23 74.19-42.43 0-11.69-4.83-19.82-9.4-25.67l23.38-1.78zm84.34 109.84l-13-1.78c-3.82-.51-4.07-1-4.07-5.09V192.52h-52.6l-2.79 20.57c15.75 5.55 17 4.86 17 10.17V298c0 5.62-.31 4.58-17 6.87v20.06h72.42zM384 315l-6.87-22.37c-40.93 15.37-37.85-12.41-37.85-16.73v-60.72h37.85v-25.41h-35.82c-2.87 0-2 2.52-2-38.63h-24.18c-2.79 27.7-11.68 38.88-34 41.42v22.62c20.47 0 19.82-.85 19.82 2.54v66.57c0 28.72 11.43 40.91 41.67 40.91 14.45 0 30.45-4.83 41.38-10.2z',
        'vw' => 448,
        'vh' => 512
    ],
    'square-github' => [
        'p' => 'M400 32H48C21.5 32 0 53.5 0 80v352c0 26.5 21.5 48 48 48h352c26.5 0 48-21.5 48-48V80c0-26.5-21.5-48-48-48zM277.3 415.7c-8.4 1.5-11.5-3.7-11.5-8 0-5.4.2-33 .2-55.3 0-15.6-5.2-25.5-11.3-30.7 37-4.1 76-9.2 76-73.1 0-18.2-6.5-27.3-17.1-39 1.7-4.3 7.4-22-1.7-45-13.9-4.3-45.7 17.9-45.7 17.9-13.2-3.7-27.5-5.6-41.6-5.6-14.1 0-28.4 1.9-41.6 5.6 0 0-31.8-22.2-45.7-17.9-9.1 22.9-3.5 40.6-1.7 45-10.6 11.7-15.6 20.8-15.6 39 0 63.6 37.3 69 74.3 73.1-4.8 4.3-9.1 11.7-10.6 22.3-9.5 4.3-33.8 11.7-48.3-13.9-9.1-15.8-25.5-17.1-25.5-17.1-16.2-.2-1.1 10.2-1.1 10.2 10.8 5 18.4 24.2 18.4 24.2 9.7 29.7 56.1 19.7 56.1 19.7 0 13.9.2 36.5.2 40.6 0 4.3-3 9.5-11.5 8-66-22.1-112.2-84.9-112.2-158.3 0-91.8 70.2-161.5 162-161.5S388 165.6 388 257.4c.1 73.4-44.7 136.3-110.7 158.3zm-98.1-61.1c-1.9.4-3.7-.4-3.9-1.7-.2-1.5 1.1-2.8 3-3.2 1.9-.2 3.7.6 3.9 1.9.3 1.3-1 2.6-3 3zm-9.5-.9c0 1.3-1.5 2.4-3.5 2.4-2.2.2-3.7-.9-3.7-2.4 0-1.3 1.5-2.4 3.5-2.4 1.9-.2 3.7.9 3.7 2.4zm-13.7-1.1c-.4 1.3-2.4 1.9-4.1 1.3-1.9-.4-3.2-1.9-2.8-3.2.4-1.3 2.4-1.9 4.1-1.5 2 .6 3.3 2.1 2.8 3.4zm-12.3-5.4c-.9 1.1-2.8.9-4.3-.6-1.5-1.3-1.9-3.2-.9-4.1.9-1.1 2.8-.9 4.3.6 1.3 1.3 1.8 3.3.9 4.1zm-9.1-9.1c-.9.6-2.6 0-3.7-1.5s-1.1-3.2 0-3.9c1.1-.9 2.8-.2 3.7 1.3 1.1 1.5 1.1 3.3 0 4.1zm-6.5-9.7c-.9.9-2.4.4-3.5-.6-1.1-1.3-1.3-2.8-.4-3.5.9-.9 2.4-.4 3.5.6 1.1 1.3 1.3 2.8.4 3.5zm-6.7-7.4c-.4.9-1.7 1.1-2.8.4-1.3-.6-1.9-1.7-1.5-2.6.4-.6 1.5-.9 2.8-.4 1.3.7 1.9 1.8 1.5 2.6z',
        'vw' => 448,
        'vh' => 512
    ],
    'square-gitlab' => [
        'p' => 'M48 32H400C426.5 32 448 53.5 448 80V432C448 458.5 426.5 480 400 480H48C21.5 480 0 458.5 0 432V80C0 53.5 21.5 32 48 32zM382.1 224.9L337.5 108.5C336.6 106.2 334.9 104.2 332.9 102.9C331.3 101.9 329.5 101.3 327.7 101.1C325.9 100.9 324 101.2 322.3 101.8C320.6 102.5 319 103.5 317.8 104.9C316.6 106.3 315.7 107.9 315.2 109.7L285 201.9H162.1L132.9 109.7C132.4 107.9 131.4 106.3 130.2 104.9C128.1 103.6 127.4 102.5 125.7 101.9C123.1 101.2 122.1 100.1 120.3 101.1C118.5 101.3 116.7 101.9 115.1 102.9C113.1 104.2 111.5 106.2 110.6 108.5L65.94 224.9L65.47 226.1C59.05 242.9 58.26 261.3 63.22 278.6C68.18 295.9 78.62 311.1 92.97 321.9L93.14 322L93.52 322.3L161.4 373.2L215.6 414.1C217.1 415.1 220.9 416.9 223.9 416.9C226.9 416.9 229.9 415.1 232.3 414.1L286.4 373.2L354.8 322L355 321.9C369.4 311 379.8 295.8 384.8 278.6C389.7 261.3 388.1 242.9 382.5 226.1L382.1 224.9z',
        'vw' => 448,
        'vh' => 512
    ],
    'square-google-plus' => [
        'p' => 'M400 32H48C21.5 32 0 53.5 0 80v352c0 26.5 21.5 48 48 48h352c26.5 0 48-21.5 48-48V80c0-26.5-21.5-48-48-48zM164 356c-55.3 0-100-44.7-100-100s44.7-100 100-100c27 0 49.5 9.8 67 26.2l-27.1 26.1c-7.4-7.1-20.3-15.4-39.8-15.4-34.1 0-61.9 28.2-61.9 63.2 0 34.9 27.8 63.2 61.9 63.2 39.6 0 54.4-28.5 56.8-43.1H164v-34.4h94.4c1 5 1.6 10.1 1.6 16.6 0 57.1-38.3 97.6-96 97.6zm220-81.8h-29v29h-29.2v-29h-29V245h29v-29H355v29h29v29.2z',
        'vw' => 448,
        'vh' => 512
    ],
    'square-hacker-news' => [
        'p' => 'M400 32H48C21.5 32 0 53.5 0 80v352c0 26.5 21.5 48 48 48h352c26.5 0 48-21.5 48-48V80c0-26.5-21.5-48-48-48zM21.2 229.2H21c.1-.1.2-.3.3-.4 0 .1 0 .3-.1.4zm218 53.9V384h-31.4V281.3L128 128h37.3c52.5 98.3 49.2 101.2 59.3 125.6 12.3-27 5.8-24.4 60.6-125.6H320l-80.8 155.1z',
        'vw' => 448,
        'vh' => 512
    ],
    'square-instagram' => [
        'p' => 'M224,202.66A53.34,53.34,0,1,0,277.36,256,53.38,53.38,0,0,0,224,202.66Zm124.71-41a54,54,0,0,0-30.41-30.41c-21-8.29-71-6.43-94.3-6.43s-73.25-1.93-94.31,6.43a54,54,0,0,0-30.41,30.41c-8.28,21-6.43,71.05-6.43,94.33S91,329.26,99.32,350.33a54,54,0,0,0,30.41,30.41c21,8.29,71,6.43,94.31,6.43s73.24,1.93,94.3-6.43a54,54,0,0,0,30.41-30.41c8.35-21,6.43-71.05,6.43-94.33S357.1,182.74,348.75,161.67ZM224,338a82,82,0,1,1,82-82A81.9,81.9,0,0,1,224,338Zm85.38-148.3a19.14,19.14,0,1,1,19.13-19.14A19.1,19.1,0,0,1,309.42,189.74ZM400,32H48A48,48,0,0,0,0,80V432a48,48,0,0,0,48,48H400a48,48,0,0,0,48-48V80A48,48,0,0,0,400,32ZM382.88,322c-1.29,25.63-7.14,48.34-25.85,67s-41.4,24.63-67,25.85c-26.41,1.49-105.59,1.49-132,0-25.63-1.29-48.26-7.15-67-25.85s-24.63-41.42-25.85-67c-1.49-26.42-1.49-105.61,0-132,1.29-25.63,7.07-48.34,25.85-67s41.47-24.56,67-25.78c26.41-1.49,105.59-1.49,132,0,25.63,1.29,48.33,7.15,67,25.85s24.63,41.42,25.85,67.05C384.37,216.44,384.37,295.56,382.88,322Z',
        'vw' => 448,
        'vh' => 512
    ],
    'square-js' => [
        'p' => 'M400 32H48C21.5 32 0 53.5 0 80v352c0 26.5 21.5 48 48 48h352c26.5 0 48-21.5 48-48V80c0-26.5-21.5-48-48-48zM243.8 381.4c0 43.6-25.6 63.5-62.9 63.5-33.7 0-53.2-17.4-63.2-38.5l34.3-20.7c6.6 11.7 12.6 21.6 27.1 21.6 13.8 0 22.6-5.4 22.6-26.5V237.7h42.1v143.7zm99.6 63.5c-39.1 0-64.4-18.6-76.7-43l34.3-19.8c9 14.7 20.8 25.6 41.5 25.6 17.4 0 28.6-8.7 28.6-20.8 0-14.4-11.4-19.5-30.7-28l-10.5-4.5c-30.4-12.9-50.5-29.2-50.5-63.5 0-31.6 24.1-55.6 61.6-55.6 26.8 0 46 9.3 59.8 33.7L368 290c-7.2-12.9-15-18-27.1-18-12.3 0-20.1 7.8-20.1 18 0 12.6 7.8 17.7 25.9 25.6l10.5 4.5c35.8 15.3 55.9 31 55.9 66.2 0 37.8-29.8 58.6-69.7 58.6z',
        'vw' => 448,
        'vh' => 512
    ],
    'square-lastfm' => [
        'p' => 'M400 32H48C21.5 32 0 53.5 0 80v352c0 26.5 21.5 48 48 48h352c26.5 0 48-21.5 48-48V80c0-26.5-21.5-48-48-48zm-92.2 312.9c-63.4 0-85.4-28.6-97.1-64.1-16.3-51-21.5-84.3-63-84.3-22.4 0-45.1 16.1-45.1 61.2 0 35.2 18 57.2 43.3 57.2 28.6 0 47.6-21.3 47.6-21.3l11.7 31.9s-19.8 19.4-61.2 19.4c-51.3 0-79.9-30.1-79.9-85.8 0-57.9 28.6-92 82.5-92 73.5 0 80.8 41.4 100.8 101.9 8.8 26.8 24.2 46.2 61.2 46.2 24.9 0 38.1-5.5 38.1-19.1 0-19.9-21.8-22-49.9-28.6-30.4-7.3-42.5-23.1-42.5-48 0-40 32.3-52.4 65.2-52.4 37.4 0 60.1 13.6 63 46.6l-36.7 4.4c-1.5-15.8-11-22.4-28.6-22.4-16.1 0-26 7.3-26 19.8 0 11 4.8 17.6 20.9 21.3 32.7 7.1 71.8 12 71.8 57.5.1 36.7-30.7 50.6-76.1 50.6z',
        'vw' => 448,
        'vh' => 512
    ],
    'square-odnoklassniki' => [
        'p' => 'M184.2 177.1c0-22.1 17.9-40 39.8-40s39.8 17.9 39.8 40c0 22-17.9 39.8-39.8 39.8s-39.8-17.9-39.8-39.8zM448 80v352c0 26.5-21.5 48-48 48H48c-26.5 0-48-21.5-48-48V80c0-26.5 21.5-48 48-48h352c26.5 0 48 21.5 48 48zm-305.1 97.1c0 44.6 36.4 80.9 81.1 80.9s81.1-36.2 81.1-80.9c0-44.8-36.4-81.1-81.1-81.1s-81.1 36.2-81.1 81.1zm174.5 90.7c-4.6-9.1-17.3-16.8-34.1-3.6 0 0-22.7 18-59.3 18s-59.3-18-59.3-18c-16.8-13.2-29.5-5.5-34.1 3.6-7.9 16.1 1.1 23.7 21.4 37 17.3 11.1 41.2 15.2 56.6 16.8l-12.9 12.9c-18.2 18-35.5 35.5-47.7 47.7-17.6 17.6 10.7 45.8 28.4 28.6l47.7-47.9c18.2 18.2 35.7 35.7 47.7 47.9 17.6 17.2 46-10.7 28.6-28.6l-47.7-47.7-13-12.9c15.5-1.6 39.1-5.9 56.2-16.8 20.4-13.3 29.3-21 21.5-37z',
        'vw' => 448,
        'vh' => 512
    ],
    'square-pied-piper' => [
        'p' => 'M32 419L0 479.2l.8-328C.8 85.3 54 32 120 32h327.2c-93 28.9-189.9 94.2-253.9 168.6C122.7 282 82.6 338 32 419M448 32S305.2 98.8 261.6 199.1c-23.2 53.6-28.9 118.1-71 158.6-28.9 27.8-69.8 38.2-105.3 56.3-23.2 12-66.4 40.5-84.9 66h328.4c66 0 119.3-53.3 119.3-119.2-.1 0-.1-328.8-.1-328.8z',
        'vw' => 448,
        'vh' => 512
    ],
    'square-pinterest' => [
        'p' => 'M448 80v352c0 26.5-21.5 48-48 48H154.4c9.8-16.4 22.4-40 27.4-59.3 3-11.5 15.3-58.4 15.3-58.4 8 15.3 31.4 28.2 56.3 28.2 74.1 0 127.4-68.1 127.4-152.7 0-81.1-66.2-141.8-151.4-141.8-106 0-162.2 71.1-162.2 148.6 0 36 19.2 80.8 49.8 95.1 4.7 2.2 7.1 1.2 8.2-3.3.8-3.4 5-20.1 6.8-27.8.6-2.5.3-4.6-1.7-7-10.1-12.3-18.3-34.9-18.3-56 0-54.2 41-106.6 110.9-106.6 60.3 0 102.6 41.1 102.6 99.9 0 66.4-33.5 112.4-77.2 112.4-24.1 0-42.1-19.9-36.4-44.4 6.9-29.2 20.3-60.7 20.3-81.8 0-53-75.5-45.7-75.5 25 0 21.7 7.3 36.5 7.3 36.5-31.4 132.8-36.1 134.5-29.6 192.6l2.2.8H48c-26.5 0-48-21.5-48-48V80c0-26.5 21.5-48 48-48h352c26.5 0 48 21.5 48 48z',
        'vw' => 448,
        'vh' => 512
    ],
    'square-reddit' => [
        'p' => 'M283.2 345.5c2.7 2.7 2.7 6.8 0 9.2-24.5 24.5-93.8 24.6-118.4 0-2.7-2.4-2.7-6.5 0-9.2 2.4-2.4 6.5-2.4 8.9 0 18.7 19.2 81 19.6 100.5 0 2.4-2.3 6.6-2.3 9 0zm-91.3-53.8c0-14.9-11.9-26.8-26.5-26.8-14.9 0-26.8 11.9-26.8 26.8 0 14.6 11.9 26.5 26.8 26.5 14.6 0 26.5-11.9 26.5-26.5zm90.7-26.8c-14.6 0-26.5 11.9-26.5 26.8 0 14.6 11.9 26.5 26.5 26.5 14.9 0 26.8-11.9 26.8-26.5 0-14.9-11.9-26.8-26.8-26.8zM448 80v352c0 26.5-21.5 48-48 48H48c-26.5 0-48-21.5-48-48V80c0-26.5 21.5-48 48-48h352c26.5 0 48 21.5 48 48zm-99.7 140.6c-10.1 0-19 4.2-25.6 10.7-24.1-16.7-56.5-27.4-92.5-28.6l18.7-84.2 59.5 13.4c0 14.6 11.9 26.5 26.5 26.5 14.9 0 26.8-12.2 26.8-26.8 0-14.6-11.9-26.8-26.8-26.8-10.4 0-19.3 6.2-23.8 14.9l-65.7-14.6c-3.3-.9-6.5 1.5-7.4 4.8l-20.5 92.8c-35.7 1.5-67.8 12.2-91.9 28.9-6.5-6.8-15.8-11-25.9-11-37.5 0-49.8 50.4-15.5 67.5-1.2 5.4-1.8 11-1.8 16.7 0 56.5 63.7 102.3 141.9 102.3 78.5 0 142.2-45.8 142.2-102.3 0-5.7-.6-11.6-2.1-17 33.6-17.2 21.2-67.2-16.1-67.2z',
        'vw' => 448,
        'vh' => 512
    ],
    'square-snapchat' => [
        'p' => 'M384,32H64A64,64,0,0,0,0,96V416a64,64,0,0,0,64,64H384a64,64,0,0,0,64-64V96A64,64,0,0,0,384,32Zm-3.907,319.309-.083.1a32.364,32.364,0,0,1-8.717,6.823,90.26,90.26,0,0,1-20.586,8.2,12.694,12.694,0,0,0-3.852,1.76c-2.158,1.909-2.1,4.64-4.4,8.55a23.137,23.137,0,0,1-6.84,7.471c-6.707,4.632-14.244,4.923-22.23,5.23-7.214.274-15.39.581-24.729,3.669-3.761,1.245-7.753,3.694-12.377,6.533-11.265,6.9-26.68,16.353-52.3,16.353s-40.925-9.4-52.106-16.279c-4.657-2.888-8.675-5.362-12.543-6.64-9.339-3.08-17.516-3.4-24.729-3.67-7.986-.307-15.523-.6-22.231-5.229a23.085,23.085,0,0,1-6.01-6.11c-3.2-4.632-2.855-7.8-5.254-9.895a13.428,13.428,0,0,0-4.1-1.834,89.986,89.986,0,0,1-20.313-8.127,32.905,32.905,0,0,1-8.3-6.284c-6.583-6.757-8.276-14.776-5.686-21.824,3.436-9.338,11.571-12.111,19.4-16.262,14.776-8.027,26.348-18.055,34.433-29.884a68.236,68.236,0,0,0,5.985-10.567c.789-2.158.772-3.329.241-4.416a7.386,7.386,0,0,0-2.208-2.217c-2.532-1.676-5.113-3.353-6.882-4.5-3.27-2.141-5.868-3.818-7.529-4.98-6.267-4.383-10.65-9.04-13.4-14.245a28.4,28.4,0,0,1-1.369-23.584c4.134-10.924,14.469-17.706,26.978-17.706a37.141,37.141,0,0,1,7.845.83c.689.15,1.37.307,2.042.482-.108-7.43.058-15.357.722-23.119,2.358-27.261,11.912-41.589,21.874-52.994a86.836,86.836,0,0,1,22.28-17.931C188.254,100.383,205.312,96,224,96s35.828,4.383,50.944,13.016a87.169,87.169,0,0,1,22.239,17.9c9.961,11.406,19.516,25.709,21.874,52.995a231.194,231.194,0,0,1,.713,23.118c.673-.174,1.362-.332,2.051-.481a37.131,37.131,0,0,1,7.844-.83c12.5,0,22.82,6.782,26.971,17.706a28.37,28.37,0,0,1-1.4,23.559c-2.74,5.2-7.123,9.861-13.39,14.244-1.668,1.187-4.258,2.864-7.529,4.981-1.835,1.187-4.541,2.947-7.164,4.682a6.856,6.856,0,0,0-1.951,2.034c-.506,1.046-.539,2.191.166,4.208a69.015,69.015,0,0,0,6.085,10.792c8.268,12.1,20.188,22.313,35.454,30.407,1.486.772,2.98,1.5,4.441,2.258.722.332,1.569.763,2.491,1.3,4.9,2.723,9.2,6.01,11.455,12.153C387.821,336.915,386.269,344.7,380.093,351.309Zm-16.719-18.461c-50.313-24.314-58.332-61.918-58.689-64.749-.431-3.379-.921-6.035,2.806-9.472,3.594-3.328,19.541-13.19,23.965-16.278,7.33-5.114,10.534-10.219,8.16-16.495-1.66-4.316-5.686-5.976-9.961-5.976a18.5,18.5,0,0,0-3.993.448c-8.035,1.743-15.838,5.769-20.354,6.857a7.1,7.1,0,0,1-1.66.224c-2.408,0-3.279-1.071-3.088-3.968.564-8.783,1.759-25.925.373-41.937-1.884-22.032-8.99-32.948-17.432-42.6-4.051-4.624-23.135-24.654-59.536-24.654S168.53,134.359,164.479,139c-8.434,9.654-15.531,20.57-17.432,42.6-1.386,16.013-.141,33.147.373,41.937.166,2.756-.68,3.968-3.088,3.968a7.1,7.1,0,0,1-1.66-.224c-4.507-1.087-12.31-5.113-20.346-6.856a18.494,18.494,0,0,0-3.993-.449c-4.25,0-8.3,1.636-9.961,5.977-2.374,6.276.847,11.381,8.168,16.494,4.425,3.088,20.371,12.958,23.966,16.279,3.719,3.437,3.237,6.093,2.805,9.471-.356,2.79-8.384,40.394-58.689,64.749-2.946,1.428-7.96,4.45.88,9.331,13.88,7.628,23.111,6.807,30.3,11.43,6.093,3.927,2.5,12.394,6.923,15.449,5.454,3.76,21.583-.266,42.335,6.6,17.433,5.744,28.116,22.015,58.963,22.015s41.788-16.3,58.938-21.973c20.795-6.865,36.89-2.839,42.336-6.6,4.433-3.055.822-11.522,6.923-15.448,7.181-4.624,16.411-3.8,30.3-11.472C371.36,337.355,366.346,334.333,363.374,332.848Z',
        'vw' => 448,
        'vh' => 512
    ],
    'square-steam' => [
        'p' => 'M185.2 356.5c7.7-18.5-1-39.7-19.6-47.4l-29.5-12.2c11.4-4.3 24.3-4.5 36.4.5 12.2 5.1 21.6 14.6 26.7 26.7 5 12.2 5 25.6-.1 37.7-10.5 25.1-39.4 37-64.6 26.5-11.6-4.8-20.4-13.6-25.4-24.2l28.5 11.8c18.6 7.8 39.9-.9 47.6-19.4zM400 32H48C21.5 32 0 53.5 0 80v160.7l116.6 48.1c12-8.2 26.2-12.1 40.7-11.3l55.4-80.2v-1.1c0-48.2 39.3-87.5 87.6-87.5s87.6 39.3 87.6 87.5c0 49.2-40.9 88.7-89.6 87.5l-79 56.3c1.6 38.5-29.1 68.8-65.7 68.8-31.8 0-58.5-22.7-64.5-52.7L0 319.2V432c0 26.5 21.5 48 48 48h352c26.5 0 48-21.5 48-48V80c0-26.5-21.5-48-48-48zm-99.7 222.5c-32.2 0-58.4-26.1-58.4-58.3s26.2-58.3 58.4-58.3 58.4 26.2 58.4 58.3-26.2 58.3-58.4 58.3zm.1-14.6c24.2 0 43.9-19.6 43.9-43.8 0-24.2-19.6-43.8-43.9-43.8-24.2 0-43.9 19.6-43.9 43.8 0 24.2 19.7 43.8 43.9 43.8z',
        'vw' => 448,
        'vh' => 512
    ],
    'square-threads' => [
        'p' => 'M64 32C28.7 32 0 60.7 0 96V416c0 35.3 28.7 64 64 64H384c35.3 0 64-28.7 64-64V96c0-35.3-28.7-64-64-64H64zM294.2 244.3c19.5 9.3 33.7 23.5 41.2 40.9c10.4 24.3 11.4 63.9-20.2 95.4c-24.2 24.1-53.5 35-95.1 35.3h-.2c-46.8-.3-82.8-16.1-106.9-46.8C91.5 341.8 80.4 303.7 80 256v-.1-.1c.4-47.7 11.5-85.7 33-113.1c24.2-30.7 60.2-46.5 106.9-46.8h.2c46.9 .3 83.3 16 108.2 46.6c12.3 15.1 21.3 33.3 27 54.4l-26.9 7.2c-4.7-17.2-11.9-31.9-21.4-43.6c-19.4-23.9-48.7-36.1-87-36.4c-38 .3-66.8 12.5-85.5 36.2c-17.5 22.3-26.6 54.4-26.9 95.5c.3 41.1 9.4 73.3 26.9 95.5c18.7 23.8 47.4 36 85.5 36.2c34.3-.3 56.9-8.4 75.8-27.3c21.5-21.5 21.1-47.9 14.2-64c-4-9.4-11.4-17.3-21.3-23.3c-2.4 18-7.9 32.2-16.5 43.2c-11.4 14.5-27.7 22.4-48.4 23.5c-15.7 .9-30.8-2.9-42.6-10.7c-13.9-9.2-22-23.2-22.9-39.5c-1.7-32.2 23.8-55.3 63.5-57.6c14.1-.8 27.3-.2 39.5 1.9c-1.6-9.9-4.9-17.7-9.8-23.4c-6.7-7.8-17.1-11.8-30.8-11.9h-.4c-11 0-26 3.1-35.6 17.6l-23-15.8c12.8-19.4 33.6-30.1 58.5-30.1h.6c41.8 .3 66.6 26.3 69.1 71.8c1.4 .6 2.8 1.2 4.2 1.9l.1 .5zm-71.8 67.5c17-.9 36.4-7.6 39.7-48.8c-8.8-1.9-18.6-2.9-29-2.9c-3.2 0-6.4 .1-9.6 .3c-28.6 1.6-38.1 15.5-37.4 27.9c.9 16.7 19 24.5 36.4 23.6l-.1-.1z',
        'vw' => 448,
        'vh' => 512
    ],
    'square-tumblr' => [
        'p' => 'M400 32H48C21.5 32 0 53.5 0 80v352c0 26.5 21.5 48 48 48h352c26.5 0 48-21.5 48-48V80c0-26.5-21.5-48-48-48zm-82.3 364.2c-8.5 9.1-31.2 19.8-60.9 19.8-75.5 0-91.9-55.5-91.9-87.9v-90h-29.7c-3.4 0-6.2-2.8-6.2-6.2v-42.5c0-4.5 2.8-8.5 7.1-10 38.8-13.7 50.9-47.5 52.7-73.2.5-6.9 4.1-10.2 10-10.2h44.3c3.4 0 6.2 2.8 6.2 6.2v72h51.9c3.4 0 6.2 2.8 6.2 6.2v51.1c0 3.4-2.8 6.2-6.2 6.2h-52.1V321c0 21.4 14.8 33.5 42.5 22.4 3-1.2 5.6-2 8-1.4 2.2.5 3.6 2.1 4.6 4.9l13.8 40.2c1 3.2 2 6.7-.3 9.1z',
        'vw' => 448,
        'vh' => 512
    ],
    'square-twitter' => [
        'p' => 'M64 32C28.7 32 0 60.7 0 96V416c0 35.3 28.7 64 64 64H384c35.3 0 64-28.7 64-64V96c0-35.3-28.7-64-64-64H64zM351.3 199.3v0c0 86.7-66 186.6-186.6 186.6c-37.2 0-71.7-10.8-100.7-29.4c5.3 .6 10.4 .8 15.8 .8c30.7 0 58.9-10.4 81.4-28c-28.8-.6-53-19.5-61.3-45.5c10.1 1.5 19.2 1.5 29.6-1.2c-30-6.1-52.5-32.5-52.5-64.4v-.8c8.7 4.9 18.9 7.9 29.6 8.3c-9-6-16.4-14.1-21.5-23.6s-7.8-20.2-7.7-31c0-12.2 3.2-23.4 8.9-33.1c32.3 39.8 80.8 65.8 135.2 68.6c-9.3-44.5 24-80.6 64-80.6c18.9 0 35.9 7.9 47.9 20.7c14.8-2.8 29-8.3 41.6-15.8c-4.9 15.2-15.2 28-28.8 36.1c13.2-1.4 26-5.1 37.8-10.2c-8.9 13.1-20.1 24.7-32.9 34c.2 2.8 .2 5.7 .2 8.5z',
        'vw' => 448,
        'vh' => 512
    ],
    'square-viadeo' => [
        'p' => 'M400 32H48C21.5 32 0 53.5 0 80v352c0 26.5 21.5 48 48 48h352c26.5 0 48-21.5 48-48V80c0-26.5-21.5-48-48-48zM280.7 381.2c-42.4 46.2-120 46.6-162.4 0-68-73.6-19.8-196.1 81.2-196.1 13.3 0 26.6 2.1 39.1 6.7-4.3 8.4-7.3 17.6-8.4 27.1-9.7-4.1-20.2-6-30.7-6-48.8 0-84.6 41.7-84.6 88.9 0 43 28.5 78.7 69.5 85.9 61.5-24 72.9-117.6 72.9-175 0-7.3 0-14.8-.6-22.1-11.2-32.9-26.6-64.6-44.2-94.5 27.1 18.3 41.9 62.5 44.2 94.1v.4c7.7 22.5 11.8 46.2 11.8 70 0 54.1-21.9 99-68.3 128.2l-2.4.2c50 1 86.2-38.6 86.2-87.2 0-12.2-2.1-24.3-6.9-35.7 9.5-1.9 18.5-5.6 26.4-10.5 15.3 36.6 12.6 87.3-22.8 125.6zM309 233.7c-13.3 0-25.1-7.1-34.4-16.1 21.9-12 49.6-30.7 62.3-53 1.5-3 4.1-8.6 4.5-12-12.5 27.9-44.2 49.8-73.9 56.7-4.7-7.3-7.5-15.5-7.5-24.3 0-10.3 5.2-24.1 12.9-31.6 21.6-20.5 53-8.5 72.4-50 32.5 46.2 13.1 130.3-36.3 130.3z',
        'vw' => 448,
        'vh' => 512
    ],
    'square-vimeo' => [
        'p' => 'M400 32H48C21.5 32 0 53.5 0 80v352c0 26.5 21.5 48 48 48h352c26.5 0 48-21.5 48-48V80c0-26.5-21.5-48-48-48zm-16.2 149.6c-1.4 31.1-23.2 73.8-65.3 127.9-43.5 56.5-80.3 84.8-110.4 84.8-18.7 0-34.4-17.2-47.3-51.6-25.2-92.3-35.9-146.4-56.7-146.4-2.4 0-10.8 5-25.1 15.1L64 192c36.9-32.4 72.1-68.4 94.1-70.4 24.9-2.4 40.2 14.6 46 51.1 20.5 129.6 29.6 149.2 66.8 90.5 13.4-21.2 20.6-37.2 21.5-48.3 3.4-32.8-25.6-30.6-45.2-22.2 15.7-51.5 45.8-76.5 90.1-75.1 32.9 1 48.4 22.4 46.5 64z',
        'vw' => 448,
        'vh' => 512
    ],
    'square-whatsapp' => [
        'p' => 'M224 122.8c-72.7 0-131.8 59.1-131.9 131.8 0 24.9 7 49.2 20.2 70.1l3.1 5-13.3 48.6 49.9-13.1 4.8 2.9c20.2 12 43.4 18.4 67.1 18.4h.1c72.6 0 133.3-59.1 133.3-131.8 0-35.2-15.2-68.3-40.1-93.2-25-25-58-38.7-93.2-38.7zm77.5 188.4c-3.3 9.3-19.1 17.7-26.7 18.8-12.6 1.9-22.4.9-47.5-9.9-39.7-17.2-65.7-57.2-67.7-59.8-2-2.6-16.2-21.5-16.2-41s10.2-29.1 13.9-33.1c3.6-4 7.9-5 10.6-5 2.6 0 5.3 0 7.6.1 2.4.1 5.7-.9 8.9 6.8 3.3 7.9 11.2 27.4 12.2 29.4s1.7 4.3.3 6.9c-7.6 15.2-15.7 14.6-11.6 21.6 15.3 26.3 30.6 35.4 53.9 47.1 4 2 6.3 1.7 8.6-1 2.3-2.6 9.9-11.6 12.5-15.5 2.6-4 5.3-3.3 8.9-2 3.6 1.3 23.1 10.9 27.1 12.9s6.6 3 7.6 4.6c.9 1.9.9 9.9-2.4 19.1zM400 32H48C21.5 32 0 53.5 0 80v352c0 26.5 21.5 48 48 48h352c26.5 0 48-21.5 48-48V80c0-26.5-21.5-48-48-48zM223.9 413.2c-26.6 0-52.7-6.7-75.8-19.3L64 416l22.5-82.2c-13.9-24-21.2-51.3-21.2-79.3C65.4 167.1 136.5 96 223.9 96c42.4 0 82.2 16.5 112.2 46.5 29.9 30 47.9 69.8 47.9 112.2 0 87.4-72.7 158.5-160.1 158.5z',
        'vw' => 448,
        'vh' => 512
    ],
    'square-x-twitter' => [
        'p' => 'M64 32C28.7 32 0 60.7 0 96V416c0 35.3 28.7 64 64 64H384c35.3 0 64-28.7 64-64V96c0-35.3-28.7-64-64-64H64zm297.1 84L257.3 234.6 379.4 396H283.8L209 298.1 123.3 396H75.8l111-126.9L69.7 116h98l67.7 89.5L313.6 116h47.5zM323.3 367.6L153.4 142.9H125.1L296.9 367.6h26.3z',
        'vw' => 448,
        'vh' => 512
    ],
    'square-xing' => [
        'p' => 'M400 32H48C21.5 32 0 53.5 0 80v352c0 26.5 21.5 48 48 48h352c26.5 0 48-21.5 48-48V80c0-26.5-21.5-48-48-48zM140.4 320.2H93.8c-5.5 0-8.7-5.3-6-10.3l49.3-86.7c.1 0 .1-.1 0-.2l-31.4-54c-3-5.6.2-10.1 6-10.1h46.6c5.2 0 9.5 2.9 12.9 8.7l31.9 55.3c-1.3 2.3-18 31.7-50.1 88.2-3.5 6.2-7.7 9.1-12.6 9.1zm219.7-214.1L257.3 286.8v.2l65.5 119c2.8 5.1.1 10.1-6 10.1h-46.6c-5.5 0-9.7-2.9-12.9-8.7l-66-120.3c2.3-4.1 36.8-64.9 103.4-182.3 3.3-5.8 7.4-8.7 12.5-8.7h46.9c5.7-.1 8.8 4.7 6 10z',
        'vw' => 448,
        'vh' => 512
    ],
    'square-youtube' => [
        'p' => 'M186.8 202.1l95.2 54.1-95.2 54.1V202.1zM448 80v352c0 26.5-21.5 48-48 48H48c-26.5 0-48-21.5-48-48V80c0-26.5 21.5-48 48-48h352c26.5 0 48 21.5 48 48zm-42 176.3s0-59.6-7.6-88.2c-4.2-15.8-16.5-28.2-32.2-32.4C337.9 128 224 128 224 128s-113.9 0-142.2 7.7c-15.7 4.2-28 16.6-32.2 32.4-7.6 28.5-7.6 88.2-7.6 88.2s0 59.6 7.6 88.2c4.2 15.8 16.5 27.7 32.2 31.9C110.1 384 224 384 224 384s113.9 0 142.2-7.7c15.7-4.2 28-16.1 32.2-31.9 7.6-28.5 7.6-88.1 7.6-88.1z',
        'vw' => 448,
        'vh' => 512
    ],
    'squarespace' => 'm11.63 21.44 9.8-9.82q1.82-1.8 4.38-1.8t4.38 1.8T32 16t-1.81 4.38l-7.44 7.43q-.94.94-2.22.94t-2.16-.94L28 18.2q.94-.94.94-2.19t-.9-2.16-2.2-.9-2.21.87l-9.82 9.82q-.44.5-1.1.5t-1.09-.47-.43-1.1.43-1.12zM26.9 14.9q.46.46.46 1.09t-.43 1.06l-9.88 9.88q-1.75 1.81-4.31 1.81t-4.38-1.81q-.5-.5-.5-1.13t.47-1.1 1.1-.46 1.12.5q.88.87 2.16.87t2.22-.87l9.81-9.81q.44-.5 1.06-.5t1.1.47zm-16.35 5.46Q8.75 22.2 6.2 22.2t-4.4-1.83T0 16t1.81-4.38L9.25 4.2q.94-.94 2.22-.94t2.16.94L4 13.8q-.94.94-.94 2.19t.9 2.16 2.2.9 2.22-.87l9.8-9.82q.45-.43 1.1-.43t1.1.43.44 1.07-.44 1.12zM5.1 17.1q-.47-.46-.47-1.09t.44-1.06l9.88-9.88q1.75-1.81 4.31-1.81t4.38 1.81q.43.5.43 1.13t-.43 1.1-1.07.46-1.12-.5q-.88-.88-2.16-.88t-2.22.88l-9.81 9.81q-.44.5-1.06.5t-1.1-.47z',
    'stack-exchange' => 'M1.13 20.75h25.75v1.38q0 1.8-1.2 3.03t-2.87 1.21h-1.19L16.2 32v-5.63h-11q-1.7 0-2.88-1.21t-1.19-3.04v-1.37zm0-1.44V14h25.75v5.31H1.13zm0-6.87V7.13h25.75v5.3H1.13zM22.8 0q1.7 0 2.88 1.25t1.19 3v1.38H1.13V4.25q0-1.75 1.18-3T5.2 0h17.6z',
    'stack-overflow' => [
        'p' => 'm18.19 19.44-.57 2.44L5.45 19.3l.5-2.44zM21.37 14l-1.62 1.94-9.56-8.06L11.75 6zm-2 2.5-1 2.25L7 13.5l1.06-2.31zm-3-14.5 7.44 10-2 1.5-7.43-10zm1.25 20.5V25H5.13v-2.5h12.5zm2.5 5V20h2.5v10H.2V20h2.5v7.5h17.43z',
        'w' => 0.75,
        'vw' => 24
    ],
    'stackpath' => [
        'p' => 'M15.29 14.52c0 .53-.27 1.28-1.34 1.28h-1.22v-2.59h1.22c1.07 0 1.34.77 1.34 1.31ZM28 2v28H0V2ZM9.46 17.99c0-1.33-.76-2.16-2.92-2.8-1.29-.47-1.63-.69-1.63-1.17s.44-.91 1.28-.91c.88 0 1.3.53 1.3 1.15H9.4v-.04c.04-1.22-.93-2.6-3.19-2.6-1.46 0-3.28.67-3.28 2.4 0 1.2.58 1.94 3.17 2.77 1.08.38 1.37.64 1.37 1.21 0 .95-1.2.9-1.22.9-1.28 0-1.6-.58-1.6-1.38H2.73v.04c-.05 1.96 1.77 2.82 3.53 2.82 1.88 0 3.2-.84 3.2-2.4Zm7.83-3.48c0-1.58-1.15-2.84-3.34-2.84h-3.23v8.64h2v-2.96h1.23c1.9 0 3.34-1 3.34-2.84Zm1.33 5.8 3.07-8.64h-1.94l-3 8.64Zm6.66-8.64h-1.94l-3 8.64h1.87Zm0 0',
        'vw' => 28
    ],
    'staylinked' => [
        'p' => 'm23.94 18.25 2.75 2.63q.44.37-.07.75L14.26 31.88q-.25.18-.53.21t-.4-.09l-.13-.06L.13 19.69q-.38-.44.12-.82l2.81-2.12q.5-.38.88 0l9.5 9.25q.44.37.87 0l5.82-4.63q.5-.37.06-.75l-5.25-5.12q-.38-.38-.84 0t-.91 0l-4.25-4.06q-.38-.38.06-.75l3.63-2.75q.43-.32.81.06l10.63 10.44zm3.37-5.19q.44.38-.06.75l-2.81 2.25q-.5.38-.88 0L13.5 6.13q-.38-.44-.88-.07l-5.75 4.5q-.43.38-.06.75l5.94 5.75q.37.38.87 0t.88 0l3.56 3.44q.44.38-.06.75l-.44.38-3.12 2.43q-.5.38-.94 0L2.8 13.75.12 11.12q-.37-.37.13-.74L13.12.24q.57-.44 1.07-.13z',
        'w' => 0.857,
        'vw' => 27
    ],
    'steam' => 'M31 16q0 4.19-2.06 7.78t-5.66 5.66-7.78 2.06q-5.31 0-9.44-3.16t-5.5-8.09l5.94 2.44q.31 1.5 1.5 2.5t2.81 1q1.82 0 3.13-1.35t1.25-3.22l5.25-3.75q2.5.07 4.25-1.68t1.75-4.2-1.72-4.12-4.13-1.68-4.12 1.68T14.75 12v.06l-3.69 5.38q-1.5-.07-2.75.75L0 14.75q.5-6 4.94-10.13T15.5.5q6.44 0 10.97 4.53T31 16zM9.75 24q.94.38 1.84 0t1.32-1.31 0-1.85-1.28-1.28l-2-.81q1.25-.5 2.43 0 1.25.56 1.82 1.81.5 1.25-.03 2.5t-1.78 1.78-2.57.03q-1.12-.5-1.69-1.62zm10.84-8.13q-1.59 0-2.75-1.12T16.7 12t1.15-2.75 2.75-1.13 2.75 1.13T24.5 12t-1.16 2.75-2.75 1.13zm0-.93q1.22 0 2.07-.88T23.5 12t-.84-2.06-2.07-.88-2.06.88T17.7 12t.84 2.06 2.06.88z',
    'steam-square' => 'M11.56 22.28q.38-.84.03-1.72t-1.21-1.25l-1.88-.75q1.13-.44 2.25 0 1.19.5 1.69 1.69t0 2.37-1.69 1.66-2.37-.03q-1.07-.44-1.57-1.5l1.82.75q.87.37 1.71 0t1.22-1.22zM25 2q1.25 0 2.13.88T28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27v-7.06l5.56 2.31q.2.94.75 1.69t1.44 1.19 1.88.43q1.68 0 2.9-1.25t1.16-3.06l4.94-3.5q2.3.06 3.97-1.56t1.65-3.94q0-2.25-1.6-3.84t-3.87-1.6-3.87 1.6-1.6 3.84v.06l-3.5 5.07q-1.37-.13-2.5.68l-7.31-3V5q0-1.25.88-2.12T3 2h22zm-6.25 13.88q-1.5 0-2.56-1.07t-1.07-2.56 1.07-2.56 2.56-1.07 2.6 1.07 1.09 2.56-1.1 2.56-2.59 1.06zm0-.88q1.19 0 1.97-.81t.78-1.94-.78-1.94-1.94-.81-1.97.81-.81 1.94q0 .75.37 1.37t1 1 1.38.38z',
    'steam-symbol' => 'M24.75 11.1q0 1.59-1.13 2.68t-2.71 1.1-2.7-1.1-1.09-2.69 1.1-2.72 2.69-1.12 2.71 1.13 1.13 2.71zm3.25.02q0 2.94-2.1 5.04t-5.02 2.09l-6.82 4.94q-.19 2.06-1.72 3.44T8.74 28q-.86 0-1.74-.3t-1.56-.88-1.16-1.34-.66-1.66L0 22.38v-6.7l6.06 2.45q1.44-.88 3.25-.75l4.44-6.32q.06-2.94 2.13-5t5-2.06q1.93 0 3.59.94t2.6 2.6.93 3.59zM12.69 22.7q0-1.63-1.13-2.78t-2.81-1.16q-.38 0-.81.13l1.62.62q1.19.5 1.69 1.69t.03 2.37-1.65 1.7-2.38 0q-.25-.07-.9-.35t-1.04-.4q1.13 2.12 3.44 2.12 1.69 0 2.81-1.16t1.13-2.78zm12.94-11.6q0-1.97-1.38-3.34t-3.38-1.38q-1.25 0-2.37.63t-1.75 1.72-.63 2.4q0 1.94 1.41 3.35t3.34 1.4q2 0 3.38-1.4t1.38-3.38z',
    'sticker-mule' => [
        'p' => 'M34.75 7.63q.25.75.5 1.3l.38.85.21.5.13.32.03.15v.19q0 .25-.44 1.06-.25.44-.44.5l-.87.13q-.69.12-.94-.57v-.28l-.03-.28-.19-.37-.47-.5q-.31-.32-.65-.47t-.75-.31-.6-.29-.31-.15-.25.03-.19.12-.19.22-.18.28q-.5.57-1.07 1.38t-1.15 1.6-1.03 1.34q-1.57 2-2.13 3-.12.18-.25.62t-.25.81-.12.44q-.2.57-.44 1.78t-.63 2.47q-.06.38-.28 1.13t-.22.87q-.18.5-.25 1.82t.2 1.93q.05.13.33.25t.41.25l.13.28.28.6.15.37q0 .2-.15.25t-.35.1l-.19.03q-.18.12-.25.22t-.09.15-.1.13q-.24.12-1.15.12t-1.1-.18q-.3-.2-.12-.94.13-.32.06-.47t-.12-.16h-.25l-.31-.06q-.2-.13-.25-.63l-.13-.25q-.06-.37.22-1.5t.28-1.18q.19-.88.13-1.82t0-1.12q.06-.25.12-.4t.06-.2.04-.18.03-.38v-.72q-.07-.68 0-1.31t.1-.78-.04-.34q-.25 0-1.6.15t-2.5.19-2.96-.22q-1.25-.25-2.16-.69T9.03 18t-.54-.44q-.5-.19-1-.03t-.68.47q-.07.06-.6 1.19t-1.15 2.47-.75 1.84q-.44 1.38.25 4.19l.1.22.24.53.28.44q.06.06.16.09t.19.1.15.12q.19.19.56 1 .32.5-.06.62-.31.2-.94.2-.06 0-.09.12t-.1.19q-.12.12-.74.21t-1.07-.09q-.37-.13-.44-.31l.04-.38q.04-.3-.1-.37-.06-.07-.31-.13t-.44-.22-.31-.4q-.13-.38-.13-3.13 0-1.06-.56-2.87-.13-.57 0-1l.1-.13.06-.12.1-.25.05-.32q-.06-.06-.18-.06t-.13-.06q-.31-2.25-.4-3.56t-.07-1.57-.03-1l-.22-1.22q-.16-.9-.22-1.43t0-1.41q.32-1.38 1.5-2.38t1.97-1.34 2.72-.28 5.25.87q.38.13 1.25.32t2.66.06 3.03-.94q.56-.37 1.31-.62t.88-.38q.12-.06 1.06-.84t2.31-1.81 2.13-1.41q1-.63 1.9-.94t1.4-.31.91.06.47.06q.13-.06.53-.84t.54-.78.15.34 0 .7.03.59q.07.3.38.3l.69-.3q.18-.13.43-.41t.47-.47.35-.19q.31.19.03.72t-.75 1.31-.47.91l.31.56q.19.38.38 1.25.18.7.28.97t.15.35.16.3.28.7z',
        'w' => 1.125,
        'vw' => 36
    ],
    'strava' => [
        'p' => 'm9.44 0 9.44 18.25H13.3L9.44 11l-3.88 7.25H0zm9.44 18.25h4.18L16.13 32l-7-13.75h4.18l2.82 5.5z',
        'w' => 0.72,
        'vw' => 23
    ],
    'stripe' => [
        'p' => 'M10.31 9.06v2.38h2.13v2.37H10.3v3.94q0 .56.28.88t.66.3.69-.05.5-.13v2.13q-.69.37-1.94.37t-2.06-.75-.82-2V9.63zm5.57 3.19q.74-1.38 2.3-.81V14q-1.3-.44-2.12.56V21h-2.75v-9.56h2.38zm5.74-4.5V10l-2.75.63V8.37zm-18.8 6.5q0 .44.65.72t1.47.5 1.47.94.65 1.84q0 .94-.5 1.63T5.2 20.9t-2.07.34q-1.56 0-3-.69v-2.5Q1.75 19 3.2 19q1.06 0 1.06-.69 0-.43-.66-.75T2.13 17t-1.47-.9T0 14.36q0-1.43 1.03-2.28t2.72-.84q1.5 0 2.75.5v2.63q-1.31-.7-2.75-.7-.94 0-.94.57zM40 16.38q0 .56-.06 1H34.5Q34.69 19 36.69 19q1.5 0 2.75-.56v2.06q-1.32.75-3.13.75-1.37 0-2.37-.53t-1.6-1.66-.6-2.75q0-2.18 1.16-3.62t3.1-1.44q1.87 0 2.94 1.44T40 16.38zm-5.5-1.07h2.88q0-.87-.41-1.34T36 13.5q-.63 0-1.06.47t-.44 1.34zm-7.06-4.06q1.56 0 2.69 1.34t1.12 3.66q0 2.5-1.1 3.75t-2.71 1.25q-1 0-1.94-.75l-.06 3.13-2.75.62V11.44h2.43l.13.68q1-.87 2.19-.87zm-.63 7.69q.75 0 1.22-.75t.47-1.88-.47-1.84-1.22-.72q-.87 0-1.37.5l.06 4.13q.5.56 1.31.56zM18.88 21v-9.56h2.75V21h-2.75z',
        'w' => 1.25,
        'vw' => 40
    ],
    'stripe-s' => [
        'p' => 'M9 9.69q0 .87 1.03 1.56t2.53 1.13 3.28 1.18 3.28 1.7 2.5 2.77 1 4.35q0 4.69-3.37 7.15T10.12 32q-5.06 0-9.75-2v-8.06q5.25 2.87 9.75 2.87 3.44 0 3.44-2.18 0-1-1-1.75t-2.5-1.22-3.28-1.22-3.28-1.66T1 14.13t-1-4.07Q0 5.37 3.28 2.7T12.06 0q4.75 0 8.88 1.63V10q-4.25-2.25-8.88-2.25Q9 7.75 9 9.69z',
        'w' => 0.707,
        'vw' => 23
    ],
    'stubber' => [
        'p' => 'M136.5 294.2l58.8 22.9c9.1-36.8 25.4-61.1 55-61.1c49.4 0 71.4 63.6 142.4 63.6c15.6 0 35.9-2.8 55.3-13.3V368c0 61.8-50.4 112-112.3 112H0l41.8-56L0 368l41.7-56L0 256.1l41.8-56L0 144.1 41.8 88 0 32H335.7C397.6 32 448 82.3 448 144.1v51.3c-9.2 36.3-25.9 60.6-55 60.6c-49.6 0-71.6-63.5-142.4-63.5c-35.9 0-95.2 14.6-114.1 101.6h0z',
        'vw' => 448,
        'vh' => 512
    ],
    'studiovinari' => 'm30 11.75.25 1.75v1.75L28.69 18l-2.44 4.88-3.5 4.25-5 2.37-1.06 1.5-.5.75-.63.25q1.13-4 1.25-4l1.25.06 4.19-2.44-5.81 1.7-3.5 2.24-1.38 1.75-.44.13 1.32-4.63.43-.31 1.25.81 5.57-2.87 3.56-3.19.88-4.25-7.82 1.44.94-1.13-10.81-3.37 5.12-.63L1.2 5.62l7.19 1.13L2 0l15.81 7.94-1.93-2.38 7.75 4.63L14.69 4l1.12 2.38-3.06-1.13L9.87 0l12.2 7.63-2.7-3.5 6.75 6.06.75-.57-1.3-1.06.24-2.31 2.38-.69 1.8 1.56.76.25-.44.38.5.75L30 9l-.63 1.25z',
    'stumbleupon' => 'M31.44 16.63V21q0 2.88-2.07 4.94T24.41 28t-4.97-2.03-2.07-4.9v-4.38l2.13 1 3.25-1v4.44q0 .68.47 1.15t1.15.47 1.2-.47.5-1.15v-4.5h5.37zM17.37 13v-1.94q0-.69-.46-1.19t-1.2-.5-1.18.5-.47 1.2v10q0 2.87-2.06 4.9T7 28q-2.88 0-4.94-2.06T0 21v-4.38h5.37v4.32q0 .68.5 1.15t1.19.47q.31 0 .62-.12t.54-.35.34-.53.12-.62v-10.2q0-2.8 2.1-4.77T15.72 4t4.93 2 2.1 4.81v2.25L19.5 14z',
    'stumbleupon-circle' => 'M16 .5q6.44 0 10.97 4.53T31.5 16t-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5zm0 11.13q.44 0 .78.3t.34.76V14l1.44.63L20.7 14v-1.44q0-1.87-1.38-3.22T16 8t-3.31 1.31-1.38 3.2v6.74q0 .5-.31.81t-.78.32-.78-.32-.32-.8v-2.82H5.5v2.87q0 1.94 1.37 3.32T10.2 24t3.31-1.34 1.38-3.28v-6.7q0-.43.34-.74t.78-.31zm10.5 7.68v-2.87h-3.63v3q0 .44-.3.75t-.79.31-.78-.31-.31-.75V16.5l-2.13.63-1.44-.7v2.95q0 1.93 1.38 3.28T21.8 24q1.25 0 2.35-.63t1.72-1.71.62-2.35z',
    'superpowers' => 'm28 2-5.44 5.44q3.94 3.87 3.57 9.25-.25 4.37-3.22 7.53t-7.29 3.72l-5 .69Q7.81 29 5 29.34T0 30l5.44-5.44Q1.5 20.75 1.8 15.31q.25-4.37 3.25-7.53t7.32-3.72q3-.37 8.43-1.1T28 2zm-5 14.5q.25-3.69-2.25-6.44t-6.25-3q-2.44-.12-4.56.94t-3.5 3.06-1.5 4.44q-.2 3.75 2.31 6.47t6.25 2.97q3.69.19 6.5-2.25t3-6.19z',
    'supple' => [
        'p' => 'M40 16.38q0 2-2.03 3.68t-5.53 2.66-7.69.97q-2.19 0-4.44-.32h1.5q6.32 0 10.75-2.3T37 15.43q0-2.7-3.19-4.88 2.88 1 4.53 2.53T40 16.37zM3 14.25Q3 17 6.19 19.13q-2.88-1-4.53-2.54T0 13.31q0-3 4.44-5.15T15.19 6q2.31 0 4.44.31-.82-.06-1.44-.06-6.31.06-10.75 2.38T3 14.25zm3.75 2.5.56-.69q.57.63 1.25.63.82 0 .82-.63 0-.37-.63-.65t-1.25-.72-.62-1.13q0-.62.5-1.06t1.25-.44q.56 0 1.06.25t.5.81v.5H9.3v-.25q0-.43-.68-.43-.82 0-.82.56 0 .31.4.53t.88.4.88.57.4 1q0 .69-.5 1.12t-1.3.44q-.5 0-.98-.22t-.65-.4zm7.56-4.63h.94v3.44q0 .88-.56 1.44t-1.5.56-1.53-.56-.6-1.44v-3.44h1v3.44q0 1.07 1.13 1.07t1.12-1.07v-3.44zm2.13 5.38V13h-.5v-.88h2.44q.75 0 1.25.5t.5 1.26-.5 1.24-1.25.5h-.94v1.88h-1zm1-2.69h.81q.88 0 .88-.93 0-.88-.88-.88h-.81v1.81zM21 17.5V13h-.5v-.88h2.44q.75 0 1.22.5t.47 1.26-.47 1.24-1.22.5H22v1.88h-1zm1-2.69h.75q.88 0 .88-.93 0-.88-.88-.88H22v1.81zm3.56 2.13v-3.81q0-.13-.18-.13h-.32v-.87H26q.56 0 .56.5v3.87q0 .13.13.13h1.19q.12 0 .12-.13v-.38h.94v.82q0 .56-.56.56h-2.32q-.5 0-.5-.56zm3.94 0V13H29v-.88h3.25q.5 0 .5.5v.82h-.88v-.31q0-.13-.18-.13h-1.2v1.38h1.76v.8H30.5v1.32q0 .13.19.13h1.18q.2 0 .2-.13v-.37h.87v.8q0 .57-.5.57H30q-.5 0-.5-.56zm3.69-4.75H33v-.06h.44v.06h-.13v.37h-.12v-.37zm.37-.06h.13l.12.3.13-.3h.19v.43H34v-.31l-.12.31h-.13l-.12-.31v.31h-.07v-.43z',
        'w' => 1.25,
        'vw' => 40
    ],
    'suse' => [
        'p' => 'm29.19 6.44.06-.07q5.69.82 8.06 2.2 1.13.62 2.2 4.37 0 .19-.26.37-.81.57-1.4.78t-1.66.32-2.56-.53-3.44-1.94q.62 1.5.69 1.56 3.44 1.81 5.12 1.81 1.57-.06 3.13-1l.25.2q-.25.93-.44 1.06-.19.18-.78.53t-2.22.78-3.44.44q-.62 0-1.53-.1t-1.4-.12-1.04.03-.81.37-.4.94q-.26 1.44 1.18 3.44h-3.06q-.13-.25-.5-1.07t-.66-1.21-.78-.94-1.25-.78-1.75-.25q-1.5 0-2.22.87t-.65 1.81.19 1.57h-2.88q-.81-2.88-2.66-4.53t-3.72-1.97q-3.37-.57-5.06 1.6t-.37 4.52q.5 1 1.4 1.57t1.82.68 1.71-.12 1.25-.81q.94-1.2.35-2.6t-2.1-1.53q-.62-.06-1.12.22t-.5.84q0 .32.19.53t.34.29.22.06q.25 0 .38-.03t.21-.07.16-.03q.31-.06.69.16t.37.72q0 .69-.8.94-.5.12-1 .12-.95 0-1.76-.72t-.87-1.9q0-1.57 1.37-2.44.81-.56 1.94-.6t2.37.66q1.38.88 1.88 2.38t.12 2.8q-.56 1.76-2.06 2.63t-3.56.7q-3.19-.26-4.88-2.76Q.5 22.13.2 20.94q-.57-2.24.3-4.44 1.63-3.87 6.25-6.87Q9 8.25 11.13 7.44t3.72-1.06 3.18-.35 2.66.03q3.94.13 8.5 1.63V6.44zM33 11q-.06 1.56 1.38 2.22t2.56-.44.56-2.56-2.12-1.53q-.44 0-.91.18t-.78.5-.5.72T33 11zm2.38-1.63q.68.07 1.15.57t.44 1.15-.53 1.13-1.19.47q-.44 0-.81-.25t-.57-.63-.18-.81.21-.81.63-.6.84-.22zm.28 1.82q.72 0 .72-.47t-.72-.47-.72.47.72.47z',
        'w' => 1.232,
        'vw' => 39
    ],
    'swift' => [
        'p' => 'm28 9.75-.01-.84c-.01-.62-.06-1.23-.16-1.84a5.87 5.87 0 0 0-4.9-4.9 12.7 12.7 0 0 0-1.84-.16L20.24 2H7.76l-.85.01-.46.02c-.46.02-.92.07-1.38.14a6.5 6.5 0 0 0-1.75.58A5.99 5.99 0 0 0 .17 7.07C.07 7.68.02 8.3.02 8.91L0 9.75v12.5c0 .28 0 .56.02.84 0 .62.06 1.23.16 1.84a5.88 5.88 0 0 0 4.9 4.9c.6.1 1.22.15 1.83.16l.85.01h12.49l.84-.01c.62-.01 1.23-.06 1.84-.17a5.89 5.89 0 0 0 4.9-4.9c.1-.6.15-1.22.16-1.83l.01-.84V9.75Zm-4.37 15.07c-1.25-2.43-3.58-1.83-4.77-1.22l-.32.19-.03.01c-2.47 1.32-5.78 1.41-9.12-.02-2.71-1.17-5-3.14-6.58-5.65.77.57 1.59 1.06 2.45 1.46 3.52 1.65 7.06 1.54 9.56 0-3.56-2.74-6.53-6.3-8.82-9.2a12.6 12.6 0 0 1-1.17-1.62 65.25 65.25 0 0 0 8.6 6.51 61.55 61.55 0 0 1-6.05-7.6 60.45 60.45 0 0 0 10.37 8.4c.1-.25.18-.51.26-.78.83-3.01-.1-6.47-2.2-9.32 4.8 2.88 7.63 8.35 6.46 12.98-.02.11-.06.22-.09.32 2.41 2.97 1.76 6.14 1.45 5.54Zm0 0',
        'vw' => 28
    ],
    'symfony' => 'M16 .5a15.5 15.5 0 1 0 0 31 15.5 15.5 0 0 0 0-31Zm8.36 8.97c-.72.03-1.22-.4-1.23-1.05-.02-.58.4-.84.4-1.18-.01-.41-.64-.43-.8-.42-2.49.08-3.04 3.56-3.68 7.12 1.34.2 2.29-.05 2.82-.4.75-.48-.21-.97-.1-1.52.26-1.14 2.04-1.2 2 .32-.01 1.12-1.61 2.62-4.84 2.24-.67 3.72-1.15 7.19-3.64 10.1-1.81 2.16-3.65 2.5-4.47 2.52-1.54.05-2.57-.77-2.6-1.86-.04-1.07.9-1.64 1.52-1.67 1.37-.05 1.88 1.6.93 2.13-.76.6 0 .79.13.78.65-.02 1.08-.34 1.38-.56 1.5-1.25 2.08-3.43 2.84-7.4.5-3.1 1.06-4.87 1.14-5.12-1.06-.8-1.7-1.79-3.12-2.17-.97-.27-1.57-.04-1.99.49-.5.62-.33 1.43.15 1.91l.79.88c.97 1.12 1.5 2 1.3 3.16-.32 1.87-2.54 3.31-5.18 2.5-2.25-.7-2.67-2.29-2.4-3.16.47-1.51 2.65-.73 2.17.85-.18.54-.31.54-.4.81-.28.93 2.62 1.78 3.2-.08.27-.9-.34-1.36-1.4-2.5-1.78-1.98-1-4.08.18-4.97 3.3-2.46 6.29 1.1 6.91 1.61 2.33-6.81 6.29-6.6 6.4-6.6 1.58-.05 2.77.67 2.8 1.8.02.48-.25 1.4-1.21 1.44Zm0 0',
    'teamspeak' => 'M15.25 21.69q-.19 1.06-1.38 1.5-.68.25-3.87.06-1.81-.12-2.69-1.5-1.37-2.12-.94-4.62.25-1.13 1.07-1.54t1.81.16q1.75.81 4 2.88 2.19 2.06 2 3.06zM28.06 23q.7.5.75 2 0 1.44-.5 2.56t-.97 1.57-1.47 1.18q-2.5 1.94-8.87 1.7-3.25-.2-3.31-.63 0-.44 3.12-.63 2.75-.25 4.9-1.19t3.32-2.3 1.69-2.32.84-1.82q.13-.43.5-.12zm3.88-7.69q0 2.88-.07 3.25-.12.75-.62 1.35t-1.22.84-1.47.13q-.56-.2-.56-.94v-5.25q-.2-7.19-6.44-10.63-2.5-1.36-5.36-1.36t-5.32 1.15T6.6 7.22t-2.34 5.1q-.19.68-.22 1.34T4 15.13v5.06q-.06.69-.75.69-1.44.06-2.35-.78T0 17.8v-1.3l.06-1.63q.19-1.5 1.56-2.13.32-.19.38-.5.56-3.06 2.44-5.75.25-.31.06-.56-.25-.38.19-.94Q7.44 1.94 11.06.81q9-2.75 15.82 3.82l.06.06q.87.75.37 1.44-.06.06.13.31 1.94 2.62 2.5 5.87.06.25.37.38 1.63.81 1.63 2.62zM22 23.56l-3.25-.5q-.88-.25-1.34-.75t.09-1.37 2.63-1.69q.06 0 .3-.13 5.07-2 5.2.63.06 1.63-.97 2.78T22 23.56z',
    'telegram' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm7.63 10.63q.12-.76-.22-1.1t-.85-.15l-14.94 5.8q-.8.26-.84.63t.66.57l3.8 1.18 8.88-5.56q.38-.25.57-.06.06.06-.07.19l-7.18 6.5-.32 3.93q.44 0 .82-.37l1.87-1.81 3.88 2.87q1.12.63 1.37-.62z',
    'telegram-plane' => [
        'p' => 'm27.94 6.19-4.25 19.87q-.5 2.13-2.32 1.13l-6.43-4.75-3.13 3q-.62.62-1.31.62l.5-6.56L22.94 8.75q.43-.44 0-.5-.38-.13-.82.19L7.37 17.75l-6.37-2q-1.06-.31-1-.94t1.31-1.06l24.82-9.63q.93-.3 1.5.22t.3 1.85z',
        'w' => 0.873,
        'vw' => 28
    ],
    'tencent-weibo' => [
        'p' => 'M4.5 31q.06.94-.84 1t-.97-.81q-.57-5.94 1.15-10.97T9 11.62q-.75-1.68.25-3.28t2.94-1.6q1.37 0 2.4 1.04t1.03 2.47q0 2.06-1.84 3.03t-3.6-.22Q3.5 20.25 4.5 31zM12 0q4.31 0 7.38 3.06t3.06 7.38q0 4.87-3.85 8.03t-8.9 2.1q-.94-.26-.72-1.13t1.1-.69q3.87 1 7.18-1.56t3.31-6.75q0-3.57-2.5-6.07T12 1.87Q9.7 1.87 7.75 3t-3 2.9-1.28 4 .9 4.41q.26.5-.06.94t-.81.44-.75-.57q-1.44-2.5-1.16-5.18t1.5-4.91 3.63-3.62T12 0z',
        'w' => 0.748,
        'vw' => 24
    ],
    'the-red-yeti' => 'M30.25 15.13q.31.37.44.8.44 1.13.56 2.13l-.12.07-.82-.94q-.56.87-1.06 1.56 0 3.81-1.9 6.88t-5.1 4.72T15.31 32q-5.75 0-9.87-3.75-4.07-3.81-4.07-9.44v-.12l-.06-.19-1.25 1.38q-.44-2.82 2.06-5 0-.07.04-.13l.03-.06-1.82-.07-.12-.25q.06-.06.44-.21t.65-.29.69-.25.72-.12q.69-1.38 1.75-2.5.06-.56.37-1.63l-.43.25-.13-.06.13-.56q1-2.44 2.87-4.19l-2.06-.87h.25q1.44-.38 3.06 0-.25-.44-.53-1.22t.03-1.6q.5.44.94 1.75.31-1.5 2.12-2.87h.13l-.75 2.8q2.87-1.69 6.12-.75l-1.21 1.22-.63.47-.84.37q1.75.5 2.18.94l-1.56-.44q1.38 1.25 1.56 1.57 2.32.12 4.25.8.25-.12.79-.43t.71-.37.63-.2.72-.02.75.28 1.1.56q0 .31-.04.6t-.03.43v.44l.03.1.03.09.06.1.13.12Q26 8 26.81 7.63q1.2-.57 1.94.37.5 0 .78.53t.35 1.16q.43.06.46.56t-.12.94-.22.44l-.44 1.12q1.44 1.44 2 2.81zm-5.63-8.2q-1.25-.8-1.75-.5.82.7.82 1.26.12.75-.63 1.37l-.25.25q.38.7.5 1.44h.2l-.13-.72q0-.03.12-.22.88.5 1 1.25 1.44-.68 3.44-1.31l-1.13 1q1.63 0 2.88.5.25-1.25.19-1.25l-.32.69q-.18-1.06-.18-1.56-.07-.63-.63-.63 0 .56.13.94l-.13.37q-.31-1.12-.56-1.75-.19-.43-.57-.37t-.68.28-.75.5-.7.28q.76.06 1.7 0l-1.38.38.19.25q-1.06 0-1.38-1.07V6.94zm-2.43-.62-1.32.75q1 .63 1.75 1.94.63-.25.75-.5.13-.12.2-.44.06-.12-.2-.68-.3-.44-1.18-1.07zM5.44 8.94q-.69 2.06-.82 4.69l.82.3.06-1.68q.63 1.06.81 1.75.63-.44 1.38-.69l.06.13-.44.69q.94-.38 1.75-.07l-.62.5.47.2.37.15.32.12q.22.1.28.16t.22.15.25.22.21.28.25.41l-.75-.12q.44.25 1.1.68t1.12.72 1.25.69 1.5.72 1.72.62 2.13.57q.87-1.38.87-1.44l.07-.06h-.07q-.06-.25-1.12-1.25.5.06.69.18l-1.07-1-.19-.18-.12-.07q1.56.63 2.06 1.75l.13-1.93L18.69 15l.44-.38h-.75l-.53.07-.63.15-.6.32-.56.47h-.18q.68-1.38 1-2.88.19-.94.19-1.75 0-1.81-.82-3.56-.62-1.25-1.87-2.19-.75-.56-1.32-.88-.12-.06-.3-.06-.45 0-1.26-.06v.19q-1 .12-1.56.93h-.06l-.38.2q-.13-.26.25-.57l.69-.69h-.19l.13-.19q-.13 0-.26-.03t-.18-.03V4l.12-.13q.06 0 .19.07h.25q.44-.2.88-.13l.18.07v.06q.63-.06 1.5-.19 1.88-.25 3-1.56l-1.31-.07q-2.44 0-4.44 1.2l-.19-.13.07-.32.06-.43q.25-1 .44-1.57-.13 0-.6.7t-.65 1.18q-.13.56-.31.94l-.88-1.82.5 2.25v.07q-.5 0-1-.13Q7.37 4 6.44 4h-.2l1.57.5.38-.06-1.25 1.25q-.88.75-2.07 3.12l.25-.06.7-.63-.38.32.81-.75L5.5 8.8zm17.31 20.62.22-1.31.06-1.19-.06-.87-.13-1.1-.15-1.1q-1.88.63-2 .63l.69.94-.13.07q-1.06-.32-1.75-.94.31.93.69 1.31t1.19.63q-.7.68-2.13-.75l.13 1.43-.25.2q-.13-.63-.5-1.23t-.66-.9-.4-.32q-3.57 1.07-7.13-.68l-.56.12q-.31.06-1.25.06t-1.78-.15-1.97-.75-2-1.6q1.44 3.75 4.5 5.2-.44-.95-1.32-1.7.07-.18.07-.62 1.81 1.81 5.81 3.75-1.81-.69-2.69-1.07t-1.06-.46l-.56-.29 1 1.32q-.25.12-.63-.07-1.12-.62-2.19-1.56.13.19.32.38l.72.72 1.09.93 1.47 1 1.78.91 2.1.66 2.4.28 2.69-.25q-1-.44-1.72-1.13t-1.06-1.15-.79-1.25-.62-1.04l1.12.5-.75-1.5q1.13 1.57 2.38 2.82l-1.88-1.07q1.2 2.57 4 3.7l.13-.04.34-.1.6-.18 1.06-.44 1.56-.75zm7.19-13.37.56.81-.75-2.06q.56 0 1.31.19-.8-1.07-1.68-2l-1.7-1.44.07-.19 1.5-.13q-.94-.24-2.06-.24h-1.13v-.2l.94-.8-2.62 1.24-.75-1.12.18 1-.18.06-.38-.31h-.62l.37.31v.13l-.87 1.68q-.25.88-.63 2t-.81 2.5-.75 2.38l.06-1-.06.19v.06q-.63 1.5-1.32 2.44l.57-1.38-2.94 1.32q1.13-1.25 2.19-1.2-2-.3-3.69-1t-2.31-1.15-.88-.72q.88 1.5 1 2.5l-.12.07q-.75-2.75-3.5-4.38l1.5.19q-.44-.56-1.35-.84t-1.22-.16l-.12-.19.5-.5-1.38.38.32-.88q-.38.25-1.07.56H6q.06-.06-.19-.62 0 .62-.06.75-.07.06-.19.12-1.19-.62-2.16-.71t-1.28 0-1.12.4q.25 0 .56.06t.5.1.34.06.25.03.2.03.12.03.1.04.09.06.09.06.12.1q-1 .5-1.34.72t-.81 1.06-1.03 2.6l1.37-1.38.19 1q.56 2.69 2.84 4.15t5.03 1.1L8 22.63l2.81 1.44-.12.18.87.25q2.82.82 5.82.38l-2-1.31q.3.06.65.18t.63.25.53.29.47.34.34.31.31.35.25.28.2.25.12.25l-.44-1.82.06-.18 1.19 1.75-.56-1.7 1.37.88h.06l-.25-.25.44.25-.87-1 .06-.18.5.43q2.75-.75 5.06-2.5 2.7-2 4.5-5l-.25-.56h.2zm-.38-4.5-1.12.12.87.7zm-15.94.44q.44.8 1.13 1.18-.38.63-1.06.94-.13.06-.2.06-.74.82-1.05.94l.43-.81q-.75.25-1.56.19l-.06-.13q1.81-.19 2.94-1-.38-.37-.57-1.37zm.25-6.38q.57.31.88.88 1.44 2.3.69 3-.75.75-1.63-.2-.87-1-.72-2.5t.78-1.18zm-.3 2.88q.37 1.12 1.18 1 .75-.07.5-1.25-.37-1.57-.97-2.07t-.65-.25q-.25.25-.25 1.57.5-.75.8-.32l-.05.07q-.07.18.12.37l.06-.06V8q-.12.94-.75.63zm-2.82-.32q-.25-.06-.31-.25-.2-.43-.03-.87t.53-.44q.18 0 .31.25-.06.06-.13.19t.2.19q0 .37-.13.62t-.25.28-.19.03zm2.5-3.12L13 5v-.13q1.38 0 1.75.88-.75-.38-1.63-.38V5.2h.13zm-4.5 3.06q.13.13.31.44-.87.19-1.31.47t-.56.28q.5-1 1.5-1L8.25 8q.37-.06.72-.03t.53.4q0-.74.19-1.5Q9.8 6.2 10 6l.31-.28q.25-.22.7-.56t.8-.53q.38-.25.82-.13.06.06.06.13l-.81.5q.69 1.06.56 2.56-.12 1.06-.94 1.75l.19.31q-.56-.06-.87-.12t-.57-.2-.37-.24-.44-.4-.69-.54zm1.19-1.13q-.06.38-.19 1.63l.31.38.44.18q.44.07.75.07l.38-.5q.5-.57.53-1.44T12 5.88q-.31-.75-.62-.57-.57.13-1 .69-.32.44-.44 1.13zm2.62 3.5-.18.38q.68-.44 1.18-.5l.44-.06.75.5h-.12q-.88-.38-1.94.43-.06.44.37 1.88h.57q-2 .56-3.07.31-1.5-.43-1.68-1.5-.26-1.06-.13-2.12-.56.06-.94.69-.19-.44-.37-.63.93-.44 2.06-.25v.13l-.38.06v.5l.32.78.87.28q.2-.06 2.25-.87z',
    'themeco' => [
        'p' => 'M12.5.75q1.06-.63 2.19 0l11.75 6.63q.5.25.81.8t.31 1.07v13.38q0 .56-.3 1.12t-.82.81l-11.75 6.7q-1.13.62-2.19 0l-11.38-6.7q-.5-.25-.8-.8T0 22.62V9.25q0-.25.1-.53t.24-.53.35-.47.44-.34zM7.75 13.06h-3v6.25h1.19V17.5h1.81q1.44 0 2.03-1.13t0-2.21-2.03-1.1zm8.63 6.19-1.94-2.13q.81-.18 1.25-.84t.37-1.37-.69-1.29-1.56-.56h-3v6.25H12v-2.06h1.12l1.82 2.06h1.43v-.06zm-2.57-5.06q.63 0 .9.5t0 1-.9.5H12v-2h1.81zm-5.93-.13q.62 0 .87.6t0 1.18-.88.6H5.95v-2.38h1.93zm11.87-1.12q-1.5 0-2.38 1.03t-.87 2.25.84 2.22 2.41 1q1.25 0 2.06-.66t1-1.62 0-1.9-1-1.63-2.06-.7zm0 1.06q1.31 0 1.84 1.1t0 2.18-1.84 1.1-1.84-1.1 0-2.19T19.75 14z',
        'w' => 0.861,
        'vw' => 28
    ],
    'themeisle' => 'M13 5.53q0-.53.31-.94t.82-.4.78.4.28.9q0 .38-.13.66t-.37.5-.56.22q-.5 0-.82-.4T13 5.53zm19 9.97q0 2-.6 4.47t-1.65 3.66q-1.06 1.12-3.5 2.53t-4.06 1.9h-.07v2q0 .56-.25 1t-.65.69-.9.25q-1 0-1.57-.88-.56.88-1.56.88t-1.57-.88q-.5.88-1.5.88t-1.56-.88Q12 32 11 32q-.57 0-.97-.28t-.63-.75-.22-1.03q-1.62 1.25-3.56 1.25-2.12 0-3.75-1.44.94-.06 1.88-.38-2-.56-3.25-2.25 2.25.5 4.18-.68Q1.31 23 .43 20.94q-.43-1-.43-2.88 0-1.31.28-2.87t.84-3.22 1.69-2.78 2.56-1.13q1.07 0 1.44.94.38-.5.75-.94l.5-.5 1.31-2.12q1.57-2 3.78-3.2t4.66-1.18q.63 0 1.13.06Q20 0 21.56 0q1.44 0 2.5 1 .06.12.06.25t-.81 1q1 .31 1 .75 0 .12-.31.44 2.81 2.5 3.56 6.75.44-.5 1.06-.5.82 0 1.75.87Q32 11.94 32 15.5zM11.75 5.6q0 .9.56 1.62t1.44.72 1.47-.72.6-1.66q0-.56-.26-1.06t-.72-.84-1.09-.35q-.87 0-1.44.69t-.56 1.6zm3.13 6.53q0-1.57.5-3.63-1.2 1.81-1.41 3.88t.4 3.74 2.04 2.79T19.69 20q2 0 3.75-1.25.25-.88.25-1.38 0-1.37-1.32-1.37-.3 0-.8.13-3.26.62-4.98-.2t-1.72-3.8zM12.3 28.05q0-.56-.37-.93t-.9-.38-.91.38-.38.93v2q0 .32.1.53t.28.41.4.31.47.13q.56 0 .94-.4t.37-.98v-2zm3.06 0q0-.37-.15-.65t-.47-.47-.63-.19q-.56 0-.93.38t-.38.93v2q0 .57.38.97t.9.4.9-.4.38-.97v-2zm3.07 0q0-.56-.35-.93t-.87-.38-.9.38-.38.93v2q0 .57.37.97t.9.4.88-.4.35-.97v-2zm3.12 0q0-.56-.37-.93t-.9-.38-.88.38-.35.93v2q0 .57.35.97t.87.4.9-.4.38-.97v-2zm4.82-10.18q-1 1.87-2.47 3.3t-3.44 2.29-4.03.84q-4.38 0-7.2-3.28t-2.8-7.78q0-1.56.37-3.12-1.87 3.12-1.87 7 0 2.87 1.15 5.56t3.28 4.5q.57-1.06 1.63-1.06 1 0 1.56.93.56-.93 1.56-.93t1.5.93q.57-.93 1.57-.93t1.56.93q.56-.93 1.56-.93 1.07 0 1.57 1 2-1.7 3.18-4.13t1.32-5.12zm.62-6.25q0-2.63-1.6-4.6t-4.15-1.97q-2.81 0-4.94 2.25-2.56 8.57 2.57 8.57.37 0 1.3-.16t1.6-.22 1.38.1.97.77.06 1.82Q27 15.62 27 11.62z',
    'think-peaks' => 'M27.63 25.56 14.74 3.5 2 25.5l-2-.06L14.75 0l12.88 22.12 3.5-5.93h2zm-22.2-2.75L14.82 6.7 29.51 32h-2L14.8 10.2 7.44 22.8h-2z',
    'threads' => [
        'p' => 'M331.5 235.7c2.2 .9 4.2 1.9 6.3 2.8c29.2 14.1 50.6 35.2 61.8 61.4c15.7 36.5 17.2 95.8-30.3 143.2c-36.2 36.2-80.3 52.5-142.6 53h-.3c-70.2-.5-124.1-24.1-160.4-70.2c-32.3-41-48.9-98.1-49.5-169.6V256v-.2C17 184.3 33.6 127.2 65.9 86.2C102.2 40.1 156.2 16.5 226.4 16h.3c70.3 .5 124.9 24 162.3 69.9c18.4 22.7 32 50 40.6 81.7l-40.4 10.8c-7.1-25.8-17.8-47.8-32.2-65.4c-29.2-35.8-73-54.2-130.5-54.6c-57 .5-100.1 18.8-128.2 54.4C72.1 146.1 58.5 194.3 58 256c.5 61.7 14.1 109.9 40.3 143.3c28 35.6 71.2 53.9 128.2 54.4c51.4-.4 85.4-12.6 113.7-40.9c32.3-32.2 31.7-71.8 21.4-95.9c-6.1-14.2-17.1-26-31.9-34.9c-3.7 26.9-11.8 48.3-24.7 64.8c-17.1 21.8-41.4 33.6-72.7 35.3c-23.6 1.3-46.3-4.4-63.9-16c-20.8-13.8-33-34.8-34.3-59.3c-2.5-48.3 35.7-83 95.2-86.4c21.1-1.2 40.9-.3 59.2 2.8c-2.4-14.8-7.3-26.6-14.6-35.2c-10-11.7-25.6-17.7-46.2-17.8H227c-16.6 0-39 4.6-53.3 26.3l-34.4-23.6c19.2-29.1 50.3-45.1 87.8-45.1h.8c62.6 .4 99.9 39.5 103.7 107.7l-.2 .2zm-156 68.8c1.3 25.1 28.4 36.8 54.6 35.3c25.6-1.4 54.6-11.4 59.5-73.2c-13.2-2.9-27.8-4.4-43.4-4.4c-4.8 0-9.6 .1-14.4 .4c-42.9 2.4-57.2 23.2-56.2 41.8l-.1 .1z',
        'vw' => 448,
        'vh' => 512
    ],
    'tiktok' => [
        'p' => 'M28 13.12c-2.75 0-5.44-.85-7.67-2.45v11.17a10.16 10.16 0 1 1-8.77-10.07v5.62a4.66 4.66 0 1 0 3.27 4.45V0h5.5a7.63 7.63 0 0 0 3.48 6.4A7.57 7.57 0 0 0 28 7.66Zm0 0',
        'vw' => 28
    ],
    'trade-federation' => 'M15.5.56q6.44 0 10.97 4.53T31 16.06t-4.53 10.97-10.97 4.53-10.97-4.53T0 16.06 4.53 5.1 15.5.56zm0 30.19q6.06 0 10.38-4.31t4.3-10.38-4.3-10.37T15.5 1.37 5.12 5.7.83 16.06t4.3 10.38 10.38 4.31zm9.69-20.56h-9.13v2.19h5.25v2.93h-5.25v12.25h-3v-12.3H5.7l-2.32-2.88h9.7V7.25H25.2v2.94zM20.62 13h-5.18V9.5h9.06V8H13.75v5h-9L6 14.5h7.75v12.38h1.69V14.5h5.19V13zM10.5 8.69l1.38-.56-.94 1.12.93 1.13-1.37-.57-.75 1.25.13-1.5-1.44-.31 1.43-.38-.12-1.5zM8.69 7.5l.56.63-.81-.38-.44.81.06-.93-.87-.2.87-.24L8 6.37l.5.75.75-.37zM19 16.56l-.56.7.56.62-.81-.32-.44.75.06-.87-.87-.19.87-.25-.06-.88.44.75zm6.69-4.81-.94 1.06.94 1.07-1.32-.57-.68 1.25.06-1.43-1.38-.32 1.38-.31-.13-1.44.75 1.25zM15.5 1.87q5.88 0 10.03 4.16t4.16 10.03-4.16 10.03-10.03 4.16-10.03-4.16T1.3 16.06 5.47 6.03 15.5 1.87zm5.94 10.38h-5.25v-1.94h9.12V7.12H12.94v5.13H3.13l2.5 3.12h7.3V27.7h3.26V15.44h5.25v-3.2zM15.5 2.69q-4.38 0-7.88 2.56t-4.8 6.63h9.8V6.75h12.5Q21.2 2.7 15.5 2.7zm0 26.69q5.5 0 9.4-3.91t3.91-9.4q0-4.82-3.06-8.57v3.19h-9.13v1.19h5.25v3.93h-5.25v12.25h-4v-12.3H5.44l-2.75-3.5q-.57 1.8-.57 3.8 0 5.5 3.94 9.41t9.44 3.9zm-4.31-12.2 1.44-.55-1.07 1.06.94 1.19-1.31-.7-.82 1.32.25-1.5-1.5-.37 1.5-.25-.12-1.5z',
    'trello' => [
        'p' => 'M24.5 2q1.44 0 2.47 1.03T28 5.5v21q0 .94-.47 1.75t-1.28 1.28-1.75.47h-21q-1.44 0-2.47-1.03T0 26.5v-21q0-1.44 1.03-2.47T3.5 2h21zM12.31 23.19V7.3q0-.69-.5-1.19t-1.18-.5h-5.2q-.68 0-1.18.5t-.5 1.2v15.87q0 .5.25.87t.62.6.82.22h5.18q.7 0 1.2-.5t.5-1.2zm12.06-7V7.3q0-.69-.5-1.19t-1.18-.5h-5.07q-.43 0-.84.25t-.63.63-.21.81v8.88q0 .5.25.87t.62.6.81.22h5.07q.68 0 1.18-.5t.5-1.2z',
        'w' => 0.873,
        'vw' => 28
    ],
    'tripadvisor' => [
        'p' => 'M10.38 17.56q0 .38-.2.72t-.52.53-.79.2q-.56 0-1.03-.45t-.47-1.03.47-1.03 1.07-.44 1.03.44.44 1.06zm16.56-1.5q.44 0 .78.2t.53.52.19.78q0 .57-.44 1t-1.06.44-1.07-.44-.43-1.03.43-1.03 1.07-.44zm5.62 8.69q-2.94 2.31-6.65 1.84t-6.03-3.4l-1.94 2.94-1.88-2.88q-2.25 2.81-5.84 3.28t-6.53-1.65Q.69 22.68.13 19t1.62-6.69Q1.19 10.37 0 8.7h5.63q2.25-1.4 5.71-2.4t6.16-1h.88q2.56 0 5.96 1T30 8.7h6q-1.19 1.62-1.81 3.56 2.19 2.94 1.72 6.6t-3.35 5.9zM16.2 17.59q0-2.96-2.1-5.06t-5.06-2.1-5.1 2.1-2.12 5.06 2.13 5.1 5.1 2.12 5.05-2.12 2.1-5.1zm1.81-.65q.25-3.44 2.75-5.82t5.94-2.5q-1.7-.67-4.14-1.2t-4.31-.54h-.5q-1.88 0-4.38.5T9.2 8.63q1.75 0 3.31.7t2.72 1.77 1.9 2.63.88 3.22zm15.72-1.78q-1.03-2.79-3.72-4.04t-5.5-.21-4.06 3.72-.22 5.5 3.75 4.06 5.5.22 4.03-3.75.22-5.5zm-20.34 2.37q0 1.84-1.32 3.16T8.91 22t-3.16-1.31-1.31-3.16 1.31-3.15 3.16-1.32 3.15 1.32 1.32 3.15zm-1.57.03v-.06q0-1.19-.84-2.03t-2.1-.85q-.74 0-1.43.38t-1.07 1.06-.37 1.5q0 1.2.84 2.03t2.07.85 2.06-.85.84-2.03zm19.63 0q0 1.2-.63 2.22t-1.62 1.63-2.25.6q-1.81 0-3.13-1.32t-1.31-3.16 1.31-3.15 3.16-1.32 3.12 1.32 1.35 3.18zm-1.56-.03q0-1.22-.85-2.06t-2.06-.85-2.06.85-.85 2.06.85 2.06 2.06.85 2.06-.85.85-2.06z',
        'w' => 1.125,
        'vw' => 36
    ],
    'tumblr' => [
        'p' => 'M19.38 30q-.75.81-2.41 1.4t-3.72.6q-2.63 0-4.56-.88t-2.88-2.3T4.47 26t-.4-2.81v-9H1.11q-.62 0-.62-.63V9.31q0-.75.69-1 2.62-.87 3.87-2.94T6.5 1q.06-1 1-1h4.38q.62 0 .62.62v7.2h5.25q.63 0 .63.62v5.12q0 .63-.63.63H12.5v8.3q0 1.7 1.1 2.35t3.15-.1q.56-.24.81-.12.32.07.44.5l1.38 4q.25.63 0 .88z',
        'w' => 0.623,
        'vw' => 20
    ],
    'tumblr-square' => 'M25 2q1.25 0 2.13.88T28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22zm-5.13 22.75q.13-.13 0-.56l-.87-2.5q-.06-.25-.25-.32t-.5.07q-1.31.5-2 .1t-.69-1.48v-5.19h3.25q.38 0 .38-.37v-3.25q0-.38-.38-.38h-3.25v-4.5q0-.37-.37-.37h-2.75q-.57 0-.63.62-.25 3.5-3.31 4.57-.44.19-.44.62v2.69q0 .38.38.38h1.87v5.62q0 .88.25 1.75T11.4 24t1.79 1.44 2.87.56q2.63 0 3.81-1.25z',
    'twitch' => 'M2.5 2h24.88v17.13l-7.32 7.3h-5.44L11.06 30H7.31v-3.56H.62V6.8zm22.38 15.88V4.5H4.8v17.56h5.63v3.57L14 22.06h6.69zm-4.2-8.57v7.32h-2.5V9.3h2.5zm-6.68 0v7.32h-2.5V9.3H14z',
    'twitter' => 'M28.69 9.5q.06.25.06.81 0 3.32-1.25 6.6t-3.53 6-5.9 4.4-8 1.7Q4.56 29 0 26.05q.69.07 1.56.07 4.57 0 8.2-2.82-2.2 0-3.85-1.28T3.63 18.8q.62.07 1.19.07.87 0 1.75-.2-1.5-.3-2.7-1.24T2 15.16t-.69-2.9v-.13q1.38.8 2.94.87-2.94-2-2.94-5.5 0-1.75.94-3.31Q4.7 7.25 8.22 9.06t7.53 2q-.12-.75-.12-1.5 0-2.69 1.9-4.62T22.13 3Q25 3 26.94 5.06q2.25-.43 4.19-1.56-.75 2.31-2.88 3.63 1.88-.25 3.75-1.07-1.37 2-3.31 3.44z',
    'twitter-square' => 'M25 2q1.25 0 2.13.88T28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22zm-3.06 9.94q1.19-.88 2.06-2.13-1.19.5-2.37.63 1.3-.81 1.8-2.25-1.24.75-2.62 1-1.18-1.31-2.93-1.31-1.94 0-3.2 1.5t-.8 3.56q-5.2-.31-8.5-4.31-.57.93-.57 2.06 0 .62.25 1.28t.69 1.22.94.94q-1-.07-1.88-.57v.07q0 1.5.94 2.62t2.38 1.44q-.88.19-1.88.06.38 1.25 1.44 2.03t2.37.78Q7.81 22.31 5 22.31q-.56 0-1-.06 2.88 1.88 6.31 1.88 3.57 0 6.32-1.82t4.03-4.43 1.28-5.44v-.5z',
    'typo3' => [
        'p' => 'M10.75 4.88q0 1.56.88 4.34t2.15 5.5 2.88 4.72 2.84 2q.75 0 1.19-.13-2.38 3.75-4.94 6.22T11.5 30q-2.25 0-5.03-3.94t-4.63-8.93T0 8.88q0-1.44.56-2.32.82-.94 2.63-1.81t4-1.47 4.44-.84q-.88.68-.88 2.43zM18.38 2q8.68 0 8.68 3.25 0 2.44-.87 5.19t-2.03 4.5-2.03 1.75q-1.07 0-2.22-1.38t-2.07-3.28-1.46-4.06-.57-3.72q0-1.38.57-1.81t2-.44z',
        'w' => 0.845,
        'vw' => 27
    ],
    'uber' => [
        'p' => 'M25.88 2q.87 0 1.5.63t.62 1.5v23.75q0 .87-.63 1.5t-1.5.62H2.13q-.87 0-1.5-.63T0 27.88V4.13q0-.87.63-1.5T2.13 2h23.75zm-11 22.44q3.5-.38 5.71-3.1t1.85-6.22q-.31-3.25-2.75-5.43T14 7.49 8.31 9.7t-2.75 5.43h5.88v-1.24q0-.44.43-.44h4.26q.43 0 .43.43v4.26q0 .43-.43.43h-4.25q-.44 0-.44-.43v-1.25H5.56q.38 3.5 3.1 5.71t6.22 1.85z',
        'w' => 0.873,
        'vw' => 28
    ],
    'ubuntu' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm3.28 5.81q-.4.7-.19 1.5t.94 1.22 1.5.19 1.22-.94.22-1.5-.94-1.19T20 5.41t-1.22.9zM5.44 18q.87 0 1.44-.6t.56-1.4q0-.56-.25-1t-.72-.72T5.44 14q-.81 0-1.38.6T3.5 16q0 .56.25 1t.72.72.97.28zm1.81.19q.81 3.12 3.5 4.94l1.5-2.5Q9.81 18.93 9.81 16t2.44-4.62l-1.5-2.5q-2.69 1.8-3.5 4.93 1 .88 1 2.22t-1 2.16zm14.28 8.22q.72-.41.94-1.2t-.22-1.5-1.22-.93-1.5.19-.94 1.22.2 1.5 1.21.9 1.53-.18zm-.03-4.35q2.31-2.25 2.56-5.5l-2.87-.06q-.25 2.94-2.82 4.4t-5.25.23l-1.43 2.56q2.94 1.44 6.06.56.13-.87.72-1.5t1.4-.78 1.63.1zm-.31-6.56 2.87-.06q-.25-3.25-2.56-5.5-1.25.5-2.4-.2t-1.35-2q-3.13-.87-6.06.57l1.44 2.56q2.68-1.3 5.25.2t2.8 4.43z',
    'uikit' => [
        'p' => 'M27.75 8v16l-14.13 8L0 24V10.62l5.5 2.82v7.31l8.31 4.69 8.5-4.69v-9.44L16 7.7l5.44-3.32zm-8.44-4.94-5.5 3.32-5.37-2.94L14 0z',
        'w' => 0.865,
        'vw' => 28
    ],
    'umbraco' => 'M16.02.5A15.53 15.53 0 0 0 .44 15.98C.44 24.54 7.4 31.49 16 31.5c8.59 0 15.55-6.92 15.56-15.48A15.53 15.53 0 0 0 16.02.5Zm9.1 16.63a9.23 9.23 0 0 1-.88 4.06 4.66 4.66 0 0 1-2.73 2.18 16.8 16.8 0 0 1-5.36.66h-.29c-2.29 0-4.08-.21-5.36-.66a4.66 4.66 0 0 1-2.73-2.18 9.26 9.26 0 0 1-.88-4.07 25.7 25.7 0 0 1 .57-6.66l.11-.53a.32.32 0 0 1 .3-.24h.05l2.01.31c.15.03.26.16.26.32v.04l-.1.55c-.11.55-.2 1.26-.3 2.11-.11.87-.17 1.75-.18 2.63a9.74 9.74 0 0 0 .52 3.74c.35.81 1.08 1.4 1.96 1.57 1.18.25 2.39.36 3.6.33h.64c1.2.03 2.41-.08 3.6-.33a2.72 2.72 0 0 0 1.95-1.57c.35-.81.52-2.06.5-3.74 0-.88-.05-1.76-.16-2.63-.1-.85-.2-1.56-.3-2.1l-.1-.56a.1.1 0 0 1 0-.04.3.3 0 0 1 .25-.32l2.01-.3h.05c.15 0 .28.09.31.23.04.13.05.23.11.53a25.78 25.78 0 0 1 .56 6.66Zm0 0',
    'uncharted' => [
        'p' => 'M10.73 14.55c.14 0 .26-.08.31-.2a3 3 0 0 1 .94-1.58c.07-.06.1-.16.1-.26a.27.27 0 0 0-.13-.2l-7.3-4.28 4.66-2.72c.19-.1.3-.3.31-.51a.57.57 0 0 0-.31-.52l-1.4-.8a.6.6 0 0 0-.6 0L1.06 7.1a.6.6 0 0 0-.3.52v7.19c0 .22.1.42.3.52l1.4.8c.1.05.2.08.31.08.15 0 .3-.05.42-.16a.67.67 0 0 0 .18-.44v-5.33l7.25 4.24c.03.02.08.03.12.03Zm9.47 9.06c-.1 0-.2.03-.3.08l-4.67 2.69v-8.39c0-.18-.18-.36-.38-.28-.6.17-1.22.18-1.81.02a.3.3 0 0 0-.27.05.3.3 0 0 0-.12.24v8.33l-4.66-2.69a.56.56 0 0 0-.31-.08.6.6 0 0 0-.6.6v1.63c0 .22.12.42.31.51l6.27 3.6c.09.06.2.08.3.08.11 0 .22-.03.32-.08l6.26-3.6c.19-.1.3-.3.31-.51v-1.63c-.05-.29-.33-.57-.65-.57ZM17.88 4.88a1.44 1.44 0 1 0 0-2.88 1.44 1.44 0 0 0 0 2.88Zm3.97-.63a1.44 1.44 0 1 0 0 2.87 1.44 1.44 0 0 0 0-2.87Zm3.95 5.23a1.44 1.44 0 1 0 0-2.88 1.44 1.44 0 0 0 0 2.88Zm-3.95-.58a1.44 1.44 0 1 0 0 2.88 1.44 1.44 0 0 0 0-2.88Zm-3.98 5.2a1.44 1.44 0 1 0 0-2.87 1.44 1.44 0 0 0 0 2.87ZM14 16.38a1.44 1.44 0 1 0 0-2.88 1.44 1.44 0 0 0 0 2.88Zm11.8-5.15a1.44 1.44 0 1 0 0 2.87 1.44 1.44 0 0 0 0-2.87Zm0 4.52a1.44 1.44 0 1 0 0 2.87 1.44 1.44 0 0 0 0-2.87Zm0 0',
        'vw' => 28
    ],
    'uniregistry' => [
        'p' => 'M12 30q-3.69 0-6.69-2H18.7q-3 2-6.7 2zM6.44 17.94v.44H0v-1.25h6.44v.8zm1.25 3.56q.87 1 2.06 1.56H1.13Q.75 22.31.5 21.5h7.2zm-1.07-2.19q.2.75.57 1.44H.3l-.25-1.44h6.57zM6.45 15v1.06H0V15h6.44zm0-10.81v.31H0v-.3h6.44zm0-2.19v.13H0V2h6.44zm0 4.31v.5H0v-.5h6.44zm0 6.5v.94H0v-.94h6.44zm0-4.31v.63H0V8.5h6.44zm0 2.19v.75H0v-.75h6.44zM16.8 20.75q.38-.69.57-1.44h6.56q-.06.75-.25 1.44H16.8zm.75-11.63V8.5H24v.63h-6.44zm0 2.32v-.75H24v.75h-6.44zm0-4.63v-.5H24v.5h-6.44zm0-4.81H24v.13h-6.44V2zM4.88 27.69q-1.25-.94-2.07-1.88h18.31q-.87 1-2 1.88H4.88zM17.55 4.5v-.31H24v.31h-6.44zm0 9.25v-.94H24v.94h-6.44zM2.44 25.38q-.25-.32-1-1.75h21.19q-.5.93-1.13 1.75H2.44zm11.81-2.32q1.19-.56 2-1.56h7.25l-.56 1.56h-8.7zm3.31-5.12v-.82H24v1.25h-6.44v-.43zm0-1.88V15H24v1.06h-6.44z',
        'w' => 0.75,
        'vw' => 24
    ],
    'unity' => [
        'p' => 'M31.13 12.9 27.83.92l-12.32 3.2-1.82 3.14-3.7-.03L.97 16 10 24.77l3.7-.03 1.83 3.13 12.31 3.2 3.3-11.97-1.87-3.1ZM13.98 7.76l9.43-2.35L18 14.52H7.18Zm0 16.48-6.8-6.76H18l5.41 9.11Zm12.07.87L20.64 16l5.4-9.11L28.67 16Zm0 0',
        'vw' => 36
    ],
    'unsplash' => [
        'p' => 'M28 14.39V30H0V14.39h8.82v7.8h10.36v-7.8ZM19.18 2H8.82v7.8h10.36Zm0 0',
        'vw' => 28
    ],
    'untappd' => [
        'p' => 'M25.06 3.13q-1.12 2.3-2.1 4.15t-1.5 2.81-.93 1.7-.56 1.02-.25.57-.16.56L19.25 16q-.19 1.13-.88 2.07l-9.25 13Q8.32 32.13 7 32q-1.88-.25-3.5-1.43t-2.44-2.88q-.56-1.19.19-2.25l9.31-13q.69-.94 1.69-1.5l1.81-.94q.38-.19.57-.37t.68-.72 1.16-1.28 2.31-2.47 3.78-3.85q.13-.5.25-.5.32-.18.38-.3l-.06-.26q0-.25.25-.25.68 0 1.56.63.94.62 1.12 1.3.07.2-.12.26l-.31.06q-.2.07-.13.5 0 .13-.44.38zm-10.68-.88q-.44-.19-.44-.31.25-.7 1.12-1.32T16.7 0q.18 0 .12.5 0 .19.44.31.12 0 .19.5l1.62 1.63q.13.12 0 .31-1.25 1.25-2.56 2.63-.13.12-.19 0l-1.37-2.75q-.07-.07-.13-.07-.31-.19-.31-.31.06-.44-.13-.5zm24.37 23.19q.75 1.06.19 2.25-.5 1.12-1.44 2.06t-2.1 1.5T33 32q-1.31.13-2.06-.94l-9.32-13q-.68-.93-.87-2.06-.19-1 0-1.88 0-.18.06-.3t.13-.32.28-.56.5-.88.84-1.56 1.25-2.32q.07-.18.19 0l.44.47.37.41.53.53.25.19.35.22 1.56.78.56.34.53.4.41.38.44.53z',
        'w' => 1.25,
        'vw' => 40
    ],
    'ups' => [
        'p' => 'M6.19 18.94v-7.69h2.06v8.81q-.5.38-1.44.6t-2.03.25-1.87-.82-.78-2.46v-6.38h2.06v6.5q0 .88.44 1.19t.87.22.69-.22zM0 4.62Q4.63 2.2 11.19 2.04t12.25 2.6v13.8q0 1.07-.19 2.04t-.44 1.69-.87 1.5-1.03 1.25-1.44 1.12-1.56 1-1.91 1-2 .94-2.25.97l-2.31-.97q-.68-.28-2-.9t-1.91-1-1.56-1-1.4-1.16-1.07-1.28-.88-1.47-.43-1.69-.2-2.03V4.63zm22.38 13.82V4.94q-2.44-.31-5.32-.19t-5.5.56-5.4 1.78-5.1 3.47v7.88q0 1.62.47 2.87t1.13 2.16 2.18 1.87 2.91 1.63 4 1.84l3.34-1.56q1.21-.56 2.63-1.34t2.12-1.47 1.38-1.6.9-1.97.25-2.43zM9.24 11.75q.69-.44 1.69-.63t2.12 0T15 12.5t.81 3.38q0 3.12-1.46 4.21t-3.04.66v4.31H9.25V11.75zm2.06 7.31.35.13q.15.06.56 0t.72-.32.56-1.03.25-1.96q0-1.07-.22-1.82t-.5-1.03-.65-.37-.6-.03-.47.18v6.25zm4.94-5.37q0-1.88 1.84-2.5t3.16.5v1.75q-.81-.88-1.9-.85t-1.1 1.03q-.06.5.47.94t1.19.72 1.15 1.03.5 1.88q-.06 1.87-1.78 2.5t-3.4-.44v-1.88q.93.88 2.06.82t1.06-1.13q0-.56-.5-1t-1.1-.72-1.12-.97-.53-1.68z',
        'w' => 0.732,
        'vw' => 23
    ],
    'usb' => [
        'p' => 'M40.13 16q0 .31-.32.5l-5.56 3.31q-.19.07-.31.07t-.25-.07q-.32-.12-.32-.43v-2.25H18.5q.5.8 1.31 2.68t1.5 2.94 1.5 1.06h1.69v-1.68q0-.57.56-.57h5.56q.57 0 .57.57v5.56q0 .56-.57.56h-5.56q-.56 0-.56-.56V26h-1.7q-1.56 0-2.78-1.37t-1.84-3.07-1.44-3.06-1.69-1.38h-6.3q-.38 1.44-1.57 2.38t-2.75.94q-1.8 0-3.12-1.32T0 16t1.3-3.13 3.13-1.3q1.57 0 2.75.93t1.57 2.38q.31 0 .84.03t.69 0 .53-.07.5-.18.4-.41.54-.69.53-1.03.65-1.4q.88-1.94 1.54-3t1.43-1.54 1.35-.56T19.56 6h.69q.37-1 1.22-1.63t1.9-.62q.7 0 1.32.28t1.06.72.72 1.06.28 1.31q0 .88-.47 1.66T25.06 10t-1.69.44q-1.06 0-1.9-.63t-1.22-1.56h-1.88q-.81 0-1.5 1.03t-1.5 2.9-1.37 2.7h19.37v-2.2q0-.3.28-.5t.6 0l5.56 3.38q.31.13.31.44z',
        'w' => 1.251,
        'vw' => 40
    ],
    'usps' => 'M27.06 15.13q-.12.24-.18.12-.07-.06 0-.13 0-.18.12-.5t.19-.53l.12-.43.03-.38-.22-.16q-.43-.12-1.9-.06t-2.4.13-1.2-.07q0-.06.44-.12t1.03-.19.91-.25q.5-.5.69-.5h1.69q1.37 0 1.59.56t-.9 2.5zm-7 1.37-1.62.63-1.97.8-2.03.91-2.69 1.32-3.03 1.5-4 1.97L0 26l3.25-15.37h5q3.44 0 6.38.03t5.09.12 3.37.4 1.29.88H12.13L14 17.88q1.38-.63 3.1-1.32t3.96-1.34 3.75-.72q1.7.06 1.63.38-.06.06-.81.18t-2.29.5-3.28.94zM4.25 6h28.38l-4.26 20H1.95l25.43-10q.07-.13.22-.34t.47-.82.47-1.1.13-1.08-.4-.91q-.13-.19-.32-.25t-.66-.13-.87-.06H25q-.07-.69-2.6-1.34t-10.9-2.4Q7 6.56 4.24 6z',
    'ussunnah' => 'm9.81 17.81.32.94h-.5q-.13-.44-.25-.62-.13-.38 0-.63l.25-.31q0 .25.18.62zM28.37 19q-.18 1.25-.62 2.5l2.69 1.44-2.82-1.19q-.5 1.25-1.25 2.31L28.5 26l-2.31-1.69q-.75 1.13-1.75 2.07L26 28.5l-1.81-2q-1 .94-2.25 1.63l1 2.3-1.2-2.24q-1.24.69-2.62 1l.44 2.44-.69-2.38q-1.37.38-2.75.38L16 32l-.13-2.37q-1.37-.07-2.75-.38l-.68 2.38.43-2.44q-1.37-.31-2.62-.94l-1.19 2.19.94-2.31q-1.19-.7-2.25-1.57L6 28.5l1.56-2.12q-1-.94-1.81-2L3.5 26l2.12-1.87Q4.87 23 4.37 21.75l-2.8 1.19 2.68-1.44q-.5-1.25-.69-2.5l-3.19.56 3.2-.87q-.13-.81-.13-1.63v-.87L0 16l3.5-.19q.12-1.19.5-2.37l-3.63-1 3.7.62Q4.43 12 5 10.94L1.56 9.06l3.69 1.57q.56-1 1.31-1.88L3.5 6l3.37 2.44Q7.62 7.63 8.5 7L6 3.5l2.87 3.25q.88-.62 1.88-1.12L9.06 1.56 11.2 5.5q1-.44 2-.69L12.44.38l1.18 4.37q1.07-.19 2.13-.25L16 0l.25 4.5q1.06 0 2.06.25L19.56.37l-.75 4.44q1.07.25 2 .7l2.13-3.95-1.7 4.07q.95.5 1.88 1.12L26 3.5 23.5 7q.87.69 1.62 1.44L28.5 6l-3.06 2.75q.75.88 1.3 1.88l3.7-1.57-3.5 1.88q.56 1.06.93 2.19l3.75-.7-3.62 1q.38 1.2.5 2.38l3.5.2-3.5.18v.87q0 .75-.06 1.63l3.19.87zM27 17.06q0-4.56-3.22-7.81T16 6 8.19 9.25t-3.25 7.81 3.25 7.78T16 28.06t7.78-3.22T27 17.06zm-1.5 0q0 3.94-2.78 6.75t-6.75 2.82-6.75-2.82-2.78-6.75 2.78-6.75 6.75-2.81 6.75 2.81 2.78 6.75zm-13.31-2q0 .2.25.32.31.18.5.5.25-.5.25-.7t-.19-.43-.19-.5q-.62.63-.62.81zm-2.56-1.19q0 .2.24.32.32.19.5.5.26-.5.26-.69t-.2-.44-.18-.5q-.63.63-.63.81zm-1.2 0q0 .2.26.32.31.19.5.5.25-.5.25-.69t-.19-.44-.19-.5q-.62.69-.62.81zm12.82 5.5q-.5-.18-.5-1v-7q-.81 1.2-1.13 1.2.57 1.3.57 1.74v4q0 .44-.32.44h-1.12q-.13-.31-.94-2.56-.12.12-.69.87.25.2.38.38.37 1.06.44 1.31h-.57q-.43-1.19-.75-2-.06.19-.62.88.19.18.31.37 0 .06.1.31t.15.38H16q-.06-.13-.25-.66t-.25-.72q-.13.13-.69.88.25.19.31.37 0 .07.04.1l.03.03h-1.13l-.31-.81q-.2-.5-.28-.82t-.1-.5q0-.25.07-.5-.88 1.07-.88 1.57 0 .25.31 1.09t.32.97q0 .19-.07.31-1-1.31-1.18-1.31h-.75l-.7-1.75q-.37-.88-.37-1.19t.13-.75l-.16.19-.22.25q0-.06-.12-.31t-.13-.38l-.75.88q.07 0 .13.15l.12.32.07.28q-.07.06-.38.47t-.44.59q-.43.62-.18 1.37.18.44-.13.88h3.88q.25 0 .59.5t.34.87q.75-.8 1.07-1.3h6.68q.38 0 .63-.38zm1.25-1.68v-6.32q-.81 1.13-1.25 1.13.63 1.38.63 1.81v4.13q0 .75.37 1t1.13.25.87-.13l-.25-.1-.22-.05q-.1-.02-.16-.1-1.12-.44-1.12-1.62z',
    'vaadin' => 'M14 8.81q.06-.62.16-1.03t.37-1.03.97-.94 1.63-.31h6.18q2 0 2-1.37v-.75q0-.57.38-.97t.94-.41q.37 0 .68.19t.5.5.2.69v2.3q0 2-1.04 2.95t-3.15.93h-6.25q-.7 0-1.13.13t-.63.4-.24.54-.07.62v.19q-.06.63-.5 1.03t-1 .4-1-.4-.44-1.03q-.06-.12-.06-.19 0-.5-.1-.81t-.59-.6-1.31-.28H4.19q-2.13 0-3.16-.93T0 5.69V3.37Q0 3 .19 2.7t.5-.5.69-.2.65.2.47.5.19.68v.75q0 1.38 2 1.38h6.19q1 0 1.65.34t.94.94.38 1T14 8.81zm0 19.69q-1.13 0-1.63-1 0-.06-1.53-2.72l-3.06-5.31-1.53-2.72q-.56-.88 0-1.81T7.94 14q1.31 0 1.87 1.25l4.2 7.37 4.18-7.37Q18.75 14 20.06 14q1.13 0 1.7.94t0 1.8q-.07.07-3.1 5.38t-3.03 5.38q-.5 1-1.63 1z',
    'viacoin' => [
        'p' => 'm24 2-3 7h3v3h-4.25l-.88 2H24v3h-6.44L12 30 6.44 17H0v-3h5.12l-.87-2H0V9h3L0 2h4l5.06 12h5.88L20 2h4zM12 21l1.69-4H10.3z',
        'w' => 0.75,
        'vw' => 24
    ],
    'viadeo' => [
        'p' => 'M17.25 9.38v.06Q15.69 4.8 12.81 0q1.94 1.25 3.1 4t1.34 5.38zM19.31 17q1.38-.25 2.63-1.06.81 1.93.9 4.03t-.68 4.4-2.47 4.13q-3.2 3.5-8.13 3.5t-8.12-3.5q-2.5-2.63-3-6.13t.68-6.43 3.97-5 6.47-2.07q2.07 0 3.88.7-.69 1.3-.82 2.68-1.43-.56-3.06-.56-3.62 0-6.06 2.62t-2.44 6.25q0 2.13.88 3.97t2.46 3.06 3.66 1.6q2.19-.88 3.72-2.9t2.25-4.7 1-5.06.28-4.84q0-1.5-.06-2.25 1.19 3.43 1.19 7 0 8.56-6.82 12.8l-.25.07q3.7.07 6.16-2.5T20 20.56q0-1.87-.69-3.56zM26.13.75Q27.68 3 27.75 6.09t-1.4 5.38-3.85 2.28q-1.75 0-3.44-1.63 4.69-2.56 6.25-5.24.38-.82.44-1.25-.94 2.06-3 3.62t-4.38 2.06q-.75-1.18-.75-2.44 0-1.87 1.25-3.12.82-.75 2.32-1.19T24 3.34t2.13-2.6z',
        'w' => 0.873,
        'vw' => 28
    ],
    'viadeo-square' => 'M25 2q1.25 0 2.13.88T28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22zm-7.44 21.81q1.57-1.68 1.88-3.94T19 16q-.81.44-1.69.63.44 1 .44 2.25 0 2.3-1.53 3.9t-3.84 1.53h.12q4.25-2.69 4.25-8 0-2.25-.69-4.37v-.07q-.12-1.06-.44-2.15t-.93-2.13T13.25 6q1.81 3.07 2.81 5.95v1.37q0 1.56-.19 3.03t-.62 3.16-1.4 2.94-2.35 1.81q-1.88-.38-3.1-1.88t-1.22-3.5q0-2.25 1.54-3.9t3.78-1.66q1 0 1.87.38.13-.88.57-1.69-1.2-.44-2.44-.44-2.32 0-4.07 1.28t-2.5 3.13-.4 4.03 1.84 3.81q2 2.19 5.1 2.19t5.09-2.19zm1.75-9.19q1.5 0 2.4-1.43t.88-3.38-1.03-3.37q-.5 1.18-1.31 1.65t-1.75.75-1.44.72q-.81.81-.81 2 0 .81.5 1.5 1.44-.31 2.72-1.28t1.84-2.22q0 .25-.25.75-1 1.69-3.87 3.31 1 1 2.12 1z',
    'viber' => 'M27.75 3.13q2.06 1.8 2.94 5.71t.19 9.66q-.2 1.44-.6 2.63t-.9 2.03-1.16 1.53-1.25 1.1-1.28.71-1.13.47-.96.28-.6.13q-.44.18-1.5.37t-3.37.4-4.63-.02l-1.87 2.06q-.2.19-.5.53t-.44.47l-.4.4-.41.28-.44.13-.44-.06q-.25-.07-.44-.25t-.25-.4-.1-.45-.02-.34l-.07-.13.07-3.25h-.07q-1.3-.37-2.43-1.03t-1.85-1.47-1.31-1.78-.88-1.97-.5-2.06-.25-2.06-.06-1.94.03-1.69Q1 6.31 4 3.31 5.87 1.62 9 .8t5.37-.8h2.19q3.25.06 5.9.72t3.66 1.25 1.63 1.15zm.88 14.8q.62-4.87-.1-8.27t-2.47-4.9q-.44-.45-1.37-.95t-3.22-1-5.03-.56h-.78l-1.94.1-2.69.3-2.78.82L5.8 5q-2.55 2.5-2.68 8.2.07 1.12 0 2.4t.25 2.76.85 2.75T6 23.47t3.06 1.6L9 30.43q0 .44.25.53t.5-.22L14 25.81q1.94.13 3.94-.03t2.93-.34 1.32-.31q.06 0 .6-.13t.93-.28 1.1-.47 1.21-.84 1.1-1.25.93-1.82.57-2.4zm-8.7-5.05q0 .43-.4.43t-.4-.37q-.13-2-2.07-2.13-.25 0-.34-.22t.03-.4.38-.19q2.68.13 2.8 2.88zm1.26.75q.06-2.07-1.19-3.44t-3.56-1.57q-.25 0-.32-.21t.04-.41.34-.19q2.62.19 4.1 1.81t1.4 4q0 .25-.19.35t-.4 0-.22-.35zm2.94.8q0 .26-.2.35t-.4 0-.22-.34q0-3.69-2.15-5.78t-5.41-2.1q-.38 0-.38-.4t.38-.4q3.63 0 6 2.34t2.38 6.34zm-.7 6.13q-.62 1.07-1.5 1.72t-1.74.35q-1.06-.32-3.03-1.38t-3.35-2.12q-1.5-1.2-2.68-2.7-1-1.3-1.88-2.87-.62-1.06-1.03-1.93t-.53-1.25l-.06-.32q-.32-.87.34-1.75t1.72-1.5q.87-.44 1.5.25.75.94 1.12 1.38.44.62.94 1.5.56 1.06-.25 1.69l-.75.56q-.19.12-.25.34t-.06.4v.13l.15.47.5 1.07.94 1.43 1.5 1.35 2.16 1q.56 0 .87-.38l.63-.75q.62-.81 1.62-.25 1.75 1 2.88 2.07.69.62.25 1.5z',
    'vimeo' => 'M25.19 2q1.19 0 2 .81t.81 2V27.2q0 1.19-.81 2t-2 .81H2.8q-1.19 0-2-.81t-.81-2V4.8q0-1.19.81-2t2-.81h22.4zm-1.63 9.31q.2-4-2.94-4.06-4.18-.12-5.68 4.75.94-.56 1.97-.28t.84 1.65q-.06 1.07-1.31 3.07-1.7 2.62-2.44 1.75-.75-.94-1.81-7.44-.57-3.5-2.88-3.25-.44.06-.94.31t-1.15.78-1.13.94l-1.4 1.22-1.38 1.19 1 1.25q1.31-.94 1.56-.94.75 0 1.32 1.47t1.62 5.34q.44 1.57.63 2.44 1.25 3.25 3 3.25 2.87 0 7-5.37 4-5.13 4.12-8.07z',
    'vimeo-square' => 'M25 2q1.25 0 2.13.88T28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22zm-1 9.38q.19-3.94-2.94-4-4.12-.2-5.62 4.68.94-.56 1.97-.28t.84 1.66q-.06 1-1.31 3-1.7 2.62-2.44 1.69t-1.75-7.32q-.25-1.75-.97-2.53t-1.9-.65q-.38 0-.88.25t-1.16.78-1.12.93-1.38 1.22T4 12.01l.94 1.18q1.31-.94 1.56-.94.44 0 .75.38t.69 1.47.65 1.96.79 2.94l.68 2.44q1.2 3.19 2.94 3.19 2.81 0 6.88-5.25 4-5.13 4.12-8z',
    'vimeo-v' => 'M28 9.63q-.19 4.06-5.75 11.18-5.69 7.44-9.63 7.44-2.43 0-4.12-4.56-.38-1.19-.94-3.38-.75-2.81-1.12-4.1t-.9-2.77-1-2.03-1.04-.53q-.31 0-2.19 1.3L0 10.5q.62-.5 1.87-1.66t1.91-1.71 1.6-1.32T7 4.75t1.25-.38q3.25-.3 4 4.44 1.5 9.44 2.62 10.44 1 .94 3.13-2.37l.12-.13q1.75-2.81 1.88-4.25.12-1.31-.53-1.9t-1.56-.5-1.85.46Q18.12 3.81 23.94 4q4.3.13 4.06 5.63z',
    'vine' => [
        'p' => 'M24 15.94v3.25q-1.69.37-3.25.37-1.56 3.25-4.03 6.13t-3.78 3.62-2.69-.06q-.69-.37-1.53-1.12t-2.19-2.6-2.5-4.19-2.28-6.28T0 6.44h4.63q.8 6.81 2.65 11.15t4.53 7.66q2.82-2.75 4.75-6.63-2.31-1.18-3.65-3.62t-1.35-5.5q0-3.13 1.72-5.16t4.66-2.03q2.5 0 4 1.28t1.81 3.16.13 3.69-.82 3.19l-.87.12q-.5.06-1.57-.25t-1.56-1.12q.5-1.7.5-3.07 0-2.87-1.75-2.87-.87 0-1.4.81t-.54 2.31q0 3.63 2.38 5.4t5.75.98z',
        'w' => 0.75,
        'vw' => 24
    ],
    'vk' => [
        'p' => 'M34.06 7.38q-.18.93-1.47 3.06t-2.34 3.72-1.13 1.65q-.37.57-.37.82t.38.87l.4.44.97 1 1.25 1.31 1.28 1.47 1.06 1.47.66 1.25Q35.19 26 33.5 26h-3.69q-.44 0-.75-.12t-.69-.44-.68-.69-.94-1.06-1.25-1.31Q23 20 21.8 20q-.75 0-.87.44t-.13 2.5q.06 1 .06 1.62 0 .75-.5 1.1T18.2 26q-2.94 0-5.94-1.75t-5.25-5q-2.13-2.87-3.53-5.72T1.62 9.2t-.44-2.13Q1.18 6 2.43 6h3.7q.68 0 1.02.31t.6 1.13q1.31 3.81 3.22 6.81t2.9 3q.38 0 .53-.28t.16-1.1v-5.43q-.06-.88-.31-1.5t-.5-.9-.47-.6-.22-.56.12-.44.35-.32T14 6h5.8q1 0 1 1.38v7.25q0 1.06.6 1.06t1.72-1.13q1.19-1.31 2.32-3.19t1.68-3.12l.57-1.19Q28.12 6 29.25 6h3.69q1.5 0 1.12 1.37z',
        'w' => 1.123,
        'vw' => 36
    ],
    'vnv' => [
        'p' => 'M6.56 22q-.93 0-1.65-.47t-.97-.97l-.31-.44-3.44-7q-.07-.06-.1-.12t-.06-.25 0-.35.22-.28T.8 12h2.06q.2 0 .32.06t.25.19.19.22.22.34.21.32l2.25 4.68q.38.82 1.32.82.43 0 .78-.22t.47-.4l.12-.2 2.25-4.68q.06-.07.19-.32t.19-.34.19-.22.28-.19.34-.06h2.06q.57 0 .72.28t.03.6l-3.62 7.24q0 .07-.06.2t-.32.43-.56.56-.81.47-1.13.22H6.56zm24.69 0q-.94 0-1.66-.47t-.96-.97l-.25-.44-3.5-7q-.7-1.12.62-1.12h2.06q.2 0 .32.06t.25.19.18.22.22.34.22.32L31 17.8q.38.82 1.31.82.44 0 .79-.22t.46-.4l.07-.2 2.31-4.68.22-.32q.16-.25.19-.34t.19-.22.28-.19.31-.06h2.06q.57 0 .72.28t.03.6l-.12.24-3.44 7q-.06.07-.12.2t-.32.43-.56.56-.81.47-1.13.22h-2.19zM21.12 12q.94 0 1.66.47t.97.97l.25.44 3.5 7q.63 1.12-.63 1.12h-2.06q-.19 0-.31-.06t-.22-.13-.19-.18-.15-.22-.16-.29-.16-.25l-2.3-4.68q-.32-.82-1.32-.82-.44 0-.78.22t-.4.4l-.13.2-2.32 4.68q0 .07-.15.32t-.22.34-.19.22-.28.19-.28.06h-2.06q-.63 0-.79-.28t-.03-.6L16 13.89q.06-.07.12-.2t.32-.43.53-.56.81-.47 1.1-.22h2.24z',
        'w' => 1.25,
        'vw' => 40
    ],
    'vuejs' => 'M22.31 4h5.7L14 28 0 4h11l3 5.56L17.5 4h4.8zM3.5 6 14 24 24.5 6h-3.38L14 18.38 6.87 6H3.5z',
    'watchman-monitoring' => 'M16 1a15 15 0 1 0 0 30 15 15 0 0 0 0-30ZM7.6 26.82a13.68 13.68 0 0 1-4.7-14.83l6.43-1.12-.01.69-.87.12-.01 1.23c0 .25.1.49.3.64l.6.47Zm6.61-17.7.53-.48s.44-.34 0-.59c-.46-.25-2.48-2.15-2.48-2.15-.33-.34-.51-.46-.96 0 0 0-2.03 1.9-2.48 2.15-.44.25 0 .6 0 .6l.53.47v.27l-4.6-1.2A13.7 13.7 0 0 1 26.31 7L14.2 9.33Zm2.13 20.58-2.12-15.68.6-.47c.2-.15.3-.39.3-.64l-.01-1.23-.87-.12-.01-.65 15.11 1.96a13.72 13.72 0 0 1-13 16.83Zm0 0',
    'waze' => 'M31.39 12.6c.9 5.37-1.94 10.5-7.08 13.01a3.24 3.24 0 1 1-6.24 1.33c-.4 0-4.01 0-4.77-.04A3.23 3.23 0 1 1 7 25.75a13 13 0 0 1-6.23-4.43 1.63 1.63 0 0 1 1.3-2.6c2.9 0 2.02-3.4 2.7-6.9C5.93 5.95 12.07 2 18 2c6.41 0 12.33 4.42 13.39 10.6Zm-8.05 11.67c2.63-1.2 5.09-3.55 6.02-6.39C31.9 10.2 25.36 3.63 18 3.63c-5.21 0-10.64 3.47-11.63 8.5-.59 3.06.32 8.21-4.3 8.21a11.56 11.56 0 0 0 5.87 4 3.24 3.24 0 0 1 4.99.9c.89.06 4.95.08 5.5.05a3.23 3.23 0 0 1 4.91-1.02ZM12.82 11.7c0-2.18 3.18-2.18 3.18 0 0 2.17-3.18 2.17-3.18 0Zm7.29 0c0-2.18 3.18-2.18 3.18 0 0 2.17-3.18 2.17-3.18 0Zm-7.67 4.41c-.21-1.06 1.39-1.38 1.6-.32v.01c.26 1.34 1.87 2.75 4.01 2.7 2.24-.06 3.7-1.4 4.01-2.68.28-1 1.79-.64 1.6.38-.33 1.39-1.96 3.87-5.72 3.93-2.66 0-5.06-1.74-5.5-4.02Zm0 0',
    'wechat' => 'weixin',
    'weebly' => 'M26.56 4.13q2.38 0 3.9 1.46T32 9.31q0 .82-.31 1.82T29.56 17q-.44 1.19-.72 1.9t-.75 2.07-.9 2.53q-1.63 4.38-5.82 4.38-3.8 0-5.37-3.38-1.56 3.38-5.38 3.38-4.18 0-5.8-4.32-.57-1.68-1.23-3.4t-1.03-2.78-.78-2.16-.62-1.72-.44-1.28-.32-1-.21-.69-.13-.6-.03-.46T0 8.94q0-2 1.6-3.4t3.9-1.41q1.87 0 3.28 1.09t1.9 2.9q.57-1.62 1.7-2.59t1.96-1.19T16 4.13q.62 0 1.25.12t1.44.5T20.22 6t1.1 2.13q.5-1.82 1.93-2.91t3.31-1.1zm1.63 7.18q.5-1.5.5-1.75 0-.69-.32-1.19t-.84-.75-1.22-.25q-1.81 0-2.25 1.82l-2.75 8.75h-.06l-2.81-8.57q-.63-2-2.44-2t-2.44 2l-2.81 8.57h-.07L7.93 9.19Q7.5 7.37 5.68 7.37q-1 0-1.68.54t-.7 1.28q0 .81.5 2.19L7.88 22.3q.88 2.32 2.88 2.32 2.06 0 2.75-2.13l2.43-7.31H16l2.5 7.31q.62 2.13 2.75 2.13 2 0 2.87-2.32z',
    'weibo' => 'M25.44 11.13q.37-1.13-.4-2t-1.91-.63q-.7.13-1.07-.31t-.22-1.03.79-.72q2.37-.5 3.96 1.28t.85 4.03q-.19.62-.78.72t-1-.32-.22-1.03zm-12 16.8q-5.32 0-9.38-2.37T0 19.38q0-4.2 4.75-8.94 4.5-4.5 8.25-4.75 2.75-.2 2.94 2.25.06.87-.31 2.12-.2.69.74.38 3.75-1.7 6.04-.78t1.09 3.96q-.13.44 0 .57t.5.31q4.13 1.25 3.88 4.56-.2 3.32-4.2 6-4.24 2.88-10.24 2.88zm8.97-9.15q-.29-2.6-3.25-4.19t-6.97-1.15q-4 .37-6.6 2.5t-2.34 4.72 3.22 4.18T13.44 26q4-.38 6.62-2.5t2.35-4.72zm-.66-16.6q3.13-.62 5.78.82t3.84 4.25.2 5.81q-.38 1.2-1.54.82t-.78-1.57q1.06-3.3-1.25-5.87t-5.75-1.88q-.75.2-1.19-.34t-.25-1.19.94-.84zm-4.94 19.45q-.75 1.8-2.78 2.68t-4.03.2q-1.87-.57-2.6-2.23t.1-3.34 2.69-2.5 3.75-.32q2 .5 2.84 2.1t.03 3.4zm-5.37-1.88q-.63-.25-1.32 0t-1.06.81q-.37.63-.22 1.22t.78.9 1.35.07 1.1-.87.18-1.25-.81-.88zm2.06-.81q-.56-.19-.94.31-.31.56.25.81.63.25.94-.3.31-.63-.25-.82z',
    'weixin' => [
        'p' => 'M24.06 10.5q-4.31 0-7.31 2.81t-3 6.82q0 1.25.38 2.5-.63.06-1.13.06-1.19 0-4.25-.56L4.5 24.25l1.19-3.69Q.8 17.13.8 12.31.81 8 4.38 5T13 2q4.44 0 7.97 2.44t4.28 6.12q-.56-.06-1.19-.06zm-6.5-3.31q-.75 0-1.31.4t-.56 1.1.56 1.1 1.31.4q.7 0 1.1-.4t.4-1.1-.4-1.1-1.1-.4zm-8.56 3q.69 0 1.12-.4t.44-1.1-.44-1.1T9 7.2t-1.25.4-.56 1.1.56 1.1 1.25.4zm26.19 9.75q0 3.81-4.25 7.06l.94 3-3.38-1.81q-2.5.62-3.62.62-4.38 0-7.38-2.6t-3-6.24 3-6.25 7.38-2.6q4.12 0 7.21 2.63t3.1 6.19zm-13.69-1.5q.44 0 .78-.19t.53-.47.19-.6q0-.18-.1-.4t-.28-.4-.5-.29-.62-.09q-.44 0-.81.38t-.38.8q0 .26.1.48t.25.4.37.28.47.1zm6.69 0q.44 0 .78-.19t.53-.47.19-.6q0-.43-.44-.8T28.19 16q-.44 0-.82.37t-.37.85.37.84.82.38z',
        'w' => 1.125,
        'vw' => 36
    ],
    'whatsapp' => 'M23.81 6.06q4.2 4.2 4.2 9.82 0 5.75-4.13 9.8T14 29.76q-3.56 0-6.62-1.69L0 30l2-7.19Q.13 19.63.13 15.88q0-5.75 4.06-9.82T14 2t9.82 4.06zM14 27.44q4.75 0 8.19-3.4t3.44-8.16q.06-4.63-3.5-8.13Q18.75 4.38 14 4.38T5.84 7.75t-3.4 8.13q0 3.3 1.81 6.12l.25.44-1.19 4.25 4.38-1.13.44.25q2.68 1.63 5.87 1.63zm6.31-8.69q.63.31.7.44.18.37-.26 1.69-.19.5-.97 1.03t-1.34.6q-1.63.24-4.19-.88-3.06-1.32-5.88-5.25-1.43-1.94-1.43-3.57 0-1.56 1.18-2.94.44-.37.94-.37h.7q.55 0 .74.56l1.06 2.57q.2.37.07.62-.25.56-.63.97t-.44.47.07.44q.87 1.62 1.9 2.43t2.78 1.7q.5.24.75-.07.82-.94 1.13-1.38.25-.37.75-.18t2.37 1.12z',
    'whatsapp-square' => 'M13.97 7.69q3.4 0 5.84 2.44 2.5 2.5 2.5 5.8 0 3.38-2.44 5.79T14 24.12q-2.25 0-4.19-1.12l-.31-.19-3.13.82.82-3-.2-.32q-1.24-2-1.24-4.37 0-3.44 2.4-5.85t5.82-2.4zm4.9 11.75q.25-.88.13-1.19-.06-.06-.5-.25-1.38-.75-1.69-.81-.37-.2-.56.12t-.75.94q-.19.25-.56.06-1.25-.62-1.97-1.22t-1.4-1.72q-.07-.3-.04-.34t.31-.34.47-.7q.13-.18 0-.37-.06-.06-.19-.4t-.3-.82-.32-.65q-.13-.38-.5-.44h-.5q-.38 0-.69.31-.88.94-.88 2.07 0 1.25 1.07 2.56l.1.16.18.28.28.37.4.5.5.5.6.53.63.53.72.47.78.4q1.87.82 3 .63.37-.06.93-.4t.75-.78zM25 2q1.25 0 2.13.88T28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22zM14 25.81q4.13 0 7.06-2.9T24 15.94q0-4-3-7.07Q18.13 6 14 6T6.97 8.9t-2.9 6.98q0 2.68 1.3 5L4 26l5.25-1.38q2.19 1.2 4.75 1.2z',
    'whmcs' => 'm28 10.06-1.81.44-.13.75 1.32 1.19-.7 1.25-1.8-.56-.44.5.56 1.75-1.31.75-1.32-1.32-.68.2-.44 1.8h-1.38l-.44-1.8-.75-.2-1.18 1.32-1.25-.75.5-1.7-.44-.5-1.88.45-.62-1.13 1.25-1.25-.19-.75-1.75-.5V8.63l1.75-.44.19-.69-1.25-1.31.69-1.2 1.75.5.5-.55-.5-1.7 1.25-.68 1.18 1.31.75-.25.44-1.62h1.38l.44 1.62.68.25 1.32-1.31 1.18.69L24.44 5l.43.5L26.7 5l.62 1.25-1.25 1.25.19.69 1.75.56v1.31zm-7.44 2.81q.94 0 1.75-.46t1.32-1.28.5-1.82q0-1.44-1.07-2.47t-2.53-1.03-2.5 1.03T17 9.34t1.03 2.5 2.53 1.03zm4.5 1.07 2.94.87v2.5l-3.44.82-.25 1.43 2.44 2.2-1.25 2.37-3.37-1-.88.93 1.06 3.25-2.43 1.44-2.44-2.5-1.32.38-.8 3.37h-2.63l-.82-3.37-1.37-.38-2.25 2.44-2.31-1.37.93-3.13L6 23.13l-3.44.94-1.25-2.2L3.7 19.5l-.32-1.37L0 17.25V14.7l3.37-.94.38-1.25-2.44-2.44 1.31-2.25 3.32.88.93-1-.93-3.25 2.3-1.25 2.26 2.44 1.44-.38.75-3.25h2.62l.75 2.88-1.44-.38-.37-.12-.25.37-.69 1.19-.19.37.32.25L14.5 7.7v.06l-1.5.38-.38.12v.38l-.06.81q-2.25.5-3.72 2.31t-1.47 4.2q0 2.68 1.94 4.62T14 22.5q2.31 0 4.12-1.46t2.32-3.66h1.25l.12-.44.31-1.5h.13l1.37 1.38.38-.2 1.25-.74.37-.25-.12-.38z',
    'wikipedia-w' => [
        'p' => 'M40 3.19v.75Q37.56 4 36.5 6.5q-.38.81-1.72 3.9t-2.53 5.91-2.75 6.4-2.69 6.16h-.87L20.8 16.81q-.44.88-2.75 5.31t-3.44 6.75q0 .07-.46.04l-.47-.04q-1.44-3.43-4.47-10.25T4.75 8.32q-.63-1.38-2.13-2.88T0 4v-.88h10.12v.82Q9 4 8 4.63t-.63 1.5q.44.87 3.75 8.56t4.13 9.44q.44-.82 1.34-2.57t1.94-3.65 1.4-2.72q-.3-.63-2.15-4.85T15.44 5.2q-.7-1.2-3.5-1.2v-.88h8.87v.82q-1 .06-1.6.47T19 5.56l1.5 3.32q.94 2.07 1.5 3.25 1.81-3.44 3-6.32.31-.62.12-1.03t-.81-.6T22.62 4v-.88h7.69v.82q-2.38.12-3.63 2L23 13.63q.18.56 2.19 5.03t2.12 4.72l7.63-17.63q-.32-.94-1.25-1.34t-1.7-.4v-.88z',
        'w' => 1.25,
        'vw' => 40
    ],
    'windows' => 'm0 5.88 11.5-1.63v11.13H0v-9.5zm0 20.25v-9.38h11.5v11zm12.75 1.75V16.74H28V30zm0-23.75L28 2v13.38H12.75V4.12z',
    'wirsindhandwerk' => [
        'p' => 'M50.77161,479.81213h83.36071V367.84741l-83.36071,47.009Zm329.04675,0h82.35022V414.85645l-82.35022-47.009Zm.00568-448V251.568L256.1759,179.1861,134.50378,251.568V31.81213H50.77161V392.60565L256.1759,270.31909,462.16858,392.60565V31.81213Z',
        'vw' => 512,
        'vh' => 512
    ],
    'wix' => [
        'p' => 'M24.56 8.25v.13q0 .56-.03.9t-.47.88-1.25.84q-.31.06-.56.22t-.31.28l-.13.06q0-1.43.22-2.1t.85-.96l.4-.12.44-.1q.22-.04.44-.03h.4zm-7.18 2.19q.25-1.19 1.25-1.75t2.18-.44l-4.12 15.5q-1.32.06-2-.25-.75-.38-1.13-.9t-.75-2.04l-.65-2.65-.85-3.44q-.39-1.54-.44-1.66-.18-.68-.46-.68t-.47.68q-.07.13-.44 1.66t-.85 3.47T8 20.56q-.38 1.5-.75 2.03t-1.13.91q-.3.13-.8.19t-.88.06h-.32L0 8.25q1.19-.12 2.19.44t1.25 1.75l2.18 8.31 1.75-6.81q.38-1.56 1.07-2.28t1.97-.72 1.97.72 1.06 2.28l1.75 6.81zm7.18.31v10q0 1.38-.18 1.9t-.88.85q-.31.19-.72.25t-.72 0h-.25V13.09l.03-.28.1-.22.15-.22.22-.15.31-.16.35-.19q.22-.13.4-.18t.38-.2q.25-.12.47-.3t.28-.32zm10.25 5.31 5.2 7.7q-2.26.24-3.38-.7-.7-.62-3.32-4.43-.37-.63-.81 0-2.69 3.87-3.25 4.43-.5.44-1.34.6t-1.47.15l-.57-.06 5.2-7.69-5.2-7.75q2.2-.37 3.25.63.7.56 3.38 4.5.44.62.81 0 2.75-3.94 3.44-4.5.44-.44 1.25-.6t1.44-.1l.56.07z',
        'w' => 1.25,
        'vw' => 40
    ],
    'wizards-of-the-coast' => [
        'p' => 'M13.69 21.63q.69-.57 1.37.43.63.94 0 1.44-.68.56-1.37-.4t0-1.47zm21.06 4.75q.25-.88.25-1.32 0-1.56-1.78-2.81t-4.47-1.44q-6.06-.37-11.5 4.63h-.06q.06-.7-.25-1.32L17 24q5.56-4.44 11.75-3.75 2.88.31 4.69 1.69t1.69 3.18q-.07.75-.32 1.25h-.06zM24.5 15q0-.5.44-.94l.94-.69q-.07 1.7-.13 1.88-.12.44-.69.44-.5-.07-.56-.7zm-3.13-9.44q6.94 0 12.22 3.6t5.97 8.72q-.18.12-.56.3.06.7 0 1h-.06q0-.37-.13-.93-.5.31-1 .75.25.63.25 1.63 0 1.75-1 2.37v-.06q.75-.75.75-2 0-1.57-.87-2.57.25.5.43 1.13-1.37 1.31-1.68 3.88-.25-.57-.7-1 .76.93.76 2.37 0 1.75-1.75 3 1.37-1.19 1.37-2.94 0-1.94-2.28-3.47t-5.84-1.53q-4.31 0-8.19 2.22t-6.62 5.78h-.07q-1.75-5.19-5.31-8.47T.12 15.2l-.06-.13q.5-.56 1.28-1.34t2.88-2.35 4.37-2.75 5.75-2.12 7.03-.94zM15.2 23.7q1.06-.94.31-1.94-.81-1.19-1.94-.37-.5.37-.56 1t.25 1.06q.37.44.87.53t1.07-.28zm.12-8.5.13.5-.25-.31-.88 1.68 1.57 1.82.75-.38-.7-4.44zm1.19 7.31q.06-.06.19-.13t.25-.15l.12-.1v-.06q-.06.07-.12.03t-.22-.28-.4-.75q.3-.25.68-.31v-.06q-.06-.2-.12-.32-.2.13-.7.5h-.06q-.25-.56-.3-.62v-.06q.37-.38.74-.5.19-.07.31 0l.07-.07q-.07-.06-.13-.15t-.12-.1v-.06q-1.07.56-1.63 1v.06q.2-.06.32.13.25.44.5.81t.34.5.19.31.06.22-.03.16h.06zm3.44-2v-.06q-.07 0-.13-.03t-.25-.47-.56-1.38v-.06q.63-.19.81-.13v-.06l-.18-.31h-.07q-1.44.62-1.75.81l.07.38h.06q0-.13.15-.22t.28-.16.2-.06q.8 1.87.74 2v.06zm2.81-1.06q-.13-.07-.19-.25t-.19-.85-.18-.78q-.07-.18 0-.31-.13 0-.7.13v.06q.13 0 .2.25l.18.56h-.06l-.31.1-.5.15q0-.06-.06-.31l-.13-.25v-.32q-.19 0-.69.2.13 0 .19.24 0 .07.13.44t.28.78.15.47q.07.19 0 .31.5-.18.63-.25-.13-.06-.2-.25 0-.06-.05-.25l-.2-.56.07-.06.75-.25.06.06.2.81v.32q.24-.13.62-.2zm2-.44.13-.38h-.07q-.06.2-.47.29t-.53.03l-.03-.25-.06-.35-.03-.22q.44-.06.62 0 0-.12.03-.22l.03-.09h-.06q-.12.06-.62.13h-.07q-.06-.38-.06-.63.81-.19 1 0h.06q-.06-.19-.12-.31v-.07q-.88.07-1.56.2h-.07q.13.06.19.24 0 .2.06.5l.13.63.12.5-.06.25v.06q1-.25 1.44-.25V19zm.69-2.75q.44-.25.44-.38l.25.5 1.3.07v-.32q-.24 0-.3-.06-.07-.12-.13-.37 0-.88.06-1.32l.07-.68q.07-.62.06-.75.06-.57 0-.75-.13-.57-.56-.82t-1.1-.22-1.47.03l-.3 1 .18.2q.87-.88 1.69-.32.37.31.12.69-.12.12-.34.28t-.44.25-.5.22-.47.31q-.37.38-.5.9t.13 1.1q.37.69 1.12.63.31 0 .69-.2zm2.31 1.31v-.37q-.69-.38-1.19-.16t-.62.78q-.13.75.5 1.13t1.12.06v-.06l.13-.38q-.32.5-.88.32t-.44-.94q.13-.63.6-.75t.78.37zm.38-5.5q.3.07.3.25l-.3 3.44q-.07.44-.5.44l-.07.31q1.19.19 2.32.44l.06-.31q-.57-.13-.6-.6t.28-2.28q.57-.44 1.32-.06l.69-.69-.16-.16-.34-.21-.44-.07-.94.7.06-1.13q-.87-.44-1.69-.44v.37zm.68 7.19q.88.19 1.13-.75.12-.38-.07-.72t-.56-.47q-.87-.31-1.19.63-.12.31-.06.56t.28.47.47.28zm2.63.56q-.07-.06-.07-.18l.07-1.63q-.07 0-.22-.06t-.28-.13V18q-.38.63-.88 1.25l-.18.13q.12 0 .25.06l.12.06v-.06l.06-.19q.07-.06.13-.19t.12-.19q.07 0 .44.2v.43l-.12.13v.06q.06 0 .15.03t.22.06.19.03zm-.32-5.56q-.62.94-.43 2.1t1.06 1.15q.13.06.28 0t.22-.13l.06-.06-.12.44 1.12.5.13-.25q-.19-.25-.1-.63t.78-2.25l.22-.68.28-.75.38-.94q-.44-.44-1.44-1.31l-.12.25q.43.3.31.68l-.5 1.63q-.13-.63-.88-.6t-1.24.85zm1.82 4.94.19-.19v-.06q-.2-.32-.5-.35t-.44.22q-.13.19.12.57.2.37.13.56-.06.12-.19.12t-.25-.15-.06-.28h-.07q-.12.25-.12.3.12.26.47.29t.47-.22q.19-.31-.07-.62-.18-.44-.12-.5t.16-.07.18.1.1.28zm1.37.81.07-.25q-.25-.19-.82-.56l-.18.25q.24 0 .37.12l-.56 1.07-.13.06-.06.06.37.25q0-.12.07-.25.25-.5.56-1 .19.13.31.25zM36 17.25l.5-.63q-.06-.18-.19-.37t-.5-.47-.75-.22-.53.32-.12.59.12.62.22.57.19.28q.12.56-.25.62-.5.13-.63-.56v-.38l-.18-.12-.32.88.25.21q.12.1.47.29t.6.18q.8-.12.8-1.06 0-.25-.21-.9t-.22-.79q0-.43.44-.25.31.2.25.57 0 .06-.13.43zm2.19-.63h.06q-.94-2.93-3.19-5t-4.62-2.9-4.69-.85q-4.19-.06-8.25 2.25l1.31-3.8-2.12 4.05-.56-1.3.43-.13-.18-.5q-.2 0-.44.12l-.75-1.81.81 3.81h-.06q-1-.5-.32-1.19l-.18-.68q-.13.06-.72.21t-1.28.41L12 8.56l3.69 2.69-.32.12-1.06-.87q.13.25.31 1.25l-1.81.81-1.88-2.25-1.06.56.88 2.63-.75.31L15.94 12l-.25.63-.07-.2-.81.26.31 1.5-.25.81-1.62-1.94-2.5.63 1.18 3.69q-.74-.75-1.8-1.88-1.63-1.69-1.95-2.1t-.12-.77l.5-.44-.25-.44q-1.69 1.19-3.63 2.69l.32.37.15-.12.32-.16q.22-.1.34-.03h.06q.25 0 .63.31t2.44 2.2q2.25 2.05 4 3.62l.75-.5-1.75-5.88 2.25 2.69-.32.81 2.63-5.5q1.87 4.38 2.06 4.94l-2.12-4.25-.13.31 1.44 3.7q.06.5 0 .55t-.38.25l.2.32q1.12-.5 2.43-.94l-.06-.31q-.63.12-.75-.25-.13-.25-1.32-3.7l-.62-.3-.13-.5 3.82 2.3-.82 2.7q1.44-.38 3.13-.63l-.06-1.69-.32.13q-.06 1.31-1.19 1.37l1.38-5q-1.88 0-4.19.57l.25 1-1.56-1.2 5.69-1.3-5.32.56q4.2-2.44 8.32-2.38 2.81.07 5.15 1.03t3.7 2.44 2.05 2.56 1.04 2.1zM12.69 9.5l.19.56q.43-.12.68-.12l-.56-.5zm9 2.44L21 14.3l-1.38-1.06q.2-.75.72-1.1t.88-.27.47.06zm-1.81-7.75q4.06 0 7.5.97t5.71 2.53 3.88 3.47 2.31 3.75.72 3.53q0 1-.06 1.19h-.06V19q0-2.5-1.38-5t-3.78-4.5-6-3.25T21 5q-3.75 0-6.88.88h-.06q-.5-.5-1.37-.75v-.07q3.44-.87 7.18-.87zM33.06 14.8q-.25 1-.47 1.5t-.34.6-.31.1q-.5 0-.38-.95.07-.62.38-1.37t.68-.69q.63.06.44.81zm-3.81 2.69q.5.13.31.88-.18.8-.75.68t-.37-.93q.25-.82.81-.63zm1.44 1.31v-.06q.25-.38.44-.63l-.07.75q-.12 0-.22-.03t-.15-.03z',
        'w' => 1.25,
        'vw' => 40
    ],
    'wodu' => [
        'p' => 'M11.15 21.23H8.82l-1.8-7.26h-.04L5.2 21.23H2.82L0 10.56h2.35l1.69 7.26h.03l1.85-7.26h2.2l1.82 7.35h.03l1.75-7.35h2.3Zm5.81-7.94c2.44 0 4 1.62 4 4.09 0 2.45-1.56 4.06-4 4.06-2.42 0-3.99-1.61-3.99-4.07s1.57-4.08 4-4.08Zm0 6.55c1.45 0 1.89-1.24 1.89-2.46 0-1.25-.44-2.49-1.89-2.49-1.73 0-1.87 1.24-1.87 2.48 0 1.23.44 2.47 1.87 2.47Zm10.23.4h-.03c-.5.84-1.36 1.2-2.34 1.2-2.34 0-3.47-2-3.47-4.14 0-2.08 1.15-4 3.42-4 .91 0 1.8.38 2.3 1.15h.02v-3.9h2.12v10.68H27.2Zm-1.85-5.35c-1.4 0-1.87 1.2-1.87 2.47 0 1.21.55 2.48 1.87 2.48 1.4 0 1.82-1.22 1.82-2.5 0-1.25-.45-2.45-1.82-2.45Zm11.72 6.34h-2.02v-1.07H35a2.75 2.75 0 0 1-2.36 1.28c-2.26 0-2.83-1.27-2.83-3.18V13.5h2.12v4.37c0 1.27.38 1.9 1.36 1.9 1.16 0 1.65-.64 1.65-2.22V13.5h2.12Zm.6-2.3H40v2.3h-2.35Zm0 0',
        'vw' => 40
    ],
    'wolf-pack-battalion' => [
        'p' => 'm15 29.5-.75 1.31-.69-1.31-.69.94-.3-.75-.32.44V32q-.38-.12-.66-.6t-.44-1.21-.22-1.4-.15-1.57-.16-1.28q0-.19-.1-1.13t-.18-1.43-.44-1.5-.81-1.75-1.38-1.72-2.15-1.79q.94-2.5.19-5-1.07.13-1.72.35t-1.32.97T1.75 15l1.93.5Q2.5 17.88 2.81 20l1.75-.44.37 2.13 1.13-.44.56 2.06 1.19-.43-1.63 1.3-.56-1.74-1.5.31-.5-2.19-1.62.88q0-.19-.07-1.07t-.06-1.43.16-1.38.53-1.37l-1.63.31q-.25-1.25.44-3.56L0 13.56q0-1.25 1.22-2.4t2.65-1.29q1.06-.06 2.28-1.06t1.1-2.19q-.07-.43-.28-.9t-.54-.9-.8-.79-1.07-.47q1.25 1 1.28 2.06t-.5 1.66-1.16.84Q4 7.62 3.81 6.6t-.16-2.93T4.18 0q5.57 3.13 5.88 9.25l3.62 2.44.2 5.81-6.7-3.75-.43.44 1.06.25 2.75 2.37 1-.3 1.75 3.05.25 7.5-1.32 1 2 1 2.07-1-1.31-1 .18-7.5 1.75-3.06 1 .31 2.75-2.37 1.13-.25-.44-.44-6.75 3.75.25-5.81 3.63-2.44q.25-6.13 5.8-9.25.5 1.75.54 3.66t-.13 2.93-.4 1.53q-.63-.25-1.16-.84t-.47-1.66T24 3.56q-.57.13-1.07.47t-.84.78-.53.9-.25.91.1.85.43.75.63.65.75.53.78.35.68.12q1.44.13 2.63 1.28t1.25 2.41l-1.44-.62q.75 2.3.5 3.56L26 16.19q.3.56.46 1.37t.16 1.38-.03 1.43-.1 1.07l-1.62-.88-.44 2.19-1.56-.31-.5 1.75-1.69-1.32 1.25.44.5-2.06 1.19.44.38-2.13 1.68.44q.32-2.13-.81-4.5l1.88-.5q-.38-1.88-1.35-2.53t-2.6-.85q-.74 2.5.2 5-1.26.82-2.16 1.79t-1.4 1.71-.82 1.75-.4 1.5-.2 1.44-.15 1.13q-.06.37-.12 1.28t-.16 1.56-.25 1.4-.4 1.23-.63.6v-1.88l-.31-.44-.38.75zm4.94-5.94-1-.68 1.19-.75q-.07 1.06-.2 1.43zm1.75-3.12q-.07 1.25-.13 2-.93-.25-1.31-.44.5-.5 1.44-1.56zm1.56.37-1.5-.43q.56-.7 1.13-1.7.37.95.37 2.13zm.19-5.06q1.62 1.06 1.87 3.5-1.56-.63-2.31-.69.44-.94.44-2.81zm-.13-2.88q2.07.26 2.88 1.76-2.44.74-2.75.87 0-.63-.13-2.63zM8.63 23.57q-.13-.38-.26-1.45.57.38 1.26.75zm-1.76-3.13Q7.75 21.5 8.25 22q-.31.19-1.31.44-.07-.75-.07-2zm-1.56.37q0-1.18.32-2.12.62 1 1.18 1.69zm-.25-5.06q.07 1.88.44 2.81-.69.07-2.31.7.25-2.45 1.87-3.5zm.13-2.88q-.13 2-.13 2.63-.31-.13-2.75-.88.88-1.5 2.88-1.74z',
        'w' => 0.89,
        'vw' => 29
    ],
    'wordpress' => 'm3.88 10.56 6.3 17.38q-3.37-1.63-5.43-4.85T2.69 16q0-2.81 1.18-5.44zM25 15.31q0 1.07-1 4.32l-1.38 4.43-4.75-14.3 1.5-.13q.5-.07.47-.6t-.59-.47q-2.13.13-3.5.13l-3.5-.13q-.38-.06-.53.22t-.03.57.44.28q.68.12 1.43.12l2.07 5.69-2.94 8.75L7.88 9.75l1.5-.12q.56-.07.5-.6t-.57-.47q-2.19.13-3.56.13h-.88Q6.7 5.94 9.63 4.3T16 2.7q5.19 0 9 3.5h-.19q-.94 0-1.6.72t-.65 1.65q0 1 1.32 3.13Q25 13.56 25 15.3zm-8.75 1.88 4.06 11.19q.07.06.13.18-2.19.75-4.44.75-1.88 0-3.75-.56zm11.44-7.56Q29.3 12.56 29.3 16q0 3.63-1.81 6.69t-4.81 4.81l4.06-11.75q1-2.5 1-4.75 0-.75-.06-1.37zM31.5 16q0 6.44-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5t10.97 4.53T31.5 16zm-.69 0q0-6.13-4.34-10.47T16 1.2 5.53 5.53 1.2 16t4.34 10.47T16 30.8t10.47-4.34T30.8 16z',
    'wordpress-simple' => 'M16 .5q6.44 0 10.97 4.53T31.5 16t-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5zM2.06 16q0 4.06 2.16 7.44t5.72 5.12L3.25 10.31q-1.19 2.7-1.19 5.7zM16 29.94q2.38 0 4.63-.82-.07-.06-.13-.18l-4.25-11.7-4.19 12.13q1.94.57 3.94.57zM17.94 9.5l5 14.94 1.44-4.63q1.06-3.37 1.06-4.5 0-1.87-1.19-3.87-1.37-2.19-1.37-3.25 0-1 .68-1.75t1.7-.75q.05 0 .18.06-4-3.69-9.44-3.69-2.37 0-4.6.78t-4 2.2-3.02 3.34h.87L9 8.18q.56-.05.6.5t-.54.63q-.75.13-1.62.2l5.12 15.05 3-9.12-2.12-5.94-1.5-.19q-.31 0-.44-.28t.03-.56.53-.28q2.32.19 3.63.19l3.75-.2q.56-.06.6.5t-.54.63q-.75.13-1.56.2zM23 28.06q3.19-1.87 5.06-5.06t1.88-7q0-3.56-1.69-6.69.06.69.06 1.44 0 2.31-1.06 5z',
    'wpbeginner' => 'M28.94 20.13q1.25 1.5 1.37 3.25t-.68 3.21-2.54 2.44-3.9.97q-1.88 0-3.6-.81t-2.65-2.31h-1.88q-.93 1.5-2.65 2.3T8.8 30q-2.18 0-3.9-.97T2.38 26.6t-.7-3.21 1.38-3.25q-1.43-2.38-1.6-5t.88-4.94 2.94-4.16T10 3.1 16 2t6 1.1 4.72 2.94 2.94 4.16.87 4.94-1.6 5zm-21.25-5.2h2.62v-3.62H7.7v3.63zm13.56 5.38v-1.43q-2.81 1-5.97.43t-5.72-2.56v1.56q2.38 2.13 5.75 2.7t5.94-.7zm-9.38-5.37h10.38V11.3H11.87v3.63z',
    'wpexplorer' => 'M32 16q0 6.63-4.69 11.31T16 32.01 4.69 27.3-.01 16 4.7 4.69 16-.01t11.32 4.7T32 16zm-2 0q0-5.75-4.1-9.88T16 2 6.1 6.1 2 16t4.1 9.9T16 30t9.9-4.1T30 16zM10.06 7.81l5.44 2.32-2.31 5.43-5.44-2.3zm6.88 10.56 2.94 5.88h-.94L15.8 18l-3.06 6.25h-.88l3.2-6.69-1.38-.62.37-.88 4.25 1.82-.37.93zm-.75-7.3 4.31 1.87-1.87 4.25-4.25-1.82zm5 2.68 3.44 1.44-1.44 3.44-3.44-1.44z',
    'wpforms' => 'M28 4.69V27.3q0 1.13-.78 1.9t-1.9.79H2.68q-1.13 0-1.9-.78T0 27.32V4.68q0-1.13.78-1.9T2.68 2h22.63q1.13 0 1.9.78t.79 1.9zM25.69 27.3V4.7q0-.38-.38-.38h-.56l-6.94 4.7L14 5.87 10.19 9 3.25 4.31h-.56q-.38 0-.38.38V27.3q0 .38.38.38H25.3q.38 0 .38-.38zM9.37 11.63v2.3H4.82v-2.3h4.57zm0 4.62v2.38H4.82v-2.38h4.57zm.7-9.19 3.37-2.75H6.06zm13.12 4.57v2.3H10.94v-2.3h12.25zm0 4.62v2.38H10.94v-2.38h12.25zm-5.25-9.19 4-2.75h-7.38zm5.25 13.88v2.31H17v-2.31h6.19z',
    'wpressr' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm10.69 9.94q.25-.56-.31-.56h-8.2q-.37 0-.56.37-.18.5-1.06 2.44-.25.56.31.56h4.75q.57 0 .38.5-.13.31-.63.31h-4.93q-.5 0-.7.44-1.87 4.38-2 4.57-.05.18-.3.25t-.38-.1 0-.53q.06-.19 1.56-3.5.25-.56-.37-.56h-2.81q-.32 0-.44.25l-1.88 4.25q-.19.31-.53.19t-.15-.57q.5-1.12 1.06-2.44l2.56-5.75q.13-.25 0-.46t-.44-.22H8.44q-.38 0-.5.37l-2.44 5.5q-1.2 2.69-1.63 3.69-.12.31.06.56.13.19.91 1.03t1.22 1.28q.19.25.5.25h8.56q.38 0 .5-.3 1.94-4.32 1.94-4.45.19-.43.69-.43h4.31q.56 0 .81-.5 1.07-2.38 2.82-6.44z',
    'x-twitter' => [
        'p' => 'M389.2 48h70.6L305.6 224.2 487 464H345L233.7 318.6 106.5 464H35.8L200.7 275.5 26.8 48H172.4L272.9 180.9 389.2 48zM364.4 421.8h39.1L151.1 88h-42L364.4 421.8z',
        'vw' => 512,
        'vh' => 512
    ],
    'xbox' => 'M23.13 19.88q4.43 5.43 3.37 7.43-.38.7-2.22 1.82t-3.53 1.68q-3.06.94-6.31.63-3.7-.38-6.88-2.44-1.31-.87-1.72-1.34t-.4-1.22q0-2.94 5.56-8.88 1.5-1.56 3.19-3.06t1.94-1.44q.43.06 3.06 2.63t3.94 4.18zM11.8 9q-5.6 6.81-7.55 12.13-1 2.8-.56 4 .12.43-.07.12-.18-.19-.5-.75-2.06-3.13-2.56-7.56-.12-1.2.25-3.75.19-1 .53-2.07t.85-2.06 1.12-1.9T4.62 5.5q.57-.56.82-.63t.93.13q1.7.56 5.07 3.62zm19.44 3.94q.25 1.06.25 2.94t-.19 2.93q-.37 1.94-1.19 3.63-.75 1.69-1.56 2.75-.19.19-.19.12l.13-.43q.19-1.13-.5-3.25Q26.3 16.5 20.3 9q2.7-2.5 4.07-3.3 1.38-.88 2.25-.88.25 0 1.13.94t1.97 2.94 1.53 4.25zM8.81 2.69l-.56-.06L9.44 2Q13.38.25 17.16.53t6.9 2.22q-.62 0-1.25-.06t-2.15.34-3.35 1.28q-1.25.63-1.31.56-.06 0-.84-.37t-1.75-.78-2.25-.75-2.35-.28z',
    'xing' => [
        'p' => 'M10.19 13.13 5.8 20.87q-.5.75-1.12.75H.62q-.37 0-.53-.28t-.03-.59l4.32-7.63-2.75-4.74q-.2-.32 0-.6t.56-.28h4.06q.69 0 1.13.75zM23.94 2.87l-9 15.82 5.69 10.44q.18.3.03.59t-.53.28H16q-.69 0-1.12-.75L9.13 18.7q.06-.13 1.03-1.81t3.22-5.66 4.8-8.47Q18.64 2 19.26 2h4.13q.37 0 .53.28t.03.6z',
        'w' => 0.75,
        'vw' => 24
    ],
    'xing-square' => 'M25 2q1.25 0 2.13.88T28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22zM8.75 20q.5 0 .81-.56 3-5.32 3.13-5.5l-2-3.44q-.31-.56-.81-.56H7q-.31 0-.4.18t.03.44l1.93 3.38-3.06 5.44q-.13.25 0 .43t.38.2h2.87zM22.5 6.62q.13-.25 0-.43t-.38-.2H19.2q-.44 0-.75.57l-6.5 11.38 4.12 7.5q.32.56.82.56h2.93q.25 0 .38-.19t0-.44l-4.13-7.43z',
    'y-combinator' => 'M28 2v28H0V2h28zM14.75 18l4.88-9.13h-2.07l-2.87 5.7q-.44.87-.82 1.68l-.75-1.69-2.8-5.69h-2.2l4.75 9v5.88h1.88V18z',
    'yahoo' => [
        'p' => 'M15.75 18.25 16 32q-1.44-.25-2-.25t-2.06.25l.25-13.75q-.7-1.19-1.7-2.97t-1.74-3.1T7 9.17 5.15 6.03t-1.84-3-2-3.03q.94.25 2.13.25 1 0 2.12-.25 2 3.56 8.44 14.13.56-.94 2.4-3.94t3.32-5.47T22.44 0q.93.25 2.06.25 1.19 0 2.12-.25-2.3 3.25-10.87 18.25z',
        'w' => 0.873,
        'vw' => 28
    ],
    'yammer' => 'M26.36 9.51c.54-.43.7-1.18.38-1.8A1.44 1.44 0 0 0 25.05 7l-.16.09c-.95.52-7.4 5.04-6.67 5.55.73.5 5.13-1.48 8.14-3.13Zm0 13.57c-3-1.65-7.41-3.63-8.14-3.12-.72.5 5.72 5.02 6.67 5.54l.16.1a1.45 1.45 0 0 0 1.3-2.52ZM29 14.81h-.19c-1.08 0-8.9.86-8.5 1.65.38.8 5.2 1.18 8.63 1.18a1.44 1.44 0 0 0 .06-2.82ZM1.94 6.04a1.56 1.56 0 0 1 2.88-1.15l5.06 12.82h.08l4.81-12.73a1.47 1.47 0 0 1 2.78.96L10.7 23.03C9.54 25.98 8.38 28 4.87 28a9 9 0 0 1-1.49-.12 1.34 1.34 0 0 1 .43-2.61l.87.04c1.9 0 2.73-1.18 3.63-3.72Zm0 0',
    'yandex' => [
        'p' => 'M9.56 19.75 4.12 32h-4l6-13.13q-4.68-2.37-4.68-8.8 0-4.82 2.65-7.45T10.75 0h5.12v32h-3.43V19.75H9.56zm2.88-16.88h-1.81q-2.38 0-3.94 1.75t-1.57 5.44q0 6.82 5.5 6.82h1.82v-14z',
        'w' => 0.5,
        'vw' => 16
    ],
    'yandex-international' => [
        'p' => 'M8.13 32V21.62L1.13 3h3.5l5.12 14.38L15.63 0h3.24l-7.56 21.75V32H8.12z',
        'w' => 0.625,
        'vw' => 20
    ],
    'yarn' => 'M24.63 21.56q-1 .25-1.91.78t-2.03 1.16-2.57 1.06q-.18.25-.62.32-1.31.3-4.31.37-1.13.06-1.38-.5-.12-.25-.1-.47t.13-.4.22-.29.25-.15l.07-.06q-.25-.2-.5-.44l-.13-.06q0 .12-.1.5t-.21.68-.32.5q-.75.75-2.18.07-.63-.38.06-1.25-.38.25-.69-.2-.25-.43-.37-.93t.09-1.37 1.03-1.57q-.44-3.31 2.5-5.37-.12-.13-.28-.4t-.5-1.04-.03-1.25q.31-.81.56-.87.69-.25.94-.57.63-.62 1.34-.9t1.16-.28l.44.06.69-1.44q.43-.69.8-.5.07.06.2.28t.28.5.28.53.19.44l.12.19.94-.31q.81 1.62.31 4.06-.12.69-.37 1.31t-.47.97-.5.78-.35.56q-.06 0 .22.22t.72.85.75 1.5q.32.75.35 1.6t-.07 1.34l-.09.5q.12 0 .31.03t.75-.22 1.25-.63q1.7-1.06 2.82-1.06.75 0 .84.62t-.53.75zM31 16q0 6.44-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5t10.97 4.53T31 16zm-4.94 4.69q-.12-.63-.6-1.03t-1.15-.4q-1.37.05-3.31 1.24-.44.25-.75.38.19-2.94-1.81-4.94 1.12-1.63 1.43-3.31.25-1 .13-2.41t-.56-2.28q-.38-.69-1.32-.44-.56-1.25-.93-1.5-.07-.06-.13-.06t-.25-.07-.34-.06-.4.1-.5.28-.5.59-.5.97q-1.76.12-2.95 1.44-.12.12-.62.31-.63.25-1.06 1.44-.5 1.31.43 2.81-2.31 2.13-2.25 5.19-.75.75-1 1.62t-.1 1.57.35 1.12.38.63q-.13 1.06.75 1.5 1.5.81 2.75.19.62.62 1.87.62 2.94 0 4.57-.38.5-.12.93-.43 2.13-.75 3.88-1.82 1.25-.87 2.31-1.12.44-.06.72-.34t.4-.63.13-.78z',
    'yelp' => [
        'p' => 'm2.63 15 6.25 3.06q.87.44.78 1.44t-1.1 1.25l-6.69 1.63q-.68.18-1.21-.2t-.6-1q-.31-2.74.56-5.3.25-.63.85-.88t1.15 0zm2.75 15q-.57-.38-.63-1.03t.38-1.16l4.62-5.12q.69-.75 1.6-.38t.9 1.38L12 30.63q0 .62-.5 1.03t-1.19.34q-2.69-.5-4.94-2zm9.06-6.88q-.5-.87.12-1.65t1.57-.47l6.56 2.13q.62.18.87.78t0 1.21q-1.18 2.44-3.3 4.2-.5.43-1.13.3t-1-.62zm9.31-8.3q.25.62-.03 1.21t-.9.78l-6.7 1.88q-.93.31-1.53-.5t-.03-1.69l3.88-5.69q.37-.56 1-.65t1.12.34q2.07 1.81 3.2 4.31zM3.81 1.86Q7.2.31 10.81 0q.63-.06 1.1.38t.47 1.06v13q0 1.12-1.07 1.4T9.7 15.2L3.19 3.88q-.32-.5-.13-1.13t.75-.88z',
        'w' => 0.744,
        'vw' => 24
    ],
    'yoast' => 'M5.69 4.75H17.3l-.43 1.19H5.68q-1.8 0-3.15 1.28t-1.34 3.1v12.87q0 1.81 1.34 3.1t3.9 1.27v1.2H5.7q-2.32 0-4-1.67T0 23.13V10.3q0-1.5.76-2.78T2.8 5.5t2.88-.75zM20 1.25h4.19l-2.44 6.4-2 5.29-1.56 4.15q-.81 2.15-1.28 3.35t-.97 2.44-.82 1.9l-.62 1.31-.6 1.07-.56.78-.72.75q-1.87 1.94-4.87 2.06v-3.19q2.06-.31 3.03-1.75t.97-2.94l.06-.06q0-.69-.31-1.53T9.44 16Q7.8 11.94 6.62 8.87h3.88l3.12 9.82zm8 8.88v18.62H14.62q.63-.88.76-1.19H26.8V10.12q0-2.87-3.06-3.93l.44-1.13Q28 6.31 28 10.12z',
    'youtube' => [
        'p' => 'M34.38 7.75q.18.75.34 1.88t.22 2.21.1 2.03.02 1.57V16q0 5.63-.68 8.31-.32 1.07-1.13 1.88t-1.94 1.12q-1.19.32-4.5.47t-6.06.22H18q-10.7 0-13.31-.69-2.44-.68-3.07-3-.31-1.18-.47-3.28T.94 17.5V16q0-5.56.68-8.25.32-1.12 1.13-1.94T4.69 4.7q1.18-.31 4.5-.47T15.25 4H18q10.69 0 13.31.69 1.13.31 1.94 1.12t1.13 1.94zM14.5 21.13 23.44 16l-8.94-5.06v10.19z',
        'w' => 1.125,
        'vw' => 36
    ],
    'youtube-square' => 'M11.69 12.63 17.63 16l-5.94 3.38v-6.75zM28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22q1.25 0 2.13.88T28 5zm-2.63 11q0-3.69-.5-5.5-.37-1.63-2-2-.75-.25-2.96-.38T15.8 8H14q-7.13 0-8.88.5-1.62.38-2 2-.25.81-.37 2.19t-.13 2.37V16q0 3.75.5 5.5.38 1.63 2 2 .75.25 2.97.38t4.1.12H14q7.13 0 8.88-.5 1.62-.44 2-2 .25-.75.37-2.12t.13-2.38v-1z',
    'zhihu' => [
        'p' => 'M10.69 9.25h7.62v13.63h-3.06l-2.63 1.62-.5-1.63H10.7V9.26zm6.06 12.13V10.74h-4.56v10.63h.75l.31 1.12L15 21.37h1.75zM9.37 15.5q.2 0 .32.34t.12.72v.38H6q-.13 1.25-.5 2.56l.94-.75q.5.56 2.25 2.63.12.18.19.4t.09.47 0 .47l-.06.44-.07.37-.03.28v.07L5.44 19.8q-.38 1.38-.94 2.44t-.94 1.5l-.37.38q-1.32 1.18-3.2.62 4-3.06 4.45-7.81H.3q.25-1.44 1-1.44h3.25q.13-1.44.13-4.62H2.87l-.3.93q-.7 1.2-2.32 1.38.37-.75.72-1.63t.56-1.43.4-1.13.26-.69q.18-.62.68-.97T3.81 7l.38-.06q-.5 1.31-.82 2.56h5.5q.32 0 .44.34t.13.72v.32h-3.2q0 2.43-.12 4.62h3.25zm25.76-.13-1.2-.87q1.88-2.5 2.26-3.06.12-.13.28-.2t.31-.02.34.06.32.13.25.15.19.13l.06.06q-1.5 2-2.81 3.63zm-9.38-3.68q.13.12.4.47t.6.75.56.78.44.62l.13.25-1.2.88-2.56-3.63.29-.15q.16-.1.65-.2t.69.23zM40 16.13q.06 1.5-.56 1.5H31.8v4.56q0 .87-.31 1.53t-.94 1-1.56.34q-.81 0-1.63-.43-.75-.44-1.18-1.38.3.06.84.1t.94.02h.78q.69.07 1.12-.3t.44-1v-4.45h-6.75q-.62 0-1.03-.37t-.47-.69l-.12-.37h8.37V9.94q-1.19 0-3.56.18-2.56.07-3.06-.06-.82-.25-1-1.25 1.87 0 5.5-.28T34.55 8l2.69-.31 1.44-.25q.37.18.37.68t-.19.94l-.12.44q-1.75.13-5.5.31-1 .07-1.44.07v6.3h.13l8.06-.05z',
        'w' => 1.25,
        'vw' => 40
    ]
];
    }

    public static function far() {
        return [
            'heart' => 'M28.63 4q2.25 1.94 3 4.6t.06 5.09-2.38 4.19L18.38 29q-1 1-2.38 1t-2.38-1L2.7 17.88Q.99 16.13.3 13.66t.06-5.1 3-4.56q2.7-2.25 6.35-1.97T16 4.93q2.63-2.62 6.28-2.9T28.62 4zm-1.44 11.75Q29 13.87 29 11.15t-2.25-4.84Q24.88 4.81 22.41 5t-4.22 2L16 9.19',
            'star' => [
                'p' => 'M33 10.75q1.25.13 1.63 1.31t-.5 2.07l-6.63 6.43 1.56 9.13q.2 1.19-.81 1.9t-2.06.16l-8.2-4.25-8.18 4.25q-1.06.63-2.06-.12t-.81-1.94l1.56-9.13-6.63-6.43q-.87-.88-.5-2.07T3 10.75l9.12-1.38 4.07-8.25Q16.75 0 17.99 0t1.82 1.12l4.06 8.25zm-8.69 8.75 6.25-6.13-8.68-1.24L18 4.25l-3.88 7.88-8.68 1.24 6.25 6.13-1.44 8.69L18 24.06l7.75 4.13',
                'w' => 1.125,
                'vw' => 36
            ],
            'user' => [
                'p' => 'M19.63 19q3.43 0 5.9 2.47t2.47 5.9V29q0 1.25-.88 2.13T25 32H3q-1.25 0-2.13-.88T0 29v-1.63q0-3.43 2.47-5.9T8.37 19q.82 0 2.41.5T14 20t3.25-.5 2.38-.5zM25 29v-1.63q0-2.18-1.6-3.78T19.64 22q-.2 0-1.97.5T14 23t-3.66-.5-1.97-.5Q6.2 22 4.6 23.6T3 27.36V29h22zM14 18q-3.75 0-6.38-2.63T5 9t2.63-6.38T14 0t6.38 2.63T23 9t-2.63 6.38T14 18zm0-15q-2.5 0-4.25 1.75T8 9t1.75 4.25T14 15t4.25-1.75T20 9t-1.75-4.25T14 3z',
                'w' => .875,
                'vw' => 28
            ],
            'clock' => 'M16 .5q6.44 0 10.97 4.53T31.5 16t-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5zm0 28q5.19 0 8.84-3.66T28.5 16t-3.66-8.84T16 3.5 7.16 7.16 3.5 16t3.66 8.84T16 28.5zm3.88-6.5-5.32-3.88q-.31-.25-.31-.62V7.25q0-.75.75-.75h2q.75 0 .75.75v8.88l4.19 3q.56.43.12 1.06l-1.12 1.62q-.44.63-1.07.2z',
            'list-alt' => 'M29 2q1.25 0 2.13.88T32 5v22q0 1.25-.88 2.13T29 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h26zm-.38 25q.38 0 .38-.38V5.38Q29 5 28.62 5H3.38Q3 5 3 5.38v21.25q0 .37.38.37h25.25zM26 21.25v1.5q0 .75-.75.75h-12.5q-.75 0-.75-.75v-1.5q0-.75.75-.75h12.5q.75 0 .75.75zm0-6v1.5q0 .75-.75.75h-12.5q-.75 0-.75-.75v-1.5q0-.75.75-.75h12.5q.75 0 .75.75zm0-6v1.5q0 .75-.75.75h-12.5q-.75 0-.75-.75v-1.5q0-.75.75-.75h12.5q.75 0 .75.75zM10.25 10q0 .94-.66 1.6T8 12.24t-1.6-.66T5.76 10t.66-1.6T8 7.76t1.6.66.65 1.59zm0 6q0 .94-.66 1.6T8 18.24t-1.6-.66T5.76 16t.66-1.6T8 13.76t1.6.66.65 1.59zm0 6q0 .94-.66 1.6T8 24.24t-1.6-.66T5.76 22t.66-1.6T8 19.76t1.6.66.65 1.59z',
            'flag' => 'M21 5q2.75 0 6.81-1.75 1.5-.63 2.85.25T32 6v15q0 1.63-1.37 2.5-3.82 2.5-8 2.5-2.32 0-5.54-1t-4.59-1q-4.06 0-8 1.81v5.2q0 .43-.28.71T3.5 32h-1q-.44 0-.72-.28T1.5 31V5.63Q0 4.75 0 3 0 1.7.94.81T3.19 0q1.06.06 1.87.81T6 2.63V3q0 .5-.12.94 2.44-.94 5-.94 2.31 0 5.53 1T21 5zm8 16V6q-4.31 2-8 2-1.88 0-5.06-1t-5.07-1Q7.31 6 4.5 8v14.5q3.25-1.5 8-1.5 1.87 0 5.06 1t5.06 1q3.57 0 6.38-2z',
            'bookmark' => [
                'p' => 'M21 0q1.25 0 2.13.88T24 3v29l-12-7-12 7V3Q0 1.75.88.87T3 0h18zm0 26.75V3.37Q21 3 20.62 3H3.38Q3 3 3 3.38v23.37l9-5.25z',
                'w' => .75,
                'vw' => 24
            ],
            'image' => 'M29 4q1.25 0 2.13.88T32 7v18q0 1.25-.88 2.13T29 28H3q-1.25 0-2.13-.88T0 25V7q0-1.25.88-2.13T3 4h26zm-.38 21q.38 0 .38-.38V7.38Q29 7 28.62 7H3.38Q3 7 3 7.38v17.25q0 .37.38.37h25.25zM8 9.5q1.06 0 1.78.72T10.5 12t-.72 1.78T8 14.5t-1.78-.72T5.5 12t.72-1.78T8 9.5zM6 22v-3l2.5-2.5q.5-.5 1 0L12 19l7.5-7.5q.5-.5 1 0L26 17v5H6z',
            'picture-o' => 'image',
            'photo' => 'image',
            'edit' => [
                'p' => 'm25.13 21.56 2-2q.25-.25.56-.12t.31.5V29q0 1.25-.87 2.13T25 32H3q-1.25 0-2.12-.87T0 29V7q0-1.25.88-2.12T3 4h17.13q.3 0 .43.31t-.12.57l-2 2q-.13.12-.31.12H3v22h22v-7.12q0-.2.13-.32zm9.8-12.62L18.5 25.38l-5.63.62q-1.24.13-2.12-.75T10 23.13l.63-5.63L27.05 1.06Q28.13 0 29.63 0t2.62 1.06l2.69 2.7Q36 4.8 36 6.34t-1.06 2.59zm-6.18 1.93-3.63-3.62L13.5 18.88l-.44 4.06 4.06-.44zm4.06-5L30.12 3.2q-.5-.44-.93 0l-1.94 1.93 3.62 3.63 1.94-1.94q.44-.44 0-.94z',
                'w' => 1.123,
                'vw' => 36
            ],
            'times-circle' => 'M16 .5q6.44 0 10.97 4.53T31.5 16t-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5zm0 28q5.19 0 8.84-3.66T28.5 16t-3.66-8.84T16 3.5 7.16 7.16 3.5 16t3.66 8.84T16 28.5zm6.38-16.38L18.5 16l3.88 3.88q.5.56 0 1.06l-1.44 1.44q-.5.5-1.07 0L16 18.5l-3.88 3.88q-.56.5-1.06 0l-1.44-1.44q-.5-.5 0-1.07L13.5 16l-3.88-3.88q-.5-.56 0-1.06l1.44-1.44q.5-.5 1.06 0L16 13.5l3.87-3.88q.57-.5 1.07 0l1.43 1.44q.5.5 0 1.06z',
            'check-circle' => 'M16 .5q6.44 0 10.97 4.53T31.5 16t-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5zm0 3q-5.19 0-8.84 3.66T3.5 16t3.66 8.84T16 28.5t8.84-3.66T28.5 16t-3.66-8.84T16 3.5zm8.75 8.13q.56.56 0 1.06L14 23.38q-.56.56-1.06 0l-5.7-5.7q-.55-.55 0-1.05l1.45-1.44q.5-.5 1.06 0L13.44 19l8.87-8.81q.5-.5 1.07.06z',
            'question-circle' => 'M16 .5q6.44 0 10.97 4.53T31.5 16t-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5zm0 28q5.19 0 8.84-3.66T28.5 16t-3.66-8.84T16 3.5 7.16 7.16 3.5 16t3.66 8.84T16 28.5zm6.69-15.94Q22.69 14 22 15t-1.56 1.44-1.57.94-.68 1v.37q0 .31-.22.53t-.53.22h-2.88q-.31 0-.53-.22t-.22-.53v-.56q0-.75.25-1.38t.81-1.12.88-.72 1.06-.6l.75-.46.53-.38.38-.47.1-.56q0-.75-.79-1.25t-1.72-.5q-.69 0-1.22.22t-.9.6-.94 1.06q-.44.56-1 .12l-1.75-1.31q-.56-.44-.19-1.06Q12.37 7 16.37 7q2.38 0 4.35 1.63t1.97 3.93zM18.62 23q0 1.06-.78 1.84t-1.84.79-1.84-.79-.79-1.84.79-1.84 1.84-.79 1.84.79.79 1.84z',
            'eye' => [
                'p' => 'M18 9q2.94 0 4.97 2.03T25 15.97t-2.03 4.94-4.94 2.03-4.94-2.03-2.03-4.9q0-1 .25-1.95.82.44 1.69.44 1.44 0 2.47-1.03T16.5 11q0-.87-.44-1.69Q17 9 18 9zm17.81 6.06q.2.44.2.94t-.2.94q-2.62 5-7.37 8.03T18 28q-3.75 0-7.22-1.4t-6.15-3.88-4.44-5.78q-.2-.44-.2-.94t.2-.94q2.62-5 7.37-8.03T18 4t10.44 3.03 7.37 8.03zM18 25q4.63 0 8.6-2.4t6.27-6.6q-2.3-4.19-6.28-6.6T18 7 9.4 9.4 3.13 16q1.5 2.75 3.82 4.78t5.15 3.13T18 25z',
                'w' => 1.125,
                'vw' => 36
            ],
            'eye-slash' => [
                'p' => 'M39.63 29.44q.37.31.37.75 0 .37-.25.69l-.63.75q-.24.37-.74.37-.38 0-.63-.25L.37 2.56Q0 2.25 0 1.81q0-.37.25-.69L.88.38Q1.11 0 1.61 0 2 0 2.25.25zM18.55 9.12Q19.26 9 20 9q2.81 0 4.85 1.97T27 15.75zm2.88 13.76Q20.75 23 20 23q-2.81 0-4.84-1.97T13 16.25zM20 7q-1.69 0-3.63.44l-2.93-2.25Q16.75 4 20 4q3.75 0 7.22 1.4t6.16 3.88 4.43 5.78q.2.44.2.94t-.2.94q-1.37 2.56-3.31 4.69l-2.38-1.88q1.63-1.69 2.76-3.75-2.32-4.19-6.29-6.6T20 7zm0 18q1.69 0 3.63-.44l2.93 2.25Q23.25 28 20 28q-5.69 0-10.44-3.03T2.2 16.94Q2 16.5 2 16t.2-.94q1.37-2.56 3.31-4.69l2.38 1.88Q6.25 13.94 5.13 16q2.3 4.19 6.28 6.6T20 25z',
                'w' => 1.25,
                'vw' => 40
            ],
            'calendar-alt' => [
                'p' => 'M9.25 18h-2.5Q6 18 6 17.25v-2.5q0-.75.75-.75h2.5q.75 0 .75.75v2.5q0 .75-.75.75zm6.75-.75q0 .75-.75.75h-2.5q-.75 0-.75-.75v-2.5q0-.75.75-.75h2.5q.75 0 .75.75v2.5zm6 0q0 .75-.75.75h-2.5q-.75 0-.75-.75v-2.5q0-.75.75-.75h2.5q.75 0 .75.75v2.5zm-6 6q0 .75-.75.75h-2.5q-.75 0-.75-.75v-2.5q0-.75.75-.75h2.5q.75 0 .75.75v2.5zm-6 0q0 .75-.75.75h-2.5Q6 24 6 23.25v-2.5q0-.75.75-.75h2.5q.75 0 .75.75v2.5zm12 0q0 .75-.75.75h-2.5q-.75 0-.75-.75v-2.5q0-.75.75-.75h2.5q.75 0 .75.75v2.5zM28 7v22q0 1.25-.88 2.13T25 32H3q-1.25 0-2.13-.88T0 29V7q0-1.25.88-2.13T3 4h3V.75Q6 0 6.75 0h2.5Q10 0 10 .75V4h8V.75q0-.75.75-.75h2.5Q22 0 22 .75V4h3q1.25 0 2.13.88T28 7zm-3 21.63V10H3v18.63q0 .37.38.37h21.25q.37 0 .37-.38z',
                'w' => .875,
                'vw' => 28
            ],
            'comment' => 'M16 2q6.6 0 11.3 3.8T32 15t-4.7 9.2T16 28q-3 0-5.8-.9-4 2.9-8.7 2.9-1 0-1.4-.9t.3-1.6l.6-.7 1.2-1.6q.7-1 1.1-2.3Q0 19.4 0 15q0-5.4 4.7-9.2T16 2zm0 23',
            'folder' => 'M29 8q1.25 0 2.13.88T32 11v14q0 1.25-.88 2.13T29 28H3q-1.25 0-2.13-.88T0 25V7q0-1.25.88-2.13T3 4h9.19q.81 0 1.37.56L17 8h12zm0 17V11H16.56q-.81 0-1.37-.56L11.75 7H3v18h26z',
            'folder-open' => [
                'p' => 'M33 14q1.75 0 2.63 1.53t-.07 3.03l-5 8Q29.7 28 28 28H3q-1.25 0-2.13-.88T0 25V7q0-1.25.87-2.13T3 4h10l4 4h10q1.25 0 2.12.88T30 11v3h3zM3 7.37v14.57l3.94-6.5Q7.8 14 9.5 14H27v-2.63q0-.37-.37-.37H15.75l-4-4H3.38Q3 7 3 7.38zM28 25l5-8H9.31L4.5 25H28z',
                'w' => 1.123,
                'vw' => 36
            ],
            'bar-chart' => 'chart-bar',
            'chart-bar' => 'M24.81 22q-.31 0-.56-.25t-.25-.56V6.8q0-.31.25-.56t.56-.25h1.38q.31 0 .56.25t.25.56v14.4q0 .31-.25.56t-.56.25H24.8zm-12 0q-.31 0-.56-.25t-.25-.56V8.8q0-.31.25-.56t.56-.25h1.38q.31 0 .56.25t.25.56v12.4q0 .31-.25.56t-.56.25H12.8zm6 0q-.31 0-.56-.25t-.25-.56V12.8q0-.31.25-.56t.56-.25h1.38q.31 0 .56.25t.25.56v8.38q0 .31-.25.56t-.56.25H18.8zM31 25q.44 0 .72.28T32 26v1q0 .44-.28.72T31 28H2q-.81 0-1.4-.6T0 26V5q0-.44.28-.72T1 4h1q.44 0 .72.28T3 5v20h28zM6.81 22q-.31 0-.56-.25T6 21.19V16.8q0-.31.25-.56t.56-.25H8.2q.31 0 .56.25t.25.56v4.38q0 .31-.25.56T8.2 22H6.8z',
            'comments' => [
                'p' => 'M33.25 24.13q.44.87 1.03 1.71t.94 1.25.4.47q.63.7.26 1.57t-1.32.87q-4.06 0-7.81-2.44Q24.87 28 23 28q-4 0-7.22-1.69t-4.72-4.43q-.94-.13-1.81-.32Q5.56 24 1.44 24q-.94 0-1.32-.88t.25-1.56q1.32-1.25 2.38-3.44Q0 15.44 0 12q0-4.13 3.8-7.06T13 2q4 0 7.25 1.69T25 8.12q4.69.57 7.84 3.35T36 18q0 3.44-2.75 6.12zM8.69 18.38l1.25.25q1.5.37 3.06.37 4.06 0 7.03-2.06T23 12t-2.97-4.94T13 5 5.97 7.06 3 12q0 2.19 1.88 4l1.5 1.5-.94 2-.5.87q1.37-.43 2.69-1.3zM31.13 22Q33 20.19 33 18q0-2.25-2-4.1t-5.06-2.53q.06.5.06.63 0 2.5-1.47 4.66t-4.03 3.53-5.69 1.68Q17.87 25 23 25q1.56 0 3.06-.38l1.25-.25 1.07.7q1.3.87 2.68 1.3l-.5-.87-.93-2z',
                'w' => 1.125,
                'vw' => 36
            ],
            'star-half' => [
                'p' => 'M18 24.06v3.44l-8.19 4.25q-1.06.56-2.06-.16t-.81-1.96l1.56-9.07-6.63-6.44q-.87-.87-.5-2.06T3 10.7l9.12-1.32 4.07-8.25Q16.75 0 18 0v4.25l-3.88 7.87-8.68 1.25 6.25 6.13-1.44 8.69z',
                'w' => .601,
                'vw' => 19
            ],
            'lemon' => 'M30.25 1.75q1.31 1.31 1.66 3t-.47 3.06q-.25.5-.32 1.66t.2 2.4.12 3-.44 3.5-1.53 3.88-3.16 4.06-4.06 3.13-3.88 1.53-3.5.47-3-.13-2.4-.18-1.66.3q-1.37.82-3.06.48t-3-1.66-1.66-3 .47-3.06q.25-.5.32-1.66t-.2-2.4-.12-3 .44-3.5 1.53-3.88T5.7 5.7t4.06-3.13 3.88-1.53 3.5-.47 3 .13 2.4.19T24.2.56q1.37-.81 3.06-.47t3 1.66zm-1.38 4.56q.63-1.12-.75-2.47t-2.43-.71q-1.07.68-2.66.71t-3.28-.15-3.56-.1-4.07 1.1-4.3 3.12-3.13 4.32-1.1 4.06.1 3.56.15 3.28-.72 2.66q-.62 1.12.75 2.47t2.44.72q.94-.57 2.25-.7t2.56.07 2.85.19 3.15-.22 3.41-1.25 3.66-2.78q2.12-2.13 3.12-4.31t1.1-4.07-.1-3.56-.15-3.28.71-2.66zm-10.62.53q.06.53-.25.94t-.88.47q-2.56.31-5.56 3.31t-3.31 5.57q-.06.56-.47.87t-.94.25-.84-.47-.25-.9q.44-3.44 4.06-7.07t7.06-4.06q.5-.06.91.25t.47.85z',
            'credit-card' => [
                'p' => 'M33 2q.81 0 1.5.4t1.1 1.1T36 5v22q0 1.25-.88 2.13T33 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h30zM3.37 5Q3 5 3 5.38V8h30V5.37Q33 5 32.62 5H3.39zm29.25 22q.38 0 .38-.38V16H3v10.63q0 .37.38.37h29.25zM12 20.75v2.5q0 .75-.75.75h-4.5Q6 24 6 23.25v-2.5q0-.75.75-.75h4.5q.75 0 .75.75zm12 0v2.5q0 .75-.75.75h-8.5q-.75 0-.75-.75v-2.5q0-.75.75-.75h8.5q.75 0 .75.75z',
                'w' => 1.125,
                'vw' => 36
            ],
            'hdd' => [
                'p' => 'M35.44 14.75q.56.75.56 1.69V25q0 1.25-.87 2.13T33 28H3q-1.25 0-2.12-.87T0 25v-8.56q0-.94.56-1.69l6.57-9.44q.37-.56 1.1-.94T9.56 4h16.87q.63 0 1.34.37t1.1.94zM9.56 7 4.7 14h26.6l-4.87-7H9.56zM33 25v-8H3v8h30zm-2-4q0 .81-.6 1.4T29 23t-1.4-.6T27 21t.6-1.4T29 19t1.4.6.6 1.4zm-6 0q0 .81-.6 1.4T23 23t-1.4-.6T21 21t.6-1.4T23 19t1.4.6.6 1.4z',
                'w' => 1.125,
                'vw' => 36
            ],
            'hand-point-right' => 'M26.81 8.63q2.13 0 3.66 1.53T32 13.8q0 1.06-.4 2t-1.13 1.66-1.66 1.12-2 .4h-2.25q-.12.76-.5 1.38.25 2.13-1 3.94-.06 2.5-1.72 4.1T16.81 30H15.5q-2.13 0-4.72-1.25T7.56 27.5h-.19Q6.81 28 6 28H2q-.81 0-1.4-.53T0 26.19V11.8q0-.75.6-1.28T2 10h4q.81 0 1.37.5h.2q.55 0 1.87-1.5 1.06-1.19 1.62-2.06.25-.38.82-1.5t.93-1.72 1.2-1.16T15.8 2q2.63 0 4.2 1.47t1.55 3.72q0 .75-.12 1.44h5.37zm0 7.37q.88 0 1.53-.66t.66-1.53-.66-1.53-1.53-.65h-9.93q0-.44.84-1.82t.84-2.62Q18.56 5 15.81 5q-.25 0-.56.53t-.81 1.5-.82 1.47q-.68 1.13-1.93 2.5-2 2.25-3.7 2.5v11q1.07.13 2.38.75T13 26.44t2.5.56h1.31q3.81 0 3.19-3.81.75-.44 1.03-1.44t-.22-1.94q.88-.81.9-2T21.13 16h5.7zM5.5 24q0-.63-.44-1.06T4 22.5t-1.06.44T2.5 24t.44 1.06T4 25.5t1.06-.44T5.5 24z',
            'hand-point-left' => 'M0 13.81q0-2.12 1.53-3.65T5.2 8.63h5.37q-.12-.7-.12-1.44 0-2.25 1.56-3.72T16.19 2q1 0 1.81.56t1.19 1.16.94 1.72.81 1.5q.56.87 1.62 2.06 1.32 1.5 1.88 1.5h.19q.56-.5 1.37-.5h4q.44 0 .78.16t.63.37.44.57.15.71V26.2q0 .75-.6 1.28T30 28h-4q-.81 0-1.37-.5h-.19q-.63 0-3.22 1.25T16.5 30h-1.3q-2.88 0-4.53-1.6t-1.72-4.09q-1.25-1.81-1-3.94-.38-.62-.5-1.37H5.19q-2.13 0-3.66-1.53T0 13.8zm3 0q0 .88.66 1.53t1.53.66h5.68q-.62.63-.59 1.81t.9 2q-.5.94-.21 1.94T12 23.2q-.63 3.8 3.19 3.8h1.31q1.5 0 3.88-1.19t3.62-1.3v-11q-1.69-.26-3.69-2.5-1.25-1.38-1.94-2.5-.3-.5-.8-1.48t-.82-1.5-.56-.53q-2.75 0-2.75 2.19 0 1.25.84 2.62t.84 1.82H5.18q-.43 0-.84.15t-.69.47-.47.72-.18.85zm25 8.69q-.63 0-1.06.44T26.5 24t.44 1.06 1.06.44 1.06-.44.44-1.06-.44-1.06T28 22.5z',
            'hand-point-up' => [
                'p' => 'M6.63 5.19q0-2.13 1.53-3.66T11.8 0q1.38 0 2.6.69t1.9 1.9.69 2.6v2.25q.75.12 1.38.5 2.12-.25 3.93 1 2.5.06 4.1 1.72t1.6 4.53v1.31q0 2.13-1.26 4.72t-1.25 3.22v.19q.5.56.5 1.37v4q0 .57-.25 1t-.65.72-.91.28H9.8q-.75 0-1.28-.6T8 30v-4q0-.81.5-1.37v-.19q0-.56-1.5-1.88-1.19-1.06-2.06-1.62-.44-.25-1.25-.66t-1.35-.71-1.12-.79-.9-1.12T0 16.19Q0 13.56 1.47 12t3.72-1.56q.75 0 1.44.12V5.2zm7.37 0q0-.88-.66-1.53T11.81 3t-1.53.66-.65 1.53v9.94q-.44 0-1.82-.85t-2.62-.84Q3 13.44 3 16.19q0 .25.53.56t1.5.81 1.47.82q1.13.68 2.5 1.93 2.25 2 2.5 3.7h11q.13-1.26 1.31-3.63t1.2-3.88v-1.3q0-3.81-3.82-3.19-.44-.75-1.44-1.03t-1.94.22q-.81-.88-2-.9t-1.81.59v-5.7zm8 21.31q-.63 0-1.06.44T20.5 28t.44 1.06 1.06.44 1.06-.44.44-1.06-.44-1.06T22 26.5z',
                'w' => .875,
                'vw' => 28
            ],
            'hand-point-down' => [
                'p' => 'M11.81 32q-2.12 0-3.65-1.53T6.63 26.8v-5.37q-.7.12-1.44.12-2.25 0-3.72-1.56T0 15.81q0-.81.31-1.47t.9-1.12 1.13-.78 1.35-.72 1.25-.66Q5.8 10.5 7 9.44q1.5-1.32 1.5-1.88v-.19Q8 6.81 8 6V2q0-.81.53-1.4T9.81 0H24.2q.75 0 1.28.6T26 2v4q0 .81-.5 1.37v.2q0 .62 1.25 3.21T28 15.5v1.31q0 2.88-1.6 4.53t-4.09 1.72q-1.81 1.25-3.94 1-.62.38-1.37.5v2.25q0 2.13-1.53 3.66T11.8 32zm0-3q.88 0 1.53-.66t.66-1.53v-5.68q.63.62 1.81.59t2-.9q.94.5 1.94.21T21.2 20q3.81.63 3.81-3.19V15.5q0-1.5-1.19-3.88T22.51 8h-11q-.26 1.69-2.5 3.69-1.38 1.25-2.5 1.94-.5.3-1.48.8t-1.5.82-.53.57q0 2.75 2.19 2.75 1.25 0 2.62-.85t1.82-.84v9.94q0 .87.65 1.53t1.53.65zM20.5 4q0 .63.44 1.06T22 5.5t1.06-.44T23.5 4t-.44-1.06T22 2.5t-1.06.44T20.5 4z',
                'w' => .875,
                'vw' => 28
            ],
            'copy' => [
                'p' => 'M27.13 4.13Q28 5 28 6.25V23q0 1.25-.88 2.13T25 26h-5v3q0 1.25-.88 2.13T17 32H3q-1.25 0-2.13-.88T0 29V9q0-1.25.88-2.13T3 6h5V3q0-1.25.88-2.13T11 0h10.75Q23 0 23.88.88zM16.63 29q.37 0 .37-.38V26h-6q-1.25 0-2.13-.88T8 23V9H3.37Q3 9 3 9.38v19.24q0 .38.38.38h13.25zm8-6q.37 0 .37-.38V10h-5.5q-.63 0-1.06-.44T18 8.5V3h-6.63Q11 3 11 3.37v19.25q0 .38.37.38h13.25zM25 7v-.63q0-.12-.13-.25l-3-3Q21.75 3 21.63 3H21v4h4z',
                'w' => .875,
                'vw' => 28
            ],
            'save' => [
                'p' => 'M27.13 8.13Q28 9 28 10.25V27q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h16.75q1.25 0 2.13.88zM17 5H9v5h8V5zm7.63 22q.37 0 .37-.38V10.38q0-.12-.13-.24L20 5.25v6.25q0 .63-.44 1.06T18.5 13h-11q-.63 0-1.06-.44T6 11.5V5H3.37Q3 5 3 5.38v21.25q0 .37.37.37h21.25zM14 14.5q2.25 0 3.88 1.63T19.5 20t-1.63 3.88T14 25.5t-3.88-1.63T8.5 20t1.63-3.88T14 14.5zm0 8q1.06 0 1.78-.72T16.5 20t-.72-1.78T14 17.5t-1.78.72T11.5 20t.72 1.78 1.78.72z',
                'w' => .875,
                'vw' => 28
            ],
            'square' => [
                'p' => 'M25 2q1.25 0 2.13.88T28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22zm-.38 25q.38 0 .38-.38V5.38Q25 5 24.62 5H3.38Q3 5 3 5.38v21.25q0 .37.38.37h21.25z',
                'w' => .875,
                'vw' => 28
            ],
            'envelope' => 'M29 4q1.25 0 2.13.88T32 7v18q0 1.25-.88 2.13T29 28H3q-1.25 0-2.13-.88T0 25V7q0-1.25.88-2.13T3 4h26zm0 3H3v2.56q2.19 1.75 8.44 6.63 0 .06.87.78t1.32 1 1.15.66T16 19t1.22-.38 1.16-.65 1.3-1 .88-.78Q26.81 11.3 29 9.56V7zM3 25h26V13.37q-2.19 1.76-6.56 5.2-.13.06-.9.74t-1.23 1-1.25.78-1.56.7T16 22t-1.53-.22-1.56-.72-1.22-.81-1.22-.97-.9-.72Q5.18 15.12 3 13.37V25z',
            'lightbulb' => [
                'p' => 'M11 5q.44 0 .72.28T12 6t-.28.72T11 7Q9.37 7 8.19 8.19T6.99 11q0 .44-.27.72T6 12t-.72-.28T5 11q0-2.5 1.75-4.25T11 5zM6 28.69V26h10v2.69q0 .31-.19.56l-1.5 2.31q-.31.44-.87.44H8.56q-.56 0-.87-.44l-1.5-2.3q-.2-.26-.2-.57zM11 0q4.56 0 7.78 3.22T22 11q0 4.13-2.75 7.25-.88 1-1.84 2.66T16 24h-3q0-.44.13-.88 1.18-3.8 3.87-6.87Q19 14 19 11q0-3.31-2.34-5.66T11 3Q7.75 3 5.37 5.31T3 11.01q0 3 2 5.24 2.69 3.06 3.88 6.88.12.43.12.87H6q-.31-1-.9-2.16t-1.22-2.06-1.13-1.53Q0 15.13 0 11q0-4.44 3.16-7.72T11 0z',
                'w' => .687,
                'vw' => 22
            ],
            'bell' => [
                'p' => 'M27.44 22.63q.56.62.56 1.37 0 .81-.56 1.4T26 26H2q-.56 0-1.03-.28t-.72-.75T0 24q0-.75.56-1.38.07-.06.2-.25.93-.93 1.46-1.68t1.16-2.85T4 13q0-3.63 2.25-6.31T12 3.3V2q0-.81.6-1.4T14 0t1.4.6T16 2v1.3q2.31.44 4.13 1.81t2.84 3.44T24 13q0 2.75.63 4.84t1.15 2.85 1.47 1.69l.19.25zM4.25 23h19.5Q21 19.31 21 13.06V13q0-2.88-2.06-4.94T14 6 9.06 8.06 7 13v.06Q7 19.31 4.25 23zM14 32q-1.63 0-2.81-1.16T9.99 28h8q0 1.69-1.18 2.84T14 32z',
                'w' => .875,
                'vw' => 28
            ],
            'hospital' => [
                'p' => 'M8 15.25v-2.5q0-.75.75-.75h2.5q.75 0 .75.75v2.5q0 .75-.75.75h-2.5Q8 16 8 15.25zm8.75.75q-.75 0-.75-.75v-2.5q0-.75.75-.75h2.5q.75 0 .75.75v2.5q0 .75-.75.75h-2.5zM12 21.25q0 .75-.75.75h-2.5Q8 22 8 21.25v-2.5q0-.75.75-.75h2.5q.75 0 .75.75v2.5zm4.75.75q-.75 0-.75-.75v-2.5q0-.75.75-.75h2.5q.75 0 .75.75v2.5q0 .75-.75.75h-2.5zM28 29.75V32H0v-2.25Q0 29 .75 29H2V5.31q0-.56.44-.94T3.5 4H9V1.5q0-.63.44-1.06T10.5 0h7q.63 0 1.07.44T19 1.5V4h5.5q.63 0 1.07.37t.43.94V29h1.25q.75 0 .75.75zm-23-.81h7v-4.19q0-.75.75-.75h2.5q.75 0 .75.75v4.19h7V7h-4v1.5q0 .63-.44 1.06T17.5 10h-7q-.63 0-1.06-.44T9 8.5V7H5v21.94zM16.63 4H15V2.37Q15 2 14.62 2h-1.24Q13 2 13 2.38V4h-1.63Q11 4 11 4.38v1.25q0 .37.38.37H13v1.63q0 .37.38.37h1.24Q15 8 15 7.62V6h1.63Q17 6 17 5.62V4.38Q17 4 16.62 4z',
                'w' => .875,
                'vw' => 28
            ],
            'plus-square' => [
                'p' => 'M22 15v2q0 .75-.75.75h-5.5v5.5q0 .75-.75.75h-2q-.75 0-.75-.75v-5.5h-5.5Q6 17.75 6 17v-2q0-.75.75-.75h5.5v-5.5q0-.75.75-.75h2q.75 0 .75.75v5.5h5.5q.75 0 .75.75zm6-10v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22q1.25 0 2.13.88T28 5zm-3 21.63V5.38Q25 5 24.62 5H3.38Q3 5 3 5.38v21.25q0 .37.38.37h21.25q.37 0 .37-.38z',
                'w' => .875,
                'vw' => 28
            ],
            'circle' => 'M16 .5q6.44 0 10.97 4.53T31.5 16t-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5zm0 28q5.19 0 8.84-3.66T28.5 16t-3.66-8.84T16 3.5 7.16 7.16 3.5 16t3.66 8.84T16 28.5z',
            'smile' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm0 28q5.19 0 8.84-3.66T28 16t-3.66-8.84T15.5 3.5 6.66 7.16 3 16t3.66 8.84 8.84 3.66zm-5-13.5q-.81 0-1.4-.6T8.5 13t.6-1.4 1.4-.6 1.4.6.6 1.4-.6 1.4-1.4.6zm10 0q-.81 0-1.4-.6t-.6-1.4.6-1.4 1.4-.6 1.4.6.6 1.4-.6 1.4-1.4.6zm.25 4.56q.38-.5 1-.56t1.1.34.52 1.04-.3 1.06Q20.11 25 15.5 25t-7.56-3.56q-.38-.44-.32-1.07t.53-1.03 1.1-.34 1 .56Q12.3 22 15.5 22t5.25-2.44z',
            'frown' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm0 28q5.19 0 8.84-3.66T28 16t-3.66-8.84T15.5 3.5 6.66 7.16 3 16t3.66 8.84 8.84 3.66zm-5-13.5q-.81 0-1.4-.6T8.5 13t.6-1.4 1.4-.6 1.4.6.6 1.4-.6 1.4-1.4.6zm10-4q.81 0 1.4.6t.6 1.4-.6 1.4-1.4.6-1.4-.6-.6-1.4.6-1.4 1.4-.6zm-5 8q3.94 0 6.5 3.06.38.44.31 1.07t-.5 1q-.5.43-1.12.37t-1-.56Q18.06 22 15.5 22t-4.19 1.94q-.37.5-1 .56t-1.1-.34-.52-1.03.31-1.07Q11.56 19 15.5 19z',
            'meh' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm0 28q5.19 0 8.84-3.66T28 16t-3.66-8.84T15.5 3.5 6.66 7.16 3 16t3.66 8.84 8.84 3.66zm-5-13.5q-.81 0-1.4-.6T8.5 13t.6-1.4 1.4-.6 1.4.6.6 1.4-.6 1.4-1.4.6zm10-4q.81 0 1.4.6t.6 1.4-.6 1.4-1.4.6-1.4-.6-.6-1.4.6-1.4 1.4-.6zm.5 9q.63 0 1.06.44t.44 1.06-.44 1.06T21 23H10q-.63 0-1.06-.44T8.5 21.5t.44-1.06T10 20h11z',
            'keyboard' => [
                'p' => 'M33 4q1.25 0 2.13.88T36 7v18q0 1.25-.88 2.13T33 28H3q-1.25 0-2.13-.88T0 25V7q0-1.25.88-2.13T3 4h30zm.5 21V7q0-.5-.5-.5H3q-.5 0-.5.5v18q0 .5.5.5h30q.5 0 .5-.5zm-22.88-8.13q0 .75-.74.75H8.12q-.74 0-.74-.75v-1.75q0-.74.75-.74h1.74q.76 0 .76.74v1.76zm6 0q0 .75-.75.75h-1.74q-.76 0-.76-.75v-1.75q0-.74.76-.74h1.74q.76 0 .76.74v1.76zm6 0q0 .75-.75.75h-1.75q-.75 0-.75-.75v-1.75q0-.74.75-.74h1.75q.75 0 .75.74v1.76zm6 0q0 .75-.75.75h-1.75q-.75 0-.75-.75v-1.75q0-.74.75-.74h1.75q.75 0 .75.74v1.76zM7.63 22q0 .75-.75.75H5.13q-.75 0-.75-.75v-1.75q0-.75.75-.75h1.75q.75 0 .75.75V22zm24 0q0 .75-.75.75h-1.75q-.75 0-.75-.75v-1.75q0-.75.75-.75h1.75q.75 0 .75.75V22zm-24-10.25q0 .75-.75.75H5.13q-.75 0-.75-.75V10q0-.75.75-.75h1.75q.75 0 .75.75v1.75zm6 0q0 .75-.74.75h-1.76q-.74 0-.74-.75V10q0-.75.74-.75h1.76q.74 0 .74.75v1.75zm6 0q0 .75-.75.75h-1.75q-.75 0-.75-.75V10q0-.75.75-.75h1.75q.75 0 .75.75v1.75zm6 0q0 .75-.75.75h-1.75q-.75 0-.75-.75V10q0-.75.75-.75h1.75q.75 0 .75.75v1.75zm6 0q0 .75-.75.75h-1.75q-.75 0-.75-.75V10q0-.75.75-.75h1.75q.75 0 .75.75v1.75zm-6.12 9.88q0 .75-.75.75h-13.5q-.75 0-.75-.75v-1q0-.75.75-.75h13.5q.75 0 .75.75v1z',
                'w' => 1.125,
                'vw' => 36
            ],
            'calendar' => [
                'p' => 'M25 4q1.25 0 2.13.88T28 7v22q0 1.25-.88 2.13T25 32H3q-1.25 0-2.13-.88T0 29V7q0-1.25.88-2.13T3 4h3V.75Q6 0 6.75 0h2.5Q10 0 10 .75V4h8V.75q0-.75.75-.75h2.5Q22 0 22 .75V4h3zm-.38 25q.38 0 .38-.38V10H3v18.63q0 .37.38.37h21.25z',
                'w' => .875,
                'vw' => 28
            ],
            'play-circle' => 'M23.25 14.88q.75.43.75 1.3t-.75 1.32l-11 6.31q-.56.25-1.06.16t-.85-.5-.34-.97v-13q0-.87.75-1.31t1.5 0zM31.5 16q0 6.44-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5t10.97 4.53T31.5 16zm-28 0q0 5.19 3.66 8.84T16 28.5t8.84-3.66T28.5 16t-3.66-8.84T16 3.5 7.16 7.16 3.5 16z',
            'minus-square' => [
                'p' => 'M6.75 17.75Q6 17.75 6 17v-2q0-.75.75-.75h14.5q.75 0 .75.75v2q0 .75-.75.75H6.75zM28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22q1.25 0 2.13.88T28 5zm-3 21.63V5.38Q25 5 24.62 5H3.38Q3 5 3 5.38v21.25q0 .37.38.37h21.25q.37 0 .37-.38z',
                'w' => .875,
                'vw' => 28
            ],
            'check-square' => [
                'p' => 'M25 2q1.25 0 2.13.88T28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22zm0 25V5H3v22h22zm-2.25-15.13L12 22.57q-.56.56-1.06 0l-5.7-5.7q-.55-.56 0-1.06l1.45-1.44q.5-.5 1.06 0l3.69 3.82 8.87-8.75q.5-.57 1.07 0l1.37 1.37q.56.56 0 1.06z',
                'w' => .875,
                'vw' => 28
            ],
            'share-square' => [
                'p' => 'M35.13 9.88q.87.87.87 2.12t-.88 2.13l-9 9q-1.43 1.43-3.28.65T21 21v-3.38q-3.06.25-4.4.85t-1.35 1.6.5 2.5q.69 2-1 3.27t-3.44.22q-2.5-1.56-3.9-3.9T6 17q0-3.69 2.25-6.19 1.88-2.06 5.19-3.12 3-.94 7.56-1.13V3q0-2 1.84-2.75t3.29.63zM24 21l9-9-9-9v6.5q-2.06 0-3.56.1t-3.38.34-3.19.78-2.5 1.31T9.6 14.1 9 17q0 4.06 3.88 6.5-.43-1.37-.56-2.53t.13-2.03.75-1.56 1.3-1.16 1.79-.78 2.25-.53 2.56-.28 2.9-.13V21zm1.56 5.25q.7-.19 1.32-.56.37-.25.75-.03t.37.65V29q0 1.25-.87 2.13T25 32H3q-1.25 0-2.12-.88T0 29V7q0-1.25.88-2.13T3 4h8.25q.75 0 .75.75V5q0 .56-.5.75-1.25.5-2.37 1.13-.2.12-.38.12H3.38Q3 7 3 7.38v21.25q0 .37.38.37h21.25q.37 0 .37-.38V27q0-.56.56-.75z',
                'w' => 1.125,
                'vw' => 36
            ],
            'compass' => 'M21.75 8.13q.75-.38 1.38.24t.25 1.38l-4.13 9q-.31.69-1 1l-9 4.13q-.75.37-1.38-.25t-.25-1.38l4.13-9q.31-.69 1-1zm-4.84 9.28q.59-.6.59-1.41t-.6-1.4-1.4-.6-1.4.6-.6 1.4.6 1.4 1.4.6 1.4-.6zM15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm0 28q5.19 0 8.84-3.66T28 16t-3.66-8.84T15.5 3.5 6.66 7.16 3 16t3.66 8.84 8.84 3.66z',
            'caret-square-down' => [
                'p' => 'M7.81 13H20.2q.5 0 .69.47t-.2.78l-6.18 6.19q-.5.5-1 0l-6.19-6.19q-.37-.31-.19-.78t.7-.47zM28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22q1.25 0 2.13.88T28 5zm-3 21.63V5.38Q25 5 24.62 5H3.38Q3 5 3 5.38v21.25q0 .37.38.37h21.25q.37 0 .37-.38z',
                'w' => .875,
                'vw' => 28
            ],
            'caret-square-up' => [
                'p' => 'M20.19 19H7.8q-.5 0-.69-.47t.2-.78l6.18-6.19q.5-.5 1 0l6.19 6.19q.37.31.19.78t-.7.47zM28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22q1.25 0 2.13.88T28 5zm-3 21.63V5.38Q25 5 24.62 5H3.38Q3 5 3 5.38v21.25q0 .37.38.37h21.25q.37 0 .37-.38z',
                'w' => .875,
                'vw' => 28
            ],
            'caret-square-right' => [
                'p' => 'M11 22.19V9.8q0-.5.47-.68t.78.18l6.19 6.2q.5.5 0 1l-6.19 6.18q-.31.37-.78.19t-.47-.7zM28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22q1.25 0 2.13.88T28 5zm-3 21.63V5.38Q25 5 24.62 5H3.38Q3 5 3 5.38v21.25q0 .37.38.37h21.25q.37 0 .37-.38z',
                'w' => .875,
                'vw' => 28
            ],
            'file' => [
                'p' => 'M23.13 6.13Q24 7 24 8.25V29q0 1.25-.88 2.13T21 32H3q-1.25 0-2.13-.88T0 29V3Q0 1.75.88.87T3 0h12.75Q17 0 17.88.88zM20.75 8 16 3.25V8h4.75zM3 29h18V11h-6.5q-.63 0-1.06-.44T13 9.5V3H3v26z',
                'w' => .75,
                'vw' => 24
            ],
            'file-o' => 'file',
            'file-alt' => [
                'p' => 'M18 15.5v1.75q0 .75-.75.75H6.75Q6 18 6 17.25V15.5q0-.75.75-.75h10.5q.75 0 .75.75zm-.75 4.5q.75 0 .75.75v1.75q0 .75-.75.75H6.75q-.75 0-.75-.75v-1.75q0-.75.75-.75h10.5zM24 8.25V29q0 1.25-.88 2.13T21 32H3q-1.25 0-2.13-.88T0 29V3Q0 1.75.88.87T3 0h12.75Q17 0 17.88.88l5.25 5.25Q24 7 24 8.25zm-8-5V8h4.75zM21 29V11h-6.5q-.63 0-1.06-.44T13 9.5V3H3v26h18z',
                'w' => .75,
                'vw' => 24
            ],
            'file-text-o' => 'file-alt',
            'fa-file-text' => 'file-alt',
            'thumbs-up' => [
                'p' => 'M29.13 17.94q.43 2.06-.57 3.94.32 2.25-1.06 4.18-.06 1.75-.78 3.07t-2.31 2.1-3.97.77h-1.38q-2.81 0-4.97-.62t-3.62-1.25-2.53-.63q-.2.63-.72 1.06T6 31H2q-.81 0-1.4-.6T0 29V14q0-.81.6-1.4T2 12h6.19q.5-.44 1.25-1.44t1.59-2.09 1.47-1.66q.19-.25.56-2.06t1.28-3.28T16.94 0q2.81 0 4.4 1.63t1.6 4.93q0 1.44-.57 2.88h2.25q1.5 0 2.75.78t1.94 2 .69 2.6q0 1.62-.88 3.12zM25.3 21.3q.94-.93.97-2.22t-.65-1.9q.43 0 .9-.75T27 14.8t-.69-1.62-1.68-.75h-6.5q0-1.13.9-2.9t.9-2.98q0-.62-.02-.97t-.2-.9-.43-.88-.87-.56T16.94 3q-.5.5-.94 2.72t-1.44 3.22q-.44.44-1.31 1.6t-1.57 2.06-1.53 1.65-1.47.75H8v11.63q1.56 0 4.96 1.18t5.79 1.2h2.3q4 0 3.38-4.13.75-.44 1.07-1.5t-.2-2.07zM5.5 27q0-.63-.44-1.06T4 25.5t-1.06.44T2.5 27t.44 1.06T4 28.5t1.06-.44T5.5 27z',
                'vw' => 30
            ],
            'thumbs-down' => [
                'p' => 'M29.13 14.06q.87 1.5.87 3.13 0 2.12-1.56 3.75t-3.82 1.62h-2.25q.57 1.44.57 2.88 0 3.31-1.6 4.94T16.94 32q-1.7 0-2.6-1.47t-1.28-3.28-.56-2.06q-.69-.69-2.19-2.66T8.2 20h-.44Q7.12 21 6 21H2q-.81 0-1.4-.6T0 19V4q0-.81.6-1.4T2 2h4q.75 0 1.31.5H8q.94 0 2.37-.62T14.03.63 19.06 0h1.38q3.5 0 5.25 1.6t1.8 4.34q1.38 1.94 1.07 4.19 1 1.87.56 3.93zM4 18.5q.63 0 1.06-.44T5.5 17t-.44-1.06T4 15.5t-1.06.44T2.5 17t.44 1.06T4 18.5zm20.63 1.06q.62 0 1.18-.34t.88-.9.31-1.13q0-.88-.47-1.63t-.9-.75q.68-.62.65-1.9t-.97-2.22q.5-1 .2-2.07t-1.07-1.5Q25.06 3 21.06 3h-2.31q-2.38 0-5.78 1.19T8 5.38V17h.69q.75 0 1.81 1.1t2.31 2.78 1.75 2.18q1 1 1.44 3.22t.94 2.72q1.06 0 1.72-.31t.9-.94.32-1.06.06-1.25q0-1.19-.9-2.97t-.91-2.9h6.5z',
                'vw' => 30
            ],
            'sun' => 'M30.88 13.88q1.12.74 1.12 2.09t-1.13 2.1l-3.75 2.56.88 4.43q.25 1.32-.69 2.25t-2.25.7l-4.44-.88-2.5 3.75Q17.32 32 15.97 32t-2.1-1.12l-2.5-3.75-4.43.87q-1.32.25-2.25-.69t-.7-2.3l.88-4.38-3.75-2.57Q0 17.31 0 15.96t1.12-2.09l3.75-2.56L4 6.87q-.25-1.3.69-2.25t2.25-.68l4.43.87 2.5-3.75Q14.62 0 16 0t2.12 1.06l2.5 3.75 4.44-.87q1.32-.25 2.25.68T28 6.87l-.88 4.44zm-7.07 5.37 4.82-3.31-4.82-3.25 1.13-5.7-5.69 1.13L16 3.32l-3.25 4.8L7.06 7l1.13 5.69-4.82 3.25 4.82 3.25-1.13 5.75 5.69-1.13L16 28.63l3.25-4.82 5.69 1.13zM16 9.5q2.69 0 4.6 1.9t1.9 4.6-1.9 4.6-4.6 1.9-4.6-1.9T9.5 16t1.9-4.6T16 9.5zm0 10q1.44 0 2.47-1.03T19.5 16t-1.03-2.47T16 12.5t-2.47 1.03T12.5 16t1.03 2.47T16 19.5z',
            'moon' => 'M17.44 32q-6.63 0-11.32-4.69T1.44 16 6.13 4.69t11.3-4.7q1.5 0 2.95.26 1.3.25 2 1.34t.4 2.35-1.47 1.87q-1.94 1.13-3.1 3.06t-1.15 4.2q0 3.93 3.03 6.43T27 21.25q2.06-.38 3.13 1.4t-.25 3.41q-2.25 2.81-5.5 4.38T17.43 32zm0-29q-5.38 0-9.19 3.81t-3.81 9.2 3.81 9.18T17.44 29q3 0 5.65-1.28t4.47-3.53q-5.25 1-9.37-2.4t-4.13-8.72q0-3.13 1.57-5.75t4.18-4.13Q18.63 3 17.44 3z',
            'caret-square-left' => [
                'p' => 'M17 9.81V22.2q0 .5-.47.69t-.78-.2L9.56 16.5q-.5-.5 0-1l6.19-6.19q.31-.37.78-.19t.47.7zM28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22q1.25 0 2.13.88T28 5zm-3 21.63V5.38Q25 5 24.62 5H3.38Q3 5 3 5.38v21.25q0 .37.38.37h21.25q.37 0 .37-.38z',
                'w' => .875,
                'vw' => 28
            ],
            'dot-circle' => 'M16 3.5q-5.19 0-8.84 3.66T3.5 16t3.66 8.84T16 28.5t8.84-3.66T28.5 16t-3.66-8.84T16 3.5zm0-3q6.44 0 10.97 4.53T31.5 16t-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5zM16 11q2.06 0 3.53 1.47T21 16t-1.47 3.53T16 21t-3.53-1.47T11 16t1.47-3.53T16 11z',
            'building' => [
                'p' => 'M8 9.25v-2.5Q8 6 8.75 6h2.5q.75 0 .75.75v2.5q0 .75-.75.75h-2.5Q8 10 8 9.25zm8.75.75Q16 10 16 9.25v-2.5q0-.75.75-.75h2.5q.75 0 .75.75v2.5q0 .75-.75.75h-2.5zm-8 6Q8 16 8 15.25v-2.5q0-.75.75-.75h2.5q.75 0 .75.75v2.5q0 .75-.75.75h-2.5zm8 0q-.75 0-.75-.75v-2.5q0-.75.75-.75h2.5q.75 0 .75.75v2.5q0 .75-.75.75h-2.5zM12 21.25q0 .75-.75.75h-2.5Q8 22 8 21.25v-2.5q0-.75.75-.75h2.5q.75 0 .75.75v2.5zm4.75.75q-.75 0-.75-.75v-2.5q0-.75.75-.75h2.5q.75 0 .75.75v2.5q0 .75-.75.75h-2.5zM28 29.75V32H0v-2.25Q0 29 .75 29H2V1.5q0-.63.44-1.06T3.5 0h21q.63 0 1.06.44T26 1.5V29h1.25q.75 0 .75.75zm-23-.81h7v-4.19q0-.75.75-.75h2.5q.75 0 .75.75v4.19h7V3.06L5 3v25.94z',
                'w' => .875,
                'vw' => 28
            ],
            'file-pdf' => [
                'p' => 'M23.13 6.13Q24 7 24 8.25V29q0 1.25-.88 2.13T21 32H3q-1.25 0-2.13-.88T0 29V3Q0 1.75.88.87T3 0h12.75Q17 0 17.88.88zM20.75 8 16 3.25V8h4.75zM3 29h18V11h-6.5q-.63 0-1.06-.44T13 9.5V3H3v26zm15.63-9q.62.5.3 1.47t-1.37.97q-1.56 0-3.5-1.06-.94.18-4.31 1.25-2.44 4.25-4.13 3.18-.75-.43-.62-1.25.31-1.37 3.19-2.87 1-1.75 2.25-4.63-.81-3-.44-4.18.19-.94 1.31-.88t1.32 1.25q.3.88-.32 3.5.75 1.94 2.32 2.88 3.18-.44 4 .37zM6.24 24.88q0 .06.22-.1t.69-.69.97-1.4q-1.63 1.56-1.88 2.19zM11.38 13q-.13 0-.13.44t.03 1.06.22 1q.19-.37.19-1.44T11.38 13zM9.8 21.5q2.82-1.06 3.44-1.19-1.12-.81-1.87-2.25-.82 2.19-1.57 3.44zm8.25-.31q.32-.19-.34-.38t-1.97-.12q.88.37 1.47.5t.72.06z',
                'w' => .75,
                'vw' => 24
            ],
            'file-word' => [
                'p' => 'M23.13 6.13Q24 7 24 8.25V29q0 1.25-.88 2.13T21 32H3q-1.25 0-2.13-.88T0 29V3Q0 1.75.88.87T3 0h12.75Q17 0 17.88.88zM20.75 8 16 3.25V8h4.75zM3 29h18V11h-6.5q-.63 0-1.06-.44T13 9.5V3H3v26zm13.75-13h1q.25 0 .44.13t.25.34.06.47l-2.19 8.5q-.12.56-.75.56h-1.5q-.62 0-.75-.56-.44-1.81-1.12-4.5L12 19.87q0 .5-.19 1.07l-1.12 4.5q-.13.56-.7.56H8.45q-.63 0-.75-.56-.2-.69-.6-2.4t-.84-3.48-.63-2.62q-.12-.38.13-.66T6.3 16h1.06q.63 0 .75.63 1.2 6 1.2 6.18 0 .07.05.25 0-.37 1.5-6.5.2-.56.76-.56h.8q.63 0 .76.56 1.44 6.07 1.5 6.25v.25q.06-.37 1.31-6.43.13-.63.75-.63z',
                'w' => .75,
                'vw' => 24
            ],
            'file-excel' => [
                'p' => 'M23.13 6.13Q24 7 24 8.25V29q0 1.25-.88 2.13T21 32H3q-1.25 0-2.13-.88T0 29V3Q0 1.75.88.87T3 0h12.75Q17 0 17.88.88zM20.75 8 16 3.25V8h4.75zM3 29h18V11h-6.5q-.63 0-1.06-.44T13 9.5V3H3v26zm13.25-15q.44 0 .66.38t-.04.74Q14.7 18.95 14 20l2.88 4.88q.25.37.03.75t-.66.37h-1.81q-.44 0-.63-.38Q13.7 25.45 12 22q-.75 1.69-1.81 3.63-.2.37-.63.37H7.75q-.44 0-.66-.38t.03-.75L10 20l-2.88-4.88q-.25-.37-.03-.74t.66-.38h1.81q.44 0 .69.38L12 18q.56-1.38 1.81-3.63.19-.37.63-.37h1.81z',
                'w' => .75,
                'vw' => 24
            ],
            'file-powerpoint' => [
                'p' => 'M23.13 6.13Q24 7 24 8.25V29q0 1.25-.88 2.13T21 32H3q-1.25 0-2.13-.88T0 29V3Q0 1.75.88.87T3 0h12.75Q17 0 17.88.88zM20.75 8 16 3.25V8h4.75zM3 29h18V11h-6.5q-.63 0-1.06-.44T13 9.5V3H3v26zm4.5-3.75v-10.5q0-.75.75-.75h4.31q1.75 0 2.85 1.16t1.1 2.97q0 .68-.13 1.25t-.32.96-.53.75-.62.54-.75.34-.78.22-.82.1-.75.02H10.5v2.94q0 .31-.22.53t-.53.22h-1.5q-.75 0-.75-.75zm3-5.44H12q.69 0 1.1-.47t.4-1.18-.4-1.16-1.1-.44h-1.5v3.25z',
                'w' => .75,
                'vw' => 24
            ],
            'file-image' => [
                'p' => 'M23.13 6.13Q24 7 24 8.25V29q0 1.25-.88 2.13T21 32H3q-1.25 0-2.13-.88T0 29V3Q0 1.75.88.87T3 0h12.75Q17 0 17.88.88zM20.75 8 16 3.25V8h4.75zM3 29h18V11h-6.5q-.63 0-1.06-.44T13 9.5V3H3v26zm2-3v-4l2.5-2.5q.5-.5 1 0L11 22l5.5-5.5q.5-.5 1 0L19 18v8H5zm3-15q1.25 0 2.13.88T11 14t-.88 2.13T8 17t-2.13-.88T5 14t.88-2.13T8 11z',
                'w' => .75,
                'vw' => 24
            ],
            'file-photo' => 'file-image',
            'file-photo-o' => 'file-image',
            'file-picture-o' => 'file-image',
            'file-archive' => [
                'p' => 'M8 10h2v2H8v-2zm4-6v2h-2V4h2zM8 6h2v2H8V6zm4 2v2h-2V8h2zm11.13-1.88Q24 7 24 8.25V29q0 1.25-.88 2.13T21 32H3q-1.25 0-2.13-.88T0 29V3Q0 1.75.88.87T3 0h12.75Q17 0 17.88.88zM16 3.25V8h4.75zM21 29V11h-6.5q-.63 0-1.06-.44T13 9.5V3h-3v1H8V3H3v26h18zm-8.88-12.38 1.13 5.44q.19 1-.22 1.9t-1.22 1.48T10 26q-1.56 0-2.53-1.22t-.66-2.72L8 16v-2h2v2h1.38q.62 0 .74.63zm-2.09 7.75q.85 0 1.44-.5t.6-1.18q0-.44-.29-.85t-.75-.62T10 21t-1 .22-.72.62-.28.85q0 .68.6 1.18t1.43.5zM12 12v2h-2v-2h2z',
                'w' => .75,
                'vw' => 24
            ],
            'file-audio' => [
                'p' => 'M23.13 6.13Q24 7 24 8.25V29q0 1.25-.88 2.13T21 32H3q-1.25 0-2.13-.88T0 29V3Q0 1.75.88.87T3 0h12.75Q17 0 17.88.88zM20.75 8 16 3.25V8h4.75zM3 29h18V11h-6.5q-.63 0-1.06-.44T13 9.5V3H3v26zm9-4.75q0 .5-.47.69t-.78-.19L8.5 22.5H6.75q-.31 0-.53-.22T6 21.75v-3.5q0-.25.22-.5t.53-.25H8.5l2.25-2.25q.31-.38.78-.19t.47.69v8.5zm2.56-2.94q.44-.43.44-1.06t-.44-1.06q-.62-.63-.34-1.47t1.03-1.03 1.5.43Q18 18.45 18 20.25t-1.25 3.13q-.69.68-1.5.43t-1.03-1 .34-1.5z',
                'w' => .75,
                'vw' => 24
            ],
            'file-video' => [
                'p' => 'M23.13 6.13Q24 7 24 8.25V29q0 1.25-.88 2.13T21 32H3q-1.25 0-2.13-.88T0 29V3Q0 1.75.88.87T3 0h12.75Q17 0 17.88.88zM20.75 8 16 3.25V8h4.75zM3 29h18V11h-6.5q-.63 0-1.06-.44T13 9.5V3H3v26zm14.31-13.19q.44-.5 1.07-.25t.62.94v7q0 .69-.62.94t-1.07-.25L14 20.94v2.31q0 .5-.38.88t-.87.37h-6.5q-.5 0-.88-.37T5 23.25v-6.5q0-.5.38-.87t.87-.38h6.5q.5 0 .88.38t.37.87v2.31z',
                'w' => .75,
                'vw' => 24
            ],
            'file-code' => [
                'p' => 'M9.38 21.81q.37.38 0 .82L8.3 23.75q-.19.19-.44.19-.18 0-.37-.19l-3.56-3.37q-.44-.38 0-.75l3.56-3.38q.44-.44.81 0l1.06 1.13v.06q.38.37 0 .75L7.31 20zM23.12 6.12Q24 7 24 8.25V29q0 1.25-.88 2.13T21 32H3q-1.25 0-2.13-.88T0 29V3Q0 1.75.88.87T3 0h12.75Q17 0 17.88.88zM16 3.25V8h4.75zM21 29V11h-6.5q-.63 0-1.06-.44T13 9.5V3H3v26h18zm-7.88-15.63 1.5.44q.57.13.38.7l-3.44 11.74q-.12.56-.69.38l-1.5-.44q-.56-.13-.37-.69l3.44-11.75q.12-.5.68-.37zm1.5 4.82q-.37-.38 0-.82l1.07-1.12q.37-.44.81 0l3.56 3.38q.44.37 0 .75l-3.56 3.37q-.44.44-.81 0l-1.06-1.13v-.06q-.38-.37 0-.75L16.68 20z',
                'w' => .75,
                'vw' => 24
            ],
            'life-ring' => 'M16 31.5q-6.44 0-10.97-4.53T.5 16 5.03 5.03 16 .5t10.97 4.53T31.5 16t-4.53 10.97T16 31.5zm-6.44-4.81Q12.5 28.5 16 28.5t6.44-1.81l-3.31-3.32q-3.13 1.32-6.25 0zM21 16q0-2.06-1.47-3.53T16 11t-3.53 1.47T11 16t1.47 3.53T16 21t3.53-1.47T21 16zm5.69 6.44Q28.5 19.5 28.5 16t-1.8-6.44l-3.31 3.32q1.3 3.12 0 6.25zM22.44 5.3Q19.5 3.5 16 3.5T9.56 5.31l3.31 3.32q3.13-1.32 6.26 0zM5.3 9.56Q3.5 12.5 3.5 16t1.81 6.44l3.32-3.31q-1.32-3.13 0-6.25z',
            'paper-plane' => 'M27.5.38q1.63-.88 3.19.15T32 3.43L28.25 27.7q-.25 1.43-1.5 2.12-.69.38-1.44.38-.62 0-1.18-.2l-7-2.87-2.7 3.7q-.87 1.06-2.15 1.18T10 31.25 9 29v-5.25l-7.13-3Q.13 20.06 0 18.19t1.5-2.82zM12 29l2.31-3.13-2.31-.93V29zm13.31-1.81L29.01 3 3 18l6.74 2.81 13.2-11.56q.69-.56 1.34.03t.16 1.35l-8.75 12.62z',
            'futbol' => 'M30.25 11.19q2 6.12-.94 11.84t-9 7.72q-2.37.75-4.81.75-4.94 0-9.06-2.94t-5.7-7.75q-2-6.12.95-11.84t9-7.72Q13.06.5 15.49.5q3.26 0 6.23 1.31t5.22 3.75 3.31 5.63zM25.56 23.3q.75-1 1.07-1.62 1.3-2.63 1.37-5.56l-2.44-2.32-4.43 1.38-1.57 4.75 2.75 3.81zm0-14.68q-2.31-3.25-6.12-4.5L16.5 5.75v4.63l4 2.93 4.44-1.37zm-14-4.5q-3.75 1.25-6.12 4.5l.62 3.3 4.44 1.38 4-2.93V5.75zM8.7 23.75l2.69-3.88-1.5-4.62-4.44-1.44L3 16.12q.06 3.94 2.38 7.2zm3 4.13q3.62 1.18 7.62 0l1.38-3-2.75-3.82H13l-2.69 3.82z',
            'newspaper' => [
                'p' => 'M34.5 4q.63 0 1.06.44T36 5.5v21q0 .63-.44 1.06T34.5 28h-31q-1.44 0-2.47-1.03T0 24.5v-17q0-.62.44-1.06T1.5 6h2.7q.31-.87 1.1-1.44T7 4h27.5zM3 24.5q0 .5.5.5t.5-.5V9H3v15.5zm30 .5V7H7v17.5q0 .25-.06.5H33zm-22.25-7.5q-.75 0-.75-.75v-6q0-.75.75-.75h8.5q.75 0 .75.75v6q0 .75-.75.75h-8.5zm1.75-5V15h5v-2.5h-5zM10 21.25v-1.5q0-.75.75-.75h8.5q.75 0 .75.75v1.5q0 .75-.75.75h-8.5q-.75 0-.75-.75zm12 0v-1.5q0-.75.75-.75h6.5q.75 0 .75.75v1.5q0 .75-.75.75h-6.5q-.75 0-.75-.75zm0-9v-1.5q0-.75.75-.75h6.5q.75 0 .75.75v1.5q0 .75-.75.75h-6.5q-.75 0-.75-.75zm0 4.5v-1.5q0-.75.75-.75h6.5q.75 0 .75.75v1.5q0 .75-.75.75h-6.5q-.75 0-.75-.75z',
                'w' => 1.125,
                'vw' => 36
            ],
            'bell-slash' => [
                'p' => 'M39.63 29.44q.74.62.12 1.44l-.63.75q-.62.75-1.37.12L.37 2.56q-.74-.62-.12-1.44L.88.37Q1.5-.37 2.25.25zM10.25 23h11.38l3.87 3H8q-.88 0-1.44-.6T6 24q0-.75.56-1.38l.19-.18.78-.88q.28-.3.78-1.12t.78-1.63.57-2.12.28-2.82l2.87 2.2q-.56 4.24-2.56 6.93zM20 6q-2.13 0-3.94 1.19L13.7 5.37q1.93-1.56 4.3-2.06V2q0-.81.6-1.4T20 0t1.4.6T22 2v1.3q2.31.44 4.13 1.82t2.84 3.43T30 13q0 3.44.88 5.81l-3.7-2.93Q27 14.44 27 13.06V13q0-2.88-2.06-4.94T20 6zm0 26q-1.63 0-2.81-1.16T15.99 28h8q0 1.06-.52 2T22 31.47 20 32z',
                'w' => 1.25,
                'vw' => 40
            ],
            'copyright' => 'M16 .5q6.44 0 10.97 4.53T31.5 16t-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5zm0 28q5.19 0 8.84-3.66T28.5 16t-3.66-8.84T16 3.5 7.16 7.16 3.5 16t3.66 8.84T16 28.5zm6.69-6.31q-2.57 2.56-6.5 2.56-3.82 0-6.28-2.53t-2.47-6.28q0-2.44 1.12-4.44t3.13-3.15 4.5-1.16q3.69 0 6.06 2.19.25.25.25.56 0 .19-.12.38l-1.13 1.8q-.44.7-1.12.13-1.75-1.37-3.82-1.37-1.5 0-2.62.65T12 13.35t-.56 2.53q0 2.12 1.31 3.69t3.56 1.56 4.13-1.69q.25-.25.56-.22t.56.28l1.25 1.7q.13.24.13.5t-.25.5z',
            'closed-captioning' => 'M29 4q1.25 0 2.13.88T32 7v18q0 1.25-.88 2.13T29 28H3q-1.25 0-2.13-.88T0 25V7q0-1.25.88-2.13T3 4h26zm-.38 21q.38 0 .38-.38V7.38Q29 7 28.62 7H3.38Q3 7 3 7.38v17.25q0 .37.38.37h25.25zm-13.18-5.38q.19.25-.06.5-1.63 1.75-4.2 1.91T6.63 20.5t-2-4.63q0-2.87 2.03-4.5t4.6-1.53 4.12 1.66q.2.19.07.38l-1.07 1.87q-.25.38-.62.13-1.88-1.5-3.88-.85t-2 2.78q0 2.32 1.91 3.03t3.84-.97q.32-.3.57.07zm11.87 0q.2.25 0 .5-1.68 1.75-4.25 1.91T18.5 20.5t-2-4.63q0-2.87 2.03-4.5t4.6-1.53 4.12 1.66q.19.19.13.38l-1.13 1.87q-.25.38-.56.13-1.94-1.5-3.94-.85t-2 2.78q0 2.32 1.94 3.03t3.81-.97q.38-.3.63.07z',
            'object-group' => 'M31.25 8h-.75v16h.75q.75 0 .75.75v4.5q0 .75-.75.75h-4.5q-.75 0-.75-.75v-.75H6v.75q0 .75-.75.75H.75Q0 30 0 29.25v-4.5Q0 24 .75 24h.75V8H.75Q0 8 0 7.25v-4.5Q0 2 .75 2h4.5Q6 2 6 2.75v.75h20v-.75q0-.75.75-.75h4.5q.75 0 .75.75v4.5q0 .75-.75.75zM28 4v2h2V4h-2zM2 4v2h2V4H2zm2 24v-2H2v2h2zm26 0v-2h-2v2h2zm-2.5-4V8h-.75Q26 8 26 7.25V6.5H6v.75Q6 8 5.25 8H4.5v16h.75q.75 0 .75.75v.75h20v-.75q0-.75.75-.75h.75zm-2.25-12q.75 0 .75.75v10.5q0 .75-.75.75h-12.5q-.75 0-.75-.75V20H6.75Q6 20 6 19.25V8.75Q6 8 6.75 8h12.5q.75 0 .75.75V12h5.25zM8.5 10.5v7h9v-7h-9zm15 11v-7H20v4.75q0 .75-.75.75H14.5v1.5h9z',
            'object-ungroup' => [
                'p' => 'M35.25 14h-.75v10h.75q.75 0 .75.75v4.5q0 .75-.75.75h-4.5q-.75 0-.75-.75v-.75H16v.75q0 .75-.75.75h-4.5q-.75 0-.75-.75v-4.5q0-.75.75-.75h.75v-1.5H6v.75q0 .75-.75.75H.75Q0 24 0 23.25v-4.5Q0 18 .75 18h.75V8H.75Q0 8 0 7.25v-4.5Q0 2 .75 2h4.5Q6 2 6 2.75v.75h14v-.75q0-.75.75-.75h4.5q.75 0 .75.75v4.5q0 .75-.75.75h-.75v1.5H30v-.75q0-.75.75-.75h4.5q.75 0 .75.75v4.5q0 .75-.75.75zM22 4v2h2V4h-2zm0 16v2h2v-2h-2zM4 22v-2H2v2h2zM4 6V4H2v2h2zm2 13.5h14v-.75q0-.75.75-.75h.75V8h-.75Q20 8 20 7.25V6.5H6v.75Q6 8 5.25 8H4.5v10h.75q.75 0 .75.75v.75zm8 8.5v-2h-2v2h2zm17.5-4V14h-.75q-.75 0-.75-.75v-.75h-5.5V18h.75q.75 0 .75.75v4.5q0 .75-.75.75h-4.5q-.75 0-.75-.75v-.75h-5.5V24h.75q.75 0 .75.75v.75h14v-.75q0-.75.75-.75h.75zm2.5 4v-2h-2v2h2zm0-16v-2h-2v2h2z',
                'w' => 1.125,
                'vw' => 36
            ],
            'sticky-note' => [
                'p' => 'M28 21.75q0 1.25-.88 2.13l-5.25 5.25Q21 30 19.75 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22q1.25 0 2.13.88T28 5v16.75zm-8 5L24.75 22H20v4.75zM25 5H3v22h14v-6.5q0-.63.44-1.06T18.5 19H25V5z',
                'w' => .875,
                'vw' => 28
            ],
            'clone' => 'M29 0q1.25 0 2.13.88T32 3v20q0 1.25-.88 2.13T29 26h-3v3q0 1.25-.88 2.13T23 32H3q-1.25 0-2.13-.88T0 29V9q0-1.25.88-2.13T3 6h3V3q0-1.25.88-2.13T9 0h20zm-6.38 29q.38 0 .38-.38V26H9q-1.25 0-2.13-.88T6 23V9H3.37Q3 9 3 9.38v19.24q0 .38.38.38h19.25zm6-6q.38 0 .38-.38V3.38Q29 3 28.62 3H9.38Q9 3 9 3.38v19.25q0 .37.38.37h19.24z',
            'hourglass' => [
                'p' => 'M23 3q0 3.94-1.56 7.53T16.94 16q2.94 1.88 4.5 5.47T23 29h.25q.75 0 .75.75v1.5q0 .75-.75.75H.75Q0 32 0 31.25v-1.5Q0 29 .75 29H1q0-3.94 1.56-7.53T7.06 16q-2.94-1.88-4.5-5.47T1 3H.75Q0 3 0 2.25V.75Q0 0 .75 0h22.5Q24 0 24 .75v1.5q0 .75-.75.75H23zM4 3q0 4.75 2.34 8.13T12 14.5t5.66-3.38T20 3H4zm16 26q0-4.75-2.34-8.13T12 17.5t-5.66 3.38T4 29h16z',
                'w' => .75,
                'vw' => 24
            ],
            'hand-rock' => 'M25.56 4.94q2.25-1 4.35.37T32 9.2v7.12q0 1.5-.57 2.82l-2.69 6.25q-.25.56-.25 2.62 0 .81-.6 1.4t-1.4.6H10.75q-.81 0-1.4-.6t-.6-1.4v-1.66l-.03-.15-.03-.1-.03-.06-.1-.1-6.12-5.3Q0 18.5 0 15.25v-3.56q0-2.2 1.75-3.57t3.94-.87q.12-2.56 2.4-3.81t4.54 0Q14 2 16 2t3.38 1.44q1.5-.88 3.3-.47t2.88 1.97zM29 16.3V9.2q0-1-.81-1.4t-1.63 0-.81 1.34q0 .43-.28.72t-.72.28h-.44q-.44 0-.72-.28t-.28-.72V7.5q0-1.06-.81-1.47t-1.63 0-.81 1.4v1.7q0 .43-.28.71t-.72.29h-.44q-.44 0-.72-.29t-.28-.71v-2.5q0-1-.81-1.44t-1.63-.03-.8 1.4v2.57q0 .43-.29.71t-.72.29h-.44q-.43 0-.72-.29t-.28-.71V7.5q0-1.06-.81-1.47t-1.62 0-.82 1.4v7.26q0 .44-.28.72t-.72.28q-.37 0-.69-.25l-.43-.38q-.32-.25-.32-.75v-2.56q0-1-.81-1.4t-1.62 0-.82 1.34v3.56q0 1.88 1.44 3.13l6.06 5.3q1.25 1.07 1.25 2.7V27H25.5v-.44q0-1.25.5-2.37l2.7-6.25q.3-.75.3-1.63z',
            'hand-paper' => [
                'p' => 'M23.31 7.06q1.82-.31 3.25.97T28 11.2v8.25q0 1.44-.31 2.81l-1.88 8.19q-.18.69-.72 1.12t-1.28.44H9.94q-1.06 0-1.69-.81L.81 20.63q-1-1.38-.75-3.1t1.63-2.72q1.81-1.31 3.87-.56V6.37q0-2.06 1.6-3.3t3.59-.76Q11.88.06 14.38 0t3.74 2.25q1.94-.5 3.57.78t1.62 3.34v.7zM25 19.44v-8.25q0-.75-.56-1.07t-1.16 0-.6 1.13V16q0 .44-.27.72t-.72.28h-.38q-.44 0-.72-.28t-.28-.72V6.37q0-.75-.6-1.09t-1.15-.03T18 6.31v9.7q0 .43-.28.71T17 17h-.44q-.37 0-.69-.28t-.31-.72V4.25q0-.75-.56-1.1t-1.13-.02-.56 1.06V16q0 .44-.31.72t-.7.28h-.43q-.44 0-.72-.28t-.28-.72V6.44q0-.75-.6-1.1t-1.15-.03-.56 1.07v12.87q0 .75-.7.97t-1.12-.34L5.06 17.5q-.25-.37-.63-.47t-.69.03-.53.41-.22.66.25.75L10.44 29h12.62l1.68-7.44q.25-1.06.25-2.12z',
                'w' => .875,
                'vw' => 28
            ],
            'hand-scissors' => 'M16 30q-2.13 0-3.38-1.69T12 24.62q-1-.62-1.53-1.72t-.35-2.34h-5.5q-1.93 0-3.28-1.34T0 16t1.34-3.22 3.29-1.34l5.87.56-4.94-1.19Q3.75 10.12 3 8.37t-.06-3.5 2.43-2.5 3.57-.06L18 5.8q2.69-3.31 5.75-1.56l6.13 3.43Q32 8.88 32 11.31v12.81q0 1.44-.9 2.6t-2.35 1.47l-7.44 1.68q-.44.13-.94.13H16zm0-3h4.38q.12 0 .25-.06l7.43-1.7q.94-.18.94-1.12v-12.8q0-.7-.56-1L22.3 6.86q-.93-.5-1.56.25l-1.38 1.7q-.3.37-.75.37-.18 0-.37-.07l-10.38-4q-1.43-.56-2.06.7-.69 1.3.5 2.05.19.07.31.2l9.75 3.68q.63.25.63.94v.75q0 .37-.28.69t-.72.3H4.62q-1 0-1.43.79t0 1.56 1.44.78H16q.44 0 .72.32t.28.68V19q0 .44-.28.72T16 20h-1.75q-.75 0-1.06.57t0 1.15 1.06.6H16q.44 0 .72.28t.28.72v.37q0 .44-.28.72t-.72.28q-.75 0-1.06.6t0 1.15T16 27z',
            'hand-lizard' => [
                'p' => 'M34.81 18.19q1.2 1.81 1.2 4.06V30H22v-4.5l-6.26-2.37q-.25-.13-.5-.13H7q-1.44 0-2.47-1.03T3.5 19.5q0-2.25 1.63-3.87T9 14h7.13l1.12-3H5q-2.06 0-3.53-1.47T0 6v-.5q0-1.44 1.03-2.47T3.5 2h18.38q2.43 0 3.75 2.06zM33 27v-4.75q0-1.31-.75-2.44L23.12 5.7q-.43-.7-1.25-.7H3.5Q3 5 3 5.5V6q0 .82.6 1.42T5 8h13.31q1.2 0 1.88.97t.25 2.1l-1.5 4q-.25.55-.69 1t-1 .68-1.12.25H9q-1.06 0-1.78.72T6.5 19.5q0 .5.5.5h8.25q.81 0 1.63.31l6.18 2.44q.88.31 1.4 1.06t.54 1.7V27h8z',
                'w' => 1.125,
                'vw' => 36
            ],
            'hand-spock' => 'M1.31 23.88Q.06 22.68 0 20.93t1.13-3q1.12-1.2 2.78-1.32t2.96.94l-1.93-8.5q-.44-1.94.78-3.5t3.22-1.62Q9 2.56 9.9 1.5T12.19.12q1.68-.37 3.15.54t1.85 2.59l1.37 8.25 1.07-6.56q.37-1.7 1.87-2.6t3.19-.47q1.25.32 2.06 1.25t1.06 2.13q2 0 3.28 1.6t.79 3.59l-2 8.3q-.07.38-.07.76v2.31q0 1.07-.31 2.07l-2 6.68q-.19.63-.72 1.03t-1.16.41H10.76q-.81 0-1.38-.56zM3.34 20q-.34.38-.34.84t.38.85L11.13 29h13.74l1.75-6q.2-.56.2-1.19V19.5q0-.75.18-1.44l1.94-8.37q.19-.7-.31-1.13t-1.16-.28-.78.9L25 16.26q-.19.75-1 .75h-.62q-.44 0-.75-.38t-.2-.87l2.38-9.5q.13-.44-.06-.81t-.5-.53-.66-.16-.65.22-.44.66l-2.63 10.62q-.18.75-1 .75h-.8q-.82 0-1-.75L14.24 3.94q-.19-.75-.81-.9t-1.16.27-.34 1.13l2.62 11.31q.06.25 0 .47t-.19.4-.34.29-.47.09h-.5q-.81 0-1-.75l-1.88-8.38q-.18-.75-.84-.9t-1.15.28-.38 1.13l2.2 9.36v4.56q0 .7-.6.94t-1.09-.19L5 19.94q-.38-.32-.85-.32t-.81.38z',
            'hand-pointer' => [
                'p' => 'M22.38 11.19q2.06-.75 3.84.5T28 15.13v5.25q0 .5-.13.93l-1.68 7.44q-.32 1.44-1.47 2.35t-2.6.9H11q-2.13 0-3.38-1.75l-6.8-9.62q-1-1.38-.76-3.1t1.63-2.72q1.8-1.3 3.87-.56V4.2q0-1.75 1.22-2.97T9.72 0t2.94 1.22 1.22 2.97V8.8q2.06-.31 3.5 1.2 1.3-.63 2.71-.29t2.29 1.47zM5.05 17.5q-.44-.63-1.1-.5t-.87.69.16 1.19l6.81 9.62q.38.5.94.5h11.13q.93 0 1.12-.94l1.69-7.43q.06-.13.06-.25v-5.25q0-.76-.56-1.07t-1.16 0-.6 1.06q0 .26-.12.5t-.34.38-.53.13h-.38q-.43 0-.71-.29t-.29-.72v-1.3q0-.76-.59-1.07t-1.16 0T18 13.8v1.31q0 .44-.28.72t-.72.28h-.44q-.25 0-.5-.12t-.37-.38-.13-.5v-2.18q0-.75-.56-1.07t-1.12 0-.57 1.07v2.19q0 .24-.15.5t-.38.37-.47.13h-.44q-.43 0-.71-.29t-.29-.72V4.2q0-.75-.59-1.07t-1.16 0-.56 1.07v15.06q0 .75-.69.97t-1.12-.34zM11 25v-6q0-.44.25-.72t.63-.28h.37q.38 0 .63.28t.24.72v6q0 .19-.06.38t-.19.3-.28.23-.34.1h-.38q-.37 0-.62-.29T11 25zm4.75 0v-6q0-.44.25-.72t.63-.28H17q.31 0 .6.28t.27.72v6q0 .44-.28.72T17 26h-.38q-.37 0-.62-.28t-.25-.72zm4.69 0v-6q0-1 .87-1h.38q.87 0 .87 1v6q0 1-.87 1h-.38q-.87 0-.87-1z',
                'w' => .875,
                'vw' => 28
            ],
            'hand-peace' => [
                'p' => 'M22.63 12q1.3-.44 2.56 0t2.03 1.53T28 16v4.38q0 .5-.13.93l-1.68 7.44q-.32 1.44-1.47 2.34t-2.6.91H9.32q-2.45 0-3.63-2.12l-3.44-6.13Q.5 20.7 3.8 18L.31 8.94q-.69-1.82.06-3.57t2.5-2.43q1.25-.44 2.41-.28t2.13.93 1.4 1.97L10 10.5l-.56-5.87q0-1.94 1.34-3.29T14 0t3.22 1.34 1.34 3.29v5.5q1.25-.2 2.35.34T22.63 12zM25 20.38V16q0-.75-.56-1.06t-1.16 0-.6 1.06q0 .44-.27.72t-.72.28h-.38q-.44 0-.72-.28t-.28-.72v-1.75q0-.75-.6-1.06t-1.15 0-.56 1.06V16q0 .44-.28.72T17 17h-.44q-.37 0-.69-.28t-.31-.72V4.62q0-1-.78-1.43t-1.56 0-.79 1.44V16q0 .44-.3.72t-.7.28h-.75q-.68 0-.93-.63l-3.7-9.75Q5.69 5.7 4.82 5.6t-1.47.6-.22 1.69l4 10.37q.06.19.06.38 0 .43-.37.75l-1.69 1.37q-.75.63-.25 1.56l3.44 6.13q.31.56 1 .56h12.81q.94 0 1.13-.94l1.68-7.43q.07-.13.07-.25z',
                'w' => .875,
                'vw' => 28
            ],
            'registered' => 'M16 .5q6.44 0 10.97 4.53T31.5 16t-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5zm0 28q5.19 0 8.84-3.66T28.5 16t-3.66-8.84T16 3.5 7.16 7.16 3.5 16t3.66 8.84T16 28.5zm6.88-5.13q.12.2.12.41t-.1.38-.27.25-.38.09h-2.69q-.44 0-.62-.38l-3-5.62h-2v5.25q0 .31-.22.53t-.53.22h-2.44q-.25 0-.5-.22t-.25-.53V8.25q0-.31.25-.53t.5-.22h4.94q6.37 0 6.37 5.31 0 3.2-2.5 4.57.13.18 3.32 6zM16.05 15q2 0 2-2.06 0-.63-.15-1.03t-.72-.66-1.57-.25h-1.68v4h2.12z',
            'calendar-plus' => [
                'p' => 'M21 18.25v1.5q0 .75-.75.75H15.5v4.75q0 .75-.75.75h-1.5q-.75 0-.75-.75V20.5H7.75q-.75 0-.75-.75v-1.5q0-.75.75-.75h4.75v-4.75q0-.75.75-.75h1.5q.75 0 .75.75v4.75h4.75q.75 0 .75.75zM28 7v22q0 1.25-.88 2.13T25 32H3q-1.25 0-2.13-.88T0 29V7q0-1.25.88-2.13T3 4h3V.75Q6 0 6.75 0h2.5Q10 0 10 .75V4h8V.75q0-.75.75-.75h2.5Q22 0 22 .75V4h3q1.25 0 2.13.88T28 7zm-3 21.63V10H3v18.63q0 .37.38.37h21.25q.37 0 .37-.38z',
                'w' => .875,
                'vw' => 28
            ],
            'calendar-minus' => [
                'p' => 'M7.75 20.5q-.75 0-.75-.75v-1.5q0-.75.75-.75h12.5q.75 0 .75.75v1.5q0 .75-.75.75H7.75zM28 7v22q0 1.25-.88 2.13T25 32H3q-1.25 0-2.13-.88T0 29V7q0-1.25.88-2.13T3 4h3V.75Q6 0 6.75 0h2.5Q10 0 10 .75V4h8V.75q0-.75.75-.75h2.5Q22 0 22 .75V4h3q1.25 0 2.13.88T28 7zm-3 21.63V10H3v18.63q0 .37.38.37h21.25q.37 0 .37-.38z',
                'w' => .875,
                'vw' => 28
            ],
            'calendar-times' => [
                'p' => 'm19.5 23.44-1.06 1.06q-.57.5-1.07 0L14 21.13l-3.38 3.37q-.5.5-1.06 0L8.5 23.44q-.5-.57 0-1.07L11.87 19 8.5 15.62q-.5-.5 0-1.06l1.06-1.06q.56-.5 1.06 0L14 16.87l3.37-3.37q.5-.5 1.07 0l1.06 1.06q.5.56 0 1.06L16.12 19l3.38 3.38q.5.5 0 1.06zM28 7v22q0 1.25-.88 2.13T25 32H3q-1.25 0-2.13-.88T0 29V7q0-1.25.88-2.13T3 4h3V.75Q6 0 6.75 0h2.5Q10 0 10 .75V4h8V.75q0-.75.75-.75h2.5Q22 0 22 .75V4h3q1.25 0 2.13.88T28 7zm-3 21.63V10H3v18.63q0 .37.38.37h21.25q.37 0 .37-.38z',
                'w' => .875,
                'vw' => 28
            ],
            'calendar-check' => [
                'p' => 'M25 4q1.25 0 2.13.88T28 7v22q0 1.25-.88 2.13T25 32H3q-1.25 0-2.13-.88T0 29V7q0-1.25.88-2.13T3 4h3V.75Q6 0 6.75 0h2.5Q10 0 10 .75V4h8V.75q0-.75.75-.75h2.5Q22 0 22 .75V4h3zm-.38 25q.38 0 .38-.38V10H3v18.63q0 .37.38.37h21.25zm-3.3-12.56-8.88 8.87q-.57.5-1.07-.06L6.7 20.56q-.57-.56 0-1.06l1.43-1.44q.5-.5 1.07 0l2.75 2.81L18.88 14q.5-.56 1.06 0l1.37 1.37q.57.57 0 1.07z',
                'w' => .875,
                'vw' => 28
            ],
            'map' => [
                'p' => 'M35 2q.38 0 .69.28T36 3v21.63q0 .37-.15.75t-.47.68-.63.44l-9.5 3.31q-.62.2-1.25.2t-1.25-.2L12 26.01 1.38 29.93Q1.18 30 1 30q-.37 0-.69-.28T0 29V7.38q0-.38.16-.75t.46-.7.63-.43l9.5-3.31Q11.38 2 12 2t1.25.19L24 6l10.63-3.94Q34.8 2 35 2zM14 5.63V23.5l8 2.88V8.5zM3 26.13l7.94-2.94H11V5.3L3 8.06v18.06zm30-2.2V5.89L25.06 8.8H25v17.9z',
                'w' => 1.125,
                'vw' => 36
            ],
            'comment-alt' => 'M28 0q1.6 0 2.8 1.2T32 4v18q0 1.6-1.2 2.8T28 26h-9l-7.8 6-.4.1q-.8 0-.8-.8V26H4q-1.6 0-2.8-1.2T0 22V4q0-1.6 1.2-2.8T4 0h24zm1 22V4q0-.4-.3-.7T28 3H4q-.4 0-.7.3T3 4v18q0 .4.3.7t.7.3h9v3.8l4.2-3.2.8-.6h10q.4 0 .7-.3t.3-.7z',
            'pause-circle' => 'M16 .5q6.44 0 10.97 4.53T31.5 16t-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5zm0 28q5.19 0 8.84-3.66T28.5 16t-3.66-8.84T16 3.5 7.16 7.16 3.5 16t3.66 8.84T16 28.5zM22 11v10q0 .44-.28.72T21 22h-3q-.44 0-.72-.28T17 21V11q0-.44.28-.72T18 10h3q.44 0 .72.28T22 11zm-7 0v10q0 .44-.28.72T14 22h-3q-.44 0-.72-.28T10 21V11q0-.44.28-.72T11 10h3q.44 0 .72.28T15 11z',
            'stop-circle' => 'M31.5 16q0 6.44-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5t10.97 4.53T31.5 16zm-28 0q0 5.19 3.66 8.84T16 28.5t8.84-3.66T28.5 16t-3.66-8.84T16 3.5 7.16 7.16 3.5 16zM22 11v10q0 .44-.28.72T21 22H11q-.44 0-.72-.28T10 21V11q0-.44.28-.72T11 10h10q.44 0 .72.28T22 11z',
            'handshake' => [
                'p' => 'M32.44 8H40v16h-4q-.81 0-1.4-.6T34 22h-3.62q-.2.81-.7 1.44l-1.62 2q-.75.94-1.84 1.22t-2.16-.16Q22.88 28 21 28q-1.38 0-2.56-1-1.44 1.06-3.28 1t-3.28-1.25L6.55 22H6q0 .56-.28 1t-.72.72T4 24H0V8h7.37l3-3q1-1 2.44-1H27q1.44 0 2.5 1zM3 22q.44 0 .72-.31t.28-.72-.28-.69T3 20t-.72.28-.28.69.28.72T3 22zm24.38-.44q.3-.43-.07-.75l-6.75-5.43-.56.43q-1.13 1.07-2.53 1.16t-2.31-.38-1.47-1.03q-1.25-1.37-1.2-3.22t1.45-3.15L16.37 7h-3.56q-.06 0-.16.06t-.15.07L8.62 11H6v8h1.75l6.06 5.5q.75.63 1.78.5t1.66-.94l1-1.18 2.31 2q.56.43 1.13-.2l1.87-2.24 1.5 1.18q.38.32.69-.06zM34 19v-8h-2.81l-3.82-3.88Q27.2 7 27 7h-5.25q-.94 0-1.69.69L16 11.44q-.5.37-.5 1t.44 1.12q.37.44 1.03.4t1.1-.4l3.43-3.19q.75-.68 1.44.07l.68.75q.7.75-.06 1.37l-.81.75 6.44 5.2.5.5H34zm3 3q.44 0 .72-.31t.28-.72-.28-.69T37 20t-.72.28-.28.69.28.72.72.31z',
                'w' => 1.25,
                'vw' => 40
            ],
            'envelope-open' => 'M30.94 10.31Q32 11.2 32 12.63V29q0 1.25-.87 2.13T29 32H3q-1.25 0-2.12-.87T0 29V12.56q0-1.37 1.13-2.31 2.5-2.06 8.43-6.75.2-.19.75-.66l.94-.78 1.03-.75 1.2-.68 1.21-.44L16-.01l1.32.2 1.25.43 1.18.72 1 .75.94.75.75.66q6.25 4.94 8.5 6.81zM29 28.62V12.76q0-.13-.13-.25-5.06-4.13-8.3-6.69-.07 0-.91-.72t-1.29-1.03-1.15-.69T16 3t-1.22.37-1.15.66-1.29 1.03-.9.75Q8.5 8.12 3.13 12.5q-.13.06-.13.25v15.87q0 .38.38.38h25.25q.37 0 .37-.38zm-2-11.75q.5.63-.13 1.07-3.43 2.81-4.43 3.56-.2.19-.75.66l-.94.78-1.03.75-1.19.69-1.22.43-1.31.2q-.75 0-1.53-.23t-1.57-.72-1.21-.8-1.25-.98-.88-.78q-1-.75-4.44-3.56-.62-.44-.12-1.07l.94-1.12q.5-.56 1.06-.13 3.44 2.75 4.44 3.57.06 0 .9.72t1.28 1.03 1.16.69T16 22q.37 0 .78-.16t.78-.37.84-.56.82-.6.75-.62.6-.5q1-.82 4.43-3.57.56-.43 1.06.13z',
            'address-book' => [
                'p' => 'M27.25 10H26v4h1.25q.75 0 .75.75v2.5q0 .75-.75.75H26v4h1.25q.75 0 .75.75v2.5q0 .75-.75.75H26v3q0 1.25-.88 2.13T23 32H3q-1.25 0-2.13-.88T0 29V3Q0 1.75.88.87T3 0h20q1.25 0 2.13.88T26 3v3h1.25q.75 0 .75.75v2.5q0 .75-.75.75zM23 29V3H3v26h20zM13 16q-1.63 0-2.81-1.19T8.99 12t1.2-2.81T13 7.99t2.81 1.2 1.2 2.8-1.2 2.82T13 16zm-5.63 8q-.56 0-.96-.34T6 22.8v-1.19q0-1.5 1.22-2.56T10.19 18q.25 0 1.06.25t1.75.25 1.75-.25 1.06-.25q1.75 0 2.97 1.06T20 21.62v1.2q0 .5-.4.84t-.97.34H7.38z',
                'w' => .875,
                'vw' => 28
            ],
            'address-card' => [
                'p' => 'M33 2q1.25 0 2.13.88T36 5v22q0 1.25-.88 2.13T33 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h30zm0 25V5H3v22h30zM13 16q-1.63 0-2.81-1.19T8.99 12t1.2-2.81T13 7.99t2.81 1.2 1.2 2.8-1.2 2.82T13 16zm-5.63 8q-.56 0-.96-.34T6 22.8v-1.19q0-1.5 1.22-2.56T10.19 18q.25 0 1.06.25t1.75.25 1.75-.25 1.06-.25q1.75 0 2.97 1.06T20 21.62v1.2q0 .5-.4.84t-.97.34H7.38zm15.13-4q-.5 0-.5-.5v-1q0-.5.5-.5h7q.5 0 .5.5v1q0 .5-.5.5h-7zm0-4q-.5 0-.5-.5v-1q0-.5.5-.5h7q.5 0 .5.5v1q0 .5-.5.5h-7zm0-4q-.5 0-.5-.5v-1q0-.5.5-.5h7q.5 0 .5.5v1q0 .5-.5.5h-7z',
                'w' => 1.125,
                'vw' => 36
            ],
            'user-circle' => 'M15.5 6.5q2.5 0 4.25 1.75t1.75 4.25-1.75 4.25-4.25 1.75-4.25-1.75T9.5 12.5t1.75-4.25T15.5 6.5zm0 9q1.25 0 2.13-.88t.87-2.12-.88-2.13-2.12-.87-2.13.88-.87 2.12.88 2.13 2.12.87zm0-15q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm0 28q4.63 0 8.13-3-1.5-2.38-4.38-2.5-1.88.63-3.75.63-1.81 0-3.75-.63-2.81.06-4.38 2.5 3.5 3 8.13 3zm10.19-5.25Q28 20 28 16q0-5.19-3.65-8.84T15.5 3.5 6.66 7.16 3 16q0 4 2.31 7.25Q7.88 20 11.88 20q.18 0 1.34.31t2.28.32 2.28-.32 1.35-.31q4 0 6.56 3.25z',
            'id-badge' => [
                'p' => 'M21 0q1.25 0 2.13.88T24 3v26q0 1.25-.88 2.13T21 32H3q-1.25 0-2.13-.88T0 29V3Q0 1.75.88.87T3 0h18zm0 29V3H3v26h18zM9 7q-.44 0-.72-.28T8 6t.28-.72T9 5h6q.44 0 .72.28T16 6t-.28.72T15 7H9zm3 11q-1.63 0-2.81-1.19T7.99 14t1.2-2.81T12 9.99t2.81 1.2 1.2 2.8-1.2 2.82T12 18zm-5.63 8q-.56 0-.96-.34T5 24.8v-1.19q0-1.5 1.22-2.56T9.19 20q.25 0 1.06.25t1.75.25 1.75-.25 1.06-.25q1.75 0 2.97 1.06T19 23.62v1.2q0 .5-.4.84t-.97.34H6.38z',
                'w' => .75,
                'vw' => 24
            ],
            'id-card' => [
                'p' => 'M33 2q1.25 0 2.13.88T36 5v22q0 1.25-.88 2.13T33 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h30zm0 25V9H3v18h2.06Q5 26.75 5 25.62q0-1.5 1.22-2.56T9.19 22q.25 0 1.06.25t1.75.25 1.75-.25 1.06-.25q1.75 0 2.97 1.06T19 25.63q0 1.12-.06 1.37H33zm-10.5-5q-.5 0-.5-.5v-1q0-.5.5-.5h7q.5 0 .5.5v1q0 .5-.5.5h-7zm0-4q-.5 0-.5-.5v-1q0-.5.5-.5h7q.5 0 .5.5v1q0 .5-.5.5h-7zm0-4q-.5 0-.5-.5v-1q0-.5.5-.5h7q.5 0 .5.5v1q0 .5-.5.5h-7zM12 20q-1.63 0-2.81-1.19T7.99 16t1.2-2.81 2.81-1.2 2.81 1.2 1.2 2.8-1.2 2.82T12 20z',
                'w' => 1.125,
                'vw' => 36
            ],
            'window-maximize' => 'M29 2q1.25 0 2.13.88T32 5v22q0 1.25-.88 2.13T29 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h26zm0 24.63V12H3v14.63q0 .37.38.37h25.25q.37 0 .37-.38z',
            'window-minimize' => 'M30 30H2q-.81 0-1.4-.6T0 28t.6-1.4T2 26h28q.81 0 1.4.6T32 28t-.6 1.4-1.4.6z',
            'window-restore' => 'M29 0q1.25 0 2.13.88T32 3v20q0 1.25-.88 2.13T29 26h-3v3q0 1.25-.88 2.13T23 32H3q-1.25 0-2.13-.88T0 29V9q0-1.25.88-2.13T3 6h3V3q0-1.25.88-2.13T9 0h20zm-6 29V16H3v13h20zm6-6V3H9v3h14q1.25 0 2.13.88T26 9v14h3z',
            'snowflake' => [
                'p' => 'M27.5 22.19q.38.25.47.62t-.1.75l-.5.88q-.5.87-1.37.37l-2.44-1.43.57 2.12q.06.31 0 .56t-.25.44-.44.25l-.94.25q-.38.13-.75-.1t-.5-.59l-1.31-5.06-4.5-2.63v5.25l3.68 3.7q.7.74 0 1.43l-.75.69q-.68.75-1.37 0l-1.56-1.56V31q0 .44-.28.72t-.66.28h-1q-.44 0-.72-.28T12.5 31v-2.87l-1.56 1.62q-.7.69-1.38 0L8.87 29q-.25-.25-.25-.69t.25-.68l3.63-3.75v-5.25L8 21.25l-1.32 5.06q-.12.38-.46.6t-.72.1l-.94-.26q-.44-.12-.63-.47t-.06-.72l.56-2.19L2 24.82q-.38.2-.79.1t-.59-.47l-.5-.88q-.5-.87.38-1.37l2.43-1.44-2.12-.56q-.38-.13-.6-.47t-.09-.78L.37 18q.13-.44.47-.63t.72-.12l5 1.37 4.5-2.62-4.5-2.63-5 1.38q-.38.06-.72-.13T.37 14l-.25-.94q-.12-.44.1-.78t.59-.47l2.12-.56L.5 9.8q-.88-.5-.38-1.38l.5-.87q.5-.87 1.38-.37l2.43 1.43-.56-2.12q-.12-.44.06-.78t.63-.47L5.5 5q.37-.13.75.1t.44.59l1.3 5.06 4.5 2.62V8.12L8.88 4.44q-.68-.75 0-1.44l.75-.69q.7-.75 1.38 0l1.56 1.56V1q0-.44.28-.72T13.5 0h1q.44 0 .72.28t.28.72v2.87l1.56-1.62q.69-.69 1.38 0l.68.75q.7.69 0 1.37L15.5 8.12v5.25l4.5-2.62 1.31-5.06q.13-.38.47-.6t.72-.1l.94.26q.44.12.62.47t.07.72l-.57 2.18 2.5-1.43q.32-.2.72-.1t.6.47l.5.88q.5.87-.38 1.37l-2.44 1.44 2.13.56q.37.13.6.47t.09.78l-.25.94q-.13.44-.47.62t-.72.13l-5-1.38-4.5 2.63 4.5 2.62 5-1.37q.37-.06.72.12t.47.63l.25.94q.12.43-.1.78t-.6.47l-2.12.56z',
                'w' => .875,
                'vw' => 28
            ],
            'trash-alt' => [
                'p' => 'M16.75 26q-.75 0-.75-.75v-13.5q0-.75.75-.75h1.5q.75 0 .75.75v13.5q0 .75-.75.75h-1.5zM27 5q.44 0 .72.28T28 6v1q0 .44-.28.72T27 8h-1v21q0 1.25-.88 2.13T23 32H5q-1.25 0-2.13-.88T2 29V8H1q-.44 0-.72-.28T0 7V6q0-.44.28-.72T1 5h5.13l2.12-3.56q.38-.57 1.13-1t1.5-.44h6.24q.5 0 1 .19t.94.53.7.72L21.87 5H27zM10.75 3.19 9.62 5h8.76l-1.13-1.8q-.13-.2-.31-.2h-5.88q-.19 0-.31.19zM23 29V8H5v21h18zM9.75 26Q9 26 9 25.25v-13.5q0-.75.75-.75h1.5q.75 0 .75.75v13.5q0 .75-.75.75h-1.5z',
                'w' => .875,
                'vw' => 28
            ],
            'images' => [
                'p' => 'M30 26v1q0 1.25-.88 2.13T27 30H3q-1.25 0-2.13-.88T0 27V11q0-1.25.88-2.13T3 8h1v3h-.63Q3 11 3 11.38v15.24q0 .38.38.38h23.25q.37 0 .37-.38V26h3zm2.63-21H9.36Q9 5 9 5.38v15.25q0 .37.38.37h23.24q.38 0 .38-.38V5.38Q33 5 32.62 5zM33 2q1.25 0 2.13.88T36 5v16q0 1.25-.88 2.13T33 24H9q-1.25 0-2.13-.88T6 21V5q0-1.25.88-2.13T9 2h24zM16.5 9q0 1.06-.72 1.78T14 11.5t-1.78-.72T11.5 9t.72-1.78T14 6.5t1.78.72T16.5 9zM12 15l2.5-2.5q.5-.5 1 0L18 15l6.5-6.5q.5-.5 1 0L30 13v5H12v-3z',
                'w' => 1.125,
                'vw' => 36
            ],
            'clipboard' => [
                'p' => 'M21 4q1.25 0 2.13.88T24 7v22q0 1.25-.88 2.13T21 32H3q-1.25 0-2.13-.88T0 29V7q0-1.25.88-2.13T3 4h5q0-1.63 1.19-2.81T12-.01t2.81 1.2T16.01 4h5zm-9-1.5q-.63 0-1.06.44T10.5 4t.44 1.06T12 5.5t1.06-.44T13.5 4t-.44-1.06T12 2.5zm9 26.13V7.38Q21 7 20.62 7H18v2.25q0 .75-.75.75H6.75Q6 10 6 9.25V7H3.37Q3 7 3 7.38v21.25q0 .37.38.37h17.25q.37 0 .37-.38z',
                'w' => .75,
                'vw' => 24
            ],
            'arrow-alt-circle-down' => 'M16 .5q6.44 0 10.97 4.53T31.5 16t-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5zm0 28q5.19 0 8.84-3.66T28.5 16t-3.66-8.84T16 3.5 7.16 7.16 3.5 16t3.66 8.84T16 28.5zM14 8.75q0-.75.75-.75h2.5q.75 0 .75.75V16h4.19q.5 0 .69.47t-.13.78L16.5 23.5q-.5.5-1 0l-6.25-6.25q-.31-.31-.12-.78T9.8 16H14V8.75z',
            'arrow-alt-circle-left' => 'M.5 16q0-6.44 4.53-10.97T16 .5t10.97 4.53T31.5 16t-4.53 10.97T16 31.5 5.03 26.97.5 16zm28 0q0-5.19-3.66-8.84T16 3.5 7.16 7.16 3.5 16t3.66 8.84T16 28.5t8.84-3.66T28.5 16zM24 14.75v2.5q0 .75-.75.75H16v4.19q0 .5-.47.69t-.78-.13L8.5 16.5q-.5-.5 0-1l6.25-6.25q.31-.31.78-.12t.47.68v4.2h7.25q.75 0 .75.74z',
            'arrow-alt-circle-right' => 'M31.5 16q0 6.44-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5t10.97 4.53T31.5 16zm-28 0q0 5.19 3.66 8.84T16 28.5t8.84-3.66T28.5 16t-3.66-8.84T16 3.5 7.16 7.16 3.5 16zM8 17.25v-2.5q0-.75.75-.75H16V9.81q0-.5.47-.69t.78.13l6.25 6.25q.5.5 0 1l-6.25 6.25q-.31.31-.78.12T16 22.2V18H8.75Q8 18 8 17.26z',
            'arrow-alt-circle-up' => 'M16 31.5q-6.44 0-10.97-4.53T.5 16 5.03 5.03 16 .5t10.97 4.53T31.5 16t-4.53 10.97T16 31.5zm0-28q-5.19 0-8.84 3.66T3.5 16t3.66 8.84T16 28.5t8.84-3.66T28.5 16t-3.66-8.84T16 3.5zM17.25 24h-2.5q-.75 0-.75-.75V16H9.81q-.5 0-.69-.47t.13-.78L15.5 8.5q.5-.5 1 0l6.25 6.25q.31.31.12.78t-.68.47h-4.2v7.25q0 .75-.74.75z',
            'gem' => [
                'p' => 'M29 0q.38 0 .63.31l6.25 9.25q.3.44-.07.88L18.56 31.75q-.19.25-.56.25t-.56-.25L.19 10.44Q-.2 10 .12 9.56L6.37.31Q6.57 0 7 0h22zm-1.19 3h-3.56l3.25 6h4.25zM15.2 3l-3.25 6h12.12l-3.25-6H15.2zm-7 0L4.25 9H8.5l3.25-6H8.19zm-2.7 9L13 22 8.75 12H5.5zm6.44 0L18 27.19 24.06 12H11.94zM23 22l7.5-10h-3.25z',
                'w' => 1.125,
                'vw' => 36
            ],
            'diamond' => 'gem',
            'money-bill-alt' => [
                'p' => 'M20 9q2.5 0 4.25 2.06T26 16t-1.75 4.94T20 23t-4.25-2.06T14 16t1.75-4.94T20 9zm2.5 10.5v-1q0-.5-.5-.5h-1v-5.5q0-.5-.5-.5h-.88q-.43 0-.8.25l-.95.63q-.25.18-.25.43 0 .13.13.25l.5.88q.19.19.44.19.18 0 .31-.07V18h-1q-.5 0-.5.5v1q0 .5.5.5h4q.5 0 .5-.5zM38 4q.81 0 1.4.6T40 6v20q0 .81-.6 1.4T38 28H2q-.81 0-1.4-.6T0 26V6q0-.81.6-1.4T2 4h36zm-1 17V11q-1.69 0-2.84-1.16T33 7H7q0 1.69-1.16 2.84T3 11v10q1.69 0 2.84 1.16T7 25h26q0-1.69 1.16-2.84T37 21z',
                'w' => 1.25,
                'vw' => 40
            ],
            'grin-tears' => [
                'p' => 'M7.31 16q.32-.06.53.16t.16.53Q7.25 21.8 6.06 23q-1 1-2.47 1t-2.53-1.06T0 20.4t1-2.47Q2.19 16.74 7.3 16zM39 17.94q1 1 1 2.47t-1.06 2.53T36.4 24t-2.47-1q-1.2-1.19-1.94-6.31-.06-.32.15-.53t.54-.16q5.12.75 6.3 1.94zm-7.94 3.81q.7 2 1.82 2.94-2.2 3.12-5.57 4.97T20 31.5t-7.31-1.84-5.57-4.97q1.13-.94 1.82-2.94 1.56 3.06 4.53 4.9T20 28.5t6.53-1.84 4.53-4.91zM7.7 14q-.69-.06-3.13.44.38-3.88 2.53-7.06t5.57-5.04T20 .5t7.35 1.84 5.56 5.04 2.53 7.12q-2.38-.56-3.13-.5-.68-4.5-4.18-7.5T20 3.5t-8.12 3-4.2 7.5zM20 25q-2.56 0-4.97-1.44T12.31 20q-.12-.5.25-.81t.81-.13q2.57.81 6.63.81t6.63-.8q.43-.2.8.15t.26.78q-.32 2.12-2.72 3.56T20 25zm8.16-10.53q-.41.1-.6-.22L27 13.19q-.81-1.38-2-1.38t-2 1.38l-.56 1.06q-.2.38-.57.25-.43-.13-.37-.56.12-1.88 1.22-3.16T25 9.5t2.28 1.28 1.22 3.16q.06.44-.35.53zM15 11.8q-1.19 0-2 1.38l-.56 1.06q-.2.38-.57.25-.43-.12-.37-.56.06-.75.31-1.44t.6-1.25.78-.94.9-.6.9-.21q1.2 0 2.29 1.28t1.22 3.16q.06.43-.34.56t-.6-.25L17 13.19q-.81-1.38-2-1.38z',
                'w' => 1.25,
                'vw' => 40
            ],
            'window-close' => 'M29 2q1.25 0 2.13.88T32 5v22q0 1.25-.88 2.13T29 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h26zm0 24.63V5.38Q29 5 28.62 5H3.38Q3 5 3 5.38v21.25q0 .37.38.37h25.25q.37 0 .37-.38zm-6.75-14.44L18.44 16l3.81 3.81q.56.57 0 1.07l-1.38 1.37q-.5.56-1.06 0L16 18.44l-3.81 3.81q-.57.56-1.07 0l-1.37-1.37q-.57-.5 0-1.07l3.81-3.8-3.81-3.82q-.57-.56 0-1.06l1.37-1.38q.5-.56 1.07 0l3.8 3.81 3.82-3.81q.57-.56 1.07 0l1.37 1.38q.56.5 0 1.06z',
            'comment-dots' => 'M9 13q.81 0 1.4.6T11 15t-.6 1.4T9 17t-1.4-.6T7 15t.6-1.4T9 13zm7 0q.81 0 1.4.6T18 15t-.6 1.4-1.4.6-1.4-.6T14 15t.6-1.4T16 13zm7 0q.81 0 1.4.6T25 15t-.6 1.4-1.4.6-1.4-.6T21 15t.6-1.4T23 13zM16 2q6.63 0 11.31 3.81t4.7 9.2-4.7 9.18T16 28q-2.94 0-5.81-.87Q6.12 30 1.49 30q-1 0-1.37-.9t.25-1.6q.13-.12.6-.65t1.19-1.7 1.15-2.27Q0 19.38 0 15q0-5.37 4.69-9.19T16 2.01zm0 23q5.38 0 9.19-2.94T29 15t-3.8-7.06T16 5 6.81 7.94 3 15q0 3.19 2.5 5.81l1.25 1.38-.63 1.75q-.5 1.25-1.25 2.56 1.88-.63 3.57-1.88l1.25-.8 1.44.43Q13.5 25 16 25z',
            'smile-wink' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm0 28q5.19 0 8.84-3.66T28 16t-3.66-8.84T15.5 3.5 6.66 7.16 3 16t3.66 8.84 8.84 3.66zm7.34-9.16q.47.41.54 1.04t-.32 1.06Q20.12 25 15.5 25t-7.56-3.56q-.38-.44-.32-1.07t.53-1.03 1.1-.34 1 .56Q12.3 22 15.5 22t5.25-2.44q.37-.5 1-.56t1.1.34zM10.5 15q-.81 0-1.4-.6T8.5 13t.6-1.4 1.4-.6 1.4.6.6 1.4-.6 1.4-1.4.6zm10-3.75q1.25 0 2.4.75t1.35 1.88q.06.5-.4.74t-.85-.06l-.63-.56q-.68-.56-1.87-.56t-1.94.56l-.56.56q-.38.31-.84.06t-.41-.74q.12-.76.72-1.38t1.4-.94 1.63-.31z',
            'angry' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm0 28q5.19 0 8.84-3.66T28 16t-3.66-8.84T15.5 3.5 6.66 7.16 3 16t3.66 8.84 8.84 3.66zm0-9q3.31 0 5.44 2.56.37.44.31 1.07t-.5 1q-.5.43-1.12.37t-1-.56q-1.2-1.38-3.13-1.38t-3.12 1.38q-.38.5-1 .56t-1.1-.34-.53-1.03.31-1.07q2.13-2.56 5.44-2.56zm-3-4.5q0 .81-.6 1.4t-1.4.6-1.4-.6-.6-1.4q0-.5.31-1l-1.75-.56q-.56-.2-.87-.72t-.13-1.13.72-.9 1.16-.13l5 1.5q.56.19.87.72t.13 1.16Q13.63 15 12.5 15zm12.44-3.4q.19.59-.13 1.12t-.87.72l-1.75.5q.31.56.31 1.06 0 .81-.6 1.4t-1.4.6-1.4-.6-.6-1.4q-1.12 0-1.44-1.06-.12-.32-.06-.6t.19-.53.34-.47.53-.28l5-1.5q.63-.19 1.16.13t.72.9z',
            'dizzy' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm0 28q5.19 0 8.84-3.66T28 16t-3.66-8.84T15.5 3.5 6.66 7.16 3 16t3.66 8.84 8.84 3.66zm-2.13-13.63q-.37.38-.87.38t-.88-.38l-1.12-1.12-1.13 1.13q-.37.37-.87.37t-.88-.38-.37-.87.38-.88L8.75 12l-1.13-1.13q-.37-.37-.37-.87t.38-.88.87-.37.88.38l1.12 1.12 1.13-1.13q.37-.37.87-.37t.88.38.37.87-.38.88L12.25 12l1.13 1.13q.37.37.37.87t-.38.88zm10-5.74q.38.37.38.87t-.38.88L22.25 12l1.13 1.13q.37.37.37.87t-.38.88-.87.37-.88-.38l-1.12-1.12-1.13 1.13q-.37.37-.87.37t-.88-.38-.37-.87.38-.88L18.75 12l-1.13-1.13q-.37-.37-.37-.87t.38-.88.87-.37.88.38l1.12 1.12 1.13-1.13q.37-.37.87-.37t.88.38zM15.5 17q1.63 0 2.81 1.19t1.2 2.81-1.2 2.81-2.81 1.2-2.81-1.2-1.2-2.8 1.2-2.82T15.5 17z',
            'flushed' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm0 28q5.19 0 8.84-3.66T28 16t-3.66-8.84T15.5 3.5 6.66 7.16 3 16t3.66 8.84 8.84 3.66zm6-19.5q2.06 0 3.53 1.47T26.5 14t-1.47 3.53T21.5 19t-3.53-1.47T16.5 14t1.47-3.53T21.5 9zm0 8q1.25 0 2.13-.88T24.5 14t-.88-2.13T21.5 11t-2.13.88T18.5 14t.88 2.13 2.12.87zm0-4.5q.63 0 1.06.44T23 14t-.44 1.06-1.06.44-1.06-.44T20 14t.44-1.06 1.06-.44zm-7 1.5q0 2.06-1.47 3.53T9.5 19t-3.53-1.47T4.5 14t1.47-3.53T9.5 9t3.53 1.47T14.5 14zm-5 3q1.25 0 2.13-.88T12.5 14t-.88-2.13T9.5 11t-2.13.88T6.5 14t.88 2.13T9.5 17zm0-4.5q.63 0 1.06.44T11 14t-.44 1.06-1.06.44-1.06-.44T8 14t.44-1.06 1.06-.44zm10 9q.63 0 1.06.44T21 23t-.44 1.06-1.06.44h-8q-.63 0-1.06-.44T10 23t.44-1.06 1.06-.44h8z',
            'frown-open' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm0 28q5.19 0 8.84-3.66T28 16t-3.66-8.84T15.5 3.5 6.66 7.16 3 16t3.66 8.84 8.84 3.66zm-3-15.5q0 .81-.6 1.4t-1.4.6-1.4-.6-.6-1.4.6-1.4 1.4-.6 1.4.6.6 1.4zm8-2q.81 0 1.4.6t.6 1.4-.6 1.4-1.4.6-1.4-.6-.6-1.4.6-1.4 1.4-.6zm-5 7q1.88 0 3.78 1.06t2.22 2.75q.06.57-.34.94t-.97.19q-3.07-.94-4.7-.94t-4.68.94q-.56.19-.97-.19t-.34-.94q.31-1.68 2.22-2.75T15.5 18z',
            'grimace' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm0 28q5.19 0 8.84-3.66T28 16t-3.66-8.84T15.5 3.5 6.66 7.16 3 16t3.66 8.84 8.84 3.66zm-5-13.5q-.81 0-1.4-.6T8.5 13t.6-1.4 1.4-.6 1.4.6.6 1.4-.6 1.4-1.4.6zm10 0q-.81 0-1.4-.6t-.6-1.4.6-1.4 1.4-.6 1.4.6.6 1.4-.6 1.4-1.4.6zm1 1q1.25 0 2.13.88T24.5 19v2q0 1.25-.88 2.13T21.5 24h-12q-1.25 0-2.13-.88T6.5 21v-2q0-1.25.88-2.13T9.5 16h12zM11 22v-1.5H8.5v.5q0 .44.28.72t.72.28H11zm0-2.5V18H9.5q-.44 0-.72.28T8.5 19v.5H11zm4 2.5v-1.5h-3V22h3zm0-2.5V18h-3v1.5h3zm4 2.5v-1.5h-3V22h3zm0-2.5V18h-3v1.5h3zm3.5 1.5v-.5H20V22h1.5q.44 0 .72-.28t.28-.72zm0-1.5V19q0-.44-.28-.72T21.5 18H20v1.5h2.5z',
            'grin' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm0 28q5.19 0 8.84-3.66T28 16t-3.66-8.84T15.5 3.5 6.66 7.16 3 16t3.66 8.84 8.84 3.66zm6.63-9.44q.43-.18.8.13t.26.81q-.32 2.13-2.72 3.56T15.5 25q-1.63 0-3.31-.62t-2.94-1.82T7.8 20q-.12-.5.25-.81t.81-.13q2.57.82 6.63.82t6.62-.82zM10.5 15q-.81 0-1.4-.6T8.5 13t.6-1.4 1.4-.6 1.4.6.6 1.4-.6 1.4-1.4.6zm10 0q-.81 0-1.4-.6t-.6-1.4.6-1.4 1.4-.6 1.4.6.6 1.4-.6 1.4-1.4.6z',
            'grin-alt' => 'M12.5 15.5q-.31.56-.97.56t-1.03-.56q-.94-1.38-1-3.5.06-2.13 1-3.5.31-.56.97-.56t1.03.56q.94 1.38 1 3.5-.06 2.13-1 3.5zm8 0q-.31.56-.97.56t-1.03-.56q-.94-1.38-1-3.5.06-2.13 1-3.5.31-.56.97-.56t1.03.56q.94 1.38 1 3.5-.06 2.13-1 3.5zm-5-15q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm0 28q5.19 0 8.84-3.66T28 16t-3.66-8.84T15.5 3.5 6.66 7.16 3 16t3.66 8.84 8.84 3.66zm6.63-9.44q.43-.18.8.13t.26.81q-.32 2.13-2.72 3.56T15.5 25t-4.97-1.44T7.81 20q-.12-.5.25-.81t.81-.13q2.57.82 6.63.82t6.63-.82z',
            'grin-beam' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm0 28q5.19 0 8.84-3.66T28 16t-3.66-8.84T15.5 3.5 6.66 7.16 3 16t3.66 8.84 8.84 3.66zm6.63-9.44q.43-.18.8.13t.26.81q-.32 2.13-2.72 3.56T15.5 25q-1.63 0-3.31-.62t-2.94-1.82T7.8 20q-.12-.5.25-.81t.81-.13q2.57.82 6.63.82t6.62-.82zM7.38 14.5q-.44-.13-.38-.56.06-.75.31-1.44t.6-1.25.78-.94.9-.6.91-.21q1.19 0 2.28 1.28T14 13.94q.06.43-.34.56t-.6-.25l-.56-1.06q-.81-1.38-2-1.38t-2 1.38l-.56 1.06q-.2.37-.57.25zm10 0q-.44-.13-.38-.56.13-1.88 1.22-3.16T20.5 9.5t2.28 1.28T24 13.94q.06.43-.34.56t-.6-.25l-.56-1.06q-.81-1.38-2-1.38t-2 1.38l-.56 1.06q-.2.38-.57.25z',
            'grin-beam-sweat' => 'M27.5 10q-1.38 0-2.34-1.06t-.97-2.63q0-1.75 2.87-6.06.44-.69.88 0Q30.8 4.56 30.8 6.3q0 1.56-.97 2.63T27.5 10zm-12 15q-1.63 0-3.31-.63t-2.9-1.78T7.8 20q-.12-.5.25-.81t.82-.13q2.56.82 6.62.82t6.63-.82q.43-.19.8.16t.26.78q-.32 2.12-2.72 3.56T15.5 25zm8.16-10.53q-.41.1-.6-.22l-.56-1.06q-.81-1.38-2-1.38t-2 1.38l-.56 1.06q-.2.38-.57.25-.43-.13-.37-.56.12-1.88 1.22-3.16T20.5 9.5t2.28 1.28T24 13.94q.06.44-.35.53zm6.59-3.28Q31 13.5 31 16q0 6.44-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5q4.19 0 7.81 2.13-.87 1.75-1.06 2.87-3.13-2-6.75-2-5.19 0-8.84 3.66T3 16t3.66 8.85 8.84 3.65 8.84-3.66T28 16q0-2.06-.69-4 1.63 0 2.94-.81zm-19.75.62q-1.19 0-2 1.38l-.56 1.06q-.2.38-.57.25-.43-.12-.37-.56.06-.75.31-1.44t.6-1.25.78-.94.9-.6.9-.21q1.2 0 2.29 1.28T14 13.94q.06.43-.34.56t-.6-.25l-.56-1.06q-.81-1.38-2-1.38z',
            'grin-hearts' => 'M22.13 19.06q.43-.18.8.13t.26.81q-.32 2.13-2.72 3.56T15.5 25t-4.97-1.44T7.81 20q-.12-.5.25-.81t.81-.13q2.57.82 6.63.82t6.63-.82zM12.55 16 8.2 14.87q-1-.24-1.44-1.15t-.12-1.85q.3-.8 1.12-1.18t1.69-.2l.44.13.12-.43q.25-.88 1-1.38t1.63-.37q1 .18 1.53 1t.28 1.8l-1.25 4.38q-.13.5-.63.38zm11.81-4.13q.32.94-.12 1.85t-1.44 1.15L18.44 16q-.5.13-.63-.38l-1.25-4.37q-.25-1 .28-1.81t1.53-1q.57-.07 1.13.12t.94.63.56 1l.12.43.44-.12q.38-.06.82-.03t.8.22.7.5.5.69zM15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm0 28q5.19 0 8.84-3.66T28 16t-3.66-8.84T15.5 3.5 6.66 7.16 3 16t3.66 8.84 8.84 3.66z',
            'grin-squint' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm0 28q5.19 0 8.84-3.66T28 16t-3.66-8.84T15.5 3.5 6.66 7.16 3 16t3.66 8.84 8.84 3.66zm6.63-9.44q.43-.18.8.13t.26.81q-.32 2.13-2.72 3.56T15.5 25q-1.63 0-3.31-.62t-2.94-1.82T7.8 20q-.12-.5.25-.81t.81-.13q2.57.82 6.63.82t6.62-.82zM7.4 16.46q-.4-.46.04-.96L9.5 13l-2.06-2.5q-.44-.5 0-.97t.93-.15l5 3q.38.18.38.62t-.38.63l-5 3q-.56.3-.96-.16zm15.21.17-5-3q-.37-.2-.37-.63t.38-.63l5-3q.56-.3.93.13.44.5 0 1L21.5 13l2.06 2.5q.44.5.03.97t-.97.16z',
            'grin-squint-tears' => 'M7.31 24q.32-.06.53.16t.16.53Q7.25 29.8 6.06 31q-1 1-2.47 1t-2.53-1.06T0 28.4t1-2.47Q2.19 24.74 7.3 24zm-2.56-2.63q-1.69.38-2.94.82-1.94-4.25-1.06-9t4.31-8.13Q9.56.5 16 .5q3.75 0 7.06 1.69-.43 1.31-.75 3.06Q19.37 3.5 16 3.5q-5.19 0-8.81 3.69Q4.3 10 3.69 13.97t1.06 7.4zm22-11.68q1.81-.38 3.06-.75 2.25 4.31 1.5 9.37T27 26.94q-4.56 4.56-11 4.56-3.19 0-6.19-1.31.44-1.19.82-2.94Q13.19 28.5 16 28.5q5.19 0 8.81-3.69 3.07-3 3.57-7.28T26.75 9.7zM24.69 8q-.32.06-.53-.16T24 7.31Q24.75 2.2 25.94 1q1-1 2.47-1t2.53 1.06T32 3.6t-1 2.47Q29.81 7.26 24.7 8zM13 14.75q.31.31.19.75l-1.44 5.63q-.13.56-.69.56H11q-.63-.07-.69-.69L10 17.75l-3.25-.31q-.63-.07-.69-.7t.56-.74l5.63-1.44q.44-.12.75.19zm2.5-1.56q-.56 0-.75-.19-.31-.31-.19-.75L16 6.62q.13-.56.75-.56.63.07.69.69l.31 3.25 3.25.31q.63.07.69.7t-.56.74zm3.25 5.56q2.88-2.94 4.06-5.25.25-.44.72-.47t.72.35q1.19 1.93.56 4.56t-2.43 4.44-4.44 2.46-4.56-.59q-.38-.25-.35-.72t.47-.72q2.38-1.25 5.25-4.06z',
            'grin-stars' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm0 28q5.19 0 8.84-3.66T28 16t-3.66-8.84T15.5 3.5 6.66 7.16 3 16t3.66 8.84 8.84 3.66zm6.63-9.44q.43-.18.8.13t.26.81q-.32 2.13-2.72 3.56T15.5 25t-4.97-1.44T7.81 20q-.12-.5.25-.81t.81-.13q2.57.82 6.63.82t6.63-.82zM7.88 15.44l.37-2.19-1.63-1.5q-.18-.25-.09-.53t.4-.34l2.2-.32.93-1.94q.13-.3.44-.3t.44.3l.94 1.94 2.19.32q.3.06.4.34t-.1.53l-1.62 1.5.38 2.19q.06.31-.2.47t-.5.03l-1.93-1-1.94 1q-.25.12-.5-.03t-.18-.47zm16.18-4.56q.32.06.4.34t-.09.53l-1.62 1.5.37 2.19q.07.19-.03.34t-.28.19-.37-.03l-1.94-1-1.94 1q-.25.12-.5-.03t-.19-.47l.38-2.19-1.63-1.5q-.18-.25-.09-.53t.4-.34l2.2-.32.93-1.93q.13-.32.44-.32t.44.32l.94 1.93z',
            'grin-tongue' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm4 25v-2.75l-1.13-.56q-.68-.32-1.34.03t-.78 1.03l-.19.75q-.12.44-.56.44t-.56-.44l-.13-.75q-.19-.69-.84-1.03t-1.35-.03l-1.12.56v2.69q0 1.62 1.15 2.84t2.79 1.22q1.12 0 2.06-.53t1.47-1.44.53-2.03zm1.75 1.56q3.06-1.56 4.9-4.53T28 16q0-5.19-3.66-8.84T15.5 3.5 6.66 7.16 3 16q0 3.56 1.84 6.53t4.91 4.53q-.25-.87-.25-1.56v-2.75Q8.06 21.5 7.81 20q-.12-.5.25-.81t.82-.13q2.56.82 6.62.82t6.63-.82q.43-.19.8.16t.26.78q-.25 1.5-1.69 2.75v2.75q0 .69-.25 1.56zM10.5 11q.81 0 1.4.6t.6 1.4-.6 1.4-1.4.6-1.4-.6-.6-1.4.6-1.4 1.4-.6zm10 0q.81 0 1.4.6t.6 1.4-.6 1.4-1.4.6-1.4-.6-.6-1.4.6-1.4 1.4-.6z',
            'grin-tongue-squint' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm4 25v-2.75l-1.13-.56q-.68-.32-1.34.03t-.78 1.03l-.19.75q-.12.44-.56.44t-.56-.44l-.13-.75q-.19-.69-.84-1.03t-1.35-.03l-1.12.56v2.69q0 1.62 1.15 2.84t2.79 1.22q1.12 0 2.06-.53t1.47-1.44.53-2.03zm1.75 1.56q3.06-1.56 4.9-4.53T28 16q0-5.19-3.66-8.84T15.5 3.5 6.66 7.16 3 16q0 3.56 1.84 6.53t4.91 4.53q-.25-.87-.25-1.56v-2.75Q8.06 21.5 7.81 20q-.12-.5.25-.81t.82-.13q2.56.82 6.62.82t6.63-.82q.43-.19.8.16t.26.78q-.25 1.5-1.69 2.75v2.75q0 .69-.25 1.56zm2.31-17.53q.44.47 0 .97L21.5 13l2.06 2.5q.44.5.03.97t-.97.16l-5-3q-.37-.2-.37-.63t.37-.63l5-3q.5-.3.94.16zm-10.19 2.84q.38.2.38.63t-.38.62l-5 3q-.56.32-.96-.15t.03-.97L9.5 13l-2.06-2.5q-.44-.5 0-.97t.93-.15z',
            'grin-tongue-wink' => 'M9.5 11.25q.81 0 1.63.31t1.4.94.72 1.38q.06.5-.4.75t-.85-.07l-.63-.56q-.68-.56-1.87-.56T7.62 14l-.62.56q-.38.38-.84.1t-.41-.78q.19-1.13 1.34-1.88t2.41-.75zM20.5 8q2.06 0 3.53 1.47T25.5 13t-1.47 3.53T20.5 18t-3.53-1.47T15.5 13t1.47-3.53T20.5 8zm0 8q1.25 0 2.13-.88T23.5 13t-.88-2.13T20.5 10t-2.13.88T17.5 13t.88 2.13 2.12.87zm0-4.5q.63 0 1.06.44T22 13t-.44 1.06-1.06.44-1.06-.44T19 13t.44-1.06 1.06-.44zm-5-11q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm4 25v-2.75l-1.13-.56q-.68-.32-1.34.03t-.78 1.03l-.19.75q-.12.44-.56.44t-.56-.44l-.13-.75q-.19-.69-.84-1.03t-1.35-.03l-1.12.56v2.69q0 1.62 1.15 2.84t2.79 1.22q1.12 0 2.06-.53t1.47-1.44.53-2.03zm1.75 1.56q3.06-1.56 4.9-4.53T28 16q0-5.19-3.66-8.84T15.5 3.5 6.66 7.16 3 16q0 3.56 1.84 6.53t4.91 4.53q-.25-.87-.25-1.56v-2.75Q8.06 21.5 7.81 20q-.12-.5.25-.81t.82-.13q2.56.82 6.62.82t6.63-.82q.43-.19.8.16t.26.78q-.25 1.5-1.69 2.75v2.75q0 .69-.25 1.56z',
            'grin-wink' => 'M20.5 11.25q1.25 0 2.4.75t1.35 1.88q.06.5-.4.78t-.85-.1l-.63-.56q-.68-.56-1.87-.56t-1.88.56l-.62.56q-.38.38-.84.13t-.41-.82q.19-1.12 1.34-1.87t2.41-.75zM10.5 15q-.81 0-1.4-.6T8.5 13t.6-1.4 1.4-.6 1.4.6.6 1.4-.6 1.4-1.4.6zm11.63 4.06q.43-.18.8.13t.26.81q-.32 2.13-2.72 3.56T15.5 25t-4.97-1.44T7.81 20q-.12-.5.25-.81t.81-.13q2.57.82 6.63.82t6.63-.82zM15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm0 28q5.19 0 8.84-3.66T28 16t-3.66-8.84T15.5 3.5 6.66 7.16 3 16t3.66 8.84 8.84 3.66z',
            'kiss' => 'M10.5 11q.81 0 1.4.6t.6 1.4-.6 1.4-1.4.6-1.4-.6-.6-1.4.6-1.4 1.4-.6zm8.5 8.25q0 1.38-2.19 2.25 2.19.88 2.19 2.25 0 1-1.28 1.84t-3.16.91h-.06q-.38 0-.5-.38-.06-.37.31-.56l1.07-.44q1.3-.56 1.3-1.34t-1.3-1.34L14.3 22q-.25-.13-.25-.47t.25-.47l1.07-.44q1.3-.56 1.3-1.37t-1.3-1.32l-1.07-.5q-.37-.12-.31-.56.12-.44.5-.37 1 .06 1.84.34t1.41.66.9.84.35.9zM15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm0 28q5.19 0 8.84-3.66T28 16t-3.66-8.84T15.5 3.5 6.66 7.16 3 16t3.66 8.84 8.84 3.66zm5-17.5q.81 0 1.4.6t.6 1.4-.6 1.4-1.4.6-1.4-.6-.6-1.4.6-1.4 1.4-.6z',
            'kiss-beam' => 'M10.5 9.5q1.19 0 2.28 1.28T14 13.94q0 .43-.38.56t-.56-.25l-.62-1.06q-.75-1.38-1.94-1.38t-2 1.38l-.56 1.06q-.2.31-.6.22T7 13.94q.06-.75.31-1.44t.6-1.25.78-.94.9-.6.9-.21zm5-9q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm0 28q5.19 0 8.84-3.66T28 16t-3.66-8.84T15.5 3.5 6.66 7.16 3 16t3.66 8.84 8.84 3.66zm3.5-9.25q0 1.38-2.19 2.25 2.19.88 2.19 2.25 0 1-1.28 1.84t-3.16.91h-.06q-.38 0-.5-.38-.06-.37.31-.56l1.07-.44q1.3-.56 1.3-1.34t-1.3-1.34L14.3 22q-.25-.13-.25-.47t.25-.47l1.07-.44q1.3-.56 1.3-1.37t-1.3-1.32l-1.07-.5q-.37-.12-.31-.56.12-.44.5-.37 1 .06 1.84.34t1.41.66.9.84.35.9zm1.5-9.75q1.19 0 2.28 1.28T24 13.94q0 .43-.38.56t-.56-.25l-.62-1.06q-.75-1.38-1.94-1.38t-2 1.38l-.56 1.06q-.2.31-.6.22t-.34-.53q.12-1.88 1.22-3.16T20.5 9.5z',
            'kiss-wink-heart' => 'M19 19.25q0 1.38-2.19 2.31 2.19.88 2.19 2.25 0 .94-1.28 1.78t-3.16.97h-.06q-.38 0-.5-.37-.06-.44.31-.57l1.07-.5q1.3-.56 1.3-1.34t-1.3-1.34L14.3 22q-.25-.13-.25-.47t.25-.47l1.07-.44q1.3-.56 1.3-1.34t-1.3-1.35l-1.07-.43q-.37-.2-.31-.57.12-.43.5-.43 1 .06 1.84.34t1.41.66.9.84.35.9zm4.38-5.19q-.7-.62-1.88-.62t-1.88.62l-.62.57q-.38.3-.84.06t-.41-.75q.13-.56.5-1.06t.88-.85 1.12-.53 1.25-.19q1.25 0 2.4.75t1.35 1.88q.06.5-.4.75t-.85-.06zM8.5 13.03q0-.84.6-1.44t1.4-.6 1.4.6.6 1.44-.6 1.4-1.4.57-1.4-.56-.6-1.4zm22.81 12.1q.44 1.12-.1 2.15t-1.65 1.34L24.38 30q-.63.13-.82-.44l-1.44-5.19q-.18-.75.04-1.46t.78-1.22 1.3-.63q1.07-.12 1.95.47t1.12 1.6l.19.56.5-.13q1-.31 1.97.16t1.34 1.4zm-10.43 2.12.8 2.94q-2.93 1.31-6.18 1.31-6.44 0-10.97-4.53T0 16 4.53 5.03 15.5.5t10.97 4.53T31 16q0 2.88-1.06 5.63-.7-.2-1.2-.2-.43-.8-1.37-1.5Q28 18 28 16q0-5.19-3.66-8.84T15.5 3.5 6.65 7.16 3 16t3.66 8.84 8.84 3.66q2.81 0 5.37-1.25z',
            'laugh' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm8.84 24.34Q28 21.2 28 16t-3.66-8.84T15.5 3.5 6.66 7.16 3 16t3.66 8.84 8.84 3.66 8.84-3.66zM20.5 14q-.81 0-1.4-.6t-.6-1.4.6-1.4 1.4-.6 1.4.6.6 1.4-.6 1.4-1.4.6zm-10 0q-.81 0-1.4-.6T8.5 12t.6-1.4 1.4-.6 1.4.6.6 1.4-.6 1.4-1.4.6zm12.13 4q.43 0 .68.28t.2.66q-.38 2.81-2.54 4.68t-5.03 1.88h-.88q-2.87 0-5.03-1.88T7.5 18.95q-.06-.38.19-.66t.69-.28h14.25z',
            'laugh-beam' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm8.84 24.34Q28 21.2 28 16t-3.66-8.84T15.5 3.5 6.66 7.16 3 16t3.66 8.84 8.84 3.66 8.84-3.66zM20.5 9.5q1.19 0 2.28 1.28T24 13.94q0 .43-.38.53t-.56-.22l-.62-1.06q-.75-1.38-1.94-1.38t-2 1.38l-.56 1.06q-.2.31-.6.22t-.34-.53q.12-1.88 1.22-3.16T20.5 9.5zM7.94 14.25q-.19.31-.6.22T7 13.94q.13-1.88 1.22-3.16T10.5 9.5t2.28 1.28T14 13.94q0 .44-.38.53t-.56-.22l-.56-1.06q-.81-1.38-2-1.38t-2 1.38zM22.62 18q.44 0 .7.28t.18.66q-.37 2.81-2.53 4.68t-5.03 1.88h-.88q-2.87 0-5.03-1.88T7.5 18.95q-.06-.38.19-.66t.69-.28h14.25z',
            'laugh-squint' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm8.84 24.34Q28 21.2 28 16t-3.66-8.84T15.5 3.5 6.66 7.16 3 16t3.66 8.84 8.84 3.66 8.84-3.66zM21.5 12.25l2.06 2.5q.44.5 0 .97t-.93.15l-5-3q-.38-.18-.38-.62t.38-.63l5-3q.5-.3.93.16t0 .97zM8.37 15.88q-.56.3-.96-.16t.03-.97l2.06-2.5-2.06-2.5q-.44-.5 0-.97t.93-.15l5 3q.38.18.38.62t-.38.63zM22.63 18q.43 0 .68.28t.2.66q-.38 2.81-2.54 4.68t-5.03 1.88h-.88q-2.87 0-5.03-1.88T7.5 18.95q-.06-.38.19-.66t.69-.28h14.25z',
            'laugh-wink' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm8.84 24.34Q28 21.2 28 16t-3.66-8.84T15.5 3.5 6.66 7.16 3 16t3.66 8.84 8.84 3.66 8.84-3.66zM20.5 10.25q.81 0 1.63.31t1.4.94.72 1.38q.06.5-.4.75t-.85-.07l-.63-.56q-.68-.56-1.87-.56t-1.94.56l-.56.56q-.38.32-.84.07t-.41-.75q.12-.75.72-1.38t1.4-.94 1.63-.31zM10.5 14q-.81 0-1.4-.6T8.5 12t.6-1.4 1.4-.6 1.4.6.6 1.4-.6 1.4-1.4.6zm12.13 4q.43 0 .68.28t.2.66q-.38 2.81-2.54 4.68t-5.03 1.88h-.88q-2.87 0-5.03-1.88T7.5 18.95q-.06-.38.19-.66t.69-.28h14.25z',
            'meh-blank' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm0 28q5.19 0 8.84-3.66T28 16t-3.66-8.84T15.5 3.5 6.66 7.16 3 16t3.66 8.84 8.84 3.66zm-5-17.5q.81 0 1.4.6t.6 1.4-.6 1.4-1.4.6-1.4-.6-.6-1.4.6-1.4 1.4-.6zm10 0q.81 0 1.4.6t.6 1.4-.6 1.4-1.4.6-1.4-.6-.6-1.4.6-1.4 1.4-.6z',
            'meh-rolling-eyes' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm0 28q5.19 0 8.84-3.66T28 16t-3.66-8.84T15.5 3.5 6.66 7.16 3 16t3.66 8.84 8.84 3.66zm5.5-19q1.88 0 3.19 1.31t1.31 3.2-1.31 3.18T21 18.5t-3.19-1.31T16.5 14t1.31-3.19T21 9.5zm0 7q1.06 0 1.78-.72T23.5 14q0-1.25-1.13-2 .13.25.13.5 0 .63-.44 1.06T21 14t-1.06-.44-.44-1.06q0-.25.12-.5-1.12.75-1.12 2 0 1.06.72 1.78t1.78.72zM14.5 14q0 1.88-1.31 3.19t-3.2 1.31-3.18-1.31T5.5 14t1.31-3.19T10 9.5t3.19 1.31T14.5 14zm-7 0q0 1.06.72 1.78t1.78.72 1.78-.72.72-1.78q0-1.25-1.13-2 .13.25.13.5 0 .63-.44 1.06T10 14t-1.06-.44-.44-1.06q0-.25.12-.5-1.12.75-1.12 2zm12 8q.63 0 1.06.44T21 23.5t-.44 1.06-1.06.44h-8q-.63 0-1.06-.44T10 23.5t.44-1.06T11.5 22h8z',
            'sad-cry' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm9 24.13Q28 21 28 16q0-5.19-3.66-8.84T15.5 3.5 6.66 7.16 3 16q0 5 3.5 8.63V17.5q0-.63.44-1.06T8 16t1.06.44.44 1.06v9.44q2.82 1.56 6 1.56t6-1.56V17.5q0-.63.44-1.06T23 16t1.07.44.43 1.06v7.13zm-11.66-9.97q-.47.28-.84-.1l-.56-.56q-.75-.56-1.94-.56T7.62 14l-.62.56q-.38.38-.85.1t-.4-.78q.12-.76.72-1.38t1.4-.94 1.63-.31q1.25 0 2.4.75t1.35 1.87q.06.5-.4.78zm8.66-3.41q1.25 0 2.4.75t1.35 1.88q.06.5-.4.74t-.85-.06l-.63-.56q-.68-.56-1.87-.56t-1.88.56l-.62.56q-.38.31-.84.06t-.41-.74q.19-1.13 1.34-1.88t2.41-.75zm-6 5.75q1.44 0 2.47 1.19T19 21t-1.03 2.81-2.47 1.2-2.47-1.2-1.03-2.8 1.03-2.82T15.5 17z',
            'sad-tear' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm0 28q5.19 0 8.84-3.66T28 16t-3.66-8.84T15.5 3.5 6.66 7.16 3 16t3.66 8.84 8.84 3.66zM16 19q3.75 0 6.13 2.88.43.5.37 1.12t-.56 1-1.1.31-.97-.5Q18.37 22 16 22q-.63 0-1.06-.44t-.44-1.06.44-1.06T16 19zm-5.5-4q-.81 0-1.4-.6T8.5 13t.6-1.4 1.4-.6 1.4.6.6 1.4-.6 1.4-1.4.6zm10-4q.81 0 1.4.6t.6 1.4-.6 1.4-1.4.6-1.4-.6-.6-1.4.6-1.4 1.4-.6zm-10.38 6.19q.38-.5.76 0 2.24 3.06 2.24 4.25 0 1.06-.78 1.81T10.5 24t-1.84-.75-.78-1.81q0-1.19 2.25-4.25z',
            'smile-beam' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm0 28q5.19 0 8.84-3.66T28 16t-3.66-8.84T15.5 3.5 6.66 7.16 3 16t3.66 8.84 8.84 3.66zm5.25-8.94q.38-.5 1-.56t1.1.34.52 1.04-.3 1.06Q20.11 25 15.5 25t-7.56-3.56q-.38-.44-.32-1.06t.53-1.04 1.1-.34 1 .56Q12.3 22 15.5 22t5.25-2.44zM8.5 13.2l-.56 1.06q-.2.38-.57.25-.43-.12-.37-.56.06-.75.31-1.44t.6-1.25.78-.94.9-.6.9-.21q1.2 0 2.29 1.28T14 13.94q.06.43-.34.56t-.6-.25l-.56-1.06q-.81-1.38-2-1.38t-2 1.38zm12-3.69q1.19 0 2.28 1.28T24 13.94q.06.43-.34.56t-.6-.25l-.56-1.06q-.81-1.38-2-1.38t-2 1.38l-.56 1.06q-.2.38-.57.25-.43-.13-.37-.56.12-1.88 1.22-3.16T20.5 9.5z',
            'surprise' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm0 28q5.19 0 8.84-3.66T28 16t-3.66-8.84T15.5 3.5 6.66 7.16 3 16t3.66 8.84 8.84 3.66zm0-11q1.63 0 2.81 1.19t1.2 2.81-1.2 2.81-2.81 1.2-2.81-1.2-1.2-2.8 1.2-2.82 2.81-1.19zm-3-4.5q0 .81-.6 1.4t-1.4.6-1.4-.6-.6-1.4.6-1.4 1.4-.6 1.4.6.6 1.4zm8-2q.81 0 1.4.6t.6 1.4-.6 1.4-1.4.6-1.4-.6-.6-1.4.6-1.4 1.4-.6z',
            'tired' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm0 28q5.19 0 8.84-3.66T28 16t-3.66-8.84T15.5 3.5 6.66 7.16 3 16t3.66 8.84 8.84 3.66zm8.1-18.97q.4.47-.04.97L21.5 13l2.06 2.5q.44.5.03.97t-.97.16l-5-3q-.37-.2-.37-.63t.37-.63l5-3q.57-.3.97.16zM13.74 13q0 .44-.38.63l-5 3q-.56.3-.96-.16t.03-.97L9.5 13l-2.06-2.5q-.44-.5 0-.97t.93-.15l5 3q.38.18.38.62zm1.75 4q2.25 0 4.34 1.69t2.41 4.12q.06.57-.25.94t-.75.19Q19.06 23 15.5 23t-5.75.94q-.25.12-.5 0t-.4-.44-.1-.69q.31-2.44 2.4-4.12T15.5 17z'
        ];
    }

    public static function fas() {
        return [
            '0' => [
                'p' => 'M0 192C0 103.6 71.6 32 160 32s160 71.6 160 160V320c0 88.4-71.6 160-160 160S0 408.4 0 320V192zM160 96c-53 0-96 43-96 96V320c0 53 43 96 96 96s96-43 96-96V192c0-53-43-96-96-96z',
                'vw' => 320,
                'vh' => 512
            ],
            '1' => [
                'p' => 'M160 64c0-11.8-6.5-22.6-16.9-28.2s-23-5-32.8 1.6l-96 64C-.5 111.2-4.4 131 5.4 145.8s29.7 18.7 44.4 8.9L96 123.8V416H32c-17.7 0-32 14.3-32 32s14.3 32 32 32h96 96c17.7 0 32-14.3 32-32s-14.3-32-32-32H160V64z',
                'vw' => 256,
                'vh' => 512
            ],
            '2' => [
                'p' => 'M142.9 96c-21.5 0-42.2 8.5-57.4 23.8L54.6 150.6c-12.5 12.5-32.8 12.5-45.3 0s-12.5-32.8 0-45.3L40.2 74.5C67.5 47.3 104.4 32 142.9 32C223 32 288 97 288 177.1c0 38.5-15.3 75.4-42.5 102.6L109.3 416H288c17.7 0 32 14.3 32 32s-14.3 32-32 32H32c-12.9 0-24.6-7.8-29.6-19.8s-2.2-25.7 6.9-34.9L200.2 234.5c15.2-15.2 23.8-35.9 23.8-57.4c0-44.8-36.3-81.1-81.1-81.1z',
                'vw' => 320,
                'vh' => 512
            ],
            '3' => [
                'p' => 'M0 64C0 46.3 14.3 32 32 32H272c13.2 0 25 8.1 29.8 20.4s1.5 26.3-8.2 35.2L162.3 208H184c75.1 0 136 60.9 136 136s-60.9 136-136 136H105.4C63 480 24.2 456 5.3 418.1l-1.9-3.8c-7.9-15.8-1.5-35 14.3-42.9s35-1.5 42.9 14.3l1.9 3.8c8.1 16.3 24.8 26.5 42.9 26.5H184c39.8 0 72-32.2 72-72s-32.2-72-72-72H80c-13.2 0-25-8.1-29.8-20.4s-1.5-26.3 8.2-35.2L189.7 96H32C14.3 96 0 81.7 0 64z',
                'vw' => 320,
                'vh' => 512
            ],
            '4' => [
                'p' => 'M189 77.6c7.5-16 .7-35.1-15.3-42.6s-35.1-.7-42.6 15.3L3 322.4c-4.7 9.9-3.9 21.5 1.9 30.8S21 368 32 368H256v80c0 17.7 14.3 32 32 32s32-14.3 32-32V368h32c17.7 0 32-14.3 32-32s-14.3-32-32-32H320V160c0-17.7-14.3-32-32-32s-32 14.3-32 32V304H82.4L189 77.6z',
                'vw' => 384,
                'vh' => 512
            ],
            '5' => [
                'p' => 'M32.5 58.3C35.3 43.1 48.5 32 64 32H256c17.7 0 32 14.3 32 32s-14.3 32-32 32H90.7L70.3 208H184c75.1 0 136 60.9 136 136s-60.9 136-136 136H100.5c-39.4 0-75.4-22.3-93-57.5l-4.1-8.2c-7.9-15.8-1.5-35 14.3-42.9s35-1.5 42.9 14.3l4.1 8.2c6.8 13.6 20.6 22.1 35.8 22.1H184c39.8 0 72-32.2 72-72s-32.2-72-72-72H32c-9.5 0-18.5-4.2-24.6-11.5s-8.6-16.9-6.9-26.2l32-176z',
                'vw' => 320,
                'vh' => 512
            ],
            '6' => [
                'p' => 'M232.4 84.7c11.4-13.5 9.7-33.7-3.8-45.1s-33.7-9.7-45.1 3.8L38.6 214.7C14.7 242.9 1.1 278.4 .1 315.2c0 1.4-.1 2.9-.1 4.3c0 .2 0 .3 0 .5c0 88.4 71.6 160 160 160s160-71.6 160-160c0-85.5-67.1-155.4-151.5-159.8l63.9-75.6zM256 320A96 96 0 1 1 64 320a96 96 0 1 1 192 0z',
                'vw' => 320,
                'vh' => 512
            ],
            '7' => [
                'p' => 'M0 64C0 46.3 14.3 32 32 32H288c11.5 0 22 6.1 27.7 16.1s5.7 22.2-.1 32.1l-224 384c-8.9 15.3-28.5 20.4-43.8 11.5s-20.4-28.5-11.5-43.8L232.3 96H32C14.3 96 0 81.7 0 64z',
                'vw' => 320,
                'vh' => 512
            ],
            '8' => [
                'p' => 'M304 160c0-70.7-57.3-128-128-128H144C73.3 32 16 89.3 16 160c0 34.6 13.7 66 36 89C20.5 272.3 0 309.8 0 352c0 70.7 57.3 128 128 128h64c70.7 0 128-57.3 128-128c0-42.2-20.5-79.7-52-103c22.3-23 36-54.4 36-89zM176.1 288H192c35.3 0 64 28.7 64 64s-28.7 64-64 64H128c-35.3 0-64-28.7-64-64s28.7-64 64-64h15.9c0 0 .1 0 .1 0h32c0 0 .1 0 .1 0zm0-64c0 0 0 0 0 0H144c0 0 0 0 0 0c-35.3 0-64-28.7-64-64c0-35.3 28.7-64 64-64h32c35.3 0 64 28.7 64 64c0 35.3-28.6 64-64 64z',
                'vw' => 320,
                'vh' => 512
            ],
            '9' => [
                'p' => 'M64 192a96 96 0 1 0 192 0A96 96 0 1 0 64 192zm87.5 159.8C67.1 347.4 0 277.5 0 192C0 103.6 71.6 32 160 32s160 71.6 160 160c0 2.6-.1 5.3-.2 7.9c-1.7 35.7-15.2 70-38.4 97.4l-145 171.4c-11.4 13.5-31.6 15.2-45.1 3.8s-15.2-31.6-3.8-45.1l63.9-75.6z',
                'vw' => 320,
                'vh' => 512
            ],
            'a' => [
                'p' => 'M221.5 51.7a31.94 31.94 0 0 0-59 0l-120 288-40 96c-6.8 16.3.9 35 17.2 41.8s35-.9 41.8-17.2L93.3 384h197.4l31.8 76.3c6.8 16.3 25.5 24 41.8 17.2s24-25.5 17.2-41.8l-40-96-120-288zM264 320H120l72-172.8L264 320z',
                'vw' => 384,
                'vh' => 512
            ],
            'ad' => 'M9.88 17 11 13.69 12.13 17H9.87zM22 16q.63 0 1.06.44t.44 1.06-.44 1.06T22 19t-1.06-.44-.44-1.06.44-1.06T22 16zm7-12q1.25 0 2.13.88T32 7v18q0 1.25-.88 2.13T29 28H3q-1.25 0-2.13-.88T0 25V7q0-1.25.88-2.13T3 4h26zM15.69 22q.5 0 .78-.4t.16-.91L13.25 11q-.13-.44-.56-.72T11.8 10h-1.6q-.32 0-.6.13t-.5.37-.34.5l-3.38 9.69q-.12.5.16.9t.78.41h1.06q.75 0 .94-.69l.5-1.31h4.38l.5 1.31q.18.69.93.69h1.07zm10.81-1V11q0-.44-.28-.72T25.5 10h-1q-.44 0-.72.28t-.28.72v2.25Q22.75 13 22 13q-1.88 0-3.19 1.31t-1.31 3.2 1.31 3.18T22 22q.94 0 1.75-.37.31.37.75.37h1q.44 0 .72-.28t.28-.72z',
            'address-book' => [
                'p' => 'M27.25 10H26v4h1.25q.75 0 .75.75v2.5q0 .75-.75.75H26v4h1.25q.75 0 .75.75v2.5q0 .75-.75.75H26v3q0 1.25-.88 2.13T23 32H3q-1.25 0-2.13-.88T0 29V3Q0 1.75.88.87T3 0h20q1.25 0 2.13.88T26 3v3h1.25q.75 0 .75.75v2.5q0 .75-.75.75zM13 8q-1.63 0-2.81 1.19T8.99 12t1.2 2.81 2.81 1.2 2.81-1.2 1.2-2.8-1.2-2.82T13 8zm7 14.81v-1.18q0-1.5-1.22-2.57T15.81 18h-.31q-1.19.5-2.5.5t-2.5-.5h-.31q-1.13 0-2.1.47t-1.53 1.31T6 21.63v1.18q0 .5.4.85t.97.34h11.25q.57 0 .97-.34t.4-.85z',
                'w' => .875,
                'vw' => 28
            ],
            'address-card' => [
                'p' => 'M33 2q1.25 0 2.13.88T36 5v22q0 1.25-.88 2.13T33 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h30zM11 8Q9.37 8 8.19 9.19T6.99 12t1.2 2.81 2.81 1.2 2.81-1.2 1.2-2.8-1.2-2.82T11 8zm7 14.81v-1.18q0-1.5-1.22-2.57T13.81 18h-.31q-1.19.5-2.5.5T8.5 18h-.31q-1.75 0-2.97 1.06T4 21.63v1.18q0 .5.4.85t.97.34h11.25q.57 0 .97-.34t.4-.85zm14-3.31v-1q0-.5-.5-.5h-9q-.5 0-.5.5v1q0 .5.5.5h9q.5 0 .5-.5zm0-4v-1q0-.5-.5-.5h-9q-.5 0-.5.5v1q0 .5.5.5h9q.5 0 .5-.5zm0-4v-1q0-.5-.5-.5h-9q-.5 0-.5.5v1q0 .5.5.5h9q.5 0 .5-.5z',
                'w' => 1.125,
                'vw' => 36
            ],
            'adjust' => 'M.5 16q0-6.44 4.53-10.97T16 .5t10.97 4.53T31.5 16t-4.53 10.97T16 31.5 5.03 26.97.5 16zM16 27.5q4.75 0 8.13-3.38T27.5 16t-3.38-8.13T16 4.5v23z',
            'air-freshener' => [
                'p' => 'M23.69 20.06q.56.63.19 1.28t-1.25.66H14v2h7q.44 0 .72.28T22 25v6q0 .44-.28.72T21 32H3q-.44 0-.72-.28T2 31v-6q0-.44.28-.72T3 24h7v-2H1.38q-.88 0-1.25-.66t.18-1.28L6.2 14H3.13q-.7 0-1-.6t.12-1.15l7.13-7.88Q9 3.7 9 3q0-1.25.88-2.13T12 0t2.13.88T15 3q0 .69-.37 1.38l7.12 7.87q.44.56.13 1.16t-1 .59H17.8zM12 2q-.44 0-.72.28T11 3t.28.72T12 4t.72-.28T13 3t-.28-.72T12 2zm7 25H5v2h14v-2z',
                'w' => .75,
                'vw' => 24
            ],
            'airplay' => 'M8.1 25.9h15.8L16 16.8Zm3.3-6.1H6.1V8.6h19.8v11.2h-5.3l.9 1h5.4V7.6H5.1v13.2h5.4Zm0 0',
            'align-center' => [
                'p' => 'M22 2.75v2.5q0 .44-.28.72t-.72.28H7q-.44 0-.72-.28T6 5.25v-2.5q0-.44.28-.72T7 1.75h14q.44 0 .72.28t.28.72zM1 14.25q-.44 0-.72-.28T0 13.25v-2.5q0-.44.28-.72T1 9.75h26q.44 0 .72.28t.28.72v2.5q0 .44-.28.72t-.72.28H1zm0 16q-.44 0-.72-.28T0 29.25v-2.5q0-.44.28-.72t.72-.28h26q.44 0 .72.28t.28.72v2.5q0 .44-.28.72t-.72.28H1zm20-12.5q.44 0 .72.28t.28.72v2.5q0 .44-.28.72t-.72.28H7q-.44 0-.72-.28T6 21.25v-2.5q0-.44.28-.72t.72-.28h14z',
                'w' => .875,
                'vw' => 28
            ],
            'align-justify' => [
                'p' => 'M0 5.25v-2.5q0-.44.28-.72T1 1.75h26q.44 0 .72.28t.28.72v2.5q0 .44-.28.72t-.72.28H1q-.44 0-.72-.28T0 5.25zm1 9q-.44 0-.72-.28T0 13.25v-2.5q0-.44.28-.72T1 9.75h26q.44 0 .72.28t.28.72v2.5q0 .44-.28.72t-.72.28H1zm0 16q-.44 0-.72-.28T0 29.25v-2.5q0-.44.28-.72t.72-.28h26q.44 0 .72.28t.28.72v2.5q0 .44-.28.72t-.72.28H1zm0-8q-.44 0-.72-.28T0 21.25v-2.5q0-.44.28-.72t.72-.28h26q.44 0 .72.28t.28.72v2.5q0 .44-.28.72t-.72.28H1z',
                'w' => .875,
                'vw' => 28
            ],
            'align-left' => [
                'p' => 'M18 2.75v2.5q0 .44-.28.72t-.72.28H1q-.44 0-.72-.28T0 5.25v-2.5q0-.44.28-.72T1 1.75h16q.44 0 .72.28t.28.72zm-18 8q0-.44.28-.72T1 9.75h26q.44 0 .72.28t.28.72v2.5q0 .44-.28.72t-.72.28H1q-.44 0-.72-.28T0 13.25v-2.5zm1 19.5q-.44 0-.72-.28T0 29.25v-2.5q0-.44.28-.72t.72-.28h26q.44 0 .72.28t.28.72v2.5q0 .44-.28.72t-.72.28H1zm16-12.5q.44 0 .72.28t.28.72v2.5q0 .44-.28.72t-.72.28H1q-.44 0-.72-.28T0 21.25v-2.5q0-.44.28-.72t.72-.28h16z',
                'w' => .875,
                'vw' => 28
            ],
            'align-right' => [
                'p' => 'M10 5.25v-2.5q0-.44.28-.72t.72-.28h16q.44 0 .72.28t.28.72v2.5q0 .44-.28.72t-.72.28H11q-.44 0-.72-.28T10 5.25zm-9 9q-.44 0-.72-.28T0 13.25v-2.5q0-.44.28-.72T1 9.75h26q.44 0 .72.28t.28.72v2.5q0 .44-.28.72t-.72.28H1zm0 16q-.44 0-.72-.28T0 29.25v-2.5q0-.44.28-.72t.72-.28h26q.44 0 .72.28t.28.72v2.5q0 .44-.28.72t-.72.28H1zm10-8q-.44 0-.72-.28t-.28-.72v-2.5q0-.44.28-.72t.72-.28h16q.44 0 .72.28t.28.72v2.5q0 .44-.28.72t-.72.28H11z',
                'w' => .875,
                'vw' => 28
            ],
            'allergies' => [
                'p' => 'M26 7q.81 0 1.4.6T28 9v11q0 1.38-.31 2.63l-1.7 7.06q-.18 1-1.02 1.65t-1.85.66H10.75q-1.5 0-2.44-1.25L.5 20q-.63-.88-.47-1.9t1-1.63 1.87-.44 1.6.97L6 19.06V4q0-.81.6-1.4T8 2t1.4.6T10 4v9.5q0 .5.5.5h1q.5 0 .5-.5V2q0-.81.6-1.4T14 0t1.4.6T16 2v11.5q0 .5.5.5h1q.5 0 .5-.5V4q0-.81.6-1.4T20 2t1.4.6T22 4v9.5q0 .5.5.5h1q.5 0 .5-.5V9q0-.81.6-1.4T26 7zM11 26q.44 0 .72-.28T12 25t-.28-.72T11 24t-.72.28T10 25t.28.72.72.28zm0-6q.44 0 .72-.28T12 19t-.28-.72T11 18t-.72.28T10 19t.28.72.72.28zm4 8q.44 0 .72-.28T16 27t-.28-.72T15 26t-.72.28T14 27t.28.72.72.28zm0-6q.44 0 .72-.28T16 21t-.28-.72T15 20t-.72.28T14 21t.28.72.72.28zm4 2q.44 0 .72-.28T20 23t-.28-.72T19 22t-.72.28T18 23t.28.72.72.28zm2 4q.44 0 .72-.28T22 27t-.28-.72T21 26t-.72.28T20 27t.28.72.72.28zm2-8q.44 0 .72-.28T24 19t-.28-.72T23 18t-.72.28T22 19t.28.72.72.28z',
                'w' => .875,
                'vw' => 28
            ],
            'ambulance' => [
                'p' => 'M39 22q.44 0 .72.28T40 23v2q0 .44-.28.72T39 26h-3q0 2.5-1.75 4.25T30 32t-4.25-1.75T24 26h-8q0 2.5-1.75 4.25T10 32t-4.25-1.75T4 26H3q-1.25 0-2.13-.88T0 23V3Q0 1.75.88.87T3 0h20q1.25 0 2.13.88T26 3v3h2.75q1.25 0 2.13.88l6.25 6.25q.87.87.87 2.12V22h1zm-29 7q1.25 0 2.13-.88T13 26t-.88-2.13T10 23t-2.13.88T7 26t.88 2.13T10 29zm9-15.5v-3q0-.5-.5-.5H15V6.5q0-.5-.5-.5h-3q-.5 0-.5.5V10H7.5q-.5 0-.5.5v3q0 .5.5.5H11v3.5q0 .5.5.5h3q.5 0 .5-.5V14h3.5q.5 0 .5-.5zM30 29q1.25 0 2.13-.88T33 26t-.88-2.13T30 23t-2.13.88T27 26t.88 2.13T30 29zm5-13v-.75L28.75 9H26v7h9z',
                'w' => 1.25,
                'vw' => 40
            ],
            'american-sign-language-interpreting' => [
                'p' => 'M18.19 11.81q-2-1-4.07-.25 3.82 0 5.32 3.2.44.8-.07 1.59t-1.37.78q.88 0 1.38.78t.06 1.6q-.69 1.43-2.03 2.3t-2.97.88l-5.88-.56-4.18 2.06q-.44.25-.88.1t-.62-.54L.13 18.2q-.5-.94.43-1.5l3.63-2.06L6.75 10q.63-5.62 5-9.19.5-.43 1.19-.37t1.12.6.38 1.21-.56 1.13q-1.25 1.06-2.07 2.25 2.13-1.44 4.63-1.75.69-.13 1.25.3t.62 1.13q.07.44-.1.85t-.52.69-.81.34q-1.44.13-2.75.94 2.8-.63 5.5.69.62.3.84.97t-.1 1.28-.96.84-1.22-.1zm-1.75 4.38q-.63-1.31-2-1.31-.94 0-1.57.65t-.62 1.6q0 .62.31 1.15t.82.78 1.06.25q1.37 0 2-1.25.44-.94 1.5-.94-1.06-.06-1.5-.93zm23.43-2.38q.5 1-.43 1.5l-3.63 2.07L33.25 22q-.63 5.63-5 9.19-.5.44-1.19.37t-1.12-.59-.38-1.22.56-1.12q1.25-1.07 2.07-2.25-2.13 1.43-4.63 1.75-.56.12-1.19-.29t-.68-1.15q-.07-.44.09-.85t.53-.68.81-.35q1.44-.12 2.75-.93-2.8.62-5.5-.7-.62-.3-.84-.96t.1-1.28.96-.85 1.22.1q2 1 4.06.25-3.8 0-5.3-3.2-.45-.8.05-1.58t1.38-.79q-.88 0-1.38-.78t-.06-1.6q.69-1.43 2.03-2.3t3.1-.88l5.75.56 4.18-2.06q.44-.25.88-.1t.63.54zm-14.3-1.12q-1.38 0-2 1.25-.44.94-1.5.94 1.06.06 1.5 1 .62 1.25 2 1.25.93 0 1.56-.66t.62-1.6-.65-1.56-1.54-.62z',
                'w' => 1.248,
                'vw' => 40
            ],
            'anchor' => [
                'p' => 'M.81 22q-.5 0-.69-.47t.13-.78L4.5 16.5q.5-.5 1 0l4.25 4.25q.31.31.13.78t-.7.47H7q.87 2.44 3.44 3.97t5.56 1.9V16h-3.25q-.75 0-.75-.75v-2.5q0-.75.75-.75H16v-.31q-1.75-.63-2.88-2.2T12 6q0-2.5 1.78-4.27T18.06 0t4.22 1.81T24 6q0 1.94-1.13 3.5T20 11.69V12h3.25q.31 0 .53.22t.22.53v2.5q0 .31-.22.53t-.53.22H20v11.88q3-.38 5.56-1.91T29 22h-2.19q-.5 0-.68-.47t.12-.78l4.25-4.25q.5-.5 1 0l4.25 4.25q.31.31.13.78t-.7.47h-2q-1 4.56-5.43 7.28T18 32t-9.75-2.72T2.8 22h-2zM18 4q-.81 0-1.4.6T16 6t.6 1.4T18 8t1.4-.6T20 6t-.6-1.4T18 4z',
                'w' => 1.123,
                'vw' => 36
            ],
            'anchor-circle-check' => [
                'p' => 'M320 96a32 32 0 1 1 -64 0 32 32 0 1 1 64 0zm21.1 80C367 158.8 384 129.4 384 96c0-53-43-96-96-96s-96 43-96 96c0 33.4 17 62.8 42.9 80H224c-17.7 0-32 14.3-32 32s14.3 32 32 32h32V448H208c-53 0-96-43-96-96v-6.1l7 7c9.4 9.4 24.6 9.4 33.9 0s9.4-24.6 0-33.9L97 263c-9.4-9.4-24.6-9.4-33.9 0L7 319c-9.4 9.4-9.4 24.6 0 33.9s24.6 9.4 33.9 0l7-7V352c0 88.4 71.6 160 160 160h80 80c8.2 0 16.3-.6 24.2-1.8c-22.2-16.2-40.4-37.5-53-62.2H320V368 240h32c17.7 0 32-14.3 32-32s-14.3-32-32-32H341.1zM640 368a144 144 0 1 0 -288 0 144 144 0 1 0 288 0zm-76.7-43.3c6.2 6.2 6.2 16.4 0 22.6l-72 72c-6.2 6.2-16.4 6.2-22.6 0l-40-40c-6.2-6.2-6.2-16.4 0-22.6s16.4-6.2 22.6 0L480 385.4l60.7-60.7c6.2-6.2 16.4-6.2 22.6 0z',
                'vw' => 640,
                'vh' => 512
            ],
            'anchor-circle-exclamation' => [
                'p' => 'M320 96a32 32 0 1 1 -64 0 32 32 0 1 1 64 0zm21.1 80C367 158.8 384 129.4 384 96c0-53-43-96-96-96s-96 43-96 96c0 33.4 17 62.8 42.9 80H224c-17.7 0-32 14.3-32 32s14.3 32 32 32h32V448H208c-53 0-96-43-96-96v-6.1l7 7c9.4 9.4 24.6 9.4 33.9 0s9.4-24.6 0-33.9L97 263c-9.4-9.4-24.6-9.4-33.9 0L7 319c-9.4 9.4-9.4 24.6 0 33.9s24.6 9.4 33.9 0l7-7V352c0 88.4 71.6 160 160 160h80 80c8.2 0 16.3-.6 24.2-1.8c-22.2-16.2-40.4-37.5-53-62.2H320V368 240h32c17.7 0 32-14.3 32-32s-14.3-32-32-32H341.1zM496 512a144 144 0 1 0 0-288 144 144 0 1 0 0 288zm0-96a24 24 0 1 1 0 48 24 24 0 1 1 0-48zm0-144c8.8 0 16 7.2 16 16v80c0 8.8-7.2 16-16 16s-16-7.2-16-16V288c0-8.8 7.2-16 16-16z',
                'vw' => 640,
                'vh' => 512
            ],
            'anchor-circle-xmark' => [
                'p' => 'M320 96a32 32 0 1 1 -64 0 32 32 0 1 1 64 0zm21.1 80C367 158.8 384 129.4 384 96c0-53-43-96-96-96s-96 43-96 96c0 33.4 17 62.8 42.9 80H224c-17.7 0-32 14.3-32 32s14.3 32 32 32h32V448H208c-53 0-96-43-96-96v-6.1l7 7c9.4 9.4 24.6 9.4 33.9 0s9.4-24.6 0-33.9L97 263c-9.4-9.4-24.6-9.4-33.9 0L7 319c-9.4 9.4-9.4 24.6 0 33.9s24.6 9.4 33.9 0l7-7V352c0 88.4 71.6 160 160 160h80 80c8.2 0 16.3-.6 24.2-1.8c-22.2-16.2-40.4-37.5-53-62.2H320V368 240h32c17.7 0 32-14.3 32-32s-14.3-32-32-32H341.1zM496 512a144 144 0 1 0 0-288 144 144 0 1 0 0 288zm59.3-180.7L518.6 368l36.7 36.7c6.2 6.2 6.2 16.4 0 22.6s-16.4 6.2-22.6 0L496 390.6l-36.7 36.7c-6.2 6.2-16.4 6.2-22.6 0s-6.2-16.4 0-22.6L473.4 368l-36.7-36.7c-6.2-6.2-6.2-16.4 0-22.6s16.4-6.2 22.6 0L496 345.4l36.7-36.7c6.2-6.2 16.4-6.2 22.6 0s6.2 16.4 0 22.6z',
                'vw' => 640,
                'vh' => 512
            ],
            'anchor-lock' => [
                'p' => 'M320 96a32 32 0 1 1 -64 0 32 32 0 1 1 64 0zm21.1 80C367 158.8 384 129.4 384 96c0-53-43-96-96-96s-96 43-96 96c0 33.4 17 62.8 42.9 80H224c-17.7 0-32 14.3-32 32s14.3 32 32 32h32V448H208c-53 0-96-43-96-96v-6.1l7 7c9.4 9.4 24.6 9.4 33.9 0s9.4-24.6 0-33.9L97 263c-9.4-9.4-24.6-9.4-33.9 0L7 319c-9.4 9.4-9.4 24.6 0 33.9s24.6 9.4 33.9 0l7-7V352c0 88.4 71.6 160 160 160h80 80c8 0 15.9-.6 23.6-1.7c-4.8-9-7.6-19.3-7.6-30.3V446.7c-5.2 .9-10.5 1.3-16 1.3H320V240h32c17.7 0 32-14.3 32-32s-14.3-32-32-32H341.1zM528 240c17.7 0 32 14.3 32 32v48H496V272c0-17.7 14.3-32 32-32zm-80 32v48c-17.7 0-32 14.3-32 32V480c0 17.7 14.3 32 32 32H608c17.7 0 32-14.3 32-32V352c0-17.7-14.3-32-32-32V272c0-44.2-35.8-80-80-80s-80 35.8-80 80z',
                'vw' => 640,
                'vh' => 512
            ],
            'angle-double-down' => [
                'p' => 'M8.94 16 .44 7.5Q0 7.06 0 6.44t.44-1.07L1.88 4q.43-.44 1.06-.44T4 4l6 6 6-6q.44-.44 1.06-.44t1.07.44l1.44 1.37q.43.44.43 1.07t-.43 1.06l-8.5 8.5q-.44.44-1.07.44T8.94 16zm2.12 12q-.43.44-1.06.44T8.94 28l-8.5-8.5Q0 19.06 0 18.44t.44-1.07L1.87 16q.44-.44 1.07-.44T4 16l6 6 6-6q.44-.44 1.06-.44t1.07.44l1.43 1.37q.44.44.44 1.07t-.44 1.06z',
                'w' => .625,
                'vw' => 20
            ],
            'angle-double-left' => [
                'p' => 'm14 14.94 8.5-8.5Q22.94 6 23.56 6t1.07.44L26 7.88q.44.43.44 1.06T26 10l-6 6 6 6q.31.32.4.69t0 .78-.4.66l-1.37 1.43q-.44.44-1.07.44t-1.06-.44l-8.5-8.5q-.44-.43-.44-1.06t.44-1.06zM2 17.06q-.44-.43-.44-1.06T2 14.94l8.5-8.5Q10.94 6 11.56 6t1.07.44L14 7.87q.44.44.44 1.07T14 10l-6 6 6 6q.44.44.44 1.06T14 24.13l-1.37 1.43q-.44.44-1.07.44t-1.06-.44z',
                'w' => .875,
                'vw' => 28
            ],
            'angle-double-right' => [
                'p' => 'm14 17.06-8.5 8.5q-.44.44-1.06.44t-1.07-.44L2 24.13q-.44-.44-.44-1.07T2 22l6-6-6-6q-.44-.44-.44-1.06T2 7.87l1.37-1.43Q3.81 6 4.44 6t1.06.44l8.5 8.5q.44.43.44 1.06T14 17.06zm12-2.12q.44.44.44 1.06T26 17.06l-8.5 8.5q-.44.44-1.06.44t-1.07-.44L14 24.13q-.44-.44-.44-1.07T14 22l6-6-6-6q-.44-.44-.44-1.06T14 7.87l1.37-1.43q.25-.2.54-.32t.56-.12.56.12.47.32z',
                'w' => .875,
                'vw' => 28
            ],
            'angle-double-up' => [
                'p' => 'm11.06 16 8.5 8.5q.44.44.44 1.06t-.44 1.07L18.13 28q-.44.44-1.07.44T16 28l-6-6-6 6q-.44.44-1.06.44T1.87 28L.43 26.63Q0 26.19 0 25.56t.43-1.06l8.5-8.5q.44-.44 1.07-.44t1.06.44zM8.94 4q.44-.44 1.06-.44t1.06.44l8.5 8.5q.44.44.44 1.06t-.44 1.07L18.13 16q-.44.44-1.07.44T16 16l-6-6-6 6q-.25.25-.53.35t-.56.09-.57-.1-.47-.34L.44 14.63Q0 14.19 0 13.56t.44-1.06z',
                'w' => .625,
                'vw' => 20
            ],
            'angle-down' => [
                'p' => 'm8.94 22-8.5-8.5Q0 13.06 0 12.44t.44-1.07L1.88 10q.43-.44 1.06-.44T4 10l6 6 6-6q.44-.44 1.06-.44t1.07.44l1.44 1.37q.43.44.43 1.07t-.43 1.06l-8.5 8.5q-.44.44-1.07.44T8.94 22z',
                'w' => .623,
                'vw' => 20
            ],
            'angle-left' => [
                'p' => 'm2 14.94 8.5-8.5Q10.94 6 11.56 6t1.07.44L14 7.88q.44.43.44 1.06T14 10l-6 6 6 6q.31.32.4.69t0 .78-.4.66l-1.37 1.43q-.44.44-1.07.44t-1.06-.44L2 17.06q-.44-.43-.44-1.06T2 14.94z',
                'w' => .5,
                'vw' => 16,
            ],
            'angle-right' => [
                'p' => 'm14 17.06-8.5 8.5q-.44.44-1.06.44t-1.07-.44L2 24.13q-.44-.44-.44-1.07T2 22l6-6-6-6q-.44-.44-.44-1.06T2 7.87l1.37-1.43Q3.81 6 4.44 6t1.06.44l8.5 8.5q.44.43.44 1.06T14 17.06z',
                'w' => .5,
                'vw' => 16,
            ],
            'angle-up' => [
                'p' => 'm11.06 10 8.5 8.5q.44.44.44 1.06t-.44 1.07L18.13 22q-.44.44-1.07.44T16 22l-6-6-6 6q-.44.44-1.06.44T1.87 22L.43 20.63Q0 20.19 0 19.56t.43-1.06l8.5-8.5q.44-.44 1.07-.44t1.06.44z',
                'w' => .625,
                'vw' => 20
            ],
            'angles-down' => [
                'p' => 'M246.6 470.6c-12.5 12.5-32.8 12.5-45.3 0l-160-160c-12.5-12.5-12.5-32.8 0-45.3s32.8-12.5 45.3 0L224 402.7 361.4 265.4c12.5-12.5 32.8-12.5 45.3 0s12.5 32.8 0 45.3l-160 160zm160-352l-160 160c-12.5 12.5-32.8 12.5-45.3 0l-160-160c-12.5-12.5-12.5-32.8 0-45.3s32.8-12.5 45.3 0L224 210.7 361.4 73.4c12.5-12.5 32.8-12.5 45.3 0s12.5 32.8 0 45.3z',
                'vw' => 448,
                'vh' => 512
            ],
            'angles-left' => [
                'p' => 'M41.4 233.4c-12.5 12.5-12.5 32.8 0 45.3l160 160c12.5 12.5 32.8 12.5 45.3 0s12.5-32.8 0-45.3L109.3 256 246.6 118.6c12.5-12.5 12.5-32.8 0-45.3s-32.8-12.5-45.3 0l-160 160zm352-160l-160 160c-12.5 12.5-12.5 32.8 0 45.3l160 160c12.5 12.5 32.8 12.5 45.3 0s12.5-32.8 0-45.3L301.3 256 438.6 118.6c12.5-12.5 12.5-32.8 0-45.3s-32.8-12.5-45.3 0z',
                'vw' => 512,
                'vh' => 512
            ],
            'angles-right' => [
                'p' => 'M470.6 278.6c12.5-12.5 12.5-32.8 0-45.3l-160-160c-12.5-12.5-32.8-12.5-45.3 0s-12.5 32.8 0 45.3L402.7 256 265.4 393.4c-12.5 12.5-12.5 32.8 0 45.3s32.8 12.5 45.3 0l160-160zm-352 160l160-160c12.5-12.5 12.5-32.8 0-45.3l-160-160c-12.5-12.5-32.8-12.5-45.3 0s-12.5 32.8 0 45.3L210.7 256 73.4 393.4c-12.5 12.5-12.5 32.8 0 45.3s32.8 12.5 45.3 0z',
                'vw' => 512,
                'vh' => 512
            ],
            'angles-up' => [
                'p' => 'M246.6 41.4c-12.5-12.5-32.8-12.5-45.3 0l-160 160c-12.5 12.5-12.5 32.8 0 45.3s32.8 12.5 45.3 0L224 109.3 361.4 246.6c12.5 12.5 32.8 12.5 45.3 0s12.5-32.8 0-45.3l-160-160zm160 352l-160-160c-12.5-12.5-32.8-12.5-45.3 0l-160 160c-12.5 12.5-12.5 32.8 0 45.3s32.8 12.5 45.3 0L224 301.3 361.4 438.6c12.5 12.5 32.8 12.5 45.3 0s12.5-32.8 0-45.3z',
                'vw' => 448,
                'vh' => 512
            ],
            'angry' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zM8.5 15q0 .81.6 1.4t1.4.6 1.4-.6.6-1.4q0-.13-.13-.5h.13q.75 0 .94-.69.06-.25.06-.44t-.1-.34-.24-.28-.35-.19l-5-1.5q-.43-.12-.78.06t-.47.57q-.06.25-.03.43t.1.35.22.28.4.19l1.88.62q-.63.57-.63 1.44zM19 24.63q.38.5.94.37t.75-.63-.19-1Q18.56 21 15.5 21t-5.06 2.38q-.13.18-.2.43t0 .44.2.38.31.28.4.12.44-.06.41-.35q1.31-1.56 3.5-1.56t3.5 1.56zm4.81-11.7q.38-.12.57-.46t.06-.78q-.06-.25-.25-.44t-.44-.22-.5.03l-5 1.5q-.44.13-.62.47t-.07.78q.2.69.94.69h.13q-.13.31-.13.5 0 .81.6 1.4t1.4.6 1.4-.6.6-1.4q0-.87-.62-1.44z',
            'ankh' => [
                'p' => 'M18.5 16q.63 0 1.06.44T20 17.5v2q0 .63-.44 1.06T18.5 21h-6v9.5q0 .63-.44 1.06T11 32H9q-.63 0-1.06-.44T7.5 30.5V21h-6q-.63 0-1.07-.44T0 19.5v-2q0-.62.43-1.06T1.5 16h2.8Q2 12.31 2 9q0-4.06 2.3-6.53T10 0t5.69 2.47T18 9q0 3.31-2.31 7h2.81zM10 5Q8.62 5 7.81 6.06T7 9q0 1.5.94 3.28T10 15.31q.69-.75 1.34-1.78t1.16-2.28T13 9q0-1.87-.81-2.94T9.99 5z',
                'w' => .625,
                'vw' => 20
            ],
            'apple-alt' => [
                'p' => 'M21.94 8.06q2.44.44 4 2.63 1.37 1.94 1.87 4.87.44 2.7-.06 5.38-.75 4.44-3 7.44Q22.06 32 18 32q-1 0-2.19-.62-.81-.5-1.81-.5t-1.81.5q-1.2.62-2.2.62-4.06 0-6.74-3.62-2.25-3-3-7.44-.5-2.69-.07-5.38.5-2.93 1.88-4.87 1.56-2.19 4-2.63 1.5-.25 4.12.44 2.25.63 3.82 1.5 1.56-.87 3.81-1.5 2.63-.69 4.13-.44zM18.5 5.5q-.88.81-2.31 1.19-1 .31-2.2.31l-.93-.06q-.12-.88 0-2 .25-2.25 1.44-3.44.87-.81 2.31-1.19Q17.81 0 19 0l.94.06L20 1q0 1.19-.31 2.19-.38 1.44-1.19 2.31z',
                'w' => .875,
                'vw' => 28
            ],
            'apple-whole' => 'apple-alt',
            'archive' => 'M2 28V10h28v18q0 .81-.6 1.4T28 30H4q-.81 0-1.4-.6T2 28zm10-13.25v.5q0 .75.75.75h6.5q.75 0 .75-.75v-.5q0-.75-.75-.75h-6.5q-.75 0-.75.75zM30 2q.81 0 1.4.6T32 4v3q0 .44-.28.72T31 8H1q-.44 0-.72-.28T0 7V4q0-.81.6-1.4T2 2h28z',
            'archway' => [
                'p' => 'M35 28q.44 0 .72.28T36 29v2q0 .44-.28.72T35 32H25q-.44 0-.72-.28T24 31V20q0-2.5-1.75-4.25T18 14t-4.25 1.75T12 20v11q0 .44-.28.72T11 32H1q-.44 0-.72-.28T0 31v-2q0-.44.28-.72T1 28h1V6h32v22h1zm0-28q.44 0 .72.28T36 1v2q0 .44-.28.72T35 4H1q-.44 0-.72-.28T0 3V1Q0 .56.28.28T1 0h34z',
                'w' => 1.125,
                'vw' => 36
            ],
            'arrow-alt-circle-down' => 'M31.5 16q0 6.44-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5t10.97 4.53T31.5 16zM13.25 8.75V16H8.81q-.5 0-.69.47t.2.78l7.18 7.19q.5.5 1 0l7.19-7.19q.37-.31.19-.78t-.7-.47h-4.43V8.75Q18.75 8 18 8h-4q-.75 0-.75.75z',
            'arrow-alt-circle-left' => 'M16 31.5q-6.44 0-10.97-4.53T.5 16 5.03 5.03 16 .5t10.97 4.53T31.5 16t-4.53 10.97T16 31.5zm7.25-18.25H16V8.81q0-.5-.47-.69t-.78.2L7.56 15.5q-.5.5 0 1l7.19 7.19q.31.37.78.19t.47-.7v-4.43h7.25q.75 0 .75-.75v-4q0-.75-.75-.75z',
            'arrow-alt-circle-right' => 'M16 .5q6.44 0 10.97 4.53T31.5 16t-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5zM8.75 18.75H16v4.44q0 .5.47.69t.78-.2l7.19-7.18q.5-.5 0-1l-7.19-7.19q-.31-.37-.78-.19t-.47.7v4.43H8.75Q8 13.25 8 14v4q0 .75.75.75z',
            'arrow-alt-circle-up' => 'M.5 16q0-6.44 4.53-10.97T16 .5t10.97 4.53T31.5 16t-4.53 10.97T16 31.5 5.03 26.97.5 16zm18.25 7.25V16h4.44q.5 0 .69-.47t-.2-.78L16.5 7.56q-.5-.5-1 0l-7.19 7.19q-.37.31-.19.78t.7.47h4.43v7.25q0 .75.75.75h4q.75 0 .75-.75z',
            'arrow-circle-down' => 'M31.5 16q0 6.44-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5t10.97 4.53T31.5 16zm-9-1.81L18 18.94V7.5q0-.63-.44-1.06T16.5 6h-1q-.63 0-1.06.44T14 7.5v11.44l-4.5-4.75q-.44-.44-1.1-.44t-1.09.44l-.69.69q-.44.43-.44 1.06T6.62 17l8.32 8.25q.43.44 1.06.44t1.06-.44L25.37 17q.44-.44.44-1.06t-.44-1.06l-.68-.7q-.44-.43-1.1-.43t-1.1.44z',
            'arrow-circle-left' => 'M16 31.5q-6.44 0-10.97-4.53T.5 16 5.03 5.03 16 .5t10.97 4.53T31.5 16t-4.53 10.97T16 31.5zm1.81-9L13.06 18H24.5q.63 0 1.06-.44T26 16.5v-1q0-.63-.44-1.06T24.5 14H13.06l4.75-4.5q.44-.44.44-1.1t-.44-1.09l-.68-.69q-.44-.44-1.07-.44T15 6.62l-8.25 8.32q-.44.43-.44 1.06t.44 1.06L15 25.37q.44.44 1.06.44t1.07-.44l.68-.68q.44-.44.44-1.1t-.44-1.1z',
            'arrow-circle-right' => 'M16 .5q6.44 0 10.97 4.53T31.5 16t-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5zm-1.81 9 4.75 4.5H7.5q-.63 0-1.06.44T6 15.5v1q0 .63.44 1.06T7.5 18h11.44l-4.75 4.5q-.44.44-.44 1.1t.44 1.09l.69.69q.43.44 1.06.44t1.06-.44l8.25-8.31q.44-.44.44-1.07t-.44-1.06L17 6.63q-.44-.44-1.06-.44t-1.06.44l-.7.68q-.43.44-.43 1.1t.44 1.1z',
            'arrow-circle-up' => 'M.5 16q0-6.44 4.53-10.97T16 .5t10.97 4.53T31.5 16t-4.53 10.97T16 31.5 5.03 26.97.5 16zm9 1.81 4.5-4.75V24.5q0 .63.44 1.06t1.06.44h1q.63 0 1.06-.44T18 24.5V13.06l4.5 4.75q.44.44 1.1.44t1.09-.44l.69-.68q.44-.44.44-1.07T25.38 15l-8.31-8.25Q16.62 6.3 16 6.3t-1.06.44L6.63 15q-.44.44-.44 1.06t.44 1.07l.68.68q.44.44 1.1.44t1.1-.44z',
            'arrow-down' => [
                'p' => 'm25.81 13.88 1.38 1.43q.44.44.44 1.07t-.44 1.06L15.06 29.56Q14.63 30 14 30t-1.06-.44L.8 17.44Q.37 17 .37 16.38t.44-1.07l1.38-1.43q.43-.44 1.06-.41t1.06.47l7.2 7.5V3.5q0-.63.43-1.06T13 2h2q.63 0 1.06.44t.44 1.06v17.94l7.2-7.5q.43-.44 1.05-.47t1.07.4z',
                'w' => .875,
                'vw' => 28
            ],
            'arrow-down-1-9' => [
                'p' => 'M450.7 38c-8.3-6-19.1-7.7-28.8-4.4l-48 16c-16.8 5.6-25.8 23.7-20.2 40.5s23.7 25.8 40.5 20.2l5.9-2V160H384c-17.7 0-32 14.3-32 32s14.3 32 32 32h48 48c17.7 0 32-14.3 32-32s-14.3-32-32-32H464V64c0-10.3-4.9-19.9-13.3-26zM160 480c9 0 17.5-3.8 23.6-10.4l88-96c11.9-13 11.1-33.3-2-45.2s-33.3-11.1-45.2 2L192 365.7V64c0-17.7-14.3-32-32-32s-32 14.3-32 32V365.7L95.6 330.4c-11.9-13-32.2-13.9-45.2-2s-13.9 32.2-2 45.2l88 96C142.5 476.2 151 480 160 480zM418.3 307a32 32 0 1 1 27.4 57.9A32 32 0 1 1 418.3 307zM405.1 419.8l-6.8 9.2c-10.5 14.2-7.5 34.2 6.7 44.8s34.2 7.5 44.8-6.7l48.8-65.8c14-18.9 21.5-41.7 21.5-65.2c0-48.6-39.4-88-88-88s-88 39.4-88 88c0 39.2 25.6 72.4 61.1 83.8z',
                'vw' => 576,
                'vh' => 512
            ],
            'arrow-down-9-1' => [
                'p' => 'M160 480c9 0 17.5-3.8 23.6-10.4l88-96c11.9-13 11.1-33.3-2-45.2s-33.3-11.1-45.2 2L192 365.7V64c0-17.7-14.3-32-32-32s-32 14.3-32 32V365.7L95.6 330.4c-11.9-13-32.2-13.9-45.2-2s-13.9 32.2-2 45.2l88 96C142.5 476.2 151 480 160 480zM450.7 294c-8.3-6-19.1-7.7-28.8-4.4l-48 16c-16.8 5.6-25.8 23.7-20.2 40.5s23.7 25.8 40.5 20.2l5.9-2V416H384c-17.7 0-32 14.3-32 32s14.3 32 32 32h48 48c17.7 0 32-14.3 32-32s-14.3-32-32-32H464V320c0-10.3-4.9-19.9-13.3-26zM418.3 91a32 32 0 1 1 27.4 57.9A32 32 0 1 1 418.3 91zM405.1 203.8l-6.8 9.2c-10.5 14.2-7.5 34.2 6.7 44.8s34.2 7.5 44.8-6.7l48.8-65.8c14-18.9 21.5-41.7 21.5-65.2c0-48.6-39.4-88-88-88s-88 39.4-88 88c0 39.2 25.6 72.4 61.1 83.8z',
                'vw' => 576,
                'vh' => 512
            ],
            'arrow-down-a-z' => [
                'p' => 'M183.6 469.6C177.5 476.2 169 480 160 480s-17.5-3.8-23.6-10.4l-88-96c-11.9-13-11.1-33.3 2-45.2s33.3-11.1 45.2 2L128 365.7V64c0-17.7 14.3-32 32-32s32 14.3 32 32V365.7l32.4-35.4c11.9-13 32.2-13.9 45.2-2s13.9 32.2 2 45.2l-88 96zM320 320c0-17.7 14.3-32 32-32H480c12.9 0 24.6 7.8 29.6 19.8s2.2 25.7-6.9 34.9L429.3 416H480c17.7 0 32 14.3 32 32s-14.3 32-32 32H352c-12.9 0-24.6-7.8-29.6-19.8s-2.2-25.7 6.9-34.9L402.7 352H352c-17.7 0-32-14.3-32-32zM416 32c12.1 0 23.2 6.8 28.6 17.7l64 128 16 32c7.9 15.8 1.5 35-14.3 42.9s-35 1.5-42.9-14.3L460.2 224H371.8l-7.2 14.3c-7.9 15.8-27.1 22.2-42.9 14.3s-22.2-27.1-14.3-42.9l16-32 64-128C392.8 38.8 403.9 32 416 32zM395.8 176h40.4L416 135.6 395.8 176z',
                'vw' => 576,
                'vh' => 512
            ],
            'arrow-down-short-wide' => [
                'p' => 'M151.6 469.6C145.5 476.2 137 480 128 480s-17.5-3.8-23.6-10.4l-88-96c-11.9-13-11.1-33.3 2-45.2s33.3-11.1 45.2 2L96 365.7V64c0-17.7 14.3-32 32-32s32 14.3 32 32V365.7l32.4-35.4c11.9-13 32.2-13.9 45.2-2s13.9 32.2 2 45.2l-88 96zM320 32h32c17.7 0 32 14.3 32 32s-14.3 32-32 32H320c-17.7 0-32-14.3-32-32s14.3-32 32-32zm0 128h96c17.7 0 32 14.3 32 32s-14.3 32-32 32H320c-17.7 0-32-14.3-32-32s14.3-32 32-32zm0 128H480c17.7 0 32 14.3 32 32s-14.3 32-32 32H320c-17.7 0-32-14.3-32-32s14.3-32 32-32zm0 128H544c17.7 0 32 14.3 32 32s-14.3 32-32 32H320c-17.7 0-32-14.3-32-32s14.3-32 32-32z',
                'vw' => 576,
                'vh' => 512
            ],
            'arrow-down-up-across-line' => [
                'p' => 'M137.4 502.6c12.5 12.5 32.8 12.5 45.3 0l96-96c12.5-12.5 12.5-32.8 0-45.3s-32.8-12.5-45.3 0L192 402.7V288H544c17.7 0 32-14.3 32-32s-14.3-32-32-32H448V109.3l41.4 41.4c12.5 12.5 32.8 12.5 45.3 0s12.5-32.8 0-45.3l-96-96c-12.5-12.5-32.8-12.5-45.3 0l-96 96c-12.5 12.5-12.5 32.8 0 45.3s32.8 12.5 45.3 0L384 109.3V224H192 128 32c-17.7 0-32 14.3-32 32s14.3 32 32 32h96V402.7L86.6 361.4c-12.5-12.5-32.8-12.5-45.3 0s-12.5 32.8 0 45.3l96 96zM128 192h64V64c0-17.7-14.3-32-32-32s-32 14.3-32 32V192zM448 320H384V448c0 17.7 14.3 32 32 32s32-14.3 32-32V320z',
                'vw' => 576,
                'vh' => 512
            ],
            'arrow-down-up-lock' => [
                'p' => 'M150.6 502.6l96-96c12.5-12.5 12.5-32.8 0-45.3s-32.8-12.5-45.3 0L160 402.7V288H416V272c0-17.2 3.9-33.5 10.8-48H352V109.3l41.4 41.4c12.5 12.5 32.8 12.5 45.3 0s12.5-32.8 0-45.3l-96-96c-6-6-14.1-9.4-22.6-9.4s-16.6 3.4-22.6 9.4l-96 96c-12.5 12.5-12.5 32.8 0 45.3s32.8 12.5 45.3 0L288 109.3V224l-128 0H96l-64 0c-17.7 0-32 14.3-32 32s14.3 32 32 32H96V402.7L54.6 361.4c-12.5-12.5-32.8-12.5-45.3 0s-12.5 32.8 0 45.3l96 96c12.5 12.5 32.8 12.5 45.3 0zM160 192V64c0-17.7-14.3-32-32-32s-32 14.3-32 32V192h64zM288 320V448c0 17.7 14.3 32 32 32s32-14.3 32-32V320H288zm240-80c17.7 0 32 14.3 32 32v48H496V272c0-17.7 14.3-32 32-32zm-80 32v48c-17.7 0-32 14.3-32 32V480c0 17.7 14.3 32 32 32H608c17.7 0 32-14.3 32-32V352c0-17.7-14.3-32-32-32V272c0-44.2-35.8-80-80-80s-80 35.8-80 80z',
                'vw' => 640,
                'vh' => 512
            ],
            'arrow-down-wide-short' => [
                'p' => 'M151.6 469.6C145.5 476.2 137 480 128 480s-17.5-3.8-23.6-10.4l-88-96c-11.9-13-11.1-33.3 2-45.2s33.3-11.1 45.2 2L96 365.7V64c0-17.7 14.3-32 32-32s32 14.3 32 32V365.7l32.4-35.4c11.9-13 32.2-13.9 45.2-2s13.9 32.2 2 45.2l-88 96zM320 480c-17.7 0-32-14.3-32-32s14.3-32 32-32h32c17.7 0 32 14.3 32 32s-14.3 32-32 32H320zm0-128c-17.7 0-32-14.3-32-32s14.3-32 32-32h96c17.7 0 32 14.3 32 32s-14.3 32-32 32H320zm0-128c-17.7 0-32-14.3-32-32s14.3-32 32-32H480c17.7 0 32 14.3 32 32s-14.3 32-32 32H320zm0-128c-17.7 0-32-14.3-32-32s14.3-32 32-32H544c17.7 0 32 14.3 32 32s-14.3 32-32 32H320z',
                'vw' => 576,
                'vh' => 512
            ],
            'arrow-down-z-a' => [
                'p' => 'M183.6 469.6C177.5 476.2 169 480 160 480s-17.5-3.8-23.6-10.4l-88-96c-11.9-13-11.1-33.3 2-45.2s33.3-11.1 45.2 2L128 365.7V64c0-17.7 14.3-32 32-32s32 14.3 32 32V365.7l32.4-35.4c11.9-13 32.2-13.9 45.2-2s13.9 32.2 2 45.2l-88 96zM320 64c0-17.7 14.3-32 32-32H480c12.9 0 24.6 7.8 29.6 19.8s2.2 25.7-6.9 34.9L429.3 160H480c17.7 0 32 14.3 32 32s-14.3 32-32 32H352c-12.9 0-24.6-7.8-29.6-19.8s-2.2-25.7 6.9-34.9L402.7 96H352c-17.7 0-32-14.3-32-32zm96 192c12.1 0 23.2 6.8 28.6 17.7l64 128 16 32c7.9 15.8 1.5 35-14.3 42.9s-35 1.5-42.9-14.3L460.2 448H371.8l-7.2 14.3c-7.9 15.8-27.1 22.2-42.9 14.3s-22.2-27.1-14.3-42.9l16-32 64-128c5.4-10.8 16.5-17.7 28.6-17.7zM395.8 400h40.4L416 359.6 395.8 400z',
                'vw' => 576,
                'vh' => 512
            ],
            'arrow-left' => [
                'p' => 'm16.13 27.81-1.44 1.38q-.44.44-1.07.44t-1.06-.44L.44 17.06Q0 16.63 0 16t.44-1.06L12.56 2.8q.44-.44 1.06-.44t1.07.44l1.44 1.38q.43.43.4 1.06t-.47 1.06l-7.5 7.2H26.5q.63 0 1.06.43T28 15v2q0 .63-.44 1.06t-1.06.44H8.56l7.5 7.2q.44.43.47 1.05t-.4 1.07z',
                'w' => .875,
                'vw' => 28
            ],
            'arrow-pointer' => [
                'p' => 'M0 55.2V426c0 12.2 9.9 22 22 22c6.3 0 12.4-2.7 16.6-7.5L121.2 346l58.1 116.3c7.9 15.8 27.1 22.2 42.9 14.3s22.2-27.1 14.3-42.9L179.8 320H297.9c12.2 0 22.1-9.9 22.1-22.1c0-6.3-2.7-12.3-7.4-16.5L38.6 37.9C34.3 34.1 28.9 32 23.2 32C10.4 32 0 42.4 0 55.2z',
                'vw' => 320,
                'vh' => 512
            ],
            'arrow-right' => [
                'p' => 'M11.88 4.19 13.3 2.8q.44-.44 1.07-.44t1.06.44l12.12 12.13q.44.44.44 1.06t-.44 1.06L15.44 29.2q-.44.44-1.06.44t-1.07-.44l-1.44-1.38q-.43-.43-.4-1.06t.47-1.06l7.5-7.19H1.5q-.63 0-1.06-.44T0 17v-2q0-.63.44-1.06t1.06-.44h17.94l-7.5-7.19q-.44-.44-.47-1.06t.4-1.07z',
                'w' => .875,
                'vw' => 28
            ],
            'arrow-right-arrow-left' => [
                'p' => 'M438.6 150.6c12.5-12.5 12.5-32.8 0-45.3l-96-96c-12.5-12.5-32.8-12.5-45.3 0s-12.5 32.8 0 45.3L338.7 96 32 96C14.3 96 0 110.3 0 128s14.3 32 32 32l306.7 0-41.4 41.4c-12.5 12.5-12.5 32.8 0 45.3s32.8 12.5 45.3 0l96-96zm-333.3 352c12.5 12.5 32.8 12.5 45.3 0s12.5-32.8 0-45.3L109.3 416 416 416c17.7 0 32-14.3 32-32s-14.3-32-32-32l-306.7 0 41.4-41.4c12.5-12.5 12.5-32.8 0-45.3s-32.8-12.5-45.3 0l-96 96c-12.5 12.5-12.5 32.8 0 45.3l96 96z',
                'vw' => 448,
                'vh' => 512
            ],
            'arrow-right-from-bracket' => [
                'p' => 'M502.6 278.6c12.5-12.5 12.5-32.8 0-45.3l-128-128c-12.5-12.5-32.8-12.5-45.3 0s-12.5 32.8 0 45.3L402.7 224 192 224c-17.7 0-32 14.3-32 32s14.3 32 32 32l210.7 0-73.4 73.4c-12.5 12.5-12.5 32.8 0 45.3s32.8 12.5 45.3 0l128-128zM160 96c17.7 0 32-14.3 32-32s-14.3-32-32-32L96 32C43 32 0 75 0 128L0 384c0 53 43 96 96 96l64 0c17.7 0 32-14.3 32-32s-14.3-32-32-32l-64 0c-17.7 0-32-14.3-32-32l0-256c0-17.7 14.3-32 32-32l64 0z',
                'vw' => 512,
                'vh' => 512
            ],
            'arrow-right-to-bracket' => [
                'p' => 'M352 96l64 0c17.7 0 32 14.3 32 32l0 256c0 17.7-14.3 32-32 32l-64 0c-17.7 0-32 14.3-32 32s14.3 32 32 32l64 0c53 0 96-43 96-96l0-256c0-53-43-96-96-96l-64 0c-17.7 0-32 14.3-32 32s14.3 32 32 32zm-9.4 182.6c12.5-12.5 12.5-32.8 0-45.3l-128-128c-12.5-12.5-32.8-12.5-45.3 0s-12.5 32.8 0 45.3L242.7 224 32 224c-17.7 0-32 14.3-32 32s14.3 32 32 32l210.7 0-73.4 73.4c-12.5 12.5-12.5 32.8 0 45.3s32.8 12.5 45.3 0l128-128z',
                'vw' => 512,
                'vh' => 512
            ],
            'arrow-right-to-city' => [
                'p' => 'M288 48c0-26.5 21.5-48 48-48h96c26.5 0 48 21.5 48 48V192h40V120c0-13.3 10.7-24 24-24s24 10.7 24 24v72h24c26.5 0 48 21.5 48 48V464c0 26.5-21.5 48-48 48H432 336c-26.5 0-48-21.5-48-48V48zm64 32v32c0 8.8 7.2 16 16 16h32c8.8 0 16-7.2 16-16V80c0-8.8-7.2-16-16-16H368c-8.8 0-16 7.2-16 16zm16 80c-8.8 0-16 7.2-16 16v32c0 8.8 7.2 16 16 16h32c8.8 0 16-7.2 16-16V176c0-8.8-7.2-16-16-16H368zM352 272v32c0 8.8 7.2 16 16 16h32c8.8 0 16-7.2 16-16V272c0-8.8-7.2-16-16-16H368c-8.8 0-16 7.2-16 16zm176-16c-8.8 0-16 7.2-16 16v32c0 8.8 7.2 16 16 16h32c8.8 0 16-7.2 16-16V272c0-8.8-7.2-16-16-16H528zM512 368v32c0 8.8 7.2 16 16 16h32c8.8 0 16-7.2 16-16V368c0-8.8-7.2-16-16-16H528c-8.8 0-16 7.2-16 16zM166.6 153.4l80 80c12.5 12.5 12.5 32.8 0 45.3l-80 80c-12.5 12.5-32.8 12.5-45.3 0s-12.5-32.8 0-45.3L146.7 288H32c-17.7 0-32-14.3-32-32s14.3-32 32-32H146.7l-25.4-25.4c-12.5-12.5-12.5-32.8 0-45.3s32.8-12.5 45.3 0z',
                'vw' => 640,
                'vh' => 512
            ],
            'arrow-rotate-left' => [
                'p' => 'M125.7 160H176c17.7 0 32 14.3 32 32s-14.3 32-32 32H48c-17.7 0-32-14.3-32-32V64c0-17.7 14.3-32 32-32s32 14.3 32 32v51.2L97.6 97.6c87.5-87.5 229.3-87.5 316.8 0s87.5 229.3 0 316.8s-229.3 87.5-316.8 0c-12.5-12.5-12.5-32.8 0-45.3s32.8-12.5 45.3 0c62.5 62.5 163.8 62.5 226.3 0s62.5-163.8 0-226.3s-163.8-62.5-226.3 0L125.7 160z',
                'vw' => 512,
                'vh' => 512
            ],
            'arrow-rotate-right' => [
                'p' => 'M386.3 160H336c-17.7 0-32 14.3-32 32s14.3 32 32 32H464c17.7 0 32-14.3 32-32V64c0-17.7-14.3-32-32-32s-32 14.3-32 32v51.2L414.4 97.6c-87.5-87.5-229.3-87.5-316.8 0s-87.5 229.3 0 316.8s229.3 87.5 316.8 0c12.5-12.5 12.5-32.8 0-45.3s-32.8-12.5-45.3 0c-62.5 62.5-163.8 62.5-226.3 0s-62.5-163.8 0-226.3s163.8-62.5 226.3 0L386.3 160z',
                'vw' => 512,
                'vh' => 512
            ],
            'arrow-trend-down' => [
                'p' => 'M384 352c-17.7 0-32 14.3-32 32s14.3 32 32 32H544c17.7 0 32-14.3 32-32V224c0-17.7-14.3-32-32-32s-32 14.3-32 32v82.7L342.6 137.4c-12.5-12.5-32.8-12.5-45.3 0L192 242.7 54.6 105.4c-12.5-12.5-32.8-12.5-45.3 0s-12.5 32.8 0 45.3l160 160c12.5 12.5 32.8 12.5 45.3 0L320 205.3 466.7 352H384z',
                'vw' => 576,
                'vh' => 512
            ],
            'arrow-trend-up' => [
                'p' => 'M384 160c-17.7 0-32-14.3-32-32s14.3-32 32-32H544c17.7 0 32 14.3 32 32V288c0 17.7-14.3 32-32 32s-32-14.3-32-32V205.3L342.6 374.6c-12.5 12.5-32.8 12.5-45.3 0L192 269.3 54.6 406.6c-12.5 12.5-32.8 12.5-45.3 0s-12.5-32.8 0-45.3l160-160c12.5-12.5 32.8-12.5 45.3 0L320 306.7 466.7 160H384z',
                'vw' => 576,
                'vh' => 512
            ],
            'arrow-turn-down' => [
                'p' => 'M32 64C14.3 64 0 49.7 0 32S14.3 0 32 0l96 0c53 0 96 43 96 96l0 306.7 73.4-73.4c12.5-12.5 32.8-12.5 45.3 0s12.5 32.8 0 45.3l-128 128c-12.5 12.5-32.8 12.5-45.3 0l-128-128c-12.5-12.5-12.5-32.8 0-45.3s32.8-12.5 45.3 0L160 402.7 160 96c0-17.7-14.3-32-32-32L32 64z',
                'vw' => 384,
                'vh' => 512
            ],
            'arrow-turn-up' => [
                'p' => 'M32 448c-17.7 0-32 14.3-32 32s14.3 32 32 32l96 0c53 0 96-43 96-96l0-306.7 73.4 73.4c12.5 12.5 32.8 12.5 45.3 0s12.5-32.8 0-45.3l-128-128c-12.5-12.5-32.8-12.5-45.3 0l-128 128c-12.5 12.5-12.5 32.8 0 45.3s32.8 12.5 45.3 0L160 109.3 160 416c0 17.7-14.3 32-32 32l-96 0z',
                'vw' => 384,
                'vh' => 512
            ],
            'arrow-up' => [
                'p' => 'M2.19 18.13.8 16.68q-.44-.44-.44-1.07t.44-1.06L12.94 2.44Q13.38 2 14 2t1.06.44L27.2 14.56q.44.44.44 1.06t-.44 1.07l-1.38 1.37q-.43.5-1.06.47t-1.06-.47l-7.19-7.5V28.5q0 .63-.44 1.06T15 30h-2q-.63 0-1.06-.44t-.44-1.06V10.56l-7.19 7.5q-.44.44-1.06.47t-1.07-.4z',
                'w' => .873,
                'vw' => 28
            ],
            'arrow-up-1-9' => [
                'p' => 'M450.7 38c8.3 6 13.3 15.7 13.3 26v96h16c17.7 0 32 14.3 32 32s-14.3 32-32 32H432 384c-17.7 0-32-14.3-32-32s14.3-32 32-32h16V108.4l-5.9 2c-16.8 5.6-34.9-3.5-40.5-20.2s3.5-34.9 20.2-40.5l48-16c9.8-3.3 20.5-1.6 28.8 4.4zM160 32c9 0 17.5 3.8 23.6 10.4l88 96c11.9 13 11.1 33.3-2 45.2s-33.3 11.1-45.2-2L192 146.3V448c0 17.7-14.3 32-32 32s-32-14.3-32-32V146.3L95.6 181.6c-11.9 13-32.2 13.9-45.2 2s-13.9-32.2-2-45.2l88-96C142.5 35.8 151 32 160 32zM445.7 364.9A32 32 0 1 0 418.3 307a32 32 0 1 0 27.4 57.9zm-40.7 54.9C369.6 408.4 344 375.2 344 336c0-48.6 39.4-88 88-88s88 39.4 88 88c0 23.5-7.5 46.3-21.5 65.2L449.7 467c-10.5 14.2-30.6 17.2-44.8 6.7s-17.2-30.6-6.7-44.8l6.8-9.2z',
                'vw' => 576,
                'vh' => 512
            ],
            'arrow-up-9-1' => [
                'p' => 'M160 32c9 0 17.5 3.8 23.6 10.4l88 96c11.9 13 11.1 33.3-2 45.2s-33.3 11.1-45.2-2L192 146.3V448c0 17.7-14.3 32-32 32s-32-14.3-32-32V146.3L95.6 181.6c-11.9 13-32.2 13.9-45.2 2s-13.9-32.2-2-45.2l88-96C142.5 35.8 151 32 160 32zM450.7 294c8.3 6 13.3 15.7 13.3 26v96h16c17.7 0 32 14.3 32 32s-14.3 32-32 32H432 384c-17.7 0-32-14.3-32-32s14.3-32 32-32h16V364.4l-5.9 2c-16.8 5.6-34.9-3.5-40.5-20.2s3.5-34.9 20.2-40.5l48-16c9.8-3.3 20.5-1.6 28.8 4.4zm-5-145.1A32 32 0 1 0 418.3 91a32 32 0 1 0 27.4 57.9zm-40.7 54.9C369.6 192.4 344 159.2 344 120c0-48.6 39.4-88 88-88s88 39.4 88 88c0 23.5-7.5 46.3-21.5 65.2L449.7 251c-10.5 14.2-30.6 17.2-44.8 6.7s-17.2-30.6-6.7-44.8l6.8-9.2z',
                'vw' => 576,
                'vh' => 512
            ],
            'arrow-up-a-z' => [
                'p' => 'M183.6 42.4C177.5 35.8 169 32 160 32s-17.5 3.8-23.6 10.4l-88 96c-11.9 13-11.1 33.3 2 45.2s33.3 11.1 45.2-2L128 146.3V448c0 17.7 14.3 32 32 32s32-14.3 32-32V146.3l32.4 35.4c11.9 13 32.2 13.9 45.2 2s13.9-32.2 2-45.2l-88-96zM320 320c0 17.7 14.3 32 32 32h50.7l-73.4 73.4c-9.2 9.2-11.9 22.9-6.9 34.9s16.6 19.8 29.6 19.8H480c17.7 0 32-14.3 32-32s-14.3-32-32-32H429.3l73.4-73.4c9.2-9.2 11.9-22.9 6.9-34.9s-16.6-19.8-29.6-19.8H352c-17.7 0-32 14.3-32 32zM416 32c-12.1 0-23.2 6.8-28.6 17.7l-64 128-16 32c-7.9 15.8-1.5 35 14.3 42.9s35 1.5 42.9-14.3l7.2-14.3h88.4l7.2 14.3c7.9 15.8 27.1 22.2 42.9 14.3s22.2-27.1 14.3-42.9l-16-32-64-128C439.2 38.8 428.1 32 416 32zM395.8 176L416 135.6 436.2 176H395.8z',
                'vw' => 576,
                'vh' => 512
            ],
            'arrow-up-from-bracket' => [
                'p' => 'M246.6 9.4c-12.5-12.5-32.8-12.5-45.3 0l-128 128c-12.5 12.5-12.5 32.8 0 45.3s32.8 12.5 45.3 0L192 109.3V320c0 17.7 14.3 32 32 32s32-14.3 32-32V109.3l73.4 73.4c12.5 12.5 32.8 12.5 45.3 0s12.5-32.8 0-45.3l-128-128zM64 352c0-17.7-14.3-32-32-32s-32 14.3-32 32v64c0 53 43 96 96 96H352c53 0 96-43 96-96V352c0-17.7-14.3-32-32-32s-32 14.3-32 32v64c0 17.7-14.3 32-32 32H96c-17.7 0-32-14.3-32-32V352z',
                'vw' => 448,
                'vh' => 512
            ],
            'arrow-up-from-ground-water' => [
                'p' => 'M288 352c17.7 0 32-14.3 32-32V109.3l25.4 25.4c12.5 12.5 32.8 12.5 45.3 0s12.5-32.8 0-45.3l-80-80c-12.5-12.5-32.8-12.5-45.3 0l-80 80c-12.5 12.5-12.5 32.8 0 45.3s32.8 12.5 45.3 0L256 109.3V320c0 17.7 14.3 32 32 32zm-18.5 69.9C247 437.4 219.5 448 192 448c-26.9 0-55.3-10.8-77.4-26.1l0 0c-11.9-8.5-28.1-7.8-39.2 1.7c-14.4 11.9-32.5 21-50.6 25.2c-17.2 4-27.9 21.2-23.9 38.4s21.2 27.9 38.4 23.9c24.5-5.7 44.9-16.5 58.2-25C126.5 501.7 159 512 192 512c31.9 0 60.6-9.9 80.4-18.9c5.8-2.7 11.1-5.3 15.6-7.7c4.5 2.4 9.7 5.1 15.6 7.7c19.8 9 48.5 18.9 80.4 18.9c33 0 65.5-10.3 94.5-25.8c13.4 8.4 33.7 19.3 58.2 25c17.2 4 34.4-6.7 38.4-23.9s-6.7-34.4-23.9-38.4c-18.1-4.2-36.2-13.3-50.6-25.2c-11.1-9.4-27.3-10.1-39.2-1.7l0 0C439.4 437.2 410.9 448 384 448c-27.5 0-55-10.6-77.5-26.1c-11.1-7.9-25.9-7.9-37 0zM192 192H48c-26.5 0-48 21.5-48 48V425c5.3-3.1 11.2-5.4 17.5-6.9c13.1-3.1 26.7-9.8 37.3-18.6c22.2-18.7 54.3-20.1 78.1-3.4c18 12.4 40.1 20.3 59.1 20.3V192zm384 48c0-26.5-21.5-48-48-48H384V416.5h0c19 0 41.2-7.9 59.2-20.3c23.8-16.7 55.8-15.3 78.1 3.4c10.6 8.8 24.2 15.6 37.3 18.6c6.3 1.5 12.1 3.8 17.5 6.9V240z',
                'vw' => 576,
                'vh' => 512
            ],
            'arrow-up-from-water-pump' => [
                'p' => 'M112 0C85.5 0 64 21.5 64 48V256H48c-26.5 0-48 21.5-48 48v96c0 8 2 15.6 5.4 22.2c3.8-1.7 7.8-3.1 12-4.1c13.1-3.1 26.7-9.8 37.3-18.6c22.2-18.7 54.3-20.1 78.1-3.4c18 12.4 40.1 20.3 59.2 20.3c21.1 0 42-8.5 59.2-20.3c22.1-15.5 51.6-15.5 73.7 0c18.4 12.7 39.6 20.3 59.2 20.3c19 0 41.2-7.9 59.2-20.3c23.8-16.7 55.8-15.3 78.1 3.4c10.6 8.8 24.2 15.6 37.3 18.6c4.2 1 8.2 2.4 12 4.1C574 415.6 576 408 576 400V304c0-26.5-21.5-48-48-48H480l0-146.7 25.4 25.4c12.5 12.5 32.8 12.5 45.3 0s12.5-32.8 0-45.3l-80-80c-12.5-12.5-32.8-12.5-45.3 0l-80 80c-12.5 12.5-12.5 32.8 0 45.3s32.8 12.5 45.3 0L416 109.3 416 256H288V48c0-26.5-21.5-48-48-48H112zM306.5 421.9c-11.1-7.9-25.9-7.9-37 0C247 437.4 219.5 448 192 448c-26.9 0-55.3-10.8-77.4-26.1l0 0c-11.9-8.5-28.1-7.8-39.2 1.7c-14.4 11.9-32.5 21-50.6 25.2c-17.2 4-27.9 21.2-23.9 38.4s21.2 27.9 38.4 23.9c24.5-5.7 44.9-16.5 58.2-25C126.5 501.7 159 512 192 512c31.9 0 60.6-9.9 80.4-18.9c5.8-2.7 11.1-5.3 15.6-7.7c4.5 2.4 9.7 5.1 15.6 7.7c19.8 9 48.5 18.9 80.4 18.9c33 0 65.5-10.3 94.5-25.8c13.4 8.4 33.7 19.3 58.2 25c17.2 4 34.4-6.7 38.4-23.9s-6.7-34.4-23.9-38.4c-18.1-4.2-36.2-13.3-50.6-25.2c-11.1-9.4-27.3-10.1-39.2-1.7l0 0C439.4 437.2 410.9 448 384 448c-27.5 0-55-10.6-77.5-26.1z',
                'vw' => 576,
                'vh' => 512
            ],
            'arrow-up-right-dots' => [
                'p' => 'M160 0c-17.7 0-32 14.3-32 32s14.3 32 32 32h50.7L9.4 265.4c-12.5 12.5-12.5 32.8 0 45.3s32.8 12.5 45.3 0L256 109.3V160c0 17.7 14.3 32 32 32s32-14.3 32-32V32c0-17.7-14.3-32-32-32H160zM576 80a48 48 0 1 0 -96 0 48 48 0 1 0 96 0zM448 208a48 48 0 1 0 -96 0 48 48 0 1 0 96 0zM400 384a48 48 0 1 0 0-96 48 48 0 1 0 0 96zm48 80a48 48 0 1 0 -96 0 48 48 0 1 0 96 0zm128 0a48 48 0 1 0 -96 0 48 48 0 1 0 96 0zM272 384a48 48 0 1 0 0-96 48 48 0 1 0 0 96zm48 80a48 48 0 1 0 -96 0 48 48 0 1 0 96 0zM144 512a48 48 0 1 0 0-96 48 48 0 1 0 0 96zM576 336a48 48 0 1 0 -96 0 48 48 0 1 0 96 0zm-48-80a48 48 0 1 0 0-96 48 48 0 1 0 0 96z',
                'vw' => 576,
                'vh' => 512
            ],
            'arrow-up-right-from-square' => [
                'p' => 'M320 0c-17.7 0-32 14.3-32 32s14.3 32 32 32h82.7L201.4 265.4c-12.5 12.5-12.5 32.8 0 45.3s32.8 12.5 45.3 0L448 109.3V192c0 17.7 14.3 32 32 32s32-14.3 32-32V32c0-17.7-14.3-32-32-32H320zM80 32C35.8 32 0 67.8 0 112V432c0 44.2 35.8 80 80 80H400c44.2 0 80-35.8 80-80V320c0-17.7-14.3-32-32-32s-32 14.3-32 32V432c0 8.8-7.2 16-16 16H80c-8.8 0-16-7.2-16-16V112c0-8.8 7.2-16 16-16H192c17.7 0 32-14.3 32-32s-14.3-32-32-32H80z',
                'vw' => 512,
                'vh' => 512
            ],
            'arrow-up-short-wide' => [
                'p' => 'M151.6 42.4C145.5 35.8 137 32 128 32s-17.5 3.8-23.6 10.4l-88 96c-11.9 13-11.1 33.3 2 45.2s33.3 11.1 45.2-2L96 146.3V448c0 17.7 14.3 32 32 32s32-14.3 32-32V146.3l32.4 35.4c11.9 13 32.2 13.9 45.2 2s13.9-32.2 2-45.2l-88-96zM320 32c-17.7 0-32 14.3-32 32s14.3 32 32 32h32c17.7 0 32-14.3 32-32s-14.3-32-32-32H320zm0 128c-17.7 0-32 14.3-32 32s14.3 32 32 32h96c17.7 0 32-14.3 32-32s-14.3-32-32-32H320zm0 128c-17.7 0-32 14.3-32 32s14.3 32 32 32H480c17.7 0 32-14.3 32-32s-14.3-32-32-32H320zm0 128c-17.7 0-32 14.3-32 32s14.3 32 32 32H544c17.7 0 32-14.3 32-32s-14.3-32-32-32H320z',
                'vw' => 576,
                'vh' => 512
            ],
            'arrow-up-wide-short' => [
                'p' => 'M151.6 42.4C145.5 35.8 137 32 128 32s-17.5 3.8-23.6 10.4l-88 96c-11.9 13-11.1 33.3 2 45.2s33.3 11.1 45.2-2L96 146.3V448c0 17.7 14.3 32 32 32s32-14.3 32-32V146.3l32.4 35.4c11.9 13 32.2 13.9 45.2 2s13.9-32.2 2-45.2l-88-96zM320 480h32c17.7 0 32-14.3 32-32s-14.3-32-32-32H320c-17.7 0-32 14.3-32 32s14.3 32 32 32zm0-128h96c17.7 0 32-14.3 32-32s-14.3-32-32-32H320c-17.7 0-32 14.3-32 32s14.3 32 32 32zm0-128H480c17.7 0 32-14.3 32-32s-14.3-32-32-32H320c-17.7 0-32 14.3-32 32s14.3 32 32 32zm0-128H544c17.7 0 32-14.3 32-32s-14.3-32-32-32H320c-17.7 0-32 14.3-32 32s14.3 32 32 32z',
                'vw' => 576,
                'vh' => 512
            ],
            'arrow-up-z-a' => [
                'p' => 'M183.6 42.4C177.5 35.8 169 32 160 32s-17.5 3.8-23.6 10.4l-88 96c-11.9 13-11.1 33.3 2 45.2s33.3 11.1 45.2-2L128 146.3V448c0 17.7 14.3 32 32 32s32-14.3 32-32V146.3l32.4 35.4c11.9 13 32.2 13.9 45.2 2s13.9-32.2 2-45.2l-88-96zM320 64c0 17.7 14.3 32 32 32h50.7l-73.4 73.4c-9.2 9.2-11.9 22.9-6.9 34.9s16.6 19.8 29.6 19.8H480c17.7 0 32-14.3 32-32s-14.3-32-32-32H429.3l73.4-73.4c9.2-9.2 11.9-22.9 6.9-34.9s-16.6-19.8-29.6-19.8H352c-17.7 0-32 14.3-32 32zm96 192c-12.1 0-23.2 6.8-28.6 17.7l-64 128-16 32c-7.9 15.8-1.5 35 14.3 42.9s35 1.5 42.9-14.3l7.2-14.3h88.4l7.2 14.3c7.9 15.8 27.1 22.2 42.9 14.3s22.2-27.1 14.3-42.9l-16-32-64-128C439.2 262.8 428.1 256 416 256zM395.8 400L416 359.6 436.2 400H395.8z',
                'vw' => 576,
                'vh' => 512
            ],
            'arrows-alt' => 'm22 26.63-4.94 4.93Q16.62 32 16 32t-1.06-.44L10 26.63q-.69-.75-.31-1.66t1.37-.9h3.19v-6.32H7.93v3.19q0 1-.9 1.37T5.37 22L.43 17.06Q0 16.62 0 16t.43-1.06L5.38 10q.75-.75 1.66-.35t.9 1.41v3.19h6.32V7.93h-3.19q-1 0-1.4-.9T10 5.37L14.94.43Q15.37 0 16 0t1.06.43L22 5.38q.69.75.31 1.66t-1.37.9h-3.19v6.32h6.31v-3.19q0-1 .9-1.38t1.66.32l4.94 4.94q.44.43.44 1.06t-.44 1.06L26.63 22q-.75.75-1.66.34t-.9-1.4v-3.19h-6.32v6.31h3.19q1 0 1.4.9T22 26.63z',
            'arrows-alt-h' => 'M23.63 10.63q0-1 .93-1.38t1.63.31l5.37 5.38q.44.44.44 1.06t-.44 1.06l-5.37 5.38q-.69.69-1.63.31t-.93-1.37V18.5H8.38v2.88q0 1-.94 1.37t-1.63-.31L.44 17.06Q0 16.63 0 16t.44-1.06L5.8 9.56q.69-.68 1.63-.31t.93 1.38v2.87h15.26v-2.88z',
            'arrows-alt-v' => [
                'p' => 'M13.38 23.63q1 0 1.37.93t-.31 1.63l-5.38 5.37Q8.62 32 8 32t-1.06-.44L1.56 26.2q-.69-.69-.31-1.63t1.37-.93H5.5V8.38H2.62q-1 0-1.37-.94t.31-1.63L6.94.44Q7.37 0 8 0t1.06.44l5.38 5.37q.68.69.31 1.63t-1.38.93H10.5v15.26h2.88z',
                'vw' => 16,
            ],
            'arrows-down-to-line' => [
                'p' => 'M544 416L32 416c-17.7 0-32 14.3-32 32s14.3 32 32 32l512 0c17.7 0 32-14.3 32-32s-14.3-32-32-32zm22.6-137.4c12.5-12.5 12.5-32.8 0-45.3s-32.8-12.5-45.3 0L480 274.7 480 64c0-17.7-14.3-32-32-32s-32 14.3-32 32l0 210.7-41.4-41.4c-12.5-12.5-32.8-12.5-45.3 0s-12.5 32.8 0 45.3l96 96c12.5 12.5 32.8 12.5 45.3 0l96-96zm-320-45.3c-12.5-12.5-32.8-12.5-45.3 0L160 274.7 160 64c0-17.7-14.3-32-32-32s-32 14.3-32 32l0 210.7L54.6 233.4c-12.5-12.5-32.8-12.5-45.3 0s-12.5 32.8 0 45.3l96 96c12.5 12.5 32.8 12.5 45.3 0l96-96c12.5-12.5 12.5-32.8 0-45.3z',
                'vw' => 576,
                'vh' => 512
            ],
            'arrows-down-to-people' => [
                'p' => 'M144 0c-13.3 0-24 10.7-24 24V142.1L97 119c-9.4-9.4-24.6-9.4-33.9 0s-9.4 24.6 0 33.9l64 64c9.4 9.4 24.6 9.4 33.9 0l64-64c9.4-9.4 9.4-24.6 0-33.9s-24.6-9.4-33.9 0l-23 23V24c0-13.3-10.7-24-24-24zM360 200a40 40 0 1 0 -80 0 40 40 0 1 0 80 0zM184 296a40 40 0 1 0 -80 0 40 40 0 1 0 80 0zm312 40a40 40 0 1 0 0-80 40 40 0 1 0 0 80zM200 441.5l26.9 49.9c6.3 11.7 20.8 16 32.5 9.8s16-20.8 9.8-32.5l-36.3-67.5c1.7-1.7 3.2-3.6 4.3-5.8L264 345.5V400c0 17.7 14.3 32 32 32h48c17.7 0 32-14.3 32-32V345.5l26.9 49.9c1.2 2.2 2.6 4.1 4.3 5.8l-36.3 67.5c-6.3 11.7-1.9 26.2 9.8 32.5s26.2 1.9 32.5-9.8L440 441.5V480c0 17.7 14.3 32 32 32h48c17.7 0 32-14.3 32-32V441.5l26.9 49.9c6.3 11.7 20.8 16 32.5 9.8s16-20.8 9.8-32.5l-37.9-70.3c-15.3-28.5-45.1-46.3-77.5-46.3H486.2c-16.3 0-31.9 4.5-45.4 12.6l-33.6-62.3c-15.3-28.5-45.1-46.3-77.5-46.3H310.2c-32.4 0-62.1 17.8-77.5 46.3l-33.6 62.3c-13.5-8.1-29.1-12.6-45.4-12.6H134.2c-32.4 0-62.1 17.8-77.5 46.3L18.9 468.6c-6.3 11.7-1.9 26.2 9.8 32.5s26.2 1.9 32.5-9.8L88 441.5V480c0 17.7 14.3 32 32 32h48c17.7 0 32-14.3 32-32V441.5zM415 153l64 64c9.4 9.4 24.6 9.4 33.9 0l64-64c9.4-9.4 9.4-24.6 0-33.9s-24.6-9.4-33.9 0l-23 23V24c0-13.3-10.7-24-24-24s-24 10.7-24 24V142.1l-23-23c-9.4-9.4-24.6-9.4-33.9 0s-9.4 24.6 0 33.9z',
                'vw' => 640,
                'vh' => 512
            ],
            'arrows-h' => 'arrows-alt-h',
            'arrows-left-right' => [
                'p' => 'M406.6 374.6l96-96c12.5-12.5 12.5-32.8 0-45.3l-96-96c-12.5-12.5-32.8-12.5-45.3 0s-12.5 32.8 0 45.3L402.7 224l-293.5 0 41.4-41.4c12.5-12.5 12.5-32.8 0-45.3s-32.8-12.5-45.3 0l-96 96c-12.5 12.5-12.5 32.8 0 45.3l96 96c12.5 12.5 32.8 12.5 45.3 0s12.5-32.8 0-45.3L109.3 288l293.5 0-41.4 41.4c-12.5 12.5-12.5 32.8 0 45.3s32.8 12.5 45.3 0z',
                'vw' => 512,
                'vh' => 512
            ],
            'arrows-left-right-to-line' => [
                'p' => 'M32 64c17.7 0 32 14.3 32 32l0 320c0 17.7-14.3 32-32 32s-32-14.3-32-32V96C0 78.3 14.3 64 32 64zm214.6 73.4c12.5 12.5 12.5 32.8 0 45.3L205.3 224l229.5 0-41.4-41.4c-12.5-12.5-12.5-32.8 0-45.3s32.8-12.5 45.3 0l96 96c12.5 12.5 12.5 32.8 0 45.3l-96 96c-12.5 12.5-32.8 12.5-45.3 0s-12.5-32.8 0-45.3L434.7 288l-229.5 0 41.4 41.4c12.5 12.5 12.5 32.8 0 45.3s-32.8 12.5-45.3 0l-96-96c-12.5-12.5-12.5-32.8 0-45.3l96-96c12.5-12.5 32.8-12.5 45.3 0zM640 96V416c0 17.7-14.3 32-32 32s-32-14.3-32-32V96c0-17.7 14.3-32 32-32s32 14.3 32 32z',
                'vw' => 640,
                'vh' => 512
            ],
            'arrows-rotate' => [
                'p' => 'M105.1 202.6c7.7-21.8 20.2-42.3 37.8-59.8c62.5-62.5 163.8-62.5 226.3 0L386.3 160H336c-17.7 0-32 14.3-32 32s14.3 32 32 32H463.5c0 0 0 0 0 0h.4c17.7 0 32-14.3 32-32V64c0-17.7-14.3-32-32-32s-32 14.3-32 32v51.2L414.4 97.6c-87.5-87.5-229.3-87.5-316.8 0C73.2 122 55.6 150.7 44.8 181.4c-5.9 16.7 2.9 34.9 19.5 40.8s34.9-2.9 40.8-19.5zM39 289.3c-5 1.5-9.8 4.2-13.7 8.2c-4 4-6.7 8.8-8.1 14c-.3 1.2-.6 2.5-.8 3.8c-.3 1.7-.4 3.4-.4 5.1V448c0 17.7 14.3 32 32 32s32-14.3 32-32V396.9l17.6 17.5 0 0c87.5 87.4 229.3 87.4 316.7 0c24.4-24.4 42.1-53.1 52.9-83.7c5.9-16.7-2.9-34.9-19.5-40.8s-34.9 2.9-40.8 19.5c-7.7 21.8-20.2 42.3-37.8 59.8c-62.5 62.5-163.8 62.5-226.3 0l-.1-.1L125.6 352H176c17.7 0 32-14.3 32-32s-14.3-32-32-32H48.4c-1.6 0-3.2 .1-4.8 .3s-3.1 .5-4.6 1z',
                'vw' => 512,
                'vh' => 512
            ],
            'arrows-spin' => [
                'p' => 'M256 96c38.4 0 73.7 13.5 101.3 36.1l-32.6 32.6c-4.6 4.6-5.9 11.5-3.5 17.4s8.3 9.9 14.8 9.9H448c8.8 0 16-7.2 16-16V64c0-6.5-3.9-12.3-9.9-14.8s-12.9-1.1-17.4 3.5l-34 34C363.4 52.6 312.1 32 256 32c-10.9 0-21.5 .8-32 2.3V99.2c10.3-2.1 21-3.2 32-3.2zM132.1 154.7l32.6 32.6c4.6 4.6 11.5 5.9 17.4 3.5s9.9-8.3 9.9-14.8V64c0-8.8-7.2-16-16-16H64c-6.5 0-12.3 3.9-14.8 9.9s-1.1 12.9 3.5 17.4l34 34C52.6 148.6 32 199.9 32 256c0 10.9 .8 21.5 2.3 32H99.2c-2.1-10.3-3.2-21-3.2-32c0-38.4 13.5-73.7 36.1-101.3zM477.7 224H412.8c2.1 10.3 3.2 21 3.2 32c0 38.4-13.5 73.7-36.1 101.3l-32.6-32.6c-4.6-4.6-11.5-5.9-17.4-3.5s-9.9 8.3-9.9 14.8V448c0 8.8 7.2 16 16 16H448c6.5 0 12.3-3.9 14.8-9.9s1.1-12.9-3.5-17.4l-34-34C459.4 363.4 480 312.1 480 256c0-10.9-.8-21.5-2.3-32zM256 416c-38.4 0-73.7-13.5-101.3-36.1l32.6-32.6c4.6-4.6 5.9-11.5 3.5-17.4s-8.3-9.9-14.8-9.9H64c-8.8 0-16 7.2-16 16l0 112c0 6.5 3.9 12.3 9.9 14.8s12.9 1.1 17.4-3.5l34-34C148.6 459.4 199.9 480 256 480c10.9 0 21.5-.8 32-2.3V412.8c-10.3 2.1-21 3.2-32 3.2z',
                'vw' => 512,
                'vh' => 512
            ],
            'arrows-split-up-and-left' => [
                'p' => 'M246.6 150.6c-12.5 12.5-32.8 12.5-45.3 0s-12.5-32.8 0-45.3l96-96c12.5-12.5 32.8-12.5 45.3 0l96 96c12.5 12.5 12.5 32.8 0 45.3s-32.8 12.5-45.3 0L352 109.3V384c0 35.3 28.7 64 64 64h64c17.7 0 32 14.3 32 32s-14.3 32-32 32H416c-70.7 0-128-57.3-128-128c0-35.3-28.7-64-64-64H109.3l41.4 41.4c12.5 12.5 12.5 32.8 0 45.3s-32.8 12.5-45.3 0l-96-96c-12.5-12.5-12.5-32.8 0-45.3l96-96c12.5-12.5 32.8-12.5 45.3 0s12.5 32.8 0 45.3L109.3 256H224c23.3 0 45.2 6.2 64 17.1V109.3l-41.4 41.4z',
                'vw' => 512,
                'vh' => 512
            ],
            'arrows-to-circle' => [
                'p' => 'M9.4 9.4C21.9-3.1 42.1-3.1 54.6 9.4L160 114.7V96c0-17.7 14.3-32 32-32s32 14.3 32 32v96c0 4.3-.9 8.5-2.4 12.2c-1.6 3.7-3.8 7.3-6.9 10.3l-.1 .1c-3.1 3-6.6 5.3-10.3 6.9c-3.8 1.6-7.9 2.4-12.2 2.4H96c-17.7 0-32-14.3-32-32s14.3-32 32-32h18.7L9.4 54.6C-3.1 42.1-3.1 21.9 9.4 9.4zM256 256a64 64 0 1 1 128 0 64 64 0 1 1 -128 0zM114.7 352H96c-17.7 0-32-14.3-32-32s14.3-32 32-32h96 0l.1 0c8.8 0 16.7 3.6 22.5 9.3l.1 .1c3 3.1 5.3 6.6 6.9 10.3c1.6 3.8 2.4 7.9 2.4 12.2v96c0 17.7-14.3 32-32 32s-32-14.3-32-32V397.3L54.6 502.6c-12.5 12.5-32.8 12.5-45.3 0s-12.5-32.8 0-45.3L114.7 352zM416 96c0-17.7 14.3-32 32-32s32 14.3 32 32v18.7L585.4 9.4c12.5-12.5 32.8-12.5 45.3 0s12.5 32.8 0 45.3L525.3 160H544c17.7 0 32 14.3 32 32s-14.3 32-32 32H448c-8.8 0-16.8-3.6-22.6-9.3l-.1-.1c-3-3.1-5.3-6.6-6.9-10.3s-2.4-7.8-2.4-12.2l0-.1v0V96zM525.3 352L630.6 457.4c12.5 12.5 12.5 32.8 0 45.3s-32.8 12.5-45.3 0L480 397.3V416c0 17.7-14.3 32-32 32s-32-14.3-32-32V320v0c0 0 0-.1 0-.1c0-4.3 .9-8.4 2.4-12.2c1.6-3.8 3.9-7.3 6.9-10.4c5.8-5.8 13.7-9.3 22.5-9.4c0 0 .1 0 .1 0h0 96c17.7 0 32 14.3 32 32s-14.3 32-32 32H525.3z',
                'vw' => 640,
                'vh' => 512
            ],
            'arrows-to-dot' => [
                'p' => 'M256 0c17.7 0 32 14.3 32 32V64h32c12.9 0 24.6 7.8 29.6 19.8s2.2 25.7-6.9 34.9l-64 64c-12.5 12.5-32.8 12.5-45.3 0l-64-64c-9.2-9.2-11.9-22.9-6.9-34.9s16.6-19.8 29.6-19.8h32V32c0-17.7 14.3-32 32-32zM169.4 393.4l64-64c12.5-12.5 32.8-12.5 45.3 0l64 64c9.2 9.2 11.9 22.9 6.9 34.9s-16.6 19.8-29.6 19.8H288v32c0 17.7-14.3 32-32 32s-32-14.3-32-32V448H192c-12.9 0-24.6-7.8-29.6-19.8s-2.2-25.7 6.9-34.9zM32 224H64V192c0-12.9 7.8-24.6 19.8-29.6s25.7-2.2 34.9 6.9l64 64c12.5 12.5 12.5 32.8 0 45.3l-64 64c-9.2 9.2-22.9 11.9-34.9 6.9s-19.8-16.6-19.8-29.6V288H32c-17.7 0-32-14.3-32-32s14.3-32 32-32zm297.4 54.6c-12.5-12.5-12.5-32.8 0-45.3l64-64c9.2-9.2 22.9-11.9 34.9-6.9s19.8 16.6 19.8 29.6v32h32c17.7 0 32 14.3 32 32s-14.3 32-32 32H448v32c0 12.9-7.8 24.6-19.8 29.6s-25.7 2.2-34.9-6.9l-64-64zM256 224a32 32 0 1 1 0 64 32 32 0 1 1 0-64z',
                'vw' => 512,
                'vh' => 512
            ],
            'arrows-to-eye' => [
                'p' => 'M15 15C24.4 5.7 39.6 5.7 49 15l63 63V40c0-13.3 10.7-24 24-24s24 10.7 24 24v96c0 13.3-10.7 24-24 24H40c-13.3 0-24-10.7-24-24s10.7-24 24-24H78.1L15 49C5.7 39.6 5.7 24.4 15 15zM133.5 243.9C158.6 193.6 222.7 112 320 112s161.4 81.6 186.5 131.9c3.8 7.6 3.8 16.5 0 24.2C481.4 318.4 417.3 400 320 400s-161.4-81.6-186.5-131.9c-3.8-7.6-3.8-16.5 0-24.2zM320 320a64 64 0 1 0 0-128 64 64 0 1 0 0 128zM591 15c9.4-9.4 24.6-9.4 33.9 0s9.4 24.6 0 33.9l-63 63H600c13.3 0 24 10.7 24 24s-10.7 24-24 24H504c-13.3 0-24-10.7-24-24V40c0-13.3 10.7-24 24-24s24 10.7 24 24V78.1l63-63zM15 497c-9.4-9.4-9.4-24.6 0-33.9l63-63H40c-13.3 0-24-10.7-24-24s10.7-24 24-24h96c13.3 0 24 10.7 24 24v96c0 13.3-10.7 24-24 24s-24-10.7-24-24V433.9L49 497c-9.4 9.4-24.6 9.4-33.9 0zm576 0l-63-63V472c0 13.3-10.7 24-24 24s-24-10.7-24-24V376c0-13.3 10.7-24 24-24h96c13.3 0 24 10.7 24 24s-10.7 24-24 24H561.9l63 63c9.4 9.4 9.4 24.6 0 33.9s-24.6 9.4-33.9 0z',
                'vw' => 640,
                'vh' => 512
            ],
            'arrows-turn-right' => [
                'p' => 'M297.4 9.4c12.5-12.5 32.8-12.5 45.3 0l96 96c12.5 12.5 12.5 32.8 0 45.3l-96 96c-12.5 12.5-32.8 12.5-45.3 0s-12.5-32.8 0-45.3L338.7 160H128c-35.3 0-64 28.7-64 64v32c0 17.7-14.3 32-32 32s-32-14.3-32-32V224C0 153.3 57.3 96 128 96H338.7L297.4 54.6c-12.5-12.5-12.5-32.8 0-45.3zm-96 256c12.5-12.5 32.8-12.5 45.3 0l96 96c12.5 12.5 12.5 32.8 0 45.3l-96 96c-12.5 12.5-32.8 12.5-45.3 0s-12.5-32.8 0-45.3L242.7 416H96c-17.7 0-32 14.3-32 32v32c0 17.7-14.3 32-32 32s-32-14.3-32-32V448c0-53 43-96 96-96H242.7l-41.4-41.4c-12.5-12.5-12.5-32.8 0-45.3z',
                'vw' => 448,
                'vh' => 512
            ],
            'arrows-turn-to-dots' => [
                'p' => 'M249.4 25.4c12.5-12.5 32.8-12.5 45.3 0s12.5 32.8 0 45.3L269.3 96 416 96c53 0 96 43 96 96v32c0 17.7-14.3 32-32 32s-32-14.3-32-32V192c0-17.7-14.3-32-32-32l-146.7 0 25.4 25.4c12.5 12.5 12.5 32.8 0 45.3s-32.8 12.5-45.3 0l-80-80c-12.5-12.5-12.5-32.8 0-45.3l80-80zm13.3 256l80 80c12.5 12.5 12.5 32.8 0 45.3l-80 80c-12.5 12.5-32.8 12.5-45.3 0s-12.5-32.8 0-45.3L242.7 416 96 416c-17.7 0-32 14.3-32 32v32c0 17.7-14.3 32-32 32s-32-14.3-32-32V448c0-53 43-96 96-96l146.7 0-25.4-25.4c-12.5-12.5-12.5-32.8 0-45.3s32.8-12.5 45.3 0zM384 384a64 64 0 1 1 128 0 64 64 0 1 1 -128 0zM64 192A64 64 0 1 1 64 64a64 64 0 1 1 0 128z',
                'vw' => 512,
                'vh' => 512
            ],
            'arrows-up-down' => [
                'p' => 'M182.6 9.4c-12.5-12.5-32.8-12.5-45.3 0l-96 96c-12.5 12.5-12.5 32.8 0 45.3s32.8 12.5 45.3 0L128 109.3V402.7L86.6 361.4c-12.5-12.5-32.8-12.5-45.3 0s-12.5 32.8 0 45.3l96 96c12.5 12.5 32.8 12.5 45.3 0l96-96c12.5-12.5 12.5-32.8 0-45.3s-32.8-12.5-45.3 0L192 402.7V109.3l41.4 41.4c12.5 12.5 32.8 12.5 45.3 0s12.5-32.8 0-45.3l-96-96z',
                'vw' => 320,
                'vh' => 512
            ],
            'arrows-up-down-left-right' => [
                'p' => 'M278.6 9.4c-12.5-12.5-32.8-12.5-45.3 0l-64 64c-12.5 12.5-12.5 32.8 0 45.3s32.8 12.5 45.3 0l9.4-9.4V224H109.3l9.4-9.4c12.5-12.5 12.5-32.8 0-45.3s-32.8-12.5-45.3 0l-64 64c-12.5 12.5-12.5 32.8 0 45.3l64 64c12.5 12.5 32.8 12.5 45.3 0s12.5-32.8 0-45.3l-9.4-9.4H224V402.7l-9.4-9.4c-12.5-12.5-32.8-12.5-45.3 0s-12.5 32.8 0 45.3l64 64c12.5 12.5 32.8 12.5 45.3 0l64-64c12.5-12.5 12.5-32.8 0-45.3s-32.8-12.5-45.3 0l-9.4 9.4V288H402.7l-9.4 9.4c-12.5 12.5-12.5 32.8 0 45.3s32.8 12.5 45.3 0l64-64c12.5-12.5 12.5-32.8 0-45.3l-64-64c-12.5-12.5-32.8-12.5-45.3 0s-12.5 32.8 0 45.3l9.4 9.4H288V109.3l9.4 9.4c12.5 12.5 32.8 12.5 45.3 0s12.5-32.8 0-45.3l-64-64z',
                'vw' => 512,
                'vh' => 512
            ],
            'arrows-up-to-line' => [
                'p' => 'M32 96l512 0c17.7 0 32-14.3 32-32s-14.3-32-32-32L32 32C14.3 32 0 46.3 0 64S14.3 96 32 96zM9.4 233.4c-12.5 12.5-12.5 32.8 0 45.3s32.8 12.5 45.3 0L96 237.3 96 448c0 17.7 14.3 32 32 32s32-14.3 32-32l0-210.7 41.4 41.4c12.5 12.5 32.8 12.5 45.3 0s12.5-32.8 0-45.3l-96-96c-12.5-12.5-32.8-12.5-45.3 0l-96 96zm320 45.3c12.5 12.5 32.8 12.5 45.3 0L416 237.3 416 448c0 17.7 14.3 32 32 32s32-14.3 32-32l0-210.7 41.4 41.4c12.5 12.5 32.8 12.5 45.3 0s12.5-32.8 0-45.3l-96-96c-12.5-12.5-32.8-12.5-45.3 0l-96 96c-12.5 12.5-12.5 32.8 0 45.3z',
                'vw' => 576,
                'vh' => 512
            ],
            'arrows-v' => 'arrows-alt-v',
            'assistive-listening-systems' => 'M13.5 16.25q0 .75-.5 1.25t-1.25.5-1.25-.5-.5-1.25q0-2.06 1.47-3.53T15 11.25t3.53 1.47T20 16.25q0 .75-.5 1.25t-1.25.5-1.25-.5-.5-1.25q0-.63-.44-1.06T15 14.75t-1.06.44-.44 1.06zm1.5-11q4.56 0 7.78 3.22T26 16.25q0 1.69-.47 3.03t-1.12 2.16-1.32 1.43-1.12 1.22-.47 1.16q0 2.81-1.97 4.78T14.75 32q-.75 0-1.25-.5t-.5-1.25.5-1.25 1.25-.5q1.37 0 2.31-.94t.94-2.31q0-1.56.72-2.72t1.56-1.75T21.81 19t.7-2.75q0-3.13-2.2-5.31T15 8.74t-5.31 2.2-2.2 5.3q0 .76-.5 1.26t-1.24.5-1.25-.5-.5-1.25q0-4.56 3.22-7.78T15 5.25zM10 20q.81 0 1.4.6T12 22t-.6 1.4-1.4.6-1.4-.6T8 22t.6-1.4T10 20zm-8 8q.81 0 1.4.6T4 30t-.6 1.4T2 32t-1.4-.6T0 30t.6-1.4T2 28zm30-11.75q0 .75-.5 1.25t-1.25.5-1.25-.5-.5-1.25V16q-.06-4.25-2.6-7.72t-6.52-4.84q-.7-.25-1-.9t-.13-1.35q.13-.32.38-.6t.53-.4.65-.16.69.1q5.06 1.68 8.25 6.06T32 15.94v.31zM9.56 27.44l-2.12 2.12-5-5 2.12-2.12z',
            'asterisk' => 'M29.88 20.88q.56.3.71.9t-.15 1.16l-1.19 2.12q-.31.56-.94.72t-1.12-.22l-8.7-5.25.2 10.13q.06.68-.38 1.12T17.2 32h-2.4q-.69 0-1.12-.44t-.38-1.12l.19-10.13-8.69 5.25q-.5.38-1.12.22t-.94-.72l-1.19-2.12q-.31-.57-.15-1.16t.71-.9L11 16l-8.88-4.88q-.56-.3-.71-.9t.15-1.16l1.19-2.12q.31-.57.94-.72t1.12.22l8.69 5.25-.19-10.13q-.06-.44.16-.78t.56-.56.78-.22h2.38q.69 0 1.12.44t.38 1.12l-.2 10.13 8.7-5.25q.5-.38 1.12-.22t.94.72l1.19 2.12q.31.57.15 1.16t-.71.9L21 16z',
            'at' => 'M16 .5q6.63 0 11.06 3.88T31.5 14.5q0 3.94-2.31 6.56t-7.06 2.63q-.94 0-1.44-.03t-1.22-.22-1.16-.66-.62-1.22q-.82 1.07-2.13 1.75t-2.56.7q-2.75 0-4.38-1.79T7 17.38q0-4.13 2.53-6.82t6.1-2.69q2.62 0 3.74 1.38.07-.75.82-.75H23q.75 0 1.19.53t.31 1.28L23 17.87q-.19.82-.03 1.32t.9.56q1.63-.19 2.63-1.75t1-3.5q0-4.62-3.31-7.31T16 4.5q-4.75 0-8.13 3.38T4.5 16t3.38 8.13T16 27.5q3.38 0 6.19-1.81.5-.32 1.06-.22t.94.53l.62.75q.44.56.32 1.22t-.7 1.03q-3.8 2.5-8.43 2.5-6.44 0-10.97-4.53T.5 16 5.03 5.03 16 .5zm-1.38 19q1.32 0 2.47-1.28T18.25 15q0-1.19-.56-1.88t-1.63-.68q-1.44 0-2.56 1.28T12.37 17q0 1.19.63 1.84t1.62.66z',
            'atlas' => [
                'p' => 'M19.88 13q-.5 3.06-3.32 4.38.75-1.82.88-4.38h2.44zm0-2h-2.44q-.13-2.56-.88-4.38 2.82 1.32 3.32 4.38zM14 6.06q.25.32.56.88t.57 1.65.3 2.41h-2.87q.07-1.88.54-3.19t.9-1.75zm-2.56.57q-.75 1.8-.88 4.37H8.12q.5-3.06 3.32-4.38zm0 10.75Q8.63 16.05 8.13 13h2.43q.13 2.56.88 4.38zm4-4.38q-.06 1.88-.53 3.19t-.9 1.75q-.26-.32-.57-.88t-.57-1.65-.3-2.4h2.87zM28 22.38q0 .93-.63 1.25-.12.56-.12 2.18t.13 2.38q.62.62.62 1.19v1q0 .75-.47 1.18t-1.16.44H6q-2.56 0-4.28-1.72T0 26V6q0-2.56 1.72-4.28T6 0h20.38Q28 0 28 1.63v20.75zM14 4q-3.31 0-5.66 2.34T6 12t2.34 5.66T14 20t5.66-2.34T22 12t-2.34-5.66T14 4zm10 24v-4H6q-.81 0-1.4.56T4 26t.56 1.44T6 28h18z',
                'w' => .875,
                'vw' => 28
            ],
            'atom' => [
                'p' => 'M25.81 16q3.44 4.69 1.57 8-1.82 3.25-7.25 2.63Q17.88 32 14 32t-6.12-5.38Q2.44 27.25.63 24q-1.88-3.31 1.56-8-3.44-4.69-1.56-8 1.8-3.25 7.25-2.63Q10.13 0 14 0t6.13 5.38q5.43-.63 7.25 2.62 1.87 3.31-1.57 8zM3.94 22q.44.75 2.75.69-.25-1.25-.44-2.5-.69-.57-1.38-1.19-1.37 2.19-.93 3zm2.31-10.19q.19-1.25.44-2.5H6.3q-1.93 0-2.37.7-.44.8.94 3 .68-.63 1.37-1.2zM14 4q-1 0-2.13 2.31l2.13.82 2.13-.82Q15 4 14 4zm0 24q1 0 2.13-2.31L14 24.87l-2.13.82Q13 28 14 28zm3.88-9.81q.18-2.19 0-4.38Q16.05 12.5 14 11.5q-2.06 1-3.88 2.31-.18 2.2 0 4.38Q11.95 19.5 14 20.5q2.06-1 3.88-2.31zM24.05 22q.44-.81-.93-3-.7.63-1.38 1.19-.19 1.25-.44 2.5 2.32.06 2.75-.69zm-.93-9q1.37-2.19.93-3-.43-.69-2.37-.69h-.38q.25 1.25.44 2.5.69.56 1.38 1.19zM14 14q.81 0 1.4.6T16 16t-.6 1.4-1.4.6-1.4-.6T12 16t.6-1.4T14 14z',
                'w' => .875,
                'vw' => 28
            ],
            'audio-description' => 'm10.19 14.94.56 1.87H9.13l.56-1.87q.12-.38.25-1 .12.62.25 1zm10.37-2q1.38 0 2.16.78T23.5 16q0 3.06-2.94 3.06h-.87v-6.12h.87zM32 7v18q0 1.25-.88 2.13T29 28H3q-1.25 0-2.13-.88T0 25V7q0-1.25.88-2.13T3 4h26q1.25 0 2.13.88T32 7zM15.31 21l-3.56-10.5q-.13-.5-.69-.5H8.81q-.5 0-.69.5L4.56 21q-.06.19-.03.38t.13.3.25.23.34.1h1.81q.57 0 .75-.57l.5-1.88h3.25l.5 1.88q.19.56.75.56h1.82q.25 0 .43-.15t.28-.38-.03-.47zm11.57-5q0-1.88-.75-3.22t-2.16-2.06-3.28-.72h-3.56q-.32 0-.54.22t-.21.53v10.5q0 .31.21.53t.54.22h3.56q2.87 0 4.53-1.63T26.88 16z',
            'austral-sign' => [
                'p' => 'M253.5 51.7C248.6 39.8 236.9 32 224 32s-24.6 7.8-29.5 19.7L122.7 224H32c-17.7 0-32 14.3-32 32s14.3 32 32 32H96L82.7 320H32c-17.7 0-32 14.3-32 32s14.3 32 32 32H56L34.5 435.7c-6.8 16.3 .9 35 17.2 41.8s35-.9 41.8-17.2L125.3 384H322.7l31.8 76.3c6.8 16.3 25.5 24 41.8 17.2s24-25.5 17.2-41.8L392 384h24c17.7 0 32-14.3 32-32s-14.3-32-32-32H365.3L352 288h64c17.7 0 32-14.3 32-32s-14.3-32-32-32H325.3L253.5 51.7zM256 224H192l32-76.8L256 224zm-90.7 64H282.7L296 320H152l13.3-32z',
                'vw' => 448,
                'vh' => 512
            ],
            'automobile' => 'car',
            'award' => [
                'p' => 'M6.06 22.69Q7.44 24 9.31 24q1.13 0 2.2-.62l-3.26 8q-.19.5-.72.59t-.9-.28L4.3 29.3l-3.25.13Q.5 29.44.2 29t-.13-.94l2.88-7q.75.63 1.56.88.19.06.56.12.57.13.82.44zm17.88 5.37q.19.5-.13.94t-.87.44l-3.25-.13-2.32 2.38q-.37.37-.9.28t-.72-.6l-3.25-8q1.06.63 2.19.63 1.87 0 3.25-1.31l.19-.19q.25-.31.8-.44.13 0 .2-.03t.15-.06.22-.03q.81-.25 1.56-.88zm-7.5-6.81q-.63.63-1.5.72t-1.63-.34q-.56-.38-1.31-.38t-1.31.38q-.75.43-1.63.34t-1.5-.72l-.31-.31-.25-.22-.25-.19-.19-.13-.28-.09-.31-.1-.38-.09-.47-.12q-1.37-.38-1.75-1.82-.37-1.3-.53-1.59t-1.1-1.28q-1.05-1.06-.68-2.44t.38-1.69-.38-1.62q-.38-1.44.69-2.5.94-.94 1.1-1.22t.52-1.66 1.75-1.8Q6.7 2 7.2 1.5q.12-.07.75-.75Q9-.32 10.37.05 11.7.44 12 .44t1.62-.37Q15-.32 16.06.75q.94 1 1.22 1.15t1.6.47q1.37.44 1.75 1.81l.28 1.07q.03.13.28.56t.34.53l.72.72q1.06 1.06.69 2.5-.38 1.31-.38 1.62t.38 1.7-.69 2.43q-.94 1-1.1 1.28t-.52 1.6q-.38 1.43-1.75 1.8-.2.07-.5.13t-.41.1-.31.12-.32.13-.25.15-.31.28zM6.12 11q0 2.5 1.72 4.25T12 17t4.16-1.75T17.87 11q0-1.63-.78-3t-2.12-2.19T12 5Q9.56 5 7.84 6.75T6.13 11z',
                'w' => .75,
                'vw' => 24
            ],
            'b' => [
                'p' => 'M64 32C28.7 32 0 60.7 0 96V256 416c0 35.3 28.7 64 64 64H192c70.7 0 128-57.3 128-128c0-46.5-24.8-87.3-62-109.7c18.7-22.3 30-51 30-82.3c0-70.7-57.3-128-128-128H64zm96 192H64V96h96c35.3 0 64 28.7 64 64s-28.7 64-64 64zM64 288h96 32c35.3 0 64 28.7 64 64s-28.7 64-64 64H64V288z',
                'vw' => 320,
                'vh' => 512
            ],
            'baby' => [
                'p' => 'M12 10q-2.06 0-3.53-1.47T7 5t1.47-3.53T12 0t3.53 1.47T17 5t-1.47 3.53T12 10zM8.69 25.56 10.5 28q.63.81.47 1.84T10 31.5q-.69.5-1.5.5-1.25 0-2-1l-3-4q-.5-.69-.5-1.53t.56-1.53l2.88-3.56 3.81 3.18zm8.87-5.18 2.88 3.56q.56.69.56 1.53T20.5 27l-3 4q-.75 1-2 1-.81 0-1.5-.5-.81-.63-.97-1.66T13.5 28l1.81-2.44-1.56-2zm5.97-11.32q.6.88.44 1.88t-1.03 1.62l-2.56 1.75q-1.44 1.07-3.38 1.75V18H7v-1.94q-1.94-.68-3.38-1.75l-2.56-1.75Q.2 11.94.03 10.94t.44-1.88 1.6-1.03 1.87.4l2.56 1.82q2.44 1.69 5.5 1.69t5.5-1.69l2.56-1.81q.88-.56 1.88-.4t1.6 1.02z',
                'w' => .75,
                'vw' => 24
            ],
            'baby-carriage' => 'M9.06 1.06 16 12H0q0-3.37 1.5-6.37T5.7.5q.75-.62 1.78-.44t1.6 1zM31 6q.44 0 .72.28T32 7v2q0 .44-.28.72T31 10h-3v4q0 4.69-3.75 8.19 1.63.37 2.69 1.72T28 27q0 2.05-1.47 3.52T23 32t-3.53-1.47T18 27q0-.75.25-1.56Q16.2 26 14 26t-4.25-.56q.25.81.25 1.56 0 2.06-1.47 3.53T5 32t-3.53-1.47T0 27q0-1.12.5-2.15t1.34-1.72 1.91-.94Q0 18.69 0 14h24v-4q0-1.63 1.19-2.81T28 5.99h3zM5 29q.81 0 1.4-.6T7 27t-.6-1.4T5 25t-1.4.6T3 27t.6 1.4T5 29zm20-2q0-.81-.6-1.4T23 25t-1.4.6T21 27t.6 1.4 1.4.6 1.4-.6.6-1.4z',
            'backspace' => [
                'p' => 'M36 4q1.69 0 2.84 1.16T40 8v16q0 1.69-1.16 2.84T36 28H12.81Q11.2 28 10 26.81L.56 17.44Q0 16.8 0 16t.56-1.44L10 5.2Q11.19 4 12.81 4H36zm-5.31 15.88L26.8 16l3.88-3.88q.75-.68 0-1.43L29.3 9.3q-.75-.75-1.43 0L24 13.2l-3.88-3.9q-.68-.75-1.43 0l-1.39 1.4q-.75.75 0 1.43L21.2 16l-3.88 3.88q-.75.68 0 1.43l1.38 1.38q.75.75 1.43 0L24 18.8l3.88 3.88q.68.75 1.43 0l1.38-1.38q.75-.75 0-1.43z',
                'w' => 1.25,
                'vw' => 40
            ],
            'backward' => 'M.75 17.56Q0 16.94 0 16t.75-1.56l12-10q.94-.82 2.1-.25T16 6v20q0 1.25-1.16 1.81t-2.09-.25zm16 0Q16 16.94 16 16t.75-1.56l12-10q.94-.82 2.1-.25T32 6v20q0 1.25-1.16 1.81t-2.09-.25z',
            'backward-fast' => [
                'p' => 'M493.6 445c-11.2 5.3-24.5 3.6-34.1-4.4L288 297.7V416c0 12.4-7.2 23.7-18.4 29s-24.5 3.6-34.1-4.4L64 297.7V416c0 17.7-14.3 32-32 32s-32-14.3-32-32V96C0 78.3 14.3 64 32 64s32 14.3 32 32V214.3L235.5 71.4c9.5-7.9 22.8-9.7 34.1-4.4S288 83.6 288 96V214.3L459.5 71.4c9.5-7.9 22.8-9.7 34.1-4.4S512 83.6 512 96V416c0 12.4-7.2 23.7-18.4 29z',
                'vw' => 512,
                'vh' => 512
            ],
            'backward-step' => [
                'p' => 'M267.5 440.6c9.5 7.9 22.8 9.7 34.1 4.4s18.4-16.6 18.4-29V96c0-12.4-7.2-23.7-18.4-29s-24.5-3.6-34.1 4.4l-192 160L64 241V96c0-17.7-14.3-32-32-32S0 78.3 0 96V416c0 17.7 14.3 32 32 32s32-14.3 32-32V271l11.5 9.6 192 160z',
                'vw' => 320,
                'vh' => 512
            ],
            'bacon' => [
                'p' => 'M13.69 21q-.81.81-1.5 1.38t-1.44.93l-1.19.6-1.25.47q-3.31 1.25-5.5 3.3L.56 25.5Q0 24.88.03 24.03T.7 22.7q2.25-1.88 5.68-3.19l.13-.06q4.31-1.44 6.06-6.07l.35-.93q.1-.25.65-1.38t1.07-1.87 1.5-1.97 2.09-2.1 2.69-1.84 3.47-1.66Q26.07 1 26.94.37q.5-.37 1.13-.37.8 0 1.37.56l2.31 2.31Q29.88 4.62 27 5.75q-1.18.44-1.87.75t-1.88 1.12-2.34 1.97-1.97 2.35-1.13 1.84-.75 1.84q-.43 1.13-.68 1.72t-.97 1.63T13.69 21zM35.44 6.5q.56.63.53 1.47T35.3 9.3q-2.25 1.88-5.68 3.2-.07 0-.13.05-4.31 1.44-6.06 6.07l-.6 1.5q-.22.56-1.22 2.25t-2.15 2.9-3.25 2.7-4.6 2.4q-1.68.62-2.56 1.25-.5.37-1.12.37-.82 0-1.38-.56l-2.31-2.31Q6.12 27.38 9 26.25q.75-.25 1.12-.4t1.07-.5 1.22-.7 1.25-.93 1.46-1.28q1.13-1.19 1.94-2.38t1.13-1.84.75-1.84q.37-1.13.65-1.72t1-1.63 1.72-2.06 2.03-1.72T26 8.28t1.69-.66q3.31-1.25 5.5-3.3z',
                'w' => 1.125,
                'vw' => 36
            ],
            'bacteria' => [
                'p' => 'M17.02 14.15c.27-.12.48-.34.59-.61.1-.28.1-.58-.02-.85l-.25-.57c.25-.08.51-.14.77-.2a5.19 5.19 0 0 0 2.11-.86l.63.56a1.11 1.11 0 0 0 1.5-1.64l-.62-.54c.55-.91.8-1.96.73-3.01l.74-.23c.6-.17.93-.79.75-1.38-.18-.58-.8-.91-1.38-.74l-.71.2a5.23 5.23 0 0 0-2.27-2.21l.18-.68A1.12 1.12 0 0 0 17.6.83l-.18.67c-.34 0-.7.02-1.04.07-.71.12-1.39.32-2.07.52l-.21-.6a1.11 1.11 0 0 0-2.1.75l.22.61c-.96.41-1.88.9-2.75 1.47l-.4-.52a1.12 1.12 0 0 0-1.99.5c-.04.3.04.6.21.83l.4.53a18.2 18.2 0 0 0-2.18 2.21l-.55-.4A1.12 1.12 0 0 0 3.2 8.54c.05.29.2.55.45.72l.55.4c-.55.88-1.04 1.8-1.44 2.75l-.65-.23a1.1 1.1 0 1 0-.73 2.1l.65.22c-.17.63-.36 1.24-.47 1.9-.06.38-.08.78-.05 1.18l-.72.2a1.1 1.1 0 0 0 .63 2.13l.7-.2c.51.95 1.3 1.73 2.28 2.21l-.18.7a1.11 1.11 0 0 0 2.15.55l.18-.69.2.02c1.01 0 2-.3 2.85-.87l.54.55a1.11 1.11 0 0 0 1.6-1.54l-.58-.59c.38-.59.64-1.25.76-1.94.06-.32.14-.65.24-.97l.56.27a1.1 1.1 0 0 0 .95-2l-.55-.27c.04-.06.07-.13.12-.19a7.11 7.11 0 0 1 1.22-1.3c.2-.17.4-.34.62-.49l.21-.13.25.55a1.11 1.11 0 0 0 1.47.57ZM8 16a2 2 0 1 1 0-4 2 2 0 0 1 0 4Zm5-6a1 1 0 0 1 0-2 1 1 0 0 1 0 2Zm26.95 2.83a1.11 1.11 0 0 0-1.38-.74l-.7.2a5.22 5.22 0 0 0-2.28-2.21l.18-.7a1.11 1.11 0 0 0-2.15-.55l-.18.69-.2-.02c-1.01 0-2 .3-2.85.87l-.54-.55a1.11 1.11 0 0 0-1.6 1.54l.58.59a5.21 5.21 0 0 0-.76 1.94c-.06.33-.14.65-.24.97l-.56-.27a1.1 1.1 0 0 0-.95 2.01l.55.26c-.04.06-.07.13-.12.19a7.26 7.26 0 0 1-1.22 1.3c-.2.17-.4.34-.62.49l-.22.13-.24-.55a1.11 1.11 0 0 0-2.06.04c-.1.28-.1.58.02.85l.25.57c-.25.08-.51.14-.77.2a5.2 5.2 0 0 0-2.11.86l-.63-.56a1.11 1.11 0 1 0-1.5 1.64l.61.55a5.1 5.1 0 0 0-.72 3l-.75.23c-.58.17-.91.8-.74 1.38.17.59.8.92 1.38.74l.7-.2a5.37 5.37 0 0 0 2.28 2.22l-.18.68a1.11 1.11 0 0 0 2.16.55l.18-.67c.35.01.7-.01 1.04-.07.71-.12 1.4-.32 2.07-.51l.21.59a1.11 1.11 0 0 0 2.1-.74l-.22-.62c.96-.4 1.88-.9 2.75-1.47l.4.53a1.12 1.12 0 0 0 1.99-.51 1.1 1.1 0 0 0-.21-.82l-.4-.54a18.2 18.2 0 0 0 2.18-2.21l.55.4a1.1 1.1 0 0 0 1.79-.76c.04-.4-.14-.8-.48-1.03l-.55-.4c.55-.88 1.03-1.8 1.44-2.75l.65.23a1.11 1.11 0 0 0 .73-2.1l-.65-.22c.17-.63.36-1.24.47-1.89.06-.4.08-.79.05-1.19l.72-.21a1.1 1.1 0 0 0 .75-1.38ZM26 26a2 2 0 1 1 0-4 2 2 0 0 1 0 4Zm0 0',
                'vw' => 40
            ],
            'bacterium' => 'M31.94 6.43a1.48 1.48 0 0 0-1.85-1l-.94.29a6.99 6.99 0 0 0-3.03-2.97l.23-.9a1.48 1.48 0 1 0-2.87-.74l-.24.9c-.46-.02-.93.01-1.4.08-.93.17-1.86.38-2.76.66L18.8 2a1.49 1.49 0 0 0-2.8.98l.27.76c-1.28.55-2.5 1.22-3.67 1.99l-.5-.67a1.49 1.49 0 0 0-2.38 1.77l.5.67c-1.05.9-2.03 1.9-2.92 2.96l-.68-.5c-.43-.33-1-.4-1.5-.19a1.48 1.48 0 0 0-.25 2.58l.69.5a23.8 23.8 0 0 0-1.93 3.67l-.81-.27a1.48 1.48 0 1 0-.97 2.79l.81.28c-.24.83-.43 1.68-.58 2.53-.08.52-.1 1.06-.06 1.59l-.97.28a1.48 1.48 0 0 0 .85 2.83l.93-.27a7 7 0 0 0 3.04 2.95l-.24.92A1.48 1.48 0 0 0 7.07 32c.68 0 1.27-.45 1.44-1.1l.24-.92.26.02a6.8 6.8 0 0 0 3.8-1.16l.72.74a1.48 1.48 0 0 0 2.13-2.06l-.77-.78a6.9 6.9 0 0 0 1.01-2.59c.08-.44.19-.87.33-1.3l.75.36a1.47 1.47 0 1 0 1.26-2.67l-.74-.35.16-.25a10.31 10.31 0 0 1 1.63-1.74c.26-.23.54-.45.83-.65l.29-.18.32.74c.33.75 1.2 1.1 1.96.76.75-.33 1.1-1.2.76-1.95l-.33-.76c.33-.1.68-.19 1.03-.25a6.99 6.99 0 0 0 2.81-1.16l.83.75a1.49 1.49 0 1 0 2-2.19l-.81-.73a6.9 6.9 0 0 0 .97-4.02l.99-.29a1.48 1.48 0 0 0 1-1.84ZM10 23a3 3 0 1 1 0-6 3 3 0 0 1 0 6Zm5-8.5a1.5 1.5 0 1 1 0-3 1.5 1.5 0 0 1 0 3Zm0 0',
            'bag-shopping' => [
                'p' => 'M160 112c0-35.3 28.7-64 64-64s64 28.7 64 64v48H160V112zm-48 48H48c-26.5 0-48 21.5-48 48V416c0 53 43 96 96 96H352c53 0 96-43 96-96V208c0-26.5-21.5-48-48-48H336V112C336 50.1 285.9 0 224 0S112 50.1 112 112v48zm24 48a24 24 0 1 1 0 48 24 24 0 1 1 0-48zm152 24a24 24 0 1 1 48 0 24 24 0 1 1 -48 0z',
                'vw' => 448,
                'vh' => 512
            ],
            'bahai' => 'm31.02 12.66-6.88-.97 2.61-6.52c.42-1.04-.72-2.01-1.66-1.41L19.21 7.5 17.1.8C16.9.27 16.46 0 16 0c-.46 0-.91.27-1.09.8L12.8 7.5 6.9 3.76c-.94-.6-2.08.37-1.66 1.41l2.6 6.52-6.87.97c-1.1.15-1.35 1.64-.37 2.16l6.13 3.3-4.65 5.22c-.68.76-.11 1.93.83 1.93.08 0 .17 0 .25-.03l6.79-1.47-.26 7.03c-.02.73.55 1.2 1.15 1.2.32 0 .65-.14.89-.45L16 26l4.26 5.55a1.14 1.14 0 0 0 2.04-.75l-.26-7.03 6.79 1.47c1.08.24 1.83-1.07 1.08-1.9l-4.65-5.23 6.13-3.29c.98-.52.72-2-.37-2.16Zm-9.86 6.82-3.25-.7.13 3.36L16 19.48l-2.04 2.66.13-3.37-3.25.71 2.23-2.5-2.93-1.57 3.28-.46-1.25-3.12 2.81 1.8 1.02-3.2 1.02 3.2 2.8-1.8-1.24 3.12 3.28.46-2.93 1.57Zm0 0',
            'baht-sign' => [
                'p' => 'M144 0c-17.7 0-32 14.3-32 32V64H37.6C16.8 64 0 80.8 0 101.6V224v41.7V288 406.3c0 23 18.7 41.7 41.7 41.7H112v32c0 17.7 14.3 32 32 32s32-14.3 32-32V448h32c61.9 0 112-50.1 112-112c0-40.1-21.1-75.3-52.7-95.1C280.3 222.6 288 200.2 288 176c0-61.9-50.1-112-112-112V32c0-17.7-14.3-32-32-32zM112 128v96H64V128h48zm64 96V128c26.5 0 48 21.5 48 48s-21.5 48-48 48zm-64 64v96H64V288h48zm64 96V288h32c26.5 0 48 21.5 48 48s-21.5 48-48 48H176z',
                'vw' => 320,
                'vh' => 512
            ],
            'balance-scale' => [
                'p' => 'M16 21q0 2.06-2.34 3.53T8 26t-5.66-1.47T0 21v-.06q0-.5.31-1.22t2.2-4.47l2.8-5.63Q6.13 8 8.01 8t2.68 1.62q1.25 2.5 2.88 5.82 1.8 3.56 2.12 4.25t.31 1.25V21zM8 11l-4.5 9h9zm32 10q0 1.38-1.06 2.53t-2.9 1.81T32 26q-3.31 0-5.66-1.47T24 21v-.06q0-.5.31-1.22t2.2-4.47l.46-.94.5-.97.6-1.15.59-1.2.65-1.37Q30.13 8 32.01 8t2.68 1.62q1.25 2.5 2.88 5.82 1.8 3.56 2.12 4.25t.31 1.25V21zm-12.5-1h9L32 11zm5.5 8q.44 0 .72.28T34 29v2q0 .44-.28.72T33 32H7q-.44 0-.72-.28T6 31v-2q0-.44.28-.72T7 28h11V9.56q-2.38-1-2.88-3.56H7q-.44 0-.72-.28T6 5V3q0-.44.28-.72T7 2h9q1.56-2 4-2t4 2h9q.44 0 .72.28T34 3v2q0 .44-.28.72T33 6h-8.13q-.5 2.56-2.87 3.56V28h11z',
                'w' => 1.25,
                'vw' => 40
            ],
            'balance-scale-left' => [
                'p' => 'M33 28H22V9.58a4.99 4.99 0 0 0 2.71-2.97l8.25-2.77a1 1 0 0 0 .63-1.26l-.63-1.9a1 1 0 0 0-1.27-.63l-7.37 2.47A4.98 4.98 0 0 0 15 5c0 .21.04.42.06.63L7.04 8.31a1 1 0 0 0-.63 1.26l.63 1.9a1 1 0 0 0 1.27.63l8.88-2.98c.25.17.52.32.81.45V31a1 1 0 0 0 1 1h14a1 1 0 0 0 1-1v-2a1 1 0 0 0-1-1Zm7-9c0-1.01.08-.55-5.32-11.34a3 3 0 0 0-5.36 0C23.87 18.55 24 18.02 24 19c0 2.76 3.58 5 8 5s8-2.24 8-5Zm-12.5-1L32 9l4.5 9Zm-16.82-2.34a3 3 0 0 0-5.36 0C-.13 26.55 0 26.02 0 27c0 2.76 3.58 5 8 5s8-2.24 8-5c0-1.01.08-.55-5.32-11.34ZM3.5 26 8 17l4.5 9Zm0 0',
                'vw' => 40
            ],
            'balance-scale-right' => [
                'p' => 'M6 29v2a1 1 0 0 0 1 1h14a1 1 0 0 0 1-1V9.58c.29-.13.56-.28.81-.45l8.88 2.98a1 1 0 0 0 1.27-.63l.63-1.9a1 1 0 0 0-.63-1.26l-8.02-2.7c.02-.2.06-.4.06-.62a5 5 0 0 0-9.32-2.48L8.31.05a1 1 0 0 0-1.27.63l-.63 1.9a1 1 0 0 0 .63 1.26l8.25 2.77A4.99 4.99 0 0 0 18 9.58V28H7a1 1 0 0 0-1 1ZM0 19c0 2.76 3.58 5 8 5s8-2.24 8-5c0-.98.13-.45-5.32-11.34a3 3 0 0 0-5.36 0C-.08 18.46 0 17.99 0 19Zm3.5-1L8 9l4.5 9ZM24 27c0 2.76 3.58 5 8 5s8-2.24 8-5c0-.98.13-.45-5.32-11.34a3 3 0 0 0-5.36 0C23.92 26.46 24 25.99 24 27Zm3.5-1 4.5-9 4.5 9Zm0 0',
                'vw' => 40
            ],
            'ban' => 'M16 .5q6.44 0 10.97 4.53T31.5 16t-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5zm8.13 7.38q-3-3-7.25-3.35t-7.5 2.03l16.06 16.07q2.37-3.25 2.03-7.5t-3.35-7.26zM7.88 24.13q3 3 7.25 3.34t7.5-2.03L6.55 9.37q-2.37 3.26-2.03 7.5t3.35 7.25z',
            'ban-smoking' => [
                'p' => 'M99.5 144.8L178.7 224l96 96 92.5 92.5C335.9 434.9 297.5 448 256 448C150 448 64 362 64 256c0-41.5 13.1-79.9 35.5-111.2zM333.3 288l-32-32H384v32H333.3zm32 32H400c8.8 0 16-7.2 16-16V240c0-8.8-7.2-16-16-16H269.3L144.8 99.5C176.1 77.1 214.5 64 256 64c106 0 192 86 192 192c0 41.5-13.1 79.9-35.5 111.2L365.3 320zM256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM272 96c-8.8 0-16 7.2-16 16c0 26.5 21.5 48 48 48h32c8.8 0 16 7.2 16 16s7.2 16 16 16s16-7.2 16-16c0-26.5-21.5-48-48-48H304c-8.8 0-16-7.2-16-16s-7.2-16-16-16zM229.5 320l-96-96H112c-8.8 0-16 7.2-16 16v64c0 8.8 7.2 16 16 16H229.5z',
                'vw' => 512,
                'vh' => 512
            ],
            'band-aid' => [
                'p' => 'M0 10q0-1.63 1.19-2.81T4 5.99h6v20H4q-1.62 0-2.81-1.18T0 22V10zm36-4q1.63 0 2.81 1.19t1.2 2.81v12q0 1.63-1.2 2.81T36 26.01h-6V6h6zM12 26V6h16v20H12zm11-14.5q-.63 0-1.06.44T21.5 13t.44 1.06 1.06.44 1.06-.44.44-1.06-.44-1.06T23 11.5zm0 6q-.63 0-1.06.44T21.5 19t.44 1.06 1.06.44 1.06-.44.44-1.06-.44-1.06T23 17.5zm-6-6q-.63 0-1.06.44T15.5 13t.44 1.06 1.06.44 1.06-.44.44-1.06-.44-1.06T17 11.5zm0 6q-.63 0-1.06.44T15.5 19t.44 1.06 1.06.44 1.06-.44.44-1.06-.44-1.06T17 17.5z',
                'w' => 1.25,
                'vw' => 40
            ],
            'bandage' => 'band-aid',
            'bangladeshi-taka-sign' => [
                'p' => 'M36 32.2C18.4 30.1 2.4 42.5 .2 60S10.5 93.6 28 95.8l7.9 1c16 2 28 15.6 28 31.8V160H32c-17.7 0-32 14.3-32 32s14.3 32 32 32H64V384c0 53 43 96 96 96h32c106 0 192-86 192-192V256c0-53-43-96-96-96H272c-17.7 0-32 14.3-32 32s14.3 32 32 32h16c17.7 0 32 14.3 32 32v32c0 70.7-57.3 128-128 128H160c-17.7 0-32-14.3-32-32V224h32c17.7 0 32-14.3 32-32s-14.3-32-32-32H128V128.5c0-48.4-36.1-89.3-84.1-95.3l-7.9-1z',
                'vw' => 384,
                'vh' => 512
            ],
            'barcode' => 'M0 28V4h1.13v24H0zm1.69 0V4h.56v24H1.7zm1.69 0V4h.56v24h-.57zm2.8 0V4h.57v24H6.2zm2.26 0V4h1.12v24H8.44zm2.81 0V4h.56v24h-.56zm1.13 0V4h.56v24h-.56zm1.12 0V4h.56v24h-.56zm2.19 0V4h1.12v24H15.7zm2.81 0V4h1.13v24H18.5zm2.25 0V4h1.13v24h-1.13zM23 28V4h1.13v24H23zm1.69 0V4h1.12v24H24.7zm2.81 0V4h1.69v24H27.5zm2.25 0V4h.56v24h-.56zm1.13 0V4H32v24h-1.13z',
            'bars' => [
                'p' => 'M1 8.25q-.44 0-.72-.28T0 7.25v-2.5q0-.44.28-.72T1 3.75h26q.44 0 .72.28t.28.72v2.5q0 .44-.28.72t-.72.28H1zm0 10q-.44 0-.72-.28T0 17.25v-2.5q0-.44.28-.72t.72-.28h26q.44 0 .72.28t.28.72v2.5q0 .44-.28.72t-.72.28H1zm0 10q-.44 0-.72-.28T0 27.25v-2.5q0-.44.28-.72t.72-.28h26q.44 0 .72.28t.28.72v2.5q0 .44-.28.72t-.72.28H1z',
                'w' => .875,
                'vw' => 28
            ],
            'bars-progress' => [
                'p' => 'M448 160H320V128H448v32zM48 64C21.5 64 0 85.5 0 112v64c0 26.5 21.5 48 48 48H464c26.5 0 48-21.5 48-48V112c0-26.5-21.5-48-48-48H48zM448 352v32H192V352H448zM48 288c-26.5 0-48 21.5-48 48v64c0 26.5 21.5 48 48 48H464c26.5 0 48-21.5 48-48V336c0-26.5-21.5-48-48-48H48z',
                'vw' => 512,
                'vh' => 512
            ],
            'bars-staggered' => [
                'p' => 'M0 96C0 78.3 14.3 64 32 64H416c17.7 0 32 14.3 32 32s-14.3 32-32 32H32C14.3 128 0 113.7 0 96zM64 256c0-17.7 14.3-32 32-32H480c17.7 0 32 14.3 32 32s-14.3 32-32 32H96c-17.7 0-32-14.3-32-32zM448 416c0 17.7-14.3 32-32 32H32c-17.7 0-32-14.3-32-32s14.3-32 32-32H416c17.7 0 32 14.3 32 32z',
                'vw' => 512,
                'vh' => 512
            ],
            'baseball' => 'baseball-ball',
            'baseball-ball' => 'M23 22.75q1.19 2.44 3.25 4.38-4.5 4.37-10.75 4.37T4.75 27.12q2-1.93 3.25-4.37l-1.81-.88q-1.07 2.07-2.82 3.82Q0 21.44 0 16t3.37-9.69Q5.12 8 6.12 10.06l1.82-.94q-1.2-2.37-3.2-4.25Q9.25.5 15.5.5t10.75 4.37q-2 1.94-3.19 4.32l1.81.87q1-2.06 2.75-3.75Q31 10.56 31 16t-3.44 9.69q-1.69-1.69-2.75-3.81zm-14.25-2q1.56-4.81 0-9.63l-1.94.63q1.38 4.19.07 8.38zm13.5-9.63q-1.56 4.82 0 9.63l1.88-.63q-1.32-4.12.06-8.37z',
            'baseball-bat-ball' => [
                'p' => 'M424 0c-12.4 0-24.2 4.9-33 13.7L233.5 171.2c-10.5 10.5-19.8 22.1-27.7 34.6L132.7 321.6c-7.3 11.5-15.8 22.2-25.5 31.9L69.9 390.7l51.3 51.3 37.3-37.3c9.6-9.6 20.3-18.2 31.9-25.5l115.8-73.1c12.5-7.9 24.1-17.2 34.6-27.7L498.3 121c8.7-8.7 13.7-20.6 13.7-33s-4.9-24.2-13.7-33L457 13.7C448.2 4.9 436.4 0 424 0zm88 432a80 80 0 1 0 -160 0 80 80 0 1 0 160 0zM15 399c-9.4 9.4-9.4 24.6 0 33.9l64 64c9.4 9.4 24.6 9.4 33.9 0s9.4-24.6 0-33.9L49 399c-9.4-9.4-24.6-9.4-33.9 0z',
                'vw' => 512,
                'vh' => 512
            ],
            'basket-shopping' => [
                'p' => 'M253.3 35.1c6.1-11.8 1.5-26.3-10.2-32.4s-26.3-1.5-32.4 10.2L117.6 192H32c-17.7 0-32 14.3-32 32s14.3 32 32 32L83.9 463.5C91 492 116.6 512 146 512H430c29.4 0 55-20 62.1-48.5L544 256c17.7 0 32-14.3 32-32s-14.3-32-32-32H458.4L365.3 12.9C359.2 1.2 344.7-3.4 332.9 2.7s-16.3 20.6-10.2 32.4L404.3 192H171.7L253.3 35.1zM192 304v96c0 8.8-7.2 16-16 16s-16-7.2-16-16V304c0-8.8 7.2-16 16-16s16 7.2 16 16zm96-16c8.8 0 16 7.2 16 16v96c0 8.8-7.2 16-16 16s-16-7.2-16-16V304c0-8.8 7.2-16 16-16zm128 16v96c0 8.8-7.2 16-16 16s-16-7.2-16-16V304c0-8.8 7.2-16 16-16s16 7.2 16 16z',
                'vw' => 576,
                'vh' => 512
            ],
            'basketball' => 'basketball-ball',
            'basketball-ball' => 'M13.25.63q-.13 4.8-2.75 8.24L5.62 4Q8.95 1.25 13.25.62zm2.25 13.24L12.62 11Q16.2 6.62 16.25.5q5.19.25 9.13 3.5zm-12-7.74L8.38 11Q4.94 13.63.12 13.75.76 9.44 3.5 6.12zm17 12.75L17.62 16l9.88-9.88q3.25 3.94 3.5 9.13-6.13.13-10.5 3.63zm-5-.75L18.38 21q-3.5 4.38-3.63 10.5-5.19-.25-9.13-3.5zm12 7.75L22.62 21q3.44-2.63 8.25-2.75-.62 4.31-3.37 7.63zm-17-12.75L13.38 16 3.5 25.88Q.25 21.93 0 16.74q6.13-.06 10.5-3.63zm7.25 18.24q.13-4.8 2.75-8.25L25.38 28q-3.32 2.75-7.63 3.38z',
            'bath' => 'M30.5 16q.63 0 1.06.44T32 17.5v1q0 .63-.44 1.06T30.5 20H30v2q0 2.69-2 4.5v2q0 .63-.44 1.06T26.5 30h-1q-.63 0-1.06-.44T24 28.5V28H8v.5q0 .63-.44 1.06T6.5 30h-1q-.63 0-1.07-.44T4 28.5v-2Q2 24.7 2 22v-2h-.5q-.63 0-1.07-.44T0 18.5v-1q0-.62.43-1.06T1.5 16H2V7q0-2.06 1.47-3.53T7 2q2.62 0 4.12 2.13 2.25-.63 4 1 .5-.5 1 0l.75.75q.5.5 0 1.06l-5.94 5.94q-.56.5-1.06 0l-.75-.75q-.5-.5 0-1-1-1.07-1.12-2.54t.68-2.65Q8.12 5 7 5q-.82 0-1.41.6T4.99 7v9h25.5z',
            'battery-empty' => [
                'p' => 'M34 10H4v12h30v-4h2v-4h-2v-4zm1-4q1.25 0 2.13.88T38 9v1h.5q.63 0 1.06.44T40 11.5v9q0 .63-.44 1.06T38.5 22H38v1q0 1.25-.88 2.13T35 26H3q-1.25 0-2.13-.87T0 23V9q0-1.25.88-2.12T3 6h32z',
                'w' => 1.25,
                'vw' => 40
            ],
            'battery-full' => [
                'p' => 'M34 10H4v12h30v-4h2v-4h-2v-4zm1-4q1.25 0 2.13.88T38 9v1h.5q.63 0 1.06.44T40 11.5v9q0 .63-.44 1.06T38.5 22H38v1q0 1.25-.88 2.13T35 26H3q-1.25 0-2.13-.87T0 23V9q0-1.25.88-2.12T3 6h32zm-3 6v8H6v-8h26z',
                'w' => 1.25,
                'vw' => 40
            ],
            'battery-half' => [
                'p' => 'M34 10H4v12h30v-4h2v-4h-2v-4zm1-4q1.25 0 2.13.88T38 9v1h.5q.63 0 1.06.44T40 11.5v9q0 .63-.44 1.06T38.5 22H38v1q0 1.25-.88 2.13T35 26H3q-1.25 0-2.13-.87T0 23V9q0-1.25.88-2.12T3 6h32zm-15 6v8H6v-8h14z',
                'w' => 1.25,
                'vw' => 40
            ],
            'battery-quarter' => [
                'p' => 'M34 10H4v12h30v-4h2v-4h-2v-4zm1-4q1.25 0 2.13.88T38 9v1h.5q.63 0 1.06.44T40 11.5v9q0 .63-.44 1.06T38.5 22H38v1q0 1.25-.88 2.13T35 26H3q-1.25 0-2.13-.87T0 23V9q0-1.25.88-2.12T3 6h32zm-21 6v8H6v-8h8z',
                'w' => 1.25,
                'vw' => 40
            ],
            'battery-three-quarters' => [
                'p' => 'M34 10H4v12h30v-4h2v-4h-2v-4zm1-4q1.25 0 2.13.88T38 9v1h.5q.63 0 1.06.44T40 11.5v9q0 .63-.44 1.06T38.5 22H38v1q0 1.25-.88 2.13T35 26H3q-1.25 0-2.13-.87T0 23V9q0-1.25.88-2.12T3 6h32zm-9 6v8H6v-8h20z',
                'w' => 1.25,
                'vw' => 40
            ],
            'bed' => [
                'p' => 'M11 16q-2.06 0-3.53-1.47T6 11t1.47-3.53T11 6t3.53 1.47T16 11t-1.47 3.53T11 16zm22-8q2.88 0 4.94 2.06T40 15v12q0 .44-.28.72T39 28h-2q-.44 0-.72-.28T36 27v-3H4v3q0 .44-.28.72T3 28H1q-.44 0-.72-.28T0 27V5q0-.44.28-.72T1 4h2q.44 0 .72.28T4 5v13h14V9q0-.44.28-.72T19 8h14z',
                'w' => 1.25,
                'vw' => 40
            ],
            'bed-pulse' => [
                'p' => 'M483.2 9.6L524 64h92c13.3 0 24 10.7 24 24s-10.7 24-24 24H512c-7.6 0-14.7-3.6-19.2-9.6L468.7 70.3l-47 99.9c-3.7 7.8-11.3 13.1-19.9 13.7s-16.9-3.4-21.7-10.6L339.2 112H216c-13.3 0-24-10.7-24-24s10.7-24 24-24H352c8 0 15.5 4 20 10.7l24.4 36.6 45.9-97.5C445.9 6.2 453.2 1 461.6 .1s16.6 2.7 21.6 9.5zM320 160h12.7l20.7 31.1c11.2 16.8 30.6 26.3 50.7 24.8s37.9-13.7 46.5-32L461.9 160H544c53 0 96 43 96 96V480c0 17.7-14.3 32-32 32s-32-14.3-32-32V448H352 320 64v32c0 17.7-14.3 32-32 32s-32-14.3-32-32V96C0 78.3 14.3 64 32 64s32 14.3 32 32V352H288V192c0-17.7 14.3-32 32-32zm-144 0a80 80 0 1 1 0 160 80 80 0 1 1 0-160z',
                'vw' => 640,
                'vh' => 512
            ],
            'beer' => [
                'p' => 'M23 6q2.06 0 3.53 1.47T28 11v8.06q0 1.5-.81 2.72t-2.13 1.85L20 25.88v2.62q0 .63-.44 1.06T18.5 30h-17q-.63 0-1.06-.44T0 28.5v-25q0-.62.44-1.06T1.5 2h17q.62 0 1.06.44T20 3.5V6h3zm1 13.06V11q0-.44-.28-.72T23 10h-3v11.5l3.44-1.56q.56-.25.56-.88zM13 24q.44 0 .72-.28T14 23V9q0-.44-.28-.72T13 8t-.72.28T12 9v14q0 .44.28.72T13 24zm-6 0q.44 0 .72-.28T8 23V9q0-.44-.28-.72T7 8t-.72.28T6 9v14q0 .44.28.72T7 24z',
                'w' => .875,
                'vw' => 28
            ],
            'beer-mug-empty' => [
                'p' => 'M32 64c0-17.7 14.3-32 32-32H352c17.7 0 32 14.3 32 32V96h51.2c42.4 0 76.8 34.4 76.8 76.8V274.9c0 30.4-17.9 57.9-45.6 70.2L384 381.7V416c0 35.3-28.7 64-64 64H96c-35.3 0-64-28.7-64-64V64zM384 311.6l56.4-25.1c4.6-2.1 7.6-6.6 7.6-11.7V172.8c0-7.1-5.7-12.8-12.8-12.8H384V311.6zM160 144c0-8.8-7.2-16-16-16s-16 7.2-16 16V368c0 8.8 7.2 16 16 16s16-7.2 16-16V144zm64 0c0-8.8-7.2-16-16-16s-16 7.2-16 16V368c0 8.8 7.2 16 16 16s16-7.2 16-16V144zm64 0c0-8.8-7.2-16-16-16s-16 7.2-16 16V368c0 8.8 7.2 16 16 16s16-7.2 16-16V144z',
                'vw' => 512,
                'vh' => 512
            ],
            'bell' => [
                'p' => 'M14 32q-1.63 0-2.81-1.16T9.99 28h8q0 1.69-1.18 2.84T14 32zm13.44-9.38q.56.63.56 1.38 0 .81-.56 1.4T26 26H2q-.56 0-1.03-.28t-.72-.75T0 24q0-.75.56-1.38.07-.06.2-.25.93-.93 1.46-1.68t1.16-2.85T4 13q0-3.63 2.25-6.31T12 3.3V2q0-.81.6-1.4T14 0t1.4.6T16 2v1.3q2.31.44 4.13 1.81t2.84 3.44T24 13q0 2.75.63 4.84t1.15 2.85 1.47 1.69l.19.25z',
                'w' => .875,
                'vw' => 28
            ],
            'bell-concierge' => [
                'p' => 'M216 64c-13.3 0-24 10.7-24 24s10.7 24 24 24h16v33.3C119.6 157.2 32 252.4 32 368H480c0-115.6-87.6-210.8-200-222.7V112h16c13.3 0 24-10.7 24-24s-10.7-24-24-24H256 216zM24 400c-13.3 0-24 10.7-24 24s10.7 24 24 24H488c13.3 0 24-10.7 24-24s-10.7-24-24-24H24z',
                'vw' => 512,
                'vh' => 512
            ],
            'bell-slash' => [
                'p' => 'M39.63 28.63q.74.62.18 1.43l-1.25 1.57q-.62.75-1.37.18L.38 3.38q-.76-.63-.2-1.44L1.45.38q.62-.76 1.37-.2l9.07 7q2.3-3.06 6.12-3.87V2q0-.81.6-1.4T20 0t1.4.6T22 2v1.3q3.5.69 5.75 3.38T30 13q0 2.75.63 4.84t1.15 2.85 1.47 1.69l.19.25q.56.62.56 1.37 0 .06-.03.16l-.03.1zM9.8 15.74 23.13 26H8q-.56 0-1.03-.28t-.72-.75T6 24q0-.75.56-1.38.88-.93 1.25-1.37t1.07-2 .93-3.5zM20 32q-1.63 0-2.81-1.16T15.99 28h8q0 1.06-.52 2T22 31.47 20 32z',
                'w' => 1.25,
                'vw' => 40
            ],
            'bezier-curve' => [
                'p' => 'M23 2q.81 0 1.4.6T25 4v6q0 .81-.6 1.4T23 12h-6q-.81 0-1.4-.6T15 10V4q0-.81.6-1.4T17 2h6zM13 5.5V10q0 .88.38 1.63Q10 14.05 8.93 18H5.8q.69-3 2.47-5.47t4.4-4.03h-5q-.18.56-.56 1.03t-.84.78-1.06.5T4 11q-1.7 0-2.85-1.16T0 7t1.15-2.84T4 3q1.25 0 2.25.69T7.69 5.5h5.3zM10 20q.81 0 1.4.6T12 22v6q0 .81-.6 1.4T10 30H4q-.81 0-1.4-.6T2 28v-6q0-.81.6-1.4T4 20h6zM36 3q1.69 0 2.84 1.16T40 7t-1.16 2.84T36 11q-1.25 0-2.25-.69T32.31 8.5h-5q2.63 1.56 4.4 4.03T34.2 18h-3.13q-1.06-3.94-4.44-6.38.38-.75.38-1.62V5.5h5.31q.44-1.13 1.44-1.81t2.25-.7zm0 17q.81 0 1.4.6T38 22v6q0 .81-.6 1.4T36 30h-6q-.81 0-1.4-.6T28 28v-6q0-.81.6-1.4T30 20h6z',
                'w' => 1.25,
                'vw' => 40
            ],
            'bible' => [
                'p' => 'M28 22.38q0 .93-.63 1.25-.12.56-.12 2.18t.13 2.38q.62.62.62 1.19v1q0 .75-.47 1.18t-1.16.44H6q-2.56 0-4.28-1.72T0 26V6q0-2.56 1.72-4.28T6 0h20.38Q28 0 28 1.63v20.75zM9 9v2q0 .44.28.72T10 12h3v7q0 .44.28.72T14 20h2q.44 0 .72-.28T17 19v-7h3q.44 0 .72-.28T21 11V9q0-.44-.28-.72T20 8h-3V5q0-.44-.28-.72T16 4h-2q-.44 0-.72.28T13 5v3h-3q-.44 0-.72.28T9 9zm14.81 19v-4H6q-.81 0-1.4.56T4 26t.56 1.44T6 28h17.81z',
                'w' => .875,
                'vw' => 28
            ],
            'bicycle' => [
                'p' => 'M32.06 12q3.32 0 5.66 2.4T40 20.13t-2.4 5.63T31.93 28q-3.25-.06-5.6-2.4T24 20q0-3.5 2.56-5.88l-.93-1.5-5.38 8.7Q19.81 22 19 22h-3.25q-.69 2.63-2.84 4.31T8 28.01q-3.31 0-5.66-2.35T0 19.94q0-3.25 2.38-5.6T8 12q1.44 0 2.69.5l1.87-3H9.5q-.63 0-1.06-.44T8 8q0-.38.22-.72t.56-.56.78-.22H15q.44 0 .72.28t.28.72v1h7.13L22.18 7h-3.2q-.43 0-.71-.28T18 6V5q0-.44.28-.72T19 4h4q.81 0 1.25.69l4.87 7.81q1.38-.5 2.94-.5zM11.7 16.63 10.19 19h2.69q-.25-1.38-1.2-2.38zM8 25q1.5 0 2.75-.81t1.81-2.2H7.5q-.88 0-1.31-.77t.06-1.53l2.81-4.57Q8.56 15 8 15q-2.06 0-3.53 1.47T3 20t1.47 3.53T8 25zm10.19-6 4.62-7.5h-8l-1.5 2.5q2.25 2 2.63 5h2.25zm13.56 6q2.13.13 3.69-1.38T37 20q0-2.06-1.47-3.53T32 15q-.62 0-1.25.13l2.81 4.5q.2.37.1.78t-.47.59l-.81.56q-.38.2-.79.1t-.59-.47l-2.81-4.44q-1.25 1.5-1.19 3.44t1.44 3.31 3.31 1.5z',
                'w' => 1.25,
                'vw' => 40
            ],
            'biking' => [
                'p' => 'M25 6a3 3 0 1 0 0-6 3 3 0 0 0 0 6Zm-.25 7.56A2 2 0 0 0 26 14h4a2 2 0 1 0 0-4h-3.3l-4.45-3.56a2 2 0 0 0-2.55.04l-7 6a2 2 0 0 0 .2 3.18l5.1 3.41V26a2 2 0 1 0 4 0v-8a2 2 0 0 0-.9-1.66l-2.58-1.73 3.64-3.12ZM32 16a8 8 0 1 0 0 16 8 8 0 0 0 0-16Zm0 12a4 4 0 1 1 0-8 4 4 0 0 1 0 8ZM8 16a8 8 0 1 0 0 16 8 8 0 0 0 0-16Zm0 12a4 4 0 1 1 0-8 4 4 0 0 1 0 8Zm0 0',
                'vw' => 40
            ],
            'binoculars' => 'M26 3v3h-6V3q0-.44.28-.72T21 2h4q.44 0 .72.28T26 3zM4 10q0-.81.6-1.4T6 8h6v10h-2v10q0 .81-.6 1.4T8 30H2q-.81 0-1.4-.6T0 28v-2.75q.06-2.25.47-4.1t.97-3.12 1.12-2.44.97-2.56T4 10zm24 0q.06 1.94.66 3.72t1.3 3.06 1.35 3.5.7 4.97V28q0 .81-.6 1.4T30 30h-6q-.82 0-1.42-.6T22 28V18h-2V8h6q.56 0 1 .28t.72.72.28 1zM11 2q.44 0 .72.28T12 3v3H6V3q0-.44.28-.72T7 2h4zm3 16V8h4v10h-4z',
            'biohazard' => [
                'p' => 'M18 7q-1.56 0-3.31.63-1.25-1-1.82-2.57Q15.31 4 18 4t5.12 1.06q-.56 1.56-1.8 2.56Q19.55 7 18 7zm-7.75 20.44q-2.31-1.69-3.69-4.19T5 17.81q.69-.19 1.44-.19t1.62.32q.38 4.18 3.69 6.81-.37 1.56-1.5 2.69zm14-2.69q3.25-2.63 3.63-6.81.93-.32 1.68-.32.7 0 1.38.2-.19 2.93-1.57 5.43t-3.62 4.19q-1.13-1.13-1.5-2.69zm7.06-12.81q3.38 2 4.63 5.81.19.38-.19.6t-.63-.1q-.87-1-1.8-1.56-2.7-1.63-5.7-.78t-4.56 3.6-.75 5.8 3.5 4.7q.94.55 2.19.8.44.07.44.5t-.44.5q-3.81.75-7.19-1.25-1.56-.93-2.81-2.5-1.25 1.57-2.81 2.5-3.38 2-7.25 1.25-.38-.06-.38-.5t.38-.5q1.3-.3 2.25-.8 2.68-1.63 3.5-4.7t-.75-5.8-4.57-3.6-5.69.78q-.93.5-1.8 1.56-.26.31-.63.1t-.25-.6q1.3-3.81 4.68-5.81 1.5-.88 3.44-1.25-.62-1.81-.62-3.5 0-3.94 2.56-7 .25-.31.63-.13t.25.63q-.44 1.25-.44 2.37 0 3.2 2.19 5.44t5.3 2.25q2.07 0 3.79-1.03t2.72-2.78 1-3.88q0-1.12-.44-2.37-.12-.44.25-.63t.63.13q2.56 3 2.56 7 0 1.69-.63 3.5 1.94.37 3.44 1.25zM18 20q1.25 0 2.13-.88T21 17t-.88-2.13T18 14t-2.13.88T15 17t.88 2.13T18 20z',
                'w' => 1.123,
                'vw' => 36
            ],
            'birthday-cake' => [
                'p' => 'M28 24q-.75 0-1.97-1t-2.69-1-2.69 1-1.96 1-1.97-1T14 22q-1.44 0-2.69 1t-1.94 1q-.81 0-2.03-1t-2.69-1-2.69 1T0 24v-5q0-1.25.87-2.13T3 16h1V7h4v9h4V7h4v9h4V7h4v9h1q1.25 0 2.12.88T28 19v5zm0 8H0v-6q1.5 0 2.72-1t1.97-1 1.97 1 2.69 1 2.68-1T14 24t1.97 1 2.69 1 2.69-1 2.03-1q.69 0 1.9 1T28 26v6zM6 6q-.81 0-1.4-.6T4 4q0-.88 1-1.75T6 0q.63 0 1.31 1.16t.7 2.34-.57 1.84T6 6zm8 0q-.81 0-1.4-.6T12 4q0-.88 1-1.75T14 0q.63 0 1.31 1.16t.7 2.34-.57 1.84T14 6zm8 0q-.81 0-1.4-.6T20 4q0-.88 1-1.75T22 0q.63 0 1.31 1.16t.7 2.34-.57 1.84T22 6z',
                'w' => .875,
                'vw' => 28
            ],
            'bitcoin-sign' => [
                'p' => 'M48 32C48 14.3 62.3 0 80 0s32 14.3 32 32V64h32V32c0-17.7 14.3-32 32-32s32 14.3 32 32V64c0 1.5-.1 3.1-.3 4.5C254.1 82.2 288 125.1 288 176c0 24.2-7.7 46.6-20.7 64.9c31.7 19.8 52.7 55 52.7 95.1c0 61.9-50.1 112-112 112v32c0 17.7-14.3 32-32 32s-32-14.3-32-32V448H112v32c0 17.7-14.3 32-32 32s-32-14.3-32-32V448H41.7C18.7 448 0 429.3 0 406.3V288 265.7 224 101.6C0 80.8 16.8 64 37.6 64H48V32zM64 224H176c26.5 0 48-21.5 48-48s-21.5-48-48-48H64v96zm112 64H64v96H208c26.5 0 48-21.5 48-48s-21.5-48-48-48H176z',
                'vw' => 320,
                'vh' => 512
            ],
            'blender' => 'M26 24q1.69 0 2.84 1.16T30 28v2q0 .81-.6 1.4T28 32H8q-.81 0-1.4-.6T6 30v-2q0-1.69 1.16-2.84T10 24h16zm-8 6q.81 0 1.4-.6T20 28t-.6-1.4T18 26t-1.4.6T16 28t.6 1.4 1.4.6zm2.5-26q-.5 0-.5.5v1q0 .5.5.5h9.88l-1.13 4H20.5q-.5 0-.5.5v1q0 .5.5.5h8.25l-1.13 4H20.5q-.5 0-.5.5v1q0 .5.5.5h6.56L26 22H10l-.56-6H3q-1.25 0-2.13-.88T0 13V3Q0 1.75.87.87T3 0h29l-1.06 4H20.5zM4 12h5.06l-.69-8H4v8z',
            'blender-phone' => [
                'p' => 'M24.5 4q-.5 0-.5.5v1q0 .5.5.5h9.88l-1.13 4H24.5q-.5 0-.5.5v1q0 .5.5.5h8.25l-1.13 4H24.5q-.5 0-.5.5v1q0 .5.5.5h6.56L30 22H12V0h24l-1.06 4H24.5zM9.94 20.94q.31.81-.44 1.25l-2.44 1.5q-.62.37-1.4.28t-1.22-.6Q.06 18.57 0 12.1T4.3.75Q4.81.19 5.6.03t1.47.28l2.44 1.5q.75.44.44 1.25L8.3 7q-.31.75-1.06.69l-2.81-.32q-1.7 4.63 0 9.25l2.81-.3q.75-.07 1.06.68zM30 24q1.69 0 2.84 1.16T34 28v2q0 .81-.6 1.4T32 32H10q-.81 0-1.4-.6T8 30v-2q0-1.69 1.16-2.84T12 24h18zm-9 6q.81 0 1.4-.6T23 28t-.6-1.4T21 26t-1.4.6T19 28t.6 1.4 1.4.6z',
                'w' => 1.125,
                'vw' => 36
            ],
            'blind' => [
                'p' => 'M23.75 31.94q-.13.06-.25.06-.31 0-.44-.25l-7.81-12.87q.5-.2.81-.63l7.88 13q.06.13.06.25 0 .31-.25.44zM8.94 19.62l3.94 9.63q.3.75-.03 1.53t-1.1 1.1-1.53-.04-1.1-1.09l-2.24-5.5zM6 5.5q-1.13 0-1.94-.81t-.81-1.94T4.06.8 6 0t1.94.8.81 1.94-.81 1.94T6 5.5zm9.69 10.56q.37.5.31 1.13t-.56 1q-.44.37-1.06.31t-1.07-.56L6.94 9.87q-.07-.12-.2-.12-.24 0-.24.25 0 .13.06.19l1.94 2.44v6.75l-4.13 11.3q-.25.76-1 1.13t-1.53.1T.7 30.88.63 29.3L4 20.01v-8.76L3 12.5v5q0 .63-.44 1.06T1.5 19t-1.06-.44T0 17.44V11.5l3.8-4.94Q4.25 6 5 6h2q.75 0 1.19.56z',
                'w' => .748,
                'vw' => 24
            ],
            'blog' => 'M10.75 14.19q3.63.69 5.75 3.81t1.31 6.94q-.56 2.56-2.44 4.44t-4.43 2.44q-4.32.87-7.63-1.85T0 23V7.5q0-.62.44-1.06T1.5 6h3q.63 0 1.06.44T6 7.5V23q0 1.25.88 2.13T9 26t2.13-.87T12 23q0-.62-.25-1.19t-.69-1-1-.62Q9 19.8 9 18.8v-3.1q0-.75.53-1.19t1.22-.31zM13.06 0q7.63.38 13.1 5.84T32 18.94q0 .44-.28.75T31 20h-2q-.44 0-.72-.28t-.28-.66q-.37-6.06-4.69-10.4T12.94 4q-.38-.06-.66-.34T12 3V1q0-.44.31-.72t.75-.28zm0 6q5.2.38 8.88 4.06T26 18.94q0 .44-.28.75T25 20h-2q-.44 0-.72-.25t-.28-.69q-.31-3.62-2.9-6.18T12.94 10q-.25 0-.47-.16t-.34-.37T12 9V7q0-.44.31-.75t.75-.25z',
            'bold' => [
                'p' => 'M19.06 15.25q2.07.69 3.25 2.44t1.2 4.19q0 2.75-1.57 4.84t-4 2.78q-1.69.5-4.19.5H1.5q-.44 0-.72-.28T.5 29v-2q0-.44.28-.72T1.5 26h2.06V6.06H1.5q-.44 0-.72-.28T.5 5.06V3q0-.44.28-.72T1.5 2h11.63q3.12 0 5 .75 2 .75 3.12 2.5t1.13 4q0 1.94-.88 3.53t-2.44 2.47zM8.88 6.31v7.07h4.8q1.57 0 2.45-.97T17 9.8q0-2.25-1.56-3.06-.88-.44-2.32-.44H8.87zm7.06 19.07q1-.38 1.6-1.44t.59-2.38q0-1.81-1.07-2.87t-2.87-1.07H8.88v8.07h5.06q1.25 0 2-.32z',
                'w' => .75,
                'vw' => 24
            ],
            'bolt' => [
                'p' => 'M18.5 10q.88 0 1.31.75t0 1.5l-11 19Q8.37 32 7.5 32q-.75 0-1.19-.56t-.25-1.32L8.94 18H1.5q-.44 0-.81-.25t-.57-.63-.12-.8l2-15q.06-.38.28-.7T2.8.15 3.5 0h9q.75 0 1.19.6t.25 1.27L11.3 10h7.19z',
                'w' => .625,
                'vw' => 20
            ],
            'bolt-lightning' => [
                'p' => 'M0 256L28.5 28c2-16 15.6-28 31.8-28H228.9c15 0 27.1 12.1 27.1 27.1c0 3.2-.6 6.5-1.7 9.5L208 160H347.3c20.2 0 36.7 16.4 36.7 36.7c0 7.4-2.2 14.6-6.4 20.7l-192.2 281c-5.9 8.6-15.6 13.7-25.9 13.7h-2.9c-15.7 0-28.5-12.8-28.5-28.5c0-2.3 .3-4.6 .9-6.9L176 288H32c-17.7 0-32-14.3-32-32z',
                'vw' => 384,
                'vh' => 512
            ],
            'bomb' => 'm27.5 5.5-3.25 3.25 1.69 1.69q.44.44.44 1.06t-.44 1.06l-1.07 1.07Q26 16.19 26 19q0 5.38-3.81 9.19T12.99 32t-9.17-3.8T0 19t3.8-9.19 9.2-3.8q2.81 0 5.37 1.12l1.07-1.07q.43-.43 1.06-.43t1.06.43l1.63 1.7 3.25-3.26zm3.75-1.75q.75 0 .75.75t-.75.75h-1.5q-.75 0-.75-.75t.75-.75h1.5zM27.5 0q.75 0 .75.75v1.5q0 .75-.75.75t-.75-.75V.75q0-.75.75-.75zm2.13 3.44q-.5.56-1.07 0t0-1.06l1.07-1.07q.5-.56 1.03 0t.03 1.06zm-4.25 0L24.3 2.38q-.5-.5.03-1.07t1.03 0l1.07 1.06q.56.5.03 1.07t-1.1 0zm4.25 2.12 1.06 1.07q.5.56-.03 1.09t-1.03-.03l-1.07-1.06q-.56-.5-.03-1.07t1.1 0zM7 17q0-1.63 1.19-2.81t2.81-1.2q.44 0 .72-.27T12 12t-.28-.72T11 11q-2.5 0-4.25 1.75T5 17q0 .44.28.72T6 18t.72-.28T7 17z',
            'bone' => [
                'p' => 'M37.44 15.31q-.44.2-.44.7t.44.68q1.19.62 1.87 1.75t.7 2.44v.5q0 1.87-1.38 3.25T35.38 26q-1.5 0-2.72-.87t-1.72-2.32q0-.06-.19-.59t-.25-.69-.25-.56-.4-.53-.54-.28-.75-.16H11.44q-.32 0-.57.06t-.43.16-.35.31-.28.35-.25.44-.15.46-.16.54-.19.5q-.5 1.43-1.72 2.3T4.62 26q-1.87 0-3.25-1.37T0 21.38v-.5q0-1.31.69-2.44t1.87-1.75Q3 16.5 3 16t-.44-.69Q1.37 14.7.7 13.56T0 11.13v-.5q0-1.88 1.38-3.25T4.62 6q1 0 1.91.4t1.56 1.13.97 1.66q0 .06.19.6t.25.68.25.56.4.53.54.28.75.16h17.12q.32 0 .6-.06t.5-.25.37-.32.31-.5.2-.5.21-.62.19-.56q.5-1.44 1.72-2.32T35.38 6q1.87 0 3.25 1.37T40 10.62v.5q0 1.32-.69 2.44t-1.87 1.75z',
                'w' => 1.25,
                'vw' => 40
            ],
            'bong' => [
                'p' => 'M18.88 32H5.13q-2.38 0-3.5-2.06Q0 27.19 0 24q0-3.25 1.63-6.03T6 13.62V4H5q-.44 0-.72-.28T4 3V1q0-.38.28-.69T5 0h14q.44 0 .72.28T20 1v2q0 .37-.28.69t-.72.3h-1v9.63q1.31.75 2.44 1.82L22.88 13l-.57-.56q-.75-.75 0-1.44l.7-.69q.68-.75 1.43 0l3.25 3.25q.75.75 0 1.44l-.69.69q-.69.75-1.44 0l-.56-.57-2.69 2.7Q24 20.68 24 24q0 3.19-1.63 5.94Q21.25 32 18.88 32zM7.5 16.19Q5.19 17.56 4 20h16q-1.19-2.44-3.5-3.81l-1.5-.81V4H9v11.38z',
                'w' => .873,
                'vw' => 28
            ],
            'book' => [
                'p' => 'M28 22.5q0 .75-.56 1.19-.2.69-.2 2.31t.2 2.31q.56.5.56 1.2v1q0 .62-.44 1.05T26.5 32H6q-2.5 0-4.25-1.75T0 26V6q0-2.5 1.75-4.25T6 0h20.5q.62 0 1.06.44T28 1.5v21zM8 8.37v1.26q0 .37.38.37h13.24q.38 0 .38-.38V8.38Q22 8 21.62 8H8.38Q8 8 8 8.38zm0 4v1.26q0 .37.38.37h13.24q.38 0 .38-.38v-1.24q0-.38-.38-.38H8.38Q8 12 8 12.38zM23.81 28q-.18-2 0-4H6q-.81 0-1.4.6T4 26t.6 1.4T6 28h17.81z',
                'w' => .875,
                'vw' => 28
            ],
            'book-atlas' => [
                'p' => 'M0 96C0 43 43 0 96 0H384h32c17.7 0 32 14.3 32 32V352c0 17.7-14.3 32-32 32v64c17.7 0 32 14.3 32 32s-14.3 32-32 32H384 96c-53 0-96-43-96-96V96zM64 416c0 17.7 14.3 32 32 32H352V384H96c-17.7 0-32 14.3-32 32zM247.4 283.8c-3.7 3.7-6.2 4.2-7.4 4.2s-3.7-.5-7.4-4.2c-3.8-3.7-8-10-11.8-18.9c-6.2-14.5-10.8-34.3-12.2-56.9h63c-1.5 22.6-6 42.4-12.2 56.9c-3.8 8.9-8 15.2-11.8 18.9zm42.7-9.9c7.3-18.3 12-41.1 13.4-65.9h31.1c-4.7 27.9-21.4 51.7-44.5 65.9zm0-163.8c23.2 14.2 39.9 38 44.5 65.9H303.5c-1.4-24.7-6.1-47.5-13.4-65.9zM368 192a128 128 0 1 0 -256 0 128 128 0 1 0 256 0zM145.3 208h31.1c1.4 24.7 6.1 47.5 13.4 65.9c-23.2-14.2-39.9-38-44.5-65.9zm31.1-32H145.3c4.7-27.9 21.4-51.7 44.5-65.9c-7.3 18.3-12 41.1-13.4 65.9zm56.1-75.8c3.7-3.7 6.2-4.2 7.4-4.2s3.7 .5 7.4 4.2c3.8 3.7 8 10 11.8 18.9c6.2 14.5 10.8 34.3 12.2 56.9h-63c1.5-22.6 6-42.4 12.2-56.9c3.8-8.9 8-15.2 11.8-18.9z',
                'vw' => 448,
                'vh' => 512
            ],
            'book-bible' => [
                'p' => 'M96 0C43 0 0 43 0 96V416c0 53 43 96 96 96H384h32c17.7 0 32-14.3 32-32s-14.3-32-32-32V384c17.7 0 32-14.3 32-32V32c0-17.7-14.3-32-32-32H384 96zm0 384H352v64H96c-17.7 0-32-14.3-32-32s14.3-32 32-32zM208 80c0-8.8 7.2-16 16-16h32c8.8 0 16 7.2 16 16v48h48c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H272V304c0 8.8-7.2 16-16 16H224c-8.8 0-16-7.2-16-16V192H160c-8.8 0-16-7.2-16-16V144c0-8.8 7.2-16 16-16h48V80z',
                'vw' => 448,
                'vh' => 512
            ],
            'book-bookmark' => [
                'p' => 'M0 96C0 43 43 0 96 0h96V190.7c0 13.4 15.5 20.9 26 12.5L272 160l54 43.2c10.5 8.4 26 .9 26-12.5V0h32 32c17.7 0 32 14.3 32 32V352c0 17.7-14.3 32-32 32v64c17.7 0 32 14.3 32 32s-14.3 32-32 32H384 96c-53 0-96-43-96-96V96zM64 416c0 17.7 14.3 32 32 32H352V384H96c-17.7 0-32 14.3-32 32z',
                'vw' => 448,
                'vh' => 512
            ],
            'book-dead' => [
                'p' => 'M17 8.5q-.44 0-.72-.28T16 7.5t.28-.72.72-.28.72.28.28.72-.28.72-.72.28zm11 13.88q0 .93-.63 1.25-.12.56-.12 2.18t.13 2.38q.62.62.62 1.19v1q0 .75-.47 1.18t-1.16.44H6q-2.56 0-4.28-1.72T0 26V6q0-2.56 1.72-4.28T6 0h20.38Q28 0 28 1.63v20.75zM15 3.5q-2.06 0-3.53 1.19T10 7.5q0 2 2 3.19v.81q0 .44.28.72t.72.28h4q.44 0 .72-.28t.28-.72v-.81q2-1.19 2-3.19 0-1.62-1.47-2.81T15 3.5zM7.81 13.94q-.18.5.25.69l4.44 1.87-4.44 1.88q-.44.18-.25.68l.38.88q.19.5.69.31L15 17.56l6.13 2.7q.5.18.68-.32l.38-.88q.19-.5-.25-.68l-4.38-1.88 4.38-1.87q.44-.2.25-.7l-.38-.87q-.18-.5-.68-.31L15 15.38l-6.12-2.63q-.5-.19-.7.31zm16 14.06v-4H6q-.81 0-1.4.56T4 26t.56 1.44T6 28h17.81zM13 8.5q-.44 0-.72-.28T12 7.5t.28-.72.72-.28.72.28.28.72-.28.72-.72.28z',
                'w' => .875,
                'vw' => 28
            ],
            'book-journal-whills' => [
                'p' => 'M0 96C0 43 43 0 96 0H384h32c17.7 0 32 14.3 32 32V352c0 17.7-14.3 32-32 32v64c17.7 0 32 14.3 32 32s-14.3 32-32 32H384 96c-53 0-96-43-96-96V96zM64 416c0 17.7 14.3 32 32 32H352V384H96c-17.7 0-32 14.3-32 32zm90.4-234.4l-21.2-21.2c-3 10.1-5.1 20.6-5.1 31.6c0 .2 0 .5 .1 .8s.1 .5 .1 .8L165.2 226c2.5 2.1 3.4 5.8 2.3 8.9c-1.3 3-4.1 5.1-7.5 5.1c-1.9-.1-3.8-.8-5.2-2l-23.6-20.6C142.8 267 186.9 304 240 304s97.3-37 108.9-86.6L325.3 238c-1.4 1.2-3.3 2-5.3 2c-2.2-.1-4.4-1.1-6-2.8c-1.2-1.5-1.9-3.4-2-5.2c.1-2.2 1.1-4.4 2.8-6l37.1-32.5c0-.3 0-.5 .1-.8s.1-.5 .1-.8c0-11-2.1-21.5-5.1-31.6l-21.2 21.2c-3.1 3.1-8.1 3.1-11.3 0s-3.1-8.1 0-11.2l26.4-26.5c-8.2-17-20.5-31.7-35.9-42.6c-2.7-1.9-6.2 1.4-5 4.5c8.5 22.4 3.6 48-13 65.6c-3.2 3.4-3.6 8.9-.9 12.7c9.8 14 12.7 31.9 7.5 48.5c-5.9 19.4-22 34.1-41.9 38.3l-1.4-34.3 12.6 8.6c.6 .4 1.5 .6 2.3 .6c1.5 0 2.7-.8 3.5-2s.6-2.8-.1-4L260 225.4l18-3.6c1.8-.4 3.1-2.1 3.1-4s-1.4-3.5-3.1-3.9l-18-3.7 8.5-14.3c.8-1.2 .9-2.9 .1-4.1s-2-2-3.5-2l-.1 0c-.7 .1-1.5 .3-2.1 .7l-14.1 9.6L244 87.9c-.1-2.2-1.9-3.9-4-3.9s-3.9 1.6-4 3.9l-4.6 110.8-12-8.1c-1.5-1.1-3.6-.9-5 .4s-1.6 3.4-.8 5l8.6 14.3-18 3.7c-1.8 .4-3.1 2-3.1 3.9s1.4 3.6 3.1 4l18 3.8-8.6 14.2c-.2 .6-.5 1.4-.5 2c0 1.1 .5 2.1 1.2 3c.8 .6 1.8 1 2.8 1c.7 0 1.6-.2 2.2-.6l10.4-7.1-1.4 32.8c-19.9-4.1-36-18.9-41.9-38.3c-5.1-16.6-2.2-34.4 7.6-48.5c2.7-3.9 2.3-9.3-.9-12.7c-16.6-17.5-21.6-43.1-13.1-65.5c1.2-3.1-2.3-6.4-5-4.5c-15.3 10.9-27.6 25.6-35.8 42.6l26.4 26.5c3.1 3.1 3.1 8.1 0 11.2s-8.1 3.1-11.2 0z',
                'vw' => 448,
                'vh' => 512
            ],
            'book-medical' => [
                'p' => 'M28 22.38q0 .93-.63 1.25-.12.56-.12 2.18t.13 2.38q.62.62.62 1.19v1q0 .75-.47 1.18t-1.16.44H6q-2.56 0-4.28-1.72T0 26V6q0-2.56 1.72-4.28T6 0h20.38Q28 0 28 1.63v20.75zM9 10.5v3q0 .5.5.5H13v3.5q0 .5.5.5h3q.5 0 .5-.5V14h3.5q.5 0 .5-.5v-3q0-.5-.5-.5H17V6.5q0-.5-.5-.5h-3q-.5 0-.5.5V10H9.5q-.5 0-.5.5zM23.81 28v-4H6q-.81 0-1.4.56T4 26t.56 1.44T6 28h17.81z',
                'w' => .875,
                'vw' => 28
            ],
            'book-open' => [
                'p' => 'M33.88 2q.87-.06 1.5.5T36 3.94v21.12q0 .75-.53 1.32t-1.34.56q-8.76.5-13.7 2.94-.3.18-.65.12t-.56-.34-.22-.6V6.31q0-.56.44-.8Q24.25 2.55 33.87 2zM16.55 5.5q.44.25.44.81v22.75q0 .5-.47.78t-.97.03q-4.94-2.43-13.69-2.93-.8 0-1.34-.57T0 25.07V3.93q0-.88.63-1.44t1.5-.5q9.62.56 14.43 3.5z',
                'w' => 1.123,
                'vw' => 36
            ],
            'book-open-reader' => [
                'p' => 'M160 96a96 96 0 1 1 192 0A96 96 0 1 1 160 96zm80 152V512l-48.4-24.2c-20.9-10.4-43.5-17-66.8-19.3l-96-9.6C12.5 457.2 0 443.5 0 427V224c0-17.7 14.3-32 32-32H62.3c63.6 0 125.6 19.6 177.7 56zm32 264V248c52.1-36.4 114.1-56 177.7-56H480c17.7 0 32 14.3 32 32V427c0 16.4-12.5 30.2-28.8 31.8l-96 9.6c-23.2 2.3-45.9 8.9-66.8 19.3L272 512z',
                'vw' => 512,
                'vh' => 512
            ],
            'book-quran' => [
                'p' => 'M352 0c53 0 96 43 96 96V416c0 53-43 96-96 96H64 32c-17.7 0-32-14.3-32-32s14.3-32 32-32V384c-17.7 0-32-14.3-32-32V32C0 14.3 14.3 0 32 0H64 352zm0 384H96v64H352c17.7 0 32-14.3 32-32s-14.3-32-32-32zM274.1 150.2l-8.9 21.4-23.1 1.9c-5.7 .5-8 7.5-3.7 11.2L256 199.8l-5.4 22.6c-1.3 5.5 4.7 9.9 9.6 6.9L280 217.2l19.8 12.1c4.9 3 10.9-1.4 9.6-6.9L304 199.8l17.6-15.1c4.3-3.7 2-10.8-3.7-11.2l-23.1-1.9-8.9-21.4c-2.2-5.3-9.6-5.3-11.8 0zM96 192c0 70.7 57.3 128 128 128c25.6 0 49.5-7.5 69.5-20.5c3.2-2.1 4.5-6.2 3.1-9.7s-5.2-5.6-9-4.8c-6.1 1.2-12.5 1.9-19 1.9c-52.4 0-94.9-42.5-94.9-94.9s42.5-94.9 94.9-94.9c6.5 0 12.8 .7 19 1.9c3.8 .8 7.5-1.3 9-4.8s.2-7.6-3.1-9.7C273.5 71.5 249.6 64 224 64C153.3 64 96 121.3 96 192z',
                'vw' => 448,
                'vh' => 512
            ],
            'book-reader' => 'M22 6q0 2.5-1.75 4.25T16 12t-4.25-1.75T10 6t1.75-4.25T16 0t4.25 1.75T22 6zm-7.38 9.06q.38.25.38.75V31.2q0 .44-.4.69t-.85 0q-4.38-2.2-12.13-2.57-.68-.06-1.15-.53T0 27.63V13.69q0-.75.56-1.25T1.88 12q8.5.5 12.75 3.06zM30.13 12q.5 0 .94.22t.7.6.24.87v13.94q0 .68-.47 1.15t-1.15.53q-7.75.44-12.07 2.57-.31.18-.62.12t-.5-.31-.19-.5V15.8q0-.5.37-.75Q21.62 12.5 30.12 12z',
            'book-skull' => [
                'p' => 'M0 96C0 43 43 0 96 0H384h32c17.7 0 32 14.3 32 32V352c0 17.7-14.3 32-32 32v64c17.7 0 32 14.3 32 32s-14.3 32-32 32H384 96c-53 0-96-43-96-96V96zM64 416c0 17.7 14.3 32 32 32H352V384H96c-17.7 0-32 14.3-32 32zM320 112c0-35.3-35.8-64-80-64s-80 28.7-80 64c0 20.9 12.6 39.5 32 51.2V176c0 8.8 7.2 16 16 16h64c8.8 0 16-7.2 16-16V163.2c19.4-11.7 32-30.3 32-51.2zM208 96a16 16 0 1 1 0 32 16 16 0 1 1 0-32zm48 16a16 16 0 1 1 32 0 16 16 0 1 1 -32 0zM134.3 209.3c-8.1-3.5-17.5 .3-21 8.4s.3 17.5 8.4 21L199.4 272l-77.7 33.3c-8.1 3.5-11.9 12.9-8.4 21s12.9 11.9 21 8.4L240 289.4l105.7 45.3c8.1 3.5 17.5-.3 21-8.4s-.3-17.5-8.4-21L280.6 272l77.7-33.3c8.1-3.5 11.9-12.9 8.4-21s-12.9-11.9-21-8.4L240 254.6 134.3 209.3z',
                'vw' => 448,
                'vh' => 512
            ],
            'book-tanakh' => [
                'p' => 'M352 0c53 0 96 43 96 96V416c0 53-43 96-96 96H64 32c-17.7 0-32-14.3-32-32s14.3-32 32-32V384c-17.7 0-32-14.3-32-32V32C0 14.3 14.3 0 32 0H64 352zm0 384H96v64H352c17.7 0 32-14.3 32-32s-14.3-32-32-32zM138.7 208l13.9 24H124.9l13.9-24zm-13.9-24L97.1 232c-6.2 10.7 1.5 24 13.9 24h55.4l27.7 48c6.2 10.7 21.6 10.7 27.7 0l27.7-48H305c12.3 0 20-13.3 13.9-24l-27.7-48 27.7-48c6.2-10.7-1.5-24-13.9-24H249.6L221.9 64c-6.2-10.7-21.6-10.7-27.7 0l-27.7 48H111c-12.3 0-20 13.3-13.9 24l27.7 48zm27.7 0l27.7-48h55.4l27.7 48-27.7 48H180.3l-27.7-48zm0-48l-13.9 24-13.9-24h27.7zm41.6-24L208 88l13.9 24H194.1zm69.3 24h27.7l-13.9 24-13.9-24zm13.9 72l13.9 24H263.4l13.9-24zm-55.4 48L208 280l-13.9-24h27.7z',
                'vw' => 448,
                'vh' => 512
            ],
            'bookmark' => [
                'p' => 'M0 32V3Q0 1.75.88.87T3 0h18q1.25 0 2.13.88T24 3v29l-12-7z',
                'w' => .75,
                'vw' => 24
            ],
            'border-all' => [
                'p' => 'M26 2H2a2 2 0 0 0-2 2v24c0 1.1.9 2 2 2h24a2 2 0 0 0 2-2V4a2 2 0 0 0-2-2Zm-2 4v8h-8V6ZM12 6v8H4V6ZM4 26v-8h8v8Zm12 0v-8h8v8Zm0 0',
                'vw' => 28
            ],
            'border-none' => [
                'p' => 'M15 14h-2a1 1 0 0 0-1 1v2a1 1 0 0 0 1 1h2a1 1 0 0 0 1-1v-2a1 1 0 0 0-1-1Zm6 0h-2a1 1 0 0 0-1 1v2a1 1 0 0 0 1 1h2a1 1 0 0 0 1-1v-2a1 1 0 0 0-1-1Zm6 0h-2a1 1 0 0 0-1 1v2a1 1 0 0 0 1 1h2a1 1 0 0 0 1-1v-2a1 1 0 0 0-1-1ZM9 14H7a1 1 0 0 0-1 1v2a1 1 0 0 0 1 1h2a1 1 0 0 0 1-1v-2a1 1 0 0 0-1-1Zm6 12h-2a1 1 0 0 0-1 1v2a1 1 0 0 0 1 1h2a1 1 0 0 0 1-1v-2a1 1 0 0 0-1-1Zm6 0h-2a1 1 0 0 0-1 1v2a1 1 0 0 0 1 1h2a1 1 0 0 0 1-1v-2a1 1 0 0 0-1-1Zm6 0h-2a1 1 0 0 0-1 1v2a1 1 0 0 0 1 1h2a1 1 0 0 0 1-1v-2a1 1 0 0 0-1-1Zm0-6h-2a1 1 0 0 0-1 1v2a1 1 0 0 0 1 1h2a1 1 0 0 0 1-1v-2a1 1 0 0 0-1-1Zm0-12h-2a1 1 0 0 0-1 1v2a1 1 0 0 0 1 1h2a1 1 0 0 0 1-1V9a1 1 0 0 0-1-1ZM15 20h-2a1 1 0 0 0-1 1v2a1 1 0 0 0 1 1h2a1 1 0 0 0 1-1v-2a1 1 0 0 0-1-1Zm0-12h-2a1 1 0 0 0-1 1v2a1 1 0 0 0 1 1h2a1 1 0 0 0 1-1V9a1 1 0 0 0-1-1ZM9 26H7a1 1 0 0 0-1 1v2a1 1 0 0 0 1 1h2a1 1 0 0 0 1-1v-2a1 1 0 0 0-1-1Zm6-24h-2a1 1 0 0 0-1 1v2a1 1 0 0 0 1 1h2a1 1 0 0 0 1-1V3a1 1 0 0 0-1-1Zm6 0h-2a1 1 0 0 0-1 1v2a1 1 0 0 0 1 1h2a1 1 0 0 0 1-1V3a1 1 0 0 0-1-1Zm6 0h-2a1 1 0 0 0-1 1v2a1 1 0 0 0 1 1h2a1 1 0 0 0 1-1V3a1 1 0 0 0-1-1ZM3 14H1a1 1 0 0 0-1 1v2a1 1 0 0 0 1 1h2a1 1 0 0 0 1-1v-2a1 1 0 0 0-1-1Zm0 12H1a1 1 0 0 0-1 1v2a1 1 0 0 0 1 1h2a1 1 0 0 0 1-1v-2a1 1 0 0 0-1-1Zm0-6H1a1 1 0 0 0-1 1v2a1 1 0 0 0 1 1h2a1 1 0 0 0 1-1v-2a1 1 0 0 0-1-1ZM3 8H1a1 1 0 0 0-1 1v2a1 1 0 0 0 1 1h2a1 1 0 0 0 1-1V9a1 1 0 0 0-1-1Zm0-6H1a1 1 0 0 0-1 1v2a1 1 0 0 0 1 1h2a1 1 0 0 0 1-1V3a1 1 0 0 0-1-1Zm6 0H7a1 1 0 0 0-1 1v2a1 1 0 0 0 1 1h2a1 1 0 0 0 1-1V3a1 1 0 0 0-1-1Zm0 0',
                'vw' => 28
            ],
            'border-style' => [
                'p' => 'M15 26h-2a1 1 0 0 0-1 1v2a1 1 0 0 0 1 1h2a1 1 0 0 0 1-1v-2a1 1 0 0 0-1-1Zm-6 0H7a1 1 0 0 0-1 1v2a1 1 0 0 0 1 1h2a1 1 0 0 0 1-1v-2a1 1 0 0 0-1-1Zm12 0h-2a1 1 0 0 0-1 1v2a1 1 0 0 0 1 1h2a1 1 0 0 0 1-1v-2a1 1 0 0 0-1-1Zm6-12h-2a1 1 0 0 0-1 1v2a1 1 0 0 0 1 1h2a1 1 0 0 0 1-1v-2a1 1 0 0 0-1-1Zm0 6h-2a1 1 0 0 0-1 1v2a1 1 0 0 0 1 1h2a1 1 0 0 0 1-1v-2a1 1 0 0 0-1-1Zm0 6h-2a1 1 0 0 0-1 1v2a1 1 0 0 0 1 1h2a1 1 0 0 0 1-1v-2a1 1 0 0 0-1-1Zm0-18h-2a1 1 0 0 0-1 1v2a1 1 0 0 0 1 1h2a1 1 0 0 0 1-1V9a1 1 0 0 0-1-1Zm0-6H2a2 2 0 0 0-2 2v25a1 1 0 0 0 1 1h2a1 1 0 0 0 1-1V6h23a1 1 0 0 0 1-1V3a1 1 0 0 0-1-1Zm0 0',
                'vw' => 28
            ],
            'border-top-left' => [
                'p' => 'M0 448c0 17.7 14.3 32 32 32s32-14.3 32-32l0-336c0-8.8 7.2-16 16-16l336 0c17.7 0 32-14.3 32-32s-14.3-32-32-32L80 32C35.8 32 0 67.8 0 112L0 448zm160 0a32 32 0 1 0 -64 0 32 32 0 1 0 64 0zm192 0a32 32 0 1 0 -64 0 32 32 0 1 0 64 0zm-96 0a32 32 0 1 0 -64 0 32 32 0 1 0 64 0zm192 0a32 32 0 1 0 -64 0 32 32 0 1 0 64 0zM416 288a32 32 0 1 0 0-64 32 32 0 1 0 0 64zm0 32a32 32 0 1 0 0 64 32 32 0 1 0 0-64zm0-128a32 32 0 1 0 0-64 32 32 0 1 0 0 64z',
                'vw' => 448,
                'vh' => 512
            ],
            'bore-hole' => [
                'p' => 'M256 0c-17.7 0-32 14.3-32 32V296.6c-19.1 11.1-32 31.7-32 55.4c0 35.3 28.7 64 64 64s64-28.7 64-64c0-23.7-12.9-44.4-32-55.4V32c0-17.7-14.3-32-32-32zM48 128c-26.5 0-48 21.5-48 48V464c0 26.5 21.5 48 48 48H464c26.5 0 48-21.5 48-48V176c0-26.5-21.5-48-48-48H384c-17.7 0-32 14.3-32 32V352c0 53-43 96-96 96s-96-43-96-96V160c0-17.7-14.3-32-32-32H48z',
                'vw' => 512,
                'vh' => 512
            ],
            'bottle-droplet' => [
                'p' => 'M96 0C82.7 0 72 10.7 72 24s10.7 24 24 24c4.4 0 8 3.6 8 8v64.9c0 12.2-7.2 23.1-17.2 30.1C53.7 174.1 32 212.5 32 256V448c0 35.3 28.7 64 64 64H224c35.3 0 64-28.7 64-64V256c0-43.5-21.7-81.9-54.8-105c-10-7-17.2-17.9-17.2-30.1V56c0-4.4 3.6-8 8-8c13.3 0 24-10.7 24-24s-10.7-24-24-24l-8 0 0 0 0 0H104l0 0 0 0L96 0zm64 382c-26.5 0-48-20.1-48-45c0-16.8 22.1-48.1 36.3-66.4c6-7.8 17.5-7.8 23.5 0C185.9 288.9 208 320.2 208 337c0 24.9-21.5 45-48 45z',
                'vw' => 320,
                'vh' => 512
            ],
            'bottle-water' => [
                'p' => 'M120 0h80c13.3 0 24 10.7 24 24V64H96V24c0-13.3 10.7-24 24-24zM32 151.7c0-15.6 9-29.8 23.2-36.5l24.4-11.4c11-5.1 23-7.8 35.1-7.8h90.6c12.1 0 24.1 2.7 35.1 7.8l24.4 11.4c14.1 6.6 23.2 20.8 23.2 36.5c0 14.4-7.5 27-18.9 34.1c11.5 8.8 18.9 22.6 18.9 38.2c0 16.7-8.5 31.4-21.5 40c12.9 8.6 21.5 23.3 21.5 40s-8.5 31.4-21.5 40c12.9 8.6 21.5 23.3 21.5 40s-8.5 31.4-21.5 40c12.9 8.6 21.5 23.3 21.5 40c0 26.5-21.5 48-48 48H80c-26.5 0-48-21.5-48-48c0-16.7 8.5-31.4 21.5-40C40.5 415.4 32 400.7 32 384s8.5-31.4 21.5-40C40.5 335.4 32 320.7 32 304s8.5-31.4 21.5-40C40.5 255.4 32 240.7 32 224c0-15.6 7.4-29.4 18.9-38.2C39.5 178.7 32 166.1 32 151.7zM96 240c0 8.8 7.2 16 16 16h96c8.8 0 16-7.2 16-16s-7.2-16-16-16H112c-8.8 0-16 7.2-16 16zm16 112c-8.8 0-16 7.2-16 16s7.2 16 16 16h96c8.8 0 16-7.2 16-16s-7.2-16-16-16H112z',
                'vw' => 320,
                'vh' => 512
            ],
            'bowl-food' => [
                'p' => 'M0 192c0-35.3 28.7-64 64-64c.5 0 1.1 0 1.6 0C73 91.5 105.3 64 144 64c15 0 29 4.1 40.9 11.2C198.2 49.6 225.1 32 256 32s57.8 17.6 71.1 43.2C339 68.1 353 64 368 64c38.7 0 71 27.5 78.4 64c.5 0 1.1 0 1.6 0c35.3 0 64 28.7 64 64c0 11.7-3.1 22.6-8.6 32H8.6C3.1 214.6 0 203.7 0 192zm0 91.4C0 268.3 12.3 256 27.4 256H484.6c15.1 0 27.4 12.3 27.4 27.4c0 70.5-44.4 130.7-106.7 154.1L403.5 452c-2 16-15.6 28-31.8 28H140.2c-16.1 0-29.8-12-31.8-28l-1.8-14.4C44.4 414.1 0 353.9 0 283.4z',
                'vw' => 512,
                'vh' => 512
            ],
            'bowl-rice' => [
                'p' => 'M176 56c0-13.3 10.7-24 24-24h16c13.3 0 24 10.7 24 24s-10.7 24-24 24H200c-13.3 0-24-10.7-24-24zm24 48h16c13.3 0 24 10.7 24 24s-10.7 24-24 24H200c-13.3 0-24-10.7-24-24s10.7-24 24-24zM56 176H72c13.3 0 24 10.7 24 24s-10.7 24-24 24H56c-13.3 0-24-10.7-24-24s10.7-24 24-24zM0 283.4C0 268.3 12.3 256 27.4 256H484.6c15.1 0 27.4 12.3 27.4 27.4c0 70.5-44.4 130.7-106.7 154.1L403.5 452c-2 16-15.6 28-31.8 28H140.2c-16.1 0-29.8-12-31.8-28l-1.8-14.4C44.4 414.1 0 353.9 0 283.4zM224 200c0-13.3 10.7-24 24-24h16c13.3 0 24 10.7 24 24s-10.7 24-24 24H248c-13.3 0-24-10.7-24-24zm-96 0c0-13.3 10.7-24 24-24h16c13.3 0 24 10.7 24 24s-10.7 24-24 24H152c-13.3 0-24-10.7-24-24zm-24-96h16c13.3 0 24 10.7 24 24s-10.7 24-24 24H104c-13.3 0-24-10.7-24-24s10.7-24 24-24zm216 96c0-13.3 10.7-24 24-24h16c13.3 0 24 10.7 24 24s-10.7 24-24 24H344c-13.3 0-24-10.7-24-24zm-24-96h16c13.3 0 24 10.7 24 24s-10.7 24-24 24H296c-13.3 0-24-10.7-24-24s10.7-24 24-24zm120 96c0-13.3 10.7-24 24-24h16c13.3 0 24 10.7 24 24s-10.7 24-24 24H440c-13.3 0-24-10.7-24-24zm-24-96h16c13.3 0 24 10.7 24 24s-10.7 24-24 24H392c-13.3 0-24-10.7-24-24s10.7-24 24-24zM296 32h16c13.3 0 24 10.7 24 24s-10.7 24-24 24H296c-13.3 0-24-10.7-24-24s10.7-24 24-24z',
                'vw' => 512,
                'vh' => 512
            ],
            'bowling-ball' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zM7.5 12q.81 0 1.4-.6t.6-1.4-.6-1.4T7.5 8t-1.4.6-.6 1.4.6 1.4 1.4.6zm4-6q0 .81.6 1.4t1.4.6 1.4-.6.6-1.4-.6-1.4-1.4-.6-1.4.6-.6 1.4zm3 9q.81 0 1.4-.6t.6-1.4-.6-1.4-1.4-.6-1.4.6-.6 1.4.6 1.4 1.4.6z',
            'box' => 'm31.88 11.56.06.44H17V0h8.81q.7 0 1.25.25t1 .72.63 1.1zM15 0v12H.06l.06-.44 3.2-9.5q.3-.94 1.09-1.5T6.19 0H15zM0 14h32v15q0 1.25-.88 2.13T29 32H3q-1.25 0-2.13-.88T0 29V14z',
            'box-archive' => [
                'p' => 'M32 32H480c17.7 0 32 14.3 32 32V96c0 17.7-14.3 32-32 32H32C14.3 128 0 113.7 0 96V64C0 46.3 14.3 32 32 32zm0 128H480V416c0 35.3-28.7 64-64 64H96c-35.3 0-64-28.7-64-64V160zm128 80c0 8.8 7.2 16 16 16H336c8.8 0 16-7.2 16-16s-7.2-16-16-16H176c-8.8 0-16 7.2-16 16z',
                'vw' => 512,
                'vh' => 512
            ],
            'box-open' => [
                'p' => 'M26.63 16q.43 0 .8-.13L36 13.45v11.12q0 .69-.44 1.22t-1.06.72l-13.56 3.38q-.94.25-1.94 0L5.5 26.5q-.63-.19-1.06-.72T4 24.56V13.44l8.56 2.44q.38.12.81.12 1.75 0 2.63-1.44l4-6.69 4 6.7Q24.87 16 26.62 16zm13.25-7q.24.44.06.88t-.69.56l-12.38 3.5q-.75.25-1.12-.44L20 4l15.63-2q.74-.06 1.06.56zM3.3 2.56q.32-.62 1.07-.56L20 4l-5.75 9.5q-.37.69-1.12.44L.75 10.44q-.5-.13-.69-.56T.13 9z',
                'w' => 1.25,
                'vw' => 40
            ],
            'box-tissue' => 'm24 17.99 4-12h-6.84c-1.9 0-3.57-1.2-4.17-3-.6-1.8-2.27-3-4.16-3H3.99l4 18Zm-24 12a2 2 0 0 0 2 2h28a2 2 0 0 0 2-2v-4H0Zm30-16h-2.57l-1.33 4h.9a1 1 0 0 1 0 2H5a1 1 0 1 1 0-2h.94l-.89-4H2a2 2 0 0 0-2 2v8h32v-8a2 2 0 0 0-2-2Zm0 0',
            'boxes' => [
                'p' => 'M35 18q.44 0 .72.28T36 19v12q0 .44-.28.72T35 32H21q-.44 0-.72-.28T20 31V19q0-.44.28-.72T21 18h5v6l2-1.31L30 24v-6h5zm-24-4q-.44 0-.72-.28T10 13V1q0-.44.28-.72T11 0h5v6l2-1.31L20 6V0h5q.44 0 .72.28T26 1v12q0 .44-.28.72T25 14H11zm4 4q.44 0 .72.28T16 19v12q0 .44-.28.72T15 32H1q-.44 0-.72-.28T0 31V19q0-.44.28-.72T1 18h5v6l2-1.31L10 24v-6h5z',
                'w' => 1.125,
                'vw' => 36
            ],
            'boxes-packing' => [
                'p' => 'M256 48c0-26.5 21.5-48 48-48H592c26.5 0 48 21.5 48 48V464c0 26.5-21.5 48-48 48H381.3c1.8-5 2.7-10.4 2.7-16V253.3c18.6-6.6 32-24.4 32-45.3V176c0-26.5-21.5-48-48-48H256V48zM571.3 347.3c6.2-6.2 6.2-16.4 0-22.6l-64-64c-6.2-6.2-16.4-6.2-22.6 0l-64 64c-6.2 6.2-6.2 16.4 0 22.6s16.4 6.2 22.6 0L480 310.6V432c0 8.8 7.2 16 16 16s16-7.2 16-16V310.6l36.7 36.7c6.2 6.2 16.4 6.2 22.6 0zM0 176c0-8.8 7.2-16 16-16H368c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H16c-8.8 0-16-7.2-16-16V176zm352 80V480c0 17.7-14.3 32-32 32H64c-17.7 0-32-14.3-32-32V256H352zM144 320c-8.8 0-16 7.2-16 16s7.2 16 16 16h96c8.8 0 16-7.2 16-16s-7.2-16-16-16H144z',
                'vw' => 640,
                'vh' => 512
            ],
            'boxes-stacked' => [
                'p' => 'M248 0H208c-26.5 0-48 21.5-48 48V160c0 35.3 28.7 64 64 64H352c35.3 0 64-28.7 64-64V48c0-26.5-21.5-48-48-48H328V80c0 8.8-7.2 16-16 16H264c-8.8 0-16-7.2-16-16V0zM64 256c-35.3 0-64 28.7-64 64V448c0 35.3 28.7 64 64 64H224c35.3 0 64-28.7 64-64V320c0-35.3-28.7-64-64-64H184v80c0 8.8-7.2 16-16 16H120c-8.8 0-16-7.2-16-16V256H64zM352 512H512c35.3 0 64-28.7 64-64V320c0-35.3-28.7-64-64-64H472v80c0 8.8-7.2 16-16 16H408c-8.8 0-16-7.2-16-16V256H352c-15 0-28.8 5.1-39.7 13.8c4.9 10.4 7.7 22 7.7 34.2V464c0 12.2-2.8 23.8-7.7 34.2C323.2 506.9 337 512 352 512z',
                'vw' => 576,
                'vh' => 512
            ],
            'braille' => [
                'p' => 'M8 16q0 1.69-1.16 2.84T4 20t-2.84-1.16T0 16t1.16-2.84T4 12t2.84 1.16T8 16zm-4 8q.81 0 1.4.6T6 26t-.6 1.4T4 28t-1.4-.6T2 26t.6-1.4T4 24zM4 2q1.69 0 2.84 1.16T8 6 6.84 8.84 4 10 1.16 8.84 0 6t1.16-2.84T4 2zm10 12q.81 0 1.4.6T16 16t-.6 1.4-1.4.6-1.4-.6T12 16t.6-1.4T14 14zm0 10q.81 0 1.4.6T16 26t-.6 1.4-1.4.6-1.4-.6T12 26t.6-1.4T14 24zm0-22q1.69 0 2.84 1.16T18 6t-1.16 2.84T14 10t-2.84-1.16T10 6t1.16-2.84T14 2zm14 12q.81 0 1.4.6T30 16t-.6 1.4-1.4.6-1.4-.6T26 16t.6-1.4T28 14zm0 10q.81 0 1.4.6T30 26t-.6 1.4-1.4.6-1.4-.6T26 26t.6-1.4T28 24zm0-22q1.69 0 2.84 1.16T32 6t-1.16 2.84T28 10t-2.84-1.16T24 6t1.16-2.84T28 2zm10 12q.81 0 1.4.6T40 16t-.6 1.4-1.4.6-1.4-.6T36 16t.6-1.4T38 14zm0 10q.81 0 1.4.6T40 26t-.6 1.4-1.4.6-1.4-.6T36 26t.6-1.4T38 24zm0-20q.81 0 1.4.6T40 6t-.6 1.4T38 8t-1.4-.6T36 6t.6-1.4T38 4z',
                'w' => 1.25,
                'vw' => 40
            ],
            'brain' => [
                'p' => 'M13 0q1.69 0 2.84 1.16T17 4v4.63q0 1.18-.75 2.12T14.37 12q-.37.13-.37.5v1q0 .25.19.4t.44.1q1.3-.31 2.37-1.19V27.5q0 1.88-1.31 3.19T12.5 32q-1.44 0-2.63-.84t-1.62-2.22q-.44.06-.75.06-1.88 0-3.19-1.31T3 24.5q0-.44.06-.88-1.37-.56-2.22-1.8T0 19q0-2.88 2.5-4.31-.5-1.07-.5-2.2 0-1.55.84-2.8t2.28-1.82Q5 7.43 5 7q0-1.7 1.16-2.85T9 3h.12q.25-.88.79-1.53T11.25.4 13 0zm23 19q0 1.56-.84 2.81t-2.22 1.82q.06.43.06.87 0 1.88-1.31 3.19T28.49 29q-.3 0-.74-.06-.44 1.37-1.63 2.22T23.5 32q-1.88 0-3.19-1.31T19 27.5V12.81q1.06.88 2.37 1.2.25.05.44-.1t.19-.4v-1q0-.38-.38-.5-.75-.2-1.34-.66t-.94-1.2T19 8.64V4q0-1.69 1.15-2.84T23 0q.56 0 1.1.16t.96.4.78.63.63.84.4.97H27q1.69 0 2.84 1.16T31 7q0 .44-.13.88 1.44.56 2.28 1.8T34 12.5q0 1.13-.5 2.19Q36 16.13 36 19z',
                'w' => 1.125,
                'vw' => 36
            ],
            'brazilian-real-sign' => [
                'p' => 'M400 0c17.7 0 32 14.3 32 32V50.2c12.5 2.3 24.7 6.4 36.2 12.1l10.1 5.1c15.8 7.9 22.2 27.1 14.3 42.9s-27.1 22.2-42.9 14.3l-10.2-5.1c-9.9-5-20.9-7.5-32-7.5h-1.7c-29.8 0-53.9 24.1-53.9 53.9c0 22 13.4 41.8 33.9 50l52 20.8c44.7 17.9 74.1 61.2 74.1 109.4v3.4c0 51.2-33.6 94.6-80 109.2V480c0 17.7-14.3 32-32 32s-32-14.3-32-32V460.6c-15-3.5-29.4-9.7-42.3-18.3l-23.4-15.6c-14.7-9.8-18.7-29.7-8.9-44.4s29.7-18.7 44.4-8.9L361.2 389c10.8 7.2 23.4 11 36.3 11c27.9 0 50.5-22.6 50.5-50.5v-3.4c0-22-13.4-41.8-33.9-50l-52-20.8C317.3 257.4 288 214.1 288 165.9C288 114 321.5 70 368 54.2V32c0-17.7 14.3-32 32-32zM0 64C0 46.3 14.3 32 32 32h80c79.5 0 144 64.5 144 144c0 58.8-35.2 109.3-85.7 131.7l51.4 128.4c6.6 16.4-1.4 35-17.8 41.6s-35-1.4-41.6-17.8L106.3 320H64V448c0 17.7-14.3 32-32 32s-32-14.3-32-32V288 64zM64 256h48c44.2 0 80-35.8 80-80s-35.8-80-80-80H64V256z',
                'vw' => 512,
                'vh' => 512
            ],
            'bread-slice' => [
                'p' => 'M18 0q4.06 0 7.53.97t5.72 2.53 3.5 3.4T36 10.57q0 1.5-1.1 2.47T32 14v16q0 .81-.66 1.4t-1.59.6H6.25q-.94 0-1.6-.6T4 30V14q-.94 0-1.69-.28t-1.25-.75-.78-1.1-.28-1.3q0-3.88 5.03-7.23T18 0z',
                'w' => 1.125,
                'vw' => 36
            ],
            'bridge' => [
                'p' => 'M32 32C14.3 32 0 46.3 0 64S14.3 96 32 96H72v64H0V288c53 0 96 43 96 96v64c0 17.7 14.3 32 32 32h32c17.7 0 32-14.3 32-32V384c0-53 43-96 96-96s96 43 96 96v64c0 17.7 14.3 32 32 32h32c17.7 0 32-14.3 32-32V384c0-53 43-96 96-96V160H504V96h40c17.7 0 32-14.3 32-32s-14.3-32-32-32H32zM456 96v64H376V96h80zM328 96v64H248V96h80zM200 96v64H120V96h80z',
                'vw' => 576,
                'vh' => 512
            ],
            'bridge-circle-check' => [
                'p' => 'M64 32C46.3 32 32 46.3 32 64s14.3 32 32 32h40v64H32V288c53 0 96 43 96 96v64c0 17.7 14.3 32 32 32h32c17.7 0 32-14.3 32-32V384c0-53 43-96 96-96c6.3 0 12.4 .6 18.3 1.7C367.1 231.8 426.9 192 496 192c42.5 0 81.6 15.1 112 40.2V160H536V96h40c17.7 0 32-14.3 32-32s-14.3-32-32-32H64zM488 96v64H408V96h80zM360 96v64H280V96h80zM232 96v64H152V96h80zM640 368a144 144 0 1 0 -288 0 144 144 0 1 0 288 0zm-76.7-43.3c6.2 6.2 6.2 16.4 0 22.6l-72 72c-6.2 6.2-16.4 6.2-22.6 0l-40-40c-6.2-6.2-6.2-16.4 0-22.6s16.4-6.2 22.6 0L480 385.4l60.7-60.7c6.2-6.2 16.4-6.2 22.6 0z',
                'vw' => 640,
                'vh' => 512
            ],
            'bridge-circle-exclamation' => [
                'p' => 'M64 32C46.3 32 32 46.3 32 64s14.3 32 32 32h40v64H32V288c53 0 96 43 96 96v64c0 17.7 14.3 32 32 32h32c17.7 0 32-14.3 32-32V384c0-53 43-96 96-96c6.3 0 12.4 .6 18.3 1.7C367.1 231.8 426.9 192 496 192c42.5 0 81.6 15.1 112 40.2V160H536V96h40c17.7 0 32-14.3 32-32s-14.3-32-32-32H64zM488 96v64H408V96h80zM360 96v64H280V96h80zM232 96v64H152V96h80zM496 512a144 144 0 1 0 0-288 144 144 0 1 0 0 288zm0-96a24 24 0 1 1 0 48 24 24 0 1 1 0-48zm0-144c8.8 0 16 7.2 16 16v80c0 8.8-7.2 16-16 16s-16-7.2-16-16V288c0-8.8 7.2-16 16-16z',
                'vw' => 640,
                'vh' => 512
            ],
            'bridge-circle-xmark' => [
                'p' => 'M64 32C46.3 32 32 46.3 32 64s14.3 32 32 32h40v64H32V288c53 0 96 43 96 96v64c0 17.7 14.3 32 32 32h32c17.7 0 32-14.3 32-32V384c0-53 43-96 96-96c6.3 0 12.4 .6 18.3 1.7C367.1 231.8 426.9 192 496 192c42.5 0 81.6 15.1 112 40.2V160H536V96h40c17.7 0 32-14.3 32-32s-14.3-32-32-32H64zM488 96v64H408V96h80zM360 96v64H280V96h80zM232 96v64H152V96h80zM496 512a144 144 0 1 0 0-288 144 144 0 1 0 0 288zm59.3-180.7L518.6 368l36.7 36.7c6.2 6.2 6.2 16.4 0 22.6s-16.4 6.2-22.6 0L496 390.6l-36.7 36.7c-6.2 6.2-16.4 6.2-22.6 0s-6.2-16.4 0-22.6L473.4 368l-36.7-36.7c-6.2-6.2-6.2-16.4 0-22.6s16.4-6.2 22.6 0L496 345.4l36.7-36.7c6.2-6.2 16.4-6.2 22.6 0s6.2 16.4 0 22.6z',
                'vw' => 640,
                'vh' => 512
            ],
            'bridge-lock' => [
                'p' => 'M32 64c0-17.7 14.3-32 32-32H576c17.7 0 32 14.3 32 32s-14.3 32-32 32H536v64h-8c-61.9 0-112 50.1-112 112v24.6c-9.9 5.8-18.2 14.1-23.8 24.1c-17.6-20-43.4-32.7-72.2-32.7c-53 0-96 43-96 96v64c0 17.7-14.3 32-32 32H160c-17.7 0-32-14.3-32-32V384c0-53-43-96-96-96V160h72V96H64C46.3 96 32 81.7 32 64zM408 96v64h80V96H408zm-48 64V96H280v64h80zM152 96v64h80V96H152zM528 240c-17.7 0-32 14.3-32 32v48h64V272c0-17.7-14.3-32-32-32zm-80 32c0-44.2 35.8-80 80-80s80 35.8 80 80v48c17.7 0 32 14.3 32 32V480c0 17.7-14.3 32-32 32H448c-17.7 0-32-14.3-32-32V352c0-17.7 14.3-32 32-32V272z',
                'vw' => 640,
                'vh' => 512
            ],
            'bridge-water' => [
                'p' => 'M0 96C0 78.3 14.3 64 32 64H544c17.7 0 32 14.3 32 32v35.6c0 15.7-12.7 28.4-28.4 28.4c-37.3 0-67.6 30.2-67.6 67.6V352.5c-12.9 0-25.8 3.9-36.8 11.7c-18 12.4-40.1 20.3-59.2 20.3h0l0-.5V256c0-53-43-96-96-96s-96 43-96 96V384l0 .5c-19 0-41.2-7.9-59.1-20.3c-11.1-7.8-24-11.7-36.9-11.7V227.6C96 190.2 65.8 160 28.4 160C12.7 160 0 147.3 0 131.6V96zM306.5 389.9C329 405.4 356.5 416 384 416c26.9 0 55.4-10.8 77.4-26.1l0 0c11.9-8.5 28.1-7.8 39.2 1.7c14.4 11.9 32.5 21 50.6 25.2c17.2 4 27.9 21.2 23.9 38.4s-21.2 27.9-38.4 23.9c-24.5-5.7-44.9-16.5-58.2-25C449.5 469.7 417 480 384 480c-31.9 0-60.6-9.9-80.4-18.9c-5.8-2.7-11.1-5.3-15.6-7.7c-4.5 2.4-9.7 5.1-15.6 7.7c-19.8 9-48.5 18.9-80.4 18.9c-33 0-65.5-10.3-94.5-25.8c-13.4 8.4-33.7 19.3-58.2 25c-17.2 4-34.4-6.7-38.4-23.9s6.7-34.4 23.9-38.4c18.1-4.2 36.2-13.3 50.6-25.2c11.1-9.4 27.3-10.1 39.2-1.7l0 0C136.7 405.2 165.1 416 192 416c27.5 0 55-10.6 77.5-26.1c11.1-7.9 25.9-7.9 37 0z',
                'vw' => 576,
                'vh' => 512
            ],
            'briefcase' => 'M20 21v-3h12v9q0 1.19-.9 2.1T29 30H3q-1.19 0-2.1-.9T0 27v-9h12v3q0 .44.28.72T13 22h6q.44 0 .72-.28T20 21zm9-13q1.19 0 2.1.9T32 11v5H0v-5q0-1.19.9-2.1T3 8h5V5q0-1.19.9-2.1T11 2h10q1.19 0 2.1.9T24 5v3h5zm-9 0V6h-8v2h8z',
            'briefcase-medical' => 'M29 8q1.25 0 2.13.88T32 11v18q0 1.25-.88 2.13T29 32H3q-1.25 0-2.13-.88T0 29V11q0-1.25.88-2.13T3 8h5V5q0-1.25.88-2.13T11 2h10q1.25 0 2.13.88T24 5v3h5zM12 6v2h8V6h-8zm10 15.5v-3q0-.5-.5-.5H18v-3.5q0-.5-.5-.5h-3q-.5 0-.5.5V18h-3.5q-.5 0-.5.5v3q0 .5.5.5H14v3.5q0 .5.5.5h3q.5 0 .5-.5V22h3.5q.5 0 .5-.5z',
            'broadcast-tower' => [
                'p' => 'M9.44 12q-.75 0-.94-.69Q8 9.7 8 8t.5-3.31q.19-.7.94-.7h2.12q.5 0 .78.41t.16.91Q12 6.62 12 8t.5 2.69q.13.5-.16.9t-.78.4H9.44zM5.63 1.44q-.75 1.37-1.22 3t-.35 4.53 1.56 5.6q.25.5-.03.96t-.84.47H2.56q-.62 0-.87-.56Q-.01 11.88-.01 8T1.63.56Q1.87 0 2.56 0h2.19q.56 0 .84.47t.04.97zM38.37.56Q40 4.13 40 8t-1.63 7.44q-.24.56-.93.56h-2.2q-.37 0-.65-.22t-.31-.56.1-.72q1.37-2.56 1.52-5.4t-.3-4.54-1.23-3.12q-.19-.31-.12-.66t.34-.56.66-.22h2.19q.68 0 .93.56zM30.57 4q.74 0 .93.69Q32 6.3 32 8t-.5 3.31q-.19.7-.94.7h-2.12q-.5 0-.78-.41t-.16-.91Q28 9.38 28 8t-.5-2.69q-.12-.5.16-.9t.78-.4h2.12zm-7.26 6.25 8.13 19.63q.19.37.03.75t-.53.5l-1.88.8q-.37.13-.75-.02t-.56-.53L24.69 24H15.3l-3.06 7.38q-.19.37-.56.53t-.75.03l-1.88-.82q-.37-.12-.53-.5t.03-.75l8.13-19.62q-.7-1-.7-2.25 0-1.06.54-2T18 4.53 20 4t2 .53T23.47 6 24 8q0 1.25-.69 2.25zM17 20h6l-3-7.25z',
                'w' => 1.25,
                'vw' => 40
            ],
            'broom' => [
                'p' => 'm16 13.56 5.44 6.82q0 .25-.1.68t-.5 1.72-.87 2.4-1.38 2.45-1.9 2.12Q15 31.07 10.84 31.6t-7.53.4L0 31.88q.25-1.44.69-3.44l5.94-7q.18-.25 0-.5t-.44-.13L2.44 22.2Q4 17.56 5.8 16.13q1.7-1.38 4.22-2t4.28-.63zM39.81 1.94q.57.81-.18 1.37l-14.5 11.13 2.12 2.69q.19.3.06.68t-.5.44l-3.69.75-5.37-6.81 1.56-3.38q.2-.37.57-.43t.62.25l2.13 2.68L37.13.2q.8-.56 1.43.19z',
                'w' => 1.25,
                'vw' => 40
            ],
            'broom-ball' => [
                'p' => 'M566.6 9.4c12.5 12.5 12.5 32.8 0 45.3l-192 192 34.7 34.7c4.2 4.2 6.6 10 6.6 16c0 12.5-10.1 22.6-22.6 22.6H364.3L256 211.7V182.6c0-12.5 10.1-22.6 22.6-22.6c6 0 11.8 2.4 16 6.6l34.7 34.7 192-192c12.5-12.5 32.8-12.5 45.3 0zm-344 225.5L341.1 353.4c3.7 42.7-11.7 85.2-42.3 115.8C271.4 496.6 234.2 512 195.5 512L22.1 512C9.9 512 0 502.1 0 489.9c0-6.3 2.7-12.3 7.3-16.5L133.7 359.7c4.2-3.7-.4-10.4-5.4-7.9L77.2 377.4c-6.1 3-13.2-1.4-13.2-8.2c0-31.5 12.5-61.7 34.8-84l8-8c30.6-30.6 73.1-45.9 115.8-42.3zM464 352a80 80 0 1 1 0 160 80 80 0 1 1 0-160z',
                'vw' => 576,
                'vh' => 512
            ],
            'brush' => [
                'p' => 'M22 0q.81 0 1.4.6T24 2v14H0V2Q0 1.19.6.6T2 0h20zM0 20v-2h24v2q0 1.69-1.16 2.84T20 24h-4v4q0 1.69-1.16 2.84T12 32t-2.84-1.16T8 28v-4H4q-1.69 0-2.84-1.16T0 20zm12 6.5q-.63 0-1.06.44T10.5 28t.44 1.06 1.06.44 1.06-.44.44-1.06-.44-1.06T12 26.5z',
                'w' => .75,
                'vw' => 24
            ],
            'bucket' => [
                'p' => 'M96 152v8H48v-8C48 68.1 116.1 0 200 0h48c83.9 0 152 68.1 152 152v8H352v-8c0-57.4-46.6-104-104-104H200C142.6 48 96 94.6 96 152zM0 224c0-17.7 14.3-32 32-32H416c17.7 0 32 14.3 32 32s-14.3 32-32 32h-5.1L388.5 469c-2.6 24.4-23.2 43-47.7 43H107.2c-24.6 0-45.2-18.5-47.7-43L37.1 256H32c-17.7 0-32-14.3-32-32z',
                'vw' => 448,
                'vh' => 512
            ],
            'bug' => 'M32 18.06q0 .82-.63 1.38t-1.43.56H26.5v1q0 2-.88 3.81l3.82 3.75Q30 29.2 30 30t-.6 1.4-1.4.6-1.44-.56L23.2 28q-2.5 2-5.7 2V14.75q0-.75-.74-.75h-1.5q-.75 0-.75.75V30q-3.19 0-5.69-2l-3.37 3.44Q4.8 32 4 32t-1.4-.6T2 30t.56-1.44l3.81-3.75q-.87-1.8-.87-3.8v-1H2.06q-.81 0-1.44-.57T0 18.06q0-.87.6-1.47T2 16h3.5v-3.69L2.56 9.44Q2 8.8 2 8t.6-1.4T4 6t1.43.56L8.81 10h14.38l3.37-3.44Q27.19 6 28 6t1.4.6T30 8t-.56 1.44L26.5 12.3V16H30q.81 0 1.4.6t.6 1.46zM16.06 0Q18.94 0 21 2.06T23.06 7h-14q0-2.87 2.07-4.94T16.07 0z',
            'bug-slash' => [
                'p' => 'M38.8 5.1C28.4-3.1 13.3-1.2 5.1 9.2S-1.2 34.7 9.2 42.9l592 464c10.4 8.2 25.5 6.3 33.7-4.1s6.3-25.5-4.1-33.7L477.4 348.9c1.7-9.4 2.6-19 2.6-28.9h64c17.7 0 32-14.3 32-32s-14.3-32-32-32H479.7c-1.1-14.1-5-27.5-11.1-39.5c.7-.6 1.4-1.2 2.1-1.9l64-64c12.5-12.5 12.5-32.8 0-45.3s-32.8-12.5-45.3 0l-64 64c-.7 .7-1.3 1.4-1.9 2.1C409.2 164.1 393.1 160 376 160H264c-8.3 0-16.3 1-24 2.8L38.8 5.1zM320 0c-53 0-96 43-96 96v3.6c0 15.7 12.7 28.4 28.4 28.4H387.6c15.7 0 28.4-12.7 28.4-28.4V96c0-53-43-96-96-96zM160.3 256H96c-17.7 0-32 14.3-32 32s14.3 32 32 32h64c0 24.6 5.5 47.8 15.4 68.6c-2.2 1.3-4.2 2.9-6 4.8l-64 64c-12.5 12.5-12.5 32.8 0 45.3s32.8 12.5 45.3 0l63.1-63.1c24.5 21.8 55.8 36.2 90.3 39.6V335.5L166.7 227.3c-3.4 9-5.6 18.7-6.4 28.7zM336 479.2c36.6-3.6 69.7-19.6 94.8-43.8L336 360.7V479.2z',
                'vw' => 640,
                'vh' => 512
            ],
            'bugs' => [
                'p' => 'M164.5 107.4l33.4-73.5c5.5-12.1 .1-26.3-11.9-31.8s-26.3-.1-31.8 11.9L128 71.7 101.9 14.1C96.4 2 82.1-3.3 70.1 2.1S52.7 21.9 58.1 33.9l33.4 73.5c-10.2 7.1-18.2 17-22.9 28.6h-17l-4.1-20.7c-2.6-13-15.2-21.4-28.2-18.8S-2.1 111.7 .5 124.7l8 40C10.7 175.9 20.6 184 32 184H64v23.3l-37.8 9.5c-9.5 2.4-16.6 10.2-17.9 19.9l-8 56c-1.9 13.1 7.2 25.3 20.4 27.2s25.3-7.2 27.2-20.4l5.7-40 18.4-4.6C82.7 274.6 103.8 288 128 288s45.3-13.4 56.1-33.2l18.4 4.6 5.7 40c1.9 13.1 14 22.2 27.2 20.4s22.2-14 20.4-27.2l-8-56c-1.4-9.7-8.5-17.5-17.9-19.9L192 207.3V184h32c11.4 0 21.3-8.1 23.5-19.3l8-40c2.6-13-5.8-25.6-18.8-28.2s-25.6 5.8-28.2 18.8L204.3 136h-17c-4.7-11.6-12.7-21.5-22.9-28.6zM496 286.5l65.6-47c10.8-7.7 13.3-22.7 5.6-33.5s-22.7-13.3-33.5-5.6l-51.4 36.8 6.1-62.9c1.3-13.2-8.4-24.9-21.6-26.2s-24.9 8.4-26.2 21.6L432.8 250c-12.3 1-24.2 5.6-34.1 13.3L384 254.8l6.8-20c4.2-12.6-2.5-26.2-15-30.4s-26.2 2.5-30.4 15l-13.1 38.6c-3.7 10.8 .8 22.8 10.7 28.5l27.7 16L359 322.7 321.5 312c-9.4-2.7-19.5 .6-25.5 8.3l-34.9 44.5c-8.2 10.4-6.4 25.5 4.1 33.7s25.5 6.4 33.7-4.1l25-31.8 18.2 5.2c-.5 22.6 11 44.7 32 56.8s45.9 11 65.2-.7l13.6 13.2-15.1 37.5c-4.9 12.3 1 26.3 13.3 31.2s26.3-1 31.2-13.3L503.5 440c3.6-9.1 1.4-19.4-5.6-26.2l-28-27.1 11.6-20.1 27.7 16c9.9 5.7 22.5 3.7 30-4.9L566.2 347c8.7-10 7.8-25.1-2.2-33.9s-25.1-7.8-33.9 2.2l-13.9 15.9-14.7-8.5c1.7-12.4-.2-25-5.5-36.2z',
                'vw' => 576,
                'vh' => 512
            ],
            'building' => [
                'p' => 'M27.25 30q.75 0 .75.75V32H0v-1.25Q0 30 .75 30H2V1.5q0-.63.44-1.06T3.5 0h21q.63 0 1.06.44T26 1.5V30h1.25zM8 4.75v2.5Q8 8 8.75 8h2.5Q12 8 12 7.25v-2.5Q12 4 11.25 4h-2.5Q8 4 8 4.75zm0 6v2.5q0 .75.75.75h2.5q.75 0 .75-.75v-2.5q0-.75-.75-.75h-2.5Q8 10 8 10.75zM11.25 20q.75 0 .75-.75v-2.5q0-.75-.75-.75h-2.5Q8 16 8 16.75v2.5q0 .75.75.75h2.5zM16 30v-5.25q0-.75-.75-.75h-2.5q-.75 0-.75.75V30h4zm4-10.75v-2.5q0-.75-.75-.75h-2.5q-.75 0-.75.75v2.5q0 .75.75.75h2.5q.75 0 .75-.75zm0-6v-2.5q0-.75-.75-.75h-2.5q-.75 0-.75.75v2.5q0 .75.75.75h2.5q.75 0 .75-.75zm0-6v-2.5Q20 4 19.25 4h-2.5Q16 4 16 4.75v2.5q0 .75.75.75h2.5Q20 8 20 7.25z',
                'w' => .875,
                'vw' => 28
            ],
            'building-circle-arrow-right' => [
                'p' => 'M0 48C0 21.5 21.5 0 48 0H336c26.5 0 48 21.5 48 48V232.2c-39.1 32.3-64 81.1-64 135.8c0 49.5 20.4 94.2 53.3 126.2C364.5 505.1 351.1 512 336 512H240V432c0-26.5-21.5-48-48-48s-48 21.5-48 48v80H48c-26.5 0-48-21.5-48-48V48zM80 224c-8.8 0-16 7.2-16 16v32c0 8.8 7.2 16 16 16h32c8.8 0 16-7.2 16-16V240c0-8.8-7.2-16-16-16H80zm80 16v32c0 8.8 7.2 16 16 16h32c8.8 0 16-7.2 16-16V240c0-8.8-7.2-16-16-16H176c-8.8 0-16 7.2-16 16zm112-16c-8.8 0-16 7.2-16 16v32c0 8.8 7.2 16 16 16h32c8.8 0 16-7.2 16-16V240c0-8.8-7.2-16-16-16H272zM64 112v32c0 8.8 7.2 16 16 16h32c8.8 0 16-7.2 16-16V112c0-8.8-7.2-16-16-16H80c-8.8 0-16 7.2-16 16zM176 96c-8.8 0-16 7.2-16 16v32c0 8.8 7.2 16 16 16h32c8.8 0 16-7.2 16-16V112c0-8.8-7.2-16-16-16H176zm80 16v32c0 8.8 7.2 16 16 16h32c8.8 0 16-7.2 16-16V112c0-8.8-7.2-16-16-16H272c-8.8 0-16 7.2-16 16zm96 256a144 144 0 1 1 288 0 144 144 0 1 1 -288 0zm140.7-67.3c-6.2 6.2-6.2 16.4 0 22.6L521.4 352H432c-8.8 0-16 7.2-16 16s7.2 16 16 16h89.4l-28.7 28.7c-6.2 6.2-6.2 16.4 0 22.6s16.4 6.2 22.6 0l56-56c6.2-6.2 6.2-16.4 0-22.6l-56-56c-6.2-6.2-16.4-6.2-22.6 0z',
                'vw' => 640,
                'vh' => 512
            ],
            'building-circle-check' => [
                'p' => 'M48 0C21.5 0 0 21.5 0 48V464c0 26.5 21.5 48 48 48h96V432c0-26.5 21.5-48 48-48s48 21.5 48 48v80h96c15.1 0 28.5-6.9 37.3-17.8C340.4 462.2 320 417.5 320 368c0-54.7 24.9-103.5 64-135.8V48c0-26.5-21.5-48-48-48H48zM64 240c0-8.8 7.2-16 16-16h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H80c-8.8 0-16-7.2-16-16V240zm112-16h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H176c-8.8 0-16-7.2-16-16V240c0-8.8 7.2-16 16-16zm80 16c0-8.8 7.2-16 16-16h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H272c-8.8 0-16-7.2-16-16V240zM80 96h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H80c-8.8 0-16-7.2-16-16V112c0-8.8 7.2-16 16-16zm80 16c0-8.8 7.2-16 16-16h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H176c-8.8 0-16-7.2-16-16V112zM272 96h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H272c-8.8 0-16-7.2-16-16V112c0-8.8 7.2-16 16-16zM640 368a144 144 0 1 0 -288 0 144 144 0 1 0 288 0zm-76.7-43.3c6.2 6.2 6.2 16.4 0 22.6l-72 72c-6.2 6.2-16.4 6.2-22.6 0l-40-40c-6.2-6.2-6.2-16.4 0-22.6s16.4-6.2 22.6 0L480 385.4l60.7-60.7c6.2-6.2 16.4-6.2 22.6 0z',
                'vw' => 640,
                'vh' => 512
            ],
            'building-circle-exclamation' => [
                'p' => 'M48 0C21.5 0 0 21.5 0 48V464c0 26.5 21.5 48 48 48h96V432c0-26.5 21.5-48 48-48s48 21.5 48 48v80h96c15.1 0 28.5-6.9 37.3-17.8C340.4 462.2 320 417.5 320 368c0-54.7 24.9-103.5 64-135.8V48c0-26.5-21.5-48-48-48H48zM64 240c0-8.8 7.2-16 16-16h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H80c-8.8 0-16-7.2-16-16V240zm112-16h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H176c-8.8 0-16-7.2-16-16V240c0-8.8 7.2-16 16-16zm80 16c0-8.8 7.2-16 16-16h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H272c-8.8 0-16-7.2-16-16V240zM80 96h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H80c-8.8 0-16-7.2-16-16V112c0-8.8 7.2-16 16-16zm80 16c0-8.8 7.2-16 16-16h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H176c-8.8 0-16-7.2-16-16V112zM272 96h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H272c-8.8 0-16-7.2-16-16V112c0-8.8 7.2-16 16-16zM496 512a144 144 0 1 0 0-288 144 144 0 1 0 0 288zm0-96a24 24 0 1 1 0 48 24 24 0 1 1 0-48zm0-144c8.8 0 16 7.2 16 16v80c0 8.8-7.2 16-16 16s-16-7.2-16-16V288c0-8.8 7.2-16 16-16z',
                'vw' => 640,
                'vh' => 512
            ],
            'building-circle-xmark' => [
                'p' => 'M48 0C21.5 0 0 21.5 0 48V464c0 26.5 21.5 48 48 48h96V432c0-26.5 21.5-48 48-48s48 21.5 48 48v80h96c15.1 0 28.5-6.9 37.3-17.8C340.4 462.2 320 417.5 320 368c0-54.7 24.9-103.5 64-135.8V48c0-26.5-21.5-48-48-48H48zM64 240c0-8.8 7.2-16 16-16h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H80c-8.8 0-16-7.2-16-16V240zm112-16h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H176c-8.8 0-16-7.2-16-16V240c0-8.8 7.2-16 16-16zm80 16c0-8.8 7.2-16 16-16h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H272c-8.8 0-16-7.2-16-16V240zM80 96h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H80c-8.8 0-16-7.2-16-16V112c0-8.8 7.2-16 16-16zm80 16c0-8.8 7.2-16 16-16h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H176c-8.8 0-16-7.2-16-16V112zM272 96h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H272c-8.8 0-16-7.2-16-16V112c0-8.8 7.2-16 16-16zM496 512a144 144 0 1 0 0-288 144 144 0 1 0 0 288zm59.3-180.7L518.6 368l36.7 36.7c6.2 6.2 6.2 16.4 0 22.6s-16.4 6.2-22.6 0L496 390.6l-36.7 36.7c-6.2 6.2-16.4 6.2-22.6 0s-6.2-16.4 0-22.6L473.4 368l-36.7-36.7c-6.2-6.2-6.2-16.4 0-22.6s16.4-6.2 22.6 0L496 345.4l36.7-36.7c6.2-6.2 16.4-6.2 22.6 0s6.2 16.4 0 22.6z',
                'vw' => 640,
                'vh' => 512
            ],
            'building-columns' => [
                'p' => 'M243.4 2.6l-224 96c-14 6-21.8 21-18.7 35.8S16.8 160 32 160v8c0 13.3 10.7 24 24 24H456c13.3 0 24-10.7 24-24v-8c15.2 0 28.3-10.7 31.3-25.6s-4.8-29.9-18.7-35.8l-224-96c-8-3.4-17.2-3.4-25.2 0zM128 224H64V420.3c-.6 .3-1.2 .7-1.8 1.1l-48 32c-11.7 7.8-17 22.4-12.9 35.9S17.9 512 32 512H480c14.1 0 26.5-9.2 30.6-22.7s-1.1-28.1-12.9-35.9l-48-32c-.6-.4-1.2-.7-1.8-1.1V224H384V416H344V224H280V416H232V224H168V416H128V224zM256 64a32 32 0 1 1 0 64 32 32 0 1 1 0-64z',
                'vw' => 512,
                'vh' => 512
            ],
            'building-flag' => [
                'p' => 'M48 0C21.5 0 0 21.5 0 48V464c0 26.5 21.5 48 48 48h96V432c0-26.5 21.5-48 48-48s48 21.5 48 48v80h96c26.5 0 48-21.5 48-48V48c0-26.5-21.5-48-48-48H48zM64 240c0-8.8 7.2-16 16-16h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H80c-8.8 0-16-7.2-16-16V240zm112-16h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H176c-8.8 0-16-7.2-16-16V240c0-8.8 7.2-16 16-16zm80 16c0-8.8 7.2-16 16-16h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H272c-8.8 0-16-7.2-16-16V240zM80 96h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H80c-8.8 0-16-7.2-16-16V112c0-8.8 7.2-16 16-16zm80 16c0-8.8 7.2-16 16-16h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H176c-8.8 0-16-7.2-16-16V112zM272 96h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H272c-8.8 0-16-7.2-16-16V112c0-8.8 7.2-16 16-16zM448 0c-17.7 0-32 14.3-32 32V512h64V192H624c8.8 0 16-7.2 16-16V48c0-8.8-7.2-16-16-16H480c0-17.7-14.3-32-32-32z',
                'vw' => 640,
                'vh' => 512
            ],
            'building-lock' => [
                'p' => 'M48 0C21.5 0 0 21.5 0 48V464c0 26.5 21.5 48 48 48h96V432c0-26.5 21.5-48 48-48s48 21.5 48 48v80h88.6c-5.4-9.4-8.6-20.3-8.6-32V352c0-23.7 12.9-44.4 32-55.4V272c0-30.5 12.2-58.2 32-78.4V48c0-26.5-21.5-48-48-48H48zM64 240c0-8.8 7.2-16 16-16h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H80c-8.8 0-16-7.2-16-16V240zm112-16h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H176c-8.8 0-16-7.2-16-16V240c0-8.8 7.2-16 16-16zm80 16c0-8.8 7.2-16 16-16h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H272c-8.8 0-16-7.2-16-16V240zM80 96h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H80c-8.8 0-16-7.2-16-16V112c0-8.8 7.2-16 16-16zm80 16c0-8.8 7.2-16 16-16h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H176c-8.8 0-16-7.2-16-16V112zM272 96h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H272c-8.8 0-16-7.2-16-16V112c0-8.8 7.2-16 16-16zM464 240c17.7 0 32 14.3 32 32v48H432V272c0-17.7 14.3-32 32-32zm-80 32v48c-17.7 0-32 14.3-32 32V480c0 17.7 14.3 32 32 32H544c17.7 0 32-14.3 32-32V352c0-17.7-14.3-32-32-32V272c0-44.2-35.8-80-80-80s-80 35.8-80 80z',
                'vw' => 576,
                'vh' => 512
            ],
            'building-ngo' => [
                'p' => 'M48 0C21.5 0 0 21.5 0 48V464c0 26.5 21.5 48 48 48h96V432c0-26.5 21.5-48 48-48s48 21.5 48 48v80h96c26.5 0 48-21.5 48-48V48c0-26.5-21.5-48-48-48H48zM64 240c0-8.8 7.2-16 16-16h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H80c-8.8 0-16-7.2-16-16V240zm112-16h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H176c-8.8 0-16-7.2-16-16V240c0-8.8 7.2-16 16-16zm80 16c0-8.8 7.2-16 16-16h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H272c-8.8 0-16-7.2-16-16V240zM168 64h48c8.8 0 16 7.2 16 16s-7.2 16-16 16H184v64h16V144c0-8.8 7.2-16 16-16s16 7.2 16 16v24c0 13.3-10.7 24-24 24H176c-13.3 0-24-10.7-24-24V80c0-8.8 7.2-16 16-16zM304 96c-8.8 0-16 7.2-16 16v32c0 8.8 7.2 16 16 16s16-7.2 16-16V112c0-8.8-7.2-16-16-16zm-48 16c0-26.5 21.5-48 48-48s48 21.5 48 48v32c0 26.5-21.5 48-48 48s-48-21.5-48-48V112zM61.3 71.1l34.7 52V80c0-8.8 7.2-16 16-16s16 7.2 16 16v96c0 7.1-4.6 13.3-11.4 15.3s-14-.6-17.9-6.4L64 132.8V176c0 8.8-7.2 16-16 16s-16-7.2-16-16V80c0-7.1 4.6-13.3 11.4-15.3s14 .6 17.9 6.4z',
                'vw' => 384,
                'vh' => 512
            ],
            'building-shield' => [
                'p' => 'M0 48C0 21.5 21.5 0 48 0H336c26.5 0 48 21.5 48 48V207l-42.4 17H304 272c-8.8 0-16 7.2-16 16v32 24.2V304c0 .9 .1 1.7 .2 2.6c2.3 58.1 24.1 144.8 98.7 201.5c-5.8 2.5-12.2 3.9-18.9 3.9H240V432c0-26.5-21.5-48-48-48s-48 21.5-48 48v80H48c-26.5 0-48-21.5-48-48V48zM80 224c-8.8 0-16 7.2-16 16v32c0 8.8 7.2 16 16 16h32c8.8 0 16-7.2 16-16V240c0-8.8-7.2-16-16-16H80zm80 16v32c0 8.8 7.2 16 16 16h32c8.8 0 16-7.2 16-16V240c0-8.8-7.2-16-16-16H176c-8.8 0-16 7.2-16 16zM64 112v32c0 8.8 7.2 16 16 16h32c8.8 0 16-7.2 16-16V112c0-8.8-7.2-16-16-16H80c-8.8 0-16 7.2-16 16zM176 96c-8.8 0-16 7.2-16 16v32c0 8.8 7.2 16 16 16h32c8.8 0 16-7.2 16-16V112c0-8.8-7.2-16-16-16H176zm80 16v32c0 8.8 7.2 16 16 16h32c8.8 0 16-7.2 16-16V112c0-8.8-7.2-16-16-16H272c-8.8 0-16 7.2-16 16zM423.1 225.7c5.7-2.3 12.1-2.3 17.8 0l120 48C570 277.4 576 286.2 576 296c0 63.3-25.9 168.8-134.8 214.2c-5.9 2.5-12.6 2.5-18.5 0C313.9 464.8 288 359.3 288 296c0-9.8 6-18.6 15.1-22.3l120-48zM527.4 312L432 273.8V461.7c68.2-33 91.5-99 95.4-149.7z',
                'vw' => 576,
                'vh' => 512
            ],
            'building-un' => [
                'p' => 'M48 0C21.5 0 0 21.5 0 48V464c0 26.5 21.5 48 48 48h96V432c0-26.5 21.5-48 48-48s48 21.5 48 48v80h96c26.5 0 48-21.5 48-48V48c0-26.5-21.5-48-48-48H48zM64 240c0-8.8 7.2-16 16-16h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H80c-8.8 0-16-7.2-16-16V240zm112-16h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H176c-8.8 0-16-7.2-16-16V240c0-8.8 7.2-16 16-16zm80 16c0-8.8 7.2-16 16-16h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H272c-8.8 0-16-7.2-16-16V240zM237.3 71.1l34.7 52V80c0-8.8 7.2-16 16-16s16 7.2 16 16v96c0 7.1-4.6 13.3-11.4 15.3s-14-.6-17.9-6.4l-34.7-52V176c0 8.8-7.2 16-16 16s-16-7.2-16-16V80c0-7.1 4.6-13.3 11.4-15.3s14 .6 17.9 6.4zM112 80v64c0 8.8 7.2 16 16 16s16-7.2 16-16V80c0-8.8 7.2-16 16-16s16 7.2 16 16v64c0 26.5-21.5 48-48 48s-48-21.5-48-48V80c0-8.8 7.2-16 16-16s16 7.2 16 16z',
                'vw' => 384,
                'vh' => 512
            ],
            'building-user' => [
                'p' => 'M48 0C21.5 0 0 21.5 0 48V464c0 26.5 21.5 48 48 48h96V432c0-26.5 21.5-48 48-48s48 21.5 48 48v80h89.9c-6.3-10.2-9.9-22.2-9.9-35.1c0-46.9 25.8-87.8 64-109.2V271.8 48c0-26.5-21.5-48-48-48H48zM64 240c0-8.8 7.2-16 16-16h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H80c-8.8 0-16-7.2-16-16V240zm112-16h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H176c-8.8 0-16-7.2-16-16V240c0-8.8 7.2-16 16-16zm80 16c0-8.8 7.2-16 16-16h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H272c-8.8 0-16-7.2-16-16V240zM80 96h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H80c-8.8 0-16-7.2-16-16V112c0-8.8 7.2-16 16-16zm80 16c0-8.8 7.2-16 16-16h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H176c-8.8 0-16-7.2-16-16V112zM272 96h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H272c-8.8 0-16-7.2-16-16V112c0-8.8 7.2-16 16-16zM576 272a80 80 0 1 0 -160 0 80 80 0 1 0 160 0zM352 477.1c0 19.3 15.6 34.9 34.9 34.9H605.1c19.3 0 34.9-15.6 34.9-34.9c0-51.4-41.7-93.1-93.1-93.1H445.1c-51.4 0-93.1 41.7-93.1 93.1z',
                'vw' => 640,
                'vh' => 512
            ],
            'building-wheat' => [
                'p' => 'M0 48C0 21.5 21.5 0 48 0H336c26.5 0 48 21.5 48 48V464c0 26.5-21.5 48-48 48H240V432c0-26.5-21.5-48-48-48s-48 21.5-48 48v80H48c-26.5 0-48-21.5-48-48V48zM80 224c-8.8 0-16 7.2-16 16v32c0 8.8 7.2 16 16 16h32c8.8 0 16-7.2 16-16V240c0-8.8-7.2-16-16-16H80zm80 16v32c0 8.8 7.2 16 16 16h32c8.8 0 16-7.2 16-16V240c0-8.8-7.2-16-16-16H176c-8.8 0-16 7.2-16 16zm112-16c-8.8 0-16 7.2-16 16v32c0 8.8 7.2 16 16 16h32c8.8 0 16-7.2 16-16V240c0-8.8-7.2-16-16-16H272zM64 112v32c0 8.8 7.2 16 16 16h32c8.8 0 16-7.2 16-16V112c0-8.8-7.2-16-16-16H80c-8.8 0-16 7.2-16 16zM176 96c-8.8 0-16 7.2-16 16v32c0 8.8 7.2 16 16 16h32c8.8 0 16-7.2 16-16V112c0-8.8-7.2-16-16-16H176zm80 16v32c0 8.8 7.2 16 16 16h32c8.8 0 16-7.2 16-16V112c0-8.8-7.2-16-16-16H272c-8.8 0-16 7.2-16 16zm384 80v16c0 44.2-35.8 80-80 80H544V272c0-44.2 35.8-80 80-80h16zm0 128c0 44.2-35.8 80-80 80H544V384c0-44.2 35.8-80 80-80h16v16zm0 112c0 44.2-35.8 80-80 80H544V496c0-44.2 35.8-80 80-80h16v16zM512 496v16H496c-44.2 0-80-35.8-80-80V416h16c44.2 0 80 35.8 80 80zm0-96H496c-44.2 0-80-35.8-80-80V304h16c44.2 0 80 35.8 80 80v16zm0-128v16H496c-44.2 0-80-35.8-80-80V192h16c44.2 0 80 35.8 80 80zM528 32c13.3 0 24 10.7 24 24V160c0 13.3-10.7 24-24 24s-24-10.7-24-24V56c0-13.3 10.7-24 24-24zm96 64v32c0 13.3-10.7 24-24 24s-24-10.7-24-24V96c0-13.3 10.7-24 24-24s24 10.7 24 24zM456 72c13.3 0 24 10.7 24 24v32c0 13.3-10.7 24-24 24s-24-10.7-24-24V96c0-13.3 10.7-24 24-24z',
                'vw' => 640,
                'vh' => 512
            ],
            'bullhorn' => [
                'p' => 'M36 15q0 2.31-2 3.44V28q0 .63-.5 1.31t-1.5.7q-.69 0-1.25-.45l-5.31-4.25q-4.13-3.3-9.44-3.3h-1.75Q14 23 14 24q0 2.68 1.62 4.8.7 1 .13 2.1T14 32H9.3q-1.25 0-1.75-1.07Q6 27.63 6 24q0-1 .12-2H4q-1.69 0-2.85-1.16T0 18v-6q0-1.69 1.15-2.84T4 8h12q5.31 0 9.44-3.31l5.3-4.25Q31.32 0 32 0q1 0 1.5.69T34 2v9.56q2 1.13 2 3.44zm-6 8.81V6.2l-2.06 1.6Q22.69 12 16 12v6q6.69 0 11.94 4.18z',
                'w' => 1.125,
                'vw' => 36
            ],
            'bullseye' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm0 27q4.75 0 8.13-3.38T27 16t-3.38-8.13T15.5 4.5 7.37 7.88 4 16t3.38 8.13 8.12 3.37zm0-19.5q3.31 0 5.66 2.34T23.5 16t-2.34 5.66T15.5 24t-5.66-2.34T7.5 16t2.34-5.66T15.5 8zm0 12q1.63 0 2.81-1.19t1.2-2.81-1.2-2.81-2.81-1.2-2.81 1.2-1.2 2.8 1.2 2.82T15.5 20z',
            'burger' => [
                'p' => 'M61.1 224C45 224 32 211 32 194.9c0-1.9 .2-3.7 .6-5.6C37.9 168.3 78.8 32 256 32s218.1 136.3 223.4 157.3c.5 1.9 .6 3.7 .6 5.6c0 16.1-13 29.1-29.1 29.1H61.1zM144 128a16 16 0 1 0 -32 0 16 16 0 1 0 32 0zm240 16a16 16 0 1 0 0-32 16 16 0 1 0 0 32zM272 96a16 16 0 1 0 -32 0 16 16 0 1 0 32 0zM16 304c0-26.5 21.5-48 48-48H448c26.5 0 48 21.5 48 48s-21.5 48-48 48H64c-26.5 0-48-21.5-48-48zm16 96c0-8.8 7.2-16 16-16H464c8.8 0 16 7.2 16 16v16c0 35.3-28.7 64-64 64H96c-35.3 0-64-28.7-64-64V400z',
                'vw' => 512,
                'vh' => 512
            ],
            'burn' => [
                'p' => 'M12 0q5.31 4.81 8.66 9.97T24 18.75q0 5.88-3.34 9.56T12 32.01t-8.66-3.7T0 18.75q0-3.63 3.31-8.78T12.01 0zm0 28q2.63 0 4.31-1.66t1.7-4.28q0-3.5-6-10.06-6 6.56-6 10.06 0 2.63 1.68 4.28T12 28z',
                'w' => .75,
                'vw' => 24
            ],
            'burst' => [
                'p' => 'M37.6 4.2C28-2.3 15.2-1.1 7 7s-9.4 21-2.8 30.5l112 163.3L16.6 233.2C6.7 236.4 0 245.6 0 256s6.7 19.6 16.6 22.8l103.1 33.4L66.8 412.8c-4.9 9.3-3.2 20.7 4.3 28.1s18.8 9.2 28.1 4.3l100.6-52.9 33.4 103.1c3.2 9.9 12.4 16.6 22.8 16.6s19.6-6.7 22.8-16.6l33.4-103.1 100.6 52.9c9.3 4.9 20.7 3.2 28.1-4.3s9.2-18.8 4.3-28.1L392.3 312.2l103.1-33.4c9.9-3.2 16.6-12.4 16.6-22.8s-6.7-19.6-16.6-22.8L388.9 198.7l25.7-70.4c3.2-8.8 1-18.6-5.6-25.2s-16.4-8.8-25.2-5.6l-70.4 25.7L278.8 16.6C275.6 6.7 266.4 0 256 0s-19.6 6.7-22.8 16.6l-32.3 99.6L37.6 4.2z',
                'vw' => 512,
                'vh' => 512
            ],
            'bus' => 'M30.5 8q.63 0 1.06.44T32 9.5v5q0 .63-.44 1.06T30.5 16H30v10.38q0 .69-.44 1.15t-1.19.47H28v2q0 .82-.6 1.4T26 32h-2q-.81 0-1.4-.6T22 30v-2H10v2q0 .82-.6 1.4T8 32H6q-.81 0-1.4-.6T4 30v-2q-.81 0-1.4-.6T2 26V16h-.5q-.63 0-1.06-.44T0 14.5v-5q0-.62.44-1.06T1.5 8H2V5q0-2.13 4.06-3.56T16 0t9.94 1.44T30 5v3h.5zM7 25q.81 0 1.4-.6T9 23t-.6-1.4T7 21t-1.4.6T5 23t.6 1.4T7 25zm1-7h16q.81 0 1.4-.6T26 16V8q0-.81-.6-1.4T24 6H8q-.81 0-1.4.6T6 8v8q0 .81.6 1.4T8 18zm17 7q.81 0 1.4-.6T27 23t-.6-1.4T25 21t-1.4.6T23 23t.6 1.4 1.4.6z',
            'bus-alt' => 'M30.5 8q.63 0 1.06.44T32 9.5v5q0 .63-.44 1.06T30.5 16H30v10.38q0 .69-.44 1.15t-1.19.47H28v2q0 .82-.6 1.4T26 32h-2q-.81 0-1.4-.6T22 30v-2H10v2q0 .82-.6 1.4T8 32H6q-.81 0-1.4-.6T4 30v-2q-.81 0-1.4-.6T2 26V16h-.5q-.63 0-1.06-.44T0 14.5v-5q0-.62.44-1.06T1.5 8H2V5q0-2.13 4.06-3.56T16 0t9.94 1.44T30 5v3h.5zM10 4.5v1q0 .5.5.5h11q.5 0 .5-.5v-1q0-.5-.5-.5h-11q-.5 0-.5.5zM7 25q.81 0 1.4-.6T9 23t-.6-1.4T7 21t-1.4.6T5 23t.6 1.4T7 25zm8-7V8H8q-.81 0-1.4.6T6 10v6q0 .81.6 1.4T8 18h7zm2 0h7q.81 0 1.4-.6T26 16v-6q0-.81-.6-1.4T24 8h-7v10zm8 7q.81 0 1.4-.6T27 23t-.6-1.4T25 21t-1.4.6T23 23t.6 1.4 1.4.6z',
            'bus-simple' => [
                'p' => 'M224 0C348.8 0 448 35.2 448 80V96 416c0 17.7-14.3 32-32 32v32c0 17.7-14.3 32-32 32H352c-17.7 0-32-14.3-32-32V448H128v32c0 17.7-14.3 32-32 32H64c-17.7 0-32-14.3-32-32l0-32c-17.7 0-32-14.3-32-32V96 80C0 35.2 99.2 0 224 0zM64 128V256c0 17.7 14.3 32 32 32H352c17.7 0 32-14.3 32-32V128c0-17.7-14.3-32-32-32H96c-17.7 0-32 14.3-32 32zM80 400a32 32 0 1 0 0-64 32 32 0 1 0 0 64zm288 0a32 32 0 1 0 0-64 32 32 0 1 0 0 64z',
                'vw' => 448,
                'vh' => 512
            ],
            'business-time' => [
                'p' => 'M31 14q3.75 0 6.38 2.63T40 23t-2.63 6.38T31 32t-6.38-2.63T22 23t2.63-6.38T31 14zm4 9.38v-.75q0-.63-.63-.63H32v-3.38q0-.62-.63-.62h-.75q-.62 0-.62.63v4.75q0 .62.63.62h3.75q.62 0 .62-.63zM31 12q-3.44 0-6.31 2H-.01V9Q0 7.81.9 6.9T3 6h5V3q0-1.19.9-2.1T11 0h10q1.19 0 2.1.9T24 3v3h5q1.19 0 2.1.9T32 9v3.06q-.56-.06-1-.06zM20 6V4h-8v2h8zm.44 14Q20 21.5 20 23q0 2.63 1.19 5H3q-1.19 0-2.1-.9T0 25v-9h12v3q0 .44.28.72T13 20h7.44z',
                'w' => 1.25,
                'vw' => 40
            ],
            'c' => [
                'p' => 'M329.1 142.9c-62.5-62.5-155.8-62.5-218.3 0s-62.5 163.8 0 226.3s155.8 62.5 218.3 0c12.5-12.5 32.8-12.5 45.3 0s12.5 32.8 0 45.3c-87.5 87.5-221.3 87.5-308.8 0s-87.5-229.3 0-316.8s221.3-87.5 308.8 0c12.5 12.5 12.5 32.8 0 45.3s-32.8 12.5-45.3 0z',
                'vw' => 384,
                'vh' => 512
            ],
            'cable-car' => [
                'p' => 'M288 0a32 32 0 1 1 0 64 32 32 0 1 1 0-64zM160 56a32 32 0 1 1 64 0 32 32 0 1 1 -64 0zM32 288c0-35.3 28.7-64 64-64H232V157.5l-203.1 42c-13 2.7-25.7-5.7-28.4-18.6s5.7-25.7 18.6-28.4l232-48 232-48c13-2.7 25.7 5.7 28.4 18.6s-5.7 25.7-18.6 28.4L280 147.5V224H416c35.3 0 64 28.7 64 64V448c0 35.3-28.7 64-64 64H96c-35.3 0-64-28.7-64-64V288zm64 0c-8.8 0-16 7.2-16 16v64c0 8.8 7.2 16 16 16h64c8.8 0 16-7.2 16-16V304c0-8.8-7.2-16-16-16H96zm112 16v64c0 8.8 7.2 16 16 16h64c8.8 0 16-7.2 16-16V304c0-8.8-7.2-16-16-16H224c-8.8 0-16 7.2-16 16zm144-16c-8.8 0-16 7.2-16 16v64c0 8.8 7.2 16 16 16h64c8.8 0 16-7.2 16-16V304c0-8.8-7.2-16-16-16H352z',
                'vw' => 512,
                'vh' => 512
            ],
            'cake-candles' => [
                'p' => 'M86.4 5.5L61.8 47.6C58 54.1 56 61.6 56 69.2V72c0 22.1 17.9 40 40 40s40-17.9 40-40V69.2c0-7.6-2-15-5.8-21.6L105.6 5.5C103.6 2.1 100 0 96 0s-7.6 2.1-9.6 5.5zm128 0L189.8 47.6c-3.8 6.5-5.8 14-5.8 21.6V72c0 22.1 17.9 40 40 40s40-17.9 40-40V69.2c0-7.6-2-15-5.8-21.6L233.6 5.5C231.6 2.1 228 0 224 0s-7.6 2.1-9.6 5.5zM317.8 47.6c-3.8 6.5-5.8 14-5.8 21.6V72c0 22.1 17.9 40 40 40s40-17.9 40-40V69.2c0-7.6-2-15-5.8-21.6L361.6 5.5C359.6 2.1 356 0 352 0s-7.6 2.1-9.6 5.5L317.8 47.6zM128 176c0-17.7-14.3-32-32-32s-32 14.3-32 32v48c-35.3 0-64 28.7-64 64v71c8.3 5.2 18.1 9 28.8 9c13.5 0 27.2-6.1 38.4-13.4c5.4-3.5 9.9-7.1 13-9.7c1.5-1.3 2.7-2.4 3.5-3.1c.4-.4 .7-.6 .8-.8l.1-.1 0 0 0 0s0 0 0 0s0 0 0 0c3.1-3.2 7.4-4.9 11.9-4.8s8.6 2.1 11.6 5.4l0 0 0 0 .1 .1c.1 .1 .4 .4 .7 .7c.7 .7 1.7 1.7 3.1 3c2.8 2.6 6.8 6.1 11.8 9.5c10.2 7.1 23 13.1 36.3 13.1s26.1-6 36.3-13.1c5-3.5 9-6.9 11.8-9.5c1.4-1.3 2.4-2.3 3.1-3c.3-.3 .6-.6 .7-.7l.1-.1c3-3.5 7.4-5.4 12-5.4s9 2 12 5.4l.1 .1c.1 .1 .4 .4 .7 .7c.7 .7 1.7 1.7 3.1 3c2.8 2.6 6.8 6.1 11.8 9.5c10.2 7.1 23 13.1 36.3 13.1s26.1-6 36.3-13.1c5-3.5 9-6.9 11.8-9.5c1.4-1.3 2.4-2.3 3.1-3c.3-.3 .6-.6 .7-.7l.1-.1c2.9-3.4 7.1-5.3 11.6-5.4s8.7 1.6 11.9 4.8l0 0 0 0 0 0 .1 .1c.2 .2 .4 .4 .8 .8c.8 .7 1.9 1.8 3.5 3.1c3.1 2.6 7.5 6.2 13 9.7c11.2 7.3 24.9 13.4 38.4 13.4c10.7 0 20.5-3.9 28.8-9V288c0-35.3-28.7-64-64-64V176c0-17.7-14.3-32-32-32s-32 14.3-32 32v48H256V176c0-17.7-14.3-32-32-32s-32 14.3-32 32v48H128V176zM448 394.6c-8.5 3.3-18.2 5.4-28.8 5.4c-22.5 0-42.4-9.9-55.8-18.6c-4.1-2.7-7.8-5.4-10.9-7.8c-2.8 2.4-6.1 5-9.8 7.5C329.8 390 310.6 400 288 400s-41.8-10-54.6-18.9c-3.5-2.4-6.7-4.9-9.4-7.2c-2.7 2.3-5.9 4.7-9.4 7.2C201.8 390 182.6 400 160 400s-41.8-10-54.6-18.9c-3.7-2.6-7-5.2-9.8-7.5c-3.1 2.4-6.8 5.1-10.9 7.8C71.2 390.1 51.3 400 28.8 400c-10.6 0-20.3-2.2-28.8-5.4V480c0 17.7 14.3 32 32 32H416c17.7 0 32-14.3 32-32V394.6z',
                'vw' => 448,
                'vh' => 512
            ],
            'calculator' => [
                'p' => 'M25 0q1.19 0 2.1.9T28 3v26q0 1.19-.9 2.1T25 32H3q-1.19 0-2.1-.9T0 29V3Q0 1.81.9.9T3 0h22zM8 27.19V24.8q0-.31-.25-.56T7.19 24H4.8q-.31 0-.56.25t-.25.56v2.38q0 .31.25.56t.56.25h2.4q.31 0 .56-.25T8 27.2zm0-8V16.8q0-.31-.25-.56T7.19 16H4.8q-.31 0-.56.25t-.25.56v2.38q0 .31.25.56t.56.25h2.4q.31 0 .56-.25T8 19.2zm8 8V24.8q0-.31-.25-.56t-.56-.25H12.8q-.31 0-.56.25t-.25.56v2.38q0 .31.25.56t.56.25h2.38q.31 0 .56-.25t.25-.56zm0-8V16.8q0-.31-.25-.56t-.56-.25H12.8q-.31 0-.56.25t-.25.56v2.38q0 .31.25.56t.56.25h2.38q.31 0 .56-.25t.25-.56zm8 8V16.8q0-.31-.25-.56t-.56-.25H20.8q-.31 0-.56.25t-.25.56v10.4q0 .31.25.56t.56.25h2.38q.31 0 .56-.25t.25-.56zm0-16V4.8q0-.31-.25-.56T23.19 4H4.8q-.31 0-.56.25T4 4.81v6.38q0 .31.25.56t.56.25H23.2q.31 0 .56-.25t.25-.56z',
                'w' => .875,
                'vw' => 28
            ],
            'calendar' => [
                'p' => 'M.75 12h26.5q.75 0 .75.75V29q0 1.25-.88 2.13T25 32H3q-1.25 0-2.13-.88T0 29V12.75Q0 12 .75 12zM28 9.25q0 .75-.75.75H.75Q0 10 0 9.25V7q0-1.25.88-2.13T3 4h3V.75Q6 0 6.75 0h2.5Q10 0 10 .75V4h8V.75q0-.75.75-.75h2.5Q22 0 22 .75V4h3q1.25 0 2.13.88T28 7v2.25z',
                'w' => .875,
                'vw' => 28
            ],
            'calendar-alt' => 'calendar-days',
            'calendar-check' => [
                'p' => 'M27.25 10H.75Q0 10 0 9.25V7q0-1.25.88-2.13T3 4h3V.75Q6 0 6.75 0h2.5Q10 0 10 .75V4h8V.75q0-.75.75-.75h2.5Q22 0 22 .75V4h3q1.25 0 2.13.88T28 7v2.25q0 .75-.75.75zM.75 12h26.5q.75 0 .75.75V29q0 1.25-.88 2.13T25 32H3q-1.25 0-2.13-.88T0 29V12.75Q0 12 .75 12zm20.81 6-1.75-1.75q-.5-.56-1.06-.06l-6.63 6.62-2.87-2.94q-.5-.5-1.06 0l-1.75 1.75q-.57.57 0 1.07l5.12 5.25q.56.5 1.06 0l8.94-8.88q.57-.56 0-1.06z',
                'w' => .875,
                'vw' => 28
            ],
            'calendar-day' => [
                'p' => 'M0 29V12h28v17q0 1.25-.88 2.13T25 32H3q-1.25 0-2.13-.88T0 29zm4-12v6q0 .44.28.72T5 24h6q.44 0 .72-.28T12 23v-6q0-.44-.28-.72T11 16H5q-.44 0-.72.28T4 17zM25 4q1.25 0 2.13.88T28 7v3H0V7q0-1.25.88-2.13T3 4h3V1q0-.44.28-.72T7 0h2q.44 0 .72.28T10 1v3h8V1q0-.44.28-.72T19 0h2q.44 0 .72.28T22 1v3h3z',
                'w' => .875,
                'vw' => 28
            ],
            'calendar-days' => [
                'p' => 'M128 0c17.7 0 32 14.3 32 32V64H288V32c0-17.7 14.3-32 32-32s32 14.3 32 32V64h48c26.5 0 48 21.5 48 48v48H0V112C0 85.5 21.5 64 48 64H96V32c0-17.7 14.3-32 32-32zM0 192H448V464c0 26.5-21.5 48-48 48H48c-26.5 0-48-21.5-48-48V192zm64 80v32c0 8.8 7.2 16 16 16h32c8.8 0 16-7.2 16-16V272c0-8.8-7.2-16-16-16H80c-8.8 0-16 7.2-16 16zm128 0v32c0 8.8 7.2 16 16 16h32c8.8 0 16-7.2 16-16V272c0-8.8-7.2-16-16-16H208c-8.8 0-16 7.2-16 16zm144-16c-8.8 0-16 7.2-16 16v32c0 8.8 7.2 16 16 16h32c8.8 0 16-7.2 16-16V272c0-8.8-7.2-16-16-16H336zM64 400v32c0 8.8 7.2 16 16 16h32c8.8 0 16-7.2 16-16V400c0-8.8-7.2-16-16-16H80c-8.8 0-16 7.2-16 16zm144-16c-8.8 0-16 7.2-16 16v32c0 8.8 7.2 16 16 16h32c8.8 0 16-7.2 16-16V400c0-8.8-7.2-16-16-16H208zm112 16v32c0 8.8 7.2 16 16 16h32c8.8 0 16-7.2 16-16V400c0-8.8-7.2-16-16-16H336c-8.8 0-16 7.2-16 16z',
                'vw' => 448,
                'vh' => 512
            ],
            'calendar-minus' => [
                'p' => 'M27.25 10H.75Q0 10 0 9.25V7q0-1.25.88-2.13T3 4h3V.75Q6 0 6.75 0h2.5Q10 0 10 .75V4h8V.75q0-.75.75-.75h2.5Q22 0 22 .75V4h3q1.25 0 2.13.88T28 7v2.25q0 .75-.75.75zM.75 12h26.5q.75 0 .75.75V29q0 1.25-.88 2.13T25 32H3q-1.25 0-2.13-.88T0 29V12.75Q0 12 .75 12zm19 12q.75 0 .75-.75v-2.5q0-.75-.75-.75H8.25q-.75 0-.75.75v2.5q0 .75.75.75h11.5z',
                'w' => .875,
                'vw' => 28
            ],
            'calendar-plus' => [
                'p' => 'M27.25 10H.75Q0 10 0 9.25V7q0-1.25.88-2.13T3 4h3V.75Q6 0 6.75 0h2.5Q10 0 10 .75V4h8V.75q0-.75.75-.75h2.5Q22 0 22 .75V4h3q1.25 0 2.13.88T28 7v2.25q0 .75-.75.75zM.75 12h26.5q.75 0 .75.75V29q0 1.25-.88 2.13T25 32H3q-1.25 0-2.13-.88T0 29V12.75Q0 12 .75 12zm19.75 8.75q0-.75-.75-.75H16v-3.75q0-.75-.75-.75h-2.5q-.75 0-.75.75V20H8.25q-.75 0-.75.75v2.5q0 .75.75.75H12v3.75q0 .75.75.75h2.5q.75 0 .75-.75V24h3.75q.75 0 .75-.75v-2.5z',
                'w' => .875,
                'vw' => 28
            ],
            'calendar-times' => [
                'p' => 'M27.25 10H.75Q0 10 0 9.25V7q0-1.25.88-2.13T3 4h3V.75Q6 0 6.75 0h2.5Q10 0 10 .75V4h8V.75q0-.75.75-.75h2.5Q22 0 22 .75V4h3q1.25 0 2.13.88T28 7v2.25q0 .75-.75.75zM.75 12h26.5q.75 0 .75.75V29q0 1.25-.88 2.13T25 32H3q-1.25 0-2.13-.88T0 29V12.75Q0 12 .75 12zm16.06 10 3-3q.57-.56 0-1.06l-1.75-1.75q-.5-.57-1.06 0l-3 3-3-3q-.56-.57-1.06 0l-1.75 1.75q-.57.5 0 1.06l3 3-3 3q-.57.56 0 1.06l1.75 1.75q.5.57 1.06 0l3-3 3 3q.56.57 1.06 0l1.75-1.75q.57-.5 0-1.06z',
                'w' => .875,
                'vw' => 28
            ],
            'calendar-week' => [
                'p' => 'M0 29V12h28v17q0 1.25-.88 2.13T25 32H3q-1.25 0-2.13-.88T0 29zm4-12v4q0 .44.28.72T5 22h18q.44 0 .72-.28T24 21v-4q0-.44-.28-.72T23 16H5q-.44 0-.72.28T4 17zM25 4q1.25 0 2.13.88T28 7v3H0V7q0-1.25.88-2.13T3 4h3V1q0-.44.28-.72T7 0h2q.44 0 .72.28T10 1v3h8V1q0-.44.28-.72T19 0h2q.44 0 .72.28T22 1v3h3z',
                'w' => .875,
                'vw' => 28
            ],
            'calendar-xmark' => [
                'p' => 'M160 0c17.7 0 32 14.3 32 32V64H320V32c0-17.7 14.3-32 32-32s32 14.3 32 32V64h48c26.5 0 48 21.5 48 48v48H32V112c0-26.5 21.5-48 48-48h48V32c0-17.7 14.3-32 32-32zM32 192H480V464c0 26.5-21.5 48-48 48H80c-26.5 0-48-21.5-48-48V192zM337 305c9.4-9.4 9.4-24.6 0-33.9s-24.6-9.4-33.9 0l-47 47-47-47c-9.4-9.4-24.6-9.4-33.9 0s-9.4 24.6 0 33.9l47 47-47 47c-9.4 9.4-9.4 24.6 0 33.9s24.6 9.4 33.9 0l47-47 47 47c9.4 9.4 24.6 9.4 33.9 0s9.4-24.6 0-33.9l-47-47 47-47z',
                'vw' => 512,
                'vh' => 512
            ],
            'camera' => 'M32 9v18q0 1.25-.88 2.13T29 30H3q-1.25 0-2.13-.88T0 27V9q0-1.25.88-2.13T3 6h5.5l.75-2.06q.25-.57.66-1t.97-.7T12.05 2h7.88q.94 0 1.69.53t1.12 1.4L23.5 6H29q1.25 0 2.13.87T32 9zm-8.5 9q0-3.13-2.19-5.31T16 10.49t-5.31 2.2T8.49 18t2.2 5.31 5.3 2.2 5.32-2.2T23.5 18zm-2 0q0 2.25-1.63 3.88T16 23.5t-3.88-1.63T10.5 18t1.63-3.88T16 12.5t3.88 1.63T21.5 18z',
            'camera-retro' => 'M3 2h26q1.25 0 2.13.88T32 5v22q0 1.25-.88 2.13T29 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2zm0 2q-.44 0-.72.28T2 5v.63Q2 6 2.38 6h7.25Q10 6 10 5.62V4.38Q10 4 9.62 4H3zm26.63 6q.37 0 .37-.38V5q0-.44-.28-.72T29 4H13.19q-.2 0-.32.19L11 7H2.37Q2 7 2 7.38v2.25q0 .37.37.37h27.25zM16 26.5q3.13 0 5.31-2.19t2.2-5.31-2.2-5.31-5.31-2.2-5.31 2.2-2.2 5.3 2.2 5.32T16 26.5zm0-13q2.25 0 3.88 1.63T21.5 19t-1.63 3.88T16 24.5t-3.88-1.63T10.5 19t1.63-3.88T16 13.5zM13 20q.44 0 .72-.28T14 19q0-.81.6-1.4T16 17q.44 0 .72-.28T17 16t-.28-.72T16 15q-1.63 0-2.81 1.19T11.99 19q0 .44.29.72T13 20z',
            'camera-rotate' => [
                'p' => 'M213.1 64.8L202.7 96H128c-35.3 0-64 28.7-64 64V416c0 35.3 28.7 64 64 64H512c35.3 0 64-28.7 64-64V160c0-35.3-28.7-64-64-64H437.3L426.9 64.8C420.4 45.2 402.1 32 381.4 32H258.6c-20.7 0-39 13.2-45.5 32.8zM448 256c0 8.8-7.2 16-16 16H355.3c-6.2 0-11.3-5.1-11.3-11.3c0-3 1.2-5.9 3.3-8L371 229c-13.6-13.4-31.9-21-51-21c-19.2 0-37.7 7.6-51.3 21.3L249 249c-9.4 9.4-24.6 9.4-33.9 0s-9.4-24.6 0-33.9l19.7-19.7C257.4 172.7 288 160 320 160c31.8 0 62.4 12.6 85 35l23.7-23.7c2.1-2.1 5-3.3 8-3.3c6.2 0 11.3 5.1 11.3 11.3V256zM192 320c0-8.8 7.2-16 16-16h76.7c6.2 0 11.3 5.1 11.3 11.3c0 3-1.2 5.9-3.3 8L269 347c13.6 13.4 31.9 21 51 21c19.2 0 37.7-7.6 51.3-21.3L391 327c9.4-9.4 24.6-9.4 33.9 0s9.4 24.6 0 33.9l-19.7 19.7C382.6 403.3 352 416 320 416c-31.8 0-62.4-12.6-85-35l-23.7 23.7c-2.1 2.1-5 3.3-8 3.3c-6.2 0-11.3-5.1-11.3-11.3V320z',
                'vw' => 640,
                'vh' => 512
            ],
            'campground' => [
                'p' => 'M39 28q.44 0 .72.28T40 29v2q0 .44-.28.72T39 32H1q-.44 0-.72-.28T0 31v-2q0-.44.28-.72T1 28h1.56L17.5 7.37l-3.3-4.62q-.56-.81.25-1.38L16 .2q.81-.57 1.44.25L20 3.94l2.56-3.5Q23.2-.38 24 .19l1.57 1.19q.8.56.25 1.37L22.5 7.38 37.44 28H39zM20 18l-7.25 10h14.5z',
                'w' => 1.25,
                'vw' => 40
            ],
            'candy-cane' => 'M31.13 5.75q1.62 3.56.34 7.28t-4.66 5.78L5.06 31.75q-.5.25-1 .25-1.12 0-1.75-1l-2-3.38q-.44-.75-.25-1.56T1 24.87l22.06-13.12q.75-.44.94-1.25T23.75 9q-.56-1-1.69-1-.56 0-1.06.25L19.31 9.3q-.5.25-1.06.25-1.13 0-1.69-.94L14.5 5.2q-.44-.7-.22-1.5t.9-1.25l1.76-1Q19.24 0 22 0q2.87 0 5.34 1.53t3.78 4.22zM20 6.56q.75-.43 1.75-.5L20.44 2.2q-1 .19-1.88.62l1.25 3.82zM9.12 27l1.94-1.13-3.81-2.43-1.94 1.12zm6.7-4 1.93-1.13L14 19.45l-1.94 1.12zm6.93-4.13 1.94-1.12-3.75-2.44L19 16.5zm1.31-12.3q.88.5 1.44 1.37l2.81-2.88q-.62-.81-1.37-1.37zm5.13 6.68q.44-.88.62-1.88L26 10.13q0 .43-.06.87-.13.5-.5 1.06z',
            'cannabis' => 'M31.44 22.5q.56.31.56.9t-.56.91q-3.75 1.94-8.32 1.94H22q.75 1.37 1.06 2.44.2.56-.25 1-.25.31-.68.31-.13 0-.32-.06-2.31-.75-4.81-2.44v4q0 .5-.5.5h-1q-.5 0-.5-.5v-4q-2.5 1.75-4.81 2.44-.2.06-.32.06-.43 0-.68-.31-.44-.44-.25-1 .3-1.07 1.06-2.44H8.87q-4.56 0-8.31-1.94Q0 24 0 23.41t.56-.91q2-1.06 4.75-1.63-1.75-2.06-3.1-4.75T.54 12.35 0 10.56v-.25q0-.37.28-.69t.72-.3q.12 0 .18.05.7.13 1.76.44t3.65 1.6 4.66 2.9v-.25q0-3.31 1.03-6.81T14 2.5 15.19.44q.25-.44.8-.44t.82.44q.25.31.53.81t.94 1.94 1.16 2.94.9 3.68.4 4.25v.25q2.07-1.62 4.66-2.9t3.66-1.6 1.75-.43q.06-.07.19-.07.44 0 .69.32.43.37.3.93-.18.7-.52 1.78t-1.69 3.79-3.1 4.75q2.82.56 4.76 1.62z',
            'capsules' => [
                'p' => 'M34.69 18.75q1.69 2.44 1.19 5.34t-2.94 4.6Q31.13 30 28.88 30q-3.75 0-5.88-3.06l-8.19-11.69q-.44-.63-.81-1.56V23q0 2.88-2.06 4.94T7 30t-4.94-2.06T0 23V9q0-2.88 2.06-4.94T7 2q2.8 0 4.8 1.94t2.13 4.69q.8-2.07 2.62-3.32Q18.37 4 20.62 4q3.75 0 5.88 3.06zM10 16V9q0-1.25-.88-2.13T7 6t-2.13.88T4 9v7h6zm12.19 2.81 5.12-3.62-4.06-5.82Q22.31 8 20.62 8q-1 0-1.75.56-1.12.75-1.34 2.03t.53 2.35z',
                'w' => 1.123,
                'vw' => 36
            ],
            'car' => 'M31.25 11q.38 0 .6.28t.15.66l-.38 1.5q-.18.56-.75.56h-1.25Q31 15.19 31 17v3q0 1.5-1 2.63V26q0 .81-.6 1.4T28 28h-2q-.81 0-1.4-.6T24 26v-2H8v2q0 .81-.6 1.4T6 28H4q-.81 0-1.4-.6T2 26v-3.38Q1 21.5 1 20v-3q0-1.81 1.38-3H1.13q-.57 0-.75-.56L0 11.94q-.06-.38.16-.66T.75 11H4.5l1.06-2.63q.38-.93 1.03-1.78t1.5-1.4 1.85-.88T12 4h8q2.13 0 3.9 1.19t2.54 3.19L27.5 11h3.75zm-22-1.13L8 13h16l-1.25-3.13q-.31-.8-1.1-1.34T20 8h-8q-.88 0-1.66.53T9.25 9.88zM6 20h1.13l.53-.03.53-.1.4-.15.32-.28L9 19q0-.94-1.06-1.97T6 16t-1.44.56T4 18t.56 1.44T6 20zm20 0q.88 0 1.44-.56T28 18t-.56-1.44T26 16q-.56 0-1.28.5t-1.22 1.22T23 19q0 .25.1.44t.3.28.41.15.53.1.53.03H26z',
            'car-alt' => [
                'p' => 'M27.44 13.25q1.12.44 1.84 1.47T30 17v3q0 1.5-1 2.63V26q0 .81-.6 1.4T27 28h-2q-.81 0-1.4-.6T23 26v-2H7v2q0 .81-.6 1.4T5 28H3q-.81 0-1.4-.6T1 26v-3.38Q0 21.5 0 20v-3q0-1.25.72-2.28t1.84-1.47l.75-1.75 1.25-3.13q.38-.93 1.04-1.78t1.5-1.4 1.84-.88T11 4h8q2.13 0 3.9 1.19t2.54 3.19l1.25 3.12zM8.25 9.87 7 13h16l-1.25-3.13q-.31-.8-1.1-1.34T19 8h-8q-.88 0-1.66.53T8.25 9.88zM5 20h1.13l.53-.03.53-.1.4-.15.32-.28L8 19q0-.94-1.06-1.97T5 16q-.44 0-.81.16t-.63.4-.4.63T3 18q0 .87.56 1.44T5 20zm20 0q.88 0 1.44-.56T27 18t-.56-1.44T25 16q-.56 0-1.28.5t-1.22 1.22T22 19q0 .25.1.44t.3.28.41.15.53.1.53.03H25z',
                'vw' => 30,
            ],
            'car-battery' => 'M30 8q.81 0 1.4.6T32 10v16q0 .81-.6 1.4T30 28H2q-.81 0-1.4-.6T0 26V10q0-.81.6-1.4T2 8h2V5q0-.44.28-.72T5 4h6q.44 0 .72.28T12 5v3h8V5q0-.44.28-.72T21 4h6q.44 0 .72.28T28 5v3h2zm-18 8.5v-1q0-.5-.5-.5h-7q-.5 0-.5.5v1q0 .5.5.5h7q.5 0 .5-.5zm16 0v-1q0-.5-.5-.5H25v-2.5q0-.5-.5-.5h-1q-.5 0-.5.5V15h-2.5q-.5 0-.5.5v1q0 .5.5.5H23v2.5q0 .5.5.5h1q.5 0 .5-.5V17h2.5q.5 0 .5-.5z',
            'car-burst' => [
                'p' => 'M176 8c-6.6 0-12.4 4-14.9 10.1l-29.4 74L55.6 68.9c-6.3-1.9-13.1 .2-17.2 5.3s-4.6 12.2-1.4 17.9l39.5 69.1L10.9 206.4c-5.4 3.7-8 10.3-6.5 16.7s6.7 11.2 13.1 12.2l78.7 12.2L90.6 327c-.5 6.5 3.1 12.7 9 15.5s12.9 1.8 17.8-2.6l35.3-32.5 9.5-35.4 10.4-38.6c8-29.9 30.5-52.1 57.9-60.9l41-59.2c11.3-16.3 26.4-28.9 43.5-37.2c-.4-.6-.8-1.2-1.3-1.8c-4.1-5.1-10.9-7.2-17.2-5.3L220.3 92.1l-29.4-74C188.4 12 182.6 8 176 8zM367.7 161.5l135.6 36.3c6.5 1.8 11.3 7.4 11.8 14.2l4.6 56.5-201.5-54 32.2-46.6c3.8-5.6 10.8-8.1 17.3-6.4zm-69.9-30l-47.9 69.3c-21.6 3-40.3 18.6-46.3 41l-10.4 38.6-16.6 61.8-8.3 30.9c-4.6 17.1 5.6 34.6 22.6 39.2l15.5 4.1c17.1 4.6 34.6-5.6 39.2-22.6l8.3-30.9 247.3 66.3-8.3 30.9c-4.6 17.1 5.6 34.6 22.6 39.2l15.5 4.1c17.1 4.6 34.6-5.6 39.2-22.6l8.3-30.9L595 388l10.4-38.6c6-22.4-2.5-45.2-19.6-58.7l-6.8-84c-2.7-33.7-26.4-62-59-70.8L384.2 99.7c-32.7-8.8-67.3 4-86.5 31.8zm-17 131a24 24 0 1 1 -12.4 46.4 24 24 0 1 1 12.4-46.4zm217.9 83.2A24 24 0 1 1 545 358.1a24 24 0 1 1 -46.4-12.4z',
                'vw' => 640,
                'vh' => 512
            ],
            'car-crash' => [
                'p' => 'm8.94 13.81-.75 2.88q-.31 1-.19 2.19l-2.19 1.8q-.31.26-.65.04t-.22-.6l.93-4.8q.13-.63-.5-.7L.5 14.2q-.38-.07-.47-.44t.22-.63l4.06-2.68q.57-.38.13-.82L1.3 5.81q-.25-.31-.03-.66t.6-.22l4.8.94q.63.13.7-.5L7.82.5q.06-.38.43-.47t.63.22l2.69 4.06q.37.56.8.13L16.2 1.3q.31-.25.66-.03t.22.6l-.7 3.5q-.3.43-.5.62l-2 2.69-.68.87q-1.5.44-2.65 1.57t-1.6 2.68zm30.94 8.25L39.05 25q-.25 1.06-1.06 1.81-.25.25-.56.44l-.88 3.25q-.12.44-.37.75-.82 1-2.07.69l-1.94-.5q-.8-.25-1.21-.97t-.22-1.53l.56-1.88-15.5-4.18-.5 1.93q-.19.82-.9 1.22t-1.54.22l-1.94-.5q-1.3-.37-1.43-1.68-.07-.38 0-.82l.87-3.25-.25-.62q-.31-1.13 0-2.2l.75-2.87q.5-1.87 2.31-2.62.57-.25 1.2-.31l1.12-1.5 2-2.7q.31-.43.69-.8.75-.75 1.68-1.2.44-.24.94-.43.94-.31 2-.38t2.06.25l7.7 2.07q1.06.25 1.93.81 1.31.88 2.19 2.19.81 1.31 1 2.94l.44 3.3.18 1.88q1 .75 1.44 1.88.44 1.19.13 2.37zM16 14.63q-.44-.13-.84-.07t-.72.25-.57.5-.37.75q-.13.56 0 1.06t.5.85.94.53q.12.06.5.16t.6.15.52.13.53.03l.44-.07.38-.18.21-.4q.2-.88-.56-2.17T16 14.62zm18.13 1.8-.38-3.37q-.06-.44-.25-.87-.56-1.32-1.94-1.63L23.87 8.5q-.37-.13-.8-.13-.57 0-1.29.35t-1.1.84l-2 2.7 10.76 2.87zm1.18 3.38q-.87-.25-2.15.5t-1.53 1.63q.06.25 0 .44t.18.37.35.28.47.22.53.16.56.12.53.16q.44.12.85.06t.72-.25.56-.5.31-.75q.25-.87-.12-1.53t-1.25-.9z',
                'w' => 1.25,
                'vw' => 40
            ],
            'car-on' => [
                'p' => 'M280 24c0-13.3-10.7-24-24-24s-24 10.7-24 24v80c0 13.3 10.7 24 24 24s24-10.7 24-24V24zM185.8 224H326.2c6.8 0 12.8 4.3 15.1 10.6L360.3 288H151.7l19.1-53.4c2.3-6.4 8.3-10.6 15.1-10.6zm-75.3-10.9L82.2 292.4C62.1 300.9 48 320.8 48 344v40 64 32c0 17.7 14.3 32 32 32H96c17.7 0 32-14.3 32-32V448H384v32c0 17.7 14.3 32 32 32h16c17.7 0 32-14.3 32-32V448 384 344c0-23.2-14.1-43.1-34.2-51.6l-28.3-79.3C390.1 181.3 360 160 326.2 160H185.8c-33.8 0-64 21.3-75.3 53.1zM128 344a24 24 0 1 1 0 48 24 24 0 1 1 0-48zm232 24a24 24 0 1 1 48 0 24 24 0 1 1 -48 0zM39 39c-9.4 9.4-9.4 24.6 0 33.9l48 48c9.4 9.4 24.6 9.4 33.9 0s9.4-24.6 0-33.9L73 39c-9.4-9.4-24.6-9.4-33.9 0zm400 0L391 87c-9.4 9.4-9.4 24.6 0 33.9s24.6 9.4 33.9 0l48-48c9.4-9.4 9.4-24.6 0-33.9s-24.6-9.4-33.9 0z',
                'vw' => 512,
                'vh' => 512
            ],
            'car-rear' => [
                'p' => 'M165.4 96H346.6c13.6 0 25.7 8.6 30.2 21.4L402.9 192H109.1l26.1-74.6c4.5-12.8 16.6-21.4 30.2-21.4zm-90.6 .3L39.6 196.8C16.4 206.4 0 229.3 0 256v80c0 23.7 12.9 44.4 32 55.4V448c0 17.7 14.3 32 32 32H96c17.7 0 32-14.3 32-32V400H384v48c0 17.7 14.3 32 32 32h32c17.7 0 32-14.3 32-32V391.4c19.1-11.1 32-31.7 32-55.4V256c0-26.7-16.4-49.6-39.6-59.2L437.2 96.3C423.7 57.8 387.4 32 346.6 32H165.4c-40.8 0-77.1 25.8-90.6 64.3zM208 272h96c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H208c-8.8 0-16-7.2-16-16V288c0-8.8 7.2-16 16-16zM48 280c0-13.3 10.7-24 24-24h32c13.3 0 24 10.7 24 24s-10.7 24-24 24H72c-13.3 0-24-10.7-24-24zm360-24h32c13.3 0 24 10.7 24 24s-10.7 24-24 24H408c-13.3 0-24-10.7-24-24s10.7-24 24-24z',
                'vw' => 512,
                'vh' => 512
            ],
            'car-side' => [
                'p' => 'M34 12q2.5 0 4.25 1.75T40 18v5q0 .44-.28.72T39 24h-3q0 2.5-1.75 4.25T30 30t-4.25-1.75T24 24h-8q0 2.5-1.75 4.25T10 30t-4.25-1.75T4 24H1q-.44 0-.72-.28T0 23v-7q0-1.38.84-2.44T3 12.12L6 4.5q.44-1.13 1.47-1.81t2.22-.7h13.37q.82 0 1.72.45t1.4 1.06L33 12h1zM10 27q1.25 0 2.13-.88T13 24t-.88-2.13T10 21t-2.13.88T7 24t.88 2.13T10 27zm4.5-15V6H9.69L7.3 12h7.19zm3 0h10.38l-4.82-6H17.5v6zM30 27q1.25 0 2.13-.88T33 24t-.88-2.13T30 21t-2.13.88T27 24t.88 2.13T30 27z',
                'w' => 1.25,
                'vw' => 40
            ],
            'car-tunnel' => [
                'p' => 'M256 0C114.6 0 0 114.6 0 256V448c0 35.3 28.7 64 64 64h42.8c-6.6-5.9-10.8-14.4-10.8-24V376c0-20.8 11.3-38.9 28.1-48.6l21-64.7c7.5-23.1 29-38.7 53.3-38.7H313.6c24.3 0 45.8 15.6 53.3 38.7l21 64.7c16.8 9.7 28.2 27.8 28.2 48.6V488c0 9.6-4.2 18.1-10.8 24H448c35.3 0 64-28.7 64-64V256C512 114.6 397.4 0 256 0zM362.8 512c-6.6-5.9-10.8-14.4-10.8-24V448H160v40c0 9.6-4.2 18.1-10.8 24H362.8zM190.8 277.5L177 320H335l-13.8-42.5c-1.1-3.3-4.1-5.5-7.6-5.5H198.4c-3.5 0-6.5 2.2-7.6 5.5zM168 408a24 24 0 1 0 0-48 24 24 0 1 0 0 48zm200-24a24 24 0 1 0 -48 0 24 24 0 1 0 48 0z',
                'vw' => 512,
                'vh' => 512
            ],
            'caravan' => [
                'p' => 'M26 13a1 1 0 0 0 0 2 1 1 0 0 0 0-2Zm13 7h-3V10A10 10 0 0 0 26 0H4a4 4 0 0 0-4 4v16a4 4 0 0 0 4 4h2a6 6 0 0 0 12 0h21a1 1 0 0 0 1-1v-2a1 1 0 0 0-1-1Zm-27 7a3 3 0 1 1 0-6 3 3 0 0 1 0 6Zm4-15a2 2 0 0 1-2 2H6a2 2 0 0 1-2-2V8c0-1.1.9-2 2-2h8a2 2 0 0 1 2 2Zm12 8h-8V8c0-1.1.9-2 2-2h4a2 2 0 0 1 2 2Zm0 0',
                'vw' => 40
            ],
            'caret-down' => [
                'p' => 'M1.94 12h16.12q.82 0 1.13.78t-.25 1.35l-8.07 8.06q-.37.37-.87.37t-.88-.37l-8.06-8.06q-.56-.57-.25-1.35T1.94 12z',
                'w' => .623,
                'vw' => 20
            ],
            'caret-left' => [
                'p' => 'M12 7.94v16.12q0 .82-.78 1.13t-1.35-.25l-8.06-8.07q-.37-.37-.37-.87t.37-.88l8.07-8.06q.56-.56 1.34-.25T12 7.94z',
                'w' => .42,
                'vw' => 13
            ],
            'caret-right' => [
                'p' => 'M0 24.06V7.94q0-.82.78-1.13t1.35.25l8.06 8.07q.37.37.37.87t-.37.88l-8.06 8.06q-.57.56-1.35.25T0 24.06z',
                'w' => .328,
                'vw' => 11,
            ],
            'caret-square-down' => [
                'p' => 'M28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22q1.25 0 2.13.88T28 5zM5.75 13.75l7.75 7.75q.5.5 1 0l7.75-7.75q.31-.31.13-.78t-.7-.47H6.32q-.5 0-.69.47t.13.78z',
                'w' => .875,
                'vw' => 28
            ],
            'caret-square-left' => [
                'p' => 'M25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22q1.25 0 2.13.88T28 5v22q0 1.25-.88 2.13T25 30zM16.25 7.75 8.5 15.5q-.5.5 0 1l7.75 7.75q.31.31.78.13t.47-.7V8.32q0-.5-.47-.69t-.78.13z',
                'w' => .875,
                'vw' => 28
            ],
            'caret-square-right' => [
                'p' => 'M3 2h22q1.25 0 2.13.88T28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2zm8.75 22.25 7.75-7.75q.5-.5 0-1l-7.75-7.75q-.31-.31-.78-.13t-.47.7v15.37q0 .5.47.69t.78-.13z',
                'w' => .875,
                'vw' => 28
            ],
            'caret-square-up' => [
                'p' => 'M0 27V5q0-1.25.88-2.13T3 2h22q1.25 0 2.13.88T28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27zm22.25-8.75L14.5 10.5q-.5-.5-1 0l-7.75 7.75q-.31.31-.13.78t.7.47h15.37q.5 0 .69-.47t-.13-.78z',
                'w' => .875,
                'vw' => 28
            ],
            'caret-up' => [
                'p' => 'M18.06 22H1.94q-.82 0-1.13-.78t.25-1.34l8.07-8.07q.37-.37.87-.37t.88.37l8.06 8.06q.56.57.25 1.35t-1.13.78z',
                'w' => .625,
                'vw' => 20
            ],
            'carrot' => 'M18.63 9.81q2.87 1.38 3.93 4.44t-.37 5.94q-1.13 2.37-3.57 3.56l-6.37 3.13-3.56-3.57q-.7-.75-1.38 0-.31.25-.31.7t.31.68l3.06 3.06L2 31.88q-.5.25-1.06.06t-.82-.75 0-1.19l8-16.43 3.2 3.12q.24.31.65.31t.72-.31q.75-.69 0-1.38l-3.44-3.5q1.69-2 4.28-2.59t5.1.6zm5.75-2.19q1.87-.93 4-.43T32 9.5q-1.94 2.38-4.69 2.5t-4.81-2l-.5-.5q-2.13-2.06-2-4.81t2.5-4.7q1.81 1.5 2.31 3.63t-.44 4z',
            'cart-arrow-down' => [
                'p' => 'M31.56 20H13.25l.38 2h16.8q.7 0 1.13.56t.32 1.25l-.38 1.57q2 .93 2 3.12 0 1.44-1.03 2.47T29.94 32q-1.38 0-2.38-1t-1.06-2.37q-.06-1.57 1.06-2.63H14.44q1.06 1 1.06 2.5t-1.1 2.53-2.59.97q-1.3-.06-2.28-1.03T8.5 28.69q0-.69.19-1.31t.6-1.1.96-.84L5.88 4H1.5Q.88 4 .44 3.56T0 2.5v-1Q0 .87.44.44T1.5 0h6.44q.31 0 .63.15T9.1.6t.28.6L9.94 4H34.5q.75 0 1.2.57t.24 1.25l-2.94 13Q32.75 20 31.57 20zm-6.37-8H22.5V8.25q0-.75-.75-.75h-1.5q-.75 0-.75.75V12h-2.69q-.5 0-.69.47t.13.78l4.25 4.25q.5.5 1 0l4.25-4.25q.31-.31.12-.78T25.2 12z',
                'w' => 1.123,
                'vw' => 36
            ],
            'cart-flatbed' => [
                'p' => 'M32 0C14.3 0 0 14.3 0 32S14.3 64 32 64H48c8.8 0 16 7.2 16 16V368c0 44.2 35.8 80 80 80h18.7c-1.8 5-2.7 10.4-2.7 16c0 26.5 21.5 48 48 48s48-21.5 48-48c0-5.6-1-11-2.7-16H450.7c-1.8 5-2.7 10.4-2.7 16c0 26.5 21.5 48 48 48s48-21.5 48-48c0-5.6-1-11-2.7-16H608c17.7 0 32-14.3 32-32s-14.3-32-32-32H144c-8.8 0-16-7.2-16-16V80C128 35.8 92.2 0 48 0H32zM192 80V272c0 26.5 21.5 48 48 48H560c26.5 0 48-21.5 48-48V80c0-26.5-21.5-48-48-48H464V176c0 5.9-3.2 11.3-8.5 14.1s-11.5 2.5-16.4-.8L400 163.2l-39.1 26.1c-4.9 3.3-11.2 3.6-16.4 .8s-8.5-8.2-8.5-14.1V32H240c-26.5 0-48 21.5-48 48z',
                'vw' => 640,
                'vh' => 512
            ],
            'cart-flatbed-suitcase' => [
                'p' => 'M0 32C0 14.3 14.3 0 32 0H48c44.2 0 80 35.8 80 80V368c0 8.8 7.2 16 16 16H608c17.7 0 32 14.3 32 32s-14.3 32-32 32H541.3c1.8 5 2.7 10.4 2.7 16c0 26.5-21.5 48-48 48s-48-21.5-48-48c0-5.6 1-11 2.7-16H253.3c1.8 5 2.7 10.4 2.7 16c0 26.5-21.5 48-48 48s-48-21.5-48-48c0-5.6 1-11 2.7-16H144c-44.2 0-80-35.8-80-80V80c0-8.8-7.2-16-16-16H32C14.3 64 0 49.7 0 32zM432 96V56c0-4.4-3.6-8-8-8H344c-4.4 0-8 3.6-8 8V96h96zM288 96V56c0-30.9 25.1-56 56-56h80c30.9 0 56 25.1 56 56V96 320H288V96zM512 320V96h16c26.5 0 48 21.5 48 48V272c0 26.5-21.5 48-48 48H512zM240 96h16V320H240c-26.5 0-48-21.5-48-48V144c0-26.5 21.5-48 48-48z',
                'vw' => 640,
                'vh' => 512
            ],
            'cart-plus' => [
                'p' => 'M31.56 20H13.25l.38 2h16.8q.7 0 1.13.56t.32 1.25l-.38 1.57q2 .93 2 3.12 0 1.44-1.03 2.47T29.94 32q-1.38 0-2.38-1t-1.06-2.37q-.06-1.57 1.06-2.63H14.44q1.06 1 1.06 2.5t-1.1 2.53-2.59.97q-1.3-.06-2.28-1.03T8.5 28.69q0-.69.19-1.31t.6-1.1.96-.84L5.88 4H1.5Q.88 4 .44 3.56T0 2.5v-1Q0 .87.44.44T1.5 0h6.44q.31 0 .63.15T9.1.6t.28.6L9.94 4H34.5q.75 0 1.2.57t.24 1.25l-2.94 13Q32.75 20 31.57 20zm-6.06-9.5h-3V8q0-.44-.28-.72T21.5 7h-1q-.44 0-.72.28T19.5 8v2.5h-3q-.44 0-.72.28t-.28.72v1q0 .44.28.72t.72.28h3V16q0 .44.28.72t.72.28h1q.44 0 .72-.28t.28-.72v-2.5h3q.44 0 .72-.28t.28-.72v-1q0-.44-.28-.72t-.72-.28z',
                'w' => 1.123,
                'vw' => 36
            ],
            'cart-shopping' => [
                'p' => 'M0 24C0 10.7 10.7 0 24 0H69.5c22 0 41.5 12.8 50.6 32h411c26.3 0 45.5 25 38.6 50.4l-41 152.3c-8.5 31.4-37 53.3-69.5 53.3H170.7l5.4 28.5c2.2 11.3 12.1 19.5 23.6 19.5H488c13.3 0 24 10.7 24 24s-10.7 24-24 24H199.7c-34.6 0-64.3-24.6-70.7-58.5L77.4 54.5c-.7-3.8-4-6.5-7.9-6.5H24C10.7 48 0 37.3 0 24zM128 464a48 48 0 1 1 96 0 48 48 0 1 1 -96 0zm336-48a48 48 0 1 1 0 96 48 48 0 1 1 0-96z',
                'vw' => 576,
                'vh' => 512
            ],
            'cash-register' => 'M31.94 23.69q.06.31.06.62v5.7q0 .43-.16.77t-.43.63-.66.43T30 32H2q-.81 0-1.4-.6T0 30v-5.69q0-.31.06-.62l1.7-10q.06-.75.62-1.22T3.69 12H9V8H3q-.43 0-.72-.28T2 7V1q0-.44.28-.72T3 0h16q.44 0 .72.28T20 1v6q0 .44-.28.72T19 8h-6v4h15.32q.75 0 1.28.47t.65 1.22zM17.5 15.5v1q0 .44.28.72t.72.28h1q.44 0 .72-.28t.28-.72v-1q0-.44-.28-.72t-.72-.28h-1q-.44 0-.72.28t-.28.72zm-2 4q-.44 0-.72.28t-.28.72v1q0 .44.28.72t.72.28h1q.44 0 .72-.28t.28-.72v-1q0-.44-.28-.72t-.72-.28h-1zm-2-5h-1q-.44 0-.72.28t-.28.72v1q0 .44.28.72t.72.28h1q.44 0 .72-.28t.28-.72v-1q0-.44-.28-.72t-.72-.28zM5 5h12V3H5v2zm2.5 12.5q.44 0 .72-.28t.28-.72v-1q0-.44-.28-.72t-.72-.28h-1q-.44 0-.72.28t-.28.72v1q0 .44.28.72t.72.28h1zm1 4q0 .44.28.72t.72.28h1q.44 0 .72-.28t.28-.72v-1q0-.44-.28-.72t-.72-.28h-1q-.44 0-.72.28t-.28.72v1zm13.5 7v-1q0-.5-.5-.5h-11q-.5 0-.5.5v1q0 .5.5.5h11q.5 0 .5-.5zm1.5-7v-1q0-.44-.28-.72t-.72-.28h-1q-.44 0-.72.28t-.28.72v1q0 .44.28.72t.72.28h1q.44 0 .72-.28t.28-.72zm3-5v-1q0-.44-.28-.72t-.72-.28h-1q-.44 0-.72.28t-.28.72v1q0 .44.28.72t.72.28h1q.44 0 .72-.28t.28-.72z',
            'cat' => 'M18.19 12q.25 1.38.97 2.56t1.75 2.03 2.34 1.32 2.75.47q.94 0 2-.25V31q0 .44-.28.72T27 32h-2q-.44 0-.72-.28T24 31v-9l-8 6h2q.81 0 1.4.6T20 30v1q0 .44-.28.72T19 32H8q-1.62 0-2.81-1.19T4 28V12q0-.81-.6-1.4T2 10 .6 9.4 0 8t.6-1.4T2 6q2.5 0 4.25 1.75T8 12v5.38q1.19-1.82 2.78-3.04t3.19-1.65 2.47-.57 1.75-.12zM28 6l4-4v8.38q0 2.5-1.75 4.24T26 16.39t-4.25-1.75T20 10.37V2l4 4h4zm-4.5 5q.44 0 .72-.28t.28-.72-.28-.72T23.5 9t-.72.28-.28.72.28.72.72.28zm5 0q.44 0 .72-.28t.28-.72-.28-.72T28.5 9t-.72.28-.28.72.28.72.72.28z',
            'cedi-sign' => [
                'p' => 'M256 32c0-17.7-14.3-32-32-32s-32 14.3-32 32V66.7C101.2 81.9 32 160.9 32 256s69.2 174.1 160 189.3V480c0 17.7 14.3 32 32 32s32-14.3 32-32V445.3c30.9-5.2 59.2-17.7 83.2-35.8c14.1-10.6 17-30.7 6.4-44.8s-30.7-17-44.8-6.4c-13.2 9.9-28.3 17.3-44.8 21.6V132c16.4 4.2 31.6 11.6 44.8 21.6c14.1 10.6 34.2 7.8 44.8-6.4s7.8-34.2-6.4-44.8c-24-18-52.4-30.6-83.2-35.8V32zM192 132V380c-55.2-14.2-96-64.3-96-124s40.8-109.8 96-124z',
                'vw' => 384,
                'vh' => 512
            ],
            'cent-sign' => [
                'p' => 'M224 0c17.7 0 32 14.3 32 32V66.7c30.9 5.2 59.2 17.7 83.2 35.8c14.1 10.6 17 30.7 6.4 44.8s-30.7 17-44.8 6.4C279.4 137.5 252.9 128 224 128c-70.7 0-128 57.3-128 128s57.3 128 128 128c28.9 0 55.4-9.5 76.8-25.6c14.1-10.6 34.2-7.8 44.8 6.4s7.8 34.2-6.4 44.8c-24 18-52.4 30.6-83.2 35.8V480c0 17.7-14.3 32-32 32s-32-14.3-32-32V445.3C101.2 430.1 32 351.1 32 256s69.2-174.1 160-189.3V32c0-17.7 14.3-32 32-32z',
                'vw' => 384,
                'vh' => 512
            ],
            'certificate' => 'm28.69 16 2.87 2.81q.63.63.38 1.47t-1.07 1.03l-3.93 1 1.12 3.88q.25.87-.37 1.5t-1.5.37l-3.88-1.12-1 3.94q-.19.87-1.06 1.09t-1.44-.4L16 28.68l-2.81 2.87q-.57.63-1.44.41t-1.07-1.1l-1-3.93-3.87 1.12q-.88.25-1.5-.37t-.38-1.5l1.13-3.88-3.94-1q-.81-.18-1.06-1.03t.37-1.47L3.31 16 .43 13.19q-.62-.63-.37-1.47t1.06-1.03l3.94-1L3.93 5.8q-.25-.87.38-1.5t1.5-.37l3.87 1.12 1-3.94q.2-.87 1.07-1.09t1.43.4L16 3.38 18.8.44q.63-.63 1.47-.38t1.03 1.06l1 3.94 3.88-1.12q.87-.25 1.5.37t.37 1.5L26.94 9.7l3.93 1q.82.19 1.07 1.03t-.38 1.47z',
            'chair' => [
                'p' => 'M7 8v8H4V8q0-3.31 2.34-5.66T12 0h4q3.31 0 5.66 2.34T24 8v8h-3V8q0-2.88-2.5-4.31V16h-3V3h-3v13h-3V3.69Q7 5.13 7 8zm20.88 13.38q.3.93-.29 1.78T26 24v7q0 .44-.28.72T25 32h-2q-.44 0-.72-.28T22 31v-7H6v7q0 .44-.28.72T5 32H3q-.44 0-.72-.28T2 31v-7q-1 0-1.63-.84t-.24-1.79l.62-2Q1.25 18 2.69 18H25.3q1.44 0 1.94 1.38z',
                'w' => .873,
                'vw' => 28
            ],
            'chalkboard' => [
                'p' => 'M6 4v22H2V2.5q0-1.06.72-1.78T4.5 0h31q1.06 0 1.78.72T38 2.5V26h-4V4H6zm33 24q.44 0 .72.28T40 29v2q0 .44-.28.72T39 32H1q-.44 0-.72-.28T0 31v-2q0-.44.28-.72T1 28h17v-4h12v4h9z',
                'w' => 1.25,
                'vw' => 40
            ],
            'chalkboard-teacher' => [
                'p' => 'M13 22q2.94 0 4.97 2.06t2.03 5q0 1.2-.88 2.07T17 32H3q-1.25 0-2.13-.87T0 29.06q0-2.94 2.03-5T7 22q.25 0 .44.06 1.31.44 2.56.44t2.56-.44q.2-.06.44-.06zm-3-2q-2.5 0-4.25-1.75T4 14t1.75-4.25T10 8t4.25 1.75T16 14t-1.75 4.25T10 20zM37 0q1.25 0 2.13.9T40 3.13v19.75q0 1.32-.88 2.22T37 26H21.69q-.57-2.38-2.44-4H24v-4h8v4h4V4H14v3.13Q12.06 6 10 6V3.12q0-1.3.87-2.21T13 0h24z',
                'w' => 1.25,
                'vw' => 40
            ],
            'chalkboard-user' => [
                'p' => 'M160 64c0-35.3 28.7-64 64-64H576c35.3 0 64 28.7 64 64V352c0 35.3-28.7 64-64 64H336.8c-11.8-25.5-29.9-47.5-52.4-64H384V320c0-17.7 14.3-32 32-32h64c17.7 0 32 14.3 32 32v32h64V64L224 64v49.1C205.2 102.2 183.3 96 160 96V64zm0 64a96 96 0 1 1 0 192 96 96 0 1 1 0-192zM133.3 352h53.3C260.3 352 320 411.7 320 485.3c0 14.7-11.9 26.7-26.7 26.7H26.7C11.9 512 0 500.1 0 485.3C0 411.7 59.7 352 133.3 352z',
                'vw' => 640,
                'vh' => 512
            ],
            'champagne-glasses' => [
                'p' => 'M155.6 17.3C163 3 179.9-3.6 195 1.9L320 47.5l125-45.6c15.1-5.5 32 1.1 39.4 15.4l78.8 152.9c28.8 55.8 10.3 122.3-38.5 156.6L556.1 413l41-15c16.6-6 35 2.5 41 19.1s-2.5 35-19.1 41l-71.1 25.9L476.8 510c-16.6 6.1-35-2.5-41-19.1s2.5-35 19.1-41l41-15-31.3-86.2c-59.4 5.2-116.2-34-130-95.2L320 188.8l-14.6 64.7c-13.8 61.3-70.6 100.4-130 95.2l-31.3 86.2 41 15c16.6 6 25.2 24.4 19.1 41s-24.4 25.2-41 19.1L92.2 484.1 21.1 458.2c-16.6-6.1-25.2-24.4-19.1-41s24.4-25.2 41-19.1l41 15 31.3-86.2C66.5 292.5 48.1 226 76.9 170.2L155.6 17.3zm44 54.4l-27.2 52.8L261.6 157l13.1-57.9L199.6 71.7zm240.9 0L365.4 99.1 378.5 157l89.2-32.5L440.5 71.7z',
                'vw' => 640,
                'vh' => 512
            ],
            'charging-station' => [
                'p' => 'M21 28q.44 0 .72.28T22 29v2q0 .44-.28.72T21 32H1q-.44 0-.72-.28T0 31v-2q0-.44.28-.72T1 28h20zM34 8h1q.44 0 .72.28T36 9v2q0 1.63-.97 2.94t-2.53 1.81v7.5q0 2.06-1.56 3.5T27.24 28q-1.8-.19-3.02-1.62T23 23.06V21.5q0-1.06-.72-1.78T20.5 19H20v7H2V4q0-1.69 1.15-2.84T6 0h10q1.69 0 2.84 1.16T20 4v12h.5q2.25 0 3.87 1.63T26 21.5v1.75q0 .81.62 1.34t1.44.41q.63-.13 1.03-.66t.4-1.21v-7.38q-1.55-.5-2.52-1.81T26 11V9q0-.44.28-.72T27 8h1V5q0-.44.28-.72T29 4t.72.28T30 5v3h2V5q0-.44.28-.72T33 4t.72.28T34 5v3zm-17.75 3q.25-.31.03-.66t-.66-.34H12l.75-3.19q.06-.31-.16-.56T12 6H7.75q-.69 0-.75.56l-1 6.69q-.06.31.19.53t.56.22h3.69L9 19.19q-.06.31.16.56t.6.25q.43 0 .62-.31z',
                'w' => 1.125,
                'vw' => 36
            ],
            'chart-area' => 'M31.25 24q.75 0 .75.75v2.5q0 .75-.75.75H.75Q0 28 0 27.25V4.75Q0 4 .75 4h2.5Q4 4 4 4.75V24h27.25zm-7.94-14q.25-.25.6-.16t.47.41L30 22H6v-6.5l5.44-9.06q.19-.32.6-.35t.65.28L18 13.5z',
            'chart-bar' => 'M20.81 20q-.31 0-.56-.25t-.25-.56V10.8q0-.31.25-.56t.56-.25h2.38q.31 0 .56.25t.25.56v8.38q0 .31-.25.56t-.56.25H20.8zm6 0q-.31 0-.56-.25t-.25-.56V4.8q0-.31.25-.56t.56-.25h2.38q.31 0 .56.25t.25.56v14.4q0 .31-.25.56t-.56.25H26.8zm-18 0q-.31 0-.56-.25T8 19.19V14.8q0-.31.25-.56t.56-.25h2.38q.31 0 .56.25t.25.56v4.38q0 .31-.25.56t-.56.25H8.8zm6 0q-.31 0-.56-.25t-.25-.56V6.8q0-.31.25-.56t.56-.25h2.38q.31 0 .56.25t.25.56v12.4q0 .31-.25.56t-.56.25H14.8zM31 24q.44 0 .72.28T32 25v2q0 .44-.28.72T31 28H2q-.81 0-1.4-.6T0 26V5q0-.44.28-.72T1 4h2q.44 0 .72.28T4 5v19h27z',
            'chart-column' => [
                'p' => 'M32 32c17.7 0 32 14.3 32 32V400c0 8.8 7.2 16 16 16H480c17.7 0 32 14.3 32 32s-14.3 32-32 32H80c-44.2 0-80-35.8-80-80V64C0 46.3 14.3 32 32 32zM160 224c17.7 0 32 14.3 32 32v64c0 17.7-14.3 32-32 32s-32-14.3-32-32V256c0-17.7 14.3-32 32-32zm128-64V320c0 17.7-14.3 32-32 32s-32-14.3-32-32V160c0-17.7 14.3-32 32-32s32 14.3 32 32zm64 32c17.7 0 32 14.3 32 32v96c0 17.7-14.3 32-32 32s-32-14.3-32-32V224c0-17.7 14.3-32 32-32zM480 96V320c0 17.7-14.3 32-32 32s-32-14.3-32-32V96c0-17.7 14.3-32 32-32s32 14.3 32 32z',
                'vw' => 512,
                'vh' => 512
            ],
            'chart-gantt' => [
                'p' => 'M32 32c17.7 0 32 14.3 32 32V400c0 8.8 7.2 16 16 16H480c17.7 0 32 14.3 32 32s-14.3 32-32 32H80c-44.2 0-80-35.8-80-80V64C0 46.3 14.3 32 32 32zm96 96c0-17.7 14.3-32 32-32l96 0c17.7 0 32 14.3 32 32s-14.3 32-32 32H160c-17.7 0-32-14.3-32-32zm96 64H352c17.7 0 32 14.3 32 32s-14.3 32-32 32H224c-17.7 0-32-14.3-32-32s14.3-32 32-32zm160 96h64c17.7 0 32 14.3 32 32s-14.3 32-32 32H384c-17.7 0-32-14.3-32-32s14.3-32 32-32z',
                'vw' => 512,
                'vh' => 512
            ],
            'chart-line' => 'M31 24q.44 0 .72.28T32 25v2q0 .44-.28.72T31 28H2q-.81 0-1.4-.6T0 26V5q0-.44.28-.72T1 4h2q.44 0 .72.28T4 5v19h27zM29 6q.44 0 .72.28T30 7v7.38q0 1-.94 1.37t-1.62-.31l-2-2-6 6Q18.8 20 18 20t-1.44-.56L12 14.8l-2.88 2.9q-.68.75-1.43 0L6.3 16.3q-.75-.75 0-1.44l4.25-4.3Q11.18 10 12 10t1.44.56L18 15.2l4.56-4.63-2-2q-.69-.69-.31-1.62T21.62 6H29z',
            'chart-pie' => [
                'p' => 'M33 18q.44 0 .75.34t.25.79q-.75 5.18-4.56 8.8-.7.63-1.38-.05L18.12 18H33zm-1-4.06q0 .44-.28.75T31 15H17V1q0-.44.31-.72t.75-.28q5.57.38 9.57 4.38T32 13.94zM14 18l9.75 9.75q.31.31.28.78t-.4.72q-3.75 2.69-8.44 2.75-6 .06-10.5-4.22T-.01 17.5q-.18-5.75 3.54-10.16t9.34-5.15q.44-.06.78.22t.35.78V18z',
                'vw' => 34,
            ],
            'chart-simple' => [
                'p' => 'M160 80c0-26.5 21.5-48 48-48h32c26.5 0 48 21.5 48 48V432c0 26.5-21.5 48-48 48H208c-26.5 0-48-21.5-48-48V80zM0 272c0-26.5 21.5-48 48-48H80c26.5 0 48 21.5 48 48V432c0 26.5-21.5 48-48 48H48c-26.5 0-48-21.5-48-48V272zM368 96h32c26.5 0 48 21.5 48 48V432c0 26.5-21.5 48-48 48H368c-26.5 0-48-21.5-48-48V144c0-26.5 21.5-48 48-48z',
                'vw' => 448,
                'vh' => 512
            ],
            'check' => 'M10.88 27.44.44 17.06Q0 16.63 0 15.94t.44-1.13l2.31-2.25q.44-.5 1.1-.5t1.15.5l7 7 15-15q.5-.5 1.16-.5t1.09.5l2.31 2.25q.44.44.44 1.13t-.44 1.12L13.13 27.44q-.44.5-1.13.5t-1.13-.5z',
            'check-circle' => 'M31.5 16q0 6.44-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5t10.97 4.53T31.5 16zm-17.31 8.19 11.5-11.5q.75-.69 0-1.38L24.3 9.88q-.75-.7-1.43 0l-9.38 9.37-4.38-4.38q-.68-.68-1.43 0L6.3 16.32q-.75.7 0 1.38l6.5 6.5q.69.75 1.38 0z',
            'check-double' => 'M31.5 10.75q.5.44.5 1.13T31.5 13L13.12 31.5q-.43.5-1.12.5t-1.13-.5L.5 21.06q-.5-.44-.5-1.12t.5-1.13l2.25-2.31q.44-.44 1.1-.44T5 16.5l7 7.06L27 8.5q.5-.5 1.16-.5t1.09.5zm-21.06 6.94L2.75 9.94q-.31-.31-.31-.78t.31-.78l2.63-2.63q.37-.31.8-.31t.76.31l4.31 4.31L21.06.31q.32-.3.79-.3t.78.3l2.62 2.63q.31.37.31.84t-.3.78L12 17.7q-.31.31-.78.31t-.78-.31z',
            'check-square' => [
                'p' => 'M25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22q1.25 0 2.13.88T28 5v22q0 1.25-.88 2.13T25 30zm-12.81-6.13 11.5-11.5q.75-.68 0-1.43L22.3 9.56q-.75-.75-1.43 0l-9.38 9.38-4.38-4.38q-.68-.75-1.43 0L4.3 15.94q-.75.75 0 1.44l6.5 6.5q.69.68 1.38 0z',
                'w' => .875,
                'vw' => 28
            ],
            'check-to-slot' => [
                'p' => 'M96 80c0-26.5 21.5-48 48-48H432c26.5 0 48 21.5 48 48V384H96V80zm313 47c-9.4-9.4-24.6-9.4-33.9 0l-111 111-47-47c-9.4-9.4-24.6-9.4-33.9 0s-9.4 24.6 0 33.9l64 64c9.4 9.4 24.6 9.4 33.9 0L409 161c9.4-9.4 9.4-24.6 0-33.9zM0 336c0-26.5 21.5-48 48-48H64V416H512V288h16c26.5 0 48 21.5 48 48v96c0 26.5-21.5 48-48 48H48c-26.5 0-48-21.5-48-48V336z',
                'vw' => 576,
                'vh' => 512
            ],
            'cheese' => 'M0 18h32v10q0 .81-.6 1.4T30 30H2q-.81 0-1.4-.6T0 28V18zM18.75 2q5.56.31 9.4 4.34T32 16H0L17.44 2.44q.5-.44 1.25-.44h.06z',
            'chess' => 'M4.75 13.13 2.62 7.18q-.18-.38.07-.7t.62-.3h3.2V4.3H5.12q-.38 0-.38-.37V2.5q0-.38.38-.38h1.8V.37q0-.37.38-.37h1.4q.37 0 .37.37v1.75h1.82q.37 0 .37.38v1.44q0 .37-.37.37H9.5V6.2h3.19q.38 0 .63.3t.06.7l-2.13 5.93h-6.5zm14.81 5.68q-.06-.06-.12-.18v-4.38q0-.25.25-.25h1.75q.25 0 .25.25v1.94h1.44v-1.94q0-.25.25-.25h3.25q.25 0 .25.25v1.94h1.43v-1.94q0-.25.25-.25h1.75q.2 0 .2.25v4.38q0 .12-.07.18l-1.56 1.07q0 3.87.75 7.12h-9.25q.8-3.31.75-7.12zm4.5 3.25v1.88h1.88v-1.88q0-.37-.28-.65t-.66-.28-.65.28-.29.65zm7.5 7.32q.07.06.07.18v2.2q0 .24-.25.24H18.63q-.25 0-.25-.25v-2.19q0-.12.06-.18l.87-.7v-.93q0-.25.25-.25h10.88q.25 0 .25.25v.94zm-16.25-.7q.2.13.2.32v2.63q0 .37-.38.37H.88q-.38 0-.38-.37V29q0-.19.19-.31l1.37-.88v-1.75q0-.37.38-.37h11.12q.38 0 .38.37v1.75zM12.5 15.64h-1.63q0 6.43 1.07 9.62H4.06q1.07-3.19 1.07-9.63H3.5q-.37 0-.37-.37V14q0-.38.37-.38h9q.38 0 .38.38v1.25q0 .38-.38.38z',
            'chess-bishop' => [
                'p' => 'M7.69 4.88q-.75-.25-1.22-.91T6 2.5q0-1.06.72-1.78T8.5 0h3q1 0 1.75.72T14 2.5q0 .81-.47 1.47t-1.22.9q2.25 1.57 4.25 5.13l-7.5 7.5q-.12.13-.12.28t.12.28l.88.88q.12.12.28.12t.28-.12l7.06-7.07q1.82 3.94 1.82 7.13 0 2.37-.91 3.62t-2.72 1.75V28H4.25v-3.63q-1.81-.5-2.72-1.75T.63 19t1.03-5.25 2.68-5.28 3.35-3.6zM20 31.25q0 .75-.75.75H.75Q0 32 0 31.25v-1.5Q0 29 .75 29h18.5q.75 0 .75.75v1.5z',
                'w' => .625,
                'vw' => 20
            ],
            'chess-board' => 'M12 16h4v4h-4v-4zm20 4h-4v-4h4v4zM0 12h4v4H0v-4zm32 0h-4V8h4v4zm0 16h-4v-4h4v4zM16 16v-4h4v4h-4zM0 32v-4h4v4H0zm8 0v-4h4v4H8zM24 0v4h-4V0h4zM8 0v4H4V0h4zm8 32v-4h4v4h-4zM0 4h4v4H0V4zm0 16h4v4H0v-4zM16 0v4h-4V0h4zm8 32v-4h4v4h-4zM4 24h4v4H4v-4zm16 0h4v4h-4v-4zm-4-12h-4V8h4v4zm8 12v-4h4v4h-4zM8 16v4H4v-4h4zm16 0v-4h4v4h-4zm8-16v4h-4V0h4zM8 8v4H4V8h4zm16-4h4v4h-4V4zM12 24h4v4h-4v-4zm-4-4h4v4H8v-4zm8 0h4v4h-4v-4zm-4-8v4H8v-4h4zm8-4h-4V4h4v4zm-8 0H8V4h4v4zm8 4V8h4v4h-4zm0 8v-4h4v4h-4z',
            'chess-king' => [
                'p' => 'M26 29.75v1.5q0 .75-.75.75H2.75Q2 32 2 31.25v-1.5q0-.75.75-.75h22.5q.75 0 .75.75zM25.5 9.5q.81 0 1.5.5t.9 1.25-.02 1.56l-5.38 15.2h-17L.12 12.8q-.43-1.18.32-2.25T2.5 9.5h10v-3H9.38Q9 6.5 9 6.12V3.88q0-.37.38-.37h3.12V.37q0-.37.38-.37h2.25q.37 0 .37.38V3.5h3.13q.37 0 .37.38v2.25q0 .37-.37.37H15.5v3h10z',
                'w' => .875,
                'vw' => 28
            ],
            'chess-knight' => [
                'p' => 'M22 14v14H2v-2.94q0-1.37.75-2.6t2-1.84l3.56-1.8q.7-.32 1.2-1.1t.5-1.6V13l-1.38.69q-.63.3-.82 1l-.75 2.93q-.06.38-.43.5l-1.82.75q-.12.07-.31.07-.13 0-.31-.07L.44 17.2Q0 17 0 16.5V6.63Q0 6 .44 5.56L1 5 .12 3.25Q0 3 0 2.75 0 2 .75 2H10q5 0 8.5 3.5T22 14zm1.25 15q.75 0 .75.75v1.5q0 .75-.75.75H.75Q0 32 0 31.25v-1.5Q0 29 .75 29h22.5zm-20-21q-.5 0-.88.38T2 9.24t.38.88.87.37.88-.38.37-.87-.38-.88T3.25 8z',
                'w' => .75,
                'vw' => 24
            ],
            'chess-pawn' => [
                'p' => 'M16.5 28h-13q.13-.13.4-.34t.91-1.13 1.13-2.06.9-3.16.4-4.31h-2q-.74 0-.74-.75v-1.5q0-.75.75-.75h2.5q-1.38-.63-2.19-1.9t-.81-2.85q0-2.19 1.53-3.72T10 4t3.72 1.53 1.53 3.72q0 1.56-.82 2.84T12.25 14h2.5q.75 0 .75.76v1.5q0 .75-.75.75h-2q0 3.69.93 6.44t1.88 3.69zm1.75 1q.75 0 .75.75v1.5q0 .75-.75.75H1.75Q1 32 1 31.25v-1.5q0-.75.75-.75h16.5z',
                'w' => .625,
                'vw' => 20
            ],
            'chess-queen' => 'M27.25 32H4.75Q4 32 4 31.25v-1.5q0-.75.75-.75h22.5q.75 0 .75.75v1.5q0 .75-.75.75zM15.97 0q1.47 0 2.5 1.03T19.5 3.5t-1.03 2.47T15.97 7 13.5 5.97t-1-2.47 1-2.47T15.97 0zm12.78 9.69 2.5 1.25q.38.25.38.69 0 .12-.07.3L25 28H7L.44 11.94q-.07-.19-.07-.31 0-.44.38-.7l2.5-1.24q.62-.38 1 .25Q5.62 12.3 7 12.3q.87 0 1.31-.47t.53-1.09.16-2q0-.19.06-.31t.16-.22.22-.16.3-.06h2.57q.5 0 .69.5.88 2.56 3 2.56t3-2.56q.19-.5.69-.5h2.56q.31 0 .53.22t.22.53q.06 2 .4 2.78t1.6.78q1.44 0 2.75-2.37.38-.63 1-.25z',
            'chess-rook' => [
                'p' => 'M5.06 13.44 2.25 10.8Q2 10.56 2 10.25v-7.5Q2 2 2.75 2h2.5Q6 2 6 2.75V5.5h3V2.75q0-.31.22-.53T9.75 2h4.5q.31 0 .53.22t.22.53V5.5h3V2.75q0-.31.22-.53t.53-.22h2.5q.31 0 .53.22t.22.53v7.5q0 .31-.25.56l-2.81 2.63Q19 21.13 21 28H3q2-6.87 2.06-14.56zM10 16v4h4v-4q0-.81-.6-1.4T12 14t-1.4.6T10 16zm14 13.75v1.5q0 .75-.75.75H.75Q0 32 0 31.25v-1.5Q0 29 .75 29h22.5q.75 0 .75.75z',
                'w' => .75,
                'vw' => 24
            ],
            'chevron-circle-down' => 'M31.5 16q0 6.44-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5t10.97 4.53T31.5 16zm-14.44 7.13 8.44-8.5q.5-.44.5-1.07t-.5-1.06l-1-1q-.5-.5-1.1-.5t-1.02.5L16 17.8l-6.37-6.3q-.44-.5-1.07-.5t-1.06.5l-1 1Q6 13 6 13.6t.5 1.02l8.44 8.5q.44.44 1.06.44t1.06-.44z',
            'chevron-circle-left' => 'M16 31.5q-6.44 0-10.97-4.53T.5 16 5.03 5.03 16 .5t10.97 4.53T31.5 16t-4.53 10.97T16 31.5zM8.87 17.06l8.5 8.44q.44.5 1.07.5t1.06-.5l1-1q.5-.5.5-1.1t-.5-1.02L14.2 16l6.31-6.37q.5-.44.5-1.07t-.5-1.06l-1-1Q19 6 18.4 6t-1.02.5l-8.5 8.44q-.44.44-.44 1.06t.44 1.06z',
            'chevron-circle-right' => 'M16 .5q6.44 0 10.97 4.53T31.5 16t-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5zm7.13 14.44-8.5-8.44q-.44-.5-1.07-.5t-1.06.5l-1 1Q11 8 11 8.6t.5 1.02L17.8 16l-6.31 6.38q-.5.43-.5 1.06t.5 1.06l1 1q.5.5 1.1.5t1.02-.5l8.5-8.44q.44-.44.44-1.06t-.44-1.06z',
            'chevron-circle-up' => 'M.5 16q0-6.44 4.53-10.97T16 .5t10.97 4.53T31.5 16t-4.53 10.97T16 31.5 5.03 26.97.5 16zm14.44-7.13-8.44 8.5q-.5.44-.5 1.07t.5 1.06l1 1q.5.5 1.1.5t1.02-.5L16 14.2l6.38 6.31q.43.5 1.06.5t1.06-.5l1-1q.5-.5.5-1.1t-.5-1.02l-8.44-8.5q-.44-.44-1.06-.44t-1.06.44z',
            'chevron-down' => [
                'p' => 'M12.94 23.81.8 11.7q-.43-.44-.43-1.06T.8 9.55L2.2 8.2q.44-.44 1.06-.47t1.06.4l9.7 9.7 9.68-9.7q.44-.43 1.06-.4t1.07.47l1.37 1.37q.44.44.44 1.06t-.44 1.07L15.07 23.8q-.44.5-1.07.5t-1.06-.5z',
                'w' => .873,
                'vw' => 28
            ],
            'chevron-left' => [
                'p' => 'M2.19 14.94 14.3 2.8q.44-.43 1.07-.43t1.06.43l1.37 1.4q.44.44.47 1.06t-.4 1.06l-9.7 9.7 9.7 9.68q.44.44.4 1.06t-.47 1.07l-1.37 1.37q-.44.44-1.06.44t-1.07-.44L2.2 17.07q-.5-.44-.5-1.07t.5-1.06z',
                'w' => .625,
                'vw' => 20
            ],
            'chevron-right' => [
                'p' => 'M17.81 17.06 5.7 29.2q-.44.44-1.07.44t-1.06-.44L2.2 27.8q-.44-.43-.47-1.06t.4-1.06l9.7-9.7-9.7-9.68q-.43-.44-.4-1.06t.47-1.06L3.56 2.8q.44-.44 1.06-.44t1.07.44L17.8 14.94q.5.43.5 1.06t-.5 1.06z',
                'w' => .623,
                'vw' => 20
            ],
            'chevron-up' => [
                'p' => 'M15.06 8.19 27.2 20.3q.44.44.44 1.07t-.44 1.06l-1.4 1.37q-.43.44-1.06.47t-1.06-.4l-9.7-9.7-9.68 9.7q-.44.44-1.06.4t-1.06-.47L.8 22.44Q.37 22 .37 21.38t.44-1.07L12.94 8.2q.43-.5 1.06-.5t1.06.5z',
                'w' => .875,
                'vw' => 28
            ],
            'child' => [
                'p' => 'M7.5 4.5q0-1.88 1.31-3.19T12.01 0t3.18 1.31T16.5 4.5t-1.3 3.19T12 9 8.81 7.7 7.5 4.5zm15.9.1q.6.59.6 1.4t-.56 1.44L17.5 13.3V30q0 .8-.6 1.4t-1.4.6h-1q-.81 0-1.4-.6t-.6-1.4v-7h-1v7q0 .8-.6 1.4t-1.4.6h-1q-.81 0-1.4-.6T6.5 30V13.3L.56 7.44Q0 6.8 0 6t.6-1.4T2 4t1.44.56L8.8 10h6.38l5.37-5.44Q21.2 4 22 4t1.4.6z',
                'w' => .75,
                'vw' => 24
            ],
            'child-combatant' => [
                'p' => 'M176 128A64 64 0 1 0 176 0a64 64 0 1 0 0 128zm-8 352V352h16V480c0 17.7 14.3 32 32 32s32-14.3 32-32V300.5L260.9 321c9.4 15 29.2 19.4 44.1 10s19.4-29.2 10-44.1l-51.7-82.1c-17.6-27.9-48.3-44.9-81.2-44.9H169.8c-33 0-63.7 16.9-81.2 44.9L36.9 287c-9.4 15-4.9 34.7 10 44.1s34.7 4.9 44.1-10L104 300.5V480c0 17.7 14.3 32 32 32s32-14.3 32-32zM448 0H432 416c-8.8 0-16 7.2-16 16s7.2 16 16 16V132.3c-9.6 5.5-16 15.9-16 27.7v32c-17.7 0-32 14.3-32 32V368c0 17.7 14.3 32 32 32h16v96c0 8.8 7.2 16 16 16h59.5c10.4 0 18-9.8 15.5-19.9L484 400h44c8.8 0 16-7.2 16-16V368c0-8.8-7.2-16-16-16H480V325.3l53.1-17.7c6.5-2.2 10.9-8.3 10.9-15.2V208c0-8.8-7.2-16-16-16H512c-8.8 0-16 7.2-16 16v56l-16 5.3V160c0-11.8-6.4-22.2-16-27.7V16c0-8.8-7.2-16-16-16z',
                'vw' => 576,
                'vh' => 512
            ],
            'child-dress' => [
                'p' => 'M224 64A64 64 0 1 0 96 64a64 64 0 1 0 128 0zM88 400v80c0 17.7 14.3 32 32 32s32-14.3 32-32V400h16v80c0 17.7 14.3 32 32 32s32-14.3 32-32V400h17.8c10.9 0 18.6-10.7 15.2-21.1l-31.1-93.4 28.6 37.8c10.7 14.1 30.8 16.8 44.8 6.2s16.8-30.7 6.2-44.8L254.6 207c-22.4-29.6-57.5-47-94.6-47s-72.2 17.4-94.6 47L6.5 284.7c-10.7 14.1-7.9 34.2 6.2 44.8s34.2 7.9 44.8-6.2l28.7-37.8L55 378.9C51.6 389.3 59.3 400 70.2 400H88z',
                'vw' => 320,
                'vh' => 512
            ],
            'child-reaching' => [
                'p' => 'M256 64A64 64 0 1 0 128 64a64 64 0 1 0 128 0zM152.9 169.3c-23.7-8.4-44.5-24.3-58.8-45.8L74.6 94.2C64.8 79.5 45 75.6 30.2 85.4s-18.7 29.7-8.9 44.4L40.9 159c18.1 27.1 42.8 48.4 71.1 62.4V480c0 17.7 14.3 32 32 32s32-14.3 32-32V384h32v96c0 17.7 14.3 32 32 32s32-14.3 32-32V221.6c29.1-14.2 54.4-36.2 72.7-64.2l18.2-27.9c9.6-14.8 5.4-34.6-9.4-44.3s-34.6-5.5-44.3 9.4L291 122.4c-21.8 33.4-58.9 53.6-98.8 53.6c-12.6 0-24.9-2-36.6-5.8c-.9-.3-1.8-.7-2.7-.9z',
                'vw' => 384,
                'vh' => 512
            ],
            'children' => [
                'p' => 'M160 0a64 64 0 1 1 0 128A64 64 0 1 1 160 0zM88 480V400H70.2c-10.9 0-18.6-10.7-15.2-21.1l31.1-93.4L57.5 323.3c-10.7 14.1-30.8 16.8-44.8 6.2s-16.8-30.7-6.2-44.8L65.4 207c22.4-29.6 57.5-47 94.6-47s72.2 17.4 94.6 47l58.9 77.7c10.7 14.1 7.9 34.2-6.2 44.8s-34.2 7.9-44.8-6.2l-28.6-37.8L265 378.9c3.5 10.4-4.3 21.1-15.2 21.1H232v80c0 17.7-14.3 32-32 32s-32-14.3-32-32V400H152v80c0 17.7-14.3 32-32 32s-32-14.3-32-32zM480 0a64 64 0 1 1 0 128A64 64 0 1 1 480 0zm-8 384v96c0 17.7-14.3 32-32 32s-32-14.3-32-32V300.5L395.1 321c-9.4 15-29.2 19.4-44.1 10s-19.4-29.2-10-44.1l51.7-82.1c17.6-27.9 48.3-44.9 81.2-44.9h12.3c33 0 63.7 16.9 81.2 44.9L619.1 287c9.4 15 4.9 34.7-10 44.1s-34.7 4.9-44.1-10L552 300.5V480c0 17.7-14.3 32-32 32s-32-14.3-32-32V384H472z',
                'vw' => 640,
                'vh' => 512
            ],
            'church' => [
                'p' => 'M29 15.44q1 .56 1 1.69V32h-6v-6q0-1.69-1.16-2.84T20 22t-2.84 1.16T16 26v6h-6V17.13q0-1.13 1-1.7l7-4.24V8h-3q-.44 0-.72-.28T14 7V5q0-.44.28-.72T15 4h3V1q0-.44.28-.72T19 0h2q.44 0 .72.28T22 1v3h3q.44 0 .72.28T26 5v2q0 .44-.28.72T25 8h-3v3.19zM0 24.75q0-.56.34-1.1t.85-.71L8 20v12H1q-.44 0-.72-.28T0 31v-6.25zm38.81-1.81q1.2.5 1.2 1.81V31q0 .44-.29.72T39 32h-7V20z',
                'w' => 1.25,
                'vw' => 40
            ],
            'circle' => 'M16 .5q6.44 0 10.97 4.53T31.5 16t-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5z',
            'circle-arrow-down' => [
                'p' => 'M256 0a256 256 0 1 0 0 512A256 256 0 1 0 256 0zM127 281c-9.4-9.4-9.4-24.6 0-33.9s24.6-9.4 33.9 0l71 71L232 136c0-13.3 10.7-24 24-24s24 10.7 24 24l0 182.1 71-71c9.4-9.4 24.6-9.4 33.9 0s9.4 24.6 0 33.9L273 393c-9.4 9.4-24.6 9.4-33.9 0L127 281z',
                'vw' => 512,
                'vh' => 512
            ],
            'circle-arrow-left' => [
                'p' => 'M512 256A256 256 0 1 0 0 256a256 256 0 1 0 512 0zM231 127c9.4-9.4 24.6-9.4 33.9 0s9.4 24.6 0 33.9l-71 71L376 232c13.3 0 24 10.7 24 24s-10.7 24-24 24l-182.1 0 71 71c9.4 9.4 9.4 24.6 0 33.9s-24.6 9.4-33.9 0L119 273c-9.4-9.4-9.4-24.6 0-33.9L231 127z',
                'vw' => 512,
                'vh' => 512
            ],
            'circle-arrow-right' => [
                'p' => 'M0 256a256 256 0 1 0 512 0A256 256 0 1 0 0 256zM281 385c-9.4 9.4-24.6 9.4-33.9 0s-9.4-24.6 0-33.9l71-71L136 280c-13.3 0-24-10.7-24-24s10.7-24 24-24l182.1 0-71-71c-9.4-9.4-9.4-24.6 0-33.9s24.6-9.4 33.9 0L393 239c9.4 9.4 9.4 24.6 0 33.9L281 385z',
                'vw' => 512,
                'vh' => 512
            ],
            'circle-arrow-up' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM385 231c9.4 9.4 9.4 24.6 0 33.9s-24.6 9.4-33.9 0l-71-71V376c0 13.3-10.7 24-24 24s-24-10.7-24-24V193.9l-71 71c-9.4 9.4-24.6 9.4-33.9 0s-9.4-24.6 0-33.9L239 119c9.4-9.4 24.6-9.4 33.9 0L385 231z',
                'vw' => 512,
                'vh' => 512
            ],
            'circle-check' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM369 209L241 337c-9.4 9.4-24.6 9.4-33.9 0l-64-64c-9.4-9.4-9.4-24.6 0-33.9s24.6-9.4 33.9 0l47 47L335 175c9.4-9.4 24.6-9.4 33.9 0s9.4 24.6 0 33.9z',
                'vw' => 512,
                'vh' => 512
            ],
            'circle-chevron-down' => [
                'p' => 'M256 0a256 256 0 1 0 0 512A256 256 0 1 0 256 0zM135 241c-9.4-9.4-9.4-24.6 0-33.9s24.6-9.4 33.9 0l87 87 87-87c9.4-9.4 24.6-9.4 33.9 0s9.4 24.6 0 33.9L273 345c-9.4 9.4-24.6 9.4-33.9 0L135 241z',
                'vw' => 512,
                'vh' => 512
            ],
            'circle-chevron-left' => [
                'p' => 'M512 256A256 256 0 1 0 0 256a256 256 0 1 0 512 0zM271 135c9.4-9.4 24.6-9.4 33.9 0s9.4 24.6 0 33.9l-87 87 87 87c9.4 9.4 9.4 24.6 0 33.9s-24.6 9.4-33.9 0L167 273c-9.4-9.4-9.4-24.6 0-33.9L271 135z',
                'vw' => 512,
                'vh' => 512
            ],
            'circle-chevron-right' => [
                'p' => 'M0 256a256 256 0 1 0 512 0A256 256 0 1 0 0 256zM241 377c-9.4 9.4-24.6 9.4-33.9 0s-9.4-24.6 0-33.9l87-87-87-87c-9.4-9.4-9.4-24.6 0-33.9s24.6-9.4 33.9 0L345 239c9.4 9.4 9.4 24.6 0 33.9L241 377z',
                'vw' => 512,
                'vh' => 512
            ],
            'circle-chevron-up' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM377 271c9.4 9.4 9.4 24.6 0 33.9s-24.6 9.4-33.9 0l-87-87-87 87c-9.4 9.4-24.6 9.4-33.9 0s-9.4-24.6 0-33.9L239 167c9.4-9.4 24.6-9.4 33.9 0L377 271z',
                'vw' => 512,
                'vh' => 512
            ],
            'circle-dollar-to-slot' => [
                'p' => 'M326.7 403.7c-22.1 8-45.9 12.3-70.7 12.3s-48.7-4.4-70.7-12.3c-.3-.1-.5-.2-.8-.3c-30-11-56.8-28.7-78.6-51.4C70 314.6 48 263.9 48 208C48 93.1 141.1 0 256 0S464 93.1 464 208c0 55.9-22 106.6-57.9 144c-1 1-2 2.1-3 3.1c-21.4 21.4-47.4 38.1-76.3 48.6zM256 91.9c-11.1 0-20.1 9-20.1 20.1v6c-5.6 1.2-10.9 2.9-15.9 5.1c-15 6.8-27.9 19.4-31.1 37.7c-1.8 10.2-.8 20 3.4 29c4.2 8.8 10.7 15 17.3 19.5c11.6 7.9 26.9 12.5 38.6 16l2.2 .7c13.9 4.2 23.4 7.4 29.3 11.7c2.5 1.8 3.4 3.2 3.7 4c.3 .8 .9 2.6 .2 6.7c-.6 3.5-2.5 6.4-8 8.8c-6.1 2.6-16 3.9-28.8 1.9c-6-1-16.7-4.6-26.2-7.9l0 0 0 0 0 0c-2.2-.7-4.3-1.5-6.4-2.1c-10.5-3.5-21.8 2.2-25.3 12.7s2.2 21.8 12.7 25.3c1.2 .4 2.7 .9 4.4 1.5c7.9 2.7 20.3 6.9 29.8 9.1V304c0 11.1 9 20.1 20.1 20.1s20.1-9 20.1-20.1v-5.5c5.3-1 10.5-2.5 15.4-4.6c15.7-6.7 28.4-19.7 31.6-38.7c1.8-10.4 1-20.3-3-29.4c-3.9-9-10.2-15.6-16.9-20.5c-12.2-8.8-28.3-13.7-40.4-17.4l-.8-.2c-14.2-4.3-23.8-7.3-29.9-11.4c-2.6-1.8-3.4-3-3.6-3.5c-.2-.3-.7-1.6-.1-5c.3-1.9 1.9-5.2 8.2-8.1c6.4-2.9 16.4-4.5 28.6-2.6c4.3 .7 17.9 3.3 21.7 4.3c10.7 2.8 21.6-3.5 24.5-14.2s-3.5-21.6-14.2-24.5c-4.4-1.2-14.4-3.2-21-4.4V112c0-11.1-9-20.1-20.1-20.1zM48 352H64c19.5 25.9 44 47.7 72.2 64H64v32H256 448V416H375.8c28.2-16.3 52.8-38.1 72.2-64h16c26.5 0 48 21.5 48 48v64c0 26.5-21.5 48-48 48H48c-26.5 0-48-21.5-48-48V400c0-26.5 21.5-48 48-48z',
                'vw' => 512,
                'vh' => 512
            ],
            'circle-dot' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zm0-352a96 96 0 1 1 0 192 96 96 0 1 1 0-192z',
                'vw' => 512,
                'vh' => 512
            ],
            'circle-down' => [
                'p' => 'M256 0a256 256 0 1 0 0 512A256 256 0 1 0 256 0zM376.9 294.6L269.8 394.5c-3.8 3.5-8.7 5.5-13.8 5.5s-10.1-2-13.8-5.5L135.1 294.6c-4.5-4.2-7.1-10.1-7.1-16.3c0-12.3 10-22.3 22.3-22.3l57.7 0 0-96c0-17.7 14.3-32 32-32l32 0c17.7 0 32 14.3 32 32l0 96 57.7 0c12.3 0 22.3 10 22.3 22.3c0 6.2-2.6 12.1-7.1 16.3z',
                'vw' => 512,
                'vh' => 512
            ],
            'circle-exclamation' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zm0-384c13.3 0 24 10.7 24 24V264c0 13.3-10.7 24-24 24s-24-10.7-24-24V152c0-13.3 10.7-24 24-24zM224 352a32 32 0 1 1 64 0 32 32 0 1 1 -64 0z',
                'vw' => 512,
                'vh' => 512
            ],
            'circle-h' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM368 152V256 360c0 13.3-10.7 24-24 24s-24-10.7-24-24V280H192l0 80c0 13.3-10.7 24-24 24s-24-10.7-24-24l0-208c0-13.3 10.7-24 24-24s24 10.7 24 24v80H320V152c0-13.3 10.7-24 24-24s24 10.7 24 24z',
                'vw' => 512,
                'vh' => 512
            ],
            'circle-half-stroke' => [
                'p' => 'M448 256c0-106-86-192-192-192V448c106 0 192-86 192-192zM0 256a256 256 0 1 1 512 0A256 256 0 1 1 0 256z',
                'vw' => 512,
                'vh' => 512
            ],
            'circle-info' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM216 336h24V272H216c-13.3 0-24-10.7-24-24s10.7-24 24-24h48c13.3 0 24 10.7 24 24v88h8c13.3 0 24 10.7 24 24s-10.7 24-24 24H216c-13.3 0-24-10.7-24-24s10.7-24 24-24zm40-208a32 32 0 1 1 0 64 32 32 0 1 1 0-64z',
                'vw' => 512,
                'vh' => 512
            ],
            'circle-left' => [
                'p' => 'M512 256A256 256 0 1 0 0 256a256 256 0 1 0 512 0zM217.4 376.9L117.5 269.8c-3.5-3.8-5.5-8.7-5.5-13.8s2-10.1 5.5-13.8l99.9-107.1c4.2-4.5 10.1-7.1 16.3-7.1c12.3 0 22.3 10 22.3 22.3l0 57.7 96 0c17.7 0 32 14.3 32 32l0 32c0 17.7-14.3 32-32 32l-96 0 0 57.7c0 12.3-10 22.3-22.3 22.3c-6.2 0-12.1-2.6-16.3-7.1z',
                'vw' => 512,
                'vh' => 512
            ],
            'circle-minus' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM184 232H328c13.3 0 24 10.7 24 24s-10.7 24-24 24H184c-13.3 0-24-10.7-24-24s10.7-24 24-24z',
                'vw' => 512,
                'vh' => 512
            ],
            'circle-nodes' => [
                'p' => 'M418.4 157.9c35.3-8.3 61.6-40 61.6-77.9c0-44.2-35.8-80-80-80c-43.4 0-78.7 34.5-80 77.5L136.2 151.1C121.7 136.8 101.9 128 80 128c-44.2 0-80 35.8-80 80s35.8 80 80 80c12.2 0 23.8-2.7 34.1-7.6L259.7 407.8c-2.4 7.6-3.7 15.8-3.7 24.2c0 44.2 35.8 80 80 80s80-35.8 80-80c0-27.7-14-52.1-35.4-66.4l37.8-207.7zM156.3 232.2c2.2-6.9 3.5-14.2 3.7-21.7l183.8-73.5c3.6 3.5 7.4 6.7 11.6 9.5L317.6 354.1c-5.5 1.3-10.8 3.1-15.8 5.5L156.3 232.2z',
                'vw' => 512,
                'vh' => 512
            ],
            'circle-notch' => 'M18 2.44q0-.75.56-1.19T19.88 1q5.06 1.31 8.34 5.47T31.5 16q0 4.19-2.06 7.75t-5.63 5.66-7.75 2.09Q9.63 31.5 5.1 27T.5 16.06q0-5.37 3.28-9.56T12.12 1q.7-.19 1.29.25T14 2.44V3.5q0 1.13-1.13 1.44-3.62 1-6 4.06T4.5 16q0 4.75 3.38 8.13T16 27.5t8.13-3.37T27.5 16q0-3.94-2.38-7t-6-4.06Q18 4.62 18 3.5V2.44z',
            'circle-pause' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM224 192V320c0 17.7-14.3 32-32 32s-32-14.3-32-32V192c0-17.7 14.3-32 32-32s32 14.3 32 32zm128 0V320c0 17.7-14.3 32-32 32s-32-14.3-32-32V192c0-17.7 14.3-32 32-32s32 14.3 32 32z',
                'vw' => 512,
                'vh' => 512
            ],
            'circle-play' => [
                'p' => 'M0 256a256 256 0 1 1 512 0A256 256 0 1 1 0 256zM188.3 147.1c-7.6 4.2-12.3 12.3-12.3 20.9V344c0 8.7 4.7 16.7 12.3 20.9s16.8 4.1 24.3-.5l144-88c7.1-4.4 11.5-12.1 11.5-20.5s-4.4-16.1-11.5-20.5l-144-88c-7.4-4.5-16.7-4.7-24.3-.5z',
                'vw' => 512,
                'vh' => 512
            ],
            'circle-plus' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM232 344V280H168c-13.3 0-24-10.7-24-24s10.7-24 24-24h64V168c0-13.3 10.7-24 24-24s24 10.7 24 24v64h64c13.3 0 24 10.7 24 24s-10.7 24-24 24H280v64c0 13.3-10.7 24-24 24s-24-10.7-24-24z',
                'vw' => 512,
                'vh' => 512
            ],
            'circle-question' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM169.8 165.3c7.9-22.3 29.1-37.3 52.8-37.3h58.3c34.9 0 63.1 28.3 63.1 63.1c0 22.6-12.1 43.5-31.7 54.8L280 264.4c-.2 13-10.9 23.6-24 23.6c-13.3 0-24-10.7-24-24V250.5c0-8.6 4.6-16.5 12.1-20.8l44.3-25.4c4.7-2.7 7.6-7.7 7.6-13.1c0-8.4-6.8-15.1-15.1-15.1H222.6c-3.4 0-6.4 2.1-7.5 5.3l-.4 1.2c-4.4 12.5-18.2 19-30.6 14.6s-19-18.2-14.6-30.6l.4-1.2zM224 352a32 32 0 1 1 64 0 32 32 0 1 1 -64 0z',
                'vw' => 512,
                'vh' => 512
            ],
            'circle-radiation' => [
                'p' => 'M256 64a192 192 0 1 1 0 384 192 192 0 1 1 0-384zm0 448A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM200 256c0-20.7 11.3-38.8 28-48.5l-36-62.3c-8.8-15.3-28.7-20.8-42-9c-25.6 22.6-43.9 53.3-50.9 88.1C95.7 241.5 110.3 256 128 256l72 0zm28 48.5l-36 62.4c-8.8 15.3-3.6 35.2 13.1 40.8c16 5.4 33.1 8.3 50.9 8.3s34.9-2.9 50.9-8.3c16.7-5.6 21.9-25.5 13.1-40.8l-36-62.4c-8.2 4.8-17.8 7.5-28 7.5s-19.8-2.7-28-7.5zM312 256l72 0c17.7 0 32.3-14.5 28.8-31.8c-7-34.8-25.3-65.5-50.9-88.1c-13.2-11.7-33.1-6.3-42 9l-36 62.3c16.7 9.7 28 27.8 28 48.5zm-56 32a32 32 0 1 0 0-64 32 32 0 1 0 0 64z',
                'vw' => 512,
                'vh' => 512
            ],
            'circle-right' => [
                'p' => 'M0 256a256 256 0 1 0 512 0A256 256 0 1 0 0 256zM294.6 135.1l99.9 107.1c3.5 3.8 5.5 8.7 5.5 13.8s-2 10.1-5.5 13.8L294.6 376.9c-4.2 4.5-10.1 7.1-16.3 7.1C266 384 256 374 256 361.7l0-57.7-96 0c-17.7 0-32-14.3-32-32l0-32c0-17.7 14.3-32 32-32l96 0 0-57.7c0-12.3 10-22.3 22.3-22.3c6.2 0 12.1 2.6 16.3 7.1z',
                'vw' => 512,
                'vh' => 512
            ],
            'circle-stop' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM192 160H320c17.7 0 32 14.3 32 32V320c0 17.7-14.3 32-32 32H192c-17.7 0-32-14.3-32-32V192c0-17.7 14.3-32 32-32z',
                'vw' => 512,
                'vh' => 512
            ],
            'circle-up' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM135.1 217.4l107.1-99.9c3.8-3.5 8.7-5.5 13.8-5.5s10.1 2 13.8 5.5l107.1 99.9c4.5 4.2 7.1 10.1 7.1 16.3c0 12.3-10 22.3-22.3 22.3H304v96c0 17.7-14.3 32-32 32H240c-17.7 0-32-14.3-32-32V256H150.3C138 256 128 246 128 233.7c0-6.2 2.6-12.1 7.1-16.3z',
                'vw' => 512,
                'vh' => 512
            ],
            'circle-user' => [
                'p' => 'M399 384.2C376.9 345.8 335.4 320 288 320H224c-47.4 0-88.9 25.8-111 64.2c35.2 39.2 86.2 63.8 143 63.8s107.8-24.7 143-63.8zM0 256a256 256 0 1 1 512 0A256 256 0 1 1 0 256zm256 16a72 72 0 1 0 0-144 72 72 0 1 0 0 144z',
                'vw' => 512,
                'vh' => 512
            ],
            'circle-xmark' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM175 175c9.4-9.4 24.6-9.4 33.9 0l47 47 47-47c9.4-9.4 24.6-9.4 33.9 0s9.4 24.6 0 33.9l-47 47 47 47c9.4 9.4 9.4 24.6 0 33.9s-24.6 9.4-33.9 0l-47-47-47 47c-9.4 9.4-24.6 9.4-33.9 0s-9.4-24.6 0-33.9l47-47-47-47c-9.4-9.4-9.4-24.6 0-33.9z',
                'vw' => 512,
                'vh' => 512
            ],
            'city' => [
                'p' => 'M38.5 12q.63 0 1.06.44T40 13.5V30q0 .81-.6 1.4T38 32H2q-.81 0-1.4-.6T0 30V7.5q0-.62.44-1.06T1.5 6H4V1q0-.44.28-.72T5 0h1q.44 0 .72.28T7 1v5h4V1q0-.44.28-.72T12 0h1q.44 0 .72.28T14 1v5h4V1.5q0-.63.44-1.06T19.5 0h9q.63 0 1.07.44T30 1.5V12h8.5zM8 25.25v-2.5Q8 22 7.25 22h-2.5Q4 22 4 22.75v2.5q0 .75.75.75h2.5Q8 26 8 25.25zm0-6v-2.5Q8 16 7.25 16h-2.5Q4 16 4 16.75v2.5q0 .75.75.75h2.5Q8 20 8 19.25zm0-6v-2.5Q8 10 7.25 10h-2.5Q4 10 4 10.75v2.5q0 .75.75.75h2.5Q8 14 8 13.25zm8 12v-2.5q0-.75-.75-.75h-2.5q-.75 0-.75.75v2.5q0 .75.75.75h2.5q.75 0 .75-.75zm0-6v-2.5q0-.75-.75-.75h-2.5q-.75 0-.75.75v2.5q0 .75.75.75h2.5q.75 0 .75-.75zm0-6v-2.5q0-.75-.75-.75h-2.5q-.75 0-.75.75v2.5q0 .75.75.75h2.5q.75 0 .75-.75zm10 6v-2.5q0-.75-.75-.75h-2.5q-.75 0-.75.75v2.5q0 .75.75.75h2.5q.75 0 .75-.75zm0-6v-2.5q0-.75-.75-.75h-2.5q-.75 0-.75.75v2.5q0 .75.75.75h2.5q.75 0 .75-.75zm0-6v-2.5Q26 4 25.25 4h-2.5Q22 4 22 4.75v2.5q0 .75.75.75h2.5Q26 8 26 7.25zm10 18v-2.5q0-.75-.75-.75h-2.5q-.75 0-.75.75v2.5q0 .75.75.75h2.5q.75 0 .75-.75zm0-6v-2.5q0-.75-.75-.75h-2.5q-.75 0-.75.75v2.5q0 .75.75.75h2.5q.75 0 .75-.75z',
                'w' => 1.25,
                'vw' => 40
            ],
            'clapperboard' => [
                'p' => 'M448 32H361.9l-1 1-127 127h92.1l1-1L453.8 32.3c-1.9-.2-3.8-.3-5.8-.3zm64 128V96c0-15.1-5.3-29.1-14-40l-104 104H512zM294.1 32H201.9l-1 1L73.9 160h92.1l1-1 127-127zM64 32C28.7 32 0 60.7 0 96v64H6.1l1-1 127-127H64zM512 192H0V416c0 35.3 28.7 64 64 64H448c35.3 0 64-28.7 64-64V192z',
                'vw' => 512,
                'vh' => 512
            ],
            'clinic-medical' => [
                'p' => 'm18 7.19 13.69 12.06.15.13q.1.07.16.06V31q0 .44-.28.72T31 32H5q-.44 0-.72-.28T4 31V19.44q.06 0 .19-.06t.12-.13zm6 16.31v-3q0-.5-.5-.5H20v-3.5q0-.5-.5-.5h-3q-.5 0-.5.5V20h-3.5q-.5 0-.5.5v3q0 .5.5.5H16v3.5q0 .5.5.5h3q.5 0 .5-.5V24h3.5q.5 0 .5-.5zm11.69-8.75q.31.31.31.75 0 .38-.25.69l-1.31 1.5q-.31.31-.75.31t-.69-.25L18.69 5.13q-.32-.25-.7-.25t-.68.25L3 17.75q-.25.25-.69.25t-.69-.31l-1.37-1.5Q0 15.88 0 15.5q0-.44.31-.75l16-14.12Q17.06 0 18 0t1.69.63z',
                'w' => 1.125,
                'vw' => 36
            ],
            'clipboard' => [
                'p' => 'M24 7v22q0 1.25-.88 2.13T21 32H3q-1.25 0-2.13-.88T0 29V7q0-1.25.88-2.13T3 4h5q0-1.63 1.19-2.81T12-.01t2.81 1.2T16.01 4h5q1.24 0 2.12.88T24 7zM12 2.5q-.63 0-1.06.44T10.5 4t.44 1.06T12 5.5t1.06-.44T13.5 4t-.44-1.06T12 2.5zm6 7.13V8.37Q18 8 17.62 8H6.38Q6 8 6 8.38v1.24q0 .38.38.38h11.25q.37 0 .37-.38z',
                'w' => .75,
                'vw' => 24
            ],
            'clipboard-check' => [
                'p' => 'M21 4q1.25 0 2.13.88T24 7v22q0 1.25-.88 2.13T21 32H3q-1.25 0-2.13-.88T0 29V7q0-1.25.88-2.13T3 4h5q0-1.63 1.19-2.81T12-.01t2.81 1.2T16.01 4h5zm-9-1.5q-.63 0-1.06.44T10.5 4t.44 1.06T12 5.5t1.06-.44T13.5 4t-.44-1.06T12 2.5zM19.56 17q.57-.56 0-1.06l-1.75-1.82q-.5-.5-1.06 0l-6.63 6.63-2.87-2.94q-.5-.5-1.06 0l-1.75 1.75q-.57.56 0 1.06l5.12 5.25q.57.5 1.07 0z',
                'w' => .75,
                'vw' => 24
            ],
            'clipboard-list' => [
                'p' => 'M21 4q1.25 0 2.13.88T24 7v22q0 1.25-.88 2.13T21 32H3q-1.25 0-2.13-.88T0 29V7q0-1.25.88-2.13T3 4h5q0-1.63 1.19-2.81T12-.01t2.81 1.2T16.01 4h5zM6 26.5q.63 0 1.06-.44T7.5 25t-.44-1.06T6 23.5t-1.06.44T4.5 25t.44 1.06T6 26.5zm0-6q.63 0 1.06-.44T7.5 19t-.44-1.06T6 17.5t-1.06.44T4.5 19t.44 1.06T6 20.5zm0-6q.63 0 1.06-.44T7.5 13t-.44-1.06T6 11.5t-1.06.44T4.5 13t.44 1.06T6 14.5zm6-12q-.63 0-1.06.44T10.5 4t.44 1.06T12 5.5t1.06-.44T13.5 4t-.44-1.06T12 2.5zm8 23v-1q0-.5-.5-.5h-9q-.5 0-.5.5v1q0 .5.5.5h9q.5 0 .5-.5zm0-6v-1q0-.5-.5-.5h-9q-.5 0-.5.5v1q0 .5.5.5h9q.5 0 .5-.5zm0-6v-1q0-.5-.5-.5h-9q-.5 0-.5.5v1q0 .5.5.5h9q.5 0 .5-.5z',
                'w' => .75,
                'vw' => 24
            ],
            'clipboard-question' => [
                'p' => 'M192 0c-41.8 0-77.4 26.7-90.5 64H64C28.7 64 0 92.7 0 128V448c0 35.3 28.7 64 64 64H320c35.3 0 64-28.7 64-64V128c0-35.3-28.7-64-64-64H282.5C269.4 26.7 233.8 0 192 0zm0 64a32 32 0 1 1 0 64 32 32 0 1 1 0-64zM105.8 229.3c7.9-22.3 29.1-37.3 52.8-37.3h58.3c34.9 0 63.1 28.3 63.1 63.1c0 22.6-12.1 43.5-31.7 54.8L216 328.4c-.2 13-10.9 23.6-24 23.6c-13.3 0-24-10.7-24-24V314.5c0-8.6 4.6-16.5 12.1-20.8l44.3-25.4c4.7-2.7 7.6-7.7 7.6-13.1c0-8.4-6.8-15.1-15.1-15.1H158.6c-3.4 0-6.4 2.1-7.5 5.3l-.4 1.2c-4.4 12.5-18.2 19-30.6 14.6s-19-18.2-14.6-30.6l.4-1.2zM160 416a32 32 0 1 1 64 0 32 32 0 1 1 -64 0z',
                'vw' => 384,
                'vh' => 512
            ],
            'clipboard-user' => [
                'p' => 'M192 0c-41.8 0-77.4 26.7-90.5 64H64C28.7 64 0 92.7 0 128V448c0 35.3 28.7 64 64 64H320c35.3 0 64-28.7 64-64V128c0-35.3-28.7-64-64-64H282.5C269.4 26.7 233.8 0 192 0zm0 64a32 32 0 1 1 0 64 32 32 0 1 1 0-64zM128 256a64 64 0 1 1 128 0 64 64 0 1 1 -128 0zM80 432c0-44.2 35.8-80 80-80h64c44.2 0 80 35.8 80 80c0 8.8-7.2 16-16 16H96c-8.8 0-16-7.2-16-16z',
                'vw' => 384,
                'vh' => 512
            ],
            'clock' => 'M16 .5q6.44 0 10.97 4.53T31.5 16t-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5zm3.56 21.88q.63.43 1.07-.13l1.75-2.44q.43-.62-.13-1.06l-4-2.88V7.25q0-.75-.75-.75h-3q-.75 0-.75.75v10.5q0 .37.31.62z',
            'clock-rotate-left' => [
                'p' => 'M75 75L41 41C25.9 25.9 0 36.6 0 57.9V168c0 13.3 10.7 24 24 24H134.1c21.4 0 32.1-25.9 17-41l-30.8-30.8C155 85.5 203 64 256 64c106 0 192 86 192 192s-86 192-192 192c-40.8 0-78.6-12.7-109.7-34.4c-14.5-10.1-34.4-6.6-44.6 7.9s-6.6 34.4 7.9 44.6C151.2 495 201.7 512 256 512c141.4 0 256-114.6 256-256S397.4 0 256 0C185.3 0 121.3 28.7 75 75zm181 53c-13.3 0-24 10.7-24 24V256c0 6.4 2.5 12.5 7 17l72 72c9.4 9.4 24.6 9.4 33.9 0s9.4-24.6 0-33.9l-65-65V152c0-13.3-10.7-24-24-24z',
                'vw' => 512,
                'vh' => 512
            ],
            'clone' => 'M29 0q1.25 0 2.13.88T32 3v18q0 1.25-.88 2.13T29 24H11q-1.25 0-2.13-.88T8 21V3q0-1.25.88-2.13T11 0h18zM11 26h13v3q0 1.25-.88 2.13T21 32H3q-1.25 0-2.13-.88T0 29V11q0-1.25.88-2.13T3 8h3v13q0 2.06 1.47 3.53T11 26z',
            'closed-captioning' => 'M29 4q1.25 0 2.13.88T32 7v18q0 1.25-.88 2.13T29 28H3q-1.25 0-2.13-.88T0 25V7q0-1.25.88-2.13T3 4h26zM13.62 18q-1.93 1.69-3.84.97t-1.9-3.03q0-2.13 2-2.78t3.87.84q.38.25.63-.12L15.44 12q.12-.19-.06-.37-1.57-1.57-4.13-1.66t-4.6 1.53T4.63 16q0 2.94 1.97 4.63t4.57 1.53 4.22-1.9q.25-.26.06-.5l-1.25-1.7q-.25-.37-.56-.06zm11.88 0q-1.88 1.69-3.81.97t-1.94-3.03q0-2.13 2-2.78t3.94.84q.31.25.56-.12L27.38 12q.06-.19-.13-.37-1.56-1.57-4.13-1.66t-4.59 1.53T16.5 16q0 2.94 2 4.63t4.56 1.53 4.25-1.9q.2-.26 0-.5l-1.18-1.7q-.25-.37-.63-.06z',
            'cloud' => [
                'p' => 'M33.63 14.19q1.8.31 3.24 1.44t2.29 2.8T40 22q0 3.31-2.34 5.66T32 30H9q-3.75 0-6.38-2.62T0 21q0-2.94 1.69-5.25T6 12.5V12q0-4.12 2.94-7.06T16 2q2.75 0 5.06 1.38T24.7 7q1.5-1 3.31-1 2.5 0 4.25 1.75T34 12q0 1.13-.37 2.19z',
                'w' => 1.25,
                'vw' => 40
            ],
            'cloud-arrow-down' => [
                'p' => 'M144 480C64.5 480 0 415.5 0 336c0-62.8 40.2-116.2 96.2-135.9c-.1-2.7-.2-5.4-.2-8.1c0-88.4 71.6-160 160-160c59.3 0 111 32.2 138.7 80.2C409.9 102 428.3 96 448 96c53 0 96 43 96 96c0 12.2-2.3 23.8-6.4 34.6C596 238.4 640 290.1 640 352c0 70.7-57.3 128-128 128H144zm79-167l80 80c9.4 9.4 24.6 9.4 33.9 0l80-80c9.4-9.4 9.4-24.6 0-33.9s-24.6-9.4-33.9 0l-39 39V184c0-13.3-10.7-24-24-24s-24 10.7-24 24V318.1l-39-39c-9.4-9.4-24.6-9.4-33.9 0s-9.4 24.6 0 33.9z',
                'vw' => 640,
                'vh' => 512
            ],
            'cloud-arrow-up' => [
                'p' => 'M144 480C64.5 480 0 415.5 0 336c0-62.8 40.2-116.2 96.2-135.9c-.1-2.7-.2-5.4-.2-8.1c0-88.4 71.6-160 160-160c59.3 0 111 32.2 138.7 80.2C409.9 102 428.3 96 448 96c53 0 96 43 96 96c0 12.2-2.3 23.8-6.4 34.6C596 238.4 640 290.1 640 352c0 70.7-57.3 128-128 128H144zm79-217c-9.4 9.4-9.4 24.6 0 33.9s24.6 9.4 33.9 0l39-39V392c0 13.3 10.7 24 24 24s24-10.7 24-24V257.9l39 39c9.4 9.4 24.6 9.4 33.9 0s9.4-24.6 0-33.9l-80-80c-9.4-9.4-24.6-9.4-33.9 0l-80 80z',
                'vw' => 640,
                'vh' => 512
            ],
            'cloud-bolt' => [
                'p' => 'M0 224c0 53 43 96 96 96h47.2L290 202.5c17.6-14.1 42.6-14 60.2 .2s22.8 38.6 12.8 58.8L333.7 320H352h64c53 0 96-43 96-96s-43-96-96-96c-.5 0-1.1 0-1.6 0c1.1-5.2 1.6-10.5 1.6-16c0-44.2-35.8-80-80-80c-24.3 0-46.1 10.9-60.8 28C256.5 24.3 219.1 0 176 0C114.1 0 64 50.1 64 112c0 7.1 .7 14.1 1.9 20.8C27.6 145.4 0 181.5 0 224zm330.1 3.6c-5.8-4.7-14.2-4.7-20.1-.1l-160 128c-5.3 4.2-7.4 11.4-5.1 17.8s8.3 10.7 15.1 10.7h70.1L177.7 488.8c-3.4 6.7-1.6 14.9 4.3 19.6s14.2 4.7 20.1 .1l160-128c5.3-4.2 7.4-11.4 5.1-17.8s-8.3-10.7-15.1-10.7H281.9l52.4-104.8c3.4-6.7 1.6-14.9-4.2-19.6z',
                'vw' => 512,
                'vh' => 512
            ],
            'cloud-download-alt' => [
                'p' => 'M33.63 14.19q1.8.31 3.24 1.44t2.29 2.8T40 22q0 3.31-2.34 5.66T32 30H9q-3.75 0-6.38-2.62T0 21q0-2.94 1.69-5.25T6 12.5V12q0-4.12 2.94-7.06T16 2q2.75 0 5.06 1.38T24.7 7q1.5-1 3.31-1 2.5 0 4.25 1.75T34 12q0 1.13-.37 2.19zm-8.32 5.5q.44-.44.2-1.06t-.95-.63H20.5v-7q0-.44-.28-.72T19.5 10h-3q-.44 0-.72.28t-.28.72v7h-4.06q-.7 0-.94.63t.19 1.06l6.62 6.62q.69.7 1.38 0z',
                'w' => 1.25,
                'vw' => 40
            ],
            'cloud-meatball' => 'M3 22q1.25 0 2.13.88T6 25t-.88 2.13T3 28t-2.13-.88T0 25t.88-2.13T3 22zm26 0q1.25 0 2.13.88T32 25t-.88 2.13T29 28t-2.13-.88T26 25t.88-2.13T29 22zm-7.44.69Q23 23.44 23 25t-1.44 2.31q.5 1.57-.62 2.63-.75.81-1.82.81-.3 0-.8-.19Q17.55 32 16 32t-2.31-1.44q-.44.2-.82.2-1.06 0-1.81-.82-1.13-1.13-.63-2.63Q9 26.56 9 25.01t1.43-2.32q-.5-1.57.63-2.66t2.62-.6Q14.44 18 16 18t2.31 1.44q1.56-.5 2.66.62t.6 2.63zM32 14q0 2.5-1.75 4.25T26 20h-2.69q-.37-.75-.94-1.38-1.3-1.3-3.18-1.3Q17.87 16 15.99 16t-3.18 1.31-3.19 1.32q-.56.56-.94 1.37H6q-2.5 0-4.25-1.75T0 14q0-1.94 1.15-3.5t2.97-2.19Q4 7.62 4 7q0-2.88 2.06-4.94T11 0q2 0 3.65 1.03t2.53 2.78Q18.68 2 21 2q2.06 0 3.53 1.47T26 7q0 .56-.13 1H26q2.5 0 4.25 1.75T32 14z',
            'cloud-moon' => [
                'p' => 'M21.44 22.06q1.94.13 3.25 1.57T26 27q0 2.06-1.47 3.53T21 32H6q-2.5 0-4.25-1.75T0 26q0-1.94 1.13-3.47t2.93-2.15q0-.07-.03-.2T4 20q0-2.5 1.75-4.25T10 14q1.69 0 3.06.84t2.2 2.28Q16.37 16 18 16t2.82 1.19T22 20q0 1.06-.56 2.06zm13.87-3.44q.38-.06.6.29t-.03.65q-3.32 4.06-8.57 4.38-1.06-2.25-3.37-3.25.06-.38.06-.7 0-2.5-1.75-4.24T18 14q-1.19 0-2.25.44-.44-.44-1.06-.88-.13-.81-.13-1.56 0-5 3.53-8.5T26.56 0q1.07 0 2.2.19.37.06.43.47t-.25.59q-2.19 1.25-3.47 3.44t-1.28 4.75q0 4.37 3.4 7.22t7.72 1.97z',
                'w' => 1.125,
                'vw' => 36
            ],
            'cloud-moon-rain' => [
                'p' => 'M21.88 14.13q1.8.3 2.96 1.68t1.16 3.2q0 2.05-1.47 3.52T21 24H5q-2.06 0-3.53-1.47T0 19q0-1.81 1.16-3.19T4 14.12V14q0-2.5 1.75-4.25T10 8q3.06 0 4.88 2.5 1-.5 2.12-.5 1.81 0 3.19 1.16t1.69 2.97zM35.5 14q.31-.06.44.19t-.06.5q-2.7 3.31-7 3.31-.2 0-.35-.03t-.34-.03-.32-.06q-.18-1.2-.78-2.28t-1.5-1.88-2.1-1.22q-1-2.69-3.68-3.87Q20 5 22.62 2.5T28.87 0q.82 0 1.63.13.31.06.37.37t-.25.44q-1.62.94-2.59 2.6t-.97 3.52q0 1.63.69 3.07t1.84 2.37 2.72 1.38 3.19.12zM22.75 26.13q.88.5.38 1.37l-2.25 4q-.32.5-.88.5-.25 0-.5-.13-.88-.5-.38-1.37l2.32-4q.19-.38.6-.47t.71.1zm-6 0q.88.5.38 1.37l-2.25 4q-.32.5-.88.5-.25 0-.5-.13-.88-.5-.38-1.37l2.32-4q.19-.38.6-.47t.71.1zm-6 0q.88.5.38 1.37l-2.26 4q-.3.5-.87.5-.25 0-.5-.13-.88-.5-.38-1.37l2.32-4q.19-.38.6-.47t.71.1zm-6 0q.88.5.38 1.37l-2.25 4q-.32.5-.88.5-.25 0-.5-.13-.88-.5-.38-1.37l2.32-4q.19-.38.6-.47t.71.1z',
                'w' => 1.123,
                'vw' => 36
            ],
            'cloud-rain' => 'M26 8q2.5 0 4.25 1.75T32 14t-1.75 4.25T26 20H6q-2.5 0-4.25-1.75T0 14q0-1.94 1.16-3.5t2.96-2.19Q4 7.62 4 7q0-2.88 2.06-4.94T11 0q2 0 3.66 1.03t2.53 2.78Q18.69 2 21 2q2.06 0 3.53 1.47T26 7q0 .56-.12 1H26zM5.5 23.38Q5.63 23 6 23t.5.38q.38 1.3 1.44 2.87T9 28.88q0 1.3-.87 2.21T6 32t-2.12-.9T3 28.86q0-1 1.06-2.59t1.44-2.9zm10 0q.13-.38.5-.38t.5.38q.38 1.3 1.44 2.87T19 28.88q0 .87-.4 1.59t-1.1 1.12T16 32q-1.25 0-2.12-.9T13 28.86q0-1 1.06-2.59t1.44-2.9zm10 0q.13-.38.5-.38t.5.38q.38 1.3 1.44 2.87T29 28.88q0 1.3-.87 2.21T26 32t-2.12-.9-.88-2.23q0-.8.44-1.59t1.1-1.78.96-2.13z',
            'cloud-showers-heavy' => 'M11.5 23.13q.38.18.47.59t-.1.78l-4 7q-.3.5-.87.5-.25 0-.5-.13-.38-.18-.47-.59t.1-.78l4-7q.18-.38.59-.47t.78.1zm6 0q.38.18.47.59t-.1.78l-4 7q-.3.5-.87.5-.25 0-.5-.13-.38-.18-.47-.59t.1-.78l4-7q.18-.38.59-.47t.78.1zm-12 0q.38.18.47.59t-.1.78l-4 7q-.3.5-.87.5-.25 0-.5-.13-.38-.18-.47-.59t.1-.78l4-7q.18-.38.59-.47t.78.1zm24 0q.38.18.47.59t-.1.78l-4 7q-.3.5-.87.5-.25 0-.5-.13-.38-.18-.47-.59t.1-.78l4-7q.18-.38.59-.47t.78.1zm-6 0q.38.18.47.59t-.1.78l-4 7q-.3.5-.87.5-.25 0-.5-.13-.38-.18-.47-.59t.1-.78l4-7q.18-.38.59-.47t.78.1zM26 8q2.5 0 4.25 1.75T32 14t-1.75 4.25T26 20H6q-2.5 0-4.25-1.75T0 14q0-1.94 1.16-3.5t2.96-2.19Q4 7.62 4 7q0-2.88 2.06-4.94T11 0q2 0 3.66 1.03t2.53 2.78Q18.69 2 21 2q2.06 0 3.53 1.47T26 7q0 .56-.12 1H26z',
            'cloud-showers-water' => [
                'p' => 'M224 0c38.6 0 71.9 22.8 87.2 55.7C325.7 41.1 345.8 32 368 32c38.7 0 71 27.5 78.4 64H448c35.3 0 64 28.7 64 64s-28.7 64-64 64H128c-35.3 0-64-28.7-64-64s28.7-64 64-64c0-53 43-96 96-96zM140.6 292.3l-48 80c-6.8 11.4-21.6 15-32.9 8.2s-15.1-21.6-8.2-32.9l48-80c6.8-11.4 21.6-15.1 32.9-8.2s15.1 21.6 8.2 32.9zm327.8-32.9c11.4 6.8 15 21.6 8.2 32.9l-48 80c-6.8 11.4-21.6 15-32.9 8.2s-15-21.6-8.2-32.9l48-80c6.8-11.4 21.6-15.1 32.9-8.2zM252.6 292.3l-48 80c-6.8 11.4-21.6 15-32.9 8.2s-15.1-21.6-8.2-32.9l48-80c6.8-11.4 21.6-15.1 32.9-8.2s15.1 21.6 8.2 32.9zm103.8-32.9c11.4 6.8 15 21.6 8.2 32.9l-48 80c-6.8 11.4-21.6 15-32.9 8.2s-15.1-21.6-8.2-32.9l48-80c6.8-11.4 21.6-15.1 32.9-8.2zM306.5 421.9C329 437.4 356.5 448 384 448c26.9 0 55.4-10.8 77.4-26.1l0 0c11.9-8.5 28.1-7.8 39.2 1.7c14.4 11.9 32.5 21 50.6 25.2c17.2 4 27.9 21.2 23.9 38.4s-21.2 27.9-38.4 23.9c-24.5-5.7-44.9-16.5-58.2-25C449.5 501.7 417 512 384 512c-31.9 0-60.6-9.9-80.4-18.9c-5.8-2.7-11.1-5.3-15.6-7.7c-4.5 2.4-9.7 5.1-15.6 7.7c-19.8 9-48.5 18.9-80.4 18.9c-33 0-65.5-10.3-94.5-25.8c-13.4 8.4-33.7 19.3-58.2 25c-17.2 4-34.4-6.7-38.4-23.9s6.7-34.4 23.9-38.4c18.1-4.2 36.2-13.3 50.6-25.2c11.1-9.4 27.3-10.1 39.2-1.7l0 0C136.7 437.2 165.1 448 192 448c27.5 0 55-10.6 77.5-26.1c11.1-7.9 25.9-7.9 37 0z',
                'vw' => 576,
                'vh' => 512
            ],
            'cloud-sun' => [
                'p' => 'M35.94 20.38q1.81.62 2.94 2.15T40 26q0 2.5-1.75 4.25T34 32H17q-2.5 0-4.25-1.75T11 26q0-2.38 1.63-4.06t3.93-1.88l-.03-.19q-.03-.13-.03-.21v-.16q0-2.88 2.07-4.94t4.93-2.06q2.13 0 3.82 1.12t2.56 3q1-.62 2.12-.62 1.63 0 2.82 1.19t1.18 2.8q0 .07-.03.2t-.03.18zM9.06 18.93q.94 1 2.25 1.5-.93.94-1.5 2.12l-4.75 1.57q-.56.18-.97-.22t-.21-.97l1.8-5.5L.5 14.87Q0 14.57 0 14t.5-.88l5.19-2.56-1.82-5.5q-.18-.56.22-.97t.97-.22l5.5 1.82 2.57-5.2q.3-.5.87-.5t.88.5l2.56 5.2 5.5-1.82q.56-.18.97.22t.22.97l-1.88 5.57q-1.06.12-1.87.5-.57-1.26-1.44-2.07Q16.88 7 14 7T9.06 9.06 7 14t2.06 4.94zM8.75 14q0-2.19 1.53-3.72T14 8.75q1.56 0 2.84.84t1.91 2.28q-3.69 2.26-4.19 6.5-.75.25-1.56.75-1.81-.3-3.03-1.78T8.75 14z',
                'w' => 1.25,
                'vw' => 40
            ],
            'cloud-sun-rain' => [
                'p' => 'M31.88 14.13q1.8.3 2.96 1.68t1.16 3.2q0 2.05-1.47 3.52T31 24H15q-2.06 0-3.53-1.47T10 19q0-1.81 1.13-3.19T14 14.12V14q0-2.5 1.75-4.25T20 8q3.06 0 4.88 2.5 1-.5 2.12-.5 1.19 0 2.25.53T31 12t.88 2.13zM7.75 16.25q.19.13.63.5Q8 17.88 8 19q0 .13.03.28l.03.16-3.69 1.25q-.5.12-.84-.22t-.22-.85l1.56-4.68-4.43-2.2Q0 12.5 0 12t.44-.75l4.43-2.19-1.56-4.69q-.12-.5.22-.84t.84-.22l4.7 1.56L11.24.43Q11.5 0 12 0t.75.43l2.19 4.44 4.69-1.56q.5-.13.84.22t.22.84L20.13 6H20q-2.3 0-4.3 1.3-1.76-1.43-4.04-1.3t-3.9 1.75Q6 9.5 6 12t1.75 4.25zm4.38-3.63q-1.7.76-2.82 2.32Q8 13.75 8 12q0-1.63 1.19-2.81T12 7.99q1.13 0 2.13.63-1.63 1.75-2 4zm20.62 13.5q.88.5.38 1.38l-2.26 4q-.3.5-.87.5-.25 0-.5-.13-.88-.5-.38-1.37l2.32-4q.19-.38.6-.47t.71.1zm-6 0q.88.5.38 1.38l-2.25 4q-.32.5-.88.5-.25 0-.5-.13-.88-.5-.38-1.37l2.32-4q.19-.38.6-.47t.71.1zm-6 0q.88.5.38 1.38l-2.25 4q-.32.5-.88.5-.25 0-.5-.13-.88-.5-.38-1.37l2.32-4q.19-.38.6-.47t.71.1zm-6 0q.88.5.38 1.38l-2.26 4q-.3.5-.87.5-.25 0-.5-.13-.88-.5-.38-1.37l2.32-4q.19-.38.6-.47t.71.1z',
                'w' => 1.125,
                'vw' => 36
            ],
            'cloud-upload-alt' => [
                'p' => 'M33.63 14.19q1.8.31 3.24 1.44t2.29 2.8T40 22q0 3.31-2.34 5.66T32 30H9q-3.75 0-6.38-2.62T0 21q0-2.94 1.69-5.25T6 12.5V12q0-4.12 2.94-7.06T16 2q2.75 0 5.06 1.38T24.7 7q1.5-1 3.31-1 2.5 0 4.25 1.75T34 12q0 1.13-.37 2.19zM24.55 18q.7 0 .94-.63t-.19-1.06L18.7 9.7q-.69-.7-1.38 0l-6.62 6.6q-.44.44-.19 1.07t.94.62h4.06v7q0 .44.28.72t.72.28h3q.44 0 .72-.28t.28-.72v-7h4.06z',
                'w' => 1.25,
                'vw' => 40
            ],
            'clover' => [
                'p' => 'M173.3 32C139.4 32 112 59.4 112 93.3v4.9c0 12 3.3 23.7 9.4 34l18.8 31.3c1.1 1.8 1.2 3.1 1 4.2c-.2 1.2-.8 2.5-2 3.6s-2.4 1.8-3.6 2c-1 .2-2.4 .1-4.2-1l-31.3-18.8c-10.3-6.2-22-9.4-34-9.4H61.3C27.4 144 0 171.4 0 205.3c0 16.2 6.5 31.8 17.9 43.3l1.2 1.2c3.4 3.4 3.4 9 0 12.4l-1.2 1.2C6.5 274.9 0 290.5 0 306.7C0 340.6 27.4 368 61.3 368h4.9c12 0 23.7-3.3 34-9.4l31.3-18.8c1.8-1.1 3.1-1.2 4.2-1c1.2 .2 2.5 .8 3.6 2s1.8 2.4 2 3.6c.2 1 .1 2.4-1 4.2l-18.8 31.3c-6.2 10.3-9.4 22-9.4 34v4.9c0 33.8 27.4 61.3 61.3 61.3c16.2 0 31.8-6.5 43.3-17.9l1.2-1.2c3.4-3.4 9-3.4 12.4 0l1.2 1.2c11.5 11.5 27.1 17.9 43.3 17.9c33.8 0 61.3-27.4 61.3-61.3v-4.9c0-12-3.3-23.7-9.4-34l-18.8-31.3c-1.1-1.8-1.2-3.1-1-4.2c.2-1.2 .8-2.5 2-3.6s2.4-1.8 3.6-2c1-.2 2.4-.1 4.2 1l31.3 18.8c10.3 6.2 22 9.4 34 9.4h4.9c33.8 0 61.3-27.4 61.3-61.3c0-16.2-6.5-31.8-17.9-43.3l-1.2-1.2c-3.4-3.4-3.4-9 0-12.4l1.2-1.2c11.5-11.5 17.9-27.1 17.9-43.3c0-33.8-27.4-61.3-61.3-61.3h-4.9c-12 0-23.7 3.3-34 9.4l-31.3 18.8c-1.8 1.1-3.1 1.2-4.2 1c-1.2-.2-2.5-.8-3.6-2s-1.8-2.4-2-3.6c-.2-1-.1-2.4 1-4.2l18.8-31.3c6.2-10.3 9.4-22 9.4-34V93.3C336 59.4 308.6 32 274.7 32c-16.2 0-31.8 6.5-43.3 17.9l-1.2 1.2c-3.4 3.4-9 3.4-12.4 0l-1.2-1.2C205.1 38.5 189.5 32 173.3 32z',
                'vw' => 448,
                'vh' => 512
            ],
            'cocktail' => [
                'p' => 'M18.5 29q1.06 0 1.78.72T21 31.5q0 .5-.5.5h-15q-.5 0-.5-.5 0-1.06.72-1.78T7.5 29H11v-7.81L.44 10.62q-.7-.75-.32-1.68T1.56 8h22.88q1.06 0 1.43.94t-.3 1.68L15 21.2V29h3.5zM27 0q3.75 0 6.38 2.63T36 9t-2.63 6.38T27 18q-2.5 0-4.69-1.31l2.25-2.25Q25.75 15 27 15q2.5 0 4.25-1.75T33 9t-1.75-4.25T27 3q-1.63 0-3 .81t-2.19 2.2h-3.25q.94-2.7 3.25-4.35T27 0z',
                'w' => 1.125,
                'vw' => 36
            ],
            'code' => [
                'p' => 'm17.44 32-3.82-1.13q-.75-.18-.5-.93L21.63.56q.2-.75.94-.56l3.82 1.12q.75.2.5.94l-8.5 29.38q-.25.75-.94.56zm-7.13-7.06q-.5.56-1.06.06l-9-8.44q-.63-.56 0-1.12l9-8.44q.56-.5 1.06.06l2.7 2.88q.55.56 0 1.06l-5.7 5 5.7 5q.55.5 0 1.06zm20.44.06q-.56.5-1.06-.06l-2.7-2.88q-.55-.56 0-1.06l5.7-5-5.7-5q-.55-.5 0-1.06l2.7-2.88q.5-.56 1.06-.06l9 8.44q.63.56 0 1.12z',
                'w' => 1.25,
                'vw' => 40
            ],
            'code-branch' => [
                'p' => 'M24 9q0 1.63-.94 2.94t-2.44 1.81q-.06 2.63-1.37 4.25-.81 1-2.22 1.6t-2.4.78-2.76.3q-3.12.32-4.12.82-.63.31-.94.81 1.44.57 2.31 1.85T10 27q0 2.07-1.47 3.53T5 32t-3.53-1.47T0 27q0-1.12.44-2.1t1.25-1.65 1.81-1V9.75Q1.94 9.31.97 8T0 5q0-2.06 1.47-3.53T5 0t3.53 1.47T10 5q0 1.7-.97 3T6.5 9.75v9q1.5-.68 5.06-1.06 1.5-.13 2.25-.22t1.7-.44 1.37-.9q.68-.82.75-2.32-1.07-.3-1.88-1t-1.28-1.68T14 9q0-2.06 1.47-3.53T19 4t3.53 1.47T24 9zM5 4q-.44 0-.72.28T4 5t.28.72T5 6t.72-.28T6 5t-.28-.72T5 4zm0 24q.44 0 .72-.28T6 27t-.28-.72T5 26t-.72.28T4 27t.28.72T5 28zM19 8q-.44 0-.72.28T18 9t.28.72.72.28.72-.28T20 9t-.28-.72T19 8z',
                'w' => .75,
                'vw' => 24
            ],
            'code-commit' => [
                'p' => 'M320 336a80 80 0 1 0 0-160 80 80 0 1 0 0 160zm156.8-48C462 361 397.4 416 320 416s-142-55-156.8-128H32c-17.7 0-32-14.3-32-32s14.3-32 32-32H163.2C178 151 242.6 96 320 96s142 55 156.8 128H608c17.7 0 32 14.3 32 32s-14.3 32-32 32H476.8z',
                'vw' => 640,
                'vh' => 512
            ],
            'code-compare' => [
                'p' => 'M320 488c0 9.5-5.6 18.1-14.2 21.9s-18.8 2.3-25.8-4.1l-80-72c-5.1-4.6-7.9-11-7.9-17.8s2.9-13.3 7.9-17.8l80-72c7-6.3 17.2-7.9 25.8-4.1s14.2 12.4 14.2 21.9v40h16c35.3 0 64-28.7 64-64V153.3C371.7 141 352 112.8 352 80c0-44.2 35.8-80 80-80s80 35.8 80 80c0 32.8-19.7 61-48 73.3V320c0 70.7-57.3 128-128 128H320v40zM456 80a24 24 0 1 0 -48 0 24 24 0 1 0 48 0zM192 24c0-9.5 5.6-18.1 14.2-21.9s18.8-2.3 25.8 4.1l80 72c5.1 4.6 7.9 11 7.9 17.8s-2.9 13.3-7.9 17.8l-80 72c-7 6.3-17.2 7.9-25.8 4.1s-14.2-12.4-14.2-21.9V128H176c-35.3 0-64 28.7-64 64V358.7c28.3 12.3 48 40.5 48 73.3c0 44.2-35.8 80-80 80s-80-35.8-80-80c0-32.8 19.7-61 48-73.3V192c0-70.7 57.3-128 128-128h16V24zM56 432a24 24 0 1 0 48 0 24 24 0 1 0 -48 0z',
                'vw' => 512,
                'vh' => 512
            ],
            'code-fork' => [
                'p' => 'M80 104a24 24 0 1 0 0-48 24 24 0 1 0 0 48zm80-24c0 32.8-19.7 61-48 73.3V192c0 17.7 14.3 32 32 32H304c17.7 0 32-14.3 32-32V153.3C307.7 141 288 112.8 288 80c0-44.2 35.8-80 80-80s80 35.8 80 80c0 32.8-19.7 61-48 73.3V192c0 53-43 96-96 96H256v70.7c28.3 12.3 48 40.5 48 73.3c0 44.2-35.8 80-80 80s-80-35.8-80-80c0-32.8 19.7-61 48-73.3V288H144c-53 0-96-43-96-96V153.3C19.7 141 0 112.8 0 80C0 35.8 35.8 0 80 0s80 35.8 80 80zm208 24a24 24 0 1 0 0-48 24 24 0 1 0 0 48zM248 432a24 24 0 1 0 -48 0 24 24 0 1 0 48 0z',
                'vw' => 448,
                'vh' => 512
            ],
            'code-merge' => [
                'p' => 'M80 56a24 24 0 1 1 0 48 24 24 0 1 1 0-48zm32.4 97.2c28-12.4 47.6-40.5 47.6-73.2c0-44.2-35.8-80-80-80S0 35.8 0 80c0 32.8 19.7 61 48 73.3V358.7C19.7 371 0 399.2 0 432c0 44.2 35.8 80 80 80s80-35.8 80-80c0-32.8-19.7-61-48-73.3V272c26.7 20.1 60 32 96 32h86.7c12.3 28.3 40.5 48 73.3 48c44.2 0 80-35.8 80-80s-35.8-80-80-80c-32.8 0-61 19.7-73.3 48H208c-49.9 0-91-38.1-95.6-86.8zM80 408a24 24 0 1 1 0 48 24 24 0 1 1 0-48zM344 272a24 24 0 1 1 48 0 24 24 0 1 1 -48 0z',
                'vw' => 448,
                'vh' => 512
            ],
            'code-pull-request' => [
                'p' => 'M305.8 2.1C314.4 5.9 320 14.5 320 24V64h16c70.7 0 128 57.3 128 128V358.7c28.3 12.3 48 40.5 48 73.3c0 44.2-35.8 80-80 80s-80-35.8-80-80c0-32.8 19.7-61 48-73.3V192c0-35.3-28.7-64-64-64H320v40c0 9.5-5.6 18.1-14.2 21.9s-18.8 2.3-25.8-4.1l-80-72c-5.1-4.6-7.9-11-7.9-17.8s2.9-13.3 7.9-17.8l80-72c7-6.3 17.2-7.9 25.8-4.1zM104 80A24 24 0 1 0 56 80a24 24 0 1 0 48 0zm8 73.3V358.7c28.3 12.3 48 40.5 48 73.3c0 44.2-35.8 80-80 80s-80-35.8-80-80c0-32.8 19.7-61 48-73.3V153.3C19.7 141 0 112.8 0 80C0 35.8 35.8 0 80 0s80 35.8 80 80c0 32.8-19.7 61-48 73.3zM104 432a24 24 0 1 0 -48 0 24 24 0 1 0 48 0zm328 24a24 24 0 1 0 0-48 24 24 0 1 0 0 48z',
                'vw' => 512,
                'vh' => 512
            ],
            'coffee' => [
                'p' => 'M12 24q-2.5 0-4.25-1.75T6 18V3.5q0-.63.44-1.06T7.5 2H32q3.31 0 5.66 2.34T40 10t-2.34 5.65T32 18h-2q0 2.5-1.75 4.25T24 24H12zM32 6h-2v8h2q1.63 0 2.81-1.19t1.2-2.81-1.2-2.81T32 5.99zm3 24H3Q.94 30 .19 28q-.57-1.5.25-1.94.12-.06.31-.06h36.5q.94 0 .69 1.44-.31 1.81-1.94 2.37-.5.19-1 .19z',
                'w' => 1.25,
                'vw' => 40
            ],
            'cog' => 'M30.44 19.75q.56.25.37.88-1.06 3.3-3.44 5.87-.37.44-.87.19l-2.69-1.57q-1.69 1.44-3.81 2.2v3.06q0 .62-.56.75-3.44.75-6.82 0Q12 31 12 30.38V27.3q-2.13-.75-3.81-2.19L5.56 26.7q-.56.25-.94-.19-2.37-2.56-3.44-5.88-.18-.56.38-.87l2.62-1.56Q4 17.12 4 15.99t.18-2.18l-2.62-1.56Q1 12 1.18 11.37 2.25 8.06 4.62 5.5q.38-.44.94-.13l2.63 1.5q1.68-1.43 3.8-2.19V1.63q0-.62.57-.75 3.44-.75 6.88 0 .56.13.56.75v3.06q2.12.76 3.81 2.2l2.63-1.57q.56-.25.93.19 2.38 2.56 3.44 5.87.2.63-.37.88L27.8 13.8q.38 2.19 0 4.38zM16 21q2.06 0 3.53-1.47T21 16t-1.47-3.53T16 11t-3.53 1.47T11 16t1.47 3.53T16 21z',
            'cogs' => [
                'p' => 'm32 11.94-.5.87q-.31.57-.94.38-1.12-.44-2-1.19-.5-.44-.19-1l.5-.88q-.62-.74-1-1.68h-1q-.68 0-.75-.7-.25-1.12 0-2.3.07-.63.75-.63h1q.38-.94 1-1.69l-.5-.93q-.3-.57.2-.94.87-.75 2-1.2.62-.24.93.32l.5.94q1-.19 2 0l.5-.94q.31-.56.94-.31 1.12.37 2 1.19.5.37.19.94l-.5.93q.62.75 1 1.7h1q.68 0 .75.62.25 1.18 0 2.3-.07.7-.75.7h-1q-.38.94-1 1.69l.5.87q.3.63-.2 1-.87.75-2 1.19-.62.19-.93-.38l-.5-.87q-1 .19-2 0zm-.63-3.69q1.13.88 2.32.5t1.47-1.47-.53-2.34q-1.13-.88-2.32-.5T30.84 5.9t.54 2.34zm-7.25 9.63 2.13 1.06q1.06.62.63 1.81-.57 1.63-2.63 4.13-.81.93-1.88.3l-1.87-1.05q-1.5 1.3-3.38 2v2.12q0 .5-.37.94t-.88.5q-2.37.44-4.74 0-.57-.06-.91-.5t-.35-.94v-2.12q-1.93-.7-3.43-2l-1.82 1.06q-1.06.62-1.87-.31-2-2.44-2.63-4.13-.43-1.19.63-1.81l2.06-1.06q-.37-2 0-3.94L.75 12.87q-1.13-.62-.63-1.8.63-1.7 2.63-4.13.75-.94 1.87-.32l1.82 1Q8 6.31 9.87 5.68V3.56q0-.56.35-.97t.9-.47q2.38-.44 4.76 0 .24 0 .46.16t.41.34.28.44.1.5v2.13q1.87.68 3.37 1.93l1.88-1q1.06-.68 1.87.32 2 2.37 2.63 4.12.43 1.19-.63 1.81l-2.13 1.07q.38 1.94 0 3.94zm-7.3 1.3q1.06-1.37 1.18-2.87t-.56-2.6-1.78-1.77-2.6-.53-2.87 1.22Q9.13 14 9 15.5t.56 2.6 1.78 1.78 2.6.53 2.87-1.22zM32 30.64l-.5.87q-.31.56-.94.38-1.12-.44-2-1.2-.5-.43-.19-1l.5-.87q-.62-.75-1-1.69h-1q-.68 0-.75-.68-.25-1.13 0-2.32.07-.62.75-.62h1q.38-.94 1-1.75l-.5-.88q-.3-.56.2-1 .87-.75 2-1.12.62-.25.93.31l.5.94q1-.19 2 0l.5-.94q.31-.56.94-.31 1.12.37 2 1.12.5.44.19 1l-.5.88q.62.81 1 1.75h1q.68 0 .75.62.25 1.2 0 2.32-.07.68-.75.68h-1q-.38.94-1 1.7l.5.87q.3.56-.2 1-.87.75-2 1.18-.62.2-.93-.37l-.5-.88q-1 .2-2 0zm-.63-3.7q1.13.88 2.32.5t1.47-1.46-.53-2.35q-1.13-.87-2.32-.5t-1.47 1.47.54 2.35z',
                'w' => 1.248,
                'vw' => 40
            ],
            'coins' => 'M0 25.31q3.81 2.7 12 2.7t12-2.7v2.7q0 1.62-3.53 2.8T12 32.01t-8.47-1.2T0 28.01v-2.7zM20 8q-4.94 0-8.47-1.19T8 4t3.53-2.81T20-.01t8.47 1.2T32 3.99t-3.53 2.82T20 8zM0 18.75q1.88 1.56 5.13 2.4T12 22q2.38 0 4.63-.34t4.18-1.1 3.2-1.81V22q0 1.62-3.54 2.81T12 26t-8.47-1.19T0 22v-3.25zm26 .69V15.5q3.94-.69 6-2.19V16q0 .56-.44 1.06t-1.22.97-1.87.81-2.47.6zM12 10q5 0 8.5 1.47T24 15t-3.5 3.53T12 20t-8.5-1.47T0 15t3.5-3.53T12 10zm13.69 3.5q-.75-2.13-3.75-3.56Q28.8 9.62 32 7.3V10q0 1.13-1.72 2.06t-4.6 1.44z',
            'colon-sign' => [
                'p' => 'M255 39.8c4.3-17.1-6.1-34.5-23.3-38.8S197.2 7.1 193 24.2L181.9 68.6C96.1 87.8 32 164.4 32 256c0 58.1 25.8 110.2 66.7 145.4L81 472.2c-4.3 17.1 6.1 34.5 23.3 38.8s34.5-6.1 38.8-23.3l13-52.1c9 3.4 18.4 6.2 28 8.2L177 472.2c-4.3 17.1 6.1 34.5 23.3 38.8s34.5-6.1 38.8-23.3l10.4-41.4c33.4-4.4 64.1-17.4 89.8-36.7c14.1-10.6 17-30.7 6.4-44.8s-30.7-17-44.8-6.4c-10.2 7.7-21.7 13.9-34 18.3L321 160c9.4-.3 18.5-4.7 24.6-12.8c10.6-14.1 7.8-34.2-6.4-44.8c-1.1-.8-2.2-1.6-3.3-2.4L351 39.8c4.3-17.1-6.1-34.5-23.3-38.8S293.2 7.1 289 24.2L277.2 71.5c-9.3-2.7-18.8-4.6-28.6-5.9L255 39.8zM163.2 143.3L117.3 326.8C103.9 306.5 96 282.2 96 256c0-48.7 27.2-91 67.2-112.7zm8.6 229.5l61.1-244.6c9.9 .7 19.5 2.5 28.7 5.3l-62 248.1c-9.7-1.9-19-4.8-27.8-8.8z',
                'vw' => 384,
                'vh' => 512
            ],
            'columns' => 'M29 2q1.25 0 2.13.88T32 5v22q0 1.25-.88 2.13T29 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h26zM14 26V10H4v16h10zm14 0V10H18v16h10z',
            'comment' => 'M16 2q6.63 0 11.31 3.81t4.7 9.2-4.7 9.18T16 28q-3.5 0-6.69-1.19-4.06 3.2-8.81 3.2-.13 0-.25-.1t-.19-.22q-.12-.31.1-.53t.75-.88 1.4-2.19 1.25-2.9Q0 19.63 0 15q0-5.37 4.69-9.19T15.99 2z',
            'comment-alt' => 'M28 0q1.63 0 2.81 1.19T32.01 4v18q0 1.63-1.2 2.81T28 26.01h-9l-7.81 5.87q-.38.25-.78.03t-.41-.66V26H4q-1.63 0-2.81-1.19t-1.2-2.8V4q0-1.63 1.2-2.82T4 0h24z',
            'comment-dollar' => 'M16 2q6.63 0 11.31 3.81t4.7 9.2-4.7 9.18T16 28q-3.5 0-6.69-1.19-4.06 3.2-8.81 3.2-.5 0-.5-.5 0-.2.12-.38.32-.25.79-.85t1.37-2.15 1.28-2.94Q0 19.63 0 15q0-5.37 4.69-9.19T15.99 2zm1.5 18.88q1.44-.13 2.34-1.29t.72-2.65q-.12-1.06-.87-1.85T17.87 14l-3.12-.94q-.38-.06-.38-.5 0-.5.57-.5H17q.31 0 .63.13.5.25.87-.13L19.56 11q.25-.25.22-.63t-.28-.56q-.87-.56-2-.69V8q0-.44-.28-.72T16.5 7h-1q-.44 0-.72.28T14.5 8v1.12q-.94.07-1.69.63t-1.12 1.44-.25 1.87q.12 1.06.87 1.85T14.13 16l3.12.94q.38.06.38.5 0 .5-.57.5h-2q-.37 0-.68-.13-.5-.25-.88.13L12.44 19q-.25.25-.22.63t.28.56q.88.56 2 .69V22q0 .44.28.72t.72.28h1q.44 0 .72-.28t.28-.72v-1.12z',
            'comment-dots' => 'M16 2q6.63 0 11.31 3.81t4.7 9.2-4.7 9.18T16 28q-3.5 0-6.69-1.19-4.06 3.2-8.81 3.2-.13 0-.25-.1t-.19-.22q-.12-.31.1-.53t.75-.88 1.4-2.19 1.25-2.9Q0 19.63 0 15q0-5.37 4.69-9.19T15.99 2zM8 17q.81 0 1.4-.6T10 15t-.6-1.4T8 13t-1.4.6T6 15t.6 1.4T8 17zm8 0q.81 0 1.4-.6T18 15t-.6-1.4T16 13t-1.4.6T14 15t.6 1.4 1.4.6zm8 0q.81 0 1.4-.6T26 15t-.6-1.4T24 13t-1.4.6T22 15t.6 1.4 1.4.6z',
            'comment-medical' => 'M16 2q6.63 0 11.31 3.81t4.7 9.2-4.7 9.18T16 28q-3.5 0-6.69-1.19-4.06 3.2-8.81 3.2-.5 0-.5-.5 0-.2.12-.38.25-.2.79-.85t1.4-2.19 1.25-2.9Q0 19.56 0 15q0-5.37 4.69-9.19T15.99 2zm6 14.5v-3q0-.5-.5-.5H18V9.5q0-.5-.5-.5h-3q-.5 0-.5.5V13h-3.5q-.5 0-.5.5v3q0 .5.5.5H14v3.5q0 .5.5.5h3q.5 0 .5-.5V17h3.5q.5 0 .5-.5z',
            'comment-slash' => [
                'p' => 'M4 15q0-1.63.56-3.38l20.32 15.76Q22.5 28 20 28q-3.5 0-6.69-1.19Q9.25 30 4.5 30q-.31 0-.44-.31t.07-.57q2.75-3.25 3.43-5.93Q4 19.62 4 14.99zm35.63 13.63q.3.25.37.65t-.19.72l-1.25 1.63q-.62.75-1.37.12L.37 3.37Q-.38 2.75.2 1.95L1.44.37Q2.06-.38 2.87.2l6.57 5.1Q14 2 20 2q6.63 0 11.31 3.81t4.7 9.19q0 4.5-3.57 8.12z',
                'w' => 1.25,
                'vw' => 40
            ],
            'comment-sms' => [
                'p' => 'M256 448c141.4 0 256-93.1 256-208S397.4 32 256 32S0 125.1 0 240c0 45.1 17.7 86.8 47.7 120.9c-1.9 24.5-11.4 46.3-21.4 62.9c-5.5 9.2-11.1 16.6-15.2 21.6c-2.1 2.5-3.7 4.4-4.9 5.7c-.6 .6-1 1.1-1.3 1.4l-.3 .3 0 0 0 0 0 0 0 0c-4.6 4.6-5.9 11.4-3.4 17.4c2.5 6 8.3 9.9 14.8 9.9c28.7 0 57.6-8.9 81.6-19.3c22.9-10 42.4-21.9 54.3-30.6c31.8 11.5 67 17.9 104.1 17.9zM202.9 176.8c6.5-2.2 13.7 .1 17.9 5.6L256 229.3l35.2-46.9c4.1-5.5 11.3-7.8 17.9-5.6s10.9 8.3 10.9 15.2v96c0 8.8-7.2 16-16 16s-16-7.2-16-16V240l-19.2 25.6c-3 4-7.8 6.4-12.8 6.4s-9.8-2.4-12.8-6.4L224 240v48c0 8.8-7.2 16-16 16s-16-7.2-16-16V192c0-6.9 4.4-13 10.9-15.2zm173.1 38c0 .2 0 .4 0 .4c.1 .1 .6 .8 2.2 1.7c3.9 2.3 9.6 4.1 18.3 6.8l.6 .2c7.4 2.2 17.3 5.2 25.2 10.2c9.1 5.7 17.4 15.2 17.6 29.9c.2 15-7.6 26-17.8 32.3c-9.5 5.9-20.9 7.9-30.7 7.6c-12.2-.4-23.7-4.4-32.6-7.4l0 0 0 0c-1.4-.5-2.7-.9-4-1.4c-8.4-2.8-12.9-11.9-10.1-20.2s11.9-12.9 20.2-10.1c1.7 .6 3.3 1.1 4.9 1.6l0 0 0 0c9.1 3.1 15.6 5.3 22.6 5.5c5.3 .2 10-1 12.8-2.8c1.2-.8 1.8-1.5 2.1-2c.2-.4 .6-1.2 .6-2.7l0-.2c0-.7 0-1.4-2.7-3.1c-3.8-2.4-9.6-4.3-18-6.9l-1.2-.4c-7.2-2.2-16.7-5-24.3-9.6c-9-5.4-17.7-14.7-17.7-29.4c-.1-15.2 8.6-25.7 18.5-31.6c9.4-5.5 20.5-7.5 29.7-7.4c10 .2 19.7 2.3 27.9 4.4c8.5 2.3 13.6 11 11.3 19.6s-11 13.6-19.6 11.3c-7.3-1.9-14.1-3.3-20.1-3.4c-4.9-.1-9.8 1.1-12.9 2.9c-1.4 .8-2.1 1.6-2.4 2c-.2 .3-.4 .8-.4 1.9zm-272 0c0 .2 0 .4 0 .4c.1 .1 .6 .8 2.2 1.7c3.9 2.3 9.6 4.1 18.3 6.8l.6 .2c7.4 2.2 17.3 5.2 25.2 10.2c9.1 5.7 17.4 15.2 17.6 29.9c.2 15-7.6 26-17.8 32.3c-9.5 5.9-20.9 7.9-30.7 7.6c-12.3-.4-24.2-4.5-33.2-7.6l0 0 0 0c-1.3-.4-2.5-.8-3.6-1.2c-8.4-2.8-12.9-11.9-10.1-20.2s11.9-12.9 20.2-10.1c1.4 .5 2.8 .9 4.1 1.4l0 0 0 0c9.5 3.2 16.5 5.6 23.7 5.8c5.3 .2 10-1 12.8-2.8c1.2-.8 1.8-1.5 2.1-2c.2-.4 .6-1.2 .6-2.7l0-.2c0-.7 0-1.4-2.7-3.1c-3.8-2.4-9.6-4.3-18-6.9l-1.2-.4 0 0c-7.2-2.2-16.7-5-24.3-9.6C80.8 239 72.1 229.7 72 215c-.1-15.2 8.6-25.7 18.5-31.6c9.4-5.5 20.5-7.5 29.7-7.4c9.5 .1 22.2 2.1 31.1 4.4c8.5 2.3 13.6 11 11.3 19.6s-11 13.6-19.6 11.3c-6.6-1.8-16.8-3.3-23.3-3.4c-4.9-.1-9.8 1.1-12.9 2.9c-1.4 .8-2.1 1.6-2.4 2c-.2 .3-.4 .8-.4 1.9z',
                'vw' => 512,
                'vh' => 512
            ],
            'comments' => [
                'p' => 'M26 12q0 4.13-3.81 7.06T12.99 22q-3.74 0-6.93-1.56Q3.37 22 .5 22q-.13 0-.25-.1t-.19-.21q-.12-.32.06-.57 1.32-1.25 2.25-3.37Q0 15.19 0 12q0-4.13 3.8-7.06T13 2t9.19 2.94T26 12zm7.63 13.75q.93 2.13 2.24 3.38.2.25.07.56t-.44.31q-2.88 0-5.56-1.56Q26.74 30 23 30q-4 0-7.25-1.69T11 23.88q1.12.12 2 .12 6.19 0 10.6-3.53T28 12q0-.69-.06-1.25 3.56 1.13 5.8 3.66T36 20q0 3.19-2.38 5.75z',
                'w' => 1.125,
                'vw' => 36
            ],
            'comments-dollar' => [
                'p' => 'M26 12q0 4.13-3.81 7.06T12.99 22q-3.74 0-6.93-1.56Q3.37 22 .5 22q-.5 0-.5-.5 0-.19.12-.37 1.38-1.44 2.25-3.38Q0 15.2 0 12q0-4.12 3.8-7.06T13 2t9.19 2.94T26 12zm-14 6q0 .5.5.5h1q.5 0 .5-.5v-1q1.13-.06 1.9-.88t.79-1.93q0-.5-.16-.94t-.4-.78-.63-.63-.81-.4l-2.82-.82q-.56-.18-.56-.8 0-.2.1-.38t.28-.32.37-.12h1.75q.44 0 .81.25.38.19.63-.07L16 8.5q.44-.44-.06-.75-.88-.7-1.94-.75V6q0-.5-.5-.5h-1q-.5 0-.5.5v1q-1.13.06-1.9.87t-.8 1.94q0 .94.56 1.69t1.44 1.06l2.82.81q.56.2.56.82 0 .31-.22.56t-.53.25h-1.75q-.44 0-.81-.25-.38-.19-.63.06l-.75.7q-.44.43.06.74.88.69 1.94.75v1zm21.63 7.75q.87 1.94 2.24 3.38.2.25.07.56t-.44.31q-2.88 0-5.56-1.56Q26.74 30 23 30q-4 0-7.25-1.69T11 23.88q1.12.12 2 .12 6.19 0 10.6-3.53T28 12q0-.63-.06-1.25 3.56 1.13 5.8 3.63T36 20q0 3.19-2.38 5.75z',
                'w' => 1.125,
                'vw' => 36
            ],
            'compact-disc' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zM5.5 16q0-4.13 2.94-7.06T15.5 6V4q-4.94 0-8.47 3.53T3.5 16h2zm10 6q2.5 0 4.25-1.75T21.5 16t-1.75-4.25T15.5 10t-4.25 1.75T9.5 16t1.75 4.25T15.5 22zm0-8q.81 0 1.4.6t.6 1.4-.6 1.4-1.4.6-1.4-.6-.6-1.4.6-1.4 1.4-.6z',
            'compass' => 'M14.1 14.6q.59-.6 1.4-.6t1.4.6.6 1.4-.6 1.4-1.4.6-1.4-.6-.6-1.4.6-1.4zM15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm7.88 9.25q.25-.5.03-1t-.66-.66-1 .03l-9 4.13q-.69.31-1 1l-4.13 9q-.37.75.25 1.38t1.38.25l9-4.13q.69-.31 1-1z',
            'compass-drafting' => [
                'p' => 'M352 96c0 14.3-3.1 27.9-8.8 40.2L396 227.4c-23.7 25.3-54.2 44.1-88.5 53.6L256 192h0 0l-68 117.5c21.5 6.8 44.3 10.5 68.1 10.5c70.7 0 133.8-32.7 174.9-84c11.1-13.8 31.2-16 45-5s16 31.2 5 45C428.1 341.8 347 384 256 384c-35.4 0-69.4-6.4-100.7-18.1L98.7 463.7C94 471.8 87 478.4 78.6 482.6L23.2 510.3c-5 2.5-10.9 2.2-15.6-.7S0 501.5 0 496V440.6c0-8.4 2.2-16.7 6.5-24.1l60-103.7C53.7 301.6 41.8 289.3 31.2 276c-11.1-13.8-8.8-33.9 5-45s33.9-8.8 45 5c5.7 7.1 11.8 13.8 18.2 20.1l69.4-119.9c-5.6-12.2-8.8-25.8-8.8-40.2c0-53 43-96 96-96s96 43 96 96zm21 297.9c32.6-12.8 62.5-30.8 88.9-52.9l43.7 75.5c4.2 7.3 6.5 15.6 6.5 24.1V496c0 5.5-2.9 10.7-7.6 13.6s-10.6 3.2-15.6 .7l-55.4-27.7c-8.4-4.2-15.4-10.8-20.1-18.9L373 393.9zM256 128a32 32 0 1 0 0-64 32 32 0 1 0 0 64z',
                'vw' => 512,
                'vh' => 512
            ],
            'compress' => [
                'p' => 'M27.25 12H19.5q-.63 0-1.06-.44T18 10.5V2.75q0-.75.75-.75h2.5q.75 0 .75.75V8h5.25q.75 0 .75.75v2.5q0 .75-.75.75zM10 10.5q0 .63-.44 1.06T8.5 12H.75Q0 12 0 11.25v-2.5Q0 8 .75 8H6V2.75Q6 2 6.75 2h2.5q.75 0 .75.75v7.75zm0 18.75q0 .75-.75.75h-2.5Q6 30 6 29.25V24H.75Q0 24 0 23.25v-2.5Q0 20 .75 20H8.5q.63 0 1.06.44T10 21.5v7.75zm12 0q0 .75-.75.75h-2.5q-.75 0-.75-.75V21.5q0-.63.44-1.06T19.5 20h7.75q.75 0 .75.75v2.5q0 .75-.75.75H22v5.25z',
                'w' => .875,
                'vw' => 28
            ],
            'compress-alt' => [
                'p' => 'm.3 26.7 6.2-6.2-2.06-1.94A1.5 1.5 0 0 1 5.5 16h7c.83 0 1.5.67 1.5 1.5v7c0 1.34-1.61 2-2.56 1.06L9.5 23.5l-6.2 6.2a1 1 0 0 1-1.42 0L.29 28.13a1 1 0 0 1 0-1.41ZM27.7 5.3l-6.2 6.2 2.06 1.94A1.5 1.5 0 0 1 22.5 16h-7a1.5 1.5 0 0 1-1.5-1.5v-7c0-1.34 1.61-2 2.56-1.06L18.5 8.5l6.2-6.2a1 1 0 0 1 1.42 0l1.59 1.58a1 1 0 0 1 0 1.41Zm0 0',
                'vw' => 28
            ],
            'compress-arrows-alt' => 'M12.5 18q.63 0 1.06.44T14 19.5v7q0 1-.94 1.38t-1.62-.32L9.5 25.5l-6.19 6.19Q3 32 2.6 32t-.72-.31L.31 30.12q-.31-.3-.31-.71t.31-.72l6.19-6.2-2.06-1.93q-.7-.69-.32-1.62T5.5 18h7zm7-4q-.63 0-1.06-.44T18 12.5v-7q0-1 .94-1.38t1.62.32L22.5 6.5 28.69.31Q29 0 29.4 0t.72.31l1.56 1.57q.31.3.31.71t-.31.72l-6.19 6.2 2.06 1.93q.7.69.32 1.62T26.5 14h-7zm6 8.5 6.19 6.19q.31.31.31.72t-.31.72l-1.57 1.56q-.68.75-1.43 0l-6.2-6.19-1.93 2.06q-.69.7-1.62.32T18 26.5v-7q0-.62.44-1.06T19.5 18h7q1 0 1.37.94t-.3 1.62zM11.44 4.44q.69-.69 1.62-.32T14 5.5v7q0 .63-.44 1.06T12.5 14h-7q-1 0-1.38-.94t.32-1.62L6.5 9.5.31 3.31Q0 3 0 2.6t.31-.72L1.88.31q.3-.31.71-.31t.72.31l6.2 6.19z',
            'computer' => [
                'p' => 'M384 96V320H64L64 96H384zM64 32C28.7 32 0 60.7 0 96V320c0 35.3 28.7 64 64 64H181.3l-10.7 32H96c-17.7 0-32 14.3-32 32s14.3 32 32 32H352c17.7 0 32-14.3 32-32s-14.3-32-32-32H277.3l-10.7-32H384c35.3 0 64-28.7 64-64V96c0-35.3-28.7-64-64-64H64zm464 0c-26.5 0-48 21.5-48 48V432c0 26.5 21.5 48 48 48h64c26.5 0 48-21.5 48-48V80c0-26.5-21.5-48-48-48H528zm16 64h32c8.8 0 16 7.2 16 16s-7.2 16-16 16H544c-8.8 0-16-7.2-16-16s7.2-16 16-16zm-16 80c0-8.8 7.2-16 16-16h32c8.8 0 16 7.2 16 16s-7.2 16-16 16H544c-8.8 0-16-7.2-16-16zm32 160a32 32 0 1 1 0 64 32 32 0 1 1 0-64z',
                'vw' => 640,
                'vh' => 512
            ],
            'computer-mouse' => [
                'p' => 'M0 192H176V0H160C71.6 0 0 71.6 0 160v32zm0 32V352c0 88.4 71.6 160 160 160h64c88.4 0 160-71.6 160-160V224H192 0zm384-32V160C384 71.6 312.4 0 224 0H208V192H384z',
                'vw' => 384,
                'vh' => 512
            ],
            'concierge-bell' => 'M18 8.19q5.13.69 8.56 4.62t3.44 9.2H2q0-5.26 3.44-9.2T14 8.2V7h-1q-.44 0-.72-.28T12 6V5q0-.44.28-.72T13 4h6q.44 0 .72.28T20 5v1q0 .44-.28.72T19 7h-1v1.19zM31 24q.44 0 .72.28T32 25v2q0 .44-.28.72T31 28H1q-.44 0-.72-.28T0 27v-2q0-.44.28-.72T1 24h30z',
            'cookie' => 'M31.88 15.94q.43 2.62-.75 5.06l-2.25 4.31q-.32.63-.91 1.35t-1.31 1.31-1.35.9L21 31.14q-2.44 1.18-5.06.75l-4.82-.75q-1.06-.2-2.4-.85t-2.1-1.47L3.2 25.38q-.82-.75-1.5-2.1t-.82-2.47l-.75-4.75Q0 15.44 0 14.75 0 12.81.87 11l2.25-4.31q.5-.94 1.53-2t2.03-1.57L11 .87Q13.44-.3 16.06.12l4.81.75q2.63.38 4.5 2.32l3.44 3.43q.81.75 1.5 2.1t.81 2.47zM11 23q.81 0 1.4-.6T13 21t-.6-1.4T11 19t-1.4.6T9 21t.6 1.4 1.4.6zm2-10q.81 0 1.4-.6T15 11t-.6-1.4T13 9t-1.4.6T11 11t.6 1.4 1.4.6zm10 8q.81 0 1.4-.6T25 19t-.6-1.4T23 17t-1.4.6T21 19t.6 1.4 1.4.6z',
            'cookie-bite' => 'M31.94 16q.37 2.63-.82 5l-2.18 4.31q-.38.63-.97 1.35t-1.31 1.31-1.35.9L21 31.14q-2.44 1.18-5.06.75l-4.82-.75q-1.06-.2-2.4-.85t-2.1-1.47l-3.44-3.44q-.8-.75-1.5-2.09t-.8-2.47l-.76-4.75Q0 15.44 0 14.75 0 12.8.87 11l2.25-4.31q.5-.94 1.53-2t2.03-1.57L11 .87q2.37-1.19 5-.75.06 3.25 2.37 5.57T24 7.99q0 3.32 2.3 5.63T31.94 16zM11 23q.81 0 1.4-.6T13 21t-.6-1.4T11 19t-1.4.6T9 21t.6 1.4 1.4.6zm2-10q.81 0 1.4-.6T15 11t-.6-1.4T13 9t-1.4.6T11 11t.6 1.4 1.4.6zm10 8q.81 0 1.4-.6T25 19t-.6-1.4T23 17t-1.4.6T21 19t.6 1.4 1.4.6z',
            'copy' => [
                'p' => 'M20 28v2.5q0 .63-.44 1.06T18.5 32h-17q-.63 0-1.06-.44T0 30.5v-23q0-.63.44-1.06T1.5 6H6v18.5q0 1.44 1.03 2.47T9.5 28H20zm0-21.5q0 .63.44 1.06T21.5 8H28v16.5q0 .63-.44 1.06T26.5 26h-17q-.63 0-1.06-.44T8 24.5v-23q0-.62.44-1.06T9.5 0H20v6.5zm7.56-1.94Q28 5 28 5.63V6h-6V0h.38q.62 0 1.06.44z',
                'w' => .875,
                'vw' => 28
            ],
            'copyright' => 'M16 .5q6.44 0 10.97 4.53T31.5 16t-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5zm7.31 21.69q.2-.25.2-.5t-.13-.44l-1.57-2.13q-.18-.25-.56-.28t-.56.22l-.22.22-.56.44-.85.53-1.12.44-1.38.19q-2.19 0-3.4-1.5t-1.22-3.5q0-2.07 1.25-3.4t3.37-1.35q2.25 0 3.88 1.5.25.25.59.18t.53-.3l1.38-2.2q.12-.18.12-.37 0-.31-.19-.56l-.15-.16-.44-.34-.69-.47-.9-.5-1.2-.47-1.43-.32L16.43 7Q12.56 7 10 9.56t-2.57 6.38q0 2.56 1.13 4.62t3.22 3.25 4.65 1.2q2 0 3.75-.76T22.46 23t.85-.81z',
            'couch' => [
                'p' => 'M10 14q0-1.63-1.19-2.81T6 9.99H4q0-2.5 1.75-4.24T10 4h20q2.5 0 4.25 1.75T36 10h-2q-1.63 0-2.81 1.19T29.99 14v4H10v-4zm26-2q1.63 0 2.81 1.19t1.2 2.81q0 2.25-2 3.44V27q0 .44-.29.72T37 28h-4q-.44 0-.72-.28T32 27v-1H8v1q0 .44-.28.72T7 28H3q-.44 0-.72-.28T2 27v-7.56Q0 18.25 0 16q0-1.62 1.19-2.81T4 12h2q.81 0 1.4.6T8 14v6h24v-6q0-.81.6-1.4T34 12h2z',
                'w' => 1.25,
                'vw' => 40
            ],
            'cow' => [
                'p' => 'M96 224v32V416c0 17.7 14.3 32 32 32h32c17.7 0 32-14.3 32-32V327.8c9.9 6.6 20.6 12 32 16.1V368c0 8.8 7.2 16 16 16s16-7.2 16-16V351.1c5.3 .6 10.6 .9 16 .9s10.7-.3 16-.9V368c0 8.8 7.2 16 16 16s16-7.2 16-16V343.8c11.4-4 22.1-9.4 32-16.1V416c0 17.7 14.3 32 32 32h32c17.7 0 32-14.3 32-32V256l32 32v49.5c0 9.5 2.8 18.7 8.1 26.6L530 427c8.8 13.1 23.5 21 39.3 21c22.5 0 41.9-15.9 46.3-38l20.3-101.6c2.6-13-.3-26.5-8-37.3l-3.9-5.5V184c0-13.3-10.7-24-24-24s-24 10.7-24 24v14.4l-52.9-74.1C496 86.5 452.4 64 405.9 64H272 256 192 144C77.7 64 24 117.7 24 184v54C9.4 249.8 0 267.8 0 288v17.6c0 8 6.4 14.4 14.4 14.4C46.2 320 72 294.2 72 262.4V256 224 184c0-24.3 12.1-45.8 30.5-58.9C98.3 135.9 96 147.7 96 160v64zM560 336a16 16 0 1 1 32 0 16 16 0 1 1 -32 0zM166.6 166.6c-4.2-4.2-6.6-10-6.6-16c0-12.5 10.1-22.6 22.6-22.6H361.4c12.5 0 22.6 10.1 22.6 22.6c0 6-2.4 11.8-6.6 16l-23.4 23.4C332.2 211.8 302.7 224 272 224s-60.2-12.2-81.9-33.9l-23.4-23.4z',
                'vw' => 640,
                'vh' => 512
            ],
            'credit-card' => [
                'p' => 'M0 27V16h36v11q0 1.25-.88 2.13T33 30H3q-1.25 0-2.13-.88T0 27zm12-4.25v2.5q0 .75.75.75h8.5q.75 0 .75-.75v-2.5q0-.75-.75-.75h-8.5q-.75 0-.75.75zm-8 0v2.5q0 .75.75.75h4.5q.75 0 .75-.75v-2.5q0-.75-.75-.75h-4.5Q4 22 4 22.75zM36 5v3H0V5q0-1.25.88-2.13T3 2h30q1.25 0 2.13.88T36 5z',
                'w' => 1.125,
                'vw' => 36
            ],
            'crop' => 'M30.5 22q.63 0 1.06.44T32 23.5v3q0 .63-.44 1.06T30.5 28H28v2.5q0 .63-.44 1.06T26.5 32h-3q-.63 0-1.06-.44T22 30.5V12.81l-9.19 9.2H20v6H5.5q-.63 0-1.06-.45T4 26.5V10H1.5q-.63 0-1.07-.44T0 8.5v-3q0-.63.43-1.06T1.5 4H4V1.5q0-.63.43-1.06T5.5 0h3q.62 0 1.06.44T10 1.5v17.69l9.19-9.2h-7.2V4h13.2L28.87.32q.7-.75 1.44 0L31.7 1.7q.75.75 0 1.43L28 6.82V22h2.5z',
            'crop-alt' => 'M30.5 22q.63 0 1.06.44T32 23.5v3q0 .63-.44 1.06T30.5 28H28v2.5q0 .63-.44 1.06T26.5 32h-3q-.63 0-1.06-.44T22 30.5V10H12V4h14q.81 0 1.4.6T28 6v16h2.5zM10 1.5V22h10v6H6q-.81 0-1.4-.6T4 26V10H1.5q-.63 0-1.06-.44T0 8.5v-3q0-.63.44-1.06T1.5 4H4V1.5q0-.63.44-1.06T5.5 0h3q.63 0 1.06.44T10 1.5z',
            'crop-simple' => [
                'p' => 'M128 32c0-17.7-14.3-32-32-32S64 14.3 64 32V64H32C14.3 64 0 78.3 0 96s14.3 32 32 32H64V384c0 35.3 28.7 64 64 64H352V384H128V32zM384 480c0 17.7 14.3 32 32 32s32-14.3 32-32V448h32c17.7 0 32-14.3 32-32s-14.3-32-32-32H448l0-256c0-35.3-28.7-64-64-64L160 64v64l224 0 0 352z',
                'vw' => 512,
                'vh' => 512
            ],
            'cross' => [
                'p' => 'M22 8q.81 0 1.4.6T24 10v4q0 .81-.6 1.4T22 16h-6v14q0 .81-.6 1.4T14 32h-4q-.81 0-1.4-.6T8 30V16H2q-.81 0-1.4-.6T0 14v-4q0-.81.6-1.4T2 8h6V2q0-.81.6-1.4T10 0h4q.81 0 1.4.6T16 2v6h6z',
                'w' => .75,
                'vw' => 24
            ],
            'crosshairs' => 'M31.25 14q.75 0 .75.75v2.5q0 .75-.75.75h-1.88q-.43 2.88-2.03 5.31t-4.03 4.03T18 29.38v1.87q0 .75-.75.75h-2.5q-.75 0-.75-.75v-1.88q-4.38-.68-7.53-3.84T2.63 18H.75Q0 18 0 17.25v-2.5Q0 14 .75 14h1.88q.68-4.38 3.84-7.53T14 2.63V.75q0-.75.75-.75h2.5Q18 0 18 .75v1.88q4.38.68 7.53 3.84T29.37 14h1.88zM18 25.31q2.75-.62 4.72-2.6t2.6-4.71h-2.57q-.75 0-.75-.75v-2.5q0-.75.75-.75h2.56q-.62-2.75-2.6-4.72T18 6.68v2.57q0 .75-.75.75h-2.5Q14 10 14 9.25V6.69q-2.75.62-4.72 2.6T6.68 14h2.57q.75 0 .75.75v2.5q0 .75-.75.75H6.69q.62 2.75 2.6 4.72t4.71 2.6v-2.57q0-.75.75-.75h2.5q.75 0 .75.75v2.56zM18 16q0 .81-.6 1.4T16 18t-1.4-.6T14 16t.6-1.4T16 14t1.4.6.6 1.4z',
            'crow' => [
                'p' => 'M34 2q2.5 0 4.25 1.16T40 6l-6 1v5q0 4.06-2.4 7.22t-6.23 4.28l2.57 7q.25.69-.44.94l-1.38.5q-.75.25-1-.44L22.38 24h-2.8l2.37 6.5q.25.69-.44.94l-1.38.5q-.75.25-1-.44L16.37 24h-6.06l-7.56 3.81q-.94.44-1.85-.1T0 26.13q0-.93.75-1.5L24 6.32V5q0-2.06 1.47-3.53T29 0q2.44 0 4 2h1zm-5 4.5q.63 0 1.06-.44T30.5 5t-.44-1.06T29 3.5t-1.06.44T27.5 5t.44 1.06T29 6.5z',
                'w' => 1.25,
                'vw' => 40
            ],
            'crown' => [
                'p' => 'M33 28q.44 0 .72.28T34 29v2q0 .44-.28.72T33 32H7q-.44 0-.72-.28T6 31v-2q0-.44.28-.72T7 28h26zm4-20q1.25 0 2.13.88T40 11t-.88 2.13T37 14q-.19 0-.5-.06L32 26H8L3.5 13.94Q3.19 14 3 14q-1.25 0-2.13-.88T0 11t.88-2.13T3 8t2.13.88T6 11q0 .63-.25 1.25l4.5 2.69q.69.44 1.53.22t1.22-.9l5.13-8.95Q17 4.38 17 3q0-1.25.88-2.13T20 0t2.13.88T23 3q0 1.38-1.13 2.31L27 14.25q.38.69 1.19.9t1.56-.21l4.5-2.69Q34 11.62 34 11q0-1.25.88-2.13T37 8z',
                'w' => 1.25,
                'vw' => 40
            ],
            'crutch' => 'M31.69 11.63q.31.24.31.68t-.31.7l-1.38 1.43q-.31.31-.72.31t-.72-.31L17.57 3.13q-.32-.32-.32-.72t.31-.72L19 .3q.25-.31.69-.31t.69.31zM20.5 15.75l3.44-3.44 2.81 2.81L19.88 22q-1.2 1.19-2.88 1.63L9.5 25.3l-6.37 6.4q-.33.3-.73.3t-.72-.31L.3 30.3Q0 30 0 29.6t.31-.71L6.7 22.5 8.38 15q.43-1.69 1.62-2.88l6.88-6.87 2.8 2.81-3.43 3.44zm-3.44 3.44.63-.63-4.25-4.25-.63.63q-.37.37-.5.94l-1.19 5 5-1.2q.57-.12.94-.5z',
            'cruzeiro-sign' => [
                'p' => 'M96 256c0-88.4 71.6-160 160-160c41 0 78.3 15.4 106.7 40.7c13.2 11.8 33.4 10.7 45.2-2.5s10.7-33.4-2.5-45.2c-39.6-35.5-92-57-149.3-57C132.3 32 32 132.3 32 256s100.3 224 224 224c57.4 0 109.7-21.6 149.3-57c13.2-11.8 14.3-32 2.5-45.2s-32-14.3-45.2-2.5C334.3 400.6 297 416 256 416V320v-8.7c0-12.8 10.4-23.3 23.3-23.3c4.6 0 9.1 1.4 12.9 3.9l10.1 6.7c14.7 9.8 34.6 5.8 44.4-8.9s5.8-34.6-8.9-44.4l-10.1-6.7c-14.3-9.6-31.2-14.7-48.4-14.7c-12.4 0-24.2 2.6-34.9 7.3c-5.5-4.5-12.6-7.3-20.3-7.3c-17.7 0-32 14.3-32 32v55.3V320v82.7C135.5 378 96 321.6 96 256z',
                'vw' => 448,
                'vh' => 512
            ],
            'cube' => 'M14.94.38Q16 0 17.06.38l13 4.87q.88.38 1.4 1.13T32 8.05v14.07q0 1.87-1.69 2.68l-13 6.5q-1.31.7-2.69 0l-13-6.5Q0 24.01 0 22.13V8.06q0-.93.53-1.68t1.4-1.13zM16 4.25 4 8.75v.13l12 4.87 12-4.88v-.12zm2 22.25 10-5v-8.38l-10 4.13v9.25z',
            'cubes' => 'M30.56 15.63q1.44.56 1.44 2.12v6.88q0 1.37-1.25 2l-6.25 3.12q-1 .5-2 0L16 26.5l-6.5 3.25q-1 .5-2 0l-6.25-3.13Q0 26 0 24.63v-6.87q0-1.56 1.44-2.13l6.06-2.24V6.62q0-1.56 1.44-2.12l6.25-2.38q.81-.3 1.62 0l6.25 2.38q1.44.56 1.44 2.13v6.75zm-8.19-2.2V8.89l-5.3 2.24v4.32zM9.63 6.5v.06L16 9.13l6.38-2.57V6.5L16 4.12zm5.26 18.19V20l-5.32 2.44v4.94zm0-7L8.5 15.25l-6.38 2.44L8.5 20.3zm15 7V20l-5.32 2.44v4.94zm0-7-6.38-2.44-6.38 2.44 6.38 2.62z',
            'cubes-stacked' => [
                'p' => 'M192 64v64c0 17.7 14.3 32 32 32h64c17.7 0 32-14.3 32-32V64c0-17.7-14.3-32-32-32H224c-17.7 0-32 14.3-32 32zM82.7 207c-15.3 8.8-20.5 28.4-11.7 43.7l32 55.4c8.8 15.3 28.4 20.5 43.7 11.7l55.4-32c15.3-8.8 20.5-28.4 11.7-43.7l-32-55.4c-8.8-15.3-28.4-20.5-43.7-11.7L82.7 207zM288 192c-17.7 0-32 14.3-32 32v64c0 17.7 14.3 32 32 32h64c17.7 0 32-14.3 32-32V224c0-17.7-14.3-32-32-32H288zm64 160c-17.7 0-32 14.3-32 32v64c0 17.7 14.3 32 32 32h64c17.7 0 32-14.3 32-32V384c0-17.7-14.3-32-32-32H352zM160 384v64c0 17.7 14.3 32 32 32h64c17.7 0 32-14.3 32-32V384c0-17.7-14.3-32-32-32H192c-17.7 0-32 14.3-32 32zM32 352c-17.7 0-32 14.3-32 32v64c0 17.7 14.3 32 32 32H96c17.7 0 32-14.3 32-32V384c0-17.7-14.3-32-32-32H32z',
                'vw' => 448,
                'vh' => 512
            ],
            'cut' => [
                'p' => 'm17.38 16 10.37 10.38q.56.56 0 1.06Q26.25 29 24.06 29t-3.69-1.56l-7.25-7.19-1.56 1.56q.44 1.07.44 2.2 0 2.5-1.75 4.24T6 30t-4.25-1.75T0 24t1.75-4.25T6 18q.44 0 .81.06L8.88 16 6.8 13.94Q6.44 14 6 14q-2.5 0-4.25-1.75T0 8t1.75-4.25T6 2t4.25 1.75T12 8q0 1.13-.44 2.19l1.56 1.56 7.25-7.19Q21.87 3 24.07 3t3.68 1.56q.56.5 0 1.07zM6 10q.81 0 1.4-.6T8 8t-.6-1.4T6 6t-1.4.6T4 8t.6 1.4T6 10zm0 16q.81 0 1.4-.6T8 24t-.6-1.4T6 22t-1.4.6T4 24t.6 1.4T6 26z',
                'w' => .873,
                'vw' => 28
            ],
            'cutlery' => 'utensils',
            'd' => [
                'p' => 'M0 96C0 60.7 28.7 32 64 32h96c123.7 0 224 100.3 224 224s-100.3 224-224 224H64c-35.3 0-64-28.7-64-64V96zm160 0H64V416h96c88.4 0 160-71.6 160-160s-71.6-160-160-160z',
                'vw' => 384,
                'vh' => 512
            ],
            'database' => [
                'p' => 'M28 4.56v2.88q0 1.87-4.1 3.22T14 12q-3.81 0-7.03-.62t-5.1-1.66T0 7.44V4.56Q0 2.7 4.1 1.35T14 0q3.81 0 7.03.63t5.1 1.65T28 4.56zM28 11v6.44q0 1.87-4.1 3.22T14 22t-9.9-1.34T0 17.44V11q4.44 3.06 14 3.06T28 11zm0 10v6.44q0 1.87-4.1 3.22T14 32t-9.9-1.34T0 27.44V21q4.44 3.06 14 3.06T28 21z',
                'w' => .875,
                'vw' => 28
            ],
            'deaf' => 'M13.5 16.25q0 .75-.5 1.25t-1.25.5-1.25-.5-.5-1.25q0-2.06 1.47-3.53T15 11.25t3.53 1.47T20 16.25q0 .75-.5 1.25t-1.25.5-1.25-.5-.5-1.25q0-.63-.44-1.06T15 14.75t-1.06.44-.44 1.06zm1.5-11q4.56 0 7.78 3.22T26 16.25q0 1.69-.47 3.03t-1.12 2.16-1.32 1.43-1.12 1.22-.47 1.16q0 2.81-1.97 4.78T14.75 32q-.75 0-1.25-.5t-.5-1.25.5-1.25 1.25-.5q1.37 0 2.31-.94t.94-2.31q0-1.56.72-2.72t1.56-1.75T21.81 19t.7-2.75q0-3.13-2.2-5.31T15 8.74t-5.31 2.2-2.2 5.3q0 .76-.5 1.26t-1.24.5-1.25-.5-.5-1.25q0-4.56 3.22-7.78T15 5.25zM31.75 2q.56.5 0 1.06L26.31 8.5q-.5.5-1.06 0L23.5 6.75q-.5-.56 0-1.06L28.94.25q.56-.56 1.06 0zM10.56 19.69l1.75 1.75q.57.56 0 1.06l-9.25 9.25q-.25.25-.56.25t-.5-.25L.25 30q-.56-.5 0-1.06l9.25-9.25q.5-.57 1.06 0z',
            'delete-left' => [
                'p' => 'M576 128c0-35.3-28.7-64-64-64H205.3c-17 0-33.3 6.7-45.3 18.7L9.4 233.4c-6 6-9.4 14.1-9.4 22.6s3.4 16.6 9.4 22.6L160 429.3c12 12 28.3 18.7 45.3 18.7H512c35.3 0 64-28.7 64-64V128zM271 175c9.4-9.4 24.6-9.4 33.9 0l47 47 47-47c9.4-9.4 24.6-9.4 33.9 0s9.4 24.6 0 33.9l-47 47 47 47c9.4 9.4 9.4 24.6 0 33.9s-24.6 9.4-33.9 0l-47-47-47 47c-9.4 9.4-24.6 9.4-33.9 0s-9.4-24.6 0-33.9l47-47-47-47c-9.4-9.4-9.4-24.6 0-33.9z',
                'vw' => 576,
                'vh' => 512
            ],
            'democrat' => [
                'p' => 'M39.81 16.06q.57.82-.25 1.38l-1.68 1.12q-.82.57-1.38-.31-.19-.25-.44-.66t-.37-.56-.32-.44-.28-.37-.22-.28-.25-.28-.28-.22-.34-.32V20H12l-3.38-6.75-2.37 2.19Q5.62 16 4.8 16H2.87q-1.18 0-1.75-1.13L.2 13.13Q-.31 12 .37 11l4.7-6.19q.05-.06.37-.31-.5-.31-.7-.5Q4 3.25 3.9 2.16T4.37.25q.32-.56.75-.13l2.7 2.7L10.43.18q.18-.2.5-.16t.43.28q.75 1.06.63 2.38t-1.06 2.25L16 10h14.75q2.37 0 4.47 1.13t3.4 3.12zM18.5 15.2q.13-.13.1-.31t-.23-.2l-1.43-.18-.63-1.31q-.12-.2-.31-.2t-.25.2l-.69 1.3-1.37.2q-.2 0-.25.18t.06.32l1 1-.25 1.44q0 .18.16.3t.34 0l1.25-.62 1.25.63q.19.12.34 0t.16-.31l-.25-1.44zm7 0q.13-.13.1-.31t-.23-.2l-1.43-.18-.63-1.31q-.12-.2-.31-.2t-.25.2l-.69 1.3-1.37.2q-.2 0-.25.18t.06.32l1 1-.25 1.44q0 .18.16.3t.34 0l1.25-.62 1.25.63q.19.12.34 0t.16-.31l-.25-1.44zm7 0q.13-.13.1-.31t-.23-.2l-1.43-.18-.63-1.31q-.12-.2-.31-.2t-.25.2l-.69 1.3-1.37.2q-.2 0-.25.18t.06.32l1 1-.25 1.44q0 .18.16.3t.34 0l1.25-.62 1.25.63q.19.12.34 0t.16-.31l-.25-1.44zM12 31v-9h22v9q0 .44-.28.72T33 32h-4q-.44 0-.72-.28T28 31v-5H18v5q0 .44-.28.72T17 32h-4q-.44 0-.72-.28T12 31z',
                'w' => 1.248,
                'vw' => 40
            ],
            'desktop' => [
                'p' => 'M33 0q1.25 0 2.13.88T36 3v20q0 1.25-.88 2.13T33 26H21l1 3h4.5q.63 0 1.06.44T28 30.5t-.44 1.06-1.06.44h-17q-.63 0-1.06-.44T8 30.5t.44-1.06T9.5 29H14l1-3H3q-1.25 0-2.13-.88T0 23V3Q0 1.75.88.87T3 0h30zm-1 22V4H4v18h28z',
                'w' => 1.125,
                'vw' => 36
            ],
            'dharmachakra' => 'M30.94 14.06q.44 0 .75.28t.31.72v1.88q0 .44-.31.72t-.75.28l-1.07-.06q-.5 3.68-2.75 6.62l.82.69q.31.31.34.75t-.28.69L26.63 28q-.25.25-.7.25t-.74-.31l-.69-.81q-2.94 2.25-6.62 2.75l.06 1.06q0 .44-.28.75t-.72.31h-1.88q-.43 0-.72-.31t-.28-.75l.07-1.06q-3.7-.5-6.63-2.75l-.69.8q-.31.32-.75.32T5.38 28L4 26.63q-.25-.25-.25-.7t.31-.74l.82-.69q-2.25-2.94-2.75-6.62l-1.07.06q-.44 0-.75-.28T0 16.94v-1.88q0-.44.31-.72t.75-.28l1.07.07q.5-3.7 2.75-6.63l-.82-.69q-.31-.31-.31-.75T4 5.38L5.38 4q.25-.25.68-.25t.75.31l.7.82q2.93-2.25 6.62-2.75l-.07-1.07q0-.44.28-.75t.72-.31h1.88q.44 0 .72.31t.28.75l-.06 1.07q3.68.5 6.62 2.75l.69-.82q.31-.31.75-.31t.69.25L28 5.37q.25.25.25.7t-.31.74l-.81.7q2.25 2.93 2.75 6.62zM17.62 6.2l-.25 4q.94.19 1.75.75l2.7-3.07q-1.88-1.3-4.2-1.68zm-3.25 0q-2.3.37-4.18 1.69l2.69 3.06q.8-.56 1.75-.75zm-6.5 4q-1.3 1.87-1.68 4.19l4 .25q.19-.94.75-1.75zM6.2 17.62q.37 2.32 1.69 4.2l3.06-2.7q-.56-.8-.75-1.75zm8.18 8.2.26-4q-.94-.2-1.76-.76l-2.68 3.07q1.87 1.3 4.18 1.68zM16 18q.81 0 1.4-.6T18 16t-.6-1.4T16 14t-1.4.6T14 16t.6 1.4 1.4.6zm1.63 7.81q2.3-.37 4.18-1.68l-2.69-3.07q-.8.57-1.75.75zm6.5-4q1.3-1.87 1.68-4.18l-4-.25q-.18.93-.75 1.75zm-2.32-7.18 4-.26q-.37-2.3-1.68-4.18l-3.07 2.69q.57.8.75 1.75z',
            'diagnoses' => [
                'p' => 'M31 16q-.44 0-.72-.28T30 15t.28-.72T31 14t.72.28.28.72-.28.72T31 16zm-11-5q-2.25 0-3.88-1.63T14.5 5.5t1.63-3.88T20 0t3.88 1.63T25.5 5.5t-1.63 3.88T20 11zM3.75 22.75l-1.13-1.69q-.43-.56-.28-1.44t.91-1.37l1.81-1q1.5 1.25 3.22.47T10 15q5.38-2 10-2 3.75 0 8.06 1.37-.3 1.57.7 2.66t2.46 1 2.35-1.47q1.74.81 3.18 1.69.82.5.94 1.37t-.31 1.44l-1.13 1.69q-.43.69-1.25.84t-1.43-.21Q30.43 21.5 27 20.3V26H13v-5.69q-3.5 1.19-6.56 3.07-.63.37-1.44.21t-1.25-.84zM23 21.5q-.63 0-1.06.44T21.5 23t.44 1.06 1.06.44 1.06-.44.44-1.06-.44-1.06T23 21.5zm-6-6q-.63 0-1.06.44T15.5 17t.44 1.06 1.06.44 1.06-.44.44-1.06-.44-1.06T17 15.5zM7 16q-.44 0-.72-.28T6 15t.28-.72T7 14t.72.28T8 15t-.28.72T7 16zm32 12q.44 0 .72.28T40 29v2q0 .44-.28.72T39 32H1q-.44 0-.72-.28T0 31v-2q0-.44.28-.72T1 28h38z',
                'w' => 1.25,
                'vw' => 40
            ],
            'diagram-next' => [
                'p' => 'M512 160c0 35.3-28.7 64-64 64H280v64h46.1c21.4 0 32.1 25.9 17 41L273 399c-9.4 9.4-24.6 9.4-33.9 0L169 329c-15.1-15.1-4.4-41 17-41H232V224H64c-35.3 0-64-28.7-64-64V96C0 60.7 28.7 32 64 32H448c35.3 0 64 28.7 64 64v64zM448 416V352H365.3l.4-.4c18.4-18.4 20.4-43.7 11-63.6l71.3 0c35.3 0 64 28.7 64 64v64c0 35.3-28.7 64-64 64L64 480c-35.3 0-64-28.7-64-64V352c0-35.3 28.7-64 64-64l71.3 0c-9.4 19.9-7.4 45.2 11 63.6l.4 .4H64v64H210.7l5.7 5.7c21.9 21.9 57.3 21.9 79.2 0l5.7-5.7H448z',
                'vw' => 512,
                'vh' => 512
            ],
            'diagram-predecessor' => [
                'p' => 'M448 416l0-64L64 352l0 64 384 0zm0 64L64 480c-35.3 0-64-28.7-64-64l0-64c0-35.3 28.7-64 64-64l384 0c35.3 0 64 28.7 64 64l0 64c0 35.3-28.7 64-64 64zM288 160c0 35.3-28.7 64-64 64L64 224c-35.3 0-64-28.7-64-64L0 96C0 60.7 28.7 32 64 32l144 0 16 0 144 0c44.2 0 80 35.8 80 80l0 16 38.1 0c21.4 0 32.1 25.9 17 41L433 239c-9.4 9.4-24.6 9.4-33.9 0L329 169c-15.1-15.1-4.4-41 17-41l38.1 0 0-16c0-8.8-7.2-16-16-16l-80 0 0 64z',
                'vw' => 512,
                'vh' => 512
            ],
            'diagram-project' => [
                'p' => 'M0 80C0 53.5 21.5 32 48 32h96c26.5 0 48 21.5 48 48V96H384V80c0-26.5 21.5-48 48-48h96c26.5 0 48 21.5 48 48v96c0 26.5-21.5 48-48 48H432c-26.5 0-48-21.5-48-48V160H192v16c0 1.7-.1 3.4-.3 5L272 288h96c26.5 0 48 21.5 48 48v96c0 26.5-21.5 48-48 48H272c-26.5 0-48-21.5-48-48V336c0-1.7 .1-3.4 .3-5L144 224H48c-26.5 0-48-21.5-48-48V80z',
                'vw' => 576,
                'vh' => 512
            ],
            'diagram-successor' => [
                'p' => 'M512 416l0-64c0-35.3-28.7-64-64-64L64 288c-35.3 0-64 28.7-64 64l0 64c0 35.3 28.7 64 64 64l384 0c35.3 0 64-28.7 64-64zM64 160l0-64 144 0 16 0 0 64L64 160zm224 0l0-64 80 0c8.8 0 16 7.2 16 16l0 16-38.1 0c-21.4 0-32.1 25.9-17 41L399 239c9.4 9.4 24.6 9.4 33.9 0L503 169c15.1-15.1 4.4-41-17-41L448 128l0-16c0-44.2-35.8-80-80-80L224 32l-16 0L64 32C28.7 32 0 60.7 0 96l0 64c0 35.3 28.7 64 64 64l160 0c35.3 0 64-28.7 64-64z',
                'vw' => 512,
                'vh' => 512
            ],
            'diamond' => [
                'p' => 'M284.3 11.7c-15.6-15.6-40.9-15.6-56.6 0l-216 216c-15.6 15.6-15.6 40.9 0 56.6l216 216c15.6 15.6 40.9 15.6 56.6 0l216-216c15.6-15.6 15.6-40.9 0-56.6l-216-216z',
                'vw' => 512,
                'vh' => 512
            ],
            'diamond-turn-right' => [
                'p' => 'M227.7 11.7c15.6-15.6 40.9-15.6 56.6 0l216 216c15.6 15.6 15.6 40.9 0 56.6l-216 216c-15.6 15.6-40.9 15.6-56.6 0l-216-216c-15.6-15.6-15.6-40.9 0-56.6l216-216zm87.6 137c-4.6-4.6-11.5-5.9-17.4-3.5s-9.9 8.3-9.9 14.8v56H224c-35.3 0-64 28.7-64 64v48c0 13.3 10.7 24 24 24s24-10.7 24-24V280c0-8.8 7.2-16 16-16h64v56c0 6.5 3.9 12.3 9.9 14.8s12.9 1.1 17.4-3.5l80-80c6.2-6.2 6.2-16.4 0-22.6l-80-80z',
                'vw' => 512,
                'vh' => 512
            ],
            'dice' => [
                'p' => 'M37 12q1.25 0 2.13.88T40 15v14q0 1.25-.88 2.13T37 32H23q-1.25 0-2.13-.88T20 29v-2.88l8.5-8.56q1.13-1.12 1.4-2.65T29.57 12H37zm-7 11.5q.63 0 1.06-.44T31.5 22t-.44-1.06T30 20.5t-1.06.44T28.5 22t.44 1.06 1.06.44zm-2.88-11.69q.88.94.88 2.2t-.88 2.18L16.2 27.13q-.94.87-2.2.87t-2.18-.87L.87 16.19Q0 15.25 0 14t.87-2.19L11.81.87Q12.75 0 14 0t2.19.88zM6 15.5q.63 0 1.06-.44T7.5 14t-.44-1.06T6 12.5t-1.06.44T4.5 14t.44 1.06T6 15.5zm8 8q.63 0 1.06-.44T15.5 22t-.44-1.06T14 20.5t-1.06.44T12.5 22t.44 1.06 1.06.44zm0-8q.63 0 1.06-.44T15.5 14t-.44-1.06T14 12.5t-1.06.44T12.5 14t.44 1.06 1.06.44zm0-8q.63 0 1.06-.44T15.5 6t-.44-1.06T14 4.5t-1.06.44T12.5 6t.44 1.06T14 7.5zm8 8q.63 0 1.06-.44T23.5 14t-.44-1.06T22 12.5t-1.06.44T20.5 14t.44 1.06 1.06.44z',
                'w' => 1.25,
                'vw' => 40
            ],
            'dice-d20' => [
                'p' => 'm6.69 13.44 6.75 11.87-13-1.37q-.25 0-.38-.25t0-.5zM.44 19.69q-.07.19-.25.12T0 19.56V9.44q0-.19.13-.25t.24 0l4.82 2.87zm.68 6.81q-.18-.12-.12-.31t.25-.2L14 27.38v4.13q0 .25-.22.4t-.47.04zM6.2 10.38 1.12 7.3Q1 7.25 1 7.12t.13-.25L10.55.75q.32-.19.6.1t.03.59zM15 11H8.19L14.13.5q.3-.5.87-.5t.88.5L21.8 11H15zm14.63-1.81q.12-.06.25 0t.12.25v10.12q0 .2-.19.25t-.25-.12l-4.75-7.63zm-5.82 1.19-5-8.94q-.12-.25-.06-.47t.28-.28.4.06l9.44 6.12q.13.13.13.25t-.13.2zM28.75 26q.19 0 .25.19t-.13.31L16.7 31.94q-.13.06-.25.06t-.22-.06-.16-.19-.06-.25v-4.12zm-5.44-12.56 6.63 9.75q.12.25 0 .5t-.38.25l-13 1.37zM15 13h6.25L15 24 8.75 13H15z',
                'vw' => 30,
            ],
            'dice-d6' => [
                'p' => 'M26.38 6.88q.25.12.25.43t-.25.44L14 15.2 1.62 7.75q-.25-.12-.25-.44t.25-.43L12 .56q2-1.25 4 0zm.87 2.8q.25-.18.5-.02t.25.47V22.5q0 2.38-2 3.56l-9.5 5.82q-.5.31-1 0t-.5-.94v-14zM0 10.14q0-.32.25-.47t.5.03L13 16.94v14q0 .62-.5.93t-1 0L2 26.07q-2-1.2-2-3.57V10.12z',
                'w' => .875,
                'vw' => 28
            ],
            'dice-five' => [
                'p' => 'M24 2q1.69 0 2.84 1.16T28 6v20q0 1.69-1.16 2.84T24 30H4q-1.69 0-2.84-1.16T0 26V6q0-1.69 1.16-2.84T4 2h20zM8 24q.81 0 1.4-.6T10 22t-.6-1.4T8 20t-1.4.6T6 22t.6 1.4T8 24zm0-12q.81 0 1.4-.6T10 10t-.6-1.4T8 8t-1.4.6T6 10t.6 1.4T8 12zm6 6q.81 0 1.4-.6T16 16t-.6-1.4T14 14t-1.4.6T12 16t.6 1.4 1.4.6zm6 6q.81 0 1.4-.6T22 22t-.6-1.4T20 20t-1.4.6T18 22t.6 1.4 1.4.6zm0-12q.81 0 1.4-.6T22 10t-.6-1.4T20 8t-1.4.6T18 10t.6 1.4 1.4.6z',
                'w' => .875,
                'vw' => 28
            ],
            'dice-four' => [
                'p' => 'M24 2q1.69 0 2.84 1.16T28 6v20q0 1.69-1.16 2.84T24 30H4q-1.69 0-2.84-1.16T0 26V6q0-1.69 1.16-2.84T4 2h20zM8 24q.81 0 1.4-.6T10 22t-.6-1.4T8 20t-1.4.6T6 22t.6 1.4T8 24zm0-12q.81 0 1.4-.6T10 10t-.6-1.4T8 8t-1.4.6T6 10t.6 1.4T8 12zm12 12q.81 0 1.4-.6T22 22t-.6-1.4T20 20t-1.4.6T18 22t.6 1.4 1.4.6zm0-12q.81 0 1.4-.6T22 10t-.6-1.4T20 8t-1.4.6T18 10t.6 1.4 1.4.6z',
                'w' => .875,
                'vw' => 28
            ],
            'dice-one' => [
                'p' => 'M24 2q1.69 0 2.84 1.16T28 6v20q0 1.69-1.16 2.84T24 30H4q-1.69 0-2.84-1.16T0 26V6q0-1.69 1.16-2.84T4 2h20zM14 18q.81 0 1.4-.6T16 16t-.6-1.4T14 14t-1.4.6T12 16t.6 1.4 1.4.6z',
                'w' => .875,
                'vw' => 28
            ],
            'dice-six' => [
                'p' => 'M24 2q1.69 0 2.84 1.16T28 6v20q0 1.69-1.16 2.84T24 30H4q-1.69 0-2.84-1.16T0 26V6q0-1.69 1.16-2.84T4 2h20zM8 24q.81 0 1.4-.6T10 22t-.6-1.4T8 20t-1.4.6T6 22t.6 1.4T8 24zm0-6q.81 0 1.4-.6T10 16t-.6-1.4T8 14t-1.4.6T6 16t.6 1.4T8 18zm0-6q.81 0 1.4-.6T10 10t-.6-1.4T8 8t-1.4.6T6 10t.6 1.4T8 12zm12 12q.81 0 1.4-.6T22 22t-.6-1.4T20 20t-1.4.6T18 22t.6 1.4 1.4.6zm0-6q.81 0 1.4-.6T22 16t-.6-1.4T20 14t-1.4.6T18 16t.6 1.4 1.4.6zm0-6q.81 0 1.4-.6T22 10t-.6-1.4T20 8t-1.4.6T18 10t.6 1.4 1.4.6z',
                'w' => .875,
                'vw' => 28
            ],
            'dice-three' => [
                'p' => 'M24 2q1.69 0 2.84 1.16T28 6v20q0 1.69-1.16 2.84T24 30H4q-1.69 0-2.84-1.16T0 26V6q0-1.69 1.16-2.84T4 2h20zM8 12q.81 0 1.4-.6T10 10t-.6-1.4T8 8t-1.4.6T6 10t.6 1.4T8 12zm6 6q.81 0 1.4-.6T16 16t-.6-1.4T14 14t-1.4.6T12 16t.6 1.4 1.4.6zm6 6q.81 0 1.4-.6T22 22t-.6-1.4T20 20t-1.4.6T18 22t.6 1.4 1.4.6z',
                'w' => .875,
                'vw' => 28
            ],
            'dice-two' => [
                'p' => 'M24 2q1.69 0 2.84 1.16T28 6v20q0 1.69-1.16 2.84T24 30H4q-1.69 0-2.84-1.16T0 26V6q0-1.69 1.16-2.84T4 2h20zM8 12q.81 0 1.4-.6T10 10t-.6-1.4T8 8t-1.4.6T6 10t.6 1.4T8 12zm12 12q.81 0 1.4-.6T22 22t-.6-1.4T20 20t-1.4.6T18 22t.6 1.4 1.4.6z',
                'w' => .875,
                'vw' => 28
            ],
            'digital-tachograph' => [
                'p' => 'M38 6q.81 0 1.4.6T40 8v16q0 .81-.6 1.4T38 26H2q-.81 0-1.4-.6T0 24V8q0-.81.6-1.4T2 6h36zM19 22v-.5q0-.5-.5-.5h-14q-.5 0-.5.5v.5q0 .5.5.5h14q.5 0 .5-.5zM4.5 18q0 .5.5.5h1q.5 0 .5-.5v-1q0-.5-.5-.5H5q-.5 0-.5.5v1zm4 0q0 .5.5.5h1q.5 0 .5-.5v-1q0-.5-.5-.5H9q-.5 0-.5.5v1zm4 0q0 .5.5.5h1q.5 0 .5-.5v-1q0-.5-.5-.5h-1q-.5 0-.5.5v1zm4 0q0 .5.5.5h1q.5 0 .5-.5v-1q0-.5-.5-.5h-1q-.5 0-.5.5v1zm2.5-4v-3q0-.44-.28-.72T18 10H5q-.44 0-.72.28T4 11v3q0 .44.28.72T5 15h13q.44 0 .72-.28T19 14zm17 8v-.5q0-.5-.5-.5h-14q-.5 0-.5.5v.5q0 .5.5.5h14q.5 0 .5-.5z',
                'w' => 1.25,
                'vw' => 40
            ],
            'directions' => 'M31.44 14.56Q32 15.2 32 16t-.56 1.44l-14 14Q16.8 32 16 32t-1.44-.56l-14-14Q0 16.8 0 16t.56-1.44l14-14Q15.2 0 16 0t1.44.56zm-6.32.82q.38-.38 0-.76l-5.3-4.87q-.2-.19-.5-.06t-.32.43v3.38h-7q-.81 0-1.4.6t-.6 1.4v5q0 .5.5.5h2q.5 0 .5-.5v-4h6v3.38q0 .3.31.43t.5-.06z',
            'disease' => 'm29.52 12.24-4.2-1.44a3.58 3.58 0 0 1-2.38-2.39l-1-3.78c-.72-2.73-4.78-3.57-6.87-1.41L12.19 6.2A4.28 4.28 0 0 1 8.8 7.41l-4.5-.35C1.05 6.8-1.12 9.86.62 12.23l2.4 3.28c.7.94.8 2.12.3 3.14l-1.78 3.56c-1.29 2.58 1.42 5.31 4.6 4.62l4.37-.96a4.4 4.4 0 0 1 3.56.74l3.4 2.55c2.46 1.84 6.31.47 6.53-2.33l.29-3.87c.09-1.1.8-2.11 1.91-2.68l3.88-1.99c2.8-1.43 2.47-5-.56-6.05ZM10 16a2 2 0 1 1 0-4 2 2 0 0 1 0 4Zm8 6a2 2 0 1 1 0-4 2 2 0 0 1 0 4Zm1-8a1 1 0 0 1 0-2 1 1 0 0 1 0 2Zm0 0',
            'display' => [
                'p' => 'M64 0C28.7 0 0 28.7 0 64V352c0 35.3 28.7 64 64 64H240l-10.7 32H160c-17.7 0-32 14.3-32 32s14.3 32 32 32H416c17.7 0 32-14.3 32-32s-14.3-32-32-32H346.7L336 416H512c35.3 0 64-28.7 64-64V64c0-35.3-28.7-64-64-64H64zM512 64V352H64V64H512z',
                'vw' => 576,
                'vh' => 512
            ],
            'divide' => [
                'p' => 'M14 22q1.69 0 2.84 1.16T18 26t-1.16 2.84T14 30t-2.84-1.16T10 26t1.16-2.84T14 22zm0-12q-1.69 0-2.84-1.16T10 6t1.16-2.84T14 2t2.84 1.16T18 6t-1.16 2.84T14 10zm12 3q.81 0 1.4.6T28 15v2q0 .81-.6 1.4T26 19H2q-.81 0-1.4-.6T0 17v-2q0-.81.6-1.4T2 13h24z',
                'w' => .875,
                'vw' => 28
            ],
            'dizzy' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm-6 12.94 1.81 1.75q.44.44.97.25t.69-.66-.28-.97L10.94 12l1.75-1.81q.31-.25.34-.57t-.15-.56-.44-.44-.56-.15-.57.34L9.5 10.56 7.69 8.81q-.44-.44-.97-.25t-.69.66.28.97l1.75 1.8-1.75 1.82q-.44.44-.28.97t.66.66 1-.25zm6 12.56q1.63 0 2.81-1.19t1.2-2.81-1.2-2.81-2.81-1.2-2.81 1.2-1.2 2.8 1.2 2.82T15.5 26zm9.19-12.19L22.94 12l1.75-1.81q.44-.44.25-.97t-.66-.69-.97.28l-1.81 1.75-1.81-1.75q-.44-.44-.97-.25t-.69.66.28.97l1.75 1.8-1.75 1.82q-.44.44-.28.97t.66.66 1-.25l1.8-1.75 1.82 1.75q.44.44.97.25t.69-.66-.28-.97z',
            'dna' => [
                'p' => 'M0 30.88q.19-1.32.63-2.79t1.43-3.59 3.07-4.4T10 16q-2.81-1.81-4.87-4.1T2.06 7.5.62 3.9 0 1.14Q-.06.69.25.33T1 0h2q.88 0 1 .88.13.5.25 1.12h19.5L24 .87Q24.13 0 25 0h2q.44 0 .75.34t.25.78q-.13 1.13-.44 2.38t-1.25 3.22-2.25 3.75-3.72 3.78-5.47 3.56q-2.75 1.38-4.87 3.2h7.94q-.82-.76-1.07-.88 2-1.07 3.7-2.25 2.62 2.25 4.4 5.06t2.31 4.56.72 3.38q.06.43-.25.78T27 32h-2q-.94 0-1-.87-.06-.38-.25-1.13H4.25L4 31.13Q3.87 32 3 32H1q-.44 0-.75-.34T0 30.88zm14-17.13q2.25-1.25 4-2.75h-8q1.75 1.5 4 2.75zM22.19 6H5.8q.57 1 1.32 2h13.75q.75-1 1.3-2zM5.8 26h16.32q-.5-1-1.32-2H7.13q-.75 1-1.32 2z',
                'w' => .875,
                'vw' => 28
            ],
            'dog' => 'M31 6q.44 0 .72.28T32 7v2q0 1.69-1.16 2.84T28 13h-2v2.25l-8-2.88V3q0-.69.63-.94t1.06.25L21.44 4h3.31q.5 0 1.03.31t.78.82L27 6h4zm-7 3q.44 0 .72-.28T25 8t-.28-.72T24 7t-.72.28T23 8t.28.72T24 9zM6 14h10.63L26 17.38V31q0 .44-.28.72T25 32h-4q-.44 0-.72-.28T20 31v-7H10v7q0 .44-.28.72T9 32H5q-.44 0-.72-.28T4 31V17.62Q2.25 17 1.12 15.47T0 12q0-.81.6-1.4T2 10t1.4.6T4 12t.6 1.4T6 14z',
            'dollar' => 'dollar-sign',
            'dollar-sign' => [
                'p' => 'M13.06 14.56q2.75.82 4.16 3.44t.4 5.44q-.75 2.06-2.56 3.28T11 28v3q0 .44-.28.72T10 32H8q-.44 0-.72-.28T7 31v-3q-3 0-5.38-1.87-.37-.32-.43-.75t.3-.75l2.13-2.13q.63-.56 1.25-.12Q5.81 23 7 23h4.12q.82 0 1.35-.53t.53-1.34q0-1.38-1.31-1.75L5.25 17.5q-2.13-.62-3.57-2.25T0 11.5q-.25-3.06 1.78-5.28T6.88 4H7V1q0-.44.28-.72T8 0h2q.43 0 .72.28T11 1v3q3 0 5.37 1.88.38.3.44.75t-.31.75L14.37 9.5q-.62.56-1.25.13Q12.18 9 11 9H6.87q-.81 0-1.34.53T5 10.88q0 .56.37 1.06t.94.69z',
                'w' => .563,
                'vw' => 18,
            ],
            'dolly' => [
                'p' => 'm18.38 17.38-3.32-10q-.06-.25-.03-.5t.22-.47.44-.29l3.87-1.25 2.07 6.2 3.8-1.26-2.05-6.19 3.8-1.3q.2-.07.38-.04t.38.1.31.21.19.35l3.75 11.31q.13.44-.06.81t-.57.5L21.5 18.88q-1.44-1.07-3.12-1.5zm17.56 3q.12.43-.06.78t-.57.53L22 26.12q-.06 2.75-2.19 4.5t-4.94 1.25q-1.81-.3-3.15-1.68t-1.66-3.25q-.5-3.44 2.25-5.63L6.56 4H1q-.44 0-.72-.28T0 3V1Q0 .56.28.28T1 0h7q1.44 0 1.87 1.37L16.12 20q2.82.06 4.57 2.31l13.37-4.44q.38-.12.75.07t.5.56zM16 29q1.25 0 2.13-.88T19 26t-.88-2.13T16 23t-2.13.88T13 26t.88 2.13T16 29z',
                'w' => 1.125,
                'vw' => 36
            ],
            'dolly-flatbed' => [
                'p' => 'M13 20q-.44 0-.72-.28T12 19V3q0-.44.28-.72T13 2h9v8l3-2 3 2V2h9q.44 0 .72.28T38 3v16q0 .44-.28.72T37 20H13zm26 4q.44 0 .72.28T40 25v2q0 .44-.28.72T39 28h-5.19q.19.56.19 1 0 1.25-.88 2.13T31 32t-2.13-.88T28 29q0-.5.19-1H15.8q.2.56.2 1 0 1.25-.88 2.13T13 32t-2.12-.88T10 29q0-.5.19-1H5q-.44 0-.72-.28T4 27V4H1q-.44 0-.72-.28T0 3V1Q0 .56.28.28T1 0h6q.44 0 .72.28T8 1v23h31z',
                'w' => 1.25,
                'vw' => 40
            ],
            'donate' => 'M16 26q-5.38 0-9.19-3.81T3 12.99t3.8-9.17T16 0t9.19 3.81T29 13t-3.81 9.19T16 26zM14.62 6.06q-1.56.07-2.62 1.2T10.94 10q0 1.31.75 2.35t1.93 1.4l3.94 1.19q.75.19.75 1.06 0 .5-.31.82t-.75.3h-2.38q-.62 0-1.12-.3t-.88.06l-1 1q-.24.25-.24.56t.3.5q1.2.88 2.7.94v1.06q0 .44.28.75t.72.31h.68q.44 0 .75-.3t.32-.76v-1q1.56-.06 2.62-1.19T21.06 16q0-1.3-.75-2.34t-2-1.4l-3.87-1.2q-.75-.18-.75-1.06 0-.5.31-.81t.7-.31h2.43q.62 0 1.12.31t.88-.06l1-1q.25-.25.22-.57t-.28-.5q-1.2-.87-2.7-.93V5.06q0-.43-.3-.75t-.7-.31h-.74q-.25 0-.5.13t-.38.37-.12.56v1zM30 22q.81 0 1.4.6T32 24v6q0 .81-.6 1.4T30 32H2q-.81 0-1.4-.6T0 30v-6q0-.81.6-1.4T2 22h2q1.88 2.44 4.56 4H4.62Q4 26 4 26.5v1q0 .5.63.5h22.75q.62 0 .62-.5v-1q0-.5-.63-.5h-4q2.75-1.56 4.63-4h2z',
            'dong-sign' => [
                'p' => 'M288 32c-17.7 0-32 14.3-32 32l-32 0c-17.7 0-32 14.3-32 32s14.3 32 32 32h32v49.1c-18.8-10.9-40.7-17.1-64-17.1c-70.7 0-128 57.3-128 128s57.3 128 128 128c24.5 0 47.4-6.9 66.8-18.8c5 11.1 16.2 18.8 29.2 18.8c17.7 0 32-14.3 32-32V288 128c17.7 0 32-14.3 32-32s-14.3-32-32-32c0-17.7-14.3-32-32-32zM128 288a64 64 0 1 1 128 0 64 64 0 1 1 -128 0zM32 448c-17.7 0-32 14.3-32 32s14.3 32 32 32H352c17.7 0 32-14.3 32-32s-14.3-32-32-32H32z',
                'vw' => 384,
                'vh' => 512
            ],
            'door-closed' => [
                'p' => 'M39 28q.44 0 .72.28T40 29v2q0 .44-.28.72T39 32H1q-.44 0-.72-.28T0 31v-2q0-.44.28-.72T1 28h7V3.19q0-1.32.88-2.25T11 0h18q1.25 0 2.13.94T32 3.19V28h7zM26 18q.81 0 1.4-.6T28 16t-.6-1.4T26 14t-1.4.6T24 16t.6 1.4 1.4.6z',
                'w' => 1.25,
                'vw' => 40
            ],
            'door-open' => [
                'p' => 'M39 28q.44 0 .72.28T40 29v2q0 .44-.28.72T39 32h-9V8h-6V4h7q1.25 0 2.13.9T34 7.07V28h5zM19.5.06q.94-.25 1.72.38T22 2.06V32H1q-.44 0-.72-.28T0 31v-2q0-.44.28-.72T1 28h5V5.2q0-.69.44-1.25T7.5 3.2zM16.5 18q.63 0 1.06-.6T18 16t-.44-1.4-1.06-.6-1.06.6T15 16t.44 1.4 1.06.6z',
                'w' => 1.25,
                'vw' => 40
            ],
            'dot-circle' => 'M16 .5q6.44 0 10.97 4.53T31.5 16t-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5zM21 16q0-2.06-1.47-3.53T16 11t-3.53 1.47T11 16t1.47 3.53T16 21t3.53-1.47T21 16z',
            'dove' => 'M18 10.44q-4.88-1-8.19-4.69 1.06-3.06 3.06-5.37.44-.5 1.04-.32t.72.82q.62 4.3 3.37 7.8v1.76zM25 4h7l-2 4v10q0 4.13-2.94 7.06T20 28h-4.81l-4.07 3.5q-.62.5-1.37.5-6.69-.25-9.56-4.75-.32-.44-.13-.94t.69-.62l9-2.25q-1.38-1-2.44-2.07Q2 16.32 2 9.81 2 6 3.62 2.56q.25-.56.9-.56t.91.62q1.88 4.32 5.79 7.07T20 12.75V9q0-2.06 1.47-3.53T25 4zm0 6q.44 0 .72-.28T26 9t-.28-.72T25 8t-.72.28T24 9t.28.72.72.28z',
            'down-left-and-up-right-to-center' => [
                'p' => 'M439 7c9.4-9.4 24.6-9.4 33.9 0l32 32c9.4 9.4 9.4 24.6 0 33.9l-87 87 39 39c6.9 6.9 8.9 17.2 5.2 26.2s-12.5 14.8-22.2 14.8H296c-13.3 0-24-10.7-24-24V72c0-9.7 5.8-18.5 14.8-22.2s19.3-1.7 26.2 5.2l39 39L439 7zM72 272H216c13.3 0 24 10.7 24 24V440c0 9.7-5.8 18.5-14.8 22.2s-19.3 1.7-26.2-5.2l-39-39L73 505c-9.4 9.4-24.6 9.4-33.9 0L7 473c-9.4-9.4-9.4-24.6 0-33.9l87-87L55 313c-6.9-6.9-8.9-17.2-5.2-26.2s12.5-14.8 22.2-14.8z',
                'vw' => 512,
                'vh' => 512
            ],
            'down-long' => [
                'p' => 'M2 334.5c-3.8 8.8-2 19 4.6 26l136 144c4.5 4.8 10.8 7.5 17.4 7.5s12.9-2.7 17.4-7.5l136-144c6.6-7 8.4-17.2 4.6-26s-12.5-14.5-22-14.5l-72 0 0-288c0-17.7-14.3-32-32-32L128 0C110.3 0 96 14.3 96 32l0 288-72 0c-9.6 0-18.2 5.7-22 14.5z',
                'vw' => 320,
                'vh' => 512
            ],
            'download' => 'M13.5 0h5q.63 0 1.06.44T20 1.5V12h5.5q.81 0 1.13.78t-.25 1.35l-9.5 9.5Q16.5 24 16 24t-.87-.37l-9.5-9.5q-.57-.57-.25-1.35T6.5 12H12V1.5q0-.62.44-1.06T13.5 0zM32 23.5v7q0 .63-.44 1.06T30.5 32h-29q-.63 0-1.06-.44T0 30.5v-7q0-.63.44-1.06T1.5 22h9.19l3.06 3.06q.94.94 2.25.94t2.25-.94L21.31 22h9.2q.62 0 1.05.44T32 23.5zM24.25 29q0-.5-.38-.88t-.87-.37-.88.38-.37.87.38.88.87.37.88-.38.37-.87zm4 0q0-.5-.38-.88t-.87-.37-.88.38-.37.87.38.88.87.37.88-.38.37-.87z',
            'drafting-compass' => 'm28.56 21.5 3.44 6-.44 3.63q-.06.5-.5.75t-.87.06L26.8 30.5l-3.44-5.94q2.75-1.06 5.2-3.06zm2.69-5.88q-1.69 2.57-4.06 4.47T21.94 23 16 24q-3.25 0-6.38-1.19L5.2 30.5l-3.4 1.44q-.19.06-.44.06t-.44-.12-.34-.32-.15-.43L0 27.5l4.43-7.75Q2.37 18 .93 15.87q-.18-.37-.09-.78t.47-.59l1.75-1q.75-.5 1.31.31.88 1.32 2.13 2.44l4.25-7.37Q10 7.5 10 6q0-2.5 1.75-4.25T16 0t4.25 1.75T22 6q0 1.5-.75 2.88l3.25 5.56q-2.19 2.25-5.19 3.06L16.12 12h-.25l-4.19 7.25q2.2.75 4.32.75 3.56 0 6.69-1.72t5.06-4.65q.5-.82 1.31-.38l1.75.94q.38.25.5.66t-.06.78zM16 4q-.81 0-1.4.6T14 6t.6 1.4T16 8t1.4-.6T18 6t-.6-1.4T16 4z',
            'dragon' => [
                'p' => 'M1.13 16q-.75.06-1.04-.63t.29-1.12l7.3-7.31q1.07-.88 2.38-.94t2.44.69l7.5 5.43v2.7q0 2.62.94 5.18H7q-.69 0-.94-.62t.25-1.07l5.7-4.31zm34.8 2.13q2 1 3.1 2.87t.97 4.06Q39.81 28 37.56 30t-5.25 2H1.12q-.5 0-.8-.31T0 30.87q0-.87.87-1.06 4.44-1 7.1-1.53t8.53-1.31T28 26q-2.81-1.88-4.4-4.84T22 14.8V8l-3.75-1.56Q18 6.3 18 6t.25-.44L22 4 18.12.87q-.18-.25-.06-.56T18.5 0h14.8q1 0 1.63.8L39.62 7q.38.56.38 1.18 0 .5-.19.94l-.94 1.75q-.18.5-.72.82t-1.03.3H35.2q-.88 0-1.44-.55L32 10h-4v2.31q0 .75.5 1.53T29.69 15zm-5.37-14q-.18.75.25 1.28t1.13.59q1.19.06 1.5-1.13z',
                'w' => 1.25,
                'vw' => 40
            ],
            'draw-polygon' => [
                'p' => 'M24 22q1.69 0 2.84 1.16T28 26t-1.16 2.84T24 30q-2.31 0-3.44-2H7.44Q6.3 30 4 30q-1.69 0-2.84-1.16T0 26q0-2.31 2-3.44V9.44Q0 8.3 0 6q0-1.7 1.16-2.85T4 2q2.3 0 3.44 2h13.12Q21.7 2 24 2q1.69 0 2.84 1.15T28 6t-1.16 2.84T24 10h-.06l-2.44 4.06Q22 15 22 16t-.5 1.94L23.94 22H24zm-18 .56q.94.5 1.44 1.44h13l-2.38-4H18q-1.69 0-2.84-1.16T14 16t1.16-2.84T18 12h.06l2.38-4h-13q-.5.94-1.44 1.44v13.12zM17 16q0 .44.28.72T18 17t.72-.28T19 16t-.28-.72T18 15t-.72.28T17 16zm8-10q0-.44-.28-.72T24 5t-.72.28T23 6t.28.72T24 7t.72-.28T25 6zM4 5q-.44 0-.72.28T3 6t.28.72T4 7t.72-.28T5 6t-.28-.72T4 5zM3 26q0 .44.28.72T4 27t.72-.28T5 26t-.28-.72T4 25t-.72.28T3 26zm21 1q.44 0 .72-.28T25 26t-.28-.72T24 25t-.72.28T23 26t.28.72.72.28z',
                'w' => .875,
                'vw' => 28
            ],
            'droplet' => [
                'p' => 'M192 512C86 512 0 426 0 320C0 228.8 130.2 57.7 166.6 11.7C172.6 4.2 181.5 0 191.1 0h1.8c9.6 0 18.5 4.2 24.5 11.7C253.8 57.7 384 228.8 384 320c0 106-86 192-192 192zM96 336c0-8.8-7.2-16-16-16s-16 7.2-16 16c0 61.9 50.1 112 112 112c8.8 0 16-7.2 16-16s-7.2-16-16-16c-44.2 0-80-35.8-80-80z',
                'vw' => 384,
                'vh' => 512
            ],
            'droplet-slash' => [
                'p' => 'M320 512c53.2 0 101.4-21.6 136.1-56.6l-298.3-235C140 257.1 128 292.3 128 320c0 106 86 192 192 192zM505.2 370.7c4.4-16.1 6.8-33.1 6.8-50.7c0-91.2-130.2-262.3-166.6-308.3C339.4 4.2 330.5 0 320.9 0h-1.8c-9.6 0-18.5 4.2-24.5 11.7C277.8 33 240.7 81.3 205.8 136L38.8 5.1C28.4-3.1 13.3-1.2 5.1 9.2S-1.2 34.7 9.2 42.9l592 464c10.4 8.2 25.5 6.3 33.7-4.1s6.3-25.5-4.1-33.7L505.2 370.7zM224 336c0 44.2 35.8 80 80 80c8.8 0 16 7.2 16 16s-7.2 16-16 16c-61.9 0-112-50.1-112-112c0-8.8 7.2-16 16-16s16 7.2 16 16z',
                'vw' => 640,
                'vh' => 512
            ],
            'drum' => [
                'p' => 'M28.63 7.56q3.43 1.13 5.4 2.82T36 14v10.06q0 2.82-4 5v-6q0-.8-.6-1.4t-1.4-.6-1.4.6-.6 1.4v7.63Q24.37 31.8 20 32v-6.94q0-.8-.6-1.4t-1.4-.6-1.4.6-.6 1.4V32q-4.38-.19-8-1.31v-7.63q0-.8-.6-1.4t-1.4-.6-1.4.6-.6 1.4v6q-4-2.18-4-5V14q0-3.31 5.28-5.66T18 6q2.5 0 4.94.31L33 .31q.69-.43 1.5-.25t1.22.91.22 1.53T35 3.7zM18 19q6.19 0 10.6-1.47T33 14q0-.94-.97-1.78t-2.75-1.5-4.1-1.1L15 15.7q-.63.44-1.44.25T12.31 15q-.44-.69-.25-1.5t.94-1.19L18.44 9H18q-6.2 0-10.6 1.47T3 14t4.4 3.53T18 19z',
                'w' => 1.125,
                'vw' => 36
            ],
            'drum-steelpan' => [
                'p' => 'M18 2q7.44 0 12.72 2.34T36 10v12q0 3.31-5.28 5.66T18 30 5.28 27.66 0 22V10q0-3.31 5.28-5.66T18 2zm-5.19 9.88q.5-1.82-.44-3.5l-1.62-2.76q-3.5.63-5.63 1.82T3 10q0 1.44 2.19 2.63t5.87 1.8q1.32-1 1.75-2.55zM18 15q1.88 0 3.81-.19-.37-1.25-1.43-2.03T18 12q-.88 0-1.66.38t-1.34 1-.81 1.43q1.94.2 3.81.2zm4-9v-.81q-2-.2-4-.2t-4 .2V6q0 1.63 1.19 2.81t2.81 1.2 2.81-1.2T22.01 6zm2.94 8.44q2.37-.44 4.19-1.1t2.84-1.53T33 10q0-1.38-2.12-2.56t-5.63-1.82l-1.62 2.75q-.94 1.7-.44 3.5.37 1.57 1.75 2.57z',
                'w' => 1.125,
                'vw' => 36
            ],
            'drumstick-bite' => 'M28.94 3.13Q32.13 6.3 32 10.88q-1.94-1.07-4.19-.79t-3.87 1.85q-1.88 1.87-1.9 4.62t1.84 4.82q-1.82.62-3.75.62h-5.5l-2.5 2.56q-.94.88-.38 2.25.25.7.25 1.44 0 1.56-1.1 2.66T8.26 32t-2.69-1.13q-1.43-1.38-.93-3.5-2.13.5-3.5-.94Q0 25.3 0 23.75q0-1 .5-1.87t1.38-1.38 1.87-.5q.75 0 1.44.25 1.37.56 2.31-.37l2.5-2.5V12q0-4.94 3.94-8.87.87-.88 2.16-1.6T18.8.41t2.63-.4q1.81 0 4 .9t3.5 2.22z',
            'dumbbell' => [
                'p' => 'M6.5 6q.63 0 1.06.44T8 7.5v17q0 .63-.44 1.06T6.5 26h-3q-.63 0-1.06-.44T2 24.5V18H.5q-.5 0-.5-.5v-3q0-.5.5-.5H2V7.5q0-.62.44-1.06T3.5 6h3zm33 8q.5 0 .5.5v3q0 .5-.5.5H38v6.5q0 .63-.44 1.06T36.5 26h-3q-.63 0-1.06-.44T32 24.5v-17q0-.63.44-1.06T33.5 6h3q.62 0 1.06.44T38 7.5V14h1.5zm-11-12q.63 0 1.06.44T30 3.5v25q0 .63-.44 1.06T28.5 30h-3q-.63 0-1.06-.44T24 28.5V18h-8v10.5q0 .63-.44 1.06T14.5 30h-3q-.63 0-1.06-.44T10 28.5v-25q0-.62.43-1.06T11.5 2h3q.62 0 1.06.44T16 3.5V14h8V3.5q0-.62.44-1.06T25.5 2h3z',
                'w' => 1.25,
                'vw' => 40
            ],
            'dumpster' => [
                'p' => 'M35 10h-6.06L27.3 2h6.19q.75 0 1 .75l1.5 6q.06.5-.22.88T35 10zM17 2v8H9.12l1.63-8H17zm8.25 0 1.63 8H19V2h6.25zM1 10q-.5 0-.78-.38T0 8.76l1.5-6Q1.75 2 2.5 2h6.19l-1.63 8H1zm34 4q.44 0 .72.28T36 15v2q0 .44-.28.72T35 18h-1.75L32 28v1q0 .44-.28.72T31 30h-2q-.44 0-.72-.28T28 29v-1H8v1q0 .44-.28.72T7 30H5q-.44 0-.72-.28T4 29v-1L2.75 18H1q-.44 0-.72-.28T0 17v-2q0-.44.28-.72T1 14h1.25L2 12h32l-.25 2H35z',
                'w' => 1.125,
                'vw' => 36
            ],
            'dumpster-fire' => [
                'p' => 'M26.19 6.5Q24.37 8.06 22.8 10H19V2h6.25zM17 2v8H9.12l1.63-8H17zm11.81 4.5-.75-.63L27.31 2h6.2q.8 0 .93.75l1.5 6q.06 0 .03.06l-.03.07-.19-.2-1.31-1.18-1.31 1.19q-.44.37-1.13 1.06-1.44-1.69-3.19-3.25zM1 10q-.5 0-.78-.38T0 8.76l1.5-6Q1.75 2 2.5 2h6.19l-1.63 8H1zm20.31 2Q18 16.69 18 20.13q0 4.5 3.06 7.87H8v1q0 .44-.28.72T7 30H5q-.44 0-.72-.28T4 29v-1L2.75 18H1q-.44 0-.72-.28T0 17v-2q0-.44.28-.72T1 14h1.25L2 12h19.31zm13.13-1.81q2.31 2.06 3.94 5.06T40 20.13q0 4.06-2.94 6.97T30 30t-7.06-2.9T20 20.13q0-2.38 2.12-5.78T27.5 8q2.5 2.25 4.44 4.75 1.18-1.37 2.5-2.56zm-1.19 14.37q1.56-1.12 1.94-3.1t-.5-3.65q-.19-.43-.5-.87l-2.44 2.94q-3.94-5.2-4.25-5.5-1.69 2.06-2.37 3.3t-.7 2.45q0 2.43 1.6 3.97T30 25.63q1.81 0 3.25-1.07z',
                'w' => 1.248,
                'vw' => 40
            ],
            'dungeon' => 'M8.06 12.19q.7.44.38 1.25t-.38 1.69q-.12.87-.93.87H1.06q-.31 0-.53-.15t-.37-.38-.1-.53q.19-2.88 1.38-5.56.19-.38.62-.5t.82.12zM19.94.5q.44.13.62.53t0 .78l-2.31 6.5Q18 9 17.31 9H14.7q-.7 0-.94-.69l-2.31-6.5q-.2-.37 0-.78t.62-.53Q14 0 16 0t3.94.5zM7 18q.44 0 .72.28T8 19v4q0 .44-.28.72T7 24H1q-.44 0-.72-.28T0 23v-4q0-.44.28-.72T1 18h6zm0 8q.44 0 .72.28T8 27v4q0 .44-.28.72T7 32H1q-.44 0-.72-.28T0 31v-4q0-.44.28-.72T1 26h6zm4.81-17.75q.32.75-.37 1.25-.57.38-1.07.88-.56.56-1.25.12L3.94 7.31q-.38-.25-.47-.69t.22-.8q1.87-2.26 4.5-3.76.37-.19.78-.03t.6.6zM24.87 16q-.8 0-.93-.88-.07-.87-.38-1.68t.38-1.25l5.18-3.2q.38-.24.82-.12t.62.5q1.2 2.7 1.38 5.57.06.43-.25.75t-.75.31h-6.07zm3.44-10.19q.32.38.22.82t-.47.68l-5.18 3.2q-.7.43-1.25-.13-.5-.5-1.07-.88-.69-.5-.37-1.25l2.25-5.62q.18-.44.6-.6t.77.03q2.63 1.5 4.5 3.75zM31 18q.44 0 .72.28T32 19v4q0 .44-.28.72T31 24h-6q-.44 0-.72-.28T24 23v-4q0-.44.28-.72T25 18h6zm0 8q.44 0 .72.28T32 27v4q0 .44-.28.72T31 32h-6q-.44 0-.72-.28T24 31v-4q0-.44.28-.72T25 26h6zM15 11.12q.63-.12 1-.12t1 .13V29.5q0 .5-.5.5h-1q-.5 0-.5-.5V11.12zm-4 2.57q.75-1.13 2-1.88v17.7q0 .5-.5.5h-1q-.5 0-.5-.5V13.68zm8-1.88q1.25.75 2 1.88V29.5q0 .5-.5.5h-1q-.5 0-.5-.5V11.81z',
            'e' => [
                'p' => 'M64 32C28.7 32 0 60.7 0 96V256 416c0 35.3 28.7 64 64 64H288c17.7 0 32-14.3 32-32s-14.3-32-32-32H64V288H224c17.7 0 32-14.3 32-32s-14.3-32-32-32H64V96H288c17.7 0 32-14.3 32-32s-14.3-32-32-32H64z',
                'vw' => 320,
                'vh' => 512
            ],
            'ear-deaf' => [
                'p' => 'M502.6 54.6l-40 40c-12.5 12.5-32.8 12.5-45.3 0s-12.5-32.8 0-45.3l40-40c12.5-12.5 32.8-12.5 45.3 0s12.5 32.8 0 45.3zm-320 320l-128 128c-12.5 12.5-32.8 12.5-45.3 0s-12.5-32.8 0-45.3l128-128c12.5-12.5 32.8-12.5 45.3 0s12.5 32.8 0 45.3zM240 128c-57.6 0-105.1 43.6-111.3 99.5c-1.9 17.6-17.8 30.2-35.3 28.3s-30.2-17.8-28.3-35.3C74.8 132.5 149.4 64 240 64c97.2 0 176 78.8 176 176c0 46-17.7 87.9-46.6 119.3c-12 13-17.4 24.8-17.4 34.7V400c0 61.9-50.1 112-112 112c-17.7 0-32-14.3-32-32s14.3-32 32-32c26.5 0 48-21.5 48-48v-6.1c0-32.9 17.4-59.6 34.4-78c18.4-20 29.6-46.6 29.6-75.9c0-61.9-50.1-112-112-112zm0 80c-17.7 0-32 14.3-32 32c0 13.3-10.7 24-24 24s-24-10.7-24-24c0-44.2 35.8-80 80-80s80 35.8 80 80c0 13.3-10.7 24-24 24s-24-10.7-24-24c0-17.7-14.3-32-32-32z',
                'vw' => 512,
                'vh' => 512
            ],
            'ear-listen' => [
                'p' => 'M398.3 3.4c-15.8-7.9-35-1.5-42.9 14.3c-7.9 15.8-1.5 34.9 14.2 42.9l.4 .2c.4 .2 1.1 .6 2.1 1.2c2 1.2 5 3 8.7 5.6c7.5 5.2 17.6 13.2 27.7 24.2C428.5 113.4 448 146 448 192c0 17.7 14.3 32 32 32s32-14.3 32-32c0-66-28.5-113.4-56.5-143.7C441.6 33.2 427.7 22.2 417.3 15c-5.3-3.7-9.7-6.4-13-8.3c-1.6-1-3-1.7-4-2.2c-.5-.3-.9-.5-1.2-.7l-.4-.2-.2-.1-.1 0 0 0c0 0 0 0-14.3 28.6L398.3 3.4zM128.7 227.5c6.2-56 53.7-99.5 111.3-99.5c61.9 0 112 50.1 112 112c0 29.3-11.2 55.9-29.6 75.9c-17 18.4-34.4 45.1-34.4 78V400c0 26.5-21.5 48-48 48c-17.7 0-32 14.3-32 32s14.3 32 32 32c61.9 0 112-50.1 112-112v-6.1c0-9.8 5.4-21.7 17.4-34.7C398.3 327.9 416 286 416 240c0-97.2-78.8-176-176-176C149.4 64 74.8 132.5 65.1 220.5c-1.9 17.6 10.7 33.4 28.3 35.3s33.4-10.7 35.3-28.3zM32 512a32 32 0 1 0 0-64 32 32 0 1 0 0 64zM192 352a32 32 0 1 0 -64 0 32 32 0 1 0 64 0zM41.4 361.4c-12.5 12.5-12.5 32.8 0 45.3l64 64c12.5 12.5 32.8 12.5 45.3 0s12.5-32.8 0-45.3l-64-64c-12.5-12.5-32.8-12.5-45.3 0zM208 240c0-17.7 14.3-32 32-32s32 14.3 32 32c0 13.3 10.7 24 24 24s24-10.7 24-24c0-44.2-35.8-80-80-80s-80 35.8-80 80c0 13.3 10.7 24 24 24s24-10.7 24-24z',
                'vw' => 512,
                'vh' => 512
            ],
            'earth-africa' => [
                'p' => 'M177.8 63.2l10 17.4c2.8 4.8 4.2 10.3 4.2 15.9v41.4c0 3.9 1.6 7.7 4.3 10.4c6.2 6.2 16.5 5.7 22-1.2l13.6-17c4.7-5.9 12.9-7.7 19.6-4.3l15.2 7.6c3.4 1.7 7.2 2.6 11 2.6c6.5 0 12.8-2.6 17.4-7.2l3.9-3.9c2.9-2.9 7.3-3.6 11-1.8l29.2 14.6c7.8 3.9 12.6 11.8 12.6 20.5c0 10.5-7.1 19.6-17.3 22.2l-35.4 8.8c-7.4 1.8-15.1 1.5-22.4-.9l-32-10.7c-3.3-1.1-6.7-1.7-10.2-1.7c-7 0-13.8 2.3-19.4 6.5L176 212c-10.1 7.6-16 19.4-16 32v28c0 26.5 21.5 48 48 48h32c8.8 0 16 7.2 16 16v48c0 17.7 14.3 32 32 32c10.1 0 19.6-4.7 25.6-12.8l25.6-34.1c8.3-11.1 12.8-24.6 12.8-38.4V318.6c0-3.9 2.6-7.3 6.4-8.2l5.3-1.3c11.9-3 20.3-13.7 20.3-26c0-7.1-2.8-13.9-7.8-18.9l-33.5-33.5c-3.7-3.7-3.7-9.7 0-13.4c5.7-5.7 14.1-7.7 21.8-5.1l14.1 4.7c12.3 4.1 25.7-1.5 31.5-13c3.5-7 11.2-10.8 18.9-9.2l27.4 5.5C432 112.4 351.5 48 256 48c-27.7 0-54 5.4-78.2 15.2zM0 256a256 256 0 1 1 512 0A256 256 0 1 1 0 256z',
                'vw' => 512,
                'vh' => 512
            ],
            'earth-americas' => [
                'p' => 'M57.7 193l9.4 16.4c8.3 14.5 21.9 25.2 38 29.8L163 255.7c17.2 4.9 29 20.6 29 38.5v39.9c0 11 6.2 21 16 25.9s16 14.9 16 25.9v39c0 15.6 14.9 26.9 29.9 22.6c16.1-4.6 28.6-17.5 32.7-33.8l2.8-11.2c4.2-16.9 15.2-31.4 30.3-40l8.1-4.6c15-8.5 24.2-24.5 24.2-41.7v-8.3c0-12.7-5.1-24.9-14.1-33.9l-3.9-3.9c-9-9-21.2-14.1-33.9-14.1H257c-11.1 0-22.1-2.9-31.8-8.4l-34.5-19.7c-4.3-2.5-7.6-6.5-9.2-11.2c-3.2-9.6 1.1-20 10.2-24.5l5.9-3c6.6-3.3 14.3-3.9 21.3-1.5l23.2 7.7c8.2 2.7 17.2-.4 21.9-7.5c4.7-7 4.2-16.3-1.2-22.8l-13.6-16.3c-10-12-9.9-29.5 .3-41.3l15.7-18.3c8.8-10.3 10.2-25 3.5-36.7l-2.4-4.2c-3.5-.2-6.9-.3-10.4-.3C163.1 48 84.4 108.9 57.7 193zM464 256c0-36.8-9.6-71.4-26.4-101.5L412 164.8c-15.7 6.3-23.8 23.8-18.5 39.8l16.9 50.7c3.5 10.4 12 18.3 22.6 20.9l29.1 7.3c1.2-9 1.8-18.2 1.8-27.5zM0 256a256 256 0 1 1 512 0A256 256 0 1 1 0 256z',
                'vw' => 512,
                'vh' => 512
            ],
            'earth-asia' => [
                'p' => 'M51.7 295.1l31.7 6.3c7.9 1.6 16-.9 21.7-6.6l15.4-15.4c11.6-11.6 31.1-8.4 38.4 6.2l9.3 18.5c4.8 9.6 14.6 15.7 25.4 15.7c15.2 0 26.1-14.6 21.7-29.2l-6-19.9c-4.6-15.4 6.9-30.9 23-30.9h2.3c13.4 0 25.9-6.7 33.3-17.8l10.7-16.1c5.6-8.5 5.3-19.6-.8-27.7l-16.1-21.5c-10.3-13.7-3.3-33.5 13.4-37.7l17-4.3c7.5-1.9 13.6-7.2 16.5-14.4l16.4-40.9C303.4 52.1 280.2 48 256 48C141.1 48 48 141.1 48 256c0 13.4 1.3 26.5 3.7 39.1zm407.7 4.6c-3-.3-6-.1-9 .8l-15.8 4.4c-6.7 1.9-13.8-.9-17.5-6.7l-2-3.1c-6-9.4-16.4-15.1-27.6-15.1s-21.6 5.7-27.6 15.1l-6.1 9.5c-1.4 2.2-3.4 4.1-5.7 5.3L312 330.1c-18.1 10.1-25.5 32.4-17 51.3l5.5 12.4c8.6 19.2 30.7 28.5 50.5 21.1l2.6-1c10-3.7 21.3-2.2 29.9 4.1l1.5 1.1c37.2-29.5 64.1-71.4 74.4-119.5zM0 256a256 256 0 1 1 512 0A256 256 0 1 1 0 256zm144.5 92.1c-2.1 8.6 3.1 17.3 11.6 19.4l32 8c8.6 2.1 17.3-3.1 19.4-11.6s-3.1-17.3-11.6-19.4l-32-8c-8.6-2.1-17.3 3.1-19.4 11.6zm92-20c-2.1 8.6 3.1 17.3 11.6 19.4s17.3-3.1 19.4-11.6l8-32c2.1-8.6-3.1-17.3-11.6-19.4s-17.3 3.1-19.4 11.6l-8 32zM343.2 113.7c-7.9-4-17.5-.7-21.5 7.2l-16 32c-4 7.9-.7 17.5 7.2 21.5s17.5 .7 21.5-7.2l16-32c4-7.9 .7-17.5-7.2-21.5z',
                'vw' => 512,
                'vh' => 512
            ],
            'earth-europe' => [
                'p' => 'M266.3 48.3L232.5 73.6c-5.4 4-8.5 10.4-8.5 17.1v9.1c0 6.8 5.5 12.3 12.3 12.3c2.4 0 4.8-.7 6.8-2.1l41.8-27.9c2-1.3 4.4-2.1 6.8-2.1h1c6.2 0 11.3 5.1 11.3 11.3c0 3-1.2 5.9-3.3 8l-19.9 19.9c-5.8 5.8-12.9 10.2-20.7 12.8l-26.5 8.8c-5.8 1.9-9.6 7.3-9.6 13.4c0 3.7-1.5 7.3-4.1 10l-17.9 17.9c-6.4 6.4-9.9 15-9.9 24v4.3c0 16.4 13.6 29.7 29.9 29.7c11 0 21.2-6.2 26.1-16l4-8.1c2.4-4.8 7.4-7.9 12.8-7.9c4.5 0 8.7 2.1 11.4 5.7l16.3 21.7c2.1 2.9 5.5 4.5 9.1 4.5c8.4 0 13.9-8.9 10.1-16.4l-1.1-2.3c-3.5-7 0-15.5 7.5-18l21.2-7.1c7.6-2.5 12.7-9.6 12.7-17.6c0-10.3 8.3-18.6 18.6-18.6H400c8.8 0 16 7.2 16 16s-7.2 16-16 16H379.3c-7.2 0-14.2 2.9-19.3 8l-4.7 4.7c-2.1 2.1-3.3 5-3.3 8c0 6.2 5.1 11.3 11.3 11.3h11.3c6 0 11.8 2.4 16 6.6l6.5 6.5c1.8 1.8 2.8 4.3 2.8 6.8s-1 5-2.8 6.8l-7.5 7.5C386 262 384 266.9 384 272s2 10 5.7 13.7L408 304c10.2 10.2 24.1 16 38.6 16H454c6.5-20.2 10-41.7 10-64c0-111.4-87.6-202.4-197.7-207.7zm172 307.9c-3.7-2.6-8.2-4.1-13-4.1c-6 0-11.8-2.4-16-6.6L396 332c-7.7-7.7-18-12-28.9-12c-9.7 0-19.2-3.5-26.6-9.8L314 287.4c-11.6-9.9-26.4-15.4-41.7-15.4H251.4c-12.6 0-25 3.7-35.5 10.7L188.5 301c-17.8 11.9-28.5 31.9-28.5 53.3v3.2c0 17 6.7 33.3 18.7 45.3l16 16c8.5 8.5 20 13.3 32 13.3H248c13.3 0 24 10.7 24 24c0 2.5 .4 5 1.1 7.3c71.3-5.8 132.5-47.6 165.2-107.2zM0 256a256 256 0 1 1 512 0A256 256 0 1 1 0 256zM187.3 100.7c-6.2-6.2-16.4-6.2-22.6 0l-32 32c-6.2 6.2-6.2 16.4 0 22.6s16.4 6.2 22.6 0l32-32c6.2-6.2 6.2-16.4 0-22.6z',
                'vw' => 512,
                'vh' => 512
            ],
            'earth-oceania' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM208.6 357.3l-39-13.5c-6.5-2.2-13.6-2.3-20.1-.3l-15.3 4.9c-18.5 5.9-38.5-2.4-47.5-19.5l-3.3-6.2c-10.6-20.1-2.3-45 18.2-54.7l35.3-16.8c2.3-1.1 4.4-2.8 5.9-4.8l5.3-7c7.2-9.6 18.6-15.3 30.6-15.3s23.4 5.7 30.6 15.3l4.6 6.1c2 2.6 4.9 4.5 8.1 5.1c7.8 1.6 15.7-1.5 20.4-7.9l10.4-14.2c2-2.8 5.3-4.4 8.7-4.4c4.4 0 8.4 2.7 10 6.8l10.1 25.9c2.8 7.2 6.7 14 11.5 20.2L311 299.8c5.8 7.4 9 16.6 9 26s-3.2 18.6-9 26L299 367.2c-8.3 10.6-21 16.8-34.4 16.8c-8.4 0-16.6-2.4-23.7-7l-25.4-16.4c-2.2-1.4-4.5-2.5-6.9-3.4zm65.2-214.8L296 164.7c10.1 10.1 2.9 27.3-11.3 27.3H254.8c-5.6 0-11.1-1.2-16.2-3.4l-42.8-19c-14.3-6.3-11.9-27.3 3.4-30.3l38.5-7.7c13.1-2.6 26.7 1.5 36.1 10.9zM248 432c0-8.8 7.2-16 16-16h16c8.8 0 16 7.2 16 16s-7.2 16-16 16H264c-8.8 0-16-7.2-16-16zM431.2 298.9l8 24c2.8 8.4-1.7 17.4-10.1 20.2s-17.4-1.7-20.2-10.1l-8-24c-2.8-8.4 1.7-17.4 10.1-20.2s17.4 1.7 20.2 10.1zm-19.9 80.4l-32 32c-6.2 6.2-16.4 6.2-22.6 0s-6.2-16.4 0-22.6l32-32c6.2-6.2 16.4-6.2 22.6 0s6.2 16.4 0 22.6z',
                'vw' => 512,
                'vh' => 512
            ],
            'edit' => [
                'p' => 'm25.19 5.19 5.62 5.62q.44.44 0 .88L17.13 25.38l-5.75.62q-.38.06-.72-.12t-.53-.53-.13-.72l.63-5.75L24.3 5.18q.44-.43.88 0zM35.3 3.75q.7.75.7 1.75t-.7 1.75l-2.25 2.19q-.43.44-.87 0L26.56 3.8q-.43-.43 0-.87L28.81.75q.7-.75 1.7-.75t1.74.75zM24 21.63q0-.32.25-.5l2.5-2.5q.31-.38.78-.2t.47.7V29q0 1.25-.88 2.13T25 32H3q-1.25 0-2.13-.88T0 29V7q0-1.25.88-2.13T3 4h17.88q.5 0 .68.47t-.18.78l-2.5 2.5q-.2.25-.5.25H4v20h20v-6.38z',
                'w' => 1.123,
                'vw' => 36
            ],
            'egg' => [
                'p' => 'M12 0q3.06 0 5.97 3.66t4.47 8.28T24 20q0 5-3.5 8.5T12 32t-8.5-3.5T0 20q0-3.44 1.56-8.06t4.47-8.28T12 0z',
                'w' => .75,
                'vw' => 24
            ],
            'eject' => [
                'p' => 'M28 24v4q0 .81-.6 1.4T26 30H2q-.81 0-1.4-.6T0 28v-4q0-.81.6-1.4T2 22h24q.81 0 1.4.6T28 24zM3 20q-1.94 0-2.72-1.78T.81 15l11-12Q12.7 2 14 2t2.19 1l11 12q1.31 1.44.53 3.22T25 20H3z',
                'w' => .875,
                'vw' => 28
            ],
            'elevator' => [
                'p' => 'M132.7 4.7l-64 64c-4.6 4.6-5.9 11.5-3.5 17.4s8.3 9.9 14.8 9.9H208c6.5 0 12.3-3.9 14.8-9.9s1.1-12.9-3.5-17.4l-64-64c-6.2-6.2-16.4-6.2-22.6 0zM64 128c-35.3 0-64 28.7-64 64V448c0 35.3 28.7 64 64 64H448c35.3 0 64-28.7 64-64V192c0-35.3-28.7-64-64-64H64zm96 96a48 48 0 1 1 0 96 48 48 0 1 1 0-96zM80 400c0-26.5 21.5-48 48-48h64c26.5 0 48 21.5 48 48v16c0 17.7-14.3 32-32 32H112c-17.7 0-32-14.3-32-32V400zm192 0c0-26.5 21.5-48 48-48h64c26.5 0 48 21.5 48 48v16c0 17.7-14.3 32-32 32H304c-17.7 0-32-14.3-32-32V400zm32-128a48 48 0 1 1 96 0 48 48 0 1 1 -96 0zM356.7 91.3c6.2 6.2 16.4 6.2 22.6 0l64-64c4.6-4.6 5.9-11.5 3.5-17.4S438.5 0 432 0H304c-6.5 0-12.3 3.9-14.8 9.9s-1.1 12.9 3.5 17.4l64 64z',
                'vw' => 512,
                'vh' => 512
            ],
            'ellipsis' => [
                'p' => 'M8 256a56 56 0 1 1 112 0A56 56 0 1 1 8 256zm160 0a56 56 0 1 1 112 0 56 56 0 1 1 -112 0zm216-56a56 56 0 1 1 0 112 56 56 0 1 1 0-112z',
                'vw' => 448,
                'vh' => 512
            ],
            'ellipsis-h' => 'M20.5 16q0 1.88-1.31 3.19t-3.2 1.31-3.18-1.31T11.5 16t1.31-3.19T16 11.5t3.19 1.31T20.5 16zm6.5-4.5q1.88 0 3.19 1.31t1.31 3.2-1.31 3.18T27 20.5t-3.19-1.31T22.5 16t1.31-3.19T27 11.5zm-22 0q1.88 0 3.19 1.31t1.31 3.2-1.31 3.18T5 20.5t-3.19-1.3T.5 16t1.31-3.19T5 11.5z',
            'ellipsis-v' => [
                'p' => 'M6 11.5q1.88 0 3.19 1.31t1.31 3.2-1.31 3.18T6 20.5t-3.19-1.3T1.5 16t1.31-3.19T6 11.5zM1.5 5q0-1.88 1.31-3.19T6.01.5t3.17 1.3T10.5 5 9.2 8.19 6 9.5 2.81 8.2 1.5 5zm0 22q0-1.88 1.31-3.19t3.2-1.31 3.18 1.31T10.5 27t-1.3 3.19T6 31.5t-3.19-1.3T1.5 27z',
                'w' => .375,
                'vw' => 12,
            ],
            'ellipsis-vertical' => [
                'p' => 'M64 360a56 56 0 1 0 0 112 56 56 0 1 0 0-112zm0-160a56 56 0 1 0 0 112 56 56 0 1 0 0-112zM120 96A56 56 0 1 0 8 96a56 56 0 1 0 112 0z',
                'vw' => 128,
                'vh' => 512
            ],
            'envelope' => 'M31.38 11.94q.18-.19.4-.06t.22.37V25q0 1.25-.88 2.13T29 28H3q-1.25 0-2.13-.88T0 25V12.25q0-.25.22-.34t.4.03Q2.7 13.5 10.26 19q.13.13.84.69t1.16.84 1.13.69 1.34.6T16 22q.56 0 1.13-.15t1.06-.38 1.06-.6.94-.65l.87-.65.7-.57q7.37-5.31 9.62-7.06zM16 20q-.5 0-1.22-.34t-1.15-.63-1.29-.94-.9-.65Q3.8 11.87.56 9.37 0 8.94 0 8.2V7q0-1.24.88-2.12T3 4h26q1.25 0 2.13.87T32 7v1.19q0 .75-.56 1.18-3.07 2.38-10.88 8.07-.06 0-.9.65t-1.28.94-1.16.63T16 20z',
            'envelope-circle-check' => [
                'p' => 'M48 64C21.5 64 0 85.5 0 112c0 15.1 7.1 29.3 19.2 38.4L236.8 313.6c11.4 8.5 27 8.5 38.4 0l57.4-43c23.9-59.8 79.7-103.3 146.3-109.8l13.9-10.4c12.1-9.1 19.2-23.3 19.2-38.4c0-26.5-21.5-48-48-48H48zM294.4 339.2c-22.8 17.1-54 17.1-76.8 0L0 176V384c0 35.3 28.7 64 64 64H360.2C335.1 417.6 320 378.5 320 336c0-5.6 .3-11.1 .8-16.6l-26.4 19.8zM640 336a144 144 0 1 0 -288 0 144 144 0 1 0 288 0zm-76.7-43.3c6.2 6.2 6.2 16.4 0 22.6l-72 72c-6.2 6.2-16.4 6.2-22.6 0l-40-40c-6.2-6.2-6.2-16.4 0-22.6s16.4-6.2 22.6 0L480 353.4l60.7-60.7c6.2-6.2 16.4-6.2 22.6 0z',
                'vw' => 640,
                'vh' => 512
            ],
            'envelope-open' => 'M32 29q0 1.25-.88 2.13T29 32H3q-1.25 0-2.13-.88T0 29V12.56q0-1.5 1.13-2.37 2.56-2 10.3-7.63l.88-.65 1.32-.94 1.15-.63L16 0l1.22.34 1.16.63 1.3.94.88.65q7.75 5.63 10.32 7.63 1.12.87 1.12 2.37V29zm-4.13-12.31q-.25-.38-.68-.13-2.2 1.7-6.63 4.88l-.87.65-1.32.94-1.15.63L16 24q-.44 0-.94-.19t-1.03-.53-.94-.62-.97-.72-.68-.5Q7 18.25 4.8 16.56q-.44-.25-.69.13l-.56.87q-.13.13-.13.25 0 .25.2.44 2.24 1.63 6.62 4.75.12.13.81.66t1.13.84 1.15.72 1.38.6T16 26t1.28-.19 1.37-.59 1.16-.72 1.12-.84.82-.66q4.37-3.12 6.62-4.75.19-.19.19-.44 0-.12-.13-.25z',
            'envelope-open-text' => 'M11 13.5q-.44 0-.72-.28T10 12.5v-1q0-.44.28-.72t.72-.28h10q.44 0 .72.28t.28.72v1q0 .44-.28.72t-.72.28H11zm-1 5v-1q0-.44.28-.72t.72-.28h10q.44 0 .72.28t.28.72v1q0 .44-.28.72t-.72.28H11q-.44 0-.72-.28T10 18.5zm6 7.56q1.63 0 2.94-.93L32 15.69V29q0 1.25-.87 2.13T29 32H3q-1.25 0-2.12-.88T0 29V15.69l13.06 9.44q1.32.93 2.94.93zM30.88 10.2q1.12.87 1.12 2.37v.63l-6 4.37V6H6v11.56L0 13.2v-.63q0-1.5 1.13-2.37 1-.81 1.87-1.44V6q0-1.25.88-2.13T6 3h4.88q.06-.06.28-.22t.25-.19.9-.68 1.32-.94 1.15-.63T16 0t1.22.34 1.16.63 1.3.93.91.7.25.18.28.22H26q1.25 0 2.13.88T29 6v2.75q.88.69 1.88 1.44z',
            'envelope-square' => [
                'p' => 'M25 2q1.25 0 2.13.88T28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22zM11.12 16.38q.07 0 .6.43t.78.6.72.37.78.22.78-.22.72-.37.78-.6.6-.43q4.62-3.32 7.12-5.32V9.5q0-.63-.44-1.06T22.5 8h-17q-.63 0-1.06.44T4 9.5v1.56q2.5 2 7.12 5.32zM24 13.63q-1.38 1.06-5.94 4.37-.06.06-.56.44t-.78.56-.78.47-.97.4T14 20t-.97-.12-.97-.41-.75-.47-.78-.56-.6-.44Q5.38 14.69 4 13.62v8.88q0 .63.44 1.06T5.5 24h17q.62 0 1.06-.44T24 22.5v-8.88z',
                'w' => .875,
                'vw' => 28
            ],
            'envelopes-bulk' => [
                'p' => 'M128 0C110.3 0 96 14.3 96 32V224h96V192c0-35.3 28.7-64 64-64H480V32c0-17.7-14.3-32-32-32H128zM256 160c-17.7 0-32 14.3-32 32v32h96c35.3 0 64 28.7 64 64V416H576c17.7 0 32-14.3 32-32V192c0-17.7-14.3-32-32-32H256zm240 64h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H496c-8.8 0-16-7.2-16-16V240c0-8.8 7.2-16 16-16zM64 256c-17.7 0-32 14.3-32 32v13L187.1 415.9c1.4 1 3.1 1.6 4.9 1.6s3.5-.6 4.9-1.6L352 301V288c0-17.7-14.3-32-32-32H64zm288 84.8L216 441.6c-6.9 5.1-15.3 7.9-24 7.9s-17-2.8-24-7.9L32 340.8V480c0 17.7 14.3 32 32 32H320c17.7 0 32-14.3 32-32V340.8z',
                'vw' => 640,
                'vh' => 512
            ],
            'equals' => [
                'p' => 'M26 19q.81 0 1.4.6T28 21v2q0 .81-.6 1.4T26 25H2q-.81 0-1.4-.6T0 23v-2q0-.81.6-1.4T2 19h24zm0-12q.81 0 1.4.6T28 9v2q0 .81-.6 1.4T26 13H2q-.81 0-1.4-.6T0 11V9q0-.81.6-1.4T2 7h24z',
                'w' => .875,
                'vw' => 28
            ],
            'eraser' => 'M31.13 17.13 22.24 26h9q.75 0 .75.75v2.5q0 .75-.75.75H9q-1.25 0-2.13-.88l-6-6Q0 22.25 0 21t.88-2.13l16-16Q17.75 2 19 2t2.13.88l10 10q.87.87.87 2.12t-.88 2.13zm-18.94-3.94L4.44 21l5 5h7.12l4.25-4.19z',
            'ethernet' => 'M31 12q.44 0 .72.28T32 13v14q0 .44-.28.72T31 28h-5v-8h-2v8h-4v-8h-2v8h-4v-8h-2v8H8v-8H6v8H1q-.44 0-.72-.28T0 27V13q0-.44.28-.72T1 12h3V9q0-.44.28-.72T5 8h3V5q0-.44.28-.72T9 4h14q.44 0 .72.28T24 5v3h3q.44 0 .72.28T28 9v3h3z',
            'euro-sign' => [
                'p' => 'm19.44 25.88.56 2.75q.13.68-.56.87-1.88.5-3.82.5-4.8 0-8.3-2.69t-4.7-7.06H.75q-.31 0-.53-.22T0 19.5v-1.75Q0 17 .75 17h1.31q-.06-1.25.07-2.63H.75q-.31 0-.53-.21T0 13.62v-1.87Q0 11 .75 11h2.06Q4.2 7 7.66 4.5T15.63 2q1.56 0 3.25.31.25.07.4.19t.22.34 0 .41L18.75 6q-.19.69-.87.56-1.32-.25-2.38-.25-2.5 0-4.44 1.25T8.13 11h8.62q.38 0 .6.28t.15.66l-.38 1.81q-.12.63-.75.63H7.2q-.13 1.18 0 2.62h8.37q.38 0 .6.28t.15.66l-.37 1.75q-.13.56-.75.56H8.06q.5 1.19 1.25 2.19t1.72 1.69 2.16 1.1 2.5.4q1.37 0 2.81-.38.75-.12.94.63z',
                'w' => .623,
                'vw' => 20
            ],
            'exchange-alt' => 'M0 10.5v-1q0-.63.44-1.06T1.5 8H24V5q0-1 .94-1.38t1.62.32l5 5Q32 9.37 32 10t-.44 1.06l-5 5q-.68.7-1.62.32T24 15v-3H1.5q-.62 0-1.06-.44T0 10.5zM30.5 20q.63 0 1.06.44T32 21.5v1q0 .63-.44 1.06T30.5 24H8v3q0 1-.94 1.38t-1.62-.32l-5-5Q0 22.63 0 22t.44-1.06l5-5q.68-.69 1.62-.32T8 17v3h22.5z',
            'exclamation' => [
                'p' => 'M11 27q0 2.06-1.47 3.53T6 32t-3.53-1.47T1 27t1.47-3.53T6 22t3.53 1.47T11 27zM1.56 1.56q0-.62.44-1.1T3.06 0h5.88q.62 0 1.06.47t.44 1.1l-.87 17q0 .62-.44 1.02T8.06 20H3.94q-.63 0-1.06-.4t-.44-1.04z',
                'w' => .375,
                'vw' => 12,
            ],
            'exclamation-circle' => 'M31.5 16q0 6.44-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5t10.97 4.53T31.5 16zM16 19.13q-1.19 0-2.03.84T13.12 22t.85 2.03 2.03.84 2.03-.84.84-2.03-.84-2.03-2.03-.84zM13.25 8.8l.5 8.5q0 .7.75.7h3q.75 0 .75-.7l.5-8.5q0-.37-.22-.6T18 8h-4q-.31 0-.53.22t-.22.6z',
            'exclamation-triangle' => [
                'p' => 'M35.63 27.5q.8 1.5-.04 3T33 32H3q-1.75 0-2.6-1.5t-.02-3l15-26Q16.25 0 18 0t2.63 1.5zM18 22.12q-1.19 0-2.03.85T15.12 25t.85 2.03 2.03.84 2.03-.84.84-2.03-.84-2.03-2.03-.84zm-2.75-10.3.5 8.5q0 .68.75.68h3q.75 0 .75-.69l.5-8.5q0-.37-.22-.6T20 11h-4q-.31 0-.53.22t-.22.6z',
                'w' => 1.125,
                'vw' => 36
            ],
            'expand' => [
                'p' => 'M0 11.25V3.5q0-.63.44-1.06T1.5 2h7.75q.75 0 .75.75v2.5Q10 6 9.25 6H4v5.25q0 .75-.75.75H.75Q0 12 0 11.25zm18-8.5q0-.75.75-.75h7.75q.63 0 1.06.44T28 3.5v7.75q0 .75-.75.75h-2.5q-.75 0-.75-.75V6h-5.25Q18 6 18 5.25v-2.5zM27.25 20q.75 0 .75.75v7.75q0 .63-.44 1.06T26.5 30h-7.75q-.75 0-.75-.75v-2.5q0-.75.75-.75H24v-5.25q0-.75.75-.75h2.5zM10 29.25q0 .75-.75.75H1.5q-.63 0-1.06-.44T0 28.5v-7.75Q0 20 .75 20h2.5q.75 0 .75.75V26h5.25q.75 0 .75.75v2.5z',
                'w' => .875,
                'vw' => 28
            ],
            'expand-alt' => [
                'p' => 'm13.3 19.7-5.8 5.8 2.06 1.94A1.5 1.5 0 0 1 8.5 30h-7A1.5 1.5 0 0 1 0 28.5v-7c0-1.34 1.61-2 2.56-1.06L4.5 22.5l5.8-5.8a1 1 0 0 1 1.4 0l1.6 1.6a1 1 0 0 1 0 1.4Zm1.4-7.4 5.8-5.8-2.06-1.94A1.5 1.5 0 0 1 19.5 2h7c.83 0 1.5.67 1.5 1.5v7c0 1.34-1.61 2-2.56 1.06L23.5 9.5l-5.8 5.8a1 1 0 0 1-1.4 0l-1.6-1.6a1 1 0 0 1 0-1.4Zm0 0',
                'vw' => 28
            ],
            'expand-arrows-alt' => [
                'p' => 'M28 21.5v7q0 .63-.44 1.06T26.5 30h-7q-1 0-1.38-.94t.32-1.62l2.25-2.25-6.7-6.69-6.68 6.69 2.25 2.25q.69.69.31 1.62T8.5 30h-7q-.63 0-1.06-.44T0 28.5v-7q0-1 .94-1.38t1.62.32l2.25 2.25L11.5 16 4.8 9.31l-2.25 2.25q-.69.69-1.62.31T0 10.5v-7q0-.63.44-1.06T1.5 2h7q1 0 1.37.94t-.3 1.62L7.3 6.81 14 13.5l6.69-6.69-2.25-2.25q-.7-.69-.32-1.62T19.5 2h7q.62 0 1.06.44T28 3.5v7q0 1-.94 1.38t-1.62-.32l-2.25-2.25-6.7 6.69 6.7 6.69 2.25-2.25q.68-.69 1.62-.32T28 21.5z',
                'w' => .875,
                'vw' => 28
            ],
            'explosion' => [
                'p' => 'M499.6 11.3c6.7-10.7 20.5-14.5 31.7-8.5s15.8 19.5 10.6 31L404.8 338.6c2.2 2.3 4.3 4.7 6.3 7.1l97.2-54.7c10.5-5.9 23.6-3.1 30.9 6.4s6.3 23-2.2 31.5l-87 87H378.5c-13.2-37.3-48.7-64-90.5-64s-77.4 26.7-90.5 64H117.8L42.3 363.7c-9.7-6.7-13.1-19.6-7.9-30.3s17.4-15.9 28.7-12.4l97.2 30.4c3-3.9 6.1-7.7 9.4-11.3L107.4 236.3c-6.1-10.1-3.9-23.1 5.1-30.7s22.2-7.5 31.1 .1L246 293.6c1.5-.4 3-.8 4.5-1.1l13.6-142.7c1.2-12.3 11.5-21.7 23.9-21.7s22.7 9.4 23.9 21.7l13.5 141.9L499.6 11.3zM64 448v0H512v0h32c17.7 0 32 14.3 32 32s-14.3 32-32 32H32c-17.7 0-32-14.3-32-32s14.3-32 32-32H64zM288 0c13.3 0 24 10.7 24 24V72c0 13.3-10.7 24-24 24s-24-10.7-24-24V24c0-13.3 10.7-24 24-24z',
                'vw' => 576,
                'vh' => 512
            ],
            'external-link-alt' => [
                'p' => 'M36 1.5v8q0 1-1 1.4t-1.6-.3l-2.2-2.3L16 23.6q-.4.4-1 .4t-1.1-.4L12.4 22q-.4-.4-.4-1t.4-1.1L27.7 4.8l-2.3-2.2q-.6-.7-.3-1.7t1.4-.9h8q.6 0 1 .4t.5 1.1zM25.4 16.9q.7-.7 1.7-.3T28 18v11q0 1.3-.9 2.1T25 32H3q-1.3 0-2.1-.9T0 29V7q0-1.3.9-2.1T3 4h18.5q1 0 1.4 1t-.3 1.6l-1 1q-.5.4-1.1.4H4v20h20v-9q0-.6.4-1z',
                'w' => 1.125,
                'vw' => 36
            ],
            'external-link-square-alt' => [
                'p' => 'M28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22q1.25 0 2.13.88T28 5zm-5.5 1h-7q-1 0-1.38.94t.32 1.62l2 2-12.19 12.2q-.56.55 0 1.05l1.94 1.94q.5.56 1.06 0l12.19-12.19 2 2q.69.7 1.62.32T24 14.5v-7q0-.62-.44-1.06T22.5 6z',
                'w' => .875,
                'vw' => 28
            ],
            'eye' => [
                'p' => 'M35.81 15.06q.2.44.2.94t-.2.94q-2.62 5-7.37 8.03T18 28q-3.75 0-7.22-1.4t-6.15-3.88-4.44-5.78q-.2-.44-.2-.94t.2-.94q2.62-5 7.37-8.03T18 4t10.44 3.03 7.37 8.03zM18 25q3.75 0 6.38-2.63T27 16t-2.63-6.38T18 7t-6.38 2.63T9 16t2.63 6.38T18 25zm0-15q2.5 0 4.25 1.75T24 15.97t-1.75 4.25T18.03 22t-4.25-1.78T12 16q0-.81.25-1.56Q13 15 14 15q1.25 0 2.13-.88T17 12q0-1-.56-1.75Q17.19 10 18 10z',
                'w' => 1.125,
                'vw' => 36
            ],
            'eye-dropper' => 'm3.19 20.81 7.87-7.87 8 8-7.87 7.87q-1.19 1.2-2.88 1.2H5.5L2 32l-2-2 2-3.5v-2.82Q2 22 3.19 20.8zM30.25 1.75Q32 3.5 32 6t-1.75 4.25l-4.81 4.81.81.82q.44.43.44 1.06T26.25 18l-2.56 2.56q-.44.44-1.07.44t-1.06-.44L11.44 10.44Q11 10 11 9.38t.44-1.07L14 5.75q.44-.44 1.06-.44t1.07.44l.8.81 4.82-4.8Q23.5 0 26 0t4.25 1.75z',
            'eye-low-vision' => [
                'p' => 'M38.8 5.1C28.4-3.1 13.3-1.2 5.1 9.2S-1.2 34.7 9.2 42.9l592 464c10.4 8.2 25.5 6.3 33.7-4.1s6.3-25.5-4.1-33.7L525.6 386.7c39.6-40.6 66.4-86.1 79.9-118.4c3.3-7.9 3.3-16.7 0-24.6c-14.9-35.7-46.2-87.7-93-131.1C465.5 68.8 400.8 32 320 32c-68.2 0-125 26.3-169.3 60.8L38.8 5.1zM223 149.5c48.6-44.3 123-50.8 179.3-11.7c60.8 42.4 78.9 123.2 44.2 186.9L408 294.5c8.4-19.3 10.6-41.4 4.8-63.3c-11.1-41.5-47.8-69.4-88.6-71.1c-5.8-.2-9.2 6.1-7.4 11.7c2.1 6.4 3.3 13.2 3.3 20.3c0 10.2-2.4 19.8-6.6 28.3L223 149.5zm223.1 298L83.1 161.5c-11 14.4-20.5 28.7-28.4 42.2l339 265.7c18.7-5.5 36.2-13 52.6-21.8zM34.5 268.3c14.9 35.7 46.2 87.7 93 131.1C174.5 443.2 239.2 480 320 480c3.1 0 6.1-.1 9.2-.2L33.1 247.8c-1.8 6.8-1.3 14 1.4 20.5z',
                'vw' => 640,
                'vh' => 512
            ],
            'eye-slash' => [
                'p' => 'M20 25q.81 0 1.63-.19l3.25 2.57Q22.3 28 20 28q-5.69 0-10.44-3.03T2.2 16.94Q2 16.5 2 16t.2-.94q.87-1.62 2.31-3.44l6.56 5.07q.2 2.31 1.4 4.22t3.23 3T20 25zm19.63 3.63q.37.3.37.8 0 .32-.19.63l-1.25 1.57q-.31.37-.81.37-.31 0-.56-.19L.37 3.38Q0 3.06 0 2.56q0-.31.19-.62L1.44.37q.3-.37.8-.37.32 0 .57.19l8 6.18q1.13-.68 2.78-1.21t3.38-.85T20 4q5.69 0 10.44 3.03t7.37 8.03q.19.44.19.94t-.19.94q-.81 1.5-2.31 3.37t-2.81 3zm-11.5-8.88Q29 17.94 29 16q0-3.75-2.63-6.38T20 7q-1.31 0-3 .6t-2.69 1.46l4.56 3.57Q19 12.3 19 12q0-1-.56-1.75.81-.25 1.62-.25 2.44 0 4.19 1.75T26 15.94V16q0 .94-.31 1.88z',
                'w' => 1.25,
                'vw' => 40
            ],
            'f' => [
                'p' => 'M64 32C28.7 32 0 60.7 0 96V256 448c0 17.7 14.3 32 32 32s32-14.3 32-32V288H224c17.7 0 32-14.3 32-32s-14.3-32-32-32H64V96H288c17.7 0 32-14.3 32-32s-14.3-32-32-32H64z',
                'vw' => 320,
                'vh' => 512
            ],
            'face-angry' => [
                'p' => 'M0 256a256 256 0 1 1 512 0A256 256 0 1 1 0 256zM338.7 395.9c6.6-5.9 7.1-16 1.2-22.6C323.8 355.4 295.7 336 256 336s-67.8 19.4-83.9 37.3c-5.9 6.6-5.4 16.7 1.2 22.6s16.7 5.4 22.6-1.2c11.7-13 31.6-26.7 60.1-26.7s48.4 13.7 60.1 26.7c5.9 6.6 16 7.1 22.6 1.2zM176.4 272c17.7 0 32-14.3 32-32c0-1.5-.1-3-.3-4.4l10.9 3.6c8.4 2.8 17.4-1.7 20.2-10.1s-1.7-17.4-10.1-20.2l-96-32c-8.4-2.8-17.4 1.7-20.2 10.1s1.7 17.4 10.1 20.2l30.7 10.2c-5.8 5.8-9.3 13.8-9.3 22.6c0 17.7 14.3 32 32 32zm192-32c0-8.9-3.6-17-9.5-22.8l30.2-10.1c8.4-2.8 12.9-11.9 10.1-20.2s-11.9-12.9-20.2-10.1l-96 32c-8.4 2.8-12.9 11.9-10.1 20.2s11.9 12.9 20.2 10.1l11.7-3.9c-.2 1.5-.3 3.1-.3 4.7c0 17.7 14.3 32 32 32s32-14.3 32-32z',
                'vw' => 512,
                'vh' => 512
            ],
            'face-dizzy' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zm0-224a64 64 0 1 1 0 128 64 64 0 1 1 0-128zM100.7 132.7c6.2-6.2 16.4-6.2 22.6 0L160 169.4l36.7-36.7c6.2-6.2 16.4-6.2 22.6 0s6.2 16.4 0 22.6L182.6 192l36.7 36.7c6.2 6.2 6.2 16.4 0 22.6s-16.4 6.2-22.6 0L160 214.6l-36.7 36.7c-6.2 6.2-16.4 6.2-22.6 0s-6.2-16.4 0-22.6L137.4 192l-36.7-36.7c-6.2-6.2-6.2-16.4 0-22.6zm192 0c6.2-6.2 16.4-6.2 22.6 0L352 169.4l36.7-36.7c6.2-6.2 16.4-6.2 22.6 0s6.2 16.4 0 22.6L374.6 192l36.7 36.7c6.2 6.2 6.2 16.4 0 22.6s-16.4 6.2-22.6 0L352 214.6l-36.7 36.7c-6.2 6.2-16.4 6.2-22.6 0s-6.2-16.4 0-22.6L329.4 192l-36.7-36.7c-6.2-6.2-6.2-16.4 0-22.6z',
                'vw' => 512,
                'vh' => 512
            ],
            'face-flushed' => [
                'p' => 'M0 256a256 256 0 1 1 512 0A256 256 0 1 1 0 256zM176 384c0 8.8 7.2 16 16 16H320c8.8 0 16-7.2 16-16s-7.2-16-16-16H192c-8.8 0-16 7.2-16 16zm-16-88a72 72 0 1 0 0-144 72 72 0 1 0 0 144zm264-72a72 72 0 1 0 -144 0 72 72 0 1 0 144 0zm-288 0a24 24 0 1 1 48 0 24 24 0 1 1 -48 0zm192 0a24 24 0 1 1 48 0 24 24 0 1 1 -48 0z',
                'vw' => 512,
                'vh' => 512
            ],
            'face-frown' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM159.3 388.7c-2.6 8.4-11.6 13.2-20 10.5s-13.2-11.6-10.5-20C145.2 326.1 196.3 288 256 288s110.8 38.1 127.3 91.3c2.6 8.4-2.1 17.4-10.5 20s-17.4-2.1-20-10.5C340.5 349.4 302.1 320 256 320s-84.5 29.4-96.7 68.7zM144.4 208a32 32 0 1 1 64 0 32 32 0 1 1 -64 0zm192-32a32 32 0 1 1 0 64 32 32 0 1 1 0-64z',
                'vw' => 512,
                'vh' => 512
            ],
            'face-frown-open' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM176.4 176a32 32 0 1 1 0 64 32 32 0 1 1 0-64zm128 32a32 32 0 1 1 64 0 32 32 0 1 1 -64 0zm-122 174.5c-12.4 5.2-26.5-4.1-21.1-16.4c16-36.6 52.4-62.1 94.8-62.1s78.8 25.6 94.8 62.1c5.4 12.3-8.7 21.6-21.1 16.4c-22.4-9.5-47.4-14.8-73.7-14.8s-51.3 5.3-73.7 14.8z',
                'vw' => 512,
                'vh' => 512
            ],
            'face-grimace' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zm96-112h-8V360l55.3 0c-3.8 22.7-23.6 40-47.3 40zm47.3-56L344 344V304h8c23.8 0 43.5 17.3 47.3 40zM328 344H264V304h64v40zm0 56H264V360h64v40zm-80-96v40l-64 0V304h64zm0 56v40H184V360l64 0zm-80-16H112.7c3.8-22.7 23.6-40 47.3-40h8v40zm0 56h-8c-23.8 0-43.5-17.3-47.3-40H168v40zM144.4 208a32 32 0 1 1 64 0 32 32 0 1 1 -64 0zm192-32a32 32 0 1 1 0 64 32 32 0 1 1 0-64z',
                'vw' => 512,
                'vh' => 512
            ],
            'face-grin' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM388.1 312.8c12.3-3.8 24.3 6.9 19.3 18.7C382.4 390.6 324.2 432 256.3 432s-126.2-41.4-151.1-100.5c-5-11.8 7-22.5 19.3-18.7c39.7 12.2 84.5 19 131.8 19s92.1-6.8 131.8-19zM144.4 208a32 32 0 1 1 64 0 32 32 0 1 1 -64 0zm192-32a32 32 0 1 1 0 64 32 32 0 1 1 0-64z',
                'vw' => 512,
                'vh' => 512
            ],
            'face-grin-beam' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM388.1 312.8c12.3-3.8 24.3 6.9 19.3 18.7C382.4 390.6 324.2 432 256.3 432s-126.2-41.4-151.1-100.5c-5-11.8 7-22.5 19.3-18.7c39.7 12.2 84.5 19 131.8 19s92.1-6.8 131.8-19zm-170.5-84l0 0 0 0-.2-.2c-.2-.2-.4-.5-.7-.9c-.6-.8-1.6-2-2.8-3.4c-2.5-2.8-6-6.6-10.2-10.3c-8.8-7.8-18.8-14-27.7-14s-18.9 6.2-27.7 14c-4.2 3.7-7.7 7.5-10.2 10.3c-1.2 1.4-2.2 2.6-2.8 3.4c-.3 .4-.6 .7-.7 .9l-.2 .2 0 0 0 0 0 0c-2.1 2.8-5.7 3.9-8.9 2.8s-5.5-4.1-5.5-7.6c0-17.9 6.7-35.6 16.6-48.8c9.8-13 23.9-23.2 39.4-23.2s29.6 10.2 39.4 23.2c9.9 13.2 16.6 30.9 16.6 48.8c0 3.4-2.2 6.5-5.5 7.6s-6.9 0-8.9-2.8l0 0 0 0zm160 0l0 0-.2-.2c-.2-.2-.4-.5-.7-.9c-.6-.8-1.6-2-2.8-3.4c-2.5-2.8-6-6.6-10.2-10.3c-8.8-7.8-18.8-14-27.7-14s-18.9 6.2-27.7 14c-4.2 3.7-7.7 7.5-10.2 10.3c-1.2 1.4-2.2 2.6-2.8 3.4c-.3 .4-.6 .7-.7 .9l-.2 .2 0 0 0 0 0 0c-2.1 2.8-5.7 3.9-8.9 2.8s-5.5-4.1-5.5-7.6c0-17.9 6.7-35.6 16.6-48.8c9.8-13 23.9-23.2 39.4-23.2s29.6 10.2 39.4 23.2c9.9 13.2 16.6 30.9 16.6 48.8c0 3.4-2.2 6.5-5.5 7.6s-6.9 0-8.9-2.8l0 0 0 0 0 0z',
                'vw' => 512,
                'vh' => 512
            ],
            'face-grin-beam-sweat' => [
                'p' => 'M476.8 126.3c-4.1 1.1-8.4 1.7-12.8 1.7c-26.5 0-48-21-48-47c0-5 1.8-11.3 4.6-18.1c.3-.7 .6-1.4 .9-2.1c9-20.2 26.5-44.9 36-57.5c3.2-4.4 9.6-4.4 12.8 0C483.4 20.6 512 61 512 81c0 21.7-14.9 39.8-35.2 45.3zM256 0c51.4 0 99.3 15.2 139.4 41.2c-1.5 3.1-3 6.2-4.3 9.3c-3.4 8-7.1 19-7.1 30.5c0 44.3 36.6 79 80 79c9.6 0 18.8-1.7 27.4-4.8c13.3 30.9 20.6 65 20.6 100.8c0 141.4-114.6 256-256 256S0 397.4 0 256S114.6 0 256 0zM383.8 317.8C345.3 329.4 301.9 336 256 336s-89.3-6.6-127.8-18.2c-12.3-3.7-24.3 7-19.2 18.7c24.5 56.9 81.1 96.7 147 96.7s122.5-39.8 147-96.7c5.1-11.8-6.9-22.4-19.2-18.7zm-166.2-89l0 0 0 0c2.1 2.8 5.7 3.9 8.9 2.8s5.5-4.1 5.5-7.6c0-17.9-6.7-35.6-16.6-48.8c-9.8-13-23.9-23.2-39.4-23.2s-29.6 10.2-39.4 23.2C126.7 188.4 120 206.1 120 224c0 3.4 2.2 6.5 5.5 7.6s6.9 0 8.9-2.8l0 0 0 0 0 0 .2-.2c.2-.2 .4-.5 .7-.9c.6-.8 1.6-2 2.8-3.4c2.5-2.8 6-6.6 10.2-10.3c8.8-7.8 18.8-14 27.7-14s18.9 6.2 27.7 14c4.2 3.7 7.7 7.5 10.2 10.3c1.2 1.4 2.2 2.6 2.8 3.4c.3 .4 .6 .7 .7 .9l.2 .2 0 0 0 0zm160 0l0 0 0 0 0 0c2.1 2.8 5.7 3.9 8.9 2.8s5.5-4.1 5.5-7.6c0-17.9-6.7-35.6-16.6-48.8c-9.8-13-23.9-23.2-39.4-23.2s-29.6 10.2-39.4 23.2C286.7 188.4 280 206.1 280 224c0 3.4 2.2 6.5 5.5 7.6s6.9 0 8.9-2.8l0 0 0 0 0 0 .2-.2c.2-.2 .4-.5 .7-.9c.6-.8 1.6-2 2.8-3.4c2.5-2.8 6-6.6 10.2-10.3c8.8-7.8 18.8-14 27.7-14s18.9 6.2 27.7 14c4.2 3.7 7.7 7.5 10.2 10.3c1.2 1.4 2.2 2.6 2.8 3.4c.3 .4 .6 .7 .7 .9l.2 .2 0 0z',
                'vw' => 512,
                'vh' => 512
            ],
            'face-grin-hearts' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM388.1 312.8c12.3-3.8 24.3 6.9 19.3 18.7C382.4 390.6 324.2 432 256.3 432s-126.2-41.4-151.1-100.5c-5-11.8 7-22.5 19.3-18.7c39.7 12.2 84.5 19 131.8 19s92.1-6.8 131.8-19zM199.3 129.1c17.8 4.8 28.4 23.1 23.6 40.8l-17.4 65c-2.3 8.5-11.1 13.6-19.6 11.3l-65.1-17.4c-17.8-4.8-28.4-23.1-23.6-40.8s23.1-28.4 40.8-23.6l16.1 4.3 4.3-16.1c4.8-17.8 23.1-28.4 40.8-23.6zm154.3 23.6l4.3 16.1 16.1-4.3c17.8-4.8 36.1 5.8 40.8 23.6s-5.8 36.1-23.6 40.8l-65.1 17.4c-8.5 2.3-17.3-2.8-19.6-11.3l-17.4-65c-4.8-17.8 5.8-36.1 23.6-40.8s36.1 5.8 40.9 23.6z',
                'vw' => 512,
                'vh' => 512
            ],
            'face-grin-squint' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM388.1 312.8c12.3-3.8 24.3 6.9 19.3 18.7C382.4 390.6 324.2 432 256.3 432s-126.2-41.4-151.1-100.5c-5-11.8 7-22.5 19.3-18.7c39.7 12.2 84.5 19 131.8 19s92.1-6.8 131.8-19zM133.5 146.7l89.9 47.9c10.7 5.7 10.7 21.1 0 26.8l-89.9 47.9c-7.9 4.2-17.5-1.5-17.5-10.5c0-2.8 1-5.5 2.8-7.6l36-43.2-36-43.2c-1.8-2.1-2.8-4.8-2.8-7.6c0-9 9.6-14.7 17.5-10.5zM396 157.1c0 2.8-1 5.5-2.8 7.6l-36 43.2 36 43.2c1.8 2.1 2.8 4.8 2.8 7.6c0 9-9.6 14.7-17.5 10.5l-89.9-47.9c-10.7-5.7-10.7-21.1 0-26.8l89.9-47.9c7.9-4.2 17.5 1.5 17.5 10.5z',
                'vw' => 512,
                'vh' => 512
            ],
            'face-grin-squint-tears' => [
                'p' => 'M426.8 14.2C446-5 477.5-4.6 497.1 14.9s20 51 .7 70.3c-6.8 6.8-21.4 12.4-37.4 16.7c-16.3 4.4-34.1 7.5-46.3 9.3c-1.6 .2-3.1 .5-4.6 .6c-4.9 .8-9.1-2.8-9.5-7.4c-.1-.7 0-1.4 .1-2.1c1.6-11.2 4.6-29.6 9-47c.3-1.3 .7-2.6 1-3.9c4.3-15.9 9.8-30.5 16.7-37.4zm-44.7 19c-1.5 4.8-2.9 9.6-4.1 14.3c-4.8 18.9-8 38.5-9.7 50.3c-4 26.8 18.9 49.7 45.7 45.8c11.9-1.6 31.5-4.8 50.4-9.7c4.7-1.2 9.5-2.5 14.3-4.1C534.2 227.5 520.2 353.8 437 437c-83.2 83.2-209.5 97.2-307.2 41.8c1.5-4.8 2.8-9.6 4-14.3c4.8-18.9 8-38.5 9.7-50.3c4-26.8-18.9-49.7-45.7-45.8c-11.9 1.6-31.5 4.8-50.4 9.7c-4.7 1.2-9.5 2.5-14.3 4.1C-22.2 284.5-8.2 158.2 75 75C158.2-8.3 284.5-22.2 382.2 33.2zM51.5 410.1c18.5-5 38.8-8.3 50.9-10c.4-.1 .7-.1 1-.1c5.1-.2 9.2 4.3 8.4 9.6c-1.7 12.1-5 32.4-10 50.9C97.6 476.4 92 491 85.2 497.8C66 517 34.5 516.6 14.9 497.1s-20-51-.7-70.3c6.8-6.8 21.4-12.4 37.4-16.7zM416.9 209c-4.7-11.9-20.8-11-26.8 .3c-19 35.5-45 70.8-77.5 103.3S244.8 371.1 209.3 390c-11.3 6-12.2 22.1-.3 26.8c57.6 22.9 125.8 11 172.3-35.5s58.4-114.8 35.5-172.3zM87.1 285.1c2 2 4.6 3.2 7.3 3.4l56.1 5.1 5.1 56.1c.3 2.8 1.5 5.4 3.4 7.3c6.3 6.3 17.2 3.6 19.8-4.9l29.7-97.4c3.5-11.6-7.3-22.5-19-19L92 265.3c-8.6 2.6-11.3 13.4-4.9 19.8zM265.3 92l-29.7 97.4c-3.5 11.6 7.3 22.5 19 19l97.4-29.7c8.6-2.6 11.3-13.4 4.9-19.8c-2-2-4.6-3.2-7.3-3.4l-56.1-5.1-5.1-56.1c-.3-2.8-1.5-5.4-3.4-7.3c-6.3-6.3-17.2-3.6-19.8 4.9z',
                'vw' => 512,
                'vh' => 512
            ],
            'face-grin-stars' => [
                'p' => 'M0 256a256 256 0 1 1 512 0A256 256 0 1 1 0 256zm407.4 75.5c5-11.8-7-22.5-19.3-18.7c-39.7 12.2-84.5 19-131.8 19s-92.1-6.8-131.8-19c-12.3-3.8-24.3 6.9-19.3 18.7c25 59.1 83.2 100.5 151.1 100.5s126.2-41.4 151.1-100.5zM160 120c-3.1 0-5.9 1.8-7.2 4.6l-16.6 34.7-38.1 5c-3.1 .4-5.6 2.5-6.6 5.5s-.1 6.2 2.1 8.3l27.9 26.5-7 37.8c-.6 3 .7 6.1 3.2 7.9s5.8 2 8.5 .6L160 232.5l33.8 18.3c2.7 1.5 6 1.3 8.5-.6s3.7-4.9 3.2-7.9l-7-37.8L226.4 178c2.2-2.1 3.1-5.3 2.1-8.3s-3.5-5.1-6.6-5.5l-38.1-5-16.6-34.7c-1.3-2.8-4.1-4.6-7.2-4.6zm192 0c-3.1 0-5.9 1.8-7.2 4.6l-16.6 34.7-38.1 5c-3.1 .4-5.6 2.5-6.6 5.5s-.1 6.2 2.1 8.3l27.9 26.5-7 37.8c-.6 3 .7 6.1 3.2 7.9s5.8 2 8.5 .6L352 232.5l33.8 18.3c2.7 1.5 6 1.3 8.5-.6s3.7-4.9 3.2-7.9l-7-37.8L418.4 178c2.2-2.1 3.1-5.3 2.1-8.3s-3.5-5.1-6.6-5.5l-38.1-5-16.6-34.7c-1.3-2.8-4.1-4.6-7.2-4.6z',
                'vw' => 512,
                'vh' => 512
            ],
            'face-grin-tears' => [
                'p' => 'M548.6 371.4C506.4 454.8 419.9 512 320 512s-186.4-57.2-228.6-140.6c4.5-2.9 8.7-6.3 12.7-10.3c8.1-8.1 13.2-18.6 16.5-26.6c3.6-8.8 6.5-18.4 8.8-27.5c4.6-18.2 7.7-37 9.3-48.2c3.9-26.5-18.8-49.2-45.2-45.4c-6.8 .9-16.2 2.4-26.6 4.4C85.3 94.5 191.6 0 320 0S554.7 94.5 573.2 217.7c-10.3-2-19.8-3.5-26.6-4.4c-26.5-3.9-49.2 18.8-45.2 45.4c1.6 11.3 4.6 30 9.3 48.2c2.3 9.1 5.2 18.8 8.8 27.5c3.3 8.1 8.4 18.5 16.5 26.6c3.9 3.9 8.2 7.4 12.7 10.3zM107 254.1c-3.1 21.5-11.4 70.2-25.5 84.4c-.9 1-1.9 1.8-2.9 2.7C60 356.7 32 355.5 14.3 337.7c-18.7-18.7-19.1-48.8-.7-67.2c8.6-8.6 30.1-15.1 50.5-19.6c13-2.8 25.5-4.8 33.9-6c5.4-.8 9.9 3.7 9 9zm454.5 87.1c-.8-.6-1.5-1.3-2.3-2c-.2-.2-.5-.4-.7-.7c-14.1-14.1-22.5-62.9-25.5-84.4c-.8-5.4 3.7-9.9 9-9c1 .1 2.2 .3 3.3 .5c8.2 1.2 19.2 3 30.6 5.5c20.4 4.4 41.9 10.9 50.5 19.6c18.4 18.4 18 48.5-.7 67.2c-17.7 17.7-45.7 19-64.2 3.4zm-90.1-9.7c5-11.8-7-22.5-19.3-18.7c-39.7 12.2-84.4 19-131.8 19s-92.1-6.8-131.8-19c-12.3-3.8-24.3 6.9-19.3 18.7c25 59.1 83.2 100.5 151.1 100.5s126.2-41.4 151.1-100.5zM281.6 228.8l0 0 0 0 0 0c2.1 2.8 5.7 3.9 8.9 2.8s5.5-4.1 5.5-7.6c0-17.9-6.7-35.6-16.6-48.8c-9.8-13-23.9-23.2-39.4-23.2s-29.6 10.2-39.4 23.2C190.7 188.4 184 206.1 184 224c0 3.4 2.2 6.5 5.5 7.6s6.9 0 8.9-2.8l0 0 0 0 0 0 .2-.2c.2-.2 .4-.5 .7-.9c.6-.8 1.6-2 2.8-3.4c2.5-2.8 6-6.6 10.2-10.3c8.8-7.8 18.8-14 27.7-14s18.9 6.2 27.7 14c4.2 3.7 7.7 7.5 10.2 10.3c1.2 1.4 2.2 2.6 2.8 3.4c.3 .4 .6 .7 .7 .9l.2 .2 0 0zm160 0l0 0 0 0c2.1 2.8 5.7 3.9 8.9 2.8s5.5-4.1 5.5-7.6c0-17.9-6.7-35.6-16.6-48.8c-9.8-13-23.9-23.2-39.4-23.2s-29.6 10.2-39.4 23.2C350.7 188.4 344 206.1 344 224c0 3.4 2.2 6.5 5.5 7.6s6.9 0 8.9-2.8l0 0 0 0 0 0 .2-.2c.2-.2 .4-.5 .7-.9c.6-.8 1.6-2 2.8-3.4c2.5-2.8 6-6.6 10.2-10.3c8.8-7.8 18.8-14 27.7-14s18.9 6.2 27.7 14c4.2 3.7 7.7 7.5 10.2 10.3c1.2 1.4 2.2 2.6 2.8 3.4c.3 .4 .6 .7 .7 .9l.2 .2 0 0 0 0z',
                'vw' => 640,
                'vh' => 512
            ],
            'face-grin-tongue' => [
                'p' => 'M0 256C0 368.9 73.1 464.7 174.5 498.8C165.3 484 160 466.6 160 448V400.7c-24-17.5-43.1-41.4-54.8-69.2c-5-11.8 7-22.5 19.3-18.7c39.7 12.2 84.5 19 131.8 19s92.1-6.8 131.8-19c12.3-3.8 24.3 6.9 19.3 18.7c-11.8 28-31.1 52-55.4 69.6V448c0 18.6-5.3 36-14.5 50.8C438.9 464.7 512 368.9 512 256C512 114.6 397.4 0 256 0S0 114.6 0 256zm176.4-80a32 32 0 1 1 0 64 32 32 0 1 1 0-64zm128 32a32 32 0 1 1 64 0 32 32 0 1 1 -64 0zM320 448V402.6c0-14.7-11.9-26.6-26.6-26.6h-2c-11.3 0-21.1 7.9-23.6 18.9c-2.8 12.6-20.8 12.6-23.6 0c-2.5-11.1-12.3-18.9-23.6-18.9h-2c-14.7 0-26.6 11.9-26.6 26.6V448c0 35.3 28.7 64 64 64s64-28.7 64-64z',
                'vw' => 512,
                'vh' => 512
            ],
            'face-grin-tongue-squint' => [
                'p' => 'M0 256C0 368.9 73.1 464.7 174.5 498.8C165.3 484 160 466.6 160 448V400.7c-24-17.5-43.1-41.4-54.8-69.2c-5-11.8 7-22.5 19.3-18.7c39.7 12.2 84.5 19 131.8 19s92.1-6.8 131.8-19c12.3-3.8 24.3 6.9 19.3 18.7c-11.8 28-31.1 52-55.4 69.6V448c0 18.6-5.3 36-14.5 50.8C438.9 464.7 512 368.9 512 256C512 114.6 397.4 0 256 0S0 114.6 0 256zM116 141.1c0-9 9.6-14.7 17.5-10.5l89.9 47.9c10.7 5.7 10.7 21.1 0 26.8l-89.9 47.9c-7.9 4.2-17.5-1.5-17.5-10.5c0-2.8 1-5.5 2.8-7.6l36-43.2-36-43.2c-1.8-2.1-2.8-4.8-2.8-7.6zm262.5-10.5c7.9-4.2 17.5 1.5 17.5 10.5c0 2.8-1 5.5-2.8 7.6l-36 43.2 36 43.2c1.8 2.1 2.8 4.8 2.8 7.6c0 9-9.6 14.7-17.5 10.5l-89.9-47.9c-10.7-5.7-10.7-21.1 0-26.8l89.9-47.9zM320 448V402.6c0-14.7-11.9-26.6-26.6-26.6h-2c-11.3 0-21.1 7.9-23.6 18.9c-2.8 12.6-20.8 12.6-23.6 0c-2.5-11.1-12.3-18.9-23.6-18.9h-2c-14.7 0-26.6 11.9-26.6 26.6V448c0 35.3 28.7 64 64 64s64-28.7 64-64z',
                'vw' => 512,
                'vh' => 512
            ],
            'face-grin-tongue-wink' => [
                'p' => 'M174.5 498.8C73.1 464.7 0 368.9 0 256C0 114.6 114.6 0 256 0S512 114.6 512 256c0 112.9-73.1 208.7-174.5 242.8C346.7 484 352 466.6 352 448V401.1c24.3-17.5 43.6-41.6 55.4-69.6c5-11.8-7-22.5-19.3-18.7c-39.7 12.2-84.5 19-131.8 19s-92.1-6.8-131.8-19c-12.3-3.8-24.3 6.9-19.3 18.7c11.7 27.8 30.8 51.7 54.8 69.2V448c0 18.6 5.3 36 14.5 50.8zm20.7-265.2c5.3 7.1 15.3 8.5 22.4 3.2s8.5-15.3 3.2-22.4c-30.4-40.5-91.2-40.5-121.6 0c-5.3 7.1-3.9 17.1 3.2 22.4s17.1 3.9 22.4-3.2c17.6-23.5 52.8-23.5 70.4 0zM336 272a64 64 0 1 0 0-128 64 64 0 1 0 0 128zM320 402.6V448c0 35.3-28.7 64-64 64s-64-28.7-64-64V402.6c0-14.7 11.9-26.6 26.6-26.6h2c11.3 0 21.1 7.9 23.6 18.9c2.8 12.6 20.8 12.6 23.6 0c2.5-11.1 12.3-18.9 23.6-18.9h2c14.7 0 26.6 11.9 26.6 26.6zM336 184a24 24 0 1 1 0 48 24 24 0 1 1 0-48z',
                'vw' => 512,
                'vh' => 512
            ],
            'face-grin-wide' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM388.1 312.8c12.3-3.8 24.3 6.9 19.3 18.7C382.4 390.6 324.2 432 256.3 432s-126.2-41.4-151.1-100.5c-5-11.8 7-22.5 19.3-18.7c39.7 12.2 84.5 19 131.8 19s92.1-6.8 131.8-19zM208 192c0 35.3-14.3 64-32 64s-32-28.7-32-64s14.3-64 32-64s32 28.7 32 64zm128 64c-17.7 0-32-28.7-32-64s14.3-64 32-64s32 28.7 32 64s-14.3 64-32 64z',
                'vw' => 512,
                'vh' => 512
            ],
            'face-grin-wink' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM388.1 312.8c12.3-3.8 24.3 6.9 19.3 18.7C382.4 390.6 324.2 432 256.3 432s-126.2-41.4-151.1-100.5c-5-11.8 7-22.5 19.3-18.7c39.7 12.2 84.5 19 131.8 19s92.1-6.8 131.8-19zm-16.9-79.2c-17.6-23.5-52.8-23.5-70.4 0c-5.3 7.1-15.3 8.5-22.4 3.2s-8.5-15.3-3.2-22.4c30.4-40.5 91.2-40.5 121.6 0c5.3 7.1 3.9 17.1-3.2 22.4s-17.1 3.9-22.4-3.2zM176.4 176a32 32 0 1 1 0 64 32 32 0 1 1 0-64z',
                'vw' => 512,
                'vh' => 512
            ],
            'face-kiss' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zm48.7-198.3c4.3 5.1 7.3 11.4 7.3 18.3s-3.1 13.2-7.3 18.3c-4.3 5.2-10.1 9.7-16.7 13.4c-2.7 1.5-5.7 3-8.7 4.3c3.1 1.3 6 2.7 8.7 4.3c6.6 3.7 12.5 8.2 16.7 13.4c4.3 5.1 7.3 11.4 7.3 18.3s-3.1 13.2-7.3 18.3c-4.3 5.2-10.1 9.7-16.7 13.4C274.7 443.1 257.4 448 240 448c-3.6 0-6.8-2.5-7.7-6s.6-7.2 3.8-9l0 0 0 0 0 0 0 0 .2-.1c.2-.1 .5-.3 .9-.5c.8-.5 2-1.2 3.4-2.1c2.8-1.9 6.5-4.5 10.2-7.6c3.7-3.1 7.2-6.6 9.6-10.1c2.5-3.5 3.5-6.4 3.5-8.6s-1-5-3.5-8.6c-2.5-3.5-5.9-6.9-9.6-10.1c-3.7-3.1-7.4-5.7-10.2-7.6c-1.4-.9-2.6-1.6-3.4-2.1c-.4-.2-.7-.4-.9-.5l-.2-.1 0 0 0 0 0 0c-2.5-1.4-4.1-4.1-4.1-7s1.6-5.6 4.1-7l0 0 0 0 0 0 0 0 0 0 .2-.1 .3-.2 .6-.4c.8-.5 2-1.2 3.4-2.1c2.8-1.9 6.5-4.5 10.2-7.6c3.7-3.1 7.2-6.6 9.6-10.1c2.5-3.5 3.5-6.4 3.5-8.6s-1-5-3.5-8.6c-2.5-3.5-5.9-6.9-9.6-10.1c-3.7-3.1-7.4-5.7-10.2-7.6c-1.4-.9-2.6-1.6-3.4-2.1l-.4-.3-.5-.3-.2-.1 0 0 0 0 0 0c-3.2-1.8-4.7-5.5-3.8-9s4.1-6 7.7-6c17.4 0 34.7 4.9 47.9 12.3c6.6 3.7 12.5 8.2 16.7 13.4zM144.4 208a32 32 0 1 1 64 0 32 32 0 1 1 -64 0zm192-32a32 32 0 1 1 0 64 32 32 0 1 1 0-64z',
                'vw' => 512,
                'vh' => 512
            ],
            'face-kiss-beam' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zm48.7-198.3c4.3 5.1 7.3 11.4 7.3 18.3s-3.1 13.2-7.3 18.3c-4.3 5.2-10.1 9.7-16.7 13.4c-2.7 1.5-5.7 3-8.7 4.3c3.1 1.3 6 2.7 8.7 4.3c6.6 3.7 12.5 8.2 16.7 13.4c4.3 5.1 7.3 11.4 7.3 18.3s-3.1 13.2-7.3 18.3c-4.3 5.2-10.1 9.7-16.7 13.4C274.7 443.1 257.4 448 240 448c-3.6 0-6.8-2.5-7.7-6s.6-7.2 3.8-9l0 0 0 0 0 0 0 0 .2-.1c.2-.1 .5-.3 .9-.5c.8-.5 2-1.2 3.4-2.1c2.8-1.9 6.5-4.5 10.2-7.6c3.7-3.1 7.2-6.6 9.6-10.1c2.5-3.5 3.5-6.4 3.5-8.6s-1-5-3.5-8.6c-2.5-3.5-5.9-6.9-9.6-10.1c-3.7-3.1-7.4-5.7-10.2-7.6c-1.4-.9-2.6-1.6-3.4-2.1c-.4-.2-.7-.4-.9-.5l-.2-.1 0 0 0 0 0 0c-2.5-1.4-4.1-4.1-4.1-7s1.6-5.6 4.1-7l0 0 0 0 0 0 0 0 0 0 .2-.1 .3-.2 .6-.4c.8-.5 2-1.2 3.4-2.1c2.8-1.9 6.5-4.5 10.2-7.6c3.7-3.1 7.2-6.6 9.6-10.1c2.5-3.5 3.5-6.4 3.5-8.6s-1-5-3.5-8.6c-2.5-3.5-5.9-6.9-9.6-10.1c-3.7-3.1-7.4-5.7-10.2-7.6c-1.4-.9-2.6-1.6-3.4-2.1l-.4-.3-.5-.3-.2-.1 0 0 0 0 0 0c-3.2-1.8-4.7-5.5-3.8-9s4.1-6 7.7-6c17.4 0 34.7 4.9 47.9 12.3c6.6 3.7 12.5 8.2 16.7 13.4zm-87.1-84.9l0 0 0 0-.2-.2c-.2-.2-.4-.5-.7-.9c-.6-.8-1.6-2-2.8-3.4c-2.5-2.8-6-6.6-10.2-10.3c-8.8-7.8-18.8-14-27.7-14s-18.9 6.2-27.7 14c-4.2 3.7-7.7 7.5-10.2 10.3c-1.2 1.4-2.2 2.6-2.8 3.4c-.3 .4-.6 .7-.7 .9l-.2 .2 0 0 0 0 0 0c-2.1 2.8-5.7 3.9-8.9 2.8s-5.5-4.1-5.5-7.6c0-17.9 6.7-35.6 16.6-48.8c9.8-13 23.9-23.2 39.4-23.2s29.6 10.2 39.4 23.2c9.9 13.2 16.6 30.9 16.6 48.8c0 3.4-2.2 6.5-5.5 7.6s-6.9 0-8.9-2.8l0 0 0 0zm160 0l0 0-.2-.2c-.2-.2-.4-.5-.7-.9c-.6-.8-1.6-2-2.8-3.4c-2.5-2.8-6-6.6-10.2-10.3c-8.8-7.8-18.8-14-27.7-14s-18.9 6.2-27.7 14c-4.2 3.7-7.7 7.5-10.2 10.3c-1.2 1.4-2.2 2.6-2.8 3.4c-.3 .4-.6 .7-.7 .9l-.2 .2 0 0 0 0 0 0c-2.1 2.8-5.7 3.9-8.9 2.8s-5.5-4.1-5.5-7.6c0-17.9 6.7-35.6 16.6-48.8c9.8-13 23.9-23.2 39.4-23.2s29.6 10.2 39.4 23.2c9.9 13.2 16.6 30.9 16.6 48.8c0 3.4-2.2 6.5-5.5 7.6s-6.9 0-8.9-2.8l0 0 0 0 0 0z',
                'vw' => 512,
                'vh' => 512
            ],
            'face-kiss-wink-heart' => [
                'p' => 'M498 339.7c9.1-26.2 14-54.4 14-83.7C512 114.6 397.4 0 256 0S0 114.6 0 256S114.6 512 256 512c35.4 0 69.1-7.2 99.7-20.2c-4.8-5.5-8.5-12.2-10.4-19.7l-22.9-89.3c-10-39 11.8-80.9 51.8-92.1c37.2-10.4 73.8 10.1 87.5 44c12.7-1.6 25.1 .4 36.2 5zM296 332c0 6.9-3.1 13.2-7.3 18.3c-4.3 5.2-10.1 9.7-16.7 13.4c-2.7 1.5-5.7 3-8.7 4.3c3.1 1.3 6 2.7 8.7 4.3c6.6 3.7 12.5 8.2 16.7 13.4c4.3 5.1 7.3 11.4 7.3 18.3s-3.1 13.2-7.3 18.3c-4.3 5.2-10.1 9.7-16.7 13.4C258.7 443.1 241.4 448 224 448c-3.6 0-6.8-2.5-7.7-6s.6-7.2 3.8-9l0 0 0 0 0 0 0 0 .2-.1c.2-.1 .5-.3 .9-.5c.8-.5 2-1.2 3.4-2.1c2.8-1.9 6.5-4.5 10.2-7.6c3.7-3.1 7.2-6.6 9.6-10.1c2.5-3.5 3.5-6.4 3.5-8.6s-1-5-3.5-8.6c-2.5-3.5-5.9-6.9-9.6-10.1c-3.7-3.1-7.4-5.7-10.2-7.6c-1.4-.9-2.6-1.6-3.4-2.1l-.6-.4-.3-.2-.2-.1 0 0 0 0 0 0c-2.5-1.4-4.1-4.1-4.1-7s1.6-5.6 4.1-7l0 0 0 0 0 0 0 0 0 0 .2-.1c.2-.1 .5-.3 .9-.5c.8-.5 2-1.2 3.4-2.1c2.8-1.9 6.5-4.5 10.2-7.6c3.7-3.1 7.2-6.6 9.6-10.1c2.5-3.5 3.5-6.4 3.5-8.6s-1-5-3.5-8.6c-2.5-3.5-5.9-6.9-9.6-10.1c-3.7-3.1-7.4-5.7-10.2-7.6c-1.4-.9-2.6-1.6-3.4-2.1c-.4-.2-.7-.4-.9-.5l-.2-.1 0 0 0 0 0 0c-3.2-1.8-4.7-5.5-3.8-9s4.1-6 7.7-6c17.4 0 34.7 4.9 47.9 12.3c6.6 3.7 12.5 8.2 16.7 13.4c4.3 5.1 7.3 11.4 7.3 18.3zM176.4 176a32 32 0 1 1 0 64 32 32 0 1 1 0-64zm194.8 57.6c-17.6-23.5-52.8-23.5-70.4 0c-5.3 7.1-15.3 8.5-22.4 3.2s-8.5-15.3-3.2-22.4c30.4-40.5 91.2-40.5 121.6 0c5.3 7.1 3.9 17.1-3.2 22.4s-17.1 3.9-22.4-3.2zM434 352.3c-6-23.2-28.8-37-51.1-30.8s-35.4 30.1-29.5 53.4l22.9 89.3c2.2 8.7 11.2 13.9 19.8 11.4l84.9-23.8c22.2-6.2 35.4-30.1 29.5-53.4s-28.8-37-51.1-30.8l-20.2 5.6-5.4-21z',
                'vw' => 512,
                'vh' => 512
            ],
            'face-laugh' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM96.8 314.1c-3.8-13.7 7.4-26.1 21.6-26.1H393.6c14.2 0 25.5 12.4 21.6 26.1C396.2 382 332.1 432 256 432s-140.2-50-159.2-117.9zM144.4 192a32 32 0 1 1 64 0 32 32 0 1 1 -64 0zm192-32a32 32 0 1 1 0 64 32 32 0 1 1 0-64z',
                'vw' => 512,
                'vh' => 512
            ],
            'face-laugh-beam' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM96.8 314.1c-3.8-13.7 7.4-26.1 21.6-26.1H393.6c14.2 0 25.5 12.4 21.6 26.1C396.2 382 332.1 432 256 432s-140.2-50-159.2-117.9zM217.6 212.8l0 0 0 0-.2-.2c-.2-.2-.4-.5-.7-.9c-.6-.8-1.6-2-2.8-3.4c-2.5-2.8-6-6.6-10.2-10.3c-8.8-7.8-18.8-14-27.7-14s-18.9 6.2-27.7 14c-4.2 3.7-7.7 7.5-10.2 10.3c-1.2 1.4-2.2 2.6-2.8 3.4c-.3 .4-.6 .7-.7 .9l-.2 .2 0 0 0 0 0 0c-2.1 2.8-5.7 3.9-8.9 2.8s-5.5-4.1-5.5-7.6c0-17.9 6.7-35.6 16.6-48.8c9.8-13 23.9-23.2 39.4-23.2s29.6 10.2 39.4 23.2c9.9 13.2 16.6 30.9 16.6 48.8c0 3.4-2.2 6.5-5.5 7.6s-6.9 0-8.9-2.8l0 0 0 0zm160 0l0 0-.2-.2c-.2-.2-.4-.5-.7-.9c-.6-.8-1.6-2-2.8-3.4c-2.5-2.8-6-6.6-10.2-10.3c-8.8-7.8-18.8-14-27.7-14s-18.9 6.2-27.7 14c-4.2 3.7-7.7 7.5-10.2 10.3c-1.2 1.4-2.2 2.6-2.8 3.4c-.3 .4-.6 .7-.7 .9l-.2 .2 0 0 0 0 0 0c-2.1 2.8-5.7 3.9-8.9 2.8s-5.5-4.1-5.5-7.6c0-17.9 6.7-35.6 16.6-48.8c9.8-13 23.9-23.2 39.4-23.2s29.6 10.2 39.4 23.2c9.9 13.2 16.6 30.9 16.6 48.8c0 3.4-2.2 6.5-5.5 7.6s-6.9 0-8.9-2.8l0 0 0 0 0 0z',
                'vw' => 512,
                'vh' => 512
            ],
            'face-laugh-squint' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM96.8 314.1c-3.8-13.7 7.4-26.1 21.6-26.1H393.6c14.2 0 25.5 12.4 21.6 26.1C396.2 382 332.1 432 256 432s-140.2-50-159.2-117.9zm36.7-199.4l89.9 47.9c10.7 5.7 10.7 21.1 0 26.8l-89.9 47.9c-7.9 4.2-17.5-1.5-17.5-10.5c0-2.8 1-5.5 2.8-7.6l36-43.2-36-43.2c-1.8-2.1-2.8-4.8-2.8-7.6c0-9 9.6-14.7 17.5-10.5zM396 125.1c0 2.8-1 5.5-2.8 7.6l-36 43.2 36 43.2c1.8 2.1 2.8 4.8 2.8 7.6c0 9-9.6 14.7-17.5 10.5l-89.9-47.9c-10.7-5.7-10.7-21.1 0-26.8l89.9-47.9c7.9-4.2 17.5 1.5 17.5 10.5z',
                'vw' => 512,
                'vh' => 512
            ],
            'face-laugh-wink' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM96.8 314.1c-3.8-13.7 7.4-26.1 21.6-26.1H393.6c14.2 0 25.5 12.4 21.6 26.1C396.2 382 332.1 432 256 432s-140.2-50-159.2-117.9zM144.4 192a32 32 0 1 1 64 0 32 32 0 1 1 -64 0zm156.4 25.6c-5.3 7.1-15.3 8.5-22.4 3.2s-8.5-15.3-3.2-22.4c30.4-40.5 91.2-40.5 121.6 0c5.3 7.1 3.9 17.1-3.2 22.4s-17.1 3.9-22.4-3.2c-17.6-23.5-52.8-23.5-70.4 0z',
                'vw' => 512,
                'vh' => 512
            ],
            'face-meh' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM176.4 176a32 32 0 1 1 0 64 32 32 0 1 1 0-64zm128 32a32 32 0 1 1 64 0 32 32 0 1 1 -64 0zM160 336H352c8.8 0 16 7.2 16 16s-7.2 16-16 16H160c-8.8 0-16-7.2-16-16s7.2-16 16-16z',
                'vw' => 512,
                'vh' => 512
            ],
            'face-meh-blank' => [
                'p' => 'M0 256a256 256 0 1 1 512 0A256 256 0 1 1 0 256zm208.4-48a32 32 0 1 0 -64 0 32 32 0 1 0 64 0zm128 32a32 32 0 1 0 0-64 32 32 0 1 0 0 64z',
                'vw' => 512,
                'vh' => 512
            ],
            'face-rolling-eyes' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM192 368H320c8.8 0 16 7.2 16 16s-7.2 16-16 16H192c-8.8 0-16-7.2-16-16s7.2-16 16-16zm32-144c0 35.3-28.7 64-64 64s-64-28.7-64-64c0-26 15.5-48.4 37.8-58.4c-3.7 5.2-5.8 11.6-5.8 18.4c0 17.7 14.3 32 32 32s32-14.3 32-32c0-6.9-2.2-13.2-5.8-18.4C208.5 175.6 224 198 224 224zm128 64c-35.3 0-64-28.7-64-64c0-26 15.5-48.4 37.8-58.4c-3.7 5.2-5.8 11.6-5.8 18.4c0 17.7 14.3 32 32 32s32-14.3 32-32c0-6.9-2.2-13.2-5.8-18.4C400.5 175.6 416 198 416 224c0 35.3-28.7 64-64 64z',
                'vw' => 512,
                'vh' => 512
            ],
            'face-sad-cry' => [
                'p' => 'M352 493.4c-29.6 12-62.1 18.6-96 18.6s-66.4-6.6-96-18.6V288c0-8.8-7.2-16-16-16s-16 7.2-16 16V477.8C51.5 433.5 0 350.8 0 256C0 114.6 114.6 0 256 0S512 114.6 512 256c0 94.8-51.5 177.5-128 221.8V288c0-8.8-7.2-16-16-16s-16 7.2-16 16V493.4zM195.2 233.6c5.3 7.1 15.3 8.5 22.4 3.2s8.5-15.3 3.2-22.4c-30.4-40.5-91.2-40.5-121.6 0c-5.3 7.1-3.9 17.1 3.2 22.4s17.1 3.9 22.4-3.2c17.6-23.5 52.8-23.5 70.4 0zm121.6 0c17.6-23.5 52.8-23.5 70.4 0c5.3 7.1 15.3 8.5 22.4 3.2s8.5-15.3 3.2-22.4c-30.4-40.5-91.2-40.5-121.6 0c-5.3 7.1-3.9 17.1 3.2 22.4s17.1 3.9 22.4-3.2zM208 336v32c0 26.5 21.5 48 48 48s48-21.5 48-48V336c0-26.5-21.5-48-48-48s-48 21.5-48 48z',
                'vw' => 512,
                'vh' => 512
            ],
            'face-sad-tear' => [
                'p' => 'M0 256a256 256 0 1 0 512 0A256 256 0 1 0 0 256zm240 80c0-8.8 7.2-16 16-16c45 0 85.6 20.5 115.7 53.1c6 6.5 5.6 16.6-.9 22.6s-16.6 5.6-22.6-.9c-25-27.1-57.4-42.9-92.3-42.9c-8.8 0-16-7.2-16-16zm-80 80c-26.5 0-48-21-48-47c0-20 28.6-60.4 41.6-77.7c3.2-4.4 9.6-4.4 12.8 0C179.6 308.6 208 349 208 369c0 26-21.5 47-48 47zM367.6 208a32 32 0 1 1 -64 0 32 32 0 1 1 64 0zm-192-32a32 32 0 1 1 0 64 32 32 0 1 1 0-64z',
                'vw' => 512,
                'vh' => 512
            ],
            'face-smile' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM164.1 325.5C182 346.2 212.6 368 256 368s74-21.8 91.9-42.5c5.8-6.7 15.9-7.4 22.6-1.6s7.4 15.9 1.6 22.6C349.8 372.1 311.1 400 256 400s-93.8-27.9-116.1-53.5c-5.8-6.7-5.1-16.8 1.6-22.6s16.8-5.1 22.6 1.6zM144.4 208a32 32 0 1 1 64 0 32 32 0 1 1 -64 0zm192-32a32 32 0 1 1 0 64 32 32 0 1 1 0-64z',
                'vw' => 512,
                'vh' => 512
            ],
            'face-smile-beam' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM164.1 325.5C182 346.2 212.6 368 256 368s74-21.8 91.9-42.5c5.8-6.7 15.9-7.4 22.6-1.6s7.4 15.9 1.6 22.6C349.8 372.1 311.1 400 256 400s-93.8-27.9-116.1-53.5c-5.8-6.7-5.1-16.8 1.6-22.6s16.8-5.1 22.6 1.6zm53.5-96.7l0 0 0 0-.2-.2c-.2-.2-.4-.5-.7-.9c-.6-.8-1.6-2-2.8-3.4c-2.5-2.8-6-6.6-10.2-10.3c-8.8-7.8-18.8-14-27.7-14s-18.9 6.2-27.7 14c-4.2 3.7-7.7 7.5-10.2 10.3c-1.2 1.4-2.2 2.6-2.8 3.4c-.3 .4-.6 .7-.7 .9l-.2 .2 0 0 0 0 0 0c-2.1 2.8-5.7 3.9-8.9 2.8s-5.5-4.1-5.5-7.6c0-17.9 6.7-35.6 16.6-48.8c9.8-13 23.9-23.2 39.4-23.2s29.6 10.2 39.4 23.2c9.9 13.2 16.6 30.9 16.6 48.8c0 3.4-2.2 6.5-5.5 7.6s-6.9 0-8.9-2.8l0 0 0 0zm160 0l0 0-.2-.2c-.2-.2-.4-.5-.7-.9c-.6-.8-1.6-2-2.8-3.4c-2.5-2.8-6-6.6-10.2-10.3c-8.8-7.8-18.8-14-27.7-14s-18.9 6.2-27.7 14c-4.2 3.7-7.7 7.5-10.2 10.3c-1.2 1.4-2.2 2.6-2.8 3.4c-.3 .4-.6 .7-.7 .9l-.2 .2 0 0 0 0 0 0c-2.1 2.8-5.7 3.9-8.9 2.8s-5.5-4.1-5.5-7.6c0-17.9 6.7-35.6 16.6-48.8c9.8-13 23.9-23.2 39.4-23.2s29.6 10.2 39.4 23.2c9.9 13.2 16.6 30.9 16.6 48.8c0 3.4-2.2 6.5-5.5 7.6s-6.9 0-8.9-2.8l0 0 0 0 0 0z',
                'vw' => 512,
                'vh' => 512
            ],
            'face-smile-wink' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM164.1 325.5C182 346.2 212.6 368 256 368s74-21.8 91.9-42.5c5.8-6.7 15.9-7.4 22.6-1.6s7.4 15.9 1.6 22.6C349.8 372.1 311.1 400 256 400s-93.8-27.9-116.1-53.5c-5.8-6.7-5.1-16.8 1.6-22.6s16.8-5.1 22.6 1.6zM144.4 208a32 32 0 1 1 64 0 32 32 0 1 1 -64 0zm156.4 25.6c-5.3 7.1-15.3 8.5-22.4 3.2s-8.5-15.3-3.2-22.4c30.4-40.5 91.2-40.5 121.6 0c5.3 7.1 3.9 17.1-3.2 22.4s-17.1 3.9-22.4-3.2c-17.6-23.5-52.8-23.5-70.4 0z',
                'vw' => 512,
                'vh' => 512
            ],
            'face-surprise' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM176.4 176a32 32 0 1 1 0 64 32 32 0 1 1 0-64zm128 32a32 32 0 1 1 64 0 32 32 0 1 1 -64 0zM256 288a64 64 0 1 1 0 128 64 64 0 1 1 0-128z',
                'vw' => 512,
                'vh' => 512
            ],
            'face-tired' => [
                'p' => 'M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM164.7 328.7c22-22 53.9-40.7 91.3-40.7s69.3 18.7 91.3 40.7c11.1 11.1 20.1 23.4 26.4 35.4c6.2 11.7 10.3 24.4 10.3 35.9c0 5.2-2.6 10.2-6.9 13.2s-9.8 3.7-14.7 1.8l-20.5-7.7c-26.9-10.1-55.5-15.3-84.3-15.3h-3.2c-28.8 0-57.3 5.2-84.3 15.3L149.6 415c-4.9 1.8-10.4 1.2-14.7-1.8s-6.9-7.9-6.9-13.2c0-11.6 4.2-24.2 10.3-35.9c6.3-12 15.3-24.3 26.4-35.4zm-31.2-182l89.9 47.9c10.7 5.7 10.7 21.1 0 26.8l-89.9 47.9c-7.9 4.2-17.5-1.5-17.5-10.5c0-2.8 1-5.5 2.8-7.6l36-43.2-36-43.2c-1.8-2.1-2.8-4.8-2.8-7.6c0-9 9.6-14.7 17.5-10.5zM396 157.1c0 2.8-1 5.5-2.8 7.6l-36 43.2 36 43.2c1.8 2.1 2.8 4.8 2.8 7.6c0 9-9.6 14.7-17.5 10.5l-89.9-47.9c-10.7-5.7-10.7-21.1 0-26.8l89.9-47.9c7.9-4.2 17.5 1.5 17.5 10.5z',
                'vw' => 512,
                'vh' => 512
            ],
            'fan' => 'M22.04 8c-1.76 0-3.38.28-4.82.8L18 1.1c.06-.64-.51-1.17-1.17-1.1A10.06 10.06 0 0 0 8 9.97c0 1.76.28 3.38.8 4.82L1.1 14c-.64-.06-1.17.51-1.1 1.17A10.06 10.06 0 0 0 9.97 24c1.76 0 3.38-.28 4.82-.8L14 30.9c-.06.64.51 1.17 1.17 1.1A10.06 10.06 0 0 0 24 22.03c0-1.76-.28-3.38-.8-4.82l7.7.78c.64.06 1.17-.51 1.1-1.17A10.06 10.06 0 0 0 22.03 8ZM16 18a2 2 0 1 1 0-4 2 2 0 0 1 0 4Zm0 0',
            'fast-backward' => 'M0 27.25V4.75Q0 4 .75 4h2.5Q4 4 4 4.75v9.5l10.75-9.81q.94-.82 2.1-.25T18 6v8.25l10.75-9.81q.94-.82 2.1-.25T32 6v20q0 1.25-1.16 1.81t-2.09-.25L18 17.81v8.2q0 1.24-1.16 1.8t-2.09-.25L4 17.81v9.44q0 .75-.75.75H.75Q0 28 0 27.25z',
            'fast-forward' => 'M32 4.75v22.5q0 .75-.75.75h-2.5q-.75 0-.75-.75v-9.5l-10.75 9.81q-.94.82-2.1.25T14 26v-8.25L3.25 27.56q-.94.82-2.1.25T0 26V6q0-1.25 1.16-1.81t2.09.25L14 14.19v-8.2q0-1.24 1.16-1.8t2.09.25L28 14.19V4.75q0-.75.75-.75h2.5q.75 0 .75.75z',
            'faucet' => 'M22 16h-2.41A8.3 8.3 0 0 0 16 14.25v-2.97l-2-.21-2 .2v2.98c-1.37.31-2.6.91-3.59 1.75H1a1 1 0 0 0-1 1v6a1 1 0 0 0 1 1h5.8a8.12 8.12 0 0 0 7.2 4 8.12 8.12 0 0 0 7.2-4h.8a2 2 0 0 1 2 2c0 1.1.9 2 2 2h4a2 2 0 0 0 2-2 10 10 0 0 0-10-10ZM5.1 10l8.9-.94 8.9.94c.59.06 1.1-.43 1.1-1.05v-1.9c0-.62-.51-1.1-1.1-1.05l-6.9.73V5a1 1 0 0 0-1-1h-2a1 1 0 0 0-1 1v1.73L5.1 6C4.5 5.94 4 6.43 4 7.05v1.9c0 .62.51 1.1 1.1 1.05Zm0 0',
            'faucet-drip' => [
                'p' => 'M224 0c17.7 0 32 14.3 32 32V44l96-12c17.7 0 32 14.3 32 32s-14.3 32-32 32L256 84l-31-3.9-1-.1-1 .1L192 84 96 96C78.3 96 64 81.7 64 64s14.3-32 32-32l96 12V32c0-17.7 14.3-32 32-32zM0 224c0-17.7 14.3-32 32-32h96l22.6-22.6c6-6 14.1-9.4 22.6-9.4H192V116.2l32-4 32 4V160h18.7c8.5 0 16.6 3.4 22.6 9.4L320 192h32c88.4 0 160 71.6 160 160c0 17.7-14.3 32-32 32H416c-17.7 0-32-14.3-32-32s-14.3-32-32-32H315.9c-20.2 29-53.9 48-91.9 48s-71.7-19-91.9-48H32c-17.7 0-32-14.3-32-32V224zM436.8 423.4c1.9-4.5 6.3-7.4 11.2-7.4s9.2 2.9 11.2 7.4l18.2 42.4c1.8 4.1 2.7 8.6 2.7 13.1V480c0 17.7-14.3 32-32 32s-32-14.3-32-32v-1.2c0-4.5 .9-8.9 2.7-13.1l18.2-42.4z',
                'vw' => 512,
                'vh' => 512
            ],
            'fax' => 'M4 8q.81 0 1.4.6T6 10v20q0 .81-.6 1.4T4 32H2q-.81 0-1.4-.6T0 30V10q0-.81.6-1.4T2 8h2zm26 2q.81 0 1.4.6T32 12v18q0 .81-.6 1.4T30 32H10q-.81 0-1.4-.6T8 30V2q0-.81.6-1.4T10 0h15.19q.81 0 1.37.56l2.88 2.88Q30 4 30 4.8V10zM18 27v-2q0-.44-.28-.72T17 24h-2q-.44 0-.72.28T14 25v2q0 .44.28.72T15 28h2q.44 0 .72-.28T18 27zm0-8v-2q0-.44-.28-.72T17 16h-2q-.44 0-.72.28T14 17v2q0 .44.28.72T15 20h2q.44 0 .72-.28T18 19zm8 8v-2q0-.44-.28-.72T25 24h-2q-.44 0-.72.28T22 25v2q0 .44.28.72T23 28h2q.44 0 .72-.28T26 27zm0-8v-2q0-.44-.28-.72T25 16h-2q-.44 0-.72.28T22 17v2q0 .44.28.72T23 20h2q.44 0 .72-.28T26 19zm1-7V6h-2q-.44 0-.72-.28T24 5V3H11v9h16z',
            'feather' => 'M29.19 2.81q2.44 2.44 2.75 5.66t-1.75 6.78L22 18h6.13l-2.94 2.94-9.2 3.06h6.13q-1.68 1.63-3.9 2.6t-4.38 1.21-3.97.13-3.18-.5l-4.13 4.12Q2.12 32 1.5 32t-1.06-.44T0 30.5t.44-1.06l16.25-16.19q.68-.75-.04-1.44t-1.4 0L4.06 22.94q-.37-3.32.53-6.94t3.47-6.25q.44-.38 1.47-1.44t3.9-3.87Q16.2 1.69 19.16.66t5.5-.44 4.54 2.6z',
            'feather-alt' => 'M32 0q-.56 7.81-3.38 14.69L22 18h5.06q-.87 1.56-1.87 2.94L16 24h6.31Q18.7 27.07 14 27.57q-1.94.18-3.9.3T6.11 28l-3.56 3.57Q2.12 32 1.5 32t-1.06-.43T0 30.5t.44-1.06l16.25-16.19q.68-.75-.04-1.44t-1.4 0L4.06 23.01q.13-2.88.38-5 .3-3.2 1.87-5.91T10 7.63t4.97-3.2 5.31-2.18T25.44.94 29.5.22 32 0z',
            'feather-pointed' => [
                'p' => 'M278.5 215.6L23 471c-9.4 9.4-9.4 24.6 0 33.9s24.6 9.4 33.9 0l74.8-74.8c7.4 4.6 15.3 8.2 23.8 10.5C200.3 452.8 270 454.5 338 409.4c12.2-8.1 5.8-25.4-8.8-25.4l-16.1 0c-5.1 0-9.2-4.1-9.2-9.2c0-4.1 2.7-7.6 6.5-8.8l97.7-29.3c3.4-1 6.4-3.1 8.4-6.1c4.4-6.4 8.6-12.9 12.6-19.6c6.2-10.3-1.5-23-13.5-23l-38.6 0c-5.1 0-9.2-4.1-9.2-9.2c0-4.1 2.7-7.6 6.5-8.8l80.9-24.3c4.6-1.4 8.4-4.8 10.2-9.3C494.5 163 507.8 86.1 511.9 36.8c.8-9.9-3-19.6-10-26.6s-16.7-10.8-26.6-10C391.5 7 228.5 40.5 137.4 131.6C57.3 211.7 56.7 302.3 71.3 356.4c2.1 7.9 12 9.6 17.8 3.8L253.6 195.8c6.2-6.2 16.4-6.2 22.6 0c5.4 5.4 6.1 13.6 2.2 19.8z',
                'vw' => 512,
                'vh' => 512
            ],
            'female' => [
                'p' => 'M8 0q1.69 0 2.84 1.16T12 4t-1.16 2.84T8 8 5.16 6.84 4 4t1.16-2.84T8 0zm7.44 22.13q.19.75-.25 1.3T14 24h-3.5v6.5q0 .63-.44 1.06T9 32H7q-.63 0-1.06-.44T5.5 30.5V24H2q-.75 0-1.19-.56t-.25-1.31l3-12q.12-.44.53-.79T5 9h.7q2.3 1.06 4.62 0H11q.5 0 .9.34t.53.78z',
                'w' => .5,
                'vw' => 16,
            ],
            'ferry' => [
                'p' => 'M224 0H352c17.7 0 32 14.3 32 32h75.1c20.6 0 31.6 24.3 18.1 39.8L456 96H120L98.8 71.8C85.3 56.3 96.3 32 116.9 32H192c0-17.7 14.3-32 32-32zM96 128H480c17.7 0 32 14.3 32 32V283.5c0 13.3-4.2 26.3-11.9 37.2l-51.4 71.9c-1.9 1.1-3.7 2.2-5.5 3.5c-15.5 10.7-34 18-51 19.9H375.6c-17.1-1.8-35-9-50.8-19.9c-22.1-15.5-51.6-15.5-73.7 0c-14.8 10.2-32.5 18-50.6 19.9H183.9c-17-1.8-35.6-9.2-51-19.9c-1.8-1.3-3.7-2.4-5.6-3.5L75.9 320.7C68.2 309.8 64 296.8 64 283.5V160c0-17.7 14.3-32 32-32zm32 64v96H448V192H128zM306.5 421.9C329 437.4 356.5 448 384 448c26.9 0 55.3-10.8 77.4-26.1l0 0c11.9-8.5 28.1-7.8 39.2 1.7c14.4 11.9 32.5 21 50.6 25.2c17.2 4 27.9 21.2 23.9 38.4s-21.2 27.9-38.4 23.9c-24.5-5.7-44.9-16.5-58.2-25C449.5 501.7 417 512 384 512c-31.9 0-60.6-9.9-80.4-18.9c-5.8-2.7-11.1-5.3-15.6-7.7c-4.5 2.4-9.7 5.1-15.6 7.7c-19.8 9-48.5 18.9-80.4 18.9c-33 0-65.5-10.3-94.5-25.8c-13.4 8.4-33.7 19.3-58.2 25c-17.2 4-34.4-6.7-38.4-23.9s6.7-34.4 23.9-38.4c18.1-4.2 36.2-13.3 50.6-25.2c11.1-9.4 27.3-10.1 39.2-1.7l0 0C136.7 437.2 165.1 448 192 448c27.5 0 55-10.6 77.5-26.1c11.1-7.9 25.9-7.9 37 0z',
                'vw' => 576,
                'vh' => 512
            ],
            'fighter-jet' => [
                'p' => 'M34 14q6 1.31 6 2t-6 2l-8 1-3 1h-1.5l-7.31 9.25h2.5q1.31 0 1.31.38t-1.31.37h-7.2v-.75h1V19h-3l-4.18 5H1.2l-.7-.69V19H1v-1h3v-.19l-4-.5V14.7l4-.5V14H1v-1H.5V8.7l.7-.7h2.1l4.19 5h3V2.76h-1V2h7.19Q18 2 18 2.37t-1.31.38h-2.5L21.5 12H23l3 1z',
                'w' => 1.25,
                'vw' => 40
            ],
            'file' => [
                'p' => 'M14 8.5q0 .63.44 1.06t1.06.44H24v20.5q0 .63-.44 1.06T22.5 32h-21q-.62 0-1.06-.44T0 30.5v-29Q0 .88.44.44T1.5 0H14v8.5zm10-.88V8h-8V0h.38q.62 0 1.06.44l6.12 6.12Q24 7 24 7.63z',
                'w' => .75,
                'vw' => 24
            ],
            'file-alt' => [
                'p' => 'M14 8.5q0 .63.44 1.06t1.06.44H24v20.5q0 .63-.44 1.06T22.5 32h-21q-.62 0-1.06-.44T0 30.5v-29Q0 .88.44.44T1.5 0H14v8.5zm4 14.75v-.5q0-.75-.75-.75H6.75Q6 22 6 22.75v.5q0 .75.75.75h10.5q.75 0 .75-.75zm0-4v-.5q0-.75-.75-.75H6.75Q6 18 6 18.75v.5q0 .75.75.75h10.5q.75 0 .75-.75zm0-4.5q0-.75-.75-.75H6.75Q6 14 6 14.75v.5q0 .75.75.75h10.5q.75 0 .75-.75v-.5zm6-7.13V8h-8V0h.38q.62 0 1.06.44l6.12 6.12Q24 7 24 7.63z',
                'w' => .75,
                'vw' => 24
            ],
            'file-archive' => [
                'p' => 'M23.56 6.56Q24 7 24 7.63V8h-8V0h.38q.62 0 1.06.44zM8.03 21q.53 0 1 .22t.75.62.28.85q0 .69-.6 1.19t-1.43.5-1.44-.5-.6-1.2q0-.43.29-.84t.75-.62 1-.22zM14 8.5q0 .63.44 1.06t1.06.44H24v20.5q0 .63-.44 1.06T22.5 32h-21q-.62 0-1.06-.44T0 30.5v-29Q0 .88.44.44T1.5 0H8v2h2V0h4v8.5zM6 2v2h2V2H6zm2 24q1.56 0 2.53-1.19t.72-2.75l-1.13-5.44Q10 16 9.38 16H8v-2h2v-2H8v-2h2V8H8V6h2V4H8v2H6v2h2v2H6v2h2v2H6v2l-1.19 6.06q-.31 1.5.66 2.72T8 26z',
                'w' => .75,
                'vw' => 24
            ],
            'file-arrow-down' => [
                'p' => 'M64 0C28.7 0 0 28.7 0 64V448c0 35.3 28.7 64 64 64H320c35.3 0 64-28.7 64-64V160H256c-17.7 0-32-14.3-32-32V0H64zM256 0V128H384L256 0zM216 232V334.1l31-31c9.4-9.4 24.6-9.4 33.9 0s9.4 24.6 0 33.9l-72 72c-9.4 9.4-24.6 9.4-33.9 0l-72-72c-9.4-9.4-9.4-24.6 0-33.9s24.6-9.4 33.9 0l31 31V232c0-13.3 10.7-24 24-24s24 10.7 24 24z',
                'vw' => 384,
                'vh' => 512
            ],
            'file-arrow-up' => [
                'p' => 'M64 0C28.7 0 0 28.7 0 64V448c0 35.3 28.7 64 64 64H320c35.3 0 64-28.7 64-64V160H256c-17.7 0-32-14.3-32-32V0H64zM256 0V128H384L256 0zM216 408c0 13.3-10.7 24-24 24s-24-10.7-24-24V305.9l-31 31c-9.4 9.4-24.6 9.4-33.9 0s-9.4-24.6 0-33.9l72-72c9.4-9.4 24.6-9.4 33.9 0l72 72c9.4 9.4 9.4 24.6 0 33.9s-24.6 9.4-33.9 0l-31-31V408z',
                'vw' => 384,
                'vh' => 512
            ],
            'file-audio' => [
                'p' => 'M14 8.5q0 .63.44 1.06t1.06.44H24v20.5q0 .63-.44 1.06T22.5 32h-21q-.62 0-1.06-.44T0 30.5v-29Q0 .88.44.44T1.5 0H14v8.5zm-4 16.75v-8.5q0-.5-.47-.69t-.78.19L6.5 18.5H4.75q-.75 0-.75.75v3.5q0 .75.75.75H6.5l2.25 2.25q.31.38.78.19t.47-.69zm2.06-3q-.62.69-.34 1.5t1.03 1 1.5-.38q1.25-1.3 1.25-3.15t-1.25-3.1q-.69-.68-1.5-.43t-1.03 1 .34 1.5q.44.43.44 1.03t-.44 1.03zm5.38-7.31q-.63-.63-1.47-.38t-1.03 1 .37 1.5q1.7 1.75 1.7 4.2t-1.7 4.18q-.68.69-.37 1.5t1.06.97 1.44-.4Q20 24.87 20 21.21t-2.56-6.28zM24 7.63V8h-8V0h.38q.62 0 1.06.44l6.12 6.12Q24 7 24 7.63z',
                'w' => .75,
                'vw' => 24
            ],
            'file-circle-check' => [
                'p' => 'M0 64C0 28.7 28.7 0 64 0H224V128c0 17.7 14.3 32 32 32H384v38.6C310.1 219.5 256 287.4 256 368c0 59.1 29.1 111.3 73.7 143.3c-3.2 .5-6.4 .7-9.7 .7H64c-35.3 0-64-28.7-64-64V64zm384 64H256V0L384 128zM288 368a144 144 0 1 1 288 0 144 144 0 1 1 -288 0zm211.3-43.3c-6.2-6.2-16.4-6.2-22.6 0L416 385.4l-28.7-28.7c-6.2-6.2-16.4-6.2-22.6 0s-6.2 16.4 0 22.6l40 40c6.2 6.2 16.4 6.2 22.6 0l72-72c6.2-6.2 6.2-16.4 0-22.6z',
                'vw' => 576,
                'vh' => 512
            ],
            'file-circle-exclamation' => [
                'p' => 'M0 64C0 28.7 28.7 0 64 0H224V128c0 17.7 14.3 32 32 32H384v38.6C310.1 219.5 256 287.4 256 368c0 59.1 29.1 111.3 73.7 143.3c-3.2 .5-6.4 .7-9.7 .7H64c-35.3 0-64-28.7-64-64V64zm384 64H256V0L384 128zm48 96a144 144 0 1 1 0 288 144 144 0 1 1 0-288zm0 240a24 24 0 1 0 0-48 24 24 0 1 0 0 48zm0-192c-8.8 0-16 7.2-16 16v80c0 8.8 7.2 16 16 16s16-7.2 16-16V288c0-8.8-7.2-16-16-16z',
                'vw' => 576,
                'vh' => 512
            ],
            'file-circle-minus' => [
                'p' => 'M0 64C0 28.7 28.7 0 64 0H224V128c0 17.7 14.3 32 32 32H384v38.6C310.1 219.5 256 287.4 256 368c0 59.1 29.1 111.3 73.7 143.3c-3.2 .5-6.4 .7-9.7 .7H64c-35.3 0-64-28.7-64-64V64zm384 64H256V0L384 128zM288 368a144 144 0 1 1 288 0 144 144 0 1 1 -288 0zm224 0c0-8.8-7.2-16-16-16H368c-8.8 0-16 7.2-16 16s7.2 16 16 16H496c8.8 0 16-7.2 16-16z',
                'vw' => 576,
                'vh' => 512
            ],
            'file-circle-plus' => [
                'p' => 'M0 64C0 28.7 28.7 0 64 0H224V128c0 17.7 14.3 32 32 32H384v38.6C310.1 219.5 256 287.4 256 368c0 59.1 29.1 111.3 73.7 143.3c-3.2 .5-6.4 .7-9.7 .7H64c-35.3 0-64-28.7-64-64V64zm384 64H256V0L384 128zm48 96a144 144 0 1 1 0 288 144 144 0 1 1 0-288zm16 80c0-8.8-7.2-16-16-16s-16 7.2-16 16v48H368c-8.8 0-16 7.2-16 16s7.2 16 16 16h48v48c0 8.8 7.2 16 16 16s16-7.2 16-16V384h48c8.8 0 16-7.2 16-16s-7.2-16-16-16H448V304z',
                'vw' => 576,
                'vh' => 512
            ],
            'file-circle-question' => [
                'p' => 'M0 64C0 28.7 28.7 0 64 0H224V128c0 17.7 14.3 32 32 32H384v38.6C310.1 219.5 256 287.4 256 368c0 59.1 29.1 111.3 73.7 143.3c-3.2 .5-6.4 .7-9.7 .7H64c-35.3 0-64-28.7-64-64V64zm384 64H256V0L384 128zm48 96a144 144 0 1 1 0 288 144 144 0 1 1 0-288zm0 240a24 24 0 1 0 0-48 24 24 0 1 0 0 48zM368 321.6V328c0 8.8 7.2 16 16 16s16-7.2 16-16v-6.4c0-5.3 4.3-9.6 9.6-9.6h40.5c7.7 0 13.9 6.2 13.9 13.9c0 5.2-2.9 9.9-7.4 12.3l-32 16.8c-5.3 2.8-8.6 8.2-8.6 14.2V384c0 8.8 7.2 16 16 16s16-7.2 16-16v-5.1l23.5-12.3c15.1-7.9 24.5-23.6 24.5-40.6c0-25.4-20.6-45.9-45.9-45.9H409.6c-23 0-41.6 18.6-41.6 41.6z',
                'vw' => 576,
                'vh' => 512
            ],
            'file-circle-xmark' => [
                'p' => 'M0 64C0 28.7 28.7 0 64 0H224V128c0 17.7 14.3 32 32 32H384v38.6C310.1 219.5 256 287.4 256 368c0 59.1 29.1 111.3 73.7 143.3c-3.2 .5-6.4 .7-9.7 .7H64c-35.3 0-64-28.7-64-64V64zm384 64H256V0L384 128zm48 96a144 144 0 1 1 0 288 144 144 0 1 1 0-288zm59.3 107.3c6.2-6.2 6.2-16.4 0-22.6s-16.4-6.2-22.6 0L432 345.4l-36.7-36.7c-6.2-6.2-16.4-6.2-22.6 0s-6.2 16.4 0 22.6L409.4 368l-36.7 36.7c-6.2 6.2-6.2 16.4 0 22.6s16.4 6.2 22.6 0L432 390.6l36.7 36.7c6.2 6.2 16.4 6.2 22.6 0s6.2-16.4 0-22.6L454.6 368l36.7-36.7z',
                'vw' => 576,
                'vh' => 512
            ],
            'file-code' => [
                'p' => 'M24 7.63V8h-8V0h.38q.62 0 1.06.44l6.12 6.12Q24 7 24 7.63zM15.5 10H24v20.5q0 .63-.44 1.06T22.5 32h-21q-.63 0-1.06-.44T0 30.5v-29Q0 .87.44.44T1.5 0H14v8.5q0 .62.44 1.06t1.06.44zM7.69 25.06l1.25-1.31q.06-.13.06-.25 0-.19-.12-.25L6.38 21l2.5-2.25q.12-.06.12-.25 0-.13-.06-.25l-1.25-1.31q-.06-.07-.25-.07-.13 0-.19.07l-4.06 3.81q-.13.13-.13.25t.13.25l4.06 3.81q.06.07.19.07.19 0 .25-.07zm3.25 3.13H11q.25 0 .31-.25l3.88-13.19v-.13q0-.25-.25-.3l-1.75-.5h-.06q-.25 0-.32.24L8.94 27.25v.13q0 .25.25.3zm10-6.94q.12-.13.12-.25t-.12-.25l-4.07-3.81q-.06-.07-.18-.07-.2 0-.25.07l-1.25 1.31q-.07.13-.07.25 0 .19.13.25l2.5 2.25-2.5 2.25q-.13.06-.13.25 0 .13.07.25l1.25 1.31q.06.07.25.07.12 0 .19-.07z',
                'w' => .75,
                'vw' => 24
            ],
            'file-contract' => [
                'p' => 'M14 8.5q0 .63.44 1.06t1.06.44H24v20.5q0 .63-.44 1.06T22.5 32h-21q-.62 0-1.06-.44T0 30.5v-29Q0 .88.44.44T1.5 0H14v8.5zm-10-4v1q0 .5.5.5h5q.5 0 .5-.5v-1q0-.5-.5-.5h-5Q4 4 4 4.5zm0 4v1q0 .5.5.5h5q.5 0 .5-.5v-1q0-.5-.5-.5h-5Q4 8 4 8.5zM16.06 24q-.43 0-.62-.38-.57-1.12-1.9-1.4t-2.23.47l-.87-2.56q-.38-1.07-1.44-1.07t-1.44 1.07L6.44 23.5q-.2.5-.7.5H5q-.44 0-.72.28T4 25t.28.72T5 26h.75q.87 0 1.6-.5t.96-1.38L9 22.2l1.06 3.12q.2.63.88.69H11q.63 0 .88-.56l.5-.94q.18-.38.62-.38t.63.38q.75 1.5 2.43 1.5H19q.44 0 .72-.28T20 25t-.28-.72T19 24h-2.94zm7.5-17.44Q24 7 24 7.63V8h-8V0h.38q.62 0 1.06.44z',
                'w' => .75,
                'vw' => 24
            ],
            'file-csv' => [
                'p' => 'M14 8.5q0 .63.44 1.06t1.06.44H24v20.5q0 .63-.44 1.06T22.5 32h-21q-.62 0-1.06-.44T0 30.5v-29Q0 .88.44.44T1.5 0H14v8.5zm-6 9v-1q0-.5-.5-.5H7q-1.25 0-2.13.88T4 19v2q0 1.25.88 2.13T7 24h.5q.5 0 .5-.5v-1q0-.5-.5-.5H7q-.44 0-.72-.28T6 21v-2q0-.44.28-.72T7 18h.5q.5 0 .5-.5zm2.75 6.5q1.13 0 1.9-.72t.79-1.72-.88-1.75l-1.37-1.19q-.13-.06-.13-.18 0-.44.69-.44h.75q.5 0 .5-.5v-1q0-.5-.5-.5h-.75q-1.12 0-1.9.72t-.79 1.72.88 1.75l1.37 1.18q.13.07.13.2 0 .43-.69.43H10q-.5 0-.5.5v1q0 .5.5.5h.75zM16 16.5q0-.5-.5-.5h-1q-.5 0-.5.5v1.31q0 3.44 2.25 5.88.31.31.75.31t.75-.31Q20 21.25 20 17.8v-1.3q0-.5-.5-.5h-1q-.5 0-.5.5v1.31q0 1.94-1 3.57-1-1.63-1-3.57V16.5zm7.56-9.94Q24 7 24 7.63V8h-8V0h.38q.62 0 1.06.44z',
                'w' => .75,
                'vw' => 24
            ],
            'file-download' => [
                'p' => 'M14 8.5q0 .63.44 1.06t1.06.44H24v20.5q0 .63-.44 1.06T22.5 32h-21q-.62 0-1.06-.44T0 30.5v-29Q0 .88.44.44T1.5 0H14v8.5zm4.75 13.19q.5-.44.25-1.06t-.94-.63H14v-5q0-.44-.28-.72T13 14h-2q-.44 0-.72.28T10 15v5H5.94q-.7 0-.94.63t.25 1.06l6 6q.31.31.75.31t.75-.31zm4.81-15.13Q24 7 24 7.63V8h-8V0h.38q.62 0 1.06.44z',
                'w' => .75,
                'vw' => 24
            ],
            'file-excel' => [
                'p' => 'M14 8.5q0 .63.44 1.06t1.06.44H24v20.5q0 .63-.44 1.06T22.5 32h-21q-.62 0-1.06-.44T0 30.5v-29Q0 .88.44.44T1.5 0H14v8.5zm3.75 6.63q.25-.38.03-.76t-.66-.37h-2.18q-.44 0-.63.38L12 18.68q0-.06-2.31-4.3-.2-.38-.63-.38H6.87q-.43 0-.65.4t.03.79L10 21l-3.75 5.88q-.25.37-.03.75t.65.37h2.2q.43 0 .62-.38 2.12-3.8 2.3-4.3 1.07 2.06 2.26 4.3.25.38.69.38h2.19q.43 0 .65-.38t-.03-.75L14 21zM24 7.63V8h-8V0h.38q.62 0 1.06.44l6.12 6.12Q24 7 24 7.63z',
                'w' => .75,
                'vw' => 24
            ],
            'file-export' => [
                'p' => 'M24 7.63V8h-8V0h.38q.62 0 1.06.44l6.12 6.12Q24 7 24 7.63zm11.69 11.62q.75.75 0 1.5l-6 6q-.44.5-1.06.25t-.63-.94V22h-4v-4h4v-4.06q0-.7.63-.94t1.06.25zM12 21q0 .44.28.72T13 22h11v8.5q0 .63-.44 1.06T22.5 32h-21q-.63 0-1.06-.44T0 30.5v-29Q0 .87.44.44T1.5 0H14v8.5q0 .62.44 1.06t1.06.44H24v8H13q-.44 0-.72.28T12 19v2z',
                'w' => 1.123,
                'vw' => 36
            ],
            'file-image' => [
                'p' => 'M24 7.63V8h-8V0h.38q.62 0 1.06.44l6.12 6.12Q24 7 24 7.63zM15.5 10H24v20.5q0 .63-.44 1.06T22.5 32h-21q-.63 0-1.06-.44T0 30.5v-29Q0 .87.44.44T1.5 0H14v8.5q0 .62.44 1.06t1.06.44zm-8.47 1q-1.22 0-2.1.88T4.07 14t.88 2.13 2.1.87 2.12-.88.9-2.12-.9-2.13T7.03 11zm13.03 15v-7l-2.5-2.5q-.5-.5-1.06 0L10.06 23l-2.5-2.5q-.5-.5-1 0L4.06 23v3h16z',
                'w' => .75,
                'vw' => 24
            ],
            'file-import' => 'M1 18h7v4H1q-.44 0-.72-.28T0 21v-2q0-.44.28-.72T1 18zM31.56 6.56Q32 7 32 7.63V8h-8V0h.38q.62 0 1.06.44zM22 8.5q0 .63.44 1.06t1.06.44H32v20.5q0 .63-.44 1.06T30.5 32h-21q-.63 0-1.06-.44T8 30.5V22h8v4.06q0 .7.62.94t1.07-.25l6-6q.75-.75 0-1.5l-6-6q-.44-.5-1.07-.25t-.62.94V18H8V1.5q0-.62.44-1.06T9.5 0H22v8.5z',
            'file-invoice' => [
                'p' => 'M18 16v4H6v-4h12zm5.56-9.44Q24 7 24 7.63V8h-8V0h.38q.62 0 1.06.44zM14 8.5q0 .63.44 1.06t1.06.44H24v20.5q0 .63-.44 1.06T22.5 32h-21q-.62 0-1.06-.44T0 30.5v-29Q0 .88.44.44T1.5 0H14v8.5zm-10-4v1q0 .5.5.5h5q.5 0 .5-.5v-1q0-.5-.5-.5h-5Q4 4 4 4.5zm0 4v1q0 .5.5.5h5q.5 0 .5-.5v-1q0-.5-.5-.5h-5Q4 8 4 8.5zm16 19v-1q0-.5-.5-.5h-5q-.5 0-.5.5v1q0 .5.5.5h5q.5 0 .5-.5zM20 15q0-.44-.28-.72T19 14H5q-.44 0-.72.28T4 15v6q0 .44.28.72T5 22h14q.44 0 .72-.28T20 21v-6z',
                'w' => .75,
                'vw' => 24
            ],
            'file-invoice-dollar' => [
                'p' => 'M23.56 6.56Q24 7 24 7.63V8h-8V0h.38q.62 0 1.06.44zM14 8.5q0 .63.44 1.06t1.06.44H24v20.5q0 .63-.44 1.06T22.5 32h-21q-.62 0-1.06-.44T0 30.5v-29Q0 .88.44.44T1.5 0H14v8.5zm-10-4v1q0 .5.5.5h5q.5 0 .5-.5v-1q0-.5-.5-.5h-5Q4 4 4 4.5zm0 5q0 .5.5.5h5q.5 0 .5-.5v-1q0-.5-.5-.5h-5Q4 8 4 8.5v1zM13 26q1.13-.06 1.9-.88t.79-1.93q0-.63-.25-1.2t-.72-.96-1.03-.6l-2.82-.8q-.56-.2-.56-.82 0-.31.22-.56t.53-.25h1.75q.44 0 .81.25.38.19.63-.07l.75-.68q.44-.44-.06-.75-.88-.7-1.94-.75v-1.5q0-.5-.5-.5h-1q-.5 0-.5.5V16q-1.13.06-1.9.87t-.79 1.94q0 .94.56 1.69t1.44 1.06l2.81.81q.57.2.57.82 0 .3-.22.56t-.53.25h-1.75q-.44 0-.81-.25-.38-.19-.63.06L9 24.5q-.44.44.06.75.88.69 1.94.75v1.5q0 .5.5.5h1q.5 0 .5-.5V26z',
                'w' => .75,
                'vw' => 24
            ],
            'file-lines' => [
                'p' => 'M64 0C28.7 0 0 28.7 0 64V448c0 35.3 28.7 64 64 64H320c35.3 0 64-28.7 64-64V160H256c-17.7 0-32-14.3-32-32V0H64zM256 0V128H384L256 0zM112 256H272c8.8 0 16 7.2 16 16s-7.2 16-16 16H112c-8.8 0-16-7.2-16-16s7.2-16 16-16zm0 64H272c8.8 0 16 7.2 16 16s-7.2 16-16 16H112c-8.8 0-16-7.2-16-16s7.2-16 16-16zm0 64H272c8.8 0 16 7.2 16 16s-7.2 16-16 16H112c-8.8 0-16-7.2-16-16s7.2-16 16-16z',
                'vw' => 384,
                'vh' => 512
            ],
            'file-medical' => [
                'p' => 'M23.56 6.56Q24 7 24 7.63V8h-8V0h.38q.62 0 1.06.44zM14 8.5q0 .63.44 1.06t1.06.44H24v20.5q0 .63-.44 1.06T22.5 32h-21q-.62 0-1.06-.44T0 30.5v-29Q0 .88.44.44T1.5 0H14v8.5zm4 10q0-.5-.5-.5H14v-3.5q0-.5-.5-.5h-3q-.5 0-.5.5V18H6.5q-.5 0-.5.5v3q0 .5.5.5H10v3.5q0 .5.5.5h3q.5 0 .5-.5V22h3.5q.5 0 .5-.5v-3z',
                'w' => .75,
                'vw' => 24
            ],
            'file-medical-alt' => [
                'p' => 'M18 8.5q0 .63.44 1.06t1.06.44H28v20.5q0 .63-.44 1.06T26.5 32h-21q-.63 0-1.06-.44T4 30.5V20h4.37l2.2 4.38q.12.25.43.25t.44-.25L15 17.25 16.38 20H22q.44 0 .72-.28T23 19t-.28-.72T22 18h-4.37l-2.2-4.37q-.12-.25-.43-.25t-.44.25L11 20.75l-1.25-2.5Q9.62 18 9.3 18H.5q-.5 0-.5-.5v-1q0-.5.5-.5H4V1.5q0-.62.44-1.06T5.5 0H18v8.5zm9.56-1.94Q28 7 28 7.63V8h-8V0h.38q.62 0 1.06.44z',
                'w' => .875,
                'vw' => 28
            ],
            'file-pdf' => [
                'p' => 'M11.38 16q-.13-.44-.2-1.16t-.06-1.25.13-.53q.37 0 .37 1.25t-.25 1.7zm-.13 2.94q.88 1.62 2.19 2.56-.75.13-3.94 1.38.81-1.5 1.75-3.94zm-5.88 7.81q.25-.69 2.2-2.5-.38.63-.76 1.1t-.62.74-.44.44-.28.19zM15.5 10H24v20.5q0 .63-.44 1.06T22.5 32h-21q-.63 0-1.06-.44T0 30.5v-29Q0 .87.44.44T1.5 0H14v8.5q0 .62.44 1.06t1.06.44zM15 20.75q-1.81-1.13-2.69-3.38.75-2.93.44-4-.19-.87-.84-1.21t-1.35-.1-.81.88q-.38 1.37.5 4.81-1.44 3.38-2.56 5.38-1.82.87-3 2.06t-.44 2.19q.5.62 1.37.62 1.57 0 3.82-3.87 3.87-1.25 4.93-1.44 2.25 1.19 4 1.19 1.25 0 1.57-1.07t-.32-1.62q-1-.94-4.62-.44zm8.56-14.19Q24 7 24 7.63V8h-8V0h.38q.62 0 1.06.44zM18.94 22.5q-.38.44-2.69-.56 2.69-.07 2.69.56z',
                'w' => .75,
                'vw' => 24
            ],
            'file-pen' => [
                'p' => 'M0 64C0 28.7 28.7 0 64 0H224V128c0 17.7 14.3 32 32 32H384V285.7l-86.8 86.8c-10.3 10.3-17.5 23.1-21 37.2l-18.7 74.9c-2.3 9.2-1.8 18.8 1.3 27.5H64c-35.3 0-64-28.7-64-64V64zm384 64H256V0L384 128zM549.8 235.7l14.4 14.4c15.6 15.6 15.6 40.9 0 56.6l-29.4 29.4-71-71 29.4-29.4c15.6-15.6 40.9-15.6 56.6 0zM311.9 417L441.1 287.8l71 71L382.9 487.9c-4.1 4.1-9.2 7-14.9 8.4l-60.1 15c-5.5 1.4-11.2-.2-15.2-4.2s-5.6-9.7-4.2-15.2l15-60.1c1.4-5.6 4.3-10.8 8.4-14.9z',
                'vw' => 576,
                'vh' => 512
            ],
            'file-powerpoint' => [
                'p' => 'M12.13 16.94q.8 0 1.24.53t.44 1.37-.47 1.38-1.28.53h-1.68v-3.81h1.74zM23.55 6.56Q24 7 24 7.63V8h-8V0h.38q.62 0 1.06.44zM14 8.5q0 .63.44 1.06t1.06.44H24v20.5q0 .63-.44 1.06T22.5 32h-21q-.62 0-1.06-.44T0 30.5v-29Q0 .88.44.44T1.5 0H14v8.5zm3.31 10.31q0-2.06-1.25-3.43T12.75 14H7.69q-.13 0-.28.06t-.25.16-.16.25-.06.28v12.5q0 .31.22.53t.53.22h1.94q.3 0 .53-.22t.22-.53v-3.56h1.68l1.1-.07 1.22-.25 1.09-.53.97-.87.62-1.31.25-1.85z',
                'w' => .75,
                'vw' => 24
            ],
            'file-prescription' => [
                'p' => 'M14 8.5q0 .63.44 1.06t1.06.44H24v20.5q0 .63-.44 1.06T22.5 32h-21q-.62 0-1.06-.44T0 30.5v-29Q0 .88.44.44T1.5 0H14v8.5zm4.31 11.19q-.75-.69-1.43 0L15 21.56l-2.06-2.06Q15 18.37 15 16q0-1.69-1.16-2.84T11 12H6q-.44 0-.72.28T5 13v10q0 .44.28.72T6 24h1q.44 0 .72-.28T8 23v-3h1.19l3.69 3.69L11 25.56q-.69.75 0 1.44l.69.69q.75.75 1.44 0L15 25.8l1.88 1.88q.69.75 1.44 0L19 27q.7-.69 0-1.44l-1.87-1.87L19 21.8q.7-.68 0-1.37zM11 17H8v-2h3q.44 0 .72.28T12 16t-.28.72T11 17zm13-9.38V8h-8V0h.38q.62 0 1.06.44l6.12 6.12Q24 7 24 7.63z',
                'w' => .75,
                'vw' => 24
            ],
            'file-shield' => [
                'p' => 'M0 64C0 28.7 28.7 0 64 0H224V128c0 17.7 14.3 32 32 32H384v47l-92.8 37.1c-21.3 8.5-35.2 29.1-35.2 52c0 56.6 18.9 148 94.2 208.3c-9 4.8-19.3 7.6-30.2 7.6H64c-35.3 0-64-28.7-64-64V64zm384 64H256V0L384 128zm39.1 97.7c5.7-2.3 12.1-2.3 17.8 0l120 48C570 277.4 576 286.2 576 296c0 63.3-25.9 168.8-134.8 214.2c-5.9 2.5-12.6 2.5-18.5 0C313.9 464.8 288 359.3 288 296c0-9.8 6-18.6 15.1-22.3l120-48zM527.4 312L432 273.8V461.7c68.2-33 91.5-99 95.4-149.7z',
                'vw' => 576,
                'vh' => 512
            ],
            'file-signature' => [
                'p' => 'M13.63 26.5q.37.69 1 1.1T16 28h8v2.5q0 .63-.44 1.06T22.5 32h-21q-.63 0-1.06-.44T0 30.5v-29Q0 .87.44.44T1.5 0H14v8.5q0 .62.44 1.06t1.06.44H24v2.94l-8 7.94V26q-.38-.06-.56-.37-.57-1.13-1.9-1.41t-2.23.47l-.87-2.56q-.38-1.07-1.44-1.07t-1.44 1.07L6.44 25.5q-.2.5-.7.5H5q-.44 0-.72.28T4 27t.28.72T5 28h.75q.87 0 1.6-.5t.96-1.37L9 24.19l1.06 3.12q.19.63.88.7t.94-.57l.5-.94q.18-.37.62-.37t.63.37zM24 7.62V8h-8V0h.38q.62 0 1.06.44l6.12 6.12Q24 7 24 7.63zM18 21.7l10.19-10.13 4.25 4.25-10.13 10.2H18V21.7zm17.56-11.25q.44.5.44 1.15t-.44 1.1l-1.75 1.75-4.25-4.25 1.75-1.75q.44-.44 1.1-.44t1.15.44z',
                'w' => 1.125,
                'vw' => 36
            ],
            'file-upload' => [
                'p' => 'M14 8.5q0 .63.44 1.06t1.06.44H24v20.5q0 .63-.44 1.06T22.5 32h-21q-.62 0-1.06-.44T0 30.5v-29Q0 .88.44.44T1.5 0H14v8.5zM18.06 22q.7 0 .94-.63t-.25-1.06l-6-6Q12.44 14 12 14t-.75.31l-6 6q-.5.44-.25 1.07t.94.62H10v5q0 .44.28.72T11 28h2q.44 0 .72-.28T14 27v-5h4.06zm5.5-15.44Q24 7 24 7.63V8h-8V0h.38q.62 0 1.06.44z',
                'w' => .75,
                'vw' => 24
            ],
            'file-video' => [
                'p' => 'M24 7.63V8h-8V0h.38q.62 0 1.06.44l6.12 6.12Q24 7 24 7.63zM14 8.5q0 .63.44 1.06t1.06.44H24v20.5q0 .63-.44 1.06T22.5 32h-21q-.62 0-1.06-.44T0 30.5v-29Q0 .88.44.44T1.5 0H14v8.5zm6 9q0-1-.94-1.38t-1.62.32L14 19.88V17.5q0-.62-.44-1.06T12.5 16h-7q-.63 0-1.07.44T4 17.5v7q0 .63.43 1.06T5.5 26h7q.62 0 1.06-.44T14 24.5v-2.37l3.44 3.43q.68.7 1.62.32T20 24.5v-7z',
                'w' => .75,
                'vw' => 24
            ],
            'file-waveform' => [
                'p' => 'M96 0C60.7 0 32 28.7 32 64V288H144c6.1 0 11.6 3.4 14.3 8.8L176 332.2l49.7-99.4c2.7-5.4 8.3-8.8 14.3-8.8s11.6 3.4 14.3 8.8L281.9 288H352c8.8 0 16 7.2 16 16s-7.2 16-16 16H272c-6.1 0-11.6-3.4-14.3-8.8L240 275.8l-49.7 99.4c-2.7 5.4-8.3 8.8-14.3 8.8s-11.6-3.4-14.3-8.8L134.1 320H32V448c0 35.3 28.7 64 64 64H352c35.3 0 64-28.7 64-64V160H288c-17.7 0-32-14.3-32-32V0H96zM288 0V128H416L288 0z',
                'vw' => 448,
                'vh' => 512
            ],
            'file-word' => [
                'p' => 'M14 8.5q0 .63.44 1.06t1.06.44H24v20.5q0 .63-.44 1.06T22.5 32h-21q-.62 0-1.06-.44T0 30.5v-29Q0 .88.44.44T1.5 0H14v8.5zm3.56 7.5q-.62 0-.68.63-1.38 6.3-1.38 8.06-.13-.94-1.88-8.13-.12-.56-.68-.56h-1.82q-.62 0-.75.56l-1.87 7.7q-.06-1.7-1.25-7.63Q7.12 16 6.5 16H4.94q-.38 0-.6.28t-.1.66l2.32 10.5q.13.56.75.56h2.31q.57 0 .75-.56 1.5-6.19 1.57-6.94H12q.25 1.44 1.56 6.94.2.56.75.56h2.38q.62 0 .75-.56l2.37-10.5q.07-.25-.03-.47t-.28-.35-.44-.12h-1.5zM24 7.62V8h-8V0h.38q.62 0 1.06.44l6.12 6.12Q24 7 24 7.63z',
                'w' => .75,
                'vw' => 24
            ],
            'file-zipper' => [
                'p' => 'M64 0C28.7 0 0 28.7 0 64V448c0 35.3 28.7 64 64 64H320c35.3 0 64-28.7 64-64V160H256c-17.7 0-32-14.3-32-32V0H64zM256 0V128H384L256 0zM96 48c0-8.8 7.2-16 16-16h32c8.8 0 16 7.2 16 16s-7.2 16-16 16H112c-8.8 0-16-7.2-16-16zm0 64c0-8.8 7.2-16 16-16h32c8.8 0 16 7.2 16 16s-7.2 16-16 16H112c-8.8 0-16-7.2-16-16zm0 64c0-8.8 7.2-16 16-16h32c8.8 0 16 7.2 16 16s-7.2 16-16 16H112c-8.8 0-16-7.2-16-16zm-6.3 71.8c3.7-14 16.4-23.8 30.9-23.8h14.8c14.5 0 27.2 9.7 30.9 23.8l23.5 88.2c1.4 5.4 2.1 10.9 2.1 16.4c0 35.2-28.8 63.7-64 63.7s-64-28.5-64-63.7c0-5.5 .7-11.1 2.1-16.4l23.5-88.2zM112 336c-8.8 0-16 7.2-16 16s7.2 16 16 16h32c8.8 0 16-7.2 16-16s-7.2-16-16-16H112z',
                'vw' => 384,
                'vh' => 512
            ],
            'fill' => 'M31.44 13.56q.56.57.56 1.4t-.56 1.42L17.56 30.24Q15.81 32 13.31 32t-4.25-1.75l-7.31-7.31Q0 21.19 0 18.69t1.75-4.25L7.69 8.5 2.3 3.12q-.75-.68 0-1.43L3.7.3q.75-.75 1.44 0l5.36 5.4L15.63.56q.56-.56 1.4-.56t1.4.56zM24.12 18l3.07-3L17 4.81 13.31 8.5 17 12.19q.56.56.56 1.4T17 15t-1.4.56-1.41-.56l-3.69-3.69-5.06 5.13-.88.81q-.31.31-.44.75h20z',
            'fill-drip' => [
                'p' => 'M32 20q4 5.81 4 8 0 1.69-1.16 2.84T32 32t-2.84-1.16T28 28q0-1 1-3t2-3.5zm-.56-6.44q.56.57.56 1.4t-.56 1.41L17.56 30.26Q15.81 32 13.31 32t-4.25-1.75l-7.31-7.31Q0 21.19 0 18.69t1.75-4.25L7.69 8.5 2.3 3.12q-.75-.68 0-1.43L3.7.3q.75-.75 1.44 0l5.36 5.4L15.63.56q.56-.56 1.4-.56t1.4.56zM24.12 18l3.07-3L17 4.81 13.31 8.5 17 12.19q.56.56.56 1.4T17 15t-1.4.56-1.41-.56l-3.69-3.69-5.06 5.13-.88.81q-.31.31-.44.75h20z',
                'w' => 1.125,
                'vw' => 36
            ],
            'film' => 'M30.5 4q.63 0 1.06.44T32 5.5v21q0 .63-.44 1.06T30.5 28H30v-1.25q0-.75-.75-.75h-2.5q-.75 0-.75.75V28H6v-1.25Q6 26 5.25 26h-2.5Q2 26 2 26.75V28h-.5q-.63 0-1.06-.44T0 26.5v-21q0-.62.44-1.06T1.5 4H2v1.25Q2 6 2.75 6h2.5Q6 6 6 5.25V4h20v1.25q0 .75.75.75h2.5Q30 6 30 5.25V4h.5zM6 23.25v-2.5Q6 20 5.25 20h-2.5Q2 20 2 20.75v2.5q0 .75.75.75h2.5Q6 24 6 23.25zm0-6v-2.5Q6 14 5.25 14h-2.5Q2 14 2 14.75v2.5q0 .75.75.75h2.5Q6 18 6 17.25zm0-6v-2.5Q6 8 5.25 8h-2.5Q2 8 2 8.75v2.5q0 .75.75.75h2.5Q6 12 6 11.25zm17 13v-6q0-.75-.75-.75H9.75q-.75 0-.75.75v6q0 .75.75.75h12.5q.75 0 .75-.75zm0-10.5v-6Q23 7 22.25 7H9.75Q9 7 9 7.75v6q0 .75.75.75h12.5q.75 0 .75-.75zm7 9.5v-2.5q0-.75-.75-.75h-2.5q-.75 0-.75.75v2.5q0 .75.75.75h2.5q.75 0 .75-.75zm0-6v-2.5q0-.75-.75-.75h-2.5q-.75 0-.75.75v2.5q0 .75.75.75h2.5q.75 0 .75-.75zm0-6v-2.5Q30 8 29.25 8h-2.5Q26 8 26 8.75v2.5q0 .75.75.75h2.5q.75 0 .75-.75z',
            'filter' => 'M30.5 0q1 0 1.38.94t-.32 1.62L20 14.13V30.5q0 .94-.81 1.34t-1.57-.09l-5-3.5Q12 27.75 12 27V14.13L.44 2.56Q-.26 1.87.12.94T1.5 0h29z',
            'filter-circle-dollar' => [
                'p' => 'M3.9 22.9C10.5 8.9 24.5 0 40 0H472c15.5 0 29.5 8.9 36.1 22.9s4.6 30.5-5.2 42.5L396.4 195.6C316.2 212.1 256 283 256 368c0 27.4 6.3 53.4 17.5 76.5c-1.6-.8-3.2-1.8-4.7-2.9l-64-48c-8.1-6-12.8-15.5-12.8-25.6V288.9L9 65.3C-.7 53.4-2.8 36.8 3.9 22.9zM288 368a144 144 0 1 1 288 0 144 144 0 1 1 -288 0zm120.8-32.6c.6-.9 1.8-2.1 4.2-3.4c5.1-2.7 12.5-4.1 18.7-4c8.2 .1 17.1 1.8 26.4 4.1c8.6 2.1 17.3-3.1 19.4-11.7s-3.1-17.3-11.7-19.4c-5.6-1.4-11.6-2.7-17.9-3.7V288c0-8.8-7.2-16-16-16s-16 7.2-16 16v9.5c-6.1 1.2-12.3 3.2-18 6.3c-11.8 6.3-23 18.4-21.8 37.2c1 16 11.7 25.3 21.6 30.7c8.8 4.7 19.7 7.8 28.6 10.3l1.8 .5c10.3 2.9 17.9 5.2 23.2 8.3c4.5 2.7 4.7 4.2 4.7 5.6c.1 2.4-.5 3.7-1 4.5c-.6 1-1.8 2.2-4 3.3c-4.7 2.5-11.8 3.8-18.5 3.6c-9.5-.3-18.5-3.1-29.9-6.8c-1.9-.6-3.8-1.2-5.8-1.8c-8.4-2.6-17.4 2.1-20 10.5s2.1 17.4 10.5 20c1.6 .5 3.3 1 5 1.6l0 0 0 0c7 2.3 15.1 4.8 23.7 6.6v11.4c0 8.8 7.2 16 16 16s16-7.2 16-16V438.7c6.2-1.1 12.5-3.1 18.3-6.2c12.1-6.5 22.3-18.7 21.7-36.9c-.5-16.2-10.3-26.3-20.5-32.3c-9.4-5.6-21.2-8.9-30.5-11.5l-.2 0c-10.4-2.9-18.3-5.2-23.9-8.2c-4.8-2.6-4.8-4-4.8-4.5l0-.1c-.1-1.9 .3-2.9 .8-3.6z',
                'vw' => 576,
                'vh' => 512
            ],
            'filter-circle-xmark' => [
                'p' => 'M3.9 22.9C10.5 8.9 24.5 0 40 0H472c15.5 0 29.5 8.9 36.1 22.9s4.6 30.5-5.2 42.5L396.4 195.6C316.2 212.1 256 283 256 368c0 27.4 6.3 53.4 17.5 76.5c-1.6-.8-3.2-1.8-4.7-2.9l-64-48c-8.1-6-12.8-15.5-12.8-25.6V288.9L9 65.3C-.7 53.4-2.8 36.8 3.9 22.9zM432 224a144 144 0 1 1 0 288 144 144 0 1 1 0-288zm59.3 107.3c6.2-6.2 6.2-16.4 0-22.6s-16.4-6.2-22.6 0L432 345.4l-36.7-36.7c-6.2-6.2-16.4-6.2-22.6 0s-6.2 16.4 0 22.6L409.4 368l-36.7 36.7c-6.2 6.2-6.2 16.4 0 22.6s16.4 6.2 22.6 0L432 390.6l36.7 36.7c6.2 6.2 16.4 6.2 22.6 0s6.2-16.4 0-22.6L454.6 368l36.7-36.7z',
                'vw' => 576,
                'vh' => 512
            ],
            'fingerprint' => 'M16 15.38q.63 0 1.06.43t.44 1.07q.13 7.12-1.81 14Q15.37 32 14.25 32q-.94 0-1.28-.65t-.22-1.22q1.87-6.57 1.75-13.25 0-.63.44-1.07t1.06-.43zm-.06-5.13q2.69.06 4.62 1.94t1.94 4.5q.06 4.81-.69 9.5-.06.25-.18.53t-.57.53-1 .19q-.56-.13-.94-.63t-.25-1.12q.7-4.44.63-8.94 0-1.44-1.06-2.44t-2.5-1.06q-1.57 0-2.5 1t-.94 2.31q.06 4.81-.88 9.56-.12.57-.65.91t-1.13.22-.94-.63-.21-1.12q.87-4.38.8-8.94-.05-2.56 1.79-4.44t4.66-1.87zm-6.9-1.22q.46.4.52 1t-.31 1.1Q7.44 13.3 7.5 16.18q.06 3.87-.63 7.62 0 .2-.15.47t-.6.53-1 .2q-.56-.13-.93-.63t-.25-1.13q.62-3.44.56-7-.06-3.94 2.44-7 .37-.5 1-.56t1.1.34zm6.84-3.9q4.74.06 8.15 3.4t3.47 7.97v.88q0 3.3-.38 6.62-.06.5-.5.94t-1.12.37q-.63-.06-1.03-.56t-.28-1.13q.37-3.5.31-7.06-.06-3.44-2.6-5.9t-6.09-2.53q-1.12 0-2.12.25-.57.12-1.1-.2t-.68-.93.18-1.16.91-.65q1.5-.32 2.88-.32zm15.74 7.62q.38 1.75.38 4.5 0 .63-.44 1.06t-1.06.44-1.06-.47-.44-1.03q0-2.44-.31-3.87-.13-.63.21-1.16t.97-.66q.25-.06.57 0t.53.22.4.4.25.57zm-2.5-5.69q.38.5.25 1.13t-.62.97-1.13.25-.93-.66q-.75-1.06-1.7-1.94t-2-1.56-2.24-1.19-2.44-.78T15.75 3q-5.5-.06-9.25 3.69-3.57 3.56-3.5 8.69v1.5q0 .37-.22.72t-.56.53-.72.18h-.07q-.62 0-1.03-.46t-.4-1.1v-1.31Q-.07 9.06 4.37 4.56 9.06-.06 15.81 0q1.94 0 3.88.5t3.62 1.4 3.19 2.22 2.62 2.94z',
            'fire' => [
                'p' => 'M13.5 1.5q0 1.75 1.1 3.31t2.62 2.94 3.06 2.94 2.63 3.97T24 20q0 4.94-3.54 8.47T12 32t-8.47-3.53T0 20q0-4.87 3.44-8.44.69-.75 1.62-.37T6 12.63v5.31q0 1.69 1.16 2.88T9.94 22q1.69 0 2.87-1.15t1.2-2.85q0-1.06-.6-2.06T12 14.13t-1.73-1.7-1.5-1.93T8 8.13t.47-3.22T10.75.69q.63-.94 1.69-.63T13.5 1.5z',
                'w' => .75,
                'vw' => 24
            ],
            'fire-alt' => [
                'p' => 'M20.25 3.19q3.25 3 5.5 7.37T28 17.63q0 5.93-4.1 10.15T14 32t-9.9-4.22T0 17.63q0-3.38 2.97-8.38T10.5 0q3.56 3.31 6.19 6.94 1.69-2 3.56-3.75zM19 24.5q2.25-1.63 2.84-4.4t-.71-5.23q-.38-.56-.75-1.24l-3.63 4.18Q11 10.5 10.56 10 8.87 12 8 13.25t-1.44 2.5T6 18.19q0 3.5 2.34 5.65T14.2 26q2.68 0 4.8-1.5z',
                'w' => .875,
                'vw' => 28
            ],
            'fire-burner' => [
                'p' => 'M293.5 3.8c19.7 17.8 38.2 37 55.5 57.7c7.9-9.9 16.8-20.7 26.5-29.5c5.6-5.1 14.4-5.1 20 0c24.7 22.7 45.6 52.7 60.4 81.1c14.5 28 24.2 58.8 24.2 79C480 280 408.7 352 320 352c-89.7 0-160-72.1-160-159.8c0-26.4 12.7-60.7 32.4-92.6c20-32.4 48.1-66.1 81.4-95.8c2.8-2.5 6.4-3.8 10-3.7c3.5 0 7 1.3 9.8 3.8zM370 273c30-21 38-63 20-96c-2-4-4-8-7-12l-36 42s-58-74-62-79c-30 37-45 58-45 82c0 49 36 78 81 78c18 0 34-5 49-15zM32 288c0-17.7 14.3-32 32-32H96c17.7 0 32 14.3 32 32s-14.3 32-32 32v64H544V320c-17.7 0-32-14.3-32-32s14.3-32 32-32h32c17.7 0 32 14.3 32 32v96c17.7 0 32 14.3 32 32v64c0 17.7-14.3 32-32 32H32c-17.7 0-32-14.3-32-32V416c0-17.7 14.3-32 32-32V288zM320 480a32 32 0 1 0 0-64 32 32 0 1 0 0 64zm160-32a32 32 0 1 0 -64 0 32 32 0 1 0 64 0zM192 480a32 32 0 1 0 0-64 32 32 0 1 0 0 64z',
                'vw' => 640,
                'vh' => 512
            ],
            'fire-extinguisher' => [
                'p' => 'M27.13 1.63q.37-.07.62.18t.25.57v7.25q0 .3-.25.56t-.63.19l-10.5-1.75Q16 8.5 16 7.5h-2.5v1.7q.94.25 1.78.81t1.44 1.32.94 1.72T18 15v15.5q0 .63-.44 1.07T16.5 32h-9q-.62 0-1.06-.43T6 30.5V15.07Q6 13 7.28 11.35t3.22-2.16V7.5H8.19l-.94.07-.94.15-.75.28-.78.5-.63.72-.68 1-.6 1.35q-.19.56-.78.8t-1.15.04-.82-.81 0-1.16Q2 5.7 5.75 4.75q-.7-1.69.34-3.22T9 0q1.76 0 2.82 1.4t.56 3.1H16q0-1 .62-1.12zM9 4.5q.44 0 .72-.28T10 3.5t-.28-.72T9 2.5t-.72.28T8 3.5t.28.72.72.28z',
                'w' => .875,
                'vw' => 28
            ],
            'fire-flame-curved' => [
                'p' => 'M153.6 29.9l16-21.3C173.6 3.2 180 0 186.7 0C198.4 0 208 9.6 208 21.3V43.5c0 13.1 5.4 25.7 14.9 34.7L307.6 159C356.4 205.6 384 270.2 384 337.7C384 434 306 512 209.7 512H192C86 512 0 426 0 320v-3.8c0-48.8 19.4-95.6 53.9-130.1l3.5-3.5c4.2-4.2 10-6.6 16-6.6C85.9 176 96 186.1 96 198.6V288c0 35.3 28.7 64 64 64s64-28.7 64-64v-3.9c0-18-7.2-35.3-19.9-48l-38.6-38.6c-24-24-37.5-56.7-37.5-90.7c0-27.7 9-54.8 25.6-76.9z',
                'vw' => 384,
                'vh' => 512
            ],
            'fire-flame-simple' => [
                'p' => 'M372.5 256.5l-.7-1.9C337.8 160.8 282 76.5 209.1 8.5l-3.3-3C202.1 2 197.1 0 192 0s-10.1 2-13.8 5.5l-3.3 3C102 76.5 46.2 160.8 12.2 254.6l-.7 1.9C3.9 277.3 0 299.4 0 321.6C0 426.7 86.8 512 192 512s192-85.3 192-190.4c0-22.2-3.9-44.2-11.5-65.1zm-90.8 49.5c4.1 9.3 6.2 19.4 6.2 29.5c0 53-43 96.5-96 96.5s-96-43.5-96-96.5c0-10.1 2.1-20.3 6.2-29.5l1.9-4.3c15.8-35.4 37.9-67.7 65.3-95.1l8.9-8.9c3.6-3.6 8.5-5.6 13.6-5.6s10 2 13.6 5.6l8.9 8.9c27.4 27.4 49.6 59.7 65.3 95.1l1.9 4.3z',
                'vw' => 384,
                'vh' => 512
            ],
            'first-aid' => [
                'p' => 'M0 5q0-1.25.88-2.13T3 2h3v28H3q-1.25 0-2.13-.88T0 27V5zm8 25V2h20v28H8zm4-15.5v3q0 .5.5.5H16v3.5q0 .5.5.5h3q.5 0 .5-.5V18h3.5q.5 0 .5-.5v-3q0-.5-.5-.5H20v-3.5q0-.5-.5-.5h-3q-.5 0-.5.5V14h-3.5q-.5 0-.5.5zM33 2q1.25 0 2.13.88T36 5v22q0 1.25-.88 2.13T33 30h-3V2h3z',
                'w' => 1.125,
                'vw' => 36
            ],
            'fish' => [
                'p' => 'M20.44 6q3 0 6.03 1.28t5.03 3 3.25 3.31T36 16t-1.25 2.4-3.25 3.32-5.03 3T20.44 26q-3.88 0-7.38-1.88t-5.87-4.5l-5.5 4.13q-.57.44-1.19.1t-.5-.98L1.56 16 0 9.12q-.13-.62.5-.96t1.25.09l5.44 4.13q2.37-2.63 5.87-4.5T20.44 6zm5.47 11.5q.59 0 1.06-.44t.47-1.06-.47-1.06-1.03-.44q-.32 0-.6.12t-.5.32-.3.47-.1.59q0 .62.43 1.06t1.04.44z',
                'w' => 1.125,
                'vw' => 36
            ],
            'fish-fins' => [
                'p' => 'M275.2 38.4c-10.6-8-25-8.5-36.3-1.5S222 57.3 224.6 70.3l9.7 48.6c-19.4 9-36.9 19.9-52.4 31.5c-15.3 11.5-29 23.9-40.7 36.3L48.1 132.4c-12.5-7.3-28.4-5.3-38.7 4.9S-3 163.3 4.2 175.9L50 256 4.2 336.1c-7.2 12.6-5 28.4 5.3 38.6s26.1 12.2 38.7 4.9l93.1-54.3c11.8 12.3 25.4 24.8 40.7 36.3c15.5 11.6 33 22.5 52.4 31.5l-9.7 48.6c-2.6 13 3.1 26.3 14.3 33.3s25.6 6.5 36.3-1.5l77.6-58.2c54.9-4 101.5-27 137.2-53.8c39.2-29.4 67.2-64.7 81.6-89.5c5.8-9.9 5.8-22.2 0-32.1c-14.4-24.8-42.5-60.1-81.6-89.5c-35.8-26.8-82.3-49.8-137.2-53.8L275.2 38.4zM384 256a32 32 0 1 1 64 0 32 32 0 1 1 -64 0z',
                'vw' => 576,
                'vh' => 512
            ],
            'fist-raised' => [
                'p' => 'M16 10h-3q-.5 0-1 .19V1q0-.44.28-.72T13 0h2q.44 0 .72.28T16 1v9zm8 6v5.5q0 1.06-.5 2.28t-1.25 1.97L20 28v4H4v-4l-1.69-1.69Q0 24 0 20.7v-4.9q.5.2 1 .2h2q1.12 0 2-.75.87.75 2 .75h2q.69 0 1.37-.38.75 1.63 2.44 2.2-1.31 1.12-3 3.62l-.37.56q-.07.13-.07.31 0 .25.2.44l.87.5q.12.13.25.13.25 0 .44-.25l.37-.57q1.13-1.68 1.88-2.53t1.75-1.4T17.5 18q.5 0 .5-.5v-1q0-.5-.81-.5h-2.2q-1.24 0-2.12-.87T12 13q0-.44.28-.72T13 12h7q1.62 0 2.81 1.16T24 16zm-2-5.63Q20.94 10 20 10h-2V3q0-.44.28-.72T19 2h2q.44 0 .72.28T22 3v7.38zM1 14q-.44 0-.72-.28T0 13V5q0-.44.28-.72T1 4h2q.44 0 .72.28T4 5v8q0 .44-.28.72T3 14H1zm6 0q-.44 0-.72-.28T6 13V3q0-.44.28-.72T7 2h2q.44 0 .72.28T10 3v10q0 .44-.28.72T9 14H7z',
                'w' => .75,
                'vw' => 24
            ],
            'flag' => 'M21.88 6.19q3 0 7.3-2 .95-.44 1.88.12t.94 1.7v15.18q0 1.06-.87 1.62-3.7 2.57-7.63 2.57-2.25 0-5.5-1.1t-4.87-1.1q-3.88 0-7.13 1.45v5.87q0 .63-.44 1.06T4.5 32h-1q-.63 0-1.06-.44T2 30.5V6.38Q.5 5.3.5 3.5.5 2 1.56.97T4.12 0Q5.5.07 6.47 1T7.5 3.25v.25q0 .63-.25 1.25Q9.31 4 11.53 4T17 5.1t4.88 1.09z',
            'flag-checkered' => 'M15.19 11.88q.56.12 2.12.65t2.5.72v4.25q-.56-.13-2.12-.66t-2.5-.71v-4.25zm13.94-7.7q1-.43 1.93.13t.94 1.7v15.18q0 1.06-.87 1.62-3.7 2.57-7.63 2.57-2.25 0-5.53-1.07t-4.84-1.06q-3.88 0-7.13 1.38v5.87q0 .63-.44 1.06T4.5 32h-1q-.63 0-1.06-.44T2 30.5V6.38Q.5 5.3.5 3.5.5 2 1.56.97T4.12 0Q5 .06 5.75.5t1.22 1.16.53 1.6q.06.8-.25 1.5Q9.31 4 11.53 4T17 5.1t4.88 1.09q3 0 7.25-2zm-18.5 16.2v-4.5Q8.5 16.05 6 17v4.38q2.19-.82 4.63-1zM29 11.94V7.5q-2.44 1.06-4.63 1.44v4.5q-2.18.37-4.56-.19V9q-.75-.13-4.62-1.38v4.25q-2.38-.68-4.57-.37V7.06Q9.44 7.13 6 8.44v4.37q2.87-1.06 4.62-1.31v4.37q2.5-.18 4.57.25v4.25q.69.13 4.62 1.38V17.5q2.38.69 4.57.37v4.5q2.3-.25 4.62-1.75v-4.37q-2 1.25-4.63 1.62v-4.43q1.7-.25 4.63-1.5z',
            'flag-usa' => 'M2 0q.81 0 1.4.6T4 2v29q0 .44-.28.72T3 32H1q-.44 0-.72-.28T0 31V2Q0 1.19.6.6T2 0zm16.75 19q3.88 1 6.25 1 3.19 0 7-1.5v2.25q0 1.31-1.13 1.81-3.06 1.32-5.65 1.4t-4.56-.5-4-1.12-4.75-.22T6 24v-4.31q2.69-1.2 5.16-1.47t3.87-.06 3.72.84zm0-6q3.88 1 6.25 1 3.19 0 7-1.5v3.88q-2.69 1.12-5.16 1.4t-3.87.06-3.72-.78q-2.38-.62-3.81-.87t-4.07 0T6 17.5v-3.8q2.69-1.19 5.15-1.47t3.88-.06 3.72.84zm.56-7.88q-.75-.25-1.31-.37v2.06q.31.07.75.2 3.31 1 5.38 1Q27.5 8 32 6.05v4.32q-2.69 1.12-5.16 1.4t-3.87.07-3.72-.79q-2.38-.62-3.81-.87t-4.07 0T6 11.5V2Q9.06.63 11.5.22t3.87-.06 2.97.81 2.78.88 3.28.06T28.94.25Q30-.3 31 .22t1 1.72v1.94q-1.94.87-3.56 1.37t-2.75.66-2.41.03-1.97-.28-2-.53zM10 8q.44 0 .72-.28T11 7t-.28-.72T10 6t-.72.28T9 7t.28.72T10 8zm0-3.5q.44 0 .72-.28t.28-.69-.28-.72T10 2.5t-.72.31-.28.72.28.69.72.28zm4 3q.44 0 .72-.28t.28-.69-.28-.72T14 5.5t-.72.31-.28.72.28.69.72.28zM14 4q.44 0 .72-.28t.28-.69-.28-.72T14 2t-.72.31T13 3q0 .25.1.44t.21.31.32.19T14 4z',
            'flask' => [
                'p' => 'M27.31 25.25q1.38 2.19.13 4.47T23.56 32H4.44Q1.8 32 .56 29.72t.13-4.47L8 13.44V4h-.5q-.62 0-1.06-.44T6 2.5v-1q0-.63.44-1.06T7.5 0h13q.63 0 1.06.44T22 1.5v1q0 .62-.44 1.06T20.5 4H20v9.44zM8.63 20h10.74l-3-4.88Q16 14.63 16 14V4h-4v10q0 .63-.38 1.13z',
                'w' => .875,
                'vw' => 28
            ],
            'flask-vial' => [
                'p' => 'M175 389.4c-9.8 16-15 34.3-15 53.1c-10 3.5-20.8 5.5-32 5.5c-53 0-96-43-96-96V64C14.3 64 0 49.7 0 32S14.3 0 32 0H96h64 64c17.7 0 32 14.3 32 32s-14.3 32-32 32V309.9l-49 79.6zM96 64v96h64V64H96zM352 0H480h32c17.7 0 32 14.3 32 32s-14.3 32-32 32V214.9L629.7 406.2c6.7 10.9 10.3 23.5 10.3 36.4c0 38.3-31.1 69.4-69.4 69.4H261.4c-38.3 0-69.4-31.1-69.4-69.4c0-12.8 3.6-25.4 10.3-36.4L320 214.9V64c-17.7 0-32-14.3-32-32s14.3-32 32-32h32zm32 64V224c0 5.9-1.6 11.7-4.7 16.8L330.5 320h171l-48.8-79.2c-3.1-5-4.7-10.8-4.7-16.8V64H384z',
                'vw' => 640,
                'vh' => 512
            ],
            'floppy-disk' => [
                'p' => 'M64 32C28.7 32 0 60.7 0 96V416c0 35.3 28.7 64 64 64H384c35.3 0 64-28.7 64-64V173.3c0-17-6.7-33.3-18.7-45.3L352 50.7C340 38.7 323.7 32 306.7 32H64zm0 96c0-17.7 14.3-32 32-32H288c17.7 0 32 14.3 32 32v64c0 17.7-14.3 32-32 32H96c-17.7 0-32-14.3-32-32V128zM224 288a64 64 0 1 1 0 128 64 64 0 1 1 0-128z',
                'vw' => 448,
                'vh' => 512
            ],
            'florin-sign' => [
                'p' => 'M314.7 32c-38.8 0-73.7 23.3-88.6 59.1L170.7 224H64c-17.7 0-32 14.3-32 32s14.3 32 32 32h80L98.9 396.3c-5 11.9-16.6 19.7-29.5 19.7H32c-17.7 0-32 14.3-32 32s14.3 32 32 32H69.3c38.8 0 73.7-23.3 88.6-59.1L213.3 288H320c17.7 0 32-14.3 32-32s-14.3-32-32-32H240l45.1-108.3c5-11.9 16.6-19.7 29.5-19.7H352c17.7 0 32-14.3 32-32s-14.3-32-32-32H314.7z',
                'vw' => 384,
                'vh' => 512
            ],
            'flushed' => 'M21.5 12.5q.63 0 1.06.44T23 14t-.44 1.06-1.06.44-1.06-.44T20 14t.44-1.06 1.06-.44zm-12 0q.63 0 1.06.44T11 14t-.44 1.06-1.06.44-1.06-.44T8 14t.44-1.06 1.06-.44zm6-12q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zM5 14q0 1.88 1.31 3.19t3.2 1.31 3.18-1.31T14 14t-1.31-3.19T9.5 9.5t-3.19 1.3T5 14zm14.5 11q1 0 1-1t-1-1h-8q-1 0-1 1t1 1h8zm2-6.5q1.88 0 3.19-1.31t1.31-3.2-1.31-3.18T21.5 9.5t-3.19 1.31T17 14t1.31 3.19 3.19 1.31z',
            'folder' => 'M29 8q1.25 0 2.13.88T32 11v14q0 1.25-.88 2.13T29 28H3q-1.25 0-2.13-.88T0 25V7q0-1.25.88-2.13T3 4h10l4 4h12z',
            'folder-closed' => [
                'p' => 'M448 480H64c-35.3 0-64-28.7-64-64V192H512V416c0 35.3-28.7 64-64 64zm64-320H0V96C0 60.7 28.7 32 64 32H192c20.1 0 39.1 9.5 51.2 25.6l19.2 25.6c6 8.1 15.5 12.8 25.6 12.8H448c35.3 0 64 28.7 64 64z',
                'vw' => 512,
                'vh' => 512
            ],
            'folder-minus' => 'M29 8q1.25 0 2.13.88T32 11v14q0 1.25-.88 2.13T29 28H3q-1.25 0-2.13-.88T0 25V7q0-1.25.88-2.13T3 4h10l4 4h12zm-6 10.5v-1q0-.44-.28-.72T22 16.5H10q-.44 0-.72.28T9 17.5v1q0 .44.28.72t.72.28h12q.44 0 .72-.28t.28-.72z',
            'folder-open' => [
                'p' => 'M35.81 18.25 31.25 26q-.44.81-1.47 1.4t-1.97.6h-25q-.87 0-1.31-.75t0-1.5L6.06 18q.44-.81 1.47-1.4T9.5 16h25q.88 0 1.31.75t0 1.5zM9.5 14q-1.06 0-2.06.38T5.66 15.4 4.3 17L0 24.38V7q0-1.25.87-2.13T3 4h10l4 4h10q1.25 0 2.12.88T30 11v3H9.5z',
                'w' => 1.125,
                'vw' => 36
            ],
            'folder-plus' => 'M29 8q1.25 0 2.13.88T32 11v14q0 1.25-.88 2.13T29 28H3q-1.25 0-2.13-.88T0 25V7q0-1.25.88-2.13T3 4h10l4 4h12zm-6 10.5v-1q0-.44-.28-.72T22 16.5h-4.5V12q0-.44-.28-.72T16.5 11h-1q-.44 0-.72.28t-.28.72v4.5H10q-.44 0-.72.28T9 17.5v1q0 .44.28.72t.72.28h4.5V24q0 .44.28.72t.72.28h1q.44 0 .72-.28t.28-.72v-4.5H22q.44 0 .72-.28t.28-.72z',
            'folder-tree' => [
                'p' => 'M64 32C64 14.3 49.7 0 32 0S0 14.3 0 32v96V384c0 35.3 28.7 64 64 64H256V384H64V160H256V96H64V32zM288 192c0 17.7 14.3 32 32 32H544c17.7 0 32-14.3 32-32V64c0-17.7-14.3-32-32-32H445.3c-8.5 0-16.6-3.4-22.6-9.4L409.4 9.4c-6-6-14.1-9.4-22.6-9.4H320c-17.7 0-32 14.3-32 32V192zm0 288c0 17.7 14.3 32 32 32H544c17.7 0 32-14.3 32-32V352c0-17.7-14.3-32-32-32H445.3c-8.5 0-16.6-3.4-22.6-9.4l-13.3-13.3c-6-6-14.1-9.4-22.6-9.4H320c-17.7 0-32 14.3-32 32V480z',
                'vw' => 576,
                'vh' => 512
            ],
            'font' => [
                'p' => 'M27 26q.44 0 .72.28T28 27v2q0 .44-.28.72T27 30h-8.5q-.44 0-.72-.28T17.5 29v-2q0-.44.28-.72t.72-.28h1.44l-1.63-5.06H9.7L8 26h1.5q.44 0 .72.28t.28.72v2q0 .44-.28.72T9.5 30H1q-.44 0-.72-.28T0 29v-2q0-.44.28-.72T1 26h1.69l8.06-23.31q.25-.7 1-.7h4.5q.75 0 1 .7L25.31 26h1.7zm-16.13-9.25h6.2L14.5 9q-.38-1.06-.5-2.13-.19 1.07-.5 2.13z',
                'w' => .875,
                'vw' => 28
            ],
            'font-awesome' => [
                'p' => 'M448 48V384c-63.1 22.5-82.3 32-119.5 32c-62.8 0-86.6-32-149.3-32c-20.6 0-36.6 3.6-51.2 8.2v-64c14.6-4.6 30.6-8.2 51.2-8.2c62.7 0 86.5 32 149.3 32c20.4 0 35.6-3 55.5-9.3v-208c-19.9 6.3-35.1 9.3-55.5 9.3c-62.8 0-86.6-32-149.3-32c-50.8 0-74.9 20.6-115.2 28.7V448c0 17.7-14.3 32-32 32s-32-14.3-32-32V64C0 46.3 14.3 32 32 32s32 14.3 32 32V76.7c40.3-8 64.4-28.7 115.2-28.7c62.7 0 86.5 32 149.3 32c37.1 0 56.4-9.5 119.5-32z',
                'vw' => 448,
                'vh' => 512
            ],
            'football' => [
                'p' => 'M247.5 25.4c-13.5 3.3-26.4 7.2-38.6 11.7C142.9 61.6 96.7 103.6 66 153.6c-18.3 29.8-30.9 62.3-39.2 95.4L264.5 486.6c13.5-3.3 26.4-7.2 38.6-11.7c66-24.5 112.2-66.5 142.9-116.5c18.3-29.8 30.9-62.3 39.1-95.3L247.5 25.4zM495.2 205.3c6.1-56.8 1.4-112.2-7.7-156.4c-2.7-12.9-13-22.9-26.1-25.1c-58.2-9.7-109.9-12-155.6-7.9L495.2 205.3zM206.1 496L16.8 306.7c-6.1 56.8-1.4 112.2 7.7 156.4c2.7 12.9 13 22.9 26.1 25.1c58.2 9.7 109.9 12 155.6 7.9zm54.6-331.3c6.2-6.2 16.4-6.2 22.6 0l64 64c6.2 6.2 6.2 16.4 0 22.6s-16.4 6.2-22.6 0l-64-64c-6.2-6.2-6.2-16.4 0-22.6zm-48 48c6.2-6.2 16.4-6.2 22.6 0l64 64c6.2 6.2 6.2 16.4 0 22.6s-16.4 6.2-22.6 0l-64-64c-6.2-6.2-6.2-16.4 0-22.6zm-48 48c6.2-6.2 16.4-6.2 22.6 0l64 64c6.2 6.2 6.2 16.4 0 22.6s-16.4 6.2-22.6 0l-64-64c-6.2-6.2-6.2-16.4 0-22.6z',
                'vw' => 512,
                'vh' => 512
            ],
            'football-ball' => 'M30.13 3.75q.8 3.25.87 6.56L21.19.5q3.37.06 6.56.94 1.88.5 2.38 2.31zM.88 28.25Q.06 25 0 21.69l9.81 9.81q-3.37-.06-6.56-.94-1.87-.5-2.37-2.31zM.24 17.69Q1.3 10.75 6.05 6.22T17.26.75l13.5 13.56q-1.07 6.94-5.82 11.47t-11.19 5.47zM20.13 10l-1.82 1.75L16.56 10q-.37-.38-.69 0l-.75.69q-.3.37 0 .75l1.82 1.75-1.44 1.44-1.75-1.82q-.38-.31-.75 0l-.69.7q-.37.37 0 .74L14.06 16l-1.37 1.44-1.82-1.81q-.31-.32-.68 0l-.7.75q-.37.3 0 .68l1.76 1.75-1.75 1.82q-.38.31 0 .68l.68.7q.38.37.7 0l1.8-1.76L14.45 22q.37.38.68 0l.75-.69q.32-.37 0-.68l-1.8-1.82 1.43-1.37 1.75 1.75q.37.37.75 0l.69-.69q.37-.37 0-.75L16.94 16l1.37-1.44 1.82 1.82q.3.3.68 0l.7-.75q.37-.32 0-.7l-1.76-1.74 1.75-1.82q.38-.3 0-.68l-.69-.7q-.37-.37-.68 0z',
            'forward' => 'M31.25 14.44q.75.62.75 1.56t-.75 1.56l-12 10q-.94.82-2.1.25T16 26.01V6q0-1.26 1.16-1.82t2.09.25zm-16 0q.75.62.75 1.56t-.75 1.56l-12 10q-.94.82-2.1.25T0 26.01V6q0-1.26 1.16-1.82t2.09.25z',
            'forward-fast' => [
                'p' => 'M18.4 445c11.2 5.3 24.5 3.6 34.1-4.4L224 297.7V416c0 12.4 7.2 23.7 18.4 29s24.5 3.6 34.1-4.4L448 297.7V416c0 17.7 14.3 32 32 32s32-14.3 32-32V96c0-17.7-14.3-32-32-32s-32 14.3-32 32V214.3L276.5 71.4c-9.5-7.9-22.8-9.7-34.1-4.4S224 83.6 224 96V214.3L52.5 71.4c-9.5-7.9-22.8-9.7-34.1-4.4S0 83.6 0 96V416c0 12.4 7.2 23.7 18.4 29z',
                'vw' => 512,
                'vh' => 512
            ],
            'forward-step' => [
                'p' => 'M52.5 440.6c-9.5 7.9-22.8 9.7-34.1 4.4S0 428.4 0 416V96C0 83.6 7.2 72.3 18.4 67s24.5-3.6 34.1 4.4l192 160L256 241V96c0-17.7 14.3-32 32-32s32 14.3 32 32V416c0 17.7-14.3 32-32 32s-32-14.3-32-32V271l-11.5 9.6-192 160z',
                'vw' => 320,
                'vh' => 512
            ],
            'franc-sign' => [
                'p' => 'M80 32C62.3 32 48 46.3 48 64V224v96H32c-17.7 0-32 14.3-32 32s14.3 32 32 32H48v64c0 17.7 14.3 32 32 32s32-14.3 32-32V384h80c17.7 0 32-14.3 32-32s-14.3-32-32-32H112V256H256c17.7 0 32-14.3 32-32s-14.3-32-32-32H112V96H288c17.7 0 32-14.3 32-32s-14.3-32-32-32H80z',
                'vw' => 320,
                'vh' => 512
            ],
            'frog' => [
                'p' => 'M27.94 6.06q3.62 1.25 6.19 2.63 1.87 1 1.87 3.19 0 .87-.53 1.8t-1.34 1.38l-9.63 5.38L30.63 27H34q.81 0 1.4.6T36 29q0 .44-.28.72T35 30h-5.62l-7.44-7.87q.12-1.5-.4-2.88t-1.07-1.97-1.1-.97q-1.68-1.3-3.87-1.3t-3.94 1.3l-2.19 1.63q-.75.62-.15 1.4t1.4.22l2-1.56q2.2-1.62 4.57-.69 2 .88 2.62 2.94T19.25 24l-2.19 3H22q.81 0 1.4.6T24 29q0 .44-.28.72T23 30H4q-1.63 0-2.81-1.16t-1.2-2.78Q0 23 .88 20.1t2.52-5.3 3.91-4.25 5-3.03 5.75-1.4q.38-1.82 1.75-2.97T23 2t3.19 1.16 1.75 2.9zM23 8.5q.63 0 1.06-.44T24.5 7t-.44-1.06T23 5.5t-1.06.44T21.5 7t.44 1.06T23 8.5z',
                'w' => 1.125,
                'vw' => 36
            ],
            'frown' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm5 10.5q-.81 0-1.4.6t-.6 1.4.6 1.4 1.4.6 1.4-.6.6-1.4-.6-1.4-1.4-.6zm-10 0q-.81 0-1.4.6T8.5 13t.6 1.4 1.4.6 1.4-.6.6-1.4-.6-1.4-1.4-.6zm10.63 13.63q.43.5 1 .37t.71-.63-.15-1Q19.87 20 15.5 20t-7.19 3.38q-.37.43-.15.96t.75.63.97-.34Q12.05 22 15.5 22t5.63 2.63z',
            'frown-open' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zM8.5 13q0 .81.6 1.4t1.4.6 1.4-.6.6-1.4-.6-1.4-1.4-.6-1.4.6-.6 1.4zm11.69 11.44q.56.19.97-.19t.34-.94q-.31-1.69-2.22-2.75T15.5 19.5q-1.19 0-2.44.4t-2.31 1.32-1.25 2.1q-.06.56.34.93t.97.19q3.07-.94 4.7-.94t4.68.94zM20.5 15q.81 0 1.4-.6t.6-1.4-.6-1.4-1.4-.6-1.4.6-.6 1.4.6 1.4 1.4.6z',
            'fullstar' => 'M17.6 2.78a1.77 1.77 0 0 0-3.2 0l-3.57 7.35-7.97 1.18a1.77 1.77 0 0 0-1 3.02l5.8 5.73-1.37 8.09a1.78 1.78 0 0 0 2.6 1.87l7.12-3.8 7.13 3.8a1.78 1.78 0 0 0 2.6-1.87l-1.38-8.1 5.79-5.72a1.78 1.78 0 0 0-.99-3.02l-7.98-1.18-3.57-7.35z',
            'funnel-dollar' => [
                'p' => 'M27.06 10.38q-3.94 1-6.5 4.21T18 22q0 5.06 3.69 8.63-.5 1-1.6 1.28T18 31.5l-5-3.75q-1-.75-1-2V16L.56 3.37q-.94-.93-.4-2.15T2 0h30q1.31 0 1.84 1.22t-.4 2.16zM30 12q4.13 0 7.06 2.94T40 22t-2.94 7.06T30 32t-7.06-2.94T20 22t2.94-7.06T30 12zm1 15q1.13-.06 1.9-.88t.79-1.93q0-.94-.57-1.7t-1.43-1.05l-2.82-.82q-.56-.18-.56-.81 0-.31.22-.56t.53-.25h1.75q.44 0 .81.25.38.19.63-.07l.75-.68q.44-.44-.07-.75-.87-.7-1.93-.75v-1q0-.5-.5-.5h-1q-.5 0-.5.5v1q-.57 0-1.07.25t-.84.62-.56.88-.22 1.06q0 .94.56 1.69t1.44 1.06l2.81.81q.56.2.56.82 0 .12-.06.28t-.15.28-.25.19-.29.06h-1.75q-.43 0-.8-.25-.38-.19-.63.06l-.75.69q-.44.44.06.75.87.69 1.94.75v1q0 .5.5.5h1q.5 0 .5-.5v-1z',
                'w' => 1.25,
                'vw' => 40
            ],
            'futbol' => 'M31.5 16q0 6.44-4.53 10.97T16 31.5 5.03 26.97.5 16 5.03 5.03 16 .5t10.97 4.53T31.5 16zm-3 0-1.63 1.38-3.93-3.63L24 8.5l2.13.19Q23.68 5.3 19.8 4.13l.88 1.93-4.7 2.63-4.68-2.63.88-1.93Q8.3 5.3 5.87 8.69l2.2-.19 1 5.25-3.95 3.63L3.5 16q0 4.13 2.37 7.38l.5-2.13 5.32.69 2.3 4.81-1.87 1.13q3.88 1.25 7.75 0L18 26.75l2.31-4.81 5.32-.69.5 2.13Q28.5 20.13 28.5 16zM13 20.38l-1.88-5.75 4.88-3.5 4.88 3.5L19 20.37h-6z',
            'g' => [
                'p' => 'M224 96C135.6 96 64 167.6 64 256s71.6 160 160 160c77.4 0 142-55 156.8-128H256c-17.7 0-32-14.3-32-32s14.3-32 32-32H400c25.8 0 49.6 21.4 47.2 50.6C437.8 389.6 341.4 480 224 480C100.3 480 0 379.7 0 256S100.3 32 224 32c57.4 0 109.7 21.6 149.3 57c13.2 11.8 14.3 32 2.5 45.2s-32 14.3-45.2 2.5C302.3 111.4 265 96 224 96z',
                'vw' => 448,
                'vh' => 512
            ],
            'gamepad' => [
                'p' => 'M30 6q4.13 0 7.06 2.94T40 16t-2.94 7.06T30 26q-4.19 0-7.13-3h-5.75Q14.2 26 10 26q-4.13 0-7.06-2.94T0 16t2.94-7.06T10 6h20zM16 17.25v-2.5q0-.75-.75-.75H12v-3.25q0-.75-.75-.75h-2.5Q8 10 8 10.75V14H4.75Q4 14 4 14.75v2.5q0 .75.75.75H8v3.25q0 .75.75.75h2.5q.75 0 .75-.75V18h3.25q.75 0 .75-.75zm11.5 4.25q1.25 0 2.13-.88t.87-2.12-.88-2.13-2.12-.87-2.13.88-.87 2.12.88 2.13 2.12.87zm5-5q1.25 0 2.13-.88t.87-2.12-.88-2.13-2.12-.87-2.13.88-.87 2.12.88 2.13 2.12.87z',
                'w' => 1.248,
                'vw' => 40
            ],
            'gas-pump' => 'M21 28q.44 0 .72.28T22 29v2q0 .44-.28.72T21 32H1q-.44 0-.72-.28T0 31v-2q0-.44.28-.72T1 28h20zm9.81-21.31q1.2 1.19 1.2 2.87V23.5q0 2-1.48 3.34T27 28q-1.69-.19-2.84-1.56T23 23.25V21.5q0-1.06-.72-1.78T20.5 19H20v7H2V4q0-1.63 1.2-2.81T6-.01h10q1.63 0 2.81 1.2T20.01 4v12h.5q2.24 0 3.87 1.63T26 21.5v2q0 .63.44 1.06t1.06.44 1.07-.44.43-1.06V13.44q-1.25-.2-2.12-1.16T26 10V6.12l-2.37-2.37q-.25-.25-.25-.69t.25-.69l.75-.75q.25-.25.69-.25t.69.25zM16 12V4H6v8h10z',
            'gauge' => [
                'p' => 'M0 256a256 256 0 1 1 512 0A256 256 0 1 1 0 256zm320 96c0-26.9-16.5-49.9-40-59.3V88c0-13.3-10.7-24-24-24s-24 10.7-24 24V292.7c-23.5 9.5-40 32.5-40 59.3c0 35.3 28.7 64 64 64s64-28.7 64-64zM144 176a32 32 0 1 0 0-64 32 32 0 1 0 0 64zm-16 80a32 32 0 1 0 -64 0 32 32 0 1 0 64 0zm288 32a32 32 0 1 0 0-64 32 32 0 1 0 0 64zM400 144a32 32 0 1 0 -64 0 32 32 0 1 0 64 0z',
                'vw' => 512,
                'vh' => 512
            ],
            'gauge-high' => [
                'p' => 'M0 256a256 256 0 1 1 512 0A256 256 0 1 1 0 256zM288 96a32 32 0 1 0 -64 0 32 32 0 1 0 64 0zM256 416c35.3 0 64-28.7 64-64c0-17.4-6.9-33.1-18.1-44.6L366 161.7c5.3-12.1-.2-26.3-12.3-31.6s-26.3 .2-31.6 12.3L257.9 288c-.6 0-1.3 0-1.9 0c-35.3 0-64 28.7-64 64s28.7 64 64 64zM176 144a32 32 0 1 0 -64 0 32 32 0 1 0 64 0zM96 288a32 32 0 1 0 0-64 32 32 0 1 0 0 64zm352-32a32 32 0 1 0 -64 0 32 32 0 1 0 64 0z',
                'vw' => 512,
                'vh' => 512
            ],
            'gauge-simple' => [
                'p' => 'M0 256a256 256 0 1 1 512 0A256 256 0 1 1 0 256zm320 96c0-26.9-16.5-49.9-40-59.3V88c0-13.3-10.7-24-24-24s-24 10.7-24 24V292.7c-23.5 9.5-40 32.5-40 59.3c0 35.3 28.7 64 64 64s64-28.7 64-64z',
                'vw' => 512,
                'vh' => 512
            ],
            'gauge-simple-high' => [
                'p' => 'M0 256a256 256 0 1 1 512 0A256 256 0 1 1 0 256zm320 96c0-15.9-5.8-30.4-15.3-41.6l76.6-147.4c6.1-11.8 1.5-26.3-10.2-32.4s-26.2-1.5-32.4 10.2L262.1 288.3c-2-.2-4-.3-6.1-.3c-35.3 0-64 28.7-64 64s28.7 64 64 64s64-28.7 64-64z',
                'vw' => 512,
                'vh' => 512
            ],
            'gavel' => 'M31.56 12.44q.44.44.44 1.06t-.44 1.06l-7.75 7.82q-.5.44-1.1.44t-1.02-.44l-1.44-1.44q-.44-.44-.44-1.06t.44-1.07l.38-.31-2.5-2.5-5.07 5.06.38.38q.56.56.56 1.4t-.56 1.41l-7.2 7.19q-.55.56-1.4.56t-1.4-.56L.56 28.56Q0 28 0 27.16t.56-1.4l7.19-7.2q.56-.56 1.4-.56t1.41.56l.38.38L16 13.87l-2.5-2.5-.31.38q-.44.44-1.07.44t-1.06-.44l-1.44-1.44q-.44-.44-.44-1.03t.44-1.1L17.44.45Q17.87 0 18.5 0t1.06.44l1.38 1.43q.43.44.43 1.07T20.94 4l-.32.31 7.07 7.07.31-.32q.44-.43 1.06-.43t1.07.43z',
            'gear' => 'cog',
            'gears' => 'cogs',
            'gem' => [
                'p' => 'M30.38 0 36 10h-6.31L25.37 0h5zm-8 0 4.3 10H9.32l4.32-10h8.75zM5.63 0h5L6.3 10H0zM0 12h6.31l7.7 15.75q.05.13-.1.22t-.22-.03zm9.25 0h17.5l-8.56 19.88q-.07.12-.2.12t-.18-.13zM22 27.75 29.69 12H36L22.31 27.94q-.06.12-.22.03t-.09-.22z',
                'w' => 1.125,
                'vw' => 36
            ],
            'genderless' => [
                'p' => 'M9 11q-2.06 0-3.53 1.47T4 16t1.47 3.53T9 21t3.53-1.47T14 16t-1.47-3.53T9 11zm0-4q3.75 0 6.38 2.63T18 16t-2.63 6.38T9 25t-6.38-2.63T0 16t2.63-6.38T9 7z',
                'w' => .563,
                'vw' => 18,
            ],
            'ghost' => [
                'p' => 'M11.63 0q5.12-.13 8.74 3.4T24 12v17q0 .69-.63.94t-1.06-.25l-1.56-1.13q-.75-.56-1.38.13l-2.68 3q-.7.75-1.38 0l-2.56-2.88q-.31-.31-.75-.31t-.75.31L8.69 31.7q-.7.75-1.38 0l-2.69-3Q4 28 3.25 28.56L1.68 29.7q-.43.5-1.06.25T0 29V12.5q0-5 3.37-8.69T11.62 0zM8 14q.81 0 1.4-.6T10 12t-.6-1.4T8 10t-1.4.6T6 12t.6 1.4T8 14zm8 0q.81 0 1.4-.6T18 12t-.6-1.4T16 10t-1.4.6T14 12t.6 1.4 1.4.6z',
                'w' => .75,
                'vw' => 24
            ],
            'gift' => 'M2 28v-8h12v10H4q-.81 0-1.4-.6T2 28zm16 2V20h12v8q0 .81-.6 1.4T28 30H18zm12-20q.81 0 1.4.6T32 12v5q0 .44-.28.72T31 18H1q-.44 0-.72-.28T0 17v-5q0-.81.6-1.4T2 10h2.75q-.63-1.19-.63-2.5 0-2.25 1.63-3.88T9.63 2q1.87 0 3.3 1t3.13 3.25Q17.76 4 19.2 3t3.31-1q2.25 0 3.88 1.63T28 7.5q0 1.25-.62 2.5H30zM9.62 10H15q-2.19-3.31-3.22-4.16T9.63 5q-.7 0-1.26.34t-.9.91-.34 1.25q0 1.06.71 1.78t1.79.72zm12.88 0q1.06 0 1.78-.72T25 7.5t-.72-1.78T22.5 5q-.75 0-1.28.22t-1.53 1.34T17.13 10h5.37z',
            'gifts' => [
                'p' => 'M15.06 12.13q-.68.18-1.25.53t-.97.84-.62 1.16T12 16v14q0 1.06.56 2H2q-.81 0-1.4-.6T0 30V8q0-.81.6-1.4T2 6h1.81L1.94 4.62q-.81-.56-.25-1.37l.56-.81q.56-.82 1.44-.25l2 1.44-.75-1.94q-.13-.38.03-.75t.53-.5l1-.38q.38-.12.75.03t.5.53L9 4 10.25.62q.13-.37.5-.53t.75-.03l1 .38q.38.12.53.5t.03.75l-.75 1.94 2-1.44q.38-.25.78-.2t.66.45l.56.81q.25.31.2.72t-.45.65L14.2 6H16q1.25 0 1.81 1.13-2.56 1.8-2.75 5zM14 30v-6h12v8H16q-.81 0-1.4-.6T14 30zm14 2v-8h12v6q0 .81-.6 1.4T38 32H28zm10-18q.81 0 1.4.6T40 16v6H28v-8h-2v8H14v-6q0-.81.6-1.4T16 14h1.25q-.25-.75-.25-1.5 0-1.75 1.25-3.13T21.5 8q3.44 0 5.5 4.69Q29.06 8 32.5 8q2 0 3.25 1.38T37 12.5q0 .69-.25 1.5H38zm-17 0h3.31Q23 11 21.5 11q-.75 0-1.13.47T20 12.5q0 .69.56 1.19.2.19.44.31zm12.44-.31q.56-.5.56-1.19 0-.56-.37-1.03T32.5 11q-1.5 0-2.81 3H33q.25-.13.44-.31z',
                'w' => 1.25,
                'vw' => 40
            ],
            'glass-cheers' => [
                'p' => 'M39.94 27.13q.19.43-.25.62l-10.13 4.19q-.5.25-.68-.25-.38-.94.03-1.9t1.34-1.35l1.38-.57-2.44-6.5-.82.07q-2.5 0-4.46-1.44t-2.66-3.88L20 11.45l-1.25 4.69q-.69 2.43-2.66 3.87t-4.47 1.44l-.8-.07-2.45 6.5 1.38.57q.94.37 1.34 1.34t.03 1.9q-.18.45-.68.26L.3 27.75q-.44-.19-.25-.63.38-1 1.31-1.37t1.94 0l1.38.56 2.43-6.37q-2.3-1.63-2.93-4.35t.75-5.22L10.37 1q.38-.63 1.1-.88t1.4 0l7.13 3 7.12-3q.7-.24 1.41 0t1.1.88l5.43 9.38q1.38 2.5.75 5.21t-2.94 4.35l2.44 6.37 1.38-.56q1-.37 1.93 0t1.32 1.38zm-22.69-17 1.19-4.44-5.94-2.44-2.25 4zm5.5 0 7-2.88-2.25-4-5.94 2.44z',
                'w' => 1.25,
                'vw' => 40
            ],
            'glass-martini' => 'M31.38 3.63 18 17v12h3.5q1.06 0 1.78.72T24 31.5q0 .5-.5.5h-15q-.5 0-.5-.5 0-1.06.72-1.78T10.5 29H14V17L.62 3.62q-1-1-.46-2.3T2.12 0h27.75q1.44 0 1.97 1.31t-.46 2.32z',
            'glass-martini-alt' => 'M31.38 3.63 18 17v12h3.5q1.06 0 1.78.72T24 31.5q0 .5-.5.5h-15q-.5 0-.5-.5 0-1.06.72-1.78T10.5 29H14V17L.62 3.62q-1-1-.46-2.3T2.12 0h27.75q1.44 0 1.97 1.31t-.46 2.32zM27.75 3H4.25l3 3h17.5z',
            'glass-water' => [
                'p' => 'M32 0C23.1 0 14.6 3.7 8.6 10.2S-.6 25.4 .1 34.3L28.9 437.7c3 41.9 37.8 74.3 79.8 74.3H275.3c42 0 76.8-32.4 79.8-74.3L383.9 34.3c.6-8.9-2.4-17.6-8.5-24.1S360.9 0 352 0H32zM73 156.5L66.4 64H317.6L311 156.5l-24.2 12.1c-19.4 9.7-42.2 9.7-61.6 0c-20.9-10.4-45.5-10.4-66.4 0c-19.4 9.7-42.2 9.7-61.6 0L73 156.5z',
                'vw' => 384,
                'vh' => 512
            ],
            'glass-water-droplet' => [
                'p' => 'M32 0C23.1 0 14.6 3.7 8.6 10.2S-.6 25.4 .1 34.3L28.9 437.7c3 41.9 37.8 74.3 79.8 74.3H275.3c42 0 76.8-32.4 79.8-74.3L383.9 34.3c.6-8.9-2.4-17.6-8.5-24.1S360.9 0 352 0H32zM83 297.5L66.4 64H317.6L301 297.5 288 304c-20.1 10.1-43.9 10.1-64 0s-43.9-10.1-64 0s-43.9 10.1-64 0l-13-6.5zM256 196c0-24-33.7-70.1-52.2-93.5c-6.1-7.7-17.5-7.7-23.6 0C161.7 125.9 128 172 128 196c0 33.1 28.7 60 64 60s64-26.9 64-60z',
                'vw' => 384,
                'vh' => 512
            ],
            'glass-whiskey' => 'M30 2q.94 0 1.53.69T32 4.25l-3.5 22.31q-.13 1-.69 1.79t-1.44 1.21-1.8.44H7.5q-1 0-1.88-.44t-1.4-1.21-.72-1.79L0 4.26q-.13-.88.47-1.57T2 2h28zm-2.31 4H4.3l1.94 12H25.8z',
            'glasses' => [
                'p' => 'M35.88 17.5q.12.5.12 1v4.38q0 2.93-2.1 5.03T28.82 30H26.5q-2.88 0-4.94-1.9t-2.25-4.66L19.12 21h-2.25l-.18 2.44q-.2 2.75-2.25 4.65T9.5 30H7.18q-3 0-5.09-2.1t-2.1-5.02V18.5q0-.5.13-1L2.93 6.19q.63-2.38 2.72-3.53T10 2.3l1 .32q.37.12.56.5t.06.75l-.31.93q-.13.38-.5.57t-.75.06l-.88-.25q-1.12-.38-2.06.06-1 .57-1.25 1.63L3.43 16.5q2.57-.75 5.13-.75 3.69 0 7.12 1.63h4.63q3.44-1.63 7.12-1.63 2.57 0 5.13.75l-2.44-9.62q-.25-1.07-1.25-1.63-.94-.44-2.06-.06l-.88.25q-.37.12-.75-.06t-.5-.57l-.31-.93q-.13-.38.06-.75t.57-.5l1-.32q2.25-.81 4.34.35t2.72 3.53zm-23.2 5.63.26-2.57q-2.19-.81-4.38-.81-2.31 0-4.56.81v2.32q0 1.3.94 2.21t2.25.91H9.5q1.25 0 2.19-.84t1-2.04zM32 22.88v-2.32q-2.25-.81-4.56-.81-2.2 0-4.38.81l.25 2.57q.07 1.18 1 2.03t2.19.84h2.31q1.32 0 2.25-.9t.94-2.23z',
                'w' => 1.125,
                'vw' => 36
            ],
            'globe' => 'M21 10H10q.69-4.25 2.19-6.88T15.5.5t3.31 2.63 2.2 6.87zM9.5 16q0-2 .19-4H21.3q.2 2 .2 4t-.2 4H9.7q-.19-2-.19-4zm20.31-6h-6.75q-.93-5.75-3.12-8.88 3.31 1 5.9 3.35T29.81 10zM11.06 1.12Q8.81 4.25 7.94 10H1.19q1.37-3.19 3.97-5.53t5.9-3.34zM30.44 12q.56 2 .56 4t-.56 4H23.3q.2-2.06.2-4t-.2-4h7.13zM7.5 16q0 1.88.19 4H.56Q0 17.94 0 16t.56-4H7.7q-.19 2.06-.19 4zm2.5 6h11q-.69 4.25-2.19 6.88T15.5 31.5t-3.31-2.63T9.99 22zm9.94 8.88q2.25-3.13 3.12-8.88h6.75q-1.37 3.19-3.97 5.53t-5.9 3.34zM1.19 22h6.75q.94 5.75 3.12 8.88-3.31-1-5.9-3.35T1.19 22z',
            'globe-africa' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm10 13.5q0-.44.28-.72t.72-.28h1.13q-.88-3.63-3.66-6.19T17.5 3.7V5q0 .44-.28.72t-.66.28H15q-.5 0-.81.44l-.5.75q-.2.31-.57.37l-.87.25q-.75.2-.75.94V9q0 .44.28.72t.72.28h5.62q.38 0 .7.31l.37.38q.31.31.68.31h.63q.44 0 .72.28t.28.66q0 .75-.69.94l-2.94 1q-.37.12-.75-.07l-.87-.43Q15.5 13 14.69 13h-.07q-1.12 0-2.06.69L10.87 15Q9.5 16 9.5 17.7v.88q0 1.43 1 2.43t2.43 1h1.57q.44 0 .72.28t.28.72v1.82q0 1.18.5 2.25.5.93 1.56.93.94 0 1.44-.75l.8-1.26q.69-1 1.56-1.8.25-.26.32-.57l.25-1.38q.06-.25.19-.43l1.18-1.57q.2-.25.2-.56V19q0-.44-.29-.72T22.5 18H22q-.5 0-.81-.44l-.81-1.25q-.25-.37-.1-.84t.6-.6l.18-.06q.13-.06.32-.06.25 0 .5.19l1.18.75q.2.19.5.19.25 0 .44-.13l.94-.5q.56-.25.56-.87V14z',
            'globe-americas' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm5.13 22.38q.87-.88.87-2.07 0-.81-.56-1.37l-.88-.88Q19.5 18 18.7 18h-4.2q-.24-.13-.93-1.06T12.5 16q-1 0-1.88-.44l-.68-.37q-.44-.2-.44-.63 0-.56.5-.75l1.93-.62q.2-.07.32-.07.37 0 .69.25l.56.5q.12.13.31.13h.38q.31 0 .44-.25t0-.5l-.94-1.94q-.07-.06-.07-.19 0-.25.13-.37l.62-.63q.13-.12.38-.12h.56q.19 0 .32-.13l.5-.5q.37-.37 0-.75l-.25-.25q-.38-.37 0-.75l.93-.94q.7-.68 0-1.37L15 3.5q-.75.06-1.5.18v.7q0 .55-.47.84t-.97.03l-1.5-.75q-4.44 1.93-6.44 6.37 1.57 2.38 2.13 3.19.5.75 1.12 1.31l.07.07q.94.8 2 1.37 1.12.56 3.06 1.63 1 .56 1 1.75v2q0 .8.56 1.37.7.69 1.1 1.66t.34 1.59v1.69q1.31 0 2.69-.31l1.06-2.94q.06-.13.31-1 .07-.44.38-.7zm5.43-5.75 1.82.5Q28 16.68 28 16q0-2.94-1.31-5.56l-.82.37q-.37.19-.62.5l-1.19 1.88q-.25.37-.25.81t.25.81l1.13 1.7q.31.5.87.62z',
            'globe-asia' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm-.69 15 1.07-1.81q.06-.13.25-.13t.25.19q.12.25.43.25h.2q.5 0 .5-.5V8.62q0-.62-.57-.87l-.69-.38q-.25-.12-.25-.4t.19-.47l3.19-2.38q-1.94-.62-3.88-.62-5.19 0-8.84 3.66T3 16q0 .94.19 2h3.88q.43 0 .74-.31l1.2-1.2q.18-.18.43-.15t.38.28l1.43 2.82q.25.56.88.56h.37q.44 0 .72-.28t.28-.72v-.56q0-.44-.3-.75l-.32-.32q-.38-.37 0-.75l.31-.3q.31-.32.75-.32.56 0 .88-.5zm10.69 6.88v-1.5q0-.44-.31-.7l-.75-.74q-.13-.2-.13-.38v-.81q0-.25-.25-.25h-.37q-.2 0-.25.19l-.25.87q-.07.19-.25.19h-.25q-.13 0-.2-.13l-.37-.8Q22 18 21.7 18h-.75q-.2 0-.32.06l-1.5 1.07q-.12.12-.31.18l-2.44 1q-.37.13-.37.5v.63q0 .19.19.31l.75.75q.25.31.68.31h.88l1.37-.37q.2-.06.38-.06.69 0 1.19.5l.8.8q.32.32.7.32h.93q.44 0 .75-.31l.57-.56q.31-.32.31-.75z',
            'globe-europe' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zM28 16q0-5.19-3.66-8.84T15.5 3.5h-.38l-1.8 1.38q-.2.12-.2.37V6.5q0 .5.5.5h1q.5 0 .5-.5V6l1-1h1.32q.25 0 .47.22t.21.5-.18.53l-1.7 1.63q-.12.12-.24.18l-2.5.82q-.38.12-.38.5 0 .62-.43 1.12l-1.25 1.25q-.32.25-.32.69V14q0 .44.28.72t.72.28h1.38q.62 0 .87-.5l.63-1.19q.12-.3.44-.3h.18q.5 0 .5.5t.5.5h1q.5 0 .5-.5v-.13q0-.38.38-.44l1.94-.69q.68-.25.68-.94V11q0-.38.29-.69t.71-.31h2.32q.25 0 .47.22t.22.53v.56q0 .31-.22.5t-.47.19h-2q-.32 0-.5.25l-.63.56q-.18.19-.18.5t.21.5.47.19h1q.32 0 .5.25l.63.56q.19.2.19.5v.57l-.75.75q-.57.56 0 1.06l2 2.06q.25.25.68.25h1.25q.7-1.94.7-4zM8.12 9.31q0 .32.22.5t.47.2h1q.32 0 .5-.2l.63-.56q.19-.25.19-.5v-1q0-.31-.22-.53T10.4 7t-.47.25L8.3 8.81q-.18.2-.18.5zm8 19.19q3.32-.19 6.04-1.94T26.44 22h-.82q-.5 0-.8-.31l-1.13-1.07Q23.12 20 22.3 20h-1.19l-2.69-2.31q-.74-.7-1.8-.7h-2q-.76 0-1.45.45l-2.68 1.62q-1.38.81-1.38 2.44V23q0 1.37 1.13 2.25l1.37 1.06q1 .69 2.25.69h1.25q.44 0 .72.31t.28.69v.5z',
            'golf-ball' => [
                'p' => 'M6 26h14q0 .81-.6 1.4T18 28h-1q-.81 0-1.4.6T15 30v1.25q0 .75-.75.75h-2.5q-.75 0-.75-.75V30q0-.81-.6-1.4T9 28H8q-.81 0-1.4-.6T6 26zm20-13q0 1.69-.44 3.34t-1.25 3.04-1.94 2.56-2.5 2.06H6.12q-2.8-1.75-4.47-4.66T0 13q0-5.37 3.8-9.19T13 0t9.19 3.81 3.81 9.2zm-11.25 2.75q.88 0 1.47-.6t.6-1.46q0-1.44-1.38-1.94.44 1.25-.47 2.16t-2.16.46q.5 1.38 1.94 1.38zm3.06 2.94q0-1.44-1.37-1.94.31.81-.03 1.53t-1.07 1.03-1.53.06q.5 1.38 1.94 1.38.88 0 1.47-.63t.6-1.43zm4-4q0-1.44-1.37-1.94.44 1.25-.47 2.16t-2.16.46q.5 1.38 1.94 1.38.88 0 1.47-.63t.6-1.43z',
                'w' => .812,
                'vw' => 26,
            ],
            'golf-ball-tee' => [
                'p' => 'M384 192c0 66.8-34.1 125.6-85.8 160H85.8C34.1 317.6 0 258.8 0 192C0 86 86 0 192 0S384 86 384 192zM242.1 256.6c0 18.5-15 33.5-33.5 33.5c-4.9 0-9.1 5.1-5.4 8.4c5.9 5.2 13.7 8.4 22.1 8.4c18.5 0 33.5-15 33.5-33.5c0-8.5-3.2-16.2-8.4-22.1c-3.3-3.7-8.4 .5-8.4 5.4zm-52.3-49.3c-4.9 0-9.1 5.1-5.4 8.4c5.9 5.2 13.7 8.4 22.1 8.4c18.5 0 33.5-15 33.5-33.5c0-8.5-3.2-16.2-8.4-22.1c-3.3-3.7-8.4 .5-8.4 5.4c0 18.5-15 33.5-33.5 33.5zm113.5-17.5c0 18.5-15 33.5-33.5 33.5c-4.9 0-9.1 5.1-5.4 8.4c5.9 5.2 13.7 8.4 22.1 8.4c18.5 0 33.5-15 33.5-33.5c0-8.5-3.2-16.2-8.4-22.1c-3.3-3.7-8.4 .5-8.4 5.4zM96 416c0-17.7 14.3-32 32-32h64 64c17.7 0 32 14.3 32 32s-14.3 32-32 32H240c-8.8 0-16 7.2-16 16v16c0 17.7-14.3 32-32 32s-32-14.3-32-32V464c0-8.8-7.2-16-16-16H128c-17.7 0-32-14.3-32-32z',
                'vw' => 384,
                'vh' => 512
            ],
            'gopuram' => 'M31 22q.44 0 .72.28T32 23v8q0 .44-.28.72T31 32h-5V22h-2v-8h-2V8h-2v6h2v8h2v10h-5v-5q0-.44-.28-.72T18 26h-4q-.44 0-.72.28T13 27v5H8V22h2v-8h2V8h-2v6H8v8H6v10H1q-.44 0-.72-.28T0 31v-8q0-.44.28-.72T1 22h1v-7q0-.44.28-.72T3 14h1V9q0-.44.28-.72T5 8h1V1q0-.44.28-.72T7 0t.72.28T8 1v1h4V1q0-.44.28-.72T13 0t.72.28T14 1v1h4V1q0-.44.28-.72T19 0t.72.28T20 1v1h4V1q0-.44.28-.72T25 0t.72.28T26 1v7h1q.44 0 .72.28T28 9v5h1q.44 0 .72.28T30 15v7h1zM14.5 11v3h3v-3q0-.44-.28-.72T16.5 10h-1q-.44 0-.72.28t-.28.72zM18 22v-4q0-.44-.28-.72T17 17h-2q-.44 0-.72.28T14 18v4h4z',
            'graduation-cap' => [
                'p' => 'M38.88 9.56Q40 9.94 40 11t-1.13 1.44l-17.43 5.31q-1.44.5-2.88 0L6.31 14q-1.19.88-1.31 2.31 1 .63 1 1.7 0 1-.88 1.62l1.63 7.19q.06.43-.22.8t-.78.38h-3.5q-.5 0-.78-.37t-.22-.81l1.62-7.2Q2 19 2 18q0-1.12 1-1.69.06-1.62 1.12-2.93l-3-.94Q0 12.06 0 11t1.12-1.44l17.44-5.31q1.44-.5 2.88 0zM22.05 19.7l9.07-2.75L32 24q0 1.69-3.5 2.84T20 28t-8.5-1.16T8 24l.88-7.06 9.06 2.75q2.06.62 4.12 0z',
                'w' => 1.25,
                'vw' => 40
            ],
            'greater-than' => [
                'p' => 'M22.88 13.13q1.12.5 1.12 1.8v2.13q0 .57-.34 1.1t-.79.72L3.7 27.8q-.75.38-1.53.1t-1.1-1.03L.2 25.06q-.38-.75-.1-1.53t1.03-1.15L14.82 16 1.18 9.63Q.37 9.3.09 8.53T.2 7l.87-1.81q.32-.81 1.1-1.1t1.53.1z',
                'w' => .75,
                'vw' => 24
            ],
            'greater-than-equal' => [
                'p' => 'M3.44 6.75q-.44-.19-.75-.44t-.47-.6-.19-.74.1-.78l.75-1.88q.3-.75 1.15-1.12t1.66-.07l18.87 7.5Q26 9.06 26 10.5v1q0 1.44-1.44 1.87l-18.81 7.5q-.88.32-1.72-.06T2.87 19.7l-.75-1.88q-.3-.81.1-1.56t1.22-1l11-4.25zM26.5 25q.63 0 1.06.44T28 26.5v3q0 .63-.44 1.06T26.5 31h-25q-.63 0-1.06-.44T0 29.5v-3q0-.62.44-1.06T1.5 25h25z',
                'w' => .875,
                'vw' => 28
            ],
            'grimace' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zM9 25v-2.5H6.5v.5q0 .81.6 1.4t1.4.6H9zm0-3.5V19h-.5q-.81 0-1.4.6T6.5 21v.5H9zM8.5 13q0 .81.6 1.4t1.4.6 1.4-.6.6-1.4-.6-1.4-1.4-.6-1.4.6-.6 1.4zM13 25v-2.5h-3V25h3zm0-3.5V19h-3v2.5h3zm4 3.5v-2.5h-3V25h3zm0-3.5V19h-3v2.5h3zm4 3.5v-2.5h-3V25h3zm0-3.5V19h-3v2.5h3zm-.5-6.5q.81 0 1.4-.6t.6-1.4-.6-1.4-1.4-.6-1.4.6-.6 1.4.6 1.4 1.4.6zm4 8v-.5H22V25h.5q.81 0 1.4-.6t.6-1.4zm0-1.5V21q0-.81-.6-1.4t-1.4-.6H22v2.5h2.5z',
            'grin' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm5 10.5q-.81 0-1.4.6t-.6 1.4.6 1.4 1.4.6 1.4-.6.6-1.4-.6-1.4-1.4-.6zm-10 0q-.81 0-1.4.6T8.5 13t.6 1.4 1.4.6 1.4-.6.6-1.4-.6-1.4-1.4-.6zm5 16q3 0 5.78-1.69t3.22-4.12q.06-.57-.34-.94t-.97-.19q-2.94.94-7.7.94t-7.68-.94q-.37-.12-.69 0t-.5.44-.12.69q.44 2.44 3.22 4.12T15.5 27z',
            'grin-alt' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm4 8.06q-.94 1.38-1 3.44.06 2.13 1 3.44.38.56 1.03.56t.97-.56q.94-1.38 1-3.44-.06-2.12-1-3.44Q21.12 8 20.47 8t-.97.56zm-10 0q-.94 1.38-1 3.44.06 2.13 1 3.44.38.56 1.03.56t.97-.56q.94-1.38 1-3.44-.06-2.12-1-3.44Q11.12 8 10.47 8t-.97.56zm6 18.44q3 0 5.78-1.69t3.22-4.12q.06-.57-.34-.94t-.97-.19q-2.94.94-7.7.94t-7.68-.94q-.37-.12-.69 0t-.5.44-.12.69q.44 2.44 3.22 4.12T15.5 27z',
            'grin-beam' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm5 9q-1.19 0-2.28 1.28T17 13.94q0 .43.38.53t.56-.22l.62-1.06q.75-1.38 1.94-1.38t2 1.38l.56 1.06q.2.31.6.22t.34-.53q-.12-1.88-1.22-3.16T20.5 9.5zm-10 0q-1.19 0-2.28 1.28T7 13.94q0 .43.38.53t.56-.22l.62-1.06q.75-1.38 1.94-1.38t2 1.38l.56 1.06q.2.31.6.22t.34-.53q-.06-.75-.31-1.44t-.6-1.25-.78-.94-.9-.6-.9-.21zm5 17.5q3 0 5.78-1.69t3.22-4.12q.06-.57-.34-.94t-.97-.19q-2.94.94-7.7.94t-7.68-.94q-.56-.19-.97.19t-.34.94q.44 2.44 3.22 4.12T15.5 27z',
            'grin-beam-sweat' => 'M28.5 8q-1.25 0-2.13-.88t-.87-2.06q0-1.44 2.63-4.87.12-.2.37-.2t.38.2q2.62 3.44 2.62 4.87 0 1.19-.88 2.07T28.5 8zm0 2q.56 0 1.25-.19Q31 12.81 31 16q0 6.44-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5q4.63 0 8.5 2.56-.5 1.13-.5 2 0 2.07 1.47 3.5T28.5 10zm-8-.5q-1.19 0-2.28 1.28T17 13.94q0 .43.38.53t.56-.22l.62-1.06q.75-1.38 1.94-1.38t2 1.38l.56 1.06q.2.37.6.25t.34-.56q-.12-1.88-1.22-3.16T20.5 9.5zm-10 0q-1.19 0-2.28 1.28T7 13.94q0 .43.38.53t.56-.22l.62-1.06q.75-1.38 1.94-1.38t2 1.38l.56 1.06q.2.37.6.25t.34-.56q-.06-.75-.31-1.44t-.6-1.25-.78-.94-.9-.6-.9-.21zm5 17.5q3 0 5.78-1.69t3.22-4.12q.06-.57-.34-.94t-.97-.19q-2.94.94-7.7.94t-7.68-.94q-.37-.12-.69 0t-.5.44-.12.69q.44 2.44 3.22 4.12T15.5 27z',
            'grin-hearts' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm-9.88 11q-.3.94.13 1.81t1.44 1.13l4.37 1.12q.5.13.63-.37l1.25-4.38q.25-.93-.28-1.78t-1.53-.97q-.88-.18-1.63.35T9 9.75l-.12.5-.44-.12q-.88-.25-1.69.15T5.62 11.5zM15.5 27q3 0 5.78-1.69t3.22-4.12q.06-.57-.34-.94t-.97-.19q-2.94.94-7.7.94t-7.68-.94q-.56-.19-.97.19t-.34.94q.44 2.44 3.22 4.12T15.5 27zm8.31-12.56q1-.25 1.44-1.13t.13-1.81q-.25-.56-.7-.94t-1-.47-1.12.03l-.43.13-.13-.5q-.25-.81-1-1.34T19.38 8q-1 .19-1.54 1.03t-.28 1.78l1.25 4.38q.13.5.63.37z',
            'grin-squint' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zm2.13 11.88q-.38.18-.38.62t.38.63l5 3q.5.3.93-.16t0-.97L21.5 13l2.06-2.5q.2-.25.2-.53t-.2-.47-.43-.25-.5.13zM7.43 10.5 9.5 13l-2.06 2.5q-.44.5 0 .97t.94.16l5-3q.37-.2.37-.63t-.37-.63l-5-3q-.5-.3-.94.16t0 .97zM15.5 27q3 0 5.78-1.69t3.22-4.12q.06-.57-.34-.94t-.97-.19q-2.94.94-7.7.94t-7.68-.94q-.56-.19-.97.19t-.34.94q.44 2.44 3.22 4.12T15.5 27z',
            'grin-squint-tears' => 'M25.63 7q-.32 0-.5-.16T25 6.38q.69-4.5 1.69-5.5.87-.88 2.19-.88t2.22.94.9 2.22-.87 2.15q-1.07 1.07-5.5 1.7zM6.38 25q.3 0 .5.16t.12.46q-.69 4.5-1.69 5.5-.87.88-2.15.88t-2.22-.94T0 28.84t.87-2.15q1.07-1.07 5.5-1.7zm19.5-16q2.06-.31 3.56-.75 2.56 4.44 1.94 9.69t-4.41 9.03-9.03 4.4-9.69-1.93q.44-1.57.75-3.57.13-1.06-.66-2.03T6.13 23q-2.07.31-3.57.75Q0 19.3.62 14.06t4.41-9.03 9.03-4.4 9.69 1.93q-.44 1.56-.75 3.56-.13 1.07.66 2.04t2.21.84zM16 6.62l-1.44 5.7q-.06.37.22.68t.72.19l5.62-1.44q.63-.12.57-.75t-.69-.69L17.75 10l-.31-3.25q-.07-.63-.7-.69t-.74.57zM10.31 21q.07.63.7.69t.74-.57l1.44-5.62q.06-.31-.06-.53t-.35-.35-.53-.06L6.63 16q-.63.13-.57.75t.7.69l3.24.31zm13.44 2.75q2.13-2.13 2.9-5.16T26 13.31q-.31-.43-.88-.4t-.8.53q-1.45 2.75-4.79 6.1t-6.1 4.77q-.5.25-.56.82t.44.87q1.32.94 3.19.94t3.88-.88 3.37-2.3z',
            'grin-stars' => 'M15.5.5q6.44 0 10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16 4.53 5.03 15.5.5zM5.94 10.56q-.32.07-.4.35t.09.47l1.62 1.56-.37 2.19q-.07.25.15.43t.53.07l1.94-1.06 1.94 1.06q.25.12.5-.06t.19-.44l-.38-2.2 1.63-1.55q.18-.2.1-.47t-.42-.35l-2.18-.31-.94-2Q9.8 8 9.5 8t-.44.25l-.94 2zM15.5 27q3 0 5.78-1.69t3.22-4.12q.06-.57-.34-.94t-.97-.19q-2.94.94-7.7.94t-7.68-.94q-.56-.19-.97.19t-.34.94q.44 2.44 3.22 4.12T15.5 27zm9.88-15.63q.18-.18.09-.46t-.4-.35l-2.2-.31-.93-2Q21.8 8 21.5 8t-.44.25l-.94 2-2.18.31q-.32.07-.41.35t.1.47l1.62 1.56-.38 2.19q-.06.25.16.43t.53.07l1.94-1.07 1.94 1.07q.3.12.53-.07t.15-.43l-.37-2.2z',
            'grin-tears' => [
                'p' => 'M6.38 16q.3 0 .5.16t.12.46q-.69 4.5-1.69 5.5-.87.88-2.15.88t-2.22-.94T0 19.84t.87-2.15q1.07-1.07 5.5-1.7zm32.74 1.69q.88.87.88 2.15t-.94 2.22-2.22.94-2.15-.87q-1.07-1.07-1.7-5.5-.05-.32.13-.5t.5-.13q4.5.69 5.5 1.69zM31 16.87q.75 5.13 2.25 6.63l.19.19q-2 3.56-5.6 5.69T20 31.5t-7.84-2.12-5.6-5.7l.19-.18Q8.25 22 9 16.87q.13-1.06-.66-2.03T6.13 14q-.7.13-1.5.25.62-5.81 5.03-9.78T20 .5q3.94 0 7.31 1.81t5.53 4.97 2.53 6.97q-.8-.12-1.5-.25-1.24-.12-2.12.75T31 16.88zM25 9.5q-1.19 0-2.28 1.28t-1.22 3.16q0 .43.38.53t.56-.22l.62-1.06q.75-1.38 1.94-1.38t2 1.38l.56 1.06q.2.37.6.25t.34-.56q-.12-1.88-1.22-3.16T25 9.5zm-10 0q-1.19 0-2.28 1.28t-1.22 3.16q0 .43.38.53t.56-.22l.62-1.06q.75-1.38 1.94-1.38t2 1.38l.56 1.06q.2.37.6.25t.34-.56q-.12-1.88-1.22-3.16T15 9.5zM20 27q3 0 5.78-1.69T29 21.2q.06-.57-.34-.94t-.97-.19q-2.94.94-7.7.94t-7.68-.94q-.56-.19-.97.19t-.34.94q.44 2.44 3.22 4.12T20 27z',
                'w' => 1.248,
                'vw' => 40
            ],
            'grin-tongue' => 'M15.5.5q6.44 0 10.97 4.53T31 16q0 4.94-2.78 8.88t-7.28 5.62q.56-1.25.56-2.5v-2.88q2.63-1.68 3-3.93.06-.57-.34-.94t-.97-.19q-2.94.94-7.69.94t-7.69-.94q-.56-.19-.97.19t-.34.94q.37 2.25 3 3.94V28q0 1.25.56 2.5-4.5-1.69-7.28-5.63T0 16Q0 9.56 4.53 5.03T15.5.5zm-5 14.5q.81 0 1.4-.6t.6-1.4-.6-1.4-1.4-.6-1.4.6-.6 1.4.6 1.4 1.4.6zm10 0q.81 0 1.4-.6t.6-1.4-.6-1.4-1.4-.6-1.4.6-.6 1.4.6 1.4 1.4.6zm-2.19 8.44h.1l.1.06.18.1.34.15.47.25v4q0 1.69-1.19 2.85T15.44 32q-.81 0-1.53-.34t-1.25-.88-.85-1.28-.3-1.56V24q1.12-.56 1.18-.56.69-.32 1.34.03t.85 1.03l.06.5q.13.44.56.44t.57-.44l.06-.5q.12-.44.47-.78t.81-.4.9.12z',
            'grin-tongue-squint' => 'M18.31 23.44h.1l.1.06.18.1.34.15.47.25v4q0 1.69-1.19 2.84T15.44 32q-.81 0-1.53-.34t-1.25-.88-.85-1.28-.3-1.56V24q1.12-.56 1.18-.56.69-.32 1.34.03t.85 1.03l.06.5q.13.44.56.44t.57-.44l.06-.5q.12-.44.47-.78t.81-.4.9.12zM15.5.5q6.44 0 10.97 4.53T31 16q0 4.94-2.78 8.88t-7.28 5.62q.56-1.25.56-2.5v-2.88q2.63-1.68 3-3.93.06-.57-.34-.94t-.97-.19q-2.94.94-7.69.94t-7.69-.94q-.56-.19-.97.19t-.34.94q.37 2.25 3 3.94V28q0 1.25.56 2.5-4.5-1.69-7.28-5.63T0 16Q0 9.56 4.53 5.03T15.5.5zm-2.13 13.13q.38-.2.38-.63t-.38-.63l-5-3q-.5-.3-.93.16t0 .97L9.5 13l-2.06 2.5q-.44.5 0 .97t.93.16zm10.2 1.87L21.5 13l2.06-2.5q.44-.5.03-.97t-.97-.15l-5 3q-.37.18-.37.62t.37.63l5 3q.5.3.94-.16t0-.97z',
            'grin-tongue-wink' => 'M21.5 11.5q.63 0 1.06.44T23 13t-.44 1.06-1.06.44-1.06-.44T20 13t.44-1.06 1.06-.44zm-6-11q6.44 0 10.97 4.53T31 16q0 4.94-2.78 8.88t-7.28 5.62q.56-1.25.56-2.5v-2.88q2.63-1.68 3-3.93.06-.57-.34-.94t-.97-.19q-2.94.94-7.69.94t-7.69-.94q-.56-.19-.97.19t-.34.94q.37 2.25 3 3.94V28q0 1.25.56 2.5-4.5-1.69-7.28-5.63T0 16Q0 9.56 4.53 5.03T15.5.5zM12 14.56q.38.38.84.13t.41-.81-.44-1.07-.9-.84-1.16-.53-1.25-.2q-1.25 0-2.4.76t-1.35 1.87q-.06.57.4.82t.85-.13l.63-.56q.68-.56 1.87-.56t1.88.56zM21.5 17q1.63 0 2.81-1.19t1.2-2.81-1.2-2.81-2.81-1.2-2.81 1.2-1.2 2.8 1.2 2.82T21.5 17zm-3.19 6.44h.1l.1.06.18.1.34.15.47.25v4q0 1.69-1.19 2.84T15.44 32q-.81 0-1.53-.34t-1.25-.88-.85-1.28-.3-1.56V24q1.12-.56 1.18-.56.69-.32 1.34.03t.85 1.03l.06.5q.13.44.56.44t.57-.44l.06-.5q.12-.44.47-.78t.81-.4.9.12z',
            'grin-wink' => 'M0 16Q0 9.56 4.53 5.03T15.5.5t10.97 4.53T31 16t-4.53 10.97T15.5 31.5 4.53 26.97 0 16zm12.5-3q0-.81-.6-1.4t-1.4-.6-1.4.6-.6 1.4.6 1.4 1.4.6 1.4-.6.6-1.4zM23 14.56q.38.38.84.1t.41-.78q-.06-.57-.44-1.07t-.9-.84-1.16-.53-1.25-.19q-1.25 0-2.4.75t-1.35 1.87q-.06.57.4.82t.85-.13l.63-.56q.68-.56 1.87-.56t1.88.56zm-15.19 5.5q-.56-.18-.97.2t-.34.93q.44 2.44 3.22 4.12t5.78 1.7 5.78-1.7 3.22-4.12q.06-.38-.13-.69t-.5-.44-.68 0q-2.94.94-7.7.94t-7.68-.94z',
            'grip' => [
                'p' => 'M128 136c0-22.1-17.9-40-40-40L40 96C17.9 96 0 113.9 0 136l0 48c0 22.1 17.9 40 40 40H88c22.1 0 40-17.9 40-40l0-48zm0 192c0-22.1-17.9-40-40-40H40c-22.1 0-40 17.9-40 40l0 48c0 22.1 17.9 40 40 40H88c22.1 0 40-17.9 40-40V328zm32-192v48c0 22.1 17.9 40 40 40h48c22.1 0 40-17.9 40-40V136c0-22.1-17.9-40-40-40l-48 0c-22.1 0-40 17.9-40 40zM288 328c0-22.1-17.9-40-40-40H200c-22.1 0-40 17.9-40 40l0 48c0 22.1 17.9 40 40 40h48c22.1 0 40-17.9 40-40V328zm32-192v48c0 22.1 17.9 40 40 40h48c22.1 0 40-17.9 40-40V136c0-22.1-17.9-40-40-40l-48 0c-22.1 0-40 17.9-40 40zM448 328c0-22.1-17.9-40-40-40H360c-22.1 0-40 17.9-40 40v48c0 22.1 17.9 40 40 40h48c22.1 0 40-17.9 40-40V328z',
                'vw' => 448,
                'vh' => 512
            ],
            'grip-horizontal' => [
                'p' => 'M6 18q.81 0 1.4.6T8 20v4q0 .81-.6 1.4T6 26H2q-.81 0-1.4-.6T0 24v-4q0-.81.6-1.4T2 18h4zm10 0q.81 0 1.4.6T18 20v4q0 .81-.6 1.4T16 26h-4q-.81 0-1.4-.6T10 24v-4q0-.81.6-1.4T12 18h4zm10 0q.81 0 1.4.6T28 20v4q0 .81-.6 1.4T26 26h-4q-.81 0-1.4-.6T20 24v-4q0-.81.6-1.4T22 18h4zM6 6q.81 0 1.4.6T8 8v4q0 .81-.6 1.4T6 14H2q-.81 0-1.4-.6T0 12V8q0-.81.6-1.4T2 6h4zm10 0q.81 0 1.4.6T18 8v4q0 .81-.6 1.4T16 14h-4q-.81 0-1.4-.6T10 12V8q0-.81.6-1.4T12 6h4zm10 0q.81 0 1.4.6T28 8v4q0 .81-.6 1.4T26 14h-4q-.81 0-1.4-.6T20 12V8q0-.81.6-1.4T22 6h4z',
                'w' => .875,
                'vw' => 28
            ],
            'grip-lines' => 'M31 18q.44 0 .72.28T32 19v2q0 .44-.28.72T31 22H1q-.44 0-.72-.28T0 21v-2q0-.44.28-.72T1 18h30zm0-8q.44 0 .72.28T32 11v2q0 .44-.28.72T31 14H1q-.44 0-.72-.28T0 13v-2q0-.44.28-.72T1 10h30z',
            'grip-lines-vertical' => [
                'p' => 'M6 31q0 .44-.28.72T5 32H3q-.44 0-.72-.28T2 31V1q0-.44.28-.72T3 0h2q.44 0 .72.28T6 1v30zm8 0q0 .44-.28.72T13 32h-2q-.44 0-.72-.28T10 31V1q0-.44.28-.72T11 0h2q.44 0 .72.28T14 1v30z',
                'w' => .5,
                'vw' => 16,
            ],
            'grip-vertical' => [
                'p' => 'M6 2q.81 0 1.4.6T8 4v4q0 .81-.6 1.4T6 10H2q-.81 0-1.4-.6T0 8V4q0-.81.6-1.4T2 2h4zm0 10q.81 0 1.4.6T8 14v4q0 .81-.6 1.4T6 20H2q-.81 0-1.4-.6T0 18v-4q0-.81.6-1.4T2 12h4zm0 10q.81 0 1.4.6T8 24v4q0 .81-.6 1.4T6 30H2q-.81 0-1.4-.6T0 28v-4q0-.81.6-1.4T2 22h4zM18 2q.81 0 1.4.6T20 4v4q0 .81-.6 1.4T18 10h-4q-.81 0-1.4-.6T12 8V4q0-.81.6-1.4T14 2h4zm0 10q.81 0 1.4.6T20 14v4q0 .81-.6 1.4T18 20h-4q-.81 0-1.4-.6T12 18v-4q0-.81.6-1.4T14 12h4zm0 10q.81 0 1.4.6T20 24v4q0 .81-.6 1.4T18 30h-4q-.81 0-1.4-.6T12 28v-4q0-.81.6-1.4T14 22h4z',
                'w' => .625,
                'vw' => 20
            ],
            'group-arrows-rotate' => [
                'p' => 'M201.1 71.9c16.9-5 26.6-22.9 21.5-39.8s-22.9-26.6-39.8-21.5c-21.5 6.4-41.8 15.5-60.6 27C114.3 34 105.4 32 96 32C60.7 32 32 60.7 32 96c0 9.4 2 18.3 5.6 26.3c-11.5 18.7-20.6 39-27 60.6c-5 16.9 4.6 34.8 21.5 39.8s34.8-4.6 39.8-21.5c4.3-14.6 10.4-28.5 17.9-41.4c2 .2 4.1 .3 6.1 .3c35.3 0 64-28.7 64-64c0-2.1-.1-4.1-.3-6.1c12.9-7.5 26.8-13.6 41.4-17.9zm128-61.3c-16.9-5-34.8 4.6-39.8 21.5s4.6 34.8 21.5 39.8c14.6 4.3 28.5 10.4 41.4 17.9c-.2 2-.3 4.1-.3 6.1c0 35.3 28.7 64 64 64c2.1 0 4.1-.1 6.2-.3c7.5 12.9 13.6 26.8 17.9 41.4c5 16.9 22.9 26.6 39.8 21.5s26.6-22.9 21.5-39.8c-6.4-21.5-15.5-41.8-27-60.6c3.6-8 5.6-16.9 5.6-26.3c0-35.3-28.7-64-64-64c-9.4 0-18.3 2-26.3 5.6c-18.7-11.5-39-20.6-60.6-27zM71.9 310.9c-5-16.9-22.9-26.6-39.8-21.5s-26.6 22.9-21.5 39.8c6.4 21.5 15.5 41.8 27 60.6C34 397.7 32 406.6 32 416c0 35.3 28.7 64 64 64c9.4 0 18.3-2 26.3-5.6c18.7 11.5 39 20.6 60.6 27c16.9 5 34.8-4.6 39.8-21.5s-4.6-34.8-21.5-39.8c-14.6-4.3-28.5-10.4-41.4-17.9c.2-2 .3-4.1 .3-6.2c0-35.3-28.7-64-64-64c-2.1 0-4.1 .1-6.2 .3c-7.5-12.9-13.6-26.8-17.9-41.4zm429.4 18.3c5-16.9-4.6-34.8-21.5-39.8s-34.8 4.6-39.8 21.5c-4.3 14.6-10.4 28.5-17.9 41.4c-2-.2-4.1-.3-6.2-.3c-35.3 0-64 28.7-64 64c0 2.1 .1 4.1 .3 6.2c-12.9 7.5-26.8 13.6-41.4 17.9c-16.9 5-26.6 22.9-21.5 39.8s22.9 26.6 39.8 21.5c21.5-6.4 41.8-15.5 60.6-27c8 3.6 16.9 5.6 26.3 5.6c35.3 0 64-28.7 64-64c0-9.4-2-18.3-5.6-26.3c11.5-18.7 20.6-39 27-60.6zM192.8 256.8c0-15.6 5.6-29.9 14.9-41.1L223 231c6.6 6.6 17.8 1.9 17.8-7.4V163.2c0-5.7-4.7-10.4-10.4-10.4H169.9c-9.3 0-13.9 11.2-7.4 17.8l11.2 11.2c-17.9 19.8-28.9 46.2-28.9 75.1c0 43.6 24.9 81.3 61.1 99.8c11.8 6 26.3 1.4 32.3-10.4s1.4-26.3-10.4-32.3c-20.8-10.6-34.9-32.2-34.9-57zm93.1-58.6c20.8 10.6 34.9 32.2 34.9 57c0 15.6-5.6 29.9-14.9 41.1L290.6 281c-6.6-6.6-17.8-1.9-17.8 7.4v60.5c0 5.7 4.7 10.4 10.4 10.4h60.5c9.3 0 13.9-11.2 7.4-17.8l-11.2-11.2c17.9-19.8 28.9-46.2 28.9-75.1c0-43.6-24.9-81.3-61.1-99.8c-11.8-6-26.3-1.4-32.3 10.4s-1.4 26.3 10.4 32.3z',
                'vw' => 512,
                'vh' => 512
            ],
            'guarani-sign' => [
                'p' => 'M192 0c-17.7 0-32 14.3-32 32V66.7C69.2 81.9 0 160.9 0 256s69.2 174.1 160 189.3V480c0 17.7 14.3 32 32 32s32-14.3 32-32V445.3c90.8-15.2 160-94.2 160-189.3c0-17.7-14.3-32-32-32H224V132c22.1 5.7 41.8 17.1 57.6 32.6c12.6 12.4 32.9 12.2 45.3-.4s12.2-32.9-.5-45.3C299 92 263.5 73.3 224 66.7V32c0-17.7-14.3-32-32-32zM160 132V380c-55.2-14.2-96-64.3-96-124s40.8-109.8 96-124zM224 380V288h92c-11.6 45-47 80.4-92 92z',
                'vw' => 384,
                'vh' => 512
            ],
            'guitar' => 'M31.44 3.44Q32 4 32 4.84t-.56 1.41l-4.25 4.25q-.63.56-1.44.56t-1.38-.56l-3.43 3.38q1.37 2.12 1.34 4.46t-1.6 3.85q-.87.94-2.12 1.31-.87.25-1.47 1t-.72 1.69q-.18 2.37-1.75 3.94-2.18 2.19-5.59 1.84T3 29 .03 22.97t1.84-5.6q1.57-1.56 3.94-1.74.94-.13 1.66-.72t1.03-1.47q.37-1.25 1.25-2.13 1.56-1.56 3.9-1.59t4.47 1.34L21.5 7.7q-.56-.63-.56-1.44t.56-1.44L25.75.56Q26.3 0 27.15 0t1.41.56zM13 22q1.25 0 2.13-.88T16 19t-.88-2.13T13 16t-2.13.88T10 19t.88 2.13T13 22z',
            'gun' => [
                'p' => 'M528 56c0-13.3-10.7-24-24-24s-24 10.7-24 24v8H32C14.3 64 0 78.3 0 96V208c0 17.7 14.3 32 32 32H42c20.8 0 36.1 19.6 31 39.8L33 440.2c-2.4 9.6-.2 19.7 5.8 27.5S54.1 480 64 480h96c14.7 0 27.5-10 31-24.2L217 352H321.4c23.7 0 44.8-14.9 52.7-37.2L400.9 240H432c8.5 0 16.6-3.4 22.6-9.4L477.3 208H544c17.7 0 32-14.3 32-32V96c0-17.7-14.3-32-32-32H528V56zM321.4 304H229l16-64h105l-21 58.7c-1.1 3.2-4.2 5.3-7.5 5.3zM80 128H464c8.8 0 16 7.2 16 16s-7.2 16-16 16H80c-8.8 0-16-7.2-16-16s7.2-16 16-16z',
                'vw' => 576,
                'vh' => 512
            ],
            'h' => [
                'p' => 'M320 256l0 192c0 17.7 14.3 32 32 32s32-14.3 32-32l0-224V64c0-17.7-14.3-32-32-32s-32 14.3-32 32V192L64 192 64 64c0-17.7-14.3-32-32-32S0 46.3 0 64V448c0 17.7 14.3 32 32 32s32-14.3 32-32l0-192 256 0z',
                'vw' => 384,
                'vh' => 512
            ],
            'h-square' => [
                'p' => 'M28 5v22q0 1.25-.88 2.13T25 30H3q-1.25 0-2.13-.88T0 27V5q0-1.25.88-2.13T3 2h22q1.25 0 2.13.88T28 5zm-7 3h-2q-.44 0-.72.28T18 9v5h-8V9q0-.44-.28-.72T9 8H7q-.44 0-.72.28T6 9v14q0 .44.28.72T7 24h2q.44 0 .72-.28T10 23v-5h8v5q0 .44.28.72T19 24h2q.44 0 .72-.28T22 23V9q0-.44-.28-.72T21 8z',
                'w' => .875,
                'vw' => 28
            ],
            'hamburger' => 'M29 16q1.25 0 2.13.88T32 19t-.88 2.13T29 22H3q-1.25 0-2.13-.88T0 19t.88-2.13T3 16h26zm1 8q.44 0 .72.28T31 25v1q0 1.63-1.19 2.81T27 30.01H5q-1.63 0-2.81-1.2T.99 26v-1q0-.44.29-.72T2 24h28zM3.69 14q-1.63 0-2.38-1.63t.2-3.12q1.93-3.13 5.87-5.19T16 2t8.63 2.06 5.87 5.19q.94 1.5.19 3.13T28.3 14H3.7zM24 7q-.44 0-.72.28T23 8t.28.72T24 9t.72-.28T25 8t-.28-.72T24 7zm-8-2q-.44 0-.72.28T15 6t.28.72T16 7t.72-.28T17 6t-.28-.72T16 5zM8 7q-.44 0-.72.28T7 8t.28.72T8 9t.72-.28T9 8t-.28-.72T8 7z',
            'hammer' => [
                'p' => 'M35.69 12.13q.75.68 0 1.43l-5.63 5.63q-.68.69-1.43 0l-1.38-1.44q-.75-.69 0-1.38l.69-.75-1.82-1.74q-2.25.56-3.87-1.07l-3.06-3.06q-1.2-1.19-1.2-2.81v-1.2l-5.62-2.8Q15.31 0 19.44 0t7.06 2.94l2.81 2.8q1.63 1.63 1.07 3.88l1.8 1.82.7-.75q.68-.7 1.43 0zm-17.88-.94 3.07 3.06q.25.25.68.56L6.7 30.75Q5.56 32 3.94 32t-2.78-1.15T0 28.06t1.25-2.75L17.2 10.44q.31.44.63.75z',
                'w' => 1.125,
                'vw' => 36
            ],
            'hamsa' => 'M31.81 19.19q.2.37.2.81 0 .81-.57 1.38l-6.38 6.87Q21.56 32 16 32t-9.06-3.75L.56 21.38Q0 20.8 0 20q0-.44.19-.81Q.69 18 2 18h4V5q0-1 .75-1.75T8.5 2.5t1.75.75T11 5v8.38q0 .62.63.62h1.24q.63 0 .63-.63V2.5q0-1 .75-1.75T16 0t1.75.75.75 1.75v10.88q0 .62.63.62h1.25q.62 0 .62-.63V5q0-1 .75-1.75t1.75-.75 1.75.75T26 5v13h4q1.31 0 1.81 1.19zM16 26q1.5 0 3-1t2.25-2l.75-1q-.31-.44-.84-1.1t-2.1-1.77T16 18t-3 1-2.25 2L10 22q.31.44.84 1.1t2.1 1.77T16 26zm0-6q.81 0 1.4.6T18 22t-.6 1.4-1.4.6-1.4-.6T14 22t.6-1.4T16 20z',
            'hand' => [
                'p' => 'M288 32c0-17.7-14.3-32-32-32s-32 14.3-32 32V240c0 8.8-7.2 16-16 16s-16-7.2-16-16V64c0-17.7-14.3-32-32-32s-32 14.3-32 32V336c0 1.5 0 3.1 .1 4.6L67.6 283c-16-15.2-41.3-14.6-56.6 1.4s-14.6 41.3 1.4 56.6L124.8 448c43.1 41.1 100.4 64 160 64H304c97.2 0 176-78.8 176-176V128c0-17.7-14.3-32-32-32s-32 14.3-32 32V240c0 8.8-7.2 16-16 16s-16-7.2-16-16V64c0-17.7-14.3-32-32-32s-32 14.3-32 32V240c0 8.8-7.2 16-16 16s-16-7.2-16-16V32z',
                'vw' => 512,
                'vh' => 512
            ],
            'hand-back-fist' => [
                'p' => 'M144 0C117.5 0 96 21.5 96 48V96v28.5V176c0 8.8-7.2 16-16 16s-16-7.2-16-16V149.3l-9 7.5C40.4 169 32 187 32 206V244c0 38 16.9 74 46.1 98.3L128 384v96c0 17.7 14.3 32 32 32H320c17.7 0 32-14.3 32-32V374.7c46.9-19 80-65 80-118.7V176 160 144c0-26.5-21.5-48-48-48c-12.4 0-23.6 4.7-32.1 12.3C350 83.5 329.3 64 304 64c-12.4 0-23.6 4.7-32.1 12.3C270 51.5 249.3 32 224 32c-12.4 0-23.6 4.7-32.1 12.3C190 19.5 169.3 0 144 0z',
                'vw' => 448,
                'vh' => 512
            ],
            'hand-dots' => [
                'p' => 'M288 32c0-17.7-14.3-32-32-32s-32 14.3-32 32V240c0 8.8-7.2 16-16 16s-16-7.2-16-16V64c0-17.7-14.3-32-32-32s-32 14.3-32 32V336c0 1.5 0 3.1 .1 4.6L67.6 283c-16-15.2-41.3-14.6-56.6 1.4s-14.6 41.3 1.4 56.6L124.8 448c43.1 41.1 100.4 64 160 64H304c97.2 0 176-78.8 176-176V128c0-17.7-14.3-32-32-32s-32 14.3-32 32V240c0 8.8-7.2 16-16 16s-16-7.2-16-16V64c0-17.7-14.3-32-32-32s-32 14.3-32 32V240c0 8.8-7.2 16-16 16s-16-7.2-16-16V32zM240 336a16 16 0 1 1 32 0 16 16 0 1 1 -32 0zm80 16a16 16 0 1 1 0 32 16 16 0 1 1 0-32zm48-16a16 16 0 1 1 32 0 16 16 0 1 1 -32 0zm-16 80a16 16 0 1 1 0 32 16 16 0 1 1 0-32zM240 432a16 16 0 1 1 32 0 16 16 0 1 1 -32 0zm-48-48a16 16 0 1 1 0 32 16 16 0 1 1 0-32z',
                'vw' => 512,
                'vh' => 512
            ],
            'hand-fist' => [
                'p' => 'M192 0c17.7 0 32 14.3 32 32V144H160V32c0-17.7 14.3-32 32-32zM64 64c0-17.7 14.3-32 32-32s32 14.3 32 32v80H64V64zm192 0c0-17.7 14.3-32 32-32s32 14.3 32 32v96c0 17.7-14.3 32-32 32s-32-14.3-32-32V64zm96 64c0-17.7 14.3-32 32-32s32 14.3 32 32v64c0 17.7-14.3 32-32 32s-32-14.3-32-32V128zm-96 88l0-.6c9.4 5.4 20.3 8.6 32 8.6c13.2 0 25.4-4 35.6-10.8c8.7 24.9 32.5 42.8 60.4 42.8c11.7 0 22.6-3.1 32-8.6V256c0 52.3-25.1 98.8-64 128v96c0 17.7-14.3 32-32 32H160c-17.7 0-32-14.3-32-32V401.6c-17.3-7.9-33.2-18.8-46.9-32.5L69.5 357.5C45.5 333.5 32 300.9 32 267V240c0-35.3 28.7-64 64-64h88c22.1 0 40 17.9 40 40s-17.9 40-40 40H128c-8.8 0-16 7.2-16 16s7.2 16 16 16h56c39.8 0 72-32.2 72-72z',
                'vw' => 448,
                'vh' => 512
            ],
            'hand-holding' => [
                'p' => 'M35.31 20.5q.75.63.72 1.56t-.78 1.5l-9.44 7.57q-1.12.87-2.5.87H1q-.44 0-.72-.28T0 31v-6q0-.44.28-.72T1 24h3.44l2.94-2.37Q9.38 20 12 20h10q.94 0 1.53.72t.47 1.6q-.12.74-.75 1.21t-1.37.47H17q-.44 0-.72.28T16 25t.28.72.72.28h7.38q1.43 0 2.5-.87l5.8-4.63q.57-.5 1.32-.5t1.32.5z',
                'w' => 1.125,
                'vw' => 36
            ],
            'hand-holding-dollar' => [
                'p' => 'M312 24V34.5c6.4 1.2 12.6 2.7 18.2 4.2c12.8 3.4 20.4 16.6 17 29.4s-16.6 20.4-29.4 17c-10.9-2.9-21.1-4.9-30.2-5c-7.3-.1-14.7 1.7-19.4 4.4c-2.1 1.3-3.1 2.4-3.5 3c-.3 .5-.7 1.2-.7 2.8c0 .3 0 .5 0 .6c.2 .2 .9 1.2 3.3 2.6c5.8 3.5 14.4 6.2 27.4 10.1l.9 .3 0 0c11.1 3.3 25.9 7.8 37.9 15.3c13.7 8.6 26.1 22.9 26.4 44.9c.3 22.5-11.4 38.9-26.7 48.5c-6.7 4.1-13.9 7-21.3 8.8V232c0 13.3-10.7 24-24 24s-24-10.7-24-24V220.6c-9.5-2.3-18.2-5.3-25.6-7.8c-2.1-.7-4.1-1.4-6-2c-12.6-4.2-19.4-17.8-15.2-30.4s17.8-19.4 30.4-15.2c2.6 .9 5 1.7 7.3 2.5c13.6 4.6 23.4 7.9 33.9 8.3c8 .3 15.1-1.6 19.2-4.1c1.9-1.2 2.8-2.2 3.2-2.9c.4-.6 .9-1.8 .8-4.1l0-.2c0-1 0-2.1-4-4.6c-5.7-3.6-14.3-6.4-27.1-10.3l-1.9-.6c-10.8-3.2-25-7.5-36.4-14.4c-13.5-8.1-26.5-22-26.6-44.1c-.1-22.9 12.9-38.6 27.7-47.4c6.4-3.8 13.3-6.4 20.2-8.2V24c0-13.3 10.7-24 24-24s24 10.7 24 24zM568.2 336.3c13.1 17.8 9.3 42.8-8.5 55.9L433.1 485.5c-23.4 17.2-51.6 26.5-80.7 26.5H192 32c-17.7 0-32-14.3-32-32V416c0-17.7 14.3-32 32-32H68.8l44.9-36c22.7-18.2 50.9-28 80-28H272h16 64c17.7 0 32 14.3 32 32s-14.3 32-32 32H288 272c-8.8 0-16 7.2-16 16s7.2 16 16 16H392.6l119.7-88.2c17.8-13.1 42.8-9.3 55.9 8.5zM193.6 384l0 0-.9 0c.3 0 .6 0 .9 0z',
                'vw' => 576,
                'vh' => 512
            ],
            'hand-holding-droplet' => [
                'p' => 'M275.5 6.6C278.3 2.5 283 0 288 0s9.7 2.5 12.5 6.6L366.8 103C378 119.3 384 138.6 384 158.3V160c0 53-43 96-96 96s-96-43-96-96v-1.7c0-19.8 6-39 17.2-55.3L275.5 6.6zM568.2 336.3c13.1 17.8 9.3 42.8-8.5 55.9L433.1 485.5c-23.4 17.2-51.6 26.5-80.7 26.5H192 32c-17.7 0-32-14.3-32-32V416c0-17.7 14.3-32 32-32H68.8l44.9-36c22.7-18.2 50.9-28 80-28H272h16 64c17.7 0 32 14.3 32 32s-14.3 32-32 32H288 272c-8.8 0-16 7.2-16 16s7.2 16 16 16H392.6l119.7-88.2c17.8-13.1 42.8-9.3 55.9 8.5zM193.6 384l0 0-.9 0c.3 0 .6 0 .9 0z',
                'vw' => 576,
                'vh' => 512
            ],
            'hand-holding-hand' => [
                'p' => 'M7.8 207.7c-13.1-17.8-9.3-42.8 8.5-55.9L142.9 58.5C166.2 41.3 194.5 32 223.5 32H384 544c17.7 0 32 14.3 32 32v64c0 17.7-14.3 32-32 32H507.2l-44.9 36c-22.7 18.2-50.9 28-80 28H304 288 224c-17.7 0-32-14.3-32-32s14.3-32 32-32h64 16c8.8 0 16-7.2 16-16s-7.2-16-16-16H183.4L63.7 216.2c-17.8 13.1-42.8 9.3-55.9-8.5zM382.4 160l0 0 .9 0c-.3 0-.6 0-.9 0zM568.2 304.3c13.1 17.8 9.3 42.8-8.5 55.9L433.1 453.5c-23.4 17.2-51.6 26.5-80.7 26.5H192 32c-17.7 0-32-14.3-32-32V384c0-17.7 14.3-32 32-32H68.8l44.9-36c22.7-18.2 50.9-28 80-28H272h16 64c17.7 0 32 14.3 32 32s-14.3 32-32 32H288 272c-8.8 0-16 7.2-16 16s7.2 16 16 16H392.6l119.7-88.2c17.8-13.1 42.8-9.3 55.9 8.5zM193.6 352l0 0-.9 0c.3 0 .6 0 .9 0z',
                'vw' => 576,
                'vh' => 512
            ],
            'hand-holding-heart' => [
                'p' => 'M17.19 15.63 10.37 8.5Q8.95 7 9.04 4.75t1.72-3.63Q12.19-.13 14.09.03t3.22 1.53l.7.75.68-.75Q20.06.2 21.94.03t3.31 1.1q1.69 1.37 1.78 3.62t-1.4 3.75l-6.82 7.13q-.37.37-.8.37t-.82-.38zM35.3 20.5q.75.63.72 1.56t-.78 1.5l-9.44 7.57q-1.12.87-2.5.87H1q-.44 0-.72-.28T0 31v-6q0-.44.28-.72T1 24h3.44l2.94-2.37Q9.38 20 12 20h10q.94 0 1.53.72t.47 1.6q-.12.74-.75 1.21t-1.37.47H17q-.44 0-.72.28T16 25t.28.72.72.28h7.38q1.43 0 2.5-.87l5.8-4.63q.57-.5 1.32-.5t1.32.5z',
                'w' => 1.125,
                'vw' => 36
            ],
            'hand-holding-medical' => [
                'p' => 'M10 10.99h4v4a1 1 0 0 0 1 1h4a1 1 0 0 0 1-1v-4h4a1 1 0 0 0 1-1v-4a1 1 0 0 0-1-1h-4v-4a1 1 0 0 0-1-1h-4a1 1 0 0 0-1 1v4h-4a1 1 0 0 0-1 1v4a1 1 0 0 0 1 1ZM35.5 21a2.5 2.5 0 0 0-3.5-.53l-7.47 5.5H17a1 1 0 1 1 0-2h4.9c1 0 1.91-.67 2.08-1.65A2 2 0 0 0 22 19.98H12c-1.69 0-3.32.58-4.63 1.64L4.46 24H.98a1 1 0 0 0-1 1v6a1 1 0 0 0 1 1h21.69c1.39 0 2.74-.45 3.86-1.27l8.44-6.22a2.5 2.5 0 0 0 .52-3.5Zm0 0',
                'vw' => 36
            ],
            'hand-holding-usd' => [
                'p' => 'M16.13 9q-1.07-.31-1.82-1.1t-.87-1.84q-.2-1.5.72-2.65t2.34-1.29V1q0-.44.28-.72T17.5 0h1q.44 0 .72.28t.28.72v1.13q1.13.12 2 .68.25.2.28.57t-.22.62L20.5 5.06q-.38.38-.88.13-.3-.13-.68-.13h-2q-.57 0-.57.5 0 .44.38.5l3.12.94q.7.19 1.28.63t.97 1.03.44 1.28q.19 1.5-.72 2.66t-2.34 1.28V15q0 .44-.28.72t-.72.28h-1q-.44 0-.72-.28T16.5 15v-1.12q-1.13-.13-2-.7-.32-.18-.32-.55t.25-.63l1.07-1.06q.37-.38.87-.13.32.13.63.13h2.06q.56 0 .56-.5 0-.44-.37-.5zm17.25 11.5q.62.63.59 1.56t-.66 1.5l-8.94 7.57Q23.32 32 22 32H.94q-.38 0-.66-.28T0 31v-6q0-.44.28-.72T.94 24h3.31l2.69-2.37Q8.88 20 11.3 20h9.44q.88 0 1.47.72t.4 1.6q-.06.5-.34.87t-.72.6-.87.21h-4.63q-.37 0-.65.28t-.28.72.28.72.65.28h7q1.32 0 2.32-.87l5.5-4.63q.56-.5 1.25-.5t1.25.5z',
                'vw' => 34,
            ],
            'hand-holding-water' => [
                'p' => 'M18 16c3.31 0 6-2.63 6-5.88 0-2.5-3.57-7.54-5.2-9.72a1 1 0 0 0-1.6 0C15.57 2.58 12 7.63 12 10.12A5.94 5.94 0 0 0 18 16Zm17.33 4.5a2.08 2.08 0 0 0-2.66 0l-5.78 4.63c-.7.57-1.59.87-2.5.87H17a1 1 0 0 1 0-2h4.9c.99 0 1.91-.68 2.08-1.66A2 2 0 0 0 22 20H12c-1.69 0-3.32.58-4.63 1.64L4.47 24H1a1 1 0 0 0-1 1v6a1 1 0 0 0 1 1h22.3a4 4 0 0 0 2.5-.88l9.45-7.56a2 2 0 0 0 .08-3.05Zm0 0',
                'vw' => 36
            ],
            'hand-lizard' => [
                'p' => 'M24 30v-3.81q0-.88-.69-1.32l-7-4.43Q15.56 20 14.7 20H5.5q-.62 0-1.05-.44T4 18.5V18q0-1.7 1.15-2.85T8 14h7.75q.5 0 1.06-.38t.75-.87l1.38-3.2q.18-.55-.13-1.05t-.94-.5H3.5Q2.06 8 1.03 6.97T0 4.5v-1q0-.63.44-1.07T1.5 2h20.87q.63 0 1.38.4t1.12.97l10.2 16.19Q36 21 36 22.75V30H24z',
                'w' => 1.125,
                'vw' => 36
            ],
            'hand-middle-finger' => 'M30 19.81v5.2q0 1.87-.94 3.5t-2.56 2.55T23 32H9.56q-1.19 0-2.66-.6t-2.28-1.46L2.7 28q-.7-.69-.7-1.62v-4.63q0-.56.38-1.19t.94-.87L5 18.75v4.75q0 .5.5.5t.5-.5v-9.19q0-.31.1-.56t.24-.47.4-.4.57-.35.63-.16L9.87 12q.5-.06 1 .12t.82.57.31.81v2q0 .5.5.5t.5-.5V3q0-1.25.88-2.13T16 0h.06q1.25.06 2.1.97T19 3.13V15.5q0 .5.5.5t.5-.5v-2q0-.69.66-1.16t1.47-.34l2.44.44q.3.06.56.19t.47.3.31.45.1.5V17l2.24.57q.7.18 1.22.84t.53 1.4z',
            'hand-paper' => [
                'p' => 'M25.56 8q.63 0 1.22.34t.9.91.32 1.25v9.38q0 1.43-.31 2.75L26 29.68q-.19 1-1.03 1.65t-1.84.66H10.74q-.63 0-1.34-.38t-1.1-.87L.5 20q-.63-.88-.47-1.9t1-1.63 1.87-.44 1.6.97l2 2.75V5q0-.69.31-1.25t.9-.9 1.23-.35q1.06 0 1.81.75t.75 1.81V16h.5V2.5q0-.69.31-1.25t.9-.9T14.45 0q.5 0 1 .19t.84.53.53.84.2 1V16h.5V4.94q0-1 .71-1.75t1.72-.75q1.06 0 1.81.75T22.5 5v11h.5v-5.44q0-1.06.75-1.8T25.56 8z',
                'w' => .873,
                'vw' => 28
            ],
            'hand-peace' => [
                'p' => 'M25.5 13.5q1.06 0 1.78.72T28 16v5q0 .31-.06.56l-2 8.5q-.2.88-.88 1.4T23.5 32h-15q-.38 0-.81-.16t-.82-.46-.56-.63l-4-7q-.44-.81-.28-1.72t.9-1.47L4.88 19 .18 7.12q-.43-1.18.07-2.3t1.65-1.6 2.28.03 1.63 1.62L10.12 16H11V3q0-1.25.87-2.13T14 0t2.12.87T17 3v13h.5v-2q0-1.06.72-1.78T20 11.5t1.78.72.72 1.78v2h.5q0-1.06.72-1.78t1.78-.72z',
                'w' => .875,
                'vw' => 28
            ],
            'hand-point-down' => [
                'p' => 'M5.75 29.19v-9.32q-.81.57-1.56.88-1.5.63-2.85-.25T0 18q0-1.75 1.81-2.5Q3 15 4.41 12.84t2.21-4.09Q7.45 7 9.38 7h10.76q.56 0 .96.38t.47.87q.2 1.13 1.32 3.34T24 16v1q0 2.5-1.12 3.72t-3.32.84q-.5.88-1.65 1.25t-2.28-.25q-.94 1.13-2.28 1.16t-2.16-.78v6.25q0 1.12-.81 1.97T8.5 32q-1.12 0-1.94-.84t-.81-1.97zM7 4.5v-3q0-.63.44-1.06T8.5 0h12q.63 0 1.06.44T22 1.5v3q0 .63-.44 1.06T20.5 6h-12q-.62 0-1.06-.44T7 4.5zM20.25 3q0-.5-.38-.88T19 1.75t-.88.38-.37.87.38.88.87.37.88-.38.37-.87z',
                'w' => .75,
                'vw' => 24
            ],
            'hand-point-left' => 'M2.81 9.75h9.32q-.57-.81-.88-1.56-.62-1.5.25-2.85T14 4q1.75 0 2.5 1.81.5 1.19 2.66 2.6t4.1 2.21q1.74.82 1.74 2.75v10.75q0 .57-.37.97t-.88.47q-1.12.2-3.34 1.32T16 28h-1q-2.5 0-3.73-1.12t-.84-3.32q-.88-.5-1.25-1.65t.25-2.28q-1.13-.94-1.16-2.28t.78-2.16H2.81q-1.12 0-1.97-.81T0 12.5q0-1.12.84-1.94t1.97-.81zM27.5 11h3q.63 0 1.06.44T32 12.5v12q0 .63-.44 1.06T30.5 26h-3q-.63 0-1.06-.44T26 24.5v-12q0-.62.44-1.06T27.5 11zM29 24.25q.5 0 .88-.38t.37-.87-.38-.88-.87-.37-.88.38-.37.87.38.88.87.37z',
            'hand-point-right' => 'M32 12.5q0 1.06-.84 1.88t-1.97.8h-6.25q.81.82.78 2.16t-1.16 2.29q.63 1.12.25 2.28t-1.25 1.65q.25 1.5-.25 2.5t-1.6 1.47T17 28h-1q-2.19 0-4.4-1.13t-3.35-1.3q-.5-.07-.88-.48T7 24.12V13.37q0-1.93 1.75-2.75 1.94-.8 4.1-2.22t2.65-2.59Q16.25 4 18 4q1.62 0 2.5 1.34t.25 2.85q-.31.75-.88 1.56h9.32q1.12 0 1.97.81T32 12.5zm-26 0v12q0 .63-.44 1.06T4.5 26h-3q-.63 0-1.06-.44T0 24.5v-12q0-.63.44-1.06T1.5 11h3q.62 0 1.06.44T6 12.5zM4.25 23q0-.5-.38-.88T3 21.75t-.88.38-.37.87.38.88.87.37.88-.38.37-.87z',
            'hand-point-up' => [
                'p' => 'M8.5 0q1.06 0 1.88.84t.8 1.97v6.25q.82-.81 2.16-.78t2.29 1.16q1.12-.63 2.28-.25t1.65 1.25q2.19-.38 3.32.84T24 15v1q0 2.19-1.13 4.4t-1.3 3.35q0 .31-.23.63t-.53.46-.69.16H9.37q-1.93 0-2.75-1.75-.8-1.94-2.21-4.1T1.8 16.5Q0 15.75 0 14q0-1.62 1.34-2.5t2.85-.25q.75.31 1.56.88V2.8q0-1.12.81-1.97T8.5 0zm0 26h12q.63 0 1.06.44T22 27.5v3q0 .63-.44 1.06T20.5 32h-12q-.63 0-1.06-.44T7 30.5v-3q0-.62.44-1.06T8.5 26zM19 27.75q-.5 0-.88.38t-.37.87.38.88.87.37.88-.38.37-.87-.38-.88-.87-.37z',
                'w' => .75,
                'vw' => 24
            ],
            'hand-pointer' => [
                'p' => 'M28 15v6q0 .31-.06.56l-2 8.5q-.2.88-.88 1.4T23.5 32h-13q-1.25 0-2-1l-8-11q-.63-.88-.47-1.9t1-1.63 1.87-.44 1.6.97l2 2.75V2.5q0-1.06.72-1.78T9 0t1.78.72.72 1.78V15h.5v-2.5q0-1.06.72-1.78T14.5 10t1.78.72.72 1.78V15h.5v-1.5q0-1.06.72-1.78T20 11t1.78.72.72 1.78V15h.5q0-1.06.72-1.78t1.78-.72 1.78.72T28 15zm-16 5h-.5v6h.5v-6zm5.5 0H17v6h.5v-6zm5.5 0h-.5v6h.5v-6z',
                'w' => .875,
                'vw' => 28
            ],
            'hand-rock' => 'M29.06 5q.82 0 1.47.44t1.06 1.12.41 1.5v8.32q0 1.25-.5 2.37l-3 7.25q-.5 1.13-.5 2.31v.2q0 .43-.19.77t-.53.53-.78.2h-15q-.62 0-1.06-.45T10 28.5v-.44q0-1.31-1-2.18l-7-6.25q-2-1.82-2-4.5V11q0-1.25.9-2.13T3.07 8q1.19 0 2.06.9T6 11.07v3l.5.44V6q0-1.25.9-2.13T9.57 3q1.19 0 2.06.9t.88 2.16V8h.5V5q0-1.25.9-2.13T16.07 2q1.19 0 2.07.9T19 5.07V8h.5V6q0-1.25.9-2.13T22.57 3q1.19 0 2.07.9t.87 2.16V8h.5q0-1.25.9-2.13T29.07 5z',
            'hand-scissors' => 'M13.5 27.5q0-1.06.72-1.78T16 25v-.5h-2q-1.06 0-1.78-.72T11.5 22t.72-1.78T14 19.5h2V19H3q-1.25 0-2.13-.88T0 16t.88-2.13T3 13h13v-.88L4.87 7.83q-.75-.32-1.25-.95t-.59-1.4.16-1.6q.5-1.12 1.62-1.62t2.32-.06L19 6.87l1.56-1.93q.57-.75 1.47-.91t1.72.28l7 4q.5.31.88.94T32 10.5v15q0 .87-.53 1.56t-1.4.88l-8.5 2Q21.3 30 21 30h-5q-1.06 0-1.78-.72t-.72-1.78z',
            'hand-sparkles' => [
                'p' => 'm6.67 10.66 3.1-1.29A.46.46 0 0 0 10 9c0-.16-.1-.3-.23-.38l-3.1-1.29-1.3-3.1a.42.42 0 0 0-.74 0l-1.3 3.1-3.1 1.3A.46.46 0 0 0 0 9c0 .16.1.3.23.38l3.1 1.28 1.3 3.1a.42.42 0 0 0 .74 0l1.3-3.1Zm22.8 18.55-.07-.02-.06-.03a2.42 2.42 0 0 1 0-4.32l.06-.04.06-.02 2.34-.98.98-2.34.03-.07.03-.05c.24-.49.66-.86 1.14-1.09L34 20V9a2 2 0 1 0-4 0v4.5a.5.5 0 0 1-.5.5h-1a.5.5 0 0 1-.5-.5V4a2 2 0 1 0-4 0v9.5a.5.5 0 0 1-.5.5h-1a.5.5 0 0 1-.5-.5V2a2 2 0 1 0-4 0v11.5a.5.5 0 0 1-.5.5h-1a.5.5 0 0 1-.5-.5V4a2 2 0 1 0-4 0v15.06l-1.47-2.03a2.5 2.5 0 0 0-4.05 2.95l7.85 10.78A3 3 0 0 0 16.75 32h12.36c1.2 0 2.23-.73 2.7-1.8Zm-7.6-8L20 22l-.78 1.86A.25.25 0 0 1 19 24a.25.25 0 0 1-.22-.14L18 22l-1.86-.78A.25.25 0 0 1 16 21c0-.1.05-.18.14-.23L18 20l.78-1.86A.25.25 0 0 1 19 18c.1 0 .18.05.22.14L20 20l1.86.77a.25.25 0 0 1 0 .45ZM40 27c0-.16-.1-.3-.23-.38l-3.1-1.3-1.3-3.1a.42.42 0 0 0-.74 0l-1.3 3.1-3.1 1.3A.46.46 0 0 0 30 27c0 .16.1.3.23.37l3.1 1.3 1.3 3.1a.41.41 0 0 0 .74 0l1.3-3.11 3.1-1.29A.46.46 0 0 0 40 27Zm0 0',
                'vw' => 40
            ],
            'hand-spock' => 'M30.06 6.06q1 .25 1.57 1.13t.3 1.87l-2.24 9.57q-.19.68-.19 1.37v2.63q0 .87-.25 1.68l-1.62 5.57q-.32.93-1.1 1.53t-1.78.6H11.2q-1.19 0-2.06-.82L.8 23.3q-.75-.68-.78-1.71T.7 19.8t1.72-.78 1.78.66L8 23.25v-5L5.56 7.56q-.12-.5-.03-1t.35-.9.65-.69.9-.4q1-.26 1.88.3t1.13 1.57L12.63 16h.62l-3-12.94q-.25-1 .31-1.87T12.13.06q.68-.12 1.3.07t1.1.65.6 1.16L18.38 16h.93l3-12.12q.25-1 1.16-1.53t1.9-.28 1.54 1.15.28 1.9L24.44 16h.69l1.93-8.06q.25-1 1.13-1.56t1.87-.32z',
            'handcuffs' => [
                'p' => 'M240 32a32 32 0 1 1 64 0 32 32 0 1 1 -64 0zM192 48a32 32 0 1 1 0 64 32 32 0 1 1 0-64zm-32 80c17.7 0 32 14.3 32 32h8c13.3 0 24 10.7 24 24v16c0 1.7-.2 3.4-.5 5.1C280.3 229.6 320 286.2 320 352c0 88.4-71.6 160-160 160S0 440.4 0 352c0-65.8 39.7-122.4 96.5-146.9c-.4-1.6-.5-3.3-.5-5.1V184c0-13.3 10.7-24 24-24h8c0-17.7 14.3-32 32-32zm0 320a96 96 0 1 0 0-192 96 96 0 1 0 0 192zm192-96c0-25.9-5.1-50.5-14.4-73.1c16.9-32.9 44.8-59.1 78.9-73.9c-.4-1.6-.5-3.3-.5-5.1V184c0-13.3 10.7-24 24-24h8c0-17.7 14.3-32 32-32s32 14.3 32 32h8c13.3 0 24 10.7 24 24v16c0 1.7-.2 3.4-.5 5.1C600.3 229.6 640 286.2 640 352c0 88.4-71.6 160-160 160c-62 0-115.8-35.3-142.4-86.9c9.3-22.5 14.4-47.2 14.4-73.1zm224 0a96 96 0 1 0 -192 0 96 96 0 1 0 192 0zM368 0a32 32 0 1 1 0 64 32 32 0 1 1 0-64zm80 48a32 32 0 1 1 0 64 32 32 0 1 1 0-64z',
                'vw' => 640,
                'vh' => 512
            ],
            'hands' => [
                'p' => 'm12.81 14.38 3.57 4.8Q18 21.32 18 24v7q0 .44-.28.72T17 32H8.75q-.75 0-.94-.75t-.62-1.31L.62 21.5Q0 20.69 0 19.62V6q0-.81.6-1.4T2 4t1.4.6T4 6v9.25L9.63 22q.24.38.68.38t.75-.32l.82-.81q.62-.56.12-1.25l-2.37-3.2q-.5-.69-.41-1.5T10 14t1.5-.38 1.31.75zM38 4q.81 0 1.4.6T40 6v13.63q0 1.06-.63 1.87l-6.56 8.44q-.44.56-.62 1.31t-.94.75H23q-.44 0-.72-.28T22 31v-7q0-2.69 1.62-4.81l3.57-4.81q.5-.63 1.31-.75T30 14t.78 1.31-.4 1.5L28 20.01q-.5.68.13 1.24l.8.81q.32.32.76.32t.69-.38L36 15.25V6q0-.81.6-1.4T38 4z',
                'w' => 1.25,
                'vw' => 40
            ],
            'hands-asl-interpreting' => [
                'p' => 'M156.6 46.3c7.9-15.8 1.5-35-14.3-42.9s-35-1.5-42.9 14.3L13.5 189.4C4.6 207.2 0 226.8 0 246.7V256c0 70.7 57.3 128 128 128h72 8v-.3c35.2-2.7 65.4-22.8 82.1-51.7c8.8-15.3 3.6-34.9-11.7-43.7s-34.9-3.6-43.7 11.7c-7 12-19.9 20-34.7 20c-22.1 0-40-17.9-40-40s17.9-40 40-40c14.8 0 27.7 8 34.7 20c8.8 15.3 28.4 20.5 43.7 11.7s20.5-28.4 11.7-43.7c-12.8-22.1-33.6-39.1-58.4-47.1l80.8-22c17-4.6 27.1-22.2 22.5-39.3s-22.2-27.1-39.3-22.5L194.9 124.6l81.6-68c13.6-11.3 15.4-31.5 4.1-45.1S249.1-3.9 235.5 7.4L133.6 92.3l23-46zM483.4 465.7c-7.9 15.8-1.5 35 14.3 42.9s35 1.5 42.9-14.3l85.9-171.7c8.9-17.8 13.5-37.4 13.5-57.2V256c0-70.7-57.3-128-128-128H440h-8v.3c-35.2 2.7-65.4 22.8-82.1 51.7c-8.9 15.3-3.6 34.9 11.7 43.7s34.9 3.6 43.7-11.7c7-12 19.9-20 34.7-20c22.1 0 40 17.9 40 40s-17.9 40-40 40c-14.8 0-27.7-8-34.7-20c-8.9-15.3-28.4-20.5-43.7-11.7s-20.5 28.4-11.7 43.7c12.8 22.1 33.6 39.1 58.4 47.1l-80.8 22c-17.1 4.7-27.1 22.2-22.5 39.3s22.2 27.1 39.3 22.5l100.7-27.5-81.6 68c-13.6 11.3-15.4 31.5-4.1 45.1s31.5 15.4 45.1 4.1l101.9-84.9-23 46z',
                'vw' => 640,
                'vh' => 512
            ],
            'hands-bound' => [
                'p' => 'M96 32C96 14.3 81.7 0 64 0S32 14.3 32 32V96v59.1 .7V192v21.9c0 14.2 5.1 27.9 14.3 38.7L131.6 352H128c-13.3 0-24 10.7-24 24s10.7 24 24 24h32H288h64H480h32c13.3 0 24-10.7 24-24s-10.7-24-24-24h-3.6l85.3-99.5c9.2-10.8 14.3-24.5 14.3-38.7V192 155.8v-.7V96 32c0-17.7-14.3-32-32-32s-32 14.3-32 32V96v48.8l-69.3 92.4c-5.7 7.6-16.1 9.6-24.2 4.8c-9.7-5.7-12.1-18.7-5.1-27.5L473 180c10.8-13.5 8.9-33.3-4.4-44.5s-33-9.8-44.5 3.2l-46.7 52.5C361 209.7 352 233.4 352 258.1V320v32H288V320 258.1c0-24.6-9-48.4-25.4-66.8l-46.7-52.5c-11.5-13-31.3-14.4-44.5-3.2s-15.2 30.9-4.4 44.5l27.6 34.5c7 8.8 4.7 21.8-5.1 27.5c-8.1 4.8-18.6 2.7-24.2-4.8L96 144.8V96 32zm64 448v32H288V480h64v32H480V480h32c13.3 0 24-10.7 24-24s-10.7-24-24-24H480 352 288 160 128c-13.3 0-24 10.7-24 24s10.7 24 24 24h32z',
                'vw' => 640,
                'vh' => 512
            ],
            'hands-bubbles' => [
                'p' => 'M416 64a32 32 0 1 0 0-64 32 32 0 1 0 0 64zm96 128a32 32 0 1 0 0-64 32 32 0 1 0 0 64zM160 464a48 48 0 1 0 -96 0 48 48 0 1 0 96 0zM32 160l.1 72.6c.1 52.2 24 101 64 133.1c-.1-1.9-.1-3.8-.1-5.7v-8c0-71.8 37-138.6 97.9-176.7l60.2-37.6c8.6-5.4 17.9-8.4 27.3-9.4l45.9-79.5c6.6-11.5 2.7-26.2-8.8-32.8s-26.2-2.7-32.8 8.8l-78 135.1c-3.3 5.7-10.7 7.7-16.4 4.4s-7.7-10.7-4.4-16.4l62-107.4c6.6-11.5 2.7-26.2-8.8-32.8S214 5 207.4 16.5l-68 117.8 0 0 0 0-43.3 75L96 160c0-17.7-14.4-32-32-32s-32 14.4-32 32zM332.1 88.5L307.5 131c13.9 4.5 26.4 13.7 34.7 27c.9 1.5 1.7 2.9 2.5 4.4l28.9-50c6.6-11.5 2.7-26.2-8.8-32.8s-26.2-2.7-32.8 8.8zm46.4 63.7l-26.8 46.4c-.6 6-2.1 11.8-4.3 17.4H352h13.3l0 0H397l23-39.8c6.6-11.5 2.7-26.2-8.8-32.8s-26.2-2.7-32.8 8.8zM315.1 175c-9.4-15-29.1-19.5-44.1-10.2l-60.2 37.6C159.3 234.7 128 291.2 128 352v8c0 8.9 .8 17.6 2.2 26.1c35.4 8.2 61.8 40 61.8 77.9c0 6.3-.7 12.5-2.1 18.4C215.1 501 246.3 512 280 512H456c13.3 0 24-10.7 24-24s-10.7-24-24-24H364c-6.6 0-12-5.4-12-12s5.4-12 12-12H488c13.3 0 24-10.7 24-24s-10.7-24-24-24H364c-6.6 0-12-5.4-12-12s5.4-12 12-12H520c13.3 0 24-10.7 24-24s-10.7-24-24-24H364c-6.6 0-12-5.4-12-12s5.4-12 12-12H488c13.3 0 24-10.7 24-24s-10.7-24-24-24H352l0 0 0 0H258.8L305 219.1c15-9.4 19.5-29.1 10.2-44.1z',
                'vw' => 576,
                'vh' => 512
            ],
            'hands-clapping' => [
                'p' => 'M336 16V80c0 8.8-7.2 16-16 16s-16-7.2-16-16V16c0-8.8 7.2-16 16-16s16 7.2 16 16zm-98.7 7.1l32 48c4.9 7.4 2.9 17.3-4.4 22.2s-17.3 2.9-22.2-4.4l-32-48c-4.9-7.4-2.9-17.3 4.4-22.2s17.3-2.9 22.2 4.4zM135 119c9.4-9.4 24.6-9.4 33.9 0L292.7 242.7c10.1 10.1 27.3 2.9 27.3-11.3V192c0-17.7 14.3-32 32-32s32 14.3 32 32V345.6c0 57.1-30 110-78.9 139.4c-64 38.4-145.8 28.3-198.5-24.4L7 361c-9.4-9.4-9.4-24.6 0-33.9s24.6-9.4 33.9 0l53 53c6.1 6.1 16 6.1 22.1 0s6.1-16 0-22.1L23 265c-9.4-9.4-9.4-24.6 0-33.9s24.6-9.4 33.9 0l93 93c6.1 6.1 16 6.1 22.1 0s6.1-16 0-22.1L55 185c-9.4-9.4-9.4-24.6 0-33.9s24.6-9.4 33.9 0l117 117c6.1 6.1 16 6.1 22.1 0s6.1-16 0-22.1l-93-93c-9.4-9.4-9.4-24.6 0-33.9zM433.1 484.9c-24.2 14.5-50.9 22.1-77.7 23.1c48.1-39.6 76.6-99 76.6-162.4l0-98.1c8.2-.1 16-6.4 16-16V192c0-17.7 14.3-32 32-32s32 14.3 32 32V345.6c0 57.1-30 110-78.9 139.4zM424.9 18.7c7.4 4.9 9.3 14.8 4.4 22.2l-32 48c-4.9 7.4-14.8 9.3-22.2 4.4s-9.3-14.8-4.4-22.2l32-48c4.9-7.4 14.8-9.3 22.2-4.4z',
                'vw' => 512,
                'vh' => 512
            ],
            'hands-helping' => [
                'p' => 'M30.5 12q.63 0 1.06.44T32 13.5v3q0 .63-.44 1.06T30.5 18H30v4q0 .81-.6 1.4T28 24h-1q0 1.63-1.19 2.81T23 28.01h-8.56L8 31.74q-.69.38-1.5.16T5.25 31l-5-8.7q-.38-.68-.16-1.5t.9-1.18l5-2.88v-3q0-2.25 1.95-3.44L12 7.88v7.62q0 1.88 1.31 3.19T16.5 20t3.19-1.31T21 15.5V12h9.5zm9.25-2.31q.38.69.16 1.5T39 12.38l-5 2.87V13.5q0-1.44-1.03-2.47T30.5 10H19v5.5q0 1.06-.72 1.78T16.5 18t-1.78-.72T14 15.5V7.63q0-1.13.94-1.7L17 4.64Q18 4 19.13 4h6.43L32 .25q.69-.38 1.5-.16t1.25.91z',
                'w' => 1.25,
                'vw' => 40
            ],
            'hands-holding' => [
                'p' => 'M80 104c0-22.1-17.9-40-40-40S0 81.9 0 104v56 64V325.5c0 25.5 10.1 49.9 28.1 67.9L128 493.3c12 12 28.3 18.7 45.3 18.7H240c26.5 0 48-21.5 48-48V385.1c0-29.7-11.8-58.2-32.8-79.2l-25.3-25.3 0 0-15.2-15.2-32-32c-12.5-12.5-32.8-12.5-45.3 0s-12.5 32.8 0 45.3l32 32 15.2 15.2c11 11 9.2 29.2-3.7 37.8c-9.7 6.5-22.7 5.2-31-3.1L98.7 309.5c-12-12-18.7-28.3-18.7-45.3V224 144 104zm480 0v40 80 40.2c0 17-6.7 33.3-18.7 45.3l-51.1 51.1c-8.3 8.3-21.3 9.6-31 3.1c-12.9-8.6-14.7-26.9-3.7-37.8l15.2-15.2 32-32c12.5-12.5 12.5-32.8 0-45.3s-32.8-12.5-45.3 0l-32 32-15.2 15.2 0 0-25.3 25.3c-21 21-32.8 49.5-32.8 79.2V464c0 26.5 21.5 48 48 48h66.7c17 0 33.3-6.7 45.3-18.7l99.9-99.9c18-18 28.1-42.4 28.1-67.9V224 160 104c0-22.1-17.9-40-40-40s-40 17.9-40 40z',
                'vw' => 640,
                'vh' => 512
            ],
            'hands-holding-child' => [
                'p' => 'M320 0a40 40 0 1 1 0 80 40 40 0 1 1 0-80zm44.7 164.3L375.8 253c1.6 13.2-7.7 25.1-20.8 26.8s-25.1-7.7-26.8-20.8l-4.4-35h-7.6l-4.4 35c-1.6 13.2-13.6 22.5-26.8 20.8s-22.5-13.6-20.8-26.8l11.1-88.8L255.5 181c-10.1 8.6-25.3 7.3-33.8-2.8s-7.3-25.3 2.8-33.8l27.9-23.6C271.3 104.8 295.3 96 320 96s48.7 8.8 67.6 24.7l27.9 23.6c10.1 8.6 11.4 23.7 2.8 33.8s-23.7 11.4-33.8 2.8l-19.8-16.7zM40 64c22.1 0 40 17.9 40 40v40 80 40.2c0 17 6.7 33.3 18.7 45.3l51.1 51.1c8.3 8.3 21.3 9.6 31 3.1c12.9-8.6 14.7-26.9 3.7-37.8l-15.2-15.2-32-32c-12.5-12.5-12.5-32.8 0-45.3s32.8-12.5 45.3 0l32 32 15.2 15.2 0 0 25.3 25.3c21 21 32.8 49.5 32.8 79.2V464c0 26.5-21.5 48-48 48H173.3c-17 0-33.3-6.7-45.3-18.7L28.1 393.4C10.1 375.4 0 351 0 325.5V224 160 104C0 81.9 17.9 64 40 64zm560 0c22.1 0 40 17.9 40 40v56 64V325.5c0 25.5-10.1 49.9-28.1 67.9L512 493.3c-12 12-28.3 18.7-45.3 18.7H400c-26.5 0-48-21.5-48-48V385.1c0-29.7 11.8-58.2 32.8-79.2l25.3-25.3 0 0 15.2-15.2 32-32c12.5-12.5 32.8-12.5 45.3 0s12.5 32.8 0 45.3l-32 32-15.2 15.2c-11 11-9.2 29.2 3.7 37.8c9.7 6.5 22.7 5.2 31-3.1l51.1-51.1c12-12 18.7-28.3 18.7-45.3V224 144 104c0-22.1 17.9-40 40-40z',
                'vw' => 640,
                'vh' => 512
            ],
            'hands-holding-circle' => [
                'p' => 'M320 0a128 128 0 1 1 0 256A128 128 0 1 1 320 0zM40 64c22.1 0 40 17.9 40 40v40 80 40.2c0 17 6.7 33.3 18.7 45.3l51.1 51.1c8.3 8.3 21.3 9.6 31 3.1c12.9-8.6 14.7-26.9 3.7-37.8l-15.2-15.2-32-32c-12.5-12.5-12.5-32.8 0-45.3s32.8-12.5 45.3 0l32 32 15.2 15.2 0 0 25.3 25.3c21 21 32.8 49.5 32.8 79.2V464c0 26.5-21.5 48-48 48H173.3c-17 0-33.3-6.7-45.3-18.7L28.1 393.4C10.1 375.4 0 351 0 325.5V224 160 104C0 81.9 17.9 64 40 64zm560 0c22.1 0 40 17.9 40 40v56 64V325.5c0 25.5-10.1 49.9-28.1 67.9L512 493.3c-12 12-28.3 18.7-45.3 18.7H400c-26.5 0-48-21.5-48-48V385.1c0-29.7 11.8-58.2 32.8-79.2l25.3-25.3 0 0 15.2-15.2 32-32c12.5-12.5 32.8-12.5 45.3 0s12.5 32.8 0 45.3l-32 32-15.2 15.2c-11 11-9.2 29.2 3.7 37.8c9.7 6.5 22.7 5.2 31-3.1l51.1-51.1c12-12 18.7-28.3 18.7-45.3V224 144 104c0-22.1 17.9-40 40-40z',
                'vw' => 640,
                'vh' => 512
            ],
            'hands-praying' => [
                'p' => 'M351.2 4.8c3.2-2 6.6-3.3 10-4.1c4.7-1 9.6-.9 14.1 .1c7.7 1.8 14.8 6.5 19.4 13.6L514.6 194.2c8.8 13.1 13.4 28.6 13.4 44.4v73.5c0 6.9 4.4 13 10.9 15.2l79.2 26.4C631.2 358 640 370.2 640 384v96c0 9.9-4.6 19.3-12.5 25.4s-18.1 8.1-27.7 5.5L431 465.9c-56-14.9-95-65.7-95-123.7V224c0-17.7 14.3-32 32-32s32 14.3 32 32v80c0 8.8 7.2 16 16 16s16-7.2 16-16V219.1c0-7-1.8-13.8-5.3-19.8L340.3 48.1c-1.7-3-2.9-6.1-3.6-9.3c-1-4.7-1-9.6 .1-14.1c1.9-8 6.8-15.2 14.3-19.9zm-62.4 0c7.5 4.6 12.4 11.9 14.3 19.9c1.1 4.6 1.2 9.4 .1 14.1c-.7 3.2-1.9 6.3-3.6 9.3L213.3 199.3c-3.5 6-5.3 12.9-5.3 19.8V304c0 8.8 7.2 16 16 16s16-7.2 16-16V224c0-17.7 14.3-32 32-32s32 14.3 32 32V342.3c0 58-39 108.7-95 123.7l-168.7 45c-9.6 2.6-19.9 .5-27.7-5.5S0 490 0 480V384c0-13.8 8.8-26 21.9-30.4l79.2-26.4c6.5-2.2 10.9-8.3 10.9-15.2V238.5c0-15.8 4.7-31.2 13.4-44.4L245.2 14.5c4.6-7.1 11.7-11.8 19.4-13.6c4.6-1.1 9.4-1.2 14.1-.1c3.5 .8 6.9 2.1 10 4.1z',
                'vw' => 640,
                'vh' => 512
            ],
            'hands-wash' => [
                'p' => 'M31 14a3 3 0 1 0 0-6 3 3 0 0 0 0 6Zm-11.53-2.85a3.52 3.52 0 0 1 2.22.07l.96-3.04a1.5 1.5 0 0 0-1.27-1.95c-.72-.08-1.4.4-1.61 1.1l-1.3 4.13Zm-13.62 13L6 24.1v-1.6c0-4.2 2.7-7.94 6.7-9.25l4.6-1.43 2.13-6.75a1.5 1.5 0 0 0-1.27-1.95c-.72-.08-1.39.4-1.6 1.1l-2.1 6.65a.5.5 0 0 1-.62.33.5.5 0 0 1-.33-.63l2.7-8.62A1.5 1.5 0 0 0 14.96.01c-.72-.08-1.39.4-1.6 1.1l-3 9.48a.5.5 0 0 1-.95-.3l2.4-7.63A1.5 1.5 0 0 0 10.53.71c-.72-.08-1.39.4-1.6 1.1L5.18 13.66l-.13-3.13c0-.83-.66-1.5-1.5-1.5-.82 0-1.54.65-1.55 1.49v7.04a7.73 7.73 0 0 0 3.85 6.57ZM32.45 21H22.5a.5.5 0 0 1-.5-.5c0-.28.22-.5.5-.5h8a1.5 1.5 0 0 0 1.47-1.8c-.13-.71-.8-1.2-1.52-1.2H18l2.94-1.07a1.5 1.5 0 1 0-.88-2.86l-6.74 2.07A7.74 7.74 0 0 0 8 22.5v1.6a5 5 0 0 1 3.63 6.8A7.4 7.4 0 0 0 15.5 32h13a1.5 1.5 0 0 0 1.47-1.8c-.13-.71-.8-1.2-1.52-1.2H22.5a.5.5 0 0 1-.5-.5c0-.28.22-.5.5-.5h8a1.5 1.5 0 0 0 1.47-1.8c-.13-.71-.8-1.2-1.52-1.2H22.5a.5.5 0 0 1-.5-.5c0-.28.22-.5.5-.5h10a1.5 1.5 0 0 0 1.47-1.8c-.13-.71-.8-1.2-1.52-1.2ZM26 4a2 2 0 1 0 0-4 2 2 0 0 0 0 4ZM7 26a3 3 0 1 0 0 6 3 3 0 0 0 0-6Zm0 0',
                'vw' => 36
            ],
            'handshake' => [
                'p' => 'M27.19 4q.81 0 1.37.56L32 8v12.13q-.25-.32-.5-.5l-9.12-7.38 1.62-1.5q.75-.69.13-1.44Q23.8 9 23.4 9t-.72.25l-5 4.56v.07q-.82.62-1.88.56t-1.69-.75q-.62-.75-.62-1.78t.81-1.72v-.06l6.13-5.63Q21 4 21.8 4h5.38zM34 8h6v16h-4q-.81 0-1.4-.6T34 22V8zm3 14q.44 0 .72-.28T38 21t-.28-.72T37 20t-.72.28T36 21t.28.72.72.28zM0 24V8h6v14q0 .81-.6 1.4T4 24H0zm3-4q-.44 0-.72.28T2 21t.28.72T3 22t.72-.28T4 21t-.28-.72T3 20zm27.25 1.19q.63.5.72 1.31T30.5 24l-.56.75q-.5.63-1.35.72T27.12 25l-.3-.25-2 2.38q-.57.75-1.54.87t-1.72-.5l-1.12-1q-1.07 1.31-2.72 1.5t-2.9-.87L9.11 22H8V8l3.44-3.44Q12 4 12.8 4h5.25l-5.12 4.69q-1.38 1.25-1.44 3.12t1.19 3.25q1.25 1.32 3.09 1.4t3.28-1.15l1.88-1.69z',
                'w' => 1.25,
                'vw' => 40
            ],
            'handshake-alt-slash' => [
                'p' => 'm22.41 12.23 9.1 7.38c.73.6 1.22 1.44 1.4 2.37H39a1 1 0 0 0 1-1V9a1 1 0 0 0-1-1h-7l-3.41-3.41A1.99 1.99 0 0 0 27.18 4H21.8a2 2 0 0 0-1.35.53l-5.51 5.04-1.6-1.24L18.06 4h-5.24c-.53 0-1.04.21-1.41.59l-1.27 1.27L2.85.2A1.01 1.01 0 0 0 2.1 0a1 1 0 0 0-.66.38L.2 1.96a1 1 0 0 0 .2 1.41L37.16 31.8a1 1 0 0 0 1.4-.18l1.23-1.57a1 1 0 0 0-.18-1.4L18.96 12.66 21 10.8l1.7-1.54a1 1 0 0 1 1.4.06 1 1 0 0 1-.06 1.41ZM1 8a1 1 0 0 0-1 1v12a1 1 0 0 0 1 1h8.14l5.66 5.11a4 4 0 0 0 5.63-.58v-.01l1.13.97c1 .8 2.46.65 3.26-.34l.55-.68L1.48 8Zm0 0',
                'vw' => 40
            ],
            'handshake-angle' => [
                'p' => 'M544 248v3.3l69.7-69.7c21.9-21.9 21.9-57.3 0-79.2L535.6 24.4c-21.9-21.9-57.3-21.9-79.2 0L416.3 64.5c-2.7-.3-5.5-.5-8.3-.5H296c-37.1 0-67.6 28-71.6 64H224V248c0 22.1 17.9 40 40 40s40-17.9 40-40V176c0 0 0-.1 0-.1V160l16 0 136 0c0 0 0 0 .1 0H464c44.2 0 80 35.8 80 80v8zM336 192v56c0 39.8-32.2 72-72 72s-72-32.2-72-72V129.4c-35.9 6.2-65.8 32.3-76 68.2L99.5 255.2 26.3 328.4c-21.9 21.9-21.9 57.3 0 79.2l78.1 78.1c21.9 21.9 57.3 21.9 79.2 0l37.7-37.7c.9 0 1.8 .1 2.7 .1H384c26.5 0 48-21.5 48-48c0-5.6-1-11-2.7-16H432c26.5 0 48-21.5 48-48c0-12.8-5-24.4-13.2-33c25.7-5 45.1-27.6 45.2-54.8v-.4c-.1-30.8-25.1-55.8-56-55.8c0 0 0 0 0 0l-120 0z',
                'vw' => 640,
                'vh' => 512
            ],
            'handshake-simple' => [
                'p' => 'M323.4 85.2l-96.8 78.4c-16.1 13-19.2 36.4-7 53.1c12.9 17.8 38 21.3 55.3 7.8l99.3-77.2c7-5.4 17-4.2 22.5 2.8s4.2 17-2.8 22.5l-20.9 16.2L550.2 352H592c26.5 0 48-21.5 48-48V176c0-26.5-21.5-48-48-48H516h-4-.7l-3.9-2.5L434.8 79c-15.3-9.8-33.2-15-51.4-15c-21.8 0-43 7.5-60 21.2zm22.8 124.4l-51.7 40.2C263 274.4 217.3 268 193.7 235.6c-22.2-30.5-16.6-73.1 12.7-96.8l83.2-67.3c-11.6-4.9-24.1-7.4-36.8-7.4C234 64 215.7 69.6 200 80l-72 48H48c-26.5 0-48 21.5-48 48V304c0 26.5 21.5 48 48 48H156.2l91.4 83.4c19.6 17.9 49.9 16.5 67.8-3.1c5.5-6.1 9.2-13.2 11.1-20.6l17 15.6c19.5 17.9 49.9 16.6 67.8-2.9c4.5-4.9 7.8-10.6 9.9-16.5c19.4 13 45.8 10.3 62.1-7.5c17.9-19.5 16.6-49.9-2.9-67.8l-134.2-123z',
                'vw' => 640,
                'vh' => 512
            ],
            'handshake-simple-slash' => [
                'p' => 'M38.8 5.1C28.4-3.1 13.3-1.2 5.1 9.2S-1.2 34.7 9.2 42.9l592 464c10.4 8.2 25.5 6.3 33.7-4.1s6.3-25.5-4.1-33.7l-135-105.8c-1.1-11.3-6.3-22.3-15.3-30.7l-134.2-123-23.4 18.2-26-20.3 77.2-60.1c7-5.4 17-4.2 22.5 2.8s4.2 17-2.8 22.5l-20.9 16.2L550.2 352H592c26.5 0 48-21.5 48-48V176c0-26.5-21.5-48-48-48H516h-4-.7l-3.9-2.5L434.8 79c-15.3-9.8-33.2-15-51.4-15c-21.8 0-43 7.5-60 21.2l-89.7 72.6-25.8-20.3 81.8-66.2c-11.6-4.9-24.1-7.4-36.8-7.4C234 64 215.7 69.6 200 80l-35.5 23.7L38.8 5.1zM0 176V304c0 26.5 21.5 48 48 48H156.2l91.4 83.4c19.6 17.9 49.9 16.5 67.8-3.1c5.5-6.1 9.2-13.2 11.1-20.6l17 15.6c19.5 17.9 49.9 16.6 67.8-2.9c.8-.8 1.5-1.7 2.2-2.6L41.2 128.5C17.9 131.8 0 151.8 0 176z',
                'vw' => 640,
                'vh' => 512
            ],
            'handshake-slash' => [
                'p' => 'M0 8.01V24h4a2 2 0 0 0 2-2V11.5L1.49 8.01Zm3 12a1 1 0 1 1 0 2 1 1 0 0 1 0-2ZM8 22h1.14l5.66 5.11a4 4 0 0 0 5.63-.58v-.01l1.13.97c1 .8 2.46.65 3.26-.34l.55-.68L8 13.05ZM34 8v14c0 1.1.9 2 2 2h4V8Zm3 14a1 1 0 1 1 0-2 1 1 0 0 1 0 2Zm-18.04-9.34 3.72-3.41a1 1 0 0 1 1.35 1.47l-1.63 1.5 9.1 7.38c.18.15.34.31.5.48V8l-3.42-3.41A1.99 1.99 0 0 0 27.17 4H21.8a2 2 0 0 0-1.35.53l-5.5 5.04-1.6-1.24L18.06 4h-5.24c-.53 0-1.04.21-1.41.59l-1.27 1.27L2.85.2A1.01 1.01 0 0 0 2.1 0a1 1 0 0 0-.66.38L.2 1.96a1 1 0 0 0 .2 1.41L37.16 31.8a1 1 0 0 0 1.4-.18l1.23-1.57a1 1 0 0 0-.18-1.4Zm0 0',
                'vw' => 40
            ],
            'hanukiah' => [
                'p' => 'M14.5 10h1q.5 0 .5.5V18h-2v-7.5q0-.5.5-.5zm-4 0h1q.5 0 .5.5V18h-2v-7.5q0-.5.5-.5zm14 0h1q.5 0 .5.5V18h-2v-7.5q0-.5.5-.5zm4 0h1q.5 0 .5.5V18h-2v-7.5q0-.5.5-.5zm5.5.5V18h-2v-7.5q0-.5.5-.5h1q.5 0 .5.5zM6.5 10h1q.5 0 .5.5V18H6v-7.5q0-.5.5-.5zM39 10q.44 0 .72.28T40 11v7q0 2.5-1.75 4.25T34 24H22v4h11q.44 0 .72.28T34 29v2q0 .44-.28.72T33 32H7q-.44 0-.72-.28T6 31v-2q0-.44.28-.72T7 28h11v-4H6q-2.5 0-4.25-1.75T0 18v-7q0-.44.28-.72T1 10h2q.44 0 .72.28T4 11v7q0 .81.6 1.4T6 20h12V8q0-.44.28-.72T19 7h2q.44 0 .72.28T22 8v12h12q.81 0 1.4-.6T36 18v-7q0-.44.28-.72T37 10h2zm-1-2q-.63 0-1.06-.5t-.44-1.19q0-.37.37-1.22t.75-1.47L38 3q1.5 2.44 1.5 3.31 0 .69-.44 1.19T38 8zM2 8q-.63 0-1.06-.5T.5 6.31q0-.37.37-1.22t.75-1.47L2 3q1.5 2.44 1.5 3.3 0 .69-.44 1.19T2 8zm18-3q-.44 0-.78-.22t-.53-.6-.19-.87q0-.37.38-1.22t.75-1.47L20 0q1.5 2.44 1.5 3.31 0 .69-.44 1.19T20 5zM7 8q-.63 0-1.06-.5T5.5 6.31q0-.37.37-1.22t.75-1.47L7 3q1.5 2.44 1.5 3.3 0 .69-.44 1.19T7 8zm4 0q-.63 0-1.06-.5T9.5 6.31q0-.37.37-1.22t.75-1.47L11 3q1.5 2.44 1.5 3.31 0 .5-.19.88t-.53.6T11 8zm4 0q-.63 0-1.06-.5t-.44-1.19q0-.37.37-1.22t.75-1.47L15 3q1.5 2.44 1.5 3.31 0 .69-.44 1.19T15 8zm10 0q-.63 0-1.06-.5t-.44-1.19q0-.37.37-1.22t.75-1.47L25 3q1.5 2.44 1.5 3.31 0 .69-.44 1.19T25 8zm4 0q-.63 0-1.06-.5t-.44-1.19q0-.37.37-1.22t.75-1.47L29 3q1.5 2.44 1.5 3.31 0 .69-.44 1.19T29 8zm4 0q-.63 0-1.06-.5t-.44-1.19q0-.37.37-1.22t.75-1.47L33 3q1.5 2.44 1.5 3.31 0 .5-.19.88t-.53.6T33 8z',
                'w' => 1.25,
                'vw' => 40
            ],
            'hard-drive' => [
                'p' => 'M0 96C0 60.7 28.7 32 64 32H448c35.3 0 64 28.7 64 64V280.4c-17-15.2-39.4-24.4-64-24.4H64c-24.6 0-47 9.2-64 24.4V96zM64 288H448c35.3 0 64 28.7 64 64v64c0 35.3-28.7 64-64 64H64c-35.3 0-64-28.7-64-64V352c0-35.3 28.7-64 64-64zM320 416a32 32 0 1 0 0-64 32 32 0 1 0 0 64zm128-32a32 32 0 1 0 -64 0 32 32 0 1 0 64 0z',
                'vw' => 512,
                'vh' => 512
            ],
            'hard-hat' => 'M30 18v4H2v-4q0-3.69 2.06-6.72t5.38-4.4L12 12V5q0-.44.28-.72T13 4h6q.44 0 .72.28T20 5v7l2.56-5.13q3.32 1.38 5.38 4.41T30 18zm1 6q.44 0 .72.28T32 25v2q0 .44-.28.72T31 28H1q-.44 0-.72-.28T0 27v-2q0-.44.28-.72T1 24h30z',
            'hashtag' => [
                'p' => 'M27.56 11.38q-.12.62-.75.62h-4.93l-1.44 8h4.68q.32 0 .57.28t.19.6l-.5 2.5q-.07.62-.7.62h-5l-.93 5.38q-.13.62-.75.62h-2.5q-.38 0-.6-.28t-.15-.6l.88-5.12H9.5l-1 5.38q-.06.62-.69.62H5.25q-.38 0-.6-.28t-.15-.6L5.44 24H.74q-.37 0-.59-.28t-.15-.6l.44-2.5q.12-.62.75-.62h4.93l1.44-8H2.88q-.32 0-.57-.28t-.19-.6l.5-2.5Q2.7 8 3.32 8h5l.93-5.38Q9.38 2 10 2h2.5q.38 0 .6.28t.15.6L12.37 8h6.13l1-5.38q.06-.62.69-.62h2.56q.38 0 .6.28t.15.6L22.56 8h4.7q.37 0 .58.28t.16.6zM16.38 20l1.43-8h-6.18l-1.44 8h6.18z',
                'w' => .875,
                'vw' => 28
            ],
            'hat-cowboy' => [
                'p' => 'M30.63 18.55C30.02 14.97 28.16 4 24.52 4c-.88 0-1.66.37-2.31.88-1.3.99-3.11.99-4.41 0A3.85 3.85 0 0 0 15.48 4c-3.64 0-5.51 10.97-6.1 14.55C11.8 19.35 15.22 20 20 20s8.2-.66 10.63-1.45Zm8.92-2.32a1 1 0 0 0-1.23.1C38.26 16.38 31.99 22 20 22 8.07 22 1.73 16.38 1.67 16.32a1 1 0 0 0-1.23-.09c-.38.26-.53.75-.38 1.18C.11 17.54 4.9 30 20 30c15.1 0 19.9-12.46 19.94-12.6a1 1 0 0 0-.38-1.17Zm0 0',
                'vw' => 40
            ],
            'hat-cowboy-side' => [
                'p' => 'M16.3 18.2a9.6 9.6 0 0 0-6.03-2.2C5.44 16 1.34 19.92.1 25.75a3.88 3.88 0 0 0 .54 2.96A2.7 2.7 0 0 0 2.87 30H37c-6.45 0-9.69-2.32-14.57-6.53Zm14.66-1.15-1.69-9.79a4 4 0 0 0-4.77-3.23L12.55 6.56a4 4 0 0 0-3.14 3.5l-.43 4.06c.43-.05.85-.12 1.3-.12 2.6 0 5.12.91 7.33 2.68l6.12 5.28C28.17 25.79 30.93 28 37 28a3 3 0 0 0 3-3c0-1.59-1.85-7.46-9.04-7.95Zm0 0',
                'vw' => 40
            ],
            'hat-wizard' => 'M31 28q.44 0 .72.28T32 29v2q0 .44-.28.72T31 32H1q-.44 0-.72-.28T0 31v-2q0-.44.28-.72T1 28h30zm-19-4 1 2H4l6.94-15.63q.44-1 1.4-2.06t1.91-1.62L26-.01l-3.5 10.57q-.25.63-.25 1.25 0 .82.38 1.57L28 26H15l1-2 4-2-4-2-2-4-2 4-4 2zm4-14-2 1 2 1 1 2 1-2 2-1-2-1-1-2z',
            'haykal' => 'M31 12.69q.81.06.97.9t-.6 1.22l-6.12 3.32 4.69 5.18q.56.63.12 1.38t-1.25.56l-6.75-1.5.25 7.06q0 .5-.34.85t-.85.34-.87-.43L16 26l-4.25 5.56q-.38.44-.88.44t-.84-.34-.34-.85l.25-7.06-6.75 1.5h-.25q-.75 0-1.07-.65t.2-1.29l4.68-5.18L.63 14.8q-.75-.37-.6-1.21t.97-.91l6.88-1-2.63-6.5q-.31-.81.31-1.35t1.38-.09L12.8 7.5 14.94.81Q15.19 0 16 0t1.06.81l2.13 6.7 5.87-3.76q.75-.44 1.38.1t.31 1.34l-2.62 6.5zm-9.81 6.81L18.94 17l2.94-1.56-3.32-.5 1.25-3.13L17 13.62l-1-3.18-1 3.19-2.81-1.82 1.25 3.13-3.32.5L13.06 17l-2.25 2.5 3.25-.75-.12 3.38L16 19.5l2.06 2.63-.12-3.38z',
            'hdd' => [
                'p' => 'M36 19v6q0 1.25-.88 2.13T33 28H3q-1.25 0-2.13-.88T0 25v-6q0-1.25.88-2.13T3 16h30q1.25 0 2.13.88T36 19zm-3-5H3q-1 0-1.94.38L7.12 5.3q.25-.31.66-.62t.9-.5.94-.2h16.76q.68 0 1.4.41t1.1.91l6.06 9.06Q34 14 33 14zm-3 6q-.81 0-1.4.6T28 22t.6 1.4 1.4.6 1.4-.6.6-1.4-.6-1.4T30 20zm-6 0q-.81 0-1.4.6T22 22t.6 1.4 1.4.6 1.4-.6.6-1.4-.6-1.4T24 20z',
                'w' => 1.125,
                'vw' => 36
            ],
            'head-side-cough' => [
                'p' => 'M38.5 19a1.5 1.5 0 1 0 0-3 1.5 1.5 0 0 0 0 3Zm-4 7a1.5 1.5 0 1 0 0 3 1.5 1.5 0 0 0 0-3Zm-4-3.5a1.5 1.5 0 1 0 0 3 1.5 1.5 0 0 0 0-3Zm8 6.5a1.5 1.5 0 1 0 0 3 1.5 1.5 0 0 0 0-3Zm0-6.5a1.5 1.5 0 1 0 0 3 1.5 1.5 0 0 0 0-3Zm-4-2.5a1.5 1.5 0 1 0 0 3 1.5 1.5 0 0 0 0-3Zm-4.67-2.81c-1.31-2.95-3.03-9.49-4.57-11.67A13.01 13.01 0 0 0 14.63 0H12a12 12 0 0 0-8 20.93V32h14v-2h4a4 4 0 0 0 4-4h-6a2 2 0 1 1 0-4h6v-2h2a2 2 0 0 0 1.83-2.81ZM18 14a2 2 0 1 1 0-4 2 2 0 0 1 0 4Zm0 0',
                'vw' => 40
            ],
            'head-side-cough-slash' => [
                'p' => 'M28.38 19.95a2 2 0 0 0 1.45-2.76c-1.31-2.95-3.03-9.49-4.57-11.67A13.01 13.01 0 0 0 14.63 0H12C9.6 0 7.25.72 5.26 2.08L2.84.21a1 1 0 0 0-1.4.18L.2 1.96a1 1 0 0 0 .2 1.41L37.16 31.8a1 1 0 0 0 1.4-.18l1.23-1.57a1 1 0 0 0-.18-1.4Zm-8.8-6.8-3.1-2.4c.36-.45.9-.75 1.52-.75a2 2 0 0 1 2 2c0 .44-.17.82-.41 1.15ZM38.5 19a1.5 1.5 0 1 0 0-3 1.5 1.5 0 0 0 0 3Zm-4 4a1.5 1.5 0 1 0 0-3 1.5 1.5 0 0 0 0 3ZM18 24c0-1.1.9-2 2-2h1.22L1.3 6.6A11.92 11.92 0 0 0 4 20.93V32h14v-2h4a4 4 0 0 0 4-4h-6a2 2 0 0 1-2-2Zm20.5-1.5a1.5 1.5 0 1 0 0 3 1.5 1.5 0 0 0 0-3Zm0 0',
                'vw' => 40
            ],
            'head-side-mask' => 'M0 11.53a11.96 11.96 0 0 0 4 9.4V32h10V19.78L.23 9.77c-.12.58-.2 1.16-.22 1.76Zm31.83 5.66c-1.31-2.95-3.03-9.49-4.57-11.67A13.01 13.01 0 0 0 16.63 0H12.5A12.6 12.6 0 0 0 .83 7.73L14.95 18h17.03c.01-.28-.04-.55-.15-.81ZM20 14a2 2 0 1 1 0-4 2 2 0 0 1 0 4Zm1 9h10l1-3H16v12h9.12a4 4 0 0 0 3.79-2.73L29 29h-8a1 1 0 0 1 0-2h8.67l.66-2H21a1 1 0 0 1 0-2Zm0 0',
            'head-side-virus' => 'M17 15a1 1 0 0 0 0 2 1 1 0 0 0 0-2Zm-4-4a1 1 0 0 0 0 2 1 1 0 0 0 0-2Zm18.82 6.19c-1.3-2.95-3.02-9.49-4.56-11.67A13 13 0 0 0 16.63 0H12a12 12 0 0 0-8 20.93V32h16v-4h4a4 4 0 0 0 4-4v-4h2a2 2 0 0 0 1.82-2.81ZM23 15h-.76a2 2 0 0 0-1.41 3.41l.53.54a1 1 0 1 1-1.41 1.41l-.54-.53A2 2 0 0 0 16 21.24V22a1 1 0 0 1-2 0v-.76a2 2 0 0 0-3.41-1.41l-.54.53a1 1 0 1 1-1.41-1.41l.53-.54A2 2 0 0 0 7.76 15H7a1 1 0 0 1 0-2h.76a2 2 0 0 0 1.41-3.41l-.53-.54a1 1 0 1 1 1.41-1.41l.54.53A2 2 0 0 0 14 6.76V6a1 1 0 0 1 2 0v.76a2 2 0 0 0 3.41 1.41l.54-.53a1 1 0 1 1 1.41 1.41l-.53.54A2 2 0 0 0 22.24 13H23a1 1 0 0 1 0 2Zm0 0',
            'heading' => 'M31 5q0 .44-.28.72T30 6h-2.31v20H30q.44 0 .72.28T31 27v2q0 .44-.28.72T30 30H20q-.44 0-.72-.28T19 29v-2q0-.44.28-.72T20 26h2.38v-8H9.63v8H12q.44 0 .72.28T13 27v2q0 .44-.28.72T12 30H2q-.44 0-.72-.28T1 29v-2q0-.44.28-.72T2 26h2.31V6H2q-.44 0-.72-.28T1 5V3q0-.44.28-.72T2 2h10q.44 0 .72.28T13 3v2q0 .44-.28.72T12 6H9.62v8h12.76V6H20q-.44 0-.72-.28T19 5V3q0-.44.28-.72T20 2h10q.44 0 .72.28T31 3v2z',
            'headphones' => 'M16 2q6.63 0 11.31 4.69T32.01 18v3q0 .5-.32 1.03t-.81.78l-.88.44q-.12 2.81-2.16 4.78T23 30h-1.5q-.63 0-1.06-.44T20 28.5v-11q0-.62.44-1.06T21.5 16H23q2.94 0 5 2.12V18q0-4.94-3.53-8.47T16 6 7.53 9.53 4 18v.12Q6.06 16 9 16h1.5q.63 0 1.06.44T12 17.5v11q0 .62-.44 1.06T10.5 30H9q-2.81 0-4.84-1.97T2 23.25l-.88-.44q-.5-.25-.8-.78T0 21v-3q0-6.62 4.69-11.3T16 2z',
            'headphones-alt' => 'M10 18q.81 0 1.4.6T12 20v8q0 .81-.6 1.4T10 30H9q-1.06 0-2-.53T5.53 28 5 26v-4q0-1.06.53-2T7 18.53 9 18h1zm13 0q1.69 0 2.84 1.19T27 22v4q0 1.06-.53 2T25 29.47 23 30h-1q-.81 0-1.4-.6T20 28v-8q0-.81.6-1.4T22 18h1zM16 2q6.56 0 11.19 4.72T32 18v7q0 .44-.28.72T31 26h-1q-.44 0-.72-.28T29 25v-7q0-5.38-3.81-9.19T16 5q-3.5 0-6.5 1.75T4.75 11.5 3 18v7q0 .44-.28.72T2 26H1q-.44 0-.72-.28T0 25v-7q.19-6.56 4.81-11.28T16.01 2z',
            'headphones-simple' => [
                'p' => 'M256 80C141.1 80 48 173.1 48 288V392c0 13.3-10.7 24-24 24s-24-10.7-24-24V288C0 146.6 114.6 32 256 32s256 114.6 256 256V392c0 13.3-10.7 24-24 24s-24-10.7-24-24V288c0-114.9-93.1-208-208-208zM80 352c0-35.3 28.7-64 64-64h16c17.7 0 32 14.3 32 32V448c0 17.7-14.3 32-32 32H144c-35.3 0-64-28.7-64-64V352zm288-64c35.3 0 64 28.7 64 64v64c0 35.3-28.7 64-64 64H352c-17.7 0-32-14.3-32-32V320c0-17.7 14.3-32 32-32h16z',
                'vw' => 512,
                'vh' => 512
            ],
            'headset' => 'M12 13v7q0 .81-.6 1.4T10 22H9q-1.69 0-2.84-1.16T5 18v-3q0-1.69 1.16-2.84T9 11h1q.81 0 1.4.6T12 13zm11 9h-1q-.81 0-1.4-.6T20 20v-7q0-.81.6-1.4T22 11h1q1.69 0 2.84 1.16T27 15v3q0 1.69-1.16 2.84T23 22zM16 0q6.56 0 11.19 4.72T32 16v10.38q0 2.3-1.66 3.96T26.38 32H15q-1.25 0-2.12-.88T12 29t.88-2.13T15 26h2q1.25 0 2.13.88T20 29h6.38q1.06 0 1.84-.78t.78-1.84V16q0-5.38-3.81-9.19T16 3 6.81 6.8 3 16v1q0 .44-.28.72T2 18H1q-.44 0-.72-.28T0 17v-1Q.19 9.44 4.8 4.72T16 0z',
            'heart' => 'M28.88 3.94Q31.8 6.44 32 10.3t-2.5 6.57l-12.06 12.5Q16.8 30 16 30t-1.44-.62L2.5 16.88Q-.2 14.18 0 10.3t3.12-6.37q2.57-2.19 5.9-1.9t5.73 2.71L16 6l1.25-1.25q2.37-2.44 5.72-2.72t5.9 1.9z',
            'heart-broken' => 'M29.63 4.63q2.25 2.3 2.37 5.59T30.06 16L16.81 29.69q-.37.31-.84.31t-.78-.31L1.94 16Q-.13 13.5 0 10.22t2.37-5.6l.2-.18q2.12-2.2 5.18-2.4t5.44 1.52L15 9l-6 4 9 9-3-8 6-4-2.12-6.5q2.37-1.69 5.4-1.47t5.16 2.4z',
            'heart-circle-bolt' => [
                'p' => 'M47.6 300.4L228.3 469.1c7.5 7 17.4 10.9 27.7 10.9s20.2-3.9 27.7-10.9l2.6-2.4C267.2 438.6 256 404.6 256 368c0-97.2 78.8-176 176-176c28.3 0 55 6.7 78.7 18.5c.9-6.5 1.3-13 1.3-19.6v-5.8c0-69.9-50.5-129.5-119.4-141C347 36.5 300.6 51.4 268 84L256 96 244 84c-32.6-32.6-79-47.5-124.6-39.9C50.5 55.6 0 115.2 0 185.1v5.8c0 41.5 17.2 81.2 47.6 109.5zM432 512a144 144 0 1 0 0-288 144 144 0 1 0 0 288zm47.9-225c4.3 3.7 5.4 9.9 2.6 14.9L452.4 356H488c5.2 0 9.8 3.3 11.4 8.2s-.1 10.3-4.2 13.4l-96 72c-4.5 3.4-10.8 3.2-15.1-.6s-5.4-9.9-2.6-14.9L411.6 380H376c-5.2 0-9.8-3.3-11.4-8.2s.1-10.3 4.2-13.4l96-72c4.5-3.4 10.8-3.2 15.1 .6z',
                'vw' => 576,
                'vh' => 512
            ],
            'heart-circle-check' => [
                'p' => 'M47.6 300.4L228.3 469.1c7.5 7 17.4 10.9 27.7 10.9s20.2-3.9 27.7-10.9l2.6-2.4C267.2 438.6 256 404.6 256 368c0-97.2 78.8-176 176-176c28.3 0 55 6.7 78.7 18.5c.9-6.5 1.3-13 1.3-19.6v-5.8c0-69.9-50.5-129.5-119.4-141C347 36.5 300.6 51.4 268 84L256 96 244 84c-32.6-32.6-79-47.5-124.6-39.9C50.5 55.6 0 115.2 0 185.1v5.8c0 41.5 17.2 81.2 47.6 109.5zM576 368a144 144 0 1 0 -288 0 144 144 0 1 0 288 0zm-76.7-43.3c6.2 6.2 6.2 16.4 0 22.6l-72 72c-6.2 6.2-16.4 6.2-22.6 0l-40-40c-6.2-6.2-6.2-16.4 0-22.6s16.4-6.2 22.6 0L416 385.4l60.7-60.7c6.2-6.2 16.4-6.2 22.6 0z',
                'vw' => 576,
                'vh' => 512
            ],
            'heart-circle-exclamation' => [
                'p' => 'M47.6 300.4L228.3 469.1c7.5 7 17.4 10.9 27.7 10.9s20.2-3.9 27.7-10.9l2.6-2.4C267.2 438.6 256 404.6 256 368c0-97.2 78.8-176 176-176c28.3 0 55 6.7 78.7 18.5c.9-6.5 1.3-13 1.3-19.6v-5.8c0-69.9-50.5-129.5-119.4-141C347 36.5 300.6 51.4 268 84L256 96 244 84c-32.6-32.6-79-47.5-124.6-39.9C50.5 55.6 0 115.2 0 185.1v5.8c0 41.5 17.2 81.2 47.6 109.5zM432 512a144 144 0 1 0 0-288 144 144 0 1 0 0 288zm0-96a24 24 0 1 1 0 48 24 24 0 1 1 0-48zm0-144c8.8 0 16 7.2 16 16v80c0 8.8-7.2 16-16 16s-16-7.2-16-16V288c0-8.8 7.2-16 16-16z',
                'vw' => 576,
                'vh' => 512
            ],
            'heart-circle-minus' => [
                'p' => 'M47.6 300.4L228.3 469.1c7.5 7 17.4 10.9 27.7 10.9s20.2-3.9 27.7-10.9l2.6-2.4C267.2 438.6 256 404.6 256 368c0-97.2 78.8-176 176-176c28.3 0 55 6.7 78.7 18.5c.9-6.5 1.3-13 1.3-19.6v-5.8c0-69.9-50.5-129.5-119.4-141C347 36.5 300.6 51.4 268 84L256 96 244 84c-32.6-32.6-79-47.5-124.6-39.9C50.5 55.6 0 115.2 0 185.1v5.8c0 41.5 17.2 81.2 47.6 109.5zM576 368a144 144 0 1 0 -288 0 144 144 0 1 0 288 0zm-64 0c0 8.8-7.2 16-16 16H368c-8.8 0-16-7.2-16-16s7.2-16 16-16H496c8.8 0 16 7.2 16 16z',
                'vw' => 576,
                'vh' => 512
            ],
            'heart-circle-plus' => [
                'p' => 'M47.6 300.4L228.3 469.1c7.5 7 17.4 10.9 27.7 10.9s20.2-3.9 27.7-10.9l2.6-2.4C267.2 438.6 256 404.6 256 368c0-97.2 78.8-176 176-176c28.3 0 55 6.7 78.7 18.5c.9-6.5 1.3-13 1.3-19.6v-5.8c0-69.9-50.5-129.5-119.4-141C347 36.5 300.6 51.4 268 84L256 96 244 84c-32.6-32.6-79-47.5-124.6-39.9C50.5 55.6 0 115.2 0 185.1v5.8c0 41.5 17.2 81.2 47.6 109.5zM432 512a144 144 0 1 0 0-288 144 144 0 1 0 0 288zm16-208v48h48c8.8 0 16 7.2 16 16s-7.2 16-16 16H448v48c0 8.8-7.2 16-16 16s-16-7.2-16-16V384H368c-8.8 0-16-7.2-16-16s7.2-16 16-16h48V304c0-8.8 7.2-16 16-16s16 7.2 16 16z',
                'vw' => 576,
                'vh' => 512
            ],
            'heart-circle-xmark' => [
                'p' => 'M47.6 300.4L228.3 469.1c7.5 7 17.4 10.9 27.7 10.9s20.2-3.9 27.7-10.9l2.6-2.4C267.2 438.6 256 404.6 256 368c0-97.2 78.8-176 176-176c28.3 0 55 6.7 78.7 18.5c.9-6.5 1.3-13 1.3-19.6v-5.8c0-69.9-50.5-129.5-119.4-141C347 36.5 300.6 51.4 268 84L256 96 244 84c-32.6-32.6-79-47.5-124.6-39.9C50.5 55.6 0 115.2 0 185.1v5.8c0 41.5 17.2 81.2 47.6 109.5zM432 512a144 144 0 1 0 0-288 144 144 0 1 0 0 288zm59.3-180.7L454.6 368l36.7 36.7c6.2 6.2 6.2 16.4 0 22.6s-16.4 6.2-22.6 0L432 390.6l-36.7 36.7c-6.2 6.2-16.4 6.2-22.6 0s-6.2-16.4 0-22.6L409.4 368l-36.7-36.7c-6.2-6.2-6.2-16.4 0-22.6s16.4-6.2 22.6 0L432 345.4l36.7-36.7c6.2-6.2 16.4-6.2 22.6 0s6.2 16.4 0 22.6z',
                'vw' => 576,
                'vh' => 512
            ],
            'heart-crack' => [
                'p' => 'M119.4 44.1c23.3-3.9 46.8-1.9 68.6 5.3l49.8 77.5-75.4 75.4c-1.5 1.5-2.4 3.6-2.3 5.8s1 4.2 2.6 5.7l112 104c2.9 2.7 7.4 2.9 10.5 .3s3.8-7 1.7-10.4l-60.4-98.1 90.7-75.6c2.6-2.1 3.5-5.7 2.4-8.8L296.8 61.8c28.5-16.7 62.4-23.2 95.7-17.6C461.5 55.6 512 115.2 512 185.1v5.8c0 41.5-17.2 81.2-47.6 109.5L283.7 469.1c-7.5 7-17.4 10.9-27.7 10.9s-20.2-3.9-27.7-10.9L47.6 300.4C17.2 272.1 0 232.4 0 190.9v-5.8c0-69.9 50.5-129.5 119.4-141z',
                'vw' => 512,
                'vh' => 512
            ],
            'heart-pulse' => [
                'p' => 'M228.3 469.1L47.6 300.4c-4.2-3.9-8.2-8.1-11.9-12.4h87c22.6 0 43-13.6 51.7-34.5l10.5-25.2 49.3 109.5c3.8 8.5 12.1 14 21.4 14.1s17.8-5 22-13.3L320 253.7l1.7 3.4c9.5 19 28.9 31 50.1 31H476.3c-3.7 4.3-7.7 8.5-11.9 12.4L283.7 469.1c-7.5 7-17.4 10.9-27.7 10.9s-20.2-3.9-27.7-10.9zM503.7 240h-132c-3 0-5.8-1.7-7.2-4.4l-23.2-46.3c-4.1-8.1-12.4-13.3-21.5-13.3s-17.4 5.1-21.5 13.3l-41.4 82.8L205.9 158.2c-3.9-8.7-12.7-14.3-22.2-14.1s-18.1 5.9-21.8 14.8l-31.8 76.3c-1.2 3-4.2 4.9-7.4 4.9H16c-2.6 0-5 .4-7.3 1.1C3 225.2 0 208.2 0 190.9v-5.8c0-69.9 50.5-129.5 119.4-141C165 36.5 211.4 51.4 244 84l12 12 12-12c32.6-32.6 79-47.5 124.6-39.9C461.5 55.6 512 115.2 512 185.1v5.8c0 16.9-2.8 33.5-8.3 49.1z',
                'vw' => 512,
                'vh' => 512
            ],
            'heartbeat' => 'M20 15.25 21.38 18h6.8L16.82 29.63q-.37.37-.84.37t-.78-.38L3.8 18h5.9l1.87-4.5 3.57 7.94q.25.56.87.56t.88-.56zm9.63-10.63q2.25 2.32 2.37 5.6T30.06 16h-7.44l-1.75-3.44Q20.62 12 20 12t-.88.56l-3.06 6.13-3.62-8.13q-.32-.56-.97-.56t-.9.62L8.3 16H1.94Q-.13 13.5 0 10.22t2.37-5.6l.2-.12Q4.93 2 8.36 2t5.88 2.5L16 6.25l1.75-1.81Q20.19 2 23.62 2t5.82 2.44z',
            'helicopter' => [
                'p' => 'M19 24q-1 0-1.63-.81L12 15.99 2 12 0 5.26q-.06-.5.22-.88T1 4h2.5q.5 0 .81.37L7.01 8h13V4H9q-.45 0-.73-.28T8 3V1q0-.44.28-.72T9 0h26q.44 0 .72.28T36 1v2q0 .44-.28.72T35 4H24v4q5.81 0 9.9 4.1T38 22q0 .81-.6 1.4T36 24H19zm7-11.75V20h7.81q-.43-1.94-1.53-3.56t-2.72-2.72T26 12.25zm13.69 16.44q.31.31.31.75t-.37.69Q37.56 32 35.38 32H15q-.44 0-.72-.28T14 31v-2q0-.44.28-.72T15 28h20.38q.8 0 1.53-.69t1.4 0z',
                'w' => 1.25,
                'vw' => 40
            ],
            'helicopter-symbol' => [
                'p' => 'M445.3 224H510C495.6 108.2 403.8 16.4 288 2V66.7C368.4 80.1 431.9 143.6 445.3 224zM510 288H445.3C431.9 368.4 368.4 431.9 288 445.4V510c115.8-14.4 207.6-106.2 222-222zM2 288C16.4 403.8 108.2 495.6 224 510V445.4C143.6 431.9 80.1 368.4 66.7 288H2zm0-64H66.7C80.1 143.6 143.6 80.1 224 66.7V2C108.2 16.4 16.4 108.2 2 224zm206-64c0-17.7-14.3-32-32-32s-32 14.3-32 32V352c0 17.7 14.3 32 32 32s32-14.3 32-32V288h96v64c0 17.7 14.3 32 32 32s32-14.3 32-32V160c0-17.7-14.3-32-32-32s-32 14.3-32 32v64H208V160z',
                'vw' => 512,
                'vh' => 512
            ],
            'helmet-safety' => [
                'p' => 'M256 32c-17.7 0-32 14.3-32 32v2.3 99.6c0 5.6-4.5 10.1-10.1 10.1c-3.6 0-7-1.9-8.8-5.1L157.1 87C83 123.5 32 199.8 32 288v64H544l0-66.4c-.9-87.2-51.7-162.4-125.1-198.6l-48 83.9c-1.8 3.2-5.2 5.1-8.8 5.1c-5.6 0-10.1-4.5-10.1-10.1V66.3 64c0-17.7-14.3-32-32-32H256zM16.6 384C7.4 384 0 391.4 0 400.6c0 4.7 2 9.2 5.8 11.9C27.5 428.4 111.8 480 288 480s260.5-51.6 282.2-67.5c3.8-2.8 5.8-7.2 5.8-11.9c0-9.2-7.4-16.6-16.6-16.6H16.6z',
                'vw' => 576,
                'vh' => 512
            ],
            'helmet-un' => [
                'p' => 'M479.5 224C471.2 98.9 367.2 0 240 0C107.5 0 0 107.5 0 240v56.3C0 344.8 39.2 384 87.7 384H200h14.9L343.5 505.4c4.5 4.2 10.4 6.6 16.5 6.6h96c13.3 0 24-10.7 24-24s-10.7-24-24-24H369.5l-1.5-1.5V288h80 32c17.7 0 32-14.3 32-32s-14.3-32-32-32h-.5zM320 417.2l-78-73.7L274.4 288H320V417.2zM285.3 103.1l34.7 52V112c0-8.8 7.2-16 16-16s16 7.2 16 16v96c0 7.1-4.6 13.3-11.4 15.3s-14-.6-17.9-6.4l-34.7-52V208c0 8.8-7.2 16-16 16s-16-7.2-16-16V112c0-7.1 4.6-13.3 11.4-15.3s14 .6 17.9 6.4zM160 112v64c0 8.8 7.2 16 16 16s16-7.2 16-16V112c0-8.8 7.2-16 16-16s16 7.2 16 16v64c0 26.5-21.5 48-48 48s-48-21.5-48-48V112c0-8.8 7.2-16 16-16s16 7.2 16 16z',
                'vw' => 512,
                'vh' => 512
            ],
            'highlighter' => [
                'p' => 'm0 30 4.25-4.38 4.19 4.2L6.25 32zm7.81-15 2.57-2.19 10.8 10.81L19 26.25q-1 1.06-2.37.63l-2.7-.82-3.18 3.19-6-6 3.19-3.19-.81-2.69Q7 17.07 7 16.7q0-1 .81-1.7zM33 4.94q.94 1 1 2.34t-.88 2.35L22.5 22.06 11.94 11.5 24.38.88q1-.94 2.34-.88t2.34 1z',
                'vw' => 34
            ],
            'hiking' => [
                'p' => 'm5.06 29.5 2.2-8.63 3.24 3.32-1.56 6.31q-.19.69-.72 1.1T7 32q-.25 0-.5-.06-.5-.13-.88-.5t-.53-.88-.03-1.06zM6 17.25q-.13.38-.47.6t-.78.15l-4-1q-.25-.06-.44-.25t-.28-.44-.03-.5L1.63 9.7q.5-1.94 2.3-3t3.88-.57q.44.13.66.47t.1.72zM23 10q.44 0 .72.28T24 11v20q0 .44-.28.72T23 32h-1q-.44 0-.72-.28T21 31V16h-3q-.81 0-1.44-.56L15.2 14l-1.25 5-.07.06L16.81 22Q18 23.19 18 24.81v5.2q0 .8-.6 1.4T16 32t-1.4-.6T14 30v-5.2l-5.44-5.37Q8 18.8 8 18q0-.25.06-.5l1.69-6.69q.31-1.25 1.31-2.03T13.31 8q1.5 0 2.57 1.06L18.8 12H21v-1q0-.44.27-.72T22 10h1zm-8-4q-1.25 0-2.13-.88T12 3t.88-2.13T15 0t2.13.88T18 3t-.88 2.13T15 6z',
                'w' => .748,
                'vw' => 24
            ],
            'hill-avalanche' => [
                'p' => 'M439.7 401.9c34.2 23.1 81.1 19.5 111.4-10.8c34.4-34.4 34.4-90.1 0-124.4c-27.8-27.8-69.5-33.1-102.6-16c-11.8 6.1-16.4 20.6-10.3 32.3s20.6 16.4 32.3 10.3c15.1-7.8 34-5.3 46.6 7.3c15.6 15.6 15.6 40.9 0 56.6s-40.9 15.6-56.6 0l-81.7-81.7C401.2 261.3 416 236.4 416 208c0-33.9-21.1-62.9-50.9-74.5c1.9-6.8 2.9-14 2.9-21.5c0-44.2-35.8-80-80-80c-27.3 0-51.5 13.7-65.9 34.6C216.3 46.6 197.9 32 176 32c-26.5 0-48 21.5-48 48c0 4 .5 7.9 1.4 11.6L439.7 401.9zM480 64a32 32 0 1 0 -64 0 32 32 0 1 0 64 0zm0 128a32 32 0 1 0 0-64 32 32 0 1 0 0 64zM68.3 87C43.1 61.8 0 79.7 0 115.3V432c0 44.2 35.8 80 80 80H396.7c35.6 0 53.5-43.1 28.3-68.3L68.3 87z',
                'vw' => 576,
                'vh' => 512
            ],
            'hill-rockslide' => [
                'p' => 'M252.4 103.8l27 48c2.8 5 8.2 8.2 13.9 8.2l53.3 0c5.8 0 11.1-3.1 13.9-8.2l27-48c2.7-4.9 2.7-10.8 0-15.7l-27-48c-2.8-5-8.2-8.2-13.9-8.2H293.4c-5.8 0-11.1 3.1-13.9 8.2l-27 48c-2.7 4.9-2.7 10.8 0 15.7zM68.3 87C43.1 61.8 0 79.7 0 115.3V432c0 44.2 35.8 80 80 80H396.7c35.6 0 53.5-43.1 28.3-68.3L68.3 87zM504.2 403.6c4.9 2.7 10.8 2.7 15.7 0l48-27c5-2.8 8.2-8.2 8.2-13.9V309.4c0-5.8-3.1-11.1-8.2-13.9l-48-27c-4.9-2.7-10.8-2.7-15.7 0l-48 27c-5 2.8-8.2 8.2-8.2 13.9v53.3c0 5.8 3.1 11.1 8.2 13.9l48 27zM192 64a32 32 0 1 0 -64 0 32 32 0 1 0 64 0zM384 288a32 32 0 1 0 0-64 32 32 0 1 0 0 64z',
                'vw' => 576,
                'vh' => 512
            ],
            'hippo' => [
                'p' => 'M36.31 6q1.07 0 1.9.63t1.32 1.59.47 2.03V16q0 .81-.6 1.4T38 18v2q0 .44-.28.72T37 21h-2q-.44 0-.72-.28T34 20v-2h-8v11q0 .44-.28.72T25 30h-4q-.44 0-.72-.28T20 29v-4.44Q16.81 26 13 26t-7-1.44V29q0 .44-.28.72T5 30H1q-.44 0-.72-.28T0 29V14q0-4.13 3.5-7.06T12 4q4.56 0 8 2.56V3.5q0-.63.44-1.06T21.5 2h1q.63 0 1.06.44T24 3.5v.87Q25.07 4 26 4q1.25 0 2.35.47t1.9 1.31 1.32 1.9q.43-.18 1.4-.65t1.78-.75T36.32 6zM28 11q.44 0 .72-.28T29 10t-.28-.72T28 9t-.72.28T27 10t.28.72.72.28z',
                'w' => 1.25,
                'vw' => 40
            ],
            'history' => 'M31.5 15.97q0 6.4-4.53 10.97T16 31.5q-5.5 0-9.75-3.44-.5-.43-.53-1.1t.47-1.15l.69-.68q.37-.38.97-.41t1.03.28Q12 27.5 16 27.5q4.75 0 8.13-3.38T27.5 16t-3.37-8.13T16 4.5q-4.56 0-7.87 3.13l3.18 3.18q.44.44.2 1.07t-.95.62H1.5q-.44 0-.72-.28T.5 11.5V2.44q0-.69.63-.94t1.06.19L5.3 4.8Q9.81.5 16.01.5q6.43 0 10.96 4.53t4.53 10.94zm-11.31 4.9q-.38.5-1 .6t-1.13-.28L14 18V9.5q0-.63.44-1.06T15.5 8h1q.63 0 1.06.44T18 9.5V16l2.56 2q.5.38.57 1t-.32 1.13z',
            'hockey-puck' => 'M0 10q0-2.5 4.69-4.25T16 4t11.31 1.75 4.7 4.25-4.7 4.25T16.01 16 4.68 14.25 0 10zm0 5.13q3.44 2.5 9.72 3.43t12.56 0T32 15.12V22q0 2.5-4.69 4.25T16 28 4.69 26.25-.01 22v-6.88z',
            'holly-berry' => [
                'p' => 'M9 12q-1.25 0-2.13-.88T6 9t.88-2.13T9 6t2.13.88T12 9t-.88 2.13T9 12zm7-3q0-1.25.88-2.13T19 6t2.13.88T22 9t-.88 2.13T19 12t-2.13-.88T16 9zm-2-3q-1.25 0-2.13-.88T11 3t.88-2.13T14 0t2.13.88T17 3t-.88 2.13T14 6zm-1 8.69q-.06 3.81 1.44 7 .19.44-.06.84t-.7.47q-1.43.13-2.8.56-.7.2-1.07.82t-.19 1.37q.32 1.44.88 2.75.19.38-.03.78t-.66.47q-3.44.32-6.56 2.13-.57.31-1.16-.03t-.6-.97q0-3.63-1.43-6.75-.13-.32-.06-.57t.28-.46.53-.22q1.37-.13 2.81-.57.69-.25 1.06-.87t.2-1.31q-.32-1.5-.88-2.75-.19-.44.03-.85t.65-.4q3.57-.38 6.82-2.32.5-.25 1 .03t.5.85zm14.19 8.19q.44 0 .69.4t0 .84q-1.38 3.07-1.38 6.7 0 .68-.6 1.02t-1.15-.03q-3.12-1.75-6.56-2.06-.44-.06-.69-.47t0-.84q.56-1.32.88-2.7.18-.74-.2-1.37t-1.06-.87q-.06 0-.93-.19.62-1.25.06-2.44Q15 18.18 15 15v-.31q0-.57.5-.88t1 0q3.25 1.94 6.81 2.25.44.06.66.47t.03.84q-.56 1.32-.88 2.7-.18.74.2 1.37t1.06.8q1.43.45 2.8.63z',
                'w' => .873,
                'vw' => 28
            ],
            'home' => [
                'p' => 'M17.5 9.25q.25-.13.5-.13t.5.13l11.5 9.5V29q0 .44-.28.72T29 30h-7q-.44 0-.72-.31t-.28-.7v-6q0-.24-.13-.5t-.37-.37T20 22h-4q-.44 0-.72.28T15 23v6q0 .37-.28.69T14 30H7q-.44 0-.72-.28T6 29V18.75zm18.25 6.44q.25.25.25.62 0 .25-.19.44l-1.56 1.94q-.25.31-.56.31t-.5-.19L18.49 6.7q-.24-.13-.5-.13t-.5.13L2.82 18.8q-.19.2-.44.2-.37 0-.62-.32L.18 16.75Q0 16.56 0 16.31q0-.37.25-.62l15.8-13Q16.94 2 18 2t1.87.69L25.5 7.3V2.75q0-.31.22-.53t.53-.22h3.5q.31 0 .53.22t.22.53v8.69z',
                'w' => 1.125,
                'vw' => 36
            ],
            'horse' => [
                'p' => 'M36 4.81v4.82q0 .37-.16.75t-.47.68-.62.44l-2.06.82q-.7.3-1.44.03t-1.06-.97L28.99 9l-1-.44V15q0 2.82-2 4.88V31q0 .44-.27.72T25 32h-4q-.44 0-.72-.28T20 31v-9.37l-8.38-1.44-1.5 4 1.63 6.56q.12.5-.16.88t-.78.37H6.7q-.75 0-.94-.75L4.19 25q-.13-.44-.13-.94 0-.75.25-1.43l1.63-4.25Q4 16.56 4 14v-.31l.06-.19Q3 14.31 3 15.5V19q0 .44-.28.72T2 20H1q-.44 0-.72-.28T0 19v-3.5q0-2.25 1.62-3.88T5.5 10v.06Q7.3 8 10 8h10q0-3.31 2.34-5.66T28 0h7.5q.25 0 .4.19t.1.44q-.25 1.18-1.38 1.87l.82.94Q36 4 36 4.8zM32 6q.44 0 .72-.28T33 5t-.28-.72T32 4t-.72.28T31 5t.28.72T32 6z',
                'w' => 1.123,
                'vw' => 36
            ],
            'horse-head' => 'M31.88 20.75q.43 1.25-.44 2.19l-2.88 2.5Q28 26 27.2 26H24q-1 0-1.62-.81l-2.87-4q-1.38.81-2.88.81-1.8 0-3.28-1t-2.22-2.56q-.06-.25-.34-.32t-.47.13l-.75.75q-.25.31-.06.63.94 1.8 2.65 3t3.85 1.3V24l2.56 5.13q.5 1-.1 1.93t-1.71.94H2q-.82 0-1.4-.6T0 30v-5.06q0-3.94.5-7t1.65-5.72 3.22-4.5 5.07-2.97L23.06.06q.63-.25.75.38.88 3.37-2.12 5.12 1.3.25 2.46.94t2.04 1.72 1.3 2.28zM20.5 14q.63 0 1.06-.44T22 12.5t-.44-1.06T20.5 11t-1.06.44T19 12.5t.44 1.06 1.06.44z',
            'hospital' => [
                'p' => 'M28 30.75V32H0v-1.25Q0 30 .75 30H2V7.5q0-.63.44-1.06T3.5 6H9V1.5q0-.63.44-1.06T10.5 0h7q.63 0 1.07.44T19 1.5V6h5.5q.63 0 1.07.44T26 7.5V30h1.25q.75 0 .75.75zM19.25 12h-2.5q-.75 0-.75.75v2.5q0 .75.75.75h2.5q.75 0 .75-.75v-2.5q0-.75-.75-.75zm-10.5 4h2.5q.75 0 .75-.75v-2.5q0-.75-.75-.75h-2.5Q8 12 8 12.75v2.5q0 .75.75.75zm6.5 8h-2.5q-.75 0-.75.75V30h4v-5.25q0-.75-.75-.75zm4-6h-2.5q-.75 0-.75.75v2.5q0 .75.75.75h2.5q.75 0 .75-.75v-2.5q0-.75-.75-.75zm-7.25.75q0-.75-.75-.75h-2.5Q8 18 8 18.75v2.5q0 .75.75.75h2.5q.75 0 .75-.75v-2.5zM11.37 6H13v1.63q0 .37.38.37h1.24Q15 8 15 7.62V6h1.63Q17 6 17 5.62V4.38Q17 4 16.62 4H15V2.37Q15 2 14.62 2h-1.24Q13 2 13 2.38V4h-1.63Q11 4 11 4.38v1.25q0 .37.38.37z',
                'w' => .875,
                'vw' => 28
            ],
            'hospital-alt' => [
                'p' => 'M34 6q.81 0 1.4.6T36 8v23q0 .44-.28.72T35 32H1q-.44 0-.72-.28T0 31V8q0-.81.6-1.4T2 6h8V2q0-.81.6-1.4T12 0h12q.81 0 1.4.6T26 2v4h8zM10 27.25v-2.5q0-.75-.75-.75h-2.5Q6 24 6 24.75v2.5q0 .75.75.75h2.5q.75 0 .75-.75zm0-8v-2.5q0-.75-.75-.75h-2.5Q6 16 6 16.75v2.5q0 .75.75.75h2.5q.75 0 .75-.75zm10 8v-2.5q0-.75-.75-.75h-2.5q-.75 0-.75.75v2.5q0 .75.75.75h2.5q.75 0 .75-.75zm0-8v-2.5q0-.75-.75-.75h-2.5q-.75 0-.75.75v2.5q0 .75.75.75h2.5q.75 0 .75-.75zm1-10.63V7.38Q21 7 20.62 7H19V5.37Q19 5 18.62 5h-1.25Q17 5 17 5.38V7h-1.63Q15 7 15 7.38v1.25q0 .37.38.37H17v1.63q0 .37.38.37h1.25q.37 0 .37-.38V9h1.63Q21 9 21 8.62zm9 18.63v-2.5q0-.75-.75-.75h-2.5q-.75 0-.75.75v2.5q0 .75.75.75h2.5q.75 0 .75-.75zm0-8v-2.5q0-.75-.75-.75h-2.5q-.75 0-.75.75v2.5q0 .75.75.75h2.5q.75 0 .75-.75z',
                'w' => 1.125,
                'vw' => 36
            ],
            'hospital-symbol' => 'M16 0q6.63 0 11.31 4.69T32.01 16t-4.7 11.31T16 32.01 4.69 27.3-.01 16 4.7 4.68 16 0zm7 23.5v-15q0-.5-.5-.5h-3q-.5 0-.5.5V14h-6V8.5q0-.5-.5-.5h-3Q9 8 9 8.5v15q0 .5.5.5h3q.5 0 .5-.5V18h6v5.5q0 .5.5.5h3q.5 0 .5-.5z',
            'hospital-user' => [
                'p' => 'M30 20a6 6 0 1 0 0-12 6 6 0 0 0 0 12Zm3 2c-.15 0-.3.02-.44.07a7.77 7.77 0 0 1-5.12 0c-.14-.05-.3-.07-.44-.07a7.01 7.01 0 0 0-7 7.04A3 3 0 0 0 23 32h14a3 3 0 0 0 3-2.96A7 7 0 0 0 33 22Zm-12.38.65A9 9 0 0 1 22 21.54V8a2 2 0 0 0-2-2h-2V2a2 2 0 0 0-2-2H6a2 2 0 0 0-2 2v4H2a2 2 0 0 0-2 2v23a1 1 0 0 0 1 1h18.02A5 5 0 0 1 18 29.05c-.02-2.4.93-4.7 2.62-6.4ZM9 25.25c0 .41-.34.75-.75.75h-2.5a.75.75 0 0 1-.75-.75v-2.5c0-.41.34-.75.75-.75h2.5c.41 0 .75.34.75.75Zm0-8c0 .41-.34.75-.75.75h-2.5a.75.75 0 0 1-.75-.75v-2.5c0-.41.34-.75.75-.75h2.5c.41 0 .75.34.75.75Zm3-7.63c0 .21-.17.38-.38.38h-1.24a.38.38 0 0 1-.38-.38V8H8.37A.38.38 0 0 1 8 7.62V6.38c0-.2.17-.37.38-.37H10V4.37c0-.2.17-.37.38-.37h1.24c.21 0 .38.17.38.38V6h1.63c.2 0 .37.17.37.38v1.25c0 .2-.17.37-.38.37H12Zm5 15.63c0 .41-.34.75-.75.75h-2.5a.75.75 0 0 1-.75-.75v-2.5c0-.41.34-.75.75-.75h2.5c.41 0 .75.34.75.75Zm0-8c0 .41-.34.75-.75.75h-2.5a.75.75 0 0 1-.75-.75v-2.5c0-.41.34-.75.75-.75h2.5c.41 0 .75.34.75.75Zm0 0',
                'vw' => 40
            ],
            'hot-tub' => 'M25.88 11.13q-.32-2.5-2-3.88-2.7-2.25-3.13-6.13-.06-.43.25-.78t.75-.34h1q.88 0 1 .88.31 2.5 2 3.87 2.69 2.25 3.13 6.13.06.43-.25.78t-.75.34h-1q-.88 0-1-.88zm-6.75 0q-.32-2.5-2-3.88Q14.43 5 14 1.12q-.06-.43.25-.78T15 0h1q.88 0 1 .88.31 2.5 2 3.87 2.69 2.25 3.13 6.13.06.43-.25.78t-.75.34h-1q-.88 0-1-.88zM30 16q.81 0 1.4.6T32 18v10q0 1.69-1.16 2.84T28 32H4q-1.69 0-2.84-1.16T0 28V14q0-1.69 1.16-2.84T4 10h2.69q1.31 0 2.37.81l6.94 5.2h14zM8 27.5v-7q0-.5-.5-.5h-1q-.5 0-.5.5v7q0 .5.5.5h1q.5 0 .5-.5zm6 0v-7q0-.5-.5-.5h-1q-.5 0-.5.5v7q0 .5.5.5h1q.5 0 .5-.5zm6 0v-7q0-.5-.5-.5h-1q-.5 0-.5.5v7q0 .5.5.5h1q.5 0 .5-.5zm6 0v-7q0-.5-.5-.5h-1q-.5 0-.5.5v7q0 .5.5.5h1q.5 0 .5-.5zM4 8Q2.31 8 1.16 6.84T0 4t1.16-2.84T4 0t2.84 1.16T8 4 6.84 6.84 4 8z',
            'hot-tub-person' => [
                'p' => 'M272 24c0-13.3-10.7-24-24-24s-24 10.7-24 24v5.2c0 34 14.4 66.4 39.7 89.2l16.4 14.8c15.2 13.7 23.8 33.1 23.8 53.5V200c0 13.3 10.7 24 24 24s24-10.7 24-24V186.8c0-34-14.4-66.4-39.7-89.2L295.8 82.8C280.7 69.1 272 49.7 272 29.2V24zM0 320v16V448c0 35.3 28.7 64 64 64H448c35.3 0 64-28.7 64-64V320c0-35.3-28.7-64-64-64H277.3c-13.8 0-27.3-4.5-38.4-12.8l-85.3-64C137 166.7 116.8 160 96 160c-53 0-96 43-96 96v64zm128 16v96c0 8.8-7.2 16-16 16s-16-7.2-16-16V336c0-8.8 7.2-16 16-16s16 7.2 16 16zm80-16c8.8 0 16 7.2 16 16v96c0 8.8-7.2 16-16 16s-16-7.2-16-16V336c0-8.8 7.2-16 16-16zm112 16v96c0 8.8-7.2 16-16 16s-16-7.2-16-16V336c0-8.8 7.2-16 16-16s16 7.2 16 16zm80-16c8.8 0 16 7.2 16 16v96c0 8.8-7.2 16-16 16s-16-7.2-16-16V336c0-8.8 7.2-16 16-16zM360 0c-13.3 0-24 10.7-24 24v5.2c0 34 14.4 66.4 39.7 89.2l16.4 14.8c15.2 13.7 23.8 33.1 23.8 53.5V200c0 13.3 10.7 24 24 24s24-10.7 24-24V186.8c0-34-14.4-66.4-39.7-89.2L407.8 82.8C392.7 69.1 384 49.7 384 29.2V24c0-13.3-10.7-24-24-24zM64 128A64 64 0 1 0 64 0a64 64 0 1 0 0 128z',
                'vw' => 512,
                'vh' => 512
            ],
            'hotdog' => 'M30.56 1.44q.25.31.53.75t.47.9.32.97T32 5q0 .88-.44 1.9t-1 1.66l-22 22q-.62.63-1.69 1.07t-1.93.43q-2.07 0-3.53-1.47t-1.47-3.53q0-.87.43-1.93t1.07-1.7l22-22q.62-.55 1.65-1T27 0t1.9.44 1.66 1zm-3.12 6q.25-.32.25-.72T27.4 6t-.7-.31T26 6q-.75.75-2.19 1-2 .38-3.22 1.6T19 11.8q-.24 1.44-1 2.2t-2.18 1q-2 .37-3.22 1.58T11 19.81q-.24 1.44-1 2.2t-2.18 1q-2 .37-3.25 1.55-.25.32-.25.72t.28.72.69.32T6 26q.75-.75 2.19-1 2-.37 3.21-1.6t1.6-3.2q.25-1.44 1-2.19t2.19-1q2-.37 3.22-1.6T21 12.2q.25-1.44 1-2.19t2.19-1q2-.37 3.25-1.56zM1.94 20.12l-.69-.68Q.06 18.24 0 16.66T1 14L14 1q1.06-1.06 2.66-1t2.78 1.25l.68.69zm28.12-8.25.7.7q1.18 1.18 1.24 2.77T31 18L18 31q-1.06 1.06-2.66 1t-2.78-1.25l-.68-.69z',
            'hotel' => [
                'p' => 'M35 4h-1v24h1q.44 0 .72.28T36 29v2q0 .44-.28.72T35 32H20v-5q0-.44-.28-.72T19 26h-2q-.44 0-.72.28T16 27v5H1q-.44 0-.72-.28T0 31v-2q0-.44.28-.72T1 28h1V4H1q-.44 0-.72-.28T0 3V1Q0 .56.28.28T1 0h34q.44 0 .72.28T36 1v2q0 .44-.28.72T35 4zM16 6.81V9.2q0 .31.25.56t.56.25h2.38q.31 0 .56-.25T20 9.2V6.8q0-.31-.25-.56T19.2 6h-2.4q-.31 0-.56.25t-.25.56zm0 6v2.38q0 .31.25.56t.56.25h2.38q.31 0 .56-.25t.25-.56V12.8q0-.31-.25-.56T19.2 12h-2.4q-.31 0-.56.25t-.25.56zm-8-6V9.2q0 .31.25.56t.56.25h2.38q.31 0 .56-.25T12 9.2V6.8q0-.31-.25-.56T11.2 6H8.8q-.31 0-.56.25T8 6.81zM11.19 16q.31 0 .56-.25t.25-.56V12.8q0-.31-.25-.56T11.2 12H8.8q